/*
 * Copyright (C) 2008 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef SMSMESSAGEBASE_H_
#define SMSMESSAGEBASE_H_

#include <cstdint>
#include <string>
#include <memory>
#include <vector>
#include "SmsAddress.h"
#include "SmsConstants.h"
#include "SmsHeader.h"
#include "HexDump.h"

class SmsMessageBase {
public:
  SmsMessageBase();
  virtual ~SmsMessageBase();
  /** TP-Message-Reference - Message Reference of sent message. @hide */
  int mMessageRef = 0;
  class SubmitPduBase {
  public:
    uint8_t* encodedScAddress = nullptr; // Null if not applicable.
    uint32_t encodedScAddress_length = 0;
    uint8_t* encodedMessage = nullptr;
    uint32_t encodedMessage_length = 0;

    std::string toString() {

      return "SubmitPdu: encodedScAddress = "
          + HexDump::dumpHexString(encodedScAddress, encodedScAddress_length)
          + ", encodedMessage = "
          + HexDump::dumpHexString(encodedMessage, encodedMessage_length);
    }
  };
  /**
   * Returns the address of the SMS service center that relayed this message
   * or null if there is none.
   */
  std::string getServiceCenterAddress() {
    return mScAddress;
  }
  std::string getOriginatingAddress() {
    if (mOriginatingAddress == nullptr) {
      return nullptr;
    }

    return mOriginatingAddress->getAddressString();
  }
  /**
   * Returns the originating address, or email from address if this message
   * was from an email gateway. Returns null if originating address
   * unavailable.
   */
  std::string getDisplayOriginatingAddress() {
    if (mIsEmail) {
      return mEmailFrom;
    } else {
      return getOriginatingAddress();
    }
  }
  /**
   * Returns the message body as a String, if it exists and is text based.
   * @return message body is there is one, otherwise null
   */
  std::string getMessageBody() {
    return mMessageBody;
  }

  /**
   * Returns the class of this message.
   */
  virtual SmsConstants::MessageClass getMessageClass() = 0;

  /**
   * Returns the message body, or email message body if this message was from
   * an email gateway. Returns null if message body unavailable.
   */
  std::string getDisplayMessageBody() {
    if (mIsEmail) {
      return mEmailBody;
    } else {
      return getMessageBody();
    }
  }

  /**
   * Unofficial convention of a subject line enclosed in parens empty string
   * if not present
   */
  std::string getPseudoSubject() {
    return mPseudoSubject.empty() ? "" : mPseudoSubject;
  }

  /**
   * Returns the service centre timestamp in currentTimeMillis() format
   */
  long getTimestampMillis() {
    return mScTimeMillis;
  }

  /**
   * Returns true if message is an email.
   *
   * @return true if this message came through an email gateway and email
   *         sender / subject / parsed body are available
   */
  bool isEmail() {
    return mIsEmail;
  }

  /**
   * @return if isEmail() is true, body of the email sent through the gateway.
   *         null otherwise
   */
  std::string getEmailBody() {
    return mEmailBody;
  }

  /**
   * @return if isEmail() is true, email from address of email sent through
   *         the gateway. null otherwise
   */
  std::string getEmailFrom() {
    return mEmailFrom;
  }

  /**
   * Get protocol identifier.
   */
  virtual int getProtocolIdentifier() = 0;

  /**
   * See TS 23.040 9.2.3.9 returns true if this is a "replace short message"
   * SMS
   */
  virtual bool isReplace() = 0;

  /**
   * Returns true for CPHS MWI toggle message.
   *
   * @return true if this is a CPHS MWI toggle message See CPHS 4.2 section
   *         B.4.2
   */
  virtual bool isCphsMwiMessage() = 0;

  /**
   * returns true if this message is a CPHS voicemail / message waiting
   * indicator (MWI) clear message
   */
  virtual bool isMWIClearMessage() = 0;

  /**
   * returns true if this message is a CPHS voicemail / message waiting
   * indicator (MWI) set message
   */
  virtual bool isMWISetMessage() = 0;

  /**
   * returns true if this message is a "Message Waiting Indication Group:
   * Discard Message" notification and should not be stored.
   */
  virtual bool isMwiDontStore() = 0;

  /**
   * returns the user data section minus the user data header if one was
   * present.
   */
  std::vector<uint8_t> getUserData() {
    return mUserData;
  }

  /**
   * Returns an object representing the user data header
   *
   * {@hide}
   */
  std::shared_ptr<SmsHeader> getUserDataHeader() {
    return mUserDataHeader;
  }

  /**
   * TODO(cleanup): The term PDU is used in a seemingly non-unique
   * manner -- for example, what is the difference between this byte
   * array and the contents of SubmitPdu objects.  Maybe a more
   * illustrative term would be appropriate.
   */

  /**
   * Returns the raw PDU for the message.
   */
  std::vector<uint8_t> getPdu() {
    return mPdu;
  }

  /**
   * For an SMS-STATUS-REPORT message, this returns the status field from
   * the status report.  This field indicates the status of a previously
   * submitted SMS, if requested.  See TS 23.040, 9.2.3.15 TP-Status for a
   * description of values.
   *
   * @return 0 indicates the previously sent message was received.
   *         See TS 23.040, 9.9.2.3.15 for a description of other possible
   *         values.
   */
  virtual int getStatus() = 0;

  /**
   * Return true iff the message is a SMS-STATUS-REPORT message.
   */
  virtual bool isStatusReportMessage() = 0;

  /**
   * Returns true iff the <code>TP-Reply-Path</code> bit is set in
   * this message.
   */
  virtual bool isReplyPathPresent() = 0;

  /**
   * Returns the status of the message on the ICC (read, unread, sent, unsent).
   *
   * @return the status of the message on the ICC.  These are:
   *         SmsManager.STATUS_ON_ICC_FREE
   *         SmsManager.STATUS_ON_ICC_READ
   *         SmsManager.STATUS_ON_ICC_UNREAD
   *         SmsManager.STATUS_ON_ICC_SEND
   *         SmsManager.STATUS_ON_ICC_UNSENT
   */
  int getStatusOnIcc() {
    return mStatusOnIcc;
  }

  /**
   * Returns the record index of the message on the ICC (1-based index).
   * @return the record index of the message on the ICC, or -1 if this
   *         SmsMessage was not created from a ICC SMS EF record.
   */
  int getIndexOnIcc() {
    return mIndexOnIcc;
  }
protected:
  void parseMessageBody();
  void extractEmailAddressFromMessageBody();
protected:
  /** {@hide} The address of the SMSC. May be null */
  std::string mScAddress;

  /** {@hide} The address of the sender */
  std::shared_ptr<SmsAddress> mOriginatingAddress = nullptr;

  /** {@hide} The message body as a string. May be null if the message isn't text */
  std::string mMessageBody;

  /** {@hide} */
  std::string mPseudoSubject;

  /** {@hide} Non-null if this is an email gateway message */
  std::string mEmailFrom;

  /** {@hide} Non-null if this is an email gateway message */
  std::string mEmailBody;

  /** {@hide} */
  bool mIsEmail = false;

  /** {@hide} Time when SC (service centre) received the message */
  long mScTimeMillis;

  /** {@hide} The raw PDU of the message */
  //uint8_t* mPdu = nullptr;
  //uint32_t mPdu_length = 0;
  std::vector<uint8_t> mPdu;

  /** {@hide} The raw bytes for the user data section of the message */
  std::vector<uint8_t> mUserData;
//  uint8_t* mUserData = nullptr;
//  uint32_t mUserData_length = 0;

  /** {@hide} */

  std::shared_ptr<SmsHeader> mUserDataHeader;

  // "Message Waiting Indication Group"
  // 23.038 Section 4
  /** {@hide} */
  bool mIsMwi = false;

  /** {@hide} */
  bool mMwiSense = false;

  /** {@hide} */
  bool mMwiDontStore = false;

  /**
   * Indicates status for messages stored on the ICC.
   */
  int mStatusOnIcc = -1;

  /**
   * Record index of message in the EF.
   */
  int mIndexOnIcc = -1;

};

#endif /* SMSMESSAGEBASE_H_ */
