/**
 * @file Hal_codec_tlv3100.c
 * @brief Implementation of Sanechips Audio Codec Function
 *
 * Copyright (C) 2017 Sanechips Technology Co., Ltd.
 * @author Author: Xinqiang Xu <xu.xinqiang@sanechips.com.cn>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. 
 *
 */
 
/*******************************************************************************
 *							 Include header files							   *
 ******************************************************************************/
#include "drvs_general.h"
#include "drvs_i2c.h"
#include "drvs_pm.h"
#include "drvs_codec.h"
#include "drvs_pow.h"
#include "drvs_ramlog.h"
#include "hal_codec_tlv3100.h"

/*******************************************************************************
 *                             Type definitions                                *
 ******************************************************************************/
typedef enum {
    CODEC_CLKIN_MCLK,
    CODEC_CLKIN_BCLK,
    CODEC_CLKIN_GPIO1,
    CODEC_CLKIN_PLLCLK,
    CODEC_CLKIN_MAX
} T_ZDrvCodec_AudioClockIn;

typedef enum {
    CODEC_IDLE,
    CODEC_OPEN,
    CODEC_ENABLE,
    CODEC_DISABLE,
    CODEC_RESET,
    CODEC_STATUS_MAX
} T_AudCodec_UseStatus;

typedef enum {
    HANDLE_FREE,
    HANDLE_IN_USE,
    HANDLE_USE_MAX
}T_AudCodec_HandleStatus;

typedef struct _T_Codec_UseInfo {
	T_ZDrvCodec_Handle handle;
	T_AudCodec_HandleStatus handleState;
	T_ZDrvCodec_AudioI2SChannel audI2SChannel;
	T_ZDrvCodec_AudioI2SMode masterSlave;
	T_ZDrvCodec_AudioSample sampleRate;
	T_ZDrvCodec_AudioClockIn clockIn;
	T_AudCodec_UseStatus useState;
	T_AudCodec_BusMode  busMode;
	T_ZDrvCodec_InDevice	inDev;
	T_ZDrvCodec_OutDevice	outDev;
	T_ZDrvCodec_SourceType srcType;
} T_Codec_UseInfo;

typedef struct _T_CLK_DIV {
	UINT32 fs;
	UINT8 ndiv;
	UINT8 mdiv;
	UINT16 sordiv;
} T_CLK_DIV;

typedef struct _T_PLL_CLK_DIV {
	UINT32 fs;
	UINT8 pll_pdiv;
	UINT8 pll_rmulti;
	UINT8  pll_jmulti;
	UINT16 pll_dval;
	UINT8 nda_div;
	UINT8 mda_div;
	UINT16 dsor_div;
	UINT8 nad_div;
	UINT8 mad_div;
	UINT16 asor_div;
} T_PLL_CLK_DIV;


/*******************************************************************************
 *                        Local function declarations                          *
 ******************************************************************************/
SINT32 aud_SetCodecPWOnOff(BOOL onOff);
SINT32 codec_SetFmInPath(T_ZDrvCodec_Handle handle, BOOL onoff);
SINT32 codec_EarpieceDetectEnable(BOOL onoff);
extern SINT32 zDrvI2S_setCodecClk(VOID);
SINT32 zDrvCodec_GetOutPathParaInfo(T_CodecPathParaInfo *pathinfo,T_ZDrvCodec_OutDevice dev,T_ZDrvCodec_SourceType sourcetype,BOOL onoff);//xiu
SINT32 zDrvCodec_GetInPathParaInfo(T_CodecPathParaInfo *pathinfo,T_ZDrvCodec_InDevice dev,BOOL onoff);//xiu
/*******************************************************************************
 *						   Local variable definitions						   *
 ******************************************************************************/
static BOOL isEarpDetect = 0;
static ZOSS_MUTEX_ID s_codecMutexID = NULL;
 #if 0
static T_CLK_DIV clkDiv[] = {
	{8000,     5,   13,   50},
	{16000,   5,   5,     61},
	{44100,   2,    5,    59},
	{48000,   2,    5,    54}
};
#endif
static T_PLL_CLK_DIV pll_clkDiv[] = {
	{8000,   2,    1,   6, 3803,   3,    27,   128,   3,    27,   128},
	{16000,   2,    1,   6, 6166,   3,    14,   128,   3,    14,   128},
	{44100,   2,    1,   6, 5132, 3 , 5, 128,  3,    5,   128},
	{48000,   2,    1,   6, 6166,   7,    2,   128,   7,    2,   128},
	{11025,   2,    1,   6, 1876,   3,    19,   128,   3,    19,   128},
	{12000,   2,    1,   6, 3803,   3,    18,   128,   3,    18,   128},
	{32000,   2,    1,   6, 6166,   3,    7,   128,   3,    7,   128}
};
extern T_ZDrvI2c_Device s_codecI2CDev ;


static T_Codec_UseInfo  voiceUseInfo = {
	(unsigned long)NULL,//xiu
	HANDLE_FREE,
	AUDIO_I2S0,
	I2S_SLAVE_MODE,
	AUDIO_SAMPLE_8K,
	CODEC_CLKIN_MCLK,/* 26MHZ */
	CODEC_IDLE,
	CODEC_BUS_I2S,
	AUDIO_INPUT_MAX,
	AUDIO_OUTPUT_MAX,
	VOICE_SOURCE
};

static T_Codec_UseInfo  audioUseInfo = {
	(unsigned long)NULL,
	HANDLE_FREE,
	AUDIO_I2S0,
	I2S_SLAVE_MODE,
	AUDIO_SAMPLE_8K,
	CODEC_CLKIN_MCLK,/* 26MHZ */
	CODEC_IDLE,
	CODEC_BUS_I2S,
	AUDIO_INPUT_MAX,
	AUDIO_OUTPUT_MAX,
	AUDIO_SOURCE
};

static T_Codec_UseInfo  fmUseInfo = {
	(unsigned long)NULL,
	HANDLE_FREE,
	AUDIO_I2S0,
	I2S_SLAVE_MODE,
	AUDIO_SAMPLE_8K,
	CODEC_CLKIN_MCLK,/* 26MHZ */
	CODEC_IDLE,
	CODEC_BUS_I2S,
	AUDIO_INPUT_MAX,
	AUDIO_OUTPUT_MAX,
	EXTERN_AUDIO
};
extern UINT8 speakerOutIirHpfPara[2][20];
extern T_Codec_Reg_Para  inPathResetPara[6] ;

/* for reference;SPK output path reference config */
#if 0//xiu
static T_Codec_Reg_Para spk_outPathParaCfgRef[] = {
	/* page0 reg63: DAC Data-Path Setup */
	{
		0, 0, 63, \
		/* bits */
		CODEC_P0_REG63_LDAC_PWON_VAL | CODEC_P0_REG63_LDAC_EQU_LRMIXER_VAL | CODEC_P0_REG63_RDAC_PWON_VAL | \
		CODEC_P0_REG63_RDAC_EQU_LRMIXER_VAL | CODEC_P0_REG63_DAC_VOLCTR_SSTEP_1FS_VAL, \
		/* mask */
		CODEC_P0_REG63_LDAC_PW_MASK | CODEC_P0_REG63_LDAC_EQU_CTRL_MASK | CODEC_P0_REG63_RDAC_PW_MASK | \
		CODEC_P0_REG63_RDAC_EQU_CTRL_MASK | CODEC_P0_REG63_DAC_VOLCTR_SSTEP_CTRL_MASK
	},
	/* page0 reg64:DAC VOLUME CONTROL */
	{
		0, 0, 64, \
		/* bits */
		CODEC_P0_REG64_DAC_LRCHN_VOLCTRL_INDEP_VAL, \
		/* mask */ 
		CODEC_P0_REG64_DAC_RCHN_VOLCTRL_MASK
	},
	/* page0 reg65:DAC Left Volume Control */
	{
		0, 0, 65, \
		CODEC_P0_REG65_DAC_LCHN_VOL_MINUS_3DB_VAL, \
		CODEC_P0_REG65_DAC_LCHN_VOL_MASK
	},
	/* page0 reg66:DAC Right Volume Control */
	{
		0, 0, 66, \
		CODEC_P0_REG66_DAC_RCHN_VOL_MINUS_3DB_VAL, \
		CODEC_P0_REG66_DAC_RCHN_VOL_MASK
	},
	/* page1 reg31:  Headphone Drivers */
	{
		0, 1, 31, \
		0x04, \
		0xff
	},
	/* page1 reg32:   Class-D Speaker Amplifier */
	{
		0, 1, 32, \
		CODEC_P1_REG32_WRITE_ONLY_VAL | CODEC_P1_REG32_SPKPM_PWON_VAL, \
		CODEC_P1_REG32_WRITE_ONLY_VAL_MASK | CODEC_P1_REG32_SPKPM_PW_MASK
	},
	/* page1 reg33:  HP Output Drivers POP Removal Settings */
	{
		0, 1, 33, \
		0x3e, \
		0xff
	},
	/* page1 reg34:  Output Driver PGA Ramp-Down Period Control */
	{
		0, 1, 34, \
		CODEC_P1_REG34_SPK_PWON_WAIT_19_8MS_VAL, \
		CODEC_P1_REG34_SPK_PWON_WAIT_MS_CTRL_MASK
	},
	/* page1 reg35:  DAC_L and DAC_R Output Mixer Routing */
	{
		0, 1, 35, \
		CODEC_P1_REG35_DACL_ROUTE_LCHN_MIXER_VAL | CODEC_P1_REG35_DACR_ROUTE_RCHN_MIXER_VAL, \
		CODEC_P1_REG35_DACL_ROUTE_CTRL_MASK | CODEC_P1_REG35_DACR_ROUTE_CTRL_MASK
	},
	/* page1 reg36: Left Analog Vol to HPL */
	{
		0, 1, 36, \
		0x7f, \
		0xff
	},
	/* page1 reg37: Right Analog Vol to HPR */
	{
		0, 1, 37, \
		0x7f, \
		0xff
	},
	/* page1 reg38: Left Analog Vol to SPK */
	{
		0, 1, 38, \
		CODEC_P1_REG38_LAVOL_TO_SPK_VAL | CODEC_P1_REG38_LAVOL_CTRL_FORSPK_0DB_VAL, \
		CODEC_P1_REG38_LAVOL_TO_SPK_CTRL_MASK | CODEC_P1_REG38_LAVOL_SPK_GAIN_MASK
	},
	/* page1 reg40: HPL Driver */
	{
		0, 1, 40, \
		0x02, \
		0xff
	},
	/* page1 reg41: HPR Driver */
	{
		0, 1, 41, \
		0x02, \
		0xff
	},
	/* page1 reg42: SPK Driver */
	{
		0, 1, 42, \
		CODEC_P1_REG42_SPK_DRV_SGAIN_6DB_VAL | CODEC_P1_REG42_SPK_DRV_UNMUTE_VAL, \
		CODEC_P1_REG42_SPK_DRV_SGAIN_MASK | CODEC_P1_REG42_SPK_DRV_MUTE_CTRL_MASK
	},
	/* page1 reg44: HP Driver Control */
	{
		0, 1, 44, \
		0x00, \
		0xff
	}
};

/* for reference;HP output path reference config */
static T_Codec_Reg_Para hp_outPathParaCfgRef[] = {
	/* page0 reg63: DAC Data-Path Setup */
	{
		0, 0, 63, \
		/* bits */
		CODEC_P0_REG63_LDAC_PWON_VAL | CODEC_P0_REG63_LDAC_EQU_LRMIXER_VAL | CODEC_P0_REG63_RDAC_PWON_VAL | \
		CODEC_P0_REG63_RDAC_EQU_LRMIXER_VAL | CODEC_P0_REG63_DAC_VOLCTR_SSTEP_1FS_VAL, \
		/* mask */
		CODEC_P0_REG63_LDAC_PW_MASK | CODEC_P0_REG63_LDAC_EQU_CTRL_MASK | CODEC_P0_REG63_RDAC_PW_MASK | \
		CODEC_P0_REG63_RDAC_EQU_CTRL_MASK | CODEC_P0_REG63_DAC_VOLCTR_SSTEP_CTRL_MASK
	},
	/* page0 reg64:DAC VOLUME CONTROL */
	{
		0, 0, 64, \
		/* bits */
		CODEC_P0_REG64_DAC_LRCHN_VOLCTRL_INDEP_VAL, \
		/* mask */
		CODEC_P0_REG64_DAC_RCHN_VOLCTRL_MASK
	},
	/* page0 reg65:DAC Left Volume Control */
	{
		0, 0, 65, \
		CODEC_P0_REG65_DAC_LCHN_VOL_MINUS_3DB_VAL, \
		CODEC_P0_REG65_DAC_LCHN_VOL_MASK
	},
	/* page0 reg66:DAC Right Volume Control */
	{
		0, 0, 66, \
		CODEC_P0_REG66_DAC_RCHN_VOL_MINUS_3DB_VAL, \
		CODEC_P0_REG66_DAC_RCHN_VOL_MASK
	},
	/* page1 reg31:  Headphone Drivers */
	{
		0, 1, 31, \
		CODEC_P1_REG31_HP_OUT_1_35V_VAL | CODEC_P1_REG31_HPL_PWON_VAL | \
		CODEC_P1_REG31_HPR_PWON_VAL , \
		CODEC_P1_REG31_HP_OUT_VOL_MASK | CODEC_P1_REG31_HPL_PW_MASK | \
		CODEC_P1_REG31_HPR_PW_MASK
	},
	/* page1 reg32:   Class-D Speaker Amplifier */
	{
		0, 1, 32, \
		0x06, \
		CODEC_P1_REG32_SPKPM_PW_MASK
	},
	/* page1 reg33:  HP Output Drivers POP Removal Settings */
	{
		0, 1, 33, \
		CODEC_P1_REG33_HPDRI_REVSEQ_AFT_VAL | CODEC_P1_REG33_HPDRI_PWON_15_3MS_VAL | \
		CODEC_P1_REG33_HPDRI_RAMPUP_3_9MS_VAL, \
		CODEC_P1_REG33_HPDRI_REVSEQ_MASK | CODEC_P1_REG33_HPDRI_PWON_TIME_MASK | \
		CODEC_P1_REG33_HPDRI_RAMPUP_TIME_MASK
	},
	/* page1 reg34:  Output Driver PGA Ramp-Down Period Control */
	{
		0, 1, 34, \
		CODEC_P1_REG34_SPK_PWON_WAIT_19_8MS_VAL, \
		CODEC_P1_REG34_SPK_PWON_WAIT_MS_CTRL_MASK
	},
	/* page1 reg35:  DAC_L and DAC_R Output Mixer Routing */
	{
		0, 1, 35, \
		CODEC_P1_REG35_DACL_ROUTE_HPL_DRV_VAL |  CODEC_P1_REG35_DACR_ROUTE_HPR_DRV_VAL | \
		CODEC_P1_REG35_HPL_OUTPUT_UNROUTE_HPR_DRV_VAL, \
		CODEC_P1_REG35_DACL_ROUTE_CTRL_MASK | CODEC_P1_REG35_DACR_ROUTE_CTRL_MASK | \
		CODEC_P1_REG35_HPL_OUTUT_ROUTE_CTRL_MASK
	},
	/* page1 reg36: Left Analog Vol to HPL */
	{
		0, 1, 36, \
		CODEC_P1_REG36_LAVOL_TO_HPL_VAL | CODEC_P1_REG36_LAVOL_CTRL_FORHPL_0DB_VAL, \
		CODEC_P1_REG36_LAVOL_TO_HPL_CTRL_MASK | CODEC_P1_REG36_LAVOL_HPL_GAIN_MASK
	},
	/* page1 reg37: Right Analog Vol to HPR */
	{
		0, 1, 37, \
		CODEC_P1_REG37_RAVOL_TO_HPR_VAL | CODEC_P1_REG37_RAVOL_CTRL_FORHPR_0DB_VAL, \
		CODEC_P1_REG37_RAVOL_TO_HPR_CTRL_MASK | CODEC_P1_REG37_RAVOL_HPR_GAIN_MASK
	},
	/* page1 reg38: Left Analog Vol to SPK */
	{
		0, 1, 38, \
		0x7f, \
		CODEC_P1_REG38_LAVOL_TO_SPK_CTRL_MASK | CODEC_P1_REG38_LAVOL_SPK_GAIN_MASK
	},
	/* page1 reg40: HPL Driver */
	{
		0, 1, 40, \
		CODEC_P1_REG40_HPL_DRV_PGA_0DB_VAL | CODEC_P1_REG40_HPL_DRV_UNMUTE_VAL, \
		CODEC_P1_REG40_HPL_DRV_PGA_CTRL_MASK | CODEC_P1_REG40_HPL_DRV_MUTE_CTRL_MASK
	},
	/* page1 reg41: HPR Driver */
	{
		0, 1, 41, \
		CODEC_P1_REG41_HPR_DRV_PGA_0DB_VAL | CODEC_P1_REG41_HPR_DRV_UNMUTE_VAL, \
		CODEC_P1_REG41_HPR_DRV_PGA_CTRL_MASK | CODEC_P1_REG41_HPR_DRV_MUTE_CTRL_MASK
	},
	/* page1 reg42: SPK Driver */
	{
		0, 1, 42, \
		CODEC_P1_REG42_SPK_DRV_MUTE_VAL | CODEC_P1_REG42_SPK_DRV_SGAIN_6DB_VAL, \
		CODEC_P1_REG42_SPK_DRV_MUTE_CTRL_MASK | CODEC_P1_REG42_SPK_DRV_SGAIN_MASK
	},
	/* page1 reg44: HP Driver Control */
	{
		0, 1, 44, \
		CODEC_P1_REG44_HP_DRV_DEBOUNCE_0US_VAL | CODEC_P1_REG44_HPL_DRV_AS_HEADPHONE_VAL | \
		CODEC_P1_REG44_HPR_DRV_AS_HEADPHONE_VAL, \
		CODEC_P1_REG44_HP_DRV_DEBOUNCE_US_CTRL_MASK | CODEC_P1_REG44_HPL_DRV_AS_CTRL_MASK | \
		CODEC_P1_REG44_HPR_DRV_AS_CTRL_MASK
	}
};
#endif
/* fm  digital loop input channel */
static UINT8 handsetFMDigitalInPathPara[] = {//xiu
	/* page1 reg46: MICBIAS 0x0a */
	(CODEC_P1_REG46_MICBIAS_OUT_PWOFF_VAL | CODEC_P1_REG46_MICBIAS_PWON_CTRL2_VAL) ,
	/* page1 reg47:MIC PGA 0x14 */
	(CODEC_P1_REG47_MICPGA_CTRL_BY_D06_VAL | CODEC_P1_REG47_MICPGA_20DB_VAL),
	/* page1 reg48:Delta-Sigma Mono ADC Channel Fine-Gain Input Selection for P-Terminal */
	(CODEC_P1_REG48_MIC1LP_SEL_10KR_VAL),
	/* page1 reg49:ADC Input Selection for M-Terminal */
	(CODEC_P1_REG49_CM_SEL_10KR_VAL),
	/* page0 reg81:ADC Digital Mic  0x80 */
	(CODEC_P0_REG81_ADC_PWON_VAL | CODEC_P0_REG81_ADC_VOLCTRL_SSTEP_1FS_VAL),
	/* page0 reg82:ADC Digital Volume Control Fine Adjust */
	(CODEC_P0_REG82_ADC_DIGVOL_FINE_0DB_VAL ),
	/* page0 reg83:ADC Digital Volume Control Coarse Adjust */
	(CODEC_P0_REG83_ADC_DIGVOL_COARSE_0DB_VAL)
};

static T_Codec_Reg_Para i2sSlaveModePara[] = {
	/* page0 reg27: Codec Interface Control */
	{
		0, 0, 27, \
		/* bits */
		CODEC_P0_REG27_BUS_I2S_VAL | CODEC_P0_REG27_BCLK_INPUT_VAL | CODEC_P0_REG27_WCLK_INPUT_VAL, \
		/* mask */
		CODEC_P0_REG27_BUS_MODE_MASK | CODEC_P0_REG27_BUS_WORD_LENGTH_MASK | CODEC_P0_REG27_BCLK_PIN_MASK | \
		CODEC_P0_REG27_WCLK_PIN_MASK
	},
	/* page0 reg28:	Data-Slot Offset Programmability */
	{
		0, 0, 28, \
		/* bits ,0 is std timing */
		CODEC_P0_REG28_DATASLOT_OFFSET_0BCLK_VAL, \
		/* mask */
		CODEC_P0_REG28_DATASLOT_OFFSET_MASK
	},
	/* page0 reg29:	 Codec Interface Control 2 */
	{
		0, 0, 29, \
		/* bits */
		CODEC_P0_REG29_BCLK_INVERTED_DIS_VAL | CODEC_P0_REG29_BDIVCLKIN_EQUAL_DACMODCLK_VAL, \
		/* mask */
		CODEC_P0_REG29_BCLK_INVERTED_CTRL_MASK | CODEC_P0_REG29_BDIVCLKIN_EQUAL_CTRL_MASK
	},
	/* page0 reg32:	 Codec Secondary Interface Control 3 */
	{
		0, 0, 32, \
		/* bits */
		CODEC_P0_REG32_PRIMBCLK_FDTO_SERINTER_VAL | CODEC_P0_REG32_PRIMWBCLK_FDTO_SERINTER_VAL | CODEC_P0_REG32_PRIMDIN_FDTO_SERINTER_VAL, \
		/* mask */
		CODEC_P0_REG32_BCLK_FDTO_SERINTER_CTRL_MASK | CODEC_P0_REG32_WCLK_FDTO_SERINTER_CTRL_MASK | CODEC_P0_REG32_DIN_FDTO_SERINTER_CTRL_MASK
	},
	/* page0 reg33:	 Codec Secondary Interface Control 3 */
	{
		0, 0, 33, \
		/* bits */
		CODEC_P0_REG33_PRIMBCLK_EQ_INTERBCLK_VAL | CODEC_P0_REG33_PRIMWCLK_EQ_INTERDACFS_VAL | CODEC_P0_REG33_PRIMDOUT_EQ_INTERDOUT_VAL, \
		/* mask */
		CODEC_P0_REG33_PRIMBCLK_EQ_CTRL_MASK | CODEC_P0_REG33_PRIMWCLK_EQ_CTRL_MASK | CODEC_P0_REG33_PRIMDOUT_EQ_CTRL_MASK
	},
	/* page0 reg53:	 DOUT (OUT Pin) Control */
	{
		0, 0, 53, \
		/* bits */
		CODEC_P0_REG53_DOUT_POUT_VAL | CODEC_P0_REG53_DOUT_KEEP_DIS_VAL, \
		/* mask */
		CODEC_P0_REG53_DOUT_CTRL_MASK | CODEC_P0_REG53_DOUT_KEEP_MASK
	}
};

static T_Codec_Reg_Para i2sMasterModePara[] = {
	/* page0 reg27: Codec Interface Control */
	{
		0, 0, 27, \
		/* bits */
		CODEC_P0_REG27_BUS_I2S_VAL | CODEC_P0_REG27_BCLK_OUTPUT_VAL | CODEC_P0_REG27_WCLK_OUTPUT_VAL, \
		/* mask */
		CODEC_P0_REG27_BUS_MODE_MASK | CODEC_P0_REG27_BUS_WORD_LENGTH_MASK | CODEC_P0_REG27_BCLK_PIN_MASK | \
		CODEC_P0_REG27_WCLK_PIN_MASK
	},
	/* page0 reg28:	Data-Slot Offset Programmability */
	{
		0, 0, 28, \
		/* bits ,0 is std timing */
		CODEC_P0_REG28_DATASLOT_OFFSET_0BCLK_VAL, \
		/* mask */
		CODEC_P0_REG28_DATASLOT_OFFSET_MASK
	},
	/* page0 reg29:	 Codec Interface Control 2 */
	{
		0, 0, 29, \
		/* bits */
		CODEC_P0_REG29_BCLK_INVERTED_DIS_VAL | CODEC_P0_REG29_BDIVCLKIN_EQUAL_DACMODCLK_VAL, \
		/* mask */
		CODEC_P0_REG29_BCLK_INVERTED_CTRL_MASK | CODEC_P0_REG29_BDIVCLKIN_EQUAL_CTRL_MASK
	},
	/* page0 reg32:	 Codec Secondary Interface Control 3 */
	{
		0, 0, 32, \
		/* bits */
		CODEC_P0_REG32_PRIMBCLK_FDTO_SERINTER_VAL | CODEC_P0_REG32_PRIMWBCLK_FDTO_SERINTER_VAL | CODEC_P0_REG32_PRIMDIN_FDTO_SERINTER_VAL, \
		/* mask */
		CODEC_P0_REG32_BCLK_FDTO_SERINTER_CTRL_MASK | CODEC_P0_REG32_WCLK_FDTO_SERINTER_CTRL_MASK | CODEC_P0_REG32_DIN_FDTO_SERINTER_CTRL_MASK
	},
	/* page0 reg33:	 Codec Secondary Interface Control 3 */
	{
		0, 0, 33, \
		/* bits */
		CODEC_P0_REG33_PRIMBCLK_EQ_INTERBCLK_VAL | CODEC_P0_REG33_PRIMWCLK_EQ_INTERDACFS_VAL | CODEC_P0_REG33_PRIMDOUT_EQ_INTERDOUT_VAL, \
		/* mask */
		CODEC_P0_REG33_PRIMBCLK_EQ_CTRL_MASK | CODEC_P0_REG33_PRIMWCLK_EQ_CTRL_MASK | CODEC_P0_REG33_PRIMDOUT_EQ_CTRL_MASK
	},
	/* page0 reg53:	 DOUT (OUT Pin) Control */
	{
		0, 0, 53, \
		/* bits */
		CODEC_P0_REG53_DOUT_POUT_VAL | CODEC_P0_REG53_DOUT_KEEP_DIS_VAL, \
		/* mask */
		CODEC_P0_REG53_DOUT_CTRL_MASK | CODEC_P0_REG53_DOUT_KEEP_MASK
	}
};

static T_Codec_Reg_Para pcmSlaveModePara[] = {
	/* page0 reg27: Codec Interface Control */
	{
		0, 0, 27, \
		/* bits */
		CODEC_P0_REG27_BUS_DSP_VAL | CODEC_P0_REG27_BCLK_INPUT_VAL | CODEC_P0_REG27_WCLK_INPUT_VAL, \
		/* mask */
		CODEC_P0_REG27_BUS_MODE_MASK | CODEC_P0_REG27_BUS_WORD_LENGTH_MASK | CODEC_P0_REG27_BCLK_PIN_MASK | \
		CODEC_P0_REG27_WCLK_PIN_MASK
	},
	/* page0 reg28:	Data-Slot Offset Programmability */
	{
		0, 0, 28, \
		/* bits ,0 is std timing */
		CODEC_P0_REG28_DATASLOT_OFFSET_0BCLK_VAL, \
		/* mask */
		CODEC_P0_REG28_DATASLOT_OFFSET_MASK
	},
	/* page0 reg29:	 Codec Interface Control 2 */
	{
		0, 0, 29, \
		/* bits */
		CODEC_P0_REG29_BCLK_INVERTED_DIS_VAL | CODEC_P0_REG29_BDIVCLKIN_EQUAL_DACMODCLK_VAL, \
		/* mask */
		CODEC_P0_REG29_BCLK_INVERTED_CTRL_MASK | CODEC_P0_REG29_BDIVCLKIN_EQUAL_CTRL_MASK
	},
	/* page0 reg32:	 Codec Secondary Interface Control 3  */
	{
		0, 0, 32, \
		/* bits */
		CODEC_P0_REG32_PRIMBCLK_FDTO_SERINTER_VAL | CODEC_P0_REG32_PRIMWBCLK_FDTO_SERINTER_VAL | CODEC_P0_REG32_PRIMDIN_FDTO_SERINTER_VAL, \
		/* mask */
		CODEC_P0_REG32_BCLK_FDTO_SERINTER_CTRL_MASK | CODEC_P0_REG32_WCLK_FDTO_SERINTER_CTRL_MASK | CODEC_P0_REG32_DIN_FDTO_SERINTER_CTRL_MASK
	},
	/* page0 reg33:	 Codec Secondary Interface Control 3 */
	{
		0, 0, 33, \
		/* bits */
		CODEC_P0_REG33_PRIMBCLK_EQ_INTERBCLK_VAL | CODEC_P0_REG33_PRIMWCLK_EQ_INTERDACFS_VAL | CODEC_P0_REG33_PRIMDOUT_EQ_INTERDOUT_VAL, \
		/* mask */
		CODEC_P0_REG33_PRIMBCLK_EQ_CTRL_MASK | CODEC_P0_REG33_PRIMWCLK_EQ_CTRL_MASK | CODEC_P0_REG33_PRIMDOUT_EQ_CTRL_MASK
	},
	/* page0 reg53:	 DOUT (OUT Pin) Control */
	{
		0, 0, 53, \
		/* bits */
		CODEC_P0_REG53_DOUT_POUT_VAL | CODEC_P0_REG53_DOUT_KEEP_DIS_VAL, \
		/* mask */
		CODEC_P0_REG53_DOUT_CTRL_MASK | CODEC_P0_REG53_DOUT_KEEP_MASK
	}
};

static T_Codec_Reg_Para pcmMasterModePara[] = {
	/* page0 reg27: Codec Interface Control */
	{
		0, 0, 27, \
		/* bits */
		CODEC_P0_REG27_BUS_I2S_VAL | CODEC_P0_REG27_BCLK_OUTPUT_VAL | CODEC_P0_REG27_WCLK_OUTPUT_VAL, \
		/* mask */
		CODEC_P0_REG27_BUS_MODE_MASK | CODEC_P0_REG27_BUS_WORD_LENGTH_MASK | CODEC_P0_REG27_BCLK_PIN_MASK | \
		CODEC_P0_REG27_WCLK_PIN_MASK
	},
	/* page0 reg28:	Data-Slot Offset Programmability */
	{
		0, 0, 28, \
		/* bits ,0 is std timing*/
		CODEC_P0_REG28_DATASLOT_OFFSET_0BCLK_VAL, \
		/* mask */
		CODEC_P0_REG28_DATASLOT_OFFSET_MASK
	},
	/* page0 reg29:	 Codec Interface Control 2 */
	{
		0, 0, 29, \
		/* bits */
		CODEC_P0_REG29_BCLK_INVERTED_DIS_VAL | CODEC_P0_REG29_BDIVCLKIN_EQUAL_DACMODCLK_VAL, \
		/* mask */
		CODEC_P0_REG29_BCLK_INVERTED_CTRL_MASK | CODEC_P0_REG29_BDIVCLKIN_EQUAL_CTRL_MASK
	},
	/* page0 reg32:	 Codec Secondary Interface Control 3 */
	{
		0, 0, 32, \
		/* bits */
		CODEC_P0_REG32_PRIMBCLK_FDTO_SERINTER_VAL | CODEC_P0_REG32_PRIMWBCLK_FDTO_SERINTER_VAL | CODEC_P0_REG32_PRIMDIN_FDTO_SERINTER_VAL, \
		/* mask */
		CODEC_P0_REG32_BCLK_FDTO_SERINTER_CTRL_MASK | CODEC_P0_REG32_WCLK_FDTO_SERINTER_CTRL_MASK | CODEC_P0_REG32_DIN_FDTO_SERINTER_CTRL_MASK
	},
	/* page0 reg33:	 Codec Secondary Interface Control 3 */
	{
		0, 0, 33, \
		/* bits */
		CODEC_P0_REG33_PRIMBCLK_EQ_INTERBCLK_VAL | CODEC_P0_REG33_PRIMWCLK_EQ_INTERDACFS_VAL | CODEC_P0_REG33_PRIMDOUT_EQ_INTERDOUT_VAL, \
		/* mask */
		CODEC_P0_REG33_PRIMBCLK_EQ_CTRL_MASK | CODEC_P0_REG33_PRIMWCLK_EQ_CTRL_MASK | CODEC_P0_REG33_PRIMDOUT_EQ_CTRL_MASK
	},
	/* page0 reg53:	 DOUT (OUT Pin) Control */
	{
		0, 0, 53, \
		/* bits */
		CODEC_P0_REG53_DOUT_POUT_VAL | CODEC_P0_REG53_DOUT_KEEP_DIS_VAL, \
		/* mask */
		CODEC_P0_REG53_DOUT_CTRL_MASK | CODEC_P0_REG53_DOUT_KEEP_MASK
	}
};

/*******************************************************************************
 *                        Global variable definitions                          *
 ******************************************************************************/

/*******************************************************************************
 *                      Local function implementations                         *
 ******************************************************************************/


/*******************************************************************************
 *                      Global function implementations                        *
 ******************************************************************************/

/**
* @brief	init mutex and get mutex.
*
* @param	void
* @return	None
*/
void codec_Claim_Mutex(void)
{
	if(NULL == s_codecMutexID) {
		s_codecMutexID = zOss_CreateMutex("codecMutex",  ZOSS_INHERIT);  /* create mutex */
	}
	zOss_GetMutex(s_codecMutexID, ZOSS_WAIT_FOREVER);
}

/**
* @brief	release mutex.
*
* @param	void
* @return	None
*/
void codec_Release_Mutex(void)
{
	zOss_PutMutex(s_codecMutexID);
}

/**
* @brief	write data to codec
*
* @param	pi2cDev	Pointer to I2c_Device
* @param	regAddr	register address
* @param	writeBuf	register value
* @param	writeNum	write the register numbers
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 codec_I2cDevWrite(T_ZDrvI2c_Device *pi2cDev, UINT16 regAddr, UINT8 *writeBuf, UINT32 writeNum)
{
	UINT8 i = 0;
	SINT32 ret = DRV_SUCCESS;
	do {
		ret = zDrvI2c_DevWrite(pi2cDev, regAddr, writeBuf, writeNum);
		if(ret != DRV_SUCCESS) {
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2c_DevWrite error i=%d,ret=%d !\n", i, ret);
			zDrvPow_ActiveSleep(2); /* 2ms */
		}
		i++;
	} while((i < 3) &&(ret != DRV_SUCCESS));

	return ret;
}

/**
* @brief	read data from codec
*
* @param	pi2cDev    Pointer to I2c_Device
* @param	regAddr    register address
* @param	readBuf    register value
* @param	readNum    read the register numbers
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 codec_I2cDevRead(T_ZDrvI2c_Device *pi2cDev, UINT16 regAddr, UINT8 *readBuf, UINT32 readNum)
{
	UINT8 i = 0;
	SINT32 ret = DRV_SUCCESS;
	do {
		ret = zDrvI2c_DevRead(pi2cDev, regAddr, readBuf, readNum);
		if (ret != DRV_SUCCESS) {
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2c_DevRead error i=%d,ret=%d !\n", i, ret);
			zDrvPow_ActiveSleep(2); /* 2ms */
		}
		i++;
	} while((i < 3) && (ret != DRV_SUCCESS));

	return ret;
}

/**
* @brief	write data to codec
*
* @param	regPage     the page number of the register
* @param	regAddress  register address
* @param	regValue    register value
* @param	regMask     Mask the register bits
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 codec_I2CWrite(
    UINT8 regPage,
    UINT8 regAddress,
    UINT8 regValue,
    UINT8 regMask
)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 regTempValue = 0;
	//SINT32 regTemp = 0;/xiu
	/* switch to reg page */
	if (regPage == 6 || regPage == 7 || regPage == 10 || regPage == 11 || regPage > 13) {
		return halRet;
	}

	regTempValue = regPage;
	halRet = codec_I2cDevWrite(&s_codecI2CDev, 0, &regTempValue, 1);
	if (halRet != DRV_SUCCESS) {
		return halRet;
	}
	/* write value to reg */
	regTempValue = 0;
	halRet = codec_I2cDevRead(&s_codecI2CDev, (UINT16)regAddress, &regTempValue, 1);
	if (halRet == DRV_SUCCESS) {
		regTempValue = regTempValue & (~regMask);
		regTempValue |= (regValue & regMask);
		halRet = codec_I2cDevWrite(&s_codecI2CDev, (UINT16)regAddress, &regTempValue, 1);
		if (halRet == DRV_SUCCESS) {
			return DRV_SUCCESS;
		} else {
			return DRV_ERR_DATA_TRANSFER;
		}
	}
	
	return halRet;
}

/**
* @brief	read data to codec
*
* @param	regPage     the page number of the register
* @param	regAddress  register address
* @param	regValue    register value
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 codec_I2CRead(
    UINT8 regPage,
    UINT8 regAddress,
    UINT8 *regValue
)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 regTempValue = 0;

	/* switch to reg page */
	if (regPage == 6 || regPage == 7 || regPage == 10 || regPage == 11 || regPage > 13) {
		return halRet;
	}

	regTempValue = regPage;
	halRet = codec_I2cDevWrite(&s_codecI2CDev, 0, &regTempValue, 1);
	if (halRet != DRV_SUCCESS) {
		return halRet;
	}
	/* read val from reg */
	halRet = codec_I2cDevRead(&s_codecI2CDev, (UINT16)regAddress, regValue, 1);

	return halRet;
}

/**
* @brief	Get Codec clock division parameters
*
* @param	fs     T_ZDrvCodec_AudioSample
* @param	pPllClkDiv  T_PLL_CLK_DIV
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 codec_GetClkDivPara(T_ZDrvCodec_AudioSample fs, T_PLL_CLK_DIV *pPllClkDiv)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 i, cur_row = 0;
	UINT32 arraySize = 0;
	UINT32 tempfs;

	if (fs == AUDIO_SAMPLE_8K) {
		tempfs = 8000;
	} else if (fs == AUDIO_SAMPLE_16K) {
		tempfs = 16000;
	} else if (fs == AUDIO_SAMPLE_32K) {
		tempfs = 32000;
	} else if (fs == AUDIO_SAMPLE_44_1K) {
		tempfs = 44100;
	} else if (fs == AUDIO_SAMPLE_48K) {
		tempfs = 48000;
	} else {
		return	DRV_ERR_INVALID_PARAM;
	}
	arraySize = sizeof(pll_clkDiv) / sizeof(pll_clkDiv[0]);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec codec_SetBclk arraySize=%d !\n", arraySize);

/* PLLCLK  ref clk */

	for (i = 0; i < arraySize; i++) {
		if (pll_clkDiv[i].fs == tempfs) {
			cur_row = i;
			break;
		}
		if (i >= arraySize) {
			return	DRV_ERROR;
		}
	}

	*pPllClkDiv = pll_clkDiv[cur_row];
	return halRet;
}

/**
* @brief	read data from codec
*
* @param	regPage    the page of the register
* @param	regAddress    register address
* @param	regValue    register value
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Read(
    UINT16 regPage,
    UINT16 regAddress,
    UINT16 *regValue
)
{
	SINT32 halRet = DRV_SUCCESS;

	halRet = codec_I2CRead((UINT8)regPage, (UINT8)regAddress, (UINT8 *)regValue);

	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodec_Read p%dreg%d=0x%x ,halRet=%d!\n", regPage, regAddress, *((UINT8 *)regValue), halRet);
	return halRet;
}

/**
* @brief	write data to codec
*
* @param	regPage    the page of the register
* @param	regAddress    register address
* @param	regValue    register value
* @param	regMask    mask the register bits
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Write(
    UINT16 regPage,
    UINT16 regAddress,
    UINT16 regValue,
    UINT16 regMask
)
{
	SINT32 halRet = DRV_SUCCESS;
	halRet = codec_I2CWrite((UINT8)regPage, (UINT8)regAddress, (UINT8)regValue, (UINT8)regMask);
	return halRet;
}

/**
* @brief	write data to codec
*
* @param	VOID
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_DumpReg(VOID)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8  reg_write = 0;

	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg setInPath!\n");
	halRet = codec_I2CRead(1, 46, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg46=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 47, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg47=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 48, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg48=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 49, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg49=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 50, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg50=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 81, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg81=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 82, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg82=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 83, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg83=0x%x ,halRet=%d!\n", reg_write, halRet);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg setOutPath!\n");
	halRet += codec_I2CRead(0, 63, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg63=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 64, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg64=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 65, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg65=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 66, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg66=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 31, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg31=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 32, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg32=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 33, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg33=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 34, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg34=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 35, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg35=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 36, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg36=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 37, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg37=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 38, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg38=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 40, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg40=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 41, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg41=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 42, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg42=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(1, 44, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p1reg44=0x%x ,halRet=%d!\n", reg_write, halRet);

	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg AudioEnable!\n");
	halRet += codec_I2CRead(0, 64, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg64=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 82, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg82=0x%x ,halRet=%d!\n", reg_write, halRet);

	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg setClock!\n");
	halRet += codec_I2CRead(0, 4, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg4=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 5, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg5=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 6, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg6=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 7, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg7=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 8, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg8=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 11, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg11=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 12, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg12=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 13, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg13=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 14, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg14=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 18, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg18=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 19, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg19=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 20, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg20=0x%x ,halRet=%d!\n", reg_write, halRet);

	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg read flag!\n");
	halRet += codec_I2CRead(0, 36, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg36=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 37, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg37=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 38, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg38=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 39, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg39=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 44, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg44=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 45, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg45=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 46, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg46=0x%x ,halRet=%d!\n", reg_write, halRet);
	halRet += codec_I2CRead(0, 47, &reg_write);
	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec zDrvCodecDumpReg p0reg47=0x%x ,halRet=%d!\n", reg_write, halRet);

	return halRet;
}

/**
* @brief	Set codec Bits clock
*
* @param	useInfo    T_Codec_UseInfo
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 codec_SetBclk(T_Codec_UseInfo useInfo)
{
	SINT32 halRet = DRV_SUCCESS;
	//T_Codec_Reg_Para * pRegPara = NULL;
	UINT8 reg_write = 0, reg_read = 0, bdiv_clkin = 0;//, i = 0, cur_row = 0;//xiu
	UINT8 reg_mask = 0;
	//UINT8  reg_book = 0, reg_page = 0, reg_index = 0;
	//UINT8 arraySize = 0;//xiu
	UINT8 bclkDiv = 0;
	T_PLL_CLK_DIV pllClkDiv = {0};

/**
*according page0 reg29:Codec Interface Control 2 D1-D0(BDIV_CLKIN) to select
*Table 7-74. Page 0 / Register 30: BCLK N_VAL D6-D0(BCLK divider N)
*/
	halRet = codec_I2CRead(0, 29, &reg_read);
	if (halRet != DRV_SUCCESS) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec codec_SetBclk readreg err halRet=%d!\n", halRet);
		return DRV_ERROR;
	}
	bdiv_clkin = (reg_read & CODEC_P0_REG29_BDIVCLKIN_EQUAL_CTRL_MASK);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec codec_SetBclk bdiv_clkin=0x%x!\n", bdiv_clkin);

	if (useInfo.masterSlave == I2S_MASTER_MODE) {
		halRet = codec_GetClkDivPara(useInfo.sampleRate, &pllClkDiv);
		if (halRet != DRV_SUCCESS) {
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec codec_SetBclk codec_GetClkDivPara err halRet=%d!\n", halRet);
			return DRV_ERROR;
		}

		if (bdiv_clkin == 0) {
			/*DAC_CLK DAC DSP clock,*/
			bclkDiv = (pllClkDiv.dsor_div * pllClkDiv.mda_div) / 32; /* divide 32 ,make bclk equal 256khz */
		} else if (bdiv_clkin == 1) {
			/*DAC_MOD_CLK*/
			bclkDiv = pllClkDiv.dsor_div / 32; /* divide 32 ,make bclk equal 256khz */
		} else if (bdiv_clkin == 2) {
			/*according actual  to calculate*/
			/*ADC_CLK ADC DSP clock*/
			bclkDiv = (pllClkDiv.asor_div * pllClkDiv.mad_div) / 32; /* divide 32 ,make bclk equal 256khz */
		} else if (bdiv_clkin == 3) {
			/*ADC_MOD_CLK*/
			bclkDiv = pllClkDiv.asor_div / 32; /* divide 32 ,make bclk equal 256khz */
		}

		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec codec_SetBclk calc bclkDiv =%d !\n", bclkDiv);

		if (bclkDiv > 128) {
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec codec_SetBclk bclkDiv =%d err !\n", bclkDiv);
			return DRV_ERROR;
		} else if (bclkDiv == 128) {
			bclkDiv = 0;
		}
		reg_write = bclkDiv | CODEC_P0_REG30_BCLK_NDIV_PU_VAL;
	} else if (useInfo.masterSlave == I2S_SLAVE_MODE) {
		reg_write |= CODEC_P0_REG30_BCLK_NDIV_PD_VAL;
	}

	reg_mask = CODEC_P0_REG30_BCLK_NDIV_VAL_MASK | CODEC_P0_REG30_BCLK_NDIV_POWER_CTRL_MASK;

	halRet = codec_I2CWrite(0, 30, reg_write, reg_mask);
	if (halRet != DRV_SUCCESS) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec codec_SetBclk set p0r30 err halRet=%d !\n", halRet);
	}

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec codec_SetBclk end!\n");
	return halRet;
}

/**
* @brief	open codec, set parameters.
*
* @param	handle    T_ZDrvCodec_Handle
* @param	srcType	  T_ZDrvCodec_SourceType
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Open(T_ZDrvCodec_Handle *handle, T_ZDrvCodec_SourceType srcType)
{
	SINT32 halRet = DRV_SUCCESS;
	//UINT8 reg_write = 0;
	//UINT8 reg_mask = 0;
	//T_Codec_UseInfo *codec_info = NULL;//xiu

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec open start srcType=%d !\n", srcType);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec open, voice sta=%x,audio sta=%x,fm sta=%x !\n", voiceUseInfo.useState, audioUseInfo.useState, fmUseInfo.useState);

	if (handle == NULL) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Open handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();

	if (srcType == VOICE_SOURCE) {
		halRet = aud_SetCodecPWOnOff(TRUE);
		if (halRet == DRV_SUCCESS) {
			voiceUseInfo.handle = (T_ZDrvCodec_Handle)(&voiceUseInfo);
			voiceUseInfo.useState = CODEC_OPEN;
			voiceUseInfo.handleState = HANDLE_IN_USE;
			*handle = voiceUseInfo.handle;
			voiceUseInfo.inDev = AUDIO_INPUT_MAX;
			voiceUseInfo.outDev = AUDIO_OUTPUT_MAX;
			voiceUseInfo.srcType = VOICE_SOURCE;
		}
	} else if (srcType == AUDIO_SOURCE) {
		halRet = aud_SetCodecPWOnOff(TRUE);
		if (halRet == DRV_SUCCESS) {
			audioUseInfo.handle = (T_ZDrvCodec_Handle)(&audioUseInfo);
			audioUseInfo.useState = CODEC_OPEN;
			audioUseInfo.handleState = HANDLE_IN_USE;
			*handle = audioUseInfo.handle;
			audioUseInfo.inDev = AUDIO_INPUT_MAX;
			audioUseInfo.outDev = AUDIO_OUTPUT_MAX;
			audioUseInfo.srcType = AUDIO_SOURCE;
		}
	} else if (srcType == EXTERN_AUDIO) {
		halRet = aud_SetCodecPWOnOff(TRUE);
		if (halRet == DRV_SUCCESS) {
			fmUseInfo.handle = (T_ZDrvCodec_Handle)(&fmUseInfo);
			fmUseInfo.useState = CODEC_OPEN;
			fmUseInfo.handleState = HANDLE_IN_USE;
			*handle = fmUseInfo.handle;
			fmUseInfo.inDev = AUDIO_INPUT_MAX;
			fmUseInfo.outDev = AUDIO_OUTPUT_MAX;
			fmUseInfo.srcType = EXTERN_AUDIO;
		}
	} else {
		halRet = DRV_ERROR;
		*handle = (T_ZDrvCodec_Handle )NULL;//xiu
	}

	codec_Release_Mutex();
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec open end handle=%x, srcType=%x !\n", *handle, srcType);
	return  halRet;
}

/**
* @brief	close codec, power off.
*
* @param	handle    T_ZDrvCodec_Handle
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Close(T_ZDrvCodec_Handle *handle)
{
	SINT32 halRet = DRV_SUCCESS;
	//UINT8 reg_write = 0;
	//UINT8 reg_mask = 0;
	
	T_Codec_UseInfo *codec_info = NULL;

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec close start *handle=%x !\n", *handle);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec close start voice.handle=%x,audio.handle=%x,fm.handle=%x !\n", voiceUseInfo.handle, audioUseInfo.handle, fmUseInfo.handle);

	if ((*handle == 0) || (handle == NULL)) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Close handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();

	codec_info = (T_Codec_UseInfo *)(*handle);

	if (codec_info->handleState == HANDLE_IN_USE) {
		if (codec_info->handle == voiceUseInfo.handle && fmUseInfo.useState == CODEC_IDLE && audioUseInfo.useState == CODEC_IDLE) {
			halRet = aud_SetCodecPWOnOff (FALSE);
		} else if (codec_info->handle == audioUseInfo.handle && fmUseInfo.useState == CODEC_IDLE && voiceUseInfo.useState == CODEC_IDLE) {
			halRet = aud_SetCodecPWOnOff (FALSE);
		} else if (codec_info->handle == fmUseInfo.handle && audioUseInfo.useState == CODEC_IDLE && voiceUseInfo.useState == CODEC_IDLE) {
			halRet = aud_SetCodecPWOnOff (FALSE);
		}
	} else {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec close err codec_info->handleState =%x !\n", codec_info->handleState);
		halRet = DRV_ERROR;
	}

	codec_info->handle = (T_ZDrvCodec_Handle)NULL;//xiu
	codec_info->useState = CODEC_IDLE;
	codec_info->handleState = HANDLE_FREE;
	codec_info->inDev = AUDIO_INPUT_MAX;
	codec_info->outDev = AUDIO_OUTPUT_MAX;
#ifdef _CONFIG_USE_CODEC_EARPIECE_DETECT
//        codec_EarpieceDetectEnable(isEarpDetect);
#endif

	codec_Release_Mutex();

	zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL, "codec close end  halRet=%x !\n", halRet);
	return  halRet;
}

/**
* @brief	power off codec, set parameters.
*
* @param	onOff    BOOL 1 - power on 0 - power off
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 aud_SetCodecPWOnOff(BOOL onOff)
{
	SINT32 halRet = DRV_SUCCESS;
	//UINT8 reg_write = 0;
	//UINT8 reg_mask = 0;//xiu
/*must be configure when coding in 7520, lvwenhua 20150317*/
	return  halRet;
}

/**
* @brief	Set i2s protocol.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	channel   T_ZDrvCodec_AudioI2SChannel
* @param	masterSlave	T_ZDrvCodec_AudioI2SMode
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetProtocal(T_ZDrvCodec_Handle handle, T_ZDrvCodec_AudioI2SChannel channel, T_ZDrvCodec_AudioI2SMode masterSlave)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	//UINT8 reg_mask = 0;
	T_Codec_UseInfo *codec_info = NULL;
	UINT8 regVal[10] = {0};

#ifdef CODEC_TEST
	UINT8  reg_readval[10] = {0};
#endif
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec Set protocal !\n");
	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec Set protocal handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;
	if (codec_info->handleState == HANDLE_IN_USE) {
		/*set codec interface control */
		if (masterSlave == I2S_SLAVE_MODE) {
			if (codec_info->busMode == CODEC_BUS_I2S) {
				*regVal = CODEC_P0_REG27_BUS_I2S_VAL | CODEC_P0_REG27_BCLK_INPUT_VAL | CODEC_P0_REG27_WCLK_INPUT_VAL; /*0x00 */
				//*(regVal+2)= CODEC_P0_REG29_BCLK_INVERTED_EN_VAL; /*p0/reg29 reset val */
				*(regVal + 2) = CODEC_P0_REG29_BCLK_INVERTED_DIS_VAL; /*p0/reg29 reset val  xxq 2016 */
			} else if (codec_info->busMode == CODEC_BUS_DSP) {
				*regVal = CODEC_P0_REG27_BUS_DSP_VAL | CODEC_P0_REG27_BCLK_INPUT_VAL | CODEC_P0_REG27_WCLK_INPUT_VAL; /*0x00 */
				*(regVal + 2) = 0; /*p0/reg29 reset val */
			} else {
				codec_Release_Mutex();
				return  DRV_ERROR;
			}

			*(regVal + 1) = CODEC_P0_REG28_DATASLOT_OFFSET_VAL; /*0x00 */
			*(regVal + 3) = 0x01 | CODEC_P0_REG30_BCLK_NDIV_PU_VAL; /*p0/reg30 reset val */
			*(regVal + 4) = 0x00; /* p0/reg31 reset val */
			*(regVal + 5) = 0x00; /* p0/reg32 reset val */
			*(regVal + 6) = 0x00; /* p0/reg33 reset val */
			*(regVal + 7) = CODEC_P0_REG53_DOUT_POUT_VAL | CODEC_P0_REG53_DOUT_KEEP_DIS_VAL; /*0x12 */
			*(regVal + 8) = CODEC_P0_REG54_DIN_EN_VAL ; /* p0/reg54  val	0x02 */

			reg_write = 0;
			halRet = codec_I2cDevWrite(&s_codecI2CDev, 0, &reg_write, 1); /*set page to 0 */
			halRet += codec_I2cDevWrite(&s_codecI2CDev, 27, regVal, 7);
#ifdef CODEC_TEST
			halRet += codec_I2cDevRead(&s_codecI2CDev, 27, reg_readval, 7);
#endif
			halRet += codec_I2cDevWrite(&s_codecI2CDev, 53, regVal + 7, 2);
#ifdef CODEC_TEST
			halRet += codec_I2cDevRead(&s_codecI2CDev, 53, reg_readval + 7, 2);
#endif
			if (halRet == DRV_SUCCESS) {
				codec_info->masterSlave = I2S_SLAVE_MODE;
			}
		} else if (masterSlave == I2S_MASTER_MODE) {
			if (codec_info->busMode == CODEC_BUS_I2S) {
				*regVal = CODEC_P0_REG27_BUS_I2S_VAL | CODEC_P0_REG27_BCLK_OUTPUT_VAL | CODEC_P0_REG27_WCLK_OUTPUT_VAL; /* 0x0c */
				*(regVal + 3) = CODEC_P0_REG30_BCLK_NDIV_VAL | CODEC_P0_REG30_BCLK_NDIV_PU_VAL;	/* 0x81 */
			} else if (codec_info->busMode == CODEC_BUS_DSP) {
				*regVal = CODEC_P0_REG27_BUS_DSP_VAL | CODEC_P0_REG27_BCLK_OUTPUT_VAL | CODEC_P0_REG27_WCLK_OUTPUT_VAL; /* 0x0c */
				*(regVal + 3) = CODEC_P0_REG30_BCLK_NDIV_VAL | CODEC_P0_REG30_BCLK_NDIV_PU_VAL;	/* 0x81 */
			} else {
				codec_Release_Mutex();
				return  DRV_ERROR;
			}

			*(regVal + 1) = CODEC_P0_REG28_DATASLOT_OFFSET_VAL; /* 0x00 */
			*(regVal + 2) = CODEC_P0_REG29_BWCLK_UNACTIVE_PD_VAL | CODEC_P0_REG29_BDIVCLKIN_EQUAL_DACMODCLK_VAL | \
			                 CODEC_P0_REG29_BCLK_INVERTED_DIS_VAL;/* 0x04 */
			*(regVal + 4) = 0x00; /* p0/reg31 reset val */
			*(regVal + 5) = CODEC_P0_REG32_PRIMBCLK_FDTO_SERINTER_VAL | CODEC_P0_REG32_PRIMWBCLK_FDTO_SERINTER_VAL | \
			                 CODEC_P0_REG32_PRIMDIN_FDTO_SERINTER_VAL;/* 0x00 */
			*(regVal + 6) = CODEC_P0_REG33_PRIMBCLK_EQ_INTERBCLK_VAL | CODEC_P0_REG33_PRIMWCLK_EQ_INTERDACFS_VAL | \
			                 CODEC_P0_REG33_PRIMDOUT_EQ_INTERDOUT_VAL;	/* 0x00 */
			*(regVal + 7) = CODEC_P0_REG53_DOUT_POUT_VAL | CODEC_P0_REG53_DOUT_KEEP_DIS_VAL; /* 0x12 */
			*(regVal + 8) = CODEC_P0_REG54_DIN_EN_VAL ; /*p0/reg54  val	0x02 */

			reg_write = 0;
			halRet = codec_I2cDevWrite(&s_codecI2CDev, 0, &reg_write, 1); /* set page to 0 */
			halRet += codec_I2cDevWrite(&s_codecI2CDev, 27, regVal, 7);
#ifdef CODEC_TEST
			halRet += codec_I2cDevRead(&s_codecI2CDev, 27, reg_readval, 7);
#endif
			halRet += codec_I2cDevWrite(&s_codecI2CDev, 53, regVal + 7, 2);
#ifdef CODEC_TEST
			halRet += codec_I2cDevRead(&s_codecI2CDev, 53, reg_readval + 7, 2);
#endif
			if (halRet == DRV_SUCCESS) {
				codec_info->masterSlave = I2S_MASTER_MODE;
			}
		} else {
			halRet = DRV_ERR_INVALID_PARAM;
		}
	} else {
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();
	return  halRet;
}

/**
* @brief	Set i2s protocol.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	useProtocol   T_ZDrvCodec_UseProtocol
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetProt(T_ZDrvCodec_Handle handle, T_ZDrvCodec_UseProtocol useProtocol)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;//xiu
	UINT8 reg_mask = 0;
	T_Codec_UseInfo *codec_info = NULL;
	T_Codec_Reg_Para *pRegPara = 0;
	UINT8  reg_page = 0, reg_index = 0;//xiu
	UINT8 arraySize = 0, i = 0;

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetProt !\n");
	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetProt handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;
	if (codec_info->handleState == HANDLE_IN_USE) {
		/*set codec interface control */
		if (useProtocol.masterSlave == I2S_SLAVE_MODE) {
			if (useProtocol.busMode == CODEC_BUS_I2S) {
				pRegPara = i2sSlaveModePara;
				arraySize = sizeof(i2sSlaveModePara) / sizeof(i2sSlaveModePara[0]);
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetProt i2sSlaveModePara arraySize=%d !\n", arraySize);
			} else if (useProtocol.busMode == CODEC_BUS_DSP) {
				pRegPara = pcmSlaveModePara;
				arraySize = sizeof(pcmSlaveModePara) / sizeof(pcmSlaveModePara[0]);
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetProt pcmSlaveModePara arraySize=%d !\n", arraySize);
			} else {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetProt I2S_SLAVE_MODE interface control=%d !\n", arraySize);
				codec_Release_Mutex();
				return  DRV_ERROR;
			}
		} else if (useProtocol.masterSlave == I2S_MASTER_MODE) {
			if (useProtocol.busMode == CODEC_BUS_I2S) {
				pRegPara = i2sMasterModePara;
				arraySize = sizeof(i2sMasterModePara) / sizeof(i2sMasterModePara[0]);
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetProt i2sMasterModePara arraySize=%d !\n", arraySize);
			} else if (useProtocol.busMode == CODEC_BUS_DSP) {
				pRegPara = pcmMasterModePara;
				arraySize = sizeof(pcmMasterModePara) / sizeof(pcmMasterModePara[0]);
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetProt pcmMasterModePara arraySize=%d !\n", arraySize);
			} else {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetProt pcmMasterModePara arraySize=%d !\n", arraySize);
				codec_Release_Mutex();
				return  DRV_ERROR;
			}
		} else {
			halRet = DRV_ERR_INVALID_PARAM;
		}
	} else {
		halRet = DRV_ERROR;
	}

	if (halRet == DRV_SUCCESS) {
		for (i = 0; i < arraySize; i++) {
			reg_page = (pRegPara + i)->reg_page;
			reg_index = (pRegPara + i)->reg_index;
			reg_write = (pRegPara + i)->reg_bits;
			reg_mask = (pRegPara + i)->reg_bitsmask;
			halRet = codec_I2CWrite(reg_page, reg_index, reg_write, reg_mask);
			if (halRet != DRV_SUCCESS) {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetProt new i=%d,halRet=%d !\n", i, halRet);
				codec_Release_Mutex();
				return	halRet;
			}
		}
		
		codec_info->audI2SChannel = useProtocol.audI2SChannel;
		codec_info->busMode = useProtocol.busMode;
		codec_info->masterSlave = useProtocol.masterSlave;

		halRet = codec_SetBclk(*codec_info);
		if (halRet != DRV_SUCCESS) {
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetProt codec_SetBclk err ,halRet=%d !\n", i, halRet);
			halRet = DRV_ERROR;
		}
	}
	
	codec_Release_Mutex();
	return  halRet;
}

/**
* @brief	Set i2s clock.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	sample   T_ZDrvCodec_AudioSample
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetClock(T_ZDrvCodec_Handle handle, T_ZDrvCodec_AudioSample sample)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
	UINT32 tempfs;
	UINT8 i, cur_row = 0;
	UINT8 regVal[12] = {0};
	UINT8 arraySize = 0;

	T_Codec_UseInfo *codec_info = NULL;

#ifdef CODEC_TEST
	UINT8  reg_readval[6] = {0};
#endif
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvCodec_SetClock codec Set clk=%x !\n", sample);
	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetClock handle is null !\n");
		return	DRV_ERROR;
	}
	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;
	if (codec_info->handleState == HANDLE_IN_USE) {
		if (sample == AUDIO_SAMPLE_8K) {
			tempfs = 8000;
		} else if (sample == AUDIO_SAMPLE_16K) {
			tempfs = 16000;
		} else if (sample == AUDIO_SAMPLE_32K) {
			tempfs = 32000;
		} else if (sample == AUDIO_SAMPLE_44_1K) {
			tempfs = 44100;
		} else if (sample == AUDIO_SAMPLE_48K) {
			tempfs = 48000;
		} else {
			codec_Release_Mutex();
			return  DRV_ERR_INVALID_PARAM;
		}

		arraySize = sizeof(pll_clkDiv) / sizeof(pll_clkDiv[0]);
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec Set clk arraySize=%d !\n", arraySize);

		/* PLLCLK  ref clk */
		for (i = 0; i < arraySize; i++) {
			if (pll_clkDiv[i].fs == tempfs) {
				cur_row = i;
				break;
			}
			if (i >= arraySize) {
				codec_Release_Mutex();
				return  DRV_ERROR;
			}
		}

		reg_write = CODEC_P0_REG4_CODEC_CLKIN_PLLCLK_VAL | CODEC_P0_REG4_PLL_CLKIN_MCLK_VAL;
		reg_mask = CODEC_P0_REG4_CODEC_CLKIN_EQCLK_MASK | CODEC_P0_REG4_PLL_CLKIN_EQCLK_MASK;
		halRet = codec_I2CWrite(0, 4, reg_write, reg_mask);

		/* pll clk set */
		if (pll_clkDiv[cur_row].pll_pdiv == 8) {
			*regVal = (0 & CODEC_P0_REG5_PLL_PDIV_MASK);
		} else {
			*regVal = ( (pll_clkDiv[cur_row].pll_pdiv << 4) & CODEC_P0_REG5_PLL_PDIV_MASK);
		}
		if (pll_clkDiv[cur_row].pll_rmulti == 16) {
			*regVal |= (0 & CODEC_P0_REG5_PLL_RMULTI_MASK);
		} else {
			*regVal |= (pll_clkDiv[cur_row].pll_rmulti & CODEC_P0_REG5_PLL_RMULTI_MASK);
		}
		*regVal |= CODEC_P0_REG5_PLL_PUP_VAL;

		*(regVal + 1) = pll_clkDiv[cur_row].pll_jmulti & CODEC_P0_REG6_PLL_JMULTI_MASK;
		*(regVal + 2) = ( (pll_clkDiv[cur_row].pll_dval & 0xff00) >> 8) & CODEC_P0_REG7_PLL_DVAL_MSB_MASK;
		*(regVal + 3) = (pll_clkDiv[cur_row].pll_dval & 0xff) & CODEC_P0_REG8_PLL_DVAL_LSB_MASK;

		/*set dac fs
		*start reg P0,reg11
		*/

		/* Page 0 / Register 11: DAC NDAC_VAL */
		if (pll_clkDiv[cur_row].nda_div == 128) {
			*(regVal + 4) = (0x00 & CODEC_P0_REG11_NDAC_VAL_MASK) | CODEC_P0_REG11_NDAC_PWON_VAL;
		} else {
			*(regVal + 4) = (pll_clkDiv[cur_row].nda_div & CODEC_P0_REG11_NDAC_VAL_MASK) | CODEC_P0_REG11_NDAC_PWON_VAL;
		}

		/* Page 0 / Register 12: DAC MDAC_VAL */
		if (pll_clkDiv[cur_row].mda_div == 128) {
			*(regVal + 5) = (0x00 & CODEC_P0_REG12_MDAC_VAL_MASK) | CODEC_P0_REG12_MDAC_PWON_VAL;
		} else {
			*(regVal + 5) = (pll_clkDiv[cur_row].mda_div & CODEC_P0_REG12_MDAC_VAL_MASK) | CODEC_P0_REG12_MDAC_PWON_VAL;
		}

		/* DAC DOSR_VAL */
		if (pll_clkDiv[cur_row].dsor_div == 1024) {
			*(regVal + 6) = 0; /*MSB */
			*(regVal + 7) = 0; /*LSB */
		} else {
			*(regVal + 6) = (pll_clkDiv[cur_row].dsor_div & 0x0300) >> 8; /*MSB */
			*(regVal + 7) = pll_clkDiv[cur_row].dsor_div & 0x00ff; /*LSB */
		}

		/* adc clk */
		if (pll_clkDiv[cur_row].nad_div == 128) {
			*(regVal + 8) = (0x00 & CODEC_P0_REG18_NADC_VAL_MASK) | CODEC_P0_REG18_NADC_PWON_VAL;
		} else {
			*(regVal + 8) = (pll_clkDiv[cur_row].nad_div & CODEC_P0_REG18_NADC_VAL_MASK) | CODEC_P0_REG18_NADC_PWON_VAL;
		}

		if (pll_clkDiv[cur_row].mad_div == 128) {
			*(regVal + 9) = (0x00 & CODEC_P0_REG19_MADC_VAL_MASK) | CODEC_P0_REG19_MADC_PWON_VAL;
		} else {
			*(regVal + 9) = (pll_clkDiv[cur_row].mad_div & CODEC_P0_REG19_MADC_VAL_MASK) | CODEC_P0_REG19_MADC_PWON_VAL;
		}

		/* ADC AOSR_VAL */
		if (pll_clkDiv[cur_row].asor_div == 256) {
			*(regVal + 10) = 0;
		} else {
			*(regVal + 10) = (pll_clkDiv[cur_row].asor_div & 0x00ff);
		}

	/* set pll */
		reg_write = 0;
		halRet = codec_I2cDevWrite (&s_codecI2CDev, 0, &reg_write, 1); /*set page */
		halRet += codec_I2cDevWrite(&s_codecI2CDev, 5, regVal, 1);
		zOss_Sleep(10);
		halRet += codec_I2cDevWrite(&s_codecI2CDev, 6, regVal + 1, 3);

#ifdef CODEC_TEST
		halRet += codec_I2cDevRead(&s_codecI2CDev, 5, reg_readval, 4);
#endif
/* set dac */
		halRet += codec_I2cDevWrite(&s_codecI2CDev, 11, regVal + 4, 1);
		halRet += codec_I2cDevWrite(&s_codecI2CDev, 12, regVal + 5, 1);
		halRet += codec_I2cDevWrite(&s_codecI2CDev, 13, regVal + 6, 2);

#ifdef CODEC_TEST
		halRet += codec_I2cDevRead(&s_codecI2CDev, 11, reg_readval, 4);
#endif
/* set dac */
		halRet += codec_I2cDevWrite(&s_codecI2CDev, 18, regVal + 8, 1);
		halRet += codec_I2cDevWrite(&s_codecI2CDev, 19, regVal + 9, 1);
		halRet += codec_I2cDevWrite(&s_codecI2CDev, 20, regVal + 10, 1);

#ifdef CODEC_TEST
		halRet += codec_I2cDevRead(&s_codecI2CDev, 18, reg_readval, 3);
#endif

		if (halRet == DRV_SUCCESS) {
			codec_info->sampleRate = sample;
		}
	} else {
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();
	return  halRet;
}

/**
* @brief	reset codec.
*
* @param	handle	 T_ZDrvCodec_Handle
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Reset(T_ZDrvCodec_Handle handle)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
	T_Codec_UseInfo *codec_info = NULL;
	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Reset handle is null !\n");
		return	DRV_ERROR;
	}
	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;
	if (codec_info->handleState == HANDLE_IN_USE) {
		/* soft reset */
		reg_write = CODEC_P0_REG1_RESET_VAL;
		reg_mask = CODEC_P0_REG1_RESET_MASK;
		halRet = codec_I2CWrite(0, 1, reg_write, reg_mask);
		if (halRet != DRV_SUCCESS) {
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Reset soft reset err halRet=%d !\n", halRet);
		}
	} else {
		halRet = DRV_ERROR;
	}
#ifdef _CONFIG_USE_CODEC_EARPIECE_DETECT
	if (halRet == DRV_SUCCESS) {
		codec_EarpieceDetectEnable(isEarpDetect);
	}
#endif
	codec_Release_Mutex();
	return  halRet;
}

/**
* @brief	Set the input path of the codec.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	dev   T_ZDrvCodec_InDevice
* @param	onoff   BOOL
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetInPath(T_ZDrvCodec_Handle handle,  T_ZDrvCodec_InDevice dev, BOOL onoff)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
	//UINT32 arraySize = 0;
	T_CodecPathParaInfo inpathinfo;

	UINT8  reg_page = 0, reg_index = 0, i = 0;//xiu
	//T_Codec_Reg_Para * pRegPara = NULL;
	T_Codec_UseInfo *codec_info = NULL;
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec SetInPath dev=%x,onoff=%x !\n", dev, onoff);

	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetInPath handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;
	if (codec_info->handleState == HANDLE_IN_USE) {

	halRet = zDrvCodec_GetInPathParaInfo(&inpathinfo,dev,onoff);
	if (halRet != DRV_SUCCESS) {
        codec_Release_Mutex();
        return DRV_ERROR;
    }
	
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec SetInPath arraySize=%d !\n", inpathinfo.arraySize);

		for (i = 0; i < inpathinfo.arraySize; i++) {
			reg_page = (inpathinfo.pRegPara + i)->reg_page;
			reg_index = (inpathinfo.pRegPara + i)->reg_index;
			reg_write = (inpathinfo.pRegPara + i)->reg_bits;
			reg_mask = (inpathinfo.pRegPara + i)->reg_bitsmask;
			halRet += codec_I2CWrite(reg_page, reg_index, reg_write , reg_mask);

			if (halRet != DRV_SUCCESS) {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec SetInputPath new i=%d,halRet=%d !\n", i, halRet);
				halRet = DRV_ERROR;
				break;
			}
		}

		if (halRet == DRV_SUCCESS) {
			if (onoff) {
				codec_info->inDev = dev;
			} else {
				codec_info->inDev = AUDIO_INPUT_MAX;
			}
			halRet = codec_I2CRead(1, 48, &reg_write);
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec SetInputPath new p1reg48 regval=0x%x ,halRet=%d!\n", reg_write, halRet);
			halRet += codec_I2CRead(1, 49, &reg_write);
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec SetInputPath new p1reg49 regval=0x%x ,halRet=%d!\n", reg_write, halRet);
			halRet += codec_I2CRead(1, 50, &reg_write);
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec SetInputPath new p1reg50 regval=0x%x ,halRet=%d!\n", reg_write, halRet);
		}
	} else {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec SetInPath write/read reg err halRet=%d !\n", halRet);
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();

	return  halRet;
}

/**
* @brief	Set the output path of the codec.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	dev   T_ZDrvCodec_InDevice
* @param	onoff   BOOL
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetOutPath(T_ZDrvCodec_Handle handle,  T_ZDrvCodec_OutDevice dev, BOOL onoff)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
	//UINT32 arraySize = 0;
	T_CodecPathParaInfo outpathinfo;

	UINT8  reg_page = 0, reg_index = 0, i = 0;//reg_book = 0,
	//T_Codec_Reg_Para *pRegPara = NULL;
	T_Codec_UseInfo *codec_info = NULL;

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec SetOutPath dev=%x,onoff=%x !\n", dev, onoff);
	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetOutPath handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;
	if (codec_info->handleState == HANDLE_IN_USE) {

	halRet = zDrvCodec_GetOutPathParaInfo(&outpathinfo,dev,codec_info->srcType,onoff);
	if (halRet != DRV_SUCCESS) {
        codec_Release_Mutex();
        return DRV_ERROR;
    }
		
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec SetOutPath arraySize=%d !\n", outpathinfo.arraySize);

	for (i = 0; i < outpathinfo.arraySize; i++) {
		reg_page = (outpathinfo.pRegPara + i)->reg_page;
		reg_index = (outpathinfo.pRegPara + i)->reg_index;
		reg_write = (outpathinfo.pRegPara + i)->reg_bits;
		reg_mask = (outpathinfo.pRegPara + i)->reg_bitsmask;
		halRet += codec_I2CWrite(reg_page, reg_index, reg_write, reg_mask);
		if (halRet != DRV_SUCCESS) {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec SetOutputPath new i=%d,halRet=%d !\n", i, halRet);
				halRet = DRV_ERROR;
				break;
			}
		}

		if (halRet == DRV_SUCCESS) {
			if (onoff) {
				codec_info->outDev = dev;
			} else {
				codec_info->outDev = AUDIO_OUTPUT_MAX;
			}
			if (dev == AUDIO_OUTPUT_HEADSET) {
				halRet = codec_I2CRead(0, 65, &reg_write);
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec SetOutputPath new p0reg65 regval=0x%x ,halRet=%d!\n", reg_write, halRet);
				halRet = codec_I2CRead(0, 66, &reg_write);
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec SetOutputPath new p0reg66 regval=0x%x ,halRet=%d!\n", reg_write, halRet);
			}
		}
	} else {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec SetOutputPath write codec reg ,halRet=%d !\n", halRet);
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();

	return  halRet;
}

/**
* @brief	Set the output Hpf of the codec.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	fs   T_ZDrvCodec_InDevice
* @param	onoff   BOOL
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetOut_Hpf(T_ZDrvCodec_Handle handle, T_ZDrvCodec_AudioSample fs, BOOL onoff)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	T_Codec_UseInfo *codec_info = NULL;

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvCodec_SetOut_Hpf fs=%d,onoff=%x !\n", fs, onoff);
	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec DrvCodec_SetOut_Hpf handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;
	if (codec_info->handleState == HANDLE_IN_USE) {
		if (onoff == TRUE) {
			reg_write = 0;
			halRet = codec_I2cDevWrite(&s_codecI2CDev, 0, &reg_write, 1); /*set page to 0 */
			reg_write = 9; /*select PRB7 */
			halRet += codec_I2cDevWrite(&s_codecI2CDev, 60, &reg_write, 1);
			if (halRet != DRV_SUCCESS) {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec DrvCodec_SetOut_Hpf set PRBX err halRet=%d!\n", halRet);
				codec_Release_Mutex();
				return	DRV_ERROR;
			}

			if ((fs == AUDIO_SAMPLE_8K) || (fs == AUDIO_SAMPLE_16K) || (fs == AUDIO_SAMPLE_32K)) {

				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec DrvCodec_SetOut_Hpf fs=%d !\n", fs);
				reg_write = 8;
				halRet = codec_I2cDevWrite(&s_codecI2CDev, 0, &reg_write, 1);
				halRet += codec_I2cDevWrite(&s_codecI2CDev, 2, speakerOutIirHpfPara[0], 10);
				halRet += codec_I2cDevWrite(&s_codecI2CDev, 66, speakerOutIirHpfPara[0] + 10, 10);

				if (halRet != DRV_SUCCESS) {
					zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec DrvCodec_SetOut_Hpf set hpf err fs=%d,halRet=%d !\n", fs, halRet);
					codec_Release_Mutex();
					return	DRV_ERROR;
				}
			} else if ((fs == AUDIO_SAMPLE_44_1K) || (fs == AUDIO_SAMPLE_48K)) {

				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec DrvCodec_SetOut_Hpf fs=%d !\n", fs);
				reg_write = 8;

				halRet = codec_I2cDevWrite(&s_codecI2CDev, 0, &reg_write, 1);
				halRet += codec_I2cDevWrite(&s_codecI2CDev, 2, speakerOutIirHpfPara[1], 10);
				halRet += codec_I2cDevWrite(&s_codecI2CDev, 66, speakerOutIirHpfPara[1] + 10, 10);

				if (halRet != DRV_SUCCESS) {
					zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec DrvCodec_SetOut_Hpf set hpf err fs=%d,halRet=%d !\n", fs, halRet);
					codec_Release_Mutex();
					return	DRV_ERROR;
				}
			} else {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec DrvCodec_SetOut_Hpf para sample not support,fs=%d!\n", fs);
				codec_Release_Mutex();
				return  DRV_ERR_INVALID_PARAM;
			}
		} else {
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec DrvCodec_SetOut_Hpf  set off  onoff=%d,!\n", onoff);
			reg_write = 0;
			halRet = codec_I2cDevWrite(&s_codecI2CDev, 0, &reg_write, 1); /*set page to 0 */
			reg_write = 1;
			halRet += codec_I2cDevWrite(&s_codecI2CDev, 60, &reg_write, 1);
			if (halRet != DRV_SUCCESS) {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec DrvCodec_SetOut_Hpf set off err halRet=%d !\n", halRet);
				codec_Release_Mutex();
				return	DRV_ERROR;
			}
		}
	} else {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec DrvCodec_SetOut_Hpf write codec reg ,halRet=%d !\n", halRet);
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();
	return  halRet;
}

/**
* @brief	Enable the audio path.
*
* @param	handle	 T_ZDrvCodec_Handle
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_AudioEnable(T_ZDrvCodec_Handle handle)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;

#ifdef CODEC_TEST
	UINT8  reg_readval[3] = {0};
#endif

	T_Codec_UseInfo *codec_info = NULL;
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_AudioEnable handle=%x !\n", handle);
	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_AudioEnable handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;
	if (codec_info->handleState == HANDLE_IN_USE) {
		if (codec_info->useState == CODEC_ENABLE) {
			codec_Release_Mutex();
			return  halRet;
		}

		if (codec_info->inDev != AUDIO_INPUT_MAX) {
			/*input path */
			reg_write = CODEC_P0_REG82_ADC_UNMUTE_VAL;
			reg_mask = CODEC_P0_REG82_ADC_MUTE_CTRL_MASK;
			halRet += codec_I2CWrite(0, 82, reg_write, reg_mask);
			if (halRet != DRV_SUCCESS) {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_AudioEnable set codec reg err1 ,halRet=0x%x !\n", halRet);
				codec_Release_Mutex();
				return  DRV_ERROR;
			}
		}

		if(codec_info->outDev != AUDIO_OUTPUT_MAX) {

			/*output path */
			reg_write = CODEC_P0_REG64_DAC_LCHN_UNMUTE_VAL | CODEC_P0_REG64_DAC_RCHN_UNMUTE_VAL;
			reg_mask = CODEC_P0_REG64_DAC_LCHN_MUTE_CTRL_MASK | CODEC_P0_REG64_DAC_RCHN_MUTE_CTRL_MASK;
			halRet = codec_I2CWrite(0, 64, reg_write, reg_mask);
#ifdef CODEC_TEST
			halRet += codec_I2cDevRead(&s_codecI2CDev, 64, reg_readval, 1);
#endif
			if (halRet != DRV_SUCCESS) {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_AudioEnable set codec reg err2 ,halRet=0x%x !\n", halRet);
				codec_Release_Mutex();
				return  DRV_ERROR;
			}
		}
		codec_info->useState = CODEC_ENABLE;
	} else {
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();
	return  halRet;

}

/**
* @brief	Disable the audio path.
*
* @param	handle	 T_ZDrvCodec_Handle
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_AudioDisable(T_ZDrvCodec_Handle handle)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
#ifdef CODEC_TEST
	UINT8  reg_readval[3] = {0};
#endif
	T_Codec_UseInfo *codec_info = NULL;
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Audiodisable handle=%x !\n", handle);

	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_AudioDisable handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;
	
	if (codec_info->handleState == HANDLE_IN_USE) {
		if (codec_info->useState == CODEC_DISABLE) {
			codec_Release_Mutex();
			return  halRet;
		}

		if (codec_info->inDev != (UINT32)NULL) {//xiu
			/*input path */
			reg_write = CODEC_P0_REG82_ADC_MUTE_VAL;
			reg_mask = CODEC_P0_REG82_ADC_MUTE_CTRL_MASK;
			halRet = codec_I2CWrite(0, 82, reg_write, reg_mask);
			reg_write = CODEC_P1_REG46_MICBIAS_OUT_PWOFF_VAL;
			reg_mask = CODEC_P1_REG46_MICBIAS_OUT_CTRL_MASK;
			halRet += codec_I2CWrite(1, 46, reg_write, reg_mask);

			if (halRet != DRV_SUCCESS) {
				codec_Release_Mutex();
				return  DRV_ERROR;
			}
		}

		if (codec_info->outDev != (UINT32)NULL) {//xiu
			/*output path */
			reg_write = CODEC_P0_REG64_DAC_LCHN_MUTE_VAL | CODEC_P0_REG64_DAC_RCHN_MUTE_VAL;
			reg_mask = CODEC_P0_REG64_DAC_LCHN_MUTE_CTRL_MASK | CODEC_P0_REG64_DAC_RCHN_MUTE_CTRL_MASK;
			halRet = codec_I2CWrite(0, 64, reg_write , reg_mask);
#ifdef CODEC_TEST
			halRet += codec_I2cDevRead(&s_codecI2CDev, 64, reg_readval, 1);
#endif
			if (halRet != DRV_SUCCESS) {
				codec_Release_Mutex();
				return  DRV_ERROR;
			}
		}
		codec_info->useState = CODEC_DISABLE;
	} else {
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();


	return  halRet;
}

/**
* @brief	Set the ClsD volume.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	vol	 T_ZDrvCodec_ClsD_Vol
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Set_ClsD_Vol(T_ZDrvCodec_Handle handle,  T_ZDrvCodec_ClsD_Vol vol)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
	/*Page 1 / Register 42: SPK Driver */
	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Set_ClsD_Vol handle is null !\n");
		return	DRV_ERROR;
	}
	codec_Claim_Mutex();

	if (vol == CLSD_GAIN_PLUS_6DB) {
		reg_write = CODEC_P1_REG42_SPK_DRV_SGAIN_6DB_VAL;
	} else if (vol == CLSD_GAIN_PLUS_12DB) {
		reg_write = CODEC_P1_REG42_SPK_DRV_SGAIN_12DB_VAL;
	} else if (vol == CLSD_GAIN_PLUS_18DB) {
		reg_write = CODEC_P1_REG42_SPK_DRV_SGAIN_18DB_VAL;
	} else if (vol == CLSD_GAIN_PLUS_24DB) {
		reg_write = CODEC_P1_REG42_SPK_DRV_SGAIN_24DB_VAL;
	} else {
		codec_Release_Mutex();
		return  DRV_ERROR;
	}
	reg_mask = CODEC_P1_REG42_SPK_DRV_SGAIN_MASK;
	halRet = codec_I2CWrite(1, 42, reg_write, reg_mask);
	codec_Release_Mutex();

	return  halRet;
}

/**
* @brief	Set the ClsAB volume.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	vol  UINT8
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Set_ClsAB_Vol(T_ZDrvCodec_Handle handle,  UINT8 vol)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;

	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Set_ClsAB_Vol handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	if (vol <= 9) {
		reg_write = vol;
	} else {
		codec_Release_Mutex();
		return  DRV_ERROR;
	}

	reg_mask = CODEC_P1_REG40_HPL_DRV_PGA_CTRL_MASK;
	halRet = codec_I2CWrite(1, 40, reg_write, reg_mask);

	reg_mask = CODEC_P1_REG41_HPR_DRV_PGA_CTRL_MASK;
	halRet = codec_I2CWrite(1, 41, reg_write, reg_mask);

	codec_Release_Mutex();
	return  halRet;
}

/**
* @brief	Set the  output Analog volume of the Output path.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	vol  SINT8
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Set_Outpath_Ana_Vol(T_ZDrvCodec_Handle handle,  SINT8 vol)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
	T_Codec_UseInfo *codec_info = NULL;

	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Set_Outpath_Ana_Vol handle is null !\n");
		return	DRV_ERROR;
	}
	
	codec_Claim_Mutex();

	if (vol > 127 || vol < 0) {
		codec_Release_Mutex();
		return  DRV_ERROR;
	}
	
	codec_info = (T_Codec_UseInfo *)handle;
	reg_write = vol;
	
	if (codec_info->outDev == AUDIO_OUTPUT_RECEIVER || codec_info->outDev == AUDIO_OUTPUT_SPEAKER) {
		reg_mask = CODEC_P1_REG38_LAVOL_SPK_GAIN_MASK;
		halRet = codec_I2CWrite(1, 38, reg_write, reg_mask);
	} else if (codec_info->outDev == AUDIO_OUTPUT_HEADSET) {
		reg_mask = CODEC_P1_REG36_LAVOL_HPL_GAIN_MASK;
		halRet = codec_I2CWrite(1, 36, reg_write, reg_mask);
		reg_mask = CODEC_P1_REG37_RAVOL_HPR_GAIN_MASK;
		halRet += codec_I2CWrite(1, 37, reg_write, reg_mask);
	} else if (codec_info->outDev == AUDIO_OUTPUT_HEADSET_SPEAKER) {
		reg_mask = CODEC_P1_REG38_LAVOL_SPK_GAIN_MASK;
		halRet = codec_I2CWrite(1, 38, reg_write, reg_mask);
		reg_mask = CODEC_P1_REG36_LAVOL_HPL_GAIN_MASK;
		halRet += codec_I2CWrite(1, 36, reg_write, reg_mask);
		reg_mask = CODEC_P1_REG37_RAVOL_HPR_GAIN_MASK;
		halRet += codec_I2CWrite(1, 37, reg_write, reg_mask);
	} else {
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();

	return  halRet;
}

/**
* @brief	Set the  Analog volume of the Input path.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	vol  SINT8
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Set_Inpath_Ana_Vol(T_ZDrvCodec_Handle handle,  SINT8 vol)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;

	T_Codec_UseInfo *codec_info = NULL;

	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Set_Inpath_Ana_Vol handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	if (vol > 59.5 || vol < 0) {
		codec_Release_Mutex();
		return  DRV_ERROR;
	}
	codec_info = (T_Codec_UseInfo *)handle;

	reg_write = vol * 2;
	if (codec_info->inDev == (T_ZDrvCodec_InDevice)AUDIO_INPUT_HANDSET || codec_info->inDev == (T_ZDrvCodec_InDevice)AUDIO_INPUT_SPEAKER || codec_info->inDev == (T_ZDrvCodec_InDevice)AUDIO_OUTPUT_HEADSET) {
		reg_mask = CODEC_P1_REG47_MICPGA_GAIN_MASK;
		halRet = codec_I2CWrite(1, 47, reg_write, reg_mask);
	} else {
		halRet = DRV_ERROR;
	}

	codec_Release_Mutex();
	return  halRet;
}

/**
* @brief	Set the  Digital volume of the Output path.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	invol    T_ZDrv_VpVol
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Set_Outpath_Dig_Vol(T_ZDrvCodec_Handle handle,  T_ZDrv_VpVol invol)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
	SINT16 voltemp = 0;
	SINT16 vol = 0;
	T_ZDrvCodec_OutPath_Digital_Gain outvol = CODEC_OUTPATH_DIG_GAIN_MAX;

	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Set_Outpath_Dig_Vol handle is null !\n");
		return	DRV_ERROR;
	}

	if (invol == VP_VOL_0) {
		outvol = CODEC_OUTPATH_DIG_GAIN_MINUS_63DB;
	} else if (invol == VP_VOL_1) {
		outvol = CODEC_OUTPATH_DIG_GAIN_MINUS_27DB;
	} else if (invol == VP_VOL_2) {
		outvol = CODEC_OUTPATH_DIG_GAIN_MINUS_24DB;
	} else if (invol == VP_VOL_3) {
		outvol = CODEC_OUTPATH_DIG_GAIN_MINUS_21DB;;
	} else if (invol == VP_VOL_4) {
		outvol = CODEC_OUTPATH_DIG_GAIN_MINUS_18DB;
	} else if (invol == VP_VOL_5) {
		outvol = CODEC_OUTPATH_DIG_GAIN_MINUS_15DB;
	} else if (invol == VP_VOL_6) {
		outvol = CODEC_OUTPATH_DIG_GAIN_MINUS_12DB;
	} else if (invol == VP_VOL_7) {
		outvol = CODEC_OUTPATH_DIG_GAIN_MINUS_9DB;;
	} else if (invol == VP_VOL_8) {
		outvol = CODEC_OUTPATH_DIG_GAIN_MINUS_6DB;
	} else if (invol == VP_VOL_9) {
		outvol = CODEC_OUTPATH_DIG_GAIN_MINUS_3DB;
	} else if (invol == VP_VOL_10) {
		outvol = CODEC_OUTPATH_DIG_GAIN_0DB;
	} else if (invol == VP_VOL_11) {
		outvol = CODEC_OUTPATH_DIG_GAIN_PLUS_3DB;
	} else {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "halAudioConfig_SetOutputVol the vol not support vol=%d !\n", invol);
		return DRV_ERROR;
	}

	if (outvol != CODEC_OUTPATH_DIG_GAIN_MAX) {
		vol = 24 - outvol * 1;
	}

	/*vol 24db~-63.5db. */
	if (vol > 24 || vol < -63) {
		return  DRV_ERR_INVALID_PARAM;
	}

	if (vol >= 0) {
		voltemp = vol * 2;
	} else {
		voltemp = (0 - vol) * 2;
		voltemp = 0xff - voltemp + 1;
	}

	codec_Claim_Mutex();
	reg_write = voltemp;
	reg_mask = CODEC_P0_REG65_DAC_LCHN_VOL_MASK;
	halRet = codec_I2CWrite(0, 65, reg_write, reg_mask);

	halRet += codec_I2CRead(0, 65, &reg_write);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec Outpath_Dig_Vol new p0,reg65 regval=0x%x !\n", reg_write);

	reg_mask = CODEC_P0_REG66_DAC_RCHN_VOL_MASK;
	halRet += codec_I2CWrite(0, 66, reg_write, reg_mask);

	halRet += codec_I2CRead(0, 66, &reg_write);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec Outpath_Dig_Vol new p0,reg66 regval=0x%x !\n", reg_write);

	codec_Release_Mutex();
	return  halRet;
}

/**
* @brief	Set the  Digital volume of the Input path.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	invol    T_ZDrvCodec_InPath_Digital_Gain
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_Set_Inpath_Dig_Vol(T_ZDrvCodec_Handle handle, T_ZDrv_VpVol vol)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
	UINT8 reg_read = 0;
	SINT8 coarsevol = 0;
    T_ZDrvCodec_InPath_Digital_Gain invol = CODEC_INPATH_DIG_GAIN_0DB;

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Set_Inpath_Dig_Vol vol=%d !\n", vol);
	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Set_Inpath_Dig_Vol handle is null !\n");
		return	DRV_ERROR;
	}
	 if (vol == VP_VOL_0)
    {
        invol = CODEC_INPATH_DIG_GAIN_MINUS_12DB;
    }
    else if (vol == VP_VOL_1)
    {
        invol = CODEC_INPATH_DIG_GAIN_MINUS_9DB;
    }
    else if (vol == VP_VOL_2)
    {
        invol = CODEC_INPATH_DIG_GAIN_MINUS_6DB;
    }
    else if (vol == VP_VOL_3)
    {
        invol = CODEC_INPATH_DIG_GAIN_MINUS_3DB;
    }
    else if (vol == VP_VOL_4)
    {
        invol = CODEC_INPATH_DIG_GAIN_0DB;
    }
    else if (vol == VP_VOL_5)
    {
        invol = CODEC_INPATH_DIG_GAIN_PLUS_3DB;
    }
	else if (vol == VP_VOL_6)
	{
        invol = CODEC_INPATH_DIG_GAIN_PLUS_6DB;
    }
    else if (vol == VP_VOL_7)
    {
        invol = CODEC_INPATH_DIG_GAIN_PLUS_9DB;
    }
    else if (vol == VP_VOL_8)
    {
        invol = CODEC_INPATH_DIG_GAIN_PLUS_12DB;
    }
    else if (vol == VP_VOL_9)
    {
        invol = CODEC_INPATH_DIG_GAIN_PLUS_15DB;
    }
    else if (vol == VP_VOL_10)
    {
        invol = CODEC_INPATH_DIG_GAIN_PLUS_18DB;
    }
	else if (vol == VP_VOL_11)
    {
        invol = CODEC_INPATH_DIG_GAIN_PLUS_20DB;
    }
    else
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "halAudioConfig_SetInputVol the vol not support vol=%d !\n",vol);
        return DRV_ERROR;
    }
	codec_Claim_Mutex();
	halRet = codec_I2CRead(0, 82, &reg_read);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Set_Inpath_Dig_Vol start p0reg82=%x ,ret=%d !\n", reg_read, halRet);

	halRet = codec_I2CRead(0, 83, &reg_read);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Set_Inpath_Dig_Vol start p0reg83=%x ,ret=%d !\n", reg_read, halRet);

	/*set fine vol */
	reg_write = CODEC_P0_REG82_ADC_DIGVOL_FINE_0DB_VAL;
	reg_mask = CODEC_P0_REG82_ADC_DIGVOL_FINE_MASK;
	halRet = codec_I2CWrite(0, 82, reg_write, reg_mask);

	if (invol != CODEC_INPATH_DIG_GAIN_MAX) {
		coarsevol = -12 + invol * 1;
	}

	/*coarsevol -12~20db */
	if (coarsevol < -12 || coarsevol > 20) {
		codec_Release_Mutex();
		return  DRV_ERR_INVALID_PARAM;
	}

	reg_write = (UINT8)(20 - coarsevol) * 2;
	reg_write = (0x28 - reg_write) & 0xff;
	reg_mask = CODEC_P0_REG83_ADC_DIGVOL_COARSE_MASK;
	halRet += codec_I2CWrite(0, 83, reg_write, reg_mask);

	halRet = codec_I2CRead(0, 82, &reg_read);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Set_Inpath_Dig_Vol end p0reg82=%x ,ret=%d !\n", reg_read, halRet);

	halRet = codec_I2CRead(0, 83, &reg_read);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_Set_Inpath_Dig_Vol end p0reg83=%x ,ret=%d !\n", reg_read, halRet);

	codec_Release_Mutex();

	return  halRet;
}

/**
* @brief	Set the Side Tone.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	onoff    BOOL
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetSideTone(T_ZDrvCodec_Handle handle, BOOL onoff)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;

	T_Codec_UseInfo *codec_info = NULL;

	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetSideTone handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;
	if (codec_info->handleState == HANDLE_IN_USE) {
		if (onoff) {
			if ((codec_info->inDev == AUDIO_INPUT_HANDSET && codec_info->outDev == AUDIO_OUTPUT_RECEIVER) ||  \
			     (codec_info->inDev == AUDIO_INPUT_SPEAKER && codec_info->outDev == AUDIO_OUTPUT_SPEAKER)) {
				reg_write = CODEC_P1_REG35_MIC1LP_INPUT_ROUTE_LCHN_MIXER_VAL;
				reg_mask = CODEC_P1_REG35_MIC1LP_INPUT_ROUTE_LCHN_MIXER_MASK;

			} else if (codec_info->inDev == AUDIO_INPUT_HEADSET || codec_info->outDev == AUDIO_OUTPUT_HEADSET) {
				reg_write = CODEC_P1_REG35_MIC1RP_INPUT_ROUTE_LCHN_MIXER_VAL | CODEC_P1_REG35_MIC1RP_INPUT_ROUTE_RCHN_MIXER_VAL;
				reg_mask = CODEC_P1_REG35_MIC1RP_INPUT_ROUTE_LCHN_MIXER_MASK | CODEC_P1_REG35_MIC1RP_INPUT_ROUTE_RCHN_MIXER_MASK;
			}
		} else {
			if ((codec_info->inDev == AUDIO_INPUT_HANDSET && codec_info->outDev == AUDIO_OUTPUT_RECEIVER) || \
			     (codec_info->inDev == AUDIO_INPUT_SPEAKER && codec_info->outDev == AUDIO_OUTPUT_SPEAKER)) {
				reg_write = CODEC_P1_REG35_MIC1LP_INPUT_UNROUTE_LCHN_VAL;
				reg_mask = CODEC_P1_REG35_MIC1LP_INPUT_ROUTE_LCHN_MIXER_MASK;

			} else if (codec_info->inDev == AUDIO_INPUT_HEADSET || codec_info->outDev == AUDIO_OUTPUT_HEADSET) {
				reg_write = CODEC_P1_REG35_MIC1RP_INPUT_UNROUTE_LCHN_VAL | CODEC_P1_REG35_MIC1RP_INPUT_UNROUTE_RCHN_VAL;
				reg_mask = CODEC_P1_REG35_MIC1RP_INPUT_ROUTE_LCHN_MIXER_MASK | \
				           CODEC_P1_REG35_MIC1RP_INPUT_ROUTE_RCHN_MIXER_MASK;
			}
		}
		halRet = codec_I2CWrite(1, 35, reg_write, reg_mask);
	} else {
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();
	return  halRet;
}

/**
* @brief	Set the Mic Bias.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	micBiasCtrl    T_ZDrvMicBias_Ctrl
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetMicBias(T_ZDrvCodec_Handle handle, T_ZDrvMicBias_Ctrl micBiasCtrl)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
	T_Codec_UseInfo *codec_info = NULL;
	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetMicBias handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;
	if (codec_info->handleState == HANDLE_IN_USE) {
		if (micBiasCtrl ==  MICBIAS_OUT_POWERDOWN) {
			reg_write = CODEC_P1_REG46_MICBIAS_OUT_PWOFF_VAL;
		} else if (micBiasCtrl ==  MICBIAS_OUT_2V) {
			reg_write = CODEC_P1_REG46_MICBIAS_OUT_PW2V_VAL;
		} else if (micBiasCtrl ==  MICBIAS_OUT_2_5V) {
			reg_write = CODEC_P1_REG46_MICBIAS_OUT_PW2_5V_VAL;
		} else if (micBiasCtrl ==  MICBIAS_OUT_AVDD) {
			reg_write = CODEC_P1_REG46_MICBIAS_OUT_PWAVDD_VAL;
		} else {
			codec_Release_Mutex();
			return DRV_ERR_INVALID_PARAM;
		}
		
		if (micBiasCtrl !=  MICBIAS_OUT_POWERDOWN) {
			reg_write |= CODEC_P1_REG46_MICBIAS_PWON_CTRL2_VAL;
		}
		reg_mask = CODEC_P1_REG46_MICBIAS_OUT_CTRL_MASK | CODEC_P1_REG46_MICBIAS_PWON_CTRL_MASK;

		halRet = codec_I2CWrite(1, 46, reg_write, reg_mask);

	} else {
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();
	return  halRet;
}

/**
* @brief	Set the Input Agc.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	AGCPara    Poiter to T_ZDrvAudCodec_AGC_Parameter
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetInputAgc(T_ZDrvCodec_Handle handle, T_ZDrvAudCodec_AGC_Parameter * AGCPara)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	//UINT8 reg_mask = 0;
	UINT8 regval[7] = {0};
	T_Codec_UseInfo *codec_info = NULL;
	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetInputAgc handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;
	if (codec_info->handleState == HANDLE_IN_USE) {
		if (AGCPara->onOff == 1) {
			*regval = CODEC_P0_REG86_AGC_DISABLED_VAL;
			/*AGC target level */
			if (AGCPara->targetLevel == -5.5) {
				*regval |= CODEC_P0_REG86_AGC_TARLVEL_MINUS_5_5DB_VAL;
			} else if (AGCPara->targetLevel == -8) {
				*regval |= CODEC_P0_REG86_AGC_TARLVEL_MINUS_8DB_VAL;
			} else if (AGCPara->targetLevel == -10) {
				*regval |= CODEC_P0_REG86_AGC_TARLVEL_MINUS_10DB_VAL;
			} else if (AGCPara->targetLevel == -12) {
				*regval |= CODEC_P0_REG86_AGC_TARLVEL_MINUS_12DB_VAL;
			} else if (AGCPara->targetLevel == -14) {
				*regval |= CODEC_P0_REG86_AGC_TARLVEL_MINUS_14DB_VAL;
			} else if (AGCPara->targetLevel == -17) {
				*regval |= CODEC_P0_REG86_AGC_TARLVEL_MINUS_17DB_VAL;
			} else if (AGCPara->targetLevel == -20) {
				*regval |= CODEC_P0_REG86_AGC_TARLVEL_MINUS_20DB_VAL;
			} else if (AGCPara->targetLevel == -24) {
				*regval |= CODEC_P0_REG86_AGC_TARLVEL_MINUS_24DB_VAL;
			} else {
				codec_Release_Mutex();
				return  DRV_ERR_INVALID_PARAM;
			}

			/*AGC hysteresis setting  AGC noise threshold */
			if (AGCPara->noiseThreshold < -90 || AGCPara->noiseThreshold > -30) {
				codec_Release_Mutex();
				return  DRV_ERR_INVALID_PARAM;
			} else {
				*(regval + 1) =( CODEC_P0_REG87_AGC_HYST_SET_1DB_VAL | (AGC_NOISE_THRESHOLD (AGCPara->noiseThreshold)));//xiu
			}

			/*AGC Maximum Gain */
			if (AGCPara->maxgain < 0 || AGCPara->maxgain > 59.5) {
				codec_Release_Mutex();
				return  DRV_ERR_INVALID_PARAM;
			} else {
				*(regval + 2) = AGC_MAX_GAIN(AGCPara->maxgain);
			}

			/*AGC Attack Time   n*(32/fs) */
			if (AGCPara->attackTime < 1 || AGCPara->attackTime > 63) {
				codec_Release_Mutex();
				return  DRV_ERR_INVALID_PARAM;
			} else {
				*(regval + 3) = AGC_ATTACK_TIME(AGCPara->attackTime);
			}

			/*AGC decay Time   n*(32/fs) */
			if (AGCPara->decayTime < 1 || AGCPara->decayTime > 63) {
				codec_Release_Mutex();
				return  DRV_ERR_INVALID_PARAM;
			} else {
				*(regval + 4) = AGC_DECAY_TIME(AGCPara->decayTime);
			}

			/*AGC Noise Debounce  n/fs */
			if (AGCPara->noiseDebounce < 0 || AGCPara->noiseDebounce > 21 * 4096) {
				codec_Release_Mutex();
				return  DRV_ERR_INVALID_PARAM;
			} else {
				if (AGCPara->noiseDebounce <= 4096) {
					*(regval + 5) = log10(AGCPara->noiseDebounce) / log10(2) - 1;
				} else {
					*(regval + 5) = (AGCPara->noiseDebounce / 4096 - 1) + 0x0b;
				}
			}
			/*AGC Signal Debounce  n/fs */
			if (AGCPara->signalDebounce < 0 || AGCPara->signalDebounce > 6 * 2048) {
				codec_Release_Mutex();
				return  DRV_ERR_INVALID_PARAM;
			} else {
				if (AGCPara->signalDebounce <= 2048) {
					*(regval + 6) = log10(AGCPara->signalDebounce) / log10(2) - 1;
				} else {
					*(regval + 6) = (AGCPara->signalDebounce / 2048 - 1) + 0x0A;
				}
			}
		} else {
			*regval = 0x00;
			*(regval + 1) = 0x00;
			*(regval + 2) = 0x7f;
			*(regval + 3) = 0x00;
			*(regval + 4) = 0x00;
			*(regval + 5) = 0x00;
			*(regval + 6) = 0x00;
		}

		/*write  7 reg, page0 start reg86 */
		reg_write = 0;
		halRet = codec_I2cDevWrite(&s_codecI2CDev, 0, &reg_write, 1); /*set page to 0 */
		halRet += codec_I2cDevWrite(&s_codecI2CDev, 86, regval, 7);
	} else {
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();
	return  halRet;
}

/**
* @brief	Set the Output Agc.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	DRCPara    Poiter to T_ZDrvAudCodec_DRC_Parameter
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetOutputDrc(T_ZDrvCodec_Handle handle, T_ZDrvAudCodec_DRC_Parameter * DRCPara)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 regval[3] = {0};
	UINT8 regHpfVal[6] = {0x7f, 0xab, 0x80, 0x55, 0x7f, 0x56};	/*page 9,reg14 */
	UINT8 regLpfVal[6] = {0x00, 0x11, 0x00, 0x11, 0x7f, 0xde};	/*page 9,reg20 */
	T_Codec_UseInfo *codec_info = NULL;
	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetOutputDrc handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;
	if (codec_info->handleState == HANDLE_IN_USE) {
		if (DRCPara->onOff == 1) {
			*regval = CODEC_P0_REG68_DRC_EN_LCHN_VAL | CODEC_P0_REG68_DRC_EN_RCHN_VAL;

			if (DRCPara->threshold > -3 || DRCPara->threshold < -24) {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetOutputDrc threshold val is out range!\n");
				codec_Release_Mutex();
				return  DRV_ERR_INVALID_PARAM;
			}
			if (DRCPara->threshold == -3) {
				*regval |= CODEC_P0_REG68_DRC_THOLD_MINUS_3DB_VAL;
			} else if (DRCPara->threshold == -6) {
				*regval |= CODEC_P0_REG68_DRC_THOLD_MINUS_6DB_VAL;
			} else if (DRCPara->threshold == -9) {
				*regval |= CODEC_P0_REG68_DRC_THOLD_MINUS_9DB_VAL;
			} else if (DRCPara->threshold == -12) {
				*regval |= CODEC_P0_REG68_DRC_THOLD_MINUS_12DB_VAL;
			} else if (DRCPara->threshold == -15) {
				*regval |= CODEC_P0_REG68_DRC_THOLD_MINUS_15DB_VAL;
			} else if (DRCPara->threshold == -18) {
				*regval |= CODEC_P0_REG68_DRC_THOLD_MINUS_18DB_VAL;
			} else if (DRCPara->threshold == -21) {
				*regval |= CODEC_P0_REG68_DRC_THOLD_MINUS_21DB_VAL;
			} else if (DRCPara->threshold == -24) {
				*regval |= CODEC_P0_REG68_DRC_THOLD_MINUS_24DB_VAL;
			} else {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetOutputDrc threshold val not support !\n");
				codec_Release_Mutex();
				return  DRV_ERR_INVALID_PARAM;
			}

			if (DRCPara->hysteresis < 0 || DRCPara->hysteresis > 3) {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetOutputDrc hysteresis val is out range!\n");
				codec_Release_Mutex();
				return  DRV_ERR_INVALID_PARAM;
			}
			if (DRCPara->hysteresis == 0) {
				*regval |= CODEC_P0_REG68_DRC_HYST_0DB_VAL;
			} else if (DRCPara->hysteresis == 1) {
				*regval |= CODEC_P0_REG68_DRC_HYST_1DB_VAL;
			} else if (DRCPara->hysteresis == 2) {
				*regval |= CODEC_P0_REG68_DRC_HYST_2DB_VAL;
			} else if (DRCPara->hysteresis == 3) {
				*regval |= CODEC_P0_REG68_DRC_HYST_3DB_VAL;
			} else {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetOutputDrc hysteresis val not support !\n");
				codec_Release_Mutex();
				return  DRV_ERR_INVALID_PARAM;
			}

			/*DRC Control 2 */
			if (DRCPara->holdTime == 0) {
				*(regval + 1) = 0;
			} else if ( (DRCPara->holdTime < 32768) && (DRCPara->holdTime >= 32)) {
				*(regval + 1) = log10(DRCPara->holdTime / 32) / log10(2) + 0x08;
			} else if ( (DRCPara->holdTime <= 5 * 32768) && (DRCPara->holdTime >= 32768)) {
				*(regval + 1) = (DRCPara->holdTime / 32768 - 1) + 0x50;
			} else {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetOutputDrc holdTime=%d val not support !\n", DRCPara->holdTime);
				codec_Release_Mutex();
				return  DRV_ERR_INVALID_PARAM;
			}
			/*DRC Control 3   reg value */
			if (DRCPara->attackRate < 0 || DRCPara->attackRate > 15) {
				codec_Release_Mutex();
				return  DRV_ERR_INVALID_PARAM;
			} else {
				*(regval + 2) = DRCPara->attackRate << 4;
			}

			if (DRCPara->decayRate < 0 || DRCPara->decayRate > 15) {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetOutputDrc decayRate=%d val not support !\n", DRCPara->decayRate);
				codec_Release_Mutex();
				return  DRV_ERR_INVALID_PARAM;
			} else {
				*(regval + 2) |= DRCPara->decayRate;
			}
		} else {
			*regval = 0x0f;
			*(regval + 1) = 0x38;
			*(regval + 2) = 0x00;
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetOutputDrc set drc off para!\n");

		}
		/*write  3 reg, page0 start reg68 */
		reg_write = 0;
		halRet = codec_I2cDevWrite(&s_codecI2CDev, 0, &reg_write, 1); /*set page to 0 */
		halRet += codec_I2cDevWrite(&s_codecI2CDev, 68, regval, 3);
		/*set HPF and LPF */
		if (DRCPara->onOff == 1) {
			reg_write = 9;
			halRet += codec_I2cDevWrite(&s_codecI2CDev, 0, &reg_write, 1); /*set page to 0 */
			halRet += codec_I2cDevWrite(&s_codecI2CDev, 14, regHpfVal, 6);
			halRet += codec_I2cDevWrite(&s_codecI2CDev, 20, regLpfVal, 6);
		}
	} else {
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();
	return  halRet;
}

/**
* @brief	Set the Loop Back.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	onoff    BOOL
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetLoopBack(T_ZDrvCodec_Handle handle, BOOL onoff)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_read = 0;
	UINT8 reg_mask = 0;
	T_Codec_UseInfo *codec_info = NULL;
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetLoopBack !\n");

	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetLoopBack handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	/*Page 1 / Register 35: DAC_L and DAC_R Output Mixer Routing */
	codec_info = (T_Codec_UseInfo *)handle;
	if (codec_info->handleState == HANDLE_IN_USE) {
		reg_write = CODEC_P0_REG29_ADC_TO_DAC_LPBACK_EN_VAL; 
		reg_mask = CODEC_P0_REG29_ADC_TO_DAC_LPBACK_CTRL_MASK; 
		halRet = codec_I2CWrite(0, 29, reg_write, reg_mask);
		halRet += codec_I2CRead(0, 29, &reg_read);
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec SetOutputPath new p0reg29 regval=0x%x !\n", reg_read);
	} else {
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();
	return  halRet;
}

/**
* @brief	Test GPIO1 Function.
*
* @param	handle	 T_ZDrvCodec_Handle
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_TestGPIO1Fun(T_ZDrvCodec_Handle handle)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
	T_Codec_UseInfo *codec_info = NULL;
	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_TestGPIO1Fun handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;
	if (codec_info->handleState == HANDLE_IN_USE) {
		/* GPO TEST */
		reg_write = CODEC_P0_REG51_GPIO1_FUN_GPO_VAL | CODEC_P0_REG51_GPIO1_OUTHIGH_VAL;
		reg_mask = CODEC_P0_REG51_GPIO1_FUN_MASK | CODEC_P0_REG51_GPIO1_OUT_MASK;
		halRet = codec_I2CWrite(0, 51, reg_write, reg_mask);
		reg_write = CODEC_P0_REG51_GPIO1_OUTLOW_VAL;
		reg_mask = CODEC_P0_REG51_GPIO1_OUT_MASK;
		halRet = codec_I2CWrite(0, 51, reg_write, reg_mask);
		/* CLK TEST */
		reg_write = CODEC_P0_REG51_GPIO1_FUN_CLKOUT_VAL;
		reg_mask = CODEC_P0_REG51_GPIO1_FUN_MASK;
		halRet = codec_I2CWrite(0, 51, reg_write, reg_mask);
	} else {
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();

	return  halRet;
}

/**
* @brief	Set the mute function in the input path.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	onoff	 BOOL
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetInpathMute(T_ZDrvCodec_Handle handle, BOOL onoff)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
	T_Codec_UseInfo *codec_info = NULL;
	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetInpathMute handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;
	if (codec_info->handleState == HANDLE_IN_USE) {
		if (onoff) {
			reg_write = CODEC_P0_REG82_ADC_MUTE_VAL;
		} else {
			reg_write = CODEC_P0_REG82_ADC_UNMUTE_VAL;
		}
		reg_mask = CODEC_P0_REG82_ADC_MUTE_CTRL_MASK;
		halRet = codec_I2CWrite(0, 82, reg_write, reg_mask);
	} else {
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();
	return  halRet;
}

/**
* @brief	Set the mute function in the output path.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	onoff	 BOOL
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_SetOutpathMute(T_ZDrvCodec_Handle handle, BOOL onoff)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
	T_Codec_UseInfo *codec_info = NULL;
	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetOutpathMute handle is null !\n");
		return DRV_ERROR;
	}

	codec_Claim_Mutex();
	codec_info = (T_Codec_UseInfo *)handle;

	if (codec_info->handleState == HANDLE_IN_USE) {
		if (onoff) {
			reg_write = CODEC_P0_REG64_DAC_LCHN_MUTE_VAL | CODEC_P0_REG64_DAC_RCHN_MUTE_VAL;
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetOutpathMute set MUTE!\n");
		} else {
			reg_write = CODEC_P0_REG64_DAC_LCHN_UNMUTE_VAL | CODEC_P0_REG64_DAC_RCHN_UNMUTE_VAL;
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec zDrvCodec_SetOutpathMute set UNMUTE !\n");
		}
		reg_mask = CODEC_P0_REG64_DAC_LCHN_MUTE_CTRL_MASK | CODEC_P0_REG64_DAC_RCHN_MUTE_CTRL_MASK;
		halRet = codec_I2CWrite(0, 64, reg_write, reg_mask);
	} else {
		halRet = DRV_ERROR;
	}
	codec_Release_Mutex();
	return  halRet;
}

/**
* @brief	Set Input path of the FM.
*
* @param	handle	 T_ZDrvCodec_Handle
* @param	onoff	 BOOL
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 codec_SetFmInPath(T_ZDrvCodec_Handle handle, BOOL onoff)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
	UINT8 *reg_val = NULL;
	T_Codec_UseInfo *codec_info = NULL;
#ifdef CODEC_TEST
	UINT8 reg_readval[8] = {0};
#endif

	if (handle == 0) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec codec_SetFmInPath handle is null !\n");
		return	DRV_ERROR;
	}

	codec_Claim_Mutex();

	codec_info = (T_Codec_UseInfo *)handle;
	/*add  fm input ctrl */
	if (onoff) {
		reg_val = handsetFMDigitalInPathPara;
		/* open digital loop channel */
		reg_write = CODEC_P0_REG29_ADC_TO_DAC_LPBACK_EN_VAL | CODEC_P0_REG29_DIN_TO_DOUT_EN_VAL;
		reg_mask = CODEC_P0_REG29_ADC_TO_DAC_LPBACK_CTRL_MASK | CODEC_P0_REG29_DIN_TO_DOUT_CTRL_MASK;
	} else {
		reg_val = (UINT8 *)inPathResetPara;//xiu
		/* close  digital loop channel */
		reg_write = CODEC_P0_REG29_ADC_TO_DAC_LPBACK_DIS_VAL | CODEC_P0_REG29_DIN_TO_DOUT_DIS_VAL;
		reg_mask = CODEC_P0_REG29_ADC_TO_DAC_LPBACK_CTRL_MASK | CODEC_P0_REG29_DIN_TO_DOUT_CTRL_MASK;
	}

	/* write first 4 reg, page1 start reg47 */
	reg_write = 1;
	halRet = codec_I2cDevWrite(&s_codecI2CDev, 0, &reg_write, 1); /*set page to 1 */
	halRet += codec_I2cDevWrite(&s_codecI2CDev, 46, reg_val, 4);
#ifdef CODEC_TEST
	halRet += codec_I2cDevRead(&s_codecI2CDev, 46, reg_readval, 4);
#endif

	/* write next 3 reg,page0  start reg81 */
	reg_write = 0;
	halRet += codec_I2cDevWrite(&s_codecI2CDev, 0, &reg_write, 1); /*set page to 0 */
	halRet += codec_I2cDevWrite(&s_codecI2CDev, 81, reg_val + 4, 3);
#ifdef CODEC_TEST
	halRet += codec_I2cDevRead(&s_codecI2CDev, 81, reg_readval + 4, 3);
#endif

	halRet += codec_I2CWrite(0, 29, reg_write, reg_mask);

	if (halRet == DRV_SUCCESS) {
		if (onoff) {
			codec_info->inDev = AUDIO_INPUT_FM_ANATODIG;
		} else {
			codec_info->inDev = AUDIO_INPUT_MAX;
		}
	}
	codec_Release_Mutex();

	return  halRet;
}

/**
* @brief	Set the Mic Bias in the earpiece case.
*
* @param	micBiasCtrl	 T_ZDrvMicBias_Ctrl
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 earp_SetMicBias(T_ZDrvMicBias_Ctrl micBiasCtrl)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec earp_SetMicBias!\n");

	if (micBiasCtrl ==  MICBIAS_OUT_POWERDOWN) {
		reg_write = CODEC_P1_REG46_MICBIAS_VOLUME_VAL_OFF;
	} else if (micBiasCtrl ==  MICBIAS_OUT_2V) {
		reg_write = CODEC_P1_REG46_MICBIAS_VOLUME_VAL_2V;
	} else if (micBiasCtrl ==  MICBIAS_OUT_2_5V) {
		reg_write = CODEC_P1_REG46_MICBIAS_VOLUME_VAL_2V5;
	} else if (micBiasCtrl ==  MICBIAS_OUT_AVDD) {
		reg_write = CODEC_P1_REG46_MICBIAS_VOLUME_VAL_AVDD;
	}

	reg_write = reg_write | CODEC_P1_REG46_MICBIAS_BEFORE_HEADSET_DETECTED; /* micbias is povewd always */
	reg_mask = CODEC_P1_REG46_MICBIAS_VOLUME_MASK | CODEC_P1_REG46_MICBIAS_HEADSET_MASK;
	halRet = codec_I2CWrite(1, 46, reg_write, reg_mask);

	if (halRet != DRV_SUCCESS) {
		return DRV_ERROR;
	}

	return DRV_SUCCESS;
}

/**
* @brief	Detect the earpiece enable.
*
* @param	onoff	 BOOL
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 codec_EarpieceDetectEnable(BOOL onoff)
{
	SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec_EarpieceDetectEnable set status is %d !\n", onoff);

	/* codec open */
	if (onoff) {
		/* open 26mhz CLK  for codec */
		halRet = zDrvI2S_setCodecClk();

		reg_write = 0x1A;
		reg_mask = 0x7f;
		halRet += codec_I2CWrite(3, 16, reg_write, reg_mask);

		reg_write = CODEC_P0_REG67_HEADSET_DETECT_EN_VAL | CODEC_P0_REG67_HEADSET_DETECT_HEADSET_DEBOUNCE_512 | CODEC_P0_REG67_HEADSET_DETECT_BTN_DEBOUNCE_32;
		reg_mask = CODEC_P0_REG67_HEADSET_DETECT_EN_MASK | CODEC_P0_REG67_HEADSET_DETECT_HEADSET_DEBOUNCE_MASK | CODEC_P0_REG67_HEADSET_DETECT_BTN_DEBOUNCE_MASK;
		halRet += codec_I2CWrite(0, 67, reg_write, reg_mask);

		/*set the GPIO TO INT1 */
		reg_write = CODEC_P0_REG51_GPIO_CTR_INT1_VAL;
		reg_mask = CODEC_P0_REG51_GPIO_CTR_MASK;
		halRet += codec_I2CWrite(0, 51, reg_write, reg_mask);

		reg_write = CODEC_P0_REG48_HEADSET_IRQ_INT1_VAL | CODEC_P0_REG48_BTN_IRQ_INT1_VAL;
		reg_mask = CODEC_P0_REG48_HEADSET_IRQ_INT1_VAL | CODEC_P0_REG48_BTN_IRQ_INT1_VAL;
		halRet += codec_I2CWrite(0, 48, reg_write, reg_mask); /*set headset and btn irq int1 enable; */
		earp_SetMicBias(MICBIAS_OUT_2_5V);
	} else {
		reg_write = 0;
		reg_mask = CODEC_P0_REG67_HEADSET_DETECT_EN_MASK;
		halRet += codec_I2CWrite(0, 67, reg_write, reg_mask);
		
		reg_write = 0;
		reg_mask = CODEC_P0_REG51_GPIO_CTR_MASK;
		halRet += codec_I2CWrite(0, 51, reg_write, reg_mask);
		
		reg_write = 0;
		reg_mask = CODEC_P0_REG48_HEADSET_IRQ_INT1_VAL | CODEC_P0_REG48_BTN_IRQ_INT1_VAL;
		halRet += codec_I2CWrite(0, 48, reg_write, reg_mask); /*set headset and btn irq int1 enable; */

		earp_SetMicBias(MICBIAS_OUT_POWERDOWN);
	}

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "codec_EarpieceDetectEnable onoff halRet is %d !\n", halRet);
	return halRet;
}

/**
* @brief	Get codec state.
*
* @param	void
* @return	CODEC_IDLE-FALSE, other-TRUE
*/
BOOL zDrvCodec_GetUsestate(VOID)
{
	if ((voiceUseInfo.useState != CODEC_IDLE)||(fmUseInfo.useState != CODEC_IDLE)||(audioUseInfo.useState != CODEC_IDLE)) {
	    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvCodec_GetUsestate true !\n");
		return TRUE;
	} else {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvCodec_GetUsestate false !\n");
        return FALSE;
	}
}

/**
* @brief	Detect the earpiece enable.
*
* @param	onoff	 BOOL
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_EarpieceDetectEnable(BOOL onoff)
{
	SINT32 halRet = DRV_SUCCESS;

	halRet = codec_EarpieceDetectEnable(onoff);
	if (DRV_SUCCESS == halRet) {
		isEarpDetect = onoff;
	}
	return halRet;
}

/**
* @brief	Detect the earpiece wakeup.
*
* @param	void
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_EarpieceDetectWakeup(VOID)
{
    SINT32 halRet = DRV_SUCCESS;

	if(isEarpDetect != 1) {
		return DRV_SUCCESS;
	}
		
    halRet = codec_EarpieceDetectEnable(1);
	
    return	halRet;
}

/**
* @brief	Detect the earpiece sleep.
*
* @param	void
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvCodec_EarpieceDetectSleep(VOID)
{
    SINT32 halRet = DRV_SUCCESS;
	UINT8 reg_write = 0;
	UINT8 reg_mask = 0;
	
	if(isEarpDetect != 1) {
		return DRV_SUCCESS;
	}

	reg_write= CODEC_P0_REG1_RESET_VAL;
	reg_mask = CODEC_P0_REG1_RESET_MASK;
	halRet=codec_I2CWrite( 0, 1,reg_write , reg_mask);

 //   halRet = codec_EarpieceDetectEnable(0);
    return	halRet;
}

