/**
 *   Copyright (C) Infineon Technologies Denmark A/S. All rights reserved.
 *
 * This document contains proprietary information belonging to Infineon
 * Technologies Denmark A/S. Passing on and copying of this document, use
 * and communication of its contents is not permitted without prior written
 * authorisation.
 *
 * Description:
 *   Main State machine for the audio driver.
 *
 * Revision Information:
 *   File name: \dwddrv\AUD\src\aud_main.c
 *   Version: \main\245
 *   Date: 2007-05-16 14:46:57
 *   Responsible: eps
 *   Comment:
 *     corrected lint warnings
 */

/*******************************************************************************
*
*                               Include files
*
*******************************************************************************/
//#include "scttypes.h"           /* must be included in all files */
#include "bastypes.h"           /* must be included in all files */
#include "aud_drv.h"            /* audio driver types and functions */
#include "aud_com.h"            /* init and trap functions */
#include "aud_data.h"           /* for access to aud_setting */
#if defined (AUD_MODULE_TEST_IFWD)
#include "atctstif.h"           /* buffer for logging */
#endif
//#include "REG_SGOLD_Regs.h"
#if defined(TRAP_HANDLING)
//#include "trap.h"               /* audio define for trap handling */
#endif
/* init functions and state machines/dispatchers */
#if defined (AUD_EXTERNAL_RINGER)
#include "aud_ringer.h"
#endif
#if defined (INTERNAL_POLYRINGER)
#include "aud_ringer.h"
#include "aud_intern_midilib.h"
#endif
#include "aud_tone.h"
#include "aud_i2s.h"
#include "aud_vms_playback.h"
#include "aud_vms_rec.h"
#include "aud_ear.h"
#include "aud_path_control.h"
#include "aud_volume.h"
#include "aud_nv.h"

#include "drvs_general.h"
#include "drvs_teak.h"
/*******************************************************************************
*
*                                Internal data
*
*******************************************************************************/
extern aud_eep_static_type aud_EEP_static;
extern SINT32 AUD_create_task(VOID);
extern void aud_nv_static_data_init(void);
extern SINT32 zDrv_VoiceMemoCreateThread(VOID);
extern SINT32 zDrv_MidiCreateThread(VOID);
extern SINT32 zDrv_ToneCreateThread(VOID);


#define AUD_MAX_NOF_DB_INDEX 20
typedef struct {
    SDL_Pid         sender;
    aud_event_enum  event;
    U32             parm1;
    U32             parm2;
    U32             parm3;
    U32             parm4;
} aud_main_db_type;
//static aud_main_db_type aud_main_db_event[AUD_MAX_NOF_DB_INDEX]; /* database with events sent to the dispatcher */
//static U16              aud_main_db_event_index; /* where to place the next event in the event database */

/*******************************************************************************
*
*                             Internal functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: aud_resize_aud_setting
* Return.....: -
* Description: Make sure that the default path settings are at the end of the
*              arrays in aud_setting. This is necessary such that more paths
*              can be defined in aud_drv.h without needing to update aud_data.c
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static void aud_resize_aud_setting(void)
{
    U32 cnt=0; /* counter for loops */

    /* find the index of the default data */
    for(cnt=0; cnt<aud_uplink_source_end; cnt++)
    {
        if(aud_setting.p_aud_audio_uplink_parms[cnt] == &aud_uplink_source_default_data)
        {
            break;
        }
    }
    /* copy default data to last array element */
    aud_setting.p_aud_audio_uplink_parms[aud_uplink_source_end]=aud_setting.p_aud_audio_uplink_parms[cnt];
    /* reset the un-initialized part of the array with null-pointers (meaning that the path does not exist) */
    for(; cnt<aud_uplink_source_end; cnt++)
    {
        aud_setting.p_aud_audio_uplink_parms[cnt] = NULL;
    }

    /* now do the same for downlink paths */
    for(cnt=0; cnt<aud_downlink_source_end; cnt++)
    {
        if(aud_setting.p_aud_audio_downlink_parms[cnt] == &aud_downlink_source_default_data)
        {
            break;
        }
    }
    aud_setting.p_aud_audio_downlink_parms[aud_downlink_source_end] = aud_setting.p_aud_audio_downlink_parms[cnt];
    for(; cnt<aud_downlink_source_end; cnt++)
    {
        aud_setting.p_aud_audio_downlink_parms[cnt]=NULL;
    }
}

/*------------------------------------------------------------------------------
* Function...: aud_init_test_handles
* Return.....: -
* Description: Initialise test handles used by PC applications such as PhoneTool.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static void aud_init_test_handles(void)
{
    aud_resource_table[aud_resource_speech].test_handle              = AUD_SPEECH_TEST_HANDLE;
    aud_resource_table[aud_resource_tone_generator].test_handle      = AUD_TONE_GENERATOR_TEST_HANDLE;
    aud_resource_table[aud_resource_ringer].test_handle              = AUD_RINGER_TEST_HANDLE;
    aud_resource_table[aud_resource_vibrator].test_handle            = AUD_VIBRATOR_TEST_HANDLE;
    aud_resource_table[aud_resource_amplifier].test_handle           = AUD_AMPLIFIER_TEST_HANDLE;
    aud_resource_table[aud_resource_radio].test_handle               = AUD_FM_RADIO_TEST_HANDLE;
    aud_resource_table[aud_resource_record_vm].test_handle           = AUD_RECORDING_VM_TEST_HANDLE;
    aud_resource_table[aud_resource_playback_vm].test_handle         = AUD_PLAYBACK_VM_TEST_HANDLE;
    aud_resource_table[aud_resource_playback_mp3].test_handle        = AUD_PLAYBACK_MP3_TEST_HANDLE;
    aud_resource_table[aud_resource_PCM_channel].test_handle         = AUD_PCM_CHANNEL_TEST_HANDLE;
    aud_resource_table[aud_resource_midi_player].test_handle         = AUD_MIDI_PLAYER_TEST_HANDLE;
    aud_resource_table[aud_resource_tty].test_handle                 = AUD_TTY_TEST_HANDLE;
    aud_resource_table[aud_resource_playback_pcm].test_handle        = AUD_PLAYBACK_PCM_TEST_HANDLE;
    aud_resource_table[aud_resource_record_pcm].test_handle          = AUD_RECORD_PCM_TEST_HANDLE;
    aud_resource_table[aud_resource_mapi].test_handle                = AUD_MAPI_TEST_HANDLE;
    aud_resource_table[aud_resource_inband].test_handle              = AUD_INBAND_TEST_HANDLE;
    aud_resource_table[aud_resource_smpower_backspeaker].test_handle = AUD_SMPOWER_BACKSPEAKER_TEST_HANDLE;
    aud_resource_table[aud_resource_sbc_encoder].test_handle         = AUD_SBC_ENCODER_TEST_HANDLE;
    aud_resource_table[aud_resource_I2S2_Rx].test_handle             = AUD_I2S2_RX_TEST_HANDLE;
    aud_resource_table[aud_resource_I2S2_Tx].test_handle             = AUD_I2S2_TX_TEST_HANDLE;
}

/*******************************************************************************
*
*                        Externally visible functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_init
* Return.....: -
* Description: Initialise the audio driver.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void AUD_init(void)
{
    static U8 audio_initialised = 0; /* prevents AUD_init from executing the functionality more than once */
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"aud_main AUD_init\n");
    if(0 == audio_initialised)
    {
        audio_initialised = 1;      /* mark audio initialised, so multiple calls doesn't crash audio */
        aud_resize_aud_setting();   /* make sure the default paths are at the end of aud_setting no matter how many paths are defined */
        if( (aud_setting.p_aud_audio_downlink_parms[aud_downlink_source_end] != &aud_downlink_source_default_data) ||
            (aud_setting.p_aud_audio_uplink_parms[aud_uplink_source_end] != &aud_uplink_source_default_data) )
        {
            AUD_ERROR_REPORTING(aud_error_grade1, TRAP_AUD_ERROR, "Wrong number of paths")
        }
        /* The enums must take up exactly 1 byte of space, otherwise E_update_uplinkpath will not function the right way */
		/*klocwork 3  INVARIANT_CONDITION.GEN delete if*/
		/*if(sizeof(aud_uplink_source_enum)!=1 || sizeof(aud_downlink_source_enum)!=1)
        {
            AUD_ERROR_REPORTING(aud_error_grade1, TRAP_AUD_ERROR, "Too many paths in enum")
        }*/
        aud_init_resource_table();  /* allocation status */
        aud_init_set_paths();       /* interface status */
        aud_path_init();            /* initialise paths */
        aud_volume_init();          /* initialise volume settings */
//        aud_sleep_timer_init();     /* place after aud_path_init(), as timer is not ready to be used */
        aud_init_SM();              /* set all resource state machines to idle */

        /********* AUDIO 1 INITIALISATION **********/
        Aud_I2S_init();
#if defined (AUD_EXTERNAL_RINGER)
       InitRinger();
#endif
#if defined (FM_RADIO_PRESENT)
        AUD_fm_radio_init();
#endif
#if defined (INTERNAL_POLYRINGER)
       InitMidi();
        AUD_update_audio_post_processor();
#endif
#if defined (AUD_ENABLE_HAL)
        AudHalInit();
#endif
        /*[Begin] [lvwenhua-2012/4/11]*/
        #if 0
        AUD_ext_amplifier_init(); /* init for boomer "3D" amplifier */
        #endif
        /*[End] [lvwenhua-2012/4/11]*/
        /********** END AUDIO 1 INITIALISATION **********/
        /********** AUDIO 2 INITIALISATION **********/
//        Initialise_tone_vars();
#if defined (VMS_PRESENT)
        InitVmsRec();
        InitVmsPlayback();
#endif
#if defined (MP3_PRESENT)
        InitMp3();
#endif
#if defined (MP3_PRESENT) || defined (INTERNAL_POLYRINGER) || defined (PCM_PLAYER) || defined (VMS_PRESENT)
        aud_init_audio_irq();
#endif
//        InitToneIrq();
        /* region specific setup for supervisory tones */
//        AUD_Update_supervisory_tone_for_region(aud_EEP_static.aud_region); /* CEPT;ANSI or Japan from eep-static */
        /********** END AUDIO 2 INITIALISATION **********/
        /********** MISC INITIALISATION **********/
        aud_init_test_handles();
        //ANA_CTRL |= 0x00000103;
        aud_init_power_save(); /* power save init */
        /********** END MISC INITIALISATION **********/
    }/* end if audio_initialised==0 */
} /* End of function AUD_init */

/*------------------------------------------------------------------------------
* Function...: AudioDispatcher
* Return.....: -
* Description: Main state machine for the audio driver.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void AudioDispatcher(
    SDL_Pid sender, /* the process that sent the event to the audio driver process */
    U8 event,       /* the event to be handled */
    U16 caller_id,  /* not used */
    U32 parm1,      /* generic parameter */
    U32 parm2,      /* generic parameter */
    U32 parm3,      /* generic parameter */
    U32 parm4,      /* generic parameter */
    U32 parm5,      /* generic parameter */
    U32 parm6,      /* generic parameter */
    U32 parm7,      /* generic parameter */
    U32 parm8,      /* generic parameter */
    U8 *ptr1,       /* generic parameter */
    U8 *ptr2)       /* generic parameter */
{
    disp_to_SM_data_type disp_to_SM_data;

    /* Load the structure to be passed to the SM's */
    disp_to_SM_data.sender = sender;
    disp_to_SM_data.caller_id = caller_id;
    disp_to_SM_data.parm1 = parm1;
    disp_to_SM_data.parm2 = parm2;
    disp_to_SM_data.parm3 = parm3;
    disp_to_SM_data.parm4 = parm4;
    disp_to_SM_data.parm5 = parm5;
    disp_to_SM_data.parm6 = parm6;
    disp_to_SM_data.parm7 = parm7;
    disp_to_SM_data.parm8 = parm8;
    disp_to_SM_data.ptr1 = ptr1;
    disp_to_SM_data.ptr2 = ptr2;

#ifdef AUDIO_LLT
    aud_common_trace(llt_group_audio, llt_type_audio_path, 2, 0,                            /* LLT params */
                     'M', 0x01,                                                             /* LIS params */
                     aud_log_lis_display, aud_log_lis_display_data, aud_log_lis_llt_trace,  /* LIS params */
                     (U32) event,
                     (U32) E_audio_llt_type_id_0 );
#endif

#if defined (AUD_MODULE_TEST_IFWD)
    if(AUD_from_MMI_log_buf.LogLevel.Level.Enabled == 1)
    {
        AUD_from_MMI_log_buf.LogValues[AUD_from_MMI_log_buf.Cntr] = event;
        if(AUD_from_MMI_log_buf.Cntr < AUD_LOG_BUFSIZE)
        {
            AUD_from_MMI_log_buf.Cntr++;
        }
        if(AUD_from_MMI_log_buf.LogLevel.Level.Parameter1)
        {
            AUD_from_MMI_log_buf.LogValues[AUD_from_MMI_log_buf.Cntr] = parm1;
            if(AUD_from_MMI_log_buf.Cntr < AUD_LOG_BUFSIZE - 1)
            {
                AUD_from_MMI_log_buf.Cntr++;
            }
        }
        if(AUD_from_MMI_log_buf.LogLevel.Level.Parameter2)
        {
            AUD_from_MMI_log_buf.LogValues[AUD_from_MMI_log_buf.Cntr] = parm2;
            if(AUD_from_MMI_log_buf.Cntr < AUD_LOG_BUFSIZE - 1)
            {
                AUD_from_MMI_log_buf.Cntr++;
            }
        }
        if(AUD_from_MMI_log_buf.LogLevel.Level.Parameter3)
        {
            AUD_from_MMI_log_buf.LogValues[AUD_from_MMI_log_buf.Cntr] = parm3;
            if(AUD_from_MMI_log_buf.Cntr < AUD_LOG_BUFSIZE - 1)
            {
                AUD_from_MMI_log_buf.Cntr++;
            }
        }
        if(AUD_from_MMI_log_buf.LogLevel.Level.Parameter4)
        {
            AUD_from_MMI_log_buf.LogValues[AUD_from_MMI_log_buf.Cntr] = parm4;
            if(AUD_from_MMI_log_buf.Cntr < AUD_LOG_BUFSIZE - 1)
            {
                AUD_from_MMI_log_buf.Cntr++;
            }
        }
    }
#endif
   /* aud_main_db_event[aud_main_db_event_index].sender = sender;
    aud_main_db_event[aud_main_db_event_index].event  = (aud_event_enum)event;
    aud_main_db_event[aud_main_db_event_index].parm1  = parm1;
    aud_main_db_event[aud_main_db_event_index].parm2  = parm2;
    aud_main_db_event[aud_main_db_event_index].parm3  = parm3;
    aud_main_db_event[aud_main_db_event_index].parm4  = parm4;

    aud_main_db_event_index++;
    if(aud_main_db_event_index >= AUD_MAX_NOF_DB_INDEX-1)
    {
        aud_main_db_event_index = 0;
    }
*/
    switch(event)
    {
        case E_generic_dsp_cmd:
            /* write dsp cmd into the DSP cmd buffer */
            /* caller_id = location of dsp cmd in the at AT cmd (1..MAX_NOF_EXT_DSP_CMD) + DSP_EXT_CMD_ID_OFFSET */
            aud_add_dsp_cmd_to_buffer((U8) caller_id);
            break;

#if defined (AUD_3D_AMP_PRESENT)
        case E_3D_mode_enable:
            amp_3d_current_3d_mode = aud_enable;
            if( (amp_3d_current_output_mode_hs == aud_enable) ||
                (amp_3d_current_output_mode_spk == aud_enable))
                AUD_control_3D_amp(amp_3d_current_input_gain_05dB);
            break;
        case E_3D_mode_disable:
            amp_3d_current_3d_mode = aud_disable;
            if( (amp_3d_current_output_mode_hs == aud_enable) ||
                (amp_3d_current_output_mode_spk == aud_enable))
            {
                AUD_control_3D_amp(amp_3d_current_input_gain_05dB);
            }
            break;
#endif /* AUD_3D_AMP_PRESENT */

        case E_interface_request_error:
            break; /* dummy - do nothing */

        default: /* the event has not been handled -> pass it on to the appropriate state machine */
            if(aud_volume_dispatcher((aud_event_enum)event, &disp_to_SM_data)) break;
            if(aud_path_dispatcher((aud_event_enum)event, &disp_to_SM_data)) break;
            /* AUDIO2 event handling */
            if(aud_speech_SM((aud_event_enum)event, &disp_to_SM_data)) break;
#if defined (AUD_TONE_GENERATOR_PRESENT)			
            if(AudToneGeneratorSM((aud_event_enum)event, &disp_to_SM_data)) break;
#endif			
#if defined (AUD_EXTERNAL_RINGER)
            if(AudRingerSM((aud_event_enum)event, &disp_to_SM_data)) break;
#endif
#if defined (FM_RADIO_PRESENT)
            if(AudFMRadioSM((aud_event_enum)event, &disp_to_SM_data)) break;
#endif
#if defined (VMS_PRESENT )
            if(AudVmsRecSM((aud_event_enum)event, &disp_to_SM_data)) break;
            if(AudVmsPlaybackSM((aud_event_enum)event, &disp_to_SM_data)) break;
#endif
#if defined (MP3_PRESENT)
            if(AudMP3SM((aud_event_enum)event, &disp_to_SM_data)) break;
#endif
#if defined (INTERNAL_POLYRINGER)
            if(AudInternalMidiPlayerSM((aud_event_enum)event, &disp_to_SM_data)) break;
#endif



            if(aud_smpower_SM((aud_event_enum)event, &disp_to_SM_data)) break;
            if(aud_non_baseband_resource_SM((aud_event_enum)event, &disp_to_SM_data)) break;
#if defined (AUD_ENABLE_HAL)
            if(AudHalTimerEvent((aud_event_enum)event)) break;
#endif
            if(Aud_I2S_dispatcher((aud_event_enum)event, &disp_to_SM_data)) break;
            /* dispatcher did not catch signal - store this as exception */
            AUD_ERROR_REPORTING(aud_error_grade2, TRAP_AUD_ERROR, "Unhandled event: %d", event);
            break;
    }
} /* End of function AudioDispatcher */


SINT32 Aud_TaskInt_Init(VOID)
{
	SINT32 ret = DRV_SUCCESS;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"Aud_TaskInt_Init !\n");
	ret += AUD_create_task();
	if(ret != ZOSS_SUCCESS)
	{
		zOss_RamLog("Aud_TaskInt_Init: AUD_create_task Fail, ret=%d!", ret);
	
	}

		
#if defined _USE_TEAKAUD
	aud_nv_static_data_init();
#endif
#if 0
	ret += zDrv_VoiceMemoCreateThread();
	if(ret != ZOSS_SUCCESS)
	{
		zOss_RamLog("Aud_TaskInt_Init: zDrv_VoiceMemoCreateThread Fail, ret=%d!", ret);
	
	}
	ret += zDrv_MidiCreateThread();
	if(ret != ZOSS_SUCCESS)
	{
		zOss_RamLog("Aud_TaskInt_Init: zDrv_MidiCreateThread Fail, ret=%d!", ret);
	
	}	
	ret += zDrv_ToneCreateThread();
	if(ret != ZOSS_SUCCESS)
	{
		zOss_RamLog("Aud_TaskInt_Init: zDrv_ToneCreateThread Fail, ret=%d!", ret);
	
	}	
#endif
	return ret;
}

