/**
 *       Copyright (C) Danish Wireless Design A/S. All rights reserved.
 *
 * This document contains proprietary information belonging to Danish Wireless
 * Design A/S. Passing on and copying of this document, use and communication
 * of its contents is not permitted without prior written authorisation.
 *
 * Description:
 *   This source file contains the external interface related functions of the 
 *   UICC Handler.
 *
 * Revision Information:
 *   File name: \dwddrv\UICC\src\uicc_ci.c
 *   Version: \main\15
 *   Date: 2006-01-25 14:38:53
 *   Responsible: mtr
 *   Comment:
 *     Merge to main branch
 */

/******************************************************************************/
/*******************************              *********************************/
/*****************************                  *******************************/
/***************************        ASSUMING       ****************************/
/*************************        CI INTERFACE      ***************************/
/***************************  ~~~~~~~~~~~~~~~~~~  *****************************/
/*****************************                  *******************************/
/*******************************              *********************************/
/******************************************************************************/
#ifdef UICC_ASSUME_CI_INTERFACE

#define NUCLEUS_HISR

/*---------------------------------------------*/
/* Include files.                              */
/*---------------------------------------------*/
#include <string.h>
#include "uicc_stub.h"
#include "eep.h"

#include "uicc.h"
#include "uicc_trc.h"
#include "uicc_ctrl.h"
#include "uicc_access.h"
#include "uicc_ptest.h"


#define VOLTAGE_NOT_INITIALISED 0xFF
#define VOLTAGE_NOT_SUPPORTED   0xF0

extern NU_HISR SIM1_HISR_control;
extern NU_HISR SIM2_HISR_control;

T_UICC_APDU_FOOTER  res_apdu;
ubyte               nof_hist_characters;
ubyte               hist_characters[20];
ushort              sim_status;
sim_voltage_type    gsm_dir_indicated_voltage; 
sim_voltage_type    activation_voltage; 
ubyte               cmd_data[265];

/* Production test interface 'hold' variable. */
volatile ubyte UICC_ptst_sem;

sim_voltage_type conv_uicc_2_ci_voltage(T_UICC_VOLTAGE_CLASS uicc_voltage)
{
  sim_voltage_type ci_voltage;
  
  switch(uicc_voltage)
  {
    case UICC_CLASS_A: ci_voltage = sim_5v; break;     /* 5v   */
    case UICC_CLASS_B: ci_voltage = sim_3v; break;     /* 3v   */
    case UICC_CLASS_C: ci_voltage = sim_1_8v; break;   /* 1.8v */
    case UICC_CLASS_D: ci_voltage = (sim_voltage_type)(sim_1_8v+1); break; /* RFU  */
    case UICC_CLASS_E:                                 /* RFU  */
    default:           ci_voltage = (sim_voltage_type)(sim_1_8v+2); break; /* RFU  */
  }
  return(ci_voltage);
}

T_UICC_VOLTAGE_CLASS conv_ci_2_uicc_voltage(sim_voltage_type ci_voltage)
{
  T_UICC_VOLTAGE_CLASS uicc_voltage;

  switch(ci_voltage)
  {
    case sim_5v:   uicc_voltage = UICC_CLASS_A; break; /* 5v */
    case sim_3v:   uicc_voltage = UICC_CLASS_B; break; /* 3v */
    case sim_1_8v: 
    default:       uicc_voltage = UICC_CLASS_C; break; /* 1.8v */
  }
  return(uicc_voltage);
}

/*******************************************************************************
* CIInit
*******************************************************************************/
void CIInit(void)
{
  T_UICC_CHARACTERISTICS characteristics;
  T_UICC_VOLTAGE_CONTROL voltage_control;

  UICC_card_profile[UICC_current_reader].used_voltage = UICC_lower_supported_voltage_level();

  #ifdef ENABLE_RUNTIME_SIM_SIMULATION
    if(UICC_simul_state == uicc_simul_active)
    { /* Simulation active - disable simulation on this driver re-initialisation. */
      UICC_simul_state = uicc_simul_inactive;
    }
  #endif

  #ifdef UICC_LLT
    LLT_MULT_PARM_TRACE_1(llt_group_sim,
                          llt_type_sim_ciinit,
                          (sizeof(ulong)),
                          (ulong)UICC_card_profile[UICC_current_reader].used_voltage);
  #endif /*UICC_LLT*/

  gsm_dir_indicated_voltage = (sim_voltage_type)(VOLTAGE_NOT_INITIALISED);

  UICC_reset(UICC_PRIMARY_CARD,
             UICC_COLD_RESET,
             UICC_card_profile[UICC_current_reader].used_voltage,
             &voltage_control,
             &nof_hist_characters,
             &hist_characters,
             &characteristics);
}

/*******************************************************************************
* SIM_WriteCmd
*******************************************************************************/
SDL_Void SIM_WriteCmd(sim_cmdDir_type sim_cmdDir,
                      SDL_Integer     sim_lgth,
                      T_BYTE_PTR      sim_cmdPtr)
{
  T_UICC_COMMAND_CASE   command_case;
  T_UICC_APDU_HEADER    c_apdu;
  T_UICC_APDU_RESULT    result;

  #ifdef UICC_LLT
    LLT_MULT_PARM_TRACE_4(llt_group_sim,
                          llt_type_sim_writecmd,
                          (4*sizeof(ulong)),
                          (ulong)sim_cmdDir,
                          (ulong)sim_lgth,
                          (ulong)UICC_UBYTE2ULONG(sim_cmdPtr[0],sim_cmdPtr[1],sim_cmdPtr[2],sim_cmdPtr[3]),
                          (ulong)UICC_UBYTE2ULONG(sim_cmdPtr[4],sim_cmdPtr[5],sim_cmdPtr[6],sim_cmdPtr[7]));

  #endif /*UICC_LLT*/

  #ifdef ENABLE_RUNTIME_SIM_SIMULATION
    if(UICC_simul_state == uicc_simul_active)
    { /* Simulation active - let's simulate..... */
      UICC_apdu.payload_ptr = &cmd_data[0];
      msap_simulate_sim_apdu((unsigned int   )sim_lgth,
                             (unsigned char *)sim_cmdPtr,
                             &UICC_apdu.footer.luicc,
                             UICC_apdu.payload_ptr);
     
      UICC_apdu.footer.luicc -= 2; /* The Length returned in the simulation includes 
                                     SW1/SW2 - the internal count is without SW1/SW2. */
      UICC_apdu.footer.sw1 = UICC_apdu.payload_ptr[UICC_apdu.footer.luicc];
      UICC_apdu.footer.sw2 = UICC_apdu.payload_ptr[UICC_apdu.footer.luicc+1];

      /* Return the simulation result to SIMAP. */
      UICC_send_command_response_signal();
      return(0);
    }
  #endif

  /* Check the input data validity. */
  if((sim_cmdPtr) &&
     ((sim_cmdDir == sim_rx_cmd) || (sim_cmdDir == sim_tx_cmd)) &&
     (sim_lgth >= 5 ) && /* Pure command header (5). */
     (sim_lgth <= 260))  /* Command header and max command data (5+255). */
  { /* Input data valid */
 
    if(UICC_main_ctrl_state != UICC_READY)
    {
      /* Received request for executing an instruction while already in the
         progress of executing an instruction. Signal to SIMAP that there is
         something wrong!! */
      #ifndef UICC_NO_DEBUG
        UICC_log_data_ptr[0] = (ubyte) UICC_main_ctrl_state;
        UICC_log_data_ptr[1] = (ubyte) UICC_card_ctrl_state[UICC_PRIMARY_CARD];        
        UICC_log_data_ptr[2] = (ubyte) (sim_cmdDir >> 8);
        UICC_log_data_ptr[3] = (ubyte) sim_cmdDir;
        UICC_log_data_ptr[4] = (ubyte) (sim_lgth >> 8);
        UICC_log_data_ptr[5] = (ubyte) sim_lgth;
        UICC_raise_exception(TRAP_UICC_CI, UICC_WRITECMD_PARAM_ERROR, 6); 
      #endif

      UICC_send_status_signal(CCI_CI_BUSY);

      return(0);
    }

    /* First retrieve CLA, INS, P1 and P2 */
    memcpy(&c_apdu.cla, sim_cmdPtr, 4);
    c_apdu.lc = 0; /* Tx-size */
    c_apdu.le = 0; /* Rx-size */
  
    /* Find the command case type. */
    switch(c_apdu.ins)
    {
      /* CASE 1 commands (no Tx data nor Rx data): */
      /* ==========================================*/
      case 0x04:  /* INVALIDATE        */
      case 0x44:  /* REHABILITATE      */
      case 0xFA:  /* SLEEP             */
        command_case = UICC_CMD_CASE_1;
        break;

      /* CASE 2 commands (Only Rx data):           */
      /* ==========================================*/
      case 0xF2:  /* STATUS            */
      case 0xC0:  /* GET RESPONSE      */
      case 0xB0:  /* READ BINARY       */
      case 0xB2:  /* READ RECORD       */
      case 0x12:  /* FETCH             */
        command_case = UICC_CMD_CASE_2;

        if(sim_cmdPtr[4])
          c_apdu.le = sim_cmdPtr[4]; /* Rx-size fetched from P3 */        
        else
          c_apdu.le = 256; /* P3=0 means 256 bytes Rx-data. */                
        break;
 
      /* CASE 3 commands (Only Tx data):           */
      /* ==========================================*/
      case 0xA4:  /* SELECT            */
      case 0xD6:  /* UPDATE BINARY     */
      case 0xDC:  /* UPDATE RECORD     */
      case 0xA2:  /* SEEK              */
      case 0x32:  /* INCREASE          */
      case 0x20:  /* VERIFY CHV        */
      case 0x24:  /* CHANGE CHV        */
      case 0x26:  /* DISABLE CHV       */
      case 0x28:  /* ENABLE CHV        */
      case 0x2C:  /* UNBLOCK CHV       */
      case 0x88:  /* RUN GSM ALGORITHM */
      case 0x10:  /* TERMINAL PROFILE  */
      case 0xC2:  /* ENVELOPE          */
      case 0x14:  /* TERMINAL RESPONSE */
        command_case = UICC_CMD_CASE_3;
        c_apdu.lc = sim_cmdPtr[4]; /* Tx-size fetched from P3 */
        memcpy(&cmd_data[0], &sim_cmdPtr[5], c_apdu.lc);
        break;

      /* CASE 4 commands (Both Tx and Rx data):    */
      /* ==========================================*/
      /* NOT supported by the SIMAP-CI interface.  */

      default:
        command_case = UICC_CMD_CASE_1;
        #ifndef UICC_NO_DEBUG
          memcpy(&UICC_log_data_ptr[0], sim_cmdPtr, 8);
          UICC_raise_exception(TRAP_UICC_CI, UICC_WRITECMD_UNKNOWN_INS, 8); 
        #endif
        break;
    }

    result = UICC_transport_APDU(UICC_PRIMARY_CARD,
                                 command_case,
                                 FALSE /* no extended length */,
                                 c_apdu,
                                 &res_apdu,
                                 &cmd_data[0]);

    if(result != UICC_TRANSFER_SUCCEEDED)
    { /* Command execution failed. */
      #ifndef UICC_NO_DEBUG
        UICC_log_data_ptr[0] = result;
        UICC_log_data_ptr[1] = command_case;
        memcpy(&UICC_log_data_ptr[2], &c_apdu, 6);
        memcpy(&UICC_log_data_ptr[8], sim_cmdPtr, 8);
        UICC_raise_exception(TRAP_UICC_CI, UICC_WRITECMD_APDU_EXECUTION_ERROR, 16); 
      #endif
    }
  }
  else
  { /* Invalid input data. */
    #ifndef UICC_NO_DEBUG
      UICC_log_data_ptr[0] = (ubyte) (sim_cmdDir >> 8);
      UICC_log_data_ptr[1] = sim_cmdDir;
      UICC_log_data_ptr[2] = (ubyte) (sim_lgth >> 8);
      UICC_log_data_ptr[3] = sim_lgth;
      if(sim_cmdPtr)
        memcpy(&UICC_log_data_ptr[4], sim_cmdPtr, 8);
      UICC_raise_exception(TRAP_UICC_CI, UICC_WRITECMD_PARAM_ERROR, 12); 
    #endif

    /* Invalid data received from SIMAP. NO command will be carried out. */
    UICC_send_status_signal(CCI_CMD_ERROR);
  }

  return(0);
}

/*******************************************************************************
* SIM_SetCardPara
*******************************************************************************/
SDL_Boolean SIM_SetCardPara (sim_voltage_type      sim_voltage,
                             sim_clkStop_mode_type sim_clkStop_mode)
{
  T_UICC_CLOCK_STOP_MODE clock_stop_mode;

  #ifdef UICC_LLT
    LLT_MULT_PARM_TRACE_4(llt_group_sim,
                          llt_type_sim_setcardpara,
                          (4*sizeof(ulong)),
                          (ulong)sim_voltage,
                          (ulong)sim_clkStop_mode,
                          (ulong)UICC_card_profile[UICC_current_reader].voltage_settled,
                          (ulong)gsm_dir_indicated_voltage);
  #endif /*UICC_LLT*/

  #ifdef ENABLE_RUNTIME_SIM_SIMULATION
    if(UICC_simul_state == uicc_simul_active)
    { /* Simulation active - let's simulate..... */
      return(FALSE); /* Simulate: current operating voltage is in line with indicated voltage. */
    }
  #endif

  switch(sim_clkStop_mode)
  {
    case sim_stop_allowed:
      clock_stop_mode = UICC_CLOCK_STOP_ALLOWED;
      break;
    case sim_high_stop_only:
      clock_stop_mode = UICC_CLOCK_STOP_ONLY_HIGH;
      break;
    case sim_high_stop_pref:
      clock_stop_mode = UICC_HIGH_LEVEL_PREFERRED;
      break;
    case sim_low_stop_only:
      clock_stop_mode = UICC_CLOCK_STOP_ONLY_LOW;
      break;
    case sim_low_stop_pref:
      clock_stop_mode = UICC_LOW_LEVEL_PREFERRED;
      break;
    case sim_no_stop:
    default:
      clock_stop_mode = UICC_CLOCK_STOP_NOT_ALLOWED;
      break;
  }

  UICC_card_characteristics(UICC_PRIMARY_CARD, clock_stop_mode, 0x21);

  if(UICC_card_profile[UICC_current_reader].voltage_settled)
  { /* Voltage setteled (the right voltage found. */
    gsm_dir_indicated_voltage = conv_uicc_2_ci_voltage(UICC_card_profile[UICC_current_reader].used_voltage);
    return(FALSE);  
  }
  else if((sim_voltage == conv_uicc_2_ci_voltage(UICC_card_profile[UICC_current_reader].used_voltage)) || 
          ((sim_voltage == sim_1_8v) && (UICC_card_profile[UICC_current_reader].used_voltage == UICC_CLASS_B)))
       /* The GSM directory indicated voltage identical to the voltage used OR
          A 1.8v Technology card shall also support 3v (according to GSM 11.18 support) and if 
          the current activation voltage is 3v, a 1.8v technology SIM should be accepted
          without further notice. */
  {
    gsm_dir_indicated_voltage = sim_voltage;
    return(FALSE);  
  }
  else
  {  
    if((conv_ci_2_uicc_voltage(sim_voltage) <= UICC_lower_supported_voltage_level()) &&
       (conv_ci_2_uicc_voltage(sim_voltage) >= UICC_higher_supported_voltage_level()))
    {
      /* We are not using the right voltage level but the GSM directory indicated voltage lies 
         within the range of supported voltagelevels. */
      gsm_dir_indicated_voltage = sim_voltage;
    }
    else
    {
      /* We are not using the right voltage level and we cannot accomodate the GSM directory indicated 
         voltage. */
      gsm_dir_indicated_voltage = (sim_voltage_type)(VOLTAGE_NOT_SUPPORTED);
    }
    return(TRUE);    
  }
}


/*******************************************************************************
* SIM_ResetCard
*******************************************************************************/
SDL_Void SIM_ResetCard(SDL_Void x)
{
  T_UICC_CHARACTERISTICS characteristics;
  T_UICC_VOLTAGE_CONTROL voltage_control;

  #ifdef UICC_LLT
    LLT_MULT_PARM_TRACE_1(llt_group_sim,
                          llt_type_sim_resetcard,
                          (sizeof(ulong)),
                          (ulong)gsm_dir_indicated_voltage);
  #endif /*UICC_LLT*/

  x = x;

  #ifdef ENABLE_RUNTIME_SIM_SIMULATION
    if(UICC_simul_state == uicc_simul_active)
    { /* Simulation active - let's simulate..... */
      UICC_send_status_signal(CCI_SIM_INSERTED);
      return(0);    
    }
  #endif  

  if(gsm_dir_indicated_voltage == VOLTAGE_NOT_SUPPORTED)
  {
    /* It was not possible to meet the voltage requirements of the used card. */
    UICC_close(UICC_PRIMARY_CARD);      
    UICC_send_status_signal(CCI_SIM_REMOVED);
  }
  else if(gsm_dir_indicated_voltage == VOLTAGE_NOT_INITIALISED)
  {
    T_UICC_VOLTAGE_CLASS used_voltage;

    /* Check if another voltage level is available. */
    if(UICC_another_voltage_level_available(&used_voltage))
    {
      UICC_reset(UICC_PRIMARY_CARD,
                 UICC_COLD_RESET,
                 used_voltage,
                 &voltage_control,
                 &nof_hist_characters,
                 &hist_characters,
                 &characteristics);
    }
    else
    {
      /* It was not possible to meet the voltage requirements of the used card. */
      UICC_close(UICC_PRIMARY_CARD);      
      UICC_send_status_signal(CCI_SIM_REMOVED);
    }
  }
  else
  {
    /* Reset the card at the indicated voltage level. */
    UICC_reset(UICC_PRIMARY_CARD,
               UICC_COLD_RESET,
               conv_ci_2_uicc_voltage(gsm_dir_indicated_voltage),
               &voltage_control,
               &nof_hist_characters,
               &hist_characters,
               &characteristics);
  }


  return(0);
}


/*******************************************************************************
* SIM_DeactivateCard
*******************************************************************************/
SDL_Void SIM_DeactivateCard (SDL_Void x)
{
  T_UICC_CLOSE_RESULT result;

  x = x;

  #ifdef ENABLE_RUNTIME_SIM_SIMULATION
    if(UICC_simul_state == uicc_simul_active)
    { /* Simulation active - let's simulate..... */
      return(0);
    }
  #endif

  gsm_dir_indicated_voltage = conv_uicc_2_ci_voltage(UICC_lower_supported_voltage_level());  

  #ifdef UICC_LLT
    LLT_MULT_PARM_TRACE_1(llt_group_sim,
                          llt_type_sim_deactivatecard,
                          (sizeof(ulong)),
                          (ulong)gsm_dir_indicated_voltage);
  #endif /*UICC_LLT*/

  result = UICC_close(UICC_PRIMARY_CARD);

  if(result != UICC_DEACTIVATION_SUCCEEDED)
  { /* Activation failed. */
      #ifndef UICC_NO_DEBUG
        UICC_log_data_ptr[0] = result;
        UICC_raise_exception(TRAP_UICC_CI, UICC_DEACTIVATECARD_FAILED, 1); 
      #endif
  }

  return(0);
}


/*******************************************************************************
* CI_GetSIMStatus
*******************************************************************************/
ci_sim_status_type CI_GetSIMStatus(void)
{
  ci_sim_status_type status;

  switch(UICC_card_ctrl_state[UICC_PRIMARY_CARD])
  {
    case UICC_INACTIVE:
      status = ci_sim_not_operational;
      break;

    case UICC_IDLE:
    case UICC_BUSY:
      status = ci_sim_operational;
      break;
 
    default:
      status = ci_not_ready;
      break;
  }

  return(status);
}

/*******************************************************************************
* CI_EnterTestMode
*******************************************************************************/
SDL_Boolean CI_EnterTestMode(void)
{
  SDL_Boolean return_state = FALSE;

  if((UICC_card_ctrl_state[UICC_PRIMARY_CARD] == UICC_INACTIVE) ||
     (UICC_card_ctrl_state[UICC_PRIMARY_CARD] == UICC_CARD_NONE)||
     (UICC_card_ctrl_state[UICC_PRIMARY_CARD] == UICC_TESTMODE))
  {
    UICC_current_reader = UICC_PRIMARY_CARD; 
    UICC_testmode_state = UICC_CARD_NONE;
    UICC_card_ctrl(UICC_TESTMODE_REQUESTED);
    return_state = TRUE;
  }

  return(return_state);
}

/*******************************************************************************
* CI_get_sim_characteristics
*******************************************************************************/
ci_sim_characteristics_type CI_get_sim_characteristics(void)
{
  static unsigned char dummy_string[] = "No Support";
  ci_sim_characteristics_type characteristics = {&dummy_string[0], &dummy_string[0], &dummy_string[0]};

  return(characteristics);  
}

/*******************************************************************************
* UICC_send_status_signal
*******************************************************************************/
void UICC_send_status_signal(ushort status)
{
  sim_status = status;

  #ifdef UICC_MODULE_TEST
    if(UICC_module_test_state != UICC_MODULE_TEST_DISABLED)
    {
      UICC_simulated_signal.status.invoked++;
      UICC_simulated_signal.status.cause = status;
    }
    else
    {
  #endif /* UICC_MODULE_TEST */

  #ifdef UICC_LLT
    LLT_MULT_PARM_TRACE_1(llt_group_sim,
                          llt_type_sim_cci_status,
                          (sizeof(ulong)),
                          (ulong)status);
  #endif /*UICC_LLT*/


  NU_Activate_HISR(&SIM1_HISR_control);

  #ifdef UICC_MODULE_TEST
    }
  #endif /* UICC_MODULE_TEST */
}

/*******************************************************************************
* UICC_send_command_response_signal
*******************************************************************************/
void UICC_send_command_response_signal(void)
{
  #ifdef UICC_MODULE_TEST
    if(UICC_module_test_state != UICC_MODULE_TEST_DISABLED)
    {
      UICC_simulated_signal.response.invoked++;
      UICC_simulated_signal.response.size = UICC_apdu.footer.luicc;
      UICC_simulated_signal.response.sw1  = UICC_apdu.footer.sw1;
      UICC_simulated_signal.response.sw2  = UICC_apdu.footer.sw2;            
      //memcpy(UICC_simulated_signal.response.data[0], UICC_apdu.payload_ptr, UICC_apdu.footer.luicc);
    }
    else
    {
  #endif /* UICC_MODULE_TEST */

  #ifdef UICC_LLT
    LLT_MULT_PARM_TRACE_2(llt_group_sim,
                          llt_type_sim_rspwrite,
                          (2*sizeof(ulong)),
                          (ulong)UICC_UBYTE2ULONG(UICC_apdu.footer.sw1,UICC_apdu.footer.sw2,0,0),
                          (ulong)UICC_apdu.footer.luicc);
  #endif /*UICC_LLT*/

  NU_Activate_HISR(&SIM2_HISR_control);

  #ifdef UICC_MODULE_TEST
    }
  #endif /* UICC_MODULE_TEST */
}

/*******************************************************************************
* hisr_prepare_SICI_CCI_STATUS_CP
*******************************************************************************/
void hisr_prepare_SICI_CCI_STATUS_CP(void)
{
  DEF_OUTSIG_PTR

  #undef SDL_SELF
  #define SDL_SELF   0
  
  ALLOC_SIGNAL(SICI_CCI_STATUS,
               ySigN_SICI_CCI_STATUS,
               SDL_SELF,
               yPDef_SICI_CCI_STATUS)

  ((yPDP_SICI_CCI_STATUS)(OUTSIGNAL_DATA_PTR))->Param1 = (SDL_Integer)sim_status;

  SDL_2OUTPUT_NO_TO(xDefaultPrioSignal,
                    (xIdNode *)0,
                    SICI_CCI_STATUS,
                    ySigN_SICI_CCI_STATUS,
                    SDL_SELF,
                    sizeof(yPDef_SICI_CCI_STATUS),
                    "SICI_CCI_STATUS")
} /* end prepare_SICI_CCI_STATUS_CP */


/*******************************************************************************
* hisr_prepare_SICI_SIM_RSPWRITE_CP
*******************************************************************************/
void hisr_prepare_SICI_SIM_RSPWRITE_CP(void)
{
  DEF_OUTSIG_PTR

  #undef SDL_SELF
  #define SDL_SELF   0

  ALLOC_SIGNAL(SICI_SIM_RSPWRITE,
               ySigN_SICI_SIM_RSPWRITE,
               SDL_SELF,
               yPDef_SICI_SIM_RSPWRITE)
 
  UICC_apdu.payload_ptr[UICC_apdu.footer.luicc]   = UICC_apdu.footer.sw1;
  UICC_apdu.payload_ptr[UICC_apdu.footer.luicc+1] = UICC_apdu.footer.sw2;

  /* The total confirm length is equal to the Rx-length and the length of
     command result SW1 and SW2 (hence +2). */
  ((yPDP_SICI_SIM_RSPWRITE)(OUTSIGNAL_DATA_PTR))->Param1 = (SDL_Integer)UICC_apdu.footer.luicc + 2;
  ((yPDP_SICI_SIM_RSPWRITE)(OUTSIGNAL_DATA_PTR))->Param2 = (T_BYTE_PTR)UICC_apdu.payload_ptr;

  SDL_2OUTPUT_NO_TO(xDefaultPrioSignal,
                    (xIdNode *)0,
                    SICI_SIM_RSPWRITE,
                    ySigN_SICI_SIM_RSPWRITE,
                    SDL_SELF,
                    sizeof(yPDef_SICI_SIM_RSPWRITE),
                    "SICI_SIM_RSPWRITE")
}

#else



/*---------------------------------------------*/
/* Include files.                              */
/*---------------------------------------------*/
#include "uicc.h"
#include "uicc_ptest.h"

/* NOTE:
   The following two functions must be available outside the scope of ASSUME_CI_INTERFACE for 
   backwards compatibility with atcptest.c. These two functions should however eventually be 
   substituted with the new production test function SIM_ptst_verify_sim_connection. */
ci_sim_status_type CI_GetSIMStatus(void)
{
  return(ci_sim_not_operational);
}

#if 0
SDL_Boolean CI_EnterTestMode(void)
{
  return(FALSE);
}
#endif
#endif /* UICC_ASSUME_CI_INTERFACE */
