/*******************************************************************************
 * Copyright (C) 2007, ZTE Corporation.
 *
 * File Name:    
 * File Mark:    
 * Description:  This source file contains the T=0 transport handler statemachine engine.
 * Others:        
 * Version:       v0.1
 * Author:        mtr
 * Date:          2007-03-28
 * History 1:      
 *     Date: 
 *     Version:
 *     Author: 
 *     Modification:  
 * History 2: 
  ********************************************************************************/


/****************************************************************************
*                                              Include files
****************************************************************************/
#include "uicc_stub.h"
    
#include "uicc.h"
#include "uicc_ctrl.h"
#include "uicc_access.h"
#include "uicc_drv.h"
#include "uicc_trc.h"


/****************************************************************************
* 	                                           Local Macros
****************************************************************************/
//#define IFDS_CHAIN_TEST
#define IFSD_TRESHOLD                    10

#define MAX_RESYNC_CNT                   3
#define MAX_RETRANS_REQ_CNT              3
#define MAX_RETRANS_CNT                  3
#define MAX_BWT_TIMEOUT_CNT              3
#define MAX_ABORT_CNT                    5 

#define DEFAULT_NAD                      0x00
#define I_BLOCK_PCB_IDENTIFIER           0x00
#define R_BLOCK_PCB_IDENTIFIER           0x80
#define S_BLOCK_PCB_IDENTIFIER           0xC0
#define I_BLOCK_MASK                     0x80
#define SR_BLOCK_MASK                    0x40
#define SR_PCB_MASK                      0x3F

#define I_PCB_SEND_SEQUENCE_NO_BIT_POS   6 
#define I_PCB_MORE_DATA_BIT_POS          5 

#define R_PCB_REC_SEQUENCE_NO_BIT_POS    4 
#define EDC_OR_PARITY_ERROR_BIT_POS      0
#define OTHER_ERROR_BIT_POS              1
#define ERROR_FREE_MASK                  0x00
#define ERROR_FREE_MASK_1                0x10
#define EDC_OR_PARITY_ERROR_MASK         0x01    
#define EDC_OR_PARITY_ERROR_MASK_1       0x11    
#define OTHER_ERROR_MASK                 0x02            
#define OTHER_ERROR_MASK_1               0x12            

#define PCB_RESPONSE_BIT_POS             5
#define RESYNCH_REQ_PATTERN              0x00
#define RESYNCH_RESP_PATTERN             0x20
#define IFS_REQ_PATTERN                  0x01
#define IFS_RESP_PATTERN                 0x21
#define ABORT_REQ_PATTERN                0x02 
#define ABORT_RESP_PATTERN               0x22 
#define WTX_REQ_PATTERN                  0x03 
#define WTX_RESP_PATTERN                 0x23 
#define VPP_ERROR_PATTERN                0x24 


/****************************************************************************
* 	                                           Local Types
****************************************************************************/
typedef enum {
  send_next,
  send_previous,
  restart_transmission
} T_SEND_IND;

typedef enum {
  last_frame,
  more_frames,
  restart_reception_last_frame,
  restart_reception_more_frames  
} T_RECEIVE_IND;

typedef enum {  
  NO_ERROR,
  EDC_ERROR,
  PARITY_ERROR,
  OTHER_PCB_ERROR
} T_R_PCB_CAUSE;

typedef enum {  
  RESYNC_REQ,
  RESYNC_RESP,
  IFS_REQ,
  IFS_RESP,
  ABORT_REQ,
  ABORT_RESP,
  WTX_REQ,
  WTX_RESP,
  ERROR_ON_VPP_STATE
} T_S_PCB_CAUSE;

typedef enum {
  I_BLOCK_RECEIVED,
  S_RESYNC_REQ_RECEIVED,
  S_RESYNC_RESP_RECEIVED,  
  S_IFS_REQ_RECEIVED,
  S_IFS_RESP_RECEIVED,
  S_ABORT_REQ_RECEIVED,
  S_ABORT_RESP_RECEIVED,
  S_WTX_REQ_RECEIVED,
  S_WTX_RESP_RECEIVED,
  S_ERROR_ON_VPP_STATE_RECEIVED,
  R_ACKNOWLEDGE_RECEIVED,
  R_EDC_OR_PARITY_ERROR_RECEIVED,
  R_OTHER_ERROR_RECEIVED,
  UNKNOWN_NAD_CODING,
  UNKNOWN_PCB_CODING,
  INCORRECT_LEN_CODING,    
  EDC_ERROR_IN_RECEIVED_BLOCK
} T_RECEIVED_BLOCK_CAUSE;


/****************************************************************************
* 	                                           Local Constants
****************************************************************************/

/****************************************************************************
* 	                                           Local Variables
****************************************************************************/
static ubyte                 resync_cnt      = 0;
static ubyte                 bwt_timeout_cnt = 0;
static ubyte                 retrans_req_cnt = 0;
static ubyte                 retrans_cnt     = 0;
static ubyte                 abort_cnt       = 0;
static ubyte                 suggested_ifsd  = 0;
static ubyte                 s_block_dummy;
static ubyte                 last_block[5] = {0,0,0,0,0};
    
#ifndef UICC_NO_DEBUG
  static T_UICC_T1_LINK_CTRL_STATE debug = UICC_T1_LINK_NONE;
#endif


/****************************************************************************
* 	                                          Global Constants
****************************************************************************/

/****************************************************************************
* 	                                          Global Variables
****************************************************************************/
T_UICC_T1_TPDU_CTRL UICC_t1_tpdu_ctrl;
ubyte *UICC_prologue = &UICC_tpdu_data[0];
ubyte *UICC_inf      = &UICC_tpdu_data[3];

T_UICC_T1_LINK_CTRL_STATE  UICC_t1_link_ctrl_state  = UICC_T1_LINK_READY;
ulong                      UICC_link_ins_cnt = 0; 


  ubyte                 UICC_t1_s_seq_no; /* Send (Terminal) sequence number. */
  ubyte                 UICC_t1_r_seq_no; /* Receive (Card) I-block sequence number */


/****************************************************************************
* 	                                          Global Function Prototypes
****************************************************************************/

/****************************************************************************
* 	                                          Function Definitions
****************************************************************************/

/*******************************************************************************
* Function:... calc_lrc
* Description: Calculates the Error Detection Character LRC for the T=1 frame.
* Created:.... 27.09.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static ubyte calc_lrc(void)
{
  ubyte i, lrc;
  
  lrc  = UICC_prologue[UICC_NAD_POS];
  lrc ^= UICC_prologue[UICC_PCB_POS];
  lrc ^= UICC_prologue[UICC_LEN_POS];
  for (i=0; i<UICC_prologue[UICC_LEN_POS]; i++)
  {
    lrc ^= UICC_inf[i];    
  }
  return(lrc);  
}

/*******************************************************************************
* Function:... invert_seq_no
* Description: Inverts sequence number (Modulus 2)
* Created:.... 27.09.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static ubyte invert_seq_no(ubyte seq_no)
{
  if(seq_no)
    return(0);
  else
    return(1);
}

#ifdef IFDS_CHAIN_TEST
  ubyte ifsd_test = 9;
#endif
/*******************************************************************************
* Function:... ifsd_change_needed
* Description: Processes the recieved I-block data.
* Created:.... 02.10.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static BOOL ifsd_change_needed(ubyte *new_ifsd_value)
{
  *new_ifsd_value = 0;

#ifdef IFDS_CHAIN_TEST
  if(UICC_card_profile[UICC_current_reader].ifsd_used == ifsd_test)
    return(FALSE);
  else
  {
    UICC_card_profile[UICC_current_reader].ifsd_used = *new_ifsd_value = ifsd_test;

    if(ifsd_test != 10)
    {
      if (ifsd_test == 1)  
        ifsd_test = 10;       
      else
        ifsd_test--;
    }
 
    return(TRUE);      
  }

#else /* NORMAL USE */  
  if(UICC_card_profile[UICC_current_reader].ifsc_used != UICC_card_profile[UICC_current_reader].ifsd_used)
  {
    *new_ifsd_value = UICC_card_profile[UICC_current_reader].ifsc_used;    
  } /* ELSE: no change. */
    
  if(*new_ifsd_value)
  {
    UICC_card_profile[UICC_current_reader].ifsd_used = *new_ifsd_value;
    return(TRUE);      
  }
  else
    return(FALSE);      
#endif
}

/*******************************************************************************
* Function:... handle_le_lc
* Description: Fills in the Le or Lc (or non) from the APDU command header depending 
*              of the type of command involved.
* Created:.... 02.09.03 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static ubyte handle_le_lc(ubyte pos)
{
  ubyte return_val;

  switch(UICC_apdu.command_case)
  {
    case UICC_CMD_CASE_2: 
      UICC_inf[pos] = UICC_apdu.header.le; 
      return_val = 1;
      break;

    case UICC_CMD_CASE_3: 
    case UICC_CMD_CASE_4:
      UICC_inf[pos] = UICC_apdu.header.lc; 
      return_val = 1;
      break;

    default:
      return_val = 0;
      break;
  } 

  return(return_val);
}

/*******************************************************************************
* Function:... fill_in_complete_header
* Description: Fills in the data from the APDU command header depending of the
*              type of command involved.
* Created:.... 27.09.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static ubyte fill_in_complete_header(void)
{
  ubyte offset = 0;

  offset = handle_le_lc(4);
  UICC_inf[3] = UICC_apdu.header.p2;
  UICC_inf[2] = UICC_apdu.header.p1;
  UICC_inf[1] = UICC_apdu.header.ins;
  UICC_inf[0] = UICC_apdu.header.cla;
  offset += 4;

  return(offset);
}

/*******************************************************************************
* Function:... fill_in_header
* Description: Fills in the data from the APDU command header depending of the
*              available payload space.
* Created:.... 02.09.03 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static ubyte fill_in_header(void)
{ 
  ubyte offset = 0;

  /* Examine if the header has already been transmitted or if it should be part of this
     frame (the whole or parts of the header). */
  switch(UICC_t1_tpdu_ctrl.max_tx_payload_size - UICC_t1_tpdu_ctrl.remaining_tx_payload_cnt)
  {
    case 0: /* No header bytes has been sent yet. */
      switch(UICC_prologue[UICC_LEN_POS])
      {
        case 5: offset = handle_le_lc(4); /* Break removed by purpose. */
          case 4: UICC_inf[3] = UICC_apdu.header.p2;  offset++; /* Break removed by purpose. */
          case 3: UICC_inf[2] = UICC_apdu.header.p1;  offset++; /* Break removed by purpose. */
          case 2: UICC_inf[1] = UICC_apdu.header.ins; offset++; /* Break removed by purpose. */
          case 1: UICC_inf[0] = UICC_apdu.header.cla; offset++; 
          break;
        default:
          offset = fill_in_complete_header();
          break;
      }
      break;

    case 1: /* CLA header byte has been sent in previous frame. */
      switch(UICC_prologue[UICC_LEN_POS])
      {
        case 4: offset = handle_le_lc(3); /* Break removed by purpose. */
        case 3: UICC_inf[2] = UICC_apdu.header.p2;  offset++; /* Break removed by purpose. */
        case 2: UICC_inf[1] = UICC_apdu.header.p1;  offset++; /* Break removed by purpose. */
        case 1: UICC_inf[0] = UICC_apdu.header.ins; offset++; 
          break;
        default:
          offset = handle_le_lc(3) + 3;
          UICC_inf[2] = UICC_apdu.header.p2;
          UICC_inf[1] = UICC_apdu.header.p1;
          UICC_inf[0] = UICC_apdu.header.ins;
          break;
      }
      break;

    case 2: /* CLA & INS header bytes has been sent in previous frame(s). */
      switch(UICC_prologue[UICC_LEN_POS])
      {
        case 3: offset = handle_le_lc(2);  /* Break removed by purpose. */
        case 2: UICC_inf[1] = UICC_apdu.header.p2; offset++; /* Break removed by purpose. */
        case 1: UICC_inf[0] = UICC_apdu.header.p1; offset++;
          break;
        default:
          offset = handle_le_lc(2) + 2;
          UICC_inf[1] = UICC_apdu.header.p2; 
          UICC_inf[0] = UICC_apdu.header.p1; 
          break;
      }
      break;

    case 3: /* CLA, INS & P1 header bytes has been sent in previous frame(s). */
      switch(UICC_prologue[UICC_LEN_POS])
      {
        case 2: offset = handle_le_lc(1); /* Break removed by purpose. */
        case 1: UICC_inf[0] = UICC_apdu.header.p2; offset++;
          break;
        default:
          offset = handle_le_lc(1) + 1;
          UICC_inf[0] = UICC_apdu.header.p2; 
          break;
      }
      break;

    case 4: /* CLA, INS, P1 & P2 header bytes has been sent in previous frame(s). */
      switch(UICC_prologue[UICC_LEN_POS])
      {
        case 1: 
        default:
          offset = handle_le_lc(0);
          break;
      }
      break;

    default: /* The entire header has been sent in previous frame(s). */
      offset = 0;
  }

  return(offset);
}

/*******************************************************************************
* Function:... build_i_block
* Description: Build a Information (application data) block frame.
* Created:.... 27.09.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void build_i_block(T_SEND_IND send_ind)
{
  ubyte header_offset = 0;

  /* First update control data according to the scenario in which the frame is to be sent. */
  switch(send_ind)
  {
    case restart_transmission:
      /* Re-establish the original Tx pointer and counter to start all over in the I-block transmission. */
      UICC_t1_tpdu_ctrl.old_current_tx_payload_ptr   = UICC_t1_tpdu_ctrl.current_tx_payload_ptr   
                                                     = UICC_t1_tpdu_ctrl.org_tx_payload_ptr;

      UICC_t1_tpdu_ctrl.old_remaining_tx_payload_cnt = UICC_t1_tpdu_ctrl.remaining_tx_payload_cnt
                                                     = UICC_t1_tpdu_ctrl.max_tx_payload_size;
                                                     
      /* Re-establish the original Rx pointer and counter. */
      UICC_t1_tpdu_ctrl.rx_payload_ptr               = UICC_t1_tpdu_ctrl.org_rx_payload_ptr;
      UICC_t1_tpdu_ctrl.current_rx_payload_cnt       = 0;
      break;

    case send_next:
      /* Store the current pointer and counter to be able to resend (if needed be). */
      UICC_t1_tpdu_ctrl.old_current_tx_payload_ptr   = UICC_t1_tpdu_ctrl.current_tx_payload_ptr;
      UICC_t1_tpdu_ctrl.old_remaining_tx_payload_cnt = UICC_t1_tpdu_ctrl.remaining_tx_payload_cnt;
      break;
      
    case send_previous:
      /* Re-store the current pointer and counter in order to resend. */
      UICC_t1_tpdu_ctrl.current_tx_payload_ptr       = UICC_t1_tpdu_ctrl.old_current_tx_payload_ptr;
      UICC_t1_tpdu_ctrl.remaining_tx_payload_cnt     = UICC_t1_tpdu_ctrl.old_remaining_tx_payload_cnt;
      break;
  }  

  /* Now build the frame. */
  
  /*  NAD ~ Node ADdress byte:
      ========================
         |07|06|05|04|03|02|01|00|
         |  |  DAD   |  |  SAD   |      SAD ~ Source NAD  -  DAD ~ Destination NAD       
          |           |
          |           |________________ Vpp state control = 0 ---| 
          |____________________________ Vpp state control = 0 -----|  NOT USED (maintain pause state). */
  UICC_prologue[UICC_NAD_POS] = DEFAULT_NAD; 

  /*  PCB ~ Protocol Control Byte - I-BLOCK:
      ======================================
         |07|06|05|04|03|02|01|00|
         |  |  |  |    RFU       |
          |  |  |
          |  |  |______________________ M-bit - More-data bit
          |  |_________________________ N(S) - Send-sequence number
          |____________________________ 0 which indicates PCB belonging to an I-block*/
  UICC_prologue[UICC_PCB_POS] = I_BLOCK_PCB_IDENTIFIER; 

  /* Initialize the sequence number */
  if(UICC_t1_s_seq_no)
  {
    UICC_SETBIT(UICC_prologue[UICC_PCB_POS], I_PCB_SEND_SEQUENCE_NO_BIT_POS);
  }
  else 
  {
    UICC_RESETBIT(UICC_prologue[UICC_PCB_POS], I_PCB_SEND_SEQUENCE_NO_BIT_POS);
  }
  
  /* Find the length (LEN) and set the more data bit - if required. */
  if(UICC_t1_tpdu_ctrl.remaining_tx_payload_cnt <= UICC_card_profile[UICC_current_reader].ifsc_used)
  { /* Remaining data can fit into this frame - clear the more-bit flag. */
    UICC_prologue[UICC_LEN_POS] = UICC_t1_tpdu_ctrl.remaining_tx_payload_cnt;
    UICC_RESETBIT(UICC_prologue[UICC_PCB_POS], I_PCB_MORE_DATA_BIT_POS);
  }
  else
  { /* Remaining data canNOT fit into this frame, more frames are needed -> set the more-bit flag. */
    UICC_prologue[UICC_LEN_POS] = UICC_card_profile[UICC_current_reader].ifsc_used;
    UICC_SETBIT(UICC_prologue[UICC_PCB_POS], I_PCB_MORE_DATA_BIT_POS);
  }

  /* Now fill in the payload data but first check if APDU command header has been taken care of (in a 
     previous frame) or if the header should be part of this frame. */
  header_offset = fill_in_header(); 
  
  /* Handle the remaining payload data (if any) - that is if there is room for it. */
  if(UICC_prologue[UICC_LEN_POS] > header_offset)
  {
    ubyte copy_len = 0;
  
    if(UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], I_PCB_MORE_DATA_BIT_POS))
    {
      copy_len = UICC_prologue[UICC_LEN_POS] - header_offset;
    }
    else
    { /* This is the last frame of the instruction -> check if we need to add the Le byte 
         at the end of the payload data (in case of a Case 4 command). */
      if(UICC_apdu.command_case == UICC_CMD_CASE_4)
      {
        copy_len = UICC_prologue[UICC_LEN_POS] - header_offset - 1;
        UICC_inf[UICC_prologue[UICC_LEN_POS] - 1] = UICC_apdu.header.le;        
      }
      else
      {
        copy_len = UICC_prologue[UICC_LEN_POS] - header_offset;
      }
    } 

    /* Copy the actual application related data. */
    memcpy(&UICC_inf[header_offset], 
           UICC_t1_tpdu_ctrl.current_tx_payload_ptr, 
           copy_len);

    /* Update the Tx payload pointer (only needed when actual Tx data has been handled. */
    UICC_t1_tpdu_ctrl.current_tx_payload_ptr += (UICC_prologue[UICC_LEN_POS] - header_offset);
  }

  /* Update the control frame control data. */
  UICC_t1_tpdu_ctrl.remaining_tx_payload_cnt -= UICC_prologue[UICC_LEN_POS];

  /* Finally calculate the frame Error Correction Code base on the frame data content. */
  #ifdef CRC_ERROR_DETECTION_SUPPORTED
  if(UICC_card_profile[UICC_current_reader].edc)
  { /* CRC supported by card. */
    // CRC!!!!!!  UICC_prologue[UICC_PROLOGUE_POS] = calc_CRC(); 
  }
  else
  #endif
  { /* LRC supported by card. */
    UICC_inf[UICC_EPILOGUE_POS] = calc_lrc(); 
  }

}                   

/*******************************************************************************
* Function:... build_s_block
* Description: Build a Supervisory block frame.
* Created:.... 27.09.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void build_s_block(T_S_PCB_CAUSE cause, ubyte data)
{
  /*  NAD ~ Node ADdress byte:
      ========================
         |07|06|05|04|03|02|01|00|
         |  |  DAD   |  |  SAD   |      SAD ~ Source NAD  -  DAD ~ Destination NAD       
          |           |
          |           |________________ Vpp state control = 0 ---| 
          |____________________________ Vpp state control = 0 -----|  NOT USED (maintain pause state). */
  UICC_prologue[UICC_NAD_POS] = DEFAULT_NAD; 

  /*  PCB ~ Protocol Control Byte - R-BLOCK:
      ======================================
         |07|06|05|04|03|02|01|00|
         |     |                |
            |          |
            |          |__________________ 000000: RESYNC request
            |                              100001: RESYNC response
            |                              000001: IFS request
            |                              100001: IFS response    
            |                              000010: ABORT request
            |                              100010: ABORT response    
            |                              000011: WTX request
            |                              100011: WTX response    
            |                              100100: Error on VPP state (NOT USED)    
            |     
            |_____________________________ 11: indicates PCB belonging to an S-block*/

 UICC_prologue[UICC_PCB_POS] = S_BLOCK_PCB_IDENTIFIER; 

  switch(cause)
  {
    /*-------------------*/
    /* Resynchronization */
    /*-------------------*/
    case RESYNC_RESP: 
      UICC_SETBIT(UICC_prologue[UICC_PCB_POS], PCB_RESPONSE_BIT_POS);
      /* Break removed by purpose! */
    case RESYNC_REQ:
      UICC_prologue[UICC_PCB_POS] |= RESYNCH_REQ_PATTERN;
      UICC_prologue[UICC_LEN_POS] = 0;
      break;

    /*------------------------*/
    /* Information field size */
    /*------------------------*/
    case IFS_RESP:
      UICC_SETBIT(UICC_prologue[UICC_PCB_POS], PCB_RESPONSE_BIT_POS);
      /* Break removed by purpose! */
    case IFS_REQ:
      UICC_prologue[UICC_PCB_POS] |= IFS_REQ_PATTERN;
      UICC_inf[0] = data;
      UICC_prologue[UICC_LEN_POS] = 1;
      break;

    /*----------------*/
    /* Chaining abort */
    /*----------------*/
    case ABORT_RESP:
      UICC_SETBIT(UICC_prologue[UICC_PCB_POS], PCB_RESPONSE_BIT_POS);
      /* Break removed by purpose! */
    case ABORT_REQ:
      UICC_prologue[UICC_PCB_POS] |= ABORT_REQ_PATTERN;
      UICC_prologue[UICC_LEN_POS] = 0;
      break;

    /*--------------------------*/
    /* Waiting time negotiation */
    /*--------------------------*/
    case WTX_RESP:
      UICC_SETBIT(UICC_prologue[UICC_PCB_POS], PCB_RESPONSE_BIT_POS);    
      /* Break removed by purpose! */
    case WTX_REQ:
      UICC_prologue[UICC_PCB_POS] |= WTX_REQ_PATTERN;
      UICC_inf[0] = data;
      UICC_prologue[UICC_LEN_POS] = 1;
      break;

    /*-----------------*/
    /* Vpp state error */
    /*-----------------*/
    case ERROR_ON_VPP_STATE:
      UICC_SETBIT(UICC_prologue[UICC_PCB_POS], PCB_RESPONSE_BIT_POS);
      UICC_prologue[UICC_PCB_POS] |= VPP_ERROR_PATTERN;
      UICC_prologue[UICC_LEN_POS] = 0;
      break;
  }

  /* Calculate the frame Error Correction Code base on the frame data content. */
  if(UICC_card_profile[UICC_current_reader].edc)
  { /* CRC supported by card. */
    // CRC!!!!!!  UICC_prologue[UICC_PROLOGUE_POS] = calc_CRC(); 
  }
  else
  { /* LRC supported by card. */
    UICC_inf[UICC_EPILOGUE_POS] = calc_lrc();
  }
}

/*******************************************************************************
* Function:... build_r_block
* Description: Build a Receive Ready block frame.
* Created:.... 27.09.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void build_r_block(T_R_PCB_CAUSE cause, ubyte receive_seq_no)
{
  /*  NAD ~ Node ADdress byte:
      ========================
         |07|06|05|04|03|02|01|00|
         |  |  DAD   |  |  SAD   |      SAD ~ Source NAD  -  DAD ~ Destination NAD       
          |           |                 Must be set to 0
          |           |________________ Vpp state control = 0 ---| 
          |____________________________ Vpp state control = 0 -----|  NOT USED (maintain pause state). */
  UICC_prologue[UICC_NAD_POS] = DEFAULT_NAD; 

  /*  PCB ~ Protocol Control Byte - S-BLOCK:
      ======================================
         |07|06|05|04|03|02|01|00|
         |     | 0|              |
            |           |
            |           |_______________ 0,N(R),0,0,0,0: Error free
            |                            0,N(R),0,0,0,1: EDC or character parity error
            |                            0,N(R),0,0,1,0: Other errors
            |   
            |___________________________ 10: indicates PCB belonging to an R-block*/

  UICC_prologue[UICC_PCB_POS] = R_BLOCK_PCB_IDENTIFIER; 

  switch(cause)
  {
    /*------------*/
    /* Error free */
    /*------------*/
    case NO_ERROR:
      break;
      
    /*--------------------------*/
    /* EDC error / Parity error */
    /*--------------------------*/
    case EDC_ERROR:
    case PARITY_ERROR:
      UICC_SETBIT(UICC_prologue[UICC_PCB_POS], EDC_OR_PARITY_ERROR_BIT_POS); 
      break;
    
    /*----------------------*/
    /* Other error scenario */
    /*----------------------*/
    case OTHER_PCB_ERROR:
      UICC_SETBIT(UICC_prologue[UICC_PCB_POS], OTHER_ERROR_BIT_POS); 
      break;
  }

  if(receive_seq_no)
  {
    UICC_SETBIT(UICC_prologue[UICC_PCB_POS], R_PCB_REC_SEQUENCE_NO_BIT_POS); 
  }
  
  UICC_prologue[UICC_LEN_POS] = 0;
  
  /* Calculate the frame Error Correction Code base on the frame data content. */
  if(UICC_card_profile[UICC_current_reader].edc)
  { /* CRC supported by card. */
    // CRC!!!!!!  UICC_prologue[UICC_PROLOGUE_POS] = calc_CRC(); 
  }
  else
  { /* LRC supported by card. */
    UICC_inf[UICC_EPILOGUE_POS] = calc_lrc(); 
  }
}

/*******************************************************************************
* Function:... evaluate_received_block
* Description: Evaluates the recived frame which also includes error detection
* Created:.... 01.10.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static T_RECEIVED_BLOCK_CAUSE evaluate_received_block(void)
{
  T_RECEIVED_BLOCK_CAUSE block_cause;

  bwt_timeout_cnt = 0;
  
  /* Check the Error Detection Code (EDC). */
  if( (UICC_card_profile[UICC_current_reader].edc == 0) && (UICC_inf[UICC_EPILOGUE_POS] != calc_lrc()))
  /* ADD also CRC check - eventually! :   || (UICC_card_profile[UICC_current_reader].edc == 1) && (UICC_inf[UICC_EPILOGUE_POS] != calc_crc()) */
  { /* Error in Error Detection Code. */ 
    block_cause = EDC_ERROR_IN_RECEIVED_BLOCK;  
  }
  else 
  { /* Error Detection Code correct. */ 

    /* Check the length (LEN) validity */
    if((UICC_prologue[UICC_LEN_POS] > UICC_card_profile[UICC_current_reader].ifsd_used) ||
       (UICC_prologue[UICC_LEN_POS] == 0xFF))       
    { /* Received length is larger than the allowed size or length is 255 (not allowed value). */
      block_cause = INCORRECT_LEN_CODING;    
    }
    else
    { /* Received lengt is within limits. */

      /* Check the NAD format. */
      if(UICC_prologue[UICC_NAD_POS] != DEFAULT_NAD)
      { /* Incorrect NAD format. */
        block_cause = UNKNOWN_NAD_CODING;
      }
      else
      { /* Correct NAD format. */
        ubyte pcb = UICC_prologue[UICC_PCB_POS];
       
        /* Now look at the actual frame type. */
        if((pcb & I_BLOCK_MASK) == 0)
        { /* I-BLOCK ~ b7=0 */
          /*=========*/  
          block_cause = I_BLOCK_RECEIVED;
        }
        else 
        { /* b7 = 1 */
          if(pcb & SR_BLOCK_MASK)
          { /* S-BLOCK ~ b7=1 and b6=1 */
            /*=========*/  
            ubyte allowed_len;

            switch(pcb & SR_PCB_MASK)
            {
              case RESYNCH_REQ_PATTERN:
                allowed_len = 0; /* The length must be 0 */
                block_cause = S_RESYNC_REQ_RECEIVED;
                break;
              case RESYNCH_RESP_PATTERN:
                allowed_len = 0; /* The length must be 0 */
                block_cause = S_RESYNC_RESP_RECEIVED;
                break;
              case IFS_REQ_PATTERN:
                allowed_len = 1; /* The length must be 1 */
                block_cause = S_IFS_REQ_RECEIVED;
                break;
              case IFS_RESP_PATTERN:
                allowed_len = 1; /* The length must be 1 */
                block_cause = S_IFS_RESP_RECEIVED;
                break;
              case ABORT_REQ_PATTERN:
                allowed_len = 0; /* The length must be 0 */
                block_cause = S_ABORT_REQ_RECEIVED;
                break;
              case ABORT_RESP_PATTERN:
                allowed_len = 0; /* The length must be 0 */
                block_cause = S_ABORT_RESP_RECEIVED;
                break;
              case WTX_REQ_PATTERN:
                allowed_len = 1; /* The length must be 1 */
                block_cause = S_WTX_REQ_RECEIVED;
                break;
              case WTX_RESP_PATTERN:
                allowed_len = 1; /* The length must be 1 */
                block_cause = S_WTX_RESP_RECEIVED;
                break;
              case VPP_ERROR_PATTERN:
                allowed_len = 0; /* The length must be 0 */
                block_cause = S_ERROR_ON_VPP_STATE_RECEIVED;
                break;
              default:
                allowed_len = UICC_prologue[UICC_LEN_POS];
                block_cause = UNKNOWN_PCB_CODING;
                break;
            }

            if(UICC_prologue[UICC_LEN_POS] != allowed_len)
            { /* The length is not as expected. */
              block_cause = INCORRECT_LEN_CODING;                    
            }
          }
          else
          { /* R-BLOCK ~ b7=1 and b6=0 */
            /*=========*/  
            /* Information field shall be absent (The length must be 0) */
            if(UICC_prologue[UICC_LEN_POS] > 0) /* The length must be 0 */
            {
              block_cause = INCORRECT_LEN_CODING;                    
            }
            else
            {
              switch(pcb & SR_PCB_MASK)
              {
                case ERROR_FREE_MASK:
                case ERROR_FREE_MASK_1:
                  block_cause = R_ACKNOWLEDGE_RECEIVED;
                  break;
                case EDC_OR_PARITY_ERROR_MASK:
                case EDC_OR_PARITY_ERROR_MASK_1:
                  block_cause = R_EDC_OR_PARITY_ERROR_RECEIVED;
                  break;
                case OTHER_ERROR_MASK:
                case OTHER_ERROR_MASK_1:
                  block_cause = R_OTHER_ERROR_RECEIVED;
                  break;
                default:
                   block_cause = UNKNOWN_PCB_CODING;
                  break;            
              }
            }
          }    
        }
      }
    } 
  }  

  return(block_cause);
}

/*******************************************************************************
* Function:... handle_received_i_block
* Description: Processes the recieved I-block data.
* Created:.... 02.10.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void handle_received_i_block(T_RECEIVE_IND receive_ind)
{
  ushort frame_len = UICC_prologue[UICC_LEN_POS];
  ubyte  copy_len = 0;

  switch(receive_ind)
  {
    /*--------------------------------*/
    case restart_reception_more_frames:
    /*--------------------------------*/
      /* Re-establish the original pointer and counter. */
      UICC_t1_tpdu_ctrl.rx_payload_ptr     = UICC_t1_tpdu_ctrl.org_rx_payload_ptr;
      UICC_t1_tpdu_ctrl.current_rx_payload_cnt = 0;
      /* break; removed by purpose to be able to continue in the following.*/
    /*--------------------------------*/
    case more_frames:
    /*--------------------------------*/
      if(frame_len)
      {
        if(UICC_t1_tpdu_ctrl.current_rx_payload_cnt + frame_len > UICC_t1_tpdu_ctrl.max_rx_payload_size)
        { /* current + len > max */
          copy_len =  UICC_t1_tpdu_ctrl.max_rx_payload_size - UICC_t1_tpdu_ctrl.current_rx_payload_cnt - 2;
          /* NOTE: DATA TRUNCATED!!!! */ 
          /* The received data length is to high. Therefore will only the requested
             data be copied. The last two words of the received frame will at this
             stage be considered as SW1 and SW2 - only to be replaced should the 
             next frame contains the real status words. */
          UICC_apdu.footer.sw1 = UICC_inf[frame_len - 2];
          UICC_apdu.footer.sw2 = UICC_inf[frame_len - 1];          
        }
        else if((UICC_t1_tpdu_ctrl.current_rx_payload_cnt + frame_len) == UICC_t1_tpdu_ctrl.max_rx_payload_size)
        { /* current + len = max */
          copy_len = frame_len - 2;
          UICC_apdu.footer.sw1 = UICC_inf[frame_len - 2];
          UICC_apdu.footer.sw2 = UICC_inf[frame_len - 1];          
        }
        else if((UICC_t1_tpdu_ctrl.current_rx_payload_cnt + frame_len) == (UICC_t1_tpdu_ctrl.max_rx_payload_size - 1))
        { /* current + len = max - 1 */
          copy_len = frame_len - 1;
          UICC_apdu.footer.sw1 = UICC_inf[frame_len - 1];
        }
        else /*(UICC_t1_tpdu_ctrl.current_rx_payload_cnt + frame_len <= UICC_t1_tpdu_ctrl.max_rx_payload_size - 2)*/
        { /* current + len <= max - 2 */
          copy_len = frame_len;
        }
        /* Copy received data. */ 
        memcpy(UICC_t1_tpdu_ctrl.rx_payload_ptr, &UICC_inf[0], copy_len);
        UICC_apdu.footer.luicc = UICC_t1_tpdu_ctrl.current_rx_payload_cnt;

        /* Update control data. */
        UICC_t1_tpdu_ctrl.current_rx_payload_cnt += copy_len;
        UICC_t1_tpdu_ctrl.rx_payload_ptr += copy_len;
      }
      /* else -> No data to handle. */ 
    break;
  
    /*--------------------------------*/
    case restart_reception_last_frame:
    /*--------------------------------*/
      /* Re-establish the original pointer and counter. */
      UICC_t1_tpdu_ctrl.rx_payload_ptr     = UICC_t1_tpdu_ctrl.org_rx_payload_ptr;
      UICC_t1_tpdu_ctrl.current_rx_payload_cnt = 0;
      /* break; removed by purpose to be able to continue in the following.*/
    /*--------------------------------*/
    case last_frame:
    /*--------------------------------*/
      /* Last frame (more-bit NOT set). */
      switch(frame_len)
      {
        case 0:
          /* No data - used by the card to force an acknowledge for the previous 
             sent I-block. */
          /* Is the maximum length trust worthy in the sense that Le known 
             beforehand (i.e Le different from 0). */
          if(UICC_t1_tpdu_ctrl.current_rx_payload_cnt == (UICC_t1_tpdu_ctrl.max_rx_payload_size-2))
          { /* Le known -> SW1 & SW2 already collected. */
            UICC_apdu.footer.luicc = UICC_t1_tpdu_ctrl.current_rx_payload_cnt;
          }
          else
          { /* SW1 & SW2 not collected. */
            UICC_apdu.footer.sw1 = *(UICC_t1_tpdu_ctrl.rx_payload_ptr-2);
            UICC_apdu.footer.sw2 = *(UICC_t1_tpdu_ctrl.rx_payload_ptr-1);
            UICC_apdu.footer.luicc = UICC_t1_tpdu_ctrl.current_rx_payload_cnt-2;
          }
          break;
            
        case 1:
          /* One byte remaining = SW2. */
          UICC_apdu.footer.sw2 = UICC_inf[0];          
          /* Is the maximum length trust worthy in the sense that Le known 
             beforehand (i.e Le different from 0). */
          if(UICC_t1_tpdu_ctrl.current_rx_payload_cnt == (UICC_t1_tpdu_ctrl.max_rx_payload_size-2))
          { /* Le known -> SW1 already collected. */
            UICC_apdu.footer.luicc = UICC_t1_tpdu_ctrl.current_rx_payload_cnt;
          }
          else
          { /* SW1 & SW2 not collected. */
            UICC_apdu.footer.sw1 = *(UICC_t1_tpdu_ctrl.rx_payload_ptr-1);
            UICC_apdu.footer.luicc = UICC_t1_tpdu_ctrl.current_rx_payload_cnt-1;
          }
          break;
 
        case 2:
          UICC_apdu.footer.sw1 = UICC_inf[0];
          UICC_apdu.footer.sw2 = UICC_inf[1];          
          UICC_apdu.footer.luicc = UICC_t1_tpdu_ctrl.current_rx_payload_cnt;
          break;
 
        default:
          if((UICC_t1_tpdu_ctrl.current_rx_payload_cnt + frame_len) <= UICC_t1_tpdu_ctrl.max_rx_payload_size)
          {           
            copy_len = frame_len - 2;
          }
          else
          { /* Not Enough room for received data. */
            copy_len =  UICC_t1_tpdu_ctrl.max_rx_payload_size - UICC_t1_tpdu_ctrl.current_rx_payload_cnt - 2;
            /* NOTE: DATA TRUNCATED!!!! */ 
          }
 
          memcpy(UICC_t1_tpdu_ctrl.rx_payload_ptr, &UICC_inf[0], copy_len);
          UICC_t1_tpdu_ctrl.current_rx_payload_cnt += copy_len;
          UICC_t1_tpdu_ctrl.rx_payload_ptr += copy_len;
          UICC_apdu.footer.sw1 = UICC_inf[frame_len - 2];
          UICC_apdu.footer.sw2 = UICC_inf[frame_len - 1];          
          UICC_apdu.footer.luicc = UICC_t1_tpdu_ctrl.current_rx_payload_cnt;
          break;
      }    
      break;
  }
}

/*******************************************************************************
* Function:... initiate_block
* Description: 
* Created:.... 10.05.04 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void initiate_block(void)
{      
  /* Store the 'footprint' of the block - to be used for potential retransmission. */
  last_block[UICC_NAD_POS] = UICC_prologue[UICC_NAD_POS];
  last_block[UICC_PCB_POS] = UICC_prologue[UICC_PCB_POS];
  last_block[UICC_LEN_POS] = UICC_prologue[UICC_LEN_POS];
  last_block[UICC_INF_POS] = UICC_inf[0];
   	
  UICC_t1_block_handler(UICC_INITIATE_BLOCK); 
}

/*******************************************************************************
* Function:... fatal_error
* Description: 
* Created:.... 27.09.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void fatal_error(void)
{      
  UICC_reset_t1_block_handler();
  UICC_t1_link_ctrl_state = UICC_T1_LINK_READY;
  UICC_t1_transport_handler(UICC_T1_CMD_FAILED);                    
}

/*******************************************************************************
* Function:... initiate_resyncronization
* Description:  Initialize resynchronisation procedure.
* Created:.... 27.09.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void initiate_resyncronization(void)
{
  resync_cnt++;

  if(resync_cnt <= MAX_RESYNC_CNT)
  {
    UICC_t1_link_ctrl_state = UICC_SENDING_RESYNC_REQ;    
    build_s_block(RESYNC_REQ, s_block_dummy);
    initiate_block(); 
  }
  else
  {
    fatal_error();
  }
}

/*******************************************************************************
* Function:... request_retransmission
* Description:  Requests block retransmission from the card by sending an R-block.
* Created:.... 17.10.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void request_retransmission(T_R_PCB_CAUSE cause)
{
  retrans_req_cnt++;

  if(retrans_req_cnt < MAX_RETRANS_REQ_CNT)
  {
    build_r_block(cause, UICC_t1_r_seq_no);
    initiate_block(); 
  }
  else
  {
    initiate_resyncronization();         
  } 
}

/*******************************************************************************
* Function:... retransmit_block
* Description: Retransmit the last block.
* Created:.... 18.05.05 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void retransmit_block(void)
{
  retrans_cnt++;

  if(retrans_cnt <= MAX_RETRANS_CNT)
  {
    /* We should only get here if a retransmission of an S- or and R-blcok is needed (not with an I-block). */
    if(  ((last_block[UICC_PCB_POS] & I_BLOCK_MASK) != 0) && 
		 ((last_block[UICC_LEN_POS] == 0) || (last_block[UICC_LEN_POS] == 1)))
    { /* Retransmit last R- or S-block */
      ubyte lrc; 
	  
      lrc  = UICC_prologue[UICC_NAD_POS] = last_block[UICC_NAD_POS];
      lrc ^= UICC_prologue[UICC_PCB_POS] = last_block[UICC_PCB_POS];
      lrc ^= UICC_prologue[UICC_LEN_POS] = last_block[UICC_LEN_POS];
 
      if(UICC_prologue[UICC_LEN_POS])
      {
        lrc ^= UICC_inf[0] = last_block[UICC_INF_POS];
      }

	  UICC_inf[UICC_EPILOGUE_POS] = lrc;
      initiate_block();  
    }
	else
    {
      request_retransmission(OTHER_PCB_ERROR);                              
    } 
  }
  else
  {
    request_retransmission(OTHER_PCB_ERROR);
  } 
}

/*******************************************************************************
* Function:... bwt_time_out
* Description:  Requests block retransmission in case a BWT timeout has occured.
* Created:.... 01.09.03 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void bwt_time_out(void)
{
  bwt_timeout_cnt++; 

  if(bwt_timeout_cnt < MAX_BWT_TIMEOUT_CNT)
  {
    build_r_block(OTHER_PCB_ERROR, UICC_t1_r_seq_no);
    initiate_block(); 
  }
  else
  {
    if(UICC_link_ins_cnt == 1)
    { /* If the instruction counter holds the value 1 this means that we are at the "start 
         of the protocol" and no further operation is accepted but reset or deactivation. */
      fatal_error();    
    }
    else
    {  
      initiate_resyncronization();         
    }
  } 
}

/*******************************************************************************
* Function:... send_abort_response
* Description:  Requests abort response retransmission from the card by sending 
*               an R-block.
* Created:.... 21.08.03 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void send_abort_response(void)
{
  abort_cnt++;

  if(abort_cnt <= MAX_ABORT_CNT)
  {
    build_s_block(ABORT_RESP, s_block_dummy);
    initiate_block();               
  }
  else
  {
    initiate_resyncronization();           
  }
}


/*******************************************************************************
********************************************************************************
* Function:... UICC_reset_t1_data_link_handler
* Parameters:. -
* Returns:.... -
* Description: Reset the Data Link state machine and initalise the local variables.
* Created:.... 25.09.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************
*******************************************************************************/
void UICC_reset_t1_data_link_handler(void)
{
  /* Initialise variables. */
  UICC_t1_link_ctrl_state = UICC_T1_LINK_READY;
  UICC_link_ins_cnt       = 0; 
  resync_cnt              = 0;
  bwt_timeout_cnt         = 0;
  retrans_req_cnt         = 0;
  retrans_cnt             = 0;
  abort_cnt               = 0;
  suggested_ifsd          = 0;

  /* Initialise no-clear declared variables. */
  UICC_t1_s_seq_no        = 1;
  UICC_t1_r_seq_no        = 0;

  /* Initialise the block handler. */
  UICC_reset_t1_block_handler();
}


/*******************************************************************************
********************************************************************************
* Function:... UICC_t1_data_link_handler
* Parameters:. event: The event triggering the statemachine.
* Returns:.... -
* Description: This statemachine/engine is controlling the data link processing
*              of the T=1 protocol.
* Created:.... 25.09.02 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************
*******************************************************************************/
void UICC_t1_data_link_handler(T_UICC_T1_LINK_EVENTS event)
{
  switch(UICC_t1_link_ctrl_state)
  {
    /*=================================================*/
    /* Inactive - ready to receive instruction request */
    /*=================================================*/
    case UICC_T1_LINK_READY:
      switch(event)
      {
        /*----------------------------*/
        case UICC_INITIATE_LINK:
        /*----------------------------*/
          UICC_link_ins_cnt++;
          resync_cnt      = 0;
          retrans_req_cnt = 0;
          retrans_cnt     = 0;
          abort_cnt       = 0;
          UICC_t1_s_seq_no = invert_seq_no(UICC_t1_s_seq_no);
          UICC_t1_tpdu_ctrl.bwt_multiplier = 1;
          
          if(ifsd_change_needed(&suggested_ifsd))
          {
            UICC_t1_link_ctrl_state = UICC_SENDING_IFX_REQ;
            build_s_block(IFS_REQ, suggested_ifsd);
          }
          else
          {
            build_i_block(send_next);
            if(UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], I_PCB_MORE_DATA_BIT_POS))
            { /* Chaining of blocks needed. */
              UICC_t1_link_ctrl_state = UICC_SENDING_I_BLOCK_CHAIN;
            }
            else
            { /* Only ONE block needed. */
              UICC_t1_link_ctrl_state = UICC_SENDING_I_BLOCK;          
            }
          }
          
          initiate_block(); 
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          break;
      }
      break;

    /*=================================================*/
    /* Sending an IFX request.                         */
    /*=================================================*/
    case UICC_SENDING_IFX_REQ:
      switch(event)
      {
        /*----------------------------*/
        case UICC_BLOCK_RECEIVED:
        /*----------------------------*/
          switch(evaluate_received_block())
          {
            /* - - - - - - - - - - - - */
            case S_IFS_RESP_RECEIVED:     
            /* - - - - - - - - - - - - */
              UICC_link_ins_cnt++;

              if(UICC_inf[0] <= suggested_ifsd)
              { /* Correct IFSD response received. Now start the block transmission. */
                UICC_card_profile[UICC_current_reader].ifsd_used = suggested_ifsd;
                build_i_block(send_next);
                if(UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], I_PCB_MORE_DATA_BIT_POS))
                { /* Chaining of blocks needed. */
                  UICC_t1_link_ctrl_state = UICC_SENDING_I_BLOCK_CHAIN;
                }
                else
                { /* Only ONE block needed. */
                  UICC_t1_link_ctrl_state = UICC_SENDING_I_BLOCK;          
                }
                initiate_block(); 
              }
              else
              { /* The returned IFSD was NOT identical to the one sent be the ME. */ 
                request_retransmission(OTHER_PCB_ERROR);                              
              }
              break;
              
            /* - - - - - - - - - - - - */
            case I_BLOCK_RECEIVED:
            case R_ACKNOWLEDGE_RECEIVED:
            case R_EDC_OR_PARITY_ERROR_RECEIVED:
            case R_OTHER_ERROR_RECEIVED:
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);                                          
              break;

            /* - - - - - - - - - - - - */
            case S_IFS_REQ_RECEIVED:
            case S_WTX_RESP_RECEIVED:     
            case S_WTX_REQ_RECEIVED:
            case S_RESYNC_RESP_RECEIVED:
            case S_RESYNC_REQ_RECEIVED:   
            case S_ABORT_REQ_RECEIVED:    
            case S_ABORT_RESP_RECEIVED:   
            case S_ERROR_ON_VPP_STATE_RECEIVED:
            case UNKNOWN_NAD_CODING:
            case UNKNOWN_PCB_CODING:
            case INCORRECT_LEN_CODING:
              request_retransmission(OTHER_PCB_ERROR);                              
              break;

            /* - - - - - - - - - - - - */
            case EDC_ERROR_IN_RECEIVED_BLOCK:
            /* - - - - - - - - - - - - */
              request_retransmission(EDC_ERROR);                              
              break;
            
            /* - - - - - - - - - - - - */
            default:
            /* - - - - - - - - - - - - */
              #ifndef UICC_NO_DEBUG
                debug = UICC_t1_link_ctrl_state;
              #endif
              initiate_resyncronization();
              break;        
          }
          break;

        /*----------------------------*/
        case UICC_BWT_TIMEOUT:
        case UICC_CWT_TIMEOUT:
        /*----------------------------*/
          bwt_time_out();                              
          break;

        /*----------------------------*/
        case UICC_CHAR_PARITY_ERROR:
        /*----------------------------*/
          request_retransmission(PARITY_ERROR);                              
          break;

        /*----------------------------*/
        case UICC_LEN_VIOLATION:
        /*----------------------------*/
          request_retransmission(OTHER_PCB_ERROR);                              
          break;

        /*----------------------------*/
        case UICC_CHAR_TIMEOUT:
        case UICC_CHAR_OVERRUN_ERROR:
        /*----------------------------*/
          /* Either TX timeout error or UART congestion. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;        
      }
      break;

    /*=================================================*/
    /* In the process of sending one block.            */
    /*=================================================*/
    case UICC_SENDING_I_BLOCK:
      switch(event)
      {
        /*----------------------------*/
        case UICC_BLOCK_RECEIVED:
        /*----------------------------*/
          switch(evaluate_received_block())
          {
            /* - - - - - - - - - - - - */
            case I_BLOCK_RECEIVED:
            /* - - - - - - - - - - - - */
              if(UICC_t1_r_seq_no == UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], I_PCB_SEND_SEQUENCE_NO_BIT_POS))
              {
                UICC_t1_r_seq_no = invert_seq_no(UICC_t1_r_seq_no); /* The next number to expect from the card. */
                retrans_req_cnt = 0;
                retrans_cnt     = 0;
              
                if(UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], I_PCB_MORE_DATA_BIT_POS))
                { /* More bit set. */
                  handle_received_i_block(more_frames);                
                  UICC_t1_link_ctrl_state = UICC_RECEIVING_I_BLOCK;
                  /* Send acknowledge. */
                  build_r_block(NO_ERROR, UICC_t1_r_seq_no);                  
                  initiate_block(); 
                }
                else
                { /* More bit NOT set. */
                  handle_received_i_block(last_frame);                                
                  UICC_t1_link_ctrl_state = UICC_LINK_SUCCEEDED;
                  UICC_t1_block_handler(UICC_STOP_BLOCK_CLOCK);
                }
              }
              else
              {
                request_retransmission(OTHER_PCB_ERROR);                                            
              }
              break;
              
            /* - - - - - - - - - - - - */
            case R_ACKNOWLEDGE_RECEIVED:
            case R_EDC_OR_PARITY_ERROR_RECEIVED:
            case R_OTHER_ERROR_RECEIVED:
            /* - - - - - - - - - - - - */
              if(UICC_t1_s_seq_no == UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], R_PCB_REC_SEQUENCE_NO_BIT_POS))
              { /* Resend the frame and stay in this state. */
                build_i_block(send_previous);
                initiate_block(); 
              }
              else
              { /* Error, because there is no next frame (there is no chain to continue). */
                retransmit_block();
              }
              break;

            /* - - - - - - - - - - - - */
            case S_IFS_REQ_RECEIVED:
            /* - - - - - - - - - - - - */
              /* Accept the resuested data size and use the value from now on. */
              if(UICC_inf[0])
              {
                UICC_card_profile[UICC_current_reader].ifsc_used = UICC_inf[0];
              }
              build_s_block(IFS_RESP,  UICC_card_profile[UICC_current_reader].ifsc_used);
              initiate_block(); 
              break;

            /* - - - - - - - - - - - - */
            case S_WTX_REQ_RECEIVED:
            /* - - - - - - - - - - - - */
              /* Accept the BWT extension (one shot extension). */
              if(UICC_inf[0])
              {
                UICC_t1_tpdu_ctrl.bwt_multiplier = UICC_inf[0];
              }
              build_s_block(WTX_RESP, UICC_t1_tpdu_ctrl.bwt_multiplier);
              initiate_block(); 
              break;

            /* - - - - - - - - - - - - */
            case S_RESYNC_REQ_RECEIVED:         /* Must NEVER be sent by the card. */
            case S_ABORT_REQ_RECEIVED:          /* Cannot happen because no chaining is going on. */
            case S_RESYNC_RESP_RECEIVED:        /*  Cannot happen in this state. */
            case S_IFS_RESP_RECEIVED:           /*  Cannot happen in this state. */
            case S_ABORT_RESP_RECEIVED:         /*  Cannot happen in this state. */            
            case S_WTX_RESP_RECEIVED:           /*  Cannot happen in this state. */
            case S_ERROR_ON_VPP_STATE_RECEIVED: /*  Must never occur! */
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);
              break;
              
            /* - - - - - - - - - - - - */
            case UNKNOWN_NAD_CODING:
            case UNKNOWN_PCB_CODING:
            case INCORRECT_LEN_CODING:
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);
              break;

            /* - - - - - - - - - - - - */
            case EDC_ERROR_IN_RECEIVED_BLOCK:
            /* - - - - - - - - - - - - */
              request_retransmission(EDC_ERROR);
              break;
            
            /* - - - - - - - - - - - - */
            default:
            /* - - - - - - - - - - - - */
              #ifndef UICC_NO_DEBUG
                debug = UICC_t1_link_ctrl_state;
              #endif
              initiate_resyncronization();
              break;        
          }
          break;
       
        /*----------------------------*/
        case UICC_BWT_TIMEOUT:
        case UICC_CWT_TIMEOUT:
        /*----------------------------*/
          bwt_time_out();                              
          break;

        /*----------------------------*/
        case UICC_CHAR_PARITY_ERROR:
        /*----------------------------*/
          request_retransmission(PARITY_ERROR);
          break;

        /*----------------------------*/
        case UICC_LEN_VIOLATION:
        /*----------------------------*/
          request_retransmission(OTHER_PCB_ERROR);                              
          break;

        /*----------------------------*/
        case UICC_CHAR_TIMEOUT:
        case UICC_CHAR_OVERRUN_ERROR:
        /*----------------------------*/
          /* Either TX timeout error or UART congestion. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;        
      }
      break;

    /*=================================================*/
    /* Sending I-Block chain                           */
    /*=================================================*/
    case UICC_SENDING_I_BLOCK_CHAIN:  
      switch(event)
      {
        /*----------------------------*/
        case UICC_BLOCK_RECEIVED:
        /*----------------------------*/
          switch(evaluate_received_block())
          {
            /* - - - - - - - - - - - - */
            case I_BLOCK_RECEIVED:
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);
              break;

            /* - - - - - - - - - - - - */
            case R_ACKNOWLEDGE_RECEIVED:
            case R_EDC_OR_PARITY_ERROR_RECEIVED:
            case R_OTHER_ERROR_RECEIVED:
            /* - - - - - - - - - - - - */
              retrans_req_cnt = 0;
              retrans_cnt     = 0;
              if(UICC_t1_s_seq_no == UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], R_PCB_REC_SEQUENCE_NO_BIT_POS))
              { /* Resend the frame and stay in this state. */
                build_i_block(send_previous);
                initiate_block(); 
              }
              else
              { /* Send the next frame and stay in this state. */
                UICC_t1_s_seq_no = invert_seq_no(UICC_t1_s_seq_no);
                build_i_block(send_next);

                /* Check if this is the last block of the chain. */
                if(!UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], I_PCB_MORE_DATA_BIT_POS))
                { /* Last block in the chain -> change state. */
                  UICC_t1_link_ctrl_state = UICC_SENDING_I_BLOCK_CHAIN_LAST;          
                } /* Else: stay in this chained-state. */
                initiate_block(); 
              }
              break;

            /* - - - - - - - - - - - - */
            case S_IFS_REQ_RECEIVED:
            /* - - - - - - - - - - - - */
              /* Accept the resuested data size and use the value from now on. */
              if(UICC_inf[0])
              {
                UICC_card_profile[UICC_current_reader].ifsc_used = UICC_inf[0];
              }
              build_s_block(IFS_RESP,  UICC_card_profile[UICC_current_reader].ifsc_used);
              initiate_block(); 
              break;

            /* - - - - - - - - - - - - */
            case S_WTX_REQ_RECEIVED:
            /* - - - - - - - - - - - - */
              /* Accept the BWT extension (one shot extension). */
              if(UICC_inf[0])
              {
                UICC_t1_tpdu_ctrl.bwt_multiplier = UICC_inf[0];
              }
              build_s_block(WTX_RESP, UICC_t1_tpdu_ctrl.bwt_multiplier);
              initiate_block(); 
              break;

            /* - - - - - - - - - - - - */
            case S_ABORT_REQ_RECEIVED: 
            /* - - - - - - - - - - - - */
              UICC_t1_link_ctrl_state = UICC_ABORT_SENDING_CHAIN;          
              send_abort_response();
              break;

            /* - - - - - - - - - - - - */
            case S_RESYNC_REQ_RECEIVED:         /* Must NEVER be sent by the card. */
            case S_RESYNC_RESP_RECEIVED:        /*  Cannot happen in this state. */
            case S_IFS_RESP_RECEIVED:           /*  Cannot happen in this state. */
            case S_ABORT_RESP_RECEIVED:         /*  Cannot happen in this state. */            
            case S_WTX_RESP_RECEIVED:           /*  Cannot happen in this state. */
            case S_ERROR_ON_VPP_STATE_RECEIVED: /*  Must never occur! */
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);
              break;

            /* - - - - - - - - - - - - */
            case UNKNOWN_NAD_CODING:
            case UNKNOWN_PCB_CODING:
            case INCORRECT_LEN_CODING:
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);
              break;

            /* - - - - - - - - - - - - */
            case EDC_ERROR_IN_RECEIVED_BLOCK:
            /* - - - - - - - - - - - - */
              request_retransmission(EDC_ERROR);
              break;

            /* - - - - - - - - - - - - */
            default:
            /* - - - - - - - - - - - - */
              #ifndef UICC_NO_DEBUG
                debug = UICC_t1_link_ctrl_state;
              #endif
              initiate_resyncronization();
              break;        
          }
          break;
       
        /*----------------------------*/
        case UICC_BWT_TIMEOUT:
        case UICC_CWT_TIMEOUT:
        /*----------------------------*/
          bwt_time_out();                              
          break;

        /*----------------------------*/
        case UICC_CHAR_PARITY_ERROR:
        /*----------------------------*/
          request_retransmission(PARITY_ERROR);
          break;

        /*----------------------------*/
        case UICC_LEN_VIOLATION:
        /*----------------------------*/
          request_retransmission(OTHER_PCB_ERROR);                              
          break;

        /*----------------------------*/
        case UICC_CHAR_TIMEOUT:
        case UICC_CHAR_OVERRUN_ERROR:
        /*----------------------------*/
          /* Either TX timeout error or UART congestion. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;        
      }
      break;

    /*=================================================*/
    /* Sending the last block in a chain.              */
    /*=================================================*/
    case UICC_SENDING_I_BLOCK_CHAIN_LAST:
      switch(event)
      {
        /*----------------------------*/
        case UICC_BLOCK_RECEIVED:
        /*----------------------------*/
          switch(evaluate_received_block())
          {
            /* - - - - - - - - - - - - */
            case I_BLOCK_RECEIVED:
            /* - - - - - - - - - - - - */
              if(UICC_t1_r_seq_no == UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], I_PCB_SEND_SEQUENCE_NO_BIT_POS))
              {
                UICC_t1_r_seq_no = invert_seq_no(UICC_t1_r_seq_no); /* The next number to expect from the card. */
                retrans_req_cnt = 0;
                retrans_cnt     = 0;
              
                if(UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], I_PCB_MORE_DATA_BIT_POS))
                { /* More bit set. */
                  handle_received_i_block(more_frames);                
                  UICC_t1_link_ctrl_state = UICC_RECEIVING_I_BLOCK;
                  /* Send acknowledge. */
                  build_r_block(NO_ERROR, UICC_t1_r_seq_no);                  
                  initiate_block(); 
                }
                else
                { /* More bit NOT set. */
                  handle_received_i_block(last_frame);                                
                  UICC_t1_link_ctrl_state = UICC_LINK_SUCCEEDED;
                  UICC_t1_block_handler(UICC_STOP_BLOCK_CLOCK);
                }
              }
              else
              {
                request_retransmission(OTHER_PCB_ERROR);                                            
              }
              break;
              
            /* - - - - - - - - - - - - */
            case R_ACKNOWLEDGE_RECEIVED:
            case R_EDC_OR_PARITY_ERROR_RECEIVED:
            case R_OTHER_ERROR_RECEIVED:
            /* - - - - - - - - - - - - */
              if(UICC_t1_s_seq_no == UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], R_PCB_REC_SEQUENCE_NO_BIT_POS))
              { /* Resend the frame and stay in this state. */
                build_i_block(send_previous);
                initiate_block(); 
              }
              else
              { /* Error, because there is no next frame (there is no chain to continue). */
                retransmit_block();
              }
              break;

            /* - - - - - - - - - - - - */
            case S_IFS_REQ_RECEIVED:
            /* - - - - - - - - - - - - */
              /* Accept the resuested data size and use the value from now on. */
              if(UICC_inf[0])
              {
                UICC_card_profile[UICC_current_reader].ifsc_used = UICC_inf[0];
              }
              build_s_block(IFS_RESP,  UICC_card_profile[UICC_current_reader].ifsc_used);
              initiate_block(); 
              break;

            /* - - - - - - - - - - - - */
            case S_WTX_REQ_RECEIVED:
            /* - - - - - - - - - - - - */
              /* Accept the BWT extension (one shot extension). */
              if(UICC_inf[0])
              {
                UICC_t1_tpdu_ctrl.bwt_multiplier = UICC_inf[0];
              }
              build_s_block(WTX_RESP, UICC_t1_tpdu_ctrl.bwt_multiplier);
              initiate_block(); 
              break;

            /* - - - - - - - - - - - - */
            case S_ABORT_REQ_RECEIVED:          
            /* - - - - - - - - - - - - */
              /* This is allowed since this is the last of a chained block sequence. */ 
              UICC_t1_link_ctrl_state = UICC_ABORT_SENDING_CHAIN;          
              send_abort_response();
              break;

            /* - - - - - - - - - - - - */
            case S_RESYNC_REQ_RECEIVED:         /* Must NEVER be sent by the card. */
            case S_RESYNC_RESP_RECEIVED:        /*  Cannot happen in this state. */
            case S_IFS_RESP_RECEIVED:           /*  Cannot happen in this state. */
            case S_ABORT_RESP_RECEIVED:         /*  Cannot happen in this state. */            
            case S_WTX_RESP_RECEIVED:           /*  Cannot happen in this state. */
            case S_ERROR_ON_VPP_STATE_RECEIVED: /*  Must never occur! */
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);
              break;
              
            /* - - - - - - - - - - - - */
            case UNKNOWN_NAD_CODING:
            case UNKNOWN_PCB_CODING:
            case INCORRECT_LEN_CODING:
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);
              break;

            /* - - - - - - - - - - - - */
            case EDC_ERROR_IN_RECEIVED_BLOCK:
            /* - - - - - - - - - - - - */
              request_retransmission(EDC_ERROR);
              break;
            
            /* - - - - - - - - - - - - */
            default:
            /* - - - - - - - - - - - - */
              #ifndef UICC_NO_DEBUG
                debug = UICC_t1_link_ctrl_state;
              #endif
              initiate_resyncronization();
              break;        
          }
          break;
       
        /*----------------------------*/
        case UICC_BWT_TIMEOUT:
        case UICC_CWT_TIMEOUT:
        /*----------------------------*/
          bwt_time_out();                              
          break;

        /*----------------------------*/
        case UICC_CHAR_PARITY_ERROR:
        /*----------------------------*/
          request_retransmission(PARITY_ERROR);
          break;

        /*----------------------------*/
        case UICC_LEN_VIOLATION:
        /*----------------------------*/
          request_retransmission(OTHER_PCB_ERROR);                              
          break;

        /*----------------------------*/
        case UICC_CHAR_TIMEOUT:
        case UICC_CHAR_OVERRUN_ERROR:
        /*----------------------------*/
          /* Either TX timeout error or UART congestion. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;        
      }
      break;

    /*=================================================*/
    /* Receiving I-Block                               */
    /*=================================================*/
    case UICC_RECEIVING_I_BLOCK:  
      switch(event)
      {
        /*----------------------------*/
        case UICC_BLOCK_RECEIVED:
        /*----------------------------*/
          switch(evaluate_received_block())
          {
            /* - - - - - - - - - - - - */
            case I_BLOCK_RECEIVED:
            /* - - - - - - - - - - - - */
              if(UICC_t1_r_seq_no == UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], I_PCB_SEND_SEQUENCE_NO_BIT_POS))
              {
                UICC_t1_r_seq_no = invert_seq_no(UICC_t1_r_seq_no); /* The next number to expect from the card. */
                retrans_req_cnt = 0;
                retrans_cnt     = 0;
              
                if(UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], I_PCB_MORE_DATA_BIT_POS))
                { /* More bit set. */
                  handle_received_i_block(more_frames);                
                  /* Send acknowledge. */
                  build_r_block(NO_ERROR, UICC_t1_r_seq_no);                  
                  initiate_block(); 
                }
                else
                { /* More bit NOT set. */
                  handle_received_i_block(last_frame);                                
                  UICC_t1_link_ctrl_state = UICC_LINK_SUCCEEDED;
                  UICC_t1_block_handler(UICC_STOP_BLOCK_CLOCK);
                }
              }
              else
              {
                request_retransmission(OTHER_PCB_ERROR);                                            
              }
              break;

            /* - - - - - - - - - - - - */
            case R_ACKNOWLEDGE_RECEIVED:
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR); 
              break;

            /* - - - - - - - - - - - - */
            case R_EDC_OR_PARITY_ERROR_RECEIVED:
            case R_OTHER_ERROR_RECEIVED:
            /* - - - - - - - - - - - - */
              retransmit_block();
              break;

            /* - - - - - - - - - - - - */
            case S_IFS_REQ_RECEIVED:
            /* - - - - - - - - - - - - */
              /* Accept the resuested data size and use the value from now on. */
              if(UICC_inf[0])
              {
                UICC_card_profile[UICC_current_reader].ifsc_used = UICC_inf[0];
              }
              build_s_block(IFS_RESP,  UICC_card_profile[UICC_current_reader].ifsc_used);
              initiate_block(); 
              break;

            /* - - - - - - - - - - - - */
            case S_WTX_REQ_RECEIVED:
            /* - - - - - - - - - - - - */
              /* Accept the BWT extension (one shot extension). */
              if(UICC_inf[0])
              {
                UICC_t1_tpdu_ctrl.bwt_multiplier = UICC_inf[0];
              }
              build_s_block(WTX_RESP, UICC_t1_tpdu_ctrl.bwt_multiplier);
              initiate_block(); 
              break;

            /* - - - - - - - - - - - - */
            case S_ABORT_REQ_RECEIVED: 
            /* - - - - - - - - - - - - */
              UICC_t1_link_ctrl_state = UICC_ABORT_RECEIVING_CHAIN;          
              send_abort_response();
              break;

            /* - - - - - - - - - - - - */
            case S_RESYNC_REQ_RECEIVED:         /* Must NEVER be sent by the card. */
            case S_RESYNC_RESP_RECEIVED:        /*  Cannot happen in this state. */
            case S_IFS_RESP_RECEIVED:           /*  Cannot happen in this state. */
            case S_ABORT_RESP_RECEIVED:         /*  Cannot happen in this state. */            
            case S_WTX_RESP_RECEIVED:           /*  Cannot happen in this state. */
            case S_ERROR_ON_VPP_STATE_RECEIVED: /*  Must never occur! */
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);
              break;
              
            /* - - - - - - - - - - - - */
            case UNKNOWN_NAD_CODING:
            case UNKNOWN_PCB_CODING:
            case INCORRECT_LEN_CODING:
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);
              break;

            /* - - - - - - - - - - - - */
            case EDC_ERROR_IN_RECEIVED_BLOCK:
            /* - - - - - - - - - - - - */
              request_retransmission(EDC_ERROR);
              break;
            
            /* - - - - - - - - - - - - */
            default:
            /* - - - - - - - - - - - - */
              #ifndef UICC_NO_DEBUG
                debug = UICC_t1_link_ctrl_state;
              #endif
              initiate_resyncronization();
              break;        
          }
          break;

        /*----------------------------*/
        case UICC_BWT_TIMEOUT:
        case UICC_CWT_TIMEOUT:
        /*----------------------------*/
          bwt_time_out();                              
          break;

        /*----------------------------*/
        case UICC_CHAR_PARITY_ERROR:
        /*----------------------------*/
          request_retransmission(PARITY_ERROR);
          break;

        /*----------------------------*/
        case UICC_LEN_VIOLATION:
        /*----------------------------*/
          request_retransmission(OTHER_PCB_ERROR);                              
          break;

        /*----------------------------*/
        case UICC_CHAR_TIMEOUT:
        case UICC_CHAR_OVERRUN_ERROR:
        /*----------------------------*/
          /* Either TX timeout error or UART congestion. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;        
      }
      break;

    /*=================================================*/
    /* Handle ABORT while sending chain.               */
    /*=================================================*/
    case UICC_ABORT_SENDING_CHAIN:
      switch(event)
      {
        /*----------------------------*/
        case UICC_BLOCK_RECEIVED:
        /*----------------------------*/
          switch(evaluate_received_block())
          {
            /* - - - - - - - - - - - - */
            case I_BLOCK_RECEIVED:
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR); 
              break;
              
            /* - - - - - - - - - - - - */
            case R_ACKNOWLEDGE_RECEIVED:
            /* - - - - - - - - - - - - */
              retrans_req_cnt = 0;
              retrans_cnt     = 0;
              UICC_t1_s_seq_no = UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], R_PCB_REC_SEQUENCE_NO_BIT_POS);

              build_i_block(restart_transmission);

              if(UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], I_PCB_MORE_DATA_BIT_POS))
              { /* Chaining of blocks needed. */
                UICC_t1_link_ctrl_state = UICC_SENDING_I_BLOCK_CHAIN;
              }
              else
              { /* Only ONE block needed. */
                UICC_t1_link_ctrl_state = UICC_SENDING_I_BLOCK;          
              }
              initiate_block(); 
              break;

            /* - - - - - - - - - - - - */
            case S_IFS_REQ_RECEIVED:
            /* - - - - - - - - - - - - */
              /* Accept the resuested data size and use the value from now on. */
              if(UICC_inf[0])
              {
                UICC_card_profile[UICC_current_reader].ifsc_used = UICC_inf[0];
              }
              build_s_block(IFS_RESP,  UICC_card_profile[UICC_current_reader].ifsc_used);
              initiate_block(); 
              break;

            /* - - - - - - - - - - - - */
            case S_WTX_REQ_RECEIVED:
            /* - - - - - - - - - - - - */
              /* Accept the BWT extension (one shot extension). */
              if(UICC_inf[0])
              {
                UICC_t1_tpdu_ctrl.bwt_multiplier = UICC_inf[0];
              }
              build_s_block(WTX_RESP, UICC_t1_tpdu_ctrl.bwt_multiplier);
              initiate_block(); 
              break;

            /* - - - - - - - - - - - - */
            case S_RESYNC_REQ_RECEIVED:         /* Must NEVER be sent by the card. */
            case S_RESYNC_RESP_RECEIVED:        /*  Cannot happen in this state. */
            case S_IFS_RESP_RECEIVED:           /*  Cannot happen in this state. */
            case S_ABORT_REQ_RECEIVED: 
            case S_ABORT_RESP_RECEIVED:         /*  Cannot happen in this state. */            
            case S_WTX_RESP_RECEIVED:           /*  Cannot happen in this state. */
            case S_ERROR_ON_VPP_STATE_RECEIVED: /*  Must never occur! */
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);
              break;
              
            /* - - - - - - - - - - - - */
            case R_EDC_OR_PARITY_ERROR_RECEIVED:
            case R_OTHER_ERROR_RECEIVED:
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);
              break;

            /* - - - - - - - - - - - - */
            case UNKNOWN_NAD_CODING:
            case UNKNOWN_PCB_CODING:
            case INCORRECT_LEN_CODING:
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);
              break;

            /* - - - - - - - - - - - - */
            case EDC_ERROR_IN_RECEIVED_BLOCK:
            /* - - - - - - - - - - - - */
              request_retransmission(EDC_ERROR);
              break;

            /* - - - - - - - - - - - - */
            default:
            /* - - - - - - - - - - - - */
              #ifndef UICC_NO_DEBUG
                debug = UICC_t1_link_ctrl_state;
              #endif
              initiate_resyncronization();
              break;        
          }
          break;

        /*----------------------------*/
        case UICC_BWT_TIMEOUT:
        case UICC_CWT_TIMEOUT:
        /*----------------------------*/
          bwt_time_out();                              
          break;

        /*----------------------------*/
        case UICC_CHAR_PARITY_ERROR:
        /*----------------------------*/
          request_retransmission(PARITY_ERROR);
          break;

        /*----------------------------*/
        case UICC_LEN_VIOLATION:
        /*----------------------------*/
          request_retransmission(OTHER_PCB_ERROR);                              
          break;

        /*----------------------------*/
        case UICC_CHAR_TIMEOUT:
        case UICC_CHAR_OVERRUN_ERROR:
        /*----------------------------*/
          /* Either TX timeout error or UART congestion. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;        
      }
      break;

    /*=================================================*/
    /*  Handle ABORT while receiving chain.            */
    /*=================================================*/
    case UICC_ABORT_RECEIVING_CHAIN:  
      switch(event)
      {
        /*----------------------------*/
        case UICC_BLOCK_RECEIVED:
        /*----------------------------*/
          switch(evaluate_received_block())
          {
            /* - - - - - - - - - - - - */
            case I_BLOCK_RECEIVED:
            /* - - - - - - - - - - - - */
              if(UICC_t1_r_seq_no == UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], I_PCB_SEND_SEQUENCE_NO_BIT_POS))
              {

                UICC_t1_r_seq_no = invert_seq_no(UICC_t1_r_seq_no); /* The next number to expect from the card. */
                retrans_req_cnt = 0;
                retrans_cnt     = 0;
              
                if(UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], I_PCB_MORE_DATA_BIT_POS))
                { /* More bit set. */
                  handle_received_i_block(restart_reception_more_frames);                
                  UICC_t1_link_ctrl_state = UICC_RECEIVING_I_BLOCK;
                  /* Send acknowledge. */
                  build_r_block(NO_ERROR, UICC_t1_r_seq_no);                  
                  initiate_block(); 
                }
                else
                { /* More bit NOT set. */
                  handle_received_i_block(restart_reception_last_frame);                                
                  UICC_t1_link_ctrl_state = UICC_LINK_SUCCEEDED;
                  UICC_t1_block_handler(UICC_STOP_BLOCK_CLOCK);
                }
              }
              else
              {
                request_retransmission(OTHER_PCB_ERROR);                                            
              }
              break;

            /* - - - - - - - - - - - - */
            case R_ACKNOWLEDGE_RECEIVED:
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR); 
              break;

            /* - - - - - - - - - - - - */
            case S_IFS_REQ_RECEIVED:
            /* - - - - - - - - - - - - */
              /* Accept the resuested data size and use the value from now on. */
              if(UICC_inf[0])
              {
                UICC_card_profile[UICC_current_reader].ifsc_used = UICC_inf[0];
              }
              build_s_block(IFS_RESP,  UICC_card_profile[UICC_current_reader].ifsc_used);
              initiate_block(); 
              break;

            /* - - - - - - - - - - - - */
            case S_WTX_REQ_RECEIVED:             
            /* - - - - - - - - - - - - */
              /* Accept the BWT extension (one shot extension). */
              if(UICC_inf[0])
              {
                UICC_t1_tpdu_ctrl.bwt_multiplier = UICC_inf[0];
              }
              build_s_block(WTX_RESP, UICC_t1_tpdu_ctrl.bwt_multiplier);
              initiate_block(); 
              break;

            /* - - - - - - - - - - - - */
            case S_RESYNC_REQ_RECEIVED:         /* Must NEVER be sent by the card. */
            case S_RESYNC_RESP_RECEIVED:        /*  Cannot happen in this state. */
            case S_IFS_RESP_RECEIVED:           /*  Cannot happen in this state. */
            case S_ABORT_REQ_RECEIVED: 
            case S_ABORT_RESP_RECEIVED:         /*  Cannot happen in this state. */            
            case S_WTX_RESP_RECEIVED:           /*  Cannot happen in this state. */
            case S_ERROR_ON_VPP_STATE_RECEIVED: /*  Must never occur! */
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);
              break;
              
            /* - - - - - - - - - - - - */
            case R_EDC_OR_PARITY_ERROR_RECEIVED:
            case R_OTHER_ERROR_RECEIVED:
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);
              break;

            /* - - - - - - - - - - - - */
            case UNKNOWN_NAD_CODING:
            case UNKNOWN_PCB_CODING:
            case INCORRECT_LEN_CODING:
            /* - - - - - - - - - - - - */
              request_retransmission(OTHER_PCB_ERROR);
              break;

            /* - - - - - - - - - - - - */
            case EDC_ERROR_IN_RECEIVED_BLOCK:
            /* - - - - - - - - - - - - */
              request_retransmission(EDC_ERROR);
              break;
            
            /* - - - - - - - - - - - - */
            default:
            /* - - - - - - - - - - - - */
              #ifndef UICC_NO_DEBUG
                debug = UICC_t1_link_ctrl_state;
              #endif
              initiate_resyncronization();
              break;        
          }
          break; 

        /*----------------------------*/
        case UICC_BWT_TIMEOUT:
        case UICC_CWT_TIMEOUT:
        /*----------------------------*/
          bwt_time_out();                              
          break;

        /*----------------------------*/
        case UICC_CHAR_PARITY_ERROR:
        /*----------------------------*/
          request_retransmission(PARITY_ERROR);
          break;

        /*----------------------------*/
        case UICC_LEN_VIOLATION:
        /*----------------------------*/
          request_retransmission(OTHER_PCB_ERROR);                              
          break;

        /*----------------------------*/
        case UICC_CHAR_TIMEOUT:
        case UICC_CHAR_OVERRUN_ERROR:
        /*----------------------------*/
          /* Either TX timeout error or UART congestion. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;        
      }
      break;

    /*=================================================*/
    /* Sending RESYNC request.                         */
    /*=================================================*/
    case UICC_SENDING_RESYNC_REQ:
      switch(event)
      {
        /*----------------------------*/
        case UICC_BLOCK_RECEIVED:
        /*----------------------------*/
          switch(evaluate_received_block())
          {
            /* - - - - - - - - - - - - */
            case S_RESYNC_RESP_RECEIVED:
            /* - - - - - - - - - - - - */
              UICC_t1_link_ctrl_state = UICC_T1_LINK_READY;
              UICC_card_profile[UICC_current_reader].ifsc_used = UICC_card_profile[UICC_current_reader].ifsc;
              UICC_card_profile[UICC_current_reader].ifsd_used = UICC_card_profile[UICC_current_reader].ifsd;
              UICC_t1_r_seq_no = 0;
              UICC_t1_s_seq_no = 0;
              retrans_req_cnt  = 0;
              retrans_cnt      = 0;
              abort_cnt        = 0;
              UICC_t1_tpdu_ctrl.bwt_multiplier = 1;
           
              build_i_block(restart_transmission);

              if(UICC_TESTBIT(UICC_prologue[UICC_PCB_POS], I_PCB_MORE_DATA_BIT_POS))
              { /* Chaining of blocks needed. */
                UICC_t1_link_ctrl_state = UICC_SENDING_I_BLOCK_CHAIN;
              }
              else
              { /* Only ONE block needed. */
                UICC_t1_link_ctrl_state = UICC_SENDING_I_BLOCK;          
              }
              initiate_block(); 
              break;
              
            /* - - - - - - - - - - - - */
            case I_BLOCK_RECEIVED:
            case R_ACKNOWLEDGE_RECEIVED:
            case R_EDC_OR_PARITY_ERROR_RECEIVED:
            case R_OTHER_ERROR_RECEIVED:
            case S_RESYNC_REQ_RECEIVED:   
            case S_WTX_REQ_RECEIVED:
            case S_WTX_RESP_RECEIVED:     
            case S_IFS_REQ_RECEIVED:
            case S_IFS_RESP_RECEIVED:     
            case S_ABORT_REQ_RECEIVED:    
            case S_ABORT_RESP_RECEIVED:   
            case S_ERROR_ON_VPP_STATE_RECEIVED:
            case UNKNOWN_NAD_CODING:
            case UNKNOWN_PCB_CODING:
            case INCORRECT_LEN_CODING:
            case EDC_ERROR_IN_RECEIVED_BLOCK:
            /* - - - - - - - - - - - - */
              initiate_resyncronization();
              break;
            
            /* - - - - - - - - - - - - */
            default:
            /* - - - - - - - - - - - - */
              #ifndef UICC_NO_DEBUG
                debug = UICC_t1_link_ctrl_state;
              #endif
              initiate_resyncronization();
              break;        
          }
          break;

        /*----------------------------*/
        case UICC_BWT_TIMEOUT:
        case UICC_CWT_TIMEOUT:
        /*----------------------------*/
          initiate_resyncronization();
          break;

        /*----------------------------*/
        case UICC_CHAR_PARITY_ERROR:
        /*----------------------------*/
          initiate_resyncronization();
          break;

        /*----------------------------*/
        case UICC_LEN_VIOLATION:
        /*----------------------------*/
          request_retransmission(OTHER_PCB_ERROR);                              
          break;

        /*----------------------------*/
        case UICC_CHAR_TIMEOUT:
        case UICC_CHAR_OVERRUN_ERROR:
        /*----------------------------*/
          /* Either TX timeout error or UART congestion. */
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;        
      }
      break;

    /*=================================================*/
    /* The LINK handling succeeded.                    */
    /*=================================================*/
    case UICC_LINK_SUCCEEDED:
      switch(event)
      {
        /*----------------------------*/
        case UICC_BLOCK_CLOCK_STOPPED:
        /*----------------------------*/
          UICC_t1_link_ctrl_state = UICC_T1_LINK_READY;
          UICC_t1_transport_handler(UICC_T1_CMD_SUCCEEDED);                  
          break;

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            debug = UICC_t1_link_ctrl_state;
          #endif
          initiate_resyncronization();   
          break;        
      }
      break;
 
    /*=================================================*/
    /* Default!!                                       */
    /*=================================================*/
    default:
      #ifndef UICC_NO_DEBUG
        debug = UICC_t1_link_ctrl_state;
      #endif
      initiate_resyncronization();   
      break;        
  }

  #ifndef UICC_NO_DEBUG
    if(debug != UICC_T1_LINK_NONE)
    {
      UICC_log_data_ptr[0] = debug;
      UICC_log_data_ptr[1] = event;
      UICC_log_data_ptr[2] = UICC_current_reader;
      UICC_log_data_ptr[3] = resync_cnt;
      UICC_log_data_ptr[4] = bwt_timeout_cnt;
      UICC_log_data_ptr[5] = retrans_req_cnt;
      UICC_log_data_ptr[6] = abort_cnt;
      UICC_raise_exception(TRAP_UICC_T1_LINK, UICC_T1_LINK_ERROR_1, 27); 
      debug = UICC_T1_LINK_NONE;
    }
  #endif
}    
                                                 /* End of file.              */
