/***********************************************************************
* Copyright (C) 2016, ZTE Corporation.
*
* File Name:    hal_tdm.c
* File Mark:
* Description:  tdm realization.
* Others:
* Version:  v1.0
* Author:   daixunkang
* Date:      2018-02-26
*

**********************************************************************/
#ifdef _OS_OSE
#include "outfmt.h"
#include "ose.h"
#include "cpu.h"
#include "dda_pci.h"
#include "dda_utils.h"
#include "dda_err.h"
#include "dda_drm.h"
#include "dda_strings.h"
#include "device.sig"
#include "inputdev.sig"
#include "ramlog.h"
#include "ose_syms.h"
#endif

#include "drvs_general.h"
#include "drvs_tdm.h"
#include "drvs_pow.h"
//#include "dma_cfg.h"
//#include "drvs_dma.h"
#include "ThreadPriority.h"
#include "drvs_chip_cfg.h"
#include "drvs_sys.h"
#include "drvs_voiceprocess.h"
#include <linux/init.h>
#include <linux/platform_device.h>
//#include <cyg/hal/tos_cache.h>
#include <linux/slab.h>
#include <linux/dma-mapping.h>
#include <linux/module.h>
#include <linux/dmaengine.h>
#include <mach/board.h>
#include <mach/dma.h>
#include <mach/iomap.h>
#include <asm/io.h>
/*************************************************************************
  *                                  Macro                                *
  *************************************************************************/
/*version and the driver name*/
#define TDM_DRIVER_NAME             "tdm"
#define TDM_DRIVER_VERSION          1

//#define ARM_TDM_LOOP_SET	*((volatile unsigned int *)(ZX_SOC_SYS_BASE + 0x60))

/* version register */
#define TDM_VERSION     				0x0104	/*TDM Version V1.4*/

/* Timing control register */

#define TDM_FS_POS                   (0x1<<0)  /*TDM FSYNC positive edge select*/
#define TDM_TX_POS                   (0x1<<1)  /*TDM TX positive edge select*/
#define TDM_RX_POS                   (0x1<<2)  /*TDM RX positive edge select*/
#define TDM_TS_WIDTH(x)              (((x)&0x3)<<5)  /* TDM timeslot width, 
                                                         00---8 clk/slot, 
                                                         01---16 clk/slot, 
                                                         10---32 clk/slot  */
#define TDM_TS_NUM(x)                (((x)&0x7F)<<7)  /*TDM timeslot NUM(+1) of one frame*/
#define TDM_FS_WIDTH(x)              (((x)&0x1F)<<14)  /*TDM FSYNC width NUM(+1)*/
#define TDM_FS_ACTIVE(x)             (((x)&0x1)<<19)         /*TDM FSYNC active low*/
#define TDM_LSB_FIRST(x)             (((x)&0x1)<<20)         /*TDM LSB first*/
#define TDM_MASTER_EN                (0x1<<21)         /*TDM master mode*/
#define TDM_TEST_MODE                (0x1<<22)		   /*test Mode*/

/* Offset register */
#define TDM_TX_OFFSET(x)             (((x)&0xFFF)<<0)  /*TDM bit offset number of TX DATA*/
#define TDM_RX_OFFSET(x)             (((x)&0xFFF)<<16) /*TDM bit offset number of RX DATA*/


/* TX fifo control register */
#define TDM_TX_FIFO_RST               (0x1<<0)   /*reset tx fifo */
#define TDM_TX_DMA_EN                 (0x1<<1)   /*Transmit DMA enable*/
#define TDM_TX_FIFO_THRESHOLD(x)      (((x)&0xf)<<2)	/*Transmit FIFO interrupt/DMA trigger threshold*/

/* RX fifo control register */
#define TDM_RX_FIFO_RST               (0x1<<0)   /*reset rx fifo */
#define TDM_RX_DMA_EN                 (0x1<<1)   /*Receive DMA enable*/
#define TDM_RX_FIFO_THRESHOLD(x)      (((x)&0xf)<<2)	/*Receive FIFO interrupt/DMA trigger threshold*/

/* tx fifo status register */
#define TDM_TX_FIFO_THRESHOLD_INT     (0x1<<0) /*TX FIFO reach threshold*/
#define TDM_TX_FIFO_EMPTY             (0x1<<2) /*TX FIFO empty*/
#define TDM_TX_FIFO_CNTR           

/* rx fifo status register */
#define TDM_RX_FIFO_THRESHOLD_INT     (0x1<<0) /*RX FIFO reach threshold*/
#define TDM_RX_FIFO_FULL             (0x1<<2) /*RX FIFO full*/
#define TDM_RX_FIFO_CNTR           

/* interrupt enable register */
#define TDM_INT_EN_TX_THRESHOLD       (0x1<<0) /*TX FIFO threshold interrup enable*/
#define TDM_INT_EN_RX_THRESHOLD       (0x1<<1) /*RX FIFO threshold interrup enable*/
#define TDM_INT_EN_TX_UNDERRUN        (0x1<<2) /*Transmit underflow interrup enable*/
#define TDM_INT_EN_RX_OVERRUN         (0x1<<3) /*receive overflow interrup enable*/
#define TDM_INT_EN_RD_CNT             (0x1<<4) /*read counter threshold interrup enable*/
#define TDM_INT_EN_WR_CNT             (0x1<<5) /*write counter threshold interrup enable*/
#define TDM_INT_EN_SYNC_FAIL          (0x1<<6) /*FS sync fail interrup enable*/

/* interrupt status */
#define TDM_INT_TX_THRESHOLD          (0x1<<0) 
#define TDM_INT_RX_THRESHOLD          (0x1<<1) 
#define TDM_INT_TX_UNDERRUN           (0x1<<2) 
#define TDM_INT_RX_OVERRUN            (0x1<<3) 
#define TDM_INT_RD_CNT                (0x1<<4) 
#define TDM_INT_WR_CNT                (0x1<<5) 
#define TDM_INT_SYNC_FAIL             (0x1<<6) 

/* frame counter */
#define TDM_FRAME_CNTR(x)              ((x)&0x3fffffff)
#define TDM_FRAME_CNTR_EN              (0x1<<30)
#define TDM_FRAME_CNTR_RST             (0x1<<31)

/* write control counter */
#define TDM_WR_CNTR_THRESHOLD(x)       ((x)&0x3fffffff)
#define TDM_WR_CNTR_EN                 (0x1<<30)
#define TDM_WR_CNTR_RST                (0x1<<31)

/* write counter */
#define TDM_WR_CNTR(x)                 ((x)&0x3fffffff)

/* write counter2 */
#define TDM_WR_CNTR2(x)                ((x)&0x3fffffff)
#define TDM_WR_CNTR2_EN                (0x1<<30)
#define TDM_WR_CNTR2_RST               (0x1<<31)

/* read control counter */
#define TDM_RD_CNTR_THRESHOLD(x)       ((x)&0x3fffffff)
#define TDM_RD_CNTR_EN                 (0x1<<30)
#define TDM_RD_CNTR_RST                (0x1<<31)

/* read counter */
#define TDM_RD_CNTR(x)                 ((x)&0x3fffffff)

/* read counter2 */
#define TDM_RD_CNTR2(x)                ((x)&0x3fffffff)
#define TDM_RD_CNTR2_EN                (0x1<<30)
#define TDM_RD_CNTR2_RST               (0x1<<31)

/* Timeslot mask 0 */
#define TS_MAKS0_TS1                   (0x1<<0)  /*time slot 1 enable*/

/* Timeslot mask 1 */
#define TS_MAKS0_TS2                   (0x1<<2)|(0x1<<0)  /*time slot 1/3 enable*/

/* process control register */
#define TDM_TX_OPEN     				(0x1<<0)	/*TX enable*/
#define TDM_RX_OPEN     				(0x1<<1)	/*RX enable*/
#define TDM_EN							(0x1<<2)

/* default rx/tx trigger threadhold */
#define TDM_RXFL_VAL                    (0x8)
#define TDM_TXFL_VAL                    (0x8)

#define TDM_ENABLE_DELAY_TIME       (10)      /* us */  //10

#define TDM_OUT_BUF_NUM             3
#define TDM_UPLINK_SEM_NUM          3
#define TDM_DOWNLINK_SEM_NUM        0
#define TDM_MAX_VOICE_GSM_MEM_SIZE	320
#define TDM_MAX_VOICE_LTE_MEM_SIZE	640
#define TDM_MAX_RX_BUF_SIZE         2048            /*2kBytes*/
#define TDM_MAX_TX_BUF_SIZE         8192           /*8kBytes*/

#define TDM_SEM_WAIT_TIME           2500

#define TDM_QUEUE_SIZE              (TDM_OUT_BUF_NUM + 1)
#define TDM_WAIT_TIME               100
#define TDM_INTCTRL_TEST_MODE    TRUE

#if defined (_USE_AUDIO_ON_7520V3)

#define TDM_CRM_CLKSEL       LSP_CRM_REG_BASE+0x50

#define TDM_CLKDIV1       LSP_CRM_REG_BASE+0x68

#define TDM_CLKDIV2       LSP_CRM_REG_BASE+0x6c
#elif defined (_USE_AUDIO_ON_7520V2)
#define TDM_CLKDIV       ((volatile UINT32 *)(LSP_CRM_REG_BASE+0x54))
#define TDM_FRA_DIV      0x00000008    /* bit 0-7 is fra div */
#define TDM_FRA_BASE     0x00000808    /* bit 8-15 is fra base */
#define TDM_INTEGER_NUM  0x00001008    /* bit 16-23 is integer num */
#define TDM_CLK_SEL_DIV  0x00001801    /* bit 24 is fra clk sel */
#define TDM_PWRDN        0x00001901    /* bit 25 is pwr dn */
#endif








/**************************************************************************
 *                                  Types                                  *
 **************************************************************************/
typedef enum
{
    TDM_REF_CLK26M,
    TDM_REF_CLK104M,

    TDM_REF_CLK_MAX
} T_Tdm_RefClkForTdm;

typedef enum
{
    BUFFER_NULL,
    BUFFER_WRITE,
    BUFFER_FULL,
    BUFFER_READ,

    MAX_BUFFER_STATUS
} T_Tdm_BufferStatus;

typedef enum
{
    TDM_IDLE,
    TDM_OPEN,
    TDM_PLAY_INUSE,
    TDM_RECORD_INUSE,
    TDM_BOTH_INUSE,

    MAX_TDM_STATUS
} T_Tdm_Status;

typedef struct
{
    volatile UINT32  tdmVersion;
    volatile UINT32  timingCtrl;
    volatile UINT32  offset;
    volatile UINT32  txFifoCtrl;
    volatile UINT32  rxFifoCtrl;
    volatile UINT32  txFifoStatus;
    volatile UINT32  rxFifoStatus;	
    volatile UINT32  intEn;
    volatile UINT32  intStatus;
    volatile UINT32  data;
    volatile UINT32  frameCntr;
	volatile UINT32  writeCntCtrl;
	volatile UINT32  writeCntr;
	volatile UINT32  writeCntr2;
	volatile UINT32  readCntCtrl;	
	volatile UINT32  readCntr;
	volatile UINT32  readCntr2;
	volatile UINT32  tsMask0;
	volatile UINT32  tsMask1;
	volatile UINT32  tsMask2;
	volatile UINT32  tsMask3;
	volatile UINT32  processCtrl;

} T_Tdm_Reg;

typedef struct
{
    T_Tdm_Reg          *regs;                      /* register pointer */
    UINT32             intline;                    /* interrupt line */
    UINT32             intprio;                   /* interrupt priority */
    OSVECTOR           vector;                   /* Interrupt vector */
#ifdef _OS_OSE
    DdaEndOfInt        *eoiFunc;               /* interrupt handler */
    DdaFuncHandle      eoiArg;                 /* interrupt handler arguments */
#endif
    PROCESS            pid;                         /* the id of the process */
} T_Tdm_Device;

typedef struct
{
    T_Tdm_Device                 *ptDev;
    T_ZDrvTdm_Cfg                tCfg;
    T_Tdm_Status                 tdmStatus;
    UINT16                       uiDmaRxChan;
    UINT16                       uiDmaTxChan;
//	zDrvDma_CallbackFunc        fDmaCbRx;
//	zDrvDma_CallbackFunc        fDmaCbTx;
    SINT32						 DmaRxID;	
    SINT32						 DmaTxID;	
    T_ZDrvI2s_TransMode          tdmTransMode;        
} T_Tdm_Instance;

typedef struct
{
    UINT8              *buf;
    T_Tdm_BufferStatus tdmBufStatus;
} T_Tdm_BufInfo;

typedef SINT32(*T_Tdm_TxFunc)(const UINT16 *pBuf, UINT32 uiLen);

typedef struct _T_Tdm_QueueNode
{
    UINT16 *pBuf;
    UINT32 len;
} T_Tdm_QueueNode;

typedef  struct _T_Tdm_Queue
{
    UINT8 readIndex;
    UINT8 writeIndex;
    T_Tdm_QueueNode *data;
} T_Tdm_Queue;

typedef struct
{
    T_Tdm_BufInfo         *txBufferArray;   /* tx buf array */
    T_Tdm_Queue           txQueue;          /* queue for tx */
    UINT32                txLen;            /* size of one buf,use in tx */
    ZOSS_SEMAPHORE_ID     txSem;            /* for the buf management */
    BOOL                  Transing;         /* use to check if datas is transing */
    BOOL                  pauseFlag;        /*  pause flag */
    T_Tdm_TxFunc          txTransmit;       /* trans function */
    T_ZDrvAudio_Channel   channel;
    T_ZDrvTdm_AUD_PLAY_CB p_cb;
	UINT8                 txRdIdx;
	UINT8                 txWrIdx;
} T_Tdm_TxStream;           /*the data stream of transing*/

typedef struct
{
    T_Tdm_BufInfo         *rxBufferArray;    /* rx buf array */
    T_Tdm_BufInfo         *currentRxBuf;     /* current rx buf */
    T_Tdm_Queue           rxQueue;           /* queue for rx */
    UINT32                rxLen;             /* size of one buf,use in rx */
	UINT8                 rxRdIdx; 
	UINT8                 rxWrIdx; 
    ZOSS_SEMAPHORE_ID     rxSem;             /* for the buf management */
    T_Tdm_TxFunc          rxTransmit;        /* trans function */
    T_ZDrvAudio_Channel   channel;
    T_ZDrvTdm_AUD_PLAY_CB p_cb;
} T_Tdm_RxStream;

#if defined (_USE_AUDIO_ON_7520V2)
typedef struct
{
    UINT32   INTEGERNUM;
    UINT32   FRACADDNUM;
    UINT32   FRACBASE;
    UINT32   CLKSELDIV;
    UINT32   PWRDN;
    T_Tdm_RefClkForTdm   refclk;
    UINT32   clkRatio;
}T_Tdm_ClkDiv;
#elif defined (_USE_AUDIO_ON_7520V3)
typedef struct {
	UINT32   FRACDIV1;
	UINT32   FRACDIV2;
	//T_Tdm_RefClkForTdm   refclk;
}T_Tdm_ClkDiv;
#endif

typedef struct
{
//    UINT32   txtotalbuffsize;
//    UINT32   rxtotalbuffsize;
//	UINT32   txsinglebuffsize;
//    UINT32   rxsinglebuffsize;
//	dma_addr_t dma_tx_phyaddr[TDM_OUT_BUF_NUM];
//	void* dma_tx_viraddr;
//	dma_addr_t dma_rx_phyaddr[TDM_OUT_BUF_NUM];
//	void* dma_rx_viraddr;
//	T_ZDrvDma_ChannelDef rxDmaChanDef;
//	T_ZDrvDma_ChannelDef txDmaChanDef;
    UINT32   totalbuffsize;
	UINT32   singlebuffsize;
	dma_addr_t dma_phyaddr;
	void* dma_viraddr;
//	T_ZDrvDma_ChannelDef dmaChanDef;
	struct dma_chan *ch;
	int channel;
}
T_Tdm_DmaInfo;

typedef enum {
    TDM_TX = 0,
    TDM_RX = 1
} T_ZDrvTDM_TransDirection;

/**************************************************************************
 *                           Global  Variable                              *
 **************************************************************************/


static T_Tdm_DmaInfo tdmDmaState[2] = {0}; 

static unsigned long long tdm_device_dma_mask = DMA_BIT_MASK(32);

static struct platform_device tdm_device = {
	.name = "v3-tdm",
	.id	= -1,
	.dev = {
		.dma_mask		= &tdm_device_dma_mask,
		.coherent_dma_mask	= DMA_BIT_MASK(32),
		.platform_data = NULL,
	 },
};

static T_Tdm_Instance  s_tdmInstance;
static T_Tdm_TxStream  s_tdmTxStream;
static T_Tdm_RxStream  s_tdmRxStream;
static const UINT16 *tdm_txBuf = NULL; /* use for playing (int mode) */
static UINT16       *tdm_rxBuf = NULL; /* use for recording (int mode) */
static UINT32       tdm_txLen;        /* use for playing (int mode) */
static UINT32       tdm_rxLen;        /* use for recording (int mode) */
static UINT8        *tdm_txBufArray[TDM_OUT_BUF_NUM] =
{
    NULL,
};
static UINT8        *tdm_rxBufArray[TDM_OUT_BUF_NUM] =
{
    NULL,
};
// 26M-->8.192M   {1,44,255,1,0,0,1}
// 26M-->4.096M   {3,89,255,0,0,0,1}
// 26M-->2.048M   {6,177,255,0,0,0,1}
// 26M-->1.024M   {12,100,255,1,0,0,1}
// 26M-->512k     {25,199,255,0,0,0,1}
// 26M-->256k     {50,143,255,1,0,0,1}
// 26M-->128k     {101,32,255,1,0,0,1}
// 26M-->64k      {203,64,255,0,0,0,1}
#if defined (_USE_AUDIO_ON_7520V3)
static T_Tdm_ClkDiv gTdmClkDiv[TDM_MAX_CLK][MAX_TDM_SLOTNUM][MAX_TDM_TS_WIDTH] =
{
	{
		{{0x80001, 0xcb},{0x100009,0x65},{0x200019, 0x32}},
		{{0x100009,0x65},{0x200019,0x32},{0x400019, 0x19}},
		{{0x180011,0x43},{0x300029,0x21},{0x600059, 0x10}},
		{{0x200019,0x32},{0x400019,0x19},{0x800059, 0xc}},
		{{0x80005, 0x28},{0x100005,0x14},{0x200005, 0xa}},

		{{0x300029,0x21},{0x600059,0x10},{0xc00059, 0x8}},
		{{0x380001,0x1d},{0x700039,0xe}, {0xe00039, 0x7}},
		{{0x400019,0x19},{0x800059,0xc}, {0x1000059,0x6}},
		{{0x480029,0x16},{0x900029,0xb}, {0x12000b9,0x5}},
		{{0x100005,0x14},{0x200005,0xa}, {0x400005, 0x5}},

		{{0x800059, 0xc},{0x1000059,0x6},{0x2000059,0x3}},
		{{0x1000059,0x6},{0x2000059,0x3},{0x4000259,0x1}},
		{{0x2000059,0x3},{0x4000259,0x1},{0x8000659,0x0}}
	},
	{
		{{0x100009, 0x65},{0x200019,0x32},{0x400019, 0x19}},
		{{0x200019,0x32},{0x400019,0x19},{0x800059, 0xc}},
		{{0x300029,0x21},{0x600059,0x10},{0xc00059, 0x8}},
		{{0x400019,0x19},{0x800059,0xc},{0x1000059, 0x6}},
		{{0x100005, 0x14},{0x200005,0xa},{0x400005, 0x5}},

		{{0x600059,0x10},{0xc00059,0x8},{0x1800059, 0x4}},
		{{0x700039,0xe},{0xe00039,0x7}, {0x1c00119, 0x3}},
		{{0x800059,0xc},{0x1000059,0x6}, {0x2000059,0x3}},
		{{0x900029,0xb},{0x12000b9,0x5}, {0x24001d9,0x2}},
		{{0x200005,0xa},{0x400005,0x5}, {0x800045, 0x2}},

		{{0x1000059, 0x6},{0x2000059,0x3},{0x4000259,0x1}},
		{{0x2000059,0x3},{0x4000259,0x1},{0x8000659,0x0}},
		{{0x4000259,0x1},{0x8000659,0x0},{0x10000659,0x0}}
	}
};
#elif defined (_USE_AUDIO_ON_7520V2)
static T_Tdm_ClkDiv gTdmClkDiv[TDM_MAX_CLK][MAX_TDM_SLOTNUM][MAX_TDM_TS_WIDTH]=
{
	{
		{{203,64,255,0,0,0,1},{101,32,255,1,0,0,1},{50,143,255,1,0,0,1}},
	    {{101,32,255,1,0,0,1},{50,143,255,1,0,0,1},{25,199,255,0,0,0,1}},
	    {{0,0,0,0,0,0,0},{0,0,0,0,0,0,0},{0,0,0,0,0,0,0}},
	    {{50,143,255,1,0,0,1},{25,199,255,0,0,0,1},{12,100,255,1,0,0,1}},
	    {{0,0,0,0,0,0,0},{0,0,0,0,0,0,0},{0,0,0,0,0,0,0}},
	    {{0,0,0,0,0,0,0},{0,0,0,0,0,0,0},{0,0,0,0,0,0,0}},
	    {{0,0,0,0,0,0,0},{0,0,0,0,0,0,0},{0,0,0,0,0,0,0}},
	    {{25,199,255,0,0,0,1},{12,100,255,1,0,0,1},{6,177,255,0,0,0,1}},
	    {{0,0,0,0,0,0,0},{0,0,0,0,0,0,0},{0,0,0,0,0,0,0}},
	    {{0,0,0,0,0,0,0},{0,0,0,0,0,0,0},{0,0,0,0,0,0,0}},
	    {{12,100,255,1,0,0,1},{6,177,255,0,0,0,1},{3,89,255,0,0,0,1}},
	    {{6,177,255,0,0,0,1},{3,89,255,0,0,0,1},{1,44,255,1,0,0,1}},
	    {{3,89,255,0,0,0,1},{1,44,255,1,0,0,1},{0,0,0,0,0,0,0}}
	},
	{

	}
};
#endif

static UINT32 audioTdmWsHandle = 0;
static UINT32 audioTdmClkHandle = 0;
static UINT32 audioTdmDinHandle = 0;
static UINT32 audioTdmDoutHandle = 0;

SINT32 gTdmCntChangeTx = 0;
SINT32 gTdmCntChangeRx = 0;

static UINT32 g_tdmClkDivRegs = LSP_CRM_REG_BASE+0x54;
static BOOL s_bTdmClkInit = FALSE;

UINT32 tdmwrite_pbuf=0;
UINT32 tdmwrite_uilen=0;

static BOOL s_bTdmInitInt = FALSE;

/**************************************************************************
 *                           Function Prototypes                           *
 **************************************************************************/
static SINT32 tdm_GetTdmDiv(T_ZDrvTdm_Cfg *ptCfg, T_Tdm_ClkDiv *pDiv);
static SINT32 tdm_SetPdiv(T_ZDrvTdm_Cfg *ptCfg);

static SINT32 tdm_QueueInit(T_Tdm_Queue *tdmQueue);
static SINT32 tdm_QueuePut(T_Tdm_Queue *tdmQueue, UINT16 *pBuf, UINT32 uiLen);
static SINT32 tdm_QueueGet(T_Tdm_Queue *tdmQueue, T_Tdm_QueueNode **queueNode);
static SINT32 tdm_QueueDestory(T_Tdm_Queue *tdmQueue);

static SINT32 tdm_SetupTxBuf();
static SINT32 tdm_SetupRxBuf();
static SINT32 tdm_Reset();

static VOID tdm_IntIsr();
static VOID tdm_IntTxWriteFifo(UINT32 chanlable);
static VOID tdm_IntRxWriteBuf();
static VOID tdm_IntTxChangeBuf();
static VOID tdm_IntRxChangeBuf();

static SINT32 tdm_TxCheckState(T_ZDrvTdm_Cfg *ptCfg);
static SINT32 tdm_RxCheckState(T_ZDrvTdm_Cfg *ptCfg);

static SINT32 tdm_TxComInit(T_ZDrvTdm_Params *params);
static SINT32 tdm_RxComInit(T_ZDrvTdm_Params *params);

static SINT32 tdm_IntRxInit();
static SINT32 tdm_IntTxInit();
static SINT32 tdm_DmaRxInit();
static SINT32 tdm_DmaTxInit();
static SINT32 tdm_DmaTx(const UINT16 *pBuf, UINT32  uiLen);
static SINT32 tdm_DmaRx(T_Tdm_Instance *ptInstance, UINT8 idx);

static SINT32 tdm_CleanTxResource();
static SINT32 tdm_CleanRxResource();

static SINT32 tdm_IntTx(const UINT16 *pBuf, UINT32 uiLen);

static VOID tdm_InitInt(VOID);
static SINT32 tdm_SetClk(T_Tdm_ClkDiv *pDiv);

static SINT32 tdm_CloseClk();
static SINT32 tdm_SetClkDiv(UINT32 divFunc, UINT32 div);

#ifdef _USE_PSM
extern SINT32 zDrvPow_SetDevActive(T_ZDrvPow_IdleFlag devId);
extern SINT32 zDrvPow_SetDevIdle(T_ZDrvPow_IdleFlag devId);
#endif

SINT32 tdm_SetTiming(T_Tdm_Reg   *ptReg, T_ZDrvTdm_Cfg *ptCfg);
static VOID tdm_ClkInit();
static VOID tdm_ClkDeinit();
static VOID isr_tdm(VOID);
SINT32 tdm_rxsem_count_Init();
SINT32 tdm_txsem_count_Init();


extern T_zDrvVoice_GbVar g_voiceVar; 
/**************************************************************************
 *                           Function Defines                              *
 **************************************************************************/
/****************************************************************************
* Function:  tdm_IsQueueEmpty
* Description: check if the queue is empty
* Parameters:
*   Input:
*             tdmQueue :queue pointer
*   Output:
*             None
*
* Returns:
*             DRV_ERR_INVALID_PARAM : NULL pointer
*
* Others:
****************************************************************************/
__inline BOOL tdm_IsQueueEmpty(T_Tdm_Queue *tdmQueue)
{
    if (NULL == tdmQueue) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }

    return (tdmQueue->writeIndex == tdmQueue->readIndex);
}

/****************************************************************************
* Function:  tdm_IsQueueFull
* Description: check if the queue is full
* Parameters:
*   Input:
*          tdmQueue : queue pointer
*   Output:
*          None
*
* Returns:
*          DRV_ERR_INVALID_PARAM : NULL pointer
*
* Others:
****************************************************************************/
__inline BOOL tdm_IsQueueFull(T_Tdm_Queue *tdmQueue)
{
    if (NULL == tdmQueue) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }

    return ((tdmQueue->writeIndex + 1) % TDM_QUEUE_SIZE == tdmQueue->readIndex);
}

/****************************************************************************
* Function:  tdm_UpdateTxBufferStatus
* Description: use to manage buffer status ,update one buffer status
* Parameters:
*   Input:
*            bufAddr : buffer pointer
*            bufferStatus : buffer status
*   Output:
*             None
*
* Returns: DRV_SUCCESS: operate success
*              DRV_ERR_INVALID_PARAM:param error
*
* Others:
****************************************************************************/
static SINT32 tdm_UpdateTxBufferStatus(UINT16 * bufAddr , T_Tdm_BufferStatus bufferStatus)
{
    T_Tdm_TxStream *tdmStream = &s_tdmTxStream;
    UINT8 bufIndex = 0;

    if (NULL == bufAddr) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }

    if(tdmStream->txBufferArray == NULL) {
         zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm_UpdateTxBufferStatus tdmStream->txBufferArray NULL return ");	
	  return DRV_ERROR;
    }

    for (; bufIndex < TDM_OUT_BUF_NUM; bufIndex++)
    {
        if (tdmStream->txBufferArray[bufIndex].buf == (UINT8 *)bufAddr)
        {
            tdmStream->txBufferArray[bufIndex].tdmBufStatus = bufferStatus;
            break;
        }
    }

    if (bufIndex == TDM_OUT_BUF_NUM)
    {
        return DRV_ERROR;
    }

    return DRV_SUCCESS;
}

/****************************************************************************
* Function:  tdm_UpdateBufferStatus
* Description: use to manage buffer status ,update one buffer status
* Parameters:
*   Input:
*            bufAddr : buffer pointer
*            bufferStatus : buffer status
*   Output:
*             None
*
* Returns: DRV_SUCCESS: operate success
*              DRV_ERR_INVALID_PARAM:param error
*
* Others:
****************************************************************************/
static SINT32 tdm_UpdateRxBufferStatus(UINT16 * bufAddr , T_Tdm_BufferStatus bufferStatus)
{
    T_Tdm_RxStream *tdmStream = &s_tdmRxStream;
    UINT8 bufIndex = 0 ;

    if (NULL == bufAddr) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }
    if(tdmStream->rxBufferArray == NULL) {
         zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm_UpdateRxBufferStatus tdmStream->rxBufferArray NULL return ");	
	  return DRV_ERROR;
    }
    for (; bufIndex < TDM_OUT_BUF_NUM; bufIndex++)
    {
        if (tdmStream->rxBufferArray[bufIndex].buf == (UINT8 *)bufAddr)
        {
            tdmStream->rxBufferArray[bufIndex].tdmBufStatus = bufferStatus;
            break;
        }
    }

    if (bufIndex == TDM_OUT_BUF_NUM)
    {
        return DRV_ERROR;
    }

    return DRV_SUCCESS;
}

/****************************************************************************
* Function:  tdm_ChangeTxBufferStatus
* Description: use to manage buffer status ,update one buffer status. if one buf status is one ,change to new status
* Parameters:
*   Input:
*            oldStatus : buffer status
*            bufferStatus : buffer status
*   Output:
*             None
*
* Returns: DRV_SUCCESS: operate success
*              DRV_ERR_INVALID_PARAM:param error
*
* Others:
****************************************************************************/
static SINT32 tdm_ChangeTxBufferStatus(T_Tdm_BufferStatus oldStatus , T_Tdm_BufferStatus newStatus)
{
    T_Tdm_TxStream *tdmStream = &s_tdmTxStream;
    UINT8 bufIndex = 0;

    for (; bufIndex < TDM_OUT_BUF_NUM; bufIndex++)
    {
        if (tdmStream->txBufferArray[bufIndex].tdmBufStatus == oldStatus)
        {
            tdmStream->txBufferArray[bufIndex].tdmBufStatus = newStatus;
            break;
        }
    }

    if (bufIndex == TDM_OUT_BUF_NUM)
    {
        return DRV_ERR_INVALID_PARAM;
    }

    return DRV_SUCCESS;
}
/****************************************************************************
* Function:  tdm_QueueInit
* Description: this function is use to init the buffer queue
* Parameters:
*   Input:
*           tdmQueue :queue pointer
*   Output:
*           None
* Returns:
*                 DRV_ERR_INVALID_PARAM : null pointer
*                 DRV_ERR_MEM_ALLOC : alloc fail
*
* Others:
****************************************************************************/
static SINT32 tdm_QueueInit(T_Tdm_Queue *tdmQueue)
{
    SINT32 ret = 0;

    if (NULL == tdmQueue) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }

    tdmQueue->data = (T_Tdm_QueueNode *)zOss_Malloc(sizeof(T_Tdm_QueueNode) * TDM_QUEUE_SIZE);
    if (NULL == tdmQueue->data)
    {
        return DRV_ERR_MEM_ALLOC;
    }

    for (ret = 0; ret < TDM_QUEUE_SIZE; ret++)
    {
        tdmQueue->data[ret].pBuf = NULL;
        tdmQueue->data[ret].len = 0;
    }

    tdmQueue->readIndex = tdmQueue->writeIndex = 0;

    return DRV_SUCCESS;
}



/****************************************************************************
* Function:  tdm_QueuePut
* Description:ָβ
* Parameters:
*   Input:
*           tdmQueue: queue pointer
*           *pBuf : buffer pointer
*           uiLen : buffer length
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
* Others:
****************************************************************************/
static SINT32 tdm_QueuePut(T_Tdm_Queue *tdmQueue, UINT16 *pBuf, UINT32 uiLen)
{
    BOOL isFull = FALSE;

    if (NULL == tdmQueue || NULL == pBuf || 0 == uiLen) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }
    isFull = tdm_IsQueueFull(tdmQueue);

    if (TRUE == isFull) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    tdmQueue->data[tdmQueue->writeIndex].pBuf = pBuf;
    tdmQueue->data[tdmQueue->writeIndex].len = uiLen;
    tdmQueue->writeIndex = (tdmQueue->writeIndex + 1) % TDM_QUEUE_SIZE;

    return DRV_SUCCESS;
}

/****************************************************************************
* Function:  tdm_QueueGet
* Description:Ӷȡ
* Parameters:
*   Input:
*           tdmQueue: queue pointer
*           *queueNode : queue node  pointer
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
* Others:
****************************************************************************/
static SINT32 tdm_QueueGet(T_Tdm_Queue *tdmQueue, T_Tdm_QueueNode **queueNode)
{
    BOOL isEmpty = FALSE;

    if (NULL == tdmQueue || NULL == queueNode) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }

    isEmpty = tdm_IsQueueEmpty(tdmQueue);

    if (TRUE == isEmpty)
    {
        *queueNode = NULL;
    }
    else
    {
        *queueNode = &tdmQueue->data[tdmQueue->readIndex];
        tdmQueue->readIndex = (tdmQueue->readIndex + 1) % TDM_QUEUE_SIZE;
    }

    return DRV_SUCCESS;
}

/****************************************************************************
* Function:  tdm_QueueDestory
* Description: destory the queue
* Parameters:
*   Input:
*           tdmQueue: queue pointer
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
* Others:
****************************************************************************/
static SINT32 tdm_QueueDestory(T_Tdm_Queue *tdmQueue)
{
    if (NULL == tdmQueue) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }

    if (tdmQueue->data != NULL)
    {
        zOss_Free(tdmQueue->data);
    }

    tdmQueue->data = NULL;
    tdmQueue->readIndex = 0;
	tdmQueue->writeIndex = 0;

    return DRV_SUCCESS;
}

static SINT32 tdm_GetTdmDiv(T_ZDrvTdm_Cfg *ptCfg, T_Tdm_ClkDiv *pDiv)
{
    T_ZDrvTdm_SlotNum tdmSlotNum = MAX_TDM_SLOTNUM;
    T_ZDrvTdm_TsWidth tdmTsWidth = MAX_TDM_TS_WIDTH;
	T_ZDrvTdm_Clk tdmClockSet = TDM_MAX_CLK;

    if (NULL == pDiv||NULL == ptCfg || ptCfg->tSlotNum>=MAX_TDM_SLOTNUM|| ptCfg->tTsWidth >= MAX_TDM_TS_WIDTH )
    {
        return DRV_ERR_INVALID_PARAM;
    }

	switch(ptCfg->sample_rate) {
	case 8000:
		tdmClockSet = TDM_CLK_8000;
		break;
	case 16000:
		tdmClockSet = TDM_CLK_16000;
		break;
	default:
		tdmClockSet = TDM_CLK_8000;
		break;
	}

	tdmSlotNum = ptCfg->tSlotNum;
	tdmTsWidth = ptCfg->tTsWidth;
	
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm_GetTdmDiv gTdmClkDiv[%d][%d][%d]\n",tdmClockSet,tdmSlotNum,tdmTsWidth);
    *pDiv = gTdmClkDiv[tdmClockSet][tdmSlotNum][tdmTsWidth];

    return DRV_SUCCESS;
}

static SINT32 tdm_SetPdiv(T_ZDrvTdm_Cfg *ptCfg)
{
    SINT32 ret = DRV_SUCCESS;
    T_Tdm_ClkDiv div = {0};

    if (NULL == ptCfg)
    {
        return DRV_ERR_INVALID_PARAM;
    }

    ret = tdm_GetTdmDiv(ptCfg, &div);

    if (ret != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, " tdm_SetPdiv err ret=0x%x",ret);
        return DRV_ERROR;
    }

    ret = tdm_SetClk(&div);
	/*klocwork 3  INVARIANT_CONDITION.UNREACH  delete if*/
	/*
    if (ret != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, " tdm_SetPdiv tdm_SetClk err ret=0x%x",ret);

        return DRV_ERROR;
    }*/

    return DRV_SUCCESS;
}

/****************************************************************************
 * Function:  tdm_CloseClk
 * Description: this function is use to close tdm divide frequency
* Parameters:
*   Input:
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
 * Others:
 ****************************************************************************/
static SINT32 tdm_CloseClk()
{
#if defined (_CHIP_ZX297520V2)
    tdm_SetClkDiv(TDM_PWRDN, 1);
#endif
	return DRV_SUCCESS;
}

static SINT32 tdm_SetClkDiv(UINT32 divFunc, UINT32 div)
{
    UINT32 bitWidth = divFunc & 0xff;
    UINT32 maxDiv = (0x1<<bitWidth)-1;
    UINT32 bitIndex = (divFunc>>8) & 0xff;
    UINT32 bitMask  = 0;
    UINT32 regBase = 0;
    UINT32 tmp= 0;

    if(div > maxDiv)
        div = maxDiv;

    if(bitIndex == 28)  /*Ƶֻ֧żƵ*/
    {
        div &=0xfe;
        div++;
    }

    bitMask  = ((0x1<<bitWidth)-1)<<bitIndex;
    regBase = g_tdmClkDivRegs;
    tmp= reg32(regBase);
    tmp &= ~bitMask;
    tmp |= div<<bitIndex;
    reg32(regBase) = tmp;

    return DRV_SUCCESS;
}

static VOID tdm_ClkInit(VOID)
{
	unsigned int val;

    if (s_bTdmClkInit)
        return;


    if (g_voiceVar.isUseSlicCodec != 1)
    {
		val = readl(TDM_CRM_CLKSEL);
		val &= ~(3 << 8);
		writel(val, TDM_CRM_CLKSEL);
		
		val = readl(TDM_CRM_CLKSEL);
		val |= (3 << 8);
		writel(val, TDM_CRM_CLKSEL);

		zOss_Sleep(1);
    }

	val = readl(TDM_CRM_CLKSEL);
	val |= ((3 << 0) | (3 << 10));
	writel(val, TDM_CRM_CLKSEL);

    s_bTdmClkInit = TRUE;
}

static VOID tdm_ClkDeinit()
{
	unsigned int val;
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm_ClkDeinit");

    if (!s_bTdmClkInit)
    {
        return;
    }


	val = readl(TDM_CRM_CLKSEL);
	val &= ~((3 << 0) | (3 << 10));
	writel(val, TDM_CRM_CLKSEL);
	
	val = readl(TDM_CRM_CLKSEL);
	val &= ~(3 << 8);
	writel(val, TDM_CRM_CLKSEL);
	
	val = readl(TDM_CRM_CLKSEL);
	val |= (3 << 8);
	writel(val, TDM_CRM_CLKSEL);

    s_bTdmClkInit = FALSE;
}

static SINT32 tdm_SetClk(T_Tdm_ClkDiv *pDiv)
{
   unsigned int val = 0;
    unsigned int div1 = 0,div2 = 0;
    if (NULL == pDiv)
    {
        return DRV_ERR_INVALID_PARAM;
    }
	
#if defined (_USE_AUDIO_ON_7520V3)

	//TDM_CLKDIV2 &= (~(0x1 << 16));
	//TDM_CLKDIV1 = pDiv->FRACDIV1;
	//TDM_CLKDIV2 = pDiv->FRACDIV2;
	//TDM_CLKDIV2 |= (0x1 << 16);
	val = readl(TDM_CLKDIV2);
	val &= ~(0x1 << 16);
	writel(val, TDM_CLKDIV2);

	writel(pDiv->FRACDIV1, TDM_CLKDIV1);
	writel( pDiv->FRACDIV2, TDM_CLKDIV2);

	val = readl(TDM_CLKDIV2);
	val |= (0x1 << 16);
	writel(val, TDM_CLKDIV2);
	
	div1 = readl(TDM_CLKDIV1);
	div2 = readl(TDM_CLKDIV2);
	
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm_SetClk,TDM_CLKDIV1=0x%x, TDM_CLKDIV2=0x%x\n",div1,div2);

#elif defined (_USE_AUDIO_ON_7520V2)

    tdm_SetClkDiv(TDM_INTEGER_NUM, pDiv->INTEGERNUM);
    tdm_SetClkDiv(TDM_FRA_DIV, pDiv->FRACADDNUM);
    tdm_SetClkDiv(TDM_FRA_BASE, pDiv->FRACBASE);
    tdm_SetClkDiv(TDM_CLK_SEL_DIV, pDiv->CLKSELDIV);
    tdm_SetClkDiv(TDM_PWRDN, pDiv->PWRDN);
    
#endif

    return DRV_SUCCESS;
}

/****************************************************************************
 * Function:  tdm_Reset
 * Description: this function is use to reset all device,contain tdm fifo and dma
* Parameters:
*   Input:
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
 * Others:
 ****************************************************************************/
static SINT32 tdm_Reset()
{
    T_Tdm_Reg  *ptReg = NULL;

    if (NULL == s_tdmInstance.ptDev) 	/*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
    	zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    ptReg = s_tdmInstance.ptDev->regs;

    ptReg->timingCtrl = 0x0;

    /* disable dma */
    ptReg->txFifoCtrl &= ~TDM_TX_DMA_EN;
    ptReg->rxFifoCtrl &= ~TDM_RX_DMA_EN;

    /* reset tx/rx fifo */
    ptReg->txFifoCtrl |= TDM_TX_FIFO_RST;
    ptReg->rxFifoCtrl |= TDM_RX_FIFO_RST;

    /* clear interrupt */
    ptReg->intStatus = 0xffff;

    /* disable interrupt */
    ptReg->intEn = 0x0;

    /* disable tdm; disable rx/tx; */
    ptReg->processCtrl = 0x0;

    /* reset offset */
    ptReg->offset = 0;  

    /* reset tsMask */
    ptReg->tsMask0 = 0;

    return DRV_SUCCESS;
}

/****************************************************************************
* Function:  tdm_SetupBuf
* Description: init buffer
* Parameters:
*   Input:
*           tdmStrem: tdm stream struct
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
 * Others:
 ****************************************************************************/
static SINT32 tdm_SetupTxBuf()
{
    SINT32 index = 0;
    T_Tdm_TxStream *tdmStream = &s_tdmTxStream;

    tdmStream->txBufferArray = (T_Tdm_BufInfo *)zOss_Malloc(TDM_OUT_BUF_NUM * sizeof(T_Tdm_BufInfo));
    if (NULL == tdmStream->txBufferArray)
    {
        return DRV_ERR_MEM_ALLOC;
    }

    for (index = 0; index < TDM_OUT_BUF_NUM; index++)
    {
        tdmStream->txBufferArray[index].buf = tdm_txBufArray[index];
        tdmStream->txBufferArray[index].tdmBufStatus = BUFFER_NULL;
       // zOss_Memset(tdmStream->txBufferArray[index].buf, 0, TDM_MAX_TX_BUF_SIZE);
    }

    return DRV_SUCCESS;
}

/****************************************************************************
* Function:  tdm_SetupBuf
* Description: init buffer
* Parameters:
*   Input:
*           tdmStrem: tdm stream struct
*   Output:
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
 * Others:
 ****************************************************************************/
static SINT32 tdm_SetupRxBuf()
{
    SINT32 index = 0;
    T_Tdm_RxStream *tdmStream = &s_tdmRxStream;

    tdmStream->rxBufferArray = (T_Tdm_BufInfo *)zOss_Malloc(TDM_OUT_BUF_NUM * sizeof(T_Tdm_BufInfo));
    if (NULL == tdmStream->rxBufferArray)
    {
        return DRV_ERR_MEM_ALLOC;
    }

    for (index = 0; index < TDM_OUT_BUF_NUM; index++)
    {
        tdmStream->rxBufferArray[index].buf = tdm_rxBufArray[index];
        tdmStream->rxBufferArray[index].tdmBufStatus = BUFFER_NULL ;
       // zOss_Memset(tdmStream->rxBufferArray[index].buf, 0, TDM_MAX_RX_BUF_SIZE);
    }
	
    tdmStream->currentRxBuf = NULL;

    return DRV_SUCCESS;
}

/****************************************************************************
* Function:  tdm_Configure
* Description: config tdm
* Parameters:
*   Input:
*           ptCfg : tdm config param
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
*
* Others:
 ****************************************************************************/
SINT32 tdm_clkConfigure(T_ZDrvTdm_Cfg *ptCfg)
{
    SINT32 ret = DRV_SUCCESS;

    if (NULL == ptCfg) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }

    if (ptCfg->bMaster)
    {
        ret = tdm_SetPdiv(ptCfg);
        if (ret != DRV_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm_Configure  tdm_SetPdiverr ret=0x%x",ret);
            return DRV_ERROR;
        }
    }

	return DRV_SUCCESS;
}


static SINT32 tdm_Configure(T_ZDrvTdm_Cfg *ptCfg)
{
    T_Tdm_Instance  *ptInstance = NULL;
    T_Tdm_Reg    *ptReg = NULL;

    SINT32 ret = DRV_SUCCESS;

    if (NULL == ptCfg) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }

    ptInstance = &s_tdmInstance;
    ptReg = ptInstance->ptDev->regs;
	/*klocwork 3  INVARIANT_CONDITION.UNREACH  delete if*/
    /*if (ret != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm_Configure  tdm_SetTiming ret=0x%x",ret);
        return DRV_ERROR;
    }*/

    if (ptCfg->bMaster)
    {
        ret = tdm_SetPdiv(ptCfg);
        if (ret != DRV_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm_Configure  tdm_SetPdiverr ret=0x%x",ret);
            return DRV_ERROR;
        }
        ptReg->timingCtrl|= TDM_MASTER_EN;   /* as master */  //test for setting master mode before gpi0/tdm
    }
    else
    {
        ptReg->timingCtrl &= ~TDM_MASTER_EN;  /* as slave */
    }

    /*transmit data at negative edge or at positive edge*/
    if (ptCfg->fsClkSel == TDM_FS_CLK_NEG)
    {
        ptReg->timingCtrl &= ~TDM_FS_POS;   
    }
    else if (ptCfg->fsClkSel == TDM_FS_CLK_POS)
    {
        ptReg->timingCtrl |= TDM_FS_POS;  /*transmit at rising edge*/
    }
    if (ptCfg->txClkSel == TDM_TX_CLK_NEG)
    {
        ptReg->timingCtrl &= ~TDM_TX_POS;   
    }
    else if (ptCfg->txClkSel == TDM_TX_CLK_POS)
    {
        ptReg->timingCtrl |= TDM_TX_POS;    /*transmit at rising edge*/
    }
    if (ptCfg->rxClkSel == TDM_RX_CLK_NEG)
    {
        ptReg->timingCtrl &= ~TDM_RX_POS;   /*receive at falling edge*/
    }
    else if (ptCfg->rxClkSel == TDM_RX_CLK_POS)
    {
        ptReg->timingCtrl |= TDM_RX_POS;
    }

    /* timeslot width */
    if (MAX_TDM_TS_WIDTH <= ptCfg->tTsWidth) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }
    else
    {
        ptReg->timingCtrl &= ~ 	TDM_TS_WIDTH(0x3);
        ptReg->timingCtrl |= TDM_TS_WIDTH(ptCfg->tTsWidth);/*TDM_TS_WIDTH*/
    }

    /* timeslot number */
    if (MAX_TDM_SLOTNUM <= ptCfg->tSlotNum) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }
    else
    {
        ptReg->timingCtrl &= ~ TDM_TS_NUM(0x7f);
        if(ptCfg->tSlotNum <= TDM_10_SLOT)
            ptReg->timingCtrl |= TDM_TS_NUM(ptCfg->tSlotNum);/*TDM_TS_NUM*/
        else if(ptCfg->tSlotNum == TDM_16_SLOT)
            ptReg->timingCtrl |= TDM_TS_NUM(0xf);/*TDM_TS_NUM*/
        else if(ptCfg->tSlotNum == TDM_32_SLOT)
            ptReg->timingCtrl |= TDM_TS_NUM(0x1f);/*TDM_TS_NUM*/
        else /*klocwork 3  INVARIANT_CONDITION.UNREACH (ptCfg->tSlotNum == TDM_64_SLOT)*/
            ptReg->timingCtrl |= TDM_TS_NUM(0x3f);/*TDM_TS_NUM*/
    }

    /* fsync width */
    if (MAX_FS_WIDTH <= ptCfg->tFsWidth) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }
    else
    {
        ptReg->timingCtrl &= ~ TDM_FS_WIDTH(0x1F);
        ptReg->timingCtrl |= TDM_FS_WIDTH(ptCfg->tFsWidth);/*TDM_TS_NUM*/
    }

    /* fsync active */
    if (MAX_FS_ACTIVE <= ptCfg->tFsActive) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/ 
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }
    else
    {
        ptReg->timingCtrl |= TDM_FS_ACTIVE(ptCfg->tFsActive);/*TDM_TS_NUM*/
    }
	
    /* LSB first */
    if (MAX_FIRSTBIT <= ptCfg->tTdmFstBit) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }
    else
    {
        ptReg->timingCtrl |= TDM_LSB_FIRST(ptCfg->tTdmFstBit); /*TDM_TS_NUM*/
    }
	
    /*test mode*/
    if(ptCfg->tTestMode == TDM_NORMAL_MODE)
    {
        ptReg->timingCtrl &= ~ TDM_TEST_MODE;
    }
    else if(ptCfg->tTestMode == TDM_LOOPBACK_MODE)
    {
        ptReg->timingCtrl |= TDM_TEST_MODE;
    }

	/* offset */
    ptReg->offset = 0;   //reset offset
	ptReg->offset |= TDM_RX_OFFSET(ptCfg->tRxOffset);
	ptReg->offset |= TDM_TX_OFFSET(ptCfg->tTxOffset);
    
	
    /*time slot mask*/
    ptReg->tsMask0 = 0;   //reset tsMask
    if(ptCfg->tTsEnable == TDM_1TS_ENABLE)
    {
        ptReg->tsMask0 |= TS_MAKS0_TS1;
    }
    else if(ptCfg->tTsEnable == TDM_2TS_ENABLE)
    {
       ptReg->tsMask0 |= TS_MAKS0_TS2;
    }

    /*enable frame counter */
    ptReg->frameCntr |= TDM_FRAME_CNTR_EN;
    
    /*enable write counter */
    ptReg->writeCntCtrl |= TDM_WR_CNTR_EN;
    ptReg->writeCntr2 |= TDM_WR_CNTR2_EN;

    /*enable read counter */
    ptReg->readCntCtrl |= TDM_RD_CNTR_EN;
    ptReg->readCntr2 |= TDM_RD_CNTR2_EN;
            
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm_Configure success");
    return DRV_SUCCESS;
}

/****************************************************************************
* Function:  tdm_CleanResource
* Description: clean resource
* Parameters:
*   Input:
*           NONE
*   Output:
*           NONE
* Returns:
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
*
* Others:
 ****************************************************************************/
static SINT32 tdm_CleanTxResource()
{
    SINT32 ret = DRV_SUCCESS;

    ret = tdm_QueueDestory(&s_tdmTxStream.txQueue);
    if (s_tdmTxStream.txBufferArray != NULL)
    {
        zOss_Free(s_tdmTxStream.txBufferArray);
        s_tdmTxStream.txBufferArray = NULL;
    }

    return ret;
}

/****************************************************************************
* Function:  tdm_CleanRxResource
* Description: clean resource
* Parameters:
*   Input:
*           NONE
* Returns:
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
*
* Others:
 ****************************************************************************/
static SINT32 tdm_CleanRxResource()
{
    SINT32 ret = DRV_SUCCESS;

    ret = tdm_QueueDestory(&s_tdmRxStream.rxQueue);
    if (s_tdmRxStream.rxBufferArray != NULL)
    {
        zOss_Free(s_tdmRxStream.rxBufferArray);
        s_tdmRxStream.rxBufferArray = NULL;
    }

    return ret;
}

/**************************************************************************
* Function:  tdm_IntIsr
* Description: interrupt handle while in int mode
* Parameters:
*   Input:
*           ptInstance : tdm instance
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
static SINT32 tdm_TxCheckState(T_ZDrvTdm_Cfg *ptCfg)
{
    SINT32          ret = DRV_SUCCESS;
    T_Tdm_Instance  *ptInstance = &s_tdmInstance;

    if (NULL == ptCfg)
    {
        return DRV_ERR_INVALID_PARAM;
    }
	
    if ((ptInstance->tdmStatus != TDM_OPEN)&&(ptInstance->tdmStatus != TDM_RECORD_INUSE))
    {
        return DRV_ERR_NOT_OPENED;
    }
	
    if (ptInstance->tdmStatus != TDM_RECORD_INUSE)
    {
        ptInstance->tCfg = *ptCfg;
        ret = tdm_Configure(ptCfg);
        if (ret != DRV_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm_TxCheckState tdm_Configure err ret=0x%x",ret);
            return DRV_ERROR;
        }
    }
    else
    {
        if ((ptCfg->bMaster!= ptInstance->tCfg.bMaster)||
            (ptCfg->tTestMode != ptInstance->tCfg.tTestMode))
        {
            return DRV_ERR_INVALID_PARAM;
        }
    }
	
    return DRV_SUCCESS;
}

/**************************************************************************
* Function:  tdm_IntIsr
* Description: interrupt handle while in int mode
* Parameters:
*   Input:
*           ptInstance : tdm instance
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
static SINT32 tdm_RxCheckState(T_ZDrvTdm_Cfg *ptCfg)
{
    SINT32          ret = DRV_SUCCESS;
    T_Tdm_Instance  *ptInstance = &s_tdmInstance;

    if (NULL == ptCfg) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }
    /* clear interrupt. must disable interrupt or tdm at first */

    if ((ptInstance->tdmStatus != TDM_OPEN)&&(ptInstance->tdmStatus != TDM_PLAY_INUSE))
    {
        return DRV_ERR_NOT_OPENED;
    }
    if (ptInstance->tdmStatus != TDM_PLAY_INUSE)
    {
        ptInstance->tCfg = *ptCfg;
        ret = tdm_Configure(ptCfg);
        if (ret != DRV_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm_RxCheckState  failed ret is %d/r/n",ret);
            return DRV_ERROR;
        }
    }
    else
    {
        if((ptCfg->bMaster!= ptInstance->tCfg.bMaster)||
           (ptCfg->tTestMode != ptInstance->tCfg.tTestMode))
        {
            return DRV_ERR_INVALID_PARAM;
        }
    }
    return DRV_SUCCESS;
}

/**************************************************************************
* Function:  tdm_IntIsr
* Description: interrupt handle while in int mode
* Parameters:
*   Input:
*           ptInstance : tdm instance
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
static SINT32 tdm_TxComInit(T_ZDrvTdm_Params *params)
{
    T_Tdm_TxStream  *txStream = &s_tdmTxStream;
    T_Tdm_Instance  *ptInstance = &s_tdmInstance;
    T_Tdm_Reg       *ptReg = NULL;
    SINT32          ret = DRV_SUCCESS;

    if (NULL == params) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }

    ptReg = ptInstance->ptDev->regs;
    txStream->Transing = FALSE;
    txStream->pauseFlag = FALSE;
    txStream->txTransmit = NULL;
    txStream->channel = params->channel;
    txStream->txLen = params->buffersize;
    txStream->p_cb = params->p_cb;

	txStream->txRdIdx = 0;
	txStream->txWrIdx = 0;

	tdm_txBufArray[0] = (UINT8 *)(tdmDmaState[TDM_TX].dma_viraddr);
	tdm_txBufArray[1] = tdm_txBufArray[0] + txStream->txLen;
	tdm_txBufArray[2] = tdm_txBufArray[1] + txStream->txLen;
    
    /* reset tx fifo to ensure there're no data of last transfer in fifo */
    ptReg->txFifoCtrl |= TDM_TX_FIFO_RST;
    /* 8 bits available for TX FIFO would generate DMA TX interrupt*/
    ptReg->txFifoCtrl |= TDM_TX_FIFO_THRESHOLD(7);  

    ret = tdm_SetupTxBuf();
    ret += tdm_QueueInit(&(txStream->txQueue));
    if (ret != DRV_SUCCESS)
    {
        tdm_CleanTxResource();
        return DRV_ERROR;
    }
    tdm_txsem_count_Init();
    return DRV_SUCCESS;
}

/**************************************************************************
* Function:  tdm_IntIsr
* Description: interrupt handle while in int mode
* Parameters:
*   Input:
*           ptInstance : tdm instance
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
static SINT32 tdm_RxComInit(T_ZDrvTdm_Params *params)
{
    T_Tdm_RxStream  *rxStream = &s_tdmRxStream;
    T_Tdm_Instance  *ptInstance = &s_tdmInstance;
    T_Tdm_Reg       *ptReg = NULL;
    SINT32          ret = DRV_SUCCESS;

    if (NULL == params) /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }
	rxStream->rxLen = params->buffersize;

	tdm_rxBufArray[0] = (UINT8 *)(tdmDmaState[TDM_RX].dma_viraddr);
	tdm_rxBufArray[1] = tdm_rxBufArray[0] + rxStream->rxLen;
	tdm_rxBufArray[2] = tdm_rxBufArray[1] + rxStream->rxLen;
    ptReg = ptInstance->ptDev->regs;
    
    /* reset rx fifo to ensure there're no data of last transfer in fifo */
    ptReg->rxFifoCtrl |= TDM_RX_FIFO_RST;         
    /* 8 bits available for RX FIFO would generate DMA RX interrupt*/
    ptReg->rxFifoCtrl |= TDM_RX_FIFO_THRESHOLD(7); 

    ret = tdm_SetupRxBuf();
    ret += tdm_QueueInit(&(rxStream->rxQueue));
    if (ret != DRV_SUCCESS)
    {
        tdm_CleanRxResource();
        return DRV_ERROR;
    }
	
    tdm_rxsem_count_Init();
    rxStream->rxLen = params->buffersize;
//    rxStream->rxIdx = 0;

	rxStream->rxRdIdx = 0;
	rxStream->rxWrIdx = 0;
//    rxStream->rxBufferArray[rxStream->rxIdx].tdmBufStatus = BUFFER_WRITE;
//    rxStream->currentRxBuf = &rxStream->rxBufferArray[rxStream->rxIdx];

	return DRV_SUCCESS;
}

/**************************************************************************
* Function:  tdm_IntIsr
* Description: interrupt handle while in int mode
* Parameters:
*   Input:
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
static SINT32 tdm_IntTxInit()
{
    T_Tdm_Instance  *ptInstance = &s_tdmInstance;
    T_Tdm_TxStream  *txStream = &s_tdmTxStream;
    T_Tdm_Device    *ptDev = NULL;
    T_Tdm_Reg       *ptReg = NULL;

    ptDev = ptInstance->ptDev;
    ptReg = ptDev->regs;
    txStream->txTransmit = tdm_IntTx;
    ptReg->intEn &= ~(TDM_INT_EN_TX_THRESHOLD | TDM_INT_EN_TX_UNDERRUN);  /* disable tx interrupt */
    ptReg->intStatus |= (TDM_INT_TX_THRESHOLD | TDM_INT_TX_UNDERRUN);    /* clear interrupt. must disable interrupt or tdm at first */

    return DRV_SUCCESS;
}

/**************************************************************************
* Function:  tdm_IntIsr
* Description: interrupt handle while in int mode
* Parameters:
*   Input:
*           ptInstance : tdm instance
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
static SINT32 tdm_IntRxInit()
{
#if 0
    T_Tdm_Instance  *ptInstance = &s_tdmInstance;
    T_Tdm_RxStream  *rxStream = &s_tdmRxStream;
    T_Tdm_Device    *ptDev = NULL;
    T_Tdm_Reg       *ptReg = NULL;

    ptDev = ptInstance->ptDev;
    ptReg = ptDev->regs;

    ptReg->intStatus |= (TDM_INT_RX_THRESHOLD | TDM_INT_RX_OVERRUN);     /* clear interrupt */
    ptReg->intEn &= ~(TDM_INT_EN_RX_THRESHOLD | TDM_INT_EN_RX_OVERRUN);   /* disable rx interrupt */

    tdm_rxBuf = (UINT16 *)rxStream->currentRxBuf->buf;
    tdm_rxLen = rxStream->rxLen/ sizeof(UINT16);

    rxStream->rxIdx++;
    ptReg->intEn |= TDM_INT_EN_RX_THRESHOLD;       /* enable rx interrupt */

    if(TDM_INTCTRL_TEST_MODE == TRUE)
    {
        ptReg->intEn |=  TDM_INT_EN_RX_OVERRUN; /* enable rx over interrupt */
    }
#endif
    return DRV_SUCCESS;
}

/**************************************************************************
* Function:  tdm_IntIsr
* Description: interrupt handle while in int mode
* Parameters:
*   Input:
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
static VOID tdm_IntIsr()
{
    T_Tdm_Reg    *ptReg = s_tdmInstance.ptDev->regs;
    T_Tdm_TxStream *tdmTxStream = &s_tdmTxStream;
    const UINT32     uiIntStatus = ptReg->intStatus;
    UINT32 channLable = 0;

    if(TDM_INTCTRL_TEST_MODE) //klocwork 3  INVARIANT_CONDITION.GEN
    {
        if (uiIntStatus & TDM_INT_TX_UNDERRUN)         /* tx err */
        {
            ptReg->intStatus |= TDM_INT_TX_UNDERRUN;   /* clear interrupt */
        }
        if (uiIntStatus & TDM_INT_RX_OVERRUN)           /* rx err */
        {
            ptReg->intStatus |= TDM_INT_RX_OVERRUN;     /* clear interrupt */
        }
    }
	
    if (uiIntStatus & TDM_INT_TX_THRESHOLD)          /* playing */
    {
        ptReg->intEn &= ~TDM_INT_EN_TX_THRESHOLD;  /* disable tx interrupt */
        ptReg->intStatus |= TDM_INT_TX_THRESHOLD;    /* clear interrupt. must disable interrupt or tdm at first */

        channLable = (tdmTxStream->channel == AUDIO_DUAL_CHANNEL) ? 2 : 1;
        tdm_IntTxWriteFifo(channLable);
        if (0 == tdm_txLen)
        {
            tdm_IntTxChangeBuf();
            return;                  /* while no data trans,then return and not enable the interrupt,so the interrupt will not generate*/
        }
        ptReg->intEn |= TDM_INT_EN_TX_THRESHOLD; /* enable tx interrupt */
        if(TDM_INTCTRL_TEST_MODE) //klocwork 3  INVARIANT_CONDITION.GEN
        {
            ptReg->intEn |=  TDM_INT_EN_TX_UNDERRUN; /* enable tx interrupt */
        }
    }
	
    if (uiIntStatus & TDM_INT_RX_THRESHOLD)                /* recording */
    {
        ptReg->intEn &= ~TDM_INT_EN_RX_THRESHOLD;        /* disable rx interrupt */
        ptReg->intStatus |= TDM_INT_RX_THRESHOLD;        /* clear interrupt */

        tdm_IntRxWriteBuf();

        if (0 == tdm_rxLen)
        {
            tdm_IntRxChangeBuf();
        }
        ptReg->intEn |= TDM_INT_EN_RX_THRESHOLD ;        /* enable rx interrupt */
        if(TDM_INTCTRL_TEST_MODE) //klocwork 3  INVARIANT_CONDITION.GEN
        {
            ptReg->intEn |=  TDM_INT_EN_RX_OVERRUN; /* enable tx interrupt */
        }
    }

    if (uiIntStatus & TDM_INT_TX_UNDERRUN)         /* tx err */
    {
        ptReg->intStatus |= TDM_INT_TX_UNDERRUN;   /* clear interrupt */
        ptReg->txFifoCtrl |= TDM_TX_FIFO_RST;          /* reset tx fifo */
    }

    if (uiIntStatus & TDM_INT_RX_OVERRUN)           /* rx err */
    {
        ptReg->intStatus |= TDM_INT_RX_OVERRUN;     /* clear interrupt */
        ptReg->rxFifoCtrl |= TDM_RX_FIFO_RST;           /* reset rx fifo */
    }
}

/****************************************************************************
* Function:  tdm_IntTxDataTrans
* Description: config interrupt handle
* Parameters:
*   Input:
*           None
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
static VOID tdm_IntTxWriteFifo(UINT32 chanlab)
{
    UINT32     uiHdledLen = TDM_TXFL_VAL;
    T_Tdm_Reg  *ptReg = s_tdmInstance.ptDev->regs;
    UINT32 chanlable = 1;
    
    chanlable = 2;

    if (tdm_txLen > TDM_TXFL_VAL*chanlable)
    {
        tdm_txLen -= uiHdledLen*chanlable;
        while (0 < uiHdledLen)
        {
            ptReg->data = *(tdm_txBuf+(chanlable-1)) |  *tdm_txBuf<<16;
            tdm_txBuf += chanlable;
            uiHdledLen--;
        }
    }
    else
    {
        while (0 < tdm_txLen)
        {
            ptReg->data = *(tdm_txBuf+(chanlable-1)) |  *tdm_txBuf<<16;
            tdm_txBuf += chanlable;
            tdm_txLen -= chanlable;
        }
    }
}

/****************************************************************************
* Function:  tdm_IntTxDataTrans
* Description: config interrupt handle
* Parameters:
*   Input:
*           None
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
static VOID tdm_IntRxWriteBuf()
{
    T_Tdm_Reg   *ptReg = s_tdmInstance.ptDev->regs;
    UINT32      uiRxledLen = TDM_RXFL_VAL;
    UINT32      data = 0;

    if (tdm_rxLen > uiRxledLen*2)
    {
        tdm_rxLen -= uiRxledLen*2;
        while (0 < uiRxledLen)
        {
            data = ptReg->data;

            *(tdm_rxBuf + 1) = data & 0xffff;
            *tdm_rxBuf = (data>>16) & 0xffff;
            tdm_rxBuf += 2;
            uiRxledLen--;
        }
    }
    else
    {
        while (0 < tdm_rxLen)
        {
            data = ptReg->data;

            *(tdm_rxBuf + 1) = data & 0xffff;
            *tdm_rxBuf = (data>>16) & 0xffff;
            tdm_rxBuf += 2;
            tdm_rxLen -= 2;
        }
    }
}

/****************************************************************************
* Function:  tdm_IntTxBufChange
* Description: config interrupt handle
* Parameters:
*   Input:
*           None
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
static VOID tdm_IntTxChangeBuf()
{
    T_Tdm_TxStream *tdmTxStream = &s_tdmTxStream;
    T_Tdm_QueueNode *queueNode = NULL;
    UINT32 sem_cnt = 0;
    SINT32 ret = DRV_SUCCESS;
    if (DRV_SUCCESS == tdm_ChangeTxBufferStatus(BUFFER_READ ,BUFFER_NULL)) /*change the BUF state from READ to FULL after the data of the buf is read over,then release the sem*/
    {
        zOss_PutSemaphore(tdmTxStream->txSem);
    }
	
    if (!(tdm_IsQueueEmpty(&tdmTxStream->txQueue)))
    {
        if (tdmTxStream->pauseFlag == FALSE)
        {
            ret = tdm_QueueGet(&tdmTxStream->txQueue, &queueNode);
	     if(queueNode != NULL)
	     {
            ret += tdm_UpdateTxBufferStatus(queueNode->pBuf, BUFFER_READ);
            ret += tdmTxStream->txTransmit(queueNode->pBuf, queueNode->len);
	     }
            if (ret != DRV_SUCCESS)
            {
                return;
            }
            tdmTxStream->Transing = TRUE;
        }
        else
        {
            tdmTxStream->Transing = FALSE;
            return;
        }
    }
    else
    {
        tdmTxStream->Transing = FALSE;
        sem_cnt = zOss_GetSemaphoreCount(tdmTxStream->txSem);
        if (sem_cnt == TDM_OUT_BUF_NUM)
        {
            if (tdmTxStream->p_cb != NULL)
            {
                tdmTxStream->p_cb();
            }
        }

    }
}

/****************************************************************************
* Function:  tdm_IntRxBufChange
* Description: config interrupt handle
* Parameters:
*   Input:
*           None
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
static VOID tdm_IntRxChangeBuf()
{
#if 0
    T_Tdm_RxStream *tdmRxStream = &s_tdmRxStream;

    if (tdmRxStream->rxBufferArray[tdmRxStream->rxIdx].tdmBufStatus == BUFFER_NULL)
    {
        tdm_QueuePut(&(tdmRxStream->rxQueue), (UINT16 *)tdmRxStream->currentRxBuf->buf, (tdmRxStream->rxLen / sizeof(UINT16))); /*ϴյ*/
        tdm_UpdateRxBufferStatus((UINT16*)tdmRxStream->currentRxBuf->buf ,BUFFER_FULL);

        zOss_PutSemaphore(tdmRxStream->rxSem);
        tdmRxStream->rxBufferArray[tdmRxStream->rxIdx].tdmBufStatus = BUFFER_WRITE;
        tdmRxStream->currentRxBuf = &tdmRxStream->rxBufferArray[tdmRxStream->rxIdx];
        tdm_rxBuf = (UINT16 *)tdmRxStream->currentRxBuf->buf;
        tdm_rxLen = tdmRxStream->rxLen / sizeof(UINT16);
        tdmRxStream->rxIdx = (tdmRxStream->rxIdx +1) % TDM_OUT_BUF_NUM;          /*use the buf in turn*/
    }
    else                       /*if the next buf is bing usedtthen rewrite the buf*/
    {
        /* for (rxIndex = 0; rxIndex < TDM_OUT_BUF_NUM ; rxIndex ++)
         {
             if (tdmRxStream->rxBufferArray[rxIndex].tdmBufStatus == BUFFER_WRITE)
             {
                 break;
             }
         }
         tdmRxStream->rxIdx = rxIndex;
         tdmRxStream->currentRxBuf = &tdmRxStream->rxBufferArray[tdmRxStream->rxIdx];*/
        tdm_rxBuf = (UINT16 *)tdmRxStream->currentRxBuf->buf;
        tdm_rxLen = tdmRxStream->rxLen/ sizeof(UINT16);
    }
#endif
}

/****************************************************************************
* Function:  isr_tdm
* Description: config interrupt handle
* Parameters:
*   Input:
*           None
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
static VOID isr_tdm(VOID)
{
    tdm_IntIsr();
}

/**************************************************************************
* Function:  tdm_IntTx
* Description: trans data from ram to fifo  while in int mode
* Parameters:
*   Input:
*           pBuf : buffer pointer
*           uiLen : buffer length
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
* Others:
****************************************************************************/
static SINT32 tdm_IntTx(const UINT16 *pBuf, UINT32 uiLen)
{
    T_Tdm_Instance  *ptInstance = NULL;
    T_Tdm_Device    *ptDev = NULL;
    T_Tdm_Reg       *ptReg = NULL;


	/*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
    if (NULL == pBuf || 0 == uiLen || TDM_MAX_TX_BUF_SIZE < uiLen) 
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }

    ptInstance = &s_tdmInstance;
    if (ptInstance->tdmStatus != TDM_PLAY_INUSE && ptInstance->tdmStatus != TDM_BOTH_INUSE)
    {
        return DRV_ERR_NOT_STARTED;
    }

    tdm_txBuf = pBuf;
    tdm_txLen = uiLen;

    ptDev = ptInstance->ptDev;
    ptReg = ptDev->regs;
    ptReg->txFifoCtrl |= TDM_RX_FIFO_THRESHOLD(7);    /*DMA MODE is also necessry*/
    /* clear tx interrupt */
    ptReg->intStatus |= (TDM_INT_TX_THRESHOLD | TDM_INT_TX_UNDERRUN);    /* clear interrupt. must disable interrupt or tdm at first */
    ptReg->intEn |= TDM_INT_EN_RX_THRESHOLD;        /* enable tx interrupt */
    if(TDM_INTCTRL_TEST_MODE) //klocwork 3  INVARIANT_CONDITION.GEN
    {
        ptReg->intEn |=  TDM_INT_EN_TX_UNDERRUN; /* enable tx under interrupt */
    }

    return DRV_SUCCESS;
}

/**************************************************************************
* Function:  tdm_DmaTxIntCb
* Description: callback function while playing
* Parameters:
*   Input:
*           tIntStatus : dma status
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
* Others:
****************************************************************************/
#if 0
static VOID tdm_DmaTxIntCb(VOID)  //(T_ZDrvDma_IntStatus tIntStatus)
{

    T_Tdm_Reg    *ptReg = s_tdmInstance.ptDev->regs;
    const UINT32     uiIntStatus = ptReg->intStatus;

    zDrv_ASSERT(tIntStatus < MAX_DMA_INT);
    if (tIntStatus >=  MAX_DMA_INT)
    {
        return;
    }
	
    switch (tIntStatus)
    {
        case DMA_INT_END:
        {
            tdm_IntTxChangeBuf();
            if (uiIntStatus & TDM_INT_TX_UNDERRUN)         /* tx err */
            {
                ptReg->intStatus |= TDM_INT_TX_UNDERRUN;   /* clear interrupt */
                ptReg->txFifoCtrl |= TDM_TX_FIFO_RST;          /* reset tx fifo */
            }
            break;
        }

        case DMA_INT_ERR:
            break;

        default:
            break;
    }

}
#endif
/**************************************************************************
* Function:  tdm_DmaRxIntCb
* Description: record callback function in dma mode,while one buffer is full, call this function
* Parameters:
*   Input:
*           ptInstance : tdm instance
*           tIntStatus : dma status
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
#if 0
static VOID tdm_DmaRxIntCb(T_Tdm_Instance *ptInstance)
{

    T_Tdm_RxStream *tdmStream = &s_tdmRxStream;
    UINT8 rxIndex = 0;

    T_Tdm_Reg    *ptReg = s_tdmInstance.ptDev->regs;
    const UINT32     uiIntStatus = ptReg->intStatus;
    zDrv_ASSERT((ptInstance != NULL) && (tIntStatus < MAX_DMA_INT));
    if ((NULL == ptInstance) || (tIntStatus >=  MAX_DMA_INT))
    {
        return;
    }

    switch (tIntStatus)
    {
        case DMA_INT_END:
        {			
			TOS_DCACHE_INVALIDATE_RANGE((UINT32)tdmStream->currentRxBuf->buf, (UINT16)tdmStream->rxLen);
			
            if (tdmStream->rxBufferArray[tdmStream->rxIdx].tdmBufStatus == BUFFER_NULL)
            {
                tdm_QueuePut(&tdmStream->rxQueue, (UINT16 *)tdmStream->currentRxBuf->buf, (tdmStream->rxLen / sizeof(UINT16))); /*put the full buf to the queue*/
                tdm_UpdateRxBufferStatus((UINT16*)tdmStream->currentRxBuf->buf, BUFFER_FULL);

                zOss_PutSemaphore(tdmStream->rxSem);          /*if the next buf status is null,then release the sem so the uplayer can get the buf to read*/
                tdm_DmaRx(ptInstance, tdmStream->rxIdx);
            }
            else
            {
                /*ϲڿbufд*/
                for (rxIndex = 0; rxIndex < TDM_OUT_BUF_NUM; rxIndex++)                      /*if the next buf is bing usedtthen rewrite the buf**/
                {
                    if (tdmStream->rxBufferArray[rxIndex].tdmBufStatus == BUFFER_WRITE)        /*it is necessary to find the index*/
                    {
                        if (uiIntStatus & TDM_INT_RX_OVERRUN)           /* rx err */
                        {
                            ptReg->intStatus |= TDM_INT_RX_OVERRUN;     /* clear interrupt */
                            ptReg->rxFifoCtrl |= TDM_RX_FIFO_RST;           /* reset rx fifo */
                            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm_DmaRxIntCb rxerr1 uiIntStatus=0x%x",uiIntStatus);
                        }
                        break;
                    }
                }
                tdmStream->rxIdx = rxIndex;
                tdm_DmaRx(ptInstance, rxIndex);
            }
            tdmStream->rxIdx = (tdmStream->rxIdx +1) % TDM_OUT_BUF_NUM;

            if (uiIntStatus & TDM_INT_RX_OVERRUN)           /* rx err */
            {
                ptReg->intStatus |= TDM_INT_RX_OVERRUN;     /* clear interrupt */
                ptReg->rxFifoCtrl |= TDM_RX_FIFO_RST;           /* reset rx fifo */
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm_DmaRxIntCb rxerr2 uiIntStatus=0x%x",uiIntStatus);
            }
            break;
        }
        case DMA_INT_ERR:
            break;
        default:
            break;
    }

}
#endif
/**************************************************************************
* Function:  dmacb_tdm_rx
* Description: set call back function
* Parameters:
*   Input:
*           tIntStatus : tdm instance
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
static VOID dmacb_tdm_rx(void *data)
{
	UINT32 sem_cnt = 0;
	T_Tdm_RxStream *tdmRxStream = &s_tdmRxStream;

    if(tdmRxStream->rxBufferArray == NULL){
        return;
    }

	tdm_UpdateRxBufferStatus((UINT16 *)(tdmRxStream->rxBufferArray[tdmRxStream->rxWrIdx].buf), BUFFER_FULL); 
	tdmRxStream->rxWrIdx = (tdmRxStream->rxWrIdx + 1) % TDM_OUT_BUF_NUM;
	sem_cnt = zOss_GetSemaphoreCount(tdmRxStream->rxSem);
	if(sem_cnt < TDM_OUT_BUF_NUM) {
		zOss_PutSemaphore(tdmRxStream->rxSem); 
	} else {
		tdmRxStream->rxRdIdx = (tdmRxStream->rxRdIdx + 1) % TDM_OUT_BUF_NUM;
	}

   // tdm_DmaRxIntCb(&s_tdmInstance, tIntStatus);
}

/****************************************************************************
* Function:  dmacb_tdm_tx
* Description: config interrupt callback
* Parameters:
*   Input:
*           None
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
static VOID dmacb_tdm_tx(void *data)
{
	UINT32 sem_cnt = 0;
	T_Tdm_TxStream *tdmTxStream = &s_tdmTxStream;

    if(tdmTxStream->txBufferArray == NULL){
        return;
    }
    
	tdm_UpdateTxBufferStatus((UINT16 *)(tdmTxStream->txBufferArray[tdmTxStream->txRdIdx].buf), BUFFER_NULL);       /*ϲ*pBufдݺ󣬸ıBUF״̬ΪFULL*/
	tdmTxStream->txRdIdx = (tdmTxStream->txRdIdx + 1)%TDM_OUT_BUF_NUM;
	sem_cnt = zOss_GetSemaphoreCount(tdmTxStream->txSem);
	if(sem_cnt < TDM_OUT_BUF_NUM) {
		zOss_PutSemaphore(tdmTxStream->txSem);
	} else {
		tdmTxStream->txWrIdx = (tdmTxStream->txWrIdx + 1) % TDM_OUT_BUF_NUM;
	}

  //  tdm_DmaTxIntCb(tIntStatus);
}
#if 0
/**************************************************************************
* Function:  tdm_DmaRx
* Description: use buffer idx to record data
* Parameters:
*   Input:
*           tIntStatus : tdm instance
*           idx : buffer index
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/
static SINT32 tdm_DmaRx(T_Tdm_Instance *ptInstance, UINT8 idx)
{
    SINT32 ret = DRV_SUCCESS;
    T_Tdm_RxStream *tdmStream = &s_tdmRxStream;
    T_ZDrvDma_ChannelDef tChanDef = {0};
    //zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"tdm_DmaRx \n");

    zDrv_ASSERT((ptInstance != NULL) && (idx < TDM_OUT_BUF_NUM));
    if (NULL == ptInstance || idx >= TDM_OUT_BUF_NUM)
    {
        return DRV_ERR_INVALID_PARAM;
    }
    
    tdmStream->rxBufferArray[idx].tdmBufStatus = BUFFER_WRITE ;
    tdmStream->currentRxBuf = &tdmStream->rxBufferArray[idx];

	TOS_DCACHE_INVALIDATE_RANGE((UINT32)tdmStream->currentRxBuf->buf, (UINT16)tdmStream->rxLen);
    //dwCnt С64k
    tChanDef.SrcAddr = (UINT32)&(ptInstance->ptDev->regs->data);
    tChanDef.DestAddr = (UINT32)tdmStream->currentRxBuf->buf;
    if(gTdmCntChangeRx>= -80)
    {
        tChanDef.Count = ((UINT16)tdmStream->rxLen) + gTdmCntChangeRx*2;//*sizeof(UINT32);//uiLen*sizeof(UINT16)
        gTdmCntChangeRx = 0;
    }
    else
    {
        tChanDef.Count = ((UINT16)tdmStream->rxLen) - 80*2;//*sizeof(UINT32);//uiLen*sizeof(UINT16)
        gTdmCntChangeRx = gTdmCntChangeRx+80;
    }

    tChanDef.LLI = NULL;

    tChanDef.CONTROL.BurstReqMod = DMA_PERIPHERAL_REQ;
    tChanDef.CONTROL.SrcMod = DMA_ADDRMOD_FIFO;
    tChanDef.CONTROL.DestMod = DMA_ADDRMOD_RAM;
    tChanDef.CONTROL.IrqMod = DMA_ALL_IRQ_ENABLE;
    tChanDef.CONTROL.SrcBurstSize = DMA_BURST_SIZE_32BIT;
    tChanDef.CONTROL.SrcBurstLen = DMA_BURST_LEN_8;
    tChanDef.CONTROL.DestBurstSize = DMA_BURST_SIZE_32BIT;
    tChanDef.CONTROL.IntSel = DMA_INT_TO_PS;
	
    ret = zDrvDma_ConfigChannel(ptInstance->DmaRxID, tChanDef);
    if (ret != DRV_SUCCESS)
    {
        zDrvDma_DeAllocChannel(ptInstance->DmaRxID);
        return ret;
    }
    //zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"tdm_DmaRx zDrvDma_ConfigChannel ret %d\n", ret);

    ret = zDrvDma_StartChannel(ptInstance->DmaRxID, ptInstance->fDmaCbRx, FALSE);
    if (ret != DRV_SUCCESS)
    {
        zDrvDma_DeAllocChannel(ptInstance->DmaRxID);
        return ret;
    }
    //zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"tdm_DmaRx zDrvDma_StartChannel ret %d\n", ret);
    return ret;
}
#endif
/**************************************************************************
* Function:  tdm_DmaRxInit
* Description: dma  init while recording
* Parameters:
*   Input:
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
* Others:
****************************************************************************/
static SINT32 tdm_DmaRxInit()
{
    T_Tdm_Instance  *ptInstance = NULL;
    T_Tdm_RxStream *tdmStream = &s_tdmRxStream;
    ptInstance = &s_tdmInstance;

	struct dma_async_tx_descriptor *desc = NULL;
	dma_cap_mask_t mask;
	
	dma_cap_zero(mask);
	dma_cap_set(DMA_CYCLIC, mask);
	
	dma_channel_def temp[3];
	int 	i;
	memset(temp, 0, sizeof(temp));
	tdmDmaState[TDM_RX].channel = DMA_CH_TDM_RX0;
	tdmDmaState[TDM_RX].ch = dma_request_channel(mask, zx29_dma_filter_fn, (void*)(tdmDmaState[TDM_RX].channel));
	
	for (i = 0; i < TDM_OUT_BUF_NUM; i++) {
		temp[i].dest_addr 	= tdmDmaState[TDM_RX].dma_phyaddr  + (tdmStream->rxLen) * i;
		temp[i].src_addr	= ZX29_TDM_PHYS + 0x24;
		temp[i].dma_control.tran_mode = TRAN_PERI_TO_MEM;
		temp[i].dma_control.src_burst_len 	= DMA_BURST_LEN_8;
		temp[i].count		= tdmStream->rxLen;
		//		temp[i].callback	= (dma_callback_func)dma_cb;
		/*
			temp[i].dma_control.tran_mode 		= (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
													? TRAN_MEM_TO_PERI : TRAN_PERI_TO_MEM);
			temp[i].dma_control.src_burst_len 	= (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
													? DMA_BURST_LEN_16 : DMA_BURST_LEN_8);
		*/
		temp[i].dma_control.src_burst_size 	= DMA_BURST_SIZE_32BIT;
		temp[i].dma_control.dest_burst_size = DMA_BURST_SIZE_32BIT;
		temp[i].dma_control.dest_burst_len 	= DMA_BURST_LEN_8;
		temp[i].dma_control.irq_mode 		= DMA_ALL_IRQ_ENABLE;
		temp[i].link_addr 		= 1;
	}

	temp[TDM_OUT_BUF_NUM - 1].link_addr 	= 0;

	dmaengine_slave_config(tdmDmaState[TDM_RX].ch, (struct dma_slave_config*)&temp);

	desc = tdmDmaState[TDM_RX].ch->device->device_prep_dma_cyclic(
           tdmDmaState[TDM_RX].ch,
           tdmDmaState[TDM_RX].dma_phyaddr, tdmStream->rxLen * 3,
           tdmStream->rxLen,
           DMA_DEV_TO_MEM,
           NULL);

	desc->callback = (dma_async_tx_callback)dmacb_tdm_rx;
	desc->callback_param = NULL;

	dmaengine_submit(desc);  

   	dma_async_issue_pending(tdmDmaState[TDM_RX].ch);
    return DRV_SUCCESS;
}

/**************************************************************************
* Function:  tdm_DmaTxInit
* Description: init dma trans
* Parameters:
*   Input:
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
* Others:
****************************************************************************/
static SINT32 tdm_DmaTxInit()
{
    T_Tdm_Instance  *ptInstance = NULL;
    ptInstance = &s_tdmInstance;
    T_Tdm_TxStream *txStream = &s_tdmTxStream;

	struct dma_async_tx_descriptor *desc = NULL;
	dma_cap_mask_t mask;	
	dma_cap_zero(mask);
	dma_cap_set(DMA_CYCLIC, mask);	
	dma_channel_def temp[3];
	int 	i;
	memset(temp, 0, sizeof(temp));
	tdmDmaState[TDM_TX].channel = DMA_CH_TDM_TX0;	
	tdmDmaState[TDM_TX].ch = dma_request_channel(mask, zx29_dma_filter_fn, (void*)(tdmDmaState[TDM_TX].channel));
	
	for (i = 0; i < TDM_OUT_BUF_NUM; i++) {
		temp[i].src_addr 	= tdmDmaState[TDM_TX].dma_phyaddr + (txStream->txLen) * i;
		temp[i].dest_addr	= ZX29_TDM_PHYS + 0x24;
		temp[i].dma_control.tran_mode = TRAN_MEM_TO_PERI;
		temp[i].dma_control.src_burst_len 	= DMA_BURST_LEN_8;
		temp[i].count		= txStream->txLen;
		//		temp[i].callback	= (dma_callback_func)dma_cb;
		/*
			temp[i].dma_control.tran_mode 		= (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
													? TRAN_MEM_TO_PERI : TRAN_PERI_TO_MEM);
			temp[i].dma_control.src_burst_len 	= (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
													? DMA_BURST_LEN_16 : DMA_BURST_LEN_8);
		*/
		temp[i].dma_control.src_burst_size 	= DMA_BURST_SIZE_32BIT;
		temp[i].dma_control.dest_burst_size = DMA_BURST_SIZE_32BIT;
		temp[i].dma_control.dest_burst_len 	= DMA_BURST_LEN_8;
		temp[i].dma_control.irq_mode 		= DMA_ALL_IRQ_ENABLE;
		temp[i].link_addr 		= 1;
	}

	temp[TDM_OUT_BUF_NUM - 1].link_addr 	= 0;

	dmaengine_slave_config(tdmDmaState[TDM_TX].ch, (struct dma_slave_config*)&temp);

	desc = tdmDmaState[TDM_TX].ch->device->device_prep_dma_cyclic(
           tdmDmaState[TDM_TX].ch,
           tdmDmaState[TDM_TX].dma_phyaddr, txStream->txLen * 3,
           txStream->txLen,
           DMA_MEM_TO_DEV,
           NULL);

	desc->callback = (dma_async_tx_callback)dmacb_tdm_tx;
	desc->callback_param = NULL;

	dmaengine_submit(desc);   
    return DRV_SUCCESS;
}
#if 0
/**************************************************************************
* Function:  tdm_DmaRx
* Description: playing dual audio data dma setting
* Parameters:
*   Input:
*           pBuf : buffer index
*           uiLen : buffer size
*   Output:
*           None
* Returns:
*           DRV_SUCCESS:operate success
* Others:
****************************************************************************/
static SINT32 tdm_DmaTx(const UINT16 *pBuf, UINT32  uiLen)
{
    SINT32 ret = DRV_SUCCESS;
    T_Tdm_Instance  *ptInstance = &s_tdmInstance;
    T_Tdm_TxStream  *txStream = &s_tdmTxStream;
    T_ZDrvDma_ChannelDef tChanDef = {0};

    zDrv_ASSERT((pBuf != NULL) && (uiLen != 0) && (uiLen <= TDM_MAX_TX_BUF_SIZE));
    if (NULL == pBuf || 0 == uiLen || TDM_MAX_TX_BUF_SIZE < uiLen)
    {
        return DRV_ERR_INVALID_PARAM;
    }
	//zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm_DmaTx buf 0x%x! \r\n", *pBuf);
	TOS_CACHE_CLEAN_RANGE((unsigned long)pBuf, uiLen * sizeof(UINT16));//xiu
	
    tChanDef.SrcAddr = (UINT32)(pBuf);
    tChanDef.DestAddr = (UINT32)&(ptInstance->ptDev->regs->data);

    if(gTdmCntChangeTx >= -80)
    {
        tChanDef.Count = uiLen*sizeof(UINT16) + gTdmCntChangeTx * 2;//ֽڴС
        gTdmCntChangeTx = 0;
    }
    else
    {
        tChanDef.Count = uiLen*sizeof(UINT16) - 80 * 2;//*sizeof(UINT32);//uiLen*sizeof(UINT16)
        gTdmCntChangeTx = gTdmCntChangeTx + 80;
    }

    tChanDef.LLI = NULL;

    tChanDef.CONTROL.BurstReqMod = DMA_PERIPHERAL_REQ;
    tChanDef.CONTROL.SrcMod = DMA_ADDRMOD_RAM;
    tChanDef.CONTROL.DestMod = DMA_ADDRMOD_FIFO;
    tChanDef.CONTROL.IrqMod = DMA_ALL_IRQ_ENABLE;
    tChanDef.CONTROL.SrcBurstSize = DMA_BURST_SIZE_32BIT;
    tChanDef.CONTROL.SrcBurstLen = DMA_BURST_LEN_8;   
    tChanDef.CONTROL.DestBurstSize = DMA_BURST_SIZE_32BIT;
    tChanDef.CONTROL.IntSel = DMA_INT_TO_PS;

    ret = zDrvDma_ConfigChannel(ptInstance->DmaTxID, tChanDef);
    if (ret != DRV_SUCCESS)
    {
        zDrvDma_DeAllocChannel(ptInstance->DmaTxID);
        return ret;
    }

    ret = zDrvDma_StartChannel(ptInstance->DmaTxID, ptInstance->fDmaCbTx, FALSE);
    if (ret != DRV_SUCCESS)
    {
        zDrvDma_DeAllocChannel(ptInstance->DmaTxID);
        return ret;
    }

    return DRV_SUCCESS;
}
#endif
static VOID tdm_InitInt()
{
    static T_Tdm_Device s_TdmDev;

    if (s_bTdmInitInt)
        return;
	
    memset(&s_TdmDev, 0, sizeof(s_TdmDev));
    s_TdmDev.intline = TDM_INT;
    s_TdmDev.intprio = 15;
    s_TdmDev.regs = (T_Tdm_Reg*)TDM_REG_BASE;            
    s_tdmInstance.ptDev = &s_TdmDev;
    s_bTdmInitInt = TRUE;
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(tdm_InitInt(TDM_REG_BASE) = %x",TDM_REG_BASE); 
}

SINT32 zDrvTDM_Init(VOID)
{
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"zDrvTDM__Init \n");

    s_tdmRxStream.rxSem = zOss_CreateSemaphore("tdm_rx_sem", 0); 
	/*klocwork 3  INVARIANT_CONDITION.UNREACH delete if*/
	/*
    if(NULL == s_tdmRxStream.rxSem)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "create tdm_rx_sem err \n");
        return DRV_ERROR;
    }*/

    s_tdmTxStream.txSem = zOss_CreateSemaphore("tdm_tx_sem", 0);
	/*klocwork 3  INVARIANT_CONDITION.UNREACH delete if*/
	/*
	if(NULL == s_tdmTxStream.txSem)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "create tdm_tx_sem err \n");
        return DRV_ERROR;
    }*/
	
    return DRV_SUCCESS;
}

SINT32 tdm_rxsem_count_Init()
{
    UINT32 i = 0;
    UINT32 semCount = 0;
    SINT32 Count = 0;

    T_Tdm_RxStream  *rxStream = &s_tdmRxStream;
//rx sem count 0
    semCount = zOss_GetSemaphoreCount(rxStream->rxSem);
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm rxStream->rxSem count=%d \r\n",semCount);

    Count = semCount;
    if (Count > 0)
    {
        for (i = 0; i < Count; i++)
        {
            zOss_GetSemaphore(rxStream->rxSem, ZOSS_NO_WAIT);
        }
    }
	
    return DRV_SUCCESS;
}

SINT32 tdm_txsem_count_Init()
{
    UINT32 i = 0;
    UINT32 semCount = 0;
    SINT32 Count = 0;

    T_Tdm_TxStream  *txStream = &s_tdmTxStream;
    //tx sem count 3

    semCount = zOss_GetSemaphoreCount(txStream->txSem);
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "tdm txStream->txSem count=%d ",semCount);

    Count=TDM_UPLINK_SEM_NUM-semCount;
    if (Count > 0)
    {
        for (i = 0; i < Count; i++)
        {
            zOss_PutSemaphore(txStream->txSem);
        }
    }
    else if (Count < 0)
    {
        Count = 0 - Count;
        for (i = 0; i < Count; i++)
        {
            zOss_GetSemaphore(txStream->txSem, ZOSS_NO_WAIT);
        }
    }
    return DRV_SUCCESS;

}

/****************************************************************************
* Function:  zDrvTDM_Open
* Description:open tdm device
* Parameters:
*   Input:
*           tdmTransFunc : callback function pointer
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
* Others:
 ****************************************************************************/
SINT32 zDrvTDM_Open(T_ZDrvI2s_TransMode tdmTransMode)
{
    T_Tdm_Instance  *ptInstance = NULL;
    T_Tdm_Device    *ptDev = NULL;
    SINT32 ret = DRV_SUCCESS;
    SINT32	returnValue = 0;
	UINT32 bufSize;
	UINT32 txBufSize;
	UINT32 rxBufSize;
    ptInstance = &s_tdmInstance;
    if (ptInstance->tdmStatus == TDM_OPEN || ptInstance->tdmStatus == TDM_PLAY_INUSE || ptInstance->tdmStatus == TDM_RECORD_INUSE || ptInstance->tdmStatus == TDM_BOTH_INUSE)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "howard zDrvTDM_Open err tdmStatus=%d",ptInstance->tdmStatus);
		zDrv_ASSERT(0);
        return DRV_ERR_OPEN_TIMES;
    }

    #ifdef _USE_PSM
    zDrvPow_SetDevActive((UINT32)IDLE_FLAG_TDM);
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Open Active(IDLE_FLAG_TDM)\n");
    #endif

    tdm_InitInt();

    tdm_ClkInit();

    ptDev = ptInstance->ptDev;
    ptInstance->tdmTransMode = tdmTransMode;
    
	if((g_voiceVar.voiceMode == VOICE_GSM_MODE)||(g_voiceVar.voiceMode == VOICE_LTE_MODE)|| \
		(g_voiceVar.voiceMode == VOICE_WCDMA_MODE)||(g_voiceVar.voiceMode == VOICE_TD_MODE)) {
		bufSize = TDM_MAX_VOICE_LTE_MEM_SIZE * 6;
		txBufSize = TDM_MAX_VOICE_LTE_MEM_SIZE;
		rxBufSize = TDM_MAX_VOICE_LTE_MEM_SIZE;
	} else {
		bufSize = (TDM_MAX_TX_BUF_SIZE + TDM_MAX_RX_BUF_SIZE) * 3;
		txBufSize = TDM_MAX_TX_BUF_SIZE;
		rxBufSize = TDM_MAX_RX_BUF_SIZE;
	}

	if((g_voiceVar.voiceMode == VOICE_GSM_MODE)||(g_voiceVar.voiceMode == VOICE_LTE_MODE)|| \
		(g_voiceVar.voiceMode == VOICE_WCDMA_MODE)||(g_voiceVar.voiceMode == VOICE_TD_MODE))
	{
		tdmDmaState[TDM_TX].totalbuffsize = TDM_MAX_VOICE_LTE_MEM_SIZE * 3;	
		tdmDmaState[TDM_RX].totalbuffsize = TDM_MAX_VOICE_LTE_MEM_SIZE * 3;		
		txBufSize = TDM_MAX_VOICE_LTE_MEM_SIZE;
		rxBufSize = TDM_MAX_VOICE_LTE_MEM_SIZE;
		tdmDmaState[TDM_TX].singlebuffsize = TDM_MAX_VOICE_LTE_MEM_SIZE;
		tdmDmaState[TDM_RX].singlebuffsize = TDM_MAX_VOICE_LTE_MEM_SIZE;
	}
	else
	{
		tdmDmaState[TDM_TX].totalbuffsize = TDM_MAX_TX_BUF_SIZE * 3;	
		tdmDmaState[TDM_RX].totalbuffsize = TDM_MAX_RX_BUF_SIZE * 3;
		txBufSize = TDM_MAX_TX_BUF_SIZE;
		rxBufSize = TDM_MAX_RX_BUF_SIZE;
		tdmDmaState[TDM_TX].singlebuffsize = TDM_MAX_TX_BUF_SIZE;
		tdmDmaState[TDM_RX].singlebuffsize = TDM_MAX_RX_BUF_SIZE;
	}

	tdmDmaState[TDM_TX].dma_viraddr = dma_alloc_writecombine(&tdm_device.dev, tdmDmaState[TDM_TX].totalbuffsize,
	                                   &(tdmDmaState[TDM_TX].dma_phyaddr), GFP_KERNEL); 
	
	tdmDmaState[TDM_RX].dma_viraddr = dma_alloc_writecombine(&tdm_device.dev, tdmDmaState[TDM_RX].totalbuffsize,
	                                   &(tdmDmaState[TDM_RX].dma_phyaddr), GFP_KERNEL); 
	
	if ((NULL == tdmDmaState[TDM_TX].dma_viraddr)||(NULL == tdmDmaState[TDM_RX].dma_viraddr))
	{
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "[XK]BUF_NUM Malloc Failed! %p,%p,%x,%x\n", tdmDmaState[TDM_TX].dma_viraddr, tdmDmaState[TDM_RX].dma_viraddr, \
								tdmDmaState[TDM_TX].dma_phyaddr, tdmDmaState[TDM_RX].dma_phyaddr);
		return DRV_ERR_MEM_ALLOC;
	}
	else		
	{	
/*		tdm_txBufArray[tId][0] = (UINT8 *)(tdmDmaState[tId][TDM_TX].dma_viraddr);
		tdm_txBufArray[tId][1] = tdm_txBufArray[tId][0] + txBufSize;
		tdm_txBufArray[tId][2] = tdm_txBufArray[tId][1] + txBufSize;
		tdm_rxBufArray[tId][0] = (UINT8 *)(tdmDmaState[tId][TDM_RX].dma_viraddr);
		tdm_rxBufArray[tId][1] = tdm_rxBufArray[tId][0] + rxBufSize;
		tdm_rxBufArray[tId][2] = tdm_rxBufArray[tId][1] + rxBufSize;

		tdmDmaState[tId].dma_tx_phyaddr[1] = tdmDmaState[tId].dma_tx_phyaddr[0] + txBufSize;
		tdmDmaState[tId].dma_tx_phyaddr[2] = tdmDmaState[tId].dma_tx_phyaddr[1] + txBufSize;
		tdmDmaState[tId].dma_rx_phyaddr[1] = tdmDmaState[tId].dma_rx_phyaddr[0] + rxBufSize;
		tdmDmaState[tId].dma_rx_phyaddr[2] = tdmDmaState[tId].dma_rx_phyaddr[1] + rxBufSize;
*/
	}

    ret = tdm_Reset();
	/*klocwork 3  INVARIANT_CONDITION.UNREACH delete if*/
    /*
    if (ret != DRV_SUCCESS)
    {
        #ifdef _USE_PSM
        if (s_tdmInstance.tdmStatus == TDM_IDLE)
        {
            zDrvPow_SetDevIdle((UINT32)IDLE_FLAG_TDM);
        }
        #endif
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "[TDM]tdm_ResetFailed!!!!\n");
		zDrv_ASSERT(0);
        return DRV_ERROR;
    }*/
    /* value of control register is 0 after reseted */
    ptInstance->tCfg.bMaster  = FALSE;
    if (ptInstance->tdmTransMode == TRANS_INT_MODE)
    {
        returnValue = zDrvInt_InstallIsr(ptDev->intline, (VOID *)isr_tdm, "tdm",INT_HIGHLEVEL);

        if (returnValue < 0)
        {
            #ifdef _USE_PSM
            if (s_tdmInstance.tdmStatus == TDM_IDLE)
            {
                zDrvPow_SetDevIdle((UINT32)IDLE_FLAG_TDM);
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Open DevIdle(IDLE_FLAG_TDM)\n");
            }
            #endif
            return DRV_ERROR;
        }
        ptDev->pid = (PROCESS)returnValue;
    }
	
#ifdef _USE_PSM
	zDrvPow_SetDevActive((UINT32) IDLE_FLAG_TDM);
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Open DevActive(IDLE_FLAG_TDM)\n");
#endif

    ptInstance->tdmStatus = TDM_OPEN;
    return DRV_SUCCESS;
}

SINT32 zDrvTDM_Reset()
{
	SINT32 ret = DRV_SUCCESS;

	ret = tdm_Reset();
	
	return ret;
}

/****************************************************************************
* Function:  zDrvTDM_Close
* Description:close  tdm device
* Parameters:
*   Input:
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
* Others:
 ****************************************************************************/
SINT32 zDrvTDM_Close()
{
    T_Tdm_Instance  *ptInstance = NULL;
    SINT32 ret = DRV_SUCCESS;
	
    ptInstance = &s_tdmInstance;
    if (ptInstance->tdmStatus != TDM_OPEN)
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Close TDM, tdmStatus=%d != TDM_OPEN",s_tdmInstance.tdmStatus);
        return DRV_ERR_NOT_OPENED;
    }
	
    if (ptInstance->tdmTransMode == TRANS_INT_MODE)
        /* reset tx/rx fifo to ensure there're no data of last transfer in fifo */
        /* set the FIFO int threshold */
    {
        ret += zDrvInt_UninstallIsr(ptInstance->ptDev->intline);
        /* enable tdm; enable tx */

    }

	if (tdmDmaState[TDM_TX].dma_viraddr != NULL) {
	dma_free_writecombine(&tdm_device.dev, tdmDmaState[TDM_TX].totalbuffsize,
		                      tdmDmaState[TDM_TX].dma_viraddr, tdmDmaState[TDM_TX].dma_phyaddr);
	tdmDmaState[TDM_TX].dma_viraddr = NULL;
	}
	if (tdmDmaState[TDM_RX].dma_viraddr != NULL) {
		dma_free_writecombine(&tdm_device.dev, tdmDmaState[TDM_RX].totalbuffsize,
			                      tdmDmaState[TDM_RX].dma_viraddr, tdmDmaState[TDM_RX].dma_phyaddr);
		tdmDmaState[TDM_RX].dma_viraddr = NULL;
	}

    ret = tdm_Reset();
	/*klocwork 3  INVARIANT_CONDITION.UNREACH delete if*/
	/*
    if (ret != DRV_SUCCESS)
    {
        return DRV_ERROR;
    }
	*/
    tdm_CloseClk();
    tdm_ClkDeinit();
    ptInstance->tdmStatus = TDM_IDLE;

	/*klocwork 3  INVARIANT_CONDITION.UNREACH ɾifҵ˳*/
#ifdef _USE_PSM
    zDrvPow_SetDevIdle((UINT32)IDLE_FLAG_TDM);
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Open DevIdle(IDLE_FLAG_TDM)\n");
#endif
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Close, tdmStatus=%d", s_tdmInstance.tdmStatus);

    return DRV_SUCCESS;
}

/****************************************************************************
* Function:  zDrvTDM_Write_Start
* Description: start to trans data,use in playing
* Parameters:
*   Input:
*           buf_size : one buffer is how many bytes
*           ptCfg : config param
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
* Others:
 ****************************************************************************/
SINT32 zDrvTDM_Write_Prepare(T_ZDrvTdm_Cfg * ptCfg)
{
    T_Tdm_Instance  *ptInstance = &s_tdmInstance;
    T_Tdm_Reg       *ptReg = NULL;
    SINT32           ret = DRV_SUCCESS;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Write_Prepare,ptCfg=%p! \r\n",ptCfg);

    if (NULL == ptCfg)
    {
        return DRV_ERR_INVALID_PARAM;
    }

    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "[TDM]zDrvTDM_Write_Prepare! \r\n");
    ptReg = ptInstance->ptDev->regs;
    ret = tdm_TxCheckState(ptCfg);
    if (ret != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Write_Prepare:  tdm_TxCheckState err ret=%d\r\n",ret);
        return DRV_ERROR;
    }

    ptReg->processCtrl |= TDM_EN;

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Write_Prepare timing ctrl) = %x",  (s_tdmInstance.ptDev)->regs->timingCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Write_Prepare txFifoCtrl) = %x", (s_tdmInstance.ptDev)->regs->txFifoCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Write_Prepare frame ctrl) = %x",(s_tdmInstance.ptDev)->regs->frameCntr);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Write_Prepare txFifoCtrl) = %x", (s_tdmInstance.ptDev)->regs->txFifoCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Write_Prepare process ctrl) = %x", (s_tdmInstance.ptDev)->regs->processCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Write_Prepare reg intEn) = %x",  (s_tdmInstance.ptDev)->regs->intEn);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Write_Prepare reg intStatus) = %x\n", (s_tdmInstance.ptDev)->regs->intStatus);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Write_Prepare reg writeCntCtrl) = %x\n",  (s_tdmInstance.ptDev)->regs->writeCntCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Write_Prepare reg writeCntr) = %x\n",  (s_tdmInstance.ptDev)->regs->writeCntr);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Write_Prepare reg writeCntr2) = %x\n",  (s_tdmInstance.ptDev)->regs->writeCntr2);
    return DRV_SUCCESS;
}

SINT32 zDrvTDM_Write_Start_Do(T_ZDrvTdm_Params *params)
{
    T_Tdm_Instance  *ptInstance = &s_tdmInstance;
    T_Tdm_TxStream  *txStream = &s_tdmTxStream;
    T_Tdm_Reg       *ptReg = NULL;
    SINT32           ret = DRV_SUCCESS;
    SINT32           waitTime = 0;
	/*klocwork 3  INVARIANT_CONDITION.UNREACH assertƵif*/
    if (NULL == params)
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }

    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Write_Start/r/n");
    ptReg = ptInstance->ptDev->regs;
    txStream->channel = params->channel;
	
    ret += tdm_TxComInit(params);
    if (ptInstance->tdmTransMode == TRANS_INT_MODE)
    {
        txStream->txTransmit = tdm_IntTx;              /*init the transter function ,it will be called in Write() function*/
        ret += tdm_IntTxInit();
        if (ret != DRV_SUCCESS)
        {
            tdm_CleanTxResource();
            return DRV_ERROR;
        }
    }
    else if (ptInstance->tdmTransMode == TRANS_DMA_MODE)
    {
   //     txStream->txTransmit = tdm_DmaTx;
        ret += tdm_DmaTxInit();
        if (ret != DRV_SUCCESS)
        {
            tdm_CleanTxResource();
            return DRV_ERROR;
        }
        ptReg->txFifoCtrl |= TDM_TX_DMA_EN;            /* enable dma transfer */
    }
    ptReg->processCtrl |=  TDM_TX_OPEN ;

    if (ptInstance->tCfg.bMaster == TRUE)                  /* wait */
    {
        //   while (0 == (ptReg->commonCtrl & TDM_EN_BACK)|| 0 == (ptReg->commonCtrl & TDM_TX_OPEN_BACK))  /* ensure tdm is enabled */   /* ensure tx is opened */
        {
            zOss_Sleep(TDM_ENABLE_DELAY_TIME);

            waitTime += TDM_ENABLE_DELAY_TIME;
			/*klocwork 3  INVARIANT_CONDITION.UNREACH delete if*/
			/*
            if (waitTime == 1000)
            {
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "waitTime error");
                return DRV_ERROR;
            }*/
        }
    }

    ptInstance->tdmStatus = ((ptInstance->tdmStatus) == TDM_RECORD_INUSE)? TDM_BOTH_INUSE : TDM_PLAY_INUSE;

    gTdmCntChangeTx = 0;
    return DRV_SUCCESS;
}

SINT32 zDrvTDM_Write_Start(T_ZDrvTdm_Params *params, T_ZDrvTdm_Cfg *ptCfg)
{
    SINT32          ret = DRV_SUCCESS;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Write_Start\r\n");

    ret = zDrvTDM_Write_Prepare(ptCfg);
    if (ret != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Read_Start  zDrvTDM_Write_Prepare error ret=%d \r\n",ret);
        return ret;
    }
	
    ret = zDrvTDM_Write_Start_Do(params);
    if (ret != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Read_Start  zDrvTDM_Write_Start_Do error ret=%d \r\n",ret);
        return ret;
    }
	
    return DRV_SUCCESS;
}

/****************************************************************************
* Function:  zDrvTDM_Read_Start
* Description: start to trans data,use in recording
* Parameters:
*   Input:
*           buf_size : one buffer is how many bytes
*           ptCfg : config param
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
* Others:
 ****************************************************************************/
SINT32 zDrvTDM_Read_Prepare(T_ZDrvTdm_Cfg *ptCfg)
{
    SINT32          ret = DRV_SUCCESS;
    T_Tdm_Reg       *ptReg = NULL;
    T_Tdm_Instance  *ptInstance = &s_tdmInstance;
	/*klocwork 3  INVARIANT_CONDITION.UNREACH assertƵif*/
    if (NULL == ptCfg )
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "[TDM]zDrvTDM_Read_Prepare \r\n");
    ptReg = ptInstance->ptDev->regs;
    ret = tdm_RxCheckState(ptCfg);
    if (ret != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Read_Prepare:  tdm_RxCheckState err ret=%d\r\n",ret);
        return DRV_ERROR;
    }
    ptReg->processCtrl |= TDM_EN;

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Read_Prepare timing ctrl) = %x",  (s_tdmInstance.ptDev)->regs->timingCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Read_Prepare rxFifoCtrl) = %x", (s_tdmInstance.ptDev)->regs->rxFifoCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Read_Prepare frame ctrl) = %x",(s_tdmInstance.ptDev)->regs->frameCntr);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Read_Prepare rxFifoCtrl) = %x", (s_tdmInstance.ptDev)->regs->rxFifoCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Read_Prepare process ctrl) = %x", (s_tdmInstance.ptDev)->regs->processCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Read_Prepare reg intEn) = %x",  (s_tdmInstance.ptDev)->regs->intEn);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Read_Prepare reg intStatus) = %x\n", (s_tdmInstance.ptDev)->regs->intStatus);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Read_Prepare reg readCntCtrl) = %x\n",  (s_tdmInstance.ptDev)->regs->readCntCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Read_Prepare reg readCntr) = %x\n",  (s_tdmInstance.ptDev)->regs->readCntr);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_Read_Prepare reg readCntr2) = %x\n",  (s_tdmInstance.ptDev)->regs->readCntr2);
    return DRV_SUCCESS;
}

SINT32 zDrvTDM_Read_Start_Do(T_ZDrvTdm_Params *params)
{

    SINT32          ret = DRV_SUCCESS;
    T_Tdm_Reg       *ptReg = NULL;
    T_Tdm_Instance  *ptInstance = &s_tdmInstance;
    SINT32           waitTime = 0;
	/*klocwork 3  INVARIANT_CONDITION.UNREACH assertƵif*/
    if (NULL == params)
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Read_Start/r/n");
    ptReg = ptInstance->ptDev->regs;
    ret = tdm_RxComInit(params);
    if (ptInstance->tdmTransMode == TRANS_INT_MODE)
    {
        ret += tdm_IntRxInit();                  /*will enable the RX interrupt*/
        if (ret != DRV_SUCCESS)
        {
            tdm_CleanRxResource();
            return DRV_ERROR;
        }
    }
    else if (ptInstance->tdmTransMode == TRANS_DMA_MODE)
    {
        ret += tdm_DmaRxInit();
        if (ret != DRV_SUCCESS)
        {
            tdm_CleanRxResource();
            return DRV_ERROR;
        }
        ptReg->rxFifoCtrl |= TDM_RX_DMA_EN;                          /* enable dma transfer */
    }
    ptReg->processCtrl |= TDM_RX_OPEN;

    if (ptInstance->tCfg.bMaster == TRUE)                         /* wait */
    {
        zOss_Sleep(TDM_ENABLE_DELAY_TIME);
        waitTime += TDM_ENABLE_DELAY_TIME;
		/*klocwork 3  INVARIANT_CONDITION.UNREACH delete if*/
		/*
        if (waitTime == 1000)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "waitTime error");
            return DRV_ERROR;
        }*/
    }

    ptInstance->tdmStatus = ((ptInstance->tdmStatus) == TDM_PLAY_INUSE)?TDM_BOTH_INUSE : TDM_RECORD_INUSE;

    /* add by lvwenhua for tdm sync   */
    gTdmCntChangeRx = 0;

    return DRV_SUCCESS;
}

SINT32 zDrvTDM_Read_Start(T_ZDrvTdm_Params *params, T_ZDrvTdm_Cfg *ptCfg)
{
    SINT32          ret = DRV_SUCCESS;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Read_Start");

    ret = zDrvTDM_Read_Prepare(ptCfg);
    if (ret != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Read_Start  zDrvTDM_Read_Prepare error ret=%d\r\n",ret);
        return ret;
    }
	
    ret = zDrvTDM_Read_Start_Do(params);
    if (ret != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Read_Start  zDrvTDM_Read_Start_Do error ret=%d\r\n",ret);
        return ret;
    }
	
    return DRV_SUCCESS;
}

/****************************************************************************
* Function:  zDrvTDM_Write
* Description: trans data to fifo from ram while playing
* Parameters:
*   Input:
*           buf_size : one buffer is how many bytes
*           uiLen : buffer length
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
 * Others:
 ****************************************************************************/
SINT32 zDrvTDM_Write(const UINT8 *pBuf, UINT32 uiLen)
{
    T_Tdm_Instance  *ptInstance = NULL;
    SINT32 ret = DRV_SUCCESS;
    UINT16 bufLen = 0 ;
    T_Tdm_QueueNode *queueNode = NULL;
    UINT16 *buf = NULL;
	T_Tdm_TxStream *tdmStream = &s_tdmTxStream;

	//zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Write pBuf 0x%x! \r\n", *pBuf);

    if ( NULL == pBuf || 0 == uiLen || TDM_MAX_TX_BUF_SIZE < uiLen)
    {
        tdmwrite_pbuf=pBuf;
        tdmwrite_uilen=uiLen;
    }

	/*klocwork 3  INVARIANT_CONDITION.UNREACH assertƵif*/
    if ( NULL == pBuf || 0 == uiLen || TDM_MAX_TX_BUF_SIZE < uiLen)
    {
   		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
    }
    ptInstance = &s_tdmInstance;
    if (ptInstance->tdmStatus != TDM_PLAY_INUSE && ptInstance->tdmStatus != TDM_BOTH_INUSE)
    {
        return DRV_ERR_NOT_STARTED;
    }
    ret = tdm_UpdateTxBufferStatus((UINT16 *)pBuf , BUFFER_FULL);       /*ϲ*pBufдݺ󣬸ıBUF״̬ΪFULL*/
    if (ret != DRV_SUCCESS)
    {
        return DRV_ERROR;
    }
	
	tdmStream->txWrIdx = (tdmStream->txWrIdx + 1)%TDM_OUT_BUF_NUM;

	if((s_tdmTxStream.pauseFlag == FALSE) && (s_tdmTxStream.Transing == FALSE)) {
		tdmStream->Transing = TRUE;
			dma_async_issue_pending(tdmDmaState[TDM_TX].ch);
			/*klocwork 3  INVARIANT_CONDITION.UNREACH delete if*/
			/*
			if(ret != DRV_SUCCESS) {
				zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Write updateTxBufSta or txTransmit err ret=%d \n", ret);
				return DRV_ERROR;
			}*/
	}
	
    return DRV_SUCCESS;
}

/****************************************************************************
* Function:  zDrvTDM_Read
* Description: get recod data
* Parameters:
*   Input:
*           pBuf : record buffer pointer
*           uiLen : buffer length
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
 * Others:
 ****************************************************************************/
SINT32 zDrvTDM_Read(UINT8 **pBuf, UINT32 *uiLen)
{
    T_Tdm_Instance  *ptInstance = NULL;
    SINT32 ret = DRV_SUCCESS;
    T_Tdm_QueueNode *queueNode = NULL;
	T_Tdm_RxStream *tdmStream = &s_tdmRxStream;
	
	/*klocwork 3  INVARIANT_CONDITION.UNREACH assertƵif*/
    if ( NULL == pBuf || NULL == uiLen)
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }

    ptInstance = &s_tdmInstance;
    if (ptInstance->tdmStatus != TDM_RECORD_INUSE && ptInstance->tdmStatus != TDM_BOTH_INUSE)
    {
        return DRV_ERR_NOT_STARTED;
    }
    //zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Read before test zOss_GetSemaphore");
    if (ZOSS_SUCCESS != zOss_GetSemaphore(s_tdmRxStream.rxSem, ZOSS_WAIT_FOREVER))
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Read zOss_GetSemaphore error");
        return DRV_ERROR;
    }
    //zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Read zOss_GetSemaphore done");

	*pBuf = tdmStream->rxBufferArray[tdmStream->rxRdIdx].buf;
	*uiLen = tdmStream->rxLen;
	tdm_UpdateRxBufferStatus((UINT16 *)tdmStream->rxBufferArray[tdmStream->rxRdIdx].buf, BUFFER_READ);

	#if 0
    ret = tdm_QueueGet(&(s_tdmRxStream.rxQueue), &queueNode);
    if (ret != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Read tdm_QueueGet error");
        return DRV_ERROR;
    }

    if (NULL != queueNode)
    {
        *pBuf = (UINT8 *)queueNode->pBuf;
        *uiLen = queueNode->len * sizeof(UINT16);
        ret = tdm_UpdateRxBufferStatus(queueNode->pBuf, BUFFER_READ);
        if (ret != DRV_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Read tdm_UpdateRxBufferStatus error");
            return DRV_ERROR;
        }
    }
    else
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_Read queueNode==NULL error");
        return DRV_ERROR;
    }
	#endif
	
    return DRV_SUCCESS;
}

/****************************************************************************
 * Function:  zDrvTDM_Stop
* Description: use to stop playing or recording
* Parameters:
*   Input:
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
 * Others:
 ****************************************************************************/
SINT32 zDrvTDM_Write_Stop()
{
    T_Tdm_Instance  *ptInstance = NULL;
    T_Tdm_Reg       *ptReg = NULL;
    SINT32 ret = DRV_SUCCESS;
    T_Tdm_TxStream  *txStream = &s_tdmTxStream;

    ptInstance = &s_tdmInstance;
    ptReg = s_tdmInstance.ptDev->regs;
    if (ptInstance->tdmStatus != TDM_PLAY_INUSE && ptInstance->tdmStatus != TDM_BOTH_INUSE)
    {
        return DRV_ERR_NOT_STARTED;
    }
    ptReg->processCtrl &= ~TDM_TX_OPEN;                /*disable tx only */
    if (ptInstance->tdmTransMode == TRANS_DMA_MODE)
    {
        ptReg->txFifoCtrl &= ~TDM_TX_DMA_EN;
		dma_release_channel(tdmDmaState[TDM_TX].ch);
    }
    /* reset tx fifo */
    ptReg->txFifoCtrl |= TDM_TX_FIFO_RST;
    ptReg->intStatus |= TDM_INT_TX_THRESHOLD;
    ptReg->intEn &= ~TDM_INT_EN_TX_THRESHOLD;       /* disable tx interrupt */

    ret = tdm_CleanTxResource();
	/*klocwork 3 INVARIANT_CONDITION.UNREACH  delete if*/
	/*
    if (ret != DRV_SUCCESS)
    {
        return DRV_ERROR;
    }*/
    txStream->txTransmit = NULL;
    txStream->channel = AUDIO_MONO_CHANNEL;
    txStream->txLen = 0;
    txStream->p_cb = NULL;
	
    if (ptInstance->tdmStatus != TDM_BOTH_INUSE)
    {
        ptInstance->tdmStatus = TDM_OPEN;
    }
    else
    {
        ptInstance->tdmStatus = TDM_RECORD_INUSE;
    }
	
    return DRV_SUCCESS;
}

/****************************************************************************
 * Function:  zDrvTDM_Stop
* Description: use to stop playing or recording
* Parameters:
*   Input:
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
 * Others:
 ****************************************************************************/
SINT32 zDrvTDM_Read_Stop()
{
    T_Tdm_Instance  *ptInstance = NULL;
    T_Tdm_Reg       *ptReg = NULL;

    SINT32 ret = DRV_SUCCESS;

    ptInstance = &s_tdmInstance;
    ptReg = s_tdmInstance.ptDev->regs;
    if (ptInstance->tdmStatus != TDM_RECORD_INUSE && ptInstance->tdmStatus != TDM_BOTH_INUSE)
    {
        return DRV_ERR_NOT_STARTED;
    }
    ptReg->processCtrl &= ~TDM_RX_OPEN;                /*disable tx only */
    if (ptInstance->tdmTransMode == TRANS_DMA_MODE)
    {
        ptReg->rxFifoCtrl &= ~TDM_RX_DMA_EN;
		dma_release_channel(tdmDmaState[TDM_RX].ch);
    }
    ptReg->rxFifoCtrl |= TDM_RX_FIFO_RST;    /* reset tx/rx fifo */
    ptReg->intStatus |= TDM_INT_RX_THRESHOLD;
    ptReg->intEn &= ~TDM_INT_EN_RX_THRESHOLD;       /* disable tx interrupt */

    ret = tdm_CleanRxResource();
	/*klocwork 3 INVARIANT_CONDITION.UNREACH  delete if*/
	/*
    if (ret != DRV_SUCCESS)
    {
        return DRV_ERROR;
    }*/

    if (ptInstance->tdmStatus != TDM_BOTH_INUSE)
    {
        ptInstance->tdmStatus = TDM_OPEN;
    }
    else
    {
        ptInstance->tdmStatus = TDM_PLAY_INUSE;
    }

    return DRV_SUCCESS;
}

/****************************************************************************
 * Function:  zDrvTDM_GetBuf
* Description: get buffer to write playing data
* Parameters:
*   Input:
*           pBuf: playing data buffer
*           uiLen : buffer length
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
 *
 * Others:
 ****************************************************************************/
SINT32 zDrvTDM_GetBuf(UINT8 **pBuf, UINT32 *uiLen)
{
    T_Tdm_TxStream *tdmStream = &s_tdmTxStream;
    UINT8 txIndex = 0 ;
    T_Tdm_Instance  *ptInstance = NULL;

    //zDrv_ASSERT((pBuf != NULL) && (uiLen != NULL));
    if ((NULL == pBuf) || (NULL == uiLen))
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_GetBuf  err return pBuf=%p,uiLen=%p \n",pBuf,uiLen);	
        return DRV_ERR_INVALID_PARAM;
    }
    ptInstance = &s_tdmInstance;
    if (ptInstance->tdmStatus != TDM_PLAY_INUSE && ptInstance->tdmStatus != TDM_BOTH_INUSE)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_GetBuf state err  return ptInstance->tdmStatus=%d \n ",ptInstance->tdmStatus);	
        return DRV_ERR_NOT_STARTED;
    }
    if (ZOSS_SUCCESS != zOss_GetSemaphore(tdmStream->txSem, TDM_SEM_WAIT_TIME))   /*changed by dangpeixia[2010.9.20]*/
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_GetBuf zOss_GetSemaphore err \n ");	
        return DRV_ERROR;
    }


	if(tdmStream->txBufferArray == NULL) {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvTDM_GetBuf tdmStream->txBufferArray == NULL  \n ");	
		return DRV_ERROR;/*  PSSTOPLTEдݷʿָ */
	}

	txIndex = tdmStream->txWrIdx;

    *pBuf = tdmStream->txBufferArray[txIndex].buf;
    *uiLen = tdmStream->txLen;
     tdmStream->txBufferArray[txIndex].tdmBufStatus = BUFFER_WRITE;

#if 0
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_GetBuf timing ctrl) = %x",  (s_tdmInstance.ptDev)->regs->timingCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_GetBuf txFifoCtrl) = %x", (s_tdmInstance.ptDev)->regs->txFifoCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_GetBuf frame ctrl) = %x",(s_tdmInstance.ptDev)->regs->frameCntr);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_GetBuf txFifoCtrl) = %x", (s_tdmInstance.ptDev)->regs->txFifoCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_GetBuf process ctrl) = %x", (s_tdmInstance.ptDev)->regs->processCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_GetBuf reg intEn) = %x",  (s_tdmInstance.ptDev)->regs->intEn);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_GetBuf reg intStatus) = %x\n", (s_tdmInstance.ptDev)->regs->intStatus);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_GetBuf reg writeCntCtrl) = %x\n",  (s_tdmInstance.ptDev)->regs->writeCntCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_GetBuf reg writeCntr) = %x\n",  (s_tdmInstance.ptDev)->regs->writeCntr);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_GetBuf reg writeCntr2) = %x\n",  (s_tdmInstance.ptDev)->regs->writeCntr2);
	
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_GetBuf rxFifoCtrl) = %x", (s_tdmInstance.ptDev)->regs->rxFifoCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_GetBuf rxFifoCtrl) = %x", (s_tdmInstance.ptDev)->regs->rxFifoCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_GetBuf reg readCntCtrl) = %x\n",  (s_tdmInstance.ptDev)->regs->readCntCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_GetBuf reg readCntr) = %x\n",  (s_tdmInstance.ptDev)->regs->readCntr);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvTDM_GetBuf reg readCntr2) = %x\n",  (s_tdmInstance.ptDev)->regs->readCntr2);
#endif
    return DRV_SUCCESS;
}

#if 0
/****************************************************************************
 * Function:  zDrvTDM_FreeBuf
* Description: if copying record data end, free buffer to continue record
 * Others:
 ****************************************************************************/
SINT32 zDrvTDM_Both_Stop()
{
    T_Tdm_Instance  *ptInstance = NULL;
    T_Tdm_Reg       *ptReg = NULL;

    SINT32 ret = DRV_SUCCESS;

    ptInstance = &s_tdmInstance;
    ptReg = s_tdmInstance.ptDev->regs;
    if (ptInstance->tdmStatus != TDM_RECORD_INUSE && ptInstance->tdmStatus != TDM_BOTH_INUSE && ptInstance->tdmStatus != TDM_PLAY_INUSE)
    {
        return DRV_ERR_NOT_STARTED;
    }
    ptReg->processCtrl &= ~ (TDM_TX_OPEN | TDM_RX_OPEN);     /*disable TDM ,Tx ,RX */
    if (ptInstance->tdmTransMode == TRANS_DMA_MODE)
    {
        ptReg->txFifoCtrl &= ~TDM_TX_DMA_EN;
        ptReg->rxFifoCtrl &= ~TDM_RX_DMA_EN;
    }
    ptReg->txFifoCtrl |= TDM_TX_FIFO_RST;  /* reset tx fifo */
    ptReg->rxFifoCtrl |= TDM_RX_FIFO_RST;  /* reset rx fifo */
    ptReg->intStatus |= TDM_INT_TX_THRESHOLD | TDM_INT_RX_THRESHOLD;
    ptReg->intEn &= ~(TDM_INT_EN_TX_THRESHOLD |TDM_INT_EN_RX_THRESHOLD);   /* disable tx interrupt */
    ptReg->processCtrl &= ~TDM_EN;                /*disable TDM ,Tx ,RX */

    ret = tdm_CleanRxResource();
    ret += tdm_CleanTxResource();

    if (ret != DRV_SUCCESS)
    {
        return DRV_ERROR;
    }

    ptInstance->tdmStatus = TDM_OPEN;

    return DRV_SUCCESS;
}
#endif

/****************************************************************************
 * Function:  zDrvTDM_FreeBuf
* Description: if copying record data end, free buffer to continue record
* Parameters:
*   Input:
*           pBuf : record buffer pointer
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
 *
 * Others:
 ****************************************************************************/
SINT32 zDrvTDM_FreeBuf(UINT8 *pBuf)
{
    T_Tdm_RxStream *tdmStream = &s_tdmRxStream;
    SINT32 i = 0;
    T_Tdm_Instance  *ptInstance = NULL;
	
	/*klocwork 3  INVARIANT_CONDITION.UNREACH assertƵif*/
    if (NULL == pBuf)
    {
        zDrv_ASSERT(0);
        return DRV_ERR_INVALID_PARAM;
    }

    ptInstance = &s_tdmInstance;
    if (ptInstance->tdmStatus != TDM_RECORD_INUSE && ptInstance->tdmStatus != TDM_BOTH_INUSE)
    {
        return DRV_ERR_NOT_STARTED;
    }

	tdm_UpdateRxBufferStatus((UINT16 *)tdmStream->rxBufferArray[tdmStream->rxRdIdx].buf, BUFFER_NULL);
	tdmStream->rxRdIdx = (tdmStream->rxRdIdx +1) % TDM_OUT_BUF_NUM ;

#if 0
    while (i < TDM_OUT_BUF_NUM)
    {
        if (tdmStream->rxBufferArray[i].buf == pBuf)
        {
            tdmStream->rxBufferArray[i].tdmBufStatus = BUFFER_NULL;
            break;
        }
        i++;
    }

    if (i == TDM_OUT_BUF_NUM)
    {
        return DRV_ERROR;
    }
#endif
    return DRV_SUCCESS;
}

#if 0
/****************************************************************************
* Function:  zDrvTDM_GetRemained
* Description: get how many data in buffer while playing
* Parameters:
*   Input:
*           len : buffer length
*   Output:
*           None
* Returns:
*           DRV_ERR_INVALID_PARAM: error param
*           DRV_ERROR : operate error
*           DRV_SUCCESS : operate success
* Others:
****************************************************************************/
SINT32 zDrvTDM_GetRemained(UINT32 *len)
{
    T_Tdm_Queue *tdmQueue = &(s_tdmTxStream.txQueue);
    UINT8 index = 0;
    UINT32 uiLen = 0;
    T_Tdm_Instance  *ptInstance = NULL;

    zDrv_ASSERT(len != NULL);
    if (NULL == len)
    {
        return DRV_ERR_INVALID_PARAM;
    }
    ptInstance = &s_tdmInstance;
    if (ptInstance->tdmStatus != TDM_PLAY_INUSE && ptInstance->tdmStatus != TDM_BOTH_INUSE)
    {
        return DRV_ERR_NOT_STARTED;
    }

    index = tdmQueue->readIndex;
    while (index != tdmQueue->writeIndex)
    {
        uiLen += tdmQueue->data[index].len;
        index = (index + 1) % TDM_QUEUE_SIZE;
    }
    *len = uiLen * sizeof(UINT16);

    return DRV_SUCCESS;
}

/****************************************************************************
 * Function:  zDrvTDM_Pause
* Description: pause while playing
* Parameters:
*   Input:
*           None
*   Output:
*           None
* Returns:
*           None
 * Others:
 ****************************************************************************/
VOID zDrvTDM_Pause()
{
    T_Tdm_Instance  *ptInstance = NULL;
	
    ptInstance = &s_tdmInstance;
    if (ptInstance->tdmStatus != TDM_PLAY_INUSE && ptInstance->tdmStatus != TDM_BOTH_INUSE)
    {
        return;
    }

    s_tdmTxStream.pauseFlag = TRUE;
}

/****************************************************************************
* Function:  zDrvTDM_Resume
* Description: resume playing if pause
* Parameters:
*   Input:
*           None
*   Output:
*           None
* Returns:
*          None
* Others:
****************************************************************************/
VOID zDrvTDM_Resume()
{
    T_Tdm_Instance  *ptInstance = NULL;
    T_Tdm_TxStream *tdmStream = &s_tdmTxStream;
    T_Tdm_QueueNode *queueNode = NULL;
    SINT32 ret = DRV_SUCCESS;

    ptInstance = &s_tdmInstance;
    if (ptInstance->tdmStatus != TDM_PLAY_INUSE && ptInstance->tdmStatus != TDM_BOTH_INUSE)
    {
        return;
    }

    ret = tdm_QueueGet(&(tdmStream->txQueue), &queueNode);
    if (ret != DRV_SUCCESS)
    {
        return;
    }

    if (queueNode != NULL)
    {
        tdmStream->Transing = TRUE;
        tdm_UpdateTxBufferStatus(queueNode->pBuf, BUFFER_READ);
        ret = tdmStream->txTransmit(queueNode->pBuf, queueNode->len);
    }
    if (ret != DRV_SUCCESS)
    {
        return;
    }
    tdmStream->pauseFlag = FALSE;

    return;
}
#endif

/**************************************************************************
* Function:  tdm_RlsRxSem
* Description: deal with read deadlock problem
* Parameters:
*   Input:
*   Output:
*           None
* Returns:
*           None
* Others:
****************************************************************************/

VOID zDrvTdm_RxRlsSemaBeforeStop()
{
    T_Tdm_RxStream *tdmRxStream = &s_tdmRxStream;

    zOss_PutSemaphore(tdmRxStream->rxSem);
}

VOID zDrvTdm_TxRlsSemaBeforeStop()
{
    T_Tdm_TxStream *tdmTxStream = &s_tdmTxStream;

    zOss_PutSemaphore(tdmTxStream->txSem);
}

VOID vp_SetTopTdmConfig(VOID)
{
#if 0
    UINT32 AmrRegBit = 0;
	int ret = 0;
	
	//tdm pin cfg
#if defined (_CHIP_ZX297520V3)
/*
	ret = gpio_request(PIN_TDM_FS, "i2s0_ws");
	if (ret < 0)
		BUG();
	ret = gpio_request(PIN_TDM_CLK, "i2s0_clk");
	if (ret < 0)
		BUG();
	ret = gpio_request(PIN_TDM_DIN, "i2s0_din");
	if (ret < 0)
		BUG();
	ret = gpio_request(PIN_TDM_DOUT, "i2s0_dout");
	if (ret < 0)
		BUG();
	zx29_gpio_config(PIN_TDM_FS, FUN_TDM_FS);
	zx29_gpio_config(PIN_TDM_CLK, FUN_TDM_CLK);
	zx29_gpio_config(PIN_TDM_DIN, FUN_TDM_DIN);
	zx29_gpio_config(PIN_TDM_DOUT, FUN_TDM_DOUT);
*/
	audioTdmWsHandle = zDrvGpio_Request("i2s0_ws");
	zDrvGpio_SetFunc(audioTdmWsHandle, GPIO35_TDM_FS);
	audioTdmClkHandle = zDrvGpio_Request("i2s0_clk");
	zDrvGpio_SetFunc(audioTdmClkHandle, GPIO36_TDM_CLK);
	audioTdmDinHandle = zDrvGpio_Request("i2s0_din");
	zDrvGpio_SetFunc(audioTdmDinHandle, GPIO37_TDM_DATA_IN);
	audioTdmDoutHandle = zDrvGpio_Request("i2s0_dout");
	zDrvGpio_SetFunc(audioTdmDoutHandle, GPIO38_TDM_DATA_OUT);

#elif defined (_CHIP_ZX297520V2)
	audioTdmWsHandle = zDrvGpio_Request("i2s0_ws");
	zDrvGpio_SetFunc(audioTdmWsHandle, GPIO38_TDM_FS);
	audioTdmClkHandle = zDrvGpio_Request("i2s0_clk");
	zDrvGpio_SetFunc(audioTdmClkHandle, GPIO39_TDM_CLK);
	audioTdmDinHandle = zDrvGpio_Request("i2s0_din");
	zDrvGpio_SetFunc(audioTdmDinHandle, GPIO40_TDM_DATA_IN);
	audioTdmDoutHandle = zDrvGpio_Request("i2s0_dout");
	zDrvGpio_SetFunc(audioTdmDoutHandle, GPIO41_TDM_DATA_OUT);
#endif

    //top tdm cfg
    AmrRegBit  = ARM_TDM_LOOP_SET;
    AmrRegBit &= 0xfffffff8;
    AmrRegBit |= 0x00000001; //  inter arm tdm1--top tdm1
    AmrRegBit &= 0xfffFfe07;
    AmrRegBit |= 0x000000a8; //   loop dsp afe(loop tdm1)--arm tdm2(loop tdm2)
    //AmrRegBit |= 0x00000150; //   loop dsp arm(loop tdm1)--afe tdm2(loop tdm2)
    ARM_TDM_LOOP_SET = AmrRegBit;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"vp_SetTopTdmConfig set top TDM,ARM_TDM_LOOP_SET=0x%x\n",ARM_TDM_LOOP_SET);

    AmrRegBit = TDM_MOD_CLK_SEL;
    AmrRegBit &= 0xfcffffff;   //set mod_clk_sel bit 25:24 to select the tdm wclk
#if defined (_CHIP_ZX297520V3)
    AmrRegBit |= 0x0000000;
#elif defined (_CHIP_ZX297520V2)
	AmrRegBit |= 0x1000000; //0x1000000(26M )0x0000000(81.92M)
#endif
    TDM_MOD_CLK_SEL = AmrRegBit;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"vp_SetTopTdmConfig set top TDM,MOD_CLK_SEL=0x%x\n",TDM_MOD_CLK_SEL);

    AmrRegBit = DMA_SEL_CFG;
#if defined (_CHIP_ZX297520V3)
    AmrRegBit &= 0xffffffe7;  
	AmrRegBit |= 0x18;		   //set dma_sel_cfg bit 3 4 to 1, to use tdm mode
#elif defined (_CHIP_ZX297520V2)
    AmrRegBit &= 0xfffffffb;   
	AmrRegBit |= 0x4;		   //set dma_sel_cfg bit 2 to 1, to use tdm mode
#endif
    DMA_SEL_CFG = AmrRegBit;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"vp_SetTopTdmConfig set top TDM,DMA_SEL_CFG=0x%x\n",DMA_SEL_CFG);
  #endif
}



static int __init tdm_init(void)
{
//	printk("i2s_init register\n");
	return platform_device_register(&tdm_device);
}

static void __exit tdm_exit(void)
{
//	printk("i2s_exit unregister\n");
	platform_device_unregister(&tdm_device);
}

module_init(tdm_init);
module_exit(tdm_exit);
