/*
 * ZTE power management common driver
 *
 * Copyright (C) 2013 ZTE Ltd.
 * 	by zxp
 *
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/types.h>
#include <linux/suspend.h>
#include <linux/tick.h>

#include <mach/irqs.h>
#include <mach/zx-pm.h>

#define	SPEED_DECODE_WAKE_CAUSE

static struct pm_wake_cause_info wake_cause_src[] = 
{
	{ 0, 9,  "sdmmc0", },
	{ 1, 10, "sd0_detect_on", },	
	{ 2, 11, "sd0_detect_off", },
	{ 3, 3,  "rtc_alarm", },
	{ 4, 4,  "rtc_timer", },
	{ 5, 13, "icp_m02ap", },
	{ 6, 14, "icp_phy2ap", },	
	{ 7, 15, "icp_ps2ap", },

#ifndef SPEED_DECODE_WAKE_CAUSE
	{ 8, 34, "spcu_pw", },
	{ 12,61, "lte_lpm1", },
	{ 13,62, "lte_lpm2", },
	{ 9, 59, "lte_lpm3", },
	{ 14,63, "lte_lpm4", },
	{ 76, "gsm_lpm", },	
#endif

	{ 10, 41, "usb0_up", },	
	{ 11, 42, "usb0_down", },
	{ 15, 12, "sdmmc1", },
	{ 16, 49, "ext0", },
	{ 17, 50, "ext1", },
	{ 18, 51, "ext2", },
	{ 19, 52, "ext3", },	
	{ 20, 53, "ext4", },
	{ 21, 54, "ext5", },	
	{ 22, 55, "ext6", },
	{ 23, 56, "ext7", },
	{ 24, 48, "ext8", },	
	{ 25, 48, "ext9", },	
	{ 26, 48, "ext10", },	
	{ 27, 48, "ext11", },	
	{ 28, 48, "ext12", },	
	{ 29, 48, "ext13", },	
	{ 30, 48, "ext14", },	
	{ 31, 48, "ext15", },	
	{ 32, 67, "sd0_data1", },
	{ 33, 68, "uart0_cts", },
	{ 34, 69, "uart1_cts", },
	{ 35, 70, "usim0_detect_on", },
	{ 36, 71, "usim0_detect_off", },
	{ 37, 72, "usim1_detect_on", },	
	{ 38, 73, "usim1_detect_off", },

	{ 64, 16, "timer2", },		
	{ 65, 17, "timer3", },	

	{ INVALID_INT_NUM, INVALID_INT_NUM, "", },
};

static unsigned pm_wake_int_no;
static char *	pm_wake_int_name;

static unsigned int gic_pending[3];
static unsigned int pcu_wakeup[3];

/** 
 * get current pll_used helper function.
 *
 * This code only used pm internel.
 *
 * return:  PCU_USE_PLL_UFI/PCU_USE_PLL_624/PCU_USE_PLL_BUS
 */
static int zx297510_pll_table[] = 
{
	PCU_USE_PLL_UFI,
	PCU_USE_PLL_MAIN,
	PCU_USE_PLL_624,
	PCU_USE_PLL_BUS,
};

/* get cur pll used for pcu setting */
int pm_get_pll_used(void)
{
	return zx297510_pll_table[pm_read_reg(TOP_UFI_SEL_REG)&3];
}
/** 
 * get sleep_time helper function.
 * used for idle sleep type.
 *
 * This code only used pm internel.
 *
 * return :  unit is 26M cycle(38.4ns)
 * note: the max value is 0x7FFFFFFF (about 82.5s)          
 */
u32 pm_get_sleep_time(void)
{
#ifdef CONFIG_CPU_IDLE
	return (u32)(idle_get_sleeptime()*26);
#else
	return 0xffffffff; 
#endif
}

/**
 * mask all and backup, then unmask wakeup interrupts.
 * 
 */
int zx297510_unmask_wakeup_interrupt(void)
{

	//backup mask reg
	//mask all
	//??need gic driver provide the interface
	
	//unmask wakeup int
	

	return 0;
}

/**
 * restore interrupt that masked. 
 * 
 */
int zx297510_interrupt_mask_restore(void)
{
	//restore mask reg from backup data

	return 0;
}

static unsigned ufi_clk_value;
void save_ufi_clk(void)
{
	ufi_clk_value = pm_read_reg(TOP_UFI_SEL_REG);
}

/* cpu clock switch to 26MHz */
void pm_switch_clk_to_26m(void)
{
	pm_write_reg(TOP_UFI_SEL_REG, 1); 
	
	pcu_set_pll_used(PCU_USE_PLL_MAIN);

	pm_ufi_pll_pwrdn();
}

/* cpu:l2=1:2, and restore the clock used before */
void pm_switch_clk_from_26m(void)
{
	pm_ufi_pll_pwron();
	
	pcu_set_pll_used(zx297510_pll_table[ufi_clk_value]);
	
	pm_write_reg(TOP_UFI_SEL_REG, ufi_clk_value);
}

/* set pcu pll used */
void pm_set_pll_used(void)
{
	pcu_set_pll_used(pm_get_pll_used());
}

/*=============================================================================
 *========  zx297510 CRM driver ===============================================
 *=============================================================================
 */
typedef struct
{
    /* 0x00 */ const volatile unsigned version;
    /* 0x04 */ volatile unsigned clkdiv;
    /* 0x08 */ volatile unsigned clken;	
    /* 0x0C */ volatile unsigned rsten;		
               char padding1[0x10];
    /* 0x20 */ volatile unsigned a9_cfg;
    /* 0x24 */ volatile unsigned a9_debug;
               char padding2[0x08];
    /* 0x30 */ const volatile unsigned a9_smpnamp;
	           char padding3[0x08];
    /* 0x3C */ const volatile unsigned a9_pmu;
    /* 0x40 */ volatile unsigned l2_cfg;
               char padding4[0x0C];
    /* 0x50 */ volatile unsigned int_mode[14];

} crm_registers;
/**
 * save & restore CRM register interface for zx297510. 
 * 
 */
void zx297510_save_crm(u32 *pointer, u32 crm_base)
{
	copy_words(pointer, (void *)crm_base, 34);
}

void zx297510_restore_crm(u32 *pointer, u32 crm_base)
{
    crm_registers *crm = (crm_registers *)crm_base;

	/* clkdiv/clken/rsten */
	crm->clkdiv 	= pointer[1];	
	crm->clken  	= pointer[2];
	crm->rsten  	= pointer[3];	

	crm->a9_cfg 	= pointer[8];
    crm->a9_debug	= pointer[9];

	copy_words(crm->int_mode, pointer+20, 14);
}

/**
 * set pll ufi==> 800M, cpu:l2==>2:1. 
 * 
 * these may be set in boot code.
 */
void pm_init_crm_temp(void)
{
	crm_registers *crm = (crm_registers *)A9_CRM_ADDRESS;

	/* cpu:l2=2:1, and restore the clock used before 
	 * this should init ok */	
	crm->clkdiv &= ~3;
	crm->clkdiv |= 1;

	//ufi 800M
#if 0	
	pm_write_reg(ZX29_TOP_VA+0x14, 0x05000000);
	pm_write_reg(ZX29_TOP_VA+0x10, 0x08346409); //800MHz
	pm_write_reg(TOP_UFI_DIV_REG, 	0);
#endif	
}

/*=============================================================================
 *========  zx297510 clock driver =============================================
 *=============================================================================
 */
void zx297510_pm_clk_init(void)
{
	/* close ddr port1/2 */
	pm_clr_reg(LPDDR_CLK_EN_REG, A9_DDRCTRL_PORT1_CLKEN|A9_DDRCTRL_PORT2_CLKEN);

	/* UFI PLL delay count configure */
	pm_write_reg(TOP_UFI_DELAY_CFG_REG, 0x3E0);

	/* close no-used uart port and clock */
	pm_clr_reg(no_use_uart_base() + zx297510_UART_CR, UART_CR_UARTEN | UART_CR_TXE | UART_CR_LBE);
#ifdef DEBUG_UART0
	pm_clr_reg(A1_CRM_PCLK_EN_REG, A1_CRM_UART1_BIT);
	pm_clr_reg(A1_CRM_WCLK_EN_REG, A1_CRM_UART1_BIT);
#else
	pm_clr_reg(A1_CRM_PCLK_EN_REG, A1_CRM_UART0_BIT);
	pm_clr_reg(A1_CRM_WCLK_EN_REG, A1_CRM_UART0_BIT);
#endif	
}

/*=============================================================================
 *========  zx297510 PM&IDLE ==================================================
 *=============================================================================
 */
/* us */ 
s64 pm_get_remainder_time(void)
{
	return div64_long(((s64)pm_read_reg(TIMER3_CUR_VALUE_REG))*1000000, 32768);
}

static inline int pm_start_wake_timer(s64 us)
{	
	/* 1 cycle == 1/32768(s)*/
	/* max setting value = 0xffffffff/32768 = 131072s = 36h */
	unsigned long cycles = div64_long(us*32768, 1000000);
	struct clock_event_device *dev = __this_cpu_read(tick_cpu_device.evtdev);
	ktime_t begin_time;	
//	s64 remainder_timer;

	dev->set_next_event(cycles, dev);

	/* wait timer refresh ok */
	begin_time = ktime_get();
	while((unsigned)ktime_to_us(ktime_sub(ktime_get(), begin_time))<60);	

/*
	remainder_timer = pm_get_remainder_time();
	pr_info("[SLP] request:%ld remainder: %d\n", us, (u32)remainder_timer);		
*/
	return 0;
}

void setup_timer_wakeup(s64 us)
{
	pm_start_wake_timer(us);
}


/**
 * set int number .
 * 
 * 
 */
static void pm_set_wakeup_int_no(unsigned int_no)
{
	pm_wake_int_no = int_no;
}

unsigned pm_get_wakeup_int_no(void)
{
	return pm_wake_int_no ;
}

char * pm_get_wakeup_int_name(void)
{
	return pm_wake_int_name ;
}

/** 
 * get gic pending state.
 *
 * This code used to get wake cause.
 */
static void pm_get_gic_pending(void)
{
	/* zx297510 has only 77 irqs */
	gic_pending[0] 	= pm_read_reg(GIC_DIST_SET_PENDING + 0x4);	/* 32~~63 */
	gic_pending[1]	= pm_read_reg(GIC_DIST_SET_PENDING + 0x8);	/* 64~~95 */
	gic_pending[2]	= pm_read_reg(GIC_DIST_SET_PENDING + 0xc);	/* 96~~127 */	
}

/** 
 * get the wake cause int number base gic-pending and pcu setting.
 *
 * return int number.
 */
static unsigned pm_decode_wake_cause(void)
{
	int 			i = 0;
	unsigned int	int_no;
	unsigned int	pcu_no;	

	while(1)
	{
		int_no = wake_cause_src[i].gic_index;
		if(int_no == INVALID_INT_NUM)
			break;

		/* now let us to find who wakeup the system... */
		pcu_no = wake_cause_src[i].pcu_index;
		
		if( (gic_pending[int_no/32]&(1<<(int_no%32))) &&  !(pcu_wakeup[pcu_no/32]&(1<<(pcu_no%32))))
		{
			pm_wake_int_name = wake_cause_src[i].int_name;
			break;
		}

		i++;
	}

	return int_no;
}

void pm_get_wake_cause(void)
{
	unsigned	int_no;

	pm_get_gic_pending();
	pcu_get_wakeup_setting(pcu_wakeup);
	
	int_no = pm_decode_wake_cause();
	if(int_no != INVALID_INT_NUM)
	{
 		pm_set_wakeup_int_no(int_no);
		
		pm_set_wakeup_reason(WR_WAKE_SRC_NORMAL);

		pr_info("[SLP] wake: %d  [%s]\n", pm_get_wakeup_int_no(), pm_get_wakeup_int_name());
	}
	else
		pm_set_wakeup_reason(WR_WAKE_SRC_UNKNOWN);
}


/*===================================================================
 *==  7510 ap interrupt arrangement =================================
 *===================================================================
 *===================================	 evb     dc	   mifi   =======
 *== timer3              -- tick		   y      y     y
 *== m02ap_icp                             y      y     y
 *== ps2ap_icp                             y      y     y
 *== ext0                -- pmu            y      n     y 
 *== ext1                -- pg(charger)    y      n     y
 *== ext2                -- wps_key   	   n      n     y 	
 *== ext3                -- rst_key	       y      n     y	
 *== ext4                -- pwr_key        y      n     n
 *== ext7                -- wps_key        y      n     n 
 *== ext6                -- wifi_wake      y      n     y
 *== alarm                                 y      n     n
 *== rtc                                   y      n     n
 *===================================================================
 */
/* evb */
#if  defined(CONFIG_ARCH_ZX297510EVB)
unsigned int wake_source_int_for_suspend[] = 
{
	/* int1 */
    WAKE_SRC_ICP_M02AP | WAKE_SRC_ICP_PS2AP | 						\
    WAKE_SRC_RTC_ALARM /*| WAKE_SRC_RTC_TIMER */| 					\
    WAKE_SRC_EXTERNAL0 | WAKE_SRC_EXTERNAL1 | WAKE_SRC_EXTERNAL3 |	\
    WAKE_SRC_EXTERNAL4 | WAKE_SRC_EXTERNAL6 | WAKE_SRC_EXTERNAL7,

	/* int2 */
	0,

	/* timer */
    WAKE_SRC_TIMER3,
};

unsigned int wake_source_int_for_dpidle[] = 
{
	/* int1 */
    WAKE_SRC_ICP_M02AP | WAKE_SRC_ICP_PS2AP | 						\
    WAKE_SRC_RTC_ALARM /*| WAKE_SRC_RTC_TIMER */| 					\
    WAKE_SRC_EXTERNAL0 | WAKE_SRC_EXTERNAL1 | WAKE_SRC_EXTERNAL3 |	\
    WAKE_SRC_EXTERNAL4 | WAKE_SRC_EXTERNAL6 | WAKE_SRC_EXTERNAL7,

	/* int2 */
	0,

	/* timer */
    WAKE_SRC_TIMER3,
};
     
#elif defined(CONFIG_ARCH_ZX297510UFI)

/* mifi */
unsigned int wake_source_int_for_suspend[] = 
{
	/* int1 */
    WAKE_SRC_ICP_M02AP | WAKE_SRC_ICP_PS2AP | 						\
    WAKE_SRC_RTC_ALARM | /*WAKE_SRC_RTC_TIMER |*/ 					\
    WAKE_SRC_EXTERNAL0 | WAKE_SRC_EXTERNAL1 | WAKE_SRC_EXTERNAL2 |	\
    WAKE_SRC_EXTERNAL4 | 											\
    WAKE_SRC_EXTERNAL3 | WAKE_SRC_EXTERNAL6 | WAKE_SRC_EXTERNAL_GPIO7,

	/* int2 */
	0,

	/* timer */
    WAKE_SRC_TIMER3,
};

unsigned int wake_source_int_for_dpidle[] = 
{
	/* int1 */
    WAKE_SRC_ICP_M02AP | WAKE_SRC_ICP_PS2AP | 						\
    WAKE_SRC_RTC_ALARM | /*WAKE_SRC_RTC_TIMER |*/ 					\
    WAKE_SRC_EXTERNAL0 | WAKE_SRC_EXTERNAL1 | WAKE_SRC_EXTERNAL2 |	\
    WAKE_SRC_EXTERNAL4 | 											\    
    WAKE_SRC_EXTERNAL3 | WAKE_SRC_EXTERNAL6 | WAKE_SRC_EXTERNAL_GPIO7,

	/* int2 */
	0,

	/* timer */
    WAKE_SRC_TIMER3,
};

#elif defined(CONFIG_ARCH_ZX297510CPE)

/* cpe */
unsigned int wake_source_int_for_suspend[] = 
{
	/* int1 */
    WAKE_SRC_ICP_M02AP | WAKE_SRC_ICP_PS2AP | 						\
    WAKE_SRC_EXTERNAL0,

	/* int2 */
	0,

	/* timer */
    WAKE_SRC_TIMER3,
};

unsigned int wake_source_int_for_dpidle[] = 
{
	/* int1 */
    WAKE_SRC_ICP_M02AP | WAKE_SRC_ICP_PS2AP | 						\
    WAKE_SRC_EXTERNAL0,

	/* int2 */
	0,

	/* timer */
    WAKE_SRC_TIMER3,
};


#else	/* CONFIG_ARCH_ZX297510DC */

/* dc */
unsigned int wake_source_int_for_suspend[] = 
{
	/* int1 */
    WAKE_SRC_ICP_M02AP | WAKE_SRC_ICP_PS2AP | 						\
    WAKE_SRC_EXTERNAL0,

	/* int2 */
	0,

	/* timer */
    WAKE_SRC_TIMER3,
};

unsigned int wake_source_int_for_dpidle[] = 
{
	/* int1 */
    WAKE_SRC_ICP_M02AP | WAKE_SRC_ICP_PS2AP | 						\
    WAKE_SRC_EXTERNAL0,

	/* int2 */
	0,

	/* timer */
    WAKE_SRC_TIMER3,
};

#endif 

/*=============================================================================
 *========  zx297510 DEBUG UART ===============================================
 *====== note: uart is in wakeup powerdomain  =================================
 *=============================================================================
 */

static struct zx_suspend_context suspend_context;

void debug_uart_suspend(void)
{
	void __iomem *uart_base = debug_uart_base();

#if 1
	suspend_context.uart.ibrd 	= pm_read_reg(uart_base + zx297510_UART_IBRD);
	suspend_context.uart.fbrd 	= pm_read_reg(uart_base + zx297510_UART_FBRD);
	suspend_context.uart.lcrh 	= pm_read_reg(uart_base + zx297510_UART_LCRH);
	suspend_context.uart.ifls 	= pm_read_reg(uart_base + zx297510_UART_IFLS);
	suspend_context.uart.imsc 	= pm_read_reg(uart_base + zx297510_UART_IMSC);
	suspend_context.uart.dmacr	= pm_read_reg(uart_base + zx297510_UART_DMACR);	
#endif	
	suspend_context.uart.cr		= pm_read_reg(uart_base + zx297510_UART_CR);

	/* disable */
	pm_clr_reg(uart_base + zx297510_UART_CR, UART_CR_UARTEN | UART_CR_TXE | UART_CR_LBE);

	/* gate pclk/wclk */
#ifdef DEBUG_UART0
	pm_clr_reg(A1_CRM_PCLK_EN_REG, A1_CRM_UART0_BIT);
	pm_clr_reg(A1_CRM_WCLK_EN_REG, A1_CRM_UART0_BIT);
#else
	pm_clr_reg(A1_CRM_PCLK_EN_REG, A1_CRM_UART1_BIT);
	pm_clr_reg(A1_CRM_WCLK_EN_REG, A1_CRM_UART1_BIT);
#endif
}

void debug_uart_resume(void)
{
	void __iomem *uart_base = debug_uart_base();

	/* open pclk/wclk */
#ifdef DEBUG_UART0
	pm_set_reg(A1_CRM_PCLK_EN_REG, A1_CRM_UART0_BIT);
	pm_set_reg(A1_CRM_WCLK_EN_REG, A1_CRM_UART0_BIT);
#else
	pm_set_reg(A1_CRM_PCLK_EN_REG, A1_CRM_UART1_BIT);
	pm_set_reg(A1_CRM_WCLK_EN_REG, A1_CRM_UART1_BIT);
#endif

#if 1
	pm_write_reg(uart_base + zx297510_UART_IBRD, suspend_context.uart.ibrd);
	pm_write_reg(uart_base + zx297510_UART_FBRD, suspend_context.uart.fbrd);
	pm_write_reg(uart_base + zx297510_UART_LCRH, suspend_context.uart.lcrh);
	pm_write_reg(uart_base + zx297510_UART_IFLS, suspend_context.uart.ifls);
	pm_write_reg(uart_base + zx297510_UART_IMSC, suspend_context.uart.imsc);
	pm_write_reg(uart_base + zx297510_UART_DMACR, suspend_context.uart.dmacr);
#endif
	pm_write_reg(uart_base + zx297510_UART_CR, suspend_context.uart.cr);
}

/*  timer register offset  */
#define TIMER_CONFIG_REG   		(0x04)
#define TIMER_LOAD_REG     		(0x08)
#define TIMER_START_REG    		(0x0C)
#define TIMER_REFRESH_REG  		(0x10)
#define TIMER_CUR_VALUE    		(0x18)

void pm_timer_suspend(void) 
{
	void __iomem *reg_base = ZX29_TIMER3_VA;

	suspend_context.timer.load	= pm_read_reg(reg_base + TIMER_LOAD_REG);

#if 0
	suspend_context.timer.cfg	= pm_read_reg(reg_base + TIMER_CONFIG_REG);	
	suspend_context.timer.start	= pm_read_reg(reg_base + TIMER_START_REG);
	suspend_context.timer.count	= pm_read_reg(reg_base + TIMER_CUR_VALUE);
#endif	
}

extern unsigned int test_timer_read( void );
void pm_timer_resume(void) 
{
	void __iomem *reg_base = ZX29_TIMER3_VA;
	unsigned int tmp=0;
    unsigned int t0;

//	pm_write_reg(reg_base + TIMER_CONFIG_REG, suspend_context.timer.cfg);
	pm_write_reg(reg_base + TIMER_LOAD_REG, suspend_context.timer.load);

	tmp	=	pm_read_reg(reg_base + TIMER_REFRESH_REG);
    tmp ^= 	0x3;
    pm_write_reg(reg_base + TIMER_REFRESH_REG, tmp);	

	t0=test_timer_read();
	while( (test_timer_read()-t0) < 31);
//	pm_write_reg(reg_base + TIMER_START_REG, suspend_context.timer.start);
}

void zx_pm_pre_suspend(void)
{
	/* backup timer3 */
//	pm_timer_suspend();
	
	setup_timer_wakeup(__SLEEP_TIME_1h__*18);	
}

void zx_pm_post_suspend(void)
{
	pcu_clr_timer3_int();

	/* restore timer3 */
//	pm_timer_resume();
}

#ifdef CONFIG_ZX_PM_DEBUG
/**
 *  print value to ram for trace state
 */
void zx_trace_flag1(unsigned flag)
{
	writel_relaxed(flag, ZX29_IRAM6_VA+1008);
}
EXPORT_SYMBOL_GPL(zx_trace_flag1);

/**
 *  print value to ram for trace flag
 */
void zx_trace_flag2(unsigned flag)
{
	writel_relaxed(flag, ZX29_IRAM6_VA+1012);
}
EXPORT_SYMBOL_GPL(zx_trace_flag2);

void zx_pm_trace_out(void)
{
	printk("\n[SLP]: sleep_cnt:%d wake_cnt:%d chipslpen_cnt:%d slpflag:%d canslptime:%d\n", 
			pm_read_reg(IRAM_ADDR_FOR_SLEEP_CNT),
			pm_read_reg(IRAM_ADDR_FOR_WAKE_CNT),
			pm_read_reg(IRAM_ADDR_FOR_CHIPSLEEPEN_CNT),
			pm_read_reg(IRAM_ADDR_FOR_g_ChipSleepFlag),
			pm_read_reg(IRAM_ADDR_FOR_g_ChipRealCanSleepTime));

	printk("[SLP]: ddr_ck1:0x%x ddr_ck2:0x%x ap_cfg:0x%x ps_cfg:0x%x phy_cfg:0x%x\n", 
			pm_read_reg(IRAM_ADDR_FOR_DENALI_DDRCTRL_89_CKE1),
			pm_read_reg(IRAM_ADDR_FOR_DENALI_DDRCTRL_89_CKE2),
			pm_read_reg(IRAM_ADDR_FOR_AP_CONFIG),
			pm_read_reg(IRAM_ADDR_FOR_PS_CONFIG),
			pm_read_reg(IRAM_ADDR_FOR_PHY_CONFIG));

	printk("[SLP]: pll_624:0x%x pll_ufi:0x%x pll_bus:0x%x pll_lte:0x%x sys_cfg:0x%x\n\n", 
			pm_read_reg(IRAM_ADDR_FOR_PLL_624_208_CONFIG1),
			pm_read_reg(IRAM_ADDR_FOR_PLL_CPU_UFI_CONFIG1),
			pm_read_reg(IRAM_ADDR_FOR_PLL_CPU_BUS_CONFIG1),
			pm_read_reg(IRAM_ADDR_FOR_PLL_TD_LTE_CONFIG1),
			pm_read_reg(IRAM_ADDR_FOR_SYSTEM_CONFIG_REG));

	printk("[SLP]: ps1:%d ps2:%d ps3:%d ps4:%d\n\n", 
			pm_read_reg(IRAM_ADDR_FOR_PS1),
			pm_read_reg(IRAM_ADDR_FOR_PS2),
			pm_read_reg(IRAM_ADDR_FOR_PS3),
			pm_read_reg(IRAM_ADDR_FOR_PS4));

	printk("[SLP]: ps5:%d ps6:%d ps7:%d ps8:%d\n\n", 
			pm_read_reg(IRAM_ADDR_FOR_PS5),
			pm_read_reg(IRAM_ADDR_FOR_PS6),
			pm_read_reg(IRAM_ADDR_FOR_PS7),
			pm_read_reg(IRAM_ADDR_FOR_PS8));	

	printk("[SLP]: ps9:%d ps10:%d ps11:%d ps12:%d\n\n", 
			pm_read_reg(IRAM_ADDR_FOR_PS9),
			pm_read_reg(IRAM_ADDR_FOR_PS10),
			pm_read_reg(IRAM_ADDR_FOR_PS11),
			pm_read_reg(IRAM_ADDR_FOR_PS12));
}

#endif

