/***********************************************************************
 * Copyright (C) 2014, ZTE Corporation.
 *
 * File Name:   lcd_comm.c
 * File Mark:
 * Description: Lcd function implementation. Adapted to LCM MODEL - LEAD T15DS26.
 * Others:      this file is in BOOTLOADER procedure.
 *                  This LCM MODEL uses SPI-4.
 *                  NOTE: 4-lines-spi uses CS, CLK, D/CX and SDI. Do not using SDO instead of D/CX.
 * Version:     V1.0
 * Author:      qihongfang
 * Date:        2017-9-18
 *
 * History 1:
 *     Date: 2015-1-16
 *     Version:
 *     Author: YinWenguan
 *     Modification: Add brightness setting.
 *
***********************************************************************/

/*************************************************************************
 * Include files
 *************************************************************************/
#include "../include/board.h"

#include <common.h>
#include <malloc.h>
#include <asm/io.h>
#include "../include/spi.h"
#include "../include/ssp.h"
#include <lcd.h>
#include <drvs_gpio.h>
#include <peripheral.h>

/*************************************************************************
 * Macro
 *************************************************************************/
 /* debug switch */

//static unsigned char g_lcd_id=0;
#define LCM_ID_LEAD			0x7C  /*йst7735s\st7735s*/

#define LCM_ID_BOE			0x7C	// 128*128 
#define LCM_ID_BOE_7735S	0x89F0	// 128*128 

/* LCD Features */
#define LCD_WIDTH			128
#define LCD_HEIGHT			128

#define LCD_RESOLUTION		(LCD_WIDTH * LCD_HEIGHT)
#define LCD_BITS_PER_PIXEL	(5 + 6 + 5)
#define LCD_PAGE_HEIGHT		8
#define LCD_PAGE_NUM		8
#define LCD_BPP				(2)

#define START_PAGE          0xB0
#define START_HIGH_BIT      0x10
#define START_LOW_BIT       0x00
#define START_LOW_PY		1


/* rotate display */
//#define LCD_ROTATE_180

#define LCD_XS				0x02
#define LCD_XE				((LCD_WIDTH)+(LCD_XS)-1)
#ifdef LCD_ROTATE_180
#define LCD_MXMYRGB 		0x1C
#define LCD_YS      		0x01
#else
#define LCD_MXMYRGB 		0xC8
#define LCD_YS      		0x03
#endif

#define LCD_YE				((LCD_HEIGHT)+(LCD_YS)-1)

struct lcd_var_screeninfo {
	unsigned int xs;	//xs_offset	
	unsigned int ys;    //ys_offset
	unsigned int width;	//128 	240 	
	unsigned int height; //128 	240		
	unsigned int xe;	 //width + xs - 1	
	unsigned int ye;     //height + ys - 1
	//unsigned int bits_per_pixel;		
};
static struct lcd_var_screeninfo var_screeninfo = {0};	/* Current var */

/**************************************************************************
 *                                  Types                                                                                   *
 **************************************************************************/
static volatile T_ZDrvLcd_Instance			g_sLcd_Instance = {false, LCD_NORMAL, LCD_XFER_POLL, LCD_DCX_CMD};
static volatile T_ZDrvLcd_BacklightStatus	g_sLcd_Backlight = LCD_BACKLIGHT_OFF;
static volatile T_ZDrvLcd_Brightness		g_sLcd_Brightness = 255;

/**************************************************************************
 * Global  Variable
 **************************************************************************/
extern volatile T_ZDrvLcd_Info s_lcd_info;
extern int zDrvLcd_Spi_WriteCmd(unsigned char cmd);
extern int zDrvLcd_Spi_WriteData(unsigned char data);

static unsigned char g_sLcdHalDataBuf[LCD_WIDTH * LCD_PAGE_NUM] =
{
    0,
};

static unsigned char g_sLcdHalDataBuf_reg[LCD_WIDTH *LCD_HEIGHT] =
{
    0,
};


/**************************************************************************
 * Function Implementation
 **************************************************************************/
void lcd_gpio_define(struct T_ZDrvLcd_gpio_define* config)
{
	config->spi_cs_gpio			= GPIO25;
	config->spi_scl_gpio		= GPIO26;
	config->spi_sda_gpio		= GPIO28;
	config->spi_dcx_gpio		= GPIO27;

	config->spi_cs_func_sel		= GPIO25_SSP0_CS;
	config->spi_scl_func_sel	= GPIO26_SSP0_CLK;
	config->spi_sda_func_sel	= GPIO28_SSP0_TXD;
	config->spi_dcx_func_sel	= GPIO27_SSP0_RXD;

	config->spi_cs_gpio_sel		= GPIO25_GPIO25;
	config->spi_scl_gpio_sel	= GPIO26_GPIO26;
	config->spi_sda_gpio_sel	= GPIO28_GPIO28;
	config->spi_dcx_gpio_sel	= GPIO27_GPIO27;

	config->lcd_reset_gpio		= GPIO120;
	config->lcd_reset_gpio_sel	= GPIO120_GPIO120;
	
	config->lcd_blg_gpio		= GPIO119;
	config->lcd_blg_gpio_sel	= GPIO119_GPIO119;

}

#if KLOCWORK_BOOT
/**************************************************************************
* Function: zDrvLcd_SetRamAddr
* Description: used to set specified start point of DISPLAY RAM for updating laterly
* Parameters:
*       (IN)
*               uiLeft      left postion
*               uiTop       top postion
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
int set_ram_addr_bw(unsigned short uiLeft, unsigned short uiTop)
{
     signed    int ret = 0;

    /*set column address*/
    zDrvLcd_Spi_WriteCmd(START_HIGH_BIT |((uiLeft>>4) & 0x0F)); // ?4?
    zDrvLcd_Spi_WriteCmd((START_LOW_BIT | (uiLeft & 0x0F))+START_LOW_PY);    // ?4?    

    /*set page address*/
    zDrvLcd_Spi_WriteCmd(START_PAGE | (uiTop & 0x0F));
    return ret ;
}
uint32_t lcd_WritePixel(unsigned short rgb565, unsigned short uiRow, unsigned short uiCol)
{
    unsigned short page_index = uiRow / LCD_PAGE_HEIGHT;

    unsigned short ram_offset = page_index * LCD_WIDTH + uiCol;

    unsigned short bit_shift = uiRow % LCD_PAGE_HEIGHT;

    if (rgb565 != 0 )
        *(g_sLcdHalDataBuf + ram_offset) &= ~(0x01 << bit_shift);
    else 
        *(g_sLcdHalDataBuf + ram_offset) |= (0x01 << bit_shift);

    return 0;
}
/**************************************************************************
* Function: zDrvLcd_UpdateRect
* Description: used to update a specified rectangle area of the LCD device opened previously.
* Parameters:
*       (IN)
*               pBuf        data buffer
*               uiLeft      left postion
*               uiTop       top postion
*               uiWidth     rectangle width
*               uiHeight    rectangle height
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
*       None.
**************************************************************************/
int update_rect_bw(unsigned char *pBuf, unsigned short uiLeft, unsigned short uiTop, unsigned short uiWidth, unsigned short uiHeight)
{

    unsigned short  *pSrcBuf = (unsigned short  *)pBuf;
    unsigned short  row,row_reg, col,col_reg, pixel_value;
    unsigned short cur_page;

    if((pBuf==NULL)|(LCD_WIDTH <=uiLeft)|(LCD_HEIGHT <=uiTop)|(LCD_WIDTH < uiLeft + uiWidth)| (LCD_HEIGHT < uiTop + uiHeight))
      {  return -1;}

    for (row = uiTop,row_reg = 0; row <(uiTop + uiHeight); row++,row_reg++)
    {
        for (col = uiLeft,col_reg = 0; col < (uiLeft + uiWidth); col++,col_reg++)
        {
           g_sLcdHalDataBuf_reg[row * LCD_WIDTH + col] = *(pSrcBuf + row_reg * uiWidth + col_reg);
        }
    }
    
    /* ???????RGB565??????,??RAM */
  
    for (row = 0; row <LCD_HEIGHT; row++)
    {
        for (col = 0; col < LCD_WIDTH; col++)
        {
           pixel_value =  g_sLcdHalDataBuf_reg[row * LCD_WIDTH + col];

            lcd_WritePixel(pixel_value, row , col );
        }
    }   

    
    /* ??,?????,??RAM???,????page */
    
    for ( cur_page =0; cur_page <LCD_PAGE_NUM; cur_page++ )
    {
        set_ram_addr_bw( 0, cur_page );
        zDrvLcd_Spi_WriteDataBlock(g_sLcdHalDataBuf + cur_page* LCD_WIDTH, LCD_WIDTH);       
    }

    return 0;
    
}
#endif

/**************************************************************************
* Function: zDrvLcd_Gpio_Init
* Description: used to initialize GPIO usage.
* Parameters:
*       (IN)
*
*       (OUT)
*
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
**************************************************************************/
int set_ram_addr_color(unsigned short uiLeft, unsigned short uiTop)
{
	int ret = DRV_SUCCESS;

	ret |= zDrvLcd_Spi_WriteCmd(0x2A);
	ret |= zDrvLcd_Spi_WriteData(0x00);
	ret |= zDrvLcd_Spi_WriteData(var_screeninfo.xs);
	ret |= zDrvLcd_Spi_WriteData((var_screeninfo.xe>>8));
	ret |= zDrvLcd_Spi_WriteData((var_screeninfo.xe&0xff));


	ret |= zDrvLcd_Spi_WriteCmd(0x2B);
	ret |= zDrvLcd_Spi_WriteData(0x00);
	ret |= zDrvLcd_Spi_WriteData(var_screeninfo.ys);
	ret |= zDrvLcd_Spi_WriteData((var_screeninfo.ye>>8));
	ret |= zDrvLcd_Spi_WriteData((var_screeninfo.ye&0xff));

	return ret;
}

int update_rect_color(unsigned char *pBuf, unsigned short uiLeft, unsigned short uiTop, unsigned short uiWidth, unsigned short uiHeight)
{
	int ret = DRV_SUCCESS;
	
    ret |= set_ram_addr_color(uiLeft, uiTop); // ָˢµʼַ

    ret |= zDrvLcd_Spi_WriteCmd(0x2c);
    ret = zDrvLcd_Spi_WriteDataBlock((unsigned char *) pBuf, uiWidth * uiHeight * 2);
	
	return ret;
}

/**************************************************************************
* Function: prj_lcm_init
* Description: used to initialize SPI usage.
* Parameters:
*       (IN)
*
*       (OUT)
*
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
**************************************************************************/

int prj_lcm_init(void)
{
	int ret = DRV_SUCCESS;
	unsigned long uiRow, uiCol;
	int i = 0;
	
    if (s_lcd_info.lcm_info.lcd_item == LEAD_ST7735_128x128_F231E)
    {
		zDrvLcd_Spi_WriteCmd(0x11); //Sleep out
		zDrvLcd_DelayMs(120);
		
		zDrvLcd_Spi_WriteCmd(0xB1); //Frame rate 80Hz
		zDrvLcd_Spi_WriteData(0x02);
		zDrvLcd_Spi_WriteData(0x35);
		zDrvLcd_Spi_WriteData(0x36);
		
		zDrvLcd_Spi_WriteCmd(0xB2); //Frame rate 80Hz
		zDrvLcd_Spi_WriteData(0x02);
		zDrvLcd_Spi_WriteData(0x35);
		zDrvLcd_Spi_WriteData(0x36);
		
		zDrvLcd_Spi_WriteCmd(0xB3); //Frame rate 80Hz
		zDrvLcd_Spi_WriteData(0x02);
		zDrvLcd_Spi_WriteData(0x35);
		zDrvLcd_Spi_WriteData(0x36);
		zDrvLcd_Spi_WriteData(0x02);
		zDrvLcd_Spi_WriteData(0x35);
		zDrvLcd_Spi_WriteData(0x36);
		
		//------End ST7735S Frame rate----//
		zDrvLcd_Spi_WriteCmd(0xB4); //Dot inversion
		zDrvLcd_Spi_WriteData(0x03);
		
		//---ST7735S Power Sequence----//
		zDrvLcd_Spi_WriteCmd(0xC0);
		zDrvLcd_Spi_WriteData(0xA2);
		zDrvLcd_Spi_WriteData(0x02);
		zDrvLcd_Spi_WriteData(0x84);
		zDrvLcd_Spi_WriteCmd(0xC1);
		zDrvLcd_Spi_WriteData(0xC5);
		zDrvLcd_Spi_WriteCmd(0xC2);
		zDrvLcd_Spi_WriteData(0x0D);
		zDrvLcd_Spi_WriteData(0x00);
		zDrvLcd_Spi_WriteCmd(0xC3);
		zDrvLcd_Spi_WriteData(0x8D);
		zDrvLcd_Spi_WriteData(0x2A);
		zDrvLcd_Spi_WriteCmd(0xC4);
		zDrvLcd_Spi_WriteData(0x8D);
		zDrvLcd_Spi_WriteData(0xEE);
		
		//--------End ST7735S Power Sequence-----//
		zDrvLcd_Spi_WriteCmd(0xC5); //VCOM
		zDrvLcd_Spi_WriteData(0x0a);
		
		zDrvLcd_Spi_WriteCmd(0x36); //MX, MY, RGB mode
		zDrvLcd_Spi_WriteData(0xC8);
		
		//-------ST7735S Gamma Sequence---------//
		zDrvLcd_Spi_WriteCmd(0xE0);
		zDrvLcd_Spi_WriteData(0x12);
		zDrvLcd_Spi_WriteData(0x1C);
		zDrvLcd_Spi_WriteData(0x10);
		zDrvLcd_Spi_WriteData(0x18);
		zDrvLcd_Spi_WriteData(0x33);
		zDrvLcd_Spi_WriteData(0x2C);
		zDrvLcd_Spi_WriteData(0x25);
		zDrvLcd_Spi_WriteData(0x28);
		zDrvLcd_Spi_WriteData(0x28);
		zDrvLcd_Spi_WriteData(0x27);
		zDrvLcd_Spi_WriteData(0x2F);
		zDrvLcd_Spi_WriteData(0x3C);
		zDrvLcd_Spi_WriteData(0x00);
		zDrvLcd_Spi_WriteData(0x03);
		zDrvLcd_Spi_WriteData(0x03);
		zDrvLcd_Spi_WriteData(0x10);
		zDrvLcd_Spi_WriteCmd(0xE1);
		zDrvLcd_Spi_WriteData(0x12);
		zDrvLcd_Spi_WriteData(0x1C);
		zDrvLcd_Spi_WriteData(0x10);
		zDrvLcd_Spi_WriteData(0x18);
		zDrvLcd_Spi_WriteData(0x2D);
		zDrvLcd_Spi_WriteData(0x28);
		zDrvLcd_Spi_WriteData(0x23);
		zDrvLcd_Spi_WriteData(0x28);
		zDrvLcd_Spi_WriteData(0x28);
		zDrvLcd_Spi_WriteData(0x26);
		zDrvLcd_Spi_WriteData(0x2F);
		zDrvLcd_Spi_WriteData(0x3B);
		zDrvLcd_Spi_WriteData(0x00);
		zDrvLcd_Spi_WriteData(0x03);
		zDrvLcd_Spi_WriteData(0x03);
		zDrvLcd_Spi_WriteData(0x10);
		//---End ST7735S Gamma Sequence------//
		zDrvLcd_Spi_WriteCmd(0x3A); //65k mode
		zDrvLcd_Spi_WriteData(0x05);
		zDrvLcd_Spi_WriteCmd(0x29); //Display on
		
	}
	else if (s_lcd_info.lcm_info.lcd_item == LCM_ID_LEAD)
	{
		ret |= zDrvLcd_Spi_WriteCmd(0x11); //Sleep out
		zDrvLcd_DelayMs(120); //Delay 120ms

		ret |= zDrvLcd_Spi_WriteCmd(0xB1);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(0x08);
		ret |= zDrvLcd_Spi_WriteData(0x05);

		ret |= zDrvLcd_Spi_WriteCmd(0xB2);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(0x08);
		ret |= zDrvLcd_Spi_WriteData(0x05);

		ret |= zDrvLcd_Spi_WriteCmd(0xB3);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(0x08);
		ret |= zDrvLcd_Spi_WriteData(0x05);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(0x08);
		ret |= zDrvLcd_Spi_WriteData(0x05);

		ret |= zDrvLcd_Spi_WriteCmd(0xB4); //Column inversion
		ret |= zDrvLcd_Spi_WriteData(0x00); // Dot inversion in Normal mode

		ret |= zDrvLcd_Spi_WriteCmd(0xC0);
		ret |= zDrvLcd_Spi_WriteData(0xA2);//a2
		ret |= zDrvLcd_Spi_WriteData(0x02);
		ret |= zDrvLcd_Spi_WriteData(0x84);

		ret |= zDrvLcd_Spi_WriteCmd(0xC1);
		ret |= zDrvLcd_Spi_WriteData(0xC5);//c5

		ret |= zDrvLcd_Spi_WriteCmd(0xC2);
		ret |= zDrvLcd_Spi_WriteData(0x0A);
		ret |= zDrvLcd_Spi_WriteData(0x00);

		ret |= zDrvLcd_Spi_WriteCmd(0xC3);
		ret |= zDrvLcd_Spi_WriteData(0x8A);
		ret |= zDrvLcd_Spi_WriteData(0x2A);

		ret |= zDrvLcd_Spi_WriteCmd(0xC4);
		ret |= zDrvLcd_Spi_WriteData(0x8A);
		ret |= zDrvLcd_Spi_WriteData(0xEE);

		ret |= zDrvLcd_Spi_WriteCmd(0xC5); //VCOM
		ret |= zDrvLcd_Spi_WriteData(0x0E);

		ret |= zDrvLcd_Spi_WriteCmd(0x36); //MX, MY, RGB mode
		ret |= zDrvLcd_Spi_WriteData(LCD_MXMYRGB);

		ret |= zDrvLcd_Spi_WriteCmd(0xE0);
		ret |= zDrvLcd_Spi_WriteData(0x12);
		ret |= zDrvLcd_Spi_WriteData(0x18);
		ret |= zDrvLcd_Spi_WriteData(0x10);
		ret |= zDrvLcd_Spi_WriteData(0x18);
		ret |= zDrvLcd_Spi_WriteData(0x33);
		ret |= zDrvLcd_Spi_WriteData(0x2c);
		ret |= zDrvLcd_Spi_WriteData(0x25);
		ret |= zDrvLcd_Spi_WriteData(0x28);
		ret |= zDrvLcd_Spi_WriteData(0x28);
		ret |= zDrvLcd_Spi_WriteData(0x27);
		ret |= zDrvLcd_Spi_WriteData(0x2f);
		ret |= zDrvLcd_Spi_WriteData(0x3c);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(0x03);
		ret |= zDrvLcd_Spi_WriteData(0x03);
		ret |= zDrvLcd_Spi_WriteData(0x10);

		ret |= zDrvLcd_Spi_WriteCmd(0xE1);
		ret |= zDrvLcd_Spi_WriteData(0x12);
		ret |= zDrvLcd_Spi_WriteData(0x18);
		ret |= zDrvLcd_Spi_WriteData(0x10);
		ret |= zDrvLcd_Spi_WriteData(0x18);
		ret |= zDrvLcd_Spi_WriteData(0x2d);
		ret |= zDrvLcd_Spi_WriteData(0x28);
		ret |= zDrvLcd_Spi_WriteData(0x23);
		ret |= zDrvLcd_Spi_WriteData(0x28);
		ret |= zDrvLcd_Spi_WriteData(0x28);
		ret |= zDrvLcd_Spi_WriteData(0x26);
		ret |= zDrvLcd_Spi_WriteData(0x2f);
		ret |= zDrvLcd_Spi_WriteData(0x3b);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(0x03);
		ret |= zDrvLcd_Spi_WriteData(0x03);
		ret |= zDrvLcd_Spi_WriteData(0x10);

		ret |= zDrvLcd_Spi_WriteCmd(0x2A);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(LCD_XS);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(LCD_XE);

		ret |= zDrvLcd_Spi_WriteCmd(0x2B);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(LCD_YS);
		ret |= zDrvLcd_Spi_WriteData(0x00);
		ret |= zDrvLcd_Spi_WriteData(LCD_YE);

		ret |= zDrvLcd_Spi_WriteCmd(0x3A); //65k mode
		ret |= zDrvLcd_Spi_WriteData(0x05);

		ret |= zDrvLcd_Spi_WriteCmd(0x29); //Display on
		zDrvLcd_DelayMs(200); //Delay 200ms

		ret |= zDrvLcd_Spi_WriteCmd(0x2C);
	}
	else
	{
		zDrvLcd_Spi_WriteCmd(0x11); //Exit Sleep
		zDrvLcd_DelayMs(150);

		zDrvLcd_Spi_WriteCmd(0x26); //Set Default Gamma
		zDrvLcd_Spi_WriteData(0x04);

		zDrvLcd_Spi_WriteCmd(0xB1);
		zDrvLcd_Spi_WriteData(0x0B);
		zDrvLcd_Spi_WriteData(0x14);

		zDrvLcd_Spi_WriteCmd(0xC0); //Set VRH1[4:0] & VC[2:0] for VCI1 & GVDD
		zDrvLcd_Spi_WriteData(0x10);
		zDrvLcd_Spi_WriteData(0x00);

		zDrvLcd_Spi_WriteCmd(0xC1); //Set BT[2:0] for AVDD & VCL & VGH & VGL
		zDrvLcd_Spi_WriteData(0x05);

		zDrvLcd_Spi_WriteCmd(0xC5); //Set VMH[6:0] & VML[6:0] for VOMH & VCOML
		zDrvLcd_Spi_WriteData(0x46);
		zDrvLcd_Spi_WriteData(0x40);

		zDrvLcd_Spi_WriteCmd(0xC7);
		zDrvLcd_Spi_WriteData(0xBD);

		zDrvLcd_Spi_WriteCmd(0xEC);
		zDrvLcd_Spi_WriteData(0x0C);

		zDrvLcd_Spi_WriteCmd(0x3a); //Set Color Format
		zDrvLcd_Spi_WriteData(0x05);

		zDrvLcd_Spi_WriteCmd(0x2A); //Set Column Address
		zDrvLcd_Spi_WriteData(0x00);
		zDrvLcd_Spi_WriteData(0x00);
		zDrvLcd_Spi_WriteData(0x00);
		zDrvLcd_Spi_WriteData(0x7F);

		zDrvLcd_Spi_WriteCmd(0x2B); //Set Page Address
		zDrvLcd_Spi_WriteData(0x00);
		zDrvLcd_Spi_WriteData(0x00);
		zDrvLcd_Spi_WriteData(0x00);
		zDrvLcd_Spi_WriteData(0x9F);

		zDrvLcd_Spi_WriteCmd(0x36); //Set Scanning Direction
		zDrvLcd_Spi_WriteData(LCD_MXMYRGB);

		zDrvLcd_Spi_WriteCmd(0xB7); //Set Source Output Direction
		zDrvLcd_Spi_WriteData(0x00);

		zDrvLcd_Spi_WriteCmd(0xf2); //Enable Gamma bit
		zDrvLcd_Spi_WriteData(0x01);

		zDrvLcd_Spi_WriteCmd(0xE0);
		zDrvLcd_Spi_WriteData(0x3F);//p1
		zDrvLcd_Spi_WriteData(0x29);//p2
		zDrvLcd_Spi_WriteData(0x26);//p3
		zDrvLcd_Spi_WriteData(0x26);//p4
		zDrvLcd_Spi_WriteData(0x26);//p5
		zDrvLcd_Spi_WriteData(0x0C);//p6
		zDrvLcd_Spi_WriteData(0x51);//p7
		zDrvLcd_Spi_WriteData(0xB8);//p8
		zDrvLcd_Spi_WriteData(0x39);//p9
		zDrvLcd_Spi_WriteData(0x17);//p10
		zDrvLcd_Spi_WriteData(0x00);//p11
		zDrvLcd_Spi_WriteData(0x00);//p12
		zDrvLcd_Spi_WriteData(0x00);//p13
		zDrvLcd_Spi_WriteData(0x00);//p14
		zDrvLcd_Spi_WriteData(0x00);//p15

		zDrvLcd_Spi_WriteCmd(0xE1);
		zDrvLcd_Spi_WriteData(0x00);//p1
		zDrvLcd_Spi_WriteData(0x16);//p2
		zDrvLcd_Spi_WriteData(0x19);//p3
		zDrvLcd_Spi_WriteData(0x19);//p4
		zDrvLcd_Spi_WriteData(0x19);//p5
		zDrvLcd_Spi_WriteData(0x13);//p6
		zDrvLcd_Spi_WriteData(0x2E);//p7
		zDrvLcd_Spi_WriteData(0x47);//p8
		zDrvLcd_Spi_WriteData(0x46);//p9
		zDrvLcd_Spi_WriteData(0x08);//p10
		zDrvLcd_Spi_WriteData(0x3F);//p11
		zDrvLcd_Spi_WriteData(0x3F);//p12
		zDrvLcd_Spi_WriteData(0x3F);//p13
		zDrvLcd_Spi_WriteData(0x3F);//p14
		zDrvLcd_Spi_WriteData(0x3F);//p15

		zDrvLcd_Spi_WriteCmd(0x29); // Display On

		zDrvLcd_Spi_WriteCmd(0x2C);
	}
	
	// ˢɺɫ
	for (uiRow = 0; uiRow < LCD_WIDTH; uiRow++)
	{
		for (uiCol = 0; uiCol < LCD_HEIGHT; uiCol++)
		{
			ret |= zDrvLcd_Spi_WriteData(0x00);
			ret |= zDrvLcd_Spi_WriteData(0x00);
		}
	}

	return ret;
}

/**************************************************************************
* Function: prj_lcd_get_info
* Description: used to get the LCD device information.
* Parameters:
*       (IN)
*               pInfo    the pointer of LCD device information to store into
*       (OUT)
*               pInfo   the pointer of LCD device information allready stored into
* Returns:
*       DRV_SUCCESS: succeeded.
* Others:
*       None.
**************************************************************************/
int prj_lcd_get_info(T_ZDrvLcd_Info* pInfo)
{
	unsigned char tmp;

	pInfo->width = LCD_WIDTH;
	pInfo->height = LCD_HEIGHT;
	pInfo->bitsPerPixel = LCD_BITS_PER_PIXEL;
	pInfo->rMask = 0xF800;
	pInfo->gMask = 0x07E0;
	pInfo->bMask = 0x001F;
	pInfo->instance = g_sLcd_Instance;
	pInfo->backlight = g_sLcd_Backlight;
	pInfo->brightness = g_sLcd_Brightness;
	
	tmp = read_st(0xda);//lcd module's manufacturer ID
	pInfo->lcm_info.man_id = tmp;
	tmp = read_st(0xdb);//lcd module's	ID
	pInfo->lcm_info.dev_id = tmp<<8;
	tmp = read_st(0xdc);//lcd module's	ID
	pInfo->lcm_info.dev_id |= tmp;
	
	if(pInfo->lcm_info.man_id == LCM_ID_LEAD)
	{		
#if 0
		var_screeninfo.xs = 2;
		#ifdef LCD_ROTATE_180
		var_screeninfo.ys = 1;
		#else
		var_screeninfo.ys = 3;
		#endif
		var_screeninfo.width = 128;
		var_screeninfo.height = 128;
		var_screeninfo.xe = var_screeninfo.width + var_screeninfo.xs - 1;
		var_screeninfo.ye = var_screeninfo.height + var_screeninfo.ys - 1;
		
		pInfo->lcm_info.lcd_item = LEAD_ST7735_128x128;
		pInfo->lcm_info.name = "LEAD_ST7735_128x128";
#else
		var_screeninfo.xs = 0;
		var_screeninfo.ys = 32;
		var_screeninfo.width = 128;
		var_screeninfo.height = 128;
		var_screeninfo.xe = var_screeninfo.width + var_screeninfo.xs - 1;
		var_screeninfo.ye = var_screeninfo.height + var_screeninfo.ys - 1;

		pInfo->lcm_info.lcd_item = LEAD_ST7735_128x128_F231E; //7C 89 F0
		pInfo->lcm_info.name = "LEAD_ST7735_128x128_F231E";
#endif
	}
	else //default s93521a 128*160
	{		
		pInfo->lcm_info.lcd_item = S93521A_128x128;
		pInfo->lcm_info.name = "128*128 lcd no found,default S93521A";
	}
		
	BOOT_PRINTF(UBOOT_NOTICE, "%s init: lcd man_id = 0x%x, dev_id=0x%x\n", pInfo->lcm_info.name, pInfo->lcm_info.man_id,pInfo->lcm_info.dev_id);
	return 0;
}

int update_rect(unsigned char *pBuf, unsigned short uiLeft, unsigned short uiTop, unsigned short uiWidth, unsigned short uiHeight)
{
	return update_rect_color(pBuf,uiLeft,uiTop,uiWidth,uiHeight);
}

