/***********************************************************************
 * Copyright (C) 2014, ZTE Corporation.
 *
 * File Name:   lcd_comm.c
 * File Mark:
 * Description: Lcd function implementation. Adapted to LCM MODEL - LEAD T15DS26.
 * Others:      this file is in BOOTLOADER procedure.
 *                  This LCM MODEL uses SPI-4.
 *                  NOTE: 4-lines-spi uses CS, CLK, D/CX and SDI. Do not using SDO instead of D/CX.
 * Version:     V1.0
 * Author:      qihongfang
 * Date:        2017-9-18
 *
 * History 1:
 *     Date: 2015-1-16
 *     Version:
 *     Author: YinWenguan
 *     Modification: Add brightness setting.
 *
***********************************************************************/

/*************************************************************************
 * Include files
 *************************************************************************/
#include "../include/board.h"

#include <common.h>
#include <malloc.h>
#include <asm/io.h>
#include "../include/spi.h"
#include "../include/ssp.h"
#include <lcd.h>
#include <drvs_gpio.h>
#include <peripheral.h>

/*************************************************************************
 * Macro
 *************************************************************************/
 /* debug switch */

//static unsigned char g_lcd_id=0;
#define LCM_ID_LEAD    		0x7C  /*йst7735s\st7735s*/
#define LCM_ID_ZGD    		0x00  /*й GC9105 IVO1.77*/
#define LCM_ID_ILI			0x54  /*й GC9105 IVO1.77*/
#define LCM_ID_ILI9342C		0xE3  /* 00 00 */

#define LCM_ID_ZGD_9105		0x9105	// 128*160 
#define LCM_ID_ILI_9163		0x8066	// 128*160 
#define LCM_ID_ZGD_9305		0x9305 	// 240*320
#define LCM_ID_ZGD_9306		0x9306 	// 240*320

#define LCM_ID_ST			0x85
#define LCM_ID_ST7789V		0x8552		//240 * 320


#define ZSP_LCDINFO_PATH "/LCDINFO.bin"
#define LCD_BITS_PER_PIXEL	(5 + 6 + 5)


/* rotate display */
//#define LCD_ROTATE_180

#ifdef LCD_ROTATE_180
#define LCD_MXMYRGB			0x14
#else
#define LCD_MXMYRGB			0xC4
#endif

struct lcd_var_screeninfo {
	unsigned int xs;	//xs_offset	
	unsigned int ys;    //ys_offset
	unsigned int width;	//128 	240 	
	unsigned int height; //128 	240		
	unsigned int xe;	 //width + xs - 1	
	unsigned int ye;     //height + ys - 1
	unsigned int xs_offset; //0x02 0x00 			
	unsigned int ys_offset; //0x03 0x00		
	//unsigned int bits_per_pixel;		
};
static struct lcd_var_screeninfo var_screeninfo = {0};	/* Current var */


/**************************************************************************
 *                                  Types                                                                                   *
 **************************************************************************/
static volatile T_ZDrvLcd_Instance			g_sLcd_Instance = {false, LCD_NORMAL, LCD_XFER_POLL, LCD_DCX_CMD};
static volatile T_ZDrvLcd_BacklightStatus	g_sLcd_Backlight = LCD_BACKLIGHT_OFF;
static volatile T_ZDrvLcd_Brightness		g_sLcd_Brightness = 255;

/**************************************************************************
 * Global  Variable
 **************************************************************************/
extern volatile T_ZDrvLcd_Info s_lcd_info;
extern int zDrvLcd_Spi_WriteCmd(unsigned char cmd);
extern int zDrvLcd_Spi_WriteData(unsigned char data);

/**************************************************************************
 * Function Implementation
 **************************************************************************/
void lcd_gpio_define(struct T_ZDrvLcd_gpio_define* config)
{
	config->spi_cs_gpio			= GPIO25;
	config->spi_scl_gpio		= GPIO26;
	config->spi_sda_gpio		= GPIO28;
	config->spi_dcx_gpio		= GPIO27;

	config->spi_cs_func_sel		= GPIO25_SSP0_CS;
	config->spi_scl_func_sel	= GPIO26_SSP0_CLK;
	config->spi_sda_func_sel	= GPIO28_SSP0_TXD;
	config->spi_dcx_func_sel	= GPIO27_SSP0_RXD;

	config->spi_cs_gpio_sel		= GPIO25_GPIO25;
	config->spi_scl_gpio_sel	= GPIO26_GPIO26;
	config->spi_sda_gpio_sel	= GPIO28_GPIO28;
	config->spi_dcx_gpio_sel	= GPIO27_GPIO27;

	config->lcd_reset_gpio		=GPIO120;
	config->lcd_reset_gpio_sel	=GPIO120_GPIO120;
	
	config->lcd_blg_gpio		=GPIO119;
	config->lcd_blg_gpio_sel	=GPIO119_GPIO119;
}

/**************************************************************************
* Function: zDrvLcd_Gpio_Init
* Description: used to initialize GPIO usage.
* Parameters:
*       (IN)
*
*       (OUT)
*
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
**************************************************************************/
int set_ram_addr_color(unsigned short uiLeft, unsigned short uiTop)
{
	int ret = DRV_SUCCESS;

	ret |= zDrvLcd_Spi_WriteCmd(0x2A);
	ret |= zDrvLcd_Spi_WriteData(0x00);
	ret |= zDrvLcd_Spi_WriteData(var_screeninfo.xs);
	ret |= zDrvLcd_Spi_WriteData((var_screeninfo.xe >> 8));
	ret |= zDrvLcd_Spi_WriteData((var_screeninfo.xe &0xff));


	ret |= zDrvLcd_Spi_WriteCmd(0x2B);
	ret |= zDrvLcd_Spi_WriteData(0x00);
	ret |= zDrvLcd_Spi_WriteData(var_screeninfo.ys);
	ret |= zDrvLcd_Spi_WriteData((var_screeninfo.ye >> 8));
	ret |= zDrvLcd_Spi_WriteData((var_screeninfo.ye & 0xff));

	return ret;
}

int update_rect_color(unsigned char *pBuf, unsigned short uiLeft, unsigned short uiTop, unsigned short uiWidth, unsigned short uiHeight)
{
	int ret = DRV_SUCCESS;
	
    ret |= set_ram_addr_color(uiLeft, uiTop); // ָˢµʼַ

    ret |= zDrvLcd_Spi_WriteCmd(0x2c);
    ret = zDrvLcd_Spi_WriteDataBlock((unsigned char *) pBuf, uiWidth * uiHeight * 2);
	
	return ret;
}

/**************************************************************************
* Function: prj_lcm_init
* Description: used to initialize SPI usage.
* Parameters:
*       (IN)
*
*       (OUT)
*
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:
**************************************************************************/
int prj_lcm_init(void)
{
    int ret = DRV_SUCCESS;
    unsigned long uiRow, uiCol;
	uint8_t page_number,column_number;
	
	if (s_lcd_info.width == 240 && s_lcd_info.height == 320) 
	{
		if(s_lcd_info.lcm_info.lcd_item == ST7789V_240x320)
		{
			zDrvLcd_Spi_WriteCmd(0x11); 
			zDrvLcd_DelayMs(150);                //Delay 120ms 
			//--------------------------------ST7789S Frame rate setting----------------------------------// 
			zDrvLcd_Spi_WriteCmd(0xb2); 
			zDrvLcd_Spi_WriteData(0x0c); 
			zDrvLcd_Spi_WriteData(0x0c); 
			zDrvLcd_Spi_WriteData(0x00); 
			zDrvLcd_Spi_WriteData(0x33); 
			zDrvLcd_Spi_WriteData(0x33); 
			zDrvLcd_Spi_WriteCmd(0xb7); 
			zDrvLcd_Spi_WriteData(0x35); 
			//---------------------------------ST7789S Power setting--------------------------------------// 
			zDrvLcd_Spi_WriteCmd(0xbb); 
			zDrvLcd_Spi_WriteData(0x2b); 
			zDrvLcd_Spi_WriteCmd(0xc0); 
			zDrvLcd_Spi_WriteData(0x2c); 
			zDrvLcd_Spi_WriteCmd(0xc2); 
			zDrvLcd_Spi_WriteData(0x01); 
			zDrvLcd_Spi_WriteCmd(0xc3); 
			zDrvLcd_Spi_WriteData(0x17); 
			zDrvLcd_Spi_WriteCmd(0xc4); 
			zDrvLcd_Spi_WriteData(0x20); 
			zDrvLcd_Spi_WriteCmd(0xc6); 
			zDrvLcd_Spi_WriteData(0x0f); 
			zDrvLcd_Spi_WriteCmd(0xca); 
			zDrvLcd_Spi_WriteData(0x0f); 
			zDrvLcd_Spi_WriteCmd(0xc8); 
			zDrvLcd_Spi_WriteData(0x08); 
			zDrvLcd_Spi_WriteCmd(0x55); 
			zDrvLcd_Spi_WriteData(0x90); 
			zDrvLcd_Spi_WriteCmd(0xd0); 
			zDrvLcd_Spi_WriteData(0xa4); 
			zDrvLcd_Spi_WriteData(0xa1); 
			//--------------------------------ST7789S gamma setting---------------------------------------// 
			zDrvLcd_Spi_WriteCmd(0xe0); 
			zDrvLcd_Spi_WriteData(0xf0); 
			zDrvLcd_Spi_WriteData(0x00); 
			zDrvLcd_Spi_WriteData(0x0a); 
			zDrvLcd_Spi_WriteData(0x10); 
			zDrvLcd_Spi_WriteData(0x12); 
			zDrvLcd_Spi_WriteData(0x1b); 
			zDrvLcd_Spi_WriteData(0x39); 
			zDrvLcd_Spi_WriteData(0x44); 
			zDrvLcd_Spi_WriteData(0x47); 
			zDrvLcd_Spi_WriteData(0x28); 
			zDrvLcd_Spi_WriteData(0x12); 
			zDrvLcd_Spi_WriteData(0x10); 
			zDrvLcd_Spi_WriteData(0x16); 
			zDrvLcd_Spi_WriteData(0x1b); 
			zDrvLcd_Spi_WriteCmd(0xe1); 
			zDrvLcd_Spi_WriteData(0xf0); 
			zDrvLcd_Spi_WriteData(0x00); 
			zDrvLcd_Spi_WriteData(0x0a); 
			zDrvLcd_Spi_WriteData(0x10); 
			zDrvLcd_Spi_WriteData(0x11); 
			zDrvLcd_Spi_WriteData(0x1a); 
			zDrvLcd_Spi_WriteData(0x3b); 
			zDrvLcd_Spi_WriteData(0x34); 
			zDrvLcd_Spi_WriteData(0x4e); 
			zDrvLcd_Spi_WriteData(0x3a); 
			zDrvLcd_Spi_WriteData(0x17); 
			zDrvLcd_Spi_WriteData(0x16); 
			zDrvLcd_Spi_WriteData(0x21); 
			zDrvLcd_Spi_WriteData(0x22); 

			zDrvLcd_Spi_WriteCmd(0x36); 
			zDrvLcd_Spi_WriteData(0x00); 
			zDrvLcd_Spi_WriteCmd(0x3A); 
			zDrvLcd_Spi_WriteData(0x05);
		       zDrvLcd_Spi_WriteCmd(0x35);/*enable TE*/
			zDrvLcd_Spi_WriteCmd(0x29);
			zDrvLcd_Spi_WriteCmd(0x2c);
		}
		else if(s_lcd_info.lcm_info.lcd_item == ZGD_GC9306_240x320)
		{
	        zDrvLcd_Spi_WriteCmd(0x11); //Exit Sleep
			zDrvLcd_DelayMs(120);
		    //--------------------------------display control setting----------------------------------------//
	        zDrvLcd_Spi_WriteCmd(0xfe);
	        zDrvLcd_Spi_WriteCmd(0xef);
	        zDrvLcd_Spi_WriteCmd(0x36);
	        zDrvLcd_Spi_WriteData(0x48);
	        zDrvLcd_Spi_WriteCmd(0x3a);
	        zDrvLcd_Spi_WriteData(0x05);
	        zDrvLcd_Spi_WriteCmd(0xad);
	        zDrvLcd_Spi_WriteData(0x33);
	        zDrvLcd_Spi_WriteCmd(0xaf);
	        zDrvLcd_Spi_WriteData(0x55);
	        zDrvLcd_Spi_WriteCmd(0xae);
	        zDrvLcd_Spi_WriteData(0x2b);
	        //------------------------------end display control setting--------------------------------//
	        //------------------------------Power Control Registers Initial------------------------------//
	        zDrvLcd_Spi_WriteCmd(0xa4);
	        zDrvLcd_Spi_WriteData(0x44);
	        zDrvLcd_Spi_WriteData(0x44);
	        zDrvLcd_Spi_WriteCmd(0xa5);
	        zDrvLcd_Spi_WriteData(0x42);
	        zDrvLcd_Spi_WriteData(0x42);
	        zDrvLcd_Spi_WriteCmd(0xaa);
	        zDrvLcd_Spi_WriteData(0x88);
	        zDrvLcd_Spi_WriteData(0x88);
	        zDrvLcd_Spi_WriteCmd(0xae);
	        zDrvLcd_Spi_WriteData(0x2b);
	        zDrvLcd_Spi_WriteCmd(0xe8);
	        zDrvLcd_Spi_WriteData(0x11);
	        zDrvLcd_Spi_WriteData(0x0b);
	        zDrvLcd_Spi_WriteCmd(0xe3);
	        zDrvLcd_Spi_WriteData(0x01);
	        zDrvLcd_Spi_WriteData(0x10);
	        zDrvLcd_Spi_WriteCmd(0xff);
	        zDrvLcd_Spi_WriteData(0x61);
	        zDrvLcd_Spi_WriteCmd(0xAC);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteCmd(0xAf);
	        zDrvLcd_Spi_WriteData(0x67);
	        zDrvLcd_Spi_WriteCmd(0xa6);
	        zDrvLcd_Spi_WriteData(0x2a);
	        zDrvLcd_Spi_WriteData(0x2a);
	        zDrvLcd_Spi_WriteCmd(0xa7);
	        zDrvLcd_Spi_WriteData(0x2b);
	        zDrvLcd_Spi_WriteData(0x2b);
	        zDrvLcd_Spi_WriteCmd(0xa8);
	        zDrvLcd_Spi_WriteData(0x18);
	        zDrvLcd_Spi_WriteData(0x18);
	        zDrvLcd_Spi_WriteCmd(0xa9);
	        zDrvLcd_Spi_WriteData(0x2a);
	        zDrvLcd_Spi_WriteData(0x2a);
	        //------------------------end Power Control Registers Initial------------------------------//
	        //----------------------------display window 240X320------------------------------------//
	        zDrvLcd_Spi_WriteCmd(0x2a);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0xef);
	        zDrvLcd_Spi_WriteCmd(0x2b);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x01);
	        zDrvLcd_Spi_WriteData(0x3f);
	        zDrvLcd_Spi_WriteCmd(0x2c);



	        zDrvLcd_Spi_WriteCmd(0xF0);
	        zDrvLcd_Spi_WriteData(0x02);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x00);
	        
	        zDrvLcd_Spi_WriteCmd(0xF1);
	        zDrvLcd_Spi_WriteData(0x01);
	        zDrvLcd_Spi_WriteData(0x02);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x09);
	        zDrvLcd_Spi_WriteData(0x1C);
	        
	        zDrvLcd_Spi_WriteCmd(0xF2);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x02);
	        zDrvLcd_Spi_WriteData(0x1D);
	        zDrvLcd_Spi_WriteData(0x05);
	        zDrvLcd_Spi_WriteData(0x04);
	        zDrvLcd_Spi_WriteData(0x2F);
	        
	        zDrvLcd_Spi_WriteCmd(0xF3);
	        zDrvLcd_Spi_WriteData(0x1A);
	        zDrvLcd_Spi_WriteData(0x0E);
	        zDrvLcd_Spi_WriteData(0x50);
	        zDrvLcd_Spi_WriteData(0x03);
	        zDrvLcd_Spi_WriteData(0x03);
	        zDrvLcd_Spi_WriteData(0x61);
	        
	        zDrvLcd_Spi_WriteCmd(0xF4);
	        zDrvLcd_Spi_WriteData(0x0E);
	        zDrvLcd_Spi_WriteData(0x1B);
	        zDrvLcd_Spi_WriteData(0x1A);
	        zDrvLcd_Spi_WriteData(0x02);
	        zDrvLcd_Spi_WriteData(0x05);
	        zDrvLcd_Spi_WriteData(0x0F);
	        
	        zDrvLcd_Spi_WriteCmd(0xF5);
	        zDrvLcd_Spi_WriteData(0x05);
	        zDrvLcd_Spi_WriteData(0x0F);
	        zDrvLcd_Spi_WriteData(0x0F);
	        zDrvLcd_Spi_WriteData(0x31);
	        zDrvLcd_Spi_WriteData(0x31);
	        zDrvLcd_Spi_WriteData(0x0F);
	        //------------------------------------end gamma setting-----------------------------------------//
	        zDrvLcd_Spi_WriteCmd(0x11);
	        zDrvLcd_DelayMs(120);
	        zDrvLcd_Spi_WriteCmd(0x29);
	        zDrvLcd_Spi_WriteCmd(0x2c);
		}
		else //if(g_lcm_id.lcd_item = ZGD_GC930x_240x320)
		{
		       zDrvLcd_Spi_WriteCmd(0x11); //Exit Sleep
			zDrvLcd_DelayMs(120);
			zDrvLcd_Spi_WriteCmd(0xfe);// internal reg enable	
			zDrvLcd_Spi_WriteCmd(0xef);//// internal reg enable	
			zDrvLcd_Spi_WriteCmd(0x36);
			zDrvLcd_Spi_WriteData(0x48);
			zDrvLcd_Spi_WriteCmd(0x3a);
			zDrvLcd_Spi_WriteData(0x05);
			zDrvLcd_Spi_WriteCmd(0xe8);
			zDrvLcd_Spi_WriteData(0x12);
			zDrvLcd_Spi_WriteData(0x22);
				//=============		

			zDrvLcd_Spi_WriteCmd(0xe3);
			zDrvLcd_Spi_WriteData(0x01);
			zDrvLcd_Spi_WriteData(0x04);
			zDrvLcd_Spi_WriteCmd(0xa5);
			zDrvLcd_Spi_WriteData(0x40);
			zDrvLcd_Spi_WriteData(0x40);
			zDrvLcd_Spi_WriteCmd(0xa4);
			zDrvLcd_Spi_WriteData(0x44);
			zDrvLcd_Spi_WriteData(0x44);
			zDrvLcd_Spi_WriteCmd(0xab);
			zDrvLcd_Spi_WriteData(0x08);
			zDrvLcd_Spi_WriteCmd(0xaa);
			zDrvLcd_Spi_WriteData(0x88);
			zDrvLcd_Spi_WriteData(0x88);
			zDrvLcd_Spi_WriteCmd(0xae);
			zDrvLcd_Spi_WriteData(0x0b);//0x1c  20151013 update
			zDrvLcd_Spi_WriteCmd(0xac);
			zDrvLcd_Spi_WriteData(0x00);
			zDrvLcd_Spi_WriteCmd(0xaf);

			zDrvLcd_Spi_WriteData(0x77);
			zDrvLcd_Spi_WriteCmd(0xad);
			zDrvLcd_Spi_WriteData(0x77);

				
			//===============gamma==========		
			zDrvLcd_Spi_WriteCmd(0xF0);
			zDrvLcd_Spi_WriteData(0x02);
			zDrvLcd_Spi_WriteData(0x02);

			zDrvLcd_Spi_WriteData(0x00);
			zDrvLcd_Spi_WriteData(0x09);
			zDrvLcd_Spi_WriteData(0x13);
			zDrvLcd_Spi_WriteData(0x0E);
			zDrvLcd_Spi_WriteCmd(0xF1);
			zDrvLcd_Spi_WriteData(0x01);
			zDrvLcd_Spi_WriteData(0x02);
			zDrvLcd_Spi_WriteData(0x00);
			zDrvLcd_Spi_WriteData(0x0F);
			zDrvLcd_Spi_WriteData(0x1C);
			zDrvLcd_Spi_WriteData(0x10);
			zDrvLcd_Spi_WriteCmd(0xF2);
			zDrvLcd_Spi_WriteData(0x0F);
			zDrvLcd_Spi_WriteData(0x08);
			zDrvLcd_Spi_WriteData(0x39);
			zDrvLcd_Spi_WriteData(0x04);
			zDrvLcd_Spi_WriteData(0x05);
			zDrvLcd_Spi_WriteData(0x49);
			zDrvLcd_Spi_WriteCmd(0xF3);
			zDrvLcd_Spi_WriteData(0x11);
			zDrvLcd_Spi_WriteData(0x0A);

			zDrvLcd_Spi_WriteData(0x41);
			zDrvLcd_Spi_WriteData(0x03);
			zDrvLcd_Spi_WriteData(0x03);
			zDrvLcd_Spi_WriteData(0x4F);
			zDrvLcd_Spi_WriteCmd(0xF4);
			zDrvLcd_Spi_WriteData(0x0F);
			zDrvLcd_Spi_WriteData(0x18);
			zDrvLcd_Spi_WriteData(0x16);
			zDrvLcd_Spi_WriteData(0x1D);
			zDrvLcd_Spi_WriteData(0x20);
			zDrvLcd_Spi_WriteData(0x0F);
			zDrvLcd_Spi_WriteCmd(0xF5);
			zDrvLcd_Spi_WriteData(0x05);
			zDrvLcd_Spi_WriteData(0x11);
			zDrvLcd_Spi_WriteData(0x11);
			zDrvLcd_Spi_WriteData(0x1C);
			zDrvLcd_Spi_WriteData(0x1F);
			zDrvLcd_Spi_WriteData(0x0F);
			zDrvLcd_Spi_WriteCmd(0x11);
			zDrvLcd_Spi_WriteCmd(0x35);
			zDrvLcd_DelayMs(120);

			zDrvLcd_Spi_WriteCmd(0x29);
			zDrvLcd_Spi_WriteCmd(0x2c);
		}
	}
	
	if (s_lcd_info.width == 128 && s_lcd_info.height == 160) 
	{
		 if(s_lcd_info.lcm_info.lcd_item == ZGD_GC910x_128x160)
		 {
	        	zDrvLcd_Spi_WriteCmd(0x11); //Exit Sleep
	        	zDrvLcd_DelayMs(150);
	/*display control setting*/
			zDrvLcd_Spi_WriteCmd(0xfe);
			zDrvLcd_Spi_WriteCmd(0xef);
			zDrvLcd_Spi_WriteCmd(0x3a);
			zDrvLcd_Spi_WriteData(0x05);
			zDrvLcd_Spi_WriteCmd(0x36);
			zDrvLcd_Spi_WriteData(0xc0);
			zDrvLcd_Spi_WriteCmd(0xb4);
			zDrvLcd_Spi_WriteData(0x00);
	/*end display controlsetting*/

	/*Power Control Register init*/
			zDrvLcd_Spi_WriteCmd(0xea);
			zDrvLcd_Spi_WriteData(0x3a);
			zDrvLcd_Spi_WriteCmd(0xff);
			zDrvLcd_Spi_WriteData(0x0b);
			zDrvLcd_Spi_WriteCmd(0xfb);
			zDrvLcd_Spi_WriteData(0x1b);
			zDrvLcd_Spi_WriteCmd(0xe3);
			zDrvLcd_Spi_WriteData(0x07);
			zDrvLcd_Spi_WriteCmd(0xa4);
			zDrvLcd_Spi_WriteData(0x0a);
			zDrvLcd_Spi_WriteCmd(0xe7);
			zDrvLcd_Spi_WriteData(0x94);
			zDrvLcd_Spi_WriteData(0x88);
			zDrvLcd_Spi_WriteCmd(0xed);
			zDrvLcd_Spi_WriteData(0x11);
			zDrvLcd_Spi_WriteCmd(0xe4);
			zDrvLcd_Spi_WriteData(0xc5);
			zDrvLcd_Spi_WriteCmd(0xe2);
			zDrvLcd_Spi_WriteData(0x80);
			zDrvLcd_Spi_WriteCmd(0xe5);
			zDrvLcd_Spi_WriteData(0x10);
	/*end Power Control Register init*/

	/*display windows 128*160*/
			zDrvLcd_Spi_WriteCmd(0x2a);
			zDrvLcd_Spi_WriteData(0x00);
			zDrvLcd_Spi_WriteData(0x00);
			zDrvLcd_Spi_WriteData(0x00);
			zDrvLcd_Spi_WriteData(0x7f);
			zDrvLcd_Spi_WriteCmd(0x2b);
			zDrvLcd_Spi_WriteData(0x00);
			zDrvLcd_Spi_WriteData(0x00);
			zDrvLcd_Spi_WriteData(0x00);
			zDrvLcd_Spi_WriteData(0x9f);
			zDrvLcd_Spi_WriteCmd(0x2c);
	/*end display windows 128*160*/

	/*gamma setting*/
			zDrvLcd_Spi_WriteCmd(0xf0);
			zDrvLcd_Spi_WriteData(0x30);
			zDrvLcd_Spi_WriteCmd(0xf1);
			zDrvLcd_Spi_WriteData(0x01);
			zDrvLcd_Spi_WriteCmd(0xf2);
			zDrvLcd_Spi_WriteData(0x00);
			zDrvLcd_Spi_WriteCmd(0xf3);
			zDrvLcd_Spi_WriteData(0x45);
			zDrvLcd_Spi_WriteCmd(0xf4);
			zDrvLcd_Spi_WriteData(0x00);
			zDrvLcd_Spi_WriteCmd(0xf5);
			zDrvLcd_Spi_WriteData(0x00);
			zDrvLcd_Spi_WriteCmd(0xf7);
			zDrvLcd_Spi_WriteData(0x35);
			zDrvLcd_Spi_WriteCmd(0xf8);
			zDrvLcd_Spi_WriteData(0x10);
			zDrvLcd_Spi_WriteCmd(0xf9);
			zDrvLcd_Spi_WriteData(0x74);
			zDrvLcd_Spi_WriteCmd(0xfa);
			zDrvLcd_Spi_WriteData(0x44);
			zDrvLcd_Spi_WriteCmd(0xfb);
			zDrvLcd_Spi_WriteData(0x00);
			zDrvLcd_Spi_WriteCmd(0xfc);
			zDrvLcd_Spi_WriteData(0x00);
	/*end gamma setting*/

			zDrvLcd_Spi_WriteCmd(0x11);
			/*Delayms(120);*/
			udelay(100000/1000 *120);
			zDrvLcd_Spi_WriteCmd(0x29);

			zDrvLcd_Spi_WriteCmd(0x2C);
		}
	    else if(s_lcd_info.lcm_info.lcd_item ==  ZGD_ST7735_128x160)
	    {
	        ret |= zDrvLcd_Spi_WriteCmd(0x11); //Sleep out
	        zDrvLcd_DelayMs(120); //Delay 120ms

	        ret |= zDrvLcd_Spi_WriteCmd(0xB1);
	        ret |= zDrvLcd_Spi_WriteData(0x0f);//0x01
	        ret |= zDrvLcd_Spi_WriteData(0x18);//0x08
	        ret |= zDrvLcd_Spi_WriteData(0x01);//0x05

	        ret |= zDrvLcd_Spi_WriteCmd(0xB2);
	        ret |= zDrvLcd_Spi_WriteData(0x05);
	        ret |= zDrvLcd_Spi_WriteData(0x3C);
	        ret |= zDrvLcd_Spi_WriteData(0x3C);

	        ret |= zDrvLcd_Spi_WriteCmd(0xB3);
	        ret |= zDrvLcd_Spi_WriteData(0x05);
	        ret |= zDrvLcd_Spi_WriteData(0x3C);
	        ret |= zDrvLcd_Spi_WriteData(0x3C);
	        ret |= zDrvLcd_Spi_WriteData(0x05);
	        ret |= zDrvLcd_Spi_WriteData(0x3C);
	        ret |= zDrvLcd_Spi_WriteData(0x3C);

	        ret |= zDrvLcd_Spi_WriteCmd(0xB4); //Column inversion
	        ret |= zDrvLcd_Spi_WriteData(0x03); // Dot inversion in Normal mode

	        ret |= zDrvLcd_Spi_WriteCmd(0xC0);
	        ret |= zDrvLcd_Spi_WriteData(0x28);//a2
	        ret |= zDrvLcd_Spi_WriteData(0x08);
	        ret |= zDrvLcd_Spi_WriteData(0x04);

	        ret |= zDrvLcd_Spi_WriteCmd(0xC1);
	        ret |= zDrvLcd_Spi_WriteData(0xC0);//c5

	        ret |= zDrvLcd_Spi_WriteCmd(0xC2);
	        ret |= zDrvLcd_Spi_WriteData(0x0D);
	        ret |= zDrvLcd_Spi_WriteData(0x00);

	        ret |= zDrvLcd_Spi_WriteCmd(0xC3);
	        ret |= zDrvLcd_Spi_WriteData(0x8D);
	        ret |= zDrvLcd_Spi_WriteData(0x2A);

	        ret |= zDrvLcd_Spi_WriteCmd(0xC4);
	        ret |= zDrvLcd_Spi_WriteData(0x8D);
	        ret |= zDrvLcd_Spi_WriteData(0xEE);

	        ret |= zDrvLcd_Spi_WriteCmd(0xC5); //VCOM
	        ret |= zDrvLcd_Spi_WriteData(0x1c);

	        ret |= zDrvLcd_Spi_WriteCmd(0x36); //MX, MY, RGB mode
	        ret |= zDrvLcd_Spi_WriteData(0xd0);
	        ret |= zDrvLcd_Spi_WriteCmd(0xE0);
	        ret |= zDrvLcd_Spi_WriteData(0x04);
	        ret |= zDrvLcd_Spi_WriteData(0x1c);
	        ret |= zDrvLcd_Spi_WriteData(0x07);
	        ret |= zDrvLcd_Spi_WriteData(0x0A);
	        ret |= zDrvLcd_Spi_WriteData(0x2E);
	        ret |= zDrvLcd_Spi_WriteData(0x30);
	        ret |= zDrvLcd_Spi_WriteData(0x25);
	        ret |= zDrvLcd_Spi_WriteData(0x2A);
	        ret |= zDrvLcd_Spi_WriteData(0x28);
	        ret |= zDrvLcd_Spi_WriteData(0x26);
	        ret |= zDrvLcd_Spi_WriteData(0x2E);
	        ret |= zDrvLcd_Spi_WriteData(0x3A);
	        ret |= zDrvLcd_Spi_WriteData(0x00);
	        ret |= zDrvLcd_Spi_WriteData(0x01);
	        ret |= zDrvLcd_Spi_WriteData(0x03);
	        ret |= zDrvLcd_Spi_WriteData(0x13);

	        ret |= zDrvLcd_Spi_WriteCmd(0xE1);
	        ret |= zDrvLcd_Spi_WriteData(0x04);
	        ret |= zDrvLcd_Spi_WriteData(0x10);
	        ret |= zDrvLcd_Spi_WriteData(0x06);
	        ret |= zDrvLcd_Spi_WriteData(0x0D);
	        ret |= zDrvLcd_Spi_WriteData(0x2D);
	        ret |= zDrvLcd_Spi_WriteData(0x26);
	        ret |= zDrvLcd_Spi_WriteData(0x23);
	        ret |= zDrvLcd_Spi_WriteData(0x27);
	        ret |= zDrvLcd_Spi_WriteData(0x27);
	        ret |= zDrvLcd_Spi_WriteData(0x25);
	        ret |= zDrvLcd_Spi_WriteData(0x2D);
	        ret |= zDrvLcd_Spi_WriteData(0x3B);
	        ret |= zDrvLcd_Spi_WriteData(0x00);
	        ret |= zDrvLcd_Spi_WriteData(0x01);
	        ret |= zDrvLcd_Spi_WriteData(0x04);
	        ret |= zDrvLcd_Spi_WriteData(0x13);

	        ret |= zDrvLcd_Spi_WriteCmd(0x35);
	        ret |= zDrvLcd_Spi_WriteData(0x00);

	        ret |= zDrvLcd_Spi_WriteCmd(0x3A); //65k mode
	        ret |= zDrvLcd_Spi_WriteData(0x05);

	        ret |= zDrvLcd_Spi_WriteCmd(0x29); //Display on
	        zDrvLcd_DelayMs(200); //Delay 200ms

	        ret |= zDrvLcd_Spi_WriteCmd(0x2C);
	    }
	    else //s93521a
	    {
	        zDrvLcd_Spi_WriteCmd(0x11); //Exit Sleep
	        zDrvLcd_DelayMs(150);

	        zDrvLcd_Spi_WriteCmd(0x26); //Set Default Gamma
	        zDrvLcd_Spi_WriteData(0x04);

	        zDrvLcd_Spi_WriteCmd(0xB1);
	        zDrvLcd_Spi_WriteData(0x0B);
	        zDrvLcd_Spi_WriteData(0x14);

	        zDrvLcd_Spi_WriteCmd(0xC0); //Set VRH1[4:0] & VC[2:0] for VCI1 & GVDD
	        zDrvLcd_Spi_WriteData(0x10);
	        zDrvLcd_Spi_WriteData(0x00);

	        zDrvLcd_Spi_WriteCmd(0xC1); //Set BT[2:0] for AVDD & VCL & VGH & VGL
	        zDrvLcd_Spi_WriteData(0x05);

	        zDrvLcd_Spi_WriteCmd(0xC5); //Set VMH[6:0] & VML[6:0] for VOMH & VCOML
	        zDrvLcd_Spi_WriteData(0x46);
	        zDrvLcd_Spi_WriteData(0x40);

	        zDrvLcd_Spi_WriteCmd(0xC7);
	        zDrvLcd_Spi_WriteData(0xBD);

	        zDrvLcd_Spi_WriteCmd(0xEC);
	        zDrvLcd_Spi_WriteData(0x0C);

	        zDrvLcd_Spi_WriteCmd(0x3a); //Set Color Format
	        zDrvLcd_Spi_WriteData(0x05);

	        zDrvLcd_Spi_WriteCmd(0x2A); //Set Column Address
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x7F);

	        zDrvLcd_Spi_WriteCmd(0x2B); //Set Page Address
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x00);
	        zDrvLcd_Spi_WriteData(0x9F);

	        zDrvLcd_Spi_WriteCmd(0x36); //Set Scanning Direction
	        zDrvLcd_Spi_WriteData(LCD_MXMYRGB);

	        zDrvLcd_Spi_WriteCmd(0xB7); //Set Source Output Direction
	        zDrvLcd_Spi_WriteData(0x00);

	        zDrvLcd_Spi_WriteCmd(0xf2); //Enable Gamma bit
	        zDrvLcd_Spi_WriteData(0x01);

	        zDrvLcd_Spi_WriteCmd(0xE0);
	        zDrvLcd_Spi_WriteData(0x3F);//p1
	        zDrvLcd_Spi_WriteData(0x29);//p2
	        zDrvLcd_Spi_WriteData(0x26);//p3
	        zDrvLcd_Spi_WriteData(0x26);//p4
	        zDrvLcd_Spi_WriteData(0x26);//p5
	        zDrvLcd_Spi_WriteData(0x0C);//p6
	        zDrvLcd_Spi_WriteData(0x51);//p7
	        zDrvLcd_Spi_WriteData(0xB8);//p8
	        zDrvLcd_Spi_WriteData(0x39);//p9
	        zDrvLcd_Spi_WriteData(0x17);//p10
	        zDrvLcd_Spi_WriteData(0x00);//p11
	        zDrvLcd_Spi_WriteData(0x00);//p12
	        zDrvLcd_Spi_WriteData(0x00);//p13
	        zDrvLcd_Spi_WriteData(0x00);//p14
	        zDrvLcd_Spi_WriteData(0x00);//p15

	        zDrvLcd_Spi_WriteCmd(0xE1);
	        zDrvLcd_Spi_WriteData(0x00);//p1
	        zDrvLcd_Spi_WriteData(0x16);//p2
	        zDrvLcd_Spi_WriteData(0x19);//p3
	        zDrvLcd_Spi_WriteData(0x19);//p4
	        zDrvLcd_Spi_WriteData(0x19);//p5
	        zDrvLcd_Spi_WriteData(0x13);//p6
	        zDrvLcd_Spi_WriteData(0x2E);//p7
	        zDrvLcd_Spi_WriteData(0x47);//p8
	        zDrvLcd_Spi_WriteData(0x46);//p9
	        zDrvLcd_Spi_WriteData(0x08);//p10
	        zDrvLcd_Spi_WriteData(0x3F);//p11
	        zDrvLcd_Spi_WriteData(0x3F);//p12
	        zDrvLcd_Spi_WriteData(0x3F);//p13
	        zDrvLcd_Spi_WriteData(0x3F);//p14
	        zDrvLcd_Spi_WriteData(0x3F);//p15

	        zDrvLcd_Spi_WriteCmd(0x29); // Display On

	        zDrvLcd_Spi_WriteCmd(0x2C);
	    }
	}
	
	if (s_lcd_info.width == 320 && s_lcd_info.height == 240)
	{
		if(s_lcd_info.lcm_info.lcd_item == ILI9342C_320x240)
		{
		    ret |= zDrvLcd_Spi_WriteCmd(0x11); 
			zDrvLcd_DelayMs(120);                //Delay 120ms 
			ret |= zDrvLcd_Spi_WriteCmd(0xC8);		
			ret |= zDrvLcd_Spi_WriteData(0xFF);
			ret |= zDrvLcd_Spi_WriteData(0x93);
			ret |= zDrvLcd_Spi_WriteData(0x42);

			ret |= zDrvLcd_Spi_WriteCmd(0xE0);
			ret |= zDrvLcd_Spi_WriteData(0x0F);
			ret |= zDrvLcd_Spi_WriteData(0x14);
			ret |= zDrvLcd_Spi_WriteData(0x1B);
			ret |= zDrvLcd_Spi_WriteData(0x07);
			ret |= zDrvLcd_Spi_WriteData(0x15);
			ret |= zDrvLcd_Spi_WriteData(0x09);
			ret |= zDrvLcd_Spi_WriteData(0x4C);
			ret |= zDrvLcd_Spi_WriteData(0x76);
			ret |= zDrvLcd_Spi_WriteData(0x5B);
			ret |= zDrvLcd_Spi_WriteData(0x05);
			ret |= zDrvLcd_Spi_WriteData(0x0E);
			ret |= zDrvLcd_Spi_WriteData(0x0A);
			ret |= zDrvLcd_Spi_WriteData(0x29);
			ret |= zDrvLcd_Spi_WriteData(0x2C);
			ret |= zDrvLcd_Spi_WriteData(0x00);

			ret |= zDrvLcd_Spi_WriteCmd(0xE1);
			ret |= zDrvLcd_Spi_WriteData(0x00);
			ret |= zDrvLcd_Spi_WriteData(0x2A);
			ret |= zDrvLcd_Spi_WriteData(0x2D);
			ret |= zDrvLcd_Spi_WriteData(0x02);
			ret |= zDrvLcd_Spi_WriteData(0x0E);
			ret |= zDrvLcd_Spi_WriteData(0x04);
			ret |= zDrvLcd_Spi_WriteData(0x41);
			ret |= zDrvLcd_Spi_WriteData(0x35);
			ret |= zDrvLcd_Spi_WriteData(0x52);
			ret |= zDrvLcd_Spi_WriteData(0x05);
			ret |= zDrvLcd_Spi_WriteData(0x10);
			ret |= zDrvLcd_Spi_WriteData(0x0D);
			ret |= zDrvLcd_Spi_WriteData(0x36);
			ret |= zDrvLcd_Spi_WriteData(0x38);
			ret |= zDrvLcd_Spi_WriteData(0x0F);

			ret |= zDrvLcd_Spi_WriteCmd(0xB1);
			ret |= zDrvLcd_Spi_WriteData(0x00);
			ret |= zDrvLcd_Spi_WriteData(0x1B);

			ret |= zDrvLcd_Spi_WriteCmd(0xB4);
			ret |= zDrvLcd_Spi_WriteData(0x02);

			// ret |= zDrvLcd_Spi_WriteCmd(0xB0);  
			// ret |= zDrvLcd_Spi_WriteData(0xE0);

			ret |= zDrvLcd_Spi_WriteCmd(0xB6);
			ret |= zDrvLcd_Spi_WriteData(0x0A);
			ret |= zDrvLcd_Spi_WriteData(0xE0);

			//ret |= zDrvLcd_Spi_WriteCmd(0xF6);
			//ret |= zDrvLcd_Spi_WriteData(0x00);
			//ret |= zDrvLcd_Spi_WriteData(0x00);
			//ret |= zDrvLcd_Spi_WriteData(0x07);

			ret |= zDrvLcd_Spi_WriteCmd(0x36); 
			ret |= zDrvLcd_Spi_WriteData(0x08); 

			ret |= zDrvLcd_Spi_WriteCmd(0x3A); 
			ret |= zDrvLcd_Spi_WriteData(0x55); 


			ret |= zDrvLcd_Spi_WriteCmd(0xC0);
			ret |= zDrvLcd_Spi_WriteData(0x0F);
			ret |= zDrvLcd_Spi_WriteData(0x0F);

			ret |= zDrvLcd_Spi_WriteCmd(0xC1);
			ret |= zDrvLcd_Spi_WriteData(0x00);

			ret |= zDrvLcd_Spi_WriteCmd(0xC5);
			ret |= zDrvLcd_Spi_WriteData(0xCC);

			ret |= zDrvLcd_Spi_WriteCmd(0x20);

			ret |= zDrvLcd_Spi_WriteCmd(0x35);
			ret |= zDrvLcd_Spi_WriteData(0x00);

			zDrvLcd_DelayMs(120); //Delay 200ms

			ret |= zDrvLcd_Spi_WriteCmd(0x29);
			
			ret |= zDrvLcd_Spi_WriteCmd(0x2C);
			}
		}

    // ˢɺɫ
    for (uiRow = 0; uiRow < var_screeninfo.width; uiRow++)
    {
        for (uiCol = 0; uiCol < var_screeninfo.height; uiCol++)
        {
            ret |= zDrvLcd_Spi_WriteData(0x00);
            ret |= zDrvLcd_Spi_WriteData(0x00);
        }
    }

    return ret;
}

/**************************************************************************
* Function: prj_lcd_get_info
* Description: used to get the LCD device information.
* Parameters:
*       (IN)
*               pInfo    the pointer of LCD device information to store into
*       (OUT)
*               pInfo   the pointer of LCD device information allready stored into
* Returns:
*       DRV_SUCCESS: succeeded.
* Others:
*       None.
**************************************************************************/
int prj_lcd_get_info(T_ZDrvLcd_Info* pInfo)
{
	unsigned char tmp;
	char cmd[64] = {0};
	unsigned char lcd_info[16] = {0};

	sprintf(cmd, "fsload resource 0x%x %s", (ulong)(&lcd_info), ZSP_LCDINFO_PATH);
	run_command(cmd, 0);
	pInfo->width = lcd_info[0]<< 8 | lcd_info[1];//LCD_WIDTH;
	pInfo->height = lcd_info[2]<< 8 | lcd_info[3];//LCD_HEIGHT;
	printf("[LCD]fsload LCDINFO.bin sucess (%d %d)\n",pInfo->width,pInfo->height);
	
	if(pInfo->width == 0 || pInfo->height == 0)
	{
		printf("[LCD]fsload LCDINFO.bin failed!!!!\n");
	}
	pInfo->bitsPerPixel = LCD_BITS_PER_PIXEL;
	pInfo->rMask = 0xF800;
	pInfo->gMask = 0x07E0;
	pInfo->bMask = 0x001F;
	pInfo->instance = g_sLcd_Instance;
	pInfo->backlight = g_sLcd_Backlight;
	pInfo->brightness = g_sLcd_Brightness;
	
	tmp = read_st(0xda);//lcd module's manufacturer ID
	pInfo->lcm_info.man_id = tmp;
	tmp = read_st(0xdb);//lcd module's	ID
	pInfo->lcm_info.dev_id = tmp<<8;
	tmp = read_st(0xdc);//lcd module's	ID
	pInfo->lcm_info.dev_id |= tmp;
	
	if (pInfo->width == 240 && pInfo->height == 320) 
	{
		if((pInfo->lcm_info.man_id == LCM_ID_ZGD)&&((pInfo->lcm_info.dev_id & 0xfff0) ==( LCM_ID_ZGD_9305 & 0xfff0)))
		{
			pInfo->lcm_info.lcd_item = ZGD_GC930x_240x320;
			pInfo->lcm_info.name = "ZGD_GC930x_240x320";
		}
		else if((pInfo->lcm_info.man_id == LCM_ID_ZGD) &&((pInfo->lcm_info.dev_id & 0xfffe) ==( LCM_ID_ZGD_9306& 0xfffe)) )
		{
			pInfo->lcm_info.lcd_item = ZGD_GC9306_240x320;
			pInfo->lcm_info.name = "ZGD_GC9306_240x320";
		}
		else if((pInfo->lcm_info.man_id == LCM_ID_ST) &&((pInfo->lcm_info.dev_id & 0xfff0) ==( LCM_ID_ST7789V& 0xfff0)) )
		{
			pInfo->lcm_info.lcd_item = ST7789V_240x320;
			pInfo->lcm_info.name = "ST7789V_240x320";
		}
		else
		{
			pInfo->lcm_info.lcd_item = ZGD_GC930x_240x320;
			pInfo->lcm_info.name = "240*320 lcd no found,default GC930x";
		}
		var_screeninfo.xs_offset = 0;
		var_screeninfo.ys_offset = 0;
	} 
	else if (pInfo->width == 128 && pInfo->height == 160) 
	{
		if(pInfo->lcm_info.man_id == LCM_ID_LEAD)
		{		
			pInfo->lcm_info.lcd_item = ZGD_ST7735_128x160;
			pInfo->lcm_info.name = "ZGD_ST7735_128x160";
		}
		else if((pInfo->lcm_info.man_id == LCM_ID_ZGD)&&((pInfo->lcm_info.dev_id & 0xfff0) ==( LCM_ID_ZGD_9105 & 0xfff0)))
		{		
			pInfo->lcm_info.lcd_item = ZGD_GC910x_128x160;
			pInfo->lcm_info.name = "ZGD_GC910x_128x160";
		}
		else //default
		{		
			pInfo->lcm_info.lcd_item = ZGD_ST7735_128x160;
			pInfo->lcm_info.name = "128*160 lcd no found,default ST7735";
		}
		var_screeninfo.xs_offset = 0;
		var_screeninfo.ys_offset = 0;
	}
	else if (pInfo->width == 320 && pInfo->height == 240) 
	{
		if (pInfo->lcm_info.man_id == LCM_ID_ILI9342C)
		{
			pInfo->lcm_info.lcd_item = ILI9342C_320x240;
			pInfo->lcm_info.name = "ILI9342C_320x240";
		}
		
		var_screeninfo.xs_offset = 0;
		var_screeninfo.ys_offset = 0;
	}
	
	var_screeninfo.xs = var_screeninfo.xs_offset;
	var_screeninfo.ys = var_screeninfo.ys_offset;
	var_screeninfo.width = pInfo->width;
	var_screeninfo.height = pInfo->height;
	var_screeninfo.xe = var_screeninfo.width + var_screeninfo.xs - 1;
	var_screeninfo.ye = var_screeninfo.height + var_screeninfo.ys - 1;
	
	BOOT_PRINTF(UBOOT_NOTICE, "%s init: lcd man_id = 0x%x, dev_id=0x%x\n", pInfo->lcm_info.name, pInfo->lcm_info.man_id,pInfo->lcm_info.dev_id);
	return 0;
}

int update_rect(unsigned char *pBuf, unsigned short uiLeft, unsigned short uiTop, unsigned short uiWidth, unsigned short uiHeight)
{
	return update_rect_color(pBuf,uiLeft,uiTop,uiWidth,uiHeight);
}

