/***********************************************************************
 * Copyright (C) 2016, ZIXC Corporation.
 *
 * File Name:   hal_visionox_m01210.c
 * File Mark:
 * Description: lcd function implementation. Adapted to model: VISIONOX m01210. Using I2C interface.
 * Others:      this file is in BOOTLOADER procedure.
 * Version:     V1.0
 * Author:      YinWenguan
 * Date:        2014-12-23
 *
 * History 1:
 *     Date: 2015-2-12
 *     Version:
 *     Author: YinWenguan
 *     Modification: rotate screen display 180C; adjust update algorithm
 *
***********************************************************************/

/*************************************************************************
 * Include files
 *************************************************************************/
#include "../include/board.h"
#ifdef CONFIG_BOARD_7520_UFI_958

#include <common.h>
#include <malloc.h>
#include <asm/io.h>
#include "../include/gpio.h"

/*************************************************************************
 * Macro
 *************************************************************************/
 /* debug switch */
#define LCD_DEBUG

#ifdef LCD_DEBUG
#define LCD_PRINTF(format, args...)                     printf(format, ##args)
#define LCD_ASSERT(condition, ret, format, args...)     
#else
#define LCD_PRINTF(format, args...)
#define LCD_ASSERT(condition, ret, format, args...)     
#endif

/* return code */
#define DRV_SUCCESS                     0         /* successed */
#define DRV_ERROR                       -1          /* failed */
#define DRV_ERR_NOT_OPENED    -9         /* the device to operate hasn't been opened yet */
#define DRV_ERR_OPEN_TIMES     -10       /* try to open a device which has been opened already */

/* LCD i2c */
#define LCD_I2C_BUSNUM                0
#define LCD_I2C_SLAVEADDR           (0x78>>1) // 0x78/0x7a
#define OP_COMMAND          0x00
#define OP_DATA             0x40
#define START_PAGE          0xB0
#define START_HIGH_BIT      0x10 // ʼַ4λ
#define START_LOW_BIT       0x00 // ʼַ4λ

/* Features */
#define LCD_WIDTH               128
#define LCD_HEIGHT              32
#define LCD_RESOLUTION          (LCD_WIDTH * LCD_HEIGHT)
#define LCD_BITS_PER_PIXEL      16 // ڰΪRGB565
#define LCD_PAGE_HEIGHT         8
#define LCD_PAGE_NUM            4

/* rotate display */
#define LCD_ROTATE_180
 #ifdef LCD_ROTATE_180
#define LCD_CFG_SEG         0xA0        //Segment do not Remap    
#define LCD_CFG_COM         0xC0        //com output scan direction
#else
#define LCD_CFG_SEG         0xA1        //Segment Remap    
#define LCD_CFG_COM         0xC8        //com output scan direction
#endif

/**************************************************************************
 *                                  Types                                                                                   *
 **************************************************************************/
#ifndef	BOOL_WAS_DEFINED
#define BOOL_WAS_DEFINED
typedef unsigned int bool;
#define false   0
#define true    1
#endif

typedef enum
{
    LCD_DCX_CMD = 0,    // ͵ƽ
    LCD_DCX_DATA,       // ߵƽ

    MAX_LCD_DCX_TYPE
} T_ZDrvLcd_DcxType; // transmission byte type

typedef enum 
{
    LCD_XFER_POLL = 0, // ѯʽ
    LCD_XFER_INT,       // жϷʽ
    LCD_XFER_DMA,   // DMAʽ
    LCD_XFER_CPU,
    
    MAX_LCD_XFER_TYPE
} T_ZDrvLcd_XferType;

/* lcd mode */
typedef enum
{
    LCD_NORMAL,        /* normal mode */
    LCD_SLEEP,          /* sleep mode */

    MAX_LCD_MODE
} T_ZDrvLcd_Mode;

/* lcd instance */
typedef struct
{
    bool                        bOpen;  /* status: open, close */
    T_ZDrvLcd_Mode          eMode;    /* working mode: normal, sleep */
    T_ZDrvLcd_XferType      eXferType; /* poll, interrupt or DMA */
    T_ZDrvLcd_DcxType       eDcxType; /* command or parameter/data */
} T_ZDrvLcd_Instance;

typedef enum
{
    LCD_SLEEP_OUT = 0,
    LCD_SLEEP_IN,
    
} T_ZDrvLcd_SleepStatus;

typedef enum
{
    LCD_BACKLIGHT_OFF = 0,
    LCD_BACKLIGHT_ON,
    
} T_ZDrvLcd_BacklightStatus;

typedef struct
{
    unsigned short  width;
    unsigned short  height;
    unsigned short  bitsPerPixel;
    unsigned short  rMask;
    unsigned short  gMask;
    unsigned short  bMask;
    T_ZDrvLcd_Instance instance;
    T_ZDrvLcd_BacklightStatus backlight;
} T_ZDrvLcd_Info;

/**************************************************************************
 * Global  Variable
 **************************************************************************/
static gpio_pin_t       g_sLcdResetPin = GPIO196;  // Reset
static gpio_pin_t       g_sLcdSa0Pin = GPIO200;  // SA0

static T_ZDrvLcd_Instance   g_sLcdInstance = {false, LCD_NORMAL, LCD_XFER_CPU, LCD_DCX_CMD};

static unsigned char g_sLcdHalDataBuf[LCD_WIDTH * LCD_PAGE_NUM] = { 0, };

/**************************************************************************
 * External Function
 **************************************************************************/
extern int i2c_simple_write(uint i2c_bus, ushort slave_addr,uchar *write_buf, uint write_len);

/**************************************************************************
 * Function Implementation
 **************************************************************************/
#if 1
/*******************************************************************************
 * Function:    zDrvLcd_I2c_WriteCmd
 * Description: 
 * Parameters:
 *      (IN)
 *              
 *      (OUT):
 *              None.
 * Returns:
 *
 *
 * Others:
 ********************************************************************************/
int zDrvLcd_I2c_WriteCmd(unsigned char cmd)
{
    int ret = DRV_SUCCESS;    
    unsigned char temp[2] = {OP_COMMAND, cmd};    
    
    ret |= i2c_simple_write(LCD_I2C_BUSNUM, LCD_I2C_SLAVEADDR, temp, 2);
    
    return ret;
}

/*******************************************************************************
 * Function:    zDrvLcd_I2c_WriteData
 * Description: 
 * Parameters:
 *      (IN)
 *              
 *      (OUT):
 *              None.
 * Returns:
 *
 *
 * Others:
 ********************************************************************************/
int zDrvLcd_I2c_WriteData(unsigned char* pData, unsigned long size)
{
    int ret = DRV_SUCCESS;
    
    unsigned long uiRest = size;
    unsigned char* pPos = pData;
    unsigned char* pBuf;
    
    pBuf = (unsigned char*)malloc(32);
    *pBuf = OP_DATA;

    while (uiRest > 0)
    {
        if (uiRest <=31)
        {
            memcpy(pBuf + 1, pPos, uiRest);
            ret |= i2c_simple_write(LCD_I2C_BUSNUM, LCD_I2C_SLAVEADDR, pBuf, uiRest + 1);
            
            pPos += uiRest;
            uiRest -= uiRest;
        }
        else
        {
            memcpy(pBuf + 1, pPos, 31);
            ret |= i2c_simple_write(LCD_I2C_BUSNUM, LCD_I2C_SLAVEADDR, pBuf, 31 + 1);
            
            pPos += 31;
            uiRest -= 31;
        }
    }

    free(pBuf);
    
    return ret;
}

/*******************************************************************************
 * Function:    zDrvLcd_DelayMs
 * Description: used to delay.
 * Parameters:
 *      (IN)
 *              ms: millisecond
 *      (OUT):
 *              None.
 * Returns:
 *
 *
 * Others:
 ********************************************************************************/
void zDrvLcd_DelayMs(unsigned long ms)
{
    volatile int i = 0;    
    while (i < (50000 * ms)) {i++;}
}
#endif

#if 1
/**************************************************************************
* Function: 
* Description: used to 
* Parameters:
*      (IN)
*               None.
*      (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERR_OPEN_TIMES: has been opened already. 
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:   
*       None.
**************************************************************************/
int zDrvLcd_Initiate(void)
{
    int ret = DRV_SUCCESS;
    
    // LCD λGPIO : rest
    gpio_set_reuse(g_sLcdResetPin, 0x0);
    _gpio_set_direction(g_sLcdResetPin, GPIO_OUT);    

    // ʼģ
    ret |= zDrvLcd_Reset();
    LCD_ASSERT(DRV_SUCCESS ==  ret, DRV_ERROR, "[LCD] Failed to reset LCM!");    

    ret |= zDrvLcd_I2c_WriteCmd(0xAE);        //Set display off
       
    LCD_ASSERT(DRV_SUCCESS == ret, DRV_ERROR, "[LCD] Failed to WriteCmd!");
    
    ret |= zDrvLcd_I2c_WriteCmd(0x00);        //Set lower column start address 
    ret |= zDrvLcd_I2c_WriteCmd(0x10);        //Set higher column start address 
    ret |= zDrvLcd_I2c_WriteCmd(0x20);        //Set page address mode 
    ret |= zDrvLcd_I2c_WriteCmd(0x02); 
    ret |= zDrvLcd_I2c_WriteCmd(0x21);        //Set coloum address start and end 
    ret |= zDrvLcd_I2c_WriteCmd(0x00); 
    ret |= zDrvLcd_I2c_WriteCmd(0x7F); 
    ret |= zDrvLcd_I2c_WriteCmd(0x22);          //Set page address start end 
    ret |= zDrvLcd_I2c_WriteCmd(0x00); 
    ret |= zDrvLcd_I2c_WriteCmd(0x03); 
    ret |= zDrvLcd_I2c_WriteCmd(0xd5);        //Display divide ratio/osc. freq. mode     
    ret |= zDrvLcd_I2c_WriteCmd(0x81);        //Divide ratio 2 
    ret |= zDrvLcd_I2c_WriteCmd(0xA8);        //Multiplex ration mode:32 
    ret |= zDrvLcd_I2c_WriteCmd(0x1F); 
    ret |= zDrvLcd_I2c_WriteCmd(0xD3);        //Set Display Offset     
    ret |= zDrvLcd_I2c_WriteCmd(0x00);        //Set vertical shift 
    ret |= zDrvLcd_I2c_WriteCmd(0x40);        //Set Display Start Line   
    ret |= zDrvLcd_I2c_WriteCmd(0xAD);        //Internal IREF Setting   
    ret |= zDrvLcd_I2c_WriteCmd(0x30);        //Selectting: 30uA 
    ret |= zDrvLcd_I2c_WriteCmd(LCD_CFG_SEG);        //Segment Remap    
    ret |= zDrvLcd_I2c_WriteCmd(LCD_CFG_COM);        //Set com output scan direction
    ret |= zDrvLcd_I2c_WriteCmd(0xDA);        //Common pads hardware: alternative   
    ret |= zDrvLcd_I2c_WriteCmd(0x12); 
    ret |= zDrvLcd_I2c_WriteCmd(0x81);        //Contrast control   
    ret |= zDrvLcd_I2c_WriteCmd(0x99); 
    ret |= zDrvLcd_I2c_WriteCmd(0x8D);       //Internal charge pump: 
    ret |= zDrvLcd_I2c_WriteCmd(0x95);        // Enable Charge Pump/10h:Disable Charge Pump 
    ret |= zDrvLcd_I2c_WriteCmd(0xD9);        //Set pre-charge period      
    ret |= zDrvLcd_I2c_WriteCmd(0x22); 
    ret |= zDrvLcd_I2c_WriteCmd(0xDB);        //VCOM deselect level mode   
    ret |= zDrvLcd_I2c_WriteCmd(0x10);        //VCOM:0.71*VCC 
    ret |= zDrvLcd_I2c_WriteCmd(0xA4);        //Set Entire Display On/Off    
    ret |= zDrvLcd_I2c_WriteCmd(0xA6);        //Set Normal Display   
    
    ret |= zDrvLcd_ClearScreen();                 //Clear Screen 
    ret |= zDrvLcd_I2c_WriteCmd(0xAF);        //Set Display On
    LCD_ASSERT(DRV_SUCCESS == ret, DRV_ERROR, "[LCD] Failed to initialize!");
    
    return ret;
}

/**************************************************************************
* Function: zDrvLcd_Open
* Description: used to open the LCD device.
* Parameters:
*      (IN)
*               devData     LCD device name
*               flags           don't care
*      (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERR_OPEN_TIMES: has been opened already. 
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:   
*       None.
**************************************************************************/
int zDrvLcd_Open()
{
    if (g_sLcdInstance.bOpen)
    {
        return DRV_ERR_OPEN_TIMES;
    }    

    // ô򿪱־͵ǰģʽ
    g_sLcdInstance.bOpen= true;
    g_sLcdInstance.eMode = LCD_NORMAL;

    return DRV_SUCCESS;
}

/**************************************************************************
* Function: zDrvLcd_Close
* Description: used to close the LCD device opened previously.
* Parameters:
*       (IN)
*               None.
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERR_NOT_OPENED: has not been opend yet.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:   
*       None.
**************************************************************************/
int zDrvLcd_Close()
{
    if (false == g_sLcdInstance.bOpen)
    {
        return DRV_ERR_NOT_OPENED;
    }

    g_sLcdInstance.bOpen = false;
    return DRV_SUCCESS;
}

/**************************************************************************
* Function: zDrvLcd_GetInfo
* Description: used to get the LCD device information.
* Parameters:
*       (IN)
*               pInfo    the pointer of LCD device information to store into
*       (OUT)
*               pInfo   the pointer of LCD device information allready stored into
* Returns:
*       DRV_SUCCESS: succeeded.
* Others:   
*       None.
**************************************************************************/
int zDrvLcd_GetInfo(T_ZDrvLcd_Info* pInfo)
{
    pInfo->width = LCD_WIDTH;
    pInfo->height = LCD_HEIGHT;
    pInfo->bitsPerPixel = LCD_BITS_PER_PIXEL;
    pInfo->rMask = 0x0000;
    pInfo->gMask = 0x00E0;
    pInfo->bMask = 0x0000;
    pInfo->instance = g_sLcdInstance;
    pInfo->backlight = LCD_BACKLIGHT_ON; // ɫ޿ر⹦ܡĬϱʼΪ
    
    return DRV_SUCCESS;
}

/**************************************************************************
* Function: zDrvLcd_SetRamAddr
* Description: used to set display ram address.
* Parameters:
*       (IN)
*               uiCol       start column
*               uiPage      start page
*       (OUT)
*               None.
* Returns:
*       None.
* Others:   
*       None.
**************************************************************************/
int zDrvLcd_SetRamAddr( unsigned short uiCol, unsigned short uiPage )
{
    int ret = DRV_SUCCESS;
    
    /*set column address*/
    ret |= zDrvLcd_I2c_WriteCmd(START_LOW_BIT | (uiCol & 0x0F));    // 4λ
    ret |= zDrvLcd_I2c_WriteCmd(START_HIGH_BIT |((uiCol>>4) & 0x0F)); // 4λ
    
    /*set page address*/
    ret |= zDrvLcd_I2c_WriteCmd(START_PAGE | (uiPage & 0x0F));

    return ret;
}

/**************************************************************************
* Function: zDrvLcd_Reset
* Description: used to reset the LCD device opened previously.
* Parameters:
*       (IN)
*               None.
*       (OUT)
*               None.
* Returns:
*       None.
* Others:   
*       None.
**************************************************************************/
int zDrvLcd_Reset(void)
{
    int ret = DRV_SUCCESS;
    
    // Ӳλ    
    gpio_set_value(g_sLcdResetPin, GPIO_HIGH);
    
    gpio_set_value(g_sLcdResetPin, GPIO_LOW);    
    zDrvLcd_DelayMs(100);

    gpio_set_value(g_sLcdResetPin, GPIO_HIGH);
    zDrvLcd_DelayMs(120);

    return ret;
}

/**************************************************************************
* Function: zDrvLcd_UpdateRect
* Description: used to update a specified rectangle area of the LCD device opened previously.
* Parameters:
*       (IN)
*               pBuf        data buffer
*               uiLeft      left postion
*               uiTop       top postion
*               uiWidth     rectangle width
*               uiHeight    rectangle height
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:   
*       None.
**************************************************************************/
int zDrvLcd_WritePixel(unsigned short rgb565, unsigned short uiRow, unsigned short uiCol)
{
    unsigned short page_index = uiRow / LCD_PAGE_HEIGHT;
    
    unsigned short ram_offset = page_index * LCD_WIDTH + uiCol;
    
    unsigned short bit_shift = uiRow % LCD_PAGE_HEIGHT;

    if (rgb565 != 0 )    
        *(g_sLcdHalDataBuf + ram_offset) |= (0x01 << bit_shift);
    else
        *(g_sLcdHalDataBuf + ram_offset) &= ~(0x01 << bit_shift);
    
    return 0;
}

int zDrvLcd_UpdateRect(unsigned char *pBuf, unsigned short uiLeft, unsigned short uiTop, unsigned short uiWidth, unsigned short uiHeight)
{
    unsigned short  *pSrcBuf = (unsigned short  *)pBuf;        
    unsigned short  row, col, pixel_value;
    unsigned short cur_page;

    LCD_ASSERT(
            (NULL != pBuf) &&
            (LCD_WIDTH >= uiLeft) &&
            (LCD_HEIGHT >= uiTop) &&            
            (LCD_WIDTH >= uiLeft + uiWidth) &&
            (LCD_HEIGHT >= uiTop + uiHeight) &&
            (0 <= uiWidth) && (0 <= uiHeight),
            DRV_ERR_INVALID_PARAM,
            "[LCD] Invalid input parameter on calling zDrvLcd_UpdateRect(...)!");

    /* صRGB565ֵתΪڰֵдRAM */
    for (row = 0; row < uiHeight; row++)
    {
        for (col = 0; col < uiWidth; col++)
        {
           pixel_value = *(pSrcBuf + row * uiWidth + col);
           
            zDrvLcd_WritePixel(pixel_value, row + uiTop, col + uiLeft);
        }
    }

    /* СRAMȥÿһpage */
    for ( cur_page = 0; cur_page < LCD_PAGE_NUM; cur_page++ )
    {
        zDrvLcd_SetRamAddr( 0, cur_page );
        zDrvLcd_I2c_WriteData(g_sLcdHalDataBuf + cur_page * LCD_WIDTH, LCD_WIDTH);
    }
    
    return DRV_SUCCESS;
}

/**************************************************************************
* Function: Lcd_EnterSleepMode
* Description: used to let the LCD device into sleep mode.
* Parameters:
*       (IN)
*               bIn     enter or exit sleep mode
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:   
*       None.
**************************************************************************/
int zDrvLcd_SleepInOut( int bIn )
{
    int ret = DRV_SUCCESS;    

    ret = zDrvLcd_I2c_WriteCmd(bIn ? 0xAE : 0xAF); // enter/exit the spleep mode 

    if (false == bIn)
    {
        g_sLcdInstance.eMode = LCD_NORMAL;
    }
    else
    {
        g_sLcdInstance.eMode = LCD_SLEEP;
    }

    return ret;
}

/**************************************************************************
* Function: zDrvLcd_ClearScreen
* Description: 
* Parameters:
*       (IN)
*               None.
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:   
*       None.
**************************************************************************/
int zDrvLcd_ClearScreen(void)
{
    int ret = DRV_SUCCESS;
    unsigned char temp = 0x00; /* ΪɫΪɫΪڵװ */
    unsigned char page_number,column_number;
    for( page_number = 0; page_number < 4; page_number++)
   {
         zDrvLcd_I2c_WriteCmd(START_PAGE + page_number);
         zDrvLcd_I2c_WriteCmd(START_HIGH_BIT);
         zDrvLcd_I2c_WriteCmd(START_LOW_BIT);
        for(column_number = 0;column_number < LCD_WIDTH; column_number++)
        {
            zDrvLcd_I2c_WriteData(&temp, 1);
        }
    }

    memset(g_sLcdHalDataBuf, 0, LCD_WIDTH * LCD_PAGE_NUM);

    return ret;
}

/**************************************************************************
* Function: zDrvLcd_PowerOnLogo, zDrvLcd_ShowUpdateWait, zDrvLcd_ShowUpdateSucc, zDrvLcd_ShowUpdateFail
* Description: 
* Parameters:
*       (IN)
*               None.
*       (OUT)
*               None.
* Returns:
*       DRV_SUCCESS: successed.
*       DRV_ERROR: others error code. For more details, please refer to the source code.
* Others:   
*       None.
**************************************************************************/
int zDrvLcd_PowerOnLogo(void)
{
    extern unsigned char bmp_128_32_logo[];    
    return zDrvLcd_UpdateRect(bmp_128_32_logo, 0, 0, LCD_WIDTH, LCD_HEIGHT); // LOGO
}

int zDrvLcd_ShowUpdateWait(void)
{
    extern unsigned char bmp_128_32_updating[];    
    return zDrvLcd_UpdateRect(bmp_128_32_updating, 0, 0, LCD_WIDTH, LCD_HEIGHT);
}

int zDrvLcd_ShowUpdateSucc(void)
{
    extern unsigned char bmp_128_32_updatesuccess[];    
    return zDrvLcd_UpdateRect(bmp_128_32_updatesuccess, 0, 0, LCD_WIDTH, LCD_HEIGHT);
}

int zDrvLcd_ShowUpdateFail(void)
{
    extern unsigned char bmp_128_32_updatefailed[];    
    return zDrvLcd_UpdateRect(bmp_128_32_updatefailed, 0, 0, LCD_WIDTH, LCD_HEIGHT);
}

int zDrvLcd_ShowCharging(void)
{
    extern unsigned char bmp_128_32_charging[];    
    return zDrvLcd_UpdateRect(bmp_128_32_charging, 0, 0, LCD_WIDTH, LCD_HEIGHT);
}

int zDrvLcd_ShowLowBattery(void)
{
    extern unsigned char bmp_128_32_lowbattery[];
    return zDrvLcd_UpdateRect(bmp_128_32_lowbattery, 0, 0, LCD_WIDTH, LCD_HEIGHT);
}

#endif
#endif
