/*
 * Freescale QuadSPI driver.
 *
 * Copyright (C) 2013 Freescale Semiconductor, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/err.h>
#include <malloc.h> 
#include <errno.h>
#include <asm/io.h> 
#include <linux/mtd/mtd.h>
#include <asm-generic/ioctl.h>
#include <config.h>
#include <common.h>
#include <command.h>
//#include <asm/arch/nand.h> 
//#include <linux/mtd/spi-nor.h>
#include <asm/arch/lsp_crpm.h>
#include <drvs_gpio.h>
#include <linux/mtd/nor_spifc.h>
#include <../drivers/dma/zx29_dma.h>
#include <nand.h>
#include <secure_verify.h>


spinor_cmd_t nor_cmd_table[]=
{	
	{CMD_WREN, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "write enable"},
	{CMD_WRDI, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "write disable"},
	{CMD_WRENVSR, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "write ensable vsr"},		
	{CMD_RDSR0, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "read SR S7-S0"},
	{CMD_RDSR1, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "read SR S15-S8"},
	{CMD_WRSR, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_EN, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "write SR S15-S0"},
	{CMD_RDB, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "read data bytes"},
	{CMD_RDFT, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_EN, 1, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "read fast"},
	{CMD_RDDFT, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_EN, 1, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_EN, TRANS_MOD_DUAL, "read dual fast"},
	{CMD_RDQFT, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_EN, 1, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_EN, TRANS_MOD_QUAD, "read quad fast"},
	{CMD_PP, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_EN, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "page program"},
	{CMD_QPP, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_EN, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_EN, TRANS_MOD_QUAD, "quad page program"},
	{CMD_SE, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "sector erase"},
	{CMD_32KBE, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "32K block erase"},
	{CMD_64KBE, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "64K block erase"},
	{CMD_CE, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "chip erase"},
	{CMD_DP, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "deep power-down"},
	{CMD_RDPRDI, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "releae DP adn RDI"},
	{CMD_REMS, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "read manufacture device id"},	
	{CMD_RDID, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "read identification"},
	{CMD_ENRESET, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "reset enable"},	
	{CMD_RESET, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_DIS, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "reset"},
	{CMD_ESR, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "erase security registers"},
	{CMD_PSR, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_EN, DATA_RX_DIS, DUMY_TX_DIS, 0, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "program security registers"},
	{CMD_RSR, TX_DMA_DIS, RX_DMA_DIS, ADDR_TX_EN, ADDR_WIDTH_24, DATA_TX_DIS, DATA_RX_EN, DUMY_TX_EN, 1, 0, ADDR_MULTI_LINE_DIS, DATA_MULTI_LINE_DIS, TRANS_MOD_SINGLE, "read security registers"},
	{0}
};


struct fsl_qspi spi_nor_flash;
extern nand_info_t nand_info[CONFIG_SYS_MAX_NAND_DEVICE];


#define	TRANS_USE_DMA	1

#if TRANS_USE_DMA
/*
 * config dma 
 *
 * dir  -- FC_DMA_RX/FC_DMA_TX
 */ 
 void spifc_config_dma(uint8_t *buf, uint32_t len, int dir)
{
	dma_channel_def chan_para;
	dma_peripheral_id channel_id = 0;
    volatile struct spifc_nor_reg_t* spi = (struct spifc_nor_reg_t*)spifc_get_reg_base();

	if(dir == FC_DMA_RX)
	{	
		chan_para.src_addr = &(spi->SFC_DATA);
		chan_para.dest_addr = (unsigned int)buf;
		channel_id	= DMA_CH_SPIFC_RX;	
		chan_para.dma_control.tran_mode = TRAN_PERI_TO_MEM;
		
	}
	else if(dir == FC_DMA_TX)
	{	
		chan_para.src_addr = (unsigned int)buf;
		chan_para.dest_addr = &(spi->SFC_DATA);
		channel_id	= DMA_CH_SPIFC_TX;	
		chan_para.dma_control.tran_mode = TRAN_MEM_TO_PERI;
	}
	else
		BUG();
		
	chan_para.count = len;
	chan_para.dma_control.irq_mode  		= DMA_ALL_IRQ_DISABLE;
	chan_para.dma_control.src_burst_size	= DMA_BURST_SIZE_32BIT;
	chan_para.dma_control.src_burst_len		= DMA_BURST_LEN_8;
	chan_para.dma_control.dest_burst_size 	= DMA_BURST_SIZE_32BIT;
	chan_para.dma_control.dest_burst_len	= DMA_BURST_LEN_8;
	
	dma_config(channel_id, &chan_para);	
	
	dma_start(channel_id, NULL, NULL);
}

 int spifc_wait_dma_done(int dir)
{
	dma_peripheral_id channel_id = 0;

	if(dir == FC_DMA_RX)
		channel_id = DMA_CH_SPIFC_RX;
	else if(dir == FC_DMA_TX)
		channel_id = DMA_CH_SPIFC_TX;
	else
		BUG();

	while(1)
    {
        if(dma_get_status(channel_id) == DMA_TRANSFER_DONE)
            break;
    }

	dma_disable_channel(channel_id);
	
	return 0;
}
#endif

#define SPI_FLASH_CONTROLLER 1

#if SPI_FLASH_CONTROLLER


 void spifc_enable(struct fsl_qspi *q)
{
   volatile struct spifc_nor_reg_t* spifc = q->iobase;
    
    if(spifc->SFC_EN & FC_EN_BACK)
    {
		printf("spifc enable error.\n");
		return;
	}   
    spifc->SFC_EN |= (1 << FC_EN);  
    spifc->SFC_CTRL0 |= (1 << FC_SCLK_PAUSE_EN);
}

 void spifc_disable(struct fsl_qspi *q)
{
    volatile struct spifc_nor_reg_t* spifc = q->iobase;
    
    if(!(spifc->SFC_EN & FC_EN_BACK))
    {
		printf("spifc disable error.\n");
		return;
	}     
    spifc->SFC_EN &= (~(1 <<FC_EN));    
}

 void spifc_config_timing(struct fsl_qspi *q, uint32_t rd_delay, uint32_t cs_setup, 
                      uint32_t cs_hold, uint32_t cs_desel)
{
    volatile struct spifc_nor_reg_t* spifc = q->iobase;
    
    spifc->SFC_TIMING = 0;
    spifc->SFC_TIMING  |= ((rd_delay&0x3) << 16)|((cs_setup&0x7) <<11)|
                        ((cs_hold&0x7) << 6)|((cs_desel&0xf) << 0);
}

 void spifc_setup_cmd(struct fsl_qspi *q, 
							spinor_cmd_t *cmd, uint32_t addr, uint32_t len)
{   
	volatile struct spifc_nor_reg_t* spifc = q->iobase;

	/* clear dma config */
	spifc->SFC_CTRL0 &= ~((1 << FC_RX_DMA_EN)|(1 << FC_TX_DMA_EN));

	/* clear fifo */
	spifc->SFC_CTRL0 |= (1 << FC_RXFIFO_CLR)|(1 << FC_TXFIFO_CLR);

	/* clear interrupt register */
	spifc->SFC_INT_SW_CLR = 0xFF;  
	
	/* dma + fifo config */
	spifc->SFC_CTRL0 |= ((cmd->tx_dma_en << FC_TX_DMA_EN) 
						  | (cmd->rx_dma_en << FC_RX_DMA_EN)
						  | (1 << FC_RXFIFO_THRES)
						  | (1 << FC_TXFIFO_THRES));

	/* addr dumy data code config */
    spifc->SFC_CTRL1 = 0;
    spifc->SFC_CTRL1 = ((cmd->addr_tx_en << FC_ADDR_TX_EN) 
						| (cmd->dumy_tx_en << FC_DUMMY_TX_EN) 
                        | (cmd->data_rx_en << FC_READ_DAT_EN) 
                        | (cmd->data_tx_en << FC_WRITE_DAT_EN));

	/* ַ(1234ֽ)ַ/߶ȡģʽ */
    spifc->SFC_CTRL2 = 0;
    spifc->SFC_CTRL2 = ((cmd->dumy_byte_num << FC_DUMMY_BYTE_NUM) 
            			| (cmd->dumy_bit_num << FC_DUMMY_BIT_NUM) 
	            		| (cmd->addr_byte_num << FC_ADDR_BYTE_NUM)
	            		| (cmd->addr_multi_line_en << FC_ADDR_MULTI_LINE_EN)
	            		| (cmd->data_multi_line_en << FC_DAT_MULTI_LINE_EN)
	            		| (cmd->trans_mod << FC_TRANS_MOD));

	/* ݳ */
    if(len)
        spifc->SFC_BYTE_NUM = len - 1;
    else
        spifc->SFC_BYTE_NUM = 0; 
	
	/* addr code register */
	spifc->SFC_ADDR = addr; 
	
	/*  */  
    spifc->SFC_INS = cmd->cmd;          
}

 int spifc_wait_cmd_end(struct fsl_qspi *q)
{
    volatile struct spifc_nor_reg_t* spifc = q->iobase;
    uint32_t intr_status = 0;
    
    while( spifc->SFC_START & FC_BUSY ); 

    while(!(spifc->SFC_INT_RAW & FC_INT_RAW_CMD_END));
    
    intr_status = spifc->SFC_INT_RAW;
    spifc->SFC_INT_SW_CLR = intr_status;           /* ж */   

	if(intr_status & FC_INT_RAW_CMD_END)
    {   
        return SUCCESS;
    }
    else if(intr_status & FC_INT_RAW_TX_UNDERRUN)
    {   
        printf("spifc error: TX_UNDERRUN\n");
        return -EIO;
    }
	else if(intr_status & FC_INT_RAW_RX_OVERRUN)
    {   
        printf("spifc error: RX_UNDERRUN\n");
        return -EIO;
    }
    else if(intr_status & FC_INT_RAW_WDOG_OVERRUN)
    {   
        printf("spifc error: WDOG_OVERRUN\n");
        return -EIO;
    }
    else if(intr_status & FC_INT_RAW_FMT_ERR)
    {   
        printf("spifc error: FMT_ERR\n");
        return -EIO;
    }
    else
    {
		printf("unknown spifc error\n");
        return -EIO;
    }   
}

uint32_t spifc_read_fifo(struct fsl_qspi *q, uint8_t *buf, uint32_t len)
{
    volatile struct spifc_nor_reg_t* spifc = q->iobase;
    uint32_t *p = (uint32_t *)buf;
    uint32_t cnt = 0;
   
   int remainder_cnt = len % 4;
   
   if(remainder_cnt != 0)
   {
	   len = len + (4 - remainder_cnt);
   }
   else
   {
	   remainder_cnt = 4;
   }
   
   while(cnt < (len>>2))  
   {
	   if(spifc->SFC_SW & (FC_RX_FIFO_CNT_MASK<<FC_RX_FIFO_CNT))//rx fifo not empty
	   {		   
		   p[cnt++]= spifc->SFC_DATA;
	   }
   }
   
   return ((cnt<<2) - (4 - remainder_cnt));

}

int spifc_write_data(struct fsl_qspi *q, uint8_t *buf, uint32_t len)
{
    uint32_t *p = (uint32_t *)buf;
    uint32_t cnt = 0;
    volatile struct spifc_nor_reg_t* spifc = q->iobase;

	int remainder_cnt = len % 4;

	if(remainder_cnt != 0)
	{
		len = len + (4 - remainder_cnt);
	}
	else
	{
		remainder_cnt = 4;
	}
	
    while(cnt < (len >> 2))  
    {
        if(spifc->SFC_SW & (FC_TX_FIFO_CNT_MASK<<FC_TX_FIFO_CNT))//tx fifo not full
        {           
            spifc->SFC_DATA = p[cnt++];
        }
    }

    return ((cnt<<2) - (4 - remainder_cnt));
}

 void spifc_start(struct fsl_qspi *q)
{
    volatile struct spifc_nor_reg_t* spifc = q->iobase;
    
    spifc->SFC_START |= FC_START;
}

#endif


 void spifc_setclk(void)
{
    uint32_t clk_reg = 0;

	clk_reg = readl(SPIFC_CLKSEL);
	clk_reg &= ~0x00000070;
	clk_reg |= (0x2 << 4);       /*124.8MHz*/
	writel(clk_reg, SPIFC_CLKSEL);  
}

 void spifc_hwinit(void)
{
    zDrvGpio_SetFunc(GPIO98, GPIO98_SPIFC_DATA3);
	zDrvGpio_SetFunc(GPIO97, GPIO97_SPIFC_DATA2);
	zDrvGpio_SetFunc(GPIO96, GPIO96_SPIFC_DATA1);
	zDrvGpio_SetFunc(GPIO95, GPIO95_SPIFC_DATA0);
	zDrvGpio_SetFunc(GPIO93,GPIO93_SPIFC_CS);
	zDrvGpio_SetFunc(GPIO94,GPIO94_SPIFC_CLK);   
}

 void fsl_qspi_set_base_addr(struct fsl_qspi *q, struct spi_nor *nor)
{
	q->chip_base_addr = q->nor_size * (nor - q->nor);
}


spinor_cmd_t *cmd_seek(u8 opcode)
{
	int i;

	for(i = 0; nor_cmd_table[i].cmd != 0; i++)
	{
		if(opcode == nor_cmd_table[i].cmd)
		{
			return (&nor_cmd_table[i]);
		}
	}

	return NULL;	
}

 int fsl_qspi_read_reg(struct spi_nor *nor, u8 opcode, u8 *buf, int len)
{
	int ret = 0;
	struct fsl_qspi *q = nor->priv;
	spinor_cmd_t *cmd = NULL;
	
	cmd = cmd_seek(opcode);
	if(cmd == NULL)
	{
		printf("cmd_seek unkown cmd = 0x%x error.\n", opcode);
		return -1;
	}
	
	spifc_setup_cmd(q, cmd, 0x0, len);            
    spifc_start(q);                          
    ret = spifc_wait_cmd_end(q);             
    if(ret != 0)
    {
		printf("spifc_wait_cmd_end error.\n");
	   	return ret;
    }
	
	ret = spifc_read_fifo(q, buf, len);
	if(ret < 0)
	{
		printf("spifc_read_fifo error.\n");
	    return ret;
	}
	
	return 0;	
}

 int fsl_qspi_write_reg(struct spi_nor *nor, u8 opcode, u8 *buf, int len)
{
	int ret;
	struct fsl_qspi *q = nor->priv;
	spinor_cmd_t *cmd = NULL;
	int nor_status = 0;
	
	cmd = cmd_seek(opcode);
	if(cmd == NULL)
	{
		printf("cmd_seek unkown cmd = 0x%x error.\n", opcode);
		return -1;
	}
	
	spifc_setup_cmd(q, cmd, 0x0, len);  	

	if(len > 0)
	{
		ret = spifc_write_data(q, buf, len);
		if(ret != len)
		{
			printf("spifc_write_data error.\n");
		    return ret;
		}
	}

	spifc_start(q);
	
    ret = spifc_wait_cmd_end(q);             
    if(ret != 0)
    {
		printf("spifc_wait_cmd_end error.\n");
	   	return ret;
    }

	do{
		ret = fsl_qspi_read_reg(nor, CMD_RDSR0, &nor_status, 1);
		if(ret != 0)
		{
			printf("read WIP fail.\n");
		}
	}while(nor_status & 0x1);
	
	return 0;
}

 int nor_write_page(struct spi_nor *nor, loff_t to,
						   size_t len, size_t *retlen, const u_char *buf)
{
	int ret;
	int nor_status = 0;
	struct fsl_qspi *q = nor->priv;
	spinor_cmd_t *cmd = NULL;
	
	cmd = cmd_seek(nor->program_opcode);
	if(cmd == NULL)
	{
		printf("cmd_seek unkown cmd = 0x%x error.\n", nor->program_opcode);
		return -1;
	}

#if TRANS_USE_DMA
	cmd->tx_dma_en = TX_DMA_EN;
#else
	cmd->tx_dma_en = TX_DMA_DIS;
#endif
	spifc_setup_cmd(q, cmd, to, len);	            
    

#if TRANS_USE_DMA
	spifc_config_dma(buf, len, FC_DMA_TX);
	spifc_start(q);
#else
	spifc_start(q);
	ret = spifc_write_data(q, buf, len);
	if(ret < 0)
	{
		printf("spifc_write_fifo error.\n");
		return ret;
	}
#endif

	ret = spifc_wait_cmd_end(q);             
    if(ret != 0)
    {
		printf("spifc_wait_cmd_end error.\n");
	   	return ret;
    }

#if TRANS_USE_DMA
	spifc_wait_dma_done(FC_DMA_TX);
#endif

	do{
		ret = fsl_qspi_read_reg(nor, CMD_RDSR0, &nor_status, 1);
		if(ret != 0)
		{
			printf("read WIP fail.\n");
		}
	}while(nor_status & 0x1);
	
	return 0;
}

 int fsl_qspi_write(struct spi_nor *nor, loff_t to,
						   size_t len, size_t *retlen, const u_char *buf)
 {
	int ret = 0;
	struct fsl_qspi *q = NULL;

	if((nor == NULL) || (nor->priv == NULL) || (retlen == NULL) || (buf == NULL))
	{
		printk("fsl_qspi_write parameter is error.\n");
		return -1;
	}
	q = nor->priv;
	memcpy((void*)q->buf.buf, buf, len);
	
	ret = nor_write_page(nor, to, len, retlen, q->buf.dma_buf);
	if(ret != 0)
	{
		printk("nor_write_page error.\n");
		return -1;
	}
	*retlen += len;

	return 0;
	
 }

 int nor_read_page(struct spi_nor *nor, loff_t from,
						 size_t len, size_t *retlen, u_char *buf)
{
	int ret;
	struct fsl_qspi *q = nor->priv;
	spinor_cmd_t *cmd = NULL;
	
	cmd = cmd_seek(nor->read_opcode);
	if(cmd == NULL)
	{
		printf("cmd_seek unkown cmd = 0x%x error.\n", nor->read_opcode);
		return -1;
	}

#if TRANS_USE_DMA
	cmd->rx_dma_en = RX_DMA_EN;
#else
	cmd->rx_dma_en = RX_DMA_DIS;
#endif
	spifc_setup_cmd(q, cmd, from, len);	            
    
#if TRANS_USE_DMA
	spifc_config_dma(buf, len, FC_DMA_RX);
	spifc_start(q);
#else
	spifc_start(q);
	ret = spifc_read_fifo(q, buf, len);
	if(ret < 0)
	{
		printf("spifc_read_fifo error.\n");
		return ret;
	}
#endif

	ret = spifc_wait_cmd_end(q);             
    if(ret != 0)
    {
		printf("spifc_wait_cmd_end error.\n");
	   	return ret;
    }
	
#if TRANS_USE_DMA
	spifc_wait_dma_done(FC_DMA_RX);
#endif

	return 0;
}

int fsl_qspi_read(struct spi_nor *nor, loff_t from,
					 size_t len, size_t *retlen, u_char *buf)
{
	int ret = 0;
	struct fsl_qspi *q = NULL;
	loff_t from_alligned = 0;

	if((nor == NULL) || (nor->priv == NULL) || (retlen == NULL) || (buf == NULL))
	{
		printk("fsl_qspi_read parameter is error.\n");
		return -1;
	}
	from_alligned = from&(~(nor->page_size - 1));
	q = nor->priv;
	
	ret = nor_read_page(nor, from_alligned, nor->page_size, retlen, q->buf.dma_buf);
	if(ret != 0)
	{
		printk("nor_read_page error.\n");
		return -1;
	}

	memcpy(buf, (void*)q->buf.buf + (from&(nor->page_size - 1)), len);
	*retlen += len;

	return 0; 
}

 int fsl_qspi_erase(struct spi_nor *nor, loff_t offs)
{
	int ret;
	int nor_status = 0;
	struct fsl_qspi *q = nor->priv;
	spinor_cmd_t *cmd = NULL;

	cmd = cmd_seek(nor->erase_opcode);
	if(cmd == NULL)
	{
		printf("cmd_seek unkown cmd = 0x%x error.\n", nor->read_opcode);
		return -1;
	}
	spifc_setup_cmd(q, cmd, offs, 0);

	spifc_start(q);
    ret = spifc_wait_cmd_end(q);             
    if(ret != 0)
    {
		printf("spifc_wait_cmd_end error.\n");
	   	return ret;
    }

	do{
		ret = fsl_qspi_read_reg(nor, CMD_RDSR0, &nor_status, 1);
		if(ret != 0)
		{
			printf("read WIP fail.\n");
		}
	}while(nor_status & 0x1);
	
	return 0;
}

 int nor_write_security_register(struct spi_nor *nor, loff_t to,
						   size_t len, size_t *retlen, const u_char *buf)
 {
	int ret;
	int nor_status = 0;
	struct fsl_qspi *q = nor->priv;
	spinor_cmd_t *cmd = NULL;
	
	cmd = cmd_seek(CMD_PSR);
	if(cmd == NULL)
	{
		printf("cmd_seek unkown cmd = 0x%x error.\n", nor->program_opcode);
		return -1;
	}
 
#if TRANS_USE_DMA
	cmd->tx_dma_en = TX_DMA_EN;
#else
	cmd->tx_dma_en = TX_DMA_DIS;
#endif
	spifc_setup_cmd(q, cmd, to, len);				
	
 
#if TRANS_USE_DMA
	spifc_config_dma(buf, len, FC_DMA_TX);
	spifc_start(q);
#else
	spifc_start(q);
	ret = spifc_write_data(q, buf, len);
	if(ret < 0)
	{
		printf("spifc_write_fifo error.\n");
		return ret;
	}
#endif
 
	ret = spifc_wait_cmd_end(q);			 
	if(ret != 0)
	{
		printf("spifc_wait_cmd_end error.\n");
		return ret;
	}
 
#if TRANS_USE_DMA
	spifc_wait_dma_done(FC_DMA_TX);
#endif
 
	do{
		ret = fsl_qspi_read_reg(nor, CMD_RDSR0, &nor_status, 1);
		if(ret != 0)
		{
			printf("read WIP fail.\n");
		}
	}while(nor_status & 0x1);
	
	return 0;
 }
 
 int fsl_qspi_write_security_register(struct spi_nor *nor, loff_t to,
						   size_t len, size_t *retlen, const u_char *buf)
 {
	int ret = 0;
	struct fsl_qspi *q = NULL;
 
	if((nor == NULL) || (nor->priv == NULL) || (retlen == NULL) || (buf == NULL))
	{
		printk("fsl_qspi_write_security_register is error.\n");
		return -1;
	}
	q = nor->priv;
	memcpy((void*)q->buf.buf, buf, len);
	
	ret = nor_write_security_register(nor, to, len, retlen, q->buf.dma_buf);
	if(ret != 0)
	{
		printk("nor_write_otp error.\n");
		return -1;
	}
	*retlen += len;
 
	return 0;
	
 }

int nor_read_security_register(struct spi_nor *nor, loff_t from,
						size_t len, size_t *retlen, u_char *buf)
{
   int ret;
   struct fsl_qspi *q = nor->priv;
   spinor_cmd_t *cmd = NULL;
   
   cmd = cmd_seek(CMD_RSR);
   if(cmd == NULL)
   {
	   printf("cmd_seek unkown cmd = 0x%x error.\n", nor->read_opcode);
	   return -1;
   }

#if TRANS_USE_DMA
   cmd->rx_dma_en = RX_DMA_EN;
#else
   cmd->rx_dma_en = RX_DMA_DIS;
#endif
   spifc_setup_cmd(q, cmd, from, len);			   
   
#if TRANS_USE_DMA
   spifc_config_dma(buf, len, FC_DMA_RX);
   spifc_start(q);
#else
   spifc_start(q);
   ret = spifc_read_fifo(q, buf, len);
   if(ret < 0)
   {
	   printf("spifc_read_fifo error.\n");
	   return ret;
   }
#endif

   ret = spifc_wait_cmd_end(q); 			
   if(ret != 0)
   {
	   printf("spifc_wait_cmd_end error.\n");
	   return ret;
   }
   
#if TRANS_USE_DMA
   spifc_wait_dma_done(FC_DMA_RX);
#endif

   return 0;
}

int fsl_qspi_read_security_register(struct spi_nor *nor, loff_t from,
					size_t len, size_t *retlen, u_char *buf)
{
   int ret = 0;
   struct fsl_qspi *q = NULL;
   loff_t from_alligned = 0;

   if((nor == NULL) || (nor->priv == NULL) || (retlen == NULL) || (buf == NULL))
   {
	   printk("fsl_qspi_read security register is error.\n");
	   return -1;
   }
   from_alligned = from&(~(nor->page_size - 1));
   q = nor->priv;
   
   ret = nor_read_security_register(nor, from_alligned, nor->page_size, retlen, q->buf.dma_buf);
   if(ret != 0)
   {
	   printk("nor_read_security_register error.\n");
	   return -1;
   }

   memcpy(buf, (void*)q->buf.buf + (from&(nor->page_size - 1)), len);
   *retlen += len;

   return 0; 
}

 int fsl_qspi_erase_security_register(struct spi_nor *nor, loff_t offs)
{
	int ret;
	int nor_status = 0;
	struct fsl_qspi *q = nor->priv;
	spinor_cmd_t *cmd = NULL;

	cmd = cmd_seek(CMD_ESR);
	if(cmd == NULL)
	{
		printf("cmd_seek unkown cmd = 0x%x error.\n", nor->read_opcode);
		return -1;
	}
	spifc_setup_cmd(q, cmd, offs, 0);

	spifc_start(q);
	ret = spifc_wait_cmd_end(q);			 
	if(ret != 0)
	{
		printf("spifc_wait_cmd_end error.\n");
		return ret;
	}

	do{
		ret = fsl_qspi_read_reg(nor, CMD_RDSR0, &nor_status, 1);
		if(ret != 0)
		{
			printf("read WIP fail.\n");
		}
	}while(nor_status & 0x1);
	
	return 0;
}

int fsl_qspi_probe(void)
{
	struct fsl_qspi *q;
	struct spi_nor *nor;
	struct mtd_info *mtd;
	int ret, i = 0;
    int32_t offset = 0x1000;
	uint32_t len = sizeof(otpInfo);
	
	q = &spi_nor_flash;

	spifc_hwinit();
	spifc_setclk();
	
	//q = kmalloc(sizeof(*q), GFP_KERNEL);
	//if(!q)
		//return -ENOMEM;

	q->nor_num = FSL_QSPI_MAX_CHIP;
	q->buf.dma_buf = (dma_addr_t)CONFIG_NAND_DMA_BUF_ADDR;
	q->buf.buf = (uint8_t *)q->buf.dma_buf;
	q->iobase = (void __iomem *)SYS_SPI_NAND_BASE;

	spifc_disable(q);
	
	nor = &q->nor[i];
	mtd = &nor->mtd;
	nor->dev = NULL;
	//nor->flash_node = NULL;
	nor->priv = q;

	/* fill the hooks */
	nor->read_reg = fsl_qspi_read_reg;
	nor->write_reg = fsl_qspi_write_reg;
	nor->read = fsl_qspi_read;
	nor->write = fsl_qspi_write;
	nor->erase = fsl_qspi_erase;
	nor->read_security_register = fsl_qspi_read_security_register;
	nor->write_security_register = fsl_qspi_write_security_register;
	nor->erase_security_register = fsl_qspi_erase_security_register;
	nor->prepare = NULL;
	nor->unprepare = NULL;
	
	/* set the chip address for READID */
	fsl_qspi_set_base_addr(q, nor);
	spifc_enable(q);

	ret = spi_nor_scan(nor, NULL, SPI_NOR_QUAD);
	if(ret)
	{
		return ret;
	}

	/*ȡOTP*/
	ret = fsl_qspi_read_security_register(nor, offset, len, (size_t *)len, &otpInfo);
	if(ret !=0){
        printf("read otp error.\n");
	}else{
        printf("otp secure_flag is 0x%x,puk hash is 0x%x.\n",otpInfo.secure_flag,otpInfo.puk_hash[0]);
	}
	
	return 0;
}


