/* ==========================================================================
 * $File: //dwh/usb_iip/dev/software/otg/linux/drivers/dwc_otg_pcd_intr.c $
 * $Revision: #116 $
 * $Date: 2012/08/10 $
 * $Change: 2047372 $
 *
 * Synopsys HS OTG Linux Software Driver and documentation (hereinafter,
 * "Software") is an Unsupported proprietary work of Synopsys, Inc. unless
 * otherwise expressly agreed to in writing between Synopsys and you.
 *
 * The Software IS NOT an item of Licensed Software or Licensed Product under
 * any End User Software License Agreement or Agreement for Licensed Product
 * with Synopsys or any supplement thereto. You are permitted to use and
 * redistribute this Software in source and binary forms, with or without
 * modification, provided that redistributions of source code must retain this
 * notice. You may not view, use, disclose, copy or distribute this file or
 * any information contained herein except pursuant to this license grant from
 * Synopsys. If you do not agree with this notice, including the disclaimer
 * below, then you are not authorized to use the Software.
 *
 * THIS SOFTWARE IS BEING DISTRIBUTED BY SYNOPSYS SOLELY ON AN "AS IS" BASIS
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE HEREBY DISCLAIMED. IN NO EVENT SHALL SYNOPSYS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 * ========================================================================== */
#ifndef DWC_HOST_ONLY
#include "dwc_otg_pcd.h"

#ifdef DWC_UTE_CFI
#include "dwc_otg_cfi.h"
#endif

#include "global.h"


/**
 * This function returns pointer to in ep struct with number ep_num
 */
static  dwc_otg_pcd_ep_t *get_in_ep(dwc_otg_pcd_t * pcd, uint32_t ep_num)
{
    int i;
    int num_in_eps = GET_CORE_IF(pcd)->dev_if->num_in_eps;
    if (ep_num == 0)
    {
        return &pcd->ep0;
    }
    else
    {
        for (i = 0; i < num_in_eps; ++i)
        {
            if (pcd->in_ep[i].dwc_ep.num == ep_num)
                return &pcd->in_ep[i];
        }
        return 0;
    }
}

/**
 * This function returns pointer to out ep struct with number ep_num
 */
static  dwc_otg_pcd_ep_t *get_out_ep(dwc_otg_pcd_t * pcd, uint32_t ep_num)
{
    int i;
    int num_out_eps = GET_CORE_IF(pcd)->dev_if->num_out_eps;
    if (ep_num == 0)
    {
        return &pcd->ep0;
    }
    else
    {
        for (i = 0; i < num_out_eps; ++i)
        {
            if (pcd->out_ep[i].dwc_ep.num == ep_num)
                return &pcd->out_ep[i];
        }
        return 0;
    }
}

/**
 * This functions gets a pointer to an EP from the wIndex address
 * value of the control request.
 */
dwc_otg_pcd_ep_t *get_ep_by_addr(dwc_otg_pcd_t * pcd, uint16_t wIndex)
{
    dwc_otg_pcd_ep_t *ep;
    uint32_t ep_num = UE_GET_ADDR(wIndex);

	if (ep_num >= MAX_EPS_CHANNELS)
		return 0;
	
    if (ep_num == 0)
    {
        ep = &pcd->ep0;
    }
    else if (UE_GET_DIR(wIndex) == UE_DIR_IN)
    {	/* in ep */
        ep = &pcd->in_ep[ep_num - 1];
    }
    else
    {
        ep = &pcd->out_ep[ep_num - 1];
    }

    return ep;
}
/**
 * This function handles the Rx Status Queue Level Interrupt, which
 * indicates that there is a least one packet in the Rx FIFO.  The
 * packets are moved from the FIFO to memory, where they will be
 * processed when the Endpoint Interrupt Register indicates Transfer
 * Complete or SETUP Phase Done.
 *
 * Repeat the following until the Rx Status Queue is empty:
 *	 -# Read the Receive Status Pop Register (GRXSTSP) to get Packet
 *		info
 *	 -# If Receive FIFO is empty then skip to step Clear the interrupt
 *		and exit
 *	 -# If SETUP Packet call dwc_otg_read_setup_packet to copy the
 *		SETUP data to the buffer
 *	 -# If OUT Data Packet call dwc_otg_read_packet to copy the data
 *		to the destination buffer
 */
int32_t dwc_otg_pcd_handle_rx_status_q_level_intr(dwc_otg_pcd_t * pcd)
{
    dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
    dwc_otg_core_global_regs_t *global_regs = core_if->core_global_regs;
    gintmsk_data_t gintmask;
    device_grxsts_data_t status;
    dwc_otg_pcd_ep_t *ep;
    gintsts_data_t gintsts;
    gintmask.d32 = 0;

    /* Disable the Rx Status Queue Level interrupt */
    gintmask.b.rxstsqlvl = 1;
    DWC_MODIFY_REG32(&global_regs->gintmsk, gintmask.d32, 0);

    /* Get the Status from the top of the FIFO */
    status.d32 = DWC_READ_REG32(&global_regs->grxstsp);

    /* Get pointer to EP structure */
    ep = get_out_ep(pcd, status.b.epnum);
	if(0==ep)
		return -1;
    switch (status.b.pktsts)
    {
    case DWC_DSTS_GOUT_NAK:
        break;
    case DWC_STS_DATA_UPDT:
        if (status.b.bcnt && ep->dwc_ep.xfer_buff)
        {
            /** @todo NGS Check for buffer overflow? */
            dwc_otg_read_packet(core_if, ep->dwc_ep.xfer_buff,status.b.bcnt);
            ep->dwc_ep.xfer_count += status.b.bcnt;
            ep->dwc_ep.xfer_buff += status.b.bcnt;
        }
        break;
    case DWC_STS_XFER_COMP:
        break;
    case DWC_DSTS_SETUP_COMP:
        break;
    case DWC_DSTS_SETUP_UPDT:
		 printf("complete_ep11\n");
        dwc_otg_read_setup_packet(core_if, pcd->setup_pkt->d32);
        ep->dwc_ep.xfer_count += status.b.bcnt;

        #if SYNC_USB_HSIC
        if(pcd->setup_pkt->req.bRequest == 0x05)
        {
        
		REG32(REG_GPIO_OUT)=0x9;
        while(REG32(REG_GPIO_IN)!=0xFF);
		usdelay(1);
		REG32(REG_GPIO_OUT)=0;
       }
       #endif
	   #if 0
	  if(DONOT_NEED_ENUM == global.g_enum)
	  {
	  	printf("unmask endpoint\n");
          	DWC_WRITE_REG32(&pcd->core_if->dev_if->dev_global_regs->daintmsk,0x3003);
	  }
	  #endif
        break;
    default:
        break;
    }

    /* Enable the Rx Status Queue Level interrupt */
    DWC_MODIFY_REG32(&global_regs->gintmsk, 0, gintmask.d32);
    /* Clear interrupt */
    gintsts.d32 = 0;
    gintsts.b.rxstsqlvl = 1;
    DWC_WRITE_REG32(&global_regs->gintsts, gintsts.d32);

    return 1;
}

/**
 * This function examines the Device IN Token Learning Queue to
 * determine the EP number of the last IN token received.  This
 * implementation is for the Mass Storage device where there are only
 * 2 IN EPs (Control-IN and BULK-IN).
 *
 * The EP numbers for the first six IN Tokens are in DTKNQR1 and there
 * are 8 EP Numbers in each of the other possible DTKNQ Registers.
 *
 * @param core_if Programming view of DWC_otg controller.
 *
 */
/**
 * This function is called when dedicated Tx FIFO Empty interrupt occurs.
 * The active request is checked for the next packet to be loaded into
 * apropriate Tx FIFO.
 */
static int32_t write_empty_tx_fifo(dwc_otg_pcd_t * pcd, uint32_t epnum)
{
    dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
    dwc_otg_dev_if_t *dev_if = core_if->dev_if;
    dtxfsts_data_t txstatus;
    dwc_otg_pcd_ep_t *ep = 0;
    uint32_t len = 0;
    int dwords;
    txstatus.d32 = 0;

	if(epnum >= MAX_EPS_CHANNELS)
		return -1;
	
    ep = get_in_ep(pcd, epnum);
	if(0==ep)
		return -1;
    len = ep->dwc_ep.xfer_len - ep->dwc_ep.xfer_count;

    if (len > ep->dwc_ep.maxpacket)
    {
        len = ep->dwc_ep.maxpacket;
    }

    dwords = (len + 3) / 4;

    /* While there is space in the queue and space in the FIFO and
     * More data to tranfer, Write packets to the Tx FIFO */
    txstatus.d32 = DWC_READ_REG32(&dev_if->in_ep_regs[epnum]->dtxfsts);
    while (txstatus.b.txfspcavail > dwords &&
            ep->dwc_ep.xfer_count < ep->dwc_ep.xfer_len &&
            ep->dwc_ep.xfer_len != 0)
    {
        /* Write the FIFO */
        dwc_otg_ep_write_packet(core_if, &ep->dwc_ep, 0);

        len = ep->dwc_ep.xfer_len - ep->dwc_ep.xfer_count;
        if (len > ep->dwc_ep.maxpacket)
        {
            len = ep->dwc_ep.maxpacket;
        }

        dwords = (len + 3) / 4;
        txstatus.d32 = DWC_READ_REG32(&dev_if->in_ep_regs[epnum]->dtxfsts);
    }


    return 1;
}

/**
 * This function configures EPO to receive SETUP packets.
 *
 * @todo NGS: Update the comments from the HW FS.
 *
 *	-# Program the following fields in the endpoint specific registers
 *	for Control OUT EP 0, in order to receive a setup packet
 *	- DOEPTSIZ0.Packet Count = 3 (To receive up to 3 back to back
 *	  setup packets)
 *	- DOEPTSIZE0.Transfer Size = 24 Bytes (To receive up to 3 back
 *	  to back setup packets)
 *		- In DMA mode, DOEPDMA0 Register with a memory address to
 *		  store any setup packets received
 *
 * @param core_if Programming view of DWC_otg controller.
 * @param pcd	  Programming view of the PCD.
 */
 void ep0_out_start(dwc_otg_core_if_t * core_if,
                           dwc_otg_pcd_t * pcd)
{
    dwc_otg_dev_if_t *dev_if = core_if->dev_if;
    deptsiz0_data_t doeptsize0;
    depctl_data_t doepctl;
    doepctl.d32 = 0;
    doeptsize0.d32 =0;


    doepctl.d32 = DWC_READ_REG32(&dev_if->out_ep_regs[0]->doepctl);
    if (doepctl.b.epena)
    {
        return;
    }

    doeptsize0.b.supcnt = 3;
    doeptsize0.b.pktcnt = 1;
    doeptsize0.b.xfersize = 8 * 3;

    /** put here as for Hermes mode deptisz register should not be written */
    DWC_WRITE_REG32(&dev_if->out_ep_regs[0]->doeptsiz,doeptsize0.d32);

    /** DOEPCTL0 Register write cnak will be set after setup interrupt */
    doepctl.d32 = 0;
    doepctl.b.epena = 1;

    DWC_MODIFY_REG32(&dev_if->out_ep_regs[0]->doepctl, 0, doepctl.d32);

}


void USB_Recovercfg(dwc_otg_pcd_t *pcd)
{
    int i = 0;
    dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
    dwc_otg_dev_if_t *dev_if = core_if->dev_if;
    depctl_data_t doepctl;
    depctl_data_t diepctl;
    doepctl.d32 = 0;
    diepctl.d32 = 0;

    for (i = 0; i < dev_if->num_out_eps; i++)
    {
        doepctl.d32= DWC_READ_REG32(&dev_if->out_ep_regs[i+1]->doepctl);
        doepctl.b.eptype = pcd->out_ep[i].dwc_ep.type;
        doepctl.b.mps = pcd->out_ep[i].dwc_ep.maxpacket;
        DWC_WRITE_REG32(&dev_if->out_ep_regs[i+1]->doepctl, doepctl.d32);

        diepctl.d32 = DWC_READ_REG32(&dev_if->in_ep_regs[i+1]->diepctl);
        diepctl.b.eptype = pcd->in_ep[i].dwc_ep.type;
        diepctl.b.mps = pcd->in_ep[i].dwc_ep.maxpacket;
        DWC_WRITE_REG32(&dev_if->in_ep_regs[i+1]->diepctl, diepctl.d32);
    }

}
int32_t dwc_otg_pcd_handle_usb_reset_intr(dwc_otg_pcd_t * pcd)
{
    dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
    dwc_otg_dev_if_t *dev_if = core_if->dev_if;
    depctl_data_t doepctl;
    daint_data_t daintmsk;
    doepmsk_data_t doepmsk;
    diepmsk_data_t diepmsk ;
    dcfg_data_t dcfg ;
    grstctl_t resetctl;
    dctl_data_t dctl;
    int i = 0;
    gintsts_data_t gintsts;
    pcgcctl_data_t power;
    doepctl.d32 = 0;
    daintmsk.d32 = 0;
    doepmsk.d32 = 0;
    diepmsk.d32 = 0;
    dcfg.d32 = 0;
    resetctl.d32 = 0;
    dctl.d32 = 0;
    power.d32 = 0;

    power.d32 = DWC_READ_REG32(core_if->pcgcctl);
    if (power.b.stoppclk)
    {
        power.d32 = 0;
        power.b.stoppclk = 1;
        DWC_MODIFY_REG32(core_if->pcgcctl, power.d32, 0);

        power.b.pwrclmp = 1;
        DWC_MODIFY_REG32(core_if->pcgcctl, power.d32, 0);

        power.b.rstpdwnmodule = 1;
        DWC_MODIFY_REG32(core_if->pcgcctl, power.d32, 0);
    }

    printf("R\n");

    /* Clear the Remote Wakeup Signalling */
    dctl.b.rmtwkupsig = 1;
    DWC_MODIFY_REG32(&core_if->dev_if->dev_global_regs->dctl, dctl.d32, 0);

    /* Set NAK for all OUT EPs */
    doepctl.b.snak = 1;
    for (i = 0; i <= dev_if->num_out_eps; i++)
    {
        DWC_WRITE_REG32(&dev_if->out_ep_regs[i]->doepctl, doepctl.d32);
    }

    /* Flush the NP Tx FIFO */
    dwc_otg_flush_tx_fifo(core_if, 0x10);
    /* Flush the Learning Queue */
    resetctl.b.intknqflsh = 1;
    DWC_WRITE_REG32(&core_if->core_global_regs->grstctl, resetctl.d32);
    {
        daintmsk.b.inep0 = 1;
        daintmsk.b.outep0 = 1;
        DWC_WRITE_REG32(&dev_if->dev_global_regs->daintmsk,daintmsk.d32);

        doepmsk.b.setup = 1;
        doepmsk.b.xfercompl = 1;
        doepmsk.b.ahberr = 1;
        doepmsk.b.epdisabled = 1;
        DWC_WRITE_REG32(&dev_if->dev_global_regs->doepmsk, doepmsk.d32);

        diepmsk.b.xfercompl = 1;
        diepmsk.b.timeout = 1;
        diepmsk.b.epdisabled = 1;
        diepmsk.b.ahberr = 1;

        DWC_WRITE_REG32(&dev_if->dev_global_regs->diepmsk, diepmsk.d32);
    }

    /* Reset Device Address */
    dcfg.d32 = DWC_READ_REG32(&dev_if->dev_global_regs->dcfg);
    dcfg.b.devaddr = 0;
    DWC_WRITE_REG32(&dev_if->dev_global_regs->dcfg, dcfg.d32);

    /* Clear interrupt */
    gintsts.d32 = 0;
    gintsts.b.usbreset = 1;
    DWC_WRITE_REG32(&core_if->core_global_regs->gintsts, gintsts.d32);

    USB_Recovercfg(pcd);

    return 1;
}

/**
 * Get the device speed from the device status register and convert it
 * to USB speed constant.
 *
 * @param core_if Programming view of DWC_otg controller.
 */
static int get_device_speed(dwc_otg_core_if_t * core_if)
{
    dsts_data_t dsts;
    int speed = 0;
    dsts.d32 = DWC_READ_REG32(&core_if->dev_if->dev_global_regs->dsts);

    switch (dsts.b.enumspd)
    {
    case DWC_DSTS_ENUMSPD_HS_PHY_30MHZ_OR_60MHZ:
        speed = USB_SPEED_HIGH;
        break;
    case DWC_DSTS_ENUMSPD_FS_PHY_30MHZ_OR_60MHZ:
    case DWC_DSTS_ENUMSPD_FS_PHY_48MHZ:
        speed = USB_SPEED_FULL;
        break;

    case DWC_DSTS_ENUMSPD_LS_PHY_6MHZ:
        speed = USB_SPEED_LOW;
        break;
    }

    return speed;
}

/**
 * Read the device status register and set the device speed in the
 * data structure.
 * Set up EP0 to receive SETUP packets by calling dwc_ep0_activate.
 */
extern void init_devspd(dwc_otg_core_if_t * core_if,uint8_t speed);


int32_t dwc_otg_pcd_handle_enum_done_intr(dwc_otg_pcd_t * pcd)
{
    dwc_otg_pcd_ep_t *ep0 = &pcd->ep0;
    gintsts_data_t gintsts;
    gusbcfg_data_t gusbcfg;
    dwc_otg_core_global_regs_t *global_regs = GET_CORE_IF(pcd)->core_global_regs;
    uint8_t utmi16b, utmi8b;
    int speed;
    printf("D\n");
    
    utmi16b = 6;	//vahrama old value was 6; data book5add by lx 2012.10.13
    utmi8b = 9;
    dwc_otg_ep0_activate(GET_CORE_IF(pcd), &ep0->dwc_ep);
    ep0_out_start(GET_CORE_IF(pcd), pcd);
#if SYNC_USB_HSIC

	REG32(REG_GPIO_OUT)=5;
	while(REG32(REG_GPIO_IN)!=0xFF);
	usdelay(1);
	REG32(REG_GPIO_OUT)=0;

	REG32(REG_GPIO_OUT)=6;
	while(REG32(REG_GPIO_IN)!=0xFF);
	usdelay(1);
	REG32(REG_GPIO_OUT)=0;

	REG32(REG_GPIO_OUT)=7;
	while(REG32(REG_GPIO_IN)!=0xFF);
	usdelay(1);
	REG32(REG_GPIO_OUT)=0;

	REG32(REG_GPIO_OUT)=8;
	while(REG32(REG_GPIO_IN)!=0xFF);
	usdelay(1);
	REG32(REG_GPIO_OUT)=0;

#endif
    if (pcd->ep0state == EP0_DISCONNECT)
    {
        pcd->ep0state = EP0_IDLE;
    }
    else if (pcd->ep0state == EP0_STALL)
    {
        pcd->ep0state = EP0_IDLE;
    }

    ep0->stopped = 0;

    speed = get_device_speed(GET_CORE_IF(pcd));
    /* Set USB turnaround time based on device speed and PHY interface. */
    gusbcfg.d32 = DWC_READ_REG32(&global_regs->gusbcfg);
    if (speed == USB_SPEED_HIGH)
    {
    
        if (GET_CORE_IF(pcd)->hwcfg2.b.hs_phy_type == DWC_HWCFG2_HS_PHY_TYPE_UTMI_ULPI)
        {
            /* UTMI+  OR  ULPI interface */
            if (gusbcfg.b.ulpi_utmi_sel == 1)
            {
                /* ULPI interface */
                gusbcfg.b.usbtrdtim = 9;
            }
            else
            {
                /* UTMI+ interface */
                if (GET_CORE_IF(pcd)->core_params->phy_utmi_width == 16)
                {
                    gusbcfg.b.usbtrdtim = utmi16b;
                }
                else
                {
                    gusbcfg.b.usbtrdtim = utmi8b;
                }
            }
        }

        printf("2\n");
        
    }
    else
    { 
        printf("1\n");
        pcd->out_ep[0].dwc_ep.maxpacket = 64;
        pcd->in_ep[0].dwc_ep.maxpacket = 64;
        pcd->out_ep[1].dwc_ep.maxpacket = 64;
        pcd->in_ep[1].dwc_ep.maxpacket = 64;   
        init_devspd(pcd->core_if,1);
        g_config_desc.atTxEP[0].wMaxPacketSize = USB_FULLSPEED_BULK_MAXSIZE;
        g_config_desc.atRxEP[0].wMaxPacketSize = USB_FULLSPEED_BULK_MAXSIZE;
        g_config_desc.atTxEP1[0].wMaxPacketSize = USB_FULLSPEED_BULK_MAXSIZE;
        g_config_desc.atRxEP1[0].wMaxPacketSize = USB_FULLSPEED_BULK_MAXSIZE;
        /* Full or low speed */
        gusbcfg.b.usbtrdtim = 9;
    }
    DWC_WRITE_REG32(&global_regs->gusbcfg, gusbcfg.d32);

    /* Clear interrupt */
    gintsts.d32 = 0;
    gintsts.b.enumdone = 1;
    DWC_WRITE_REG32(&GET_CORE_IF(pcd)->core_global_regs->gintsts,gintsts.d32);
    return 1;
}



/**
 * This funcion stalls EP0.
 */
void ep0_do_stall(dwc_otg_pcd_t * pcd, const int err_val)
{
    dwc_otg_pcd_ep_t *ep0 = &pcd->ep0;

    ep0->dwc_ep.is_in = 1;
    dwc_otg_ep_set_stall(GET_CORE_IF(pcd), &ep0->dwc_ep);
    pcd->ep0.stopped = 1;
    pcd->ep0state = EP0_IDLE;
    ep0_out_start(GET_CORE_IF(pcd), pcd);
}

static  void do_setup_in_out_status_phase(dwc_otg_pcd_t * pcd,s8 udir)
{
    dwc_otg_pcd_ep_t *ep0 = &pcd->ep0;
    if (pcd->ep0state == EP0_STALL)
    {
        return;
    }

    if(udir)
    {
     pcd->ep0state = EP0_IN_STATUS_PHASE;
    }
    else
    {
      pcd->ep0state = EP0_OUT_STATUS_PHASE;
    }

    /* Prepare for more SETUP Packets */
    ep0->dwc_ep.xfer_len = 0;
    ep0->dwc_ep.xfer_count = 0;
    ep0->dwc_ep.is_in = udir;
    dwc_otg_ep0_start_transfer(GET_CORE_IF(pcd), &ep0->dwc_ep);
}


/**
 * Clear the EP halt (STALL) and if pending requests start the
 * transfer.
 */
static  void pcd_clear_halt(dwc_otg_pcd_t * pcd, dwc_otg_pcd_ep_t * ep)
{
    if (ep->dwc_ep.stall_clear_flag == 0)
        dwc_otg_ep_clear_stall(GET_CORE_IF(pcd), &ep->dwc_ep);

    /* Reactive the EP */
    dwc_otg_ep_activate(GET_CORE_IF(pcd), &ep->dwc_ep);
    if (ep->stopped)
    {
        ep->stopped = 0;
        /* If there is a request in the EP queue start it */

        /** @todo FIXME: this causes an EP mismatch in DMA mode.
         * epmismatch not yet implemented. */

        /*
         * Above fixme is solved by implmenting a tasklet to call the
         * start_next_request(), outside of interrupt context at some
         * time after the current time, after a clear-halt setup packet.
         * Still need to implement ep mismatch in the future if a gadget
         * ever uses more than one endpoint at once
         */
        ep->queue_sof = 1;
    }
    /* Start Control Status Phase */
    do_setup_in_out_status_phase(pcd,1);
}

/**
 * This function process the GET_STATUS Setup Commands.
 */
static  void do_get_status(dwc_otg_pcd_t * pcd)
{
    usb_device_request_t ctrl = pcd->setup_pkt->req;
    dwc_otg_pcd_ep_t *ep;
    dwc_otg_pcd_ep_t *ep0 = &pcd->ep0;
    uint16_t *status = pcd->status_buf;

    switch (UT_GET_RECIPIENT(ctrl.bmRequestType))
    {
    case UT_INTERFACE:
        *status = 0;
        break;

    case UT_ENDPOINT:
        ep = get_ep_by_addr(pcd, ctrl.wIndex);
        if (ep == 0 || ctrl.wLength > 2)
        {
            ep0_do_stall(pcd, -DWC_E_NOT_SUPPORTED);
            return;
        }
        /** @todo check for EP stall */
        *status = ep->stopped;
        break;
    }
    pcd->ep0_pending = 1;
    ep0->dwc_ep.start_xfer_buff = (uint8_t *) status;
    ep0->dwc_ep.xfer_buff = (uint8_t *) status;
    ep0->dwc_ep.xfer_len = 2;
    ep0->dwc_ep.xfer_count = 0;
    ep0->dwc_ep.total_len = ep0->dwc_ep.xfer_len;
    dwc_otg_ep0_start_transfer(GET_CORE_IF(pcd), &ep0->dwc_ep);
}


static  void do_clear_feature(dwc_otg_pcd_t * pcd)
{
    usb_device_request_t ctrl = pcd->setup_pkt->req;
    dwc_otg_pcd_ep_t *ep = 0;
    switch (UT_GET_RECIPIENT(ctrl.bmRequestType))
    {
    case UT_ENDPOINT:
        ep = get_ep_by_addr(pcd, ctrl.wIndex);
        if (ep == 0)
        {
            ep0_do_stall(pcd, -DWC_E_NOT_SUPPORTED);
            return;
        }

        pcd_clear_halt(pcd, ep);

        break;

    }
}

/**
 * This function process the SET_ADDRESS Setup Commands.
 */
static  void do_set_address(dwc_otg_pcd_t * pcd)
{
    dwc_otg_dev_if_t *dev_if = GET_CORE_IF(pcd)->dev_if;
    usb_device_request_t ctrl = pcd->setup_pkt->req;
    dcfg_data_t dcfg;
    dcfg.d32 = 0;

    if (ctrl.bmRequestType == UT_DEVICE)
    {

        dcfg.b.devaddr = ctrl.wValue;
        DWC_MODIFY_REG32(&dev_if->dev_global_regs->dcfg, 0, dcfg.d32);
#if SYNC_USB_HSIC
		REG32(REG_GPIO_OUT)=0xa;
        while(REG32(REG_GPIO_IN)!=0xFF);
		usdelay(1);
		REG32(REG_GPIO_OUT)=0;

		REG32(REG_GPIO_OUT)=0xb;
        while(REG32(REG_GPIO_IN)!=0xFF);
		usdelay(1);
		REG32(REG_GPIO_OUT)=0;
		
#endif
        do_setup_in_out_status_phase(pcd,1);
#if SYNC_USB_HSIC
       if(pcd->setup_pkt->req.bRequest == 0x05)
       {
        	REG32(REG_GPIO_OUT)=0x9;
        	while(REG32(REG_GPIO_IN)!=0xFF);
        	usdelay(10);
        	REG32(REG_GPIO_OUT)=0;
        }
#endif
    }
}

/**
 *	This function processes SETUP commands. In Linux, the USB Command
 *	processing is done in two places - the first being the PCD and the
 *	second in the Gadget Driver (for example, the File-Backed Storage
 *	Gadget Driver).
 *
 * <table>
 * <tr><td>Command	</td><td>Driver </td><td>Description</td></tr>
 *
 * <tr><td>GET_STATUS </td><td>PCD </td><td>Command is processed as
 * defined in chapter 9 of the USB 2.0 Specification chapter 9
 * </td></tr>
 *
 * <tr><td>CLEAR_FEATURE </td><td>PCD </td><td>The Device and Endpoint
 * requests are the ENDPOINT_HALT feature is procesed, all others the
 * interface requests are ignored.</td></tr>
 *
 * <tr><td>SET_FEATURE </td><td>PCD </td><td>The Device and Endpoint
 * requests are processed by the PCD.  Interface requests are passed
 * to the Gadget Driver.</td></tr>
 *
 * <tr><td>SET_ADDRESS </td><td>PCD </td><td>Program the DCFG reg,
 * with device address received </td></tr>
 *
 * <tr><td>GET_DESCRIPTOR </td><td>Gadget Driver </td><td>Return the
 * requested descriptor</td></tr>
 *
 * <tr><td>SET_DESCRIPTOR </td><td>Gadget Driver </td><td>Optional -
 * not implemented by any of the existing Gadget Drivers.</td></tr>
 *
 * <tr><td>SET_CONFIGURATION </td><td>Gadget Driver </td><td>Disable
 * all EPs and enable EPs for new configuration.</td></tr>
 *
 * <tr><td>GET_CONFIGURATION </td><td>Gadget Driver </td><td>Return
 * the current configuration</td></tr>
 *
 * <tr><td>SET_INTERFACE </td><td>Gadget Driver </td><td>Disable all
 * EPs and enable EPs for new configuration.</td></tr>
 *
 * <tr><td>GET_INTERFACE </td><td>Gadget Driver </td><td>Return the
 * current interface.</td></tr>
 *
 * <tr><td>SYNC_FRAME </td><td>PCD </td><td>Display debug
 * message.</td></tr>
 * </table>
 *
 * When the SETUP Phase Done interrupt occurs, the PCD SETUP commands are
 * processed by pcd_setup. Calling the Function Driver's setup function from
 * pcd_setup processes the gadget SETUP commands.
 */
void active_allep(dwc_otg_pcd_t * pcd)
{
    int i = 0;
    dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
    dwc_otg_dev_if_t *dev_if = core_if->dev_if;
    for (i = 0; i <dev_if->num_out_eps; i++)
    {
        dwc_otg_ep_activate(pcd->core_if,&pcd->out_ep[i].dwc_ep);
        dwc_otg_ep_activate(pcd->core_if,&pcd->in_ep[i].dwc_ep);
    }

}

extern int  dwc_setup(dwc_otg_pcd_t * pcd,  usb_device_request_t * ctrl);

static  void pcd_setup(dwc_otg_pcd_t * pcd)
{
    dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
    usb_device_request_t ctrl = pcd->setup_pkt->req;
    dwc_otg_pcd_ep_t *ep0 = &pcd->ep0;

    ep0->stopped = 0;

    if (ctrl.bmRequestType & UE_DIR_IN)
    {
        ep0->dwc_ep.is_in = 1;
        pcd->ep0state = EP0_IN_DATA_PHASE;
    }
    else
    {
        ep0->dwc_ep.is_in = 0;
        pcd->ep0state = EP0_OUT_DATA_PHASE;
    }

    if (ctrl.wLength == 0)
    {
        printf("ctrl.wLength == 0\n");
        ep0->dwc_ep.is_in = 1;
        pcd->ep0state = EP0_IN_STATUS_PHASE;
    }
   if((ctrl.bmRequestType&0x60) == 0x40)
   {
	printf("vender request\n");
	 ep0_do_stall(pcd, -DWC_E_NOT_SUPPORTED);
	 return;
   }

    if( UR_GET_STATUS == ctrl.bRequest)
    {
           do_get_status(pcd);
    }

    else if( UR_CLEAR_FEATURE ==  ctrl.bRequest)
    	{
        do_clear_feature(pcd);
    	}

    else if( UR_SET_FEATURE ==ctrl.bRequest) 
    	{}

    else if( UR_SET_ADDRESS == ctrl.bRequest)
    	{
	     #if SYNC_SETADDRESS
	     pcd->request_config = 2;
	     active_allep(pcd);
	     #endif
            do_set_address(pcd); 
    	}

    else if(  UR_GET_DESCRIPTOR== ctrl.bRequest)
    	{
            dwc_setup(pcd, &ctrl);
    	}

    else if( UR_SET_CONFIG== ctrl.bRequest)
    	{
        pcd->request_config = 1;
        active_allep(pcd);
        do_setup_in_out_status_phase(pcd,1);
        ep0_out_start(core_if, pcd);
    	}
    else
  	{
        ep0_do_stall(pcd, -DWC_E_NOT_SUPPORTED);
  	}
    
}

/**
 * This function completes the ep0 control transfer.
 */
static int32_t ep0_complete_request(dwc_otg_pcd_ep_t * ep)
{
    dwc_otg_core_if_t *core_if = GET_CORE_IF(ep->pcd);
    dwc_otg_dev_if_t *dev_if = core_if->dev_if;
    dwc_otg_dev_in_ep_regs_t *in_ep_regs =
    dev_if->in_ep_regs[ep->dwc_ep.num];
    
    deptsiz0_data_t deptsiz;
    int is_last = 0;
    dwc_otg_pcd_t *pcd = ep->pcd;

    if (pcd->ep0_pending)
    {
        if (ep->dwc_ep.is_in)
        {

            do_setup_in_out_status_phase(pcd,0);
        }
        else
        {
            do_setup_in_out_status_phase(pcd,1);
        }
        pcd->ep0_pending = 0;
        return 1;
    }

    if (pcd->ep0state == EP0_OUT_STATUS_PHASE
            || pcd->ep0state == EP0_IN_STATUS_PHASE)
    {
        is_last = 1;
    }
    else if (ep->dwc_ep.is_in)
    {
        deptsiz.d32 = DWC_READ_REG32(&in_ep_regs->dieptsiz);

       if (deptsiz.b.xfersize == 0)
        {
            do_setup_in_out_status_phase(pcd,0);
        }
    }
    else
    {

        ep0_out_start(core_if, pcd);

    }

    /* Complete the request */
    if (is_last)
    {
        ep->dwc_ep.start_xfer_buff = 0;
        ep->dwc_ep.xfer_buff = 0;
        ep->dwc_ep.xfer_len = 0;
        return 1;
    }
    return 0;
}

/**
 * This function completes the request for the EP. If there are
 * additional requests for the EP in the queue they will be started.
 */
static void complete_ep(dwc_otg_pcd_ep_t * ep)
{
    dwc_otg_core_if_t *core_if = GET_CORE_IF(ep->pcd);
    dwc_otg_dev_if_t *dev_if = core_if->dev_if;
    dwc_otg_dev_in_ep_regs_t *in_ep_regs = dev_if->in_ep_regs[ep->dwc_ep.num];
    deptsiz_data_t deptsiz;

    if (ep->dwc_ep.is_in)
    {
        deptsiz.d32 = DWC_READ_REG32(&in_ep_regs->dieptsiz);
        {
            if (deptsiz.b.xfersize == 0 && deptsiz.b.pktcnt == 0)
            {
                /*      Check if the whole transfer was completed,
                 *      if no, setup transfer for next portion of data
                 */
                if (ep->dwc_ep.xfer_len < ep->dwc_ep.total_len)
                {
                    dwc_otg_ep_start_transfer(core_if,&ep->dwc_ep);
                }
                else if (ep->dwc_ep.sent_zlp)
                {
                    /*
                     * This fragment of code should initiate 0
                     * length trasfer in case if it is queued
                     * a trasfer with size divisible to EPs max
                     * packet size and with usb_request zero field
                     * is set, which means that after data is transfered,
                     * it is also should be transfered
                     * a 0 length packet at the end. For Slave and
                     * Buffer DMA modes in this case SW has
                     * to initiate 2 transfers one with transfer size,
                     * and the second with 0 size. For Desriptor
                     * DMA mode SW is able to initiate a transfer,
                     * which will handle all the packets including
                     * the last  0 legth.
                     */
                    ep->dwc_ep.sent_zlp = 0;
                    dwc_otg_ep_start_zl_transfer(core_if,
                                                 &ep->dwc_ep);
                }
                else
                {

                    if (ep->dwc_ep.fnUsbCb)
                        ep->dwc_ep.fnUsbCb( (WORD32)(ep->dwc_ep.pPara), 0, ep->dwc_ep.xfer_count, ep->dwc_ep.total_len);

                }
            }
        }
    }
    // OUT EP
    else
    {
       
        dwc_otg_dev_out_ep_regs_t *out_ep_regs = dev_if->out_ep_regs[ep->dwc_ep.num];
        {
            deptsiz.d32 = 0;
            deptsiz.d32 = DWC_READ_REG32(&out_ep_regs->doeptsiz);
            if (deptsiz.b.xfersize>0)
            {

                if (ep->dwc_ep.fnUsbCb)
                    ep->dwc_ep.fnUsbCb( (WORD32)(ep->dwc_ep.pPara), 0, ep->dwc_ep.xfer_count, ep->dwc_ep.total_len);

            }
            else
            {
                dwc_otg_ep_start_transfer(core_if,&ep->dwc_ep);
            }
        }


    }

}

/**
 * This function handles EP0 Control transfers.
 *
 * The state of the control transfers are tracked in
 * <code>ep0state</code>.
 */
static void handle_ep0(dwc_otg_pcd_t * pcd)
{
    dwc_otg_pcd_ep_t *ep0 = &pcd->ep0;

    if(EP0_DISCONNECT == pcd->ep0state )
    	{
		return;
	}
   else if(EP0_IDLE == pcd->ep0state ) 
   	{
	        pcd->request_config = 0;

	        pcd_setup(pcd);
               #if SYNC_USB_CTRL
		 REG32(ARM_PORTA)=0x55;
                #endif
   	}
	else if(EP0_IN_DATA_PHASE == pcd->ep0state)
	{
	        if (ep0->dwc_ep.xfer_count < ep0->dwc_ep.total_len)
	        {
	            dwc_otg_ep0_continue_transfer(GET_CORE_IF(pcd),&ep0->dwc_ep);
	        }
	        else if (ep0->dwc_ep.sent_zlp)
	        {
	            dwc_otg_ep0_continue_transfer(GET_CORE_IF(pcd),&ep0->dwc_ep);
	            ep0->dwc_ep.sent_zlp = 0;
	        }
	        else
	        {
	            ep0_complete_request(ep0);
	        }
	}
    else if( EP0_OUT_DATA_PHASE == pcd->ep0state)
    	{
        if (ep0->dwc_ep.xfer_count < ep0->dwc_ep.total_len)
        {
            dwc_otg_ep0_continue_transfer(GET_CORE_IF(pcd),
                                          &ep0->dwc_ep);
        }
        else if (ep0->dwc_ep.sent_zlp)
        {
            dwc_otg_ep0_continue_transfer(GET_CORE_IF(pcd), &ep0->dwc_ep);
            ep0->dwc_ep.sent_zlp = 0;
        }
        else
        {
            ep0_complete_request(ep0);
        }
    	}
	else if(EP0_IN_STATUS_PHASE == pcd->ep0state ||EP0_OUT_STATUS_PHASE == pcd->ep0state )
	{
	        ep0_complete_request(ep0);
	        pcd->ep0state = EP0_IDLE;
	        ep0->stopped = 1;
	        ep0->dwc_ep.is_in = 0;	/* OUT for next SETUP */
	}
      else if(EP0_STALL == pcd->ep0state )
      	{

	 }
	 else
	  {}

   

}

/**
 * Handler for the IN EP NAK interrupt.
 */

/**
 * Handler for the OUT EP NAK interrupt.
 */
static  int32_t handle_out_ep_nak_intr(dwc_otg_pcd_t * pcd,
                                       const uint32_t epnum)
{
    /** @todo implement ISR */
    dwc_otg_core_if_t *core_if;
    doepmsk_data_t intr_mask;
    intr_mask.d32 = 0;

    core_if = GET_CORE_IF(pcd);
    intr_mask.b.nak = 1;
    {
        DWC_MODIFY_REG32(&core_if->dev_if->dev_global_regs->doepmsk,intr_mask.d32, 0);
    }

    return 1;
}

/**
 * Handler for the OUT EP NYET interrupt.
 */
static  int32_t handle_out_ep_nyet_intr(dwc_otg_pcd_t * pcd,
                                        const uint32_t epnum)
{
    /** @todo implement ISR */
    dwc_otg_core_if_t *core_if;
    doepmsk_data_t intr_mask;
    intr_mask.d32 = 0 ;

    core_if = GET_CORE_IF(pcd);
    intr_mask.b.nyet = 1;
    {
        DWC_MODIFY_REG32(&core_if->dev_if->dev_global_regs->doepmsk, intr_mask.d32, 0);
    }

    return 1;
}

/**
 * This interrupt indicates that an IN EP has a pending Interrupt.
 * The sequence for handling the IN EP interrupt is shown below:
 * -#	Read the Device All Endpoint Interrupt register
 * -#	Repeat the following for each IN EP interrupt bit set (from
 *		LSB to MSB).
 * -#	Read the Device Endpoint Interrupt (DIEPINTn) register
 * -#	If "Transfer Complete" call the request complete function
 * -#	If "Endpoint Disabled" complete the EP disable procedure.
 * -#	If "AHB Error Interrupt" log error
 * -#	If "Time-out Handshake" log error
 * -#	If "IN Token Received when TxFIFO Empty" write packet to Tx
 *		FIFO.
 * -#	If "IN Token EP Mismatch" (disable, this is handled by EP
 *		Mismatch Interrupt)
 */
static int32_t dwc_otg_pcd_handle_in_ep_intr(dwc_otg_pcd_t * pcd)
{

    dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
    dwc_otg_dev_if_t *dev_if = core_if->dev_if;
    diepint_data_t diepint;
    depctl_data_t depctl;
    uint32_t ep_intr;
    uint32_t epnum = 0;
    dwc_otg_pcd_ep_t *ep;
    dwc_ep_t *dwc_ep;
    diepmsk_data_t diepmsk;
    uint32_t fifoemptymsk;
    diepint.d32 = 0;
    depctl.d32 = 0;
    diepmsk.d32 = 0;

    ep_intr = dwc_otg_read_dev_all_in_ep_intr(core_if);
    /* Service the Device IN interrupts for each endpoint */
    while ((ep_intr)&&(epnum<MAX_EPS_CHANNELS))
    {
        if (ep_intr & 0x1)
        {

            /* Get EP pointer */
            ep = get_in_ep(pcd, epnum);
            dwc_ep = &ep->dwc_ep;
            depctl.d32 = DWC_READ_REG32(&dev_if->in_ep_regs[epnum]->diepctl);
            diepint.d32 = dwc_otg_read_dev_in_ep_intr(core_if, dwc_ep);
            /* Transfer complete */
            if (diepint.b.xfercompl)
            {
                {
                   /* Disable the Tx FIFO Empty Interrupt for this EP */
                    fifoemptymsk = 0x1 << dwc_ep->num;
                    DWC_MODIFY_REG32(&core_if->dev_if->dev_global_regs->dtknqr4_fifoemptymsk,fifoemptymsk, 0);
                }
                /* Clear the bit in DIEPINTn for this interrupt */
                CLEAR_IN_EP_INTR(core_if, epnum, xfercompl);

                /* Complete the transfer */
                if (epnum == 0)
                {
                    handle_ep0(pcd);

                }
                else
                {

                    complete_ep(ep);
                    if (diepint.b.nak)
                        CLEAR_IN_EP_INTR(core_if, epnum, nak);
                }
            }
            /* Endpoint disable      */
            if (diepint.b.epdisabled)
            {

                CLEAR_IN_EP_INTR(core_if, epnum, epdisabled);
            }

            /* TimeOUT Handshake (non-ISOC IN EPs) */
            if (diepint.b.timeout)
            {
                CLEAR_IN_EP_INTR(core_if, epnum, timeout);
            }
            /** IN Token received with TxF Empty */
            if (diepint.b.intktxfemp)
            {
                if (!ep->stopped && epnum != 0)
                {
                    diepmsk.b.intktxfemp = 1;
                    {
                        DWC_MODIFY_REG32
                        (&dev_if->dev_global_regs->diepmsk,
                         diepmsk.d32, 0);
                    }
                }
                CLEAR_IN_EP_INTR(core_if, epnum, intktxfemp);
            }

            /** IN Endpoint NAK Effective */
            if (diepint.b.inepnakeff)
            {
                /* Periodic EP */
                if (ep->disabling)
                {
                    depctl.d32 = 0;
                    depctl.b.snak = 1;
                    depctl.b.epdis = 1;
                    DWC_MODIFY_REG32(&dev_if->in_ep_regs
                                     [epnum]->diepctl,
                                     depctl.d32,
                                     depctl.d32);
                }
                CLEAR_IN_EP_INTR(core_if, epnum, inepnakeff);

            }

            /** IN EP Tx FIFO Empty Intr */
            if (diepint.b.emptyintr)
            {
               printf("diepint.b.emptyintr\n");
                write_empty_tx_fifo(pcd, epnum);
                
                CLEAR_IN_EP_INTR(core_if, epnum, emptyintr);
 
            }
            /* NAK Interrutp */
            if (diepint.b.nak)
            {
                CLEAR_IN_EP_INTR(core_if, epnum, nak);
            }
        }
        epnum++;
        ep_intr >>= 1;
    }

    return 1;
#undef CLEAR_IN_EP_INTR
}

/**
 * This interrupt indicates that an OUT EP has a pending Interrupt.
 * The sequence for handling the OUT EP interrupt is shown below:
 * -#	Read the Device All Endpoint Interrupt register
 * -#	Repeat the following for each OUT EP interrupt bit set (from
 *		LSB to MSB).
 * -#	Read the Device Endpoint Interrupt (DOEPINTn) register
 * -#	If "Transfer Complete" call the request complete function
 * -#	If "Endpoint Disabled" complete the EP disable procedure.
 * -#	If "AHB Error Interrupt" log error
 * -#	If "Setup Phase Done" process Setup Packet (See Standard USB
 *		Command Processing)
 */
static int32_t dwc_otg_pcd_handle_out_ep_intr(dwc_otg_pcd_t * pcd)
{

    dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
    uint32_t ep_intr;
    doepint_data_t doepint;
    doepint_data_t doepint1;
    uint32_t epnum = 0;
    dwc_otg_pcd_ep_t *ep;
    dwc_ep_t *dwc_ep;

    doepint.d32 = 0;
    doepint1.d32 = 0;

    printf("dwc_otg_pcd_handle_out_ep_int\nr");
    ep_intr = dwc_otg_read_dev_all_out_ep_intr(core_if);
    while ((ep_intr)&&(epnum<MAX_EPS_CHANNELS))
    {
        if (ep_intr & 0x1)
        {
            /* Get EP pointer */
            ep = get_out_ep(pcd, epnum);
            dwc_ep = &ep->dwc_ep;
            doepint.d32 = dwc_otg_read_dev_out_ep_intr(core_if, dwc_ep);
            /* Transfer complete */
            if (doepint.b.xfercompl)
            {
                if (epnum == 0)
                {
                    /* Clear the bit in DOEPINTn for this interrupt */
                    CLEAR_OUT_EP_INTR(core_if, epnum, xfercompl);
                    doepint1.d32 = DWC_READ_REG32(&core_if->dev_if->out_ep_regs[0]->doepint);
                    if (pcd->ep0state == EP0_IDLE && doepint1.b.sr)
                    {
                        CLEAR_OUT_EP_INTR(core_if, epnum, sr);
                    }
                    else if(pcd->ep0state != EP0_IDLE)
                    {
                        handle_ep0(pcd);
                    }
                }
                else
                {
                    /* Clear the bit in DOEPINTn for this interrupt */
                    CLEAR_OUT_EP_INTR(core_if, epnum,xfercompl);
                    complete_ep(ep);
                }

            }

            /* Endpoint disable      */
            if (doepint.b.epdisabled)
            {
                /* Clear the bit in DOEPINTn for this interrupt */
                CLEAR_OUT_EP_INTR(core_if, epnum, epdisabled);
            }
            if (doepint.b.setup)
            {
                CLEAR_OUT_EP_INTR(core_if, epnum, setup);

                handle_ep0(pcd);

            }
            if (doepint.b.outtknepdis)
            {
                CLEAR_OUT_EP_INTR(core_if, epnum, outtknepdis);
            }

            /* NAK Interrutp */
            if (doepint.b.nak)
            {
                handle_out_ep_nak_intr(pcd, epnum);

                CLEAR_OUT_EP_INTR(core_if, epnum, nak);
            }

            /* NYET Interrutp */
            if (doepint.b.nyet)
            {
                handle_out_ep_nyet_intr(pcd, epnum);

                CLEAR_OUT_EP_INTR(core_if, epnum, nyet);
            }
        }

        epnum++;
        ep_intr >>= 1;
    }

    return 1;

#undef CLEAR_OUT_EP_INTR
}

/**
 * PCD interrupt handler.
 *
 * The PCD handles the device interrupts.  Many conditions can cause a
 * device interrupt. When an interrupt occurs, the device interrupt
 * service routine determines the cause of the interrupt and
 * dispatches handling to the appropriate function. These interrupt
 * handling functions are described below.
 *
 * All interrupt registers are processed from LSB to MSB.
 *
 */

extern int32_t dwc_otg_handle_usb_suspend_intr(dwc_otg_core_if_t * core_if);

int32_t dwc_otg_pcd_handle_intr(dwc_otg_pcd_t * pcd)
{
    dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);

    gintsts_data_t gintr_status;
    int32_t retval = 0;
        gintr_status.d32 = dwc_otg_read_core_intr(core_if);

        if (gintr_status.b.rxstsqlvl)
        {
            retval |=
                dwc_otg_pcd_handle_rx_status_q_level_intr(pcd);
        }
      //жϺϳһ
        if (gintr_status.b.usbsuspend)
        {
            retval |= dwc_otg_handle_usb_suspend_intr(pcd->core_if);
        }
        if (gintr_status.b.usbreset)
        {
            retval |= dwc_otg_pcd_handle_usb_reset_intr(pcd);
	       global.g_Connet =1;
        }
        if (gintr_status.b.enumdone)
        {
            retval |= dwc_otg_pcd_handle_enum_done_intr(pcd);
#if SYNC_USB_CTRL
            REG32(ARM_PORTA)=0x33;
            usdelay(1);
            REG32(ARM_PORTA)=0x66;
#endif
        }
        if (gintr_status.b.inepint)
        {
            if (!core_if->multiproc_int_enable)
            {
                retval |= dwc_otg_pcd_handle_in_ep_intr(pcd);
            }
        }
        if (gintr_status.b.outepintr)
        {
            if (!core_if->multiproc_int_enable)
            {
                retval |= dwc_otg_pcd_handle_out_ep_intr(pcd);
            }
        }
    return retval;
}

#endif /* DWC_HOST_ONLY */
