/**
* @file drvs_int.h 
* @brief Public APIs of int drivers
*
* Copyright (C) 2017 Sanechips Technology Co., Ltd.
* @author Dongdong Zhang <Zhang.Dongdong@sanechips.com.cn>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License version 2 as
* published by the Free Software Foundation. 
*
*/
#ifndef  _DRVS_INT_H
#define  _DRVS_INT_H

/*******************************************************************************
 *                           Include header files                              *
 ******************************************************************************/
#include "drvs_general.h"

#ifdef _OS_LINUX
#include <linux/irq.h>
#endif 

/*******************************************************************************
 *                             Macro definitions                               *
 ******************************************************************************/
#define INT_PRI_NUM                     15        /* 0-highest, 15-lowest, but 15 is always be masked in gic*/
                                                  /*so in fact, only 0-14 can generate interrupt*/

#define MAX_NEST_NUM                    32        /* maximum  number of  nested interrupt */


#define INT_TABLE_END                   0xFFFF
#define INT_VECTOR_DEFAULT              0xFFFF
#define INT_PRIORITY_DEFAULT            14
#define INT_INVALID_INTLINE             0xFFFFFFFF

#if 1//modify by xxx ndef _OS_LINUX

/*
 * 0-15:   SGI (software generated interrupts)
 * 16-31:  PPI (private peripheral interrupts)
 * 32+:    SPI (shared peripheral interrupts)
 */
#define GIC_PPI_START 16
#define GIC_SPI_START 32

#define RESERVED_INT 0xffff

/*
 * software generated interrupts
 */
#define SGI0_INT    (0)
#define SGI1_INT    (1)
#define SGI2_INT    (2)
#define SGI3_INT    (3)
#define SGI4_INT    (4)
#define SGI5_INT    (5)
#define SGI6_INT    (6)
#define SGI7_INT    (7)
#define SGI8_INT    (8)
#define SGI9_INT    (9)
#define SGI10_INT   (10)
#define SGI11_INT   (11)
#define SGI12_INT   (12)
#define SGI13_INT   (13)
#define SGI14_INT   (14)
#define SGI15_INT   (15)

/*
 * private peripheral interrupts
 */
#define GLOBAL_TIMER_INT       (27)
#define LEGACY_FIQ_INT         (28)
#define PRIVATE_TIMER_INT      (29)
#define PRIVATE_WDT_INT        (30)
#define LEGACY_IRQ_INT         (31)

/*
 * shared peripheral interrupts
 */
#define UART0_INT                   		(GIC_SPI_START+0)
#define UART1_INT                   		(GIC_SPI_START+1)
#define UART2_INT                   		(GIC_SPI_START+2)
#define SSP0_INT                    		(GIC_SPI_START+3)
#define PS_SYS_COUNTER						(GIC_SPI_START+4)
#define RTC_ALARM_INT               		(GIC_SPI_START+5)
#define RTC_TIMER_INT               		(GIC_SPI_START+6)
#define I2S0_INT                    		(GIC_SPI_START+7)
#define I2S1_INT                    		(GIC_SPI_START+8)
#define USIM1_INT                			(GIC_SPI_START+9)
#define I2C1_INT                    		(GIC_SPI_START+10)
#define PMIC_I2C_INT                		(GIC_SPI_START+11)
#define KEYPAD_INT                  		(GIC_SPI_START+12)
#define SD1_INT                     		(GIC_SPI_START+13)
#define WLAN_PRIORITY_POS_INT				(GIC_SPI_START+14)
#define WLAN_PRIORITY_NEG_INT				(GIC_SPI_START+15)
#define SD0_INT                     		(GIC_SPI_START+16)
#define ICP_PHY_INT                 		(GIC_SPI_START+17)
#define ICP_M0_INT                  		(GIC_SPI_START+18)
#define TIMER0_INT                  		(GIC_SPI_START+19) 	/*ps timer0*/
#define TIMER1_INT                  		(GIC_SPI_START+20)	/*ps timer1*/
#define PS_RM_TIMER_INT             		(GIC_SPI_START+21)	/*ps rm timer*/
#define GSMIF_RFSSCR_INT            		(GIC_SPI_START+22)
#define GSMIF_RFSSCT_INT            		(GIC_SPI_START+23)
#define GSMIF_GP0_INT               		(GIC_SPI_START+24)
#define GSMIF_T_INT                 		(GIC_SPI_START+25)
#define GSMIF_TL_INT                		(GIC_SPI_START+26)
#define GPRS_INT0                   		(GIC_SPI_START+27)
#define GPRS_INT1                   		(GIC_SPI_START+28)
#define DSP_INT0                    		(GIC_SPI_START+29)
#define DSP_INT1                    		(GIC_SPI_START+30)
#define DSP_INT2                    		(GIC_SPI_START+31)
#define DSP_INT3                    		(GIC_SPI_START+32)
#define DSP_INT4                    		(GIC_SPI_START+33)
#define DSP_INT6                    		(GIC_SPI_START+34)
#define DSP_INT7                    		(GIC_SPI_START+35)
#define GSM_LPM_INT0                		(GIC_SPI_START+36)
#define RCOUT1_INT                  		(GIC_SPI_START+37)
#define DMAC0_INT                   		(GIC_SPI_START+38)	/*ps dma int1 to ps*/
#define DMAC1_INT                   		(GIC_SPI_START+39) /*phy dma int1 to ps*/
#define NAND_INT                    		(GIC_SPI_START+40)
#define USB_INT                     		(GIC_SPI_START+41)
#define USB_POWERDWN_UP_INT         		(GIC_SPI_START+42)
#define USB_POWERDWN_DOWN_INT       		(GIC_SPI_START+43)
#define HSIC_INT                    		(GIC_SPI_START+44)
#define HSIC_POWERDWN_UP_INT        		(GIC_SPI_START+45)
#define HSIC_POWERDWN_DOWN_INT      		(GIC_SPI_START+46)

#define GSM_USIM_INT                		(GIC_SPI_START+48)

#define EX8IN1_INT                  		(GIC_SPI_START+51)   /*only used by int driver*/
#define EX0_INT                     		(GIC_SPI_START+52)
#define EX1_INT                     		(GIC_SPI_START+53)
#define EX2_INT                     		(GIC_SPI_START+54)
#define EX3_INT                     		(GIC_SPI_START+55)
#define EX4_INT                     		(GIC_SPI_START+56)
#define EX5_INT                     		(GIC_SPI_START+57)
#define EX6_INT                     		(GIC_SPI_START+58)
#define EX7_INT                     		(GIC_SPI_START+59)
#define SSC0_CONFLICT_INT           		(GIC_SPI_START+60)
#define TD_LPM4_INT                 		(GIC_SPI_START+61)
#define TD_FRM_INT                  		(GIC_SPI_START+62)
#define TD_FRM32K_INT               		(GIC_SPI_START+63)
#define LTE_LPM2_INT                		(GIC_SPI_START+64)
#define LTE_LPM4_INT                		(GIC_SPI_START+65)
#define LTE_LPM5_INT                		(GIC_SPI_START+66)
#define GSM_LPM_INT1                		(GIC_SPI_START+67)
#define LTE_TPU_INT                 		(GIC_SPI_START+68)
#define WD_LPM4_INT                 		(GIC_SPI_START+69)
#define WD_FRM_INT                  		(GIC_SPI_START+70)
#define EDCP_INT                    		(GIC_SPI_START+71)
#define SD1_DATA1_INT               		(GIC_SPI_START+72)

#if defined (_CHIP_ZX297520V3)
#define UART0_RXD_INT               		(GIC_SPI_START+73)
#elif defined (_CHIP_ZX297520V2)
#define UART0_CTS_INT               		(GIC_SPI_START+73)
#endif

#define SPIFC0_INT                  		(GIC_SPI_START+74)
#define TIMER2_INT     					(GIC_SPI_START+75)	/*ps timer2*/
#define PS_WDT_INT          				(GIC_SPI_START+76)
#define ICP_AP_INT         				(GIC_SPI_START+77)
#define SSP1_INT                    		(GIC_SPI_START+78)
#define SD0_DATA1_INT               		(GIC_SPI_START+79)
#define TDM_INT               				(GIC_SPI_START+80)
#define PHY_TIMER0_INT               		(GIC_SPI_START+81)
#define PHY_TIMER1_INT               		(GIC_SPI_START+82)
#define TD_MODEM_INT0						(GIC_SPI_START+83)
#define TD_MODEM_INT1						(GIC_SPI_START+84)
#define TD_MODEM_INT2						(GIC_SPI_START+85)
#define LTE_MODEM_INT0						(GIC_SPI_START+86)
#define LTE_MODEM_INT1						(GIC_SPI_START+87)
#define LTE_MODEM_INT2						(GIC_SPI_START+88)
#define WD_MODEM_INT0						(GIC_SPI_START+89)
#define WD_MODEM_INT1						(GIC_SPI_START+90)
#define WD_MODEM_INT2						(GIC_SPI_START+91)
#define TD_LPM_ZSP_EXT_INT                (GIC_SPI_START+92)
#define LTE_LPM1_INT                		(GIC_SPI_START+93)
#define WD_LPM3_INT                		(GIC_SPI_START+94)
#define EDCP_PHY_INT                    	(GIC_SPI_START+95)

#define GIC_INT_NUM                 		(EDCP_PHY_INT+1)

/*virtual external 8in1 interrupts*/
#define EX8_INT                     		(GIC_SPI_START+96)
#define EX9_INT                     		(GIC_SPI_START+97)
#define EX10_INT                    		(GIC_SPI_START+98)
#define EX11_INT                    		(GIC_SPI_START+99)
#define EX12_INT                    		(GIC_SPI_START+100)
#define EX13_INT                    		(GIC_SPI_START+101)
#define EX14_INT                    		(GIC_SPI_START+102)
#define EX15_INT                    		(GIC_SPI_START+103)

#define EX8IN1_INT_NUM              		(8)

#define INT_LINES_NUM               		(EX15_INT+1)

#if 0	/*7520V2 reserved*/
/*
 *virtual gpio interrupts
 *interrupt number should be defined by user
 *example:
 *    #define GPIO23_INT   (GPIO_INT0_START+23)
 *    #define GPIO130_INT  (GPIO_INT1_START+3)  or (GPIO_INT0_START+130)
 */

/*
 *gpio0   --- GPIO_INT0_START
 *gpio127 --- GPIO_INT0_END
 */
#define GPIO_INT0_START             (EX15_INT+1)
#define GPIO_INT0_END               (GPIO_INT0_START+127)

/*
 *gpio128 --- GPIO_INT1_START
 *gpio255 --- GPIO_INT1_END
 */
#define GPIO_INT1_START             (GPIO_INT0_END+1)
#define GPIO_INT1_END               (GPIO_INT1_START+127)


#define INT_LINES_NUM               (GPIO_INT1_END + 1) /*жϸ*/
#endif

#endif

/*******************************************************************************
*                             Type definitions                                *
******************************************************************************/
typedef struct _T_zDrvIntTable
{
	UINT32	uIntLine;				 /*the intline of the dev*/
	UINT32	uIntPri;				 /*hardware priority*/
	UINT32	vector; 				 /*OSE vector,should be convert to type OSVECTOR  */
	UINT32	level;					  /*CPU trigger level*/
}
T_zDrvIntTable;

#ifdef _OS_TOS
typedef enum _T_zDrvIntLineLevel
{
	INT_HIGHLEVEL = 0x00,			/* 00: high level */
	INT_LOWLEVEL  = 0x01,			/* 01: low level */
	INT_POSEDGE   = 0x02,			/* 10: raise edge */
	INT_NEGEDGE   = 0x03,			/* 11: fall edge */
	INT_DEFAULT_LEVEL,
}
T_zDrvIntLineLevel;
#endif

#ifdef _OS_LINUX
typedef enum _T_zDrvIntLineLevel
{
	INT_HIGHLEVEL = IRQ_TYPE_LEVEL_HIGH,		   /* 0x4: high level */
	INT_LOWLEVEL  = IRQ_TYPE_LEVEL_LOW, 		   /* 0x8: low level */
	INT_POSEDGE   = IRQ_TYPE_EDGE_RISING,		   /* 0x1: raise edge */
	INT_NEGEDGE   = IRQ_TYPE_EDGE_FALLING,		   /* 0x2: fall edge */
	INT_DUALEDGE  = IRQ_TYPE_EDGE_BOTH, 		   /* 0x3: fall and raise edge, it can only be applyed to GPIO int */
	INT_DEFAULT_LEVEL = 0xf,
}
T_zDrvIntLineLevel;
#endif

/*******************************************************************************
 *                       Global function declarations                          *
 ******************************************************************************/

/**
 *@brief		This function is used to install isr into tos
 *@param	line		interrupt line(0--INT_LINES_NUM)
 *@param	pEntry	the point to isr entry
 *@param	pName 	isr name
 *@param	level	int line trigger level
 *
 *@note		the level is source trigger level. if you don't know the int trigger level, you can set the level:INT_DEFAULT_LEVEL.
 *
 *@return 	id of interrupt handler process if successed, errcode otherwise.
 */
SINT32 zDrvInt_InstallIsr(UINT32 line, VOID *pEntry, const CHAR * pName,
                            T_zDrvIntLineLevel level);

/**
 *@brief		This function is used to uninstall isr
 *@param	line		interrupt line(0--INT_LINES_NUM)
 *
 *@return 	0 if successed, errcode otherwise.
 */
SINT32 zDrvInt_UninstallIsr(UINT32 line);

/**
 *@brief		This function is used to mask irq
 *@param	line		interrupt line(0--INT_LINES_NUM)
 *
 *@return 	0 if successed, errcode otherwise.
 */
VOID zDrvInt_MaskIrq(UINT32 line);

/**
 *@brief		This function is used to unmask irq
 *@param	line		interrupt line(0--INT_LINES_NUM)
 *
 *@return 	0 if successed, errcode otherwise.
 */	
VOID zDrvInt_UnmaskIrq(UINT32 line);

/**
 *@brief		This function is used to set int trigger level
 *@param	line		interrupt line(0--INT_LINES_NUM)
 *@param	level	int line trigger level
 *
 *@return 	0 if successed, errcode otherwise.
 */	
SINT32 zDrvInt_SetLineLevel(UINT32 line, T_zDrvIntLineLevel level);

/**
 *@brief		This function is used to clear int status
 *@param	line		interrupt line(0--INT_LINES_NUM)
 *
 *@return 	0 if successed, errcode otherwise.
 */
SINT32 zDrvInt_ClearInt(UINT32 line);

/**
 *@brief		This function is used to set int priority
 *@param	line		interrupt line(0--INT_LINES_NUM)
 *@param	pri		priority number
 *
 *@return 	0 if successed, errcode otherwise.
 */
VOID zDrvInt_SetLinePri(UINT32 line, UINT32 pri);

#ifdef _OS_LINUX
/**
 *@brief		This function is used to install isr into linux
 *@param	line		interrupt line(0--INT_LINES_NUM)
 *@param	pEntry	the point to isr entry
 *@param	pName	isr name
 *@param	level	int line trigger level
 *
 *@note		the level is source trigger level. if you don't know the int trigger level, you can set the level:INT_DEFAULT_LEVEL.
 *
 *@return 	id of interrupt handler process if successed, errcode otherwise.
 */
SINT32 zDrvInt_InstallFastIsr( UINT32 uiLine, VOID *pEntry, const CHAR * pName,
                           T_zDrvIntLineLevel level);

/**
 *@brief		This function is used to install nested isr for interrupt line uiLine.
 *@param	line		interrupt line(0--INT_LINES_NUM)
 *@param	pEntry	the point to isr entry
 *@param	pName	isr name
 *
 *@return 	id of interrupt handler process if successed, errcode otherwise.
 */
SINT32 zDrvInt_InstallNestedIsr( UINT32 uiLine, VOID *pEntry, const CHAR * pName);

#endif

#ifdef _OS_TOS
/**
 *@brief		This function is used to install isr into tos
 *@param	line		interrupt line(0--INT_LINES_NUM)
 *@param	pEntry	the point to isr entry
 *@param	pName 	isr name
 *@param	level	int line trigger level
 *
 *@note		the level is source trigger level. if you don't know the int trigger level, you can set the level:INT_DEFAULT_LEVEL.
 *
 *@return 	id of interrupt handler process if successed, errcode otherwise.
 */
SINT32 zDrvInt_InstallIsrDsr(UINT32 line, VOID *pIsrEntry,
                                VOID *pDsrEntry, T_zDrvIntLineLevel level);

/**
 *@brief		This function is used to distinguish whether this interrupt need to delay umask or not
 *@param	line		interrupt line(0--INT_LINES_NUM)
 *
 *@note		invoked by os.
 *
 *@return 	0 if need to delay umask; 1 otherwise.
 */
SINT32 zDrvInt_DelayUnmask(UINT32 intLine);

/**
 *@brief		This function is used to initiate interrupt controller, can only be used when system bringing up
 *
 *@param	none.
 *
 *@return 	none.
 */
VOID zDrvInt_Initiate(VOID);

/**
 *@brief		This function is used to delay unmask an interrupt when isr is completed.
 *@param	line		interrupt line(0--INT_LINES_NUM)
 *
 *@note		it must be used before installing irq, and you should unmask this int manually after isr is completed.
 *
 *@return 	none.
 */
VOID zDrvInt_DelayUnmaskAdd(UINT32 intLine);

/**
 *@brief		This function is used to remove the delay unmask function for an interrupt.
 *@param	line		interrupt line(0--INT_LINES_NUM)
 *
 *@return 	none.
 */	
VOID zDrvInt_DelayUnmaskRemove(UINT32 intLine);

#endif

#endif    /* DRVS_INT_H */
