/*******************************************************************************
 * Copyright (C) 2007, ZTE Corporation.
 *
 * File Name:    hal_Audio.c
 * File Mark:
 * Description:  This file contains the hal layer routines for Audio driver.
 * Others:
 * Version:       V0.5
 * Author:        WangJun
 * Date:          2008-12-12
 * History 1:
 *     Date:
 *     Version:
 *     Author:
 *     Modification:
 * History 2:
  ********************************************************************************/

/****************************************************************************
*                                             Include files
****************************************************************************/
#include "drvs_general.h"
#include "aud_nv.h"
#include "Audio_api.h"
#include "hal_Audio.h"
#include "drvs_i2s.h"
#include "oss_api.h"
#include "drvs_gpio.h"
#include "drvs_pwr.h"
#include "drvs_pow.h"

#ifdef _USE_HAL_HRDTST
#include "hal_hrdtst_pcm.h"
#endif

#include "drvs_voiceprocess.h"
#include "drvs_audiomanager.h"
#include "drvs_codec.h"

/****************************************************************************
*                                             Local Types
****************************************************************************/
//#define ARM_I2S_LOOP_CFG	*((volatile UINT32 *)(0x00140000+0x60))   

typedef enum
{
    AUDIO_IDLE = 0,
    AUDIO_OPEN,
    AUDIO_INUSE,
    AUDIO_PAUSE
}
T_Audio_Status;

typedef enum
{
    AUDIO_ARM_I2S_MASTER = 0,
    AUDIO_ARM_I2S_SLAVE

}
T_Audio_ARM_I2S_Mode;

/****************************************************************************
*                                             Local Constants
****************************************************************************/
static T_Audio_Status  s_audioRecordStatus = AUDIO_IDLE;
static T_Audio_Status  s_audioPlayStatus = AUDIO_IDLE;
static T_ZDrv_VpPath  s_outputPath = MAX_VP_PATH;
static T_ZDrvAudio_OutputVolLevel  s_outputVol = MAX_AUDIO_OUTPUT_VOL_LEVEL;

#define AUDIO_PLAYBUFFER_MAX  8196
static UINT16 s_playBuffer[AUDIO_PLAYBUFFER_MAX] = {0};
static T_ZDrvAudio_SampleRate s_audioSample = AUDIO_RATE_8_KHZ;

static ZOSS_SEMAPHORE_ID s_audioPlayEndCallbackSem= NULL;
static ZOSS_THREAD_ID s_audioPlayEndCallbackThread= NULL;
static T_ZDrvAudio_CallbackFunc  s_audioCallApplication = NULL;
static BOOL s_isMute = FALSE;
static BOOL s_isOpenCall = FALSE;
static T_ZDrvI2S_Cfg s_playI2sCfg = {TRUE, I2S_DATA_16BIT, I2S_TS_16CYCLE, 16000,     \
                                        I2S_TIME_MODE,TRANS_NEGATIVE_POSITIVE,\
                                        LEFT_TRACK,I2S_NORMAL_MODE, FIRST_DATA_ALIGN_TO_SECOND_CYCLE, \
                                        PCM_NORMAL_MODE,PCM_MSB_FIRST, PCM_FSYNC_LENGTH_1CLK,PCM_1TIME_SLOT,TIMING_I2S_MONO_LEFT,PCM_SHORT_FSYNC, \
                                        TDM_NO_EXTRA_CYCLE,DATA_ALIGN_STD_I2S,REF_CLK26M
                                       };//TEST FOR i2s  

static T_ZDrvI2S_Cfg s_recordI2sCfg = {TRUE, I2S_DATA_16BIT, I2S_TS_16CYCLE, 16000,     \
                                        I2S_TIME_MODE,TRANS_NEGATIVE_POSITIVE,\
                                        LEFT_TRACK,I2S_NORMAL_MODE, FIRST_DATA_ALIGN_TO_SECOND_CYCLE, \
                                        PCM_NORMAL_MODE,PCM_MSB_FIRST, PCM_FSYNC_LENGTH_1CLK,PCM_1TIME_SLOT,TIMING_I2S_MONO_LEFT,PCM_SHORT_FSYNC, \
                                        TDM_NO_EXTRA_CYCLE,DATA_ALIGN_STD_I2S,REF_CLK26M
                                       };//TEST FOR i2s  

static T_ZDrvAudio_InputVolLevel * s_recordVol = NULL ;
//static T_Audio_ARM_I2S_Mode  s_audArmI2SMode = AUDIO_ARM_I2S_MASTER;

#define REG_I2S0_CLK_DIV 0x01400018
#define REG_I2S1_CLK_DIV 0x01400020

static T_ZDrvI2S_Id I2S_ID = I2S_1;

/****************************************************************************
*                                             Local Function Prototypes
****************************************************************************/
static SINT32 audio_Open(VOID);
static SINT32 audio_Close(VOID);
static SINT32 audio_PlayStart(T_ZDrvAudio_PlayParam * datainfo);
static SINT32 audio_PlayStop(VOID);
static SINT32 audio_PlayPause(VOID);
static SINT32 audio_PlayResume(VOID);
static SINT32 audio_SetPlaySample(T_ZDrvAudio_SampleRate * pSampleRate);
//static SINT32 audio_SetBalance(T_ZDrvAudio_Banlance * pBanlance);//xiu
//static SINT32 audio_SetEQMODE(T_ZDrvAUDIO_EQMODE * pEQMODE);
//static SINT32 audio_Set3DEFFECT(T_ZDrvAudio_3DEFFECT * p3DEFFECT);
static SINT32 audio_SetOutputPath(T_ZDrv_AudioOutputPath * pOutputPath);
static SINT32 audio_SetOutputVol(T_ZDrvAudio_OutputVolLevel * pOutputVol);
static SINT32 audio_SetMute(BOOL * BlMute);
//static SINT32 audio_GetPlayInfo(T_ZDrvAudio_PlayInfo * pPlayInfo);
static SINT32 audio_SetCallBackFunc(T_ZDrvAudio_CallbackFunc pCallBcak);
static SINT32 audio_Write(UINT32* pDatabuff, UINT32 length);
static SINT32 audio_GetBuf(T_ZDrvAudio_BufInfo * pBufInfo);
static SINT32 audio_RecordStart(T_ZDrvRcd_InfoParam * rcdParam);
static SINT32 audio_RecordStop(VOID);
static SINT32 audio_RecordDataRead(T_ZDrvAudio_BufInfo * pBufInfo);
static SINT32 audio_FreeRecordBuf(VOID * pBuf);
//static SINT32 audio_Enable3DEffect(T_ZDrvAudio_3DEFFECT * pAudio3DEffect);
static SINT32 audio_SetRecordPath(T_ZDrv_CodecInputPath * param);
static SINT32 audio_SetRecordVol(T_ZDrvAudio_InputVolLevel * param);
static SINT32 audio_SetRecordSample(T_ZDrvAudio_SampleRate * pSampleRate);
//static SINT32 audio_SetPmicRecVol(T_ZDrvAudio_InputVolLevel * InputVol);//xiu
SINT32 halVpCfg_SetInputVol(T_ZDrv_VpVol vol);//xiu
extern SINT32 halVpCfg_Open(T_ZDrvVp_SourceType srctype);
extern SINT32 halVpCfg_Close(VOID);
extern SINT32 halVpCfg_Enable(VOID);
extern SINT32 halVpCfg_Disable(VOID);
extern SINT32 halVpCfg_SetFs(T_ZDrv_VpFs fs);
extern SINT32 halVpCfg_SetInputPath(T_ZDrv_VpPath path,BOOL enable);
extern SINT32 halVpCfg_SetOutputPath(T_ZDrv_VpPath path,BOOL enable);
extern SINT32 halVpCfg_SetMute(T_ZDrv_VpPath path,BOOL mute);
extern SINT32 halVpCfg_SetOutputVol(T_ZDrv_VpVol vol);
extern SINT32 halVpCfg_SetUseProtocol(T_ZDrvCodec_UseProtocol usePortocol);

/****************************************************************************
*                                            Global Variables
****************************************************************************/
static T_HalAudio_Opt gHalAudio_ObjOpt =
{
    audio_Open, //:1
    audio_Close, //:2
    audio_PlayStart, //:3
    audio_PlayStop, //:4
    audio_PlayPause, //:5
    audio_PlayResume, //:6
    audio_SetPlaySample, //:7
    NULL, //:8
    NULL, //:9
    NULL, //:10
    audio_SetOutputPath, //:11
    audio_SetOutputVol, //:12
    audio_SetMute, //:13
    NULL, //:14
    audio_SetCallBackFunc, //:15
    audio_Write, //:16
    NULL, //:17
    audio_SetRecordPath, //:18
    audio_SetRecordVol, //:19
    audio_GetBuf, //:20
    audio_RecordStart, //:21
    audio_RecordStop, //:22
    audio_RecordDataRead, //:23
    audio_FreeRecordBuf, //:24
    audio_SetRecordSample, //:25
    NULL, //:26
    NULL //:27
};

#ifdef _USE_HAL_HRDTST
static T_HalHrdTst_Pcm_Opt gHalHrdTst_Pcm_Opt=
{
    audio_Open,
    audio_Close,
    audio_PlayStart,
    audio_PlayStop,
    audio_PlayPause,
    audio_PlayResume,
    audio_SetPlaySample,
    audio_SetOutputPath,
    audio_SetOutputVol,
    audio_Write,
    audio_GetBuf,
    audio_SetCallBackFunc
};  /*added by dangpeixia for hardware test[2010/9/15]*/
#endif

/****************************************************************************
*                                            Global Function Prototypes
****************************************************************************/
#if defined _USE_TEAKAUD
extern aud_eep_static_type        aud_EEP_static;
#endif

/****************************************************************************
*                                            Function Definitions
****************************************************************************/

/*******************************************************************************
 * Function: audio_CallbackThread
 * Description: This function is running while I2S paly the audio data ended.
 * Parameters:
 *   Input:param reserved
 *   Output:None
 *
 * Returns: None
 *
 * Others:
 ********************************************************************************/
static VOID audio_CallbackThread(SINT32  param)
{
    while (1)
    {
        if (zOss_GetSemaphore(s_audioPlayEndCallbackSem, ZOSS_WAIT_FOREVER) == ZOSS_SUCCESS)
        {
            if (NULL!=s_audioCallApplication)
            {
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_CallbackThread !\n");
                (*s_audioCallApplication)(AUDIO_INFO_PLAY_END);
            }
        }
    }
}

/*******************************************************************************
 * Function: audio_PutSemID
 * Description: This function is used by i2s to release SemID.
 * Parameters:
 *   Input:None
 *   Output:None
 *
 * Returns: None
 *
 * Others:
 ********************************************************************************/
static void audio_PutSemID(VOID)
{
    zOss_PutSemaphore(s_audioPlayEndCallbackSem);
}

/*******************************************************************************
 * Function: audio_CreateSemAndThread
 * Description: This function is used to create SemID and the thread which waiting the SemID.
 * Parameters:
 *   Input:None
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *
 * Others:
 ********************************************************************************/
static SINT32  audio_CreateSemAndThread(VOID)
{
    s_audioPlayEndCallbackSem = zOss_CreateSemaphore("AUD_CALLBACK_SEMPID", 0);
    if (s_audioPlayEndCallbackSem == NULL)
    {
        return DRV_ERROR;
    }

    s_audioPlayEndCallbackThread = zOss_CreateThread("AUD_CALLBACK_THREAD", audio_CallbackThread, 0,1024,14,1,1);
    if (NULL == s_audioPlayEndCallbackThread)
    {
        if (NULL != s_audioPlayEndCallbackSem)
        {
            zOss_DeleteSemaphore(s_audioPlayEndCallbackSem);
            s_audioPlayEndCallbackSem = NULL;
        }
        return DRV_ERROR;
    }
	
    return DRV_SUCCESS;
}

/*******************************************************************************
 * Function: audio_RemoveSemAndThread
 * Description: This function is used to delete SemID and the thread which waiting the SemID.
 * Parameters:
 *   Input:None
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *
 * Others:
 ********************************************************************************/
static UINT32  audio_RemoveSemAndThread(VOID)
{
    UINT32 returnValue = ZOSS_SUCCESS ;
    if (NULL != s_audioPlayEndCallbackThread)
    {
        returnValue += zOss_DeleteThread(s_audioPlayEndCallbackThread);
        s_audioPlayEndCallbackThread = NULL;
    }

    if (s_audioPlayEndCallbackSem != NULL)
    {
        returnValue += zOss_DeleteSemaphore(s_audioPlayEndCallbackSem);
        s_audioPlayEndCallbackSem = NULL;
    }

    return returnValue;
}

/*------------------------------------------------------------------------------
* Function...: audio_i2s_clk_release
* Return.....: -
* Description: close i2s and pmic clock resource
* Parameters:
*   resource: aud source
*   isCheckStatus: only check the pmic status or config the pmic status
* Return:
*   none
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static void audio_i2s_clk_release(VOID)
{
    //UINT32  AmrRegBit;//xiu
#if 0
#if  defined _USE_EVB2960_02B
#ifndef _USE_EVB2963
	    reg32(GPIO_PMM_ZSPPORTA)&= 0xFFFFFF7F;
#endif
	    reg32(ARM_CLK_SELECT_1)|=  0xF0;
	    AmrRegBit = I2S_CLKOUT_SEL;
	    AmrRegBit &= 0xFFBF0FFF;
	    I2S_CLKOUT_SEL = AmrRegBit;
#endif
	    reg32(REG_I2S0_CLK_DIV) = 0x02000000;
	    reg32(REG_I2S1_CLK_DIV) = 0x02000000;
#endif

}

/*******************************************************************************
 * Function: audio_Open
 * Description: This function is used to init pmic and i2s.
 * Parameters:
 *   Input:None
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_Open(VOID)
{
    SINT32 ret = DRV_SUCCESS;
	
    ret = zDrvAudioM_GetResource(AUDIO_MANAGER_RESOURCE_AUDIO);
    if(ret != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, " audio_Open fail to get audio resource\n");
        return ret;
    }
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"ref audio_Open zDrvI2S_Open !\n");
	
    ret = zDrvI2S_Open(I2S_ID,TRANS_DMA_MODE);
    if (ret != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_Open zDrvI2S_Open error! ret=%d\n",ret);
        zDrvAudioM_FreeResource(AUDIO_MANAGER_RESOURCE_AUDIO);
        return ret;
    }

    ret = audio_CreateSemAndThread();
    if (ret != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_Open audio_CreateSemAndThread error! ret=%d\n",ret);
        ret = zDrvI2S_Close(I2S_ID);
        if (ret != DRV_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "audio_Open zDrvI2S_Close err I2S_ID=%d,ret=%d\n",I2S_ID,ret);
        }

        zDrvAudioM_FreeResource(AUDIO_MANAGER_RESOURCE_AUDIO);

        return DRV_ERROR;
    }
	
    s_isMute = FALSE ;
    s_isOpenCall = TRUE;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_Open end!\n");
    zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL,"audio_Open end!\n");

    return DRV_SUCCESS;
	
}

/*******************************************************************************
 * Function: audio_Close
 * Description: This function is used to close the pmic and i2s.
 * Parameters:
 *   Input:None
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_Close(VOID)
{
    SINT32  ret = DRV_SUCCESS, ret2 = DRV_SUCCESS;
    UINT8 statusValue = (UINT8)s_audioPlayStatus+(UINT8)s_audioRecordStatus;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_Close! s_audioPlayStatus:%d s_audioRecordStatus:%d\n", s_audioPlayStatus, s_audioRecordStatus);
	
    if (s_isOpenCall == TRUE)
    {
        if (audio_RemoveSemAndThread() != ZOSS_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_RemoveSemAndThread error!\n");
            //return DRV_ERROR;
        }
        ret = zDrvI2S_Close(I2S_ID);
        if (ret != DRV_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"zDrvI2S_Close error ret=%d!\n",ret);
            //return DRV_ERROR;
        }

        ret2 = halVpCfg_Close();
        if (ret2 != DRV_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"halVpCfg_Close error ret2=%d!\n",ret2);
			ret += ret2;
			//return DRV_ERROR;
        }

        audio_i2s_clk_release();
        s_isOpenCall = FALSE;
        s_audioPlayStatus = AUDIO_IDLE;
        s_audioRecordStatus = AUDIO_IDLE;
        zDrvAudioM_FreeResource(AUDIO_MANAGER_RESOURCE_AUDIO);
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_Close end!\n");
        zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL,"audio_Close end!\n");


        return ret;
    }
    else
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_Close time error! s_isOpenCall:%d, statusValue:%d \n",s_isOpenCall,statusValue);
        return DRV_ERROR;
    }
}

/*******************************************************************************
 * Function: audio_PlayStart
 * Description: This function is used to start audio playing.
 * Parameters:
 *   Input:datainfo include buffer size setting
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_PlayStart(T_ZDrvAudio_PlayParam *datainfo)
{
    SINT32  ret = DRV_SUCCESS;
    T_ZDrvI2s_Params parameter = {0};
    T_ZDrvCodec_UseProtocol usePortocol;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_PlayStart start!\n");
    if ((NULL == datainfo) || (datainfo->buffersize == 0))
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_PlayStart NULL err datainfo=%d!\n",datainfo);
        return DRV_ERR_INVALID_PARAM;
    }
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_PlayStart datainfo channel=%d,buffersize=%d!\n",datainfo->channel,datainfo->buffersize);

    if (s_audioPlayStatus == AUDIO_OPEN)
    {
        if (s_playI2sCfg.bMaster == FALSE)
        {
            usePortocol.masterSlave = I2S_MASTER_MODE;
        }
        else if (s_playI2sCfg.bMaster == TRUE)
        {
            usePortocol.masterSlave = I2S_SLAVE_MODE;
        }

        if(I2S_TIME_MODE == s_playI2sCfg.tClkMode)
        {
            s_playI2sCfg.tTransMode = DATA_ALIGN_STD_I2S;
            s_playI2sCfg.tTansmit_edge = TRANS_NEGATIVE_POSITIVE;//for i2s mode;
            usePortocol.busMode = CODEC_BUS_I2S;

         // usePortocol.dataLength = CODEC_DATA_LEGTH_16BIT;
        }
        else if(PCM_TIME_MODE == s_playI2sCfg.tClkMode)
        {
            s_playI2sCfg.tTransMode = DATA_ALIGN_STD_I2S; //industry standard short frame pcm timing

            s_playI2sCfg.tTansmit_edge = TRANS_POSITIVE_NEGATIVE;//for pcm mode;
			usePortocol.busMode = CODEC_BUS_DSP;

         // usePortocol.dataLength = CODEC_DATA_LEGTH_16BIT;
        }

        if(datainfo->channel == AUDIO_DUAL_CHANNEL)
        {
            s_playI2sCfg.tTrackFmt = DOUBLE_TRACK;
            s_playI2sCfg.tTimingType = TIMING_I2S_ST_2CHN_1LANE;
            usePortocol.dataTrack = CODEC_DATA_TRACK_DOUBLE;
        }
        else
        {
            s_playI2sCfg.tTrackFmt = LEFT_TRACK;//RIGHT_TRACK;//LEFT_TRACK;DOUBLE_TRACK
            s_playI2sCfg.tTimingType = TIMING_I2S_MONO_LEFT;
            usePortocol.dataTrack = CODEC_DATA_TRACK_LEFT;
        }

        ret = halVpCfg_SetUseProtocol(usePortocol);
        if (ret != DRV_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_PlayStart  halVpCfg_SetUseProtocol end! ret=%d\n",ret);
            return DRV_ERROR;
        }

        parameter.channel = datainfo->channel;
        parameter.buffersize = datainfo->buffersize;
        parameter.p_cb = audio_PutSemID;

        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_PlayStart zDrvI2S_Write_Start start!\n");
        ret = zDrvI2S_Write_Start(I2S_ID, &parameter, &s_playI2sCfg);
        if (ret != DRV_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_PlayStart zDrvI2S_Write_Start err =%d\n",ret);
            return DRV_ERROR;
        }

        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_PlayStart halVpCfg_Enable start!\n");
        ret =halVpCfg_Enable();
        if (ret != DRV_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_PlayStart  halVpCfg_Enable err =%d\n",ret);
            zDrvI2S_Write_Stop(I2S_ID);
            return DRV_ERROR;
        }

        s_audioPlayStatus = AUDIO_INUSE;
        s_audioRecordStatus = AUDIO_IDLE;
		
    }
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_PlayStart end!\n");

    return ret;
	
}

/*******************************************************************************
 * Function: audio_PlayStop
 * Description: This function is used to stop the audio playing .
 * Parameters:
 *   Input:None
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_PlayStop(VOID)
{
    SINT32  ret = DRV_SUCCESS, ret2 = DRV_SUCCESS;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_PlayStop start!\n");
    zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL,"audio_PlayStop start!\n");
	
    if (s_audioPlayStatus == AUDIO_INUSE ||s_audioPlayStatus == AUDIO_PAUSE)
    {
        /* close the audio */
        ret = zDrvI2S_Write_Stop(I2S_ID);
        zDrvI2s_RxRlsSemaBeforeStop(I2S_1);
        zDrvI2s_TxRlsSemaBeforeStop(I2S_1);
        if (ret != DRV_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_PlayStop zDrvI2S_Write_Stop err =%d!\n",ret);
           // return DRV_ERROR;
        }
#if defined _USE_FWP
        halgpio_Set_Config(AUDIO_PA_EN_PIN, SET_GPIO);
        halgpio_Set_Value(AUDIO_PA_EN_PIN, GPIO_STATE_LOW);
        halgpio_Set_Direction(AUDIO_PA_EN_PIN, GPIO_OUT);
#endif
        ret2 = halVpCfg_Disable();
        if (ret2 != DRV_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_PlayStop halVpCfg_Disable err =%d!\n",ret2);
			ret += ret2;
			//  return DRV_ERROR;			
        }
        s_audioPlayStatus = AUDIO_OPEN;
    }
	
    zOss_Printf(SUBMDL_HAL, PRINT_LEVEL_NORMAL,"audio_PlayStop end!\n");
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_PlayStop end!\n");

    return ret;
	
}

/*******************************************************************************
 * Function: audio_PlayPause
 * Description: This function is used to pause the audio playing.
 * Parameters:
 *   Input:None
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_PlayPause(VOID)
{
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_PlayPause!\n");
    if (s_audioPlayStatus == AUDIO_INUSE)
    {
        zDrvI2S_Pause(I2S_ID);

        s_audioPlayStatus = AUDIO_PAUSE;
    }
	
    return DRV_SUCCESS;
}

/*******************************************************************************
 * Function: audio_PlayResume
 * Description: This function is used to resume  the audio which is paused.
 * Parameters:
 *   Input:None
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_PlayResume(VOID)
{
    if (s_audioPlayStatus == AUDIO_PAUSE)
    {
        s_audioPlayStatus = AUDIO_INUSE;

        zDrvI2S_Resume(I2S_ID);
    }
	
    return DRV_SUCCESS;
}

/*******************************************************************************
 * Function: audio_OutputPmicInit
 * Description: This function is used to pmic while play audio
 * Parameters:
 *   Input:None
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_OutputPmicInit(VOID)//xiu
{
    SINT32  ret = DRV_SUCCESS;
    //BOOL OnOff = TRUE;//xiu
	
    if(s_audioPlayStatus==AUDIO_OPEN)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_OutputPmicInit has opened\n");
        return DRV_SUCCESS;
    }
	
    if (s_audioRecordStatus != AUDIO_IDLE)
    {
        return DRV_ERR_BUSY;
    }
	
    if (s_audioPlayStatus == AUDIO_IDLE)
    {

        ret = halVpCfg_Open(VP_AUDIO_SOURCE);
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_OutputPmicInit zDrvCodec_Open ret=%d !\n",ret);

        if (ret != DRV_SUCCESS)
        {
            halVpCfg_Disable();
            return ret;
        }

        s_audioPlayStatus = AUDIO_OPEN;
    }
	
    return DRV_SUCCESS;

}

/*******************************************************************************
 * Function: audio_SetPlaySample
 * Description: This function is used to set the audio play sample rate.
 * Parameters:
 *   Input:pSampleRate sample rate
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_SetPlaySample(T_ZDrvAudio_SampleRate * pSampleRate)
{
    SINT32  ret = DRV_SUCCESS;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetPlaySample\n");
    T_ZDrv_VpFs stdac_sample = VP_FS_8000 ;
	
    if (pSampleRate == NULL)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetPlaySample pSampleRate == NULL\n");
        return DRV_ERR_INVALID_PARAM;
    }
	
    if (audio_OutputPmicInit() != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetPlaySample audio_OutputPmicInit err\n");
        return DRV_ERROR;
    }
	
    if (s_audioPlayStatus == AUDIO_OPEN)
    {
        switch (*pSampleRate)
        {
        case AUDIO_RATE_8_KHZ:
            stdac_sample = VP_FS_8000;
            s_audioSample = AUDIO_RATE_8_KHZ;
            s_playI2sCfg.sample_rate = 8000;
            break;
        case AUDIO_RATE_11_025_KHZ:                        /*AUDIO_RATE_11_025_KHZ * 4 = 44.1K */
            stdac_sample = VP_FS_44100;
            s_audioSample = AUDIO_RATE_11_025_KHZ;
            s_playI2sCfg.sample_rate = 44100;
            break;
        case AUDIO_RATE_12_KHZ:                              /*AUDIO_RATE_12_KHZ * 4 = 48K */
            stdac_sample = VP_FS_48000;
            s_audioSample = AUDIO_RATE_12_KHZ;
            s_playI2sCfg.sample_rate = 48000;
            break;
        case AUDIO_RATE_16_KHZ:
            stdac_sample = VP_FS_16000;
            s_audioSample = AUDIO_RATE_16_KHZ;
            s_playI2sCfg.sample_rate = 16000;
            break;
        case AUDIO_RATE_22_050_KHZ:                      /*AUDIO_RATE_22_050_KHZ * 2 = 44.1K */
            stdac_sample = VP_FS_44100;
            s_audioSample = AUDIO_RATE_22_050_KHZ;
            s_playI2sCfg.sample_rate = 44100;
            break;
        case AUDIO_RATE_24_KHZ:                             /*AUDIO_RATE_24_KHZ * 2 = 48K */
            stdac_sample = VP_FS_48000;
            s_audioSample = AUDIO_RATE_24_KHZ;
            s_playI2sCfg.sample_rate = 48000;
            break;
        case AUDIO_RATE_32_KHZ:                           /*STDAC_RATE_32_KHZ / 2 = 16K */
            stdac_sample = VP_FS_32000;
            s_audioSample = AUDIO_RATE_32_KHZ;
            s_playI2sCfg.sample_rate = 32000;
            break;
        case AUDIO_RATE_44_1_KHZ:
            stdac_sample = VP_FS_44100 ;
            s_audioSample = AUDIO_RATE_44_1_KHZ;
            s_playI2sCfg.sample_rate = 44100;
            break;
        case AUDIO_RATE_48_KHZ:
            stdac_sample = VP_FS_48000;
            s_audioSample = AUDIO_RATE_48_KHZ;
            s_playI2sCfg.sample_rate = 48000;
            break;
        case AUDIO_RATE_64_KHZ:                           /*AUDIO_RATE_64_KHZ / 4 = 16K */
            stdac_sample = VP_FS_16000;
            s_audioSample = AUDIO_RATE_64_KHZ;
            s_playI2sCfg.sample_rate = 16000;
            break;
        case AUDIO_RATE_96_KHZ:                           /*AUDIO_RATE_96_KHZ / 2 = 48K */
            stdac_sample = VP_FS_48000;
            s_audioSample = AUDIO_RATE_96_KHZ;
            s_playI2sCfg.sample_rate = 48000;
            break;
        default:

            stdac_sample = VP_FS_44100;
            s_audioSample = AUDIO_RATE_44_1_KHZ;
            s_playI2sCfg.sample_rate = 44100;
            break;
        }
        ret = halVpCfg_SetFs ( stdac_sample);

        if (ret != DRV_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetPlaySample halVpCfg_SetFs err =%d\n",ret);
            return DRV_ERROR;
        }
    }
    else
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetPlaySample s_audioPlayStatus != AUDIO_OPEN");
        return DRV_ERROR;
    }
	
    return ret;
	
}
#if 0//xiu
/*******************************************************************************
 * Function: audio_SetBalance
 * Description: This function is used to set the Balance of audio .
 * Parameters:
 *   Input:pbanlance:balance param
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_SetBalance(T_ZDrvAudio_Banlance * pBanlance)
{
    SINT32  ret = DRV_SUCCESS;
    SINT8 leftRightDel = 0 ;

    if (pBanlance == NULL)
    {
        return DRV_ERR_INVALID_PARAM;
    }

#ifdef _USE_CODEC_MC13787
    leftRightDel = (SINT8)pBanlance->balanceLeft - (SINT8)pBanlance->balanceright;
    if (leftRightDel >= 0)
    {
        pBanlance->balanceLeft = AUDIO_STEREO_BALANCE_LEVEL_7 ;
        pBanlance->balanceright = (T_ZDrvAudio_BalanceLevel)(AUDIO_STEREO_BALANCE_LEVEL_7 - leftRightDel);
    }
    else
    {
        pBanlance->balanceright = AUDIO_STEREO_BALANCE_LEVEL_7 ;
        pBanlance->balanceLeft = (T_ZDrvAudio_BalanceLevel)(AUDIO_STEREO_BALANCE_LEVEL_7 + leftRightDel);
    }
    if (audio_OutputPmicInit() != DRV_SUCCESS)
    {
        return DRV_ERROR;
    }
    ret = halPmic_AudioOutputSetBalance(s_audioPlayHandler,
                                        (PMIC_AUDIO_OUTPUT_BALANCE_GAIN)pBanlance->balanceLeft,
                                        (PMIC_AUDIO_OUTPUT_BALANCE_GAIN)pBanlance->balanceright);
    if (ret != DRV_SUCCESS)
    {
        return DRV_ERROR;
    }
#endif
    if( leftRightDel );//xiu  warning:leftRightDel unused
	ret = DRV_SUCCESS;//xiu	warning:ret unused
    return ret;//xiu	

}

/*******************************************************************************
 * Function: audio_SetEQMODE
 * Description: This function is used to set the audio instance of hal layer.Now this function is not supported.
 * Parameters:
 *   Input:pEQMODE :eq mode param
 *   Output:None
 *
 * Returns:
 *              DRV_ERR_NOT_SUPPORTED: not support this function
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_SetEQMODE(T_ZDrvAUDIO_EQMODE * pEQMODE)
{
    return DRV_ERR_NOT_SUPPORTED;
}

/*******************************************************************************
 * Function: audio_Set3DEFFECT
 * Description: This function is used to set the audio instance of hal layer.Now this function is not supported.
 * Parameters:
 *   Input:p3DEFFECT :3d effect param
 *   Output:None
 *
 * Returns:
 *              DRV_ERR_NOT_SUPPORTED: not support this function
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_Set3DEFFECT(T_ZDrvAudio_3DEFFECT * p3DEFFECT)
{
    return DRV_ERR_NOT_SUPPORTED;
}
#endif
/*******************************************************************************
 * Function: audio_SetPmicOutPutPath
 * Description: This function is used to set the path between pmic and output equipment
 * Parameters:
 *   Input:pOutputPath out put path
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_SetPmicOutputPath(T_ZDrv_AudioOutputPath * pOutputPath)
{
    SINT32  ret = DRV_SUCCESS;
    //UINT32  regValue = 0 ;//xiu
	
    /* ֲŵͨ*/
    if (*pOutputPath == CODEC_OUTPUT_RECEIVER)
    {
#if defined _USE_FWP
        halgpio_Set_Config(AUDIO_PA_EN_PIN, SET_GPIO);
        halgpio_Set_Value(AUDIO_PA_EN_PIN, GPIO_STATE_LOW);
        halgpio_Set_Direction(AUDIO_PA_EN_PIN, GPIO_OUT);
#endif


        ret =halVpCfg_SetOutputPath(VP_PATH_HANDSET,TRUE);
        if (ret != DRV_SUCCESS)
        {
            return DRV_ERROR;
        }
    }
    else if (*pOutputPath == CODEC_OUTPUT_SPEAKER)
    {
#ifdef _USE_FWP
        halgpio_Set_Config(AUDIO_PA_EN_PIN, SET_GPIO);
        halgpio_Set_Value(AUDIO_PA_EN_PIN, GPIO_STATE_HIGH);
        halgpio_Set_Direction(AUDIO_PA_EN_PIN, GPIO_OUT);
#endif

        ret =halVpCfg_SetOutputPath(VP_PATH_SPEAKER,TRUE);


        if (ret != DRV_SUCCESS)
        {
            return DRV_ERROR;
        }
    }
    else if (*pOutputPath == CODEC_OUTPUT_HEADSET)
    {
        ret =halVpCfg_SetOutputPath(VP_PATH_HEADSET,TRUE);

        if (ret != DRV_SUCCESS)
        {
            return DRV_ERROR;
        }
    }
    else if (*pOutputPath == CODEC_OUTPUT_BLUETOOTH)
    {
        return DRV_ERR_NOT_SUPPORTED;
    }
    else if (*pOutputPath == CODEC_OUTPUT_HSANDSPK)
    {
        ret = halVpCfg_SetOutputPath(VP_PATH_HSANDSPK,TRUE);
        if (ret != DRV_SUCCESS)
        {
            return DRV_ERROR;
        }
    }
    else
    {
        return DRV_ERR_NOT_SUPPORTED;
    }

    return DRV_SUCCESS;

}

/*******************************************************************************
 * Function: audio_SetOutputPath
 * Description: This function is used to set the path between pmic and output equipment
 * Parameters:
 *   Input:pOutputPath pOutputPath out put path
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_SetOutputPath(T_ZDrv_AudioOutputPath * pOutputPath)
{
    SINT32  ret = DRV_SUCCESS;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetOutputPath\n");

    if (pOutputPath == NULL||*pOutputPath >= MAX_CODEC_OUTPUT_PATH)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetOutputPath err pOutputPath=%x\n",pOutputPath);
        return DRV_ERR_INVALID_PARAM;
    }

    if (audio_OutputPmicInit() != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetOutputPath audio_OutputPmicInit err \n");
        //return DRV_ERROR;
    }

    ret = audio_SetPmicOutputPath(pOutputPath);
    s_outputPath = *pOutputPath;
    if (ret != DRV_SUCCESS)
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetOutputPath audio_SetPmicOutputPath err=%d\n",ret);
    }

    return ret;
	
}

/*******************************************************************************
 * Function: audio_SetOutputVol
 * Description: This function is used to set the audio output volumn.
 * Parameters:
 *   Input:pOutputVol audio output volumn
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_SetOutputVol(T_ZDrvAudio_OutputVolLevel * pOutputVol)
{
    SINT32  ret = DRV_SUCCESS;

    if (audio_OutputPmicInit() != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetOutputVol audio_OutputPmicInit err\n");
        return DRV_ERROR;
    }

    if (s_audioPlayStatus == AUDIO_IDLE)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetOutputVol s_audioPlayStatus == AUDIO_IDLE\n");
        return DRV_ERR_NOT_OPENED;
    }
    /* ò*/
    if (pOutputVol == NULL||*pOutputVol >=  MAX_AUDIO_OUTPUT_VOL_LEVEL)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetOutputVol err pOutputVol=%x\n",pOutputVol);
        return DRV_ERR_INVALID_PARAM;
    }

    s_outputVol = *pOutputVol;
	
    ret = halVpCfg_SetOutputVol(s_outputVol);
    if (ret != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetOutputVol halVpCfg_SetOutputVol err=%d\n",ret);
        return DRV_ERROR;
    }

    return DRV_SUCCESS;
	
}

/*******************************************************************************
 * Function: audio_SetMute
 * Description: This function is used to set the sound of audio mute.
 * Parameters:
 *   Input:Blmute: mute setting
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_SetMute(BOOL *BlMute)
{
    SINT32  ret = DRV_SUCCESS;
#ifdef _USE_CODEC_SNA9029
    T_ZDrvSna9029_AudioOutputVol gain = OUTPGA_GAIN_MUTE;
#endif

    if (NULL==BlMute)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_SetMute NULL==BlMute!\n");
        return DRV_ERR_INVALID_PARAM;
    }

    ret=halVpCfg_SetMute(s_outputPath,*BlMute);
    if (ret != DRV_SUCCESS)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_SetMute halVpCfg_SetMute err=%d!\n",ret);
        return DRV_ERROR;
    }

    s_isMute = *BlMute;

    return DRV_SUCCESS;
	
}
#if 0//xiu
/*******************************************************************************
 * Function: audio_GetPlayInfo
 * Description: This function is get how many bytes in i2s buffer.
 * Parameters:
 *   Input:pPlayInfo;use to save buffer data size
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_GetPlayInfo(T_ZDrvAudio_PlayInfo * pPlayInfo)
{
    SINT32  ret = DRV_SUCCESS;
	
    if (NULL==pPlayInfo)
    {
        return DRV_ERR_INVALID_PARAM;
    }
	
    if (s_audioPlayStatus != AUDIO_INUSE &&s_audioPlayStatus != AUDIO_PAUSE)
    {
        return DRV_ERROR;
    }
	
    ret = zDrvI2S_GetRemained(I2S_ID,&(pPlayInfo->DataLenth));
	
    return ret;

}
#endif
/*******************************************************************************
 * Function: audio_SetCallBackFunc
 * Description: This function is used to set the application's call back function.
 * Parameters:
 *   Input:pCallBack :call back function
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_SetCallBackFunc(T_ZDrvAudio_CallbackFunc pCallBcak)
{
    s_audioCallApplication = pCallBcak;
	
    return DRV_SUCCESS;
}

/*******************************************************************************
 * Function: audio_Write
 * Description: This function is used to set write audio data to i2s.
 * Parameters:
 *   Input:pDatabuff: buffer point
 *   buffLength: buffer length
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/

static SINT32 audio_Write(UINT32 * pDatabuff, UINT32 buffLength)
{
    SINT32  ret = DRV_SUCCESS;
    SINT32 audioBufferNum = 0;
    SINT32 audioBufferLength = 0;
    SINT32 audioBufLengthReal = 0;
    UINT16 *pAudioBuff = NULL;

    if (pDatabuff==NULL || buffLength == 0 ||buffLength >  AUDIO_PLAYBUFFER_MAX)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_Write err pDatabuff=%x,buffLength=%d!\n",pDatabuff,buffLength);
        return DRV_ERR_INVALID_PARAM;
    }
	
    if (s_audioPlayStatus != AUDIO_INUSE)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_Write err s_audioPlayStatus=%d!\n",s_audioPlayStatus);
        return DRV_ERROR;
    }

    audioBufferLength = (UINT16)(buffLength / sizeof(UINT16));
    pAudioBuff = (UINT16 *)pDatabuff;

    switch (s_audioSample)
    {
    case AUDIO_RATE_8_KHZ:
        audioBufLengthReal = buffLength;
        break;
    case AUDIO_RATE_11_025_KHZ:                        /*AUDIO_RATE_11_025_KHZ * 4 = 44.1K */
        if((buffLength * 4) > AUDIO_PLAYBUFFER_MAX )
        {
            return DRV_ERROR;
        }

        for(audioBufferNum = 0; audioBufferNum < audioBufferLength; audioBufferNum++)
        {
            s_playBuffer[audioBufferNum*4 + 0] = s_playBuffer[audioBufferNum*4 + 1] = s_playBuffer[audioBufferNum*4 + 2] =
                    s_playBuffer[audioBufferNum*4 + 3]  =  *(UINT16 *)(pAudioBuff + audioBufferNum);

        }

        audioBufLengthReal = buffLength*4;

        zOss_Memcpy((UINT8 *)pDatabuff, (UINT8 *)s_playBuffer, audioBufLengthReal);

        break;
    case AUDIO_RATE_12_KHZ:                              /*AUDIO_RATE_12_KHZ * 4 = 48K */
        if((buffLength * 4) > AUDIO_PLAYBUFFER_MAX )
        {
            return DRV_ERROR;
        }

        for(audioBufferNum = 0; audioBufferNum < audioBufferLength; audioBufferNum++)
        {
            s_playBuffer[audioBufferNum*4 + 0] = s_playBuffer[audioBufferNum*4 + 1] = s_playBuffer[audioBufferNum*4 + 2] =
                    s_playBuffer[audioBufferNum*4 + 3]  =  *(UINT16 *)(pAudioBuff + audioBufferNum);

        }
        audioBufLengthReal = buffLength*4;
        zOss_Memcpy((UINT8 *)pDatabuff, (UINT8 *)s_playBuffer, audioBufLengthReal);

        break;
    case AUDIO_RATE_16_KHZ:
        audioBufLengthReal = buffLength;
        break;
    case AUDIO_RATE_22_050_KHZ:                      /*AUDIO_RATE_22_050_KHZ * 2 = 44.1K */
        if((buffLength * 2) > AUDIO_PLAYBUFFER_MAX )
        {
            return DRV_ERROR;
        }

        for(audioBufferNum = 0; audioBufferNum < audioBufferLength; audioBufferNum++)
        {
            s_playBuffer[audioBufferNum*2 + 0] = s_playBuffer[audioBufferNum*2 + 1]  =  *(UINT16 *)(pAudioBuff + audioBufferNum);

        }
        audioBufLengthReal = buffLength*2;
        zOss_Memcpy((UINT8 *)pDatabuff, (UINT8 *)s_playBuffer, audioBufLengthReal);

        break;
    case AUDIO_RATE_24_KHZ:                             /*AUDIO_RATE_24_KHZ * 2 = 48K */
        if((buffLength * 2) > AUDIO_PLAYBUFFER_MAX )
        {
            return DRV_ERROR;
        }

        for(audioBufferNum = 0; audioBufferNum < audioBufferLength; audioBufferNum++)
        {
            s_playBuffer[audioBufferNum*2 + 0] = s_playBuffer[audioBufferNum*2 + 1]  =  *(UINT16 *)(pAudioBuff + audioBufferNum);

        }
        audioBufLengthReal = buffLength*2;
        zOss_Memcpy((UINT8 *)pDatabuff, (UINT8 *)s_playBuffer, audioBufLengthReal);

        break;
    case AUDIO_RATE_32_KHZ:                           /*STDAC_RATE_32_KHZ / 2 = 16K */
#if 0
        for(audioBufferNum = 0; audioBufferNum < (audioBufferLength/2); audioBufferNum++)
        {
            s_playBuffer[audioBufferNum]   =  *(UINT16 *)(pAudioBuff + audioBufferNum*2);

        }
        audioBufLengthReal = buffLength/2;
        zOss_Memcpy((UINT8 *)pDatabuff, (UINT8 *)s_playBuffer, audioBufLengthReal);
#endif
        audioBufLengthReal = buffLength;

        break;
    case AUDIO_RATE_44_1_KHZ:
        audioBufLengthReal = buffLength;
        break;
    case AUDIO_RATE_48_KHZ:
        audioBufLengthReal = buffLength;
        break;

    case AUDIO_RATE_64_KHZ:                           /*AUDIO_RATE_64_KHZ / 4 = 16K */
        for(audioBufferNum = 0; audioBufferNum < (audioBufferLength/4); audioBufferNum++)
        {
            s_playBuffer[audioBufferNum]   =  *(UINT16 *)(pAudioBuff + audioBufferNum*4);

        }
        audioBufLengthReal = buffLength/4;
        zOss_Memcpy((UINT8 *)pDatabuff, (UINT8 *)s_playBuffer, audioBufLengthReal);

        break;
    case AUDIO_RATE_96_KHZ:                           /*AUDIO_RATE_96_KHZ / 2 = 48K */
        for(audioBufferNum = 0; audioBufferNum < (audioBufferLength/2); audioBufferNum++)
        {
            s_playBuffer[audioBufferNum]   =  *(UINT16 *)(pAudioBuff + audioBufferNum*2);

        }
        audioBufLengthReal = buffLength/2;
        zOss_Memcpy((UINT8 *)pDatabuff, (UINT8 *)s_playBuffer, audioBufLengthReal);
        break;

    default:

        break;

    }

    ret = zDrvI2S_Write(I2S_ID, (UINT8 *)pDatabuff, audioBufLengthReal);

    return ret;
	
}

/*******************************************************************************
 * Function: audio_GetBuf
 * Description: This function is used to get the buffer of I2S.
 * Parameters:
 *   Input:pBuf: buffer info
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_GetBuf(T_ZDrvAudio_BufInfo * pBuf)
{
    SINT32  ret = DRV_SUCCESS;
	
    if (NULL==pBuf)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_GetBuf NULL==pBuf\n");
        return DRV_ERR_INVALID_PARAM;
    }

    ret = zDrvI2S_GetBuf(I2S_ID,(UINT8 **)(&(pBuf->buf)), &(pBuf->buffersize));
	
    return ret;
}

/*******************************************************************************
 * Function: audio_RecordStart
 * Description: This function is used to start audio recording.
 * Parameters:
 *   Input:datainfo :include buffer size setting
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_RecordStart(T_ZDrvRcd_InfoParam * datainfo)
{
    SINT32  ret = DRV_SUCCESS;
    T_ZDrvI2s_Params parameter = {0};
    T_ZDrvCodec_UseProtocol usePortocol;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_RecordStart \n");
	
    if (NULL ==datainfo ||datainfo->bufsize==0)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_RecordStart datainfo=%x\n",datainfo);
        return DRV_ERR_INVALID_PARAM;
    }
	
    if (s_audioRecordStatus == AUDIO_OPEN)
    {
        if (s_recordI2sCfg.bMaster == FALSE)
        {
            usePortocol.masterSlave = I2S_MASTER_MODE;
        }
        else if (s_recordI2sCfg.bMaster == TRUE)
        {
            usePortocol.masterSlave = I2S_SLAVE_MODE;
        }

        if(I2S_TIME_MODE == s_recordI2sCfg.tClkMode)
		{
            s_recordI2sCfg.tTransMode = DATA_ALIGN_STD_I2S;
            s_recordI2sCfg.tTansmit_edge = TRANS_NEGATIVE_POSITIVE;//for i2s mode;
			usePortocol.busMode = CODEC_BUS_I2S;
        }
        else if(PCM_TIME_MODE == s_recordI2sCfg.tClkMode)
        {
            s_recordI2sCfg.tTransMode = DATA_ALIGN_STD_I2S; //industry standard short frame pcm timing
            s_recordI2sCfg.tTansmit_edge = TRANS_POSITIVE_NEGATIVE;//for pcm mode;
			usePortocol.busMode = CODEC_BUS_DSP;
        }

        if((UINT32)datainfo->channel == AUDIO_DUAL_CHANNEL)//xiu
        {
            s_recordI2sCfg.tTrackFmt = DOUBLE_TRACK;
			s_recordI2sCfg.tTimingType = TIMING_I2S_ST_2CHN_1LANE;
            usePortocol.dataTrack = CODEC_DATA_TRACK_DOUBLE;
        }
        else
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_RecordStart LEFT_TRACK\n");
            s_recordI2sCfg.tTrackFmt = LEFT_TRACK;
			s_recordI2sCfg.tTimingType = TIMING_I2S_MONO_LEFT;
            usePortocol.dataTrack = CODEC_DATA_TRACK_LEFT;
        }

        ret = halVpCfg_SetUseProtocol(usePortocol);
        if (ret != DRV_SUCCESS)
        {
            zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_PlayStart  halVpCfg_SetUseProtocol end! ret=%d\n",ret);
            return DRV_ERROR;
        }

        parameter.channel = datainfo->channel;
        //parameter.p_cb = audio_PutSemID;
        parameter.p_cb = NULL;
        parameter.buffersize = datainfo->bufsize;
        ret=halVpCfg_Enable();
        if (ret != DRV_SUCCESS)
        {
        	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_RecordStart halVpCfg_Enable err=%d\n",ret);
            return DRV_ERROR;
        }

        ret = zDrvI2S_Read_Start(I2S_ID, &parameter, &s_recordI2sCfg);
        if (ret != DRV_SUCCESS)
        {
        	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_RecordStart zDrvI2S_Read_Start err=%d\n",ret);
			halVpCfg_Disable();
            return DRV_ERROR;
        }

        s_audioRecordStatus = AUDIO_INUSE;
        s_audioPlayStatus = AUDIO_IDLE;
    }

    return ret;
}

/*******************************************************************************
 * Function: audio_RecordStop
 * Description: This function is used to stop the audio recording .
 * Parameters:
 *   Input:None
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_RecordStop(VOID)
{
    SINT32  ret = DRV_SUCCESS, ret2 = DRV_SUCCESS;
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_RecordStop start!\n");
	
    if (s_audioRecordStatus == AUDIO_INUSE)
    {
        /* close the audio */
        ret = zDrvI2S_Read_Stop(I2S_ID);
        if (ret != DRV_SUCCESS)
        {
        	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_RecordStop zDrvI2S_Read_Stop err=%d\n",ret);
           // return DRV_ERROR;
        }

        ret2 = halVpCfg_Disable();
        if (ret2 != DRV_SUCCESS)
        {
        	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_RecordStop halVpCfg_Disable err=%d\n",ret2);
			ret += ret2;
		   // return DRV_ERROR;
        }
        s_audioRecordStatus = AUDIO_OPEN;
    }
	
    return ret;
	
}

/*******************************************************************************
 * Function: audio_RecordDataRead
 * Description: This function is used to read record data from i2s.
 * Parameters:
 *   Input:pDatabuff application record data buffer
              length       buffer length
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_RecordDataRead(T_ZDrvAudio_BufInfo * pBuf)
{
    SINT32  ret = DRV_SUCCESS;
	
    if (NULL==pBuf)
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_RecordDataRead err NULL==pBuf\n");
        return DRV_ERR_INVALID_PARAM;
    }
	
    if (s_audioRecordStatus != AUDIO_INUSE)
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_RecordDataRead err s_audioRecordStatus=%d\n",s_audioRecordStatus);
        return DRV_ERROR;
    }
	
    ret = zDrvI2S_Read(I2S_ID,(UINT8 **)(&(pBuf->buf)), &(pBuf->buffersize));
	
    return ret;
}

/*******************************************************************************
 * Function: audio_FreeRecordBuf
 * Description: This function is used to free I2S record RAM buffer.
 * Parameters:
 *   Input:pVoiceBlock audio record configation
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_FreeRecordBuf(VOID * pBuf)
{
    SINT32  ret = DRV_SUCCESS;

	if (NULL == pBuf)
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_FreeRecordBuf err NULL==pBuf\n");
        return DRV_ERR_INVALID_PARAM;
    }
	
    if (s_audioRecordStatus != AUDIO_INUSE)
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_FreeRecordBuf s_audioRecordStatus=%d\n",s_audioRecordStatus);
        return DRV_ERROR;
    }
	
    ret = zDrvI2S_FreeBuf(I2S_ID,pBuf);
	
    return ret;
}
#if 0//xiu
/*******************************************************************************
 * Function: audio_Enable3DEffect
 * Description: This function is used to enable 3D audio effect.
 * Parameters:
 *   Input:None
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *              DRV_ERR_NOT_SUPPORTED : not support this function
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_Enable3DEffect(T_ZDrvAudio_3DEFFECT * p3DEFFECT)
{
    return DRV_ERR_NOT_SUPPORTED;
}
#endif
/*******************************************************************************
 * Function: audio_OutputPmicInit
 * Description: This function is used to pmic while play audio
 * Parameters:
 *   Input:None
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_InputPmicInit(void)//xiu
{
    SINT32  ret = 0;
    //BOOL OnOff = TRUE;//xiu
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_InputPmicInit\n");
	
    if(s_audioPlayStatus==AUDIO_OPEN)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_InputPmicInit has opened\n");
        return DRV_SUCCESS;
    }
	
    if (s_audioPlayStatus != AUDIO_IDLE)
    {
        return DRV_ERR_BUSY;
    }

    if (s_audioRecordStatus == AUDIO_IDLE)
    {
        ret = halVpCfg_Open(VP_AUDIO_SOURCE);
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_InputPmicInit zDrvCodec_Open ret=%d\n",ret);//modify by dl

        if (ret != DRV_SUCCESS)
        {
            halVpCfg_Disable();
            return ret;
        }

        s_audioRecordStatus = AUDIO_OPEN;
    }
	
    return DRV_SUCCESS;

}

/*******************************************************************************
 * Function: audio_SetPmicInputPath
 * Description: This function is used to set Pmic input path.
 * Parameters:
 *   Input:pInputPath input path
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_SetPmicInputPath(T_ZDrv_CodecInputPath pInputPath)
{
    SINT32  returnValue = DRV_SUCCESS;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_SetPmicInputPath\n");
    /* add up link path*/
    switch (pInputPath)
    {
    case CODEC_INPUT_MICPHONE:
        returnValue = halVpCfg_SetInputPath(VP_PATH_HANDSET,TRUE);
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_SetPmicInputPath returnValue=%d\n",returnValue);//add by dl
        if (DRV_SUCCESS!= returnValue)
        {
            return DRV_ERROR;
        }
        break;

    case CODEC_INPUT_HEADSET:
        returnValue = halVpCfg_SetInputPath(VP_PATH_HEADSET,TRUE);
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_SetPmicInputPath returnValue=%d\n",returnValue);//add by dl
        //returnValue += zDrvCodec_SetMicBias(s_audioRecordHandler, AUDIO_INPUT_HEADSET,MICBIAS_OUT_2_5V);
        if (DRV_SUCCESS!= returnValue)
        {
            return DRV_ERROR;
        }
        break;
    case CODEC_INPUT_BLUETOOTH:
        return DRV_ERR_NOT_SUPPORTED;

    default:
        return DRV_ERR_NOT_SUPPORTED;

    }

    return returnValue;
	
}

/*******************************************************************************
 * Function: audio_SetRecordPath
 * Description: This function is used to set record input path.
 * Parameters:
 *   Input:param input path param
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_SetRecordPath(T_ZDrv_CodecInputPath * param)
{
    SINT32  ret = DRV_SUCCESS;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_SetRecordPath\n");
	
    if (NULL == param)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_SetRecordPath param is NULL\n");
        return DRV_ERR_INVALID_PARAM;
    }

    if (audio_InputPmicInit()!=DRV_SUCCESS)
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_SetRecordPath audio_InputPmicInit err\n");
        return DRV_ERROR;
    }

    ret= audio_SetPmicInputPath(*param);
	
    return ret;
}

/*******************************************************************************
 * Function: audio_SetRecordSample
 * Description: This function is used to set the audio record sample rate.
 * Parameters:
 *   Input:pSampleRate sample rate
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_SetRecordSample(T_ZDrvAudio_SampleRate * pSampleRate)
{
    SINT32  ret = DRV_SUCCESS;


    T_ZDrv_VpFs vcodec_sample = VP_FS_8000;

    if (pSampleRate == NULL)
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_SetRecordSample pSampleRate == NULL\n");
        return DRV_ERR_INVALID_PARAM;
    }
	
    if (audio_InputPmicInit()!=DRV_SUCCESS)
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_SetRecordSample audio_InputPmicInit err\n");
        return DRV_ERROR;
    }
	
    if (s_audioRecordStatus == AUDIO_OPEN)
    {

        switch (*pSampleRate)
        {
        case AUDIO_RATE_8_KHZ:
            vcodec_sample = VP_FS_8000;
            s_recordI2sCfg.sample_rate = 8000;
            break;
        case AUDIO_RATE_16_KHZ:
            vcodec_sample = VP_FS_16000;
            s_recordI2sCfg.sample_rate =16000;
            break;
        case  AUDIO_RATE_44_1_KHZ:
            vcodec_sample = VP_FS_44100;
            s_recordI2sCfg.sample_rate =44100;
            break;
        case  AUDIO_RATE_48_KHZ:
            vcodec_sample = VP_FS_48000;
            s_recordI2sCfg.sample_rate =48000;
            break;
        default:
            vcodec_sample = VP_FS_8000;
            s_recordI2sCfg.sample_rate = 8000;
            break;
        }

        ret =halVpCfg_SetFs (vcodec_sample);
        if (ret != DRV_SUCCESS)
        {
        	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetRecordSample halVpCfg_SetFs err=%d\n",ret);
            return DRV_ERROR;
        }
    }
    else
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetRecordSample err s_audioRecordStatus==AUDIO_OPEN\n");
        return DRV_ERROR;
    }

    return DRV_SUCCESS;
}
/*******************************************************************************
 * Function: audio_SetRecordVol
 * Description: This function is used to set record input volumn.
 * Parameters:
 *   Input:param input gain setting
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_SetPmicRecVol(T_ZDrvAudio_InputVolLevel * InputVol)
{
    SINT32  ret = DRV_SUCCESS;

    ret = halVpCfg_SetInputVol(*InputVol);
    if (ret != DRV_SUCCESS)
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetPmicRecVol halVpCfg_SetInputVol err=%d\n",ret);
        return DRV_ERROR;
    }

    return ret;
}
/*******************************************************************************
 * Function: audio_SetRecordVol
 * Description: This function is used to set record input volumn.
 * Parameters:
 *   Input:param input gain setting
 *   Output:None
 *
 * Returns: DRV_SUCCESS: operate success
 *              DRV_ERROR:operate fail
 *              DRV_ERR_INVALID_PARAM:param error
 *
 * Others:
 ********************************************************************************/
static SINT32 audio_SetRecordVol(T_ZDrvAudio_InputVolLevel * InputVol)
{
    SINT32  ret = DRV_SUCCESS;
	
    if (NULL == InputVol)
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetRecordVol err NULL == InputVol\n");
        return DRV_ERR_INVALID_PARAM;
    }
	
    if (audio_InputPmicInit()!=DRV_SUCCESS)
    {
    	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio audio_SetRecordVol audio_InputPmicInit err\n");
        return DRV_ERROR;
    }

    s_recordVol = InputVol ;
	ret = audio_SetPmicRecVol(s_recordVol);
    return ret;
}
#if 0//xiu
/*******************************************************************************
 * Function: zDrvAudio_Init
 * Description: This function is used to init audio driver.
 * Parameters:
 *   Input:None
 *   Output:None
 *
 * Returns: None
 *
 * Others:
 ********************************************************************************/
static SINT32 halHrdTstPcm_Initiate(VOID)
{
#ifdef _USE_HAL_HRDTST
    halHrdTstPcm_SetOperations(&gHalHrdTst_Pcm_Opt);
#endif
    return DRV_SUCCESS;
}
static SINT32 audio_SetFMOutputPath(T_ZDrv_AudioOutputPath * pOutputPath)
{
    SINT32  ret = DRV_SUCCESS;
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"audio_SetOutputPath!\n");
	
    if (pOutputPath == NULL||*pOutputPath >= MAX_CODEC_OUTPUT_PATH)
    {
        return DRV_ERR_INVALID_PARAM;
    }

    if (audio_OutputPmicInit() != DRV_SUCCESS)
    {
        return DRV_ERROR;
    }

    ret = audio_SetPmicOutputPath(pOutputPath);
	
    s_outputPath = *pOutputPath;
	
#if defined _USE_TEAKAUD
    if (ret != DRV_SUCCESS)
    {
        return ret;
    }

#endif
    return ret;

}
#endif

/*******************************************************************************
 * Function: zDrvAudio_SetI2sCfg
 * Description: This function is used to set i2s parameters.
 * Parameters:
 *   Input:None
 *   Output:None
 *
 * Returns: success or error code
 *
 *
 * Others:
 ********************************************************************************/
SINT32 zDrvAudio_SetI2sCfg(T_ZDrvI2S_Cfg *pI2sCfg)
{
	SINT32 ret = DRV_SUCCESS;

	if (NULL != pI2sCfg)
	{
		s_playI2sCfg = *pI2sCfg;	 
		s_recordI2sCfg = *pI2sCfg;
	}
	else
	{	
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvAudio_SetI2sCfg error: pI2sCfg == NULL\n");
		return DRV_ERR_INVALID_PARAM;
	}
	return ret;
}

/*******************************************************************************
 * Function: zDrvAudio_Initiate
 * Description: This function is used to init audio driver.
 * Parameters:
 *   Input:None
 *   Output:None
 *
 * Returns: None
 *
 * Others:
 ********************************************************************************/
SINT32 zDrvAudio_Initiate(VOID)
{
    zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO,"zDrvAudio_Initiate!");

    zDrvAudio_SetOperations(&gHalAudio_ObjOpt);
//    halHrdTstPcm_Initiate();
    zDrvAudio_Init();

    return DRV_SUCCESS;
}
