/*
 ** Copyright 2003-2010, VisualOn, Inc.
 **
 ** Licensed under the Apache License, Version 2.0 (the "License");
 ** you may not use this file except in compliance with the License.
 ** You may obtain a copy of the License at
 **
 **     http://www.apache.org/licenses/LICENSE-2.0
 **
 ** Unless required by applicable law or agreed to in writing, software
 ** distributed under the License is distributed on an "AS IS" BASIS,
 ** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 ** See the License for the specific language governing permissions and
 ** limitations under the License.
 */

/*****************************************************************************
 *  This file contains operations in double precision.                       *
 *  These operations are not standard double precision operations.           *
 *  They are used where single precision is not enough but the full 32 bits  *
 *  precision is not necessary. For example, the function Div_32() has a     *
 *  24 bits precision which is enough for our purposes.                      *
 *                                                                           *
 *  The double precision numbers use a special representation:               *
 *                                                                           *
 *     L_32 = hi<<16 + lo<<1                                                 *
 *                                                                           *
 *  L_32 is a 32 bit integer.                                                *
 *  hi and lo are 16 bit signed integers.                                    *
 *  As the low part also contains the sign, this allows fast multiplication. *
 *                                                                           *
 *      0x8000 0000 <= L_32 <= 0x7fff fffe.                                  *
 *                                                                           *
 *  We will use DPF (Double Precision Format )in this file to specify        *
 *  this special format.                                                     *
 *****************************************************************************
*/
#include "typedef.h"
#include "basic_op.h"
#include "oper_32b.h"


  

/*****************************************************************************
 *                                                                           *
 *  Function L_Comp()                                                        *
 *                                                                           *
 *  Compose from two 16 bit DPF a 32 bit integer.                            *
 *                                                                           *
 *     L_32 = hi<<16 + lo<<1                                                 *
 *                                                                           *
 *  Arguments:                                                               *
 *                                                                           *
 *   hi        msb                                                           *
 *   lo        lsf (with sign)                                               *
 *                                                                           *
 *   Return Value :                                                          *
 *                                                                           *
 *             32 bit long signed integer (Word32) whose value falls in the  *
 *             range : 0x8000 0000 <= L_32 <= 0x7fff fff0.                   *
 *                                                                           *
 *****************************************************************************
*/

Word32 L_Comp (Word16 hi, Word16 lo)
{
	Word32 L_32;

	L_32 = L_deposit_h (hi);

	return (L_mac (L_32, lo, 1));       /* = hi<<16 + lo<<1 */
}


/*****************************************************************************
 *                                                                           *
 *   Function Name : Div_32                                                  *
 *                                                                           *
 *   Purpose :                                                               *
 *             Fractional integer division of two 32 bit numbers.            *
 *             L_num / L_denom.                                              *
 *             L_num and L_denom must be positive and L_num < L_denom.       *
 *             L_denom = denom_hi<<16 + denom_lo<<1                          *
 *             denom_hi is a normalize number.                               *
 *                                                                           *
 *   Inputs :                                                                *
 *                                                                           *
 *    L_num                                                                  *
 *             32 bit long signed integer (Word32) whose value falls in the  *
 *             range : 0x0000 0000 < L_num < L_denom                         *
 *                                                                           *
 *    L_denom = denom_hi<<16 + denom_lo<<1      (DPF)                        *
 *                                                                           *
 *       denom_hi                                                            *
 *             16 bit positive normalized integer whose value falls in the   *
 *             range : 0x4000 < hi < 0x7fff                                  *
 *       denom_lo                                                            *
 *             16 bit positive integer whose value falls in the              *
 *             range : 0 < lo < 0x7fff                                       *
 *                                                                           *
 *   Return Value :                                                          *
 *                                                                           *
 *    L_div                                                                  *
 *             32 bit long signed integer (Word32) whose value falls in the  *
 *             range : 0x0000 0000 <= L_div <= 0x7fff ffff.                  *
 *                                                                           *
 *  Algorithm:                                                               *
 *                                                                           *
 *  - find = 1/L_denom.                                                      *
 *      First approximation: approx = 1 / denom_hi                           *
 *      1/L_denom = approx * (2.0 - L_denom * approx )                       *
 *                                                                           *
 *  -  result = L_num * (1/L_denom)                                          *
 *****************************************************************************
*/
#if PV_CPU_ARCH_VERSION < 6
Word32 Div_32 (Word32 L_num, Word16 denom_hi, Word16 denom_lo)
{
	Word16 approx, hi, lo, n_hi, n_lo;
	Word32 L_32;

	/* First approximation: 1 / L_denom = 1/denom_hi */

	approx = div_s ((Word16) 0x3fff, denom_hi);

	/* 1/L_denom = approx * (2.0 - L_denom * approx) */

	L_32 = Mpy_32_16 (denom_hi, denom_lo, approx);

	L_32 = L_sub ((Word32) 0x7fffffffL, L_32);
	hi = L_32 >> 16;
	lo = (L_32 & 0xffff) >> 1;

	L_32 = Mpy_32_16 (hi, lo, approx);

	/* L_num * (1/L_denom) */
	hi = L_32 >> 16;
	lo = (L_32 & 0xffff) >> 1;
	VO_L_Extract (L_num, &n_hi, &n_lo);
	L_32 = Mpy_32 (n_hi, n_lo, hi, lo);
	L_32 = L_shl2(L_32, 2);

	return (L_32);
}
#else
Word32 Div_32(Word32 L_num,
              Word32 L_denom)
{
    Word16 approx;
    Word32 result;

    /* First approximation: 1 / L_denom = 1/L_denom_hi */
    approx = div_s((Word16) 0x3fff, (Word16)(L_denom>>16));

    /* 1/L_denom = approx * (2.0 - L_denom * approx) */
    result = zMmp_AmrWbMpy32_16(L_denom, approx); //
    /* result is > 0 , and less than 1.0 */
    result =  L_sub ((Word32) 0x7fffffffL, result);
    result = zMmp_AmrWbMpy32_16(result, approx);

    /* L_num * (1/L_denom) */
    result = zMmp_AmrWbMpy32(L_num, result);
    result = L_shl2(result, 2);

    return (result);
}

#endif

