/*******************************************************************************
* Copyright (C) 2013, ZTE Corporation.
*
* File Name:    ring_queue.c
* File Mark:
* Description:
* Others:
* Version:		 V1.0
* Author:		 geanfeng
* Date: 		 2013-09-24
********************************************************************************/

#include "drvs_general.h"
#include "drvs_assert.h"
#include "audio_ring_queue.h"

/**
 * queue_lock
 */
static SINT32 queue_lock(T_queue *queue)
{
    UINT32 old_intr;

    if (queue->multi_user_protect == QUEUE_PROTECT_MUTEX && queue->lock != NULL)
    {
        zOss_GetSemaphore(queue->lock , ZOSS_WAIT_FOREVER);
    }
    else if (queue->multi_user_protect == QUEUE_PROTECT_IRQ)
    {
        LOCK_SAVE(old_intr);
        queue->g_queueMsr = old_intr;
    }

    return 0;
}

/**
 * queue_unlock
 */
static void queue_unlock(T_queue *queue)
{
    UINT32 old_intr;

    if (queue->multi_user_protect == QUEUE_PROTECT_MUTEX && queue->lock != NULL)
    {
        zOss_PutSemaphore(queue->lock );
    }
    else if (queue->multi_user_protect == QUEUE_PROTECT_IRQ)
    {
        old_intr = queue->g_queueMsr;
        LOCK_RESTORE(old_intr);
    }
}

/**
 * queue_Create - create a new ring queue.
 *
 * Create a new special ring queue.
 */
T_queue *queue_Create(UINT32 unit_count, UINT32 unit_size, T_USER_PROTECT_POLICY multi_user_protect)
{
    T_queue *queue = NULL;
    UINT8 *buffer = NULL;

    queue = (T_queue*)zOss_Malloc(sizeof(T_queue));
    if (!queue)
        return NULL;

    zOss_Memset(queue, 0x0, sizeof(T_queue));

    buffer = (UINT8 *)zOss_Malloc((unit_count) * unit_size);
    if (!buffer)
    {
        zOss_Free(queue);
        return NULL;
    }
    
    queue->unit_status = (UINT32 *)zOss_Malloc((unit_count) * sizeof(UINT32));
    if (!queue->unit_status)
    {
        zOss_Free(queue);
        zOss_Free(buffer);
        return NULL;
    }

    queue->unit_buffer = buffer;
    queue->unit_buffer_size = (unit_count) * unit_size;
    queue->read_pos = 0;
    queue->write_pos = 0;
    queue->unit_size = unit_size;
    queue->unit_count = unit_count;
    queue->multi_user_protect = multi_user_protect;
    queue->unit_index_en = 0;
    queue->unit_index_de = 0;
    queue->g_queueMsr = 0;

    if (queue->multi_user_protect == QUEUE_PROTECT_MUTEX)
    {
        queue->lock = zOss_CreateSemaphore("ring queue lock", 1);
        zDrv_ASSERT(queue->lock );
    }

    return queue;
}

/**
 * queue_Init - initialize a new ring queue.
 *
 *
 */
SINT32 queue_Init(T_queue *queue, UINT8 *unit_buffer,
                      UINT32 unit_count, UINT32 unit_size, T_USER_PROTECT_POLICY protect_policy)
{

    if (queue == NULL || unit_buffer == NULL)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    zOss_Memset(queue, 0x0, sizeof(T_queue));

    queue->unit_buffer = unit_buffer;
    queue->unit_buffer_size = unit_count * unit_size;
    queue->read_pos = 0;
    queue->write_pos = 0;
    queue->unit_size = unit_size;
    queue->unit_count = unit_count;
    queue->multi_user_protect = protect_policy;
    queue->unit_index_en = 0;
    queue->unit_index_de = 0;

    return 0;
}

/**
 * queue_Enqueue
 *
 *
 */
SINT32 queue_Enqueue(T_queue *queue, void *unit)
{
    UINT32 pos;
    SINT32 ret = 0;

    if (queue == NULL || queue->unit_buffer == NULL || queue->unit_status == NULL || unit == NULL)
    {
        zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "queue_Enqueue queue == NULL || unit == NULL || queue->unit_buffer == NULL || queue->unit_status == NULL\n");
        //zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    if (queue->multi_user_protect > QUEUE_PROTECT_RAW)
        queue_lock(queue);

    pos = queue->write_pos + queue->unit_size;
    if (pos >= queue->unit_buffer_size)
        pos = 0;
    
    if (pos == queue->read_pos)
    {
        ret = DRV_ERROR_FULL; /*full*/
        goto quit;
    }

    zOss_Memcpy(&queue->unit_buffer[queue->write_pos], unit, queue->unit_size);

    *(queue->unit_status + queue->unit_index_en) = UNIT_NEW;
    queue->unit_index_en = queue->write_pos / queue->unit_size;
    if(queue->unit_index_en == queue->unit_count)
        queue->unit_index_en = 0;

    queue->write_pos = pos;

    ret = DRV_SUCCESS;
quit:
    if (queue->multi_user_protect > QUEUE_PROTECT_RAW)
        queue_unlock(queue);
    return ret;
}

/**
 * queue_Dequeue
 *
 *
 */
SINT32 queue_Dequeue(T_queue *queue, void *unit)
{
    SINT32 ret = 0;

    if (queue == NULL || unit == NULL)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    if (queue->multi_user_protect > QUEUE_PROTECT_RAW)
        queue_lock(queue);

    if (queue->read_pos == queue->write_pos)
    {
        ret = DRV_ERROR_EMPTY; /*empty*/
        goto quit;
    }

    zOss_Memcpy(unit, &queue->unit_buffer[queue->read_pos], queue->unit_size);
    
    *(queue->unit_status + queue->unit_index_de) = UNIT_OLD;
    queue->unit_index_de = queue->read_pos / queue->unit_size;
    if(queue->unit_index_de == queue->unit_count)
        queue->unit_index_de = 0;

    if ((queue->read_pos + queue->unit_size) >= queue->unit_buffer_size)
        queue->read_pos = 0;
    else
        queue->read_pos += queue->unit_size;

    ret = DRV_SUCCESS;
quit:
    if (queue->multi_user_protect > QUEUE_PROTECT_RAW)
        queue_unlock(queue);
    return ret;
}

/**
 * queue_Empty
 *
 *
 */
SINT32 queue_Empty(T_queue *queue)
{

    if (queue == NULL)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    if (queue->read_pos == queue->write_pos)
        return 1;
    else
        return 0;
}

/**
 * queue_Full
 *
 *
 */
SINT32 queue_Full(T_queue *queue)
{
    UINT32 pos;

    if (queue == NULL)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    pos = queue->write_pos + queue->unit_size;
    if (pos >= queue->unit_buffer_size)
        pos = 0;

    if (pos == queue->read_pos)
        return 1;
    else
        return 0;
}

/**
 * queue_Fush
 * attention: 豣֤ring_queue_dequeue
 *
 */
void queue_Fush(T_queue *queue)
{
    if (queue == NULL)
    {
        //zDrv_ASSERT(0);
        return ;
    }

    if (queue->multi_user_protect > QUEUE_PROTECT_RAW)
        queue_lock(queue);

    queue->read_pos = queue->write_pos;

    if (queue->multi_user_protect > QUEUE_PROTECT_RAW)
        queue_unlock(queue);
    return ;
}

/**
 * queue_Destroy
 *
 *
 */
void queue_Destroy(T_queue *queue)
{

    if (queue == NULL)
    {
        //zDrv_ASSERT(0);
        return ;
    }

    if (queue->multi_user_protect == QUEUE_PROTECT_MUTEX)
    {
        zOss_DeleteSemaphore(queue->lock );
        queue->lock = NULL;
    }

    zOss_Free(queue->unit_buffer);
    zOss_Free(queue->unit_status);
    //zOss_Memset(queue, 0x0, sizeof(T_queue));
    zOss_Free(queue);
	queue = NULL;

    return ;
}

