/**
 *   Copyright (C) Infineon Technologies Denmark A/S. All rights reserved.
 *
 * This document contains proprietary information belonging to Infineon
 * Technologies Denmark A/S. Passing on and copying of this document, use
 * and communication of its contents is not permitted without prior written
 * authorisation.
 *
 * Description:
 *   Main state machine for the speech resource.
 *
 * Revision Information:
 *   File name: \dwddrv\AUD\src\aud_ear.c
 *   Version: \main\145
 *   Date: 2007-08-28 11:58:29
 *   Responsible: johansek
 *   Comment:
 *     Corrected possible null pointer error (SMS 660940).
 */

/*******************************************************************************
*
*                               Include files
*
*******************************************************************************/
//#include "scttypes.h"           /* must be included in all files */
//#include "ms.h"                 /* must be included in all files */
#include "bastypes.h"           /* must be included in all files */
//#include "dsp.h"                /* for DSP command types */
#include "aud_drv.h"            /* audio types and functions */
#include "aud_data.h"           /* data describing the paths */
#include "aud_com.h"            /* audio-internal types and functions */
#include "aud_path_control.h"   /* for enabling the paths when the speech resource is active */
#include "aud_ear.h"            /* contains types needed by the speech resource */
//#include "trap.h"               /* for exception logging */
#include "aud_volume.h"         /* master volume */
#include "drvs_teak.h"

/*******************************************************************************
*
*                               Internal data
*
*******************************************************************************/
static BOOL aud_ear_HF_active = FALSE;    /* Is the hands-free algorithm active? */
static S16  aud_ear_EC_NR_mask = 0xFFFF;  /* Used to control the current setting of EC/NR - used as a mask */
/* these structures could also be static if they weren't used by aud_tst.c */
T_DSP_CMD_HF_SET_PAR aud_dsp_hf_parms;    /* parameters for programming the hands-free algorithms */
T_DSP_CMD_HF_ON_PAR  aud_dsp_hf_on_parms; /* turn on/off the hands-free algorithms */

/*******************************************************************************
*
*                         Internal/private functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: aud_ear_dsp_program_hands_free
* Return.....: -
* Description: Programs the hands-free parameters to the DSP.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static void aud_ear_dsp_program_hands_free(void) 
{
#if 0
    switch(aud_get_master_volume_analog())
    {
        case afe_earpc_gainMinus18db:
            aud_dsp_hf_parms.gain_analog = 4095;
            break;
        case afe_earpc_gainMinus12db:
            aud_dsp_hf_parms.gain_analog = 8191;
            break;
        case afe_earpc_gainMinus6db:
            aud_dsp_hf_parms.gain_analog = 16384;
            break;
        case afe_earpc_gain0db:
            aud_dsp_hf_parms.gain_analog = 32767;
            break;
        default:
            AUD_ERROR_REPORTING(aud_error_grade2, TRAP_AUD_ERROR, "Illegal enum value in Master Volume AFE setting %d in aud_ear_dsp_program_hands_free", aud_get_master_volume_analog());
            break;
    } 
#endif
    aud_dsp_hf_parms.gain_analog = 0;
    aud_dsp_hf_parms.step_width     = aud_path_settings_result.uplink_parms.p_hf->step_width;
    aud_dsp_hf_parms.lms_length     = aud_path_settings_result.uplink_parms.p_hf->lms_length;
    aud_dsp_hf_parms.lms_offset     = aud_path_settings_result.uplink_parms.p_hf->lms_offset;
    aud_dsp_hf_parms.block_length   = aud_path_settings_result.uplink_parms.p_hf->block_length;
    aud_dsp_hf_parms.rxtx_relation  = aud_path_settings_result.uplink_parms.p_hf->rxtx_relation;

    aud_dsp_hf_parms.nr_sw_2        = aud_path_settings_result.uplink_parms.p_hf->nr_sw_2;
    aud_dsp_hf_parms.nr_u_fak_0     = aud_path_settings_result.uplink_parms.p_hf->nr_u_fak_0;
    aud_dsp_hf_parms.nr_u_fak       = aud_path_settings_result.uplink_parms.p_hf->nr_u_fak;

    aud_dsp_hf_parms.add_atten      = aud_path_settings_result.uplink_parms.p_hf->add_atten;
    aud_dsp_hf_parms.min_atten      = aud_path_settings_result.uplink_parms.p_hf->min_atten;
    aud_dsp_hf_parms.max_atten      = aud_path_settings_result.uplink_parms.p_hf->max_atten;

    DSP_ASYNC_HF_SET_PAR(&aud_dsp_hf_parms) /* write parameters to DSP */
#ifdef AUDIO_LLT
    aud_common_trace(llt_group_audio, llt_type_audio_function_interface, 0, 0, /* LLT params */
                     'd', 0x9,                                                 /* LIS params */
                     aud_log_lis_display, aud_log_lis_display_nodata, aud_log_lis_trace);  /* LIS params */
#endif
} /* End of function aud_ear_dsp_program_hands_free */

/*------------------------------------------------------------------------------
* Function...: aud_ear_dsp_start_hands_free
* Return.....: -
* Description: Sets the status of the hands-free block
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static void aud_ear_dsp_start_hands_free(
    U16 ctrl_word)  /* bitmask that determines how to start the echo canceller, noise reduction algorithm and automatic gain control */
{
    ctrl_word &= aud_ear_EC_NR_mask;              /* add the aud_ear_EC_NR_mask to the control word */
//    aud_dsp_hf_on_parms.mode = ctrl_word; /* enable/restart/disable hands-free block */
    aud_dsp_hf_on_parms.mode = 0; /* enable/restart/disable hands-free block */
    DSP_ASYNC_HF_ON(&aud_dsp_hf_on_parms); /* write HF_ON status to DSP */
#ifdef AUDIO_LLT
    aud_common_trace(llt_group_audio, llt_type_audio_dsp_VB_gain, 1, 0,    /* LLT params */
                     'd', 0x9,                                             /* LIS params */
                     aud_log_lis_display, aud_log_lis_display_data, aud_log_lis_trace, /* LIS params */
                     ctrl_word );
#endif
} /* End of function aud_ear_dsp_start_hands_free */

/*******************************************************************************
*
*                        Externally visible functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: aud_speech_SM
* Return.....: TRUE/FALSE, depending on whether the event has been handled
* Description: State machine for the speech resource.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
BOOL aud_speech_SM(
    aud_event_enum event,   /* the event to be handled */
    void* p_data)           /* data to handle the event with */
{
    BOOL func_ret = TRUE; /* used to tell whether the event has been handled by this state machine */
    disp_to_SM_data_type* p_SM_data = (disp_to_SM_data_type*)p_data; /* contains the structure passed from the dispatcher */

    /* Send string : at+xl1set="l1 1l" To enable this LLT trace */
#ifdef AUDIO_LLT
    if(NULL != p_SM_data)
    {
        aud_common_trace( llt_group_audio, llt_type_audio_sm_speech, 11, 0,         /* LLT params */
                         's', 0x01,                                                 /* LIS params */
                         aud_log_lis_raw_data, aud_log_lis_display_data, aud_log_lis_llt_trace, /* LIS params */
                         (U32) aud_get_SM_state(aud_resource_speech), 
                         (U32) event,
                         (U32) p_SM_data->parm1,
                         (U32) p_SM_data->parm2,
                         (U32) p_SM_data->parm3,
                         (U32) p_SM_data->parm4,
                         (U32) p_SM_data->parm5,
                         (U32) p_SM_data->parm6,
                         (U32) p_SM_data->parm7,
                         (U32) p_SM_data->parm8,
                         (U32) E_audio_llt_type_id_0 );
    }
    else
    {
        aud_common_trace( llt_group_audio, llt_type_audio_sm_speech, 3, 0,          /* LLT params */
                         's', 0x01,                                                 /* LIS params */
                         aud_log_lis_raw_data, aud_log_lis_display_data, aud_log_lis_llt_trace, /* LIS params */
                         (U32) aud_get_SM_state(aud_resource_speech), 
                         (U32) event,
                         (U32) E_audio_llt_type_id_1 );
    }
#endif
    switch(aud_get_SM_state(aud_resource_speech))
    {
        /*--------------------------------------------------------------------*/
        /*                           Idle state                               */
        /*--------------------------------------------------------------------*/
        case S_idle:
        {
            switch(event)
            {
                case E_speech_enable: 
                    aud_disable_standby(aud_resource_speech);
                    aud_enable_path_and_resource(aud_resource_speech, S_active, TRUE);
                    break;
                case E_speech_release_disable:
                case E_speech_disable:
                    /* Event has no effect in this state, hence it's passed through */
                    break;
                case E_control_EC_NR:
                    if(NULL != p_SM_data)
                    {
                        /* make the mask */
                        if((U8)(p_SM_data->parm1)) /* EC on */
                        {
                            aud_ear_EC_NR_mask |= 0x0004;   /* set bit 2 to turn EC on */
                        }
                        else
                        {
                            aud_ear_EC_NR_mask &= 0xFFFB;   /* clear bit 2 to turn EC off */
                        }
                        if((U8)(p_SM_data->parm2)) /* NR on */
                        {
                            aud_ear_EC_NR_mask |= 0x0020;   /* set bit 5 to turn NR on */
                        }
                        else
                        {
                            aud_ear_EC_NR_mask &= 0xFFDF;   /* clear bit 5 to turn NR off */
                        }
                    }
                    break;
                default:
                    func_ret = FALSE; /* the event has not been handled */
                    break;    
            }
        }
        break;
        /*--------------------------------------------------------------------*/
        /*                            Active state                            */
        /*--------------------------------------------------------------------*/
        case S_active:
        {
            switch(event)
            {
                case E_speech_release_disable:
                case E_speech_disable:
                    aud_disable_path_and_resource(aud_resource_speech, S_idle, TRUE);
                    aud_enable_standby(aud_resource_speech);
                    break;
                case E_control_EC_NR: /* control EC and NR */
                    if(NULL != p_SM_data)
                    {
                        /* make the mask */
                        if((U8)(p_SM_data->parm1)) /* EC on */
                        {
                           aud_ear_EC_NR_mask |= 0x0004;    /* set bit 2 to turn EC on */
                        }
                        else
                        {
                           aud_ear_EC_NR_mask &= 0xFFFB;    /* clear bit 2 to turn EC off */
                        }
                        if((U8)(p_SM_data->parm2)) /* NR on */
                        {
                           aud_ear_EC_NR_mask |= 0x0020;    /* set bit 5 to turn NR on */
                        }
                        else
                        {
                           aud_ear_EC_NR_mask &= 0xFFDF;    /* clear bit 5 to turn NR off */
                        }
                        aud_drv_activate_hf();
                    }
                    break;
                case E_speech_enable:
                    /* Event has no effect in this state, hence it's passed through */
                    break;
                default:
                    func_ret = FALSE; /* the event has not been handled */
                    break;  
            }
        }
        break;
        /* Wrong state */
        default:
            func_ret = FALSE;
            break;
    }
    return (func_ret);
}

#ifdef _USE_CUT_TEAKLIT_CODE
/*------------------------------------------------------------------------------
* Function...: aud_drv_restart_hf
* Return.....: -
* Description: Called after each TCH_24 command.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void aud_drv_restart_hf(void)
{
}
#endif
/*------------------------------------------------------------------------------
* Function...: aud_drv_activate_hf
* Return.....: -
* Description: Decides whether to program the HF block, and what to write.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void aud_drv_activate_hf(void)
{
    BOOL activate = FALSE; /* should the hands-free block be activated? */

    /* if midi player and TTY are idle but another resource is active, activation is allowed */
    if( (S_idle==aud_get_SM_state(aud_resource_midi_player)) && (S_idle==aud_get_SM_state(aud_resource_tty)) )
    {
        /*[Begin]some baseband resource need not handsfree.[lvwenhua-2010/7/28]*/
        #if 0
        if(aud_one_baseband_SM_active())
        #endif
        if((aud_get_SM_state(aud_resource_speech) == S_active)
            || (aud_get_SM_state(aud_resource_record_vm) == S_active)
            || (aud_get_SM_state(aud_resource_I2S2_Tx) == S_active)
            || (aud_get_SM_state(aud_resource_I2S2_Rx) == S_active))
        {
            activate=TRUE;
        }
        /*[End] [lvwenhua-2010/7/28]*/
    }

    /* decide whether to shut down, restart or initialise */
    if(FALSE == activate)
    {
        /*[Begin]if handsfree has been closed, need not close again.  [lvwenhua-2010/7/28]*/
        if(aud_dsp_hf_on_parms.mode != 0)
        {
            aud_dsp_hf_on_parms.mode = 0; /* hands-free off */
            DSP_ASYNC_HF_ON(&aud_dsp_hf_on_parms);
        }
        /*[End] [lvwenhua-2010/7/28]*/
        
        aud_ear_HF_active = FALSE;
    }
    else
    { /* the hands-free block should be activated */
        aud_ear_dsp_program_hands_free();  /* program the parameters for the hands-free block */
        if(aud_ear_HF_active)              /* the hands-free block is already active, so just restart */
        {
            aud_ear_dsp_start_hands_free(aud_path_settings_result.uplink_parms.p_hf->hf_algorithm_restart);
        }
        else                               /* the hands-free block was shut down, so initialise it */
        {
            aud_ear_dsp_start_hands_free(aud_path_settings_result.uplink_parms.p_hf->hf_algorithm_init);
        }
        aud_ear_HF_active = TRUE;               
    }
}
#ifdef _USE_CUT_TEAKLIT_CODE
/*------------------------------------------------------------------------------
* Function...: aud_get_HF_active
* Return.....: TRUE/FALSE
* Description: Returns TRUE if the hands-free block is active.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
BOOL aud_get_HF_active(void)
{
    return (aud_ear_HF_active);
}

/*------------------------------------------------------------------------------
* Function...: aud_get_EC_NR_mask
* Return.....: the current aud_ear_EC_NR_mask
* Description: Returns the current aud_ear_EC_NR_mask.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S16 aud_get_EC_NR_mask(void)
{
    return (aud_ear_EC_NR_mask);
}
#endif
