/**
 *   Copyright (C) Infineon Technologies Denmark A/S. All rights reserved.
 *
 * This document contains proprietary information belonging to Infineon
 * Technologies Denmark A/S. Passing on and copying of this document, use
 * and communication of its contents is not permitted without prior written
 * authorisation.
 *
 * Description:
 *   Main State machine for the audio driver. The file also include the interface
 *   functions
 *
 * Revision Information:
 *   File name: \dwddrv\AUD\src\aud_intf.c
 *   Version: \main\217
 *   Date: 2007-10-02 09:06:28
 *   Responsible: johansek
 *   Comment:
 *     Removed compiler warning.
 */

/*******************************************************************************
*
*                               Include files
*
*******************************************************************************/
//#include "scttypes.h"           /* must be included in all files */
//#include "ms.h"                 /* must be included in all files */
#include "bastypes.h"           /* must be included in all files */
//#include "ms.h" 
#include "ms_os.h"              /* operating system interface wrapper */
#if !defined (XNOSTATICSTRUCTURE)
//#include "sig_code.h"
#endif
#include "aud_drv.h"            /* the interface which this file implements a part of */
#include "aud_com.h"            /* types and functions used only within the audio driver */
#include "aud_data.h"           /* data for path parallelism and I2S modes */
#include "aud_i2s.h"            /* definition of I2S modes */
#include "drvs_teak.h"

//#include "ms_crit.h"             //huojishan 2008.01.01
//#include "osa_api.h"           //huojishan 2008.01.01
/*******************************************************************************
*
*                             Type defines
*
*******************************************************************************/
/* In a reentrant mutex, the same thread can acquire the lock multiple times */

/*Change for porting.[Begin] [LvWenhua-2008/4/23]*/
#if 0
typedef struct {
    os_sem_t  sem;       /* the OS level lock, which is acquired the first time the mutex is claimed */
    void*     p_caller;  /* pointer to the PCU of the process/task/thread that holds the mutex */
    S8        depth;     /* how many times has the process claimed the mutex */
} aud_intf_reentrant_mutex_type;
#endif
typedef struct {
    ZOSS_SEMAPHORE_ID  sem;       /* the OS level lock, which is acquired the first time the mutex is claimed */
    void*     p_caller;  /* pointer to the PCU of the process/task/thread that holds the mutex */
    S8        depth;     /* how many times has the process claimed the mutex */
} aud_intf_reentrant_mutex_type;
/*[End] [LvWenhua-2008/4/23]*/

/*******************************************************************************
*
*                              Local variables
*
*******************************************************************************/
static U8         aud_intf_uplinkpath_status[32];  //aud_uplink_source_enum     /* contains the currently active paths */
static U8         aud_intf_downlinkpath_status[32];//aud_downlink_source_enum     /* contains the currently active paths */
static aud_intf_reentrant_mutex_type  aud_intf_mutex; /* reentrant mutex for preventing that two processes have concurrent access to the audio driver */
static BOOL                           aud_intf_mutex_first = FALSE;         /* has the reentrant mutex been initialised? */
static BOOL                           aud_intf_path_driver_update = TRUE;   /* is path update enabled? */

aud_resource_table_type               aud_resource_table[aud_resource_end]; /* resource allocation status */
aud_I2S_status_type                   aud_I2S_status;
#if defined (aud_old_I2S_interface)
  //U16 aud_I2S2_mode_pointer = 0;
#endif

/*******************************************************************************
*
*                             External variables
*
*******************************************************************************/
extern BOOL aud_i2s_validation_matrix[aud_I2S_entry_point_end][aud_I2S_Master_Slave_end][aud_I2S_mode_end][aud_I2S_transmission_mode_end][aud_I2S_samplerate_end];

/*******************************************************************************
*
*                      Prototypes for internal functions
*
*******************************************************************************/
static S8 aud_intf_check_handle(U8 handle, aud_resource_enum resource);
static S8 aud_intf_check_for_conflict(aud_resource_enum resource);
static U8 aud_intf_allow_sample_rate(U8 required, U8 allowed);
static void aud_intf_create_mutex(void);
static void aud_intf_claim_mutex(void);
static void aud_intf_release_mutex(void);
 
/*******************************************************************************
*
*                Resource allocation related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_allocate_resource
* Return.....: a handle to the resource (>0) or a return code (<0)
* Description: Reserve the requested resource if it is free.
* Modified...: 07.08.2007 by KBJ (status=0 means not allocated)
------------------------------------------------------------------------------*/
S8 AUD_allocate_resource(
    U16 id,                     /* ID of the calling task */
    aud_resource_enum resource, /* the resource to be allocated */
    aud_priority_enum priority) /* not used */
{
    S8 handle;
	
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(!(aud_hw_parms.audio_hw & 1<<resource)) /* when bit number <resource> is not set, the resource is not available in this system */
    {
        handle = aud_rc_resource_conflict;
    }
    /* Check parameters */
    else if(priority >= aud_priority_end || resource >= aud_resource_end)
    {
        handle = aud_rc_parameter_out_of_range;
    }
    else if(0 == aud_resource_table[resource].status)
    {
        if(aud_intf_check_for_conflict(resource))
        {
            handle = aud_rc_resource_conflict;
        }
		/*klocwork 3 INVARIANT_CONDITION.UNREACH delete else if*/
		#if 0
        else if(!(aud_hw_parms.audio_hw & 1<<resource)) /* when bit number <resource> is not set, the resource is not available in this system */
        {
            handle = aud_rc_resource_conflict;
        }
		#endif
        else
        {
            aud_resource_table[resource].status = handle = resource+1;
            aud_resource_table[resource].caller_id = id;
        }
        aud_send_mail(E_resource_allocate, 0, resource, priority, 0, 0, 0, 0, 0, 0, NULL, NULL); /* dummy - for traceability */
		
    }
    else
    {
          handle = aud_rc_resource_in_use;
    }
    #if defined (AUD_YAMAHA_MAPI)
    if(aud_resource_mapi == resource)
    {
        aud_disable_standby(aud_resource_mapi);
        aud_send_mail(E_aud_mapi_allocate, aud_resource_table[aud_resource_mapi].caller_id,
                 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
   #endif
    if(handle<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_resource_allocate, handle, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (handle);
} /* End of function AUD_allocate_resource */

/*------------------------------------------------------------------------------
* Function...: AUD_release_resource
* Return.....: return code
* Description: The resource is released and can be used by another task.
*              The resource is powered down when released.
* Modified...: 07.08.2007 by KBJ (status=0 means not allocated)
------------------------------------------------------------------------------*/
S8 AUD_release_resource(
    U8 handle)  /* the handle to the resource to be released */
{
    S8 res; /* resource */
    S8 result;
   
    if(aud_resource_table[handle-1].status == handle)
    {
        res = handle-1;
    }
    else
    {
        res = aud_rc_handle_not_used;
    }
#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_function_interface, 4, 0, /* LLT params */
                  'd', 0x06,                                                    /* LIS params */
                  aud_log_lis_raw_data, aud_log_lis_display_data, aud_log_llt_trace,        /* LIS params */
                  (U32)E_resource_release, (U32)handle, (U32)res, (U32)E_audio_llt_type_id_3 );
#endif
	
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(res<0 || res>=aud_resource_end) /* parameter range check */
    {
        result = aud_rc_parameter_out_of_range;
    }
    else
    {
        switch(res)
        {
            case aud_resource_speech:
                aud_send_mail(E_speech_release_disable, aud_resource_table[aud_resource_speech].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
            case aud_resource_tone_generator:
                aud_send_mail(E_tone_release_stop, aud_resource_table[aud_resource_tone_generator].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
            case aud_resource_ringer:
                aud_send_mail(E_ringer_release_stop, aud_resource_table[aud_resource_ringer].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
            case aud_resource_vibrator:
                aud_send_mail(E_vibrator_release_disable, aud_resource_table[aud_resource_vibrator].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
            case aud_resource_amplifier:
                aud_send_mail(E_amp_release_disable, aud_resource_table[aud_resource_amplifier].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
#if defined (FM_RADIO_PRESENT)
            case aud_resource_radio:
                aud_send_mail(E_radio_release_stop, aud_resource_table[aud_resource_radio].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
#endif
            case aud_resource_record_vm:
                aud_send_mail(E_vms_release_stop_voice_recording, aud_resource_table[aud_resource_record_vm].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
            case aud_resource_playback_vm:
                aud_send_mail(E_vms_release_stop_playback, aud_resource_table[aud_resource_playback_vm].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
#if defined (MP3_PRESENT)
            case aud_resource_playback_mp3:
                aud_send_mail(E_mp3_release_stop, aud_resource_table[aud_resource_playback_mp3].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
#endif
            case aud_resource_PCM_channel:
                aud_send_mail(E_I2S_release_disable, aud_resource_table[aud_resource_PCM_channel].caller_id, aud_resource_I2S2_Rx,0,0,0,0,0,0,0,NULL,NULL);
                aud_send_mail(E_I2S_release_disable, aud_resource_table[aud_resource_PCM_channel].caller_id, aud_resource_I2S2_Tx,0,0,0,0,0,0,0,NULL,NULL);
                break;
#if defined (INTERNAL_POLYRINGER)
            case aud_resource_midi_player:
                aud_send_mail(E_midi_release_stop, aud_resource_table[aud_resource_midi_player].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
#endif
            case aud_resource_tty:
                aud_send_mail(E_tty_release_disable, aud_resource_table[aud_resource_tty].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
            case aud_resource_playback_pcm:
                aud_send_mail(E_pcm_release_stop_playback, aud_resource_table[aud_resource_playback_pcm].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
            case aud_resource_record_pcm:
                aud_send_mail(E_pcm_release_stop_recording, aud_resource_table[aud_resource_record_pcm].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
#if defined (AUD_YAMAHA_MAPI)
            case aud_resource_mapi:
                aud_enable_standby(aud_resource_mapi);
                aud_send_mail(E_aud_mapi_release_terminate, aud_resource_table[aud_resource_mapi].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
#endif
            case aud_resource_inband:
                aud_send_mail(E_inband_release_stop, aud_resource_table[aud_resource_inband].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
            case aud_resource_smpower_backspeaker:
                aud_send_mail(E_smpower_release_disable, aud_resource_table[aud_resource_smpower_backspeaker].caller_id, 0,0,0,0,0,0,0,0,NULL,NULL);
                break;
#if defined (SBC_ENCODER)
            case aud_resource_sbc_encoder:
                aud_send_mail(E_sbc_release_stop_encoder, aud_resource_table[aud_resource_sbc_encoder].caller_id, 0,0,0,0,0,0,0,0, NULL, NULL);
                break;
#endif
            case aud_resource_I2S2_Rx:
                aud_send_mail(E_I2S_release_disable, aud_resource_table[aud_resource_I2S2_Rx].caller_id, aud_resource_I2S2_Rx,0,0,0,0,0,0,0,NULL,NULL);
                break;

            case aud_resource_I2S2_Tx:
                aud_send_mail(E_I2S_release_disable, aud_resource_table[aud_resource_I2S2_Tx].caller_id, aud_resource_I2S2_Tx,0,0,0,0,0,0,0,NULL,NULL);
                break;
            /*** make sure to update this function when a resource is added ***/
            default:
                /* already handled by parameter check */
                break;
        } /* end switch */
        result = aud_rc_ok;
        aud_resource_table[res].status = 0; /* Indicate release of resource */
        aud_send_mail(E_resource_release, 0, res, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
		
    } /* end else */
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (result);
} /* End of function AUD_release_resource */
#ifdef _USE_CUT_TEAKLIT_CODE
/*******************************************************************************
*
*                      Path related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_get_downlnk_parallel_path_configuration
* Return.....: The paths that can run in parallel with the given path
* Description: Gets the paths that can run in parallel with the given path
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
U32 AUD_get_downlnk_parallel_path_configuration(
    aud_downlink_source_enum path)  /* the path for which the parallel path configuration is requested */
{
    U32 res=0;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(path>=aud_downlink_source_end)
    {
        res = ((U32)aud_rc_parameter_out_of_range);
        aud_send_mail(E_interface_request_error, 0, E_get_downlnk_parallel_path_configuration, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    else
    {
        U8 i=0;
        while(aud_Parallel_Downlink_Paths[i]!=0)
        {
            if(aud_Parallel_Downlink_Paths[i] & (1<<path))
                res |= aud_Parallel_Downlink_Paths[i];
            i++;
        }
        aud_send_mail(E_get_downlnk_parallel_path_configuration,0,path,res,0,0,0,0,0,0,NULL,NULL); /* dummy - for traceability */
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_get_uplink_parallel_path_configuration
* Return.....: The paths that can run in parallel with the given path
* Description: Gets the paths that can run in parallel with the given path
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
U32 AUD_get_uplink_parallel_path_configuration(
    aud_uplink_source_enum path)    /* the path for which the parallel path configuration is requested */
{
    U32 res=0;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if((path>=aud_uplink_source_end))
    {
        res = ((U32)aud_rc_parameter_out_of_range);
        aud_send_mail(E_interface_request_error, 0, E_get_uplink_parallel_path_configuration, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    else
    {
        U8 i=0;
        while(aud_Parallel_Uplink_Paths[i]!=0)
        {
            if(aud_Parallel_Uplink_Paths[i] & (1<<path))
                res |= aud_Parallel_Uplink_Paths[i];
            i++;
        }
        aud_send_mail(E_get_uplink_parallel_path_configuration,0,path,res,0,0,0,0,0,0,NULL,NULL); /* dummy - for traceability */
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}
#endif
/*------------------------------------------------------------------------------
* Function...: AUD_add_uplinkpath
* Return.....: return code
* Description: If no conflict is found, the path is added to aud_intf_uplinkpath_status.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_add_uplinkpath(
    aud_uplink_source_enum path)    /* the uplink path to add */
{
    U32 parm1=0,parm2=0,parm3=0,parm4=0,parm5=0,parm6=0,parm7=0,parm8=0;
    U8 m=0, number_of_set_paths=0, n=0;
	
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */

    /* Range check */
    if(path >= aud_uplink_source_end)
    {
        aud_send_mail(E_interface_request_error, 0, E_update_uplinkpath, (U32)aud_rc_parameter_out_of_range, 0, 0, 0, 0, 0, 0, NULL, NULL);
        aud_intf_release_mutex(); /* release audio interface for other processes */
        return (aud_rc_parameter_out_of_range);
    }
    if(aud_setting.p_aud_audio_uplink_parms[path]==NULL) /* path does not exist on target */
    {
        aud_send_mail(E_interface_request_error, 0, E_update_uplinkpath, (U32)aud_rc_path_addition_conflict, 0, 0, 0, 0, 0, 0, NULL, NULL);
        aud_intf_release_mutex(); /* release audio interface for other processes */
        return (aud_rc_not_supported);
    }
    /* Is path already in use */
    for(m=0; m<aud_uplink_source_end; m++)
    {
        /* Exiting when end of struct is reached */
        if(aud_uplink_source_end == aud_intf_uplinkpath_status[m])
        {
            number_of_set_paths=m;
            break;
        }
        if(aud_intf_uplinkpath_status[m] == path) /* path has already been added */
        {
            aud_send_mail(E_interface_request_error, 0, E_update_uplinkpath, (U32)aud_rc_path_addition_conflict, 0, 0, 0, 0, 0, 0, NULL, NULL);
            aud_intf_release_mutex(); /* release audio interface for other processes */
            return (aud_rc_path_in_use); 
        }
    }
    /* Check for sample rate conflicts */
    if(FALSE == aud_intf_allow_sample_rate(aud_setting.p_aud_audio_uplink_parms[path]->required_sample_rate, aud_setting.p_aud_audio_uplink_parms[path]->allowed_sample_rates))
    {
        aud_send_mail(E_interface_request_error, 0, E_update_uplinkpath, (U32)aud_rc_path_addition_conflict, 0, 0, 0, 0, 0, 0, NULL, NULL);
        aud_intf_release_mutex(); /* release audio interface for other processes */
        return (aud_rc_path_addition_conflict);
    }
    /* Checking for conflict with existing paths */
    for(n=0; n<number_of_set_paths; n++)
    {
        parm1 |= (1<<aud_intf_uplinkpath_status[n]);
    }
    parm1 |= (1<<path);
    while(aud_Parallel_Uplink_Paths[parm2] && 0==parm3)
    {
        if((parm1 & aud_Parallel_Uplink_Paths[parm2]) == parm1)
        {
            parm3=1; /* path may run in parallel */
        }
        parm2++;
    }

    if(0 == parm3)
    {
        aud_send_mail(E_interface_request_error, 0, E_update_uplinkpath, (U32)aud_rc_path_addition_conflict, 0, 0, 0, 0, 0, 0, NULL, NULL);
        aud_intf_release_mutex(); /* release audio interface for other processes */
        return (aud_rc_path_addition_conflict); /* returning error because path may not run in parallel with an existing path */
    }
    aud_intf_uplinkpath_status[number_of_set_paths]=path;

    if(aud_intf_path_driver_update) /*klocwork 3  INVARIANT_CONDITION.GEN delete TRUE*/
    {
        memcpy(&parm1, &aud_intf_uplinkpath_status[0], sizeof(parm1));
        memcpy(&parm2, &aud_intf_uplinkpath_status[4], sizeof(parm2));
        memcpy(&parm3, &aud_intf_uplinkpath_status[8], sizeof(parm3));
        memcpy(&parm4, &aud_intf_uplinkpath_status[12], sizeof(parm4));
        memcpy(&parm5, &aud_intf_uplinkpath_status[16], sizeof(parm5));
        memcpy(&parm6, &aud_intf_uplinkpath_status[20], sizeof(parm6));
        memcpy(&parm7, &aud_intf_uplinkpath_status[24], sizeof(parm7));
        memcpy(&parm8, &aud_intf_uplinkpath_status[28], sizeof(parm8));
        aud_send_mail(E_update_uplinkpath,0,parm1,parm2,parm3,parm4,parm5,parm6,parm7,parm8,NULL,NULL); 
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (aud_rc_ok);
}

/*------------------------------------------------------------------------------
* Function...: AUD_remove_uplinkpath
* Return.....: return code
* Description: The path is removed from aud_intf_uplinkpath_status.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_remove_uplinkpath(
    aud_uplink_source_enum path)    /* the uplink path to remove */
{
    U8 m=0;
    U32 parm1=0,parm2=0,parm3=0,parm4=0,parm5=0,parm6=0,parm7=0,parm8=0;
	
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */

    if(path<aud_uplink_source_end)
    {
        /* Finding the position of the path in the aud_intf_uplinkpath_status */
        for(m=0;m<=aud_uplink_source_end;m++)
        {
            if(aud_intf_uplinkpath_status[m]==path)
            {
                break;
            }
        }
        /* If the path is not set, error is returned */
        if(m==aud_uplink_source_end)
        {
            aud_send_mail(E_interface_request_error, 0, E_update_uplinkpath, (U32)aud_rc_path_removal_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
            aud_intf_release_mutex(); /* release audio interface for other processes */
            return (aud_rc_path_removal_error);
        }
        /* Moving the last path one position up */
        for(; m<=aud_uplink_source_end; m++)
        {
            aud_intf_uplinkpath_status[m]=aud_intf_uplinkpath_status[m+1];
            /* Exit loop when last path is reach */
            if(aud_uplink_source_end == aud_intf_uplinkpath_status[m+1])
            {
                break;
            }
        }
    }
    /* Clear all paths */
    else /*klocwork 3 INVARIANT_CONDITION.GEN delete (aud_uplink_source_end == path)*/
    {
        for(m=0; m<32; m++)
        {
            aud_intf_uplinkpath_status[m]=aud_uplink_source_end;
        }
    }
	/*klocwork 3 INVARIANT_CONDITION.GEN delete TRUE*/
	#if 0
    else
    {
        aud_send_mail(E_interface_request_error, 0, E_update_uplinkpath, (U32)aud_rc_parameter_out_of_range, 0, 0, 0, 0, 0, 0, NULL, NULL);
        aud_intf_release_mutex(); /* release audio interface for other processes */
        return (aud_rc_parameter_out_of_range);
    }
	#endif
    if(aud_intf_path_driver_update)  /*klocwork 3  INVARIANT_CONDITION.GEN delete TRUE*/
    {
        memcpy(&parm1, &aud_intf_uplinkpath_status[0], sizeof(parm1));
        memcpy(&parm2, &aud_intf_uplinkpath_status[4], sizeof(parm2));
        memcpy(&parm3, &aud_intf_uplinkpath_status[8], sizeof(parm3));
        memcpy(&parm4, &aud_intf_uplinkpath_status[12], sizeof(parm4));
        memcpy(&parm5, &aud_intf_uplinkpath_status[16], sizeof(parm5));
        memcpy(&parm6, &aud_intf_uplinkpath_status[20], sizeof(parm6));
        memcpy(&parm7, &aud_intf_uplinkpath_status[24], sizeof(parm7));
        memcpy(&parm8, &aud_intf_uplinkpath_status[28], sizeof(parm8));
        aud_send_mail(E_update_uplinkpath,0,parm1,parm2,parm3,parm4,parm5,parm6,parm7,parm8,NULL,NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (aud_rc_ok);
}

/*------------------------------------------------------------------------------
* Function...: AUD_add_downlinkpath
* Return.....: return code
* Description: If no conflict is found, the path is added to aud_intf_downlinkpath_status
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_add_downlinkpath(
    aud_downlink_source_enum path)  /* the downlink path to add */
{
    U32 parm1=0,parm2=0,parm3=0,parm4=0,parm5=0,parm6=0,parm7=0,parm8=0;
    U8 m=0, number_of_set_paths=0, n=0;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */

    /* Range check */
    if(path >= aud_downlink_source_end)
    {
        aud_send_mail(E_interface_request_error, 0, E_update_downlinkpath, (U32)aud_rc_parameter_out_of_range, 0, 0, 0, 0, 0, 0, NULL, NULL);
        aud_intf_release_mutex(); /* release audio interface for other processes */
        return (aud_rc_parameter_out_of_range);
    }
    if(NULL == aud_setting.p_aud_audio_downlink_parms[path]) /* path does not exist on target */
    {
        aud_send_mail(E_interface_request_error, 0, E_update_downlinkpath, (U32)aud_rc_path_addition_conflict, 0, 0, 0, 0, 0, 0, NULL, NULL);
        aud_intf_release_mutex(); /* release audio interface for other processes */
        return (aud_rc_not_supported);
    }
    /* Is path already in use */
    for(m=0; m<aud_downlink_source_end; m++)
    {
        /* Exiting when end of struct are reach */
        if(aud_downlink_source_end == aud_intf_downlinkpath_status[m])
        {
            number_of_set_paths=m;
            break;
        }
        if(aud_intf_downlinkpath_status[m]==path)
        {   
            aud_send_mail(E_interface_request_error, 0, E_update_downlinkpath, (U32)aud_rc_path_addition_conflict, 0, 0, 0, 0, 0, 0, NULL, NULL);
            aud_intf_release_mutex(); /* release audio interface for other processes */
            return (aud_rc_path_in_use);
        }
    }
    /* Check for sample rate conflicts */
    if(FALSE == aud_intf_allow_sample_rate(0/*no requirements*/, aud_setting.p_aud_audio_downlink_parms[path]->allowed_sample_rates))
    {
        aud_send_mail(E_interface_request_error, 0, E_update_downlinkpath, (U32)aud_rc_path_addition_conflict, 0, 0, 0, 0, 0, 0, NULL, NULL);
        aud_intf_release_mutex(); /* release audio interface for other processes */
        return (aud_rc_path_addition_conflict);
    }
    /* Checking for conflict with existing paths */
    for(n=0; n<number_of_set_paths; n++)
    {
        parm1 |= (1<<aud_intf_downlinkpath_status[n]);
    }
    parm1 |= (1<<path);
    while(aud_Parallel_Downlink_Paths[parm2] && 0==parm3)
    {
        if((parm1 & aud_Parallel_Downlink_Paths[parm2])==parm1)
        {
            parm3=1; /* path may run in parallel */
        }
        parm2++;
    }
    if(0 == parm3)
    {
        aud_send_mail(E_interface_request_error, 0, E_update_downlinkpath, (U32)aud_rc_path_addition_conflict, 0, 0, 0, 0, 0, 0, NULL, NULL);
        aud_intf_release_mutex(); /* release audio interface for other processes */
        return (aud_rc_path_addition_conflict); /* returning error because path may not run in parallel with an existing path */
    }
    aud_intf_downlinkpath_status[m]=path;

    if(aud_intf_path_driver_update) /*klocwork 3 INVARIANT_CONDITION.GEN delete TRUE*/
    {
        memcpy(&parm1, &aud_intf_downlinkpath_status[0], sizeof(parm1));
        memcpy(&parm2, &aud_intf_downlinkpath_status[4], sizeof(parm2));
        memcpy(&parm3, &aud_intf_downlinkpath_status[8], sizeof(parm3));
        memcpy(&parm4, &aud_intf_downlinkpath_status[12], sizeof(parm4));
        memcpy(&parm5, &aud_intf_downlinkpath_status[16], sizeof(parm5));
        memcpy(&parm6, &aud_intf_downlinkpath_status[20], sizeof(parm6));
        memcpy(&parm7, &aud_intf_downlinkpath_status[24], sizeof(parm7));
        memcpy(&parm8, &aud_intf_downlinkpath_status[28], sizeof(parm8));
        aud_send_mail(E_update_downlinkpath,0,parm1,parm2,parm3,parm4,parm5,parm6,parm7,parm8,NULL,NULL);
		
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (aud_rc_ok);
}

/*------------------------------------------------------------------------------
* Function...: AUD_remove_downlinkpath
* Return.....: return code
* Description: The path is removed from aud_intf_downlinkpath_status
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_remove_downlinkpath(
    aud_downlink_source_enum path)  /* the downlink path to remove */
{
    U8 m=0;
    U32 parm1=0,parm2=0,parm3=0,parm4=0,parm5=0,parm6=0,parm7=0,parm8=0;
	
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */

    if(path<aud_downlink_source_end)
    {
        /* Finding the position of the path in the aud_intf_downlinkpath_status */
        for(m=0; m<=aud_downlink_source_end; m++)
        {
            if(aud_intf_downlinkpath_status[m] == path)
            {
                break;
            }
        }
        /* If the path is not set, error is returned */
        if(aud_downlink_source_end == m)
        {
            aud_send_mail(E_interface_request_error, 0, E_update_downlinkpath, (U32)aud_rc_path_removal_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
            aud_intf_release_mutex(); /* release audio interface for other processes */
            return (aud_rc_path_removal_error);
        }
        /* Moving the last path one position up */
        for(; m<=aud_downlink_source_end; m++)
        {
            aud_intf_downlinkpath_status[m]=aud_intf_downlinkpath_status[m+1];
            /* Exit loop when last path is reach  */
            if(aud_downlink_source_end == aud_intf_downlinkpath_status[m+1])
            {
                break;
            }
        }
    }
    /* Clear all paths */
    else /*klocwork 3 INVARIANT_CONDITION.UNREACH delete (path == aud_downlink_source_end)*/
    {
        for(m=0; m<32; m++)
        {
            aud_intf_downlinkpath_status[m] = aud_downlink_source_end;
        }
    }
	/*klocwork 3 INVARIANT_CONDITION.GEN delete else*/
	#if 0
    else
    {
        aud_send_mail(E_interface_request_error, 0, E_update_downlinkpath, (U32)aud_rc_parameter_out_of_range, 0, 0, 0, 0, 0, 0, NULL, NULL);
        aud_intf_release_mutex(); /* release audio interface for other processes */
        return (aud_rc_parameter_out_of_range);
    }
	#endif
    if(aud_intf_path_driver_update) /*klocwork 3 INVARIANT_CONDITION.GEN delete TRUE*/
    {
        memcpy(&parm1, &aud_intf_downlinkpath_status[0], sizeof(parm1));
        memcpy(&parm2, &aud_intf_downlinkpath_status[4], sizeof(parm2));
        memcpy(&parm3, &aud_intf_downlinkpath_status[8], sizeof(parm3));
        memcpy(&parm4, &aud_intf_downlinkpath_status[12], sizeof(parm4));
        memcpy(&parm5, &aud_intf_downlinkpath_status[16], sizeof(parm5));
        memcpy(&parm6, &aud_intf_downlinkpath_status[20], sizeof(parm6));
        memcpy(&parm7, &aud_intf_downlinkpath_status[24], sizeof(parm7));
        memcpy(&parm8, &aud_intf_downlinkpath_status[28], sizeof(parm8));
        aud_send_mail(E_update_downlinkpath,0,parm1,parm2,parm3,parm4,parm5,parm6,parm7,parm8,NULL,NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (aud_rc_ok);
}
#ifdef _USE_CUT_TEAKLIT_CODE
/*------------------------------------------------------------------------------
* Function...: AUD_Pathupdate_Enable
* Return.....: aud_rc_ok
* Description: Enables path update, i.e. that paths are being added to/removed
*              from the driver process when the interface is updated.
*              Writes the current interface settings to the driver.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_Pathupdate_Enable(void)
{
    U32 parm1=0,parm2=0,parm3=0,parm4=0,parm5=0,parm6=0,parm7=0,parm8=0;

    aud_intf_path_driver_update = TRUE;
    aud_send_mail(E_Pathupdate_Enable,0,0,0,0,0,0,0,0,0,NULL,NULL); /* dummy - for traceability */

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */

    /* Updating Uplink paths */
    memcpy(&parm1, &aud_intf_uplinkpath_status[0], sizeof(parm1));
    memcpy(&parm2, &aud_intf_uplinkpath_status[4], sizeof(parm2));
    memcpy(&parm3, &aud_intf_uplinkpath_status[8], sizeof(parm3));
    memcpy(&parm4, &aud_intf_uplinkpath_status[12], sizeof(parm4));
    memcpy(&parm5, &aud_intf_uplinkpath_status[16], sizeof(parm5));
    memcpy(&parm6, &aud_intf_uplinkpath_status[20], sizeof(parm6));
    memcpy(&parm7, &aud_intf_uplinkpath_status[24], sizeof(parm7));
    memcpy(&parm8, &aud_intf_uplinkpath_status[28], sizeof(parm8));
    aud_send_mail(E_update_uplinkpath,0,parm1,parm2,parm3,parm4,parm5,parm6,parm7,parm8,NULL,NULL);

    /* Updating Downlink paths */
    memcpy(&parm1, &aud_intf_downlinkpath_status[0], sizeof(parm1));
    memcpy(&parm2, &aud_intf_downlinkpath_status[4], sizeof(parm2));
    memcpy(&parm3, &aud_intf_downlinkpath_status[8], sizeof(parm3));
    memcpy(&parm4, &aud_intf_downlinkpath_status[12], sizeof(parm4));
    memcpy(&parm5, &aud_intf_downlinkpath_status[16], sizeof(parm5));
    memcpy(&parm6, &aud_intf_downlinkpath_status[20], sizeof(parm6));
    memcpy(&parm7, &aud_intf_downlinkpath_status[24], sizeof(parm7));
    memcpy(&parm8, &aud_intf_downlinkpath_status[28], sizeof(parm8));
    aud_send_mail(E_update_downlinkpath,0,parm1,parm2,parm3,parm4,parm5,parm6,parm7,parm8,NULL,NULL);

    aud_intf_release_mutex(); /* release audio interface for other processes */

    return (aud_rc_ok);
}

/*------------------------------------------------------------------------------
* Function...: AUD_Pathupdate_Disable
* Return.....: aud_rc_ok
* Description: Disables path update, i.e. when paths are added to/removed
*              from the interface, the driver process is unaffected.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_Pathupdate_Disable(void)
{
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    aud_intf_path_driver_update = FALSE;
    aud_send_mail(E_Pathupdate_Disable,0,0,0,0,0,0,0,0,0,NULL,NULL); /* dummy - for traceability */
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (aud_rc_ok);
}

/*------------------------------------------------------------------------------
* Function...: AUD_set_eq_vol_threshold
* Return.....: return code
* Description: Updates path's threshold value (used for choice of equalizer setting).
* Created....: 21.09.2007 by TBJ
* Modified...: 02.10.2007 by KBJ (removed warnings)
------------------------------------------------------------------------------*/
S8 AUD_set_eq_vol_threshold(
	aud_downlink_source_enum path,          /* path (aud_downlink_source_end sets eq_volume_threshold of all paths) */
	aud_volume_enum eq_volume_threshold)    /* the threshold where the other filter values should be used */
{ 
	S8 res;
	aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
	
	/* Range check */
	if(path > aud_downlink_source_end || eq_volume_threshold >= aud_volume_end)
	{
		res = aud_rc_parameter_out_of_range;
	}
	else
	{
		aud_send_mail(E_aud_eep_write_eq_vol_threshold, 0, path, eq_volume_threshold, 0, 0, 0, 0, 0, 0, NULL, NULL);
		res = aud_rc_ok;
	} 
	if(res<aud_rc_ok)
	{
		aud_send_mail(E_interface_request_error, 0, E_aud_eep_write_eq_vol_threshold, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
	}

	aud_intf_release_mutex(); /* release audio interface for other processes */
	return (res);
}
#endif
/*******************************************************************************
*
*                    Volume related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_set_resource_volume
* Return.....: return code
* Description: Controls individual gain cells for each resource.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_set_resource_volume(
    U8 handle,              /* the handle to the resource for which the volume is to be changed */
    aud_volume_enum volume) /* the new volume level */
{
    S8 res;
    aud_resource_enum current_resource = aud_resource_speech;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    for(current_resource=aud_resource_speech; current_resource<aud_resource_end; current_resource++)
    {
        if(aud_rc_ok == aud_intf_check_handle(handle, current_resource)) /* find the resource the handle is for */
        {
            break;
        }
    }
    if(aud_resource_end == current_resource) /* resource not found */
    {
        res = aud_rc_sharing_violation;
    }
    else if(volume >= aud_volume_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        if(current_resource==aud_resource_PCM_channel) /* PCM channel no longer exists, replaced by I2S2 resources */
        {
            current_resource=aud_resource_I2S2_Rx;
        }
        aud_send_mail(E_set_resource_volume,aud_resource_table[current_resource].caller_id,current_resource,volume,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_set_resource_volume, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_set_master_volume
* Return.....: return code
* Description: Controls master volume for all resources.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_set_master_volume(
    aud_volume_enum volume) /* new master volume level */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
	
    if((volume >= aud_volume_end))
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_set_master_volume, 0, volume, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_set_master_volume, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }

    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_mute_resource
* Return.....: return code
* Description: Mutes an individual resource. Previous level can be restored.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_mute_resource(
    U8 handle,                          /* handle to the resource to be muted */
    aud_mute_enum enable_disable,       /* whether muting should be enabled or disabled */
    aud_ul_dl_direction_enum direction) /* which directions (uplink/downlink) the (un)muting is for */
{
    S8 res;
    aud_resource_enum current_resource = aud_resource_speech;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    for(current_resource=aud_resource_speech; current_resource<aud_resource_end; current_resource++)
    {
        if(aud_resource_table[current_resource].status == handle ||aud_resource_table[current_resource].test_handle == handle)
        {
            break;
        }
    }
    if(aud_resource_end == current_resource) /* Resource not found */
    {
        res = aud_rc_sharing_violation;
    }
    else if(direction >= aud_direction_resource_end || enable_disable >= aud_mute_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        if( (current_resource == aud_resource_playback_pcm) ||
            (current_resource == aud_resource_midi_player) ||
            (current_resource == aud_resource_playback_mp3) ||
            (current_resource == aud_resource_ringer) ||
            (current_resource == aud_resource_mapi))
        {
            direction = aud_direction_resource_updownlink;
        }
        if(current_resource == aud_resource_PCM_channel) /* PCM channel no longer exists, replaced by I2S2 resources */
        {
            current_resource = aud_resource_I2S2_Rx;
        }
        aud_send_mail(E_mute_resource,aud_resource_table[current_resource].caller_id,current_resource,enable_disable,direction,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_mute_resource, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}
#ifdef _USE_CUT_TEAKLIT_CODE
/*------------------------------------------------------------------------------
* Function...: AUD_mute_master
* Return.....: return code
* Description: Mutes the master volume, meaning that no audio can be heard.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_mute_master(
    aud_mute_enum enable_disable)   /* whether muting should be enabled or disabled */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(enable_disable >= aud_mute_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_mute_master, 0, enable_disable, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_mute_master, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}
#endif
/*******************************************************************************
*
*              Speech and hands-free related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_speech_enable
* Return.....: return code
* Description: Enables the speech resource for voice calls.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_speech_enable(
    U8 handle)  /* handle to the speech resource */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_speech))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_speech_enable,aud_resource_table[aud_resource_speech].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_speech_enable, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_speech_enable */

/*------------------------------------------------------------------------------
* Function...: AUD_speech_disable
* Return.....: return code
* Description: Disables the speech resource.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_speech_disable(
    U8 handle)  /* handle to the speech resource */
{
    S8 res;
	
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_speech))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_speech_disable, aud_resource_table[aud_resource_speech].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_speech_disable, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_speech_disable */

/*------------------------------------------------------------------------------
* Function...: AUD_set_EC_NR
* Return.....: aud_rc_ok
* Description: Can be used to disable Echo Cancellation and Noise Reduction
*              even though path settings claim that they shoud be turned on.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_set_EC_NR(
    U8 EC_on,   /* echo cancellation, select FALSE to disable */
    U8 NR_on)   /* noise reduction, select FALSE to disable */
{
#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_function_interface, 4, 0,           /* LLT params */
                      'P', 0x05,                                                /* LIS params */
                      aud_log_lis_display, aud_log_lis_display_data, aud_log_llt_trace, /* LIS params */
                      (U32)E_control_EC_NR,
                      (U32)EC_on,
                      (U32)NR_on,
                      (U32)E_audio_llt_type_id_9 );
#endif

    aud_send_mail(E_control_EC_NR, 0, EC_on, NR_on, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_ok);
}
#ifdef _USE_CUT_TEAKLIT_CODE
/*******************************************************************************
*
*                Tone generator related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_tone_start
* Return.....: return code
* Description: Start playing the specified tone.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_tone_start(
    U8 handle,                  /* handle to the tone generator */
    aud_tone_id_enum tone_id,   /* ID of the tone to be played */
    U16 nof_repeats,            /* how many times the tone should be repeated (0 means forever) */
    S16 mix_factor)             /* not used */
{
    S8 res;
    /*************************add by zhouzhongyao  ***************************/
    //check pmic status
//    if(aud_rc_ok!= res){
//	  return res;
//    }
    /*************************            add end          ***************************/

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_tone_generator))
    {
        res = aud_rc_sharing_violation;
    }
    else if(tone_id >= aud_tone_id_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(mix_factor < 0 || mix_factor > 0x7FFF)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_tone_start, aud_resource_table[aud_resource_tone_generator].caller_id,
                 tone_id, nof_repeats, mix_factor, 0, 0, 0, 0, 0, NULL, NULL);
        res = aud_rc_ok;
	 /*************************add by zhouzhongyao  ***************************/
        //check pmic status
        /*************************            add end          ***************************/
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_tone_start, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_tone_start */


/*------------------------------------------------------------------------------
* Function...: AUD_tone_start_def
* Return.....: return code
* Description: Start playing the specified tone.
* Modified...: 18.03.2009 by wangjun
------------------------------------------------------------------------------*/
S8 AUD_tone_start_def(
    U8 handle,                  /* handle to the tone generator */
    aud_tone_id_enum tone_id,   /* ID of the tone to be played */
    U16 nof_repeats,            /* how many times the tone should be repeated (0 means forever) */
    S16 mix_factor,              /* not used */  
    UINT32 callback)             /*used for callback*/
{
    S8 res;
    /*************************add by zhouzhongyao  ***************************/
    //check pmic status 
    /*************************            add end          ***************************/

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_tone_generator))
    {
        res = aud_rc_sharing_violation;
    }
    else if(tone_id >= aud_tone_id_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(mix_factor < 0 || mix_factor > 0x7FFF)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_tone_start, aud_resource_table[aud_resource_tone_generator].caller_id,
                 tone_id, nof_repeats, mix_factor, 0, 0, 0, 0, callback, NULL, NULL);
        res = aud_rc_ok;
	 /*************************add by zhouzhongyao  ***************************/
        //check pmic status
        /*************************            add end          ***************************/
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_tone_start, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_tone_start */


/*------------------------------------------------------------------------------
* Function...: AUD_tone_start_user_tone
* Return.....: return code
* Description: Start playing a used defined tone.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_tone_start_user_tone(
    U8 handle,                  /* handle to the tone generator */
    void *p_tone_data,          /* pointer to the tone to be played */
    aud_tone_type_enum type,    /* single, dual or triple tone */
    U32 nof_tones,              /* the number of tones to be played */
    U16 nof_repeats,            /* how many times the tone should be repeated (0 means forever) */
    S16 mix_factor)             /* not used */
{
    S8 res;
	
    /*************************add by zhouzhongyao  ***************************/
    //check pmic status
//    if(aud_rc_ok!= res){
//	  return res;
//    }
    /*************************            add end          ***************************/

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_tone_generator))
    {
        res = aud_rc_sharing_violation;
    }
    else if(type >= aud_tone_type_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(mix_factor < 0 || mix_factor > 0x7FFF)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_tone_start_user_tone, aud_resource_table[aud_resource_tone_generator].caller_id,
                 (U32)p_tone_data, type, nof_tones, nof_repeats, mix_factor, 0, 0, 0, NULL, NULL);
        res = aud_rc_ok;
		
	 /*************************add by zhouzhongyao  ***************************/
        //update pmic status
        /*************************            add end          ***************************/
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_tone_start_user_tone, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_tone_start_user_tone */

/*------------------------------------------------------------------------------
* Function...: AUD_tone_start_user_tone_def
* Return.....: return code
* Description: Start playing a used defined tone.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_tone_start_user_tone_def(
    U8 handle,                  /* handle to the tone generator */
    void *p_tone_data,          /* pointer to the tone to be played */
    aud_tone_type_enum type,    /* single, dual or triple tone */
    U32 nof_tones,                /* the number of tones to be played */
    U16 nof_repeats,            /* how many times the tone should be repeated (0 means forever) */
    S16 mix_factor,              /* not used */ 
    UINT32 callback)             /*used for callback*/
{
    S8 res;
	
    /*************************add by zhouzhongyao  ***************************/
    //check pmic status
    /*************************            add end          ***************************/

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_tone_generator))
    {
        res = aud_rc_sharing_violation;
    }
    else if(type >= aud_tone_type_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(mix_factor < 0 || mix_factor > 0x7FFF)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_tone_start_user_tone, aud_resource_table[aud_resource_tone_generator].caller_id,
                 (U32)p_tone_data, type, nof_tones, nof_repeats, mix_factor, 0, 0, callback, NULL, NULL);
        res = aud_rc_ok;
		
	 /*************************add by zhouzhongyao  ***************************/
        //update pmic status
        /*************************            add end          ***************************/
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_tone_start_user_tone, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_tone_start_user_tone */

/*------------------------------------------------------------------------------
* Function...: AUD_tone_stop
* Return.....: return code
* Description: Stops tone playback.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_tone_stop(
    U8 handle)  /* handle to the tone generator */
{
    S8 res;
	
    /*************************add by zhouzhongyao  ***************************/
    //check pmic status
    /*************************            add end          ***************************/
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_tone_generator))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_tone_stop, aud_resource_table[aud_resource_tone_generator].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
	 /*************************add by zhouzhongyao  ***************************/
        //update codec status
        /*************************            add end          ***************************/
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_tone_stop, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_tone_suspend
* Return.....: return code
* Description: Suspends playback of the tone currently being played.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_tone_suspend(
    U8 handle,  /* handle to the tone generator */
    U8 slot_id) /* the slot (0-4) where the tone data is stored for later resumption */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_tone_generator))
    {
        res = aud_rc_sharing_violation;
    }
    else if(slot_id >= AUD_NOF_SUSPEND_SLOTS)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_tone_suspend, aud_resource_table[aud_resource_tone_generator].caller_id,slot_id,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_tone_suspend, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_tone_suspend */

/*------------------------------------------------------------------------------
* Function...: AUD_tone_resume
* Return.....: return code
* Description: Resumes playback of a suspended tone.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_tone_resume(
    U8 handle,  /* handle to the tone generator */
    U8 slot_id) /* the suspend slot (0-4) to resume from */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_tone_generator))
    {
        res = aud_rc_sharing_violation;
    }
    else if(slot_id >= AUD_NOF_SUSPEND_SLOTS)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_tone_resume, aud_resource_table[aud_resource_tone_generator].caller_id,slot_id,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_tone_resume, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_tone_resume */

/*------------------------------------------------------------------------------
* Function...: AUD_tone_get_total_playtime
* Return.....: return code
* Description: Returns the total playtime for the selected data in an SDL msg.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_tone_get_total_playtime(
    U8 handle,                  /* not used */
    aud_tone_id_enum tone_id,   /* ID of the tone for which the playtime should be calculated */
    void *p_tone_data,          /* pointer to the user tone for which the playtime should be calculated */
    U32 nof_tones,              /* number of tones in the user tone */
    aud_tone_type_enum type)    /* the type of user tone (single, dual, triple) */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(tone_id >= aud_tone_id_end || type >= aud_tone_type_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_tone_get_total_playtime, aud_resource_table[aud_resource_tone_generator].caller_id,
                 tone_id, nof_tones, type, 0, 0, 0, 0, (U32)p_tone_data, NULL, NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_tone_get_total_playtime, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_tone_get_total_playtime */

/*------------------------------------------------------------------------------
* Function...: AUD_tone_get_play_position
* Return.....: return code
* Description: Returns the elapsed time of the tone currently played.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_tone_get_play_position(
    U8 handle)  /* handle to the tone generator */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_tone_generator))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_tone_get_play_position,aud_resource_table[aud_resource_tone_generator].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_tone_get_play_position, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_tone_get_play_position */

/*------------------------------------------------------------------------------
* Function...: AUD_tone_set_play_position
* Return.....: return code
* Description: Change the position from where playback should be started.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_tone_set_play_position(
    U8 handle,  /* handle to the tone generator */
    U32 pos)    /* the position from where playback should be started */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_tone_generator))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_tone_set_play_position,aud_resource_table[aud_resource_tone_generator].caller_id,pos,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_tone_set_play_position, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_tone_set_play_position */

/*******************************************************************************
*
*                  Midi player related interface functions
*
*******************************************************************************/
#if defined (INTERNAL_POLYRINGER)
/*------------------------------------------------------------------------------
* Function...: AUD_midi_start
* Return.....: return code
* Description: Starts the midi player.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_midi_start(
    U8 handle,                          /* handle to the midi player */
    aud_ringer_tone_id_enum midi_id,    /* ID of the tune to be played */
    U16 nof_repeats)                    /* how many times the tune should be repeated (0 means forever) */
{
    S8 res;
	
    /*************************add by zhouzhongyao  ***************************/
    //check pmic status
    /*************************            add end          ***************************/

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        res = aud_rc_sharing_violation;
    }
    else if(midi_id >= aud_ringer_tone_id_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_midi_start,aud_resource_table[aud_resource_midi_player].caller_id,midi_id,nof_repeats,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
	 /*************************add by zhouzhongyao  ***************************/
        //check pmic status
        /*************************            add end          ***************************/
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_midi_start, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_midi_start_user_melody
* Return.....: return code
* Description: Starts midi playback of a user file.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_midi_start_user_melody(
    U8 handle,              /* handle to the midi player */
    U8 huge *p_melody_data,   /* the midi data to be played */
    U32 size,               /* size of the data file */
    aud_format_enum format, /* format of the user data */
    U16 nof_repeats)        /* how many times the tune should be repeated (0 means forever) */
{
    S8 res;
	
    /*************************add by zhouzhongyao  ***************************/
    //check pmic status
    /*************************            add end          ***************************/
	
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        res = aud_rc_sharing_violation;
    }
    else if(format >= aud_format_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_midi_start_user_melody, aud_resource_table[aud_resource_midi_player].caller_id,
                 (U32)p_melody_data, size, format, nof_repeats, 0, 0, 0, 0, NULL, NULL);
        res = aud_rc_ok;
	 /*************************add by zhouzhongyao  ***************************/
        //check pmic status
        /*************************            add end          ***************************/

    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_midi_start_user_melody, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_midi_start_user_melody_ext
* Return.....: return code
* Description: Starts midi playback of a user file.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_midi_start_user_melody_ext(
    U8 handle,              /* handle to the midi player */
    U8 huge *p_melody_data,   /* the midi data to be played */
    U32 size,               /* size of the data file */
    aud_format_enum format, /* format of the user data */
    U16 nof_repeats,/* how many times the tune should be repeated (0 means forever) */
    UINT32 PlayEndCallBack)        
{
    S8 res;
	
    /*************************add by zhouzhongyao  ***************************/
    //check pmic status
    /*************************            add end          ***************************/
	
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        res = aud_rc_sharing_violation;
    }
    else if(format >= aud_format_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_midi_start_user_melody, aud_resource_table[aud_resource_midi_player].caller_id,
                 (U32)p_melody_data, size, format, nof_repeats, PlayEndCallBack, 0, 0, 0, NULL, NULL);
        res = aud_rc_ok;
	 /*************************add by zhouzhongyao  ***************************/
        //check pmic status
        /*************************            add end          ***************************/

    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_midi_start_user_melody, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_midi_stop
* Return.....: return code
* Description: Stop midi playback.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_midi_stop(
    U8 handle)  /* handle to the midi player */
{
    S8 res;
	
    /*************************add by zhouzhongyao  ***************************/
    //check pmic status
    /*************************            add end          ***************************/

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_midi_stop, aud_resource_table[aud_resource_midi_player].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
	 /*************************add by zhouzhongyao  ***************************/
        //update codec status
        /*************************            add end          ***************************/
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_midi_stop, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_midi_suspend
* Return.....: return code
* Description: Suspends the midi file currently being played.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_midi_suspend(
    U8 handle,  /* handle to the midi player */
    U8 SlotID)  /* the slot (0-4) where the tune data is stored for later resumption */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        res = aud_rc_sharing_violation;
    }
    else if(SlotID>=AUD_NOF_SUSPENDNEST)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_midi_suspend,aud_resource_table[aud_resource_midi_player].caller_id,SlotID,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_midi_suspend, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_midi_stop_suspend
* Return.....: return code
* Description: Deletes information for a suspended melody.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_midi_stop_suspend(
    U8 handle,  /* handle to the midi player */
    U8 SlotID)  /* the suspend slot (0-4) to delete */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        res = aud_rc_sharing_violation;
    }
    else if(SlotID>=AUD_NOF_SUSPENDNEST)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_midi_stop_suspend, aud_resource_table[aud_resource_midi_player].caller_id,SlotID,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_midi_stop_suspend, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_midi_resume
* Return.....: return code
* Description: Resumes a suspended midi tune.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_midi_resume(
    U8 handle,  /* handle to the midi player */
    U8 SlotID)  /* the suspend slot (0-4) to resume from */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        res = aud_rc_sharing_violation;
    }
    else if(SlotID>=AUD_NOF_SUSPENDNEST)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_midi_resume,aud_resource_table[aud_resource_midi_player].caller_id,SlotID,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_midi_resume, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}
#endif /* INTERNAL_POLYRINGER */

/*******************************************************************************
*
*                     Ringer related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_ringer_start
* Return.....: return code
* Description: Plays the requested ringer melody.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_ringer_start(
    U8 handle,                          /* handle to the ringer/midi player resource */
    aud_ringer_tone_id_enum tone_id,    /* ID of the tune to be played */
    U16 nof_repeats,                    /* how many times the tune should be repeated (0 means forever) */
    aud_ringer_device_enum device)      /* the device to play the tune on */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (INTERNAL_POLYRINGER)
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        res = AUD_midi_start(handle, tone_id, nof_repeats); /* tone_id is the midi_id */
    }
    else
#endif
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_ringer))
    {
        res = aud_rc_sharing_violation;
    }
    else if(tone_id >= aud_ringer_tone_id_end || device >= aud_ringer_device_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_ringer_start,aud_resource_table[aud_resource_ringer].caller_id,tone_id,nof_repeats,device,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_ringer_start, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_ringer_start */

/*------------------------------------------------------------------------------
* Function...: AUD_ringer_start_user_tone
* Return.....: return code
* Description: Start playback of a user-defined tune.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_ringer_start_user_tone(
    U8 handle,                      /* handle to the ringer/midi player resource */
    U8 DWD_HUGE *p_ringer_data,     /* pointer to the tune to be played */
    U32 size,                       /* size of the data to be played */
    aud_format_enum format,         /* format of the tune */
    U16 nof_repeats,                /* how many times the tune should be repeated (0 means forever) */
    aud_ringer_device_enum device,  /* the device to play the tune on */
    U8 channel,                     /* used for smaf-phrase format (channel 1-4 available) */
    U8 channel_volume)              /* the volume level for the specified channel */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (INTERNAL_POLYRINGER)
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        res = AUD_midi_start_user_melody(handle, p_ringer_data, size, format, nof_repeats );
    }
    else
#endif
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_ringer))
    {
        res = aud_rc_sharing_violation;
    }
    else if(device >= aud_ringer_device_end || format >= aud_format_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(aud_format_smaf_phrase_l1 == format && (channel > 4 || channel_volume > 127))
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        if(aud_format_smaf_phrase_l1 != format)
        {
            channel = 1;
        }
        aud_send_mail(E_ringer_start_user_tone, aud_resource_table[aud_resource_ringer].caller_id,
                 size,format,nof_repeats,device,channel,channel_volume,(U32)p_ringer_data, 0, NULL, NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_ringer_start_user_tone, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_ringer_start_user_tone */

/*------------------------------------------------------------------------------
* Function...: AUD_ringer_start_user_tone_ext
* Return.....: return code
* Description: Start playback of a user-defined tune.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_ringer_start_user_tone_ext(
    U8 handle,                      /* handle to the ringer/midi player resource */
    U8 DWD_HUGE *p_ringer_data,     /* pointer to the tune to be played */
    U32 size,                       /* size of the data to be played */
    aud_format_enum format,         /* format of the tune */
    U16 nof_repeats,                /* how many times the tune should be repeated (0 means forever) */
    aud_ringer_device_enum device,  /* the device to play the tune on */
    U8 channel,                     /* used for smaf-phrase format (channel 1-4 available) */
    U8 channel_volume, /* the volume level for the specified channel */
    UINT32 PlayEndCallBack)              
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (INTERNAL_POLYRINGER)
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        res = AUD_midi_start_user_melody_ext(handle, p_ringer_data, size, format, nof_repeats ,PlayEndCallBack);
    }
    else
#endif
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_ringer))
    {
        res = aud_rc_sharing_violation;
    }
    else if(device >= aud_ringer_device_end || format >= aud_format_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(aud_format_smaf_phrase_l1 == format && (channel > 4 || channel_volume > 127))
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        if(aud_format_smaf_phrase_l1 != format)
        {
            channel = 1;
        }
        aud_send_mail(E_ringer_start_user_tone, aud_resource_table[aud_resource_ringer].caller_id,
                 size,format,nof_repeats,device,channel,channel_volume,(U32)p_ringer_data, 0, NULL, NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_ringer_start_user_tone, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_ringer_start_user_tone */

/*------------------------------------------------------------------------------
* Function...: AUD_ringer_stop
* Return.....: return code
* Description: Stops the tune being played.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_ringer_stop(
    U8 handle,  /* handle to the ringer/midi player resource */
    U8 channel) /* which smaf-phrase channel to stop (only for smaf-phrase format) */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (INTERNAL_POLYRINGER)
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        res = AUD_midi_stop(handle);
    }
    else
#endif
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_ringer))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_ringer_stop,aud_resource_table[aud_resource_ringer].caller_id, channel,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_ringer_stop, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_ringer_stop */

/*------------------------------------------------------------------------------
* Function...: AUD_ringer_set_channel_volume
* Return.....: return code
* Description: For smaf-phrase format, set an individual channel volume.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_ringer_set_channel_volume(
    U8 handle,          /* handle to the ringer/midi player resource */
    U8 channel_volume,  /* the new volume level */
    U8 channel)         /* the channel for which the volume should be set */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_ringer))
    {
        res = aud_rc_sharing_violation;
    }
    else if(channel > 4 || channel_volume > 127)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_ringer_set_channel_volume, aud_resource_table[aud_resource_ringer].caller_id,
                 channel_volume, channel, 0, 0, 0, 0, 0, 0, NULL, NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_ringer_set_channel_volume, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_ringer_suspend
* Return.....: return code
* Description: Suspends the active ringer melody.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_ringer_suspend(
    U8 handle,  /* handle to the ringer/midi player resource */
    U8 SlotID,  /* the slot (0-4) where the tune data is stored for later resumption */
    U8 channel) /* which channel (1-4) to suspend (0 means all) */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (INTERNAL_POLYRINGER)
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        res = AUD_midi_suspend(handle, SlotID);
    }
    else
#endif
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_ringer))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_ringer_suspend,aud_resource_table[aud_resource_ringer].caller_id,SlotID,channel,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_ringer_suspend, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_ringer_suspend */

/*------------------------------------------------------------------------------
* Function...: AUD_ringer_resume
* Return.....: return code
* Description: Resumes a suspended melody.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_ringer_resume(
    U8 handle,  /* handle to the ringer/midi player resource */
    U8 SlotID,  /* the suspend slot (0-4) to resume from */
    U8 channel) /* which channel (1-4) to resume (0 means all) */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (INTERNAL_POLYRINGER)
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        res = AUD_midi_resume(handle, SlotID);
    }
    else
#endif
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_ringer))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_ringer_resume,aud_resource_table[aud_resource_ringer].caller_id,SlotID, channel,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_ringer_resume, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_ringer_resume */

/*------------------------------------------------------------------------------
* Function...: AUD_ringer_stop_suspend
* Return.....: return code
* Description: Deletes information for a suspended melody.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_ringer_stop_suspend(
    U8 handle,  /* handle to the ringer/midi player resource */
    U8 SlotID,  /* the suspend slot (0-4) to delete */
    U8 channel) /* which channel (1-4) to delete (0 means all) */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (INTERNAL_POLYRINGER)
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        res = AUD_midi_stop_suspend(handle, SlotID);
    }
    else
#endif
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_ringer))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_ringer_stop_suspend,aud_resource_table[aud_resource_ringer].caller_id,SlotID,channel,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_ringer_stop_suspend, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_ringer_stop_suspend */

/*------------------------------------------------------------------------------
* Function...: AUD_ringer_get_total_playtime
* Return.....: return code
* Description: Returns the total playtime for the selected data.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_ringer_get_total_playtime(
    U8 handle,                          /* not used */
    U8 DWD_HUGE *p_data,                /* pointer to the melody data */
    U32 data_size,                      /* size of the melody data */
    aud_format_enum format,             /* format of the melody */
    aud_ringer_tone_id_enum tone_id)    /* id of the melody  */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (INTERNAL_POLYRINGER)
    aud_send_mail(E_midi_get_total_playtime, aud_resource_table[aud_resource_ringer].caller_id,
             tone_id, (U32)p_data, data_size, format, 0, 0, 0, 0, NULL, NULL);
    res = aud_rc_ok;
#else /* INTERNAL_POLYRINGER */
    aud_send_mail(E_ringer_get_total_playtime, aud_resource_table[aud_resource_ringer].caller_id,
             data_size, 0, format, tone_id, 0, 0, 0, (U32)p_data, NULL, NULL);
    res = aud_rc_ok;
#endif /* INTERNAL_POLYRINGER */
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_ringer_get_total_playtime_ext
* Return.....: return code
* Description: Returns the total playtime for the selected data.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_ringer_get_total_playtime_ext(
    U8 handle,                          /* not used */
    U8 DWD_HUGE *p_data,                /* pointer to the melody data */
    U32 data_size,                      /* size of the melody data */
    aud_format_enum format,             /* format of the melody */
    aud_ringer_tone_id_enum tone_id,
    UINT32 TotalTimeCallBack
    )    /* id of the melody  */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (INTERNAL_POLYRINGER)
    aud_send_mail(E_midi_get_total_playtime, aud_resource_table[aud_resource_ringer].caller_id,
             tone_id, (U32)p_data, data_size, format, TotalTimeCallBack, 0, 0, 0, NULL, NULL);
    res = aud_rc_ok;
#else /* INTERNAL_POLYRINGER */
    aud_send_mail(E_ringer_get_total_playtime, aud_resource_table[aud_resource_ringer].caller_id,
             data_size, 0, format, tone_id, 0, 0, 0, (U32)p_data, NULL, NULL);
    res = aud_rc_ok;
#endif /* INTERNAL_POLYRINGER */
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_ringer_get_playposition
* Return.....: return code
* Description: Returns the elapsed time of the melody currently played.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_ringer_get_playposition(
    U8 handle)  /* handle to the ringer/midi player resource */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (INTERNAL_POLYRINGER)
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        aud_send_mail(E_midi_get_play_position,aud_resource_table[aud_resource_ringer].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    else
#endif
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_ringer))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_ringer_get_position,aud_resource_table[aud_resource_ringer].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_ringer_get_position, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_ringer_get_playposition_ext
* Return.....: return code
* Description: Returns the elapsed time of the melody currently played.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_ringer_get_playposition_ext(
    U8 handle,/* handle to the ringer/midi player resource */
    UINT32 PlayPositionCallBack)  
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (INTERNAL_POLYRINGER)
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        aud_send_mail(E_midi_get_play_position,aud_resource_table[aud_resource_ringer].caller_id,PlayPositionCallBack,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    else
#endif
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_ringer))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_ringer_get_position,aud_resource_table[aud_resource_ringer].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_ringer_get_position, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_ringer_set_playposition
* Return.....: return code
* Description: Change the position from where playback should be started.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_ringer_set_playposition(
    U8 handle,  /* handle to the ringer/midi player resource */
    U8 channel, /* which channel (1-4) to set the play position for */
    U32 offset) /* the position from where playback should be started */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (INTERNAL_POLYRINGER)
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_midi_player))
    {
        aud_send_mail(E_midi_set_play_position, aud_resource_table[aud_resource_ringer].caller_id,
                 channel, offset, 0, 0, 0, 0, 0, 0, NULL, NULL);
        res = aud_rc_ok;
    }
    else
#endif
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_ringer))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_ringer_set_position, aud_resource_table[aud_resource_ringer].caller_id,
                 channel, offset, 0, 0, 0, 0, 0, 0, NULL, NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_ringer_set_position, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_ringer_set_playposition */

/*******************************************************************************
*
*                     Vibrator related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_vibrator_enable
* Return.....: -
* Description: Generic audio interface to activate vibrator. No handle is used
*              as this interface is only meant to be called from driver
*              generic vibrator driver and not from aplication.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void AUD_vibrator_enable(
    aud_vib_mode mode,  /* which predefined on/off sequence to use */
    U8 channel)         /* ? */
{
    aud_send_mail(E_vibrator_enable, aud_resource_table[aud_resource_vibrator].caller_id,mode,channel,0,0,0,0,0,0,NULL,NULL);
}

/*------------------------------------------------------------------------------
* Function...: AUD_vibrator_pause
* Return.....: -
* Description: Pauses the vibrator.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void AUD_vibrator_pause(void)
{
    aud_send_mail(E_vibrator_pause, aud_resource_table[aud_resource_vibrator].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
}

/*------------------------------------------------------------------------------
* Function...: AUD_vibrator_disable
* Return.....: -
* Description: Disables the vibrator and powers down the vibrator chip.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void AUD_vibrator_disable(void)
{
    aud_send_mail(E_vibrator_disable, aud_resource_table[aud_resource_vibrator].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
}

/*******************************************************************************
*
*                    FM radio related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_radio_enable
* Return.....: return code
* Description: Power on and configures the FM radio hardware.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_radio_enable(
    U8 handle)  /* handle to the FM radio */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (FM_RADIO_PRESENT)
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_radio))
    {
        res = (aud_rc_sharing_violation);
    }
    else /* All OK - pass on to FM Radio SM */
    {
        aud_send_mail(E_radio_start, aud_resource_table[aud_resource_radio].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = (aud_rc_ok);
    }
#else
    res = (aud_rc_no_hw_support);
#endif /* FM_RADIO_PRESENT */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_radio_start, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_radio_disable
* Return.....: return code
* Description: Power down the FM radio.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_radio_disable(
    U8 handle)  /* handle to the FM radio */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (FM_RADIO_PRESENT)
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_radio))
    {
        res = (aud_rc_sharing_violation);
    }
    else /* All OK - pass on to FM Radio SM */
    {
        aud_send_mail(E_radio_stop, aud_resource_table[aud_resource_radio].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = (aud_rc_ok);
    }
#else
    res = (aud_rc_no_hw_support);
#endif /* FM_RADIO_PRESENT */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_radio_stop, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_radio_rssi_subscribe
* Return.....: return code
* Description: Start subscribing to periodic RSSI information.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_radio_rssi_subscribe(
    U8 handle,                              /* handle to the FM radio resource */
    aud_fm_radio_rssi_enum lower_threshold, /* lower threshold for RSSI level changes */
    aud_fm_radio_rssi_enum upper_threshold) /* upper threshold for RSSI level changes */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (FM_RADIO_PRESENT)
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_radio))
    {
        res = (aud_rc_sharing_violation);
    }
    /* Validate radio parameters */
    else if( (lower_threshold >= aud_fm_radio_rssi_enum_end) || (upper_threshold >= aud_fm_radio_rssi_enum_end) )
    {
        res = (aud_rc_parameter_out_of_range);
    }
    /* All OK - pass on to FM Radio SM */
    else if(aud_fm_radio_rssi_0 == lower_threshold)
    {
        res = (aud_rc_ok);
        aud_send_mail(E_radio_rssi_unsubscribe,aud_resource_table[aud_resource_radio].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
    }
    else
    {
        aud_send_mail(E_radio_rssi_subscribe,  aud_resource_table[aud_resource_radio].caller_id,
                 (U32)lower_threshold, (U32)upper_threshold,0,0,0,0,0,0,NULL,NULL);
        res = (aud_rc_ok);
    }
#else
    res = (aud_rc_no_hw_support);
#endif /* FM_RADIO_PRESENT */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_radio_rssi_subscribe, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_radio_rds_subscribe
* Return.....: return code
* Description: Start subscribing to periodic RDS information.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_radio_rds_subscribe(
    U8 handle,                              /* pointer to FM radio */
    ptr_FmRadioRdsCallbackFunction pCbFunc) /* callback function for processing the RDS information */
{
    S8 res;
    aud_event_enum aud_event;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (FM_RADIO_PRESENT)
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_radio))
    {
        res = (aud_rc_sharing_violation);
    }
    /* All OK - pass on to FM Radio SM */
    else if(NULL == pCbFunc)
    {
        aud_send_mail(E_radio_rds_unsubscribe,aud_resource_table[aud_resource_radio].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = (aud_rc_ok);
        aud_event = E_radio_rds_unsubscribe;
    }
    else
    {
        aud_send_mail(E_radio_rds_subscribe,aud_resource_table[aud_resource_radio].caller_id,(U32)pCbFunc,0,0,0,0,0,0,0,NULL,NULL);
        res = (aud_rc_ok);
        aud_event = E_radio_rds_subscribe;
    }
#else
    res = (aud_rc_no_hw_support);
#endif /* FM_RADIO_PRESENT */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_radio_rds_subscribe, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
        aud_event = E_radio_rds_subscribe;
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_function_interface, 3, 0,           /* LLT params */
                      'P', 0x05,                                                /* LIS params */
                      aud_log_lis_display, aud_log_lis_display_data, aud_log_llt_trace, /* LIS params */
                      (U32)aud_event,
                      (U32)pCbFunc,
                      (U32)E_audio_llt_type_id_6 );
#endif
    if( aud_event );//xiu  warning: unused aud_event
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_radio_set_station
* Return.....: return code
* Description: Sets or finds a station, depending on the seekmode parameter.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_radio_set_station(
    U8 handle,                              /* handle to the FM radio */
    aud_fm_radio_seek_mode_enum seekmode,   /* whether to search for a station or go to a particular frequency */
    U32 frequency,                          /* the frequency to activate if seekmode=off */
    aud_fm_radio_rssi_enum detect_level,    /* the RSSI level used as a threshold when seeking for stations */
    U8 force_mono)                          /* force the radio to receive the station as mono (0 means don't care) */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (FM_RADIO_PRESENT)
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_radio))
    {
        res = (aud_rc_sharing_violation);
    }
    /* Validate radio parameters */
    else if( (seekmode >= aud_fm_radio_seek_enum_end) || (detect_level >= aud_fm_radio_rssi_enum_end) )
    {
        res = (aud_rc_parameter_out_of_range);
    }
    else /* All OK - pass on to FM Radio SM */
    {
        aud_send_mail(E_radio_set_station, aud_resource_table[aud_resource_radio].caller_id,
                 (U32)seekmode, frequency, (U32)detect_level, (U32)force_mono, 0,0,0,0, NULL, NULL);
        res = (aud_rc_ok);
    }
#else
    res = (aud_rc_no_hw_support);
#endif /* FM_RADIO_PRESENT */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_radio_set_station, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_radio_get_frequency
* Return.....: return code
* Description: Reads the current FM radio frequency.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_radio_get_frequency(
    U8 handle)  /* handle to the FM radio */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (FM_RADIO_PRESENT)
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_radio))
    {
        res = (aud_rc_sharing_violation);
    }
    else /* All OK - pass on to FM Radio SM */
    {
        aud_send_mail(E_radio_get_frequency, aud_resource_table[aud_resource_radio].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = (aud_rc_ok);
    }
#else
    res = (aud_rc_no_hw_support);
#endif /* FM_RADIO_PRESENT */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_radio_get_frequency, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*******************************************************************************
*
*                  Voice memo related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_vm_start_recording
* Return.....: return code
* Description: Setup and start voice memo recording.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_vm_start_recording(
    U8 handle,                          /* handle to the voice memo recorder */
    aud_vm_mode_enum vm_mode,           /* choose beween standby and tch mode */
    aud_media_enum media_type,          /* where to store the recording */
    aud_dsp_format_enum format,         /* which speech codec to use for recording */
    U8 rate,                            /* sample rate if AMR is chosen af format */
    U16 DWD_HUGE *p_file_handle,        /* pointer to the buffer where the recording should be stored */
    U32 buffer_size,                    /* the size of the buffer if RAM media is selected */
    U32 offset)                         /* the number of bytes to be left blank at the beginning of the buffer */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if !defined (FFS_PRESENT)
    if(aud_media_ffs == media_type)
    {
        res = aud_rc_not_supported;
    }
    else
#endif
#if !defined (MMCI_PRESENT) || !defined (FAT_FS_PRESENT)
    if( (aud_media_mmc == media_type)
#if defined (AUD_NANDDEVICE_SUPPORT)
        || (aud_media_fs_cdrive == media_type)
        || (aud_media_fs_edrive == media_type)
        || (aud_media_fs_fdrive == media_type)
#endif
      )
    {
        res = aud_rc_not_supported;
    }
    else
#endif
    if(vm_mode>=aud_vm_mode_end || format>=aud_dsp_format_end || media_type>=aud_media_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_record_vm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
#if defined (AUD_NANDDEVICE_SUPPORT)
        if(aud_media_fs_edrive == media_type)
        {
            res = aud_rc_not_supported;
        }
        else
#endif
        {
            aud_send_mail(E_vms_start_voice_recording, aud_resource_table[aud_resource_record_vm].caller_id,
                     vm_mode, media_type, format, rate, buffer_size , offset, 0, (U32)p_file_handle, NULL, NULL);
            res = aud_rc_ok;
        }
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_vms_start_voice_recording, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_vm_start_recording */


/*------------------------------------------------------------------------------
* Function...: AUD_vm_start_recording_def
* Return.....: return code
* Description: Setup and start voice memo recording.
* Modified...: 21.07.2009 by wangjun
------------------------------------------------------------------------------*/
S8 AUD_vm_start_recording_def(
    U8 handle,                          /* handle to the voice memo recorder */
    aud_vm_mode_enum vm_mode,           /* choose beween standby and tch mode */
    aud_media_enum media_type,          /* where to store the recording */
    aud_dsp_format_enum format,         /* which speech codec to use for recording */
    U8 rate,                            /* sample rate if AMR is chosen af format */
    UINT32 callbackRec,        /* pointer to the buffer where the recording should be stored */
    UINT32 callbackRate,                    /* the size of the buffer if RAM media is selected */
    U32 offset)                    /* the number of bytes to be left blank at the beginning of the buffer */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if !defined (FFS_PRESENT)
    if(aud_media_ffs == media_type)
    {
        res = aud_rc_not_supported;
    }
    else
#endif
#if !defined (MMCI_PRESENT) || !defined (FAT_FS_PRESENT)
    if( (aud_media_mmc == media_type)
#if defined (AUD_NANDDEVICE_SUPPORT)
        || (aud_media_fs_cdrive == media_type)
        || (aud_media_fs_edrive == media_type)
        || (aud_media_fs_fdrive == media_type)
#endif
      )
    {
        res = aud_rc_not_supported;
    }
    else
#endif
    if(vm_mode>=aud_vm_mode_end || format>=aud_dsp_format_end || media_type>=aud_media_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_record_vm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
#if defined (AUD_NANDDEVICE_SUPPORT)
        if(aud_media_fs_edrive == media_type)
        {
            res = aud_rc_not_supported;
        }
        else
#endif
        {
            aud_send_mail(E_vms_start_voice_recording, aud_resource_table[aud_resource_record_vm].caller_id,
                     vm_mode, media_type, format, rate, callbackRate , offset, 0, callbackRec, NULL, NULL);
            res = aud_rc_ok;
        }
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_vms_start_voice_recording, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_vm_start_recording */

/*------------------------------------------------------------------------------
* Function...: AUD_vm_stop_recording
* Return.....: return code
* Description: Stops the voice memo recording process.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_vm_stop_recording(
    U8 handle)  /* handle to the voice memo recorder */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_record_vm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_vms_stop_voice_recording, aud_resource_table[aud_resource_record_vm].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_vms_stop_voice_recording, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_stop_recording */

/*------------------------------------------------------------------------------
* Function...: AUD_vm_start_playback
* Return.....: return code
* Description: Start playback of a voice memo.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_vm_start_playback(
    U8 handle,                  /* handle to the voice memo player */
    aud_vm_mode_enum vm_mode,   /* choose beween standby and tch mode */
    aud_media_enum media_type,  /* from where to read the recording */
    aud_dsp_format_enum format, /* which speech codec to use for playback */
    U16 DWD_HUGE *p_file_handle,/* pointer to the buffer where the recording is stored */
    U32 buffer_size,            /* if RAM is the media type, this is the number of bytes in the buffer */
    U16 nof_repeats,            /* the number of times the data should be played (0 means forever) */
    U32 offset)                 /* the number of bytes that should be skipped in the start */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if !defined (FFS_PRESENT)
    if(aud_media_ffs == media_type)
    {
        res = aud_rc_not_supported;
    }
    else
#endif
#if !defined (MMCI_PRESENT) || !defined (FAT_FS_PRESENT)
    if( (aud_media_mmc == media_type)
#if defined (AUD_NANDDEVICE_SUPPORT)
        || (aud_media_fs_cdrive == media_type)
        || (aud_media_fs_edrive == media_type)
        || (aud_media_fs_fdrive == media_type)
#endif
      )
    {
        res = aud_rc_not_supported;
    }
    else
#endif
    if(vm_mode>=aud_vm_mode_end || format>=aud_dsp_format_end || media_type>=aud_media_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_vm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_vms_start_playback, aud_resource_table[aud_resource_playback_vm].caller_id,
                 vm_mode, media_type, format, buffer_size, nof_repeats, offset, 0, (U32)p_file_handle, NULL, NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_vms_start_playback, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_vm_start_playback */

/*------------------------------------------------------------------------------
* Function...: AUD_vm_stop_playback
* Return.....: return code
* Description: Stops the voice memo playback.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_vm_stop_playback(
    U8 handle)  /* handle to the voice memo player */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_vm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_vms_stop_playback, aud_resource_table[aud_resource_playback_vm].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_vms_stop_playback, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_vm_stop_playback */

#if defined (VMS_PRESENT)
/*------------------------------------------------------------------------------
* Function...: Aud_vms_set_getbuffer_function
* Return.....: -
* Description: The application is responsible for providing and handling
*              buffers for voice memo playback and recording. Control this.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void Aud_vms_set_getbuffer_function(
    U8 handle,                          /* handle to the voice memo player or recorder */
    ptr_GetAudioBufferFunction func,    /* callback function that is called when new data needs to be played */
    void *p_user_data)                  /* maintains the state of the callback functions */
{
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_playback_vm))
    {
        aud_send_mail(E_vms_pb_set_getbuffer_function, aud_resource_table[aud_resource_record_vm].caller_id,
                 0, 0, 0, 0, 0, 0, (U32 )func, (U32) p_user_data, NULL, NULL);
    }
    else if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_record_vm))
    {
        aud_send_mail(E_vms_rec_set_putbuffer_function, aud_resource_table[aud_resource_record_vm].caller_id,
                 0, 0, 0, 0, 0, 0, (U32 )func, (U32) p_user_data, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
} /* End of Aud_vms_set_getbuffer_function */
#endif

/*------------------------------------------------------------------------------
* Function...: AUD_vm_suspend
* Return.....: return code
* Description: Suspend voice memo playback.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_vm_suspend(
    U8 handle,  /* handle to the voice memo player */
    U8 slot_id) /* the slot (0-4) where the voice memo data is stored for later resumption */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(slot_id >= AUD_NOF_SUSPENDNEST)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_vm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_vms_suspend, aud_resource_table[aud_resource_playback_vm].caller_id,slot_id,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_vms_suspend, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_vm_suspend */

/*------------------------------------------------------------------------------
* Function...: AUD_vm_resume
* Return.....: return code
* Description: Resumes playback of suspended voice memo recording.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_vm_resume(
    U8 handle,  /* handle to the voice memo player */
    U8 slot_id) /* the suspend slot (0-4) to resume from */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(slot_id >= AUD_NOF_SUSPENDNEST)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_vm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_vms_resume, aud_resource_table[aud_resource_playback_vm].caller_id,slot_id,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_vms_resume, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_vm_resume */

/*------------------------------------------------------------------------------
* Function...: AUD_vm_get_total_playtime
* Return.....: return code
* Description: Returns the total playtime for the selected data.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_vm_get_total_playtime(
    U8 handle,                  /* not used */
    aud_media_enum media_type,  /* from where to read the recording */
    aud_dsp_format_enum format, /* which speech codec is used for the file */
    U16 DWD_HUGE *p_file_handle,/* pointer to the buffer where the recording is stored */
    U32 buffer_size,            /* if RAM is the media type, this is the number of bytes in the buffer */
    U32 offset)                 /* the number of bytes that should be skipped in the start */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if !defined (FFS_PRESENT)
    if(aud_media_ffs == media_type)
    {
        res = aud_rc_not_supported;
    }
    else
#endif
#if !defined (MMCI_PRESENT) || !defined (FAT_FS_PRESENT)
     if( (aud_media_mmc == media_type)
#if defined (AUD_NANDDEVICE_SUPPORT)
        || (aud_media_fs_cdrive == media_type)
        || (aud_media_fs_edrive == media_type)
        || (aud_media_fs_fdrive == media_type)
#endif
      )
    {
        res = aud_rc_not_supported;
    }
    else
#endif
    {
        aud_send_mail(E_vms_get_total_playtime, aud_resource_table[aud_resource_playback_vm].caller_id,
                 media_type, format, buffer_size, offset, 0, 0,0, (U32)p_file_handle, NULL, NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_vms_get_total_playtime, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_vm_get_play_position
* Return.....: return code
* Description: Returns the play position of the currently playing voice memo.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_vm_get_play_position(
    U8 handle)  /* handle to the voice memo player */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_vm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_vms_get_play_position, aud_resource_table[aud_resource_playback_vm].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_vms_get_play_position, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_vm_set_play_position
* Return.....: return code
* Description:
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_vm_set_play_position(
    U8 handle,                  /* handle to the voice memo player */
    U32 pos,                    /* the position to start playback from */
    U16 DWD_HUGE *p_file_handle,/* not used */
    U32 buffer_size)            /* not used */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_vm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_vms_set_play_position, aud_resource_table[aud_resource_playback_vm].caller_id,
                 pos, buffer_size,0, 0, 0, 0, 0, (U32)p_file_handle, NULL, NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_vms_set_play_position, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_vm_stop_suspend
* Return.....: return code
* Description: Clears all suspended data contents in the specified slot.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8  AUD_vm_stop_suspend(
    U8 handle,  /* handle to the voice memo player */
    U8 slot_id) /* the suspend slot (0-4) to delete */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(slot_id >= AUD_NOF_SUSPENDNEST)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_vm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_vms_stop_suspend, aud_resource_table[aud_resource_playback_vm].caller_id,slot_id,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_vms_stop_suspend, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of function AUD_vm_suspend */

/*******************************************************************************
*
*                   MP3 player related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_mp3_start
* Return.....: return code
* Description: Start playback of MP3 file.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_mp3_start(
    U8 handle,                  /* handle to the MP3 player */
    aud_media_enum media_type,  /* where is the data located */
    U16 DWD_HUGE *p_file_handle,/* file/buffer handle */
    U32 buffer_size,            /* size of buffer if RAM is used */
    U32 id_offset,              /* the offset where the audio data begins */
    U32 start_frame,            /* how many frames to skip before playback */
    U16 nof_repeats)            /* how many times should the file be repeated (0 means forever) */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if !defined (FFS_PRESENT)
    if(aud_media_ffs == media_type)
    {
        res = aud_rc_not_supported;
    }
    else
#endif
#if !defined (MMCI_PRESENT) || !defined (FAT_FS_PRESENT)
    if(aud_media_mmc == media_type)
    {
        res = aud_rc_not_supported;
    }
    else
#endif
#if defined (MP3_PRESENT )
    if(media_type >= aud_media_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_mp3))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_mp3_start, aud_resource_table[aud_resource_playback_mp3].caller_id,
                 media_type, buffer_size, id_offset, start_frame, nof_repeats,0,0, (U32)file_handle, NULL, NULL);
        res = aud_rc_ok;
    }
#else
    res = aud_rc_not_supported;
#endif /*MP3_PRESENT*/
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_mp3_start, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of  Aud_mp3_start */

/*------------------------------------------------------------------------------
* Function...: AUD_mp3_stop
* Return.....: return code
* Description: Stops MP3 playback.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_mp3_stop(
    U8 handle)  /* handle to the MP3 player */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (MP3_PRESENT)
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_mp3))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_mp3_stop, aud_resource_table[aud_resource_playback_mp3].caller_id,
                 0, 0, 0, 0, 0,0,0, 0, NULL, NULL);
        res = (aud_rc_ok);
    }
#else
    res = aud_rc_not_supported;
#endif /* MP3_PRESENT */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_mp3_stop, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_mp3_suspend
* Return.....: return code
* Description: Suspends MP3 playback to the specified slot.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_mp3_suspend(
    U8 handle,  /* handle to the MP3 player */
    U16 slot_id) /* the slot (0-4) where the MP3 data is stored for later resumption */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (MP3_PRESENT)
    if(slot_id >= AUD_NOF_SUSPENDNEST)
    {
        res = (aud_rc_parameter_out_of_range);
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_mp3))
    {
        res = (aud_rc_sharing_violation);
    }
    else
    {
        aud_send_mail(E_mp3_pause, aud_resource_table[aud_resource_playback_mp3].caller_id,
                 slot_id, 0, 0, 0, 0,0,0, 0, NULL, NULL);
        res = (aud_rc_ok);
    }
#else
    res = aud_rc_not_supported;
#endif /* MP3_PRESENT */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_mp3_pause, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of  Aud_mp3_suspend */

/*------------------------------------------------------------------------------
* Function...: AUD_mp3_resume
* Return.....: return code
* Description: Resumes playback of a suspended MP3 file.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_mp3_resume(
    U8 handle,  /* handle to the MP3 player */
    U16 slot_id) /* the suspend slot (0-4) to resume from */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (MP3_PRESENT)
    if(slot_id >= AUD_NOF_SUSPENDNEST)
    {
        res = (aud_rc_parameter_out_of_range);
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_mp3))
    {
        res = (aud_rc_sharing_violation);
    }
    else
    {
        aud_send_mail(E_mp3_resume, aud_resource_table[aud_resource_playback_mp3].caller_id,
                 slot_id, 0, 0, 0, 0,0,0, 0, NULL, NULL);
        res = (aud_rc_ok);
    }
#else
    res = aud_rc_not_supported;
#endif /*MP3_PRESENT*/
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_mp3_resume, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of  Aud_mp3_resume */

/*------------------------------------------------------------------------------
* Function...: AUD_mp3_fastforward
* Return.....: return code
* Description: Forwards a suspended MP3 file by a number of frames.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_mp3_fastforward(
    U8 handle,      /* handle to the MP3 player */
    U32 frame,      /* number of frames to skip */
    U16 slot_id)    /* the suspend slot (0-4) to resume from, 999 means skip entire file */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (MP3_PRESENT)
    if((slot_id < AUD_NOF_SUSPENDNEST) || (999 == slot_id))
    {
        if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_mp3))
        {
            res = (aud_rc_sharing_violation);
        }
        else
        {
            aud_send_mail(E_mp3_fastforward, aud_resource_table[aud_resource_playback_mp3].caller_id,
                     slot_id,frame,0, 0, 0,0,0, NULL, NULL, NULL);
            res = (aud_rc_ok);
        }
    }
    else
    {
        res = (aud_rc_parameter_out_of_range);
    }
#else
    res = aud_rc_not_supported;
#endif /*MP3_PRESENT*/
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_mp3_fastforward, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of  AUD_mp3_fastforward */

/*------------------------------------------------------------------------------
* Function...: AUD_mp3_backward
* Return.....: return code
* Description: Backwards a suspended MP3 file by a number of frames.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_mp3_backward(
    U8 handle,      /* handle to the MP3 player */
    U32 frame,      /* number of frames to skip */
    U16 slot_id)    /* the suspend slot (0-4) to resume from, 999 means skip entire file */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (MP3_PRESENT)
    if((slot_id >= AUD_NOF_SUSPENDNEST) && (slot_id != 999))
    {
        res = (aud_rc_parameter_out_of_range);
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_mp3))
    {
        res = (aud_rc_sharing_violation);
    }
    else
    {
        aud_send_mail(E_mp3_backward, aud_resource_table[aud_resource_playback_mp3].caller_id,
                 slot_id, frame,0, 0, 0,0,0,NULL, NULL, NULL);
        res = (aud_rc_ok);
    }
#else
    res = aud_rc_not_supported;
#endif /* MP3_PRESENT */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_mp3_backward, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of  AUD_mp3_backward */

/*------------------------------------------------------------------------------
* Function...: Aud_mp3_set_getbuffer_function
* Return.....: -
* Description: The application is responsible for providing and handling the
*              buffer for MP3 playback. Control this buffer.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void Aud_mp3_set_getbuffer_function(
    U8 handle,                          /* handle to the MP3 player */
    ptr_GetAudioBufferFunction func,    /* callback function that is called when new data needs to be played */
    void *p_user_data)                  /* maintains the state of the callback functions */
{
#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_function_interface, 3, 0,           /* LLT params */
                      'P', 0x05,                                                /* LIS params */
                      aud_log_lis_display, aud_log_lis_display_data, aud_log_llt_trace, /* LIS params */
                      (U32) E_mp3_set_getbuffer_function,
                      (U32) func,
                      (U32) E_audio_llt_type_id_5 );
#endif

#if defined (MP3_PRESENT)
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_mp3))
    {
        return;
    }
    aud_send_mail(E_mp3_set_getbuffer_function, aud_resource_table[aud_resource_playback_mp3].caller_id,
             0, 0, 0, 0, 0, 0, (U32 )func, (U32) p_user_data, NULL, NULL);
#endif /* MP3_PRESENT */
}

/*------------------------------------------------------------------------------
* Function...: Aud_mp3_get_current_frame
* Return.....: return code
* Description: Returns the currently playing frame's value.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 Aud_mp3_get_current_frame(
    U8 handle,      /* handle to the MP3 player */
    U16 slot_id)    /* the suspend slot (0-4) to read the frame from, 999 means currently playing file */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (MP3_PRESENT)
    if((slot_id >= AUD_NOF_SUSPENDNEST) && (slot_id != 999))
    {
        res = (aud_rc_parameter_out_of_range);
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_mp3))
    {
        res = (aud_rc_sharing_violation);
    }
    else
    {
        aud_send_mail(E_mp3_get_current_frame, aud_resource_table[aud_resource_playback_mp3].caller_id,
                 slot_id, 0, 0, 0, 0, 0, 0,0 , NULL, NULL);
        res = (aud_rc_ok);
    }
#else
    res = aud_rc_not_supported;
#endif /* MP3_PRESENT */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_mp3_get_current_frame, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_mp3_get_total_playtime
* Return.....: return code
* Description: Returns the playtime for the specified file.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_mp3_get_total_playtime(
    U8 handle,                  /* not used */
    aud_media_enum media_type,  /* where is the data located */
    U16 DWD_HUGE *p_file_handle,/* file/buffer handle */
    U32 buffer_size,            /* size of buffer if RAM is used */
    U32 id_offset)              /* the offset where the audio data begins */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if !defined (FFS_PRESENT)
    if(aud_media_ffs == media_type)
    {
        res = aud_rc_not_supported;
    }
    else
#endif
#if !defined (MMCI_PRESENT) || !defined (FAT_FS_PRESENT)
    if(aud_media_mmc == media_type)
    {
        res = aud_rc_not_supported;
    }
    else
#endif
#if defined (MP3_PRESENT)
    {
        aud_send_mail(E_mp3_get_total_playtime, aud_resource_table[aud_resource_playback_mp3].caller_id,
                 media_type, buffer_size, id_offset, 0, 0,0,0, (U32)p_file_handle, NULL, NULL);
        res = (aud_rc_ok);
    }
#else
    res = aud_rc_not_supported;
#endif /* MP3_PRESENT */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_mp3_get_total_playtime, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* Aud_mp3_get_total_playtime */

/*------------------------------------------------------------------------------
* Function...: AUD_mp3_get_play_position
* Return.....: return code
* Description: Returns the play position of the currently playing MP3 data.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_mp3_get_play_position(
    U8 handle)  /* handle to the MP3 player */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (MP3_PRESENT)
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_mp3))
    {
        res = (aud_rc_sharing_violation);
    }
    else
    {
        aud_send_mail(E_mp3_get_play_position,aud_resource_table[aud_resource_playback_mp3].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = (aud_rc_ok);
    }
#else
    res = (aud_rc_not_supported);
#endif /* MP3_PRESENT */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_mp3_get_play_position, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_mp3_set_play_position
* Return.....: return code
* Description: Sets the play position of the MP3 data to be played.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_mp3_set_play_position(
    U8 handle,                  /* handle to the MP3 player */
    U32 position,               /* the position to start playback from */
    U16 DWD_HUGE *p_file_handle,/* file/buffer handle */
    U32 buffer_size)            /* size of buffer if RAM is used */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (MP3_PRESENT)
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_mp3))
    {
        res = (aud_rc_sharing_violation);
    }
    else
    {
        aud_send_mail(E_mp3_set_play_position, aud_resource_table[aud_resource_playback_mp3].caller_id,
                 position, buffer_size, 0, 0, 0,0,0,(U32)file_handle, NULL, NULL);
        res = (aud_rc_ok);
    }
#else
    res = (aud_rc_not_supported);
#endif /* MP3_PRESENT */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_mp3_set_play_position, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_mp3_stop_suspend
* Return.....: return code
* Description: Clears all suspended data contents in the specified slot.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_mp3_stop_suspend(
    U8 handle,  /* handle to the MP3 player */
    U8 slot_id) /* the suspend slot (0-4) to delete */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (MP3_PRESENT)
    if(slot_id >= AUD_NOF_SUSPENDNEST)
    {
        res = (aud_rc_parameter_out_of_range);
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_mp3))
    {
        res = (aud_rc_sharing_violation);
    }
    else
    {
        aud_send_mail(E_mp3_stop_suspend,aud_resource_table[aud_resource_playback_mp3].caller_id,slot_id,0,0,0,0,0,0,0,NULL,NULL);
        res = (aud_rc_ok);
    }
#else
    res = (aud_rc_not_supported);
#endif /* MP3_PRESENT */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_mp3_stop_suspend, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /* End of Aud_mp3_stop_suspend */

/*******************************************************************************
*
*                       TTY related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_tty_enable
* Return.....: return code
* Description: Enables the TTY resource.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_tty_enable(
    U8 handle) /* handle to the TTY resource */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_tty))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_tty_enable,aud_resource_table[aud_resource_tty].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_tty_enable, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_tty_disable
* Return.....: return code
* Description: Disables the TTY resource.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_tty_disable(
    U8 handle)  /* handle to the TTY resource */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_tty))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_tty_disable,aud_resource_table[aud_resource_tty].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_tty_disable, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_tty_set_negotialtion
* Return.....: return code
* Description: Gives a possibility to switch off/on the negotiation parameter.
* Created....: 29.04.2007 by EPS
* Modified...: 07.08.2007 by KBJ (aligned code with coding guidelines)
------------------------------------------------------------------------------*/
S8 AUD_tty_set_negotiation(
    U8 handle,  /* handle to the TTY resource */
    U8 on_off)  /* switch on/off negotiation */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_tty))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_tty_set_negotiation,aud_resource_table[aud_resource_tty].caller_id,on_off,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_tty_set_negotiation, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*******************************************************************************
*
*               PCM player/recorder related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_pcm_intern_start_playback
* Return.....: return code
* Description:
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_pcm_intern_start_playback(
    U8 handle,                              /* handle to the PCM player */
    aud_pcm_mode_enum mode,                 /* mono/dual mono/stereo */
    aud_pcm_sample_rate_enum sample_rate,   /* sample rate */
    U8 bit_rate,                            /* the number of bits per sample */
    aud_media_enum media_type,              /* where is the data located */
    aud_pcm_format_enum format,             /* PCM/ADPCM/Wave */
    U16 DWD_HUGE *p_file_handle,            /* file/buffer handle */
    U32 buffer_size,                        /* size of buffer if RAM is used */
    U16 nof_repeats,                        /* how many times should the file be repeated (0 means forever) */
    U32 start_offset)                       /* how many bytes should be skipped in the beginning */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if !defined (FFS_PRESENT)
    if(aud_media_ffs == media_type)
    {
        res = aud_rc_not_supported;
    }
    else
#endif
#if !defined (MMCI_PRESENT) || !defined (FAT_FS_PRESENT)
    if( (aud_media_mmc == media_type)
#if defined (AUD_NANDDEVICE_SUPPORT)
        || (aud_media_fs_cdrive == media_type)
        || (aud_media_fs_edrive == media_type)
        || (aud_media_fs_fdrive == media_type)
#endif
      )
    {
        res = aud_rc_not_supported;
    }
    else
#endif
    if(mode>=aud_pcm_mode_end || format>=aud_pcm_format_end || media_type>=aud_media_end || sample_rate>=aud_pcm_rate_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_pcm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_pcm_start_playback, aud_resource_table[aud_resource_playback_pcm].caller_id,
                 mode, sample_rate, bit_rate, media_type, format, nof_repeats, buffer_size,
                 start_offset,(U8 *) p_file_handle, NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_pcm_start_playback, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_pcm_intern_stop_playback
* Return.....: return code
* Description: Stops PCM playback.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_pcm_intern_stop_playback(
    U8 handle)  /* handle to the PCM player */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_pcm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_pcm_stop_playback,aud_resource_table[aud_resource_playback_pcm].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_pcm_stop_playback, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_pcm_intern_start_recording
* Return.....: return code
* Description: Starts a PCM/ADPCM/wave recording.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_pcm_intern_start_recording(
    U8 handle,                              /* handle to the PCM recorder */
    aud_pcm_sample_rate_enum sample_rate,   /* sample rate */
    aud_media_enum media_type,              /* where should the data be placed */
    aud_pcm_format_enum format,             /* PCN/ADPCM/wave */
    U16 DWD_HUGE *p_file_handle,            /* file/buffer handle */
    U32 buffer_size,                        /* size of buffer if RAM is used */
    U32 start_offset)                       /* the number of bytes that should be skipped in the start */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if !defined (FFS_PRESENT)
    if(aud_media_ffs == media_type)
    {
        res = aud_rc_not_supported;
    }
    else
#endif
#if !defined (MMCI_PRESENT) || !defined (FAT_FS_PRESENT)
    if( (aud_media_mmc == media_type)
#if defined (AUD_NANDDEVICE_SUPPORT)
        || (aud_media_fs_cdrive == media_type)
        || (aud_media_fs_edrive == media_type)
        || (aud_media_fs_fdrive == media_type)
#endif
      )
    {
        res = aud_rc_not_supported;
    }
    else
#endif
    if(format>=aud_pcm_format_end || media_type>=aud_media_end || sample_rate>=aud_pcm_rate_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_record_pcm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
#if defined (AUD_NANDDEVICE_SUPPORT)
        if(aud_media_fs_edrive == media_type)
        {
            res = aud_rc_not_supported;
        }
        else
#endif
        {
            aud_send_mail(E_pcm_start_recording, aud_resource_table[aud_resource_record_pcm].caller_id,
                 sample_rate, media_type, format, (U32)p_file_handle, buffer_size, start_offset, 0, 0, NULL, NULL);
            res = aud_rc_ok;
      	}
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_pcm_start_recording, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_pcm_intern_stop_recording
* Return.....: return code
* Description: Stops the PCM recording process.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_pcm_intern_stop_recording(
    U8 handle)  /* handle to the PCM recorder */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_record_pcm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_pcm_stop_recording, aud_resource_table[aud_resource_record_pcm].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_pcm_stop_recording, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_pcm_intern_getbuffer_function
* Return.....: return code
* Description: The application is responsible for providing and handling the
*              buffers for PCM playback and recording. Control this buffer.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_pcm_intern_getbuffer_function(
    U8 handle,                          /* handle to the PCM player or recorder */
    ptr_GetAudioBufferFunction func,    /* callback function that is called when new data needs to be played */
    void *p_user_data)                  /* maintains the state of the callback functions */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_record_pcm))
    {
        aud_send_mail(E_pcm_rec_set_cb_func , aud_resource_table[aud_resource_playback_pcm].caller_id,
                 (U32)func, (U32)p_user_data, 0, 0, 0, 0, 0, 0, NULL, NULL);
        res = aud_rc_ok;
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_pcm))
    {
       res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_pcm_play_set_cb_func, aud_resource_table[aud_resource_record_pcm].caller_id,
                 (U32)func, (U32)p_user_data, 0, 0, 0, 0, 0, 0, NULL, NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_pcm_play_set_cb_func, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_pcm_intern_suspend
* Return.....: return code
* Description: Suspends the playback of PCM data.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_pcm_intern_suspend(
    U8 handle,  /* handle to the PCM player */
    U8 slot_id) /* the slot (0-4) where the PCM data is stored for later resumption */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(slot_id > AUD_NOF_SUSPEND_SLOTS)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_pcm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_pcm_suspend, aud_resource_table[aud_resource_playback_pcm].caller_id,slot_id,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_pcm_suspend, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_pcm_intern_resume
* Return.....: return code
* Description: Resumes PCM playback for the selected slot.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_pcm_intern_resume(
    U8 handle,  /* handle to the PCM player */
    U8 slot_id) /* the suspend slot (0-4) to resume from */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(slot_id > AUD_NOF_SUSPEND_SLOTS)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_pcm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_pcm_resume, aud_resource_table[aud_resource_playback_pcm].caller_id,slot_id,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_pcm_resume, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_pcm_intern_stop_suspend
* Return.....: return code
* Description: Clears the suspended data for the specified slot.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_pcm_intern_stop_suspend(
    U8 handle,  /* handle to the PCM player */
    U8 slot_id) /* the suspend slot (0-4) to delete */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(slot_id >= AUD_NOF_SUSPENDNEST)
    {
        res = (aud_rc_parameter_out_of_range);
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_pcm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_pcm_stop_suspend,  aud_resource_table[aud_resource_playback_pcm].caller_id,slot_id,0,0,0,0,0,0,0,NULL,NULL);
        res = (aud_rc_ok);
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_pcm_stop_suspend, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
} /*End of Aud_mp3_stop_suspend */

/*------------------------------------------------------------------------------
* Function...: AUD_pcm_intern_get_total_playtime
* Return.....: return code
* Description: Returns the total playtime for the selected data.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_pcm_intern_get_total_playtime(
    U8 handle,                              /* not used */
    aud_media_enum media_type,              /* where is the data located */
    aud_pcm_format_enum format,             /* PCM/ADPCM/Wave */
    aud_pcm_sample_rate_enum sample_rate,   /* sample rate */
    aud_pcm_mode_enum mode,                 /* mono/dual mono/stereo */
    U8 bit_rate,                            /* the number of bits per sample */
    U16 DWD_HUGE *p_file_handle,            /* file/buffer handle */
    U32 buffer_size,                        /* size of buffer if RAM is used */
    U32 offset)                             /* how many byres should be skipped in the beginning */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if !defined (FFS_PRESENT)
    if(aud_media_ffs == media_type)
    {
        res = aud_rc_not_supported;
    }
    else
#endif
#if !defined (MMCI_PRESENT) || !defined (FAT_FS_PRESENT)
    if( (aud_media_mmc == media_type)
#if defined (AUD_NANDDEVICE_SUPPORT)
        || (media_type==aud_media_fs_cdrive == media_type)
        || (aud_media_fs_edrive == media_type)
        || (aud_media_fs_fdrive == media_type)
#endif
      )
    {
        res = aud_rc_not_supported;
    }
    else
#endif
    if(format>=aud_pcm_format_end || media_type>=aud_media_end || sample_rate>=aud_pcm_rate_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else
    {
        aud_send_mail(E_pcm_get_total_playtime, aud_resource_table[aud_resource_playback_pcm].caller_id,
                 media_type, format, sample_rate, buffer_size, mode , bit_rate , offset , (U32)p_file_handle, NULL, NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_pcm_get_total_playtime, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_pcm_intern_get_play_position
* Return.....: return code
* Description: Returns the elapsed time of the currently playing file.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_pcm_intern_get_play_position(
    U8 handle)  /* handle to the PCM player */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_pcm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_pcm_get_play_position, aud_resource_table[aud_resource_playback_pcm].caller_id,0,0,0,0,0,0,0,0,NULL,NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_pcm_get_play_position, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_pcm_intern_set_play_position
* Return.....: return code
* Description: Set the position from where playback should start.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_pcm_intern_set_play_position(
    U8 handle,                  /* handle to the PCM player */
    U32 pos,                    /* the position from where playback should start */
    U16 DWD_HUGE *p_file_handle,/* not used */
    U32 buffer_size)            /* not used */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_playback_pcm))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        aud_send_mail(E_pcm_set_play_position, aud_resource_table[aud_resource_playback_pcm].caller_id,
                 pos, buffer_size, 0, 0, 0, 0, 0, (U32)p_file_handle, NULL, NULL);
        res = aud_rc_ok;
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_pcm_set_play_position, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*******************************************************************************
*
*             Yamaha Music API (mapi) related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...:
* Return.....:
* Description: Audio driver interface for MA3 through Yamaha Music API.
* Created:.... 24.11.2004 by TS (DWD)
* Modified...: dd.mm.yyyy by NN
* Note:....... This function is a different way of accessing the external
*              Yamaha ringer. Use either aud_resource_mapi for the mapi
*              interface for direct access to Yamaha's driver OR
*              aud_resource_ringer for access through the AUD_ringer_xxx functions.
------------------------------------------------------------------------------*/
S8 AUD_mapi(
    U8 handle,                  /* handle to the mapi resource */
    aud_mapi_if_enum func_id,   /* identifies which Music API function should be called */
    U32 parm1,                  /* generic parameter */
    U32 parm2,                  /* generic parameter */
    U32 parm3,                  /* generic parameter */
    U32 parm4,                  /* generic parameter */
    U32 parm5,                  /* generic parameter */
    U32 parm6,                  /* generic parameter */
    U32 parm7)                  /* generic parameter */
{
    S8 res;

#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_function_interface, 11, 0, /* LLT params */
                  'd', 0x06,                                               /* LIS params */
                  aud_log_lis_raw_data, aud_log_lis_display_data, aud_log_llt_trace, /* LIS params */
                  (U32)E_aud_mapi_msg,
                  (U32)handle,
                  (U32)func_id,
                  (U32)parm1,
                  (U32)parm2,
                  (U32)parm3,
                  (U32)parm4,
                  (U32)parm5,
                  (U32)parm6,
                  (U32)parm7,
                  (U32)E_audio_llt_type_id_7 );
#endif
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (AUD_YAMAHA_MAPI)
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_mapi))
    {
        res = (aud_rc_sharing_violation);
    }
    else if(AUD_TEST_HANDLE == handle)
    {
        res = (aud_rc_handle_not_used);
    }
    else if(func_id >= aud_mapi_melody_end)
    {
        res = (aud_rc_parameter_out_of_range);
    }
    else
    {
        aud_send_mail(E_aud_mapi_msg, aud_resource_table[aud_resource_mapi].caller_id,
                 func_id, parm1, parm2, parm3, parm4, parm5, parm6, parm7, NULL, NULL);
        res = (aud_rc_ok);
    }
#else
    res = (aud_rc_not_supported);
#endif /* AUD_YAMAHA_MAPI */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_aud_mapi_msg, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*******************************************************************************
*
*               Non-baseband resource related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_non_baseband_resource_start
* Return.....: return code
* Description: Signals that a resource outside the baseband chip has been
*              started.
* Created....: 19.03.2007 by KBJ
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_non_baseband_resource_start(
    U8 handle)  /* handle to the virtual (inband) resource */
{
    S8 res;
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    /* if inband resource not allocated, return error */
    if(aud_rc_ok != (aud_rc_ok != aud_intf_check_handle(handle, aud_resource_inband)))//xiu
    {
        res = aud_rc_sharing_violation;
    }
    else /* otherwise notify the audio driver process */
    {
        aud_send_mail(E_inband_start, 0, 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
        res = aud_rc_ok;
    }
    /* if an error has occured, notify the audio driver process about it (for debugging issues) */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_inband_start, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res); /* return the result to the calling application */
}

/*------------------------------------------------------------------------------
* Function...: AUD_non_baseband_resource_stop
* Return.....: return code
* Description: Signals that a resource outside the baseband chip has been
*              stopped.
* Created....: 19.03.2007 by KBJ
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_non_baseband_resource_stop(
    U8 handle)  /* handle to the virtual (inband) resource */
{
    S8 res;
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    /* if inband resource not allocated, return error */
    if(aud_rc_ok != (aud_rc_ok != aud_intf_check_handle(handle, aud_resource_inband)))//xiu
    {
        res = aud_rc_sharing_violation;
    }
    else /* otherwise notify the audio driver process */
    {
        aud_send_mail(E_inband_stop, 0, 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
        res = aud_rc_ok;
    }
    /* if an error has occured, notify the audio driver process about it (for debugging issues) */
    if(res<aud_rc_ok) {
        aud_send_mail(E_interface_request_error, 0, E_inband_stop, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res); /* return the result to the calling application */
}

/*******************************************************************************
*
*               SM Power backspeaker related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_SMPower_enable
* Return.....: aud_rc_ok
* Description: Equivalent to AUD_non_baseband_resource_start.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_SMPower_enable(
    U8 handle)  /* handle to the SM power backspeaker resource (not used) */
{
    aud_send_mail(E_smpower_enable, 0, 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_ok);
}

/*------------------------------------------------------------------------------
* Function...: AUD_SMPower_disable
* Return.....: aud_rc_ok
* Description: Equivalent to AUD_non_baseband_resource_stop.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_SMPower_disable(
    U8 handle)  /* handle to the SM power backspeaker resource (not used) */
{
    aud_send_mail(E_smpower_disable, 0, 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_ok);
}

/*******************************************************************************
*
*                  SBC Encoder related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_sbc_encoder_start
* Return.....: return code
* Description: Start encoding the data provided in the input buffer.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_sbc_encoder_start(
    U8 handle,                                          /* handle to the SBC Encoder */
    aud_sbc_encoder_data_mode_enum data_mode,           /* internal/external encoding */
    aud_sbc_encoder_channel_mode_enum channel_mode,     /* mono/dual mono/stereo/joint stereo */
    aud_sbc_encoder_sample_rate_enum sample_rate,       /* sample rate */
    aud_sbc_encoder_number_of_blocks_enum no_of_blocks, /* number of encoded blocks per frame */
    U8 bitpool)                                         /* number of encoded bits per block (2-53) */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (SBC_ENCODER)
    /* Check handle */
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_sbc_encoder))
    {
        if( (channel_mode >= aud_sbc_channel_mode_end) || (sample_rate >= aud_sbc_sample_rate_end) || (no_of_blocks >= aud_sbc_number_of_blocks_end) || (data_mode >= aud_sbc_data_mode_end) || (bitpool > 53) )
        {
            res=aud_rc_parameter_out_of_range;
        }
        else /* All OK */
        {
            aud_send_mail(E_sbc_start_encoder, aud_resource_table[aud_resource_sbc_encoder].caller_id, (U32)channel_mode, (U32)sample_rate, (U32)no_of_blocks, (U32)bitpool, (U32)data_mode, 0,0,0, NULL, NULL);
            res=aud_rc_ok;
        }
    }
    else
        res = aud_rc_sharing_violation;
#else
    res = aud_rc_no_hw_support;
#endif /* SBC_ENCODER */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_sbc_start_encoder, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_sbc_encoder_stop
* Return.....: return code
* Description: Stops the SBC Encoder.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_sbc_encoder_stop(
    U8 handle)  /* handle to the SBC Encoder */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (SBC_ENCODER)
    /* Check handle */
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_sbc_encoder)) /* All OK */
    {
        aud_send_mail(E_sbc_stop_encoder, aud_resource_table[aud_resource_sbc_encoder].caller_id,0,0, 0,0,0,0,0,0, NULL, NULL);
        res = aud_rc_ok;
    }
    else
    {
        res = aud_rc_sharing_violation;
    }
#else
    res=(aud_rc_no_hw_support);
#endif /* SBC_ENCODER */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_sbc_stop_encoder, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_sbc_encoder_suspend
* Return.....: return code
* Description: Suspends encoding of input data (only in external mode).
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_sbc_encoder_suspend(
    U8 handle,  /* handle to the SBC Encoder */
    U8 slot_id) /* the slot (0-4) where the data is stored for later resumption */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (SBC_ENCODER)
    /* Check handle */
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_sbc_encoder))
    {
        if(slot_id > 5)
        {
            res = (aud_rc_parameter_out_of_range);
        }
        else /* All OK */
        {
            aud_send_mail(E_sbc_suspend_encoder, aud_resource_table[aud_resource_sbc_encoder].caller_id, (U32)slot_id, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
            res = (aud_rc_ok);
        }
    }
    else
    {
        res = (aud_rc_sharing_violation);
    }
#else
    res = (aud_rc_no_hw_support);
#endif /* SBC_ENCODER */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_sbc_suspend_encoder, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_sbc_encoder_resume
* Return.....: return code
* Description: Resumes encoding of input data (only in external mode).
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_sbc_encoder_resume(
    U8 handle,  /* handle to the SBC Encoder */
    U8 slot_id) /* the suspend slot (0-4) to resume from */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (SBC_ENCODER)
    /* Check handle */
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_sbc_encoder))
    {
        if(slot_id > 5)
        {
            res = (aud_rc_parameter_out_of_range);
        }
        else /* All OK */
        {
            aud_send_mail(E_sbc_resume_encoder, aud_resource_table[aud_resource_sbc_encoder].caller_id, (U32)slot_id, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
            res = (aud_rc_ok);
        }
    }
    else
    {
        res = (aud_rc_sharing_violation);
    }
#else
    res = (aud_rc_no_hw_support);
#endif /* SBC_ENCODER */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_sbc_resume_encoder, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}


/*------------------------------------------------------------------------------
* Function...: AUD_sbc_encoder_set_getinputbuffer_function
* Return.....: return code
* Description: The application is responsible for providing and handling the
*              input buffer for SBC encoding. Control this buffer (external).
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_sbc_encoder_set_getinputbuffer_function(
    U8 handle,                                          /* handle to the SBC Encoder */
    ptr_GetAudioBufferFunction input_buffer_function,   /* callback function that is called when new data needs to be encoded */
    void* p_user_data)                                  /* one of the parameters for the callback function */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (SBC_ENCODER)
    /* Check handle */
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_sbc_encoder)) /*  All OK */
    {
        aud_send_mail(E_sbc_set_getinputbuffer_function, aud_resource_table[aud_resource_sbc_encoder].caller_id, 0, 0, 0, 0, 0, 0, (U32)input_buffer_function, (U32)p_user_data, NULL, NULL);
        res = (aud_rc_ok);
    }
    else
    {
        res = (aud_rc_sharing_violation);
    }
#else
    res = (aud_rc_no_hw_support);
#endif /* SBC_ENCODER */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_sbc_set_getinputbuffer_function, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_sbc_encoder_set_getoutputbuffer_function
* Return.....: return code
* Description: The application is responsible for providing and handling the
*              output buffer for SBC encoding. Control this buffer.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_sbc_encoder_set_getoutputbuffer_function(
    U8 handle,                                          /* handle to the SBC Encoder */
    ptr_GetAudioBufferFunction output_buffer_function,  /* callback function that is called when new data needs to be stored */
    void* p_user_data)                                    /* one of the parameters for the callback function */
{
    S8 res;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
#if defined (SBC_ENCODER)
    /* Check handle */
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_sbc_encoder)) /* All OK */
    {
        aud_send_mail(E_sbc_set_getoutputbuffer_function, aud_resource_table[aud_resource_sbc_encoder].caller_id, 0, 0, 0, 0, 0, 0, (U32)output_buffer_function, (U32)p_user_data, NULL, NULL);
        res = (aud_rc_ok);
    }
    else
    {
        res = (aud_rc_sharing_violation);
    }
#else
    res = (aud_rc_no_hw_support);
#endif /* SBC_ENCODER */
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_sbc_set_getoutputbuffer_function, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*******************************************************************************
*
*                 I2S2 Rx and Tx related interface functions
*
*******************************************************************************/
#if defined (aud_old_I2S_interface)
/*------------------------------------------------------------------------------
* Function...: AUD_configure_pcm_channel
* Return.....: return code
* Description: Configure the PCM channel (I2S2) in a certain mode.
* Modified...: dd.mm.yyyy by NN
* Note.......: Function calls are mapped to the new I2S2 interface functions.
------------------------------------------------------------------------------*/
S8 AUD_configure_pcm_channel(
    U8 handle,                  /* handle to the resource */
    aud_I2Sx_mode_enum mode)    /* sample rate, mono/stereo, master/slave etc. */
{
    S8 res = aud_rc_ok;
    U16 mode_pointer = 0;

#ifdef AUDIO_LLT
    aud_common_trace(llt_group_audio, llt_type_audio_function_interface, 4, 0,
                   (ulong)'G', 0x2,
                   aud_log_lis_display,aud_log_lis_display_data,aud_log_lis_llt_trace,
                   (U32)E_pcm_ch_configure, (U32)handle, (U32)mode, (U32)E_audio_llt_type_id_10);

#endif
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(mode >= aud_I2Sx_mode_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    /* Checking to see if the mode is allowed and finding the pointer to it */
    else
    {
        if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_PCM_channel))
        {
            res = aud_rc_sharing_violation;
        }
        else
        {
            for(mode_pointer=0; mode_pointer<=aud_number_of_temp_i2s2_modes; mode_pointer++)
            {
                if(mode_pointer==aud_number_of_temp_i2s2_modes)
                {
                    res = aud_rc_parameter_out_of_range;
                    break;
                }
                if(aud_i2s2_temp_configuration[mode_pointer].Enum==mode)
                {
                    aud_I2S2_mode_pointer = mode_pointer;
                    break;
                }
            } /* end for */
        }
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_pcm_ch_configure, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_pcm_channel_routing
* Return.....: return code
* Description: Configures when to route PCM channel input from and output to.
* Modified...: dd.mm.yyyy by NN
* Note.......: Function calls are mapped to the new I2S2 interface functions.
------------------------------------------------------------------------------*/
S8 AUD_pcm_channel_routing(
    U8 handle,                  /* handle to the resource */
    U8 input_to_uplink,         /* route the input from the PCM channel in the uplink direction */
    U8 input_to_downlink,       /* route the input from the PCM channel in the downlink direction */
    U8 output_from_uplink,      /* route uplink data to the PCM channel */
    U8 output_from_downlink)    /* route downlink data to the PCM channel */
{
    S8 res = aud_rc_ok;

#ifdef AUDIO_LLT
    aud_common_trace(llt_group_audio, llt_type_audio_function_interface, 7, 0,
                   (ulong)'G', 0x2,
                   aud_log_lis_display,aud_log_lis_display_data,aud_log_lis_llt_trace,
                   (U32)E_pcm_ch_routing, (U32)handle, (U32)input_to_uplink, (U32)input_to_downlink,
                   (U32)output_from_uplink, (U32)output_from_downlink, (U32)E_audio_llt_type_id_11);
#endif

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_PCM_channel))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        if(input_to_uplink)
        {
            AUD_mute_resource(AUD_I2S2_RX_TEST_HANDLE, aud_mute_disable, aud_direction_resource_uplink);
        }
        else
        {
            AUD_mute_resource(AUD_I2S2_RX_TEST_HANDLE, aud_mute_enable, aud_direction_resource_uplink);
        }
        if(input_to_downlink)
        {
            AUD_mute_resource(AUD_I2S2_RX_TEST_HANDLE, aud_mute_disable, aud_direction_resource_downlink);
        }
        else
        {
            AUD_mute_resource(AUD_I2S2_RX_TEST_HANDLE, aud_mute_enable, aud_direction_resource_downlink);
        }
        if(output_from_uplink)
        {
            AUD_mute_resource(AUD_I2S2_TX_TEST_HANDLE, aud_mute_disable, aud_direction_resource_uplink);
        }
        else
        {
            AUD_mute_resource(AUD_I2S2_TX_TEST_HANDLE, aud_mute_enable, aud_direction_resource_uplink);
        }
        if(output_from_downlink)
        {
            AUD_mute_resource(AUD_I2S2_TX_TEST_HANDLE, aud_mute_disable, aud_direction_resource_downlink);
        }
        else
        {
            AUD_mute_resource(AUD_I2S2_TX_TEST_HANDLE, aud_mute_enable, aud_direction_resource_downlink);
        }
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_pcm_ch_routing, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_pcm_channel_enable
* Return.....: return code
* Description: Enables the PCM channel in the specified directions.
* Modified...: dd.mm.yyyy by NN
* Note.......: Function calls are mapped to the new I2S2 interface functions.
------------------------------------------------------------------------------*/
S8 AUD_pcm_channel_enable(
    U8 handle,                          /* handle to the resource */
    aud_pcm_direction_enum direction)   /* enables rx, tx or both directions */
{
    S8 res = aud_rc_ok;

#ifdef AUDIO_LLT
    aud_common_trace(llt_group_audio, llt_type_audio_function_interface, 4, 0,
                   (ulong)'G', 0x2,
                   aud_log_lis_display,aud_log_lis_display_data,aud_log_lis_llt_trace,
                   (U32)E_pcm_ch_routing, (U32)handle, (U32)direction, (U32)E_audio_llt_type_id_12);
#endif

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(direction >= aud_pcm_direction_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    else if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_PCM_channel))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        res = AUD_I2S_setup_clock(aud_i2s2_temp_configuration[aud_I2S2_mode_pointer].clock, aud_i2s2_temp_configuration[aud_I2S2_mode_pointer].Master_Slave, aud_i2s2_temp_configuration[aud_I2S2_mode_pointer].samplerate, aud_i2s2_temp_configuration[aud_I2S2_mode_pointer].sample_width, aud_i2s2_temp_configuration[aud_I2S2_mode_pointer].transmission_mode, aud_i2s2_temp_configuration[aud_I2S2_mode_pointer].settings, aud_i2s2_temp_configuration[aud_I2S2_mode_pointer].I2S_mode);
        if(aud_rc_ok == res)
        {
            if(direction==aud_pcm_direction_rx || direction==aud_pcm_direction_rx_tx )
            {
                res = AUD_I2S_setup_resource(AUD_I2S2_RX_TEST_HANDLE, aud_i2s2_temp_configuration[aud_I2S2_mode_pointer].clock, aud_i2s2_temp_configuration[aud_I2S2_mode_pointer].entry_point);
                res |= AUD_I2S_enable_resource(AUD_I2S2_RX_TEST_HANDLE);
            }
            if(direction==aud_pcm_direction_tx || direction==aud_pcm_direction_rx_tx )
            {
                res = AUD_I2S_setup_resource(AUD_I2S2_TX_TEST_HANDLE, aud_i2s2_temp_configuration[aud_I2S2_mode_pointer].clock, aud_i2s2_temp_configuration[aud_I2S2_mode_pointer].entry_point);
                res |= AUD_I2S_enable_resource(AUD_I2S2_TX_TEST_HANDLE);
            }
        } /* end if res==aud_rc_ok */
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_pcm_enable, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_pcm_channel_disable
* Return.....: return code
* Description: Disables the PCM channel in both directions.
* Modified...: dd.mm.yyyy by NN
* Note.......: Function calls are mapped to the new I2S2 interface functions.
------------------------------------------------------------------------------*/
S8 AUD_pcm_channel_disable(
    U8 handle)  /* handle to the resource */
{
    S8 res = aud_rc_ok;

#ifdef AUDIO_LLT
    aud_common_trace(llt_group_audio, llt_type_audio_function_interface, 3, 0,
                   (ulong)'G', 0x2,
                   aud_log_lis_display,aud_log_lis_display_data,aud_log_lis_llt_trace,
                   (U32)E_pcm_ch_routing, (U32)handle, (U32)E_audio_llt_type_id_13);
#endif 

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(aud_rc_ok != aud_intf_check_handle(handle, aud_resource_PCM_channel))
    {
        res = aud_rc_sharing_violation;
    }
    else
    {
        AUD_I2S_disable_resource(AUD_I2S2_RX_TEST_HANDLE);
        AUD_I2S_disable_resource(AUD_I2S2_TX_TEST_HANDLE);
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_pcm_disable, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}
#endif /* aud_old_I2S_interface */

/*------------------------------------------------------------------------------
* Function...: AUD_I2S_enable_resource
* Return.....: return code
* Description: Enables the requested I2S2 resource.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_I2S_enable_resource(
    U8 handle)  /* handle to the rx or tx resource */
{
    aud_i2s_errors_enum error = aud_I2S_ok;
    S8 res;
    aud_resource_enum resource;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */

    /* I2S2 Rx */
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_I2S2_Rx))
    {
        resource = aud_resource_I2S2_Rx;
        if(aud_I2S_status.I2S2_Rx_enabled)
        {
            if(aud_I2S2_clk0==aud_I2S_status.I2S2_Rx_clk && aud_I2S_status.clk0_enabled)
            {
                res = aud_rc_ok;
            }
            else if(aud_I2S2_clk1==aud_I2S_status.I2S2_Rx_clk && aud_I2S_status.clk1_enabled)
            {
                res = aud_rc_ok;
            }
            else
            {
                res = aud_rc_I2S_setup_error;
                error = aud_I2S_chosen_clk_not_configured;
            }
        }
        else
        {
            res = aud_rc_I2S_setup_error;
            error = aud_I2S_resource_not_configured;
        }
    }
    /* I2S2 Tx */
    else if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_I2S2_Tx))
    {
        resource = aud_resource_I2S2_Tx;
        if(aud_I2S_status.I2S2_Tx_enabled)
        {
            if(aud_I2S2_clk0==aud_I2S_status.I2S2_Tx_clk && aud_I2S_status.clk0_enabled)
            {
                res = aud_rc_ok;
            }
            else if(aud_I2S2_clk1==aud_I2S_status.I2S2_Tx_clk && aud_I2S_status.clk1_enabled)
            {
                res = aud_rc_ok;
            }
            else
            {
                res = aud_rc_I2S_setup_error;
                error = aud_I2S_resource_not_configured;
            }
        }
        else
            res = aud_rc_I2S_setup_error;
    }
    else /* Neither I2S2 Rx nor I2S2 Tx */
    {
        resource = aud_resource_end;
        res = aud_rc_sharing_violation;
    }

    /* Send mail to audio driver */
    if(aud_rc_ok == res)
    {
        aud_send_mail(E_I2S_enable, 0, resource, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    else
    {
        aud_send_mail(E_interface_request_error, 0, E_I2S_enable, res, handle, error, resource, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_I2S_disable_resource
* Return.....: return code
* Description: Disables the requested I2S2 resource.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_I2S_disable_resource(
    U8 handle)  /* handle to the rx or tx resource */
{
    S8 res;
    aud_resource_enum resource;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */

    /* Rx */
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_I2S2_Rx))
    {
        resource = aud_resource_I2S2_Rx;
        res=aud_rc_ok;
    }
    /* Tx */
    else if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_I2S2_Tx))
    {
        resource = aud_resource_I2S2_Tx;
        res=aud_rc_ok;
    }
    else /* Error */
    {
        res = aud_rc_sharing_violation;
    }

    /* Send mail to audio driver */
    if(aud_rc_ok == res)
    {
        aud_send_mail(E_I2S_disable, 0, resource, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    else
    {
        aud_send_mail(E_interface_request_error, 0, E_I2S_disable, res, handle, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_I2S_setup_clock
* Return.....: return code
* Description: Sets up one of the clocks for the I2S2 resources.
* Modified...: 07.08.2007 by KBJ (removed lint error)
------------------------------------------------------------------------------*/
S8 AUD_I2S_setup_clock(
    aud_I2S_clk_enum clock,                             /* which clock to configure */
    aud_I2S_Master_Slave_enum Master_Slave,             /* master/slave mode */
    aud_I2S_samplerate_enum samplerate,                 /* sample rate */
    aud_I2S_sample_width_enum sample_width,             /* number of bits in each sample */
    aud_I2S_transmission_mode_emum transmission_mode,   /* PCM mode or standard I2S protocol */
    aud_I2S_setting_enum settings,                      /* normal or special settings */
    aud_I2S_mode_enum I2S_mode)                         /* mono/dual mono/stereo */
{
    S8 res = aud_rc_ok;
    U32 validation_rc = 0;
    U32 entry_point=0;
    aud_i2s_errors_enum error = aud_I2S_ok;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */

    /* Do range check */
    res = aud_rc_ok;
    if(clock >= aud_I2S_clk_end)
    {
        res = aud_rc_parameter_out_of_range;
        error = aud_I2S_clock_enum_out_of_range;
    }
    else if(Master_Slave >= aud_I2S_Master_Slave_end)
    {
        res = aud_rc_parameter_out_of_range;
        error = aud_I2S_Master_Slave_enum_out_of_range;
    }
    else if(samplerate >= aud_I2S_samplerate_end)
    {
        res = aud_rc_parameter_out_of_range;
        error = aud_I2S_samplerate_enum_out_of_range;
    }
    /* Only 16 bit sample width is possible at the moment due to FW restrictions,
        all other parts of the I2S driver are prepared */
    else if(sample_width != aud_I2S_sample_width_16)
    {
        res = aud_rc_parameter_out_of_range;
        error = aud_I2S_sample_width_enum_out_of_range;
    }
    else if(transmission_mode >= aud_I2S_transmission_mode_end)
    {
        res = aud_rc_parameter_out_of_range;
        error = aud_I2S_transmission_mode_enum_out_of_range;
    }
    else if(settings >= aud_I2S_setting_end)
    {
        res = aud_rc_parameter_out_of_range;
        error = aud_I2S_settings_enum_out_of_range;
    }
    else if(I2S_mode >= aud_I2S_mode_end)
    {
        res = aud_rc_parameter_out_of_range;
        error = aud_I2S_mode_enum_out_of_range;
    }

    /* Do a check to validate if the settings are possible */
    else if(TRUE != aud_I2S2_default_settings[settings].avaliable)
    {
        res = aud_rc_parameter_out_of_range;
        error = aud_I2S_setting_not_avaliable;
    }

    else
    {
        /* Test all different entry points */
        for(entry_point=0; entry_point<aud_I2S_entry_point_end; entry_point++)
        {
            validation_rc = validation_rc | aud_i2s_validation_matrix[entry_point][Master_Slave][I2S_mode][transmission_mode][samplerate];
        }
        if(TRUE != validation_rc)
        {
            res = aud_rc_I2S_setup_error;
            error = aud_I2S_validation_matrix_error;
        }
    }

    /* Send mail to audio driver */
    if(aud_rc_ok == res)
    {
        if(aud_I2S2_clk0 == clock)
        {
            aud_I2S_status.clk0_enabled = TRUE;
        }
        else if(aud_I2S2_clk1 == clock)
        {
            aud_I2S_status.clk1_enabled = TRUE;
        }
        aud_send_mail(E_I2S_setup_clk, 0, clock, Master_Slave, samplerate, sample_width, transmission_mode, settings, I2S_mode, 0, NULL, NULL);
    }
    else
    {
        aud_send_mail(E_interface_request_error, 0, E_I2S_setup_clk, res, error, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*------------------------------------------------------------------------------
* Function...: AUD_I2S_setup_resource
* Return.....: return code
* Description: Sets up an I2S2 resource.
* Modified...: 01.10.2007 by KBJ (added Steon2 exceptions)
------------------------------------------------------------------------------*/
S8 AUD_I2S_setup_resource(
    U8 handle,                              /* handle to the rx or tx resource */
    aud_I2S_clk_enum clock,                 /* which clock to use for this resource */
    aud_I2S_entry_point_enum entry_point)   /* which entry point (MMS/external) to route data to/from */
{
    S8 res = aud_rc_ok;
    aud_resource_enum resource = aud_resource_end;
    aud_i2s_errors_enum error = aud_I2S_ok;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */

    /* Rx */
    if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_I2S2_Rx))
    {
        resource = aud_resource_I2S2_Rx;
        res=aud_rc_ok;
    }
    /* Tx */
    else if(aud_rc_ok == aud_intf_check_handle(handle, aud_resource_I2S2_Tx))
    {
        resource = aud_resource_I2S2_Tx;
        res=aud_rc_ok;
    }
    else /* Error */
    {
        res = aud_rc_sharing_violation;
    }

    /* Parameter range check */
    if(clock >= aud_I2S_clk_end)
    {
        res=aud_rc_parameter_out_of_range;
        error = aud_I2S_clock_enum_out_of_range;
    }
    if(entry_point >= aud_I2S_entry_point_end)
    {
        res=aud_rc_parameter_out_of_range;
        error = aud_I2S_entry_point_enum_out_of_range;
    }
    /* Note: For Steon2, the Tx External entry point is allowed */
#if defined (STEON2_PROJECT)
    /* Map the entry point MMS for Steon2 project to new entry point MMS at frame based part */
    if(aud_I2S_entry_point_MMS == entry_point)
    {
        entry_point = aud_I2S_entry_point_MMS_FB;
    }
#else	
    if(aud_I2S_entry_point_External == entry_point && aud_resource_I2S2_Tx == resource)
    {
        res=aud_rc_parameter_out_of_range;
        error = aud_I2S_conflict_between_entry_point_and_Tx_resource;
    }
#endif /* STEON2_PROJECT */

    /* Send mail to audio driver */
    if(aud_rc_ok == res)
    {
        if(aud_resource_I2S2_Rx == resource)
        {
            aud_I2S_status.I2S2_Rx_enabled = TRUE;
            aud_I2S_status.I2S2_Rx_clk=clock;
        }
        else if(aud_resource_I2S2_Tx == resource)
        {
            aud_I2S_status.I2S2_Tx_enabled = TRUE;
            aud_I2S_status.I2S2_Tx_clk=clock;
        }
        aud_send_mail(E_I2S_setup_resource, 0, resource, clock, entry_point, 0, 0, 0, 0, 0, NULL, NULL);
    }
    else
    {
        aud_send_mail(E_interface_request_error, 0, E_I2S_setup_resource, res, handle, error, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}
#endif
/*******************************************************************************
*
*                I2S1 (bluetooth) related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_configure_bluetooth
* Return.....: return code
* Description: Configures the I2S1 resource used for Bluetooth.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_configure_bluetooth(
    U8 handle,                  /* not used */
    aud_I2Sx_mode_enum mode)    /* sample rate, mono/stereo, master/slave etc. */
{
    S8 res = aud_rc_ok;
    U16 mode_pointer = 0;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(mode >= aud_I2Sx_mode_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    /* Checking to see if the mode is allowed and finding the pointer to it */
    else
    {
        for(mode_pointer=0; mode_pointer<=aud_number_of_i2s1_modes; mode_pointer++)
        {
            if(aud_number_of_i2s1_modes == mode_pointer)
            {
                res = aud_rc_parameter_out_of_range;
                break;
            }
            if(aud_i2s1_configuration[mode_pointer].I2S_mode_enum == mode)
            {
                break;
            }
        }
        if(res != aud_rc_parameter_out_of_range)
        {
            aud_send_mail(E_I2S1_configure, 0, mode, 0, 0, 0, 0, 0, 0, 0, NULL, NULL); /* dummy - for traceability */
            res = aud_rc_ok;
        }
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_I2S1_configure, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*[Begin] [lvwenhua-2010/11/9]*/
/*------------------------------------------------------------------------------
* Function...: AUD_configure_afei2s
* Return.....: return code
* Description: Configures the afeI2S.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_configure_afei2s(
    U8 handle,                  /* not used */
    aud_I2Sx_mode_enum mode)    /* sample rate, mono/stereo, master/slave etc. */
{
    S8 res = aud_rc_ok;
    U16 mode_pointer = 0;

    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    if(mode >= aud_I2Sx_mode_end)
    {
        res = aud_rc_parameter_out_of_range;
    }
    /* Checking to see if the mode is allowed and finding the pointer to it */
    else
    {
        for(mode_pointer=0; mode_pointer<=aud_number_of_afei2s_modes; mode_pointer++)
        {
            if(aud_number_of_afei2s_modes == mode_pointer)
            {
                res = aud_rc_parameter_out_of_range;
                break;
            }
            if(aud_afei2s_configuration[mode_pointer].I2S_mode_enum == mode)
            {
                break;
            }
        }
        if(res != aud_rc_parameter_out_of_range)
        {
            aud_send_mail(E_AFEI2S_configure, 0, mode, 0, 0, 0, 0, 0, 0, 0, NULL, NULL); /* dummy - for traceability */
            res = aud_rc_ok;
        }
    }
    if(res<aud_rc_ok)
    {
        aud_send_mail(E_interface_request_error, 0, E_AFEI2S_configure, res, 0, 0, 0, 0, 0, 0, NULL, NULL);
    }
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (res);
}

/*[End] [lvwenhua-2010/11/9]*/
#ifdef _USE_CUT_TEAKLIT_CODE
/*******************************************************************************
*
*                  3D-amplifier related interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_3D_mode_enable
* Return.....: return code
* Description: Enables 3D stereo enhancement if the hardware supports it.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_3D_mode_enable(void)
{
#if defined (AUD_3D_AMP_PRESENT)
    aud_send_mail(E_3D_mode_enable, 0, 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_ok);
#else
    aud_send_mail(E_interface_request_error, 0, E_3D_mode_enable, (U32)aud_rc_no_hw_support, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_no_hw_support);
#endif
}

/*------------------------------------------------------------------------------
* Function...: AUD_3D_mode_disable
* Return.....: return code
* Description: Disables 3D stereo enhancement.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 AUD_3D_mode_disable(void)
{
#if defined (AUD_3D_AMP_PRESENT)
    aud_send_mail(E_3D_mode_disable, 0, 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_ok);
#else
    aud_send_mail(E_interface_request_error, 0, E_3D_mode_disable, (U32)aud_rc_no_hw_support, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_no_hw_support);
#endif
}

/*******************************************************************************
*
*                    Miscellaneous interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: AUD_info_hw_available
* Return.....: bitmask indicating available resources
* Description: Returns the resources available on the project.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
U32 AUD_info_hw_available(void)
{
    aud_send_mail(E_info_hw_available, 0, 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL); /* dummy - for traceability */
    return (aud_hw_parms.audio_hw);
} /* End of function AUD_info_hw_available */
#endif
/*------------------------------------------------------------------------------
* Function...: AUD_misc_dsp_info
* Return.....: -
* Description: Information from Layer 1.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void AUD_misc_dsp_info(
    aud_dsp_info_enum dsp_info) /* the mode of the DSP */
{
#ifdef AUDIO_LLT
    aud_common_trace(llt_group_audio, llt_type_audio_function_interface, 4, 0,          /* LLT params */
                     'P', 0x05,                                                         /* LIS params */
                     aud_log_lis_display, aud_log_lis_display_data, aud_log_llt_trace,  /* LIS params */
                     (U32) aud_get_SM_state(aud_resource_record_vm),
                     (U32) aud_get_SM_state(aud_resource_playback_vm),
                     (U32) aud_get_SM_state(aud_resource_record_pcm),
                     (U32) E_audio_llt_type_id_4 );
#endif
}

#ifdef _USE_CUT_TEAKLIT_CODE
/*------------------------------------------------------------------------------
* Function...: Aud_set_vb_on_status_function
* Return.....: aud_rc_ok
* Description: Sets the callback function which is called when the Audio
*              Front-End is turned on/off.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
S8 Aud_set_vb_on_status_function(
    ptr_SetVbOnStatusFunction func) /* the callback function */
{
#ifdef AUDIO_LLT
    aud_common_trace(llt_group_audio, llt_type_audio_function_interface, 3, 0,          /* LLT params */
                    'd', 0x06,                                                          /* LIS params */
                    aud_log_lis_raw_data, aud_log_lis_display_data, aud_log_llt_trace,  /* LIS params */
                    (U32)E_set_vb_on_function,
                    (U32)func,
                    (U32)E_audio_llt_type_id_8 );
#endif
    aud_intf_claim_mutex(); /* wait for audio interface to be available for the calling process */
    aud_vb_on_status.vb_on_status_function = func;
    aud_send_mail(E_set_vb_on_function, 0, 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
    aud_intf_release_mutex(); /* release audio interface for other processes */
    return (aud_rc_ok);
}
#endif
/*******************************************************************************
*
*                             Internal functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: aud_intf_check_handle
* Return.....: aud_rc_ok/aud_rc_sharing_violation
* Description: Determines if a handle is associated to a specified resource.
* Modified...: 21.08.2007 by KBJ (added check for handle=0 - not allocated)
------------------------------------------------------------------------------*/
static S8 aud_intf_check_handle(
    U8 handle,                  /* handle */
    aud_resource_enum resource) /* the resource which the handle is for */
{
    /* 0 is used to indicate that the resource is not allocated, so report error in this case */
    if(0 == handle)
    {
        return (aud_rc_sharing_violation);
    }
    /* if the handle provided is the generic test handle, return ok */
    if(AUD_TEST_HANDLE == handle)
    {
        return (aud_rc_ok);
    }
    /* if the handle provided is the resource handle or the resource test handle, return ok */
    if( (aud_resource_table[resource].status == handle) || (aud_resource_table[resource].test_handle == handle) )
    {
        return (aud_rc_ok);
    }
    /* otherwise, return error */
    return (aud_rc_sharing_violation);
}

/*------------------------------------------------------------------------------
* Function...: aud_intf_check_for_conflict
* Return.....: aud_rc_ok/aud_rc_resource_conflict
* Description: Checks whether a new resource is allowed to be allocated.
* Modified...: 07.08.2007 by KBJ (status=0 means not allocated)
------------------------------------------------------------------------------*/
static S8 aud_intf_check_for_conflict(
    aud_resource_enum resource) /* the new resource to be added */
{
    /* TTY cannot run in parallel with any other resource, except speech */
    aud_resource_enum current_resource = aud_resource_tone_generator; /* aud_resource_speech is allowed to run in parallel with TTY */
    if(aud_resource_tty == resource)
    {
        while(current_resource<aud_resource_end)
        {
            if(aud_resource_table[current_resource].status != 0)
            {
                return (aud_rc_resource_conflict);
            }
            current_resource++;
        }
    }
    /* besides speech, no resource can be enabled at the same time as TTY */
    if( (aud_resource_table[aud_resource_tty].status != 0) && (resource != aud_resource_speech) )
    {
        return (aud_rc_resource_conflict);
    }

    /* aud_resource_mapi and aud_resource_ringer are different ways of accessing the same driver - only use one */
#if defined (AUD_YAMAHA_MAPI)
    if(aud_resource_mapi == resource)
    {
        if( (aud_resource_table[aud_resource_ringer].status != 0)
            || (aud_resource_table[aud_resource_vibrator].status != 0)
            || (aud_resource_table[aud_resource_amplifier].status != 0) )
        {
            return (aud_rc_resource_conflict);

        }
    }
    if( (aud_resource_ringer == resource)
        || (aud_resource_vibrator == resource)
        || (aud_resource_amplifier == resource) )
    {
        if(aud_resource_table[aud_resource_mapi].status != 0)
        {
            return (aud_rc_resource_conflict);
        }
    }
#endif

    /* VM/PCM Recording and Midi player may not run simultaneously (however the firmware spec state they may!?) - affects quality */
    if(aud_resource_midi_player == resource)
    {
        if( (aud_resource_table[aud_resource_record_vm].status != 0)
            || (aud_resource_table[aud_resource_record_pcm].status != 0) )
        {
            return (aud_rc_resource_conflict);
        }
    }
    if( (aud_resource_record_vm == resource)
        || (aud_resource_record_pcm == resource) )
    {
        if(aud_resource_table[aud_resource_midi_player].status != 0)
        {
            return (aud_rc_resource_conflict);
        }
    }

    /* PCM Recording and Voice Memo recording/playback may not run simultaneously*/
    if(aud_resource_record_pcm == resource)
    {
        if( (aud_resource_table[aud_resource_record_vm].status != 0)
            || (aud_resource_table[aud_resource_playback_vm].status != 0) )
        {
            return (aud_rc_resource_conflict);
        }
    }
    if( (aud_resource_record_vm == resource)
        || (aud_resource_playback_vm == resource) )
    {
        if(aud_resource_table[aud_resource_record_pcm].status != 0)
        {
            return (aud_rc_resource_conflict);
        }
    }

    /* SBC encoder may not run in parallel with PCM recording or VM recording/playback */
    if(aud_resource_sbc_encoder == resource)
    {
        if( (aud_resource_table[aud_resource_record_vm].status != 0)
            || (aud_resource_table[aud_resource_playback_vm].status != 0)
            || (aud_resource_table[aud_resource_record_pcm].status != 0) )
        {
            return (aud_rc_resource_conflict);
        }
    }
    if( (aud_resource_record_vm == resource)
        || (aud_resource_playback_vm == resource)
        || (aud_resource_record_pcm == resource) )
    {
        if(aud_resource_table[aud_resource_sbc_encoder].status != 0)
        {
            return (aud_rc_resource_conflict);
        }
    }

    /* if no conflict was found, return OK */
    return (aud_rc_ok);
} /* End of function aud_intf_check_for_conflict */

/*------------------------------------------------------------------------------
* Function...: aud_intf_allow_sample_rate
* Return.....: TRUE/FALSE
* Description: Uses the sample rate requirements of the active paths to check
*              if a new path can be added.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static U8 aud_intf_allow_sample_rate(
    U8 required,    /* bitmask representing the required sample rate of the new path to be added */
    U8 allowed)     /* bitmask representing the allowed sample rates of the new path to be added */
{
    U8 cnt = 0;          /* Counter for loops */
    U8 requirements = 0; /* Sample rate requirements of the active paths. Default: no requirements */
    U8 allow = TRUE;     /* Return value. Is the new path allowed to be added? */

    /* Find out if any of the uplink paths has special requirements (the 'sample rate' paths are uplink paths) */
    while(aud_intf_uplinkpath_status[cnt]<aud_uplink_source_end && 0==requirements)
    {
        if(0 != aud_setting.p_aud_audio_uplink_parms[aud_intf_uplinkpath_status[cnt]]->required_sample_rate)
        {
            requirements = aud_setting.p_aud_audio_uplink_parms[aud_intf_uplinkpath_status[cnt]]->required_sample_rate;
        }
        cnt++;
    }
    /* The existing paths have requirements, check for conflicts with added path */
    if(0 != requirements)
    {
        /* Check if the added path's requirement conflicts with the existing requirements */
        if(0!=required && required!=requirements)
        {
            allow = FALSE;
        }
        /* If the required sample rate is not allowed by the added path, don't allow addition */
        else if(0==(requirements&allowed))
        {
            allow = FALSE;
        }
    }
    else /* No special requirements are found (the audio scheduler currently runs at 8 kHz) */
    {
        /* Check uplink paths for conflicts */
        cnt = 0;
        while(aud_intf_uplinkpath_status[cnt]<aud_uplink_source_end && TRUE==allow)
        {
            if(0!=required && 0==(required&aud_setting.p_aud_audio_uplink_parms[aud_intf_uplinkpath_status[cnt]]->allowed_sample_rates))
            {
                allow = FALSE; /* The requirements of an added path conflicts with an existing path */
            }
            if(0==(allowed&aud_setting.p_aud_audio_uplink_parms[aud_intf_uplinkpath_status[cnt]]->allowed_sample_rates))
            {
                allow = FALSE; /* The added path is not allowed to run at the same sample rate as this path */
            }
            cnt++;
        }
        /* Check downlink paths for conflicts */
        cnt = 0;
        while(aud_intf_downlinkpath_status[cnt]<aud_downlink_source_end && TRUE==allow)
        {
            if(0!=required && 0==(required&aud_setting.p_aud_audio_downlink_parms[aud_intf_downlinkpath_status[cnt]]->allowed_sample_rates))
            {
                allow = FALSE; /* The requirements of an added path conflicts with an existing path */
            }
            if(0==(allowed&aud_setting.p_aud_audio_downlink_parms[aud_intf_downlinkpath_status[cnt]]->allowed_sample_rates))
            {
                allow = FALSE; /* The added path is not allowed to run at the same sample rate as this path */
            }
            cnt++;
        }
    }
    return (allow);
}

/*------------------------------------------------------------------------------
* Function...: aud_intf_create_mutex
* Return.....: -
* Description: A reentrant mutex is a lock that can be claimed several times
*              by the same process (must be released the same number of times).
*              The OS wrappers cannot be used to implement the mutex functions
*              since os_sdlpid_current() can only handle 8 threads.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static void aud_intf_create_mutex(void)
{
    /*[Begin] [LvWenhua-2008/4/23]*/
    #if 0
    os_sem_create(aud_intf_mutex.sem, "AUD_intf",0/*Dummy*/, 1);
    #endif
    aud_intf_mutex.sem = zOss_CreateSemaphore("AUD_intf", 1);
    /*[End] [LvWenhua-2008/4/23]*/
    
    aud_intf_mutex.p_caller = NULL;
    aud_intf_mutex.depth = 0;
}

/*------------------------------------------------------------------------------
* Function...: aud_intf_claim_mutex
* Return.....: -
* Description: Claim the interface mutex. If the mutex is already owned by the
*              calling thread, a counter will be increased instead of
*              claiming the semaphore.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static void aud_intf_claim_mutex(void)
{
    if(FALSE == aud_intf_mutex_first)
    {
        aud_intf_create_mutex();
        aud_intf_mutex_first=TRUE;
    }
    if(aud_intf_mutex.p_caller != NU_Current_Task_Pointer())
    {
        /*[Begin] [LvWenhua-2008/4/23]*/
        #if 0
        os_sem_wait(aud_intf_mutex.sem); /* wait for other processes to release the mutex */
        #endif
        zOss_GetSemaphore(aud_intf_mutex.sem, ZOSS_WAIT_FOREVER);
        /*[End] [LvWenhua-2008/4/23]*/
    }
    aud_intf_mutex.depth++;
    aud_intf_mutex.p_caller = NU_Current_Task_Pointer();
}

/*------------------------------------------------------------------------------
* Function...: aud_intf_release_mutex
* Return.....: -
* Description: Release the interface mutex. If the thread has released the
*              mutex as many times as it claimed it, the semaphore is released.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static void aud_intf_release_mutex(void)
{
    if(FALSE == aud_intf_mutex_first)
    {
        aud_intf_create_mutex();
        aud_intf_mutex_first = TRUE;
        AUD_ERROR_REPORTING(aud_error_grade2, 345/*TRAP_AUD_ERROR*/, "Error: Mutex released before claimed");
    }
    if(aud_intf_mutex.p_caller==NU_Current_Task_Pointer())  /* only the owner of the mutex can release it */
    {
        aud_intf_mutex.depth--;
        if(0 == aud_intf_mutex.depth)                       /* only release semaphore if the mutex has been released as many times as it has been acquired */
        {
            aud_intf_mutex.p_caller = NULL;
            
            /*[Begin] [LvWenhua-2008/4/23]*/
            #if 0
            os_sem_signal(aud_intf_mutex.sem);              /* signal to other processes that the mutex is available */
            #endif
            zOss_PutSemaphore(aud_intf_mutex.sem);
            /*[End] [LvWenhua-2008/4/23]*/
        }
    }
}

/*******************************************************************************
*
*                         Non-interface functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: aud_send_mail
* Return.....: -
* Description: Send a mail to the audio dispatcher.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
extern ZOSS_TASK_ID audTaskID;
void aud_send_mail(
    U8  event,      /* ID of the event that should be handled by the audio driver process */
    U16 caller_id,  /* not used */
    U32 parm1,      /* generic parameter for handling the event */
    U32 parm2,      /* generic parameter for handling the event */
    U32 parm3,      /* generic parameter for handling the event */
    U32 parm4,      /* generic parameter for handling the event */
    U32 parm5,      /* generic parameter for handling the event */
    U32 parm6,      /* generic parameter for handling the event */
    U32 parm7,      /* generic parameter for handling the event */
    U32 parm8,      /* generic parameter for handling the event */
    U8 *ptr1,       /* generic parameter for handling the event */
    U8 *ptr2)       /* generic parameter for handling the event */
{
    T_AUD_DRV_REQ_PARAM aud_drv_req;

    aud_drv_req.parm1   = parm1;
    aud_drv_req.parm2   = parm2;
    aud_drv_req.parm3   = parm3;
    aud_drv_req.parm4   = parm4;
    aud_drv_req.parm5   = parm5;
    aud_drv_req.parm6   = parm6;
    aud_drv_req.parm7   = parm7;
    aud_drv_req.parm8   = parm8;
    aud_drv_req.ptr1    = ptr1;
    aud_drv_req.ptr2    = ptr2;
    
    zOss_TaskSend(event, (void*)(&aud_drv_req), sizeof(aud_drv_req), audTaskID);
} /* End of function aud_send_mail */
#if 0
void aud_send_mail(
    U8  event,      /* ID of the event that should be handled by the audio driver process */
    U16 caller_id,  /* not used */
    U32 parm1,      /* generic parameter for handling the event */
    U32 parm2,      /* generic parameter for handling the event */
    U32 parm3,      /* generic parameter for handling the event */
    U32 parm4,      /* generic parameter for handling the event */
    U32 parm5,      /* generic parameter for handling the event */
    U32 parm6,      /* generic parameter for handling the event */
    U32 parm7,      /* generic parameter for handling the event */
    U32 parm8,      /* generic parameter for handling the event */
    U8 *ptr1,       /* generic parameter for handling the event */
    U8 *ptr2)       /* generic parameter for handling the event */
{
    T_AUD_DRV_REQ aud_drv_req;

#if defined (SDL_SELF)
#undef SDL_SELF
#endif
//    aud_pid = os_sdlpid_current();//need modify miaolin 20130816
    aud_pid = os_sdlpid_current();
//#define SDL_SELF aud_pid//need modify miaolin 20130816
#define SDL_SELF aud_pid
    aud_drv_req.drv_id  = caller_id;
    aud_drv_req.func_id = event;
    aud_drv_req.parm1   = parm1;
    aud_drv_req.parm2   = parm2;
    aud_drv_req.parm3   = parm3;
    aud_drv_req.parm4   = parm4;
    aud_drv_req.parm5   = parm5;
    aud_drv_req.parm6   = parm6;
    aud_drv_req.parm7   = parm7;
    aud_drv_req.parm8   = parm8;
    aud_drv_req.ptr1    = (T_AUD_BUF_REQ_PTR1)ptr1;
    aud_drv_req.ptr2    = (T_AUD_BUF_REQ_PTR2)ptr2;
    
    os_signal_alloc (AUD_DRIVER_REQ, yPDef_AUD_DRIVER_REQ);
    os_signal_paramassign (AUD_DRIVER_REQ, 1, aud_drv_req);
    os_signal_route (AUD_DRIVER_REQ, "AUD_DRIVER_REQ");
} /* End of function aud_send_mail */
#endif

/*------------------------------------------------------------------------------
* Function...: aud_init_resource_table
* Return.....: -
* Description: Initialises the table holding resource allocation status.
* Modified...: 07.08.2007 by KBJ (status=0 means not allocated)
------------------------------------------------------------------------------*/
void aud_init_resource_table(void)
{
    U8 i;
    for(i=0; i < aud_resource_end; i ++)
    {
        aud_resource_table[i].status    = 0;
        aud_resource_table[i].caller_id = 0;
    }
}

/*------------------------------------------------------------------------------
* Function...: aud_init_set_paths
* Return.....: -
* Description: Initialises the array holding active paths. Default paths are
*              added as well.
* Modified...: 07.08.2007 by KBJ (removed lint error)
------------------------------------------------------------------------------*/
void aud_init_set_paths(void)
{
    U16 cnt; /* counter for loops */
    for(cnt=0; cnt<(sizeof(aud_intf_uplinkpath_status)/sizeof(aud_intf_uplinkpath_status[0])); cnt++)
    {
        aud_intf_uplinkpath_status[cnt] = aud_uplink_source_end;
    }
    for(cnt=0; cnt<(sizeof(aud_intf_downlinkpath_status)/sizeof(aud_intf_downlinkpath_status[0])); cnt++)
    {
        aud_intf_downlinkpath_status[cnt] = aud_downlink_source_end;
    }
    /* set default paths */
/***************   remove by  zhouzhongyao	*************/
//    aud_intf_uplinkpath_status[0] = aud_handset_mic;
//    aud_intf_downlinkpath_status[0] = aud_normal_earpiece;
/***************   add                       end	*************/
    
}

/*******************************************************************************
*
*                     Deprecated interface functions
*
*******************************************************************************/
/* This is old stuff that has to be removed when Apoxi has cleaned out their inclusion of old files - depending on this */
#ifdef _USE_CUT_TEAKLIT_CODE
S8 AUD_media_to_uplink(U8 enable_disable) { /* use path aud_media_to_uplink instead */
    aud_send_mail(E_interface_request_error, 0, E_media_to_uplink, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_request_error);
}
S8 AUD_media_to_downlink(U8 enable_disable) { /* use path aud_media_to_downlink_mute instead */
    aud_send_mail(E_interface_request_error, 0, E_media_to_downlink, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_request_error);
}
U32 AUD_get_resource_capability(aud_resource_enum resource, aud_resource_capability_enum capability) {
    aud_send_mail(E_interface_request_error, 0, E_get_resource_capability, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return ((U32)aud_rc_request_error);
}
S8  AUD_key_tone(aud_tone_id_enum key_tone, S16 mix_factor) {
    aud_send_mail(E_interface_request_error, 0, E_key_tone, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_request_error);
}
S8 AUD_amp_enable(U8 handle) {
    aud_send_mail(E_interface_request_error, 0, E_amp_enable, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_request_error);
}
S8 AUD_amp_disable(U8 handle) {
    aud_send_mail(E_interface_request_error, 0, E_amp_disable, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_request_error);
}
S8 AUD_vibrator_start(U8 handle, aud_vibrator_mode_enum mode,  aud_vibrator_on_enum on, aud_vibrator_off_enum off) {
    aud_send_mail(E_interface_request_error, 0, E_vib_start, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_request_error);
}
S8 AUD_vibrator_stop(U8 handle) {
    aud_send_mail(E_interface_request_error, 0, E_vib_stop, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_request_error);
}
void  AUD_misc_set_accessory(aud_accessory_enum accessory) {
    aud_send_mail(E_interface_request_error, 0, E_misc_mode, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
}
void  AUD_misc_set_vmic(U8 status) {}
S8 AUD_set_volume(U8 handle, aud_volume_enum volume) {
    aud_send_mail(E_interface_request_error, 0, E_set_volume, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_request_error);
}
S8 AUD_speech_set_volume(U8 handle, aud_speech_volume_enum volume) {
    aud_send_mail(E_interface_request_error, 0, E_speech_set_volume, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_request_error);
}
S8 AUD_ringer_set_volume(U8 handle, aud_ringer_volume_enum volume) {
    aud_send_mail(E_interface_request_error, 0, E_ringer_set_volume, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_request_error);
}
S8 AUD_radio_init(U8 handle) {
    aud_send_mail(E_interface_request_error, 0, E_radio_initialize, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_request_error);
}
S8 AUD_radio_config(U8 handle, void * AUD_radio_config_data) {
    aud_send_mail(E_interface_request_error, 0, E_radio_config, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_request_error);
}
S8 AUD_radio_read_current_status(U8 handle) {
    aud_send_mail(E_interface_request_error, 0, E_radio_read_current_status, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_request_error);
}
S8 AUD_radio_mute(U8 handle,U8 mute) {
    aud_send_mail(E_interface_request_error, 0, E_radio_mute, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_request_error);
}
S8 AUD_speech_mute(U8 handle, aud_mute_enum status, aud_direction_enum direction) {
    aud_send_mail(E_interface_request_error, 0, E_speech_mute, (U32)aud_rc_request_error, 0, 0, 0, 0, 0, 0, NULL, NULL);
    return (aud_rc_request_error);
}
#endif
#ifdef USE_VOICE_SUPPORT
void AUD_misc_layer1_info(aud_layer1_info_enum layer1_info) {}
#endif

