/**
 *   Copyright (C) Infineon Technologies Denmark A/S. All rights reserved.
 *
 * This document contains proprietary information belonging to Infineon
 * Technologies Denmark A/S. Passing on and copying of this document, use
 * and communication of its contents is not permitted without prior written
 * authorisation.
 *
 * Description:
 *   Path control
 *
 * Revision Information:
 *   File name: \dwddrv\AUD\src\aud_path_control.c
 *   Version: \main\103
 *   Date: 2007-10-23 16:48:08
 *   Responsible: johansek
 *   Comment:
 *     Corrected lint error.
 */

/*******************************************************************************
*
*                               Include files
*
*******************************************************************************/
//#include "scttypes.h"           /* must be included in all files */
//#include "ms.h"                 /* must be included in all files */
#include "bastypes.h"           /* must be included in all files */

/*changed by wangjun for NV change.[20090214]*/
#include "aud_nv.h"  //#include "eep.h"   /* contains scal_in and scal_out calibration data + the path data not included in aud_data */
//#include "dsp.h"                /* for DSP command types */
#include "aud_drv.h"            /* audio types and functions */
#include "aud_data.h"           /* data describing the paths */
#include "aud_com.h"            /* types and functions used only within the audio driver */
#include "aud_ear.h"            /* programming of the hands-free block */
#include "aud_i2s.h"            /* for enabling/disabling the I2S1 paths (BlueTooth) */
#include "aud_ratesw.h"         /* for calculating the audio scheduler sample rate */
#include "aud_gain_lib.h"       /* structures and functions for writing DSP commands */
#include "aud_path_control.h"   /* header file for path control */
#include "aud_volume.h"         /* for retrieving master volume */
#include "drvs_ramlog.h"
#include "drvs_teak.h"

/*[Begin] [lvwenhua-2010/10/28]*/
/*******************************************************************************
*******************************************************************************/
typedef struct
{
  word  Switch;
  word  afe_i2s_csel;  //afe_ext_input_ctrl;
  word  reserved_afe_tx_ctrl_1;  //afe_tx_ctrl1;
  word  afe_i2s_num0;  //afe_tx_ctrl4;
  word  afe_i2s_den0;  //afe_tx_ctrl5;
  word  afe_i2s_num1;  //afe_tx_ctrl6;
  word  afe_i2s_den1;  //afe_tx_ctrl7;
  word  reserved_afe_rx_ringer;  //afe_rx_ringer;
  word  afe_i2s_rxconf;  //afe_rx_ana1;
  word  afe_i2s_txconf;  //afe_rx_ana2;
  word  reserved_afe_rx_ana3;   //afe_rx_ana3;
  word  reserved_afe_rx_ctrl2;  //afe_rx_ctrl2;
  word  reserved_afe_rx_epref;  //afe_rx_epref;
  word  reserved_afe_rx_gain_digital;  //afe_rx_gain_digital;
  word  afe_tx_ctrl2;
  word  input_mic_mode;

  /*[Begin] [shj afe change-2010/12/28]*/
  #if 0  /*shj ýṹΪ˷ͱдVB_ONAFEö ӦðVB_ONеI2S شȥ 2010/12/28 */
  word  ratesw;
  word  out_mode;
  word  csel;
  word  num0;
  word  den0;
  word  num1;
  word  den1;
  word  rxconf;
  word  txconf;
  #if !defined (STEON2_PROJECT)
  word  i2s1_hw_mode;
  word  i2s1_chn_mode;
  #endif
  #endif
  /*[End] [shj afe change-2010/12/28]*/
} T_DSP_CMD_AFEI2S_VB_HW_PAR;
/*[End] [lvwenhua-2010/10/28]*/
/*******************************************************************************
*
*                              Local variables
*
*******************************************************************************/
aud_vb_on_status_type                aud_vb_on_status;                 /* contains VB_ON status */
aud_path_settings_result_type        aud_path_settings_result;         /* used to store the path settings - from don't care concept */
static aud_uplink_gain_cells_type    aud_path_settings_uplink_gains;   /* so that uplink gains can be referenced individually */
static aud_downlink_gain_cells_type  aud_path_settings_downlink_gains; /* so that downlink gains can be referenced individually */
static aud_path_uplink_buffer_type   aud_path_active_uplink_paths;     /* uplink path that are set */
/*[Begin]change to global variable for read scal_rec from NV in aud_volume.c.[lvwenhua-2010/9/26]*/
#if 0
static aud_path_downlink_buffer_type aud_path_active_downlink_paths;   /* downlink path that are set */
#endif
aud_path_downlink_buffer_type aud_path_active_downlink_paths;   /* downlink path that are set */
/*[End] [lvwenhua-2010/9/26]*/
/* these structures ought to be static but are accessed from aud_tst.c as well - don't use elsewhere */
T_DSP_CMD_VB_HW_PAR                  aud_dsp_vb_hw_parms_global;       /* parameters for the DSP command VB_ON */
/*[Begin] [lvwenhua-2010/10/28]*/
T_DSP_CMD_AFEI2S_VB_HW_PAR           aud_dsp_afei2s_vb_hw_parms_global = {0}; //aud_dsp_vb_hw_parms_global
aud_I2S_samplerate_enum              aud_dsp_afei2s_sample = aud_I2S_samplerate_08000;
aud_I2S_transmission_mode_emum       aud_dsp_afei2s_trans_mode = aud_I2S_transmission_mode_PCM;
/*[End] [lvwenhua-2010/10/28]*/

extern aud_eep_static_type aud_EEP_static;

T_DSP_CMD_VB_SET_BIQUAD_PAR          aud_dsp_uplink_filter_parms;
T_DSP_CMD_VB_SET_BIQUAD_PAR          aud_dsp_downlink_filter_parms;
T_DSP_CMD_VB_SET_BIQUAD_PAR          aud_dsp_downlink_cbuf_filter_parms;
#ifdef _USE_CUT_TEAKLIT_CODE
T_DSP_CMD_VB_SET_FIR				 aud_dsp_fir_filter_struct;
T_DSP_CMD_VB_AUDIO_FILTER_SWITCH     aud_dsp_filter_switch;
T_DSP_CMD_VB_ASP_EP                  aud_dsp_asp_ul_parms;
T_DSP_CMD_VB_HF_SET_GAIN             aud_dsp_hf_set_gain;
T_DSP_CMD_VB_RX_NR_CONFIG            aud_dsp_dl_nr_parms;               /* downlink noise reduction */
#endif
/*******************************************************************************
*
*                      Prototypes for internal functions
*
*******************************************************************************/
static void aud_path_populate_path_data(void);
static void aud_path_afe_dsp_programming(U32 configuration);
static void aud_path_configure_audio_path(U32 mode);
/*[Begin]no use. [lvwenhua-2010/9/16]*/
#if 0
void aud_path_set_dsp_fir_filters(const word ul_dl_select, const aud_fir_filter_type* filter); /* also used by aud_path_tst.c */
#endif
/*[End] [lvwenhua-2010/9/16]*/
static void aud_path_set_dsp_filters_uplink(const aud_ratesw_enum sample_rate);
static void aud_path_set_dsp_filters_downlink(const aud_ratesw_enum sample_rate);
/*[Begin]no use. [lvwenhua-2010/9/16]*/
#if 0
static void aud_path_program_asp(void);
static void aud_path_program_dl_nr(void);
#endif
/*[End] [lvwenhua-2010/9/16]*/
static void aud_path_enable_filters_and_gains(void);
static void aud_path_enable_gains_filters_and_path(void);
static void aud_path_set_value(word value, word position, word* dataPtr, word HowManyBits);

/*******************************************************************************
*
*                        Externally visible functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: aud_path_dispatcher
* Return.....: TRUE/FALSE, depending on whether the event has been handled
* Description: Dispatcher for path events (adding and removing)
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
BOOL aud_path_dispatcher(
    aud_event_enum event,   /* the event to be handled */
    void* p_data)           /* data to use for event handling */
{
    BOOL func_ret = TRUE; /* used to tell whether the event has been handled by this state machine */
    disp_to_SM_data_type *p_SM_data = (disp_to_SM_data_type*)p_data; /* contains the structure passed from the dispatcher */
    U32 parm1 = p_SM_data->parm1;
    U32 parm2 = p_SM_data->parm2;
    U32 parm3 = p_SM_data->parm3;
    U32 parm4 = p_SM_data->parm4;
    U32 parm5 = p_SM_data->parm5;
    U32 parm6 = p_SM_data->parm6;
    U32 parm7 = p_SM_data->parm7;
    U32 parm8 = p_SM_data->parm8;

    /* Send string : at+xl1set="l1 17l" To enable this LLT trace */
#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_path, 11, 0,                          /* LLT params */
                     's', 0x01,                                                             /* LIS params */
                     aud_log_lis_raw_data, aud_log_lis_display_data, aud_log_lis_llt_trace, /* LIS params */
                     (U32) aud_get_SM_state(aud_resource_speech), 
                     (U32) event,
                     (U32) p_SM_data->parm1,
                     (U32) p_SM_data->parm2,
                     (U32) p_SM_data->parm3,
                     (U32) p_SM_data->parm4,
                     (U32) p_SM_data->parm5,
                     (U32) p_SM_data->parm6,
                     (U32) p_SM_data->parm7,
                     (U32) p_SM_data->parm8,
                     (U32) E_audio_llt_type_id_0 );
#endif
    
    switch(event)
    {
        case E_update_uplinkpath:
            aud_uplinkpath_update_buffers(parm1,parm2,parm3,parm4,parm5,parm6,parm7,parm8);
            aud_calculate_uplink_resulting_path(); /* find valid aud_data values to use */
            aud_path_enable_gains_filters_and_path();

#ifdef AUDIO_LLT
            aud_common_trace(llt_group_audio, llt_type_audio_path, 5, 0,                       /* LLT params */
                             'b', 0x27,                                                        /* LIS params */
                             aud_log_lis_display, aud_log_lis_display_data, aud_log_llt_trace, /* LIS params */
                             (U32) E_update_uplinkpath,
                             (U32) parm1, /* path structure 1 - 4 */
                             (U32) parm2, /* path structure 5 - 8 */
                             (U32) aud_one_SM_active(),
                             (U32) E_audio_llt_type_id_1 );
#endif
            break;

        case E_update_downlinkpath:
            aud_downlinkpath_update_buffers(parm1,parm2,parm3,parm4,parm5,parm6,parm7,parm8);
            aud_calculate_downlink_resulting_path(); /* find valid aud_data values to use */
            aud_path_enable_gains_filters_and_path();

#ifdef AUDIO_LLT
            aud_common_trace(llt_group_audio, llt_type_audio_path, 5, 0,                       /* LLT params */
                             'b', 0x27,                                                        /* LIS params */
                             aud_log_lis_display, aud_log_lis_display_data, aud_log_llt_trace, /* LIS params */
                             (U32) E_update_downlinkpath,
                             (U32) parm1, /* path structure 1 - 4 */
                             (U32) parm2, /* path structure 5 - 8 */
                             (U32) aud_one_SM_active(),
                             (U32) E_audio_llt_type_id_2 );
#endif
            break;

        case E_set_vb_on_function:
            if(aud_vb_on_status.vb_on_status_function != NULL)
                (*aud_vb_on_status.vb_on_status_function)(aud_vb_on_status.vb_on);
            break;

        case E_get_downlnk_parallel_path_configuration:
        case E_get_uplink_parallel_path_configuration:
        case E_Pathupdate_Enable:
        case E_Pathupdate_Disable:
            break; /* dummy - do nothing */

        default:
            func_ret = FALSE; /* the event has not been handled */
            break;  
    }
    return (func_ret);
}

/*------------------------------------------------------------------------------
* Function...: aud_path_init
* Return.....: -
* Description: Initialises the path control system.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void aud_path_init(void)
{
    U16 cnt; /* counter for loops */
    U32 no_path_set = 0; /* used to initialise paths to default */

    /* read path data from EEPROM (?) */
    aud_path_populate_path_data();

    /* make sure space has been allocated for the gain cells of aud_path_settings_result so that they can be written individually */
    aud_path_settings_result.uplink_parms.p_uplink_gain_cells = &aud_path_settings_uplink_gains;
    aud_path_settings_result.downlink_parms.p_downlink_gain_cells = &aud_path_settings_downlink_gains;

    /* initialise uplink paths */
    aud_path_active_uplink_paths.total = 0; /* number of set paths */
    for(cnt=0; cnt<(aud_uplink_source_end+1); cnt++)
    {
        aud_path_active_uplink_paths.path[cnt] = aud_uplink_source_end;
    }

    /* initialise downlink paths */
    aud_path_active_downlink_paths.total= 0; /* number of set paths */
    for(cnt=0; cnt<(aud_downlink_source_end+1); cnt++)
    {
        aud_path_active_downlink_paths.path[cnt] = aud_downlink_source_end;
    }

    /*[Begin]It should be no default path setting, or else when playing tone, afe will been openned.[lvwenhua-2010/7/30]*/
    /* set default paths */
    no_path_set = aud_uplink_source_end | aud_uplink_source_end << 8 | aud_uplink_source_end << 16 | aud_uplink_source_end << 24; 
    aud_uplinkpath_update_buffers(no_path_set /*(U32) aud_handset_mic| aud_uplink_source_end << 8 | aud_uplink_source_end << 16 | aud_uplink_source_end << 24*/, no_path_set, no_path_set, no_path_set, no_path_set, no_path_set, no_path_set, no_path_set);
    no_path_set = aud_downlink_source_end | aud_downlink_source_end << 8 | aud_downlink_source_end << 16 | aud_downlink_source_end << 24; 
    aud_downlinkpath_update_buffers(no_path_set /*(U32) aud_normal_earpiece| aud_downlink_source_end << 8 | aud_downlink_source_end << 16 | aud_downlink_source_end << 24*/, no_path_set, no_path_set, no_path_set, no_path_set, no_path_set, no_path_set, no_path_set);
    /*[End] [lvwenhua-2010/7/30]*/

    /* initialize dynamic aud_data calculation */
    aud_calculate_uplink_resulting_path();
    aud_calculate_downlink_resulting_path();
    aud_path_enable_filters_and_gains();
    aud_path_configure_audio_path(0); /* we assume no resource is active at power-up */
}

/*------------------------------------------------------------------------------
* Function...: aud_enable_path_and_resource
* Return.....: -
* Description: Enables a resource and updates the path settings.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void aud_enable_path_and_resource(
    aud_resource_enum res,      /* the resource to enable */
    aud_SM_state_enum state,    /* the new state of the resource */
    BOOL use_power)             /* if TRUE, this function will make sure power saving is disallowed */
{
#ifdef AUDIO_LLT
    aud_common_trace(llt_group_audio, llt_type_audio_path, 4, 0,                           /* LLT params */
                     'b', 0x17,                                                            /* LIS params */
                     aud_log_lis_display, aud_log_lis_display_data, aud_log_lis_llt_trace, /* LIS params */
                     (U32) res,
                     (U32) state,
                     (U32) use_power,
                     (U32) E_audio_llt_type_id_1 );
#endif
    if(use_power)
    {
        aud_disable_standby(res);
    }
    if(res != aud_resource_end)
    {
        aud_set_SM_state(res, state);
    }
    aud_path_enable_gains_filters_and_path();
}
/*------------------------------------------------------------------------------
* Function...: aud_disable_path_and_resource
* Return.....: -
* Description: Disables a resource and updates the path settings.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void aud_disable_path_and_resource(
    aud_resource_enum res,      /* the resource to disable */
    aud_SM_state_enum state,    /* the new state of the resource */
    BOOL use_power)             /* if TRUE, this function will make sure power saving is allowed by the resource */
{
#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_path, 4, 0,                           /* LLT params */
                      'b', 0x19,                                                            /* LIS params */
                      aud_log_lis_display, aud_log_lis_display_data, aud_log_lis_llt_trace, /* LIS params */
                      (U32) res,
                      (U32) state,
                      (U32) use_power,
                      (U32) E_audio_llt_type_id_2 );
#endif

    if(res != aud_resource_end)
    {
        aud_set_SM_state(res, state);
    }
    if(FALSE == aud_one_SM_active()) /* nothing is active, so let's power everything off */
    {
        aud_path_enable_filters_and_gains();
        aud_path_configure_audio_path(0); /* Disable all paths */
        DSP_ASYNC_VB_RESET;
    }
    else
    {
        aud_drv_activate_hf();
    }
    if(use_power)
    {
        aud_enable_standby(res);
    }
}

/*------------------------------------------------------------------------------
* Function...: aud_program_afe_rx_gain
* Return.....: -
* Description: Programs the analog gain cells for the master volume.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
/*[Begin]no use [lvwenhua-2010/10/27]*/
#if 0
void aud_program_afe_rx_gain(
    unsigned char gain) /* the gain value (9=0dB, 8=-6dB, 7=-12dB, 6=-18dB) */
{
    aud_path_set_value(gain, 4, &(aud_dsp_vb_hw_parms_global.afe_rx_ana1), 4); /* afe_rx_ana1->ear_rxpath = aud_get_master_volume_analog() */
    aud_path_set_value(gain, 8, &(aud_dsp_vb_hw_parms_global.afe_rx_ana1), 4); /* afe_rx_ana1->ear_mixpath = aud_get_master_volume_analog() */
    aud_path_set_value(gain, 5, &(aud_dsp_vb_hw_parms_global.afe_rx_ana2), 4); /* afe_rx_ana2->head_rxpathL = aud_get_master_volume_analog() */
    aud_path_set_value(gain, 9, &(aud_dsp_vb_hw_parms_global.afe_rx_ana2), 4); /* afe_rx_ana2->head_mixpathL = aud_get_master_volume_analog() */
    aud_path_set_value(gain, 3, &(aud_dsp_vb_hw_parms_global.afe_rx_ana3), 4); /* afe_rx_ana3->head_rxpathR = aud_get_master_volume_analog() */
    aud_path_set_value(gain, 7, &(aud_dsp_vb_hw_parms_global.afe_rx_ana3), 4); /* afe_rx_ana3->head_mixpathR = aud_get_master_volume_analog() */

    /* Only copy the registers to the AFE, do not change the AFE state */
    aud_dsp_vb_hw_parms_global.Switch = COPYPARAMETERS_TO_AFE;
    DSP_ASYNC_VB_HW(&aud_dsp_vb_hw_parms_global);
}
#endif
/*[End] [lvwenhua-2010/10/27]*/

/*------------------------------------------------------------------------------
* Function...: aud_setup_i2s1_vb_on_parameters
* Return.....: -
* Description: Sets up the I2S1 parameters for the VB_ON call.
* Modified...: 02.10.2007 by KBJ (corrected Steon2 compile errors)
------------------------------------------------------------------------------*/
void aud_setup_i2s1_vb_on_parameters(
    U16 csel,   /* csel parameter */
    U16 den0,   /* den0 parameter */
    U16 den1,   /* den1 parameter */
    U16 num0,   /* num0 parameter */
    U16 num1,   /* num1 parameter */
    U16 rxconf, /* rxconf parameter */
    U16 txconf
#if !defined (STEON2_PROJECT)
    , /* txconf parameter */
    U16 hwmode, /* i2s1 hw mode parameter */
    U16 chnmode
#endif /* STEON2_PROJECT */
    )/* i2s1 channel mode parameter */
{
        aud_dsp_vb_hw_parms_global.csel   = csel;
        aud_dsp_vb_hw_parms_global.den0   = den0;
        aud_dsp_vb_hw_parms_global.den1   = den1;
        aud_dsp_vb_hw_parms_global.num0   = num0;
        aud_dsp_vb_hw_parms_global.num1   = num1;
        aud_dsp_vb_hw_parms_global.rxconf = rxconf;
        aud_dsp_vb_hw_parms_global.txconf = txconf;
#if !defined (STEON2_PROJECT)
        aud_dsp_vb_hw_parms_global.i2s1_hw_mode  = hwmode;
        aud_dsp_vb_hw_parms_global.i2s1_chn_mode = chnmode;
#endif /* STEON2_PROJECT */
}

/*------------------------------------------------------------------------------
* Function...: aud_setup_afei2s_vb_on_parameters
* Return.....: -
* Description: Sets up the afeI2S parameters for the VB_ON call.
* Modified...: 02.10.2007 by KBJ (corrected Steon2 compile errors)
------------------------------------------------------------------------------*/
void aud_setup_afei2s_vb_on_parameters(
    U16 csel,   /* csel parameter */
    U16 den0,   /* den0 parameter */
    U16 den1,   /* den1 parameter */
    U16 num0,   /* num0 parameter */
    U16 num1,   /* num1 parameter */
    U16 rxconf, /* rxconf parameter */
    U16 txconf,
    U16 hwmode,
    aud_I2S_samplerate_enum sample
    )/* i2s1 channel mode parameter */
{
    aud_dsp_afei2s_vb_hw_parms_global.afe_i2s_csel   = csel;
    aud_dsp_afei2s_vb_hw_parms_global.afe_i2s_den0   = den0;
    aud_dsp_afei2s_vb_hw_parms_global.afe_i2s_den1   = den1;
    aud_dsp_afei2s_vb_hw_parms_global.afe_i2s_num0   = num0;
    aud_dsp_afei2s_vb_hw_parms_global.afe_i2s_num1   = num1;
    aud_dsp_afei2s_vb_hw_parms_global.afe_i2s_rxconf = rxconf;
    aud_dsp_afei2s_vb_hw_parms_global.afe_i2s_txconf = txconf;

    if(hwmode == AUD_I2S_HW_MODE_BURST)
    {
        aud_dsp_afei2s_trans_mode = aud_I2S_transmission_mode_PCM;
    }
    else
    {
        aud_dsp_afei2s_trans_mode = aud_I2S_transmission_mode_normal;
    }

    aud_dsp_afei2s_sample = sample;
}

/*------------------------------------------------------------------------------
* Function...: aud_uplinkpath_update_buffers
* Return.....: -
* Description: Updates the buffer that holds the active uplink paths.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void aud_uplinkpath_update_buffers(
    U32 parm1,  /* path 0-3 */
    U32 parm2,  /* path 4-7 */
    U32 parm3,  /* path 8-11 */
    U32 parm4,  /* path 12-15 */
    U32 parm5,  /* path 16-19 */
    U32 parm6,  /* path 20-23 */
    U32 parm7,  /* path 24-27 */
    U32 parm8)  /* path 28-31 */
{
    U8 uplinkpath_Status_Temp[32];//aud_uplink_source_enum
    U8 m = 0;

    memcpy(&uplinkpath_Status_Temp[0], &parm1, sizeof(parm1));
    memcpy(&uplinkpath_Status_Temp[4], &parm2, sizeof(parm2));
    memcpy(&uplinkpath_Status_Temp[8], &parm3, sizeof(parm3));
    memcpy(&uplinkpath_Status_Temp[12], &parm4, sizeof(parm4));
    memcpy(&uplinkpath_Status_Temp[16], &parm5, sizeof(parm5));
    memcpy(&uplinkpath_Status_Temp[20], &parm6, sizeof(parm6));
    memcpy(&uplinkpath_Status_Temp[24], &parm7, sizeof(parm7));
    memcpy(&uplinkpath_Status_Temp[28], &parm8, sizeof(parm8));

    for(m=0; m<aud_uplink_source_end+1; m++) /* copy path to active paths */
    {
        if(aud_uplink_source_end == uplinkpath_Status_Temp[m])
        {
            break;
        }
        aud_path_active_uplink_paths.path[m]= uplinkpath_Status_Temp[m];  
    }
    aud_path_active_uplink_paths.total=m;

    for(; m<aud_uplink_source_end+1; m++)
    {
        aud_path_active_uplink_paths.path[m]=aud_uplink_source_end;
    }
}

/*------------------------------------------------------------------------------
* Function...: aud_downlinkpath_update_buffers
* Return.....: -
* Description: Updates the buffer that holds the active downlink paths.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void aud_downlinkpath_update_buffers(
    U32 parm1,  /* path 0-3 */
    U32 parm2,  /* path 4-7 */
    U32 parm3,  /* path 8-11 */
    U32 parm4,  /* path 12-15 */
    U32 parm5,  /* path 16-19 */
    U32 parm6,  /* path 20-23 */
    U32 parm7,  /* path 24-27 */
    U32 parm8)  /* path 28-31 */
{
    U8 downlinkpath_Status_Temp[32];//aud_downlink_source_enum
    U8 m = 0;

    memcpy(&downlinkpath_Status_Temp[0],  &parm1, sizeof(parm1));
    memcpy(&downlinkpath_Status_Temp[4],  &parm2, sizeof(parm2));
    memcpy(&downlinkpath_Status_Temp[8],  &parm3, sizeof(parm3));
    memcpy(&downlinkpath_Status_Temp[12], &parm4, sizeof(parm4));
    memcpy(&downlinkpath_Status_Temp[16], &parm5, sizeof(parm5));
    memcpy(&downlinkpath_Status_Temp[20], &parm6, sizeof(parm6));
    memcpy(&downlinkpath_Status_Temp[24], &parm7, sizeof(parm7));
    memcpy(&downlinkpath_Status_Temp[28], &parm8, sizeof(parm8));

    for(m=0; m<aud_downlink_source_end+1; m++) /* copy path to active paths */
    {
        if(aud_downlink_source_end == downlinkpath_Status_Temp[m])
        {
            break;
        }
        aud_path_active_downlink_paths.path[m]= downlinkpath_Status_Temp[m];  
    }
    aud_path_active_downlink_paths.total=m;

    for(; m<aud_downlink_source_end+1; m++)
    {
        aud_path_active_downlink_paths.path[m]=aud_downlink_source_end;
    }

#if defined (AUD_ENABLE_HAL)
    AudHalCheckForcePowerOff(&aud_path_active_downlink_paths.path);
#endif
}

/*------------------------------------------------------------------------------
* Function...: aud_calculate_uplink_resulting_path
* Return.....: -
* Description: Runs through the active path and identifies the values to be
*              used, and stores the result in aud_path_settings_result.
* Modified...: 10.07.2007 by KBJ (added Steon2 exceptions)
------------------------------------------------------------------------------*/
void aud_calculate_uplink_resulting_path(void)
{
    U8 active_path_counter = 0; /* used to loop through active paths */
    U8 nof_active_path = 0;     /* number of active paths */

#ifdef AUDIO_LLT
    aud_common_trace(llt_group_audio, llt_type_audio_path, 0, 0,                            /* LLT params */
                     'G', 0x51,                                                             /* LIS params */
                     aud_log_lis_display, aud_log_lis_display_nodata, aud_log_lis_trace);   /* LIS params */
#endif

    /******************** identify NOF active UL paths ********************/
    for(active_path_counter=0; active_path_counter<aud_uplink_source_end; active_path_counter++)
    {
        if(aud_path_active_uplink_paths.path[active_path_counter] != aud_uplink_source_end)
        {
            nof_active_path++;
        }
        else
        {
            break;
        }
    }
    /******************** Identify gain cells ********************/
    /* control of all gain cells follows the following pattern:
        1. set to default setting - if no valid value is identified later
        2. search through active paths to find a valid value - if none is found, the default value is used
        3. write the value to the structure for the DSP command */

    /* scal_mic */
    aud_path_settings_result.uplink_parms.p_uplink_gain_cells->scal_mic = aud_uplink_source_default_data.p_uplink_gain_cells->scal_mic;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(TRUE == aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_uplink_gain_cells->scal_mic_use)
        {
            aud_path_settings_result.uplink_parms.p_uplink_gain_cells->scal_mic = aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_uplink_gain_cells->scal_mic;
            break;
        }
    }
#if defined (STEON2_PROJECT)
    aud_steon2_exceptions(); /* For the Steon2 project, there is an exception for this gain cell */
#else
    aud_dsp_gain_parms.scal_mic = aud_path_settings_result.uplink_parms.p_uplink_gain_cells->scal_mic;
#endif /* STEON2_PROJECT */

    /* lambda0 */
    aud_path_settings_result.uplink_parms.p_uplink_gain_cells->lambda0 = aud_uplink_source_default_data.p_uplink_gain_cells->lambda0;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(TRUE == aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_uplink_gain_cells->lambda0_use)
        {
            aud_path_settings_result.uplink_parms.p_uplink_gain_cells->lambda0 = aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_uplink_gain_cells->lambda0;
            break;
        }
    }
    aud_dsp_gain_parms.lambda0 = aud_path_settings_result.uplink_parms.p_uplink_gain_cells->lambda0;

    /* gamma0 */
    aud_path_settings_result.uplink_parms.p_uplink_gain_cells->gamma0 = aud_uplink_source_default_data.p_uplink_gain_cells->gamma0;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(TRUE == aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_uplink_gain_cells->gamma0_use)
        {
            aud_path_settings_result.uplink_parms.p_uplink_gain_cells->gamma0 = aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_uplink_gain_cells->gamma0;
            break;
        }
    }
    aud_dsp_vm_parms.gamma0 = aud_path_settings_result.uplink_parms.p_uplink_gain_cells->gamma0;

    /* scal_afe */
    aud_path_settings_result.uplink_parms.p_uplink_gain_cells->scal_afe = aud_uplink_source_default_data.p_uplink_gain_cells->scal_afe;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(TRUE == aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_uplink_gain_cells->scal_afe_use)
        {
            aud_path_settings_result.uplink_parms.p_uplink_gain_cells->scal_afe = aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_uplink_gain_cells->scal_afe;
            break;
        }
    }
    aud_dsp_gain_parms.scal_afe = aud_path_settings_result.uplink_parms.p_uplink_gain_cells->scal_afe;

    /* scal_mic2 */
    aud_path_settings_result.uplink_parms.p_uplink_gain_cells->scal_mic2 = aud_uplink_source_default_data.p_uplink_gain_cells->scal_mic2;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(TRUE == aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_uplink_gain_cells->scal_mic2_use)
        {
            aud_path_settings_result.uplink_parms.p_uplink_gain_cells->scal_mic2 = aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_uplink_gain_cells->scal_mic2;
            break;
        }
    }
    aud_dsp_gain_parms.scal_mic2 = aud_path_settings_result.uplink_parms.p_uplink_gain_cells->scal_mic2;

    /* afe_tone */
    aud_path_settings_result.uplink_parms.p_uplink_gain_cells->afe_tone = aud_uplink_source_default_data.p_uplink_gain_cells->afe_tone;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(TRUE == aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_uplink_gain_cells->afe_tone_use)
        {
            aud_path_settings_result.uplink_parms.p_uplink_gain_cells->afe_tone = aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_uplink_gain_cells->afe_tone;
            break;
        }
    }
    aud_dsp_gain_parms.afe_tone = aud_path_settings_result.uplink_parms.p_uplink_gain_cells->afe_tone;

    /* mix_pcmrec - media_to_uplink*/
    aud_path_settings_result.uplink_parms.p_uplink_gain_cells->mix_pcmrec = FALSE;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(TRUE == aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_uplink_gain_cells->mix_pcmrec)
        {
            aud_path_settings_result.uplink_parms.p_uplink_gain_cells->mix_pcmrec = TRUE;
            break;
        }
    }
#if defined (STEON2_PROJECT)
    aud_steon2_exceptions(); /* For Steon2, this gain cell is controlled differently */
#else
    if(aud_path_settings_result.uplink_parms.p_uplink_gain_cells->mix_pcmrec == TRUE)
    {
        aud_dsp_cbuf_gain_parms.mix_pcmrec = aud_constant_gain_values.mix_pcmrec;
    }
    else
    {
        aud_dsp_cbuf_gain_parms.mix_pcmrec = 0;
    }
#endif /* STEON2_PROJECT */

    /******************** identify filter settings to be used ********************/
    /* biquad in filter for 8 kHz sample rate */
    aud_path_settings_result.uplink_parms.p_filter_8kHz = aud_uplink_source_default_data.p_filter_8kHz;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_filter_8kHz)
        {
            aud_path_settings_result.uplink_parms.p_filter_8kHz = aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_filter_8kHz;
            break;
        }
    } /* programmed by DspSetFilter_uplink() */
    /* biquad in filter for 16 kHz sample rate */
    aud_path_settings_result.uplink_parms.p_filter_16kHz = aud_uplink_source_default_data.p_filter_16kHz;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_filter_16kHz)
        {
            aud_path_settings_result.uplink_parms.p_filter_16kHz = aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_filter_16kHz;
            break;
        }
    } /* programmed by DspSetFilter_uplink() */
    /* biquad in filter for 48 kHz sample rate */
    aud_path_settings_result.uplink_parms.p_filter_48kHz = aud_uplink_source_default_data.p_filter_48kHz;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_filter_48kHz)
        {
            aud_path_settings_result.uplink_parms.p_filter_48kHz = aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_filter_48kHz;
            break;
        }
    } /* programmed by DspSetFilter_uplink() */
    /*[Begin]no use. [lvwenhua-2010/9/16]*/
    #if 0
    aud_path_settings_result.uplink_parms.p_fir_filter_ul = aud_uplink_source_default_data.p_fir_filter_ul;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_fir_filter_ul)
        {
            aud_path_settings_result.uplink_parms.p_fir_filter_ul = aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_fir_filter_ul;
            break;
        }
    } /* programmed by DspSetFilter_uplink() */
    #endif
    /*[End] [lvwenhua-2010/9/16]*/

    /******************** identify hf settings to be used ********************/
    /* hf */
    aud_path_settings_result.uplink_parms.p_hf = aud_uplink_source_default_data.p_hf;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_hf)
        {
            aud_path_settings_result.uplink_parms.p_hf = aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_hf;
            break;
        }
    } /* programmed by aud_drv_activate_hf() */

    /*[Begin]no use. [lvwenhua-2010/9/16]*/
    #if 0
    /******************** identify asp settings to be used ********************/
    /* acoustic shock protection */
    aud_path_settings_result.uplink_parms.p_asp_ul = aud_uplink_source_default_data.p_asp_ul;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_asp_ul)
        {
            aud_path_settings_result.uplink_parms.p_asp_ul = aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->p_asp_ul;
            break;
        }
    } /* programmed by aud_path_program_asp() */
    #endif
    /*[End] [lvwenhua-2010/9/16]*/

    /******************** Identify misc values ********************/
    /* mic_gain */
    aud_path_settings_result.uplink_parms.mic_gain = aud_uplink_source_default_data.mic_gain;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(TRUE == aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->mic_gain_use)
        {
            aud_path_settings_result.uplink_parms.mic_gain = aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->mic_gain;
            break;
        }
    } /* programmed by aud_path_afe_dsp_programming() */

    /*[Begin]no use. [lvwenhua-2010/9/16]*/
    #if 0
    /* tx_dither */
    aud_path_settings_result.uplink_parms.tx_dither = aud_uplink_source_default_data.tx_dither;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(TRUE == aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->tx_dither_use)
        {
            aud_path_settings_result.uplink_parms.tx_dither = aud_setting.p_aud_audio_uplink_parms[aud_path_active_uplink_paths.path[active_path_counter]]->tx_dither;
            break;
        }
    } /* never programmed */
    #endif
    /*[End] [lvwenhua-2010/9/16]*/

    aud_update_side_tone(); /* to make sure the side tone is muted when aud_tty_uplink is active */
}

/*------------------------------------------------------------------------------
* Function...: aud_calculate_downlink_resulting_path
* Return.....: -
* Description: Runs through the active path and identifies the values to be
*              used, and stores the result in aud_path_settings_result.
* Modified...: 01.10.2007 by KBJ (added Steon2 exceptions)
------------------------------------------------------------------------------*/
void aud_calculate_downlink_resulting_path(void)
{
    U8 active_path_counter=0; /* used to loop through active paths */
    U8 nof_active_path=0;     /* number of active paths */

#ifdef AUDIO_LLT
    aud_common_trace(llt_group_audio, llt_type_audio_path, 0, 0,                           /* LLT params */
                     'G', 0x52,                                                            /* LIS params */
                     aud_log_lis_display, aud_log_lis_display_nodata, aud_log_lis_trace ); /* LIS params */
#endif


    /******************** identify NOF active DL paths ********************/
    for(active_path_counter=0; active_path_counter<aud_downlink_source_end; active_path_counter++)
    {
        if(aud_path_active_downlink_paths.path[active_path_counter] != aud_downlink_source_end)
        {
            nof_active_path++;
        }
        else
        {
            break;
        }
    }
    /******************** Identify gain cells ********************/
    /* control of all gain cells follows the following pattern:
        1. set to default setting - if no valid value is identified later
        2. search through active paths to find a valid value - if none is found, the default value is used
        3. write the value to the structure for the DSP command */
    /* gain_out */
    aud_path_settings_result.downlink_parms.p_downlink_gain_cells->gain_out = aud_downlink_source_default_data.p_downlink_gain_cells->gain_out;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(TRUE == aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_downlink_gain_cells->gain_out_use)
        {
            aud_path_settings_result.downlink_parms.p_downlink_gain_cells->gain_out = aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_downlink_gain_cells->gain_out;
            break;
        }
    }
    aud_dsp_gain_parms.gain_out = aud_path_settings_result.downlink_parms.p_downlink_gain_cells->gain_out;

    /* kappa0 */
    aud_path_settings_result.downlink_parms.p_downlink_gain_cells->kappa0 = aud_downlink_source_default_data.p_downlink_gain_cells->kappa0;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(TRUE == aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_downlink_gain_cells->kappa0_use)
        {
            aud_path_settings_result.downlink_parms.p_downlink_gain_cells->kappa0 =aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_downlink_gain_cells->kappa0;
            break;
        }
    }
    aud_dsp_gain_parms.kappa0 = aud_path_settings_result.downlink_parms.p_downlink_gain_cells->kappa0;

    /******************** identify filter settings to be used ********************/
    /* biquad out and biquad cbuf filters for 8 and 16 kHz sample rate */
    aud_path_settings_result.downlink_parms.p_filter_8kHz       = aud_downlink_source_default_data.p_filter_8kHz;
    aud_path_settings_result.downlink_parms.p_filter_16kHz      = aud_downlink_source_default_data.p_filter_16kHz;
    aud_path_settings_result.downlink_parms.p_cbuf_filter       = aud_downlink_source_default_data.p_cbuf_filter;
    /*[Begin]no use. [lvwenhua-2010/9/16]*/
    #if 0
    aud_path_settings_result.downlink_parms.p_fir_filter_dl     = aud_downlink_source_default_data.p_fir_filter_dl;
    aud_path_settings_result.downlink_parms.p_fir_filter_cbuf   = aud_downlink_source_default_data.p_fir_filter_cbuf;
    #endif
    /*[End] [lvwenhua-2010/9/16]*/
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_filter_8kHz)
        {
            aud_path_settings_result.downlink_parms.p_filter_8kHz = aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_filter_8kHz;
            break;
        }
    } /* programmed by DspSetFilter_downlink */
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_filter_16kHz)
        {
            aud_path_settings_result.downlink_parms.p_filter_16kHz = aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_filter_16kHz;
            break;
        }
    } /* programmed by DspSetFilter_downlink */
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_cbuf_filter)
        {
            aud_path_settings_result.downlink_parms.p_cbuf_filter = aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_cbuf_filter;
            break;
        }
    } /* programmed by DspSetFilter_downlink */
    /*[Begin]no use. [lvwenhua-2010/9/16]*/
    #if 0
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_fir_filter_dl)
        {
            aud_path_settings_result.downlink_parms.p_fir_filter_dl = aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_fir_filter_dl;
            break;
        }
    } /* programmed by DspSetFilter_downlink */
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_fir_filter_cbuf)
        {
            aud_path_settings_result.downlink_parms.p_fir_filter_cbuf = aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_fir_filter_cbuf;
            break;
        }
    } /* programmed by DspSetFilter_downlink */

    /******************** identify downlink noise reduction ********************/
    /* p_dl_nr */
    aud_path_settings_result.downlink_parms.p_dl_nr = aud_downlink_source_default_data.p_dl_nr;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_dl_nr)
        {
            aud_path_settings_result.downlink_parms.p_dl_nr = aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_dl_nr;
            break;
        }
    }
    #endif
    /*[End] [lvwenhua-2010/9/16]*/

    /******************** Identify misc values ********************/
    /* side_tone_fact */
    aud_path_settings_result.downlink_parms.side_tone_fact = aud_downlink_source_default_data.side_tone_fact;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(TRUE == aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->side_tone_fact_use)
        {
            aud_path_settings_result.downlink_parms.side_tone_fact = aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->side_tone_fact;
            break;
        }
    }
    aud_update_side_tone();

    /*[Begin]no use. [lvwenhua-2010/9/16]*/
    #if 0
    /* sm_power_gain */
    aud_path_settings_result.downlink_parms.sm_pwer_gain = aud_downlink_source_default_data.sm_pwer_gain;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(TRUE == aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->sm_pwer_gain_use)
        {
            aud_path_settings_result.downlink_parms.sm_pwer_gain = aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->sm_pwer_gain;
            break;
        }
    } /* programmed by ???? */
    #endif
    /*[End] [lvwenhua-2010/9/16]*/

    /* mix_afe - media_to_downlink*/
    aud_path_settings_result.downlink_parms.p_downlink_gain_cells->mix_afe = aud_downlink_source_default_data.p_downlink_gain_cells->mix_afe;
    for(active_path_counter=0; active_path_counter<nof_active_path; active_path_counter++)
    {
        if(TRUE == aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_downlink_gain_cells->mix_afe_use)
        {
            aud_path_settings_result.downlink_parms.p_downlink_gain_cells->mix_afe = aud_setting.p_aud_audio_downlink_parms[aud_path_active_downlink_paths.path[active_path_counter]]->p_downlink_gain_cells->mix_afe;
            break;
        }
    }
#if defined (STEON2_PROJECT)
    aud_steon2_exceptions(); /* For the Steon2 project, this gain cell is controlled differently */
#else
    aud_update_mix_afe();
#endif /* STEON2_PROJECT */
}

/*******************************************************************************
*
*                           Internal functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: aud_path_populate_path_data
* Return.....: -
* Description: Updates the path pointers to point to EEPROM if this is required.
* Created....: 15.05.2006 by KBJ
* Modified...: 02.10.2007 by KBJ (corrected Steon2 compile errors)
------------------------------------------------------------------------------*/
static void aud_path_populate_path_data(void)
{
#if !defined (STEON2_PROJECT)
    U8 cnt=0;

    /*The command of DSP_CMD_VB_AUDIO_FILTER_SWITCH is no use.[Begin] [lvwenhua-2010/7/29]*/
    #if 0
    /* determine whether to use FIR and/or BiQuad filters */
    aud_dsp_filter_switch.dl_enable   = (word)aud_EEP_static.aud_path_data.aud_use_filters_dl;
    aud_dsp_filter_switch.ul_enable   = (word)aud_EEP_static.aud_path_data.aud_use_filters_ul;
    aud_dsp_filter_switch.cbuf_enable = (word)aud_EEP_static.aud_path_data.aud_use_filters_cbuf;
    DSP_ASYNC_VB_AUDIO_FILTER_SWITCH(&aud_dsp_filter_switch);
    #endif
    /*[End] [lvwenhua-2010/7/29]*/
    if(aud_EEP_static.aud_path_data.audio_parms_from_eep_used)
    {
        /* update pointers/values for uplink paths */
        for(cnt=0; cnt<aud_uplink_source_end && cnt<EEP_AUD_UPLINK_PATHS; cnt++)
        {
            if(aud_setting.p_aud_audio_uplink_parms[cnt]) /* if the path exists, replace the pointers with EEP adresses */
            {
                /*[Begin]modify for the changing of NV structure. [lvwenhua-2010/9/16]*/
                aud_setting.p_aud_audio_uplink_parms[cnt]->p_uplink_gain_cells = (aud_uplink_gain_cells_type*)&aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].uplink_gain_cells;
                if(aud_setting.p_aud_audio_uplink_parms[cnt]->p_filter_8kHz
                    /*&& aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].biquad_filter_ul_index_8kHz<EEP_AUD_BIQUAD_FILTERS_UL*/)
                {
                    aud_setting.p_aud_audio_uplink_parms[cnt]->p_filter_8kHz =
                        (T_DSP_CMD_VB_SET_BIQUAD_PAR*)&aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].biquad_filter_ul_8kHz;
                }
                if(aud_setting.p_aud_audio_uplink_parms[cnt]->p_filter_16kHz
                    /*&& aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].biquad_filter_ul_index_16kHz<EEP_AUD_BIQUAD_FILTERS_UL*/)
                {
                    aud_setting.p_aud_audio_uplink_parms[cnt]->p_filter_16kHz =
                        (T_DSP_CMD_VB_SET_BIQUAD_PAR*)&aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].biquad_filter_ul_16kHz;
                }
                if(aud_setting.p_aud_audio_uplink_parms[cnt]->p_filter_48kHz
                    /*&& aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].biquad_filter_ul_index_48kHz<EEP_AUD_BIQUAD_FILTERS_UL*/)
                {
                    aud_setting.p_aud_audio_uplink_parms[cnt]->p_filter_48kHz =
                        (T_DSP_CMD_VB_SET_BIQUAD_PAR*)&aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].biquad_filter_ul_48kHz;
                }
                /*[End] [lvwenhua-2010/9/16]*/
                /*[Begin]no use. [lvwenhua-2010/9/16]*/
                #if 0
                if(aud_setting.p_aud_audio_uplink_parms[cnt]->p_fir_filter_ul
                    && aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].fir_filter_ul_index<EEP_AUD_FIR_FILTERS_UL)
                {
                    aud_setting.p_aud_audio_uplink_parms[cnt]->p_fir_filter_ul =
                        &aud_EEP_static.aud_path_data.aud_uplink_fir_filters[aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].fir_filter_ul_index];
                }
                #endif
                /*[End] [lvwenhua-2010/9/16]*/
                if(aud_setting.p_aud_audio_uplink_parms[cnt]->p_hf
                    /*&& aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].hf_index<EEP_AUD_HF*/)
                {
                    aud_setting.p_aud_audio_uplink_parms[cnt]->p_hf =
                        (aud_hf_type*)&aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].hf;
                }
                /*[Begin]no use. [lvwenhua-2010/9/16]*/
                #if 0
                if(aud_setting.p_aud_audio_uplink_parms[cnt]->p_asp_ul
                    && aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].asp_ul_index<EEP_AUD_ASP_UL)
                {
                    aud_setting.p_aud_audio_uplink_parms[cnt]->p_asp_ul =
                        &aud_EEP_static.aud_path_data.aud_asp_ul[aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].asp_ul_index];
                }
                #endif
                /*[End] [lvwenhua-2010/9/16]*/
                aud_setting.p_aud_audio_uplink_parms[cnt]->mic_gain      =  aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].mic_gain;
                aud_setting.p_aud_audio_uplink_parms[cnt]->mic_gain_use  =  aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].mic_gain_use;
                /*[Begin]no use. [lvwenhua-2010/9/16]*/
                #if 0
                aud_setting.p_aud_audio_uplink_parms[cnt]->tx_dither     =  aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].tx_dither;
                aud_setting.p_aud_audio_uplink_parms[cnt]->tx_dither_use =  aud_EEP_static.aud_path_data.aud_audio_uplink_parms[cnt].tx_dither_use;
                #endif
                /*[End] [lvwenhua-2010/9/16]*/
            } /* if path exists */
        } /* for all uplink paths */
        /* update pointers/values for downlink paths */
        for (cnt=0; cnt<aud_downlink_source_end && cnt<EEP_AUD_DOWNLINK_PATHS; cnt++)
        {
            if (aud_setting.p_aud_audio_downlink_parms[cnt]) /* if the path exists, replace the pointers with EEP adresses */
            {
                /*[Begin]modify for the changing of NV structure. [lvwenhua-2010/9/16]*/
                aud_setting.p_aud_audio_downlink_parms[cnt]->p_downlink_gain_cells = (aud_downlink_gain_cells_type*)&aud_EEP_static.aud_path_data.aud_audio_downlink_parms[cnt].downlink_gain_cells;
                if(aud_setting.p_aud_audio_downlink_parms[cnt]->p_filter_8kHz
                    /*&& aud_EEP_static.aud_path_data.aud_audio_downlink_parms[cnt].biquad_filter_dl_index_8kHz<EEP_AUD_BIQUAD_FILTERS_DL*/)
                {
                    aud_setting.p_aud_audio_downlink_parms[cnt]->p_filter_8kHz =
                         (T_DSP_CMD_VB_SET_BIQUAD_PAR*)&aud_EEP_static.aud_path_data.aud_audio_downlink_parms[cnt].biquad_filter_dl_8kHz;
                }
                if(aud_setting.p_aud_audio_downlink_parms[cnt]->p_filter_16kHz
                    /*&& aud_EEP_static.aud_path_data.aud_audio_downlink_parms[cnt].biquad_filter_dl_index_16kHz<EEP_AUD_BIQUAD_FILTERS_DL*/)
                {
                    aud_setting.p_aud_audio_downlink_parms[cnt]->p_filter_16kHz =
                         (T_DSP_CMD_VB_SET_BIQUAD_PAR*)&aud_EEP_static.aud_path_data.aud_audio_downlink_parms[cnt].biquad_filter_dl_16kHz;
                }
                if(aud_setting.p_aud_audio_downlink_parms[cnt]->p_cbuf_filter
                    /*&& aud_EEP_static.aud_path_data.aud_audio_downlink_parms[cnt].biquad_filter_cbuf_index<EEP_AUD_BIQUAD_FILTERS_CBUF*/)
                {
                    aud_setting.p_aud_audio_downlink_parms[cnt]->p_cbuf_filter =
                         (T_DSP_CMD_VB_SET_BIQUAD_PAR*)&aud_EEP_static.aud_path_data.aud_audio_downlink_parms[cnt].biquad_filter_cbuf;
                }
                /*[End] [lvwenhua-2010/9/16]*/
                /*[Begin]no use. [lvwenhua-2010/9/16]*/
                #if 0
                if(aud_setting.p_aud_audio_downlink_parms[cnt]->p_fir_filter_dl
                    && aud_EEP_static.aud_path_data.aud_audio_downlink_parms[cnt].fir_filter_dl_index<EEP_AUD_FIR_FILTERS_DL)
                {
                    aud_setting.p_aud_audio_downlink_parms[cnt]->p_fir_filter_dl =
                        &aud_EEP_static.aud_path_data.aud_downlink_fir_filters[aud_EEP_static.aud_path_data.aud_audio_downlink_parms[cnt].fir_filter_dl_index];
                }
                if(aud_setting.p_aud_audio_downlink_parms[cnt]->p_fir_filter_cbuf
                    && aud_EEP_static.aud_path_data.aud_audio_downlink_parms[cnt].fir_filter_cbuf_index<EEP_AUD_FIR_FILTERS_CBUF)
                {
                    aud_setting.p_aud_audio_downlink_parms[cnt]->p_fir_filter_cbuf =
                        &aud_EEP_static.aud_path_data.aud_cbuf_fir_filters[aud_EEP_static.aud_path_data.aud_audio_downlink_parms[cnt].fir_filter_cbuf_index];
                }
                if(aud_setting.p_aud_audio_downlink_parms[cnt]->p_dl_nr
                    && aud_EEP_static.aud_path_data.aud_audio_downlink_parms[cnt].dl_nr_index<EEP_AUD_DL_NR)
                {
                    aud_setting.p_aud_audio_downlink_parms[cnt]->p_dl_nr =
                        &aud_EEP_static.aud_path_data.aud_dl_nr[aud_EEP_static.aud_path_data.aud_audio_downlink_parms[cnt].dl_nr_index];
                }
                #endif
                /*[End] [lvwenhua-2010/9/16]*/
                aud_setting.p_aud_audio_downlink_parms[cnt]->side_tone_fact     =  aud_EEP_static.aud_path_data.aud_audio_downlink_parms[cnt].side_tone_fact;
                aud_setting.p_aud_audio_downlink_parms[cnt]->side_tone_fact_use =  aud_EEP_static.aud_path_data.aud_audio_downlink_parms[cnt].side_tone_fact_use;
            } /* if path exists */
        } /* for all downlink paths */
    } /* if aud_EEP_static.aud_path_data.audio_parms_from_eep_used */
    else /* if aud_EEP_static.aud_path_data.audio_parms_from_eep_used==FALSE, keep the values specified in aud_data.c */
        ;
#endif /* STEON2_PROJECT */
}

/*------------------------------------------------------------------------------
* Function...: aud_path_afe_dsp_programming
* Return.....: -
* Description: Enables the AFE after configuring the control registers.
* Modified...: 23.10.2007 by KBJ (suppressed lint error)
------------------------------------------------------------------------------*/
static void aud_path_afe_dsp_programming(
    U32 configuration)  /* the 'afe_dsp_blocks' fields for active paths OR'ed together */
{
    SDL_boolean afe_ON=FALSE,I2s1_ON=FALSE,I2s1_inband_ON=FALSE;//xiu
    SDL_boolean afe_uplink_ON=FALSE,afe_downlink_ON=FALSE;

    /*These parameter is no use any more. lvwenhua*/
    //aud_dsp_afei2s_vb_hw_parms_global.reserved_afe_tx_ctrl_1 = 0;  //afe_tx_ctrl1;
    aud_dsp_afei2s_vb_hw_parms_global.reserved_afe_rx_ringer = 0;  //afe_rx_ringer;
    aud_dsp_afei2s_vb_hw_parms_global.reserved_afe_rx_ana3 = 0;   //afe_rx_ana3;
    aud_dsp_afei2s_vb_hw_parms_global.reserved_afe_rx_ctrl2 = 0;  //afe_rx_ctrl2;
    aud_dsp_afei2s_vb_hw_parms_global.reserved_afe_rx_epref = 0;  //afe_rx_epref;
    aud_dsp_afei2s_vb_hw_parms_global.reserved_afe_rx_gain_digital = 0;  //afe_rx_gain_digital;
    /*These parameter is no use any more. lvwenhua*/

    if(AUD_HW_DEP_SG_AFE & configuration)
    {
        afe_ON  = 1;
        
        if(AUD_HW_DEP_SG_AFEIIS_UL & configuration)
        {
            afe_uplink_ON  = 1;
        }
        
        if(AUD_HW_DEP_SG_AFEIIS_DL & configuration)
        {
            afe_downlink_ON = 1;
        }
    }
    if(AUD_HW_DEP_SG_I2S1 & configuration)
    {
        I2s1_ON = 1;
    }
    
    if(afe_ON || I2s1_ON)
    {
        aud_dsp_afei2s_vb_hw_parms_global.reserved_afe_tx_ctrl_1 = 0x8000;  //afe_tx_ctrl1;
    }
    else
    {
        aud_dsp_afei2s_vb_hw_parms_global.reserved_afe_tx_ctrl_1 = 0;  //afe_tx_ctrl1;
    }

    /* Configuration of AFEI2S interface */
    if(afe_ON)
    {
        AudAFEI2SSM(E_AFEI2S_enable, NULL);
    }
    else
    {
        AudAFEI2SSM(E_AFEI2S_disable, NULL);
    }

    /********############################################################################**********/
    /*[Begin] afe_tx_ctrl2 setting. [lvwenhua-2010/11/4]*/
    aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2 = 0;  //init afe_tx_ctrl2;
    if(afe_ON)
    {
        if(afe_uplink_ON || afe_downlink_ON)
        {
            aud_path_set_value(1, 0, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* afe_tx_ctrl2->txstart = on */
        }
        else
        {
            aud_path_set_value(0, 0, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* afe_tx_ctrl2->txstart = off */
        }

        if(aud_I2S_samplerate_08000 == aud_dsp_afei2s_sample)
        {
            aud_path_set_value(0, 3, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 2); /* afe_tx_ctrl2->sample_rate = 8k */
        }
        else if(aud_I2S_samplerate_16000 == aud_dsp_afei2s_sample)
        {
            aud_path_set_value(1, 3, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 2); /* afe_tx_ctrl2->sample_rate = 16k */
        }
        else if(aud_I2S_samplerate_48000 == aud_dsp_afei2s_sample)
        {
            aud_path_set_value(2, 3, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 2); /* afe_tx_ctrl2->sample_rate = 48k */
        }

        aud_path_set_value(1, 4, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* afe_tx_ctrl2->I2Sx_CTRL.I2SON = Clock On */
        
        if(afe_downlink_ON)
        {
            aud_path_set_value(1, 5, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* afe_tx_ctrl2->I2Sx_CTRL.I2STXSTART = Start tx */
        }
        else
        {
            aud_path_set_value(0, 5, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* afe_tx_ctrl2->I2Sx_CTRL.I2STXSTART = Start tx */
        }

        if(afe_uplink_ON)
        {
            aud_path_set_value(1, 6, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* afe_tx_ctrl2->I2Sx_CTRL.I2SRXSTART = Start rx */
        }
        else
        {
            aud_path_set_value(0, 6, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* afe_tx_ctrl2->I2Sx_CTRL.I2SRXSTART = Start rx */
        }

        if(aud_I2S_transmission_mode_normal == aud_dsp_afei2s_trans_mode)//normal mode
        {
            aud_path_set_value(0, 7, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* =0?? */ /* afe_tx_ctrl2->I2Sx_CTRL.TXPCMCON = Continous transmission in PCM mode */
            aud_path_set_value(0, 8, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* =0?? */ /* afe_tx_ctrl2->I2Sx_CTRL.RXPCMCON = Continous transmission in PCM mode */

            aud_path_set_value(0, 9, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* afe_tx_ctrl2->I2Sx_CTRL.TXPCM = Normal mode */
            aud_path_set_value(0, 10, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* afe_tx_ctrl2->I2Sx_CTRL.RXPCM = Normal mode */
        }
        else //pcm mode
        {
            aud_path_set_value(1, 7, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* =0?? */ /* afe_tx_ctrl2->I2Sx_CTRL.TXPCMCON = Continous transmission in PCM mode */
            aud_path_set_value(1, 8, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* =0?? */ /* afe_tx_ctrl2->I2Sx_CTRL.RXPCMCON = Continous transmission in PCM mode */
        
            aud_path_set_value(1, 9, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* afe_tx_ctrl2->I2Sx_CTRL.TXPCM = pcm mode */
            aud_path_set_value(1, 10, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* afe_tx_ctrl2->I2Sx_CTRL.RXPCM = pcm mode */
        }

        aud_path_set_value(0, 11, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 1); /* afe_tx_ctrl2->I2Sx_CTRL.DAI_EN = dai off */
        aud_path_set_value(0, 15, &(aud_dsp_afei2s_vb_hw_parms_global.afe_tx_ctrl2), 4); /* afe_tx_ctrl2->tx_init_addr = 0 */
    }
    /*[End] [lvwenhua-2010/11/4]*/
    /********############################################################################**********/
    
    /*copy afe iis modified configuration from aud_dsp_afei2s_vb_hw_parms_global to aud_dsp_vb_hw_parms_global*/
    memcpy(&aud_dsp_vb_hw_parms_global,  &aud_dsp_afei2s_vb_hw_parms_global, sizeof(aud_dsp_afei2s_vb_hw_parms_global));

    /*???????[Begin] [lvwenhua-2010/11/4]*/
    if(AUD_HW_DEP_SG_I2S1_INBAND_AFE_ON & configuration)
    {
        I2s1_inband_ON = 1;
    }
    /*[End] [lvwenhua-2010/11/4]*/

    /* Configuration of I2S1 interface */
    if(I2s1_inband_ON || I2s1_ON)
    {
        AudI2S1SM(E_I2S1_enable, NULL);
    }
    else
    {
        AudI2S1SM(E_I2S1_disable, NULL);
    }

    /* Control of VB_ON "ratesw" parameter */
    aud_dsp_vb_hw_parms_global.ratesw = aud_calculate_ratesw(); //uplink firmware sample
    /* always use stereo input in 48 kHz mode */
#if 0	
    if(8 == aud_dsp_vb_hw_parms_global.ratesw) /* 48 kHz */
    {
        aud_dsp_vb_hw_parms_global.input_mic_mode = 0; /* stereo */
    }
    else
    {
        aud_dsp_vb_hw_parms_global.input_mic_mode = 0; /* mono */   // ????? shj MIC1 or MIC2 why
    }
#endif
    aud_dsp_vb_hw_parms_global.input_mic_mode = 0; 
    if(AUD_HW_DEP_AFE_OUTPUT_STEREO & configuration) /* Control of VB_ON "out_mode" parameter (stereo or mix to mono) */
    {
        aud_dsp_vb_hw_parms_global.out_mode = 1; /* stereo */
    }
    else
    {
        aud_dsp_vb_hw_parms_global.out_mode = 0; /* mono */
    }

    /* Control of VB_ON "Switch" parameter */
    if(afe_ON)
    {
        if(I2s1_ON)
        {
            aud_dsp_vb_hw_parms_global.Switch = I2S1_ON_AFE_OUT_ON;         /* VB_ON->Switch = 4 */
        }
        else
        {
            aud_dsp_vb_hw_parms_global.Switch = INIT_AFE_COPYPARAM_I2S1OFF; /* VB_ON->Switch = 2 */
        }
    }
    else
    { /* AFE off */
        if(I2s1_ON)
        {
            aud_dsp_vb_hw_parms_global.Switch  = I2S1_ON_AFE_OFF;           /* VB_ON->Switch = 3 */
        }
        else
        {
            aud_dsp_vb_hw_parms_global.Switch  = AFE_OFF_AND_I2S1OFF;       /* VB_ON->Switch = 0 */
        }
    }

    if(AFE_OFF_AND_I2S1OFF == aud_dsp_vb_hw_parms_global.Switch)       /* Try to switch off I2S, */
    {
        if(aud_one_baseband_SM_active())                               /* but if one baseband resource is active/running */ 
        {
            aud_dsp_vb_hw_parms_global.Switch = COPYPARAMETERS_TO_AFE; /* don't do it because it could be used */
        }
    }

    /************************** execute VB_ON command *************************/
    { /* begin scope */
        static T_DSP_CMD_VB_HW_PAR aud_dsp_vb_hw_parms_mirror;          /* used to compare succeeding commands */

        if(memcmp(&aud_dsp_vb_hw_parms_global,&aud_dsp_vb_hw_parms_mirror, sizeof(T_DSP_CMD_VB_HW_PAR))) /* i.e. there are changes */
        {
#ifndef DATACARD_VERSION			
            DSP_ASYNC_VB_HW(&aud_dsp_vb_hw_parms_global);                   /* execute VB_ON command */
#endif
            aud_dsp_vb_hw_parms_mirror = aud_dsp_vb_hw_parms_global;        /* store this command so that we can compare the next time */
        }
    } /* end scope */

    if(TRUE==aud_vb_on_status.vb_on && AFE_OFF_AND_I2S1OFF==aud_dsp_vb_hw_parms_global.Switch) /* Running, switch off */
    {
        aud_vb_on_status.vb_on = FALSE;
        if(NULL != aud_vb_on_status.vb_on_status_function)
        {
            (*aud_vb_on_status.vb_on_status_function)(aud_vb_on_status.vb_on);
        }
    }
    else if(FALSE==aud_vb_on_status.vb_on && aud_dsp_vb_hw_parms_global.Switch>COPYPARAMETERS_TO_AFE) /* Not running, switch on */
    {
        aud_vb_on_status.vb_on = TRUE;
        if(NULL != aud_vb_on_status.vb_on_status_function)
        {
            (*aud_vb_on_status.vb_on_status_function)(aud_vb_on_status.vb_on);
        }

        #if 1
      	//vp_VoiceWorkStart();
	//zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "vp_VoiceWorkStart");
        #endif
    }
}

/*------------------------------------------------------------------------------
* Function...: aud_path_configure_audio_path
* Return.....: -
* Description: Sets up the AFE and external hardware according to the settings
*              of the active uplink and downlink paths.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static void aud_path_configure_audio_path(
    U32 mode)   /* 0:Off, 1:On */
{
    /*[Begin] [lvwenhua-2010/10/27]*/
    #if 0
    static unsigned long             current_system_state = 0;
    #endif
    /*[End] [lvwenhua-2010/10/27]*/
    U32                              configuration=0;       /* Default: switch off afe when mode==0 */
    /*[Begin] [lvwenhua-2010/10/27]*/
    #if 0
    unsigned long                    new_system_state = 0;  /* Default: switch off hw when mode==0 */
    unsigned long                    actbit;
    unsigned long                    state_change;
    unsigned long                    power_off;
    unsigned long                    power_on;
    unsigned char                    state_change_done = 0;
    #endif
    /*[End] [lvwenhua-2010/10/27]*/
    unsigned char                    ndlp, nulp;
    unsigned char                    pathtype;
    /*[Begin] [lvwenhua-2010/10/27]*/
    #if 0
    aud_exception_type              *p_ex_iterator;  /* exception iterator */
    aud_exception_action_table_type *p_act_iterator; /* action iterator */
    #endif
    /*[End] [lvwenhua-2010/10/27]*/

    /* Calculate the new system state from the current paths
       both up- and downlink paths */
    if(1 == mode) /* This is a request to activate path */
    {
        for(ndlp=0; aud_path_active_downlink_paths.path[ndlp]!=aud_downlink_source_end; ndlp++)
        {
            pathtype = aud_path_active_downlink_paths.path[ndlp];
            /*[Begin] [lvwenhua-2010/10/27]*/
            #if 0
            new_system_state |= aud_setting.p_aud_audio_downlink_parms[pathtype]->ext_hw_blocks;
            #endif
            /*[End] [lvwenhua-2010/10/27]*/
            configuration    |= aud_setting.p_aud_audio_downlink_parms[pathtype]->afe_dsp_blocks;
        }
        for(nulp=0; aud_path_active_uplink_paths.path[nulp]!=aud_uplink_source_end; nulp++)
        {
            pathtype = aud_path_active_uplink_paths.path[nulp];
            /*[Begin] [lvwenhua-2010/10/27]*/
            #if 0
            new_system_state |= aud_setting.p_aud_audio_uplink_parms[pathtype]->ext_hw_blocks;
            #endif
            /*[End] [lvwenhua-2010/10/27]*/
            configuration    |= aud_setting.p_aud_audio_uplink_parms[pathtype]->afe_dsp_blocks;
        }
    }

    /*[Begin] [lvwenhua-2010/10/27]*/
    #if 0
    /*
       XOR the current state mask and the new state mask to get
       an idea of what has changed since the last time

       old_state   0 0 1 1 0 1
       new_state   0 0 1 0 1 1
       ----------------------- XOR
       change      0 0 0 1 1 0
    */
    state_change = (current_system_state ^ new_system_state);

    /*
       Figure out what needs to be powered up and what needs
       to be powered down

       old_state   0 0 1 1 0 1       new_state  0 0 1 0 1 1
       change      0 0 0 1 1 0       change     0 0 0 1 1 0
       ----------------------- AND   ---------------------- AND
       power_off   0 0 0 1 0 0       power_on   0 0 0 0 1 0
    */
    power_off = (current_system_state & state_change);
    power_on = (new_system_state & state_change);

    /* Look for exceptions */

    /* Initialise the iterator */
    p_ex_iterator = aud_exception_table;
    /* Keep going until we hit the terminator */
    while(p_ex_iterator->p_table != (void *)0)
    {
        if( (p_ex_iterator->current_state == current_system_state) &&
            (p_ex_iterator->new_state     == new_system_state) )
        {
            /* A match was found, initialise the iterator */
            p_act_iterator = p_ex_iterator->p_table;
            /* Keep going until we hit the terminator */
            while( (p_act_iterator->param  != 0xFFFFFFFF) &&
                   (p_act_iterator->action != AUD_LAST_ACTION_BN) )
            {
                /* If we have a valid action function pointer, call it! */
                if(AUD_AFE_DSP_BN == p_act_iterator->action)
                {
                    aud_path_afe_dsp_programming(configuration);
                }
                else if(aud_actions[p_act_iterator->action].action)
                {
                    aud_actions[p_act_iterator->action].action(p_act_iterator->param);
                }
                p_act_iterator++;
            }
            state_change_done = 1;
            break; /* State transition is done, so break the outer loop */
        }
        p_ex_iterator++;
    }

    /* No exception was found for this state transition, so do default state transition handling */
    if(!state_change_done)
    {
        /* 1) power off hardware blocks, do it in reverse order from bit N to 0 */
        for(actbit=0; actbit<AUD_LAST_ACTION_BN; actbit++)
        {
            if(power_off & (1L<<((AUD_LAST_ACTION_BN-1)-actbit)))
            {
                aud_actions[((AUD_LAST_ACTION_BN-1)-actbit)].action(0);
            }
        }

        aud_path_afe_dsp_programming(configuration); /* Program the AFE when most blocks are powered off */

        /* 2) power on hardware blocks, do it in forward order from bit 0 to N */
        for(actbit=0; actbit<AUD_LAST_ACTION_BN; actbit++)
        {
            if(power_on & (1L<<actbit))
            {
                aud_actions[actbit].action(1);
            }
        }
    }

    /* Store the new state for the next time */
    current_system_state = new_system_state;
    #endif
    /*[End] [lvwenhua-2010/10/27]*/

    aud_path_afe_dsp_programming(configuration); /* Program the AFE when most blocks are powered off */
}
#ifdef _USE_CUT_TEAKLIT_CODE
/*------------------------------------------------------------------------------
* Function...: aud_path_set_dsp_fir_filters
* Return.....: -
* Description: Writes FIR filter settings to the DSP.
* Created....: 02.07.2007 by KBJ
* Modified...: 02.08.2007 by KBJ (removed 'static' keyword - used in aud_path_tst.c)
------------------------------------------------------------------------------*/
void aud_path_set_dsp_fir_filters(
    const word ul_dl_select,            /* which filter to program (0=dl, 1=ul, 2=cbuf) */
    const aud_fir_filter_type* filter)  /* pointer to the filter coefficients */
{
	U8 csi_count = 0;
	U8 h_count = 0;
    U8 add_one = 0;   /* 1st write has 17 parameters, 2nd, 3rd and 4th only have 16 */
	
    aud_dsp_fir_filter_struct.ul_dl_select = ul_dl_select; /* which filter to program */
    for(csi_count=0; csi_count<4; csi_count++)
    {
        aud_dsp_fir_filter_struct.csi = csi_count;
        for(h_count=0; h_count<17; h_count++)
        {
            if(0!=csi_count)
            {
                add_one=1; /* add 1 to the indexing in the h array for the chosen filter */
            }
            aud_dsp_fir_filter_struct.h[h_count] = filter->h[csi_count*16 + h_count + add_one];
        }
        DSP_ASYNC_VB_SET_FIR(&aud_dsp_fir_filter_struct);
    }
}
#endif
/*------------------------------------------------------------------------------
* Function...: aud_path_set_dsp_filters_uplink
* Return.....: -
* Description: Writes the uplink filter settings to the DSP according to the
*              sample rate of the audio scheduler.
* Modified...: 02.07.2007 by KBJ (added FIR filters)
------------------------------------------------------------------------------*/
static void aud_path_set_dsp_filters_uplink(
    const aud_ratesw_enum sample_rate)    /* the next ratesw value to be written */
{
    /*[Begin] [lvwenhua-2010/7/28]*/
    if((aud_get_SM_state(aud_resource_speech) == S_active)
        || (aud_get_SM_state(aud_resource_record_vm) == S_active)
        || (aud_get_SM_state(aud_resource_record_pcm) == S_active)
        || (aud_get_SM_state(aud_resource_I2S2_Tx) == S_active)
        || (aud_get_SM_state(aud_resource_sbc_encoder) == S_active))
    {
        switch(sample_rate)
        { /* determine which filter setting to use */
            case aud_ratesw_48khz_stereo:
                aud_dsp_uplink_filter_parms = *aud_path_settings_result.uplink_parms.p_filter_48kHz;
                break;
            case aud_ratesw_16khz_mono:
                aud_dsp_uplink_filter_parms = *aud_path_settings_result.uplink_parms.p_filter_16kHz;
                break;
            case aud_ratesw_8khz_mono:
            default:
                aud_dsp_uplink_filter_parms = *aud_path_settings_result.uplink_parms.p_filter_8kHz;
                break;
        }
        DSP_ASYNC_VB_SET_BIQUAD(&aud_dsp_uplink_filter_parms);

        #if 0
        aud_path_set_dsp_fir_filters(1/*uplink*/, aud_path_settings_result.uplink_parms.p_fir_filter_ul);
        #endif
    }
    /*[End] [lvwenhua-2010/7/28]*/
}

/*------------------------------------------------------------------------------
* Function...: aud_path_set_dsp_filters_downlink
* Return.....: -
* Description: Writes the downlink filter settings to the DSP according to the
*              sample rate of the audio scheduler.
* Modified...: 17.09.2007 by KBJ (removed 16 kHz cbuf filter)
------------------------------------------------------------------------------*/
static void aud_path_set_dsp_filters_downlink(
    const aud_ratesw_enum sample_rate)    /* the next ratesw value to be written */
{
    switch(sample_rate)
    { /* determine which filter setting to use */
        case aud_ratesw_16khz_mono:
            aud_dsp_downlink_filter_parms      = *aud_path_settings_result.downlink_parms.p_filter_16kHz;
            break;
        case aud_ratesw_48khz_stereo:
        case aud_ratesw_8khz_mono:
        default:
            aud_dsp_downlink_filter_parms      = *aud_path_settings_result.downlink_parms.p_filter_8kHz;
            break;
    }
    aud_dsp_downlink_cbuf_filter_parms = *aud_path_settings_result.downlink_parms.p_cbuf_filter; /* cbuf always runs at 48 kHz */

    /*[Begin] [lvwenhua-2010/7/28]*/
    if((aud_get_SM_state(aud_resource_speech) == S_active)
        || (aud_get_SM_state(aud_resource_tone_generator) == S_active)
        || (aud_get_SM_state(aud_resource_playback_vm) == S_active)
        || (aud_get_SM_state(aud_resource_I2S2_Rx) == S_active))
    {
        DSP_ASYNC_VB_SET_BIQUAD(&aud_dsp_downlink_filter_parms);
    }

    if(aud_one_cbuf_SM_active())
    {
        DSP_ASYNC_VB_SET_BIQUAD(&aud_dsp_downlink_cbuf_filter_parms);
    }
    
    #if 0
    aud_path_set_dsp_fir_filters(0/*downlink*/, aud_path_settings_result.downlink_parms.p_fir_filter_dl);
    aud_path_set_dsp_fir_filters(2/*cbuf*/, aud_path_settings_result.downlink_parms.p_fir_filter_cbuf);
    #endif
    /*[End] [lvwenhua-2010/7/28]*/
}

/*------------------------------------------------------------------------------
* Function...: aud_path_program_asp
* Return.....: -
* Description: Program acoustic shock protection.
* Created....: 10.07.2007 by KBJ
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
/*[Begin]no use. [lvwenhua-2010/9/16]*/
#if 0
static void aud_path_program_asp(void)
{
    static U8 asp_on = 0;
    if(NULL!=aud_path_settings_result.uplink_parms.p_asp_ul)
    {
        /* call VB_ASP_EP */
        if(0 == asp_on) /* if not active, initialise */
        {
            aud_dsp_asp_ul_parms.operation = 1; /* init & go */
            asp_on = 1;
        }
        else if(1 == asp_on) /* if already active, just update */
        {
            aud_dsp_asp_ul_parms.operation = 2; /* update */
            asp_on = 2;
        }
        aud_dsp_asp_ul_parms.attenn_dbfs = aud_path_settings_result.uplink_parms.p_asp_ul->attennuation_dbfs;
        aud_dsp_asp_ul_parms.attack_time = aud_path_settings_result.uplink_parms.p_asp_ul->attack_time;
        aud_dsp_asp_ul_parms.release_time = aud_path_settings_result.uplink_parms.p_asp_ul->release_time;
        aud_dsp_asp_ul_parms.mode = aud_path_settings_result.uplink_parms.p_asp_ul->mode;
        /* VB_HF_SET_GAIN should compensate for the attennuation done by VB_ASP_EP */
        aud_dsp_hf_set_gain.mode = aud_path_settings_result.uplink_parms.p_asp_ul->ul_speech_gain;
    }
    else /* no acoustic shock protection selected, so stop */
    {
        aud_dsp_asp_ul_parms.operation = 0; /* stop */
        asp_on = 0;
        aud_dsp_hf_set_gain.mode = 0;       /* 0dB gain */
    }
    DSP_ASYNC_VB_ASP_EP(&aud_dsp_asp_ul_parms);
    DSP_ASYNC_VB_HF_SET_GAIN(&aud_dsp_hf_set_gain);
}
#endif
/*[End] [lvwenhua-2010/9/16]*/

/*------------------------------------------------------------------------------
* Function...: aud_path_program_dl_nr
* Return.....: -
* Description: Program downlink noise reduction.
* Created....: 12.07.2007 by KBJ
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
/*[Begin]no use. [lvwenhua-2010/9/16]*/
#if 0
static void aud_path_program_dl_nr(void)
{
    static BOOL dl_nr_on = FALSE;
    if(NULL!=aud_path_settings_result.downlink_parms.p_dl_nr)
    {
        /* call VB_RX_NR_CONFIG */
        if(FALSE == dl_nr_on) /* if not active, initialise */
        {
            aud_dsp_dl_nr_parms.Switch = 0; /* init & config */
            dl_nr_on = TRUE;
            /* update audio filter switch */
            aud_dsp_filter_switch.dl_enable |= 0x0010; /* set bit 4 (activate downlink noise reduction) */
            DSP_ASYNC_VB_AUDIO_FILTER_SWITCH(&aud_dsp_filter_switch);
        }
        else /* if already active, just update */
        {
            aud_dsp_dl_nr_parms.Switch = 1; /* config */
        }
        aud_dsp_dl_nr_parms.nr_sw_2    = aud_path_settings_result.downlink_parms.p_dl_nr->nr_sw_2;
        aud_dsp_dl_nr_parms.nr_u_fak_0 = aud_path_settings_result.downlink_parms.p_dl_nr->nr_u_fak_0;
        aud_dsp_dl_nr_parms.nr_u_fak   = aud_path_settings_result.downlink_parms.p_dl_nr->nr_u_fak;
    }
    else /* no downlink noise reduction selected */
    {
        dl_nr_on = FALSE;
        /* update audio filter switch */
        aud_dsp_filter_switch.dl_enable &= 0xFFEF; /* unset bit 4 (deactivate downlink noise reduction) */
        DSP_ASYNC_VB_AUDIO_FILTER_SWITCH(&aud_dsp_filter_switch);
    }
    DSP_ASYNC_VB_RX_NR_CONFIG(&aud_dsp_dl_nr_parms);
}
#endif
/*[End] [lvwenhua-2010/9/16]*/

/*------------------------------------------------------------------------------
* Function...: aud_path_enable_filters_and_gains
* Return.....: -
* Description: Program DSP gain cells and filters used by the paths.
* Modified...: 13.07.2007 by KBJ (added downlink noise reduction)
------------------------------------------------------------------------------*/
static void aud_path_enable_filters_and_gains(void)
{
    aud_ratesw_enum sample_rate; /* sample rate for the audio scheduler */
    /* fetch gain cells from EEP */
/*[Begin]read scal_in and scal_i2s1 from NV [lvwenhua-2010/9/19]*/
#if 0
#ifndef _USE_FWP
    aud_dsp_gain_parms.scal_in  = (word) aud_EEP_static.aud_scal_in_parms[aud_path_active_uplink_paths.path[0]].scal_in;
#endif
#endif
    aud_dsp_gain_parms.scal_in  = (word) aud_EEP_static.aud_scal_in_parms[aud_path_active_uplink_paths.path[0]].scal_in;
/*[End] [lvwenhua-2010/9/19]*/
    aud_dsp_gain_parms.scal_out = (word) aud_EEP_static.aud_scal_out_parms[aud_path_active_downlink_paths.path[0]].scal_out;

    /*[Begin]read scal_in and scal_i2s1 from NV [lvwenhua-2010/9/19]*/
    aud_dsp_gain_parms.scal_I2S1 = (word) aud_EEP_static.aud_scal_i2s1_parms[aud_path_active_downlink_paths.path[0]].scal_i2s1;
    #if 0
    if(aud_dsp_gain_parms.scal_rec != 0)
    {
        aud_dsp_gain_parms.scal_rec = (word) aud_EEP_static.aud_scal_rec_parms[aud_path_active_downlink_paths.path[0]].scal_rec;
    }
    #endif
    /*[End] [lvwenhua-2010/9/16]*/

    /* calculate what the audio scheduler sample rate will be after the path update */
    sample_rate = aud_get_next_ratesw();
    /* program filter values found by aud_calculate_uplink_resulting_path() and aud_calculate_downlink_resulting_path() */
    aud_path_set_dsp_filters_uplink(sample_rate);
    aud_path_set_dsp_filters_downlink(sample_rate);
    /* program gain values found by aud_calculate_uplink_resulting_path() and aud_calculate_downlink_resulting_path() */
    aud_dsp_set_vm_gains();
    aud_dsp_set_vb_gains();
    aud_dsp_set_cbuf_gains();
    /* program hands-free parameters found by aud_calculate_uplink_resulting_path() */
    aud_drv_activate_hf();
    /*The command of DSP_CMD_VB_AUDIO_FILTER_SWITCH\DSP_CMD_VB_ASP_EP\DSP_CMD_VB_HF_SET_GAIN\DSP_CMD_VB_RX_NR_CONFIG is no use.[Begin] [lvwenhua-2010/7/29]*/
    #if 0
    /* program settings for acoustic shock protection found by aud_calculate_uplink_resulting_path() */
    aud_path_program_asp();
    /* program settings for downlink noise reduction found by aud_calculate_downlink_resulting_path() */
    aud_path_program_dl_nr();
    #endif
    /*[End] [lvwenhua-2010/7/28]*/
}

/*------------------------------------------------------------------------------
* Function...: aud_path_enable_gains_filters_and_path
* Return.....: -
* Description: Enables AFE and audio scheduler if any resource is active.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static void aud_path_enable_gains_filters_and_path(void)
{
    if(aud_one_SM_active())
    {
        aud_path_enable_filters_and_gains();
        aud_path_configure_audio_path(1);
    }
}

/*------------------------------------------------------------------------------
* Function...: aud_path_set_value
* Return.....: -
* Description: Updates the bit fields of the specified data.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static void aud_path_set_value(
    word value,         /* new value to write */
    word position,      /* the position to write the value to */
    word* p_data,       /* the address that contains the register */
    word HowManyBits)   /* how many bits in the register to update */
{
    word temp, temp1;

    switch(HowManyBits)
    {
        case 1:
            temp1 = 0x1;
            break;
        case 2:
            temp1 = 0x3;
            break;
        case 3:
            temp1 = 0x7;
            break;
        case 4:
            temp1 = 0xf;
            break;
        case 5:
            temp1 = 0x1f;
            break;
        default:
            temp1 = 0x0;
            break;
    }
    temp1 <<= (position - (HowManyBits - 1));
    *p_data &= ~temp1;
    temp = value;
    temp <<= (position - (HowManyBits - 1));
    *p_data |= temp;
}

/*******************************************************************************
*
*                 Functions only used by the test system
*
*******************************************************************************/
#ifdef _USE_CUT_TEAKLIT_CODE
/*------------------------------------------------------------------------------
* Function...: aud_enable_paths_for_test
* Return.....: -
* Description: Similar functionality as aud_path_enable_gains_filters_and_path
*              except that no resource needs to be active.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void aud_enable_paths_for_test(void)
{
    aud_path_enable_filters_and_gains();
    aud_path_configure_audio_path(1);
}
#endif
/*******************************************************************************
*
*                             Getter functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: aud_get_vb_on_ratesw
* Return.....: the ratesw value
* Description: Returns the ratesw value currently written to the DSP.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
word aud_get_vb_on_ratesw(void)
{
    return (aud_dsp_vb_hw_parms_global.ratesw);
}
#ifdef _USE_CUT_TEAKLIT_CODE
/*------------------------------------------------------------------------------
* Function...: aud_get_number_of_active_uplink_paths
* Return.....: the number of active uplink paths
* Description: Return the number of active uplink paths.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
U8 aud_get_number_of_active_uplink_paths(void)
{
    return (aud_path_active_uplink_paths.total);
}

/*------------------------------------------------------------------------------
* Function...: aud_get_number_of_active_downlink_paths
* Return.....: the number of active downlink paths
* Description: Returns the number of active downlink paths.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
U8 aud_get_number_of_active_downlink_paths(void)
{
    return (aud_path_active_downlink_paths.total);
}
#endif
/*------------------------------------------------------------------------------
* Function...: aud_get_uplink_path_with_priority
* Return.....: uplink path with specified priority
* Description: Returns the ID of the active path with the specified priority.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
aud_uplink_source_enum aud_get_uplink_path_with_priority(
    U8 priority)    /* the priority of the path */
{
    return (aud_path_active_uplink_paths.path[priority]);
}
#ifdef _USE_CUT_TEAKLIT_CODE
/*------------------------------------------------------------------------------
* Function...: aud_get_downlink_path_with_priority
* Return.....: downlink path with specified priority
* Description: Returns the ID of the active path with the specified priority.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
aud_downlink_source_enum aud_get_downlink_path_with_priority(
    U8 priority)    /* the priority of the path */
{
    return (aud_path_active_downlink_paths.path[priority]);
}
#endif
