/**
 *       Copyright (C) Infineon Technologies Denmark A/S. All rights reserved.
 *
 * This document contains proprietary information belonging to Infineon
 * Technologies Denmark A/S. Passing on and copying of this document, use
 * and communication of its contents is not permitted without prior written
 * authorisation.
 *
 * Description: 
 *   Functions needed by resources to control the audio scheduler sample rate.
 *   When the sample rate changes, resources are notified through their registered callback functions.
 *
 * Revision Information:
 *   File name: \dwddrv\AUD\src\aud_ratesw.c
 *   Version: \main\9
 *   Date: 2007-09-12 09:43:27
 *   Responsible: johansek
 *   Comment:
 *     Corrected coverity code analysis finding (SMS00660951).
 */

/*******************************************************************************
*
*                               Include files
*
*******************************************************************************/
//#include "scttypes.h"           /* must be included in all files */
//#include "ms.h"                 /* must be included in all files */
#include "bastypes.h"           /* must be included in all files */
//#include "dsp.h"                /* for DSP command types */
#include "aud_drv.h"            /* audio types and functions */
#include "aud_data.h"           /* data describing the paths */
#include "aud_com.h"            /* types and functions used only within the audio driver */
#include "aud_path_control.h"   /* for getting the current sample rate */
#include "aud_ratesw.h"         /* header file for ratesw control */

/*******************************************************************************
*
*                               Internal data
*
*******************************************************************************/
static aud_ratesw_control_type aud_ratesw_control[aud_resource_end]; /* Callback functions for resources interested in ratesw */

/*******************************************************************************
*
*                         Internal/private functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: aud_ratesw_init
* Return.....: -
* Description: Initialises the data needed to control the sample rate.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static void aud_ratesw_init(void)
{
    static BOOL AUD_ratesw_initialised = FALSE; /* Has the control array been initialised? */

    if(FALSE == AUD_ratesw_initialised)
    {
        U16 cnt=0;
        for(cnt=0; cnt<aud_resource_end; cnt++)
        {
            aud_ratesw_control[cnt].func = NULL;
        }
        AUD_ratesw_initialised = TRUE;
    }
}

/*------------------------------------------------------------------------------
* Function...: aud_notify_ratesw_change
* Return.....: -
* Description: Notify all registered callback functions that ratesw has changed.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static void aud_notify_ratesw_change(
    aud_ratesw_enum new_ratesw) /* what sample rate the audio scheduler has changed to */
{
    U16 cnt=0;
    for(cnt=0; cnt<aud_resource_end; cnt++)
    {
        if(NULL != aud_ratesw_control[cnt].func) /* if a callback function is registered for this resource, call it */
        {
            (*aud_ratesw_control[cnt].func)(new_ratesw);
        }
    }
}

/*------------------------------------------------------------------------------
* Function...: aud_calc_next_ratesw
* Return.....: the next sample rate to be written to the DSP
* Description: Based on the currently set paths, calculate the next sample rate
*              to be written.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
static aud_ratesw_enum aud_calc_next_ratesw(void)
{
    U16 cnt = 0; /* loop counter */
    U8 required = 0; /* this variable will change if any path requires a particular audio scheduler sample rate */
    aud_ratesw_enum result = aud_ratesw_8khz_mono; /* default result */
    aud_uplink_source_enum path_with_priority = aud_get_uplink_path_with_priority(cnt); /* the path linked with priority index */

    /* Check active uplink paths to see if a particular sample rate is required */
    while(path_with_priority<aud_uplink_source_end && 0==required)
    {
        required = aud_setting.p_aud_audio_uplink_parms[path_with_priority]->required_sample_rate;
        cnt++;
        path_with_priority = aud_get_uplink_path_with_priority(cnt);
    }
    /* Convert the requirement to the sample rate that is to be programmed to the DSP */
    if(0 != required)
    {
        switch(required)
        {
            case AUD_SAMPLE_RATE_48KHZ_STEREO:
                result = aud_ratesw_48khz_stereo;
                break;
            case AUD_SAMPLE_RATE_16KHZ_MONO:
                result = aud_ratesw_16khz_mono;
                break;
            case AUD_SAMPLE_RATE_8KHZ_MONO:
            default:
                result = aud_ratesw_8khz_mono;
                break;
        }
    }
    return (result);
}

/*******************************************************************************
*
*                           Global/public functions
*
*******************************************************************************/
/*------------------------------------------------------------------------------
* Function...: aud_get_current_ratesw
* Return.....: the sample rate currently written to the DSP
* Description: Get the sample rate currently programmed to the DSP.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
aud_ratesw_enum aud_get_current_ratesw(void)
{
    return ((aud_ratesw_enum)aud_get_vb_on_ratesw());
}

/*------------------------------------------------------------------------------
* Function...: aud_get_next_ratesw
* Return.....: the next sample rate to be written to the DSP
* Description: Get the sample rate to be programmed to the DSP based on the
*              currently active paths.
* Modified...: dd.mm.yyyy by NN
* Note.......: This function is required because aud_get_vb_on_ratesw is not
*              calculated at path addition unless a resource is active
*              => ratesw state not up to date.
------------------------------------------------------------------------------*/
aud_ratesw_enum aud_get_next_ratesw(void)
{
    aud_ratesw_enum result = aud_calc_next_ratesw();
    return (result);
}
#ifdef _USE_CUT_TEAKLIT_CODE
/*------------------------------------------------------------------------------
* Function...: aud_register_ratesw_observer
* Return.....: -
* Description: Register a callback function for a particular resource.
*              All registered functions will be called when ratesw changes.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
void aud_register_ratesw_observer(
    aud_resource_enum resource,                 /* the resource the callback functions is registered for */
    void (*func)(aud_ratesw_enum new_ratesw))   /* the callback function */
{
    aud_ratesw_init(); /* safe initialisation */
    aud_ratesw_control[resource].func = func;
}
#endif
/*------------------------------------------------------------------------------
* Function...: aud_calculate_ratesw
* Return.....: the sample rate to be written to the DSP
* Description: Based on active paths, calculate the sample rate to be to the DSP.
* Modified...: dd.mm.yyyy by NN
------------------------------------------------------------------------------*/
aud_ratesw_enum aud_calculate_ratesw(void)
{
    aud_ratesw_enum result = aud_ratesw_8khz_mono; /* Default result */
    aud_ratesw_init();                      /* Safe initialisation */
    result = aud_calc_next_ratesw();        /* Based on the active paths, calculate what to write to the DSP */
    if(aud_get_current_ratesw() != result)  /* If ratesw has changed, notify all registered callback functions */
    {
        aud_notify_ratesw_change(result);
    }
    return (result);
}

