/**
 *       Copyright (C) Danish Wireless Design A/S. All rights reserved.
 *
 * This document contains proprietary information belonging to Danish Wireless
 * Design A/S. Passing on and copying of this document, use and communication
 * of its contents is not permitted without prior written authorisation.
 *
 * Description:  Main State machine for the audio driver. The file also include the
 *               interface functions
 *
 * Revision Information:
 *   File name: \dwddrv\AUD\src\aud_tone.c
 *   Version: \main\52
 *   Date: 2007-10-02 09:10:50
 *   Responsible: soerenek
 *   Comment:
 *     merges steon changes
 */

//#include "scttypes.h"
//#include "ms.h"
//#include "hw_reg.h"             /* to include SGoldX registers */
//#include "REG_SGOLD_Regs.h"
#include "ptf.h"
//#include "ilevels.h"

/*changed by wangjun for NV change.[20090214]*/
#include "aud_nv.h"  //#include "eep.h" /* used for loading super visory tones */
#include "aud_drv.h"
#include "aud_com.h" 
#include "aud_data.h"
#include "aud_tone.h"
#include "aud_ext_def.h"
#include "aud_path_control.h"
//#include "trap.h"
#include "aud_ratesw.h"
#include "tone_api.h"
#include "aud_ext_intf.h"
#include "ThreadPriority.h"
#include "drvs_int.h"
#include "drvs_teak.h"




/*---------------------------------------------*/
/* Global data                                 */
/*---------------------------------------------*/
UINT16 aud_global_tone_dur_inter = AUD_GLOBAL_TONE_DUR_INTER_DEFAULT_VALUE;
aud_glob_tone_type     aud_glob_tone;
T_Tone_CallbackFunc aud_glob_tone_callback = NULL;

/*---------------------------------------------*/
/* External data                               */
/*---------------------------------------------*/
extern NU_HISR AUD_TONE_HISR_control;
extern VOID zDrv_ToneActive(VOID);


/*---------------------------------------------*/
/* Internal functions.                         */
/*---------------------------------------------*/
static void start_tone(aud_tone_id_enum tone_id);
static void stop_tone(void);
static void program_tone_generator(aud_tone_id_enum tone_id);
static void DspProgramToneGenerator(UINT16 freq1, UINT16 amp1, UINT16 freq2, UINT16 amp2, 
                                    UINT16 freq3, UINT16 amp3, UINT16 duration);
static void SuspendTone(UINT8 slot_id);
static UINT8 ResumeTone(UINT8 slot_id);

static UINT32 GetTotalPlaytimeForTone(aud_tone_id_enum tone_id, void *tone_data, UINT16 nof_tones
                                      ,aud_tone_type_enum Tone_Type);
static void SetPlayPositionForTone(aud_tone_id_enum tone_id, void *tone_data, UINT32 pos,aud_tone_type_enum Tone_Type);
static UINT32 GetPlayPositionForTone(void);
static UINT32 GetToneDuration(aud_tone_id_enum tone_id, void *tone_data, UINT16 tone_number
                              ,aud_tone_type_enum Tone_Type);
static void Set_tone_fade_in_and_fade_out_parameters(void);
static SINT16 Find_freq_prog_word_toSM(unsigned int freq);
static UINT16 Find_amp_prog_word_toSM(unsigned int freq, unsigned int amp);
static void next_tone(aud_tone_id_enum tone_id);
void Initialise_tone_vars(void);

/*---------------------------------------------*/
/* Internal data.                              */
/*---------------------------------------------*/
static aud_base_tone_ctrl_type  tone_ctrl[2];
static UINT16   duration;



/*******************************************************************/
/*                                                                                                                  */   
/*                         DTMF tones                                                                         */  
/*                                                                                                                   */   
/********************************************************************/
const aud_dual_tone_type aud_tone_DTMF_0_data[]=      { {941, 1336, 240}};
const aud_dual_tone_type aud_tone_DTMF_1_data[]=      { {697, 1209, 240}};
const aud_dual_tone_type aud_tone_DTMF_2_data[]=      { {697, 1336, 240}};
const aud_dual_tone_type aud_tone_DTMF_3_data[]=      { {697, 1477, 240}};
const aud_dual_tone_type aud_tone_DTMF_4_data[]=      { {770, 1209, 240}};
const aud_dual_tone_type aud_tone_DTMF_5_data[]=      { {770, 1336, 240}};
const aud_dual_tone_type aud_tone_DTMF_6_data[]=      { {770, 1477, 240}};
const aud_dual_tone_type aud_tone_DTMF_7_data[]=      { {852, 1209, 240}};
const aud_dual_tone_type aud_tone_DTMF_8_data[]=      { {852, 1336, 240}};
const aud_dual_tone_type aud_tone_DTMF_9_data[]=      { {852, 1477, 240}};
const aud_dual_tone_type aud_tone_DTMF_hash_data[]=   { {941, 1477, 240}};
const aud_dual_tone_type aud_tone_DTMF_asterix_data[]={ {941, 1209, 240}};
const aud_dual_tone_type aud_tone_DTMF_A_data[]=      { {697, 1633, 240}};
const aud_dual_tone_type aud_tone_DTMF_B_data[]=      { {770, 1633, 240}};
const aud_dual_tone_type aud_tone_DTMF_C_data[]=      { {852, 1633, 240}};
const aud_dual_tone_type aud_tone_DTMF_D_data[]=      { {941, 1633, 240}};

/*******************************************************************/
/*                         Key tones                                                                           */  
/********************************************************************/
const aud_single_tone_type aud_tone_key_1_data[]={ {FREQ_E2, 40},
    {FREQ_C1, 120}};

const aud_single_tone_type aud_tone_key_2_data[]={ {FREQ_C1, 40},
    {FREQ_E1, 80},
    {FREQ_A1, 80}};

const aud_single_tone_type aud_tone_key_3_data[]={ {FREQ_C1, 80},
    {FREQ_D2, 80},
    {FREQ_E2, 80}};

const aud_single_tone_type aud_tone_key_4_data[]={ {FREQ_E2, 80},
    {FREQ_G1, 40},
    {FREQ_C0, 80}};

const aud_single_tone_type aud_tone_key_5_data[]={ {988, 120}};

/*******************************************************************/
/*                         Supervisory tones                                                                */  
/********************************************************************/
const aud_dual_tone_multi_region_type aud_tone_sv_subscriber_busy_data_store[AUD_NOF_REGIONS][2]=
{
    {{0,0,500},{425,0,500}},
    {{0,0,500},{480,620,500}},
    {{0,0,500},{400,0,500}}
};

static aud_dual_tone_type aud_tone_sv_subscriber_busy_data[]=
{
    {0,0,500},
    {425,0,500}
};

const aud_dual_tone_multi_region_type aud_tone_sv_congestion_data_store[AUD_NOF_REGIONS][2]=
{
    {{0,0,200},{425,0,200}},
    {{0,0,250},{480,620,250}},
    {{0,0,200},{425,0,200}}
};

static aud_dual_tone_type aud_tone_sv_congestion_data[]=
{
    {0,0,200},
    {425,0,200}
};

const aud_single_tone_multi_region_type aud_tone_sv_radio_path_ack_data_store[AUD_NOF_REGIONS][2]=
{
    {{425,200},{0,10}},
    {{425,200},{0,10}},
    {{400,1000},{0,2000}}
};

static aud_single_tone_type aud_tone_sv_radio_path_ack_data[]=
{
    {425, 200},
    {0, 10}
};

const aud_single_tone_multi_region_type aud_tone_sv_radio_path_not_avail_data_store[AUD_NOF_REGIONS][2]=
{
    {{0,200},{425,200}},
    {{0,200},{425,200}},
    {{0,200},{425,200}}
};

static aud_single_tone_type aud_tone_sv_radio_path_not_avail_data[]=
{
    {0,   200},
    {425, 200}
};

const aud_single_tone_multi_region_type aud_tone_sv_error_info_data_store[AUD_NOF_REGIONS][4]=
{
    {{950,330},{1400,330},{1800,330},{0,1000}},
    {{950,330},{1400,330},{1800,330},{0,1000}},
    {{950,330},{1400,330},{1800,330},{0,1000}}
};

static aud_single_tone_type aud_tone_sv_error_info_data[]=
{
    {950,  330},
    {1400, 330},
    {1800, 330},
    {0,   1000}
};

const aud_single_tone_multi_region_type aud_tone_sv_call_waiting_data_store[AUD_NOF_REGIONS][8]=
{
    {{425,200},{0,600},{425,200},{0,3000},{0,10},{0,10},{0,10},{0,10}},
    {{440,300},{0,8000},{0,1700},{440,100},{0,100},{440,100},{0,8000},{0,1700}},
    {{425,200},{0,600},{425,200},{0,3000},{0,10},{0,10},{0,10},{0,10}}
};

static aud_single_tone_type aud_tone_sv_call_waiting_data[]=
{
    {425, 200},
    {0,   600},
    {425, 200},
    {0,  3000},
    {0,  10},
    {0,  10},
    {0,  10},
    {0,  10},
};

/*******************************************************************/
/*                         Info tones                                                                           */  
/********************************************************************/
const aud_single_tone_type aud_tone_info_free_tone_data[]=
{
    {425, 1000},
    {0,   4000}
};

const aud_single_tone_type aud_tone_info_connection_data[] =
{
    { FREQ_C2,      100},
    { FREQ_E2,      100},
    { FREQ_G2,      100},
    { FREQ_C2,      100},
    { FREQ_SILENT,   80}
};

const aud_single_tone_type aud_tone_info_disconnect_data[] =
{
    { FREQ_C1,       80},
    { FREQ_AS1,      80},
    { FREQ_FS1,      80},
    { FREQ_CS1,      80},
    { FREQ_SILENT, 20},
};

const aud_single_tone_type aud_tone_info_device_in_data[] =
{
    { FREQ_G2,       150},
    { FREQ_C3,        80},
    { FREQ_SILENT,    80}
};

const aud_single_tone_type aud_tone_info_device_out_data[] =
{
    { FREQ_E2,       150},
    { FREQ_C2,        80},
    { FREQ_SILENT,    80}
};

const aud_single_tone_type aud_tone_info_msg_full_data[] =
{
    { FREQ_F3,      80},
    { FREQ_SILENT, 200},  
    { FREQ_E3,      80},
    { FREQ_SILENT, 200}
};

const aud_single_tone_type aud_tone_info_ussd_data[] =
{
    { FREQ_A0 ,    100},
    { FREQ_G0 ,    100},
    { FREQ_A0 ,    100},
    { FREQ_G0 ,    100}
};

const aud_single_tone_type aud_tone_info_minutte_minder_data[] =
{
    { 1700, 90},
};

const aud_single_tone_type aud_tone_info_error_1_data[] =
{
    { 950, 240},
};

const aud_single_tone_type aud_tone_info_error_2_data[] =
{
    { 950, 240},
};

const aud_single_tone_type aud_tone_info_sms_in_call_data[] =
{
    { 550, 280},
    {   0, 450},
    {1150, 280},
};

const aud_single_tone_type aud_tone_info_broadcast_in_call_data[] =
{
    { 550, 280},
    {   0, 450},
    {1150, 280},
};

const aud_single_tone_type aud_tone_info_alarm_in_call_data[] =
{
    { 800, 280},
    {1550, 280},
    {   0, 450},
    { 800, 280},
    {1550, 280},
};

const aud_single_tone_type aud_tone_info_low_bat_in_call_data[]=
{
    {697, 60}
};

const aud_single_tone_type aud_tone_info_power_off_data[]=
{
    {750, 60}
};
const aud_single_tone_type aud_tone_info_power_on_data[]=
{
    {1700, 60}
};
const aud_single_tone_type aud_tone_info_single_beep_data[]=
{
    {941, 200},
    {  0, 200}
};
const aud_single_tone_type aud_tone_info_positive_acknowledgement_data[]=
{
    {941, 200},
    {  0, 200}
};
const aud_single_tone_type aud_tone_info_negative_acknowledgement_data[]=
{
    {988, 200},
    {  0, 200}
};
const aud_single_tone_type aud_tone_info_auto_redial_data[]=
{
    {697, 600},
    {  0, 200},
    {697, 200},
    {  0, 200}
};
const aud_single_tone_type aud_tone_info_network_attention_data[]=
{
    {697, 200},
    {  0, 200}
};


const aud_dual_tone_multi_region_type aud_tone_info_dial_tone_data_store[AUD_NOF_REGIONS][3]=
{
    {{425,0,8000}},
    {{350,440,8000}},
    {{400,0,8000}}
};

static aud_dual_tone_type aud_tone_info_dial_tone_data[]=
{
    {425,0,8000}
};


const aud_single_tone_type aud_tone_info_low_bat_data[]=
{
    {988, 200},
    {  0, 200},
    {988, 200},
    {  0,5000}
};

/*******************************************************************/
/*                         Test tones                                                                           */  
/********************************************************************/
const aud_single_tone_type aud_tone_ringing_test_data[]=
{
    {2500, 5000}
};

/*******************************************************************/
/*                         Alarm tones                                                                        */  
/********************************************************************/
const aud_single_tone_type aud_tone_alarm_1_data[]=
{
    {2500, 90},          /* (f   7    )  2500.32 Hz  ,90    mS   */
    {0   , 60},          /* (rest  !   )  0      Hz  ,60    mS   */
    {2500, 90},          /* (f   7    )  2500.32 Hz  ,90    mS   */
    {0   , 720}          /* (rest  !   )  0      Hz  ,720   mS   */
};

const aud_tone_data_type aud_tone_data[] = 
{
    {aud_dual_user_tone, (sizeof(aud_tone_DTMF_0_data) / sizeof(aud_dual_tone_type)), (void*) &aud_tone_DTMF_0_data},
    {aud_dual_user_tone, (sizeof(aud_tone_DTMF_1_data) / sizeof(aud_dual_tone_type)), (void*) &aud_tone_DTMF_1_data},
    {aud_dual_user_tone, (sizeof(aud_tone_DTMF_2_data) / sizeof(aud_dual_tone_type)), (void*) &aud_tone_DTMF_2_data},
    {aud_dual_user_tone, (sizeof(aud_tone_DTMF_3_data) / sizeof(aud_dual_tone_type)), (void*) &aud_tone_DTMF_3_data},
    {aud_dual_user_tone, (sizeof(aud_tone_DTMF_4_data) / sizeof(aud_dual_tone_type)), (void*) &aud_tone_DTMF_4_data},
    {aud_dual_user_tone, (sizeof(aud_tone_DTMF_5_data) / sizeof(aud_dual_tone_type)), (void*) &aud_tone_DTMF_5_data},
    {aud_dual_user_tone, (sizeof(aud_tone_DTMF_6_data) / sizeof(aud_dual_tone_type)), (void*) &aud_tone_DTMF_6_data},
    {aud_dual_user_tone, (sizeof(aud_tone_DTMF_7_data) / sizeof(aud_dual_tone_type)), (void*) &aud_tone_DTMF_7_data},
    {aud_dual_user_tone, (sizeof(aud_tone_DTMF_8_data) / sizeof(aud_dual_tone_type)), (void*) &aud_tone_DTMF_8_data},
    {aud_dual_user_tone, (sizeof(aud_tone_DTMF_9_data) / sizeof(aud_dual_tone_type)), (void*) &aud_tone_DTMF_9_data},
    {aud_dual_user_tone, (sizeof(aud_tone_DTMF_hash_data) / sizeof(aud_dual_tone_type)), (void*) &aud_tone_DTMF_hash_data},
    {aud_dual_user_tone, (sizeof(aud_tone_DTMF_asterix_data) / sizeof(aud_dual_tone_type)), (void*) &aud_tone_DTMF_asterix_data},

    {aud_single_user_tone, (sizeof(aud_tone_key_1_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_key_1_data},
    {aud_single_user_tone, (sizeof(aud_tone_key_2_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_key_2_data},
    {aud_single_user_tone, (sizeof(aud_tone_key_3_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_key_3_data},
    {aud_single_user_tone, (sizeof(aud_tone_key_4_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_key_4_data},
    {aud_single_user_tone, (sizeof(aud_tone_key_5_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_key_5_data},

    {aud_dual_user_tone,   (sizeof(aud_tone_sv_subscriber_busy_data) / sizeof(aud_dual_tone_type)), (void*) &aud_tone_sv_subscriber_busy_data},
    {aud_dual_user_tone,   (sizeof(aud_tone_sv_congestion_data) / sizeof(aud_dual_tone_type)), (void*) &aud_tone_sv_congestion_data},
    {aud_single_user_tone, (sizeof(aud_tone_sv_radio_path_ack_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_sv_radio_path_ack_data},
    {aud_single_user_tone, (sizeof(aud_tone_sv_radio_path_not_avail_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_sv_radio_path_not_avail_data},
    {aud_single_user_tone, (sizeof(aud_tone_sv_error_info_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_sv_error_info_data},
    {aud_single_user_tone, (sizeof(aud_tone_sv_call_waiting_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_sv_call_waiting_data},

    {aud_single_user_tone, (sizeof(aud_tone_info_free_tone_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_free_tone_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_connection_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_connection_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_disconnect_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_disconnect_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_device_in_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_device_in_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_device_out_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_device_out_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_msg_full_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_msg_full_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_ussd_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_ussd_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_minutte_minder_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_minutte_minder_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_error_1_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_error_1_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_error_2_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_error_2_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_sms_in_call_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_sms_in_call_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_broadcast_in_call_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_broadcast_in_call_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_alarm_in_call_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_alarm_in_call_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_low_bat_in_call_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_low_bat_in_call_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_power_off_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_power_off_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_power_on_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_power_on_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_single_beep_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_single_beep_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_positive_acknowledgement_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_positive_acknowledgement_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_negative_acknowledgement_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_negative_acknowledgement_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_auto_redial_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_auto_redial_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_network_attention_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_network_attention_data},
    {aud_dual_user_tone,   (sizeof(aud_tone_info_dial_tone_data) / sizeof(aud_dual_tone_type)), (void*) &aud_tone_info_dial_tone_data},
    {aud_single_user_tone, (sizeof(aud_tone_info_low_bat_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_info_low_bat_data},
    {aud_single_user_tone, (sizeof(aud_tone_ringing_test_data) / sizeof(aud_single_tone_type)), (void*) &aud_tone_ringing_test_data},
};

void AUD_TONE_int_handler( int vector)
{
    (void)vector;
    zDrv_ToneActive();
    SCU_DSPSRC3 |= CLEAR_INT;
}


/*---------------------------------------------*/
/* Code block                                                             */
/*---------------------------------------------*/
void InitToneIrq(void)
{
/*********************edit by zhouzhongyao************************/
    
   	//zDrvDsp_IntRegister(3, AUD_TONE_int_handler, INT_GSM_DSP_INT3_PRI);
   	//zDrv_InstallIsr(97, AUD_TONE_int_handler, "tone_int");
  	zDrvInt_InstallIsr(30, AUD_TONE_int_handler, "tone_int", INT_HIGHLEVEL);

   	SCU_DSPSRC3 |= ENABLE_INT;      /* enable use of DSP_INT4, Enable interrupt from DSP */
}



void AUD_tone_handle_int()
{     
#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_sm_simple_tones, 0, 0,               // LLT params 
                       'G', 0x62,                                               // LIS params 
                       aud_log_lis_display, aud_log_lis_display_nodata, aud_log_lis_trace ); // LIS params 
#endif

    HandleEarpieceTimeout(NULL);
}

void DspStartToneGenerator(void) 
{
    DSP_ASYNC_START_TONE;
}


void DspStopToneGenerator(void) 
{
    DSP_ASYNC_STOP_TONE;
}

/**********************************************************************************************************
* Function:... Find_freq_prog_word_toSM
* Parameters:. freq: Frequency of the tone to be played.
* Returns:.... Programmable word to Shared Memory, which is unique to the frequency requested.
* Description: Calculates the value to Shared memory depending upon the equation provided by the firmware. (Cosine function)
**********************************************************************************************************/
static SINT16 Find_freq_prog_word_toSM(unsigned int freq)
{

  
	int precision;
   double freq_res,x;
	int FREQ_base;

	/* implementation using the taylor sequence for cosinus. Should be good enough for 8 iterations*/

if (freq >= 4000) {
	freq =4000;
}/* the max frequency allowed is limited to 4000 Hz according to FM manual*/

switch (aud_get_next_ratesw())
   	{
   case 	aud_ratesw_8khz_mono:
		{
		FREQ_base = 8000;
   		}break;
	 case 	aud_ratesw_16khz_mono:
		{
		FREQ_base = 16000;
   		}break;
	 case 	aud_ratesw_48khz_stereo:
		{
		FREQ_base= 48000;
   		}break;
	 default:
	 	FREQ_base = 8000;
	 	break;
   	}

x = (6.2831*freq/FREQ_base);
freq_res= 0;
precision = 8;
while (precision > 0)
{
freq_res = 1 -((freq_res * x * x) /( ( 2*precision)* ( 2*precision - 1)));
precision = precision - 1;
}
freq_res =freq_res *32767;

   return (SINT16)freq_res;
}

/**********************************************************************************************************
* Function:... Find_amp_prog_word_toSM
* Parameters:. freq: Frequency of the tone to be played.
* Returns:.... -Programmable word to Shared Memory, which is unique to the frequency requested.
* Description: Calculates the value to Shared memory depending upon the equation provided by the firmware. (Sine function)
* Created:.... 14.10.2004 by (SPR - DWD)
**********************************************************************************************************/
static UINT16 Find_amp_prog_word_toSM(unsigned int freq, unsigned int amp)
{

/* implementation using the taylor sequence for sinus. Should be good enough for 8 iterations*/

   int precision;
   double amp_res,x;
	int FREQ_base;

	if (freq >= 4000) {
	freq =4000;
}/* the max frequency allowed is limited to 4000 Hz according to FM manual*/
	switch (aud_get_next_ratesw())
   	{
   case 	aud_ratesw_8khz_mono:
		{
		FREQ_base = 8000;
   		}break;
	 case 	aud_ratesw_16khz_mono:
		{
		FREQ_base = 16000;
   		}break;
	 case 	aud_ratesw_48khz_stereo:
		{
		FREQ_base= 48000;
   		}break;
	 default:
	 	FREQ_base = 8000;
	 	break;
   	}
	
	x = (6.2831*freq/FREQ_base);
	amp_res= 1;
	precision = 8;
	while (precision > 0)
	{
	 amp_res = 1 -((amp_res * x * x) /( (2 * precision)* (2 * precision + 1)));
 	precision = precision - 1;
    }
 	amp_res = x *amp_res *amp;

    return (UINT16)amp_res;
}

/* SGOLD: Below code need F12_STARTUP_V22 version of the firmware (or) newer. */
/* SGOLDlite: Below code need G14 mask. */
/**********************************************************************************************************
* Function:... Set_tone_fade_in_and_fade_out_parameters
* Parameters:. None.
* Returns:.... -None.
* Description: Calculates the values of fade in and fade out parameters to be programmed to Shared memory depending upon the 
*              duration of the tones. If the tone duration is more than 40ms, then fade in and fade out values are set to 20ms. (160 samples)
*              If the tone duration is less than 40ms, then the fade in and fade out values are fed with duration/2.
**********************************************************************************************************/
static void Set_tone_fade_in_and_fade_out_parameters(void)
{
    unsigned int fade_parameter;
	int16 samples_in_ms,faiding_duration; 

switch (aud_get_next_ratesw())
   	{
   case 	aud_ratesw_8khz_mono:
		{
		samples_in_ms = 8;
		faiding_duration = 160; /* Fad in parameter is set to 160 samples, which gives 20ms fading duration at 8 Khz sample rate.*/
   		}break;
	 case 	aud_ratesw_16khz_mono:
		{
		samples_in_ms = 16;
		faiding_duration = 320; /* Fad in parameter is set to 320 samples, which gives 20ms fading duration at 16 Khz sample rate.*/

   		}break;
	 case 	aud_ratesw_48khz_stereo:
		{
		samples_in_ms= 48;
		faiding_duration = 960; /* Fad in parameter is set to 960 samples, which gives 20ms fading duration at 48 Khz sample rate.*/

   		}break;
	 default:
	 	samples_in_ms = 8;
		faiding_duration = 160;
	 	break;
   	}

    if (duration >= 40) {
        p_l1d_sm_tone_fadin_dur->fadin_dur = faiding_duration;    /* Fad in parameter is set to 160 samples, which gives 20ms fading duration at 8 Khz sample rate.*/
        p_l1d_sm_tone_fadout_dur->fadout_dur = faiding_duration; /* Fad out parameter is set to 160 samples, which gives 20ms fading duration at 8 Khz sample rate.*/
    } else {
        fade_parameter = duration/2;
        fade_parameter *= samples_in_ms; /* 1msec contain so many samples at requested khz sample rate.*/
        p_l1d_sm_tone_fadin_dur->fadin_dur = fade_parameter;   /* Fad in parameter */
        p_l1d_sm_tone_fadout_dur->fadout_dur = fade_parameter; /* Fad out parameter*/
    }
}


/****************************************************************************************
* Function:... start_tone
* Parameters:. tone_id: ID for the tone sequence to played.
* Description: Starts the requested tone in the earpiece. Also a timer is started to
*              control duration of the tones and to secure that the next tone in the 
*              tone sequence is started.
****************************************************************************************/
static void start_tone(aud_tone_id_enum tone_id) 
{
    program_tone_generator(tone_id);   
    DspStartToneGenerator();
}

/****************************************************************************************
* Function:... stop_tone
* Description: Disable the tone generators in the DSP
****************************************************************************************/
static void stop_tone(void) 
{
    aud_glob_tone.tone_active    = FALSE;
    DspStopToneGenerator();
}

/****************************************************************************************
* Function:... next_tone
* Parameters:. tone_id: ID for the tone sequence to played.
* Description: Finds the next tone in tone sequence to be played. If it is the last 
*              tone in the sequence it is checked if the sequence should be repeated 
*              or not. If the tone sequence is finished the earpiece state machine 
*              is updated.
****************************************************************************************/
static void next_tone(aud_tone_id_enum tone_id) 
{
    unsigned int index, nof_repeats, nof_tones;

    index = tone_prio_low;
    nof_repeats = aud_glob_tone.nof_repeats;

    if (aud_glob_tone.tone_data == NULL)
        nof_tones = aud_tone_data[tone_id].nof_tones;
    else
        nof_tones = aud_glob_tone.nof_tones;

    /* check if there are more tones left */
    tone_ctrl[index].tone_number ++;
    if (tone_ctrl[index].tone_number < nof_tones) {
        program_tone_generator(tone_id);
	 /*Patch for 7374 */
      DSP_ASYNC_READ_DURATION;
      if(p_l1d_tone_data->tone_dur_out ==0) 
			   DspStartToneGenerator();
    } else {
        /* All tones have been played once - play it again ? */
        tone_ctrl[index].tone_number = 0;       
        tone_ctrl[index].repeat_number ++;
        if (tone_ctrl[index].repeat_number < nof_repeats ||
            nof_repeats == 0) {
            aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_playback_loop, 0, 0, 0, 0, NULL, NULL);
            tone_ctrl[index].tone_number = 0;
            tone_ctrl[index].elapsed_time= 0;
            program_tone_generator(tone_id);
	     /*Patch for 7374 */
	      DSP_ASYNC_READ_DURATION;
	      if(p_l1d_tone_data->tone_dur_out ==0)
				   DspStartToneGenerator();
        }
        else
        {
            aud_send_mail(E_tone_ended, aud_resource_table[aud_resource_tone_generator].caller_id, 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL);
	    /*[begin] add by wangjun 20090318*/
	    if(aud_glob_tone_callback != NULL)
	    {
	        (*aud_glob_tone_callback)(TONE_INFO_PLAY_END);
	    }
	    /*[end] add by wangjun 20090318*/
        }
    }
}

/****************************************************************************************
* Function:... program_tone_generator
* Parameters:. tone_id: ID for the tone sequence to be played.
* Description: Program the tone generator in the DSP, i.e. frequency and amplitude for
*              all tree tone generators in the DSP are set.
****************************************************************************************/
static void program_tone_generator(aud_tone_id_enum tone_id) 
{
    UINT16 freq1=0, freq2=0, freq3=0, amp1=0, amp2=0, amp3=0, index=tone_prio_low;
    int16 Freq_prog_word1=0, Freq_prog_word2=0, Freq_prog_word3=0,duration_multiplyer = 0,max_duration =0;
    UINT16 Amp_prog_word1=0,  Amp_prog_word2=0,  Amp_prog_word3=0;

		/*calculate duration*/
		/* duration is limited to max Int16, and as a consequense for 8 Khz max duration can be 8 sec, for 16 - 4 sec and for
		48 kHz - 1.3 sec. This is true untill FW changes will be introduced*/
	switch (aud_get_next_ratesw())
   	{
  	 case 	aud_ratesw_8khz_mono:
		{
		duration_multiplyer = 8;
		max_duration = 8000;
   		}break;
	 case 	aud_ratesw_16khz_mono:
		{
		duration_multiplyer=16;
		max_duration =4000;
   		}break;
	 case 	aud_ratesw_48khz_stereo:
		{
		duration_multiplyer=48;
		max_duration = 1300;
   		}break;
	 default:
		duration_multiplyer = 8;
		max_duration = 8000;
	 	break;
   	}

    if (aud_glob_tone.tone_data == NULL) {
        if (aud_tone_data[tone_id].tone_type == aud_single_user_tone) {
            duration = (((aud_single_tone_type *)aud_tone_data[tone_id].tone_data+tone_ctrl[index].tone_number)->duration);
            if (duration > max_duration) duration = max_duration;
            freq1 = ((aud_single_tone_type *)aud_tone_data[tone_id].tone_data+tone_ctrl[index].tone_number)->freq1;

            Freq_prog_word1 = Find_freq_prog_word_toSM(freq1);

            Amp_prog_word1 = Find_amp_prog_word_toSM(freq1,AUD_NORMAL_TONE_AMP); 
            DspProgramToneGenerator(Freq_prog_word1, Amp_prog_word1, AUD_NULL_FREQ, AUD_NULL_AMP, 
                                    AUD_NULL_FREQ, AUD_NULL_AMP, duration*duration_multiplyer);
        } else if (aud_tone_data[tone_id].tone_type == aud_dual_user_tone) {
            duration = (((aud_dual_tone_type *)aud_tone_data[tone_id].tone_data+tone_ctrl[index].tone_number)->duration);
             if (duration > max_duration) duration = max_duration;
            freq1 = ((aud_dual_tone_type *)aud_tone_data[tone_id].tone_data+tone_ctrl[index].tone_number)->freq1;
            freq2 = ((aud_dual_tone_type *)aud_tone_data[tone_id].tone_data+tone_ctrl[index].tone_number)->freq2;

            Freq_prog_word1 = Find_freq_prog_word_toSM(freq1);
            Freq_prog_word2 = Find_freq_prog_word_toSM(freq2);      

            Amp_prog_word1 = Find_amp_prog_word_toSM(freq1, AUD_NORMAL_TONE_AMP);
            Amp_prog_word2 = Find_amp_prog_word_toSM(freq2, AUD_NORMAL_TONE_AMP);

            DspProgramToneGenerator(Freq_prog_word1, Amp_prog_word1, Freq_prog_word2, Amp_prog_word2, 
                                    AUD_NULL_FREQ, AUD_NULL_AMP, duration*duration_multiplyer);
        } else {
            duration = (((aud_triple_tone_type *)aud_tone_data[tone_id].tone_data+tone_ctrl[index].tone_number)->duration);
            if (duration > max_duration) duration = max_duration;
            freq1 = ((aud_triple_tone_type *)aud_tone_data[tone_id].tone_data+tone_ctrl[index].tone_number)->freq1;
            freq2 = ((aud_triple_tone_type *)aud_tone_data[tone_id].tone_data+tone_ctrl[index].tone_number)->freq2;
            freq3 = ((aud_triple_tone_type *)aud_tone_data[tone_id].tone_data+tone_ctrl[index].tone_number)->freq3;

            Freq_prog_word1 = Find_freq_prog_word_toSM(freq1);
            Amp_prog_word1 = Find_amp_prog_word_toSM(freq1, AUD_NORMAL_TONE_AMP);

            Freq_prog_word2 = Find_freq_prog_word_toSM(freq2); 
            Amp_prog_word2 = Find_amp_prog_word_toSM(freq2, AUD_NORMAL_TONE_AMP);

            Freq_prog_word3 = Find_freq_prog_word_toSM(freq3); 
            Amp_prog_word3 = Find_amp_prog_word_toSM(freq3, AUD_NORMAL_TONE_AMP);


            DspProgramToneGenerator(Freq_prog_word1, Amp_prog_word1, Freq_prog_word2, Amp_prog_word2, 
                                    Freq_prog_word3, Amp_prog_word3, duration*duration_multiplyer);
        }
    } else {
        if (aud_glob_tone.tone_type == aud_single_user_tone) {
            duration = (((aud_single_user_tone_type *)aud_glob_tone.tone_data+tone_ctrl[index].tone_number)->duration);
            if (duration > max_duration) duration = max_duration;
            freq1 = ((aud_single_user_tone_type *)aud_glob_tone.tone_data+tone_ctrl[index].tone_number)->freq1;
            amp1 = ((aud_single_user_tone_type *)aud_glob_tone.tone_data+tone_ctrl[index].tone_number)->amp1;

            Freq_prog_word1= Find_freq_prog_word_toSM(freq1);
            Amp_prog_word1= Find_amp_prog_word_toSM(freq1,amp1);  

            DspProgramToneGenerator(Freq_prog_word1, Amp_prog_word1, AUD_NULL_FREQ, AUD_NULL_AMP, 
                                    AUD_NULL_FREQ, AUD_NULL_AMP, duration*duration_multiplyer);
        } else if (aud_glob_tone.tone_type == aud_dual_user_tone) {
            duration = (((aud_dual_user_tone_type *)aud_glob_tone.tone_data+tone_ctrl[index].tone_number)->duration);
            if (duration > max_duration) duration = max_duration;
            freq1 = ((aud_dual_user_tone_type *)aud_glob_tone.tone_data+tone_ctrl[index].tone_number)->freq1;
            freq2 = ((aud_dual_user_tone_type *)aud_glob_tone.tone_data+tone_ctrl[index].tone_number)->freq2;

            amp1 = ((aud_dual_user_tone_type *)aud_glob_tone.tone_data+tone_ctrl[index].tone_number)->amp1;
            amp2 = ((aud_dual_user_tone_type *)aud_glob_tone.tone_data+tone_ctrl[index].tone_number)->amp2;

            Freq_prog_word1 = Find_freq_prog_word_toSM(freq1);
            Freq_prog_word2= Find_freq_prog_word_toSM(freq2);

            Amp_prog_word1 = Find_amp_prog_word_toSM(freq1, amp1);
            Amp_prog_word2 = Find_amp_prog_word_toSM(freq2, amp2);
            DspProgramToneGenerator(Freq_prog_word1, Amp_prog_word1, Freq_prog_word2, Amp_prog_word2, 
                                    AUD_NULL_FREQ, AUD_NULL_AMP, duration*duration_multiplyer);
        } else {
            duration = (((aud_triple_user_tone_type *)aud_glob_tone.tone_data+tone_ctrl[index].tone_number)->duration);
            if (duration > max_duration) duration = max_duration;
            freq1 = ((aud_triple_user_tone_type *)aud_glob_tone.tone_data+tone_ctrl[index].tone_number)->freq1;
            freq2 = ((aud_triple_user_tone_type *)aud_glob_tone.tone_data+tone_ctrl[index].tone_number)->freq2;
            freq3 = ((aud_triple_user_tone_type *)aud_glob_tone.tone_data+tone_ctrl[index].tone_number)->freq3;
            amp1 = ((aud_triple_user_tone_type *)aud_glob_tone.tone_data+tone_ctrl[index].tone_number)->amp1;
            amp2 = ((aud_triple_user_tone_type *)aud_glob_tone.tone_data+tone_ctrl[index].tone_number)->amp2;
            amp3 = ((aud_triple_user_tone_type *)aud_glob_tone.tone_data+tone_ctrl[index].tone_number)->amp3;

            Freq_prog_word1 = Find_freq_prog_word_toSM(freq1);
            Amp_prog_word1 = Find_amp_prog_word_toSM(freq1, amp1);        

            Freq_prog_word2=  Find_freq_prog_word_toSM(freq2);
            Amp_prog_word2 = Find_amp_prog_word_toSM(freq2, amp2);

            Freq_prog_word3= Find_freq_prog_word_toSM(freq3);         
            Amp_prog_word3= Find_amp_prog_word_toSM(freq3, amp3);      

            DspProgramToneGenerator(Freq_prog_word1, Amp_prog_word1, Freq_prog_word2, Amp_prog_word2, 
                                    Freq_prog_word3, Amp_prog_word3, duration*duration_multiplyer);
        }
    }
    tone_ctrl[index].elapsed_time += duration;
}

static void DspProgramToneGenerator(UINT16 freq1, UINT16 amp1, UINT16 freq2, UINT16 amp2, 
                                    UINT16 freq3, UINT16 amp3, UINT16 duration) 
{
    p_l1d_tone_data->tone_freq_1 = freq1;
    p_l1d_tone_data->tone_freq_2 = freq2;
    p_l1d_tone_data->tone_freq_3 = freq3;

    p_l1d_tone_data->tone_amp_1  = amp1;
    p_l1d_tone_data->tone_amp_2  = amp2;
    p_l1d_tone_data->tone_amp_3  = amp3;
	if(aud_glob_tone.PlayToneInfinite ==TRUE)
	 p_l1d_tone_data->tone_dur_in  = 0xFFFF; 
	else
    p_l1d_tone_data->tone_dur_in  = duration;
    p_l1d_tone_data->tone_dur_inter = aud_global_tone_dur_inter;

    Set_tone_fade_in_and_fade_out_parameters();
} /* End of function DspProgramToneGenerator */


/****************************************************************************************
* Function:... HandleEarpieceTimeout
* Description: Handle timeout for the earpiece
****************************************************************************************/
void HandleEarpieceTimeout(void* pData) 
{
    /* Send string : at+xl1set="l1 25l" To enable this LLT trace */
#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_hisr_simple_tones, 1, 0,               // LLT params 
                       'G', 0x61,                                               // LIS params 
                       aud_log_lis_display, aud_log_lis_display_data, aud_log_lis_llt_trace, // LIS params 
                       aud_glob_tone.tone_id );
#endif
    next_tone(aud_glob_tone.tone_id);
    aud_global_tone_dur_inter = AUD_GLOBAL_TONE_DUR_INTER_DEFAULT_VALUE;
}

/****************************************************************************************
* Function:... HandleEarpieceCtrlTimeout
* Description: Handle timeout for the earpiece
****************************************************************************************/
void HandleEarpieceCtrlTimeout(void *pData) 
{
    AudToneGeneratorSM(E_timer_tone_ctrl_timeout, NULL);
}

static void SuspendTone(UINT8 slot_id) 
{
    DISABLE_CLOCK;
    aud_glob_tone.suspend[slot_id].in_use        = TRUE;
    aud_glob_tone.suspend[slot_id].tone_data     = aud_glob_tone.tone_data;
    aud_glob_tone.suspend[slot_id].nof_repeats   = aud_glob_tone.nof_repeats;
    aud_glob_tone.suspend[slot_id].nof_tones     = aud_glob_tone.nof_tones;
    aud_glob_tone.suspend[slot_id].caller_id     = aud_glob_tone.caller_id;
    aud_glob_tone.suspend[slot_id].tone_active   = aud_glob_tone.tone_active;
    aud_glob_tone.suspend[slot_id].sender        = aud_glob_tone.sender;
    aud_glob_tone.suspend[slot_id].tone_id       = aud_glob_tone.tone_id;
    aud_glob_tone.suspend[slot_id].tone_type     = aud_glob_tone.tone_type;
    aud_glob_tone.suspend[slot_id].playtime      = aud_glob_tone.playtime;
    aud_glob_tone.suspend[slot_id].elapsed_time  = tone_ctrl[tone_prio_low].elapsed_time;
    aud_glob_tone.suspend[slot_id].repeat_number = tone_ctrl[tone_prio_low].repeat_number;
    aud_glob_tone.suspend[slot_id].tone_number   = tone_ctrl[tone_prio_low].tone_number;
    ENABLE_CLOCK;
}

static UINT8 ResumeTone(UINT8 slot_id) 
{
    if (aud_glob_tone.suspend[slot_id].in_use) {
        aud_glob_tone.suspend[slot_id].in_use = FALSE;
        aud_glob_tone.tone_data   = aud_glob_tone.suspend[slot_id].tone_data;
        aud_glob_tone.nof_repeats = aud_glob_tone.suspend[slot_id].nof_repeats;
        aud_glob_tone.nof_tones   = aud_glob_tone.suspend[slot_id].nof_tones;
        aud_glob_tone.caller_id   = aud_glob_tone.suspend[slot_id].caller_id;
        aud_glob_tone.tone_active = aud_glob_tone.suspend[slot_id].tone_active;
        aud_glob_tone.sender      = aud_glob_tone.suspend[slot_id].sender;
        aud_glob_tone.tone_id     = aud_glob_tone.suspend[slot_id].tone_id;
        aud_glob_tone.tone_type   = aud_glob_tone.suspend[slot_id].tone_type;
        aud_glob_tone.playtime    = aud_glob_tone.suspend[slot_id].playtime;
        aud_glob_tone.start_pos   = 0;
        tone_ctrl[tone_prio_low].elapsed_time  = aud_glob_tone.suspend[slot_id].elapsed_time;
        tone_ctrl[tone_prio_low].repeat_number = aud_glob_tone.suspend[slot_id].repeat_number;
        tone_ctrl[tone_prio_low].tone_number   = aud_glob_tone.suspend[slot_id].tone_number;
        return(TRUE);  
    } else
        return(FALSE);  
}

/****************************************************************************************
* Function:... GetToneDuration
* Parameters:
* Returns:.... -
* Description:
* Created:.... 27.06.2000 by (PAL - DWD)
****************************************************************************************/
static UINT32 GetToneDuration(aud_tone_id_enum tone_id, void *tone_data, UINT16 tone_number,aud_tone_type_enum Tone_Type) 
{
#define M aud_tone_data[tone_id]
    UINT32 duration;

    if (tone_data == NULL)
    {
        if (M.tone_type == aud_single_user_tone)
        {
            duration = (((aud_single_tone_type *)M.tone_data+tone_number)->duration);
            if (duration > 8000) duration = 8000;
        } else if (M.tone_type == aud_dual_user_tone)
        {
            duration = (((aud_dual_tone_type *)M.tone_data+tone_number)->duration);
            if (duration > 8000) duration = 8000;
        } else
        {
            duration = (((aud_triple_tone_type *)M.tone_data+tone_number)->duration);
            if (duration > 8000) duration = 8000;
        }
    } else
    { 
        if (Tone_Type == aud_single_user_tone)
        {
            duration = (((aud_single_user_tone_type *)tone_data+tone_number)->duration);
            if (duration > 8000) duration = 8000;
        } else if (Tone_Type == aud_dual_user_tone)
        {
            duration = (((aud_dual_user_tone_type *)tone_data+tone_number)->duration);
            if (duration > 8000) duration = 8000;
        } else
        {
            duration = (((aud_triple_user_tone_type *)tone_data+tone_number)->duration);
            if (duration > 8000) duration = 8000;
        }
    }  
#undef M
    return(duration);
}

/****************************************************************************************
* Function:... GetTotalPlaytimeForTone
* Parameters:
* Returns:.... -
* Description:
* Created:.... 27.06.2000 by (PAL - DWD)
****************************************************************************************/
static UINT32 GetTotalPlaytimeForTone(aud_tone_id_enum tone_id, void *tone_data, UINT16 nof_tones,aud_tone_type_enum Tone_Type) 
{
    UINT32 len;
    UINT16 i, tones;

    len = 0;  

    if (tone_data == NULL)
        tones = aud_tone_data[tone_id].nof_tones;
    else
        tones = nof_tones;

    for (i=0; i<tones; i++)
    {
        len = len + GetToneDuration(tone_id, tone_data, i, Tone_Type);
    }
    return(len);
}

/****************************************************************************************
* Function:... SetPlayPositionForTone
* Parameters:
* Returns:.... -
* Description:
* Created:.... 27.06.2000 by (PAL - DWD)
****************************************************************************************/
static void SetPlayPositionForTone(aud_tone_id_enum tone_id, void *tone_data, UINT32 pos,aud_tone_type_enum Tone_Type) 
{
    UINT32 len, duration;
    UINT16 i;
    UINT16 index;

    len = 0;
    i   = 1;
    if (pos < aud_glob_tone.playtime)
    {
        duration = GetToneDuration(tone_id, tone_data, 0,Tone_Type);
        len = len + duration;
        while (len < pos)
        {
            duration = GetToneDuration(tone_id, tone_data, i,Tone_Type);
            i ++;
            len = len + duration;
        }
        /* if (aud_glob_key.key_tone_active)
           index = tone_prio_high;
         elseUTP138224*/
        index = tone_prio_low;

        DISABLE_CLOCK;
        if (aud_get_SM_state(aud_resource_tone_generator) == S_idle)
        {
            tone_ctrl[index].tone_number  = i;  
            tone_ctrl[index].elapsed_time = len;
        } else
        {
            tone_ctrl[index].tone_number  = i - 1;  
            tone_ctrl[index].elapsed_time = len - duration;
        }
        ENABLE_CLOCK;
    } else
    {
        aud_glob_tone.start_pos = 0;
        aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_unknown_position,
                                0, 0, 0, 0, NULL, NULL);
    }
    /* The new play position will take place from next tone timeout */
}

/**********************************************************************************************************
* Function:... GetPlayPositionForTone
* Parameters:. 
* Returns:.... 
* Description: 
* Created:.... 
**********************************************************************************************************/
static UINT32 GetPlayPositionForTone(void) 
{
    return(tone_ctrl[tone_prio_low].elapsed_time);
}

/****************************************************************************************
* Function:... AUD_Update_supervisory_tone_for_region(void)
* Parameters:  AUD_region from static eep.
* Description: AUD_region is read from static eep and based on this info the correct tones are loaded in the array used for playback.
*              All regions are stored in audio driver. If wrong region is selected - default is kept.
****************************************************************************************/
void AUD_Update_supervisory_tone_for_region(eep_aud_region_type AUD_region)
{
    if (AUD_region<EEP_AUD_REGION_END) {
        memcpy(aud_tone_sv_subscriber_busy_data, &aud_tone_sv_subscriber_busy_data_store[AUD_region][0],sizeof(aud_tone_sv_subscriber_busy_data));
        memcpy(aud_tone_sv_congestion_data, &aud_tone_sv_congestion_data_store[AUD_region][0],sizeof(aud_tone_sv_congestion_data));
        memcpy(aud_tone_info_dial_tone_data, &aud_tone_info_dial_tone_data_store[AUD_region][0],sizeof(aud_tone_info_dial_tone_data));
        memcpy(aud_tone_sv_radio_path_ack_data, &aud_tone_sv_radio_path_ack_data_store[AUD_region][0],sizeof(aud_tone_sv_radio_path_ack_data));
        memcpy(aud_tone_sv_radio_path_not_avail_data, &aud_tone_sv_radio_path_not_avail_data_store[AUD_region][0],sizeof(aud_tone_sv_radio_path_not_avail_data));
        memcpy(aud_tone_sv_error_info_data, &aud_tone_sv_error_info_data_store[AUD_region][0],sizeof(aud_tone_sv_error_info_data));
        memcpy(aud_tone_sv_call_waiting_data, &aud_tone_sv_call_waiting_data_store[AUD_region][0],sizeof(aud_tone_sv_call_waiting_data));
    } else {
        /* eep programmed wrongly - send LLT and keep default. Reused llt_type - not nice !!!!*/
        /* Send string : at+xl1set="l1 25l" To enable this LLT trace */
#ifdef AUDIO_LLT
        aud_common_trace( llt_group_audio, llt_type_audio_sm_simple_tones, 2, 0,               // LLT params 
                           'G', 0x61,                                               // LIS params 
                           aud_log_lis_display, aud_log_lis_display_data, aud_log_llt_trace, // LIS params 
                           (UINT32) AUD_region, 
                           (UINT32) E_audio_llt_type_id_0 );
#endif
        AUD_ERROR_REPORTING(aud_error_grade2, TRAP_AUD_ERROR, "Wrong region: %d",AUD_region);
    }
}

/****************************************************************************************
* Function:... AudToneGeneratorSM
* Parameters:. UINT8 event, void* pData
* Description: The Tone generator state machine
****************************************************************************************/
BOOL  AudToneGeneratorSM(aud_event_enum event, void* pData)
{
    UINT8 *Playtime_tone_data ;
    aud_tone_id_enum Playtime_tone_id;
    UINT16 Playtime_nof_tones ;
    aud_tone_type_enum Playtime_tone_type ;

    disp_to_SM_data_type *SM_data = (disp_to_SM_data_type*)pData; /* contains the structure passed from the dispatcher */
    BOOL func_ret = TRUE;

    /* Send string : at+xl1set="l1 2l" To enable this LLT trace */
#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_sm_simple_tones, 11, 0,               // LLT params 
                       'G', 0x63,                                               // LIS params 
                       aud_log_lis_raw_data, aud_log_lis_display_data, aud_log_lis_llt_trace, // LIS params 
                       (UINT32) aud_get_SM_state(aud_resource_tone_generator),
                       (UINT32) event,       
                       (UINT32) SM_data->parm1,
                       (UINT32) SM_data->parm2,
                       (UINT32) SM_data->parm3,
                       (UINT32) SM_data->parm4,
                       (UINT32) SM_data->parm5,
                       (UINT32) SM_data->parm6,
                       (UINT32) SM_data->parm7,
                       (UINT32) SM_data->parm8,
                       (UINT32) E_audio_llt_type_id_1 );
#endif
    switch (aud_get_SM_state(aud_resource_tone_generator)) {
        /*-------------------------------------------------------------------*/
        /*                           Idle state                              */
        /*-------------------------------------------------------------------*/
        case S_idle:
            switch (event) {
                case E_tone_start:
                case E_tone_start_user_tone:
                    if (event == E_tone_start) {
                        aud_glob_tone.tone_data     = NULL;
                        aud_glob_tone.tone_id       = (aud_tone_id_enum)((*SM_data).parm1);
                        aud_glob_tone.nof_repeats   = (UINT16)((*SM_data).parm2);
                        aud_glob_tone.sender        = (*SM_data).sender;
		          /*[begin] add by wangjun 20090318*/
                        aud_glob_tone_callback = (T_Tone_CallbackFunc)((*SM_data).parm8);
			  /*[end] add by wangjun 20090318*/
                    } else  { /* (event == E_tone_start_user_tone) */
                        aud_glob_tone.tone_data   = (UINT8*) SM_data->parm1;
                        aud_glob_tone.tone_type   = (aud_tone_type_enum)((*SM_data).parm2);
                        aud_glob_tone.nof_tones   = (UINT16)((*SM_data).parm3);
                        aud_glob_tone.nof_repeats = (UINT16)((*SM_data).parm4);
                        aud_glob_tone.sender      = (*SM_data).sender;
		         /*[begin] add by wangjun 20090318*/
                        aud_glob_tone_callback = (T_Tone_CallbackFunc)((*SM_data).parm8);
			  /*[end] add by wangjun 20090318*/
                        aud_glob_tone.tone_id = aud_tone_id_end;
                    }

                    aud_glob_tone.playtime = GetTotalPlaytimeForTone(aud_glob_tone.tone_id, aud_glob_tone.tone_data, 
                                                                     aud_glob_tone.nof_tones,aud_glob_tone.tone_type);
                    tone_ctrl[tone_prio_low].repeat_number = 
                    tone_ctrl[tone_prio_low].elapsed_time  = 
                    tone_ctrl[tone_prio_low].tone_number   = 0;
                    if (aud_glob_tone.start_pos != 0)
                        SetPlayPositionForTone(aud_glob_tone.tone_id, aud_glob_tone.tone_data, aud_glob_tone.start_pos
                                               ,aud_glob_tone.tone_type);
                    aud_glob_tone.tone_active   = TRUE;
                    aud_enable_path_and_resource(aud_resource_tone_generator,S_active,TRUE);
                    aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_playback_started,
                                            0, 0, 0, 0, NULL, NULL);
                    start_tone(aud_glob_tone.tone_id);
                    break;

                case E_tone_release_stop:
                case E_tone_stop:
                case E_tone_force_stop:
                    aud_glob_tone.tone_active    = FALSE;
		      aud_glob_tone.PlayToneInfinite=FALSE;
                    /* Ignore, no tones active */
                    break;

                case E_tone_get_play_position:
                case E_tone_suspend:
                    if (event == E_tone_get_play_position)
                        aud_glob_tone.sender = (*SM_data).sender;
                    aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_no_playback,
                                            0, 0, 0, 0, NULL, NULL);
                    break;

                case E_tone_set_play_position:
                    aud_glob_tone.start_pos = (UINT32)((*SM_data).parm1);
                    aud_glob_tone.sender    = (*SM_data).sender;
                    break;

                case E_tone_resume:
                    aud_glob_tone.slot_id = (UINT8)((*SM_data).parm1);
                    aud_glob_tone.sender  = (*SM_data).sender;

                    if (ResumeTone(aud_glob_tone.slot_id)) {
                        aud_glob_tone.tone_active   = TRUE;
                        aud_enable_path_and_resource(aud_resource_tone_generator,S_active,TRUE);
                        aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_playback_started,
                                                0, 0, 0, 0, NULL, NULL);
                        start_tone(aud_glob_tone.tone_id);
                    } else
                        aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_suspend_resume_error,
                                                0, 0, 0, 0, NULL, NULL);
                    break;

                case E_tone_get_total_playtime:
                    aud_glob_tone.tone_data = (UINT8*) SM_data->parm8;
                    aud_glob_tone.tone_id   = (aud_tone_id_enum)((*SM_data).parm1);
                    aud_glob_tone.nof_tones = (UINT16)((*SM_data).parm2);
                    aud_glob_tone.tone_type = (aud_tone_type_enum)((*SM_data).parm3);
                    aud_glob_tone.sender    = (*SM_data).sender;

                    aud_glob_tone.playtime = GetTotalPlaytimeForTone(aud_glob_tone.tone_id, aud_glob_tone.tone_data, 
                                                                     aud_glob_tone.nof_tones,aud_glob_tone.tone_type);
                    aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_total_playtime,
                                            aud_glob_tone.playtime, 0, 0, 0, NULL, NULL);
                    break;

                case E_tone_ended:
                case E_timer_tone_ctrl_timeout:
                    AUD_ERROR_REPORTING(aud_error_grade2, TRAP_AUD_ERROR, "Unexpected event: %d in Tone SM in idle state", event);
                    break;
                default:
                    func_ret = FALSE;
                    break;
            }
            break;

            /*-------------------------------------------------------------------*/
            /*                              Active state                         */
            /*-------------------------------------------------------------------*/
        case S_active:
            switch (event) {
                case E_tone_start:
                case E_tone_start_user_tone:
                case E_tone_resume:
                    aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_request_error,
                                            0, 0, 0, 0, NULL, NULL);
                    break;

                case E_timer_tone_ctrl_timeout:
                    tone_ctrl[tone_prio_low].repeat_number = 0;
                    tone_ctrl[tone_prio_low].tone_number   = 0;
                    aud_glob_tone.playtime = GetTotalPlaytimeForTone(aud_glob_tone.tone_id, aud_glob_tone.tone_data, 
                                                                     aud_glob_tone.nof_tones,aud_glob_tone.tone_type);
                    if (aud_glob_tone.start_pos != 0)
                        SetPlayPositionForTone(aud_glob_tone.tone_id, aud_glob_tone.tone_data, aud_glob_tone.start_pos
                                               ,aud_glob_tone.tone_type);
                    start_tone(aud_glob_tone.tone_id);
                    break;

                case E_tone_get_total_playtime:
                    Playtime_tone_data = (UINT8*) SM_data->parm8;
                    Playtime_tone_id   = (aud_tone_id_enum)((*SM_data).parm1);
                    Playtime_nof_tones = (UINT16)((*SM_data).parm2);
                    Playtime_tone_type = (aud_tone_type_enum)((*SM_data).parm3);
                    //for active tones.
                    if (Playtime_tone_data==NULL)
                    {
                        if (Playtime_tone_id==aud_glob_tone.tone_id)//request for the currently active predefined tone.
                        {
                            aud_glob_tone.sender    = (*SM_data).sender;                          
                            aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_total_playtime,
                                                    aud_glob_tone.playtime, 0, 0, 0, NULL, NULL);
                        } else
                        {
                            aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_total_playtime,
                                                    GetTotalPlaytimeForTone(Playtime_tone_id, Playtime_tone_data, 
                                                                            Playtime_nof_tones,Playtime_tone_type), 0, 0, 0, NULL, NULL);
                        }
                    } else if (Playtime_tone_data!=NULL)//active user define tone.
                    {
                        if (Playtime_tone_data ==aud_glob_tone.tone_data)
                            //currently active user defined tone.
                        {
                            aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_total_playtime,
                                                    aud_glob_tone.playtime, 0, 0, 0, NULL, NULL);      
                        } else
                        {
                            aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_total_playtime,
                                                    GetTotalPlaytimeForTone(Playtime_tone_id, Playtime_tone_data, 
                                                                            Playtime_nof_tones,Playtime_tone_type), 0, 0, 0, NULL, NULL);
                        }
                    }
                    break;

                case E_tone_suspend:
                    aud_glob_tone.slot_id = (UINT8)((*SM_data).parm1);
                    aud_glob_tone.sender  =(*SM_data).sender;
                    SuspendTone(aud_glob_tone.slot_id);
                    aud_glob_tone.tone_active    = FALSE;
                    stop_tone();
                    aud_disable_path_and_resource(aud_resource_tone_generator,S_idle,TRUE); 
			   #ifdef IFWD_AUD_NEW_RC_PARAMETER		
                    aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_playback_suspended, GetPlayPositionForTone(), 0, 0, 0, NULL, NULL);
			   #else
			   aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_playback_suspended, 0, 0, 0, 0, NULL, NULL);
			  #endif   
                    break;

                case E_tone_get_play_position:
                    aud_glob_tone.sender = (*SM_data).sender;
                    aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_elapsed_time,
                                            GetPlayPositionForTone(), 0, 0, 0, NULL, NULL);
                    break;

                case E_tone_set_play_position:
                    aud_glob_tone.start_pos = (UINT32)((*SM_data).parm1);
                    aud_glob_tone.sender    = (*SM_data).sender;
                    SetPlayPositionForTone(aud_glob_tone.tone_id, aud_glob_tone.tone_data, aud_glob_tone.start_pos
                                           ,aud_glob_tone.tone_type);
                    break;
   #ifdef IFWD_AUD_NEW_RC_PARAMETER
                case E_tone_release_stop:
                case E_tone_stop:
                case E_tone_force_stop:
			aud_glob_tone.PlayToneInfinite=FALSE;
                    if (E_tone_force_stop == event)
                        aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_playback_stopped,GetPlayPositionForTone(), 0, 0,0, NULL, NULL);
                    else
                        aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_playback_stopped,GetPlayPositionForTone(), 0, 0,0, NULL, NULL);
                    stop_tone();
                    aud_disable_path_and_resource(aud_resource_tone_generator,S_idle,TRUE);
                    break;
#else
case E_tone_release_stop:
                case E_tone_stop:
                case E_tone_force_stop:
			aud_glob_tone.PlayToneInfinite=FALSE;
                    if (E_tone_force_stop == event)
                        aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_playback_stopped, 0, 0, 0, AUD_STOPPED_BY_DRV, NULL, NULL);
                    else
                        aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_playback_stopped, 0, 0, 0, AUD_STOPPED_BY_APP, NULL, NULL);
                    stop_tone();
                    aud_disable_path_and_resource(aud_resource_tone_generator,S_idle,TRUE);
                    break;
#endif
					
                case E_tone_ended:
					  #ifdef IFWD_AUD_NEW_RC_PARAMETER
                    aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_playback_finish,  GetPlayPositionForTone(), 0, 0, 0, NULL, NULL);
				#else
			aud_send_response_signal(aud_glob_tone.sender, aud_resource_tone_generator, aud_rc_playback_finish, 0, 0, 0, 0, NULL, NULL);
			#endif
                    stop_tone();
                    aud_disable_path_and_resource(aud_resource_tone_generator,S_idle,TRUE);
                    break;
                default:
                    func_ret = FALSE;
                    break;
            }
    }
    return( func_ret );
}

 void Initialise_tone_vars()
{
	aud_glob_tone.PlayToneInfinite =FALSE; 
}

