/**
 *       Copyright (C) Danish Wireless Design A/S. All rights reserved.
 *
 * This document contains proprietary information belonging to Danish Wireless
 * Design A/S. Passing on and copying of this document, use and communication
 * of its contents is not permitted without prior written authorisation.                                                                                                                                   
 *
 * Description:
 *   State machine for the voice memo service (vms). Everything concerning the vms is
 *              controlled from this code.
 *
 * Revision Information:
 *   File name: \dwddrv\AUD\src\aud_vms_playback.c
 *   Version: \main\79
 *   Date: 2007-10-10 13:32:54
 */
#ifdef VMS_PRESENT
/*---------------------------------------------
 Include files.                              
---------------------------------------------*/
//#include "string.h"
//#include "icu_funct.h"
//#include "scttypes.h"
//#include "ms.h"
#include "bastypes.h"
//#include "dsp.h"                /* for DSP command types */
#include "aud_drv.h"
#include "aud_data.h"           /* data describing the paths */
#include "aud_com.h"
//#include "dsp.h"

/*changed by wangjun for NV change.[20090214]*/
#include "aud_nv.h"  //#include "eep.h"

#include "aud_path_control.h"
#include "aud_vms_playback.h"
#include "aud_ext_def.h"
#include "drvs_teak.h"
//#include "trap.h"

/*---------------------------------------------*/
/* Struct's.                                   */
/*---------------------------------------------*/
const UINT8 aud_vms_null_array[]=
{0x7c, 0, 0, 0, 0,
    0, 0, 0, 0, 0,
    0, 0, 0, 0, 0,
    0, 0, 0, 0, 0,
    0, 0, 0, 0, 0,
    0, 0, 0, 0, 0,
    0, 0, 0, 0, 0};

/*---------------------------------------------*/
/* Global data.                                */
/*---------------------------------------------*/
aud_glob_vms_playback_type  aud_glob_vms_playback;
aud_vms_total_playtime_type AUD_vms_total_playtime_in_general;

/*---------------------------------------------*/
/* Internal functions.                         */
/*---------------------------------------------*/
static void stop_voice_memo_playback(void);
static void play_memo(void);
static void copy_playback_buffer(void);
static UINT16 CalcNofBytesToRead(UINT32 bytes_read);
static void vms_event_handle(aud_event_enum event, void *pData);
static void check_frame_number(void);
static void find_nof_bytes_per_frame(UINT8 header, UINT8* bytes_per_frame);
static void DspStartVmPlayback(aud_dsp_format_enum codec_type);
void DspStopPlaybackVm(void);
static void SaveVmSuspendData(UINT8 slot_id);
static void RestoreVmInternal_SuspendData(UINT8 slot_id);
static void RestoreVmExternal_SuspendData(UINT8 slot_id);
static void StopVmSuspendData(UINT8 slot_id);
static void  GetVoiceMemoPlayPosition(void);

static UINT32 CalcTotalPlayTimeForRamFileForVM(UINT16 DWD_HUGE* file_handle, UINT32 start_offset, UINT8* bytes_per_frame, UINT16 buffer_size);
static UINT32 CalcTotalPlayTimeForFfsFileForVM(UINT16 DWD_HUGE* input_file_handle, UINT32 start_offset, UINT32 file_size, UINT8* bytes_per_frame);
static UINT32 CalcTotalPlayTimeForMMCFileForVM(UINT16 DWD_HUGE* input_file_handle, UINT32 start_offset, UINT32 file_size, UINT8* bytes_per_frame);

static UINT32 SetPositionForRamFile(UINT32 pos);
static UINT32 SetPositionForFfsFile(UINT32 pos);
static void SetPositionForMMFFile(UINT32 pos);
static UINT32 SetPositionForMMCFile(UINT32 pos);
static UINT16 DWD_HUGE *mmf_null_handle_function(void);
static UINT16 DWD_HUGE *vms_streaming_buffer_function(void);
static UINT8 find_header_info_playback(UINT8 *pData);
static void headerstrncpy(SINT8 *p_dest, huge UINT8 *p_source, UINT16 num);
static void vm_send_response_signal( SINT8 rc, UINT32 value);
static void vms_intialize_playback(void);
static void InitVms_Repeat(void);
static void vms_setposition_ssal_callback(SINT16 result
#ifdef FFS_EXTENDED_CALLBACK
                                          ,void *input_ptr
#endif
                                         );
static void vms_read_ssal_callback(SINT16 result
#ifdef FFS_EXTENDED_CALLBACK
                                   ,void *input_ptr
#endif
                                  );
static void AUD_set_internal_vms_pb_state(UINT8 new_state);
static void disable_vms_audio_path(void);
static void enable_vms_audio_path(void) ;

#ifdef VM_MMF_DATA_VERIFY
static void vm_data_verify_init();
void vms_write_ssal_callback_ver_vm(SINT16 result
#ifdef FFS_EXTENDED_CALLBACK
    ,void *input
#endif
);
void write_vm_data_to_ffs_file(UINT8 buffer_selector, UINT16 size);
#endif
static void aud_vms_playback_ram_readrequest(UINT8 * p_dest, UINT32 buf_size);
static void aud_vms_playback_ram_read(UINT8 *p_dest, UINT32 buf_size);
static void aud_vms_playback_ram_init_datacopy(void);

/*---------------------------------------------*/
/* Internal data.                              */
/*---------------------------------------------*/
static UINT8 DWD_HUGE         aud_vms_temp_buf[34];
static aud_vms_data_type      aud_vms_data_array[2];
static vms_playback_ctrl_type vms_pb_ctrl;

#define AUD_VMS_SIZE_OF_TOTAL_PLAYTIME_BUFFER  2048
static UINT8   aud_vms_total_playtime_buffer[AUD_VMS_SIZE_OF_TOTAL_PLAYTIME_BUFFER];


#ifdef VM_MMF_DATA_VERIFY
VM_streaming validate_vm[2];    
UINT16 vm_next_location=0;
UINT8 vm_write_index = 0;
UINT16 valid_data_size = 0;
#endif

/****************************************************************************************
* Function:... AudVmsPlaybackSM
* Parameters:. event: The reason why the function was called
* Description: State machine for the voice memo playback service control
****************************************************************************************/
BOOL AudVmsPlaybackSM(aud_event_enum event, void *pData)
{
    const aud_resource_enum me = aud_resource_playback_vm;
    disp_to_SM_data_type *SM_data = (disp_to_SM_data_type*)pData; //contains the structure passed from the dispatcher
    BOOL func_ret = TRUE;

#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_sm_vm_playback, 10, 0,               // LLT params 
                       'm', 0x17,                                               // LIS params 
                       aud_log_lis_raw_data, aud_log_lis_display_data, aud_log_lis_llt_trace, // LIS params 
                     (UINT32) vms_pb_ctrl.aud_vms_state,
                     (UINT32) event,
                     (UINT32) aud_get_SM_state(me),
                     (UINT32) SM_data->parm1,
                     (UINT32) SM_data->parm2,
                     (UINT32) SM_data->parm3,
                     (UINT32) SM_data->parm4,
                     (UINT32) SM_data->parm5,
                     (UINT32) SM_data->parm6,
                     (UINT32) SM_data->parm7 );
#endif
    switch (aud_get_SM_state(me)) 
    {
        case S_idle:
            {
                switch (event) 
                {
#ifdef VM_MMF_DATA_VERIFY   
                    case E_vm_write_to_ffs:            
                        write_vm_data_to_ffs_file(SM_data->parm1,SM_data->parm2);            
                        break;
#endif
                    case E_vms_resume:
                        aud_disable_standby(me);
                        aud_glob_vms_playback.sender         = (*SM_data).sender;
                        aud_glob_vms_playback.slot_id = (UINT16)((*SM_data).parm1);            

                        if (aud_glob_vms_playback.suspend_data[aud_glob_vms_playback.slot_id].in_use) {
                            InitVmsPlayback();
                            RestoreVmExternal_SuspendData(aud_glob_vms_playback.slot_id);  
                            RestoreVmInternal_SuspendData(aud_glob_vms_playback.slot_id);             
                            aud_glob_vms_playback.pos = aud_glob_vms_playback.suspend_data[aud_glob_vms_playback.slot_id].frame_number * 20;

                            switch (aud_glob_vms_playback.media_type) {
                                case aud_media_ram:                   
                                    vms_pb_ctrl.playback_frame =  0;
                                    SetPositionForRamFile(aud_glob_vms_playback.pos);
                                    aud_glob_vms_playback.pos = 0;           
                                    vms_intialize_playback();
                                    break;
                                case aud_media_mmc:   
#ifdef  AUD_NANDDEVICE_SUPPORT                                	
                                case aud_media_fs_cdrive:
                                case aud_media_fs_edrive:
                                case aud_media_fs_fdrive:		
#endif                                 	
                                    SetPositionForMMCFile(aud_glob_vms_playback.pos);
                                    break;
                                case aud_media_ffs:
                                    SetPositionForFfsFile(aud_glob_vms_playback.pos);
                                    break;
                                case aud_media_mmf:
                                    vms_pb_ctrl.check_header_flag = FALSE;
                                    aud_glob_vms_playback.pos = 0;           
                                    vms_intialize_playback();
                                    break;
                            }
                        } else {
                            vm_send_response_signal(aud_rc_suspend_resume_error,0);           
                        }      
                        break;

                    case E_vms_start_playback:
                        aud_disable_standby(me);
                        aud_glob_vms_playback.sender          = (*SM_data).sender;
                        aud_glob_vms_playback.vm_mode         = (aud_vm_mode_enum)((*SM_data).parm1);
                        aud_glob_vms_playback.media_type      = (aud_media_enum)((*SM_data).parm2);
                        aud_glob_vms_playback.format          = (aud_dsp_format_enum)((*SM_data).parm3);
                        aud_glob_vms_playback.file_handle     = (UINT16 DWD_HUGE*)(SM_data->parm8);
                        aud_glob_vms_playback.buffer_size     = (UINT32)((*SM_data).parm4);
                        aud_glob_vms_playback.ssal_media_type = aud_media_type_to_ssal_type(aud_glob_vms_playback.media_type);

                        switch (aud_glob_vms_playback.media_type) {
                            case aud_media_ram:
                                aud_glob_vms_playback.fp_voice_memo_getpacket = &voice_memo_getpacket_in_ram;
                                vms_pb_ctrl.data_pagememory = FALSE;
                                aud_check_pageaddr_copy((UINT8 DWD_HUGE *)aud_glob_vms_playback.file_handle, aud_glob_vms_playback.buffer_size,&vms_pb_ctrl.data_pagememory);
                                break;
                            case aud_media_mmf:
                            case aud_media_mmf_test:
                                aud_glob_vms_playback.fp_voice_memo_getpacket = &voice_memo_getpacket_in_mmf;
                                break;
                            case aud_media_ffs:
                            case aud_media_mmc:
#ifdef  AUD_NANDDEVICE_SUPPORT                            	
                            case aud_media_fs_cdrive:
                            case aud_media_fs_edrive:
                            case aud_media_fs_fdrive:
#endif                             	
                                aud_glob_vms_playback.fp_voice_memo_getpacket = &voice_memo_getpacket_in_ffs;
                                break;
                        }            
                        aud_glob_vms_playback.nof_repeats = (UINT16)((*SM_data).parm5);
                        aud_glob_vms_playback.start_offset = (UINT32)((*SM_data).parm6);

#ifdef VM_MMF_DATA_VERIFY
                        vm_data_verify_init();
#endif
                        InitVmsPlayback();
                        if (aud_glob_vms_playback.media_type != aud_media_mmf) {
                            switch (aud_glob_vms_playback.media_type) {
                                case aud_media_ffs:
                                    vms_pb_ctrl.vms_total_playtime = CalcTotalPlayTimeForFfsFileForVM(aud_glob_vms_playback.file_handle, aud_glob_vms_playback.start_offset, vms_pb_ctrl.vms_file_size, NULL);
                                    break;
                                case aud_media_mmc:                                 
#ifdef  AUD_NANDDEVICE_SUPPORT                                	
                                case aud_media_fs_cdrive:
                                case aud_media_fs_edrive:
                                case aud_media_fs_fdrive:	
#endif                                 	
                                    vms_pb_ctrl.vms_total_playtime = CalcTotalPlayTimeForMMCFileForVM(aud_glob_vms_playback.file_handle, aud_glob_vms_playback.start_offset, vms_pb_ctrl.vms_file_size, NULL);
                                    break;
                                case aud_media_ram:
                                    vms_pb_ctrl.vms_total_playtime = CalcTotalPlayTimeForRamFileForVM(aud_glob_vms_playback.file_handle, aud_glob_vms_playback.start_offset, NULL, aud_glob_vms_playback.buffer_size);
                                    break;
                            }
                        }
                        if (aud_glob_vms_playback.pos > 0) {
                            switch (aud_glob_vms_playback.media_type) {
                                case aud_media_ram:                   
                                    SetPositionForRamFile(aud_glob_vms_playback.pos);
                                    vms_intialize_playback();                  
                                    break;
                                case aud_media_ffs:
                                    SetPositionForFfsFile(aud_glob_vms_playback.pos);
                                    break;
                                case aud_media_mmc:   
#ifdef  AUD_NANDDEVICE_SUPPORT                                	
                                case aud_media_fs_cdrive:
                                case aud_media_fs_edrive:
                                case aud_media_fs_fdrive:	
#endif                                 	
                                    SetPositionForMMCFile(aud_glob_vms_playback.pos);
                                    break;
                                case aud_media_mmf:
                                    SetPositionForMMFFile(vms_pb_ctrl.vms_bytes_played);
                                    vms_intialize_playback();                  
                                    break;
                            }
                        } else
                            vms_intialize_playback();                  
                        aud_glob_vms_playback.pos = 0;        
                        vms_pb_ctrl.lNof_repeatfactor = aud_glob_vms_playback.nof_repeats;              
                        break;

                    case E_vms_repeatevent:
                        vms_intialize_playback();                  
                        break;

                    case E_vms_get_total_playtime:
                        aud_glob_vms_playback.sender         = (*SM_data).sender;
                        aud_glob_vms_playback.media_type     = (aud_media_enum)((*SM_data).parm1);
                        aud_glob_vms_playback.format         = (aud_dsp_format_enum)((*SM_data).parm2);
                        aud_glob_vms_playback.file_handle    = (UINT16*)(SM_data->parm8);
                        aud_glob_vms_playback.buffer_size    = (UINT32)((*SM_data).parm3);
                        aud_glob_vms_playback.start_offset   = (UINT32)((*SM_data).parm4);
                        switch (aud_glob_vms_playback.media_type) {
                            case aud_media_mmf:
                                vm_send_response_signal(aud_rc_request_error, 0);            
                                break;
                            case aud_media_ffs:
                                vms_pb_ctrl.vms_total_playtime = CalcTotalPlayTimeForFfsFileForVM(aud_glob_vms_playback.file_handle, aud_glob_vms_playback.start_offset, vms_pb_ctrl.vms_file_size, NULL);
                                vm_send_response_signal( aud_rc_total_playtime, vms_pb_ctrl.vms_total_playtime);
                                break;
                            case aud_media_mmc: 
#ifdef  AUD_NANDDEVICE_SUPPORT                            	
                            case aud_media_fs_cdrive:
                            case aud_media_fs_edrive:
                            case aud_media_fs_fdrive:		
#endif                             	
                                vms_pb_ctrl.vms_total_playtime = CalcTotalPlayTimeForMMCFileForVM(aud_glob_vms_playback.file_handle, aud_glob_vms_playback.start_offset, vms_pb_ctrl.vms_file_size, NULL);
                                vm_send_response_signal( aud_rc_total_playtime, vms_pb_ctrl.vms_total_playtime);
                                break;
                            case aud_media_ram:
                                vms_pb_ctrl.vms_total_playtime = CalcTotalPlayTimeForRamFileForVM(aud_glob_vms_playback.file_handle, aud_glob_vms_playback.start_offset, NULL, aud_glob_vms_playback.buffer_size);
                                vm_send_response_signal(aud_rc_total_playtime, vms_pb_ctrl.vms_total_playtime);
                                break;
                        }
                        break;
                    case E_timer_vms_playback_timeout:
                        AUD_ERROR_REPORTING(aud_error_grade2, TRAP_AUD_ERROR, "Unwanted event: %d in VM Playback SM in Idle  state ", event)
                        break;
                    case E_vms_suspend:              
                    case E_vms_get_play_position:
                        vm_send_response_signal( aud_rc_request_error, 0);    
                        break;

                    case E_vms_set_play_position:
                        aud_glob_vms_playback.sender  = (*SM_data).sender;
                        aud_glob_vms_playback.pos     = (UINT32)((*SM_data).parm1);          
                        vm_send_response_signal( aud_rc_ok, 0);    
                        break;

                    case E_vms_pb_set_getbuffer_function :
                        aud_glob_vms_playback.get_buffer_func   = (ptr_GetAudioBufferFunction )SM_data->parm7;
                        aud_glob_vms_playback.mmf_user_data     =  (UINT16 *)SM_data->parm8;
                        vm_send_response_signal( aud_rc_ok, 0);
                        break;

                    case E_vms_release_stop_playback:
                    case E_vms_stop_playback:
                    case E_vms_pb_ram_data_ready:
                        break;
                        
                    case E_vms_stop_suspend:     
                        {
                            UINT8 slot_id;
                            UINT8 power_down;
                            aud_glob_vms_playback.sender = (*SM_data).sender;
                            aud_glob_vms_playback.slot_id  = (UINT16)((*SM_data).parm1);            
                            StopVmSuspendData(aud_glob_vms_playback.slot_id);
                            vm_send_response_signal(aud_rc_playback_stopped,0);
                            InitVmsPlayback();
                            power_down = TRUE;  
                            for (slot_id=0;slot_id<AUD_NOF_SUSPENDNEST;slot_id++) {
                                if (aud_glob_vms_playback.suspend_data[slot_id].in_use)
                                    power_down = FALSE;
                            }
                            if (power_down)
                                aud_enable_standby(me);
                        }
                        break;

                    default:
                        func_ret = FALSE;
                        break;
                }/*Event handle*/
            }/* switch event of S_idle */
            break;
        case S_active:
            {
                switch (vms_pb_ctrl.aud_vms_state) {
                    /*-------wait for reading header byte ----------------*/
                    case S_aud_vms_wait_for_reading_header_byte:
                        switch (event) {
                            case E_vms_get_total_playtime:
                            case E_vms_release_stop_playback:
                            case E_vms_stop_playback:
                            case E_vms_suspend :
                            case E_vms_set_play_position:
                            case E_vms_get_play_position:
                            case E_vms_stop_suspend:
                                vms_event_handle(event,SM_data);
                                break;

                            case E_vms_resume:
                                aud_glob_vms_playback.sender = (*SM_data).sender;
                                aud_glob_vms_playback.slot_id  = (UINT16)((*SM_data).parm1);            
                                vm_send_response_signal(aud_rc_request_error, 0);
                                break;

                            case E_timer_vms_playback_timeout:
                                if (!vms_pb_ctrl.data_outstanding) { //data ready or not
                                    if (vms_pb_ctrl.check_header_flag == TRUE) {
                                        vms_pb_ctrl.vms_header_size = 
                                        vms_pb_ctrl.ptr_vms_header  = 
                                        vms_pb_ctrl.playback_frame  = find_header_info_playback(aud_vms_data_array[vms_pb_ctrl.array_number].aud_vms_data);                
                                    }
                                    vms_pb_ctrl.vms_bytes_played += vms_pb_ctrl.ptr_vms_header;
                                    find_nof_bytes_per_frame(aud_vms_data_array[vms_pb_ctrl.array_number].aud_vms_data[vms_pb_ctrl.ptr_vms_header],NULL);
                                    copy_playback_buffer();
                                    AUD_set_internal_vms_pb_state(S_aud_vms_wait_for_playback);
                                }
                                break;
                            case E_vms_start_playback:  
                                aud_send_response_signal(aud_glob_vms_playback.sender, aud_resource_playback_vm,aud_rc_request_error, 0, 0, 0,  0, NULL, NULL);
                                break;
                            case E_vms_repeatevent:  
                            case E_vms_pb_set_getbuffer_function:
                                case E_vms_pb_ram_data_ready:
                                AUD_ERROR_REPORTING(aud_error_grade2, TRAP_AUD_ERROR, "Unhandled event: %d in VM Playback SM in %d state", event,S_aud_vms_wait_for_reading_header_byte)
                                break;

                            default:
                                func_ret = FALSE;
                                break;
                        }      
                        break; /*S_aud_vms_wait_for_reading_header_byte*/
                        /*-------------------------------------------------------------------*/
                        /*                           wait for playback.                      */
                        /*-------------------------------------------------------------------*/
                    case S_aud_vms_wait_for_playback:
                        switch (event) {
                            case E_vms_get_total_playtime:
                            case E_vms_release_stop_playback:
                            case E_vms_stop_playback:
                            case E_vms_suspend:                      
                            case E_vms_set_play_position:
                            case E_vms_get_play_position: 
                            case E_vms_stop_suspend:    
                                vms_event_handle(event,SM_data);
                                break;      
                            case E_vms_resume:
                                aud_glob_vms_playback.sender  = (*SM_data).sender;
                                aud_glob_vms_playback.slot_id   = (UINT16)((*SM_data).parm1);            
                                vm_send_response_signal(aud_rc_request_error, 0);
                                break;         
                            case E_timer_vms_playback_timeout:
                                if (!vms_pb_ctrl.data_outstanding)
                                    { //data ready or not
                                        if(vms_pb_ctrl.data_pagememory)
                                        {
                                        /*Copy data from the File handle to the local ping pong buffer */
                                          aud_vms_playback_ram_init_datacopy();
                                          vms_pb_ctrl.next_bytes_per_frame = aud_vms_data_array[vms_pb_ctrl.array_number_read].aud_vms_data[0];
                                         }
                                        else
                                        {
                                         vms_pb_ctrl.data_pending--;
                                        }
                                    check_frame_number();
                                }
                                break;
                            case E_vms_start_playback:  
                                aud_send_response_signal(aud_glob_vms_playback.sender, aud_resource_playback_vm,aud_rc_request_error, 0, 0, 0,  0, NULL, NULL);
                                break;
                            case E_vms_repeatevent:  
                            case E_vms_pb_set_getbuffer_function:
                                case E_vms_pb_ram_data_ready:
                                AUD_ERROR_REPORTING(aud_error_grade2, TRAP_AUD_ERROR, "Unhandled event: %d in VM Playback SM in %d state ", event,S_aud_vms_wait_for_playback)
                                break;               
                            default:
                                func_ret = FALSE;
                                break;
                        }
                        break;    
                    case S_aud_vms_playback:
                        switch (event) {
                            case E_vms_release_stop_playback:
                            case E_vms_stop_playback:
                            case E_vms_suspend :
                            case E_vms_set_play_position:
                            case E_vms_stop_suspend:    
                            case E_vms_get_play_position: 
                                case E_vms_pb_ram_data_ready:
                                vms_event_handle(event,SM_data);
                                break;    
    
                                case E_vms_resume:
                                aud_glob_vms_playback.sender = (*SM_data).sender;
                                aud_glob_vms_playback.slot_id  = (UINT16)((*SM_data).parm1);            
                                vm_send_response_signal(aud_rc_request_error, 0);
                                break;              
                            case E_vms_start_playback: 
                                aud_send_response_signal(aud_glob_vms_playback.sender, aud_resource_playback_vm,aud_rc_request_error, 0, 0, 0,  0, NULL, NULL);
                                break;
                            case E_vms_repeatevent:  
                            case E_vms_pb_set_getbuffer_function:                                
                            case E_timer_vms_playback_timeout:
                                AUD_ERROR_REPORTING(aud_error_grade2, TRAP_AUD_ERROR, "Unhandled event: %d in VM Playback SM in %d state", event,S_aud_vms_playback)
                                break;
  
                            case E_vms_get_total_playtime:
                                {
                                UINT8 bytes_in_frame;
                                AUD_vms_total_playtime_in_general.sender         = (*SM_data).sender;
                                AUD_vms_total_playtime_in_general.media_type     = (aud_media_enum)((*SM_data).parm1);
                                AUD_vms_total_playtime_in_general.format         = (aud_dsp_format_enum)((*SM_data).parm2);
                                AUD_vms_total_playtime_in_general.file_handle    = (UINT16*)SM_data->parm8;
                                AUD_vms_total_playtime_in_general.buffer_size    = (UINT32)((*SM_data).parm3);
                                AUD_vms_total_playtime_in_general.start_offset   = (UINT32)((*SM_data).parm4);
                                switch (AUD_vms_total_playtime_in_general.media_type) {
                                    case aud_media_mmf:
                                        vm_send_response_signal(aud_rc_request_error, 0);            
                                        break;
                                    case aud_media_ffs:
                                        AUD_vms_total_playtime_in_general.vms_total_playtime = CalcTotalPlayTimeForFfsFileForVM(AUD_vms_total_playtime_in_general.file_handle, AUD_vms_total_playtime_in_general.start_offset, AUD_vms_total_playtime_in_general.file_size, &bytes_in_frame);
                                        aud_send_response_signal(AUD_vms_total_playtime_in_general.sender, aud_resource_playback_vm, aud_rc_total_playtime, AUD_vms_total_playtime_in_general.vms_total_playtime, 0, 0, 0, NULL, NULL);
                                        break;
                                    case aud_media_mmc:                       
#ifdef  AUD_NANDDEVICE_SUPPORT                                	
                                case aud_media_fs_cdrive:
                                case aud_media_fs_edrive:
                                case aud_media_fs_fdrive:		
#endif                                 	                                    	
                                        AUD_vms_total_playtime_in_general.vms_total_playtime = CalcTotalPlayTimeForMMCFileForVM(AUD_vms_total_playtime_in_general.file_handle, AUD_vms_total_playtime_in_general.start_offset, AUD_vms_total_playtime_in_general.file_size, &bytes_in_frame);
                                        aud_send_response_signal(AUD_vms_total_playtime_in_general.sender, aud_resource_playback_vm, aud_rc_total_playtime, AUD_vms_total_playtime_in_general.vms_total_playtime, 0, 0, 0, NULL, NULL);
                                        break;
                                    case aud_media_ram:
                                        AUD_vms_total_playtime_in_general.vms_total_playtime = CalcTotalPlayTimeForRamFileForVM(AUD_vms_total_playtime_in_general.file_handle, AUD_vms_total_playtime_in_general.start_offset, &bytes_in_frame, AUD_vms_total_playtime_in_general.buffer_size);
                                        aud_send_response_signal(AUD_vms_total_playtime_in_general.sender, aud_resource_playback_vm, aud_rc_total_playtime, AUD_vms_total_playtime_in_general.vms_total_playtime, 0, 0, 0, NULL, NULL);
                                        break;
                                }
                                }
                                break;
                            default:
                                func_ret = FALSE;
                                break;
                        } 
                        break;
                        /*-------------------------------------------------------------------*/
                        /*                           Temp state.                             */
                        /*-------------------------------------------------------------------*/
                    case S_aud_vms_temp_state:
                        switch (event) {
                            case E_vms_release_stop_playback:
                            case E_vms_stop_playback:
                                stop_voice_memo_playback();
                                InitVmsPlayback();
                                aud_enable_standby(me);
                                break;
                            case E_vms_get_total_playtime:
                            case E_vms_suspend :
                            case E_vms_set_play_position:
                            case E_vms_get_play_position:
                            case E_vms_repeatevent:  
                            case E_vms_pb_set_getbuffer_function:
                            case E_vms_stop_suspend:    
                                AUD_ERROR_REPORTING(aud_error_grade2, TRAP_AUD_ERROR, "Unhandled event: %d in VM Playback SM in %d state", event,S_aud_vms_temp_state)
                                break;        
                            case E_vms_start_playback: 
                                aud_send_response_signal(aud_glob_vms_playback.sender, aud_resource_playback_vm,aud_rc_request_error, 0, 0, 0,  0, NULL, NULL);
                                break;        
                            default:
                                func_ret = FALSE;
                                break;
                        }
                        break;
                }    
            }/*Case S_active*/
    }
    return(func_ret);
}

/****************************************************************************************
* Function:... check_frame_number
* Parameters:. 
* Description: 
****************************************************************************************/
static void check_frame_number(void)
{
    if ((aud_glob_vms_playback.media_type == aud_media_ram) || 
        (aud_glob_vms_playback.media_type == aud_media_mmf)) 
    {
        find_nof_bytes_per_frame(vms_pb_ctrl.next_bytes_per_frame, NULL);
    }
    play_memo();
    AUD_set_internal_vms_pb_state(S_aud_vms_playback);
} /* End of function check_frame_number */

/****************************************************************************************
* Function:... play_memo
* Parameters:. 
* Description: 
****************************************************************************************/
static void play_memo(void)
{
    UINT16 DWD_HUGE* p_data; 

    p_data = aud_glob_vms_playback.fp_voice_memo_getpacket(0);
    if (p_data != NULL) {
        memcpy(&(p_l1d_vm_data->vm_buffer_1), p_data, vms_pb_ctrl.nof_bytes_per_frame); 
        vms_pb_ctrl.vms_frames_played ++;
        find_nof_bytes_per_frame(vms_pb_ctrl.next_bytes_per_frame,NULL); 
        DspStartVmPlayback(aud_glob_vms_playback.format);
        if (!vms_pb_ctrl.rc_check_flag) {
            vm_send_response_signal( aud_rc_playback_started, 0);
            vms_pb_ctrl.rc_check_flag = TRUE;
        }

#ifdef VM_MMF_DATA_VERIFY
	if(aud_glob_vms_playback.file_created == TRUE && aud_glob_vms_playback.file_id_created == TRUE)
	  	{
	    memcpy(&validate_vm[vm_write_index].VM_data[vm_next_location], &p_l1d_vm_data->vm_buffer_1,vms_pb_ctrl.nof_bytes_per_frame);
	    vm_next_location+= vms_pb_ctrl.nof_bytes_per_frame;
	    if((VALIDATE_VM_BUFFFER_SIZE - vm_next_location) < vms_pb_ctrl.nof_bytes_per_frame)
	        {
	         aud_send_mail(E_vm_write_to_ffs, aud_resource_table[aud_resource_playback_vm].caller_id,vm_write_index,vm_next_location,0,0,0,0,0, 0, NULL, NULL);
	         vm_next_location =0;       
		  vm_write_index = !vm_write_index;
	         }  
	  	}
#endif
    }
} /* End of function play_memo */

/****************************************************************************************
* Function:... vms_event_handle
* Parameters:. 
* Description: 
****************************************************************************************/
void vms_event_handle(aud_event_enum event, void *pData)
{
    UINT32  lTemp_vms_total_playtime;
    SINT16 temp_result; 
#ifdef IFWD_AUD_NEW_RC_PARAMETER
UINT32 FramesPlayed=0;UINT32 BytesPlayed=0;
    UINT32 pos=0;
#endif

    UINT8  lTemp_vms_state;
    disp_to_SM_data_type *SM_data = (disp_to_SM_data_type*)pData; //contains the structure passed from the dispatcher
    const aud_resource_enum me = aud_resource_playback_vm;

    switch (event) {
        case E_vms_get_total_playtime:
            lTemp_vms_state = vms_pb_ctrl.aud_vms_state;
            vms_pb_ctrl.aud_vms_state =S_aud_vms_temp_state;
            if (aud_glob_vms_playback.media_type != aud_media_mmf) {
                vm_send_response_signal( aud_rc_total_playtime, vms_pb_ctrl.vms_total_playtime);                   
            } else {
                vm_send_response_signal( aud_rc_request_error, 0);
            }                       
            vms_pb_ctrl.aud_vms_state = lTemp_vms_state;      
            break;  

        case E_vms_release_stop_playback:
        case E_vms_stop_playback:         
            vms_pb_ctrl.aud_vms_state =S_aud_vms_temp_state;
            stop_voice_memo_playback();
            vm_send_response_signal( aud_rc_playback_stopped, 0);
            InitVmsPlayback();    
            aud_enable_standby(me);
            break;

        case E_vms_suspend :        
            aud_glob_vms_playback.sender = (*SM_data).sender;
            aud_glob_vms_playback.slot_id = (UINT16)((*SM_data).parm1);            
            vms_pb_ctrl.aud_vms_state =S_aud_vms_temp_state;
            SaveVmSuspendData(aud_glob_vms_playback.slot_id);
            stop_voice_memo_playback();
            vm_send_response_signal(aud_rc_playback_suspended, 0);
            InitVmsPlayback();
            aud_enable_standby(me);         
            break;  

        case E_vms_set_play_position:
            aud_glob_vms_playback.sender  = (*SM_data).sender;
            aud_glob_vms_playback.pos       = (UINT32)((*SM_data).parm1);        
             lTemp_vms_total_playtime             = vms_pb_ctrl.vms_total_playtime;

            if ((vms_pb_ctrl.aud_vms_state == S_aud_vms_playback) && 
                (aud_glob_vms_playback.media_type != aud_media_mmf)) {
                vms_pb_ctrl.aud_vms_state = S_aud_vms_temp_state;
                DspStopPlaybackVm();       
            }
            switch (aud_glob_vms_playback.media_type) {
                case aud_media_ffs:
                case aud_media_mmc:   
#ifdef  AUD_NANDDEVICE_SUPPORT                	
                case aud_media_fs_cdrive:
                case aud_media_fs_edrive:
                case aud_media_fs_fdrive:		
#endif                 	
                    if (!vms_pb_ctrl.data_outstanding) 
                      {
                        UINT32 ltemp;
                        InitVmsPlayback();

                        ltemp =  aud_glob_vms_playback.pos / 20;
                        if (ltemp >= aud_glob_vms_playback.buffer_size) {
                            vm_send_response_signal(aud_rc_parameter_out_of_range, 0);  
                            break;
                        } else {
                            vms_pb_ctrl.rc_check_flag = TRUE;
                            if (aud_glob_vms_playback.media_type == aud_media_ffs)
                                temp_result = SetPositionForFfsFile(aud_glob_vms_playback.pos);
                            else
                                temp_result = SetPositionForMMCFile(aud_glob_vms_playback.pos);                                     
                            vms_pb_ctrl.vms_total_playtime = lTemp_vms_total_playtime;
                            if (temp_result) {
                                vms_pb_ctrl.aud_vms_state = S_aud_vms_playback_idle;
                                aud_set_SM_state(me,S_idle);
                                aud_glob_vms_playback.pos = 0;           
                                vm_send_response_signal( aud_rc_ok, 0);                
                            } else {
                                vm_send_response_signal(aud_rc_storage_problems, vms_pb_ctrl.ffs_result);
                                stop_voice_memo_playback();
                                InitVmsPlayback();
                                aud_enable_standby(me);      
                            }
                        }
                    } 
                  else 
                    {
                        vm_send_response_signal(aud_rc_storage_problems, vms_pb_ctrl.ffs_result);
                        stop_voice_memo_playback();
                        InitVmsPlayback();
                        aud_enable_standby(me);
                    }         
                    break;

                case aud_media_ram:
                    {
                        UINT32 ltemp;
                        InitVmsPlayback();
                        ltemp =  aud_glob_vms_playback.pos / 20;
                        if (ltemp >= aud_glob_vms_playback.buffer_size) 
                        {
                            vm_send_response_signal( aud_rc_parameter_out_of_range, 0);  
                            break;
                        }
                         else
                         {
                            vms_pb_ctrl.aud_vms_state = S_aud_vms_temp_state; 
                            temp_result = SetPositionForRamFile(aud_glob_vms_playback.pos);
                            aud_glob_vms_playback.pos = 0;           
                            vms_pb_ctrl.rc_check_flag = TRUE;/*Ensuring that aud_rc_playback RC is not sent again */
                            vms_pb_ctrl.vms_total_playtime = lTemp_vms_total_playtime;                               
                            if( !vms_pb_ctrl.data_pagememory)
                                {
                                  vms_pb_ctrl.aud_vms_state = S_aud_vms_wait_for_playback;
                                   aud_send_mail(E_timer_vms_playback_timeout, 0, 0, 0, 0, 0, 0, 0, 0, 0,NULL, NULL);
                                  if (!temp_result)
                                    {
                                  vm_send_response_signal(aud_rc_unknown_position, temp_result);
                                  stop_voice_memo_playback();
                                  InitVmsPlayback();
                                  aud_enable_standby(me);
                               }
                               else
                               { 
                                  vm_send_response_signal(aud_rc_ok, 0);    
                                }
                           }
                          else
                          {
                             vms_intialize_playback();	
                            vm_send_response_signal(aud_rc_ok, 0);    
                          }
                        }
                    }
                    break;

                case aud_media_mmf:
                    vm_send_response_signal(aud_rc_request_error, 0);
                    break;

                default:
                    break;
            }
            break;   

        case E_vms_get_play_position:
            if (aud_glob_vms_playback.media_type == aud_media_mmf)
            {
#ifdef IFWD_AUD_NEW_RC_PARAMETER
                  pos =FramesPlayed= vms_pb_ctrl.vms_frames_played;
     	              BytesPlayed = vms_pb_ctrl.vms_bytes_played;
	              pos *= 20;
			aud_send_response_signal(aud_glob_vms_playback.sender, 
						aud_resource_playback_vm,aud_rc_elapsed_time,pos ,FramesPlayed,
						BytesPlayed,0, NULL, NULL);	
		#else
		aud_send_response_signal(aud_glob_vms_playback.sender, 
					aud_resource_playback_vm,aud_rc_elapsed_time,vms_pb_ctrl.vms_frames_played,
					 vms_pb_ctrl.vms_bytes_played, 0,  0, NULL, NULL);				
//                vm_send_response_signal(aud_rc_request_error, 0);
        #endif
				
            } else {
                lTemp_vms_state  =  vms_pb_ctrl.aud_vms_state;
                vms_pb_ctrl.aud_vms_state = S_aud_vms_temp_state;
                GetVoiceMemoPlayPosition();
                vms_pb_ctrl.aud_vms_state = lTemp_vms_state;
            }
            break;

        case E_vms_stop_suspend:                
            aud_glob_vms_playback.sender         = (*SM_data).sender;
            aud_glob_vms_playback.slot_id = (UINT16)((*SM_data).parm1);            
            vms_pb_ctrl.aud_vms_state =S_aud_vms_temp_state;   
            StopVmSuspendData(aud_glob_vms_playback.slot_id);
            vm_send_response_signal(aud_rc_playback_stopped,0);
            InitVmsPlayback();
            aud_enable_standby(me);
            break;
        case E_vms_pb_ram_data_ready:
             aud_vms_playback_ram_read((UINT8 DWD_HUGE *)SM_data->ptr2, (UINT32)((*SM_data).parm1));
             break;
        default:
            break;
    }
}

/****************************************************************************************
* Function:... stop_voice_memo_playback
* Description: This function stop recording
****************************************************************************************/
static void stop_voice_memo_playback(void)
{
    DspStopPlaybackVm();
    disable_vms_audio_path();  

#ifdef VM_MMF_DATA_VERIFY
    aud_send_mail(E_vm_write_to_ffs, aud_resource_table[aud_resource_playback_vm].caller_id,vm_write_index,vm_next_location,0,0,0,0,0, 0, NULL, NULL);
    aud_glob_vms_playback.close_file = TRUE;
#endif
} /* End of function stop_voice_memo */

/****************************************************************************************
* Function:... CalcTotalPlayTimeForRamFileForVM
* Description: Calculate total playtime
****************************************************************************************/
static UINT32 CalcTotalPlayTimeForRamFileForVM(UINT16 DWD_HUGE* file_handle, UINT32 start_offset, UINT8* bytes_per_frame, UINT16 buffer_size)
{
    UINT32 i=0,j=0;
    UINT8 temp_header_size;
    UINT8 *ptr = NULL;

    ptr = (UINT8 *)file_handle;
    ptr += start_offset;
    temp_header_size = find_header_info_playback(ptr);
    i= temp_header_size + start_offset;
    do {
        ptr = (UINT8 *)file_handle;
        ptr += i;
        find_nof_bytes_per_frame(*ptr, bytes_per_frame);
        j ++;
        if(bytes_per_frame == NULL)        
                i += vms_pb_ctrl.nof_bytes_per_frame;
        else
                i += (*bytes_per_frame);    
    }
    while (i<buffer_size);
    j *= 20; /* j counts the nof 20 msec frame - return the time in msec */
    return(j);
}

/****************************************************************************************
* Function:... SetPositionForRamFile
* Description: Sets play position for RAM file
****************************************************************************************/
static UINT32 SetPositionForRamFile(UINT32 pos)
{
    UINT32 i, j=0;
    UINT8 temp_header_size;
    UINT8 *ptr = NULL;

    pos /= 20;
    ptr = (UINT8 *)aud_glob_vms_playback.file_handle;
    ptr += aud_glob_vms_playback.start_offset;
    temp_header_size = find_header_info_playback(ptr);  

    j = temp_header_size + aud_glob_vms_playback.start_offset;
    for (i=0; i<pos; i++) {
        ptr = (UINT8 *)aud_glob_vms_playback.file_handle;
        ptr += j;   
        find_nof_bytes_per_frame(*ptr,NULL);
        j += vms_pb_ctrl.nof_bytes_per_frame;
        if (j >= aud_glob_vms_playback.buffer_size)
            return(FALSE);
    }
   if(vms_pb_ctrl.data_pagememory)
   {
	vms_pb_ctrl.ram_playback_index = j;
   }
   else
   {
   	vms_pb_ctrl.playback_frame = j;
   }
    vms_pb_ctrl.vms_header_size = temp_header_size;
    vms_pb_ctrl.check_header_flag = FALSE;
    vms_pb_ctrl.vms_bytes_played  =
    vms_pb_ctrl.vms_bytes_read    = j;
    vms_pb_ctrl.vms_frames_played = i;

    return(TRUE);
}

/****************************************************************************************
* Function:... SetPositionForMMFFile
* Description: Sets play position for MMF file
****************************************************************************************/
static void SetPositionForMMFFile(UINT32 bytes_tobe_moved)
{
    vms_pb_ctrl.playback_frame = bytes_tobe_moved;
    vms_pb_ctrl.check_header_flag = FALSE;
}

/****************************************************************************************
* Function:... SetPositionForFfsFile
* Parameters:. 
* Description: 
****************************************************************************************/
static UINT32 SetPositionForFfsFile(UINT32 pos)
{
    UINT32 i=0,j=0;
    aud_ffs_error_code_type rc;
    aud_ffs_file_info_type ffs_info;
    UINT8 dst[8];
    UINT16 *file_handle;
    UINT16  temp_header_size;
    UINT16 file_id;

    pos /= 20;

    rc = (aud_ffs_error_code_type)Aud_FFS_get_file_id_from_handle((aud_ffs_user_type)AUD_FFS_VR, (UINT16 *)aud_glob_vms_playback.file_handle, &file_id);
    if (rc != AUD_FFS_SUCCESS)
        return(0);
    rc = (aud_ffs_error_code_type)Aud_FFS_close((aud_ffs_user_type)AUD_FFS_VR, file_id, (UINT16 *)aud_glob_vms_playback.file_handle, NULL
#ifdef FFS_EXTENDED_CALLBACK
                                                ,NULL
#endif
                                                );
    if (rc != AUD_FFS_SUCCESS)
        return(0);
    /* Find the file length */
    rc = (aud_ffs_error_code_type)Aud_FFS_get_file_info(AUD_FFS_VR, file_id, &ffs_info); 
    if (rc != AUD_FFS_SUCCESS)
        return(0);
    vms_pb_ctrl.vms_file_size = ffs_info.size;

    rc = (aud_ffs_error_code_type)Aud_FFS_open(AUD_FFS_VR, file_id, &file_handle, AUD_FFS_FILE_OPEN_READ);
    if (rc != AUD_FFS_SUCCESS)
        return(0);
    /* Read the first byte - to determine the frame length */

    i = aud_glob_vms_playback.start_offset;
    rc = (aud_ffs_error_code_type)Aud_FFS_read(AUD_FFS_VR_READ, file_id, file_handle, dst, i, 7, NULL
#ifdef FFS_EXTENDED_CALLBACK
                                               ,NULL
#endif
                                               );
    vms_pb_ctrl.vms_header_size = temp_header_size = find_header_info_playback(dst);
    if (rc != AUD_FFS_SUCCESS)
        return(0);
    i += temp_header_size;

    for (j=1; j<=pos; j++) {
        find_nof_bytes_per_frame(dst[temp_header_size],NULL);
        i += vms_pb_ctrl.nof_bytes_per_frame;
        rc = (aud_ffs_error_code_type)Aud_FFS_read(AUD_FFS_VR_READ, file_id, file_handle, dst, i, 1, NULL
#ifdef FFS_EXTENDED_CALLBACK
                          ,NULL
#endif
                          );
        temp_header_size = 0;
        if (rc != AUD_FFS_SUCCESS || i >= vms_pb_ctrl.vms_file_size)
            return(0);
    }  
    rc = (aud_ffs_error_code_type)Aud_FFS_close(AUD_FFS_VR, file_id, file_handle, NULL
#ifdef FFS_EXTENDED_CALLBACK
                                                ,NULL
#endif
                                                );
    if (rc != AUD_FFS_SUCCESS)
        return(0);
    rc = (aud_ffs_error_code_type)Aud_FFS_open(AUD_FFS_VR, file_id, &file_handle, AUD_FFS_FILE_OPEN_READ_STREAMING);
    if (rc != AUD_FFS_SUCCESS)
        return(0);
    aud_glob_vms_playback.file_handle = file_handle;

    /* Set file pointer */
    rc = (aud_ffs_error_code_type)Aud_SSAL_set_file_pos(aud_glob_vms_playback.ssal_media_type,
                                                        (aud_ssal_user_type) AUD_SSAL_VR,  
                                                        (void*)aud_glob_vms_playback.file_handle,
                                                        i,&vms_setposition_ssal_callback
#ifdef FFS_EXTENDED_CALLBACK
                                                        ,NULL
#endif
                                                        );
    if (rc != AUD_FFS_SUCCESS)
        return(0);
    vms_pb_ctrl.vms_bytes_played  = vms_pb_ctrl.vms_bytes_read    = i;
    vms_pb_ctrl.vms_frames_played = j;
    vms_pb_ctrl.check_header_flag = FALSE;
    vms_pb_ctrl.ptr_vms_header    = 0;

    return(TRUE);
}

/****************************************************************************************
* Function:... CalcTotalPlayTimeForFfsFileForVM
* Parameters:. 
* Description: 
****************************************************************************************/
static UINT32 CalcTotalPlayTimeForFfsFileForVM(UINT16 DWD_HUGE* input_file_handle, UINT32 start_offset, UINT32 file_size, UINT8* bytes_per_frame)
{    
   	UINT32 i,j,nof_buffers_read, read_ptr, nof_bytes_to_read;
   	aud_ffs_error_code_type rc;   
   	aud_ffs_file_info_type ffs_info;    
    	UINT8 temp_header_size;   
   UINT16 *file_handle; 
 UINT16 file_id;   		   	
   	
    i=j=nof_buffers_read=0;
 
    rc =  (aud_ffs_error_code_type)Aud_FFS_get_file_id_from_handle(AUD_FFS_VR, (UINT16 *)input_file_handle, &file_id);
    if (rc != AUD_FFS_SUCCESS)
        return(0);
    rc =(aud_ffs_error_code_type)Aud_FFS_close(AUD_FFS_VR, file_id, (UINT16 *)input_file_handle, NULL
#ifdef FFS_EXTENDED_CALLBACK
                       ,NULL
#endif
                       );
    if (rc != AUD_FFS_SUCCESS)
        return(0);
    rc = (aud_ffs_error_code_type)Aud_FFS_open(AUD_FFS_VR, file_id, &file_handle, AUD_FFS_FILE_OPEN_READ);
    if (rc != AUD_FFS_SUCCESS)
        return(0);
    /* File the file length */
    rc = (aud_ffs_error_code_type)Aud_FFS_get_file_info(AUD_FFS_VR, file_id, &ffs_info); 
    if (rc != AUD_FFS_SUCCESS)
        return(0);
    file_size = ffs_info.size;

      read_ptr = start_offset;
    if (sizeof(aud_vms_total_playtime_buffer) < file_size)
      nof_bytes_to_read = sizeof(aud_vms_total_playtime_buffer);
    else
      nof_bytes_to_read = file_size;
    
   
    rc = (aud_ffs_error_code_type)Aud_FFS_read(AUD_FFS_VR_READ, file_id, file_handle, aud_vms_total_playtime_buffer, read_ptr, nof_bytes_to_read, NULL
#ifdef FFS_EXTENDED_CALLBACK	        
        , NULL
#endif 
        );
    if (rc != AUD_FFS_SUCCESS) {
        return(0);
    }
    nof_buffers_read ++;
    temp_header_size = find_header_info_playback(aud_vms_total_playtime_buffer);
    i = temp_header_size;
    do
    {
       find_nof_bytes_per_frame(aud_vms_total_playtime_buffer[i], bytes_per_frame);
       j ++;
    if(bytes_per_frame == NULL)               
       i += (vms_pb_ctrl.nof_bytes_per_frame);
    else
       i += (*bytes_per_frame);
    }
    while( i < nof_bytes_to_read );
    
    read_ptr += nof_bytes_to_read;
    while (read_ptr < file_size)
    {
        i = i - nof_bytes_to_read; /* New read position in new buffer */
        if (read_ptr + sizeof(aud_vms_total_playtime_buffer) < file_size)
          nof_bytes_to_read = sizeof(aud_vms_total_playtime_buffer);
        else
          nof_bytes_to_read = file_size - read_ptr;
        rc = (aud_ffs_error_code_type)Aud_FFS_read(AUD_FFS_VR_READ, file_id, file_handle, aud_vms_total_playtime_buffer, read_ptr, nof_bytes_to_read, NULL
#ifdef FFS_EXTENDED_CALLBACK	        
        , NULL
#endif 
        );
        if (rc != AUD_FFS_SUCCESS) {
            return(0);
        }
        nof_buffers_read++;
        read_ptr += nof_bytes_to_read;
        do
        {
           find_nof_bytes_per_frame(aud_vms_total_playtime_buffer[i], bytes_per_frame);
           j ++;
            if(bytes_per_frame == NULL)               
               i += (vms_pb_ctrl.nof_bytes_per_frame);
            else
               i += (*bytes_per_frame);
        }
        while( i < nof_bytes_to_read );
    }  
    rc = (aud_ffs_error_code_type)Aud_FFS_close(AUD_FFS_VR, file_id, file_handle, NULL
#ifdef FFS_EXTENDED_CALLBACK	        
        , NULL
#endif 
        );
    if (rc != AUD_FFS_SUCCESS)
        return(0);
    rc = (aud_ffs_error_code_type)Aud_FFS_open(AUD_FFS_VR, file_id, &file_handle, AUD_FFS_FILE_OPEN_READ_STREAMING);
    if (rc != AUD_FFS_SUCCESS)
        return(0);
    j *= 20; /* j counts the nof 20 msec frames - return the time in msec */
    return(j);
}

/****************************************************************************************
* Function:... SetPositionForMMCFile
* Parameters:. 
* Description: 
****************************************************************************************/
static UINT32 SetPositionForMMCFile(UINT32 pos)
{
    UINT32 i=0,j=0;
    sshort rc;
    BOOL result = TRUE;
    UINT8 dst[8];
    UINT16 temp_header_size;

    pos /= 20;
    i = aud_glob_vms_playback.start_offset;

    /*Calculate the file size */
    vms_pb_ctrl.vms_file_size = (UINT32)Aud_FS_seek ((sshort)(*aud_glob_vms_playback.file_handle), 0, AUD_FS_SEEK_END, &rc);

    if (rc != AUD_FS_SUCCESS)
        return(0);
    result = (BOOL) Aud_FS_seek ((sshort)(*aud_glob_vms_playback.file_handle), i, AUD_FS_SEEK_SET, &rc);

    if (rc != AUD_FS_SUCCESS)
        return(0);
    result = Aud_FS_read((sshort)(*aud_glob_vms_playback.file_handle), dst, 7);

    if (result == FALSE)
        return(0);
    vms_pb_ctrl.vms_header_size =
    temp_header_size = find_header_info_playback(dst);    
    i += temp_header_size;

    for (j=1; j<=pos; j++) {
        find_nof_bytes_per_frame(dst[temp_header_size], NULL);
        i += vms_pb_ctrl.nof_bytes_per_frame;

        result = (BOOL) Aud_FS_seek ((sshort)*aud_glob_vms_playback.file_handle, i, AUD_FS_SEEK_SET, &rc);
        if (rc != AUD_FS_SUCCESS)
            return(0);
        result = Aud_FS_read((sshort)*aud_glob_vms_playback.file_handle, dst, 1);
        if (result == FALSE || i >= vms_pb_ctrl.vms_file_size)
            return(0);
    }
    /* To retun to the last byte of the last frame */
    result = Aud_FS_seek ((sshort)*aud_glob_vms_playback.file_handle, i, AUD_FS_SEEK_SET, &rc);
    if (result == FALSE)
        return(0);

    vms_pb_ctrl.vms_bytes_played  = vms_pb_ctrl.vms_bytes_read    = i;
    vms_pb_ctrl.vms_frames_played = j;
    vms_pb_ctrl.check_header_flag = FALSE;
    vms_pb_ctrl.ptr_vms_header    = 0;

    aud_send_mail(E_vms_repeatevent, 0, 0, 0, 0, 0, 0, 0, 0, 0,NULL, NULL);
    return(TRUE);
}

/****************************************************************************************
* Function:... CalcTotalPlayTimeForMMCFileForVM
* Parameters:. 
* Description: 
****************************************************************************************/
static UINT32 CalcTotalPlayTimeForMMCFileForVM(UINT16 DWD_HUGE* input_file_handle, UINT32 start_offset, UINT32 file_size, UINT8* bytes_per_frame)
{
   UINT32 i,j,nof_buffers_read, read_ptr, nof_bytes_to_read;

    sshort rc;
    BOOL result;
    UINT8 temp_header_size;

    i=j=nof_buffers_read=0;

    file_size = Aud_FS_seek ((sshort)*input_file_handle, 0, AUD_FS_SEEK_END, &rc);


    if (rc != AUD_FS_SUCCESS) {
        return(0);
    }

    read_ptr = start_offset;
    result = Aud_FS_seek ((sshort)*input_file_handle, read_ptr, AUD_FS_SEEK_SET, &rc);
    if (rc != AUD_FS_SUCCESS) {
        return(0);
    }


    if (sizeof(aud_vms_total_playtime_buffer) < file_size)
      nof_bytes_to_read = sizeof(aud_vms_total_playtime_buffer);
    else
      nof_bytes_to_read = file_size;
    result = Aud_FS_read((sshort)*input_file_handle, aud_vms_total_playtime_buffer, nof_bytes_to_read);
    if (result == FALSE)
        return(0);

    nof_buffers_read ++;
    temp_header_size = find_header_info_playback(aud_vms_total_playtime_buffer);
    i = temp_header_size;
    do
    {
       find_nof_bytes_per_frame(aud_vms_total_playtime_buffer[i], bytes_per_frame);
       j ++;
       if(bytes_per_frame == NULL)
            i += (vms_pb_ctrl.nof_bytes_per_frame);
       else
             i += (*bytes_per_frame);
    }
    while( i < nof_bytes_to_read );
    

    read_ptr += nof_bytes_to_read;
    while (read_ptr < file_size)
    {
        i = i - nof_bytes_to_read; /* New read position in new buffer */
        if (read_ptr + sizeof(aud_vms_total_playtime_buffer) < file_size)
          nof_bytes_to_read = sizeof(aud_vms_total_playtime_buffer);
        else
          nof_bytes_to_read = file_size - read_ptr;
        result = Aud_FS_read((sshort)*input_file_handle, aud_vms_total_playtime_buffer, nof_bytes_to_read);
        if (result == FALSE)
            return(0);
        nof_buffers_read++;
        read_ptr += nof_bytes_to_read;
        do
        {
           find_nof_bytes_per_frame(aud_vms_total_playtime_buffer[i],bytes_per_frame);
           j ++;
       if(bytes_per_frame == NULL)
             i += (vms_pb_ctrl.nof_bytes_per_frame);
       else
             i += (*bytes_per_frame);
        }
        while( i < nof_bytes_to_read );
    }

    
    result = Aud_FS_seek ((sshort)*input_file_handle, 0, AUD_FS_SEEK_SET, &rc);
    if (rc != AUD_FS_SUCCESS) {
        return(0);
    }
    j *= 20; /* j counts the nof 20 msec frames - return the time in msec */
    return(j);

}

/****************************************************************************************
* Function:... GetVoiceMemoPlayPosition
* Parameters:. 
* Description: 
****************************************************************************************/
void GetVoiceMemoPlayPosition(void)
{
    UINT32 pos;  
    DISABLE_CLOCK;
    pos = vms_pb_ctrl.vms_frames_played;
    ENABLE_CLOCK;
    pos *= 20;
    vm_send_response_signal( aud_rc_elapsed_time, pos);
}

/****************************************************************************************
* Function:... copy_playback_buffer
* Parameters:. 
* Description: 
****************************************************************************************/
static void copy_playback_buffer(void)
{
    UINT16 DWD_HUGE * p_data;
    UINT16 read_nof_bytes;
    UINT8 lTemp_var;

    if (vms_pb_ctrl.aud_vms_state == S_aud_vms_wait_for_reading_header_byte) 
    {
        if ( (aud_glob_vms_playback.media_type == aud_media_ffs )  || 
        	    (aud_glob_vms_playback.media_type == aud_media_mmc )
#ifdef  AUD_NANDDEVICE_SUPPORT
         ||(aud_glob_vms_playback.media_type == aud_media_fs_cdrive )  ||
        	    (aud_glob_vms_playback.media_type == aud_media_fs_edrive )  ||
        	    (aud_glob_vms_playback.media_type == aud_media_fs_fdrive )  
#endif         	    
        	    )        	    
        {
            if (vms_pb_ctrl.check_header_flag == TRUE)
                lTemp_var = 7;
            else
                lTemp_var = 1;

            read_nof_bytes = CalcNofBytesToRead(lTemp_var);      
            if (read_nof_bytes == AUD_NOF_BYTES_IN_VMPLAYBACK_BUFFER)
                read_nof_bytes -= (lTemp_var);

            vms_pb_ctrl.data_outstanding = TRUE;         
            vms_pb_ctrl.data_pending = FALSE;
            vms_pb_ctrl.ffs_result = Aud_SSAL_read((aud_ssal_media_type)aud_glob_vms_playback.ssal_media_type,
                                                   (aud_ssal_user_type)AUD_SSAL_VR_READ,               
                                                   (void*)aud_glob_vms_playback.file_handle,        
                                                   (UINT8*) &aud_vms_data_array[vms_pb_ctrl.array_number].aud_vms_data[lTemp_var],                                       
                                                   (UINT32)read_nof_bytes,         
                                                   &vms_read_ssal_callback
#ifdef FFS_EXTENDED_CALLBACK
                                                   , NULL
#endif
                                                  );     

            if ((vms_pb_ctrl.ffs_result == AUD_FFS_SUCCESS) || (vms_pb_ctrl.ffs_result == AUD_FS_SUCCESS)) {
                vms_pb_ctrl.vms_bytes_read += (UINT32)read_nof_bytes;
                vms_pb_ctrl.playback_frame = vms_pb_ctrl.ptr_vms_header;
                vms_pb_ctrl.array_number   =! vms_pb_ctrl.array_number;
            } else {
                vm_send_response_signal( aud_rc_performance_problems, vms_pb_ctrl.ffs_result);
            }
        }
    } else {
           p_data =  aud_glob_vms_playback.fp_voice_memo_getpacket(0);
        if (p_data != NULL) {
            memcpy(&(p_l1d_vm_data->vm_buffer_1), p_data, vms_pb_ctrl.nof_bytes_per_frame);
            vms_pb_ctrl.vms_frames_played ++;     
            find_nof_bytes_per_frame(vms_pb_ctrl.next_bytes_per_frame,NULL);    

#ifdef VM_MMF_DATA_VERIFY
    if(aud_glob_vms_playback.file_created == TRUE && aud_glob_vms_playback.file_id_created == TRUE)
    {
        memcpy(&validate_vm[vm_write_index].VM_data[vm_next_location], &p_l1d_vm_data->vm_buffer_1,vms_pb_ctrl.nof_bytes_per_frame);
        vm_next_location+= vms_pb_ctrl.nof_bytes_per_frame;
        if((VALIDATE_VM_BUFFFER_SIZE - vm_next_location) < vms_pb_ctrl.nof_bytes_per_frame)
        {
            aud_send_mail(E_vm_write_to_ffs, aud_resource_table[aud_resource_playback_vm].caller_id,vm_write_index,vm_next_location,0,0,0,0,0, 0, NULL, NULL);
            vm_next_location =0;       
            vm_write_index = !vm_write_index;
        }  
    }
#endif
        }
    }   
}

/****************************************************************************************
* Function:... find_nof_bytes_per_frame
* Parameters:. 
* Description: 
****************************************************************************************/
static void find_nof_bytes_per_frame(UINT8 header, UINT8* bytes_per_frame)
{
    UINT8 sample_rate;
    UINT16 bytes_in_each_frame;

    if (aud_glob_vms_playback.format == aud_dsp_format_amr) {
        sample_rate   = header;
        sample_rate  &= 0x78;
        sample_rate >>= 3;

        switch (sample_rate) {
            case 0:    /* 4,75 kbit/sec */
                bytes_in_each_frame = 13;
                break;
            case 1:    /* 5,15 kbit/sec */
                bytes_in_each_frame = 14;
                break;
            case 2:    /* 5,90 kbit/sec */
                bytes_in_each_frame = 16;
                break;
            case 3:    /* 6,70 kbit/sec */
                bytes_in_each_frame = 18;
                break;
            case 4:    /* 7,40 kbit/sec */
                bytes_in_each_frame = 20;
                break;
            case 5:    /* 7,95 kbit/sec */
                bytes_in_each_frame = 21;
                break;
            case 6:    /*10,2 kbit/sec  */
                bytes_in_each_frame = 27;
                break;
            case 7:    /*12,2 kbit/sec  */
                bytes_in_each_frame = 32;
                break;
            case 15:   /* NO-DATA frame */
                bytes_in_each_frame = 1;
                break;
            default:   /* SID frame */
                bytes_in_each_frame = 6;
                break;
        }
    } 
    else if(aud_glob_vms_playback.format == aud_dsp_format_amr_if2) {
        sample_rate   = header;
        sample_rate  &= 0xF0;
        sample_rate >>= 4;

        switch (sample_rate) {
            case 0:    /* 4,75 kbit/sec */
                bytes_in_each_frame = 13;
                break;
            case 1:    /* 5,15 kbit/sec */
                bytes_in_each_frame = 14;
                break;
            case 2:    /* 5,90 kbit/sec */
                bytes_in_each_frame = 16;
                break;
            case 3:    /* 6,70 kbit/sec */
                bytes_in_each_frame = 18;
                break;
            case 4:    /* 7,40 kbit/sec */
                bytes_in_each_frame = 19;
                break;
            case 5:    /* 7,95 kbit/sec */
                bytes_in_each_frame = 21;
                break;
            case 6:    /*10,2 kbit/sec  */
                bytes_in_each_frame = 26;
                break;
            case 7:    /*12,2 kbit/sec  */
                bytes_in_each_frame = 31;
                break;
            case 15:   /* NO-DATA frame */
                bytes_in_each_frame = 1;
                break;
            default:   /* SID frame */
                bytes_in_each_frame = 6;
                break;
        }
      }else{
        bytes_in_each_frame = 34;
        }

    //store the result
    if(bytes_per_frame == NULL)
        vms_pb_ctrl.nof_bytes_per_frame = bytes_in_each_frame;
    else
        *bytes_per_frame = bytes_in_each_frame;
}

/****************************************************************************************
* Function:... CalcNofBytesToRead
* Parameters:. 
* Description: 
****************************************************************************************/
static UINT16 CalcNofBytesToRead(UINT32 bytes_read)
{
    if (vms_pb_ctrl.vms_file_size-bytes_read > AUD_NOF_BYTES_IN_VMPLAYBACK_BUFFER)
        return(AUD_NOF_BYTES_IN_VMPLAYBACK_BUFFER);
    else
        return(vms_pb_ctrl.vms_file_size-bytes_read);  
}

/****************************************************************************************
* Function:... AUD_set_internal_vms_pb_state
* Parameters:. 
* Description: 
****************************************************************************************/
static void AUD_set_internal_vms_pb_state(UINT8 new_state)
{
    vms_pb_ctrl.aud_vms_state = new_state;
}

/****************************************************************************************
* Function:... find_header_info_playback
* Parameters:. 
* Description: 
****************************************************************************************/
static UINT8 find_header_info_playback(UINT8 *pData)
{
    SINT8 StringBuf[7];  
    UINT8 offset = 0;
    StringBuf[6] = '\0';
    headerstrncpy(StringBuf, pData, 6);
    if (strcmp((char *)StringBuf, "#!AMR\n") != 0)
        offset = 0;
    else
        offset = 6; 
    return offset;
}

/****************************************************************************************
* Function:... headerstrncpy
* Parameters:. 
* Description: 
****************************************************************************************/
static void headerstrncpy(SINT8 *p_dest, huge UINT8 *p_source, UINT16 num)
{
    UINT16 i;
    for (i=0; i<num;i++)
        *p_dest++=*p_source++;
}

/****************************************************************************************
* Function:... vms_intialize_playback
* Parameters:. 
* Description: 
****************************************************************************************/
static void vms_intialize_playback(void)
{
    UINT16 size=0;
    UINT8 *aud_ptr=NULL;
    UINT8 *ptr = NULL;
    UINT8 lTemp_var;
    const aud_resource_enum me = aud_resource_playback_vm;

    switch (aud_glob_vms_playback.media_type) {
        case aud_media_ram:    
                enable_vms_audio_path();
             if(vms_pb_ctrl.data_pagememory)
             {
                vms_pb_ctrl.vms_file_size = aud_glob_vms_playback.buffer_size;
               if (vms_pb_ctrl.check_header_flag == TRUE) 
                {
                    ptr = (UINT8 *)aud_glob_vms_playback.file_handle;
                    ptr += aud_glob_vms_playback.start_offset;   		  
                    vms_pb_ctrl.ram_playback_index =
                    vms_pb_ctrl.vms_header_size      = find_header_info_playback(ptr);		  
                    vms_pb_ctrl.ram_playback_index += aud_glob_vms_playback.start_offset;
                    vms_pb_ctrl.vms_bytes_read       = vms_pb_ctrl.ram_playback_index;
                }
              }
             else
              {
              if (vms_pb_ctrl.check_header_flag == TRUE) 
              {
                    ptr = (UINT8 *)aud_glob_vms_playback.file_handle;
                    ptr += aud_glob_vms_playback.start_offset;
                    vms_pb_ctrl.vms_header_size = 
                    vms_pb_ctrl.playback_frame  = find_header_info_playback(ptr);
                    vms_pb_ctrl.playback_frame += aud_glob_vms_playback.start_offset;
              }
            }
            AUD_set_internal_vms_pb_state(S_aud_vms_wait_for_playback);		
            aud_send_mail(E_timer_vms_playback_timeout, 0, 0, 0, 0, 0, 0, 0, 0, 0,NULL, NULL);
            break;

        case aud_media_mmf:
            if (vms_pb_ctrl.vms_bytes_played == 0) 
         {
              (*aud_glob_vms_playback.get_buffer_func)(&aud_ptr, &size, aud_glob_vms_playback.mmf_user_data);
               aud_glob_vms_playback.file_handle = (UINT16 *)aud_ptr;
                aud_glob_vms_playback.buffer_size = size;
                if ((size == 0) || (aud_glob_vms_playback.file_handle == NULL))
               {
                    vm_send_response_signal(aud_rc_playback_stopped,0);                        
                }
              else 
                {
                    vms_pb_ctrl.vms_header_size =                                   
                    vms_pb_ctrl.playback_frame = find_header_info_playback((UINT8 *)aud_glob_vms_playback.file_handle);              
                    enable_vms_audio_path();
                    vms_pb_ctrl.next_bytes_per_frame = ((UINT8 DWD_HUGE *)aud_glob_vms_playback.file_handle)[vms_pb_ctrl.playback_frame];                                   
                    AUD_set_internal_vms_pb_state(S_aud_vms_wait_for_playback);
                    aud_send_mail(E_timer_vms_playback_timeout, 0, 0, 0, 0, 0, 0, 0, 0, 0,NULL, NULL);      
                }
            } 
            else
            {
                enable_vms_audio_path();
                vms_pb_ctrl.next_bytes_per_frame = ((UINT8 DWD_HUGE *)aud_glob_vms_playback.file_handle)[vms_pb_ctrl.playback_frame];                                   
                AUD_set_internal_vms_pb_state(S_aud_vms_wait_for_playback);
                aud_send_mail(E_timer_vms_playback_timeout, 0, 0, 0, 0, 0, 0, 0, 0, 0,NULL, NULL);      
            }
            break;

        case aud_media_mmc:
        case aud_media_ffs:  
#ifdef  AUD_NANDDEVICE_SUPPORT        	
        case aud_media_fs_cdrive:
        case aud_media_fs_edrive:
        case aud_media_fs_fdrive:		
#endif         	
            vms_pb_ctrl.data_outstanding = TRUE;
            aud_glob_vms_playback.buffer_size =
            vms_pb_ctrl.vms_file_size = Aud_SSAL_get_file_length((aud_ssal_media_type)aud_glob_vms_playback.ssal_media_type,
                                                                 (aud_ssal_user_type)AUD_SSAL_VR,
                                                                 (void *)aud_glob_vms_playback.file_handle);                
            if ((aud_glob_vms_playback.start_offset > 0) && (vms_pb_ctrl.check_header_flag == TRUE)) {
                vms_pb_ctrl.ffs_result = (aud_ffs_error_code_type)Aud_SSAL_set_file_pos(aud_glob_vms_playback.ssal_media_type,
                                                                                        (aud_ssal_user_type) AUD_SSAL_VR,  
                                                                                        (void*)aud_glob_vms_playback.file_handle,
                                                                                        aud_glob_vms_playback.start_offset,
                                                                                        NULL
#ifdef FFS_EXTENDED_CALLBACK
                                                                                        , NULL
#endif
                                                                                       );               
            }
            if (vms_pb_ctrl.check_header_flag == TRUE) {
                lTemp_var = 7;
                vms_pb_ctrl.vms_bytes_read = aud_glob_vms_playback.start_offset + lTemp_var;                   
                vms_pb_ctrl.vms_bytes_played = aud_glob_vms_playback.start_offset;
            } else {
                lTemp_var = 1;
                vms_pb_ctrl.vms_bytes_read += lTemp_var;                   
            }
            /* Read header byte (for finding the sample rate) */
            vms_pb_ctrl.ffs_result= Aud_SSAL_read((aud_ssal_media_type)aud_glob_vms_playback.ssal_media_type,
                                                  (aud_ssal_user_type)AUD_SSAL_VR_READ,
                                                  (void *)aud_glob_vms_playback.file_handle,
                                                  (UINT8 *)aud_vms_data_array[vms_pb_ctrl.array_number].aud_vms_data,
                                                  lTemp_var,
                                                  &vms_read_ssal_callback
#ifdef FFS_EXTENDED_CALLBACK
                                                  , NULL
#endif
                                                 );

            if ((vms_pb_ctrl.ffs_result == AUD_FFS_SUCCESS) || (vms_pb_ctrl.ffs_result == AUD_FS_SUCCESS)) {
                enable_vms_audio_path();
                AUD_set_internal_vms_pb_state(S_aud_vms_wait_for_reading_header_byte); 
            } else {
                vm_send_response_signal(aud_rc_storage_problems, vms_pb_ctrl.ffs_result);
                aud_set_SM_state(me, S_idle);             
                InitVmsPlayback();
                aud_enable_standby(me);
            }
    }
    return;
} /* End of fucntion */

/****************************************************************************************
* Function:... InitVmsPlayback
* Parameters:. 
* Description: 
****************************************************************************************/
void InitVmsPlayback(void)
{
    memset(aud_vms_data_array, 0,sizeof(aud_vms_data_array));

    vms_pb_ctrl.vm_result           	 	= 0;
    vms_pb_ctrl.vms_file_size        	 	= 0;
    vms_pb_ctrl.vms_offset           	 	= 0;
    vms_pb_ctrl.vms_bytes_played     	= 0;
    vms_pb_ctrl.vms_bytes_read        	= 0;
    vms_pb_ctrl.vms_total_playtime     	= 0;
    vms_pb_ctrl.vms_header_size        	= 0;
    vms_pb_ctrl.vms_frames_played    	= 0;
    vms_pb_ctrl.nof_bytes_per_frame  	= 0;
    vms_pb_ctrl.next_bytes_per_frame 	= 0;
    vms_pb_ctrl.mmf_synch_result     	= TRUE;
    vms_pb_ctrl.mmf_data             		= FALSE;
    vms_pb_ctrl.data_outstanding     		= FALSE;
    vms_pb_ctrl.data_pending         		= FALSE;
    vms_pb_ctrl.memo_end             		= FALSE;
    vms_pb_ctrl.aud_vm_error         		= FALSE;
    vms_pb_ctrl.ptr_vms_header       		= 0;  
    vms_pb_ctrl.array_number         		= 0;
    vms_pb_ctrl.array_number_read    	= 0;  
    vms_pb_ctrl.sync_counter         		= 0;  
    vms_pb_ctrl.playback_frame       		= 0;
    vms_pb_ctrl.ram_data_outstanding 	= FALSE;
    vms_pb_ctrl.ram_data_pending		= FALSE;
    vms_pb_ctrl.ram_playback_index      	= 0;	
    vms_pb_ctrl.aud_vms_state        		= S_aud_vms_playback_idle;  
    vms_pb_ctrl.ffs_result           		= AUD_FFS_SUCCESS;
    vms_pb_ctrl.rc_check_flag        		= FALSE;
    vms_pb_ctrl.check_header_flag    	= TRUE;
}/* End of function InitVms */

/****************************************************************************************
* Function:... vms_read_ssal_callback
* Parameters:. 
* Description: 
****************************************************************************************/
static void vms_read_ssal_callback(SINT16 result
#ifdef FFS_EXTENDED_CALLBACK
                                   , void *input_ptr
#endif
                                  )
{
    UINT16 rc_check;
    vms_pb_ctrl.ffs_result = result;
    rc_check = Aud_SSAL_result_ok(aud_glob_vms_playback.ssal_media_type,vms_pb_ctrl.ffs_result);
    if (rc_check)
      {
        vms_pb_ctrl.data_outstanding = FALSE;
        vms_pb_ctrl.data_pending++;

        if ((vms_pb_ctrl.aud_vms_state == S_aud_vms_playback_idle) ||
            (vms_pb_ctrl.aud_vms_state == S_aud_vms_wait_for_reading_header_byte) ||
            (vms_pb_ctrl.aud_vms_state == S_aud_vms_wait_for_playback) ||
            (vms_pb_ctrl.aud_vms_state == S_aud_vms_temp_state)) {
            aud_send_mail(E_timer_vms_playback_timeout, 0, 0, 0, 0, 0, 0, 0, 0, 0,NULL, NULL);
        }
     }
    else 
    {
        vms_pb_ctrl.aud_vm_error = TRUE;
    }
}

/****************************************************************************************
* Function:... enable_vms_audio_path
* Parameters:. audio_mode: The actual audio mode for which the gain cells should be set.
*              volume: The actual volume step.
* Description: Enable the earpiece, i.e. all voiceband parameters in the DSP are set          
****************************************************************************************/
static void enable_vms_audio_path(void)
{
    aud_enable_path_and_resource(aud_resource_playback_vm,S_active,FALSE);
}/* End of enable_vms_audio_path */

/****************************************************************************************
* Function:... disable_vms_audio_path
* Description: Disable the earpiece, i.e. audio part in DSP and GAIM are powered down
****************************************************************************************/
static void disable_vms_audio_path(void)
{
    aud_disable_path_and_resource(aud_resource_playback_vm,S_idle,FALSE);    
} /* End of function disable_vms_audio_path */

/****************************************************************************************
* Function:... DspStartVmPlayback
* Parameters:. 
* Description: 
****************************************************************************************/
static void DspStartVmPlayback(aud_dsp_format_enum codec_type)
{
#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_sm_vm_playback, 1, 0,               // LLT params 
                       'd', 0x03,                                               // LIS params 
                       aud_log_lis_display, aud_log_lis_display_data, aud_log_lis_trace, // LIS params 
                       (UINT32) codec_type );
#endif
    DISABLE_CLOCK;
    if (codec_type == aud_dsp_format_fr) {
        aud_dsp_vm_parms.vm_mode =  aud_dsp_vm_parms.vm_mode | 0x0002;
    } 
    if (codec_type == aud_dsp_format_amr) { /* IF1*/
        aud_dsp_vm_parms.vm_mode =  aud_dsp_vm_parms.vm_mode | 0x0200;
    }  
    if (codec_type == aud_dsp_format_amr_if2) { /* IF2*/
        aud_dsp_vm_parms.vm_mode =  aud_dsp_vm_parms.vm_mode | 0x0020; //change! for IF2 support bit5
    }  
    DSP_ASYNC_VM_CMD(&aud_dsp_vm_parms);
    ENABLE_CLOCK;
} /* End of function DspStartVmPlayback */

/****************************************************************************************
* Function:... DspStopPlaybackVm
* Parameters:. 
* Description: 
****************************************************************************************/
void DspStopPlaybackVm(void)
{
#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_sm_vm_playback, 0, 0,               // LLT params 
                       'd', 0x12,                                               // LIS params 
                       aud_log_lis_display, aud_log_lis_display_data, aud_log_lis_trace ); // LIS params 
#endif
    DISABLE_CLOCK;
    aud_dsp_vm_parms.vm_mode = aud_dsp_vm_parms.vm_mode &  0xFDD5; 
    DSP_ASYNC_VM_CMD(&aud_dsp_vm_parms);
    ENABLE_CLOCK;
} /* End of function DspStopPlaybackVm */

/****************************************************************************************
* Function   :  AudVmsPlaybackL1EventHandle
* Description: - Voice Memo Playback L1 Events handling Routine 
****************************************************************************************/
void AudVmsPlaybackL1EventHandle(aud_event_enum event)
{
// Should be removed after the reference to this function from base file is removed.
}

/****************************************************************************************
* Function   :  AudVmsPlaybackInterruptHandle
* Description: - Voice Memo Playback Interrrupt Service Routine 
****************************************************************************************/
void AudVmsPlaybackInterruptHandle(void)
{
    SINT8 rc;
    const aud_resource_enum me = aud_resource_playback_vm;
#ifdef AUD_HISR_TIMER_MEASUREMENT    
    UINT32 time_start;
    UINT32 time_end;
    time_start=RTT_stm_get_time_10us_resolution(); //Get start time
#endif

#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_hisr_vm_playback, 3, 0,               // LLT params 
                       'P', 0x06,                                               // LIS params 
                       aud_log_lis_display, aud_log_lis_display_data, aud_log_lis_llt_trace, // LIS params 
                       (UINT32) vms_pb_ctrl.aud_vms_state,
                       (UINT32) vms_pb_ctrl.aud_vm_error,
                       (UINT32) vms_pb_ctrl.vm_result );
#endif
    switch (aud_get_SM_state(me)) {
        case S_idle:
            break;

        case S_active:
            {  
                switch (vms_pb_ctrl.aud_vms_state) {
                    case S_aud_vms_playback:
                        if (vms_pb_ctrl.aud_vm_error) {
                            switch (vms_pb_ctrl.vm_result) {
                                case AUD_VM_PERFORMANCE_PROBLEMS:
                                    rc = aud_rc_performance_problems;
                                    vm_send_response_signal(rc, 0);
                                    stop_voice_memo_playback();
                                    InitVmsPlayback();              
                                    /* Allow general power down */
                                    aud_enable_standby(me);
                                    break;

                                case AUD_VM_PLAYBACK_STOPPED:
                                    rc = aud_rc_playback_stopped;
                                    vm_send_response_signal( rc, 0);
                                    stop_voice_memo_playback();
                                    InitVmsPlayback();                      
                                    /* Allow general power down */
                                    aud_enable_standby(me);               
                                    break;

                                case AUD_VM_PLAYED_TO_END: 
                                    vms_pb_ctrl.aud_vms_state =S_aud_vms_temp_state;
                                    stop_voice_memo_playback();             
                                    if (aud_glob_vms_playback.nof_repeats != 0)
                                    	{
                                        vms_pb_ctrl.lNof_repeatfactor --;
                                    	}
                                    if ( (aud_glob_vms_playback.nof_repeats == vms_pb_ctrl.lNof_repeatfactor) ||
                                        (vms_pb_ctrl.lNof_repeatfactor > 0) )
                                    {
                                        rc = aud_rc_playback_loop;
                                        vm_send_response_signal(rc, 0);
                                        InitVms_Repeat();               
                                        if (aud_glob_vms_playback.media_type == aud_media_ram)
                                       {
                                       if(!vms_pb_ctrl.data_pagememory)
                                        {
                                              aud_send_mail(E_vms_repeatevent, 0, 0, 0, 0, 0, 0, 0, 0, 0,NULL, NULL);
                                        }
                                        else
                                        {                                       
                                           /*The code has been changed to support ping pong buffer on NAND system */
                                           /* The VM playback is stopped and all the variables are reinitialized  */
                                              DspStopPlaybackVm();
                                             /* aud_send_mail(E_vms_repeatevent, 0, 0, 0, 0, 0, 0, 0, 0, 0,NULL, NULL);*/
                                               vms_intialize_playback();                  
                                          }
                                        }
                                        else if ( (aud_glob_vms_playback.media_type == aud_media_ffs) ||
                                                  	(aud_glob_vms_playback.media_type == aud_media_mmc)
#ifdef  AUD_NANDDEVICE_SUPPORT
                                        ||(aud_glob_vms_playback.media_type == aud_media_fs_cdrive )  ||
                                           (aud_glob_vms_playback.media_type == aud_media_fs_edrive )  ||
                                            (aud_glob_vms_playback.media_type == aud_media_fs_fdrive )  
#endif
                                        )    
                                        {
                                           vms_pb_ctrl.ffs_result = (aud_ffs_error_code_type)Aud_SSAL_set_file_pos(aud_glob_vms_playback.ssal_media_type,
                                                                                                                    (aud_ssal_user_type) AUD_SSAL_VR,  
                                                                                                                    (void*)aud_glob_vms_playback.file_handle,
                                                                                                                    0, &vms_setposition_ssal_callback
#ifdef FFS_EXTENDED_CALLBACK
                                                                                                                    ,NULL
#endif
                                                                                                                   );
                                        }
                                        else
                                        	{ /*Request for any other media */
                                            vm_send_response_signal(aud_rc_request_error, 0);                                 
                                        }
                                    } 
                                    else 
                                    {
                                        rc = aud_rc_playback_finish;
                                        vm_send_response_signal( rc, 0);
                                        InitVmsPlayback();    
                                        /* Allow general power down */
                                        aud_enable_standby(me);
                                    }
                                    break;

                                default:
                                    rc = aud_rc_storage_problems;
                                    vm_send_response_signal(rc, 0);
                                    stop_voice_memo_playback();
                                    InitVmsPlayback();                              
                                    /* Allow general power down */
                                    aud_enable_standby(me);

                                    break;
                            }
                        } else {
                            /* Here we send the synch signal before copying the next set of data */
                            if (aud_glob_vms_playback.media_type == aud_media_mmf) {
                                switch (vms_pb_ctrl.mmf_synch_result) {
                                    case TRUE:
                                        if (vms_pb_ctrl.sync_counter ==0)
                                            vms_pb_ctrl.sync_counter++;
                                        break;
                                    case FALSE:
                                        vm_send_response_signal( aud_rc_missing_dsp_resources, 0);
                                        break;             
                                }               
                            }
                            copy_playback_buffer();              
                        }
                        break; /*S_aud_vms_playback*/
                    case S_aud_vms_temp_state:      
                        break;
                }
            }
            break;
    }
#ifdef AUD_HISR_TIMER_MEASUREMENT    
    time_end= RTT_stm_get_time_10us_resolution();

    //check for wrap around of STM timer - happen app. each 5.8 hours
    if (time_start>time_end) {
        time_end=0xFFFFFFFF-(time_start-time_end); //store result in start_end - 32bit counter
        time_start=0;   // do not mess up reading out of result later
    }

    aud_common_trace( llt_group_audio2, llt_type_audio_timing_measurement, 3, 0,               // LLT params 
                       'P', 0x06,                                               // LIS params 
                       aud_log_lis_display, aud_log_lis_display_data, aud_log_llt_trace, // LIS params 
                       (UINT32) vms_pb_ctrl.aud_vms_state,
                       (UINT32) time_end-time_start,
                       (UINT32) E_audio_llt_type_id_2 );

#endif
}/*AudVmsPlaybackInterruptHandle*/

/****************************************************************************************
* Function:... voice_memo_getpacket
* Description: ONLY FOR TEST - MUST BE IMPLEMENTED IN APP
* Created:.... 27.05.2002 by (MT - DWD)
****************************************************************************************/
UINT16 DWD_HUGE * voice_memo_getpacket_in_ffs(UINT16 result)
{
    UINT16 DWD_HUGE *r;
    UINT16 i,j;
    UINT16 read_nof_bytes;
    UINT8 *r_ptr;

    if ((vms_pb_ctrl.data_outstanding == FALSE) && 
        (vms_pb_ctrl.data_pending == FALSE) && 
        (vms_pb_ctrl.vms_bytes_read < vms_pb_ctrl.vms_file_size)) {
        read_nof_bytes = CalcNofBytesToRead(vms_pb_ctrl.vms_bytes_read);
        vms_pb_ctrl.data_outstanding = TRUE;
        vms_pb_ctrl.ffs_result = Aud_SSAL_read((aud_ssal_media_type)aud_glob_vms_playback.ssal_media_type,
                                               (aud_ssal_user_type)AUD_SSAL_VR_READ,
                                               (void*)aud_glob_vms_playback.file_handle,
                                               (UINT8*) aud_vms_data_array[vms_pb_ctrl.array_number].aud_vms_data,
                                               read_nof_bytes,
                                               &vms_read_ssal_callback
#ifdef FFS_EXTENDED_CALLBACK
                                               ,NULL
#endif
                                              );
        if ((vms_pb_ctrl.ffs_result == AUD_FFS_SUCCESS) || (vms_pb_ctrl.ffs_result  == AUD_FS_SUCCESS)) {
            vms_pb_ctrl.vms_bytes_read += (UINT32)read_nof_bytes;    
        } else {
            vms_pb_ctrl.aud_vm_error = TRUE;
            vms_pb_ctrl.vm_result = AUD_VM_PERFORMANCE_PROBLEMS;
            vms_pb_ctrl.vms_bytes_read += (UINT32)read_nof_bytes;
            return NULL;
        }
    } else if ((vms_pb_ctrl.playback_frame + (UINT16)vms_pb_ctrl.nof_bytes_per_frame >= AUD_NOF_BYTES_IN_VMPLAYBACK_BUFFER) &&
               (vms_pb_ctrl.data_outstanding == FALSE) && (vms_pb_ctrl.data_pending == TRUE)) {
        r = (UINT16 DWD_HUGE *)aud_vms_temp_buf;
        for (i=0; i< AUD_NOF_BYTES_IN_VMPLAYBACK_BUFFER - vms_pb_ctrl.playback_frame; i++) {
            aud_vms_temp_buf[i] = aud_vms_data_array[vms_pb_ctrl.array_number_read].aud_vms_data[vms_pb_ctrl.playback_frame + i];
        }
        vms_pb_ctrl.array_number_read = !vms_pb_ctrl.array_number_read;
        for (j=0; j<vms_pb_ctrl.nof_bytes_per_frame-(AUD_NOF_BYTES_IN_VMPLAYBACK_BUFFER - vms_pb_ctrl.playback_frame); j++) {
            aud_vms_temp_buf[i] = aud_vms_data_array[vms_pb_ctrl.array_number_read].aud_vms_data[j];
            i++;
        }
        vms_pb_ctrl.playback_frame = j;
        vms_pb_ctrl.array_number =!vms_pb_ctrl.array_number;
        vms_pb_ctrl.data_pending = FALSE;
        vms_pb_ctrl.next_bytes_per_frame = aud_vms_data_array[vms_pb_ctrl.array_number_read].aud_vms_data[vms_pb_ctrl.playback_frame];
        vms_pb_ctrl.vms_bytes_played += (UINT16)vms_pb_ctrl.nof_bytes_per_frame;

        if (vms_pb_ctrl.vms_bytes_played >= vms_pb_ctrl.vms_file_size) {
            vms_pb_ctrl.aud_vm_error = TRUE;
            vms_pb_ctrl.vm_result = AUD_VM_PLAYED_TO_END;
        } else {
            read_nof_bytes = CalcNofBytesToRead(vms_pb_ctrl.vms_bytes_read);    
            vms_pb_ctrl.data_outstanding = TRUE;
            vms_pb_ctrl.ffs_result = Aud_SSAL_read((aud_ssal_media_type)aud_glob_vms_playback.ssal_media_type,
                                                   (aud_ssal_user_type)AUD_SSAL_VR_READ,
                                                   (void*)aud_glob_vms_playback.file_handle,
                                                   (UINT8*) aud_vms_data_array[vms_pb_ctrl.array_number].aud_vms_data,
                                                   read_nof_bytes,
                                                   &vms_read_ssal_callback
#ifdef FFS_EXTENDED_CALLBACK
                                                   ,NULL
#endif
                                                  );
            if ((vms_pb_ctrl.ffs_result == AUD_FFS_SUCCESS) || (vms_pb_ctrl.ffs_result  == AUD_FS_SUCCESS)) {
                vms_pb_ctrl.vms_bytes_read += (UINT32)read_nof_bytes;    
                return r;
            } else {
                vms_pb_ctrl.aud_vm_error = TRUE;
                vms_pb_ctrl.vm_result = AUD_VM_PERFORMANCE_PROBLEMS;
                vms_pb_ctrl.vms_bytes_read += (UINT32)read_nof_bytes;
                return NULL;
            }
        }
    } else if ((vms_pb_ctrl.playback_frame+(UINT16)vms_pb_ctrl.nof_bytes_per_frame >= AUD_NOF_BYTES_IN_VMPLAYBACK_BUFFER) && 
               (vms_pb_ctrl.data_outstanding == TRUE)) {
        /*STOP and call back the mmi*/
        vms_pb_ctrl.vm_result = AUD_VM_PERFORMANCE_PROBLEMS;
        vms_pb_ctrl.aud_vm_error = TRUE;
        return NULL;
    }

    r_ptr = (&aud_vms_data_array[vms_pb_ctrl.array_number_read].aud_vms_data[vms_pb_ctrl.playback_frame]);
    r = (UINT16 *)r_ptr;

    vms_pb_ctrl.playback_frame += vms_pb_ctrl.nof_bytes_per_frame;
    vms_pb_ctrl.vms_bytes_played += (UINT16)vms_pb_ctrl.nof_bytes_per_frame;

    if (vms_pb_ctrl.vms_bytes_played >= vms_pb_ctrl.vms_file_size) {
        vms_pb_ctrl.aud_vm_error = TRUE;
        vms_pb_ctrl.vm_result = AUD_VM_PLAYED_TO_END;
    } else {
        if (vms_pb_ctrl.playback_frame > AUD_NOF_BYTES_IN_VMPLAYBACK_BUFFER-1)
            vms_pb_ctrl.next_bytes_per_frame = aud_vms_data_array[!vms_pb_ctrl.array_number_read].aud_vms_data[0];
        else
            vms_pb_ctrl.next_bytes_per_frame = aud_vms_data_array[vms_pb_ctrl.array_number_read].aud_vms_data[vms_pb_ctrl.playback_frame];
    }  
    return r;
}/* End of function voice_memo_getpacket */

/****************************************************************************************
* Function:... voice_memo_getpacket_in_ram
* Parameters:. 
* Description: 
****************************************************************************************/
UINT16 DWD_HUGE * voice_memo_getpacket_in_ram(UINT16 result)
{
   UINT8 *ptr;
   UINT16 DWD_HUGE *p;
   UINT16 i,j;
   UINT16 read_nof_bytes;

    if(!vms_pb_ctrl.data_pagememory)
    {
        ptr =  &((UINT8 DWD_HUGE *)aud_glob_vms_playback.file_handle)[vms_pb_ctrl.playback_frame]; 
        p = (UINT16 *)ptr;
        vms_pb_ctrl.playback_frame += vms_pb_ctrl.nof_bytes_per_frame;
        vms_pb_ctrl.vms_bytes_played += (UINT16)vms_pb_ctrl.nof_bytes_per_frame;
        vms_pb_ctrl.next_bytes_per_frame = ((UINT8 DWD_HUGE *)aud_glob_vms_playback.file_handle)[vms_pb_ctrl.playback_frame];
        if (vms_pb_ctrl.playback_frame >= aud_glob_vms_playback.buffer_size)
        {
            vms_pb_ctrl.aud_vm_error = TRUE;
            vms_pb_ctrl.vm_result = AUD_VM_PLAYED_TO_END;
         }
        return p;
    }
    else
    {
        if ((vms_pb_ctrl.ram_data_outstanding == FALSE) && 
             (vms_pb_ctrl.ram_data_pending == FALSE) && 
             (vms_pb_ctrl.vms_bytes_read < vms_pb_ctrl.vms_file_size)) 
             {
                read_nof_bytes = CalcNofBytesToRead(vms_pb_ctrl.vms_bytes_read);
                vms_pb_ctrl.ram_data_outstanding = TRUE;
                aud_vms_playback_ram_readrequest((UINT8*) aud_vms_data_array[vms_pb_ctrl.array_number].aud_vms_data,read_nof_bytes );
                vms_pb_ctrl.vms_bytes_read += (UINT32)read_nof_bytes;    
              }
         else if ((vms_pb_ctrl.playback_frame + (UINT16)vms_pb_ctrl.nof_bytes_per_frame >= AUD_NOF_BYTES_IN_VMPLAYBACK_BUFFER) &&
                    (vms_pb_ctrl.ram_data_outstanding == FALSE) && (vms_pb_ctrl.ram_data_pending == TRUE)) 
         {
            p = (UINT16 DWD_HUGE *)aud_vms_temp_buf;
            for (i=0; i< AUD_NOF_BYTES_IN_VMPLAYBACK_BUFFER - vms_pb_ctrl.playback_frame; i++) 
            {
                aud_vms_temp_buf[i] = aud_vms_data_array[vms_pb_ctrl.array_number_read].aud_vms_data[vms_pb_ctrl.playback_frame + i];
            }
            vms_pb_ctrl.array_number_read = !vms_pb_ctrl.array_number_read;
            for (j=0; j<vms_pb_ctrl.nof_bytes_per_frame-(AUD_NOF_BYTES_IN_VMPLAYBACK_BUFFER - vms_pb_ctrl.playback_frame); j++) 
            {
                aud_vms_temp_buf[i] = aud_vms_data_array[vms_pb_ctrl.array_number_read].aud_vms_data[j];
                i++;
            }
            vms_pb_ctrl.playback_frame = j;
            vms_pb_ctrl.array_number =!vms_pb_ctrl.array_number;
            vms_pb_ctrl.ram_data_pending = FALSE;
            vms_pb_ctrl.next_bytes_per_frame = aud_vms_data_array[vms_pb_ctrl.array_number_read].aud_vms_data[vms_pb_ctrl.playback_frame];
            vms_pb_ctrl.vms_bytes_played += (UINT16)vms_pb_ctrl.nof_bytes_per_frame;
            if (vms_pb_ctrl.vms_bytes_played >= vms_pb_ctrl.vms_file_size)
            {
                vms_pb_ctrl.aud_vm_error = TRUE;
                vms_pb_ctrl.vm_result = AUD_VM_PLAYED_TO_END;
            }
            else
            {
                read_nof_bytes = CalcNofBytesToRead(vms_pb_ctrl.vms_bytes_read);    
                vms_pb_ctrl.ram_data_outstanding = TRUE;
                aud_vms_playback_ram_readrequest((UINT8*) aud_vms_data_array[vms_pb_ctrl.array_number].aud_vms_data, read_nof_bytes);
                vms_pb_ctrl.vms_bytes_read += (UINT32)read_nof_bytes;    
                return p;
             }
         } 
         else if ((vms_pb_ctrl.playback_frame+(UINT16)vms_pb_ctrl.nof_bytes_per_frame >= AUD_NOF_BYTES_IN_VMPLAYBACK_BUFFER) && 
                (vms_pb_ctrl.ram_data_outstanding == TRUE)) 
         {
               /*STOP and call back the mmi*/
              vms_pb_ctrl.vm_result = AUD_VM_PERFORMANCE_PROBLEMS;
              vms_pb_ctrl.aud_vm_error = TRUE;
              return NULL;
         }
         ptr = (&aud_vms_data_array[vms_pb_ctrl.array_number_read].aud_vms_data[vms_pb_ctrl.playback_frame]);
         p = (UINT16 *)ptr;
         vms_pb_ctrl.playback_frame += vms_pb_ctrl.nof_bytes_per_frame;
         vms_pb_ctrl.vms_bytes_played += (UINT16)vms_pb_ctrl.nof_bytes_per_frame;
         if (vms_pb_ctrl.vms_bytes_played >= vms_pb_ctrl.vms_file_size) 
         {
            vms_pb_ctrl.aud_vm_error = TRUE;
            vms_pb_ctrl.vm_result = AUD_VM_PLAYED_TO_END;
         }
         else 
         {
            if (vms_pb_ctrl.playback_frame > AUD_NOF_BYTES_IN_VMPLAYBACK_BUFFER-1)
            {
                vms_pb_ctrl.next_bytes_per_frame = aud_vms_data_array[!vms_pb_ctrl.array_number_read].aud_vms_data[0];
             }
            else
            {  
                vms_pb_ctrl.next_bytes_per_frame = aud_vms_data_array[vms_pb_ctrl.array_number_read].aud_vms_data[vms_pb_ctrl.playback_frame];
             }
          }  
         return p;
    	} 
}/* End of function voice_memo_getpacket */

/****************************************************************************************
* Function:... aud_vms_playback_ram_readrequest
* Description: Initiate the buffer copy from the file handle during HISR context.
****************************************************************************************/
static void aud_vms_playback_ram_readrequest(
                UINT8 * p_dest, /*Destination pointer to be copied */
                UINT32 buf_size /*Buffer size */
                )
{
    aud_send_mail(E_vms_pb_ram_data_ready, 0, buf_size,0,0,0,0,0,0, 0, NULL,p_dest); 
}
/****************************************************************************************
* Function:... aud_vms_playback_ram_read
* Description: copy the data from the application file handle to the local ping pong buffer.
****************************************************************************************/
static void aud_vms_playback_ram_read(
                UINT8 *p_dest,    /*Destination pointer to be copied */
                UINT32 buf_size  /*Buffer size */
                )
{
    UINT8 *p_src;
    p_src = &((UINT8 DWD_HUGE *)aud_glob_vms_playback.file_handle)[vms_pb_ctrl.ram_playback_index],
    memcpy(p_dest, p_src, buf_size);
    vms_pb_ctrl.ram_playback_index += buf_size;
    vms_pb_ctrl.ram_data_outstanding = FALSE;
    vms_pb_ctrl.ram_data_pending = TRUE;
}

/********************************************************************************************
* Function:... aud_vms_playback_ram_init_datacopy()
* Parameters:. result: The File pointer and buffer point to which the data has to be copied and also the buffer size
* Description: This function is called to copy data from the application file pointer to the first local buffer. This function
*			is called during initialisation of the playback, set play position
*********************************************************************************************/
static void aud_vms_playback_ram_init_datacopy(void)
{
   UINT16 read_nof_bytes;
   UINT8 DWD_HUGE *p_src, *p_dest;
   vms_pb_ctrl.ram_data_outstanding = FALSE;
   vms_pb_ctrl.ram_data_pending = FALSE;
   read_nof_bytes      = CalcNofBytesToRead(vms_pb_ctrl.vms_bytes_read);    
   p_src = &((UINT8 DWD_HUGE *)aud_glob_vms_playback.file_handle)[vms_pb_ctrl.ram_playback_index]; 
   p_dest = &aud_vms_data_array[vms_pb_ctrl.array_number].aud_vms_data[0];
   memcpy(p_dest, p_src, read_nof_bytes);
   vms_pb_ctrl.ram_playback_index +=  read_nof_bytes;
   vms_pb_ctrl.array_number   = ! vms_pb_ctrl.array_number;
   vms_pb_ctrl.vms_bytes_read+= (UINT32)read_nof_bytes;   
}

/****************************************************************************************
* Function:... mmf_null_handle_function
* Description: Handling NULL pointer during Streaming
****************************************************************************************/
static UINT16 DWD_HUGE *mmf_null_handle_function(void)
{
    return((UINT16 *)aud_vms_null_array);
}

/****************************************************************************************
* Function:... vms_streaming_buffer_function
* Parameters:. 
* Description: 
****************************************************************************************/
static UINT16 DWD_HUGE *vms_streaming_buffer_function(void)
{
    UINT8 *aud_ptr = NULL;
    UINT16 size;

    (*aud_glob_vms_playback.get_buffer_func)(&aud_ptr, &size, aud_glob_vms_playback.mmf_user_data);
    vms_pb_ctrl.playback_frame = 0;       

    if ((size == 0) && (aud_ptr != NULL)) {
        vms_pb_ctrl.aud_vm_error = TRUE;
        vms_pb_ctrl.vm_result    = AUD_VM_PLAYBACK_STOPPED;
        aud_glob_vms_playback.buffer_size  = 0;
        vms_pb_ctrl.mmf_synch_result = FALSE;
        aud_glob_vms_playback.file_handle      = mmf_null_handle_function();
    } else {
        if (aud_ptr == NULL) {
            aud_glob_vms_playback.file_handle = mmf_null_handle_function();
            aud_glob_vms_playback.buffer_size = 0;
        } else {
            /* Normal case - data is ready from application */
            aud_glob_vms_playback.buffer_size  = size;
            vms_pb_ctrl.mmf_synch_result = TRUE;
            aud_glob_vms_playback.file_handle = (UINT16 DWD_HUGE *)aud_ptr;
//            vms_pb_ctrl.vms_bytes_played = 0;         
        }
    }
    return((UINT16 DWD_HUGE *)aud_ptr);
}

/****************************************************************************************
* Function:... voice_memo_getpacket_in_mmf
* Description: ONLY FOR TEST - MUST BE IMPLEMENTED IN APP
****************************************************************************************/
UINT16 DWD_HUGE * voice_memo_getpacket_in_mmf(UINT16 result)
{
    UINT8 DWD_HUGE *p;

#ifdef AUDIO_LLT
    aud_common_trace( llt_group_audio, llt_type_audio_sm_vm_playback, 0, 0,               // LLT params 
                       'm', 0x0e,                                               // LIS params 
                       aud_log_lis_display, aud_log_lis_display_nodata, aud_log_lis_trace ); // LIS params 
#endif
    vms_pb_ctrl.mmf_synch_result = TRUE;

    /* Check if header and needed bytes are available in the buffer */
    if (vms_pb_ctrl.playback_frame < aud_glob_vms_playback.buffer_size) {
        /* Minimum one byte in the buffer reamains to be played */
        /* Find the size of the frame */
        find_nof_bytes_per_frame(((UINT8 DWD_HUGE *)aud_glob_vms_playback.file_handle)[vms_pb_ctrl.playback_frame], NULL);    

        /* Check if the buffer cointains enough bytes for this frame */
        if ((vms_pb_ctrl.playback_frame + vms_pb_ctrl.nof_bytes_per_frame) > aud_glob_vms_playback.buffer_size) {
            /* we need more data to play this frame */
            /* First store the data we already have in temp local buffer */
            p = &((UINT8 DWD_HUGE *)aud_glob_vms_playback.file_handle)[vms_pb_ctrl.playback_frame];
            vms_pb_ctrl.vms_offset = aud_glob_vms_playback.buffer_size - vms_pb_ctrl.playback_frame;
            memcpy(aud_vms_temp_buf, (void *)p, vms_pb_ctrl.vms_offset);
            vms_streaming_buffer_function();

            /* Check if we got what we wanted (enough data) */
            if (vms_pb_ctrl.vms_offset < aud_glob_vms_playback.buffer_size) {
                memcpy(&aud_vms_temp_buf[vms_pb_ctrl.vms_offset], (void *)aud_glob_vms_playback.file_handle,
                       (vms_pb_ctrl.nof_bytes_per_frame - vms_pb_ctrl.vms_offset) );
                vms_pb_ctrl.playback_frame       = vms_pb_ctrl.nof_bytes_per_frame - vms_pb_ctrl.vms_offset;
                vms_pb_ctrl.next_bytes_per_frame = ((UINT8 DWD_HUGE *)aud_glob_vms_playback.file_handle)[vms_pb_ctrl.playback_frame];
                vms_pb_ctrl.vms_bytes_played    += (UINT16)vms_pb_ctrl.nof_bytes_per_frame;
                return((UINT16 DWD_HUGE *)aud_vms_temp_buf);
            } else {
                /* There is not enough data to play the frame - discard
                   all data and play "comfort noise" instead */
                aud_glob_vms_playback.file_handle = mmf_null_handle_function();
                aud_glob_vms_playback.buffer_size = 0;
                p = &((UINT8 DWD_HUGE *)aud_glob_vms_playback.file_handle)[vms_pb_ctrl.playback_frame];
                return((UINT16 DWD_HUGE *)p);
            }
        } else {
            /* There is enough bytes to play this frame */
            p = &((UINT8 DWD_HUGE *)aud_glob_vms_playback.file_handle)[vms_pb_ctrl.playback_frame];
            vms_pb_ctrl.playback_frame   += vms_pb_ctrl.nof_bytes_per_frame;
            vms_pb_ctrl.vms_bytes_played += vms_pb_ctrl.nof_bytes_per_frame;
            
            return((UINT16 DWD_HUGE *)p);
        }
    } else {
        /* Exactly all bytes in the buffer has been played. */
        /* Get the next buffer */
        vms_streaming_buffer_function();            
        find_nof_bytes_per_frame(((UINT8 DWD_HUGE *)aud_glob_vms_playback.file_handle)[vms_pb_ctrl.playback_frame],NULL);    

        /* Check if the application provided enough data for a full frame */
        if (vms_pb_ctrl.nof_bytes_per_frame <= aud_glob_vms_playback.buffer_size) {
            /* Yes, we can play a frame */
            p = &((UINT8 DWD_HUGE *)aud_glob_vms_playback.file_handle)[vms_pb_ctrl.playback_frame];
            vms_pb_ctrl.playback_frame   += vms_pb_ctrl.nof_bytes_per_frame;
            vms_pb_ctrl.vms_bytes_played += vms_pb_ctrl.nof_bytes_per_frame;
            return((UINT16 DWD_HUGE *)p);
        } else {
            /*application did not provide enough data 
              to play a frame - discard the data (if any) and play 
              "comfort noise" instead */
            aud_glob_vms_playback.file_handle = mmf_null_handle_function();
            aud_glob_vms_playback.buffer_size = 0;
            p = &((UINT8 DWD_HUGE *)aud_glob_vms_playback.file_handle)[vms_pb_ctrl.playback_frame];
            return((UINT16 DWD_HUGE *)p);
        }
    }
}/* End of function voice_memo_getpacket_MMF */

/****************************************************************************************
* Function:... InitVms_Repeat
* Parameters:. 
* Description: 
****************************************************************************************/
static void InitVms_Repeat(void)
{
    memset(aud_vms_data_array, 0,sizeof(aud_vms_data_array));
    aud_glob_vms_playback.vm_mode	= aud_vm_mode_standby;
    aud_glob_vms_playback.pos        	= 0;
    vms_pb_ctrl.vm_result            		= 0;
    vms_pb_ctrl.array_number         		= 0;
    vms_pb_ctrl.mmf_data             		= FALSE;
    vms_pb_ctrl.data_outstanding     		= FALSE;
    vms_pb_ctrl.data_pending         		= FALSE;
    vms_pb_ctrl.memo_end             		= FALSE;
    vms_pb_ctrl.aud_vm_error         		= FALSE;  
    vms_pb_ctrl.check_header_flag    	= TRUE;
    vms_pb_ctrl.sync_counter         		= 0;
    vms_pb_ctrl.vms_bytes_played     	=
    vms_pb_ctrl.vms_bytes_read       		=
    vms_pb_ctrl.playback_frame       		= vms_pb_ctrl.vms_header_size + aud_glob_vms_playback.start_offset;
    vms_pb_ctrl.vms_frames_played    	= 0;
    vms_pb_ctrl.array_number_read    	= 0;
    vms_pb_ctrl.nof_bytes_per_frame  	= 0;
    vms_pb_ctrl.next_bytes_per_frame 	= 0;
    vms_pb_ctrl.mmf_synch_result     	= TRUE;
    vms_pb_ctrl.aud_vms_state        		= S_aud_vms_playback_idle;  
    vms_pb_ctrl.ffs_result           		= AUD_FFS_SUCCESS;
    vms_pb_ctrl.vms_offset           		= 0;
    vms_pb_ctrl.ptr_vms_header       		= 0;  
}

/****************************************************************************************
* Function:... StopVmSuspendData
* Parameters:. 
* Description: 
****************************************************************************************/
static void StopVmSuspendData(UINT8 slot_id)
{
    aud_glob_vms_playback.suspend_data[slot_id].in_use            	 	= FALSE;
    aud_glob_vms_playback.suspend_data[slot_id].media              	 	= 0;
    aud_glob_vms_playback.suspend_data[slot_id].vm_mode             	= (aud_vm_mode_enum)0;
    aud_glob_vms_playback.suspend_data[slot_id].file_handle        		= 0;
    aud_glob_vms_playback.suspend_data[slot_id].vms_getpacket_func 	= NULL;
    aud_glob_vms_playback.suspend_data[slot_id].buffer_size        		= 0;
    aud_glob_vms_playback.suspend_data[slot_id].get_buffer_func    	= NULL;
    aud_glob_vms_playback.suspend_data[slot_id].format             		= (aud_dsp_format_enum)0;
    aud_glob_vms_playback.suspend_data[slot_id].frame_number      	= 0;
    aud_glob_vms_playback.suspend_data[slot_id].vms_file_size      		= 0;
    aud_glob_vms_playback.suspend_data[slot_id].play_time          		= 0;
    aud_glob_vms_playback.suspend_data[slot_id].vms_header_size    	= 0;
    aud_glob_vms_playback.suspend_data[slot_id].vms_bytes_played   	= 0;
    aud_glob_vms_playback.suspend_data[slot_id].nof_repeats        		= 0;
    aud_glob_vms_playback.suspend_data[slot_id].start_offset       		= 0;
    aud_glob_vms_playback.suspend_data[slot_id].lNof_repeatfactor  	= 0;
}

/****************************************************************************************
* Function:... SaveVmSuspendData
* Description: Disable the earpiece, i.e. audio part in DSP and GAIM are powered down
****************************************************************************************/
static void SaveVmSuspendData(UINT8 slot_id)
{
    aud_glob_vms_playback.suspend_data[slot_id].in_use                			= TRUE;
    aud_glob_vms_playback.suspend_data[slot_id].media                 		= aud_glob_vms_playback.media_type;
    aud_glob_vms_playback.suspend_data[slot_id].vm_mode               		= aud_glob_vms_playback.vm_mode;
    aud_glob_vms_playback.suspend_data[slot_id].file_handle           		= aud_glob_vms_playback.file_handle;
    aud_glob_vms_playback.suspend_data[slot_id].vms_getpacket_func   		= aud_glob_vms_playback.fp_voice_memo_getpacket;
    aud_glob_vms_playback.suspend_data[slot_id].buffer_size           		= aud_glob_vms_playback.buffer_size;
    aud_glob_vms_playback.suspend_data[slot_id].get_buffer_func       		= aud_glob_vms_playback.get_buffer_func ;
    aud_glob_vms_playback.suspend_data[slot_id].format                			= aud_glob_vms_playback.format;  
    aud_glob_vms_playback.suspend_data[slot_id].frame_number          		= vms_pb_ctrl.vms_frames_played;
    aud_glob_vms_playback.suspend_data[slot_id].vms_file_size         		= vms_pb_ctrl.vms_file_size;    
    aud_glob_vms_playback.suspend_data[slot_id].play_time             		= vms_pb_ctrl.vms_total_playtime;
    aud_glob_vms_playback.suspend_data[slot_id].vms_header_size        	= vms_pb_ctrl.vms_header_size;  
    aud_glob_vms_playback.suspend_data[slot_id].vms_bytes_played      	= vms_pb_ctrl.vms_bytes_played;  
    aud_glob_vms_playback.suspend_data[slot_id].nof_repeats           		= aud_glob_vms_playback.nof_repeats;
    aud_glob_vms_playback.suspend_data[slot_id].start_offset          		= aud_glob_vms_playback.start_offset;
    aud_glob_vms_playback.suspend_data[slot_id].lNof_repeatfactor     		= vms_pb_ctrl.lNof_repeatfactor;
    aud_glob_vms_playback.suspend_data[slot_id].playback_frame        		= vms_pb_ctrl.playback_frame;
    aud_glob_vms_playback.suspend_data[slot_id].data_pagememory_status   = vms_pb_ctrl.data_pagememory;

}

/****************************************************************************************
* Function:... RestoreVmExternal_SuspendData
* Parameters:. 
* Description: 
****************************************************************************************/
static void RestoreVmExternal_SuspendData(UINT8 slot_id)
{
    aud_glob_vms_playback.suspend_data[slot_id].in_use 		= FALSE;
    aud_glob_vms_playback.media_type                   			= (aud_media_enum)aud_glob_vms_playback.suspend_data[slot_id].media;
    aud_glob_vms_playback.vm_mode                      			= aud_glob_vms_playback.suspend_data[slot_id].vm_mode;
    aud_glob_vms_playback.file_handle                  			= aud_glob_vms_playback.suspend_data[slot_id].file_handle;
    aud_glob_vms_playback.format                       			= aud_glob_vms_playback.suspend_data[slot_id].format;
    aud_glob_vms_playback.fp_voice_memo_getpacket      		= aud_glob_vms_playback.suspend_data[slot_id].vms_getpacket_func;  
    aud_glob_vms_playback.get_buffer_func              			= aud_glob_vms_playback.suspend_data[slot_id].get_buffer_func;
    aud_glob_vms_playback.buffer_size                  			= aud_glob_vms_playback.suspend_data[slot_id].buffer_size;
    aud_glob_vms_playback.nof_repeats                  			= aud_glob_vms_playback.suspend_data[slot_id].nof_repeats ; 
    aud_glob_vms_playback.start_offset                 			= aud_glob_vms_playback.suspend_data[slot_id].start_offset;
}

/****************************************************************************************
* Function:... RestoreVmInternal_SuspendData
* Parameters:. 
* Description: 
****************************************************************************************/
static void RestoreVmInternal_SuspendData(UINT8 slot_id)
{
    vms_pb_ctrl.vms_frames_played		= aud_glob_vms_playback.suspend_data[slot_id].frame_number;  
    vms_pb_ctrl.vms_file_size            	= aud_glob_vms_playback.suspend_data[slot_id].vms_file_size ;
    vms_pb_ctrl.vms_total_playtime   	= aud_glob_vms_playback.suspend_data[slot_id].play_time;  
    vms_pb_ctrl.vms_header_size         	= aud_glob_vms_playback.suspend_data[slot_id].vms_header_size;  
    vms_pb_ctrl.lNof_repeatfactor       	= aud_glob_vms_playback.suspend_data[slot_id].lNof_repeatfactor;
    vms_pb_ctrl.vms_bytes_played   		= aud_glob_vms_playback.suspend_data[slot_id].vms_bytes_played ;
    vms_pb_ctrl.playback_frame 		= aud_glob_vms_playback.suspend_data[slot_id].playback_frame;
    vms_pb_ctrl.data_pagememory 		= aud_glob_vms_playback.suspend_data[slot_id].data_pagememory_status;
}

/****************************************************************************************
* Function:... vms_setposition_ssal_callback
* Parameters:. FFS result 
* Description: This is a callback function invoked by FFS once the FFS getpositon has been State machine for the vms handling
****************************************************************************************/
static void vms_setposition_ssal_callback(SINT16 result
#ifdef FFS_EXTENDED_CALLBACK
                                          ,void *inptr
#endif
                                         )
{
    UINT16 rc_check;
    vms_pb_ctrl.ffs_result = result;
    rc_check = Aud_SSAL_result_ok(aud_glob_vms_playback.ssal_media_type, vms_pb_ctrl.ffs_result);
    if (!rc_check)
        vms_pb_ctrl.aud_vm_error = TRUE;
    else
        aud_send_mail(E_vms_repeatevent, 0, 0, 0, 0, 0, 0, 0, 0, 0,NULL, NULL);
}

/***********************************************************************
* NAME   :  vm_send_response_signal
* INPUT  :  the state for voicememo
* DESCR. :  handles internal voice memo 
************************************************************************/
static void vm_send_response_signal(SINT8 rc, UINT32 value)
{

#ifdef IFWD_AUD_NEW_RC_PARAMETER
UINT32 FramesPlayed =0;
UINT32 BytesPlayed = 0;
UINT32 pos=0;
#endif
    switch (rc) {
        case aud_rc_storage_problems:    
        case aud_rc_playback_started:
        case aud_rc_performance_problems:
        case aud_rc_ram_buffer_used:
        case aud_rc_playback_loop:
        case aud_rc_parameter_out_of_range: 
        case aud_rc_unknown_position:
        case aud_rc_suspend_resume_error:
        case aud_rc_missing_dsp_resources:
        case aud_rc_total_playtime:
        case aud_rc_request_error:
        case aud_rc_ok:    
            aud_send_response_signal(aud_glob_vms_playback.sender, aud_resource_playback_vm, rc, value, 0, 0, 0, NULL, NULL);
            break;

        case aud_rc_playback_finish:   
        case aud_rc_elapsed_time:
#ifdef IFWD_AUD_NEW_RC_PARAMETER
	              pos = FramesPlayed=vms_pb_ctrl.vms_frames_played;
                     BytesPlayed = vms_pb_ctrl.vms_bytes_played;
	              pos *= 20;
  aud_send_response_signal(aud_glob_vms_playback.sender, aud_resource_playback_vm, rc, pos,FramesPlayed,
	 	 BytesPlayed, 0, NULL, NULL);
  #else
    aud_send_response_signal(aud_glob_vms_playback.sender, aud_resource_playback_vm, rc, value, 0, 0, 0, NULL, NULL);
  #endif
			
			break;
	 case aud_rc_playback_suspended:
	 	if (aud_glob_vms_playback.media_type == aud_media_mmf)
	 		{
	#ifdef IFWD_AUD_NEW_RC_PARAMETER
	              pos = FramesPlayed=vms_pb_ctrl.vms_frames_played;
	              BytesPlayed = vms_pb_ctrl.vms_bytes_played;
	              pos *= 20;
	 	aud_send_response_signal(aud_glob_vms_playback.sender,	aud_resource_playback_vm,rc,pos,
	 	FramesPlayed, BytesPlayed,0, NULL, NULL);
	#else
			
		 	aud_send_response_signal(aud_glob_vms_playback.sender,	aud_resource_playback_vm,rc,vms_pb_ctrl.vms_frames_played, vms_pb_ctrl.vms_bytes_played,
 			 0,  0, NULL, NULL);
	#endif
			
	 		}
		else
			{
			#ifdef IFWD_AUD_NEW_RC_PARAMETER
	              pos =FramesPlayed= vms_pb_ctrl.vms_frames_played;
			  BytesPlayed = vms_pb_ctrl.vms_bytes_played;
	              pos *= 20;
	            aud_send_response_signal(aud_glob_vms_playback.sender, aud_resource_playback_vm, rc, pos,
	 	FramesPlayed, BytesPlayed, 0, NULL, NULL);
				#else
  			aud_send_response_signal(aud_glob_vms_playback.sender, aud_resource_playback_vm, rc, value, 0, 0, 0, NULL, NULL);
			#endif
			}
				
		break;		
		
        case aud_rc_playback_stopped:
		if (aud_glob_vms_playback.media_type == aud_media_mmf)
			{
			#ifdef IFWD_AUD_NEW_RC_PARAMETER
	           pos =FramesPlayed= vms_pb_ctrl.vms_frames_played;
	              pos *= 20;
		    aud_send_response_signal(aud_glob_vms_playback.sender, aud_resource_playback_vm,rc, 
	            pos, FramesPlayed,vms_pb_ctrl.vms_bytes_played, 0, NULL, NULL);
					
				  #else
	            aud_send_response_signal(aud_glob_vms_playback.sender, aud_resource_playback_vm,rc, 
	            vms_pb_ctrl.vms_frames_played, vms_pb_ctrl.vms_bytes_played, 0,  AUD_STOPPED_BY_APP, NULL, NULL);
			#endif				  
			}
		else
			{
			#ifdef IFWD_AUD_NEW_RC_PARAMETER
	              pos =FramesPlayed= vms_pb_ctrl.vms_frames_played;
			BytesPlayed = vms_pb_ctrl.vms_bytes_played;
	              pos *= 20;
			aud_send_response_signal(aud_glob_vms_playback.sender, aud_resource_playback_vm,rc,
				pos,FramesPlayed, BytesPlayed,0, NULL, NULL);
			#else
		       aud_send_response_signal(aud_glob_vms_playback.sender, aud_resource_playback_vm,rc, value, 0, 0,  AUD_STOPPED_BY_APP, NULL, NULL);
        		#endif
			}
            break;
    }
}


#ifdef VM_MMF_DATA_VERIFY
/****************************************************************************************
* Function:... vm_data_verify_init
* Description:  Initialisation routine to verify MMF data.
****************************************************************************************/
static void vm_data_verify_init()
{
  aud_ffs_error_code_type error_type,rc; 
  UINT8 *ptr = NULL;
  
  if(aud_glob_vms_playback.media_type == aud_media_mmf)
  {
      aud_glob_vms_playback.file_created = TRUE;
      aud_glob_vms_playback.close_file = FALSE;
      error_type =  FFS_create_fn(AUD_FFS_VR, "vm_data_verify",&aud_glob_vms_playback.data_ver_file_handle,FFS_FILETYPE_AMR,AUD_FFS_STREAMING,NULL);
      if(error_type == AUD_FFS_SUCCESS || error_type == AUD_FFS_EXISTS)
      {
          aud_glob_vms_playback.FFS_data_outstanding = FALSE;
          aud_glob_vms_playback.ssal_media_type_mine= aud_media_type_to_ssal_type(aud_media_ffs); 
      } else
      {
          aud_glob_vms_playback.file_created = FALSE;  // Error in creating the file.
      }
      if(aud_glob_vms_playback.file_created == TRUE)
      {
          aud_glob_vms_playback.file_id_created = TRUE;
          rc = Aud_FFS_get_file_id_from_handle(AUD_FFS_VR, (UINT16 *)aud_glob_vms_playback.data_ver_file_handle, &aud_glob_vms_playback.data_ver_file_id);
          if (rc != AUD_FFS_SUCCESS)  
              aud_glob_vms_playback.file_id_created = FALSE;
      }

      ptr = (UINT8 *)aud_glob_vms_playback.file_handle;
      ptr += aud_glob_vms_playback.start_offset;
      vm_next_location = find_header_info_playback(ptr);
  }
}

/****************************************************************************************
* Function:... write_to_ffs_file
* Description:  Writing MMF data into the FFS file.
****************************************************************************************/
void write_vm_data_to_ffs_file(UINT8 buffer_selector, UINT16 size)
{
  static UINT8 Error  =0;
  static UINT16 Counter_inside=0;
  SINT16 Ffs_result; 

  if(Error == FALSE && aud_glob_vms_playback.file_created == TRUE && aud_glob_vms_playback.FFS_data_outstanding == FALSE) 
  {
      aud_glob_vms_playback.FFS_data_outstanding = TRUE;
      //FFS replacement 
      Ffs_result = Aud_SSAL_write((aud_ssal_media_type)aud_glob_vms_playback.ssal_media_type_mine, (aud_ssal_user_type)AUD_SSAL_VR_WRITE, (void*)aud_glob_vms_playback.data_ver_file_handle, (UINT8*)validate_vm[buffer_selector].VM_data,size, &vms_write_ssal_callback_ver_vm
#ifdef FFS_EXTENDED_CALLBACK 
                                  , NULL
#endif
                                   );
      if (!((Ffs_result == AUD_FFS_SUCCESS)||(Ffs_result == AUD_FS_SUCCESS)))
           Error = TRUE;
  } 
}

/****************************************************************************************
* Function:... vms_write_ssal_callback_ver_vm
* Description:  Call back func called by FFS after writing the data to the media.
****************************************************************************************/
void vms_write_ssal_callback_ver_vm(SINT16 result
#ifdef FFS_EXTENDED_CALLBACK
                                               ,void *input
#endif
)
{  
    UINT16 rc_check;
    aud_ffs_error_code_type rc;
    SINT16 ffs_result = result;

    rc_check  = Aud_SSAL_result_ok(aud_glob_vms_playback.ssal_media_type_mine, ffs_result);
    if(rc_check )
    { 
        aud_glob_vms_playback.FFS_data_outstanding= FALSE;
        if(aud_glob_vms_playback.media_type == aud_media_mmf && aud_glob_vms_playback.file_created == TRUE &&  aud_glob_vms_playback.file_id_created == TRUE && aud_glob_vms_playback.close_file == TRUE)
        {
            rc = Aud_FFS_close(AUD_FFS_VR, aud_glob_vms_playback.data_ver_file_id, (UINT16 *)aud_glob_vms_playback.data_ver_file_handle, NULL);
            if(rc == AUD_FFS_SUCCESS)
            {
                aud_glob_vms_playback.close_file = FALSE;
                vm_next_location = 0;
                vm_write_index = 0;
                aud_glob_vms_playback.file_created = FALSE;
                aud_glob_vms_playback.file_id_created = FALSE;
                memset(validate_vm[0].VM_data, 0x00,VALIDATE_VM_BUFFFER_SIZE);
                memset(validate_vm[1].VM_data, 0x00,VALIDATE_VM_BUFFFER_SIZE);
            }
            else
            { 
                aud_glob_vms_playback.vm_result = aud_rc_performance_problems;
            }
        }
    } 
    else
        aud_glob_vms_playback.vm_result = aud_rc_performance_problems;
} 
#endif /*VM_MMF_DATA_VERIFY*/

#endif 

