/*******************************************************************************
 * Copyright (C) 2007, ZTE Corporation.
 *
 * File Name:    
 * File Mark:    
 * Description:  This source file contains the main control statemachine of the Universal Integrated Circuit Card (UICC).
 * Others:        
 * Version:       v0.1
 * Author:        mtr
 * Date:          2007-03-28
 * History 1:      
 *     Date: 
 *     Version:
 *     Author: 
 *     Modification:  
 * History 2: 
  ********************************************************************************/


/****************************************************************************
*                                              Include files
****************************************************************************/
#include "drvs_general.h"
#include "drvs_cfg.h"/*  for register base addr define  */

#include "uicc_stub.h"


#include "uicc.h"
#include "uicc_ctrl.h"
#include "uicc_access.h"
#include "uicc_drv.h"
#include "uicc_trc.h"

#include "USIM_reg.h"
#include "drvs_ramlog.h"

#ifdef USE_EEPROM_PARAMETERS
#include "eep.h"
#endif
#include "drvs_uicc.h"
#include "drvs_sys.h"

/****************************************************************************
* 	                                           Local Macros
****************************************************************************/
#define MAX_NOF_COLD_ATR_ATTEMPT        3
#define MAX_NOF_WARM_ATR_ATTEMPT        2
#define MAX_NOF_PPS_ATTEMPT             3
#define STATE_NOCLEAR_VALID             0xA52C
    
#ifdef SW_RESET_CHV
  #define AUTO_CHV_VERIFICATION_PATTERN 0xF000
#endif

/****************************************************************************
* 	                                           Local Types
****************************************************************************/

/****************************************************************************
* 	                                           Local Constants
****************************************************************************/

/****************************************************************************
* 	                                           Local Variables
****************************************************************************/
#ifndef UICC_NO_DEBUG
  static T_UICC_CARD_CONTROL_STATE card_ctrl_debug = UICC_CARD_NONE;
  static T_UICC_MAIN_CONTROL_STATE main_ctrl_debug = UICC_MAIN_NONE;
#endif

/****************************************************************************
* 	                                           Local Function Prototypes
****************************************************************************/


/****************************************************************************
* 	                                          Global Constants
****************************************************************************/

/****************************************************************************
* 	                                          Global Variables
****************************************************************************/
ubyte activation_phase = TRUE;  /* Some 'early' UICC cards cannot accept that the clock
                                                           is stopped in between activation and the very first
                                                           command issued - in spite the fact that the clock stop
                                                           conditions is given in the ATR sequence. This variable
                                                           is a counter measure for this scenario. */
    
#ifdef SILENT_RESET
  extern unsigned int NU_silent_reset_flag;
#endif
    
extern T_Drv_Hisr s_simHalHisr;
extern ushort err_type_chtimeout;
T_UICC_CARD_CONTROL_STATE UICC_testmode_state = UICC_CARD_NONE;

 T_UICC_CARD_CONTROL_STATE UICC_card_ctrl_state[UICC_NOF_CARD_SLOTS];
 ushort state_validity_noclear;

#ifdef UICC_ENABLE_IN_OUT_DETECTION
  uicc_card_presence_ctrl_state_type UICC_card_presence_ctrl_state = ready;
#endif

T_UICC_MAIN_CONTROL_STATE UICC_main_ctrl_state = UICC_READY;

/* Flag used to ensure the validity of the no clear state variable UICC_card_ctrl_state */
ushort state_validity_volatile = FALSE;

T_UICC_PROFILE_PARAMETER UICC_profile_parameter;

T_UICC_CARD_SELECTOR        UICC_current_reader = UICC_PRIMARY_CARD;

T_UICC_APDU                 UICC_apdu;

ubyte                       UICC_power_down_allowed_flag = FALSE;

ubyte                       UICC_transport_apdu_result;
ubyte                       UICC_activation_result;
ubyte                       UICC_deactivation_result;

T_UICC_T0_TPDU_CTRL         UICC_t0_tpdu_ctrl;

T_UICC_TPDU_FOOTER          UICC_tpdu_footer;
#if 0//modify by shideyou 20120718 #if 0 #else
#if defined (UICC_FIFO_MODE)
  ubyte __align(4)       UICC_tpdu_data[UICC_MAX_PAYLOAD_SIZE]; /*Should be aligned to 4 when using FIFO mode */
  ubyte                      *UICC_tpdu_data_ptr = &UICC_tpdu_data[UICC_DATA_POS];
#else
ubyte                       UICC_tpdu_data[UICC_MAX_PAYLOAD_SIZE];
ubyte                       *UICC_tpdu_data_ptr = &UICC_tpdu_data[UICC_DATA_POS];
#endif
#else
#if defined (UICC_FIFO_MODE)
  ubyte 
  #ifdef __ARMCC_VERSION /*rvct*/
  __align(4)
  #endif
  UICC_tpdu_data[UICC_MAX_PAYLOAD_SIZE]
  #ifdef __GNUC__ /* gcc */
  __attribute__ ((aligned(4)))
  #endif
  ; /*Should be aligned to 4 when using FIFO mode */
  ubyte                      *UICC_tpdu_data_ptr = &UICC_tpdu_data[UICC_DATA_POS];
#else
ubyte                       UICC_tpdu_data[UICC_MAX_PAYLOAD_SIZE];
ubyte                       *UICC_tpdu_data_ptr = &UICC_tpdu_data[UICC_DATA_POS];
#endif
#endif

ubyte                       UICC_nof_cold_atr_attempt;
ubyte                       UICC_nof_warm_atr_attempt;
ubyte                       UICC_nof_pps_attempt;

#ifdef UICC_IT3_27_12_1_PATCH
  ubyte UICC_authentication_cmd_executing = FALSE;
#endif

T_UICC_BASEBAND_PROFILE     UICC_baseband_profile;

#if defined(POWER_CONCEPT_3)
ubyte clock_setting_initialized = FALSE;
#endif

/****************************************************************************
* 	                                          Function Definitions
****************************************************************************/
//bb_ver_enum SYS_GetBBVersion() { return SYS_BB_VER_INVALID; };
//bb_rev_enum SYS_GetBBRevision() { return SYS_BB_VER_INVALID; };

/*******************************************************************************
* Function:... voltage_info_in_atr
*
* Description: Function capable of determine whether or clock stop and voltage
*              characteristics was part of the ATR string - and if so extracting
*              the clock stop characteristics.
* Created:.... 13.09.01 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static BOOL voltage_clock_info_in_atr(void)
{
  BOOL result = FALSE;
  ushort class_clock = UICC_card_profile[UICC_current_reader].class_clock;
  T_UICC_CLOCK_STOP_MODE clock_mode;

  if(class_clock)
  {
    result = TRUE;

    switch((class_clock & 0x00C0)>>6)
    {
       case 3: /* No preference. */
         clock_mode = UICC_NO_PREFERRED_LEVEL;
         break;
       case 2: /* State H (high). */
         clock_mode = UICC_CLOCK_STOP_ONLY_HIGH;
         break;
       case 1: /* State L (low). */
         clock_mode = UICC_CLOCK_STOP_ONLY_LOW;
         break;
       case 0: /* Not supported. */
       default:
         clock_mode = UICC_CLOCK_STOP_NOT_ALLOWED;
         break;
    }

    UICC_card_profile[UICC_current_reader].used_clock_stop_mode = clock_mode;
  }

  return(result);
}

/*******************************************************************************
* Function:... correct_voltage_level
*
* Description: Function used for checking if the indicated voltage level (indicated
*              within the ATR characters) also is the voltage level used in the
*              current activation.
* Created:.... 13.09.01 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static BOOL correct_voltage_level(void)
{
  BOOL result = FALSE;

  /* Check if the indicated voltage level (indicated within the ATR characters)
     also is the voltage level that is curently used in this activation. The voltage
     part (the class) part is represented by b5-b0 where:
     B0=1 indicates 5v (class A) support.
     B1=1 indicates 3v (class B) support
     B2=1 indicates 1.8v (class C) support
     B3=1 indicates RFUv (class D) support etc. */

  if((UICC_card_profile[UICC_current_reader].class_clock & 0x003F) &
     (1<<(UICC_card_profile[UICC_current_reader].used_voltage)))
  {
    result = TRUE;
  }

  return(result);
}

/*******************************************************************************
* Function:... deactivate_card_connections
*
* Description: Function for deactivating all physical signal towards the used card.
* Created:.... 13.09.01 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
void deactivate_card_connections(void)
{
  /* If automatic powerdown has been utilised prior to this initialization, it
     is necessary to reset the register controlling the PIN connections.      */
  UICC_set_low_clock_stop_level(UICC_current_reader);
  UICC_deactivate_rst(UICC_current_reader);
  UICC_deactivate_clk(UICC_current_reader);
  UICC_select_high_clock_freq();
  UICC_disable_io(UICC_current_reader);
  UICC_deactivate_vcc(UICC_current_reader);
}

#if defined(POWER_CONCEPT_3)
/*******************************************************************************
* Function:... initialize_clock_settings
*
* Description: Initliazes the USIM block related clock sources - if not alread 
*              initialzed.
* Created:.... 20.11.06 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
void initialize_clock_settings(void)
{
  if(!clock_setting_initialized)
  {
    clock_setting_initialized = TRUE;
  }
}
#endif

/*******************************************************************************
* Function:... force_signup_for_power_saving
*
* Description: Sign up for (allow) powersaving - with no reference to the clock
*              stop conditions dictated by the used card(s).
* Created:.... 13.09.01 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void force_signup_for_power_saving(void)
{
  /* UICC can accept that the the 13MHz clock is turned off .*/
  UICC_power_down_allowed_flag = TRUE;

}

/*******************************************************************************
* Function:... signup_for_power_saving
*
* Description: Sign up for (allow) powersaving - provided the clock stop conditions
*              dictated by the used card(s) would allow this action.
* Created:.... 13.09.01 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void signup_for_power_saving(void)
{
  #ifdef UICC_IT3_27_12_1_PATCH
    /* Refrain from signing up for powersaving if the instruction just executed was the RUN GSM
       ALGORITHM ('88') because the SIM clock is still running. This is to accomodate the Orga
       IT3 FTA test requirements. The clock will be stopped again after the following command.  */
    if(UICC_authentication_cmd_executing)
    { /* Do not sign up for power saving because the clock towards the card must not be stopped. */
      return;
    }
  #endif

  /* If the card supports clockstop, power saving is accepted! That
     implies that power saving is NOT accepted if clock stop is not
     supported by the card (should however never be the case using
     3v card). */
  if((UICC_card_profile[UICC_current_reader].used_clock_stop_mode != UICC_CLOCK_STOP_NOT_ALLOWED) &&
     (!activation_phase))
  {
    force_signup_for_power_saving();
  }
}

/*******************************************************************************
* Function:... UICC_resign_from_power_saving
*
* Description: Resign from (do not perform) powersaving.
* Created:.... 13.09.01 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
void UICC_resign_from_power_saving(void)
{
  /* UICC can no longer accept that the the 13MHz clock is turned off .*/
  UICC_power_down_allowed_flag = FALSE;
}

#ifdef UICC_ENABLE_IN_OUT_DETECTION
/****************************************************************************************
* Function:... UICC_in_out_control
* Parameters:. event: the event triggering the statemachine.
* Returns:.... None.
* Description: This function contains a statemachine for controllong the card in/out 
*              behaivour.
* Created:.... 08.12.06 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
****************************************************************************************/
#define DEBOUNCE_DELAY 300000
uicc_card_presence_ctrl_state_type UICC_inout_ctrl_state = ready;

void UICC_in_out_control(T_UICC_CTRL_CARD_PRESENCE_EVENTS event)
{
  switch(UICC_inout_ctrl_state)
  {
    case ready:
      switch(event)
      {
        case UICC_CARD_IN:
        case UICC_CARD_OUT:
          UICC_disable_usim_in_isr();
          UICC_inout_ctrl_state = debounce_check;
          deactivate_card_connections();
          UICC_start_inout_timer(DEBOUNCE_DELAY);
          break;

        default:
          break;
      }
      break;

    case debounce_check:
      switch(event)
      {
        case UICC_CARD_IN_OUT_TIMER_EXPIRED:
          UICC_inout_ctrl_state = ready;

          if(UICC_executing != UICC_EXECUTING)
          {
            if(UICC_check_card_presence())
            {
              UICC_card_ctrl(UICC_CARD_INSERTED);
            }
            else
            {
              UICC_card_ctrl(UICC_CARD_REMOVED);
            }
          }
          UICC_enable_usim_in_isr();
          break;

        default:
          break;
      }
      break;

    default:
      break;
  }
}
#endif

#ifdef UICC_ENABLE_IN_OUT_DETECTION
/****************************************************************************************
* Function:... UICC_ctrl_card_presence
* Parameters:. event: the event triggering the switch.
* Returns:.... None.
* Description: This function contains a function switch which is used to determine
*              which statemachine is going to receive the event - based on the
*              main control state.
* Created:.... 19.12.05 by Manh Tran (MTR)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
****************************************************************************************/
void UICC_ctrl_card_presence(T_UICC_CTRL_CARD_PRESENCE_EVENTS event)
{
  switch(UICC_card_presence_ctrl_state)
  {
    case ready:
      switch(event)
      {
        case UICC_CARD_IN:
        case UICC_CARD_OUT:
          UICC_card_presence_ctrl_state = debounce_check;
          UICC_disable_usim_in_isr();
          deactivate_card_connections();
          UICC_start_inout_timer(100000);/*100000 usec*/
          break;

        default:
          break;
      }
      break;

    case debounce_check:
      switch(event)
      {
        case UICC_CARD_IN_OUT_TIMER_EXPIRED:
          UICC_card_presence_ctrl_state = ready;
          if(UICC_check_card_presence())
          {
            UICC_card_ctrl(UICC_CARD_INSERTED);
          }
          else
          {
            /*Enable IN/OUT interrupt again here when Card removed.
                In the case card insert the IN/OUT interrupt first enable when UICC finished with the ATR process to ensure
                prelle do not disturb the ATR process */
            UICC_enable_usim_in_isr();
            UICC_card_ctrl(UICC_CARD_REMOVED);
          }
          break;

        default:
          break;
      }
      break;

    default:
      break;
  }
}
#endif


/*******************************************************************************
* Function:... init_eep_prarameter
*
* Description: read and initiale the eeprom parameters.
* Created:.... 09.11.05 by Manh Tran (MTR)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void uicc_init_eep_parameter(void)
{

#ifdef USE_EEPROM_PARAMETERS
  if (EEP_static.uicc_profile.code_word == 0xBEEF) /*If code word OK then use the parameters from eeprom*/
  {
    UICC_profile_parameter.cwt_extension =         EEP_static.uicc_profile.CWT_extension; /* increase the calculated cwt with this value */
    UICC_profile_parameter.bwt_extension = EEP_static.uicc_profile.BWT_extension;  /* increase the calculated bwt with this value */
    UICC_profile_parameter.wwt_extension = EEP_static.uicc_profile.WWT_extension;
    UICC_profile_parameter.max_speed = EEP_static.uicc_profile.max_speed; /* 0 = use cards speed or default; other value ect. 0x94 mean F=512,D=8*/
    UICC_profile_parameter.preferred_protocol = EEP_static.uicc_profile.preferred_protocol; /*If both protocol support then 0: T=0; 1: T=1 */
    UICC_profile_parameter.bgt = EEP_static.uicc_profile.BGT_extension + 22; /* 22 [ETU] = default */
    UICC_profile_parameter.support_voltage = EEP_static.uicc_profile.supported_voltage_levels;
    UICC_profile_parameter.reserved_1 = EEP_static.uicc_profile.reserved[0];
    UICC_profile_parameter.reserved_2 = EEP_static.uicc_profile.reserved[1];
    UICC_profile_parameter.reserved_3 = EEP_static.uicc_profile.reserved[2];
    UICC_profile_parameter.reserved_4 = EEP_static.uicc_profile.reserved[3];

  }
  else /*Default parameters*/
  {
    UICC_profile_parameter.cwt_extension = 0; /* increase the calculated cwt with this value */
    UICC_profile_parameter.bwt_extension = 0;  /* increase the calculated bwt with this value */
    UICC_profile_parameter.wwt_extension = 0;
    UICC_profile_parameter.max_speed = 0; /* 0 = use cards speed or default; other value ect. 0x94 mean F=512,D=8*/
    UICC_profile_parameter.preferred_protocol = 1; /*If both protocol support then 0: T=0; 1: T=1 */
    UICC_profile_parameter.bgt = 42 ;/* 42 [ETU] = default */
    UICC_profile_parameter.support_voltage = 0x03;
    UICC_profile_parameter.reserved_1 = 0;
    UICC_profile_parameter.reserved_2 = 0;
    UICC_profile_parameter.reserved_3 = 0;
    UICC_profile_parameter.reserved_4 = 0;

  }
#else /*Not read the eeprom parameter use the default*/
    UICC_profile_parameter.cwt_extension = 0; /* increase the calculated cwt with this value */
    UICC_profile_parameter.bwt_extension = 0;  /* increase the calculated bwt with this value */
    UICC_profile_parameter.wwt_extension = 0;
    UICC_profile_parameter.max_speed = 0; /* 0 = use cards speed or default; other value ect. 0x94 mean F=512,D=8*/
    UICC_profile_parameter.preferred_protocol = 1; /*If both protocol support then 0: T=0; 1: T=1 */
    UICC_profile_parameter.bgt = 42;/* 42 [ETU] = default */
    UICC_profile_parameter.support_voltage = 0x03;
    UICC_profile_parameter.reserved_1 = 0;
    UICC_profile_parameter.reserved_2 = 0;
    UICC_profile_parameter.reserved_3 = 0;
    UICC_profile_parameter.reserved_4 = 0;
#endif
}


/*******************************************************************************
* Function:... initialize_module
*
* Description: Performs the initial initialization of the UICC Handler.
* Created:.... 13.09.01 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
static void initialize_module(void)
{
  #if defined(POWER_CONCEPT_3)
    initialize_clock_settings();
  #endif

  UICC_resign_from_power_saving();          /* Power down NOT allowed */
  
  UICC_init_timer();                   /* Set up the interrupt related to the low
                                        level timer.                          */

  UICC_reset_card_interface();         /* Reset the HW to the initial value. */
  UICC_init_card_interface();          /* Set up the interrupts related to the card
                                        interface block.                      */
  UICC_reset_atr();                    /* Initialize the ATR statemachine.      */
  UICC_reset_pps();                    /* Initialize the PPS statemachine.      */
  UICC_reset_t0_cmd_handler();         /* Initialize the INS statemachine.      */

  UICC_t0_trans_ctrl_state = UICC_T0_TRANS_READY;

  UICC_reset_t1_transport_handler();   /* Initialise the T=1 protocol controllers. */

  deactivate_card_connections();       /* Disable the pin connections to the card.*/

  /* Initiate a Answer To Reset sequence */
  UICC_nof_cold_atr_attempt = MAX_NOF_COLD_ATR_ATTEMPT;
  UICC_nof_warm_atr_attempt = MAX_NOF_WARM_ATR_ATTEMPT;
  UICC_nof_pps_attempt = MAX_NOF_PPS_ATTEMPT;

  UICC_disable_dma_channel();
}


#ifdef SILENT_RESET
  /*******************************************************************************
  * Function:... silent_reset_initialise
  *
  * Description: Module initialisation after the occurence of a Silent Reset.
  * Created:.... 13.09.01 by Knud Nymann Mortensen (KNM)
  * Modified:... DD.MM.YY by (Full name / initials)
  *                Modifications note.
  *******************************************************************************/
  static void silent_reset_initialise(void)
  {
    #if defined(POWER_CONCEPT_3)
      initialize_clock_settings();
    #endif

    UICC_resign_from_power_saving();        /* Power down NOT allowed */
 
    UICC_init_timer();                   /* Set up the interrupt related to the low
                                        level timer.                          */

    UICC_init_card_interface();          /* Set up the interrupts related to the card
                                        interface block.                      */
    UICC_reset_atr();                    /* Initialize the ATR statemachine.      */
    UICC_reset_pps();                    /* Initialize the PPS statemachine.      */
    UICC_reset_t0_cmd_handler();         /* Initialize the INS statemachine.      */

    /* Initiate a Answer To Reset sequence */
    UICC_nof_cold_atr_attempt = MAX_NOF_COLD_ATR_ATTEMPT;
    UICC_nof_warm_atr_attempt = MAX_NOF_WARM_ATR_ATTEMPT;
    UICC_nof_pps_attempt = MAX_NOF_PPS_ATTEMPT;

    UICC_disable_dma_channel();

     UICC_activation_result = UICC_ACTIVATION_SUCCEEDED;
     #ifdef UICC_MODULE_TEST
       if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
       {
         UICC_simulated_function.activation_result_invoked = TRUE;
         return;
       }
     #endif
     /* RELEASE ACTIVATION SEMAPHORE. */
     UICC_release_semaphore(UICC_ACTIVATION_SEM);
  }
#endif

/****************************************************************************************
* Function:... no_card_access
* Parameters:. event: the event triggering the statemachine.
* Returns:.... None.
* Description: Dummy function used to handle and detect unexpected events.
* Created:.... 29.05.00 by Knud Nymann Mortensen
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
****************************************************************************************/
static void no_card_access(T_UICC_INT_EVENTS event)
{
  #ifndef UICC_NO_DEBUG
    UICC_log_data_ptr[0] = UICC_card_ctrl_state[UICC_current_reader];
    UICC_log_data_ptr[1] = event;
    UICC_raise_exception(TRAP_UICC_CTRL, UICC_NO_SIM_ACCESS_EVENT, 2);
  #else
    event = event;
  #endif
}

/****************************************************************************************
* Function:... fetch_baseband_profile
* Parameters:. None.
* Returns:.... None.
* Description: Fetches and caches the Baseband profile (Gold type and revision) and stores
*              the data in a global variable structure.
* Created:.... 25.08.04 by Knud Nymann Mortensen
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
****************************************************************************************/
static void fetch_baseband_profile(void)
{
  #ifndef UICC_NO_DEBUG
    ubyte raise = FALSE;
  #endif

  UICC_baseband_profile.version  = SYS_BB_VER_INVALID;
  UICC_baseband_profile.revision = SYS_BB_VER_INVALID;

  /* Check the validity of the Baseband data. */
  if(UICC_baseband_profile.version == SYS_BB_VER_INVALID)
  { /* Invalid Baseband version - set the cashed version to a 'best guess' value. */
    UICC_baseband_profile.version = SYS_BB_VER_SGOLD;
    #ifndef UICC_NO_DEBUG
      raise = TRUE;
    #endif
  }

  if(UICC_baseband_profile.revision == SYS_BB_REV_INVALID)
  { /* Invalid Baseband revision - set the cashed version to a 'best guess' value. */
    UICC_baseband_profile.revision = SYS_BB_REV_10;
    #ifndef UICC_NO_DEBUG
      raise = TRUE;
    #endif
  }

  #ifndef UICC_NO_DEBUG
    if(raise)
    { /* Raise an exception indicating problems with the Baseband information. */
      UICC_log_data_ptr[0] = (ubyte) SYS_BB_VER_INVALID;
      UICC_log_data_ptr[1] = (ubyte) SYS_BB_VER_INVALID;
      UICC_raise_exception(TRAP_UICC_CTRL, UICC_INVALID_BASEBAND_INFO, 2);
    }
  #endif
}

/*---------------------------------------------*/
/* Global functions.                           */
/*---------------------------------------------*/

/****************************************************************************************
* Function:... UICC_etu2usec
* Parameters:. etu: ETU value that needs to be converted to the corresponding uSEC value.
* Returns:.... the converted uSEC value.
* Description: Converts from ETU to uSEC.
* Created:.... 19.05.04 by Knud Nymann Mortensen
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
****************************************************************************************/
ulong UICC_etu2usec(ulong etu)
{
  if(etu < 23091) // Max value on top of brk (etu)23091*(clk)1860*100 = 0xFFFF5EB0
  {
    return((ulong)( 1+ (etu * UICC_clock_rate_table[UICC_card_profile[UICC_current_reader].f_used] * 100)/
                   (UICC_baud_rate_table[UICC_card_profile[UICC_current_reader].d_used] * 325)));
  }
  else
  {
    // OK if etu*10 < 0x100000000.
    return((ulong)( 1+  ((etu * 10)/
                     (UICC_baud_rate_table[UICC_card_profile[UICC_current_reader].d_used] * 325))
                     * UICC_clock_rate_table[UICC_card_profile[UICC_current_reader].f_used] * 10));
  }
}

/****************************************************************************************
* Function:... UICC_usec2etu
* Parameters:. usec: uSEC value that needs to be converted to the corresponding ETU value.
* Returns:.... the converted ETU value.
* Description: Converts from uSEC to ETU.
* Created:.... 19.05.04 by Knud Nymann Mortensen
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
****************************************************************************************/
ulong UICC_usec2etu(ulong usec)
{
  if(usec < 412977) // Max value on top of brk 412977*325*32 < 0xFFFFE6A0
  {
    return((ulong) ((usec * 325 * UICC_baud_rate_table[UICC_card_profile[UICC_current_reader].d_used]) /
                     (UICC_clock_rate_table[UICC_card_profile[UICC_current_reader].f_used] * 100)) + 1);
  }
  else
  {
    // OK if usec*32 < 0x100000000
    return((ulong) ( 1+ (((usec * UICC_baud_rate_table[UICC_card_profile[UICC_current_reader].d_used]) /
                       UICC_clock_rate_table[UICC_card_profile[UICC_current_reader].f_used]) * 325) / 100));
  }
}

/****************************************************************************************
* Function:... UICC_validate_state
* Parameters:. None.
* Returns:.... None.
* Description: This function validates the card control state after a Reset (either as a
*              result of a normal poweron or after a Silent Reset).
* Created:.... 14.11.03 by Knud Nymann Mortensen
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
****************************************************************************************/
void UICC_validate_state(void)
{
  if(state_validity_volatile == FALSE)
  { /* State not valid, check for Silent Reset */
    ubyte i;

    uicc_init_eep_parameter(); /*Read and initiale the eeprom parameters*/
    #ifdef SILENT_RESET
    if((NU_silent_reset_flag) && (state_validity_noclear == STATE_NOCLEAR_VALID))
    {
      /* Silent Reset and states are valid - CONTINUE with these states.*/
    }
    else
    #endif
    { /* Must be initial reset (or the state(s) was not not initialised
         correct -> initialise the card control state(s) and other no-clear
         declared variables. */
      for(i=0; i<UICC_NOF_CARD_SLOTS; i++)
      {
        UICC_card_ctrl_state[i] = UICC_INACTIVE;
        memset(&UICC_card_profile[i], 0x00, sizeof(T_UICC_CARD_PROFILE));
      }
      UICC_t1_s_seq_no        = 1;
      UICC_t1_r_seq_no        = 0;
      UICC_nof_atr_characters = 0;

      state_validity_noclear = STATE_NOCLEAR_VALID;
    }

    state_validity_volatile = TRUE;
  } /* else: State is valid (has been validated) -> continue. */


  /* Now cach the Baseband verson and revision. */
  fetch_baseband_profile();
}

/*******************************************************************************
* Function:... UICC_another_voltage_level_available
*
* Created:.... 13.09.01 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
BOOL UICC_another_voltage_level_available(T_UICC_VOLTAGE_CLASS *used_voltage)
{
  BOOL voltage_found = FALSE;

  T_UICC_VOLTAGE_CLASS i;
  ushort offered_voltage = UICC_card_profile[UICC_current_reader].class_clock & 0x003F;
  ushort current_voltage = UICC_card_profile[UICC_current_reader].used_voltage;

  /*-------------------------------------------------------------------------**
  **                                                                         **
  **   0        1        2        3        4           <- Numerical value    **
  **   |--------|--------|--------|--------|------->                         **
  **   5v      3v       1.8v     0.9v     0.Xv         <- Translated meaning **
  **   A       B         C        D        E                                 **
  **                                                                         **
  **       e.g.: MAX =   3v (Num=1).                                         **
  **             MIN = 1.8v (Num=2).                                         **
  **                                                                         **
  ** The activation should always start at lowest possible voltage level and **
  ** then climb to a higher level - if needed.                               **
  **-------------------------------------------------------------------------*/
  for(i = UICC_lower_supported_voltage_level(); (i >= UICC_higher_supported_voltage_level() && !voltage_found); i--)
  {
    /* The following conditions must be fullfilled in the search for another
       available voltage level:
       - first see if the voltage is within the higher and lower limits supported by the terminal.
       - and then should the new voltage be different from and at a higher voltage level than the
         currently used.  */
    if(i < current_voltage)
    {
      if(voltage_clock_info_in_atr())
      {
        /* Verify that the voltage is supported. */
        if(offered_voltage & 1<<i)
        {
          voltage_found = TRUE;
          *used_voltage = i;
        } /* else: found voltages not the right one -> continue the search. */
      }
      else
      { /* No voltage information in ATR characters. */
        voltage_found = TRUE;
        *used_voltage = i;
      }
    }
  }
  return(voltage_found);
}

/****************************************************************************************
* Function:... UICC_lower_supported_voltage_level
* Parameters:. None.
* Returns:.... Returnes the lower voltage level supported by the terminal.
* Description: This function detirmines the lower voltage level supported by the terminal -
*              partly via the define level but also based on the used SM Power.
* Created:.... 27.10.03 by Knud Nymann Mortensen
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
****************************************************************************************/
T_UICC_VOLTAGE_CLASS UICC_lower_supported_voltage_level(void)
{
#if defined (USE_SMPOWER) && !defined (SGOLDRADIO) && !defined (POWER_CONCEPT_3)
  if((UICC_MIN_SUPPORTED_VOLTAGE == UICC_CLASS_B) ||
     (POW_get_power_chip_version() == POW_chip_version_10))
#else
  if(UICC_MIN_SUPPORTED_VOLTAGE == UICC_CLASS_B)
#endif
  {
    return(UICC_CLASS_B);
  }
  else
  {
    return(UICC_CLASS_C);
  }
}

/****************************************************************************************
* Function:... UICC_higher_supported_voltage_level
* Parameters:. None.
* Returns:.... Returnes the higher voltage level supported by the terminal.
* Description: This function detirmines the higher voltage level supported by the terminal -
*              based on the define level.
* Created:.... 27.10.03 by Knud Nymann Mortensen
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
****************************************************************************************/
T_UICC_VOLTAGE_CLASS UICC_higher_supported_voltage_level(void)
{
  return(UICC_MAX_SUPPORTED_VOLTAGE);
}

/****************************************************************************************
* Function:... UICC_stop_clock_conditional
* Parameters:. clock_stop_mode: The clock stop mode for the operative card
*              current_reader:  The current operative card.
* Returns:.... None.
* Description: This function detirmines (and executes) the clock stop mode for the
*              operative card.
* Created:.... 17.12.04 by Knud Nymann Mortensen
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
****************************************************************************************/
void UICC_stop_clock_conditional(T_UICC_CLOCK_STOP_MODE clock_stop_mode,
                                 T_UICC_CARD_SELECTOR   current_reader)
{
  switch(clock_stop_mode)
  {
    case UICC_CLOCK_STOP_ALLOWED:
    case UICC_NO_PREFERRED_LEVEL:
    case UICC_LOW_LEVEL_PREFERRED:
    case UICC_CLOCK_STOP_ONLY_LOW:
      UICC_set_low_clock_stop_level(current_reader);
      UICC_deactivate_clk(current_reader);
      break;

    case UICC_HIGH_LEVEL_PREFERRED:
    case UICC_CLOCK_STOP_ONLY_HIGH:
      UICC_set_high_clock_stop_level(current_reader);
      UICC_deactivate_clk(current_reader);
      break;

    case UICC_CLOCK_STOP_NOT_ALLOWED:
    default:
      #ifdef UICC_LOW_CLK_FREQ_SUPPORTED
        /* Clock stop NOT allowed - switch to 1.08MHz */
        UICC_select_low_clock_freq();
      #endif
      break;
  }
}

/****************************************************************************************
* Function:... UICC_control_card_access
* Parameters:. event: the event triggering the switch.
* Returns:.... None.
* Description: This function contains a function switch wich is used to determine
*              which statemachine is going to receive the hardware generated event - based
*              on the main control state.
* Created:.... 16.09.01 by Knud Nymann Mortensen
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
****************************************************************************************/
void UICC_control_card_access(T_UICC_INT_EVENTS event)
{
#ifdef UICC_DEBUG_INF
UICC_debug[UICC_debug_count][0] = 7;  
UICC_debug[UICC_debug_count][1] = UICC_card_ctrl_state[UICC_current_reader];
UICC_debug[UICC_debug_count][2] = event;
UICC_debug_count = (UICC_debug_count+1) % 200;
#endif

  switch(UICC_card_ctrl_state[UICC_current_reader])
  {
    case UICC_COLD_ACTIVATION:
      UICC_ctrl_atr((T_UICC_ATR_EVENTS)event);
      break;
    case UICC_WARM_ACTIVATION:
      UICC_ctrl_atr((T_UICC_ATR_EVENTS)event);
      break;
    case UICC_COLD_PPS_DETERMINATION:
      UICC_ctrl_pps((T_UICC_PPS_EVENTS)event);
      break;
    case UICC_WARM_PPS_DETERMINATION:
      UICC_ctrl_pps((T_UICC_PPS_EVENTS)event);
      break;
    case UICC_BUSY:
      if(UICC_card_profile[UICC_current_reader].T_used == UICC_T_0)
      {
        UICC_t0_cmd_handler((T_UICC_T0_CMD_EVENTS)event);
      }
      else
      {
        UICC_t1_block_handler((T_UICC_T1_BLOCK_EVENTS)event);
      }
      break;
    case UICC_TESTMODE:
      if(UICC_testmode_state == UICC_COLD_ACTIVATION)
        UICC_ctrl_atr((T_UICC_ATR_EVENTS)event);
      else if(UICC_testmode_state == UICC_BUSY)
        if(UICC_card_profile[UICC_current_reader].T_used == UICC_T_0)
        {
          UICC_t0_cmd_handler((T_UICC_T0_CMD_EVENTS)event);
        }
        else
        {
          UICC_t1_block_handler((T_UICC_T1_BLOCK_EVENTS)event);
        }
      else
        no_card_access(event);
      break;
    case UICC_INACTIVE:
    case UICC_IDLE:
    default:
      no_card_access(event);
      break;
  }
}


static recoverCallback_func recoverCallback = NULL;

void uicc_setRecover_callback(recoverCallback_func func)
{
	recoverCallback = func;
}

typedef struct _T_UiccDebugInfo
{
	CHAR debugInfo[41];
}
T_UiccDebugInfo;

T_UiccDebugInfo UiccControlStatusInfo[] =
{
    "UICC_CARD_NONE",
    "UICC_INACTIVE",
    "UICC_COLD_ACTIVATION",
    "UICC_WARM_ACTIVATION",
    "UICC_COLD_PPS_DETERMINATION",

    "UICC_WARM_PPS_DETERMINATION",
    "UICC_IDLE",
    "UICC_BUSY",
    "UICC_TESTMODE",
    "UICC_CARD_IN_OUT_DETECTING"
};


T_UiccDebugInfo UiccControlEventInfo[] =
{
    "UICC_DUMMY_EVENT",
    "UICC_ATR_OK",
    "UICC_ATR_FAILED_NO_ANSWER",
    "UICC_ATR_FAILED",
    
    "UICC_PPS_OK",
    "UICC_PPS_NOT_NEEDED",
    "UICC_PPS_NOT_COMPLETE",
    "UICC_PPS_FAILED",
    "UICC_PPS_INITIATE_WARM_RESET",

    "UICC_INSTRUCTION_SUCCEEDED",
    "UICC_INSTRUCTION_FAILED",
	
    "UICC_INSTRUCTION_REQUESTED",
    "UICC_COLD_RESET_REQUESTED",
    "UICC_WARM_RESET_REQUESTED",
    "UICC_DEACTIVATION_REQUESTED",

    "UICC_TESTMODE_REQUESTED",
    
    "UICC_CARD_INSERTED",
    "UICC_CARD_REMOVED"	
};

/****************************************************************************************
* Function:... UICC_card_ctrl
* Parameters:. event: the event triggering the statemachine.
* Returns:.... None.
* Description: This function has the control over the different supported cards (one
*              statemachine instance per supported card).
* Created:.... 26.05.00 by Knud Nymann Mortensen
* Modified:... 29.11.02 by (Full name / initials)
*                - Card instances added.
****************************************************************************************/
void UICC_card_ctrl(T_UICC_CARD_CTRL_EVENT event)
{
#ifdef UICC_DEBUG_INF
  UICC_debug[UICC_debug_count][0] = 0;  
UICC_debug[UICC_debug_count][1] = UICC_card_ctrl_state[UICC_current_reader];
UICC_debug[UICC_debug_count][2] = event;
UICC_debug_count = (UICC_debug_count+1) % 200;
#endif
    
  int i = 0;
    
    i = UICC_card_ctrl_state[UICC_current_reader];
    
    
    //uiccDebug_Printf("[%d]state=%s,event=%s!\n", zOss_GetTickCount(), &UiccControlStatusInfo[UICC_current_reader],&UiccEventInfo[event]);
    zDrvRamlog_PRINTF(RAMLOG_MOD_UICC,"[%d]state=%s,event=%s\n", zOss_GetTickCount(), (char *)(&UiccControlStatusInfo[i]),(char *)(&UiccControlEventInfo[event]));
    uiccramlog_Printf("[%d]state=%s,event=%s\n", zOss_GetTickCount(), (char *)(&UiccControlStatusInfo[i]),(char *)(&UiccControlEventInfo[event]));

  switch(UICC_card_ctrl_state[UICC_current_reader])
  {
    /*===============*/
    /* UICC inactive */
    /*===============*/
    case UICC_INACTIVE:
      switch(event)
      {
        /*----------------------------*/
        case UICC_COLD_RESET_REQUESTED:
        case UICC_WARM_RESET_REQUESTED: /* Should not be the case at this point. */
        /*----------------------------*/
          activation_phase = TRUE;
          #ifdef SILENT_RESET
            #ifdef SW_RESET_CHV
              if(NU_silent_reset_flag)
              { /* Setting the bits will indicate to SI that an auto verification of the PIN is required. */
                NU_silent_reset_flag |= AUTO_CHV_VERIFICATION_PATTERN;
              }
            #else
              {
                /* Silent reset can not be performed in this state because the SIM is not up and running. The silent
                   reset falg is as a result cleard resulting in a normal mobile startup sequence. */
                NU_silent_reset_flag = FALSE;
              }
            #endif
          #endif
          UICC_card_ctrl_state[UICC_current_reader] = UICC_COLD_ACTIVATION;
          initialize_module();
          UICC_card_profile[UICC_current_reader].used_voltage =
                      UICC_card_profile [UICC_current_reader].suggested_voltage;
          UICC_select_operative_card(UICC_current_reader);
          UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
          break;

        /*----------------------------*/
        case UICC_DEACTIVATION_REQUESTED:
        /*----------------------------*/
          /* Deactivation of the card. */
          force_signup_for_power_saving(); /* Power down allowed. */
          UICC_deactivation_result = UICC_DEACTIVATION_SUCCEEDED;
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.deactivation_result_invoked = TRUE;
              break;
            }
          #endif
          /* RELEASE DEACTIVATION SEMAPHORE. */
          UICC_release_semaphore(UICC_DEACTIVATION_SEM);
          break;

        /*----------------------------*/
        case UICC_TESTMODE_REQUESTED:
        /*----------------------------*/
          UICC_card_ctrl_state[UICC_current_reader] = UICC_TESTMODE;
          break;

        #ifdef UICC_ENABLE_IN_OUT_DETECTION
            /*----------------------------*/
            case UICC_CARD_INSERTED:
            /*----------------------------*/
            UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
            UICC_activate_inout_HISR(CARD_INSERTED);
            break;
            /*----------------------------*/
            case UICC_CARD_REMOVED:
            /*----------------------------*/
            UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
            UICC_activate_inout_HISR(CARD_REMOVED);
            break;
        #endif

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            card_ctrl_debug = UICC_card_ctrl_state[UICC_current_reader];
          #endif
          break;
      }
      break;

    /*=================*/
    /* COLD Activation */
    /*================*/
    case UICC_COLD_ACTIVATION:
      switch(event)
      {
        /*----------------------------*/
        case UICC_ATR_OK:
        /*----------------------------*/
          #ifdef UICC_ENABLE_IN_OUT_DETECTION
            UICC_enable_usim_in_isr();
          #endif
           /* Check if VOLTAGE INFO was enclosed in the ATR information. */
          if(voltage_clock_info_in_atr())
          {
            if(correct_voltage_level())
            {
              /* The correct voltage level has been found. */
              UICC_card_profile[UICC_current_reader].voltage_settled = TRUE;
              UICC_card_ctrl_state[UICC_current_reader] = UICC_COLD_PPS_DETERMINATION;
              UICC_ctrl_pps(UICC_INITIATE_PPS);
            }
            else
            {
              /* Check if another voltage level is available. */
              if(UICC_another_voltage_level_available(&UICC_card_profile[UICC_current_reader].used_voltage))
              {
                /* Stay in this state and re-activate. */
                deactivate_card_connections();
                UICC_nof_cold_atr_attempt = MAX_NOF_COLD_ATR_ATTEMPT;
                UICC_nof_pps_attempt = MAX_NOF_PPS_ATTEMPT;
                UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
              }
              else
              { /* No alternative voltage level available - deactivate card. */
                deactivate_card_connections();
                UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
                force_signup_for_power_saving(); /* Power down allowed. */
                UICC_activation_result = UICC_ACTIVATION_FAILED;
                #ifdef UICC_MODULE_TEST
                  if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
                  {
                    UICC_simulated_function.activation_result_invoked = TRUE;
                    break;
                  }
                #endif
                /* RELEASE ACTIVATION SEMAPHORE. */
                UICC_release_semaphore(UICC_ACTIVATION_SEM);
              }
            }
          }
          else
          {  /* No voltage information in ATR characters. */
            UICC_card_ctrl_state[UICC_current_reader] = UICC_COLD_PPS_DETERMINATION;
            UICC_ctrl_pps(UICC_INITIATE_PPS);
          }
          break;

        /*----------------------------*/
        case UICC_ATR_FAILED_NO_ANSWER:
        /*----------------------------*/
        {
          #ifdef UICC_ENABLE_IN_OUT_DETECTION
            UICC_enable_usim_in_isr();
          #endif

          deactivate_card_connections(); /* Deactivate the Card */

           /* Check if another voltage level is available. */
          if(UICC_another_voltage_level_available(&UICC_card_profile[UICC_current_reader].used_voltage))
          {
            /* Stay in this state and re-activate. */
            UICC_nof_cold_atr_attempt = MAX_NOF_COLD_ATR_ATTEMPT;
            UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
          }
          else
          { /* No alternative voltage level available - deactivate card. */
            UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
            force_signup_for_power_saving(); /* Power down allowed. */
            UICC_activation_result = UICC_ACTIVATION_FAILED;
            #ifdef UICC_MODULE_TEST
              if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
              {
                UICC_simulated_function.activation_result_invoked = TRUE;
                break;
              }
            #endif
            /* RELEASE ACTIVATION SEMAPHORE. */
            UICC_release_semaphore(UICC_ACTIVATION_SEM);
          }
          break;
        }
        /*----------------------------*/
        case UICC_ATR_FAILED:
        /*----------------------------*/
        {
          #ifdef UICC_ENABLE_IN_OUT_DETECTION
            UICC_enable_usim_in_isr();
          #endif

          deactivate_card_connections(); /* Deactivate the Card */

          /* Check the number of remaining ATR attempts. */
          if(--UICC_nof_cold_atr_attempt)
          {
            /* There are still ATR retrial attemts left. Therefore stay in this
               state and redo the ATR sequence - at the same voltage level. */
            UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
          }
          else
          {
            /* The maximum number of ATR retrial attempts has been reached. Try on
               another voltage level if available or discard the card. */
            if(UICC_another_voltage_level_available(&UICC_card_profile[UICC_current_reader].used_voltage))
            {
              /* Stay in this state and re-activate. */
              UICC_nof_cold_atr_attempt = MAX_NOF_COLD_ATR_ATTEMPT;
              UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
            }
            else
            { /* No alternative voltage level available - deactivate card. */
              UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
              force_signup_for_power_saving(); /* Power down allowed. */
              UICC_activation_result = UICC_ACTIVATION_FAILED;
              #ifdef UICC_MODULE_TEST
                if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
                {
                  UICC_simulated_function.activation_result_invoked = TRUE;
                  break;
                }
              #endif
              /* RELEASE ACTIVATION SEMAPHORE. */
              UICC_release_semaphore(UICC_ACTIVATION_SEM);
            }
          }
          break;
        }
        /*----------------------------*/
        case UICC_DEACTIVATION_REQUESTED:
        /*----------------------------*/
          /* Deactivation of the card. */
          UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
          UICC_select_operative_card(UICC_current_reader);
          deactivate_card_connections(); /* Deactivate the card. */
          force_signup_for_power_saving(); /* Power down allowed. */
          UICC_deactivation_result = UICC_DEACTIVATION_SUCCEEDED;
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.deactivation_result_invoked = TRUE;
              break;
            }
          #endif

          /* RELEASE DEACTIVATION SEMAPHORE. */
          UICC_release_semaphore(UICC_DEACTIVATION_SEM);

          break;

        /*----------------------------*/
        case UICC_COLD_RESET_REQUESTED:
        /*----------------------------*/
          activation_phase = TRUE;
          #ifdef SILENT_RESET
            #ifdef SW_RESET_CHV
              if(NU_silent_reset_flag)
              { /* Setting the bits will indicate to SI that an auto verification of the PIN is required. */
                NU_silent_reset_flag |= AUTO_CHV_VERIFICATION_PATTERN;
              }
            #else
              {
                /* Silent reset can not be performed in this state because the SIM is not up and running. The silent
                   reset falg is as a result cleard resulting in a normal mobile startup sequence. */
                NU_silent_reset_flag = FALSE;
              }
            #endif
          #endif
          UICC_card_ctrl_state[UICC_current_reader] = UICC_COLD_ACTIVATION;
          initialize_module();
          UICC_card_profile[UICC_current_reader].used_voltage =
		  	UICC_card_profile[UICC_current_reader].suggested_voltage;
          UICC_select_operative_card(UICC_current_reader);
          UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
          break;

          #ifdef UICC_ENABLE_IN_OUT_DETECTION
              /*----------------------------*/
              case UICC_CARD_INSERTED:
              /*----------------------------*/
              UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
              UICC_activate_inout_HISR(CARD_INSERTED);
              break;
              /*----------------------------*/
              case UICC_CARD_REMOVED:
              /*----------------------------*/
              UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
              UICC_activate_inout_HISR(CARD_REMOVED);
              break;
          #endif /*UICC_ENABLE_IN_OUT_DETECTION*/

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            card_ctrl_debug = UICC_card_ctrl_state[UICC_current_reader];
          #endif
          break;
      }
      break;

    /*=================*/
    /* Warm Activation */
    /*=================*/
    case UICC_WARM_ACTIVATION:
      switch(event)
      {
        /*----------------------------*/
        case UICC_ATR_OK:
        /*----------------------------*/
          /* Since this is a warm reset, the voltage level is the same and must
             therefore be marked as settled. */
          #ifdef UICC_ENABLE_IN_OUT_DETECTION
            UICC_enable_usim_in_isr();
          #endif

          UICC_card_profile[UICC_current_reader].voltage_settled = TRUE;
          UICC_card_ctrl_state[UICC_current_reader] = UICC_WARM_PPS_DETERMINATION;
          UICC_ctrl_pps(UICC_INITIATE_PPS);
          break;

        /*----------------------------*/
        case UICC_ATR_FAILED:
        case UICC_ATR_FAILED_NO_ANSWER:
        /*----------------------------*/
          #ifdef UICC_ENABLE_IN_OUT_DETECTION
            UICC_enable_usim_in_isr();
          #endif

          deactivate_card_connections(); /* Deactivate the Card */
          if(UICC_nof_cold_atr_attempt)
            UICC_nof_cold_atr_attempt--;
          UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
          break;

        /*----------------------------*/
        case UICC_DEACTIVATION_REQUESTED:
        /*----------------------------*/
          /* Deactivation of the card. */
          UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
          UICC_select_operative_card(UICC_current_reader);
          deactivate_card_connections(); /* Deactivate the card */
          force_signup_for_power_saving(); /* Power down allowed. */
          UICC_deactivation_result = UICC_DEACTIVATION_SUCCEEDED;
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.deactivation_result_invoked = TRUE;
              break;
            }
          #endif
          /* RELEASE DEACTIVATION SEMAPHORE. */
          UICC_release_semaphore(UICC_DEACTIVATION_SEM);
          break;

        /*----------------------------*/
        case UICC_COLD_RESET_REQUESTED:
        /*----------------------------*/
          activation_phase = TRUE;
          #ifdef SILENT_RESET
            #ifdef SW_RESET_CHV
              if(NU_silent_reset_flag)
              { /* Setting the bits will indicate to SI that an auto verification of the PIN is required. */
                NU_silent_reset_flag |= AUTO_CHV_VERIFICATION_PATTERN;
              }
            #else
              {
                /* Silent reset can not be performed in this state because the SIM is not up and running. The silent
                   reset falg is as a result cleard resulting in a normal mobile startup sequence. */
                NU_silent_reset_flag = FALSE;
              }
            #endif
          #endif
          UICC_card_ctrl_state[UICC_current_reader] = UICC_COLD_ACTIVATION;
          initialize_module();
          UICC_card_profile[UICC_current_reader].used_voltage =
            UICC_card_profile[UICC_current_reader].suggested_voltage;
          UICC_select_operative_card(UICC_current_reader);
          UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
          break;

          #ifdef UICC_ENABLE_IN_OUT_DETECTION
              /*----------------------------*/
              case UICC_CARD_INSERTED:
              /*----------------------------*/
                UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
                UICC_activate_inout_HISR(CARD_INSERTED);
                break;
                /*----------------------------*/
                case UICC_CARD_REMOVED:
                /*----------------------------*/
                UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
                UICC_activate_inout_HISR(CARD_REMOVED);
                break;
          #endif /*UICC_ENABLE_IN_OUT_DETECTION*/

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            card_ctrl_debug = UICC_card_ctrl_state[UICC_current_reader];
          #endif
          break;
      }
      break;

    /*==========================*/
    /* PPS Determination (Cold) */
    /*==========================*/
    case UICC_COLD_PPS_DETERMINATION:
      switch(event)
      {
        /*----------------------------*/
        case UICC_PPS_OK:
        case UICC_PPS_NOT_NEEDED:
        /*----------------------------*/
          /* PPS succeeded or was not needed. */
          UICC_card_ctrl_state[UICC_current_reader] = UICC_IDLE;
          UICC_nof_cold_atr_attempt = MAX_NOF_COLD_ATR_ATTEMPT;
          UICC_nof_warm_atr_attempt = MAX_NOF_WARM_ATR_ATTEMPT;
          UICC_nof_pps_attempt = MAX_NOF_PPS_ATTEMPT;
          UICC_activation_result = UICC_ACTIVATION_SUCCEEDED;
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.activation_result_invoked = TRUE;
              break;
            }
          #endif
          /* RELEASE ACTIVATION SEMAPHORE. */
          UICC_release_semaphore(UICC_ACTIVATION_SEM);
          break;

        /*----------------------------*/
        case UICC_PPS_INITIATE_WARM_RESET:
        /*----------------------------*/
          /* A warm reset is required but first check if this is acceptable. */
          if(UICC_nof_warm_atr_attempt--)
          {
            UICC_card_ctrl_state[UICC_current_reader] = UICC_WARM_ACTIVATION;
            UICC_ctrl_atr(UICC_INITIATE_WARM_ATR);
          }
          else
          {
            UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
            deactivate_card_connections(); /* Deactivate the card. */
            force_signup_for_power_saving(); /* Power down allowed. */
            UICC_activation_result = UICC_ACTIVATION_FAILED;
            #ifdef UICC_MODULE_TEST
              if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
              {
                UICC_simulated_function.activation_result_invoked = TRUE;
                break;
              }
            #endif
            /* RELEASE ACTIVATION SEMAPHORE. */
            UICC_release_semaphore(UICC_ACTIVATION_SEM);
          }
          break;

        /*----------------------------*/
        case UICC_PPS_NOT_COMPLETE:
        /*----------------------------*/
          /* PPS failed - re-activation the card. */
          deactivate_card_connections(); /* Deactivate the card. */
          if(UICC_nof_pps_attempt)
            UICC_nof_pps_attempt--;
          UICC_card_ctrl_state[UICC_current_reader] = UICC_COLD_ACTIVATION;
          UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
          break;

        /*----------------------------*/
        case UICC_PPS_FAILED:
        /*----------------------------*/
          /* PPS failed - check if re-activation is required. */
          deactivate_card_connections(); /* Deactivate the card. */

          UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
          deactivate_card_connections(); /* Deactivate the card. */
          force_signup_for_power_saving(); /* Power down allowed. */
          UICC_activation_result = UICC_ACTIVATION_FAILED;
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.activation_result_invoked = TRUE;
              break;
            }
          #endif
          /* RELEASE ACTIVATION SEMAPHORE. */
          UICC_release_semaphore(UICC_ACTIVATION_SEM);
          break;

        /*----------------------------*/
        case UICC_DEACTIVATION_REQUESTED:
        /*----------------------------*/
          /* Deactivation of the card is requested. */
          UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
          UICC_select_operative_card(UICC_current_reader);
          deactivate_card_connections(); /* Deactivate the card. */
          force_signup_for_power_saving(); /* Power down allowed. */
          UICC_deactivation_result = UICC_DEACTIVATION_SUCCEEDED;
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.deactivation_result_invoked = TRUE;
              break;
            }
          #endif
          /* RELEASE DEACTIVATION SEMAPHORE. */
          UICC_release_semaphore(UICC_DEACTIVATION_SEM);
          break;

        /*----------------------------*/
        case UICC_COLD_RESET_REQUESTED:
        /*----------------------------*/
          activation_phase = TRUE;
          #ifdef SILENT_RESET
            #ifdef SW_RESET_CHV
              if(NU_silent_reset_flag)
              { /* Setting the bits will indicate to SI that an auto verification of the PIN is required. */
                NU_silent_reset_flag |= AUTO_CHV_VERIFICATION_PATTERN;
              }
            #else
              {
                /* Silent reset can not be performed in this state because the SIM is not up and running. The silent
                   reset falg is as a result cleard resulting in a normal mobile startup sequence. */
                NU_silent_reset_flag = FALSE;
              }
            #endif
          #endif
          UICC_card_ctrl_state[UICC_current_reader] = UICC_COLD_ACTIVATION;
          initialize_module();
          UICC_card_profile[UICC_current_reader].used_voltage =
            UICC_card_profile[UICC_current_reader].suggested_voltage;
          UICC_select_operative_card(UICC_current_reader);
          UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
          break;

          #ifdef UICC_ENABLE_IN_OUT_DETECTION
              /*----------------------------*/
              case UICC_CARD_INSERTED:
              /*----------------------------*/
                UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
                UICC_activate_inout_HISR(CARD_INSERTED);
                break;
              /*----------------------------*/
              case UICC_CARD_REMOVED:
              /*----------------------------*/
                UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
                UICC_activate_inout_HISR(CARD_REMOVED);
                break;
          #endif /*UICC_ENABLE_IN_OUT_DETECTION*/

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            card_ctrl_debug = UICC_card_ctrl_state[UICC_current_reader];
          #endif
          break;
      }
      break;

    /*==========================*/
    /* PPS Determination (Warm) */
    /*==========================*/
    case UICC_WARM_PPS_DETERMINATION:
      switch(event)
      {
        /*----------------------------*/
        case UICC_PPS_OK:
        case UICC_PPS_NOT_NEEDED:
        /*----------------------------*/
          /* PPS succeeded or was not needed. */
          UICC_card_ctrl_state[UICC_current_reader] = UICC_IDLE;
          UICC_nof_cold_atr_attempt = MAX_NOF_COLD_ATR_ATTEMPT;
          UICC_nof_warm_atr_attempt = MAX_NOF_WARM_ATR_ATTEMPT;
          UICC_nof_pps_attempt = MAX_NOF_PPS_ATTEMPT;
          UICC_activation_result = UICC_ACTIVATION_SUCCEEDED;
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.activation_result_invoked = TRUE;
              break;
            }
          #endif
          /* RELEASE ACTIVATION SEMAPHORE. */
          UICC_release_semaphore(UICC_ACTIVATION_SEM);
          break;

        /*----------------------------*/
        case UICC_PPS_INITIATE_WARM_RESET:
        case UICC_PPS_NOT_COMPLETE:
        case UICC_PPS_FAILED:
        /*----------------------------*/
          /* PPS failed. */
          UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
          deactivate_card_connections(); /* Deactivate the card. */
          force_signup_for_power_saving(); /* Power down allowed. */
          UICC_activation_result = UICC_ACTIVATION_FAILED;
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.activation_result_invoked = TRUE;
              break;
            }
          #endif
          /* RELEASE ACTIVATION SEMAPHORE. */
          UICC_release_semaphore(UICC_ACTIVATION_SEM);
          break;

        /*----------------------------*/
        case UICC_DEACTIVATION_REQUESTED:
        /*----------------------------*/
          /* Deactivation of the card is requested. */
          UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
          UICC_select_operative_card(UICC_current_reader);
          deactivate_card_connections(); /* Deactivate the card. */
          force_signup_for_power_saving(); /* Power down allowed. */
          UICC_deactivation_result = UICC_DEACTIVATION_SUCCEEDED;
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.deactivation_result_invoked = TRUE;
              break;
            }
          #endif
          /* RELEASE DEACTIVATION SEMAPHORE. */
          UICC_release_semaphore(UICC_DEACTIVATION_SEM);
          break;

        /*----------------------------*/
        case UICC_COLD_RESET_REQUESTED:
        /*----------------------------*/
          activation_phase = TRUE;
          #ifdef SILENT_RESET
            #ifdef SW_RESET_CHV
              if(NU_silent_reset_flag)
              { /* Setting the bits will indicate to SI that an auto verification of the PIN is required. */
                NU_silent_reset_flag |= AUTO_CHV_VERIFICATION_PATTERN;
              }
            #else
              {
                /* Silent reset can not be performed in this state because the SIM is not up and running. The silent
                   reset falg is as a result cleard resulting in a normal mobile startup sequence. */
                NU_silent_reset_flag = FALSE;
              }
            #endif
          #endif
          UICC_card_ctrl_state[UICC_current_reader] = UICC_COLD_ACTIVATION;
          initialize_module();
          UICC_card_profile[UICC_current_reader].used_voltage =
            UICC_card_profile[UICC_current_reader].suggested_voltage;
          UICC_select_operative_card(UICC_current_reader);
          UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
          break;

          #ifdef UICC_ENABLE_IN_OUT_DETECTION
              /*----------------------------*/
              case UICC_CARD_INSERTED:
              /*----------------------------*/
                UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
                UICC_activate_inout_HISR(CARD_INSERTED);
                break;
              /*----------------------------*/
              case UICC_CARD_REMOVED:
              /*----------------------------*/
                UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
                UICC_activate_inout_HISR(CARD_REMOVED);
                break;
          #endif /*UICC_ENABLE_IN_OUT_DETECTION*/

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            card_ctrl_debug = UICC_card_ctrl_state[UICC_current_reader];
          #endif
          break;
      }
      break;

    /*====================================*/
    /* Idle ready to execute instructions */
    /*====================================*/
    case UICC_IDLE:
      switch(event)
      {
        /*----------------------------*/
        case UICC_INSTRUCTION_REQUESTED:
        /*----------------------------*/
          /* Received request for executiong an APDU instruction. */
          UICC_resign_from_power_saving(); /* Power down NOT allowed. */
          /* COPY (OR CHECK) THE INSTRUCTION DATA! */
          activation_phase = FALSE;
          UICC_card_ctrl_state[UICC_current_reader] = UICC_BUSY;
          UICC_select_operative_card(UICC_current_reader);
          if(UICC_card_profile[UICC_current_reader].T_used == UICC_T_0)
          { /* T=0 supported - and used. */
            UICC_t0_transport_handler(UICC_T0_INITIATE_INS);
          }
          else
          { /* T=1 supported - and used. */
            UICC_t1_transport_handler(UICC_T1_INITIATE_INS);
          }
          break;

        /*----------------------------*/
        case UICC_WARM_RESET_REQUESTED:
        /*----------------------------*/
          /* A warm reset is required. */
          activation_phase = TRUE;
          UICC_resign_from_power_saving(); /* Power down NOT allowed. */
          UICC_card_ctrl_state[UICC_current_reader] = UICC_WARM_ACTIVATION;
          UICC_select_operative_card(UICC_current_reader);
          UICC_ctrl_atr(UICC_INITIATE_WARM_ATR);
          break;

        /*----------------------------*/
        case UICC_COLD_RESET_REQUESTED:
        /*----------------------------*/
          /* A cold reset is required. */
          activation_phase = TRUE;
          #ifdef SILENT_RESET
            if(NU_silent_reset_flag)
            { /* Soft reset -> stay in this state and perform a soft poweron. */
              silent_reset_initialise();
            }
            else
          #endif
          {
            UICC_card_ctrl_state[UICC_current_reader] = UICC_COLD_ACTIVATION;
            initialize_module();
            UICC_card_profile[UICC_current_reader].used_voltage =
              UICC_card_profile[UICC_current_reader].suggested_voltage;
            UICC_select_operative_card(UICC_current_reader);
            UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
          }
          break;

        /*----------------------------*/
        case UICC_DEACTIVATION_REQUESTED:
        /*----------------------------*/
          /* Deactivation of the card is requested. */
          UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
          UICC_select_operative_card(UICC_current_reader);
          deactivate_card_connections(); /* Deactivate the card. */
          force_signup_for_power_saving(); /* Power down allowed. */
          UICC_deactivation_result = UICC_DEACTIVATION_SUCCEEDED;
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.deactivation_result_invoked = TRUE;
              break;
            }
          #endif
          /* RELEASE DEACTIVATION SEMAPHORE. */
          UICC_release_semaphore(UICC_DEACTIVATION_SEM);
          break;

          #ifdef UICC_ENABLE_IN_OUT_DETECTION
              /*----------------------------*/
              case UICC_CARD_INSERTED:
              /*----------------------------*/
                UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
                UICC_activate_inout_HISR(CARD_INSERTED);
                break;
              /*----------------------------*/
              case UICC_CARD_REMOVED:
              /*----------------------------*/
                UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
                UICC_activate_inout_HISR(CARD_REMOVED);
                break;
          #endif /*UICC_ENABLE_IN_OUT_DETECTION*/

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            card_ctrl_debug = UICC_card_ctrl_state[UICC_current_reader];
          #endif
          break;
      }
      break;

    /*============================*/
    /* Busy executing instruction */
    /*============================*/
    case UICC_BUSY:
      switch(event)
      {
        /*----------------------------*/
        case UICC_INSTRUCTION_SUCCEEDED:
        /*----------------------------*/
          /* The instruction execution was successfully. */
          UICC_card_ctrl_state[UICC_current_reader] = UICC_IDLE;
          /* PREPARE RESPONSE APDU */
          signup_for_power_saving(); /* Power down allowed. */
          UICC_transport_apdu_result = UICC_TRANSFER_SUCCEEDED;
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.apdu_result_invoked = TRUE;
              break;
            }
          #endif
          /* RELEASE INSTRUCTION SEMAPHORE. */
          UICC_release_semaphore(UICC_TRANSPORT_APDU_SEM);
          break;

        /*----------------------------*/
        case UICC_INSTRUCTION_FAILED:
        /*----------------------------*/
          /* The instruction execution was UNsuccessfully. */
          UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
          /* The deactivation is carried out here to ensure that the card is deactivated within WWT+5ETU. */
          deactivate_card_connections(); /* Deactivate the card. */
          signup_for_power_saving(); /* Power down allowed. */
          UICC_transport_apdu_result = UICC_TRANSFER_FAILED;
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.apdu_result_invoked = TRUE;
              break;
            }
          #endif
          /* RELEASE INSTRUCTION SEMAPHORE. */
          UICC_release_semaphore(UICC_TRANSPORT_APDU_SEM);
#if 0
		  if(NULL == recoverCallback)
        	  	invoke_callback( CARD_INSERTED, UICC_current_reader);
		  else
		  	  recoverCallback();
#else
		zDrv_HisrActivate(&s_simHalHisr);
		err_type_chtimeout = 1;
		uiccramlog_Printf("[%d][SIM] transfer failed, recover sim\n",zOss_GetTickCount());
#endif		  
          break;

        /*----------------------------*/
        case UICC_COLD_RESET_REQUESTED:
        /*----------------------------*/
          activation_phase = TRUE;
          #ifdef SILENT_RESET
            #ifdef SW_RESET_CHV
              if(NU_silent_reset_flag)
              { /* Setting the bits will indicate to SI that an auto verification of the PIN is required. */
                NU_silent_reset_flag |= AUTO_CHV_VERIFICATION_PATTERN;
              }
            #else
              {
                /* Silent reset can not be performed in this state because the SIM is not up and running. The silent
                   reset falg is as a result cleard resulting in a normal mobile startup sequence. */
                NU_silent_reset_flag = FALSE;
              }
            #endif
          #endif
          UICC_card_ctrl_state[UICC_current_reader] = UICC_COLD_ACTIVATION;
          initialize_module();
          UICC_card_profile[UICC_current_reader].used_voltage =
            UICC_card_profile[UICC_current_reader].suggested_voltage;
          UICC_select_operative_card(UICC_current_reader);
          UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
          break;

        /*----------------------------*/
        case UICC_DEACTIVATION_REQUESTED:
        /*----------------------------*/
          /* Deactivation of the card is requested. */
          UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
          UICC_select_operative_card(UICC_current_reader);
          deactivate_card_connections(); /* Deactivate the card. */
          force_signup_for_power_saving(); /* Power down allowed. */
          UICC_deactivation_result = UICC_DEACTIVATION_SUCCEEDED;
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.deactivation_result_invoked = TRUE;
              break;
            }
          #endif
          /* RELEASE DEACTIVATION SEMAPHORE. */
          UICC_release_semaphore(UICC_DEACTIVATION_SEM);
          break;

          #ifdef UICC_ENABLE_IN_OUT_DETECTION
              /*----------------------------*/
              case UICC_CARD_INSERTED:
              /*----------------------------*/
                UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
                UICC_activate_inout_HISR(CARD_INSERTED);
                break;
              /*----------------------------*/
              case UICC_CARD_REMOVED:
              /*----------------------------*/
                UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
                UICC_activate_inout_HISR(CARD_REMOVED);
                break;
          #endif /*UICC_ENABLE_IN_OUT_DETECTION*/

        /*----------------------------*/
        default:
        /*----------------------------*/
          #ifndef UICC_NO_DEBUG
            card_ctrl_debug = UICC_card_ctrl_state[UICC_current_reader];
          #endif
          break;
      }
      break;

    /*===========*/
    /* Test mode */
    /*===========*/
    case UICC_TESTMODE:
      switch(event)
      {
        /*----------------------------*/
        case UICC_COLD_RESET_REQUESTED:
        /*----------------------------*/
          /* Remain in this state until the activation has been
             carried out. and commence activation of the SIM to
             see if the electrical interface is operational. */
          activation_phase = TRUE;
          initialize_module();
          UICC_card_profile[UICC_current_reader].used_voltage =
            UICC_card_profile[UICC_current_reader].suggested_voltage;
          UICC_select_operative_card(UICC_current_reader);
          UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
          break;

        /*----------------------------*/
        case UICC_ATR_OK:
        /*----------------------------*/
          /* ATR succeeded - Since this is test mode NO PPS procedure
             will be executed (eventhoug the SIM says so)! */
          #ifdef UICC_ENABLE_IN_OUT_DETECTION
            UICC_enable_usim_in_isr();
          #endif

          signup_for_power_saving(); /* Power down allowed. */
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.activation_result_invoked = TRUE;
              break;
            }
          #endif
          /* RELEASE ACTIVATION SEMAPHORE. */
          UICC_release_semaphore(UICC_ACTIVATION_SEM);
          break;

        /*----------------------------*/
        case UICC_ATR_FAILED_NO_ANSWER:
        /*----------------------------*/
          /* ATR failed - check to see if reactivation can occur on another voltage level. */
          #ifdef UICC_ENABLE_IN_OUT_DETECTION
            UICC_enable_usim_in_isr();
          #endif

          deactivate_card_connections(); /* Deactivate the Card */

           /* Check if another voltage level is available. */
          if(UICC_another_voltage_level_available(&UICC_card_profile[UICC_current_reader].used_voltage))
          {
            /* Stay in this state and re-activate. */
            UICC_nof_cold_atr_attempt = MAX_NOF_COLD_ATR_ATTEMPT;
            UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
          }
          else
          { /* No alternative voltage level available - deactivate card. */
            UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
            force_signup_for_power_saving(); /* Power down allowed. */
            #ifdef UICC_MODULE_TEST
              if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
              {
                UICC_simulated_function.activation_result_invoked = TRUE;
                break;
              }
            #endif
            /* RELEASE ACTIVATION SEMAPHORE. */
            UICC_release_semaphore(UICC_ACTIVATION_SEM);
          }
          break;

        /*----------------------------*/
        case UICC_ATR_FAILED:
        /*----------------------------*/
          /* ATR failed - check to see if reactivation is required or if activation on another voltage
             level is possible. */
          #ifdef UICC_ENABLE_IN_OUT_DETECTION
            UICC_enable_usim_in_isr();
          #endif

          deactivate_card_connections(); /* Deactivate the Card */

          /* Check the number of remaining ATR attempts. */
          if(--UICC_nof_cold_atr_attempt)
          {
            /* There are still ATR retrial attemts left. Therefore stay in this
               state and redo the ATR sequence - at the same voltage level. */
            UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
          }
          else
          {
            /* The maximum number of ATR retrial attempts has been reached. Try on
               another voltage level if available or discard the card. */
            if(UICC_another_voltage_level_available(&UICC_card_profile[UICC_current_reader].used_voltage))
            {
              /* Stay in this state and re-activate. */
              UICC_nof_cold_atr_attempt = MAX_NOF_COLD_ATR_ATTEMPT;
              UICC_ctrl_atr(UICC_INITIATE_COLD_ATR);
            }
            else
            { /* No alternative voltage level available - deactivate card. */
              UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
              force_signup_for_power_saving(); /* Power down allowed. */
              UICC_activation_result = UICC_ACTIVATION_FAILED;
              #ifdef UICC_MODULE_TEST
                if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
                {
                  UICC_simulated_function.activation_result_invoked = TRUE;
                  break;
                }
              #endif
              /* RELEASE ACTIVATION SEMAPHORE. */
              UICC_release_semaphore(UICC_ACTIVATION_SEM);
            }
          }
          break;

        /*----------------------------*/
        case UICC_INSTRUCTION_REQUESTED:
        /*----------------------------*/
          /* Received request for executiong an APDU instruction. */
          UICC_resign_from_power_saving(); /* Power down NOT allowed. */
          /* COPY (OR CHECK) THE INSTRUCTION DATA! */
          activation_phase = FALSE;
          UICC_select_operative_card(UICC_current_reader);
          if(UICC_card_profile[UICC_current_reader].T_used == UICC_T_0)
          { /* T=0 supported - and used. */
            UICC_t0_transport_handler(UICC_T0_INITIATE_INS);
          }
          else
          { /* T=1 supported - and used. */
            UICC_t1_transport_handler(UICC_T1_INITIATE_INS);
          }
          break;

        /*----------------------------*/
        case UICC_INSTRUCTION_SUCCEEDED:
        /*----------------------------*/
          /* The instruction execution was successfully. */
          /* PREPARE RESPONSE APDU */
          signup_for_power_saving(); /* Power down allowed. */
          UICC_transport_apdu_result = UICC_TRANSFER_SUCCEEDED;
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.apdu_result_invoked = TRUE;
              break;
            }
          #endif
          /* RELEASE INSTRUCTION SEMAPHORE. */
          UICC_release_semaphore(UICC_TRANSPORT_APDU_SEM);
          break;

        /*----------------------------*/
        case UICC_INSTRUCTION_FAILED:
        /*----------------------------*/
          /* The instruction execution was UNsuccessfully. */
          UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
          /* The deactivation is carried out here to ensure that the card is deactivated within WWT+5ETU. */
          deactivate_card_connections(); /* Deactivate the card. */
          signup_for_power_saving(); /* Power down allowed. */
          UICC_transport_apdu_result = UICC_TRANSFER_FAILED;
          #ifdef UICC_MODULE_TEST
            if(UICC_module_test_state == UICC_MODULE_TEST_HW_DECOUPLED)
            {
              UICC_simulated_function.apdu_result_invoked = TRUE;
              break;
            }
          #endif
          /* RELEASE INSTRUCTION SEMAPHORE. */
          UICC_release_semaphore(UICC_TRANSPORT_APDU_SEM);
          break;

        /*----------------------------*/
        case UICC_DEACTIVATION_REQUESTED:
        /*----------------------------*/
          /* Deactivation of the SIM card. */
          UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
          deactivate_card_connections(); /* Deactivate the SIM */
          force_signup_for_power_saving(); /* Power down allowed. */
          UICC_deactivation_result = UICC_DEACTIVATION_SUCCEEDED;
          /* RELEASE DEACTIVATION SEMAPHORE. */
          UICC_release_semaphore(UICC_DEACTIVATION_SEM);
          break;

        /*----------------------------*/
        case UICC_TESTMODE_REQUESTED:
        /*----------------------------*/
          /* OK - stay here. */
          break;

          #ifdef UICC_ENABLE_IN_OUT_DETECTION
              /*----------------------------*/
              case UICC_CARD_INSERTED:
              /*----------------------------*/
                UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
                UICC_activate_inout_HISR(CARD_INSERTED);
                break;
              /*----------------------------*/
              case UICC_CARD_REMOVED:
              /*----------------------------*/
                UICC_card_ctrl_state[UICC_current_reader] = UICC_INACTIVE;
                UICC_activate_inout_HISR(CARD_REMOVED);
                break;
          #endif /*UICC_ENABLE_IN_OUT_DETECTION*/

        default:
          #ifndef UICC_NO_DEBUG
            card_ctrl_debug = UICC_card_ctrl_state[UICC_current_reader];
          #endif
          break;
      }
      break;

	  default:
	  	break;

  }

  #ifndef UICC_NO_DEBUG
    if(card_ctrl_debug != UICC_CARD_NONE)
    {
      UICC_log_data_ptr[0] = card_ctrl_debug;
      UICC_log_data_ptr[1] = event;
      UICC_log_data_ptr[2] = UICC_nof_cold_atr_attempt;
      UICC_log_data_ptr[3] = UICC_nof_warm_atr_attempt;
      UICC_log_data_ptr[4] = UICC_nof_pps_attempt;
      UICC_raise_exception(TRAP_UICC_CTRL, UICC_UNEXPECTED_EVENT, 5);

      card_ctrl_debug = UICC_CARD_NONE;
    }
  #endif
}


/****************************************************************************************
* Function:... UICC_main_control
* Parameters:. event: the event triggering the statemachine.
* Returns:.... None.
* Description: This function holds the main control state machine of the UICC Handler. From
*              this state machine it is possible to keep track on which card is currently
*              being operated.
* Created:.... 29.05.00 by Knud Nymann Mortensen
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
****************************************************************************************/
ubyte UICC_main_control(T_UICC_MAIN_EVENT event, T_UICC_CARD_SELECTOR card_selector)
{
  ubyte result = FALSE;

#ifdef UICC_DEBUG_INF
  UICC_debug[UICC_debug_count][0] = 2;  
  UICC_debug[UICC_debug_count][1] = UICC_main_ctrl_state;
  UICC_debug[UICC_debug_count][2] = event;
  UICC_debug_count = (UICC_debug_count+1) % 200;
#endif

  switch(UICC_main_ctrl_state)
  {
    /*======================*/
    case UICC_READY:
    /*======================*/
    {
      switch(event)
      {
        case UICC_OPERATE_CARD:
        {
          UICC_resign_from_power_saving(); /* Power down NOT allowed. */
          if(card_selector==UICC_PRIMARY_CARD)
            UICC_main_ctrl_state = UICC_CARD1_OPERATING;
          else
            UICC_main_ctrl_state = UICC_CARD2_OPERATING;
          result = TRUE;
          break;
          }

        default:
        {
          #ifndef UICC_NO_DEBUG
            main_ctrl_debug = UICC_main_ctrl_state;
          #endif
          }
      }
      break;
      }

    /*======================*/
    case UICC_CARD1_OPERATING:
    /*======================*/
    {
      switch(event)
      {
        case UICC_OPERATION_ENDED:
        {
          if(card_selector==UICC_PRIMARY_CARD)
          {
            UICC_main_ctrl_state = UICC_READY;
            signup_for_power_saving(); /* Power down allowed. */
            result = TRUE;
          }
          else
          {
            /* Second card not allowed here, return FALSE. */
            result = FALSE;
          }
          break;
          }

        default:
        {
          #ifndef UICC_NO_DEBUG
            main_ctrl_debug = UICC_main_ctrl_state;
          #endif
          }
      }
      break;
      }

    /*======================*/
    case UICC_CARD2_OPERATING:
    /*======================*/
    {
      switch(event)
      {
        case UICC_OPERATION_ENDED:
        {
          if(card_selector==UICC_PRIMARY_CARD)
          {
            /* Primary card not allowed here, return FALSE. */
            result = FALSE;
          }
          else
          { /* Second card. */
            UICC_main_ctrl_state = UICC_READY;
            signup_for_power_saving(); /* Power down allowed. */
            result = TRUE;
          }
          break;
          }

        default:
        {
          #ifndef UICC_NO_DEBUG
            main_ctrl_debug = UICC_main_ctrl_state;
          #endif
          }
      }
      break;
      }

    /*======================*/
    default:
    /*======================*/
    {
      #ifndef UICC_NO_DEBUG
        main_ctrl_debug = UICC_main_ctrl_state;
      #endif
      break;
      }
  }

  #ifndef UICC_NO_DEBUG
    if(main_ctrl_debug != UICC_MAIN_NONE)
    {
	  UICC_log_data_ptr[0] = main_ctrl_debug;
	  UICC_log_data_ptr[1] = event;
	  UICC_raise_exception(TRAP_UICC_CTRL, UICC_MAIN_UNEXPT_EVENT, 4);

      main_ctrl_debug = UICC_MAIN_NONE;
    }
  #endif

  return(result);
}

void UICC_ISR_Process(void)
{

  //UINT32 dwSimStatus;
  //UINT32 dwFifoStatus; 
  //dwSimStatus = SIM_MIS;
  //dwFifoStatus = SIM_FIFO_MIS;

#ifdef UICC_DEBUG_INF
UICC_debug[UICC_debug_count][0] = 11;  
UICC_debug[UICC_debug_count][1] = (UINT8)SIM_MIS;
UICC_debug[UICC_debug_count][2] = (UINT8)SIM_FIFO_MIS;
UICC_debug_count = (UICC_debug_count+1) % 200;
#endif

   
while(SIM_FIFO_MIS!=0)
{
		UICC_usim_fifo_lisr();
		//dwFifoStatus = SIM_FIFO_MIS;

#ifdef UICC_DEBUG_INF
UICC_debug[UICC_debug_count][0] = 11;  
UICC_debug[UICC_debug_count][1] = 0xFE;
UICC_debug[UICC_debug_count][2] = (UINT8)SIM_FIFO_MIS;
UICC_debug_count = (UICC_debug_count+1) % 200;	
#endif
}

  if(SIM_MIS)
  {
	while(SIM_FIFO_MIS!=0)
	{
			UICC_usim_fifo_lisr();
			//dwFifoStatus = SIM_FIFO_MIS;

	}
    if(SIM_MIS & 0x1)
	{
        UICC_usim_err_lisr();
	}
    if(SIM_MIS & 0x4)
	{
        UICC_usim_ok_lisr();
	}
    if(SIM_MIS & 0x2)
	{
      UICC_usim_in_out_lisr();
	}
  }
  else
  {
  //  int err
  }
  SIM_ICR = 0x7;

#ifdef UICC_DEBUG_INF
//dwSimStatus = SIM_MIS;
//dwFifoStatus = SIM_FIFO_MIS;
UICC_debug[UICC_debug_count][0] = 11;  
UICC_debug[UICC_debug_count][1] = (UINT8)SIM_MIS;
UICC_debug[UICC_debug_count][2] = (UINT8)SIM_FIFO_MIS;
UICC_debug_count = (UICC_debug_count+1) % 200; 
#endif
}

/* End of file.              */
