/*******************************************************************************
 * Copyright (C) 2007, ZTE Corporation.
 *
 * File Name:    
 * File Mark:    
 * Description:  This source file contains the Protocol and Parameter Selection (PPS) statemachine engine. 
 * Others:        
 * Version:       v0.1
 * Author:        mtr
 * Date:          2006-06-01
 * History 1:      
 *     Date: 
 *     Version:
 *     Author: 
 *     Modification:  
 * History 2: 
  ********************************************************************************/


/****************************************************************************
*                                              Include files
****************************************************************************/
#include "uicc_stub.h"
#include "uicc.h"
#include "uicc_ctrl.h"
#include "uicc_access.h"
#include "uicc_drv.h"
#include "uicc_trc.h"
#include "drvs_ramlog.h"


/****************************************************************************
* 	                                           Local Macros
****************************************************************************/
//#define UICC_SUPPORT_CRC  /*  Use this define when the SIM driver support CRC checksum*/
#define T_CHAR               200000 /* [us] The character transmission guard time. */
#define T_ATR_PPS_SPACE        5000 /* [us] The spacing between ATR and PPS. */
#define T_SPEED_CHANGE_DELAY   2000 /* [us] Time value to ensure that the UART speed change 
                                                is requested after the last PPS response character
                                                has been received from the SIM. */
#define T_TXSPC                400 /* [us] The space bewteen Tx characters. */
    
#define PPSS                0xFF /* The value of the Initial chacarter PPSS. */
#define F_372_IDX              1 /* Table lookup value for F = 372. */
#define F_512_IDX              9 /* Table lookup value for F = 512. */
#define D_1_IDX                1 /* Table lookup value for D = 1. */
#define D_8_IDX                4 /* Table lookup value for D = 8. */
    
    /*  Parameter character masks  */
#define PPS123_MASK(mask) (mask & 0x70)
#define PPS23_MASK(mask)  (mask & 0x60)
#define PPS3_MASK(mask)   (mask & 0x40)
#define PPS2_MASK(mask)   (mask & 0x20)
#define PPS1_MASK(mask)   (mask & 0x10)
    
#define PROTOCOL_MASK(T)  (T    & 0x0F)
#define BIT5_MASK(mask)   (mask & 0x0010)
#define BIT8_MASK(mask)   (mask & 0x0080)


/****************************************************************************
* 	                                           Local Types
****************************************************************************/
typedef enum {
  INT_VALUE,
  BIT_MASK
} T_INDICATION;


/****************************************************************************
* 	                                           Local Constants
****************************************************************************/

/****************************************************************************
* 	                                           Local Variables
****************************************************************************/
static ubyte pck_sum;
static ubyte pps0_request;
static ubyte pps1_request;
static ubyte ppss_response;
static ubyte pps0_response;
static ubyte pps1_response;
static ubyte pck_response;
static ubyte f;
static ubyte d;
    
#ifndef UICC_NO_DEBUG
  static T_UICC_PPS_CTRL_STATE debug = UICC_PPS_NONE;
#endif
    

/****************************************************************************
* 	                                          Global Constants
****************************************************************************/
ushort const UICC_clock_rate_table[] =
{ /*   FI,          Fi */
  /*{  0 },*/       372,
  /*{  1 },*/       372,
  /*{  2 },*/       558,
  /*{  3 },*/       744,
  /*{  4 },*/      1116,
  /*{  5 },*/      1488,
  /*{  6 },*/      1860,
  /*{  7 },*/    0xFFFF,
  /*{  8 },*/    0xFFFF,
  /*{  9 },*/       512,
  /*{ 10 },*/       768,
  /*{ 11 },*/      1024,
  /*{ 12 },*/      1536,
  /*{ 13 },*/      2048,
  /*{ 14 },*/    0xFFFF,
  /*{ 15 },*/    0xFFFF  
};

ushort const UICC_baud_rate_table[] =
{ /*   DI,          Di */
  /*{  0 },*/    0xFFFF,
  /*{  1 },*/         1,
  /*{  2 },*/         2,
  /*{  3 },*/         4,
  /*{  4 },*/         8,
  /*{  5 },*/        16,
  /*{  6 },*/        32,
  /*{  7 },*/        64,
  /*{  8 },*/        12,
  /*{  9 },*/        20,
  /*{ 10 },*/    0xFFFF,
  /*{ 11 },*/    0xFFFF,
  /*{ 12 },*/    0xFFFF,
  /*{ 13 },*/    0xFFFF,
  /*{ 14 },*/    0xFFFF,
  /*{ 15 },*/    0xFFFF
};


/****************************************************************************
* 	                                          Global Variables
****************************************************************************/
T_UICC_PPS_CTRL_STATE UICC_pps_ctrl_state = UICC_PPS_READY;

/****************************************************************************
* 	                                          Global Function Prototypes
****************************************************************************/

/****************************************************************************
* 	                                          Function Definitions
****************************************************************************/

/****************************************************************************************
* Function:... find_best_lower_parameter_pair
* Parameters:. *f_idx: clock rate convention factor (coding: as received from the card).
*              *d_idx: baud rate adjustment factor (coding: as received from the card).
*              terminal_baud_max: The macimum baudrate supported by the terminal.
* Returns:.... None.
* Description: This function find the better match of F and D - under the following 
*              conditions:
*              - The baud rate comming from new F & D must be below the greater of either 
*                the terminal max. baud rate or the card max. baud rate.
*              - New F must be less than or equal to Fi (the value indicated by the card).
*              - New D must be less than or equal to Di (the value indicated by the card).
* 
* Created:.... 24.01.05 by Knud Nymann Mortensen
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
****************************************************************************************/
static void find_best_lower_parameter_pair(ubyte *F_idx, ubyte *D_idx, ulong terminal_baud_max)
{
  ubyte F_cnt_idx, D_cnt_idx;
  ulong baud_max;
  ubyte current_best_fit_F_idx = F_372_IDX;
  ubyte current_best_fit_D_idx = D_1_IDX;
  ulong F_val = (ulong)UICC_clock_rate_table[*F_idx];
  ulong D_val = (ulong)UICC_baud_rate_table[*D_idx];

  baud_max = ((3250000 * D_val) / F_val); /* Calculate the baud rate supported by the card. */ 

  if(terminal_baud_max < baud_max)
    baud_max = terminal_baud_max; /* The terminal sets the max boundary. */
  /* else -> the card will form the max boundary. */    

  /* Examine all values of F & D to find the best fit i.e. the closest lower value to MAX. */
  for(F_cnt_idx=0; F_cnt_idx<0x0F; F_cnt_idx++)
  {
    F_val = (ulong)UICC_clock_rate_table[F_cnt_idx];

    for(D_cnt_idx=0; D_cnt_idx<0x0F; D_cnt_idx++)
    {
      D_val = (ulong)UICC_baud_rate_table[D_cnt_idx];

    /* Check:
       - if the values F & D values are valid ISO values (different from 0xFFFF).
       - if the values of F and D can meet the requirements in terms of a valid BRF 
         value (F/(4*D) must be an integer) 
       - if the valid operational speed is below the maximum threshold (terminal or card).
       - if F lies within the limits of Fd and Fi.
       - if D lies within the limits of Dd and Di. */
      if(((F_val != 0xFFFF) && (D_val != 0xFFFF)) &&
         (F_val % (4 * D_val) == 0) &&
         (((3250000 * D_val) / F_val) < baud_max) &&
         (F_val <= UICC_clock_rate_table[*F_idx]) &&
         (D_val <= UICC_baud_rate_table[*D_idx]))
      { /* A valid combination. Now check if this combination results in a 'better' (larger) baud 
           rate than the compared to the previous found. */
        if(((3250000 * D_val) / F_val) >
           ((3250000 * (ulong)UICC_baud_rate_table[current_best_fit_D_idx]) / 
             (ulong)UICC_clock_rate_table[current_best_fit_F_idx]))
        {
          current_best_fit_F_idx = F_cnt_idx;
          current_best_fit_D_idx = D_cnt_idx;
        }
      }
    } 
  } 
  
  *F_idx = current_best_fit_F_idx;
  *D_idx = current_best_fit_D_idx;
} 

/****************************************************************************************
* Function:... parameters_supported
* Parameters:. f: clock rate convention factor (coding: as received from the card).
*              d: baud rate adjustment factor (coding: as received from the card).
*              adjustment_allowed: Indicates wheter or not it is allowed by thios function 
*              to suggest another suitable set of F & D values.
* Returns:.... Whether or not the parameters are supported.
* Description: This function investigates if the transmission speed parameters can be 
*              meet by the mobile. If allowed, the function will return alternative F & D
*              values.
* 
* Created:.... 04.09.01 by Knud Nymann Mortensen
* Modified:... 24.01.05 by Knud Nymann Mortensen
*                Enhnaced the selection of alternative values for F & D.
****************************************************************************************/
static BOOL parameters_supported(ubyte *f, ubyte *d, BOOL adjustment_allowed)
{
  BOOL return_value = FALSE;

  /*  Check that both factors are valid according to ISO/IEC7816-3 and that the terminal 
      can meet the requested speed. */
  if(UICC_HIGHSPEED_SUPPORTED && 
    (UICC_baud_rate_table[*d] != 0xFFFF) && (UICC_clock_rate_table[*f] != 0xFFFF))
  {
    ulong f_val = (ulong)UICC_clock_rate_table[*f];
    ulong d_val = (ulong)UICC_baud_rate_table[*d];

    /* First check if the values of F and D can meet the requirements in terms of a 
       valid BRF value (F//4*D) must be an integer) but also in terms of the valid 
       operational speed that can be accepted by the terminal. */
    if((f_val % (4 * d_val) == 0) &&
       (((3250000 * d_val) / f_val) <= UICC_MAX_BIT_RATE))
    {
      return_value = TRUE; /* Parameters supported. */
    }
    else
    {
      if(adjustment_allowed)
      {
        /* Find suitable values of F & D in the interval: Fd <= F <= Fi and Dd <= D <= Di */
        find_best_lower_parameter_pair(f, d, UICC_MAX_BIT_RATE);
        return_value = TRUE; /* Parameters supported. */      
      }
    }
  }
  return(return_value); 
}

/*****************************************add 071219***********************************************
* Function:... T1_supported
* Parameters:. protocol: The protocol that needs to be checked.
*              symbolic: indicates the coding of the input protocol parameter (unique 
*                        value or bit field). 
* Returns:.... whether or not the T1 protocol is supported or not.
* Description: This function investigates if the T1 protocol is suppported by the card.
****************************************************************************************/
static BOOL T1_supported(ushort protocol, T_INDICATION symbolic)
{
  #ifdef UICC_SUPPORT_CRC
    if((((symbolic==INT_VALUE) && ((protocol & 0x000F) == 1)) ||
         ((symbolic==BIT_MASK)  && (protocol & 0x0002))) && UICC_T_1_SUPPORTED)
  #else
    /* If NOT support CRC then only using T1 if edc = 0 = LRC*/
    if((((symbolic==INT_VALUE) && ((protocol & 0x000F) == 1)) ||
      ((symbolic==BIT_MASK)  && (protocol & 0x0002))) && UICC_T_1_SUPPORTED && 
      ((UICC_card_profile[UICC_current_reader].edc & 1) == 0))
  #endif
  {
    return(TRUE);
  }
  else
    return(FALSE);      
}
/*******************************************add 071219*********************************************
* Function:... T0_supported
* Parameters:. protocol: The protocol that needs to be checked.
*              symbolic: indicates the coding of the input protocol parameter (unique 
*                        value or bit field). 
* Returns:.... whether or not the T0 protocol is supported or not.
* Description: This function investigates if the T0 protocol is suppported by the card.
****************************************************************************************/
static BOOL T0_supported(ushort protocol, T_INDICATION symbolic)
{
  if(((symbolic==INT_VALUE) && (((protocol & 0x000F) == 0) || (protocol & 0x000F) == 15)) ||
      ((symbolic==BIT_MASK)  && (protocol & 0x0001))) 
  {
    return(TRUE);
  }
  else
    return(FALSE);      
}
/****************************************************************************************
* Function:... protocol_supported
* Parameters:. protocol: The protocol that needs to be checked.
*              symbolic: indicates the coding of the input protocol parameter (unique 
*                        value or bit field). 
* Returns:.... whether or not the given protocol is supported or not.
* Description: This function investigates if a protocol is suppported by the card.* 
* Created:.... 04.09.01 by Knud Nymann Mortensen
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
****************************************************************************************/
static BOOL protocol_supported(ushort protocol, T_INDICATION symbolic)
{
  BOOL return_value = TRUE;

  /* If present T=1 will always be the selected protocol over T=0, that is of course if the
     terminal supports the T=1 protocol. change 071219*/

  /* Base the protocol selection on  */
  switch(UICC_card_profile[UICC_current_reader].protocol_mode)
  {
    case UICC_AUTOMATIC_PROTOCOL_SELECTION:
    case UICC_T1_PROTOCOL_SELECTION:
      if(T1_supported(protocol, symbolic))
      { /* T=1 Supported. */
        UICC_card_profile[UICC_current_reader].T_used = UICC_T_1;
      }
      else if(T0_supported(protocol, symbolic))
      { /* T=0 Supported */
        UICC_card_profile[UICC_current_reader].T_used = UICC_T_0;    
      }
      else
      { /* No protocol supported. */
        return_value = FALSE;
      }
      break;

    case UICC_T0_PROTOCOL_SELECTION:
    default:
      if(T0_supported(protocol, symbolic))
      { /* T=0 Supported. */
        UICC_card_profile[UICC_current_reader].T_used = UICC_T_0;
      }
      else if(T1_supported(protocol, symbolic))
      { /* T=1 Supported */
        UICC_card_profile[UICC_current_reader].T_used = UICC_T_1;    
      }
      else
      { /* No protocol supported. */
        return_value = FALSE;
      }
      break;
  } 
  return(return_value);
}
    
/****************************************************************************************
* Function:... evaluate_atr_data
* Parameters:. None
* Returns:.... the next main control state.
* Description: This function investigates the data received from the card in the ATR 
*              sequence in order to establish the further activity.
* Created:.... 04.09.01 by Knud Nymann Mortensen
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
****************************************************************************************/
static T_UICC_PPS_CTRL_STATE evaluate_atr_data(void)
{
  T_UICC_PPS_CTRL_STATE return_state = UICC_PPS_READY;
  ushort           ta2          = UICC_card_profile[UICC_current_reader].ta2_specific;

  switch(UICC_nof_pps_attempt)
  {
    case 0:
      /* Last attempt - no PPS should be initiated. */
      UICC_card_profile[UICC_current_reader].f_used = UICC_F_DEFAULT; 
      UICC_card_profile[UICC_current_reader].d_used = UICC_D_DEFAULT; 
      return(UICC_NO_PPS_NEEDED);
      /*break; removed to avoid warning. */   

    case 1:
      /* Perfom PPS with default F and D values. */
      f = UICC_F_DEFAULT;
      d = UICC_D_DEFAULT;
      break;

    default:
      f  = UICC_card_profile[UICC_current_reader].fi;
      d  = UICC_card_profile[UICC_current_reader].di;  
      break;
  }         
  
  /* First investigate if the current card session takes place in specific mode (TA2 
     present) or in negotiable ode (TA2 absent). */
  if(ta2) 
  { /*===================*/
    /* --SPECIFIC MODE-- */
    /*===================*/
    if(BIT5_MASK(ta2))  /* Investigate b5 of TA(2). */
    { /* b5=1 -> F&D implicit defined (Fd & Dd). */
      UICC_card_profile[UICC_current_reader].f_used = UICC_F_DEFAULT; 
      UICC_card_profile[UICC_current_reader].d_used = UICC_D_DEFAULT; 

      if(protocol_supported(ta2, INT_VALUE)) /* Is the suggested protocol supported? */
      {
        return_state = UICC_NO_PPS_NEEDED; /* Continue card session */
      }       
      else
      { /* Protocol not supported -> can the mode be changed? */
        if(BIT8_MASK(ta2))  /* Investigate b8 of TA(2). */
        { /* b8=1 -> Unable to change mode. */
          return_state = UICC_PROTOCOL_PARAMETERS_NOT_SUPPORTED;
        }
        else
        { /* b8=0 -> Capable of changing mode. */
          return_state = UICC_WARM_RESET_REQUIRED;
        }
      }
    } 
    else
    { /* b5=0 -> F&D defined by TA(1). */
      if(protocol_supported(ta2, INT_VALUE) && parameters_supported(&f, &d, FALSE))
      {
        /* Protocol supported and the speed parameters can be meet -> change the 
           transmission setup of the UART. */
        UICC_card_profile[UICC_current_reader].f_used = f; 
        UICC_card_profile[UICC_current_reader].d_used = d; 
        UICC_transmission_rate_control(UICC_clock_rate_table[f] / (4 * UICC_baud_rate_table[d]));
        return_state = UICC_NO_PPS_NEEDED; /* Continue card session */ 
      }
      else
      { /* Protocol or parameter not supported -> can the mode be changed? */
        if(BIT8_MASK(ta2))  /* Investigate b8 of TA(2). */
        { /* b8=1 -> Unable to change mode. */
          return_state = UICC_PROTOCOL_PARAMETERS_NOT_SUPPORTED;
        }
        else
        { /* b8=0 -> Capable of changing mode. */
          return_state = UICC_WARM_RESET_REQUIRED;
        }
      }
    }
  } /* End Specific mode */
  else
  { /*=====================*/
    /* --NEGOTIABLE MODE-- */
    /*=====================*/
    if(protocol_supported(UICC_card_profile[UICC_current_reader].T_indicated, BIT_MASK))
    { /* Protocol supported */
    
      /* Protocol and Parametor selection is NOT required: 
         T=1 supported: If only T=0 OR T=1 was offered
         T=1 not supported: If T=0 protocol was offered 
         or if alternative transmissions speed parameters has been suggested
         or if this is going to be the third PPS request in a row (27.11.2.6). */
/*change 071219*/
      if( 
          ((UICC_T_1_SUPPORTED && 
           (((UICC_card_profile[UICC_current_reader].T_indicated  & 3)== 1)   ||
            ((UICC_card_profile[UICC_current_reader].T_indicated & 3) == 2)))||
           (!UICC_T_1_SUPPORTED &&
            ((UICC_card_profile[UICC_current_reader].T_indicated & 0x01) == 1))) 
          && 
          ((f <= 1) && (d == 1)) /* F=0 or F=1 -> Fi=372; D=1 -> Di=1) */
          && 
          (UICC_nof_pps_attempt != 1)
        )
/*end change*/
      { /* No PPS required. */
        UICC_card_profile[UICC_current_reader].f_used = UICC_F_DEFAULT; 
        UICC_card_profile[UICC_current_reader].d_used = UICC_D_DEFAULT; 
        return_state = UICC_NO_PPS_NEEDED; /* Continue card session */ 
      }
      else
      { /* Initiate PPS - but first chech the validity of the suggested bitrate- and clock-conversion 
           factor parameters. */
       if((UICC_nof_pps_attempt > 1) && (parameters_supported(&f, &d, TRUE)))
       { /* Suggested parameters accepted (supported).*/
          pps0_request = 0x10; /* Set the PPS1_request bit. */
        }
        else
        {
          /* One or both of the suggested factor values are not supported or are not valid (RFU) 
             and according to the Type Approval test 27.11.2.4 should the ME perform the PPS 
             procedure as if speed enhancement was not suggested (PPS0=0x00).
             OR this is the 3rd PPS request (here should PPS0=0X00). */
          pps0_request = 0x00; /* Clear the PPS1_request bit. */
        }

         /* Chech if the protocol can/needs to be changed to T1 */
        if(UICC_card_profile[UICC_current_reader].T_used == UICC_T_1)
        { /* T=1 supported and indicated. */
          pps0_request |= 0x01; /* Mark T=1 to be used in the PPS request. */
        }
        
        return_state = UICC_PREPARING_PPS;
      }      
    }
    else
    { /* Protocol(s) not supported. */
       return_state = UICC_PROTOCOL_PARAMETERS_NOT_SUPPORTED;
    }
  }

  return(return_state);
}


/*---------------------------------------------*/
/* Global functions.                            */
/*---------------------------------------------*/

void UICC_reset_pps(void)
{
  UICC_pps_ctrl_state = UICC_PPS_READY;
}


ubyte UICC_get_PPS_info(ubyte *dest)
{
  *(dest++) = PPSS;
  *(dest++) = pps0_request;
  *(dest++) = pps1_request;
  *(dest++) = pck_sum;

  *(dest++) = ppss_response;
  *(dest++) = pps0_response;
  *(dest++) = pps1_response;
  *(dest++) = pck_response;
  return 8;
}

typedef struct _T_UiccDebugInfo
{
	char debugInfo[41];
}
T_UiccDebugInfo;

T_UiccDebugInfo UiccPpsStatusInfo[] =
{
    "UICC_PPS_NONE",
  "UICC_PPS_READY",
  "UICC_NO_PPS_NEEDED",
  "UICC_PROTOCOL_PARAMETERS_NOT_SUPPORTED",       
  "UICC_WARM_RESET_REQUIRED",       

  "UICC_PREPARING_PPS",
  "UICC_PPSS_REQ",
  "UICC_PPS0_REQ",
  "UICC_PPS1_REQ",
  "UICC_PCK_REQ",

  "UICC_PPSS_RESP",
  "UICC_PPS0_RESP",
  "UICC_PPS1_RESP",
  "UICC_PPS2_RESP",
  "UICC_PPS3_RESP",

  "UICC_PCK_RESP",
  "UICC_PPS_COMPLETED"
};


T_UiccDebugInfo UiccPpsEventInfo[] =
{
    "UICC_DUMMY_INT_EVENT",
  "UICC_TIMER_EXPIRED",          
  "UICC_GUARD_TIMER_EXPIRED",
  "UICC_CHARACTER_OK",
  "UICC_T0_END",
  "UICC_T1_END",
  "UICC_DMA_END",
  "UICC_PARITY_ERROR",
  "UICC_OVERRUN_ERROR",
  "UICC_CARD_IN",
  "UICC_CARD_OUT",
  "UICC_CHTIMEOUT",
  "UICC_T1_PARITY_ERROR",
  "UICC_T1_BWT_EXPIRED",
  "UICC_UNKNOWN_INT",
  "UICC_FIFO_INT",
  "UICC_RX_LSREQ",  
  "UICC_TX_LSREQ",  
  "UICC_RX_SREQ",  
  "UICC_TX_SREQ",    
  "UICC_RX_LBREQ",  
  "UICC_TX_LBREQ",  
  "UICC_RX_BREQ",  
  "UICC_TX_BREQ",  
  "UICC_TXF_OFL_IRQ",  
  "UICC_RXF_UFL_IRQ",
  "UICC_FIFO_UNKNOWN_INT",
    "UICC_LAST_INT_EVENT",
    "UICC_INITIATE_PPS",
  "UICC_RESET_PPS",              
  "UICC_LAST_PPS_EVENT"
};


/*******************************************************************************
* Function:... UICC_ctrl_pps
* Parameters:. event: The event triggering the statemachine.
* Returns:.... -
* Description: This statemachine/engine is controlling the Protocol and
*              Parameter Selection (PPS) procedure.
* Created:.... 02.06.00 by Knud Nymann Mortensen (KNM)
* Modified:... DD.MM.YY by (Full name / initials)
*                Modifications note.
*******************************************************************************/
void UICC_ctrl_pps(T_UICC_PPS_EVENTS event)
{  
#ifdef UICC_DEBUG_INF
UICC_debug[UICC_debug_count][0] = 3;  
UICC_debug[UICC_debug_count][1] = UICC_pps_ctrl_state;
UICC_debug[UICC_debug_count][2] = event;
UICC_debug_count = (UICC_debug_count+1) % 200;
#endif
  
  int i = 0;
  i = UICC_pps_ctrl_state;
    
zDrvRamlog_PRINTF(RAMLOG_MOD_UICC, "[%d]state=%s,event=%s\n", zOss_GetTickCount(), (char *)(&UiccPpsStatusInfo[i]),(char *)(&UiccPpsEventInfo[event]));
uiccramlog_Printf("[%d]state=%s,event=%s\n", zOss_GetTickCount(), (char *)(&UiccPpsStatusInfo[i]),(char *)(&UiccPpsEventInfo[event]));

  switch(UICC_pps_ctrl_state)
  {
    /*===================================*/
    /* Ready to handle the PPS procedure */
    /*===================================*/
    case UICC_PPS_READY:
      switch(event)
      {
        case UICC_INITIATE_PPS:
          pps0_request  = 0x00;
          ppss_response = 0x00;
          pps0_response = 0x00;
          pps1_response = 0x00;
          pck_response  = 0x00; 
          UICC_pps_ctrl_state = evaluate_atr_data();     
          UICC_start_timer(T_ATR_PPS_SPACE);
#ifdef UICC_NOT_TIMER
         UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif


          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;

    /*===================================*/
    /* PPS is not needed after all.      */
    /*===================================*/
    case UICC_NO_PPS_NEEDED:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          UICC_pps_ctrl_state = UICC_PPS_READY;
          UICC_card_ctrl(UICC_PPS_NOT_NEEDED);
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;
        
    /*==================================================*/
    /* The protocol and/or parameters is not supported. */
    /*==================================================*/
    case UICC_PROTOCOL_PARAMETERS_NOT_SUPPORTED:       
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          UICC_pps_ctrl_state = UICC_PPS_READY;
          UICC_card_ctrl(UICC_PPS_FAILED);        
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;
        
    /*======================================*/
    /* A Warm reset is required to continue. */
    /*======================================*/
    case UICC_WARM_RESET_REQUIRED:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          UICC_pps_ctrl_state = UICC_PPS_READY;
          UICC_card_ctrl(UICC_PPS_INITIATE_WARM_RESET);       
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;       
  
    /*======================================*/
    /* Getting ready for the PPS procedure. */
    /*======================================*/
    case UICC_PREPARING_PPS:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
          UICC_pps_ctrl_state = UICC_PPSS_REQ;
          pck_sum = PPSS;
          UICC_start_timer(T_CHAR);
          UICC_send_character(PPSS);
          break;       

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;

    /*======================*/
    /* Sending PPSS request */
    /*======================*/
    case UICC_PPSS_REQ:
      switch(event)
      {
        case UICC_CHARACTER_OK:
          /* The character interrupt from transmitting PPSS. Now make
             appropriate distance before sending PPS0. */
          UICC_pps_ctrl_state = UICC_PPS0_REQ;
          UICC_start_guard_timer(T_TXSPC);
#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_GUARD_TIMER_EXPIRED);
#endif

      break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(T_CHAR);
#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

          break;

        case UICC_TIMER_EXPIRED:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          UICC_pps_ctrl_state = UICC_PPS_READY;
          UICC_card_ctrl(UICC_PPS_NOT_COMPLETE);        
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;

    /*======================*/
    /* Sending PPS0 request */
    /*======================*/
    case UICC_PPS0_REQ:
      switch(event)
      {
        case UICC_GUARD_TIMER_EXPIRED:
          /* Send PPS0 request. */
          pck_sum ^= pps0_request;
          UICC_start_timer(T_CHAR);
          UICC_send_character(pps0_request);
          break;

        case UICC_CHARACTER_OK:
          /* The character interrupt from transmitting PPS0. Now make
             appropriate distance before sending PPS1 or PCK. */
          if(pps0_request & 0xF0)
            UICC_pps_ctrl_state = UICC_PPS1_REQ;
          else
            UICC_pps_ctrl_state = UICC_PCK_REQ;

          UICC_start_guard_timer(T_TXSPC);

#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_GUARD_TIMER_EXPIRED);
#endif

          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(T_CHAR);

#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

  
          break;

        case UICC_TIMER_EXPIRED:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          UICC_pps_ctrl_state = UICC_PPS_READY;
          UICC_card_ctrl(UICC_PPS_NOT_COMPLETE);        
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;

    /*======================*/
    /* Sending PPS1 request */
    /*======================*/
    case UICC_PPS1_REQ:
      switch(event)
      {
        case UICC_GUARD_TIMER_EXPIRED:
          /* Send PPS1 request. */
          pck_sum ^= pps1_request = d | (f << 4);
          UICC_start_timer(T_CHAR);
          UICC_send_character(pps1_request);
          break;

        case UICC_CHARACTER_OK:
          /* The character interrupt from transmitting PPS1. Now make
             appropriate distance before sending PCK. */
          UICC_pps_ctrl_state = UICC_PCK_REQ;
          UICC_start_guard_timer(T_TXSPC);


#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_GUARD_TIMER_EXPIRED);
#endif

		  
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(T_CHAR);


#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

	  
          break;

        case UICC_TIMER_EXPIRED:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          UICC_pps_ctrl_state = UICC_PPS_READY;
          UICC_card_ctrl(UICC_PPS_NOT_COMPLETE);        
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;

    /*=====================*/
    /* Sending PCK request */
    /*=====================*/
    case UICC_PCK_REQ:
      switch(event)
      {
        case UICC_GUARD_TIMER_EXPIRED:
          /* Send PCK request. */
          UICC_start_timer(T_CHAR);
          UICC_send_character(pck_sum);
          break;

        case UICC_CHARACTER_OK:
          /* The character interrupt from transmitting PCK. Now make ready to
             receive the PPS respons (PPSS response). */
          UICC_pps_ctrl_state = UICC_PPSS_RESP;
          UICC_start_timer(UICC_T_IWT);
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(T_CHAR);


#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

	  
          break;

        case UICC_TIMER_EXPIRED:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          UICC_pps_ctrl_state = UICC_PPS_READY;
          UICC_card_ctrl(UICC_PPS_NOT_COMPLETE);        
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;

    /*=========================*/
    /* Receiving PPSS response */
    /*=========================*/
    case UICC_PPSS_RESP:
      switch(event)
      {
        case UICC_CHARACTER_OK:
          /* The character interrupt originates from PPSS response. Now make
             ready to receive PPS0 response. */
          pck_sum = ppss_response = UICC_get_character();
          UICC_pps_ctrl_state = UICC_PPS0_RESP;
          UICC_start_timer(UICC_T_IWT);
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);


#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

	  
          break;

        case UICC_TIMER_EXPIRED:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          UICC_pps_ctrl_state = UICC_PPS_READY;
          UICC_card_ctrl(UICC_PPS_NOT_COMPLETE);        
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;

    /*=========================*/
    /* Receiving PPS0 response */
    /*=========================*/
    case UICC_PPS0_RESP:
      switch(event)
      {
        case UICC_CHARACTER_OK:
          /* The character interrupt originates from PPS0 response. Now make
             ready for PPS1, PPS2, PPS3 or PCK response. */
          pck_sum ^= pps0_response = UICC_get_character();

          if (PPS123_MASK(pps0_response) == 0x00)
            UICC_pps_ctrl_state = UICC_PCK_RESP;
          else if(PPS1_MASK(pps0_response))
            UICC_pps_ctrl_state = UICC_PPS1_RESP;
          else if(PPS2_MASK(pps0_response))
            UICC_pps_ctrl_state = UICC_PPS2_RESP;
          else if(PPS3_MASK(pps0_response))
            UICC_pps_ctrl_state = UICC_PPS3_RESP;

          UICC_start_timer(UICC_T_IWT);
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);


#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

	  
          break;

        case UICC_TIMER_EXPIRED:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          UICC_pps_ctrl_state = UICC_PPS_READY;
          UICC_card_ctrl(UICC_PPS_NOT_COMPLETE);        
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;

    /*=========================*/
    /* Receiving PPS1 response */
    /*=========================*/
    case UICC_PPS1_RESP:
      switch(event)
      {
        case UICC_CHARACTER_OK:
          /* The character interrupt originates from PPS1 response. Now make
             ready for PPS2, PPS3, PCK response. */
          pck_sum ^= pps1_response = UICC_get_character();

          if (PPS23_MASK(pps0_response) == 0x00)
            UICC_pps_ctrl_state = UICC_PCK_RESP;
          else if(PPS2_MASK(pps0_response))
            UICC_pps_ctrl_state = UICC_PPS2_RESP;
          else if(PPS3_MASK(pps0_response))
            UICC_pps_ctrl_state = UICC_PPS3_RESP;

          UICC_start_timer(UICC_T_IWT);
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);


#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

	  
          break;

        case UICC_TIMER_EXPIRED:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          UICC_pps_ctrl_state = UICC_PPS_READY;
          UICC_card_ctrl(UICC_PPS_NOT_COMPLETE);        
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;

    /*=========================*/
    /* Receiving PPS2 response */
    /*=========================*/
    case UICC_PPS2_RESP:
      switch(event)
      {
        case UICC_CHARACTER_OK:
          /* The character interrupt originates from PPS2 response. Now make
             ready for PPS3 or PCK response. */             
          
          pck_sum ^= UICC_get_character();

          if(PPS3_MASK(pps0_response))
            UICC_pps_ctrl_state = UICC_PPS3_RESP;
          else
            UICC_pps_ctrl_state = UICC_PCK_RESP;

          UICC_start_timer(UICC_T_IWT);
        break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);


#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

  
          break;

        case UICC_TIMER_EXPIRED:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          UICC_pps_ctrl_state = UICC_PPS_READY;
          UICC_card_ctrl(UICC_PPS_NOT_COMPLETE);        
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;

    /*=========================*/
    /* Receiving PPS3 response */
    /*=========================*/
    case UICC_PPS3_RESP:
      switch(event)
      {
        case UICC_CHARACTER_OK:
          /* The character interrupt originates from PPS2 response. Now make
             ready for PCK response. */
          
          pck_sum ^= UICC_get_character();

          UICC_pps_ctrl_state = UICC_PCK_RESP;
          UICC_start_timer(UICC_T_IWT);
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);


#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

	  
          break;

        case UICC_TIMER_EXPIRED:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          UICC_pps_ctrl_state = UICC_PPS_READY;
          UICC_card_ctrl(UICC_PPS_NOT_COMPLETE);        
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;

    /*========================*/
    /* Receiving PCK response */
    /*========================*/
    case UICC_PCK_RESP:
      switch(event)
      {
        case UICC_CHARACTER_OK:
          /* The character interrupt originates from PPS2 response. Now get
             ready to evaluate the received PPS response. This should be done
             on the lower priority interrupt timeot. */
          pck_response = UICC_get_character();
          UICC_pps_ctrl_state = UICC_PPS_COMPLETED;
          UICC_start_timer(T_SPEED_CHANGE_DELAY);


#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

	  
          break;

        case UICC_PARITY_ERROR:
          /* A parity error occured - just restart the Initial Waiting timer. */
          UICC_start_timer(UICC_T_IWT);


#ifdef UICC_NOT_TIMER
		  UICC_control_card_access(UICC_TIMER_EXPIRED);
#endif

 
          break;

        case UICC_TIMER_EXPIRED:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          UICC_pps_ctrl_state = UICC_PPS_READY;
          UICC_card_ctrl(UICC_PPS_NOT_COMPLETE);        
          break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;

    /*==============*/
    /* PPS succeded */
    /*==============*/
    case UICC_PPS_COMPLETED:
      switch(event)
      {
        case UICC_TIMER_EXPIRED:
        {
          ubyte status = FALSE;

          UICC_pps_ctrl_state = UICC_PPS_READY;

          /* The following check will identify if the received PPS response can
             satisfy the success citeria.*/
             
          /* First check:
             - was the received checksum correct.
             - was the received PPSS = 0xFF 
             - was PPS2 and PPS3 absent in the response (must not be present when not sent by 
               the terminal). 
             - was the received protocol identical to the suggested protocol (the protocol sent). */
          if((pck_response == pck_sum) && (ppss_response == PPSS) && !PPS23_MASK(pps0_response) &&
             (PROTOCOL_MASK(pps0_request) == PROTOCOL_MASK(pps0_response)))
          {
            if(PPS1_MASK(pps0_response)) /* Was PPS1 send by the card? */
            { /* Did the card send PPS1? And if so is it identical with the one sent
                 by the ME? */
              if((PPS1_MASK(pps0_request)) && (pps1_request == pps1_response))
              { /* PPS successfull -> f and d should be used as indicated
                   in PPS1. */
                status = TRUE;
              }
              else
              { /* Is the default values indicated in the response? */
                if((pps1_response == 0x11) || (pps1_response == 0x01))
                { /* PPS successfull -> use default f and d values. */
                  f = UICC_F_DEFAULT;
                  d = UICC_D_DEFAULT;
                  status = TRUE;
                }
              }
            }
            else
            { /* PPS successfull -> use default f and d values. */
              f = UICC_F_DEFAULT;
              d = UICC_D_DEFAULT;
              status = TRUE;
            }
          }

          if(status)
          {/* PPS succeded! */
            /*=============*/

            /* Request the UART to operate at the the negotiated speed. */
            /* BRF = F/4*D */
            UICC_card_profile[UICC_current_reader].f_used = f;
            UICC_card_profile[UICC_current_reader].d_used = d;

            UICC_transmission_rate_control(UICC_clock_rate_table[f] / (4 * UICC_baud_rate_table[d]));

            /* Signal the main control statemachine of the PPS result*/
            UICC_card_ctrl(UICC_PPS_OK);
          }
          else
          { /* PPS failed! */
            /*============*/
            /* Signal the main control statemachine of the PPS result*/
            #ifndef UICC_NO_DEBUG
              UICC_log_data_ptr[0]  = debug;
              UICC_log_data_ptr[1]  = event;
              UICC_log_data_ptr[2]  = pck_sum;
              UICC_log_data_ptr[3]  = pps0_request;
              UICC_log_data_ptr[4]  = pps1_request;
              UICC_log_data_ptr[5]  = ppss_response;
              UICC_log_data_ptr[6]  = pps0_response;
              UICC_log_data_ptr[7]  = pps1_response;
              UICC_log_data_ptr[8]  = pck_response;
              UICC_log_data_ptr[9]  = f;
              UICC_log_data_ptr[10] = d;
              UICC_raise_exception(TRAP_UICC_PPS, UICC_PROTOCOL_SELECTION_FAILED_2, 11); 
            #endif

            UICC_card_ctrl(UICC_PPS_FAILED);
          }
        }
        break;

        default:
          #ifndef UICC_NO_DEBUG
            debug = UICC_pps_ctrl_state;
          #endif
          break;
      }
      break;

    default:
      #ifndef UICC_NO_DEBUG
        debug = UICC_pps_ctrl_state;
      #endif
      break;
  }

  #ifndef UICC_NO_DEBUG
    if(debug != UICC_PPS_NONE)
    {
      UICC_log_data_ptr[0]  = debug;
      UICC_log_data_ptr[1]  = event;
      UICC_log_data_ptr[2]  = pck_sum;
      UICC_log_data_ptr[3]  = pps0_request;
      UICC_log_data_ptr[4]  = pps1_request;
      UICC_log_data_ptr[5]  = ppss_response;
      UICC_log_data_ptr[6]  = pps0_response;
      UICC_log_data_ptr[7]  = pps1_response;
      UICC_log_data_ptr[8]  = pck_response;
      UICC_log_data_ptr[9]  = f;
      UICC_log_data_ptr[10] = d;
      UICC_raise_exception(TRAP_UICC_PPS, UICC_PROTOCOL_SELECTION_FAILED_1, 11); 

      debug = UICC_PPS_NONE;
    }
  #endif
}
                                                 /* End of file.              */
