/*******************************************************************************
* Copyright (C) 2013, ZTE Corporation.
*
* File Name:    io_dev.c
* File Mark:
* Description:
* Others:
* Version:		 V1.0
* Author:		 geanfeng
* Date: 		 2013-10-21
********************************************************************************/

/****************************************************************************
* 	                                           Include files
****************************************************************************/
#include "drvs_general.h"
#include "drvs_bitops.h"

/****************************************************************************
* 	                              Local Macros
****************************************************************************/
#define DEVICE_NAME_LENGTH  (32)

#define DEVICE_MAGIC_FLAG 	(0x12340000)
#define DEVICE_MAGIC(x) (DEVICE_MAGIC_FLAG | ((UINT32)x & 0x0000FFFF))

#define BIT_DISCONNECT  (0x0)
#define BIT_CLOSED  (0x1)

#ifdef _OS_WIN
#define LOCK_SAVE(x) x=0
#define LOCK_RESTORE(x)
#endif
/****************************************************************************
* 	                               Local Types
****************************************************************************/
typedef enum _T_DRVIO_CONNECT_STATE
{
    DRVIO_DEVICE_DISCONNECT=0,
    DRVIO_DEVICE_PRE_DISCONNECT,
    DRVIO_DEVICE_CONNECT,
} T_DRVIO_CONNECT_STATE;

typedef enum _T_DRVIO_DEV_STATE
{
    DRVIO_DEVICE_CLOSED=0,
    DRVIO_DEVICE_OPENED,
    DRVIO_DEVICE_CLOSING,
} T_DRVIO_DEV_STATE;

typedef struct _T_ZDrvIODev_Entry
{
    struct list_head node;	/* node*/
    char name[DEVICE_NAME_LENGTH];
	UINT32  magic;
    T_ZDRVIO_FLAGS flags;

    volatile T_DRVIO_CONNECT_STATE devConnectState;
    volatile T_DRVIO_DEV_STATE userOpenState;
    volatile T_DRVIO_DEV_STATE devOpenState;
	volatile UINT32 refCount;

	volatile UINT32 bitCtrl;
    volatile UINT32 readDepth;
    volatile UINT32 writeDepth;
    volatile UINT32 ctrlDepth;
    volatile SINT32 disallowDisconnect;
	
    ZOSS_SEMAPHORE_ID  waitConnectSem;
    ZOSS_MUTEX_ID  waitConnectMutex;
	
    ZOSS_MUTEX_ID  devProtectMutex;
	
    ZOSS_SEMAPHORE_ID  waitDisconnectSem;
    ZOSS_SEMAPHORE_ID  waitCloseSem;

    T_ZDrvIO_DevNotify notify;
	
    T_ZDrvIODev_Ops *ops;
    VOID *devData;
}
T_ZDrvIODev_Entry;


typedef struct _T_zDrvIO_GNotify
{
    struct list_head node;	/* node*/
    T_ZDrvIO_GNotifyCallBack notifyFn;
}
T_zDrvIO_GNotify;

typedef struct _T_zDrvIO_TAB
{
    struct list_head devList;
    ZOSS_SEMAPHORE_ID  opMutex;
    UINT32 devListCount;

    struct list_head notifyChain;
    UINT32 notifyCount;
}
T_zDrvIO_TAB;


T_zDrvIO_TAB  g_drvIOTable = {0,};

#define DEVICE_LIST_LOCK()  zOss_GetMutex(g_drvIOTable.opMutex,  ZOSS_WAIT_FOREVER)
#define DEVICE_LIST_UNLOCK()  zOss_PutMutex(g_drvIOTable.opMutex)

volatile T_ZDrvIODev_Entry *g_ioDevCurrentDisconnectIoEntry = NULL;

/****************************************************************************
* 	                                          Global Variables
****************************************************************************/

/****************************************************************************
* 	                                          macro Definitions
****************************************************************************/

/****************************************************************************
* 	                                          Function Definitions
****************************************************************************/
/*******************************************************************************
 * Function: _IO_GetDevStatus
 * Description:.
 * Input:
    name:
    flags:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static T_DRVIO_DEV_STATUS _IO_GetDevStatus(T_ZDrvIODev_Entry *pIOEntry)
{
    T_DRVIO_DEV_STATUS  status = {0,};

    if (pIOEntry->devConnectState == DRVIO_DEVICE_CONNECT)
    {
        status.CONNECT = 1;
    }
    return status;
}

/*******************************************************************************
 * Function: _IO_WaitDevConnect
 * Description:.
 * Input:
    name:
    flags:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static SINT32 _IO_WaitDevConnect(T_ZDrvIODev_Entry *pIOEntry)
{

    zOss_GetMutex(pIOEntry->waitConnectMutex, ZOSS_WAIT_FOREVER);

    while (pIOEntry->devConnectState != DRVIO_DEVICE_CONNECT)
    {
        zOss_GetSemaphore(pIOEntry->waitConnectSem, ZOSS_WAIT_FOREVER);
    }

    zOss_PutMutex(pIOEntry->waitConnectMutex);

    return DRV_SUCCESS;
}
/*******************************************************************************
 * Function: _IO_WakeupProcess
 * Description:.
 * Input:
    name:
    flags:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static VOID _IO_WakeupProcess(T_ZDrvIODev_Entry *pIOEntry)
{
	UINT32 bitCtrl = pIOEntry->bitCtrl;
	
	if (bitCtrl & (0x1 << BIT_DISCONNECT))
    	zOss_PutSemaphore(pIOEntry->waitDisconnectSem);
	
	if (bitCtrl & (0x1 << BIT_CLOSED))
    	zOss_PutSemaphore(pIOEntry->waitCloseSem);
}
/*******************************************************************************
 * Function: _IO_AllocDev
 * Description:.
 * Input:
    name:
    flags:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static T_ZDrvIODev_Entry* _IO_AllocDev(const char *name)
{
    T_ZDrvIODev_Entry  *pIOEntry = NULL;
    UINT32 len = 0;

    pIOEntry = (T_ZDrvIODev_Entry*)zOss_Malloc(sizeof(T_ZDrvIODev_Entry));
    if (!pIOEntry)
    {
        zDrv_ASSERT(0);
        return NULL;
    }

    zOss_Memset((void*)pIOEntry, 0x0, sizeof(T_ZDrvIODev_Entry));

    len = strlen(name);
    if ((len + 1) <= DEVICE_NAME_LENGTH)
    {
        zOss_Memcpy(pIOEntry->name, name, len+1);
    }
    else
    {
        zOss_Memcpy(pIOEntry->name, name, DEVICE_NAME_LENGTH);
        pIOEntry->name[DEVICE_NAME_LENGTH-1]='\0';
    }

    pIOEntry->waitConnectSem = zOss_CreateSemaphore("Connect Wait Sema", 0);
    pIOEntry->waitConnectMutex= zOss_CreateMutex("Wait Mutex", ZOSS_INHERIT);
    pIOEntry->devProtectMutex= zOss_CreateMutex("Protect Mutex", ZOSS_INHERIT);

    pIOEntry->waitDisconnectSem = zOss_CreateSemaphore("Disconnect Sema", 0);
    pIOEntry->waitCloseSem= zOss_CreateSemaphore("Clos Sema", 0);

    pIOEntry->devConnectState = DRVIO_DEVICE_DISCONNECT;
    pIOEntry->userOpenState = DRVIO_DEVICE_CLOSED;
    pIOEntry->devOpenState = DRVIO_DEVICE_CLOSED;
	set_bit(BIT_DISCONNECT, &pIOEntry->bitCtrl);
	set_bit(BIT_CLOSED, &pIOEntry->bitCtrl);
	pIOEntry->refCount = 0;
	pIOEntry->magic = DEVICE_MAGIC(pIOEntry);

    return pIOEntry;
}

/*******************************************************************************
 * Function: _IO_FreeDev
 * Description:.
 * Input:
    name:
    flags:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static  VOID _IO_FreeDev(T_ZDrvIODev_Entry* pIOEntry)
{

    if (pIOEntry->waitConnectSem)
        zOss_DeleteSemaphore(pIOEntry->waitConnectSem);
    if (pIOEntry->waitConnectMutex)
        zOss_DeleteMutex(pIOEntry->waitConnectMutex);
    if (pIOEntry->devProtectMutex)
        zOss_DeleteMutex(pIOEntry->devProtectMutex);
    if (pIOEntry->waitDisconnectSem)
        zOss_DeleteSemaphore(pIOEntry->waitDisconnectSem);
    if (pIOEntry->waitCloseSem)
        zOss_DeleteSemaphore(pIOEntry->waitCloseSem);

	pIOEntry->magic = 0x0;

    zOss_Free(pIOEntry);

    return ;
}

/*******************************************************************************
 * Function: _IO_DelDev
 * Description:.
 * Input:
    name:
    flags:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static  VOID _IO_DelDev(T_ZDrvIODev_Entry* pIOEntry)
{
    DEVICE_LIST_LOCK();

    if (pIOEntry->userOpenState == DRVIO_DEVICE_CLOSED && pIOEntry->devConnectState == DRVIO_DEVICE_DISCONNECT)
    {
        zDrv_ASSERT(pIOEntry->devOpenState == DRVIO_DEVICE_CLOSED);
        list_del(&pIOEntry->node);
        g_drvIOTable.devListCount--;
        _IO_FreeDev(pIOEntry);
    }

    DEVICE_LIST_UNLOCK();
}
/*******************************************************************************
 * Function: _IO_SearchDev
 * Description:.
 * Input:
    name:
    flags:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static T_ZDrvIODev_Entry* _IO_SearchDev(const char *name)
{
    struct list_head *_entry, *_next_entry;
    T_ZDrvIODev_Entry *pIOEntry = ZDRVIO_INVALID_HANDLE;
    T_ZDrvIODev_Entry *pEntry = NULL;

    list_for_each_safe(_entry, _next_entry, &g_drvIOTable.devList)
    {
        pEntry = list_entry(_entry, T_ZDrvIODev_Entry, node);
        if (!strcmp(name, pEntry->name))
        {
            pIOEntry = pEntry;
            break;
        }
    }

    return pIOEntry;
}
/*******************************************************************************
 * Function: _IO_GetDev
 * Description:.
 * Input:
    name:
    flags:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static VOID _IO_GetDev(T_ZDrvIODev_Entry* pIOEntry)
{
    zOss_GetMutex(pIOEntry->devProtectMutex, ZOSS_WAIT_FOREVER);
}
/*******************************************************************************
 * Function: _IO_PutDev
 * Description:.
 * Input:
    name:
    flags:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static VOID _IO_PutDev(T_ZDrvIODev_Entry* pIOEntry)
{
    zOss_PutMutex(pIOEntry->devProtectMutex);
}
/*******************************************************************************
 * Function: _IO_Ret
 * Description:.
 * Input:
    name:
    flags:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static SINT32 _IO_Ret(UINT32 ctrlVal)
{
	if (ctrlVal & (0x1 << BIT_DISCONNECT))
	{
		return DRV_ERROR_NOCONNECT;
	}
	else if (ctrlVal & (0x1 << BIT_CLOSED))
	{
		return DRV_ERR_NOT_OPENED;
	}
	else
	{
		return DRV_ERROR;
	}
}

/*******************************************************************************
 * Function: _IO_Ctrl
 * Description:.
 * Input:
    name:
    flags:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static SINT32 _IO_Ctrl(T_ZDrvIODev_Entry* pIOEntry, T_DRVIO_CTRL_KEY  function, VOID *arg)
{
    SINT32 ret = DRV_SUCCESS;
    UINT32 old_intr;

    /*IO DAL control function*/
    switch (function)
    {
    case IOCTL_IODEV_REGISTER_NOTIFY:
        pIOEntry->notify =* (T_ZDrvIO_DevNotify*)arg;
        break;

    case IOCTL_IODEV_GET_DEV_STATUS:
        *(T_DRVIO_DEV_STATUS*)arg = _IO_GetDevStatus(pIOEntry);
        break;

    case IOCTL_IODEV_WAIT_CONNECT:
        ret = _IO_WaitDevConnect(pIOEntry);
        break;
		
	case IOCTL_IODEV_DISALLOW_DISCONNECT:
		LOCK_SAVE(old_intr);
		pIOEntry->disallowDisconnect++;
		LOCK_RESTORE(old_intr);
		break;
		
	case IOCTL_IODEV_ALLOW_DISCONNECT:
		LOCK_SAVE(old_intr);
		pIOEntry->disallowDisconnect--;
		LOCK_RESTORE(old_intr);
		
		zDrv_ASSERT(pIOEntry->disallowDisconnect>=0);
		
		if (pIOEntry->bitCtrl)
		{
			_IO_WakeupProcess(pIOEntry);
		}
		break;
		
    default:
        zDrv_ASSERT(0);
        break;
    }

    return ret;
}
/*******************************************************************************
 * Function: zDrvIO_Open
 * Description:search device by device name, open it, and set the action flags.
 * Input:
    name: device name.
    flags: open flags, control the device action.
 * Output:None
 *
 * Returns:
              T_ZDrvIO_Handle: the opened device handle. if error happened, return NULL.
 * Others:
 ********************************************************************************/
T_ZDrvIO_Handle zDrvIO_Open(const char *name, T_ZDRVIO_FLAGS flags)
{
    T_ZDrvIODev_Entry *pIOEntry = ZDRVIO_INVALID_HANDLE;

    if (name == NULL)
    {
        zDrv_ASSERT(0);
        return ZDRVIO_INVALID_HANDLE;
    }

    DEVICE_LIST_LOCK();

    pIOEntry = _IO_SearchDev(name);
    if (pIOEntry == ZDRVIO_INVALID_HANDLE)
    {
        pIOEntry = _IO_AllocDev(name);

        list_add(&pIOEntry->node, &g_drvIOTable.devList);
        g_drvIOTable.devListCount++;
    }

    _IO_GetDev(pIOEntry);

	if (pIOEntry->userOpenState == DRVIO_DEVICE_OPENED)
	{
		zDrv_ASSERT(pIOEntry->flags.SHARED);
		zDrv_ASSERT(pIOEntry->flags.READ == flags.READ);
		zDrv_ASSERT(pIOEntry->flags.WRITE == flags.WRITE);
		zDrv_ASSERT(pIOEntry->flags.MULTI_READ == flags.MULTI_READ);
		zDrv_ASSERT(pIOEntry->flags.MULTI_WRITE == flags.MULTI_WRITE);		
	}
	
    pIOEntry->flags = flags;
    pIOEntry->userOpenState = DRVIO_DEVICE_OPENED;
	clear_bit(BIT_CLOSED,&pIOEntry->bitCtrl);
	
	pIOEntry->refCount++;

    DEVICE_LIST_UNLOCK();


    if (pIOEntry->devConnectState == DRVIO_DEVICE_CONNECT && pIOEntry->devOpenState == DRVIO_DEVICE_CLOSED)
    {
        if (pIOEntry->ops && pIOEntry->ops->open)
        {
            pIOEntry->ops->open(pIOEntry->devData, flags);
        }
		pIOEntry->devOpenState = DRVIO_DEVICE_OPENED;
    }
	
    _IO_PutDev(pIOEntry);

    return (T_ZDrvIO_Handle)pIOEntry;
}

/*******************************************************************************
 * Function: zDrvIO_Read
 * Description:read data from device.
 * Input:
 	handle:the device handle for read.
 	buffer: the read data buffer. the buffer space supplied by high layer.
 	length:the data length need to read.
 * Output:None
 *
 * Returns:
              ret >= 0: return the actual read length.
              DRV_ERROR_NOCONNECT: the device disconnect event happend, need return error to inform higher user.
              other: other error maked by device.
 * Others:
 ********************************************************************************/
SINT32 zDrvIO_Read(T_ZDrvIO_Handle handle, UINT8* buffer, UINT32 length)
{
    SINT32 ret = 0;
    T_ZDrvIODev_Entry *pIOEntry = (T_ZDrvIODev_Entry *)handle;
    UINT32 old_intr;
	UINT32 bitCtrl;
	
    if (pIOEntry == NULL || pIOEntry->magic != DEVICE_MAGIC(pIOEntry))
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    LOCK_SAVE(old_intr);
    if (pIOEntry->bitCtrl)
    {
    	bitCtrl = pIOEntry->bitCtrl;
    	LOCK_RESTORE(old_intr);
    	return _IO_Ret(bitCtrl);
    }
    pIOEntry->readDepth++;
    LOCK_RESTORE(old_intr);

    /*start read operations*/
    if (pIOEntry->ops && pIOEntry->ops->read)
    {
        ret = pIOEntry->ops->read(pIOEntry->devData, buffer, length);
    }

    LOCK_SAVE(old_intr);
    pIOEntry->readDepth--;
    LOCK_RESTORE(old_intr);

    if (pIOEntry->bitCtrl)
	{
		_IO_WakeupProcess(pIOEntry);
	}
    return ret;
}

/*******************************************************************************
 * Function: zDrvIO_Write
 * Description:write data to IO handle
 * Input:
  	handle:the device handle for write.
 	buffer: the write data buffer. the buffer space supplied by high layer.
 	length:the write data length.
* Output:None
 *
 * Returns:
              ret >= 0: return the actual write length.
              DRV_ERROR_NOCONNECT: the device disconnect event happend, need return error to inform higher user.
              other: other error maked by device.
 * Others:
 ********************************************************************************/
SINT32 zDrvIO_Write(T_ZDrvIO_Handle handle, UINT8* buffer, UINT32 length)
{
    SINT32 ret = 0;
    T_ZDrvIODev_Entry *pIOEntry = (T_ZDrvIODev_Entry *)handle;
    UINT32 old_intr;
	UINT32 bitCtrl;

    if (pIOEntry == NULL || pIOEntry->magic != DEVICE_MAGIC(pIOEntry))
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    LOCK_SAVE(old_intr);
    if (pIOEntry->bitCtrl)
    {
    	bitCtrl = pIOEntry->bitCtrl;
    	LOCK_RESTORE(old_intr);
    	return _IO_Ret(bitCtrl);
    }
    pIOEntry->writeDepth++;
    LOCK_RESTORE(old_intr);

    /*start write operations*/
    if (pIOEntry->ops && pIOEntry->ops->write)
    {
        ret = pIOEntry->ops->write(pIOEntry->devData, buffer, length);
    }

    LOCK_SAVE(old_intr);
    pIOEntry->writeDepth--;
    LOCK_RESTORE(old_intr);

    if (pIOEntry->bitCtrl)
	{
		_IO_WakeupProcess(pIOEntry);
	}
    return ret;
}

/*******************************************************************************
 * Function: zDrvIO_BRead
 * Description:read data from device.
 * Input:
 	handle:the device handle for read.
 	buffer: the read data buffer. the buffer space supplied by high layer.
 	blkCnt:the data block count need to read.
 	pos:the read data position in device storage space.
 * Output:None
 *
 * Returns:
              ret >= 0: return the actual read length.
              DRV_ERROR_NOCONNECT: the device disconnect event happend, need return error to inform higher user.
              other: other error maked by device.
 * Others:
 ********************************************************************************/
SINT32 zDrvIO_BRead(T_ZDrvIO_Handle handle, UINT8* buffer, UINT32 blkCnt, UINT32 pos)
{
    SINT32 ret = 0;
    T_ZDrvIODev_Entry *pIOEntry = (T_ZDrvIODev_Entry *)handle;
    UINT32 old_intr;
	UINT32 bitCtrl;

    if (pIOEntry == NULL || pIOEntry->magic != DEVICE_MAGIC(pIOEntry))
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    LOCK_SAVE(old_intr);
    if (pIOEntry->bitCtrl)
    {
    	bitCtrl = pIOEntry->bitCtrl;
    	LOCK_RESTORE(old_intr);
    	return _IO_Ret(bitCtrl);
    }
    pIOEntry->readDepth++;
    LOCK_RESTORE(old_intr);

    /*start read operations*/
    if (pIOEntry->ops && pIOEntry->ops->bread)
    {
        ret = pIOEntry->ops->bread(pIOEntry->devData, buffer, blkCnt, pos);
    }

    LOCK_SAVE(old_intr);
    pIOEntry->readDepth--;
    LOCK_RESTORE(old_intr);

    if (pIOEntry->bitCtrl)
	{
		_IO_WakeupProcess(pIOEntry);
	}
    return ret;
}

/*******************************************************************************
 * Function: zDrvIO_BWrite
 * Description:write data to IO handle
 * Input:
  	handle:the device handle for write.
 	buffer: the write data buffer. the buffer space supplied by high layer.
 	blkCnt:the write data block count.
 	pos:the write data position in device storage space.
* Output:None
 *
 * Returns:
              ret >= 0: return the actual write length.
              DRV_ERROR_NOCONNECT: the device disconnect event happend, need return error to inform higher user.
              other: other error maked by device.
 * Others:
 ********************************************************************************/
SINT32 zDrvIO_BWrite(T_ZDrvIO_Handle handle, UINT8* buffer, UINT32 blkCnt, UINT32 pos)
{
    SINT32 ret = 0;
    T_ZDrvIODev_Entry *pIOEntry = (T_ZDrvIODev_Entry *)handle;
    UINT32 old_intr;
	UINT32 bitCtrl;

    if (pIOEntry == NULL || pIOEntry->magic != DEVICE_MAGIC(pIOEntry))
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    LOCK_SAVE(old_intr);
    if (pIOEntry->bitCtrl)
    {
    	bitCtrl = pIOEntry->bitCtrl;
    	LOCK_RESTORE(old_intr);
    	return _IO_Ret(bitCtrl);
    }
    pIOEntry->writeDepth++;
    LOCK_RESTORE(old_intr);

    /*start write operations*/
    if (pIOEntry->ops && pIOEntry->ops->bwrite)
    {
        ret = pIOEntry->ops->bwrite(pIOEntry->devData, buffer, blkCnt, pos);
    }

    LOCK_SAVE(old_intr);
    pIOEntry->writeDepth--;
    LOCK_RESTORE(old_intr);

    if (pIOEntry->bitCtrl)
	{
		_IO_WakeupProcess(pIOEntry);
	}
    return ret;
}

/*******************************************************************************
 * Function: zDrvIO_Ctrl
 * Description:control device.
 * Input:
  	handle:the device handle for control.
   	function:the control function.
  	handle:the control function arguments.
* Output:None
 *
 * Returns:
              DRV_SUCCESS: success.
              DRV_ERROR: error.
 * Others:
 ********************************************************************************/
SINT32 zDrvIO_Ctrl(T_ZDrvIO_Handle handle, T_DRVIO_CTRL_KEY  function, VOID *arg)
{
    SINT32 ret = DRV_ERROR;
    T_ZDrvIODev_Entry *pIOEntry = (T_ZDrvIODev_Entry *)handle;
    UINT32 old_intr;
	UINT32 bitCtrl;

    if (pIOEntry == NULL || pIOEntry->magic != DEVICE_MAGIC(pIOEntry))
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    if (function < IOCTL_IODEV_VENDOR)
    {
        return _IO_Ctrl(pIOEntry, function, arg);
    }

    LOCK_SAVE(old_intr);
    if (pIOEntry->bitCtrl)
    {
    	bitCtrl = pIOEntry->bitCtrl;
    	LOCK_RESTORE(old_intr);
    	return _IO_Ret(bitCtrl);
    }
    pIOEntry->ctrlDepth++;
    LOCK_RESTORE(old_intr);

    /*start device ctrl operations*/
    if (pIOEntry->ops && pIOEntry->ops->ctrl)
    {
        ret = pIOEntry->ops->ctrl(pIOEntry->devData, function, arg);
    }

    LOCK_SAVE(old_intr);
    pIOEntry->ctrlDepth--;
    LOCK_RESTORE(old_intr);

    if (pIOEntry->bitCtrl)
	{
		_IO_WakeupProcess(pIOEntry);
	}
    return ret;
}

/*******************************************************************************
 * Function: zDrvIO_Close
 * Description:close the device
 * Input:
 	handle:the device handle to be closed.
 * Output:None
 *
 * Returns:
              DRV_SUCCESS: success.
              DRV_ERROR: error.
 * Others:
 ********************************************************************************/
SINT32 zDrvIO_Close(T_ZDrvIO_Handle handle)
{
    T_ZDrvIODev_Entry *pIOEntry = (T_ZDrvIODev_Entry *)handle;
	SINT32 ret = DRV_SUCCESS;
	
    if (pIOEntry == NULL || pIOEntry->magic != DEVICE_MAGIC(pIOEntry))
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    zDrv_ASSERT(pIOEntry->userOpenState == DRVIO_DEVICE_OPENED);

    _IO_GetDev(pIOEntry);

	pIOEntry->refCount--;

	if (pIOEntry->refCount == 0)
	{
		pIOEntry->userOpenState = DRVIO_DEVICE_CLOSING;
		set_bit(BIT_CLOSED,&pIOEntry->bitCtrl);
		
		if (pIOEntry->devConnectState == DRVIO_DEVICE_CONNECT)
		{
			zDrv_ASSERT(pIOEntry->disallowDisconnect == 0);
			while (pIOEntry->readDepth || pIOEntry->writeDepth || pIOEntry->ctrlDepth)
			{
				zDrvDebug_Printf("DRVIO(%d): closed %s, wait finish...", zOss_GetTickCount(), pIOEntry->name);
				zOss_GetSemaphore(pIOEntry->waitCloseSem, ZOSS_WAIT_FOREVER); //wait device operation exist
			}
			
			if (pIOEntry->devOpenState == DRVIO_DEVICE_OPENED)
			{
			    if (pIOEntry->ops && pIOEntry->ops->close)
			    {
			        ret = pIOEntry->ops->close(pIOEntry->devData);
			    }
				pIOEntry->devOpenState = DRVIO_DEVICE_CLOSED;
			}
		}
		
		pIOEntry->notify.notify_fn = NULL;
		pIOEntry->userOpenState = DRVIO_DEVICE_CLOSED;
	}
    _IO_PutDev(pIOEntry);

    _IO_DelDev(pIOEntry);
    return ret;
}
/*******************************************************************************
 * Function: zDrvIO_RegisterGlobalNotify
 * Description: register global notifier.
 * Input:
 *        globalNotifyFn: global noitifier callback function
 * Output:None
 *
 * Returns:
 *       T_ZDrvIO_GNotifyHandle : the global notifier handler.
 * Others:
 ********************************************************************************/
T_ZDrvIO_GNotifyHandle zDrvIO_RegisterGlobalNotify(T_ZDrvIO_GNotifyCallBack globalNotifyFn)
{
    struct list_head *_entry, *_next_entry;
    T_ZDrvIODev_Entry *pEntry = NULL;
    T_zDrvIO_GNotify *pNotify = NULL;

    pNotify = (T_zDrvIO_GNotify *)zOss_Malloc(sizeof(T_zDrvIO_GNotify));
    if (!pNotify)
        return NULL;

    zOss_Memset((void*)pNotify, 0x0, sizeof(T_zDrvIO_GNotify));

    pNotify->notifyFn= globalNotifyFn;

    DEVICE_LIST_LOCK();

    list_add(&pNotify->node, &g_drvIOTable.notifyChain);
    g_drvIOTable.notifyCount++;

    DEVICE_LIST_UNLOCK();

    list_for_each_safe(_entry, _next_entry, &g_drvIOTable.devList)
    {
        pEntry = list_entry(_entry, T_ZDrvIODev_Entry, node);
        if (pNotify->notifyFn)
        {
        	if(pEntry->devConnectState ==  DRVIO_DEVICE_CONNECT)
            	pNotify->notifyFn(pEntry->name, DRVIO_EV_CONNECT);
			else
            	pNotify->notifyFn(pEntry->name, DRVIO_EV_DISCONNECT);
        }
    }


    return (T_ZDrvIO_GNotifyHandle)pNotify;
}
/*******************************************************************************
 * Function: zDrvIO_UnregisterGlobalNotify
 * Description: unregister global notifier.
 * Input:
 *        gNotifyHandle: the global notifier handler
 * Output:None
 *
 * Returns:
 *
 * Others:
 ********************************************************************************/
VOID zDrvIO_UnregisterGlobalNotify(T_ZDrvIO_GNotifyHandle gNotifyHandle)
{
    T_zDrvIO_GNotify *pNotify = (T_zDrvIO_GNotify*)gNotifyHandle;

    if (!pNotify)
        return ;

    DEVICE_LIST_LOCK();

    list_del(&pNotify->node);
    g_drvIOTable.notifyCount--;

    DEVICE_LIST_UNLOCK();

    zOss_Free(pNotify);
    return ;
}
/*******************************************************************************
 * Function: zDrvIODev_Connect
 * Description:install and connect io device to device table.
 * Input:
     name: the device name to connect.
     devData: the device private data to be combined.
     ops: the device I/O operation.
 * Output:None
 *
 * Returns:
              T_ZDrvIODev_Handle: none zero.
              NULL: error.
 * Others:
 ********************************************************************************/
T_ZDrvIODev_Handle zDrvIODev_Connect(const char *name, VOID *devData, T_ZDrvIODev_Ops *ops)
{
    struct list_head *_entry, *_next_entry;
    T_ZDrvIODev_Entry *pIOEntry = NULL;
    T_zDrvIO_GNotify *pNotify = NULL;

    if (name == NULL || g_drvIOTable.opMutex == NULL)
    {
        zDrv_ASSERT(0);
        return NULL;
    }

    DEVICE_LIST_LOCK();

    pIOEntry = _IO_SearchDev(name);
    if (pIOEntry == NULL)
    {
        pIOEntry = _IO_AllocDev(name);
        list_add(&pIOEntry->node, &g_drvIOTable.devList);
        g_drvIOTable.devListCount++;
    }
    zDrv_ASSERT(pIOEntry->devConnectState == DRVIO_DEVICE_DISCONNECT);

    _IO_GetDev(pIOEntry);

    pIOEntry->devData = devData;
    pIOEntry->ops = ops;
    pIOEntry->devConnectState = DRVIO_DEVICE_CONNECT;
	
    DEVICE_LIST_UNLOCK();


    if (pIOEntry->userOpenState == DRVIO_DEVICE_OPENED && pIOEntry->devOpenState == DRVIO_DEVICE_CLOSED)
    {
        if (pIOEntry->ops && pIOEntry->ops->open)
        {
            pIOEntry->ops->open(pIOEntry->devData, pIOEntry->flags);
        }
		pIOEntry->devOpenState = DRVIO_DEVICE_OPENED;
    }
	
	clear_bit(BIT_DISCONNECT, &pIOEntry->bitCtrl);

    _IO_PutDev(pIOEntry);

    /*notify current device user*/
    zOss_PutSemaphore(pIOEntry->waitConnectSem);

    /*notify device user*/
    if (pIOEntry->notify.notify_fn)
    {
        pIOEntry->notify.notify_fn(pIOEntry->notify.privData, DRVIO_EV_CONNECT);
    }

    /*notify all*/
    list_for_each_safe(_entry, _next_entry, &g_drvIOTable.notifyChain)
    {
        pNotify = list_entry(_entry, T_zDrvIO_GNotify, node);
        if (pNotify->notifyFn)
            pNotify->notifyFn(pIOEntry->name, DRVIO_EV_CONNECT);
    }

    return (T_ZDrvIODev_Handle)pIOEntry;
}

/*******************************************************************************
 * Function: zDrvIODev_Disconnect
 * Description:uninstall the io device.
 * Input:
      handle: the device handle to disconnect.
 * Output:None
 *
 * Returns:
              NULL: error.
 * Others:
 ********************************************************************************/
VOID zDrvIODev_Disconnect(T_ZDrvIODev_Handle handle)
{
    T_ZDrvIODev_Entry *pIOEntry = (T_ZDrvIODev_Entry *)handle;
    struct list_head *_entry, *_next_entry;
    T_zDrvIO_GNotify *pNotify = NULL;
	UINT32 param;

    if (!handle || pIOEntry->devConnectState != DRVIO_DEVICE_CONNECT)
    {
        //zDrv_ASSERT(0);
        return ;
    }
	
	g_ioDevCurrentDisconnectIoEntry = pIOEntry;

    _IO_GetDev(pIOEntry);
    pIOEntry->devConnectState = DRVIO_DEVICE_PRE_DISCONNECT;
	set_bit(BIT_DISCONNECT, &pIOEntry->bitCtrl);
    _IO_PutDev(pIOEntry);

    /*notify current device user*/
    if (pIOEntry->notify.notify_fn)
    {
        pIOEntry->notify.notify_fn(pIOEntry->notify.privData, DRVIO_EV_PRE_DISCONNECT);
    }
    /*notify all*/
    list_for_each_safe(_entry, _next_entry, &g_drvIOTable.notifyChain)
    {
        pNotify = list_entry(_entry, T_zDrvIO_GNotify, node);
        if (pNotify->notifyFn)
            pNotify->notifyFn(pIOEntry->name, DRVIO_EV_PRE_DISCONNECT);
    }

    _IO_GetDev(pIOEntry);
    if (pIOEntry->devOpenState == DRVIO_DEVICE_OPENED && pIOEntry->ops->ctrl)
    {
    	param = ZOSS_NO_WAIT;
        pIOEntry->ops->ctrl(pIOEntry->devData, IOCTL_IODEV_SET_BLOCKTIME, (VOID*)&param);
    }
    _IO_PutDev(pIOEntry);

    /*notify current device user*/
    if (pIOEntry->notify.notify_fn)
    {
        pIOEntry->notify.notify_fn(pIOEntry->notify.privData, DRVIO_EV_DISCONNECT);
    }

    /*notify all*/
    list_for_each_safe(_entry, _next_entry, &g_drvIOTable.notifyChain)
    {
        pNotify = list_entry(_entry, T_zDrvIO_GNotify, node);
        if (pNotify->notifyFn)
            pNotify->notifyFn(pIOEntry->name, DRVIO_EV_DISCONNECT);
    }

    while (pIOEntry->readDepth || pIOEntry->writeDepth || pIOEntry->ctrlDepth || pIOEntry->disallowDisconnect)
    {
        zDrvDebug_Printf("DRVIO(%d): disconnect %s, wait finish...", zOss_GetTickCount(), pIOEntry->name);
        zDrvDebug_Printf("readDepth=%d, writeDepth=%d, ctrlDepth=%d, disallowDisconnect=%d", pIOEntry->readDepth, pIOEntry->writeDepth, pIOEntry->ctrlDepth, pIOEntry->disallowDisconnect);
        zOss_GetSemaphore(pIOEntry->waitDisconnectSem, ZOSS_WAIT_FOREVER);
    }


    _IO_GetDev(pIOEntry);
    if (pIOEntry->devOpenState == DRVIO_DEVICE_OPENED)
    {
        if (pIOEntry->ops && pIOEntry->ops->close)
        {
            pIOEntry->ops->close(pIOEntry->devData);
        }
		pIOEntry->devOpenState = DRVIO_DEVICE_CLOSED;
    }
    pIOEntry->devConnectState = DRVIO_DEVICE_DISCONNECT;
    pIOEntry->devData = NULL;
    pIOEntry->ops = NULL;
    _IO_PutDev(pIOEntry);

    _IO_DelDev(pIOEntry);
    return ;
}

/*******************************************************************************
 * Function: printBuffer
 * Description:.
 * Input:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static VOID printBuffer(UINT8 *pBuf, UINT32 len)
{
	UINT32 printPos = 0;
	UINT8 tmpCh = 0;
	
	while(printPos < len)
	{
		tmpCh = pBuf[printPos+96];
		pBuf[printPos+96] = '\0';
		zDrvDebug_Printf("%s", &pBuf[printPos]);
		pBuf[printPos+96] = tmpCh;
		printPos += 96;
	}

}

/*******************************************************************************
 * Function: printDevList
 * Description:show the device debug information.
 * Input:
 * Output:None
 *
 * Returns:None
 * Others:
 ********************************************************************************/
static VOID printDevList(VOID)
{
    struct list_head *_entry, *_next_entry;
    T_ZDrvIODev_Entry *pIOEntry = NULL;

    zDrvDebug_Printf("----------DRVIO DEVICE LIST----------");

    DEVICE_LIST_LOCK();

    zDrvDebug_Printf("Device List Count: %d", g_drvIOTable.devListCount);

    list_for_each_safe(_entry, _next_entry, &g_drvIOTable.devList)
    {
        pIOEntry = list_entry(_entry, T_ZDrvIODev_Entry, node);

		zDrvDebug_Printf("[%s]", pIOEntry->name);
    }

    DEVICE_LIST_UNLOCK();

    return ;
}

/*******************************************************************************
 * Function: printDevDebugInfo
 * Description:show the device debug information.
 * Input:
 * Output:None
 *
 * Returns:None
 * Others:
 ********************************************************************************/
static VOID printDevDebugInfo(VOID)
{
    struct list_head *_entry, *_next_entry;
    T_ZDrvIODev_Entry *pIOEntry = NULL;

    zDrvDebug_Printf("----------DRVIO DEBUG INFO----------");

    DEVICE_LIST_LOCK();

    zDrvDebug_Printf("Device List Count: %d", g_drvIOTable.devListCount);

    list_for_each_safe(_entry, _next_entry, &g_drvIOTable.devList)
    {
        pIOEntry = list_entry(_entry, T_ZDrvIODev_Entry, node);

		zDrvDebug_Printf("[%s] INFO:", pIOEntry->name);
		
        if (pIOEntry->devConnectState == DRVIO_DEVICE_CONNECT)
        {
            zDrvDebug_Printf("Device is connect");
        }
        else
        {
            zDrvDebug_Printf("Device is disconnect");
        }

        if (pIOEntry->userOpenState > DRVIO_DEVICE_CLOSED)
        {
            zDrvDebug_Printf("Device is user opend");
        }
        else
        {
            zDrvDebug_Printf("Device is user closed");
        }
		
        if (pIOEntry->devOpenState > DRVIO_DEVICE_CLOSED)
        {
            zDrvDebug_Printf("Device is dev opend");
        }
        else
        {
            zDrvDebug_Printf("Device is dev closed");
        }
		
        zDrvDebug_Printf("Device Refer Count: %d", pIOEntry->refCount);
        zDrvDebug_Printf("Device Read Depth: %d", pIOEntry->readDepth);
        zDrvDebug_Printf("Device Write Depth: %d", pIOEntry->writeDepth);
        zDrvDebug_Printf("Device Control Depth: %d", pIOEntry->ctrlDepth);
        zDrvDebug_Printf("Device Disallow Disconnect: %d", pIOEntry->disallowDisconnect);
		zDrvDebug_Printf("\n");
    }

    DEVICE_LIST_UNLOCK();

    return ;
}

/*******************************************************************************
 * Function: lsDevCmdEntry
 * Description:add DRV IO debug cmd.
 * Input:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static VOID lsDevCmdEntry(T_Shell_CommandMessage *CmdMsg)
{
	UINT32 i = 0;

	if (CmdMsg->paraCount == 0)
	{
		zDrvDebug_Printf("lsdev support arg:");
		zDrvDebug_Printf("	-l: show the io devices list");
		zDrvDebug_Printf("	-d: show the io devices debug info");
		printDevList();
		return;
	}
	
	for (i=0; i<CmdMsg->paraCount; i++)
	{
		if (strcmp((const char*)CmdMsg->para[i], "-l") == 0)
		{
			printDevList();
		}
		else if (strcmp((const char*)CmdMsg->para[i], "-d") == 0)
		{
			printDevDebugInfo();
		}
		else
		{
			zDrvDebug_Printf("Unknown para!!!");
		}
	}
    return ;
}
/*******************************************************************************
 * Function: catCDevCmdEntry
 * Description:add DRV IO debug cmd.
 * Input:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/

static VOID catCDevCmdEntry(T_Shell_CommandMessage *CmdMsg)
{
	T_ZDrvIO_Handle pIOHnd = NULL;
    T_ZDRVIO_FLAGS flags = {0};
	UINT32 i = 0;
	UINT8 *pBuf = NULL;
	UINT8 *pChBuf = NULL;
	SINT32 ret = DRV_SUCCESS;
	SINT32 readLen = 0;
	UINT32 cnt = 0;
	UINT32 chPos = 0;
	
	if (CmdMsg->paraCount == 0) 
	{
		zDrvDebug_Printf("example: catdev uart/0 200");
		return;
	}

	if (CmdMsg->paraCount != 2)
	{
		zDrvDebug_Printf("catdev invalid param !!!");
		return ;
	}

	pBuf = (UINT8*)zOss_Malloc(2048);
	pChBuf = (UINT8*)zOss_Malloc(4096);
	if (!pBuf || !pChBuf)
	{
		if (pChBuf)
			zOss_Free(pChBuf);
		if (pBuf)
			zOss_Free(pBuf);
		zDrvDebug_Printf("malloc buffer fail !!!");
		return ;
	}

	flags.READ = 1;
	flags.WRITE = 1;
	
	pIOHnd = zDrvIO_Open((const char*)CmdMsg->para[0], flags);
	if (pIOHnd == NULL)
	{
		zDrvDebug_Printf("Open Device %s Error !!!", CmdMsg->para[0]);
		zOss_Free(pChBuf);
		zOss_Free(pBuf);
		return ;
	}
	
	sscanf(CmdMsg->para[1], "%d", (int*)&readLen);
	zDrvDebug_Printf("Start Read Device Length %d", readLen);

	while(cnt<readLen)
	{
		ret = zDrvIO_Read(pIOHnd, pBuf, 2048);
		if (ret <= 0)
		{
			zDrvDebug_Printf("Read Error Ret = %d !!!\n", ret);
			goto exit;
		}
		zDrvDebug_Printf("\nRead Length %d", ret);
		
		zDrvDebug_Printf("ASCII:");
		pChBuf[ret]='\0';
		for (i=0; i<ret; i++)
		{
			if (pBuf[i] >= 32 && pBuf[i] <= 126)
				pChBuf[i] = pBuf[i];
			else
				pChBuf[i] = 32;
		}
		printBuffer(pChBuf, ret);

		zDrvDebug_Printf("HEX:");
		chPos = 0;
		for (i=0; i<ret; i++)
		{
			sprintf((char*)&pChBuf[chPos],"%02x ", pBuf[i]);
			chPos += 3;				
		}
		printBuffer(pChBuf, chPos + 1);
		cnt += ret;
	}

exit:
	zOss_Free(pChBuf);
	zOss_Free(pBuf);
	zDrvIO_Close(pIOHnd);
	
    return ;
}
/*******************************************************************************
 * Function: catBDevCmdEntry
 * Description:add DRV IO debug cmd.
 * Input:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static VOID catBDevCmdEntry(T_Shell_CommandMessage *CmdMsg)
{
	T_ZDrvIO_Handle pIOHnd = NULL;
	T_ZDRVIO_FLAGS flags = {0};
	UINT32 i = 0;
	UINT8 *pBuf = NULL;
	UINT8 *pChBuf = NULL;
	SINT32 ret = DRV_SUCCESS;
	SINT32 readCnt = 0;
	SINT32 readBase = 0;
	UINT32 cnt = 0;
	T_BlkDev_Info blkDevInfo = {0};
	UINT32 chPos = 0;
	
	if (CmdMsg->paraCount == 0) 
	{
		zDrvDebug_Printf("example: catbdev sd 200 1");
		return;
	}

	if (CmdMsg->paraCount != 3)
	{
		zDrvDebug_Printf("catdev invalid param !!!");
		return ;
	}

	pBuf = (UINT8*)zOss_Malloc(2048);
	pChBuf = (UINT8*)zOss_Malloc(6144);
	if (!pBuf || !pChBuf)
	{
		if (pChBuf)
			zOss_Free(pChBuf);
		if (pBuf)
			zOss_Free(pBuf);
		zDrvDebug_Printf("malloc buffer fail !!!");
		return ;
	}

	flags.READ = 1;
	flags.WRITE = 1;
	
	pIOHnd = zDrvIO_Open((const char*)CmdMsg->para[0], flags);
	if (pIOHnd == NULL)
	{
		zDrvDebug_Printf("Open Device %s Error !!!", CmdMsg->para[0]);
		zOss_Free(pChBuf);
		zOss_Free(pBuf);
		return ;
	}
	
	zDrvIO_Ctrl(pIOHnd,IOCTL_BLKDEV_GET_INFO,&blkDevInfo);
	if (blkDevInfo.blkSize > 2048 || blkDevInfo.blkSize == 0)
	{
		zDrvDebug_Printf("Error Block Size %d !!!", blkDevInfo.blkSize);
		goto exit;
	}

	sscanf(CmdMsg->para[1], "%d", (int*)&readBase);
	sscanf(CmdMsg->para[2], "%d", (int*)&readCnt);
	
	zDrvDebug_Printf("Start Read Device Block Cnt %d, Base %d", readCnt, readBase);
	
	while(cnt < readCnt)
	{
		ret = zDrvIO_BRead(pIOHnd, pBuf, 1, readBase + cnt);
		if (ret <= 0)
		{
			zDrvDebug_Printf("Read Error Ret = %d !!!\n", ret);
			goto exit;
		}
		zDrvDebug_Printf("\nRead Block Cnt 1, Length %d", blkDevInfo.blkSize);
		
		zDrvDebug_Printf("ASCII:");
		for (i=0; i<blkDevInfo.blkSize; i++)
		{
			if (pBuf[i] >= 32 && pBuf[i] <= 126)
				pChBuf[i] = pBuf[i];
			else
				pChBuf[i] = 32;
		}
		printBuffer(pChBuf, blkDevInfo.blkSize);
		
		zDrvDebug_Printf("HEX:");
		chPos = 0;
		for (i=0; i<blkDevInfo.blkSize; i++)
		{
			sprintf((char*)&pChBuf[chPos],"%02x ", pBuf[i]);
			chPos += 3; 			
		}
		printBuffer(pChBuf, chPos + 1);
		cnt++;
	}

exit:
	zOss_Free(pChBuf);
	zOss_Free(pBuf);
	zDrvIO_Close(pIOHnd);
	
    return ;
}

/*******************************************************************************
 * Function: echoCDevCmdEntry
 * Description:add DRV IO debug cmd.
 * Input:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static VOID echoCDevCmdEntry(T_Shell_CommandMessage *CmdMsg)
{
	T_ZDrvIO_Handle pIOHnd = NULL;
	T_ZDRVIO_FLAGS flags = {0};
	SINT32 ret = DRV_SUCCESS;
	SINT32 writeLen = 0;
	UINT32 len = 0;
	UINT8 *pBuf = NULL;
	
	if (CmdMsg->paraCount == 0) 
	{
		zDrvDebug_Printf("example: echodev uart/0 test ");
		return;
	}

	if (CmdMsg->paraCount != 2)
	{
		zDrvDebug_Printf("echodev invalid param !!!");
		return ;
	}

	flags.READ = 1;
	flags.WRITE = 1;
	
	pIOHnd = zDrvIO_Open((const char*)CmdMsg->para[0], flags);
	if (pIOHnd == NULL)
	{
		zDrvDebug_Printf("Open Device %s Error !!!", CmdMsg->para[0]);
		return ;
	}

	writeLen = strlen(CmdMsg->para[1]);
	pBuf = (UINT8*)CmdMsg->para[1];
	
	while(len < writeLen)
	{
		ret = zDrvIO_Write(pIOHnd, &pBuf[len], writeLen);
		if (ret <= 0)
		{
			zDrvDebug_Printf("Write Error Ret = %d !!!\n", ret);
			goto exit;
		}
		zDrvDebug_Printf("Write Length %d", ret);
		len  += ret;
	}
	
exit:
	zDrvIO_Close(pIOHnd);
	
    return ;
}
/*******************************************************************************
 * Function: echoBDevCmdEntry
 * Description:add DRV IO debug cmd.
 * Input:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static VOID echoBDevCmdEntry(T_Shell_CommandMessage *CmdMsg)
{
	T_ZDrvIO_Handle pIOHnd = NULL;
	T_ZDRVIO_FLAGS flags = {0};
	SINT32 ret = DRV_SUCCESS;
	SINT32 writeLen = 0;
	UINT32 cnt = 0;
	UINT32 writeBase = 0;
	UINT8 *pBuf = NULL;
	UINT8 *pWBuf = NULL;
	T_BlkDev_Info blkDevInfo = {0};
	UINT32 blockCnt = 0;
	
	if (CmdMsg->paraCount == 0) 
	{
		zDrvDebug_Printf("example: echobdev sd 200 Teststring ");
		return;
	}

	if (CmdMsg->paraCount != 3)
	{
		zDrvDebug_Printf("echodev invalid param !!!");
		return ;
	}
	pWBuf = (UINT8*)zOss_Malloc(2048);
	if (pWBuf == NULL)
	{
		zDrvDebug_Printf("malloc buffer fail !!!");
		return ;
	}
	zOss_Memset(pWBuf, 0x0, 2048);
	
	flags.READ = 1;
	flags.WRITE = 1;
	
	pIOHnd = zDrvIO_Open((const char*)CmdMsg->para[0], flags);
	if (pIOHnd == NULL)
	{
	       zOss_Free(pWBuf);
		zDrvDebug_Printf("Open Device %s Error !!!", CmdMsg->para[0]);
		return ;
	}

	sscanf(CmdMsg->para[1], "%d", (int*)&writeBase);
	
	writeLen = strlen(CmdMsg->para[2]);
	pBuf = (UINT8*)CmdMsg->para[2];
	zOss_Memcpy(pWBuf, pBuf, writeLen);
	
	zDrvIO_Ctrl(pIOHnd,IOCTL_BLKDEV_GET_INFO,&blkDevInfo);
	if (blkDevInfo.blkSize > 2048 || blkDevInfo.blkSize == 0)
	{
		zDrvDebug_Printf("Error Block Size %d !!!", blkDevInfo.blkSize);
		goto exit;
	}
	blockCnt = writeLen/blkDevInfo.blkSize;
	blockCnt += writeLen % blkDevInfo.blkSize ? 1: 0;
	
	while(cnt < blockCnt)
	{
		ret = zDrvIO_BWrite(pIOHnd, &pWBuf[cnt*blkDevInfo.blkSize], 1, writeBase + cnt);
		if (ret <= 0)
		{
			zDrvDebug_Printf("Write Error Ret = %d !!!\n", ret);
			goto exit;
		}
		zDrvDebug_Printf("Write Length %d", ret);
		cnt++;
	}
	
exit:
	zDrvIO_Close(pIOHnd);
	
	//if (pWBuf != NULL)
	{
	       zOss_Free(pWBuf);
	}
	
    return ;
}

/*******************************************************************************
 * Function: notifier
 * Description:.
 * Input:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static VOID notifier(const char* devName, T_DRVIO_EVENT event)
{
    if (event == DRVIO_EV_CONNECT)
    {
        zDrvDebug_Printf("DRVIO(%d): Device [%s] connect", zOss_GetTickCount(), devName);
    }
    else if (event == DRVIO_EV_PRE_DISCONNECT)
    {
        zDrvDebug_Printf("DRVIO(%d): Device [%s] pre disconnect", zOss_GetTickCount(), devName);
    }
    else if (event == DRVIO_EV_DISCONNECT)
    {
        zDrvDebug_Printf("DRVIO(%d): Device [%s] disconnect", zOss_GetTickCount(), devName);
    }
    else
    {
        zDrvDebug_Printf("DRVIO(%d): Device [%s] unknown event", zOss_GetTickCount(), devName);
    }

}

/*******************************************************************************
 * Function: zDrvIODev_Initiate
 * Description:initialize the io base structure.
 * Input:
 * Output:None
 *
 * Returns:
              DRV_SUCCESS: success.
              DRV_ERROR: error.
 * Others:
 ********************************************************************************/
SINT32 zDrvIODev_Initiate(VOID)
{
    if (g_drvIOTable.opMutex)
    {
        return DRV_SUCCESS;
    }

    g_drvIOTable.opMutex = zOss_CreateMutex("IO table mutex", ZOSS_INHERIT);
    g_drvIOTable.devListCount = 0;
    INIT_LIST_HEAD(&g_drvIOTable.devList);
    g_drvIOTable.notifyCount= 0;
    INIT_LIST_HEAD(&g_drvIOTable.notifyChain);

    zOss_AddShellCmd("lsdev", lsDevCmdEntry, "鿴IO豸б");
    zOss_AddShellCmd("catcdev", catCDevCmdEntry, "ַ豸");
    zOss_AddShellCmd("catbdev", catBDevCmdEntry, "豸");
    zOss_AddShellCmd("echocdev", echoCDevCmdEntry, "дַ豸");
    zOss_AddShellCmd("echobdev", echoBDevCmdEntry, "д豸");

    zDrvIO_RegisterGlobalNotify(notifier);

    return DRV_SUCCESS;
}


