/*******************************************************************************
* Copyright (C) 2013, ZTE Corporation.
*
* File Name:    ring_queue.c
* File Mark:
* Description:
* Others:
* Version:		 V1.0
* Author:		 geanfeng
* Date: 		 2013-09-24
********************************************************************************/

#include "drvs_general.h"
#include "drvs_assert.h"
#include "ring_queue.h"

UINT32 g_ringQueueMsr = 0;

/**
 * queue_lock
 */
static SINT32 queue_lock(T_Ring_Queue *queue)
{
    UINT32 old_intr;

    if (queue->multi_user_protect == QUEUE_PROTECT_MUTEX)
    {
        zOss_GetSemaphore(queue->lock , ZOSS_WAIT_FOREVER);
    }
    else if (queue->multi_user_protect == QUEUE_PROTECT_IRQ)
    {
        LOCK_SAVE(old_intr);
        g_ringQueueMsr = old_intr;
    }

    return 0;
}

/**
 * queue_unlock
 */
static void queue_unlock(T_Ring_Queue *queue)
{
    UINT32 old_intr;

    if (queue->multi_user_protect == QUEUE_PROTECT_MUTEX)
    {
        zOss_PutSemaphore(queue->lock );
    }
    else if (queue->multi_user_protect == QUEUE_PROTECT_IRQ)
    {
        old_intr = g_ringQueueMsr;
        LOCK_RESTORE(old_intr);
    }
}


/**
 * ringQueue_Create - create a new ring queue.
 *
 * Create a new special ring queue.
 */
T_Ring_Queue *ringQueue_Create(UINT32 unit_count, UINT32 unit_size, T_USER_PROTECT_POLICY multi_user_protect)
{
    T_Ring_Queue *queue = NULL;
    UINT8 *buffer = NULL;

    queue = (T_Ring_Queue*)zOss_Malloc(sizeof(T_Ring_Queue));
    if (!queue)
        return NULL;

    zOss_Memset(queue, 0x0, sizeof(T_Ring_Queue));

    buffer = (UINT8 *)zOss_Malloc((unit_count + 1) * unit_size);
    if (!buffer)
    {
        zOss_Free(queue);
        return NULL;
    }

    queue->unit_buffer = buffer;
    queue->unit_buffer_size = (unit_count +1) * unit_size;
    queue->read_pos = 0;
    queue->write_pos = 0;
    queue->unit_size = unit_size;
    queue->unit_count = unit_count + 1;
    queue->multi_user_protect = multi_user_protect;

    if (queue->multi_user_protect == QUEUE_PROTECT_MUTEX)
    {
        queue->lock = zOss_CreateSemaphore("ring queue lock", 1);
        zDrv_ASSERT(queue->lock );
    }

    return queue;
}

/**
 * ringQueue_Init - initialize a new ring queue.
 *
 *
 */
SINT32 ringQueue_Init(T_Ring_Queue *queue, UINT8 *unit_buffer,
                      UINT32 unit_count, UINT32 unit_size, T_USER_PROTECT_POLICY protect_policy)
{

    if (queue == NULL || unit_buffer == NULL)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    zOss_Memset(queue, 0x0, sizeof(T_Ring_Queue));

    queue->unit_buffer = unit_buffer;
    queue->unit_buffer_size = unit_count * unit_size;
    queue->read_pos = 0;
    queue->write_pos = 0;
    queue->unit_size = unit_size;
    queue->unit_count = unit_count;
    queue->multi_user_protect = protect_policy;

    if (queue->multi_user_protect > QUEUE_PROTECT_RAW)
    {
        queue->lock = zOss_CreateSemaphore("ring queue lock", 1);
        zDrv_ASSERT(queue->lock );
    }

    return 0;
}

/**
 * ringQueue_Enqueue
 *
 *
 */
SINT32 ringQueue_Enqueue(T_Ring_Queue *queue, void *unit)
{
    UINT32 pos;
    SINT32 ret = 0;

    if (queue == NULL || unit == NULL)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    if (queue->multi_user_protect > QUEUE_PROTECT_RAW)
        queue_lock(queue);

    pos = queue->write_pos + queue->unit_size;
    if (pos >= queue->unit_buffer_size)
        pos = 0;

    if (pos == queue->read_pos)
    {
        ret = DRV_ERROR_FULL; /*full*/
        goto quit;
    }

    zOss_Memcpy(&queue->unit_buffer[queue->write_pos], unit, queue->unit_size);

    queue->write_pos = pos;

    ret = DRV_SUCCESS;
quit:
    if (queue->multi_user_protect > QUEUE_PROTECT_RAW)
        queue_unlock(queue);
    return ret;
}

/**
 * ringQueue_Dequeue
 *
 *
 */
SINT32 ringQueue_Dequeue(T_Ring_Queue *queue, void *unit)
{
    SINT32 ret = 0;

    if (queue == NULL || unit == NULL)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    if (queue->multi_user_protect > QUEUE_PROTECT_RAW)
        queue_lock(queue);

    if (queue->read_pos == queue->write_pos)
    {
        ret = DRV_ERROR_EMPTY; /*empty*/
        goto quit;
    }

    zOss_Memcpy(unit, &queue->unit_buffer[queue->read_pos], queue->unit_size);

    if ((queue->read_pos + queue->unit_size) >= queue->unit_buffer_size)
        queue->read_pos = 0;
    else
        queue->read_pos += queue->unit_size;

    ret = DRV_SUCCESS;
quit:
    if (queue->multi_user_protect > QUEUE_PROTECT_RAW)
        queue_unlock(queue);
    return ret;
}

/**
 * ringQueue_Empty
 *
 *
 */
SINT32 ringQueue_Empty(T_Ring_Queue *queue)
{

    if (queue == NULL)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    if (queue->read_pos == queue->write_pos)
        return 1;
    else
        return 0;
}

/**
 * ringQueue_Full
 *
 *
 */
SINT32 ringQueue_Full(T_Ring_Queue *queue)
{
    UINT32 pos;

    if (queue == NULL)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    pos = queue->write_pos + queue->unit_size;
    if (pos >= queue->unit_buffer_size)
        pos = 0;

    if (pos == queue->read_pos)
        return 1;
    else
        return 0;
}

/**
 * ringQueue_Fush
 * attention: 豣֤ring_queue_dequeue
 *
 */
void ringQueue_Fush(T_Ring_Queue *queue)
{
    if (queue == NULL)
    {
        //zDrv_ASSERT(0);
        return ;
    }

    if (queue->multi_user_protect > QUEUE_PROTECT_RAW)
        queue_lock(queue);

    queue->read_pos = queue->write_pos;

    if (queue->multi_user_protect > QUEUE_PROTECT_RAW)
        queue_unlock(queue);
    return ;
}

/**
 * ringQueue_Destroy
 *
 *
 */
void ringQueue_Destroy(T_Ring_Queue *queue)
{

    if (queue == NULL)
    {
        //zDrv_ASSERT(0);
        return ;
    }

    if (queue->multi_user_protect == QUEUE_PROTECT_MUTEX)
    {
        zOss_DeleteSemaphore(queue->lock );
    }

    zOss_Free(queue->unit_buffer);
    zOss_Memset(queue, 0x0, sizeof(T_Ring_Queue));
    zOss_Free(queue);

    return ;
}

