/*
 * Ethernet driver for zte zx2975xx gmac on chip network device
 * (c)2008 http://www.zte.com.cn
 * Authors:	zhang dongdong <zhang.dongdong16@zte.com.cn>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/phy.h>
#include <linux/platform_device.h>
#include <linux/gmac/gmac.h>
#include "zx29_gmac.h"

#define gmac_printk(_format, _args...)		do{printk(KERN_INFO"gmac," _format "\n",##_args);}while(0)

struct tasklet_struct	*g_gmac_tasklet = NULL;

static void gmac_hw_deinit(struct net_device *dev);

struct zx29_gmac_dev	*g_gmac_dev = NULL;

static u8 zx29_gmac_addr[MAC_ADDR_LENTH] = {0xec,0x1d,0x7f,0xb0,0x2f,0x32};

// ȡõǰĽ
static struct bd_rx *get_rx_bd(struct net_device *dev)
{
    struct zx29_gmac_dev* prv 	= (struct zx29_gmac_dev*)netdev_priv(dev);
    int   n						= prv->rx_bd_offset;
    struct bd_rx *d				= (struct bd_rx*)prv->dma_rx_vir;

    if(d[n].RDES0 & DMA_OWNER) 
    {
        return 0;
    }
    else	
    {
        return &d[n];
    }
}

static inline int mod_sub(int left, int right, int mod)
{
    return (mod - right + left) % mod;
}

// ȡõǰķ
static inline struct bd_tx *get_tx_bd(struct net_device *dev)
{
    struct zx29_gmac_dev* prv 	= (struct zx29_gmac_dev*)netdev_priv(dev);
    int   n						= prv->tx_bd_offset;
    struct bd_tx *d				= (struct bd_tx*)prv->dma_tx_vir;
	
    if( mod_sub(prv->tx_bd_offset, prv->txed_bd, GMAC_TX_BD_NUM) > GMAC_TX_BD_NUM - 2 )  
	{
		return 0;
	}
		
    if(d[n].TDES0 & DMA_OWNER) 	
    {
        return 0;
    }
    else
    {
        return &d[n];
    }
}

// ȡѷ
static struct bd_tx *get_txed_bd(struct net_device *dev)
{
    struct bd_tx *d;
    struct zx29_gmac_dev* prv 	= (struct zx29_gmac_dev*)netdev_priv(dev);
    int n						= prv->txed_bd;

    d	= (struct bd_tx*)prv->dma_tx_vir;

	if(n == prv->tx_bd_offset) return 0;
	
    if(d[n].TDES0 & DMA_OWNER)	return 0;
	
	if(d[n].skb == NULL) return 0; 

    prv->txed_bd++;
    prv->txed_bd %= GMAC_TX_BD_NUM;

    return &d[n];
}

static void mac_init(struct net_device *dev)
{
    volatile unsigned *gmac = (unsigned*)dev->base_addr;
    unsigned int i=0, j=0, mac_rst=0;

    mac_provide_clock();
#ifdef __DEAD_LOOP_POLL__                       //////ܻѭ޷˳ں˳ʼ޷/////////
    mac_reset();  								// λ
    mac_set_gmii_mode();						// MIIӿ
    mac_wait_reset_finished();
#else                                           /////޸ĴҪϸ////////
    //LOG_INFO("wait for gmac reseting");
    for(i=0; i<MAC_RESET_NUM; i++)              /////λԴ/////
    {
        mac_reset();
        mac_set_mii_mode();                 //Ϊmiiģʽ10m100m
        for(j=0; j<MAC_WAIT_TIME; j++)          /////ѯλ״̬ʱѭ10.1ms////////
        {
            printk(".");
            if(!((MAC(0x1000)) & 1))
            {
                printk("ok\n");
                mac_rst = 1;
                goto mac_reset_option;
            } 
            udelay(100);                        ////ȴ 0.1ms
        }
    }
mac_reset_option:
    if(!mac_rst)                                ////gmacmacоƬûиλɹ//////
    {
        printk("gmac reset failed!\n");
    }
#endif
    while(mac_mii_is_busy());
}

static int gmac_init_rx_bd(struct net_device *dev, struct zx29_gmac_dev* prv)
{
    struct sk_buff	*skb = NULL;
    struct bd_rx *rx  = (struct bd_rx *)prv->dma_rx_vir;
    int 			i = 0;	

    prv->rx_bd_offset		= 0;

	for(i = 0; i < GMAC_RX_BD_NUM; i++)
	{
		// ʼ
		// ݻΪ SKBdataΣȷӦskb
		// ڽϣGMACӲݱ󣬽skbȡ£͸Эϲ
		// ͬʱҪ·һSKBٹҵ

		skb = netdev_alloc_skb(dev, GMAC_FRAME_LEN + NET_IP_ALIGN);

		if(unlikely(!skb))
		{
		   // LOG_ERR("kmalloc memory failed\n");
			gmac_hw_deinit(dev);
			return -1;
		}

		skb_reserve(skb, NET_IP_ALIGN);

		rx[i].RDES0 	   |= DMA_OWNER;
		rx[i].RDES1  = 0;
		rx[i].RDES1 	   = GMAC_FRAME_LEN | 1 << 14;
		rx[i].dma_buf		= virtaddr_to_phys((unsigned)skb->data);
		rx[i].next			= prv->dma_rx_phy + ((i + 1) << 5);
		rx[i].skb			= skb;
#if 0
		if(i%4 != 0)
		{
	       rx[i].RDES1  |= 0x80000000;
		}
#endif
		dma_sync_single_for_device(&dev->dev, rx[i].dma_buf, GMAC_FRAME_LEN, DMA_TO_DEVICE);

	}

	// ´Ƕлش
	rx[GMAC_RX_BD_NUM-1].next			= prv->dma_rx_phy;
	rx[GMAC_RX_BD_NUM-1].RDES1 			= GMAC_FRAME_LEN	  | 1 << 14 | 1 << 15;

	return 0;
}

static void gmac_init_tx_bd(struct zx29_gmac_dev* prv)
{
    struct bd_tx *tx  = (struct bd_tx *)prv->dma_tx_vir;

    int 			i = 0;

    prv->tx_bd_offset		= 0;
    prv->txed_bd			= 0;

    for(i = 0; i < GMAC_TX_BD_NUM; i++)
    {
        // ʼ
        // ݻΪ SKBdataΣʲҪݻ
        // ʱSKBڷϣGMACӲݷͺݲȻƻʱٴ

        tx[i].TDES0       	= (1 << 20 | 1 << 30);
        tx[i].TDES1       	= GMAC_FRAME_LEN;
        tx[i].next			= prv->dma_tx_phy + ((i + 1) << 5);
    }

    // ´Ƕлش
    tx[GMAC_TX_BD_NUM-1].next		    	= prv->dma_tx_phy;
    tx[GMAC_TX_BD_NUM-1].TDES0            	= 1 << 20 | 1 << 21 | 1 << 30;	
}

static void gmac_stop(void* io)
{
	volatile unsigned *gmac = (unsigned*)io;

    dma_disable();
    mac_disable();
    mac_int_disable();

    // շFIFO
    dma_clear_tx_fifo();
    dma_wait_tx_fifo_cleared();
}

static void gmac_start(void* io)
{
	volatile unsigned *gmac = (unsigned*)io;

    mac_int_enable();
    dma_enable();
    mac_enable();
}

static void gmac_trig_transmit(void* io)
{
    volatile unsigned *gmac 	= (unsigned*)io;
    register unsigned status	= (MAC(0x1014) >> 20) & 0x07;
    switch(status)
    {
        case 0:		// ֹͣλֹͣ
            dma_enable();
            break;
        case 6:		// 𣻴ûߴ仺硣
            dma_continue_tx();
            break;
        case 1:		// ڽУȡ
        case 2:		// ڽУȴ״̬
        case 3:		// ڽУӷͻȡݲ͵ FIFO(TxFIFO)
        case 4:		// дʱ״̬
        case 5:		// 
        case 7:		// Уرմ
        default:
            break;
    }
}

static void gmac_trig_receive(void* io)
{
    volatile unsigned *gmac 	= (unsigned*)io;
    register unsigned status	= (MAC(0x1014) >> 17) & 0x07;;
    switch(status)
    {
        case 0:		// ֹͣ
            dma_enable();
            break;
        case 4:		// 
            dma_continue_rx();
            break;
        default:
            break;
    }
}

// ӲMACַ
static inline void gmac_update_mac(struct net_device *dev)
{
    volatile unsigned *gmac = (unsigned*)dev->base_addr;
    unsigned char *mac		= (unsigned char *)(dev->dev_addr);

    MAC(0x0044)	= mac[0] | mac[1] << 8 | mac[2] << 16 | mac[3] << 24;
    MAC(0x0040) = mac[4] | mac[5] << 8;

    //LOG_INFO("MAC: %02X-%02X-%02X-%02X-%02X-%02X\n", mac[5], mac[4], mac[3], mac[2], mac[1], mac[0]);
}

static void gmac_hw_deinit(struct net_device *dev)
{
    int 				i;
    struct bd_rx 		*rx_bd;
	struct bd_tx 		*tx_bd;
    volatile unsigned 	*gmac	= (unsigned*)dev->base_addr;
    struct zx29_gmac_dev *prv	= (struct zx29_gmac_dev*)netdev_priv(dev);

    gmac_stop((void*)dev->base_addr);

    // עrx tx һ
    if(prv->dma_rx_phy) {
        rx_bd   = (struct bd_rx*)prv->dma_rx_vir;

        for(i = 0; i < GMAC_RX_BD_NUM; i++) {
        	if(rx_bd[i].skb != NULL)      
 	           dev_kfree_skb_any(rx_bd[i].skb);
        }

		tx_bd   = (struct bd_tx*)prv->dma_tx_vir;
		
        for(i = 0; i < GMAC_TX_BD_NUM; i++) {
        	if(tx_bd[i].skb != NULL)
            	dev_kfree_skb_any(tx_bd[i].skb);
        }

       // dma_free_coherent(NULL, GMAC_BUF_LEN, prv->dma_rx_vir, prv->dma_rx_phy);   //ͷdma
    }

    dma_set_tx_buffer(0);   //׸BDĻΪ0
    dma_set_rx_buffer(0);

    prv->rx_bd_offset	= 0;
    prv->tx_bd_offset	= 0;
    prv->txed_bd		= 0;
    prv->dma_rx_phy		= 0;
    prv->dma_rx_vir		= 0;
    prv->dma_tx_phy		= 0;
    prv->dma_tx_vir		= 0;
}

void gmac_set_speed_duplex(struct net_device *dev, int speed, int duplex)
{
    unsigned			val;
    struct zx_net_dev* 	prv 	= (struct zx_net_dev*)netdev_priv(dev);
    volatile unsigned 	*gmac 	= (unsigned*)dev->base_addr;

    val = MAC(0x0000) | 1<<11 | 1<<14;  // ĬΪ100Mȫ˫
    if(SPEED_10 == speed)
    {
        val &= ~(1 << 14);                  // 10M
    }
    if(DUPLEX_HALF == duplex)
    {
        val &= (~(1 << 11));                    // ˫
        val |= (1 << 16);
    }
    MAC(0x0000) = val;
}

static int gmac_hw_init(struct net_device *dev)
{
	int ret = -1;
    unsigned 		val;

    volatile unsigned *gmac	= (unsigned*)dev->base_addr;   //WL  豸IOַ
    struct zx29_gmac_dev* prv 	= (struct zx29_gmac_dev*)netdev_priv(dev);

    if(prv->dma_rx_phy)
        gmac_hw_deinit(dev);

    prv->dma_rx_vir = prv->dma_rx_vir_init;
	prv->dma_rx_phy = prv->dma_rx_phy_init;
    prv->dma_tx_vir	= prv->dma_rx_vir + GMAC_RX_BUF_LEN;
    prv->dma_tx_phy	= prv->dma_rx_phy + GMAC_RX_BUF_LEN;

    // 
    memset(prv->dma_rx_vir, 0, GMAC_BUF_LEN);

	//ec616000600086,should check init_rx result
    ret = gmac_init_rx_bd(dev, prv);
	if(ret < 0)
	{
		printk("hw_net_init,init_rx_bd fail\n");
		return ret;
	}
    gmac_init_tx_bd(prv);

    mac_init(dev);

    /*رGMACй*/
    dma_disable();
    mac_disable();
    mac_int_disable();

    /*ͻ䳤*/
    val		 	= MAC(0x1000);
    val			&= ~(0x3F << 8);
    val			|=  (0x10 << 8);    //burst 16
    MAC(0x1000)	 = val;

    /*  ý뷢*/
    dma_set_rx_buffer(prv->dma_rx_phy);
    dma_set_tx_buffer(prv->dma_tx_phy);

    mac_int_clear(0x0001FFFF);								// ж
    while(mac_mii_is_busy());
	
	//set speed and duplex mode
	gmac_set_speed_duplex(dev, prv->phydev->speed, prv->phydev->duplex);
	
    mac_rece_all_data();									// 
    //gmac_set_filter(dev);                                 //ʱԲô

	gmac_start((void*)dev->base_addr);
    return 0;
}

static int zx29mii_read(struct mii_bus *bus, int phy_addr, int regnum)
{
	unsigned long flags;
	struct zx29_gmac_dev *prv = (struct zx29_gmac_dev*)bus->priv;
	volatile unsigned 	*gmac	= (unsigned*)prv->base_addr;

	unsigned val	= ( 1 << 0 					|		// busy λ
					    0 << 1 					|		// R/Wָʾλ
					    PHY_CLOCK << 2)			|		// ʱλ
					    (regnum & 0x1F) << 6	|		// Ĵ
					    (phy_addr & 0x1F) << 11;		// оƬ

	spin_lock_irqsave(&prv->lock,flags);
	while(mac_mii_is_busy());
	MAC(0x0010) 	= val;
	spin_unlock_irqrestore(&prv->lock,flags);

	while(mac_mii_is_busy());

	return (MAC(0x0014) & 0xFFFF);
}

static int zx29mii_write(struct mii_bus *bus, int phy_addr, int regnum, u16 value)
{
	struct zx29_gmac_dev *prv = (struct zx29_gmac_dev*)bus->priv;
	volatile unsigned 	*gmac	= (unsigned*)prv->base_addr;

	unsigned data	= ( 1 << 0 					|		// busy λ
					    1 << 1 					|		// R/Wָʾλ
					    PHY_CLOCK << 2)			|		// ʱλ
					    (regnum & 0x1F) << 6	|		// Ĵ
					    (phy_addr & 0x1F) << 11;		// оƬ

	spin_lock_irq(&prv->lock);

	while(mac_mii_is_busy());
	MAC(0x0014)		= value;
	MAC(0x0010)		= data;

	spin_unlock_irq(&prv->lock);

	while(mac_mii_is_busy());

	return 0;
}
static int zx29mii_reset(struct mii_bus *bus)
{
	struct zx29_gmac_dev *priv = bus->priv;
	volatile unsigned *gmac = (unsigned*)priv->base_addr;
	gmac_start(priv->base_addr);
	while(mac_mii_is_busy());
	return 0;
}

static inline void zx29_gmac_set_macaddr(struct net_device* ndev)
{
    // ȡMAC ַ: 
    // MACĴеĵַЧʹøMAC
    // Ч//zx297520v2   ֱmac ַ
 
   int i =0;

#if  MAC_ADDR_SET
    for(i = 0;i<MAC_ADDR_LENTH;i++)
    {
     ndev->dev_addr[i]   = zx29_gmac_addr[i];
    }

    if(!is_valid_ether_addr(ndev->dev_addr))
    {
	    random_ether_addr(ndev->dev_addr);
    }
#else
    random_ether_addr(ndev->dev_addr);
#endif
}


static inline void zx29_gmac_linkisup(struct net_device *dev, int isup)
{
	struct zx29_gmac_dev *priv = netdev_priv(dev);
	struct phy_device *phydev = priv->phydev;

	priv->link.duplex = phydev->duplex;
	priv->link.giga = (phydev->speed == 100);
	if (priv->link.speed != phydev->speed) {
		priv->link.speed = phydev->speed;
		//zx29_gmac_set_rgmii_txclock(priv);
	}
	priv->link.isup = isup;
	if (isup)
		netif_carrier_on(dev);
	phy_print_status(phydev);
}

static void zx29_gmac_stats_printk(struct net_device *dev)
{
	printk("dev->stats.rx_errors = %d.\n", dev->stats.rx_errors);
	printk("dev->stats.rx_dropped = %d.\n", dev->stats.rx_dropped);
	printk("dev->stats.rx_packets = %d.\n", dev->stats.rx_packets);
	printk("dev->stats.tx_packets = %d.\n", dev->stats.tx_packets);
}
#if 0
static void zx29_gmac_dump_reg(struct net_device *dev)
{
	volatile unsigned 	*gmac 	= (unsigned*)dev->base_addr;
	u32 i;

	printk("MAC:");
	for(i=0;i<11;i++){
		if(!(i%4)){printk("\n");}
		printk("[%04x]:%04x",(0x1000+i*4),);
	}
	
}
#endif
static void zx29_gmac_adjust_link(struct net_device *dev)
{
	struct zx29_gmac_dev *priv = netdev_priv(dev);
	struct phy_device *phydev = priv->phydev;
	volatile unsigned 	*gmac 	= (unsigned*)dev->base_addr;

	//printk("@@@@@@@@@@zx29_gmac_adjust_link.\n");
	//zx29_gmac_stats_printk(dev);
	
	if (priv->link.isup &&
			(!phydev->link ||
			(priv->link.speed != phydev->speed) ||
			(priv->link.duplex != phydev->duplex))) {
		priv->link.isup = 0;
		netif_tx_disable(dev);
		if (!phydev->link) {
			netif_carrier_off(dev);
			phy_print_status(phydev);
			//gmac_event_notify(GMAC_ETH_PLUGOUT, NULL);
		}
	}

	if (!priv->link.isup && phydev->link) {
		if (priv->link.duplex != phydev->duplex) {
			if (phydev->duplex)
				mac_set_full_duplex_mode();
			else
				mac_set_half_duplex_mode();
		}

		if (priv->link.giga != (phydev->speed == 100)) {
			if (phydev->speed == 100) 
				mac_set_speed_100m_mode();
			else 
				mac_set_speed_10m_mode();
		}

		zx29_gmac_linkisup(dev, 1);
		//gmac_event_notify(GMAC_ETH_PLUGIN, NULL);
	}
}

static inline int zx29_gmac_phy_start(struct net_device *dev)
{
	struct zx29_gmac_dev *priv = netdev_priv(dev);
	int i = 0;
	struct phy_device *p = NULL;
	int ret= 0;

	if(priv->nports == 1) {
		p = phy_find_first(priv->mii.bus);
 	} else {
 		if(priv->rmii_port < PHY_MAX_ADDR)
			p = priv->mii.bus->phy_map[priv->rmii_port];
	}

	if(!p) {
		printk("%s: no PHY found\n", dev->name);
		return -ENODEV;
	}

	ret = phy_connect_direct(dev, p, &zx29_gmac_adjust_link, 0,
			PHY_INTERFACE_MODE_RMII);

	if (ret) {
		printk(KERN_ERR "%s: Could not attach to PHY\n", dev->name);
		return ret;
	}
	p->supported &= PHY_BASIC_FEATURES;
	p->advertising = p->supported;
	priv->phydev = p;
	return 0;
}

static int zx29_gmac_get_settings(struct net_device *dev, struct ethtool_cmd *cmd)
{
    struct zx29_gmac_dev* prv = (struct zx29_gmac_dev*)netdev_priv(dev);
    int ret = 0;

    spin_lock_irq(&prv->lock);
    cmd->advertising = prv->set_duplex_mode;
    cmd->autoneg = prv->autoneg;
    spin_unlock_irq(&prv->lock);

    return ret;
}

static int zx29_gmac_do_settings(struct net_device *dev, struct ethtool_cmd *cmd)
{
	u32 speed = ethtool_cmd_speed(cmd);
	struct zx29_gmac_dev* prv = (struct zx29_gmac_dev*)netdev_priv(dev);

	spin_lock_irq(&prv->lock);
	if(cmd->advertising == prv->set_duplex_mode)
	{
		spin_unlock_irq(&prv->lock);
		return 0;
	}
	spin_unlock_irq(&prv->lock);

	//verify the setting we care about
	if(cmd->autoneg == AUTONEG_DISABLE &&
	((speed != SPEED_100 &&
	speed != SPEED_10) ||
	(cmd->duplex != DUPLEX_HALF &&
	cmd->duplex != DUPLEX_FULL)))
	return -EINVAL;

	prv->set_duplex_mode = cmd->advertising;
	prv->autoneg = prv->phydev->autoneg = cmd->autoneg;
	prv->phydev->speed = speed;
	prv->phydev->duplex = cmd->duplex;
	return genphy_config_aneg(prv->phydev);
}


static u32 zx29_gmac_get_link(struct net_device *dev)
{
	struct zx29_gmac_dev* prv = (struct zx29_gmac_dev*)netdev_priv(dev);

	return prv->link.isup;
}

// պ
static int zx29_gmac_rx(struct net_device *dev)
{
	struct bd_rx 	*rx;
	struct sk_buff	*skb;
	struct sk_buff	*skb_new;
	unsigned		len;
	int   exhausted = 0;

	struct zx29_gmac_dev* prv = (struct zx29_gmac_dev*)netdev_priv(dev);

    rx  = get_rx_bd(dev);

    if(unlikely(!rx))	goto rcv_done;

    while(rx)
    {
        if((rx->RDES0 & ERR_RX_ES) || (rx->RDES0 & ERR_RX_LE))
        {
            dev->stats.rx_errors++;
            if(rx->RDES0 & ERR_RX_LE)	dev->stats.rx_length_errors++;
            if(rx->RDES0 & ERR_RX_OE)	dev->stats.rx_over_errors++;
            if(rx->RDES0 & ERR_RX_IPC)	dev->stats.rx_frame_errors++;
            if(rx->RDES0 & ERR_RX_LC)	dev->stats.rx_fifo_errors++;
            if(rx->RDES0 & ERR_RX_CE)	dev->stats.rx_crc_errors++;
        }
        else
        {
            // Խݽд
            // еskbȡ·skbϣڽ
            // ʱѾskb->dataˣݳskb֪ݻûCACHEͬ
            // skbݳȡͬ
            // ȻϱЭϲ

            len = ((rx->RDES0 >> 16) & 0x3FFF) - 4;
            if(len  > (ETH_FRAME_LEN+8))  
            {
            //LOG_INFO("rx data length more than %d\n", ETH_FRAME_LEN);
                dev->stats.rx_dropped++;
                goto rx_bd_reset;
            }

            skb_new = netdev_alloc_skb(dev, GMAC_FRAME_LEN + NET_IP_ALIGN);
            if(unlikely(!skb_new))
            {
                //LOG_DBG(3, "memory squeeze, dropping packet\n");
                dev->stats.rx_dropped++;
				exhausted++;
            }
            else
            {	
            	exhausted = 0;
            	dev->stats.rx_packets++;
				dev->stats.rx_bytes += len;
				// ݶȡͬDMA buffer ¸cpu
				dma_sync_single_for_cpu(&dev->dev, rx->dma_buf, GMAC_FRAME_LEN, DMA_FROM_DEVICE);
                skb				= rx->skb;
                skb_put(skb, len);
                skb->protocol 		= eth_type_trans(skb, dev);
                netif_rx(skb);		
				
                skb_reserve(skb_new, NET_IP_ALIGN);
                //rx->dma_buf    		= __virt_to_phys((unsigned)skb_new->data);
                rx->dma_buf    		= virtaddr_to_phys((unsigned)skb_new->data);
                rx->skb			= skb_new;
				wmb();
                // ݶȡͬ豸
				dma_sync_single_for_device(&dev->dev, rx->dma_buf, GMAC_FRAME_LEN, DMA_TO_DEVICE); //TODO
            }
        }
        rx_bd_reset:
        rx->RDES0 = rx->RDES0 | DMA_OWNER;
        prv->rx_bd_offset++;
        prv->rx_bd_offset %= GMAC_RX_BD_NUM;
		wmb();
		
		if(exhausted >= 10)
			break;
	
        gmac_trig_receive((void*)dev->base_addr);     //գȡմ
        rx	= get_rx_bd(dev);
    }

    rcv_done:
    gmac_trig_receive((void*)dev->base_addr);
	
	return (exhausted >= 10);
}


// жϴ
static void zx29_gmac_tx(struct net_device *dev)
{
    register unsigned status;
    struct net_device_stats	s	= dev->stats;
    struct bd_tx *tx 			= get_txed_bd(dev);

    while(tx)
    {
        status	= tx->TDES0;

        if(tx->TDES0 & ERR_TX_ES)
        {
            s.tx_errors++;
            if(status & ERR_TX_LC)		s.tx_carrier_errors++;
            if(status & ERR_TX_NC)		s.tx_carrier_errors++;
            if(status & ERR_TX_EC)		s.tx_window_errors++;
            if(status & ERR_TX_LATECOL)	s.tx_window_errors++;
            if(status & ERR_TX_UF)		s.tx_aborted_errors++;
            if(status & ERR_TX_ED)		s.tx_aborted_errors++;
            if(status & ERR_TX_JT)		s.tx_fifo_errors++;
            if(status & ERR_TX_FF)		s.tx_fifo_errors++;

			printk("%s, status=0x%x, err_cnt=%ld\n", __FUNCTION__,status, s.tx_errors);
            //LOG_DBG(3, "%s %ld\n", __FUNCTION__, s.tx_errors);
        }
	dev_kfree_skb_any( tx->skb);  //wl
	tx->skb = NULL;
	tx = get_txed_bd(dev);      
    }

   if(netif_queue_stopped(dev))
  // if (netif_carrier_ok(dev))
    {
        netif_wake_queue(dev);    //ʹϲЭʼµ
    }
}

#ifndef GMAC_NO_INT
static irqreturn_t zx29_gmac_interrupt(int irq, void *dev_id)
{
	struct net_device* dev 		= (struct net_device*)dev_id;
	struct zx29_gmac_dev* prv 	= (struct zx29_gmac_dev*)netdev_priv(dev);
	volatile unsigned *gmac 	= (unsigned*)dev->base_addr;

	prv->int_event				= MAC(0x1014);									// ȡжϢ
	MAC(0x1014)					= prv->int_event;								// ж

	// ȽGMACжΣȻҪ̷ж
	// жɺٴGAMCж

	mac_int_disable();
	tasklet_schedule(&prv->tasklet);

	return IRQ_HANDLED;
}
void zx29_gmac_tasklet(unsigned long dev_id)
{
    struct net_device* dev 		= (struct net_device*)dev_id;
    struct zx29_gmac_dev* prv 	= (struct zx29_gmac_dev*)netdev_priv(dev);
    volatile unsigned *gmac 	= (unsigned*)dev->base_addr;

    unsigned events				= prv->int_event;								// ȡжбжϱ

    // GMACжϽֱûж()Ҫ
    do
    {
        if(events & INT_ST_TX)
        {
            zx29_gmac_tx(dev); 
        }                                                     
        if(events & INT_ST_RX)	    
        {
            zx29_gmac_rx(dev);
        }

        events			= MAC(0x1014);
        MAC(0x1014)		= events;

    } while( events & (INT_ST_TX | INT_ST_RX));

    // ʱGMACжǹرյģҪٴδ
    mac_int_enable();															// GMACٴβж
}

#else

void zx29_gmac_tasklet(unsigned long dev_id)
{
    struct net_device* dev 		= (struct net_device*)dev_id;
    struct zx29_gmac_dev* prv 	= (struct zx29_gmac_dev*)netdev_priv(dev);
    volatile unsigned *gmac 	= (unsigned*)dev->base_addr;
	int rx_result = 0;
    unsigned events				= prv->int_event;								// ȡжбжϱ

    // GMACжϽֱûж()Ҫ
    do
    {
        if(events & INT_ST_TX)
        {
            zx29_gmac_tx(dev); 
        }                                                     
        if(events & INT_ST_RX)	    
        {
           rx_result = zx29_gmac_rx(dev);
			if(rx_result)//this means skb is exhausted,so break task
				break;
        }

        events			= MAC(0x1014);
        MAC(0x1014)		= events;

    } while( events & (INT_ST_TX | INT_ST_RX));

    // ʱGMACжǹرյģҪٴδ
   // mac_int_enable();															// GMACٴβж
}

enum hrtimer_restart gmac_timer_callback(struct hrtimer *timer)
{
	unsigned long delay_in_us = GTIMER_INTERVAL;
	ktime_t gmac_schdule_time;
	
	gmac_schdule_time = ktime_set(0, delay_in_us * 1000);
	hrtimer_forward_now(timer, gmac_schdule_time);

	//gmac_kick_thread();
    tasklet_schedule(g_gmac_tasklet);
    //gmac_kick_plug_thread();
    //tasklet_schedule(&gmac_net_dev_prv->tasklet);
	return HRTIMER_RESTART;
}

#endif

static int zx29_gmac_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
	unsigned long		flags;
	unsigned			len;
	struct sk_buff		*skb_old;
	struct bd_tx		*tx;
	struct zx29_gmac_dev*	prv = (struct zx29_gmac_dev*)netdev_priv(dev);
	
	if(0 == prv->link.isup)
	{
		/*for 616000599226,phy not link ,so free skb 
			*/
		dev_kfree_skb_any(skb);
		//netif_stop_queue(dev);
		printk("TSP zx29 gmac xmit  phy not link\n");	 
		return NETDEV_TX_OK;
	}
	
	// ȡʱҪ
	spin_lock_irqsave(&prv->lock, flags);

	if(prv->stopped)
	{
		spin_unlock_irqrestore(&prv->lock,flags);
		dev_kfree_skb_any(skb);
		
		printk("zx_net_start_xmit when stopped\n");
		
		return NETDEV_TX_OK;
	}

	tx	= get_tx_bd(dev);

	if(!tx)
	{
		spin_unlock_irqrestore(&prv->lock,flags);
		dev_kfree_skb_any(skb);
		return NETDEV_TX_OK;
	}
	prv->tx_bd_offset++;
	prv->tx_bd_offset %= GMAC_TX_BD_NUM;
	spin_unlock_irqrestore(&prv->lock, flags);

	if(skb->len > ETH_FRAME_LEN + 4)	//LOG_INFO("tx length too big\n");
	printk("TSP zx29 gmac start xmit len too long\n");

	// ȡϹŵSKBȻ󽫸SKB(SKBѷ)
	// ٽµSKBȥ(SKBַԼskb->dataεַGMAC֮)
	// עڽskb->dataεݷȨGMAC֮ǰҪdataεݽCACHEͬ
	// Ϊskb->datacached

	//wl skb_old		 = tx->skb;
	//flush_skbuf(skb);

	skb= flush_skbuf(skb);
	if(NULL == skb)
		BUG_ON(1);	 
	len 				= MIN(skb->len, GMAC_FRAME_LEN - NET_IP_ALIGN);
	tx->TDES0		  |= (0x07 << 28);					// ״̬Ϊ֡ĵһݰһɺж(һ֡һݰ)
	//tx->dma_buf	 = __virt_to_phys((unsigned)skb->data);
	tx->dma_buf  = virtaddr_to_phys((unsigned)skb->data);
	tx->skb 	 = skb;

	// дͬ
	// dma_sync_single_for_device(&dev->dev, tx->dma_buf, len, DMA_TO_DEVICE);
	//wl dev_kfree_skb_any(skb_old);
	  
	tx->TDES1	 = len;
	tx->TDES0	|= DMA_OWNER;

	//prv->tx_bd_offset++;
	//prv->tx_bd_offset %= GMAC_TX_BD_NUM;
	wmb();
	dev->stats.tx_bytes 		+= len;
	dev->stats.tx_packets++;
	dev->trans_start			= jiffies;

	gmac_trig_transmit((void*)dev->base_addr);
	
	return NETDEV_TX_OK;
}


static void zx29_gmac_tx_timeout(struct net_device *dev)
{
	struct zx29_gmac_dev* prv	= (struct zx29_gmac_dev*)netdev_priv(dev);

	genphy_update_link(prv->phydev);
	prv->link.isup= prv->phydev->link;
	
	if(0 == prv->link.isup) {	
		printk("TSP zx29 gmac net timeout phy not link\n");						// PHY δ
		netif_stop_queue(dev);
		netif_carrier_off(dev);
	} else {
		printk("TSP zx29 gmac net timeout phy linked\n"); 
		gmac_trig_transmit(dev);
		gmac_trig_receive(dev);

		netif_carrier_on(dev);
		netif_wake_queue(dev);
		dev->trans_start		= jiffies;
		dev->stats.tx_errors++;
		dev->stats.tx_dropped++;
	}
}

int zx29_gmac_open(struct net_device *dev)
{
	struct zx29_gmac_dev *prv = netdev_priv(dev);
	unsigned long flags;
	int ret;
	int err = 0;
#ifdef GMAC_NO_INT
	unsigned long delay_in_us = GTIMER_INTERVAL;
	ktime_t gmac_schdule_time;
#endif

	err = phy_read_status(prv->phydev);
    if (err < 0)
	    return err;
		
	spin_lock_irqsave(&prv->lock, flags);
	prv->link.speed = 0;

	zx29_gmac_linkisup(dev, prv->phydev->link);

	ret	= gmac_hw_init(dev);

	if(ret) {
		spin_unlock_irqrestore(&prv->lock, flags);
		return ret;
	}
	
    netif_carrier_on(dev);
	spin_unlock_irqrestore(&prv->lock, flags);
	phy_start(prv->phydev);
	netif_start_queue(dev);

#ifdef GMAC_NO_INT
	gmac_schdule_time = ktime_set(0, delay_in_us * 1000);
	if(prv->timer)
		hrtimer_start(prv->timer, gmac_schdule_time, HRTIMER_MODE_REL );
#endif

	prv->stopped = 0;

	printk("TSP zx29 gmac net open\n");

	return 0;
}

static int zx29_gmac_stop(struct net_device *dev)
{
	unsigned long flags = 0;
	int ret = 0;
	struct zx29_gmac_dev* prv = (struct zx29_gmac_dev*)netdev_priv(dev);
	
	spin_lock_irqsave(&prv->lock, flags);
	
#ifdef GMAC_NO_INT
	ret = hrtimer_cancel(prv->timer);
	if(ret < 0) {
		BUG_ON(1);
		spin_unlock_irqrestore(&prv->lock,flags);
		return ret;
	}
#endif

	prv->stopped = 1;
	
	netif_stop_queue(dev);
	netif_carrier_off(dev);
	phy_stop(prv->phydev);

	gmac_hw_deinit(dev);

	memset(&dev->stats, 0, sizeof(struct net_device_stats));
	
	spin_unlock_irqrestore(&prv->lock,flags);
	printk("TSP zx29 gmac net stop\n");

	return 0;
}

static int zx29_gmac_set_mac_address(struct net_device *dev, void *p)
{
    int ret = eth_mac_addr(dev, p);
	
    if(!ret) {
		gmac_update_mac(dev);
		printk(" zx29 gmac set mac addr ok\n");
    }

    return ret;
}
/*
static struct net_device_stats *zx29_gmac_stats(struct net_device *dev)
{

}
*/
static int zx29_gmac_suspend(struct device *dev)
{
    struct platform_device *pdev	= to_platform_device(dev);
    struct net_device 	*ndev		= platform_get_drvdata(pdev);

    if(ndev) {
    	if(netif_running(ndev)) {
    		netif_device_detach(ndev);
    		gmac_stop((void*)ndev->base_addr);
    	}
    }
    return 0;
}

static int zx29_gmac_resume(struct device *dev)
{
    struct platform_device *pdev	= to_platform_device(dev);
    struct net_device *ndev 		= platform_get_drvdata(pdev);

    if(ndev) {
    	if(netif_running(ndev)) {
        	gmac_start((void*)ndev->base_addr);
        	netif_device_attach(ndev);
    	}
    }
    return 0;
}

static int zx29_gmac_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd)
{
	struct zx29_gmac_dev *priv = netdev_priv(dev);
	if (!(netif_running(dev)))
		return -EINVAL;
	if (!priv->phydev)
		return -EINVAL;
	return phy_mii_ioctl(priv->phydev, ifr, cmd);
}

static const struct ethtool_ops zx29_gmac_ethtool_ops = 
{
    .get_settings	= zx29_gmac_get_settings,
    .set_settings	= zx29_gmac_do_settings,
    .get_link		= zx29_gmac_get_link,
};

static const struct net_device_ops zx29_gmac_netdev_ops =
{
    .ndo_open               = zx29_gmac_open,
    .ndo_stop               = zx29_gmac_stop,
    .ndo_start_xmit         = zx29_gmac_start_xmit,
    .ndo_tx_timeout         = zx29_gmac_tx_timeout,
    .ndo_do_ioctl             =  zx29_gmac_ioctl,
   // .ndo_set_multicast_list =  zx_set_multicast,
    .ndo_change_mtu         = eth_change_mtu,
    .ndo_validate_addr      = eth_validate_addr,
    .ndo_set_mac_address    = zx29_gmac_set_mac_address,
};

static int __devinit zx29_gmac_probe(struct platform_device *pdev)
{
	struct zx29_gmac_dev *prv = NULL;
	struct net_device* ndev = alloc_etherdev(sizeof(struct zx29_gmac_dev));
	volatile unsigned	*gmac = NULL;
	struct gmac_platform_data *pdata = dev_get_platdata(&pdev->dev);
	int ret;
	unsigned long i;
	struct mii_bus *mb;
    struct resource *iomem;

	printk("#########zx29_gmac_probe begin.\n");

	if (!ndev)
		return -ENOMEM;
	iomem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if(!iomem)
		return -ENXIO;

	ndev->base_addr = iomem->start;
	if(!ndev->base_addr)
		return -ENXIO;		
#ifndef GMAC_NO_INT
	ndev->irq = platform_get_irq(pdev, 0);
#endif
	ndev->netdev_ops = &zx29_gmac_netdev_ops;
	ndev->ethtool_ops = &zx29_gmac_ethtool_ops;

	gmac 	= (unsigned*)ndev->base_addr;

    /*رGMACй*/
    dma_disable();
    mac_disable();
    mac_int_disable();

	prv = netdev_priv(ndev);
	memset(prv, 0, sizeof(*prv));
	spin_lock_init(&prv->lock);

	wake_lock_init(&prv->wake_lock, WAKE_LOCK_SUSPEND, "gmac_pm");
    wake_lock(&prv->wake_lock);

	zx29_gmac_set_macaddr(ndev);

#ifndef GMAC_NO_INT
	ret = request_irq(ndev->irq, zx29_gmac_interrupt, 0, ndev->name, ndev);
	if (ret) {
		printk(KERN_ERR "irq request failed: %d\n", ndev->irq);
		goto errirq;
	}
#endif

	ret = register_netdev(ndev);
	if (ret) {
		printk(KERN_ERR "error registering device %s\n",
			ndev->name);
		goto errdev;
	}

	prv->nports = pdata->nports;
	prv->rmii_port = pdata->rmii_port;
	prv->base_addr = ndev->base_addr;

	prv->netdev = ndev;

	mb = mdiobus_alloc();
	if (!mb) {
		printk(KERN_ERR "error allocating mii bus\n");
		goto errmii;
	}
	mb->name = "zx29_gmac_mii";
	mb->read = zx29mii_read;
	mb->write = zx29mii_write;
	mb->reset = zx29mii_reset;
	mb->priv = prv;
	snprintf(mb->id, MII_BUS_ID_SIZE, "%s-%x", pdev->name, pdev->id);
	mb->phy_mask = pdata->port_mask;
	mb->irq = &prv->mii.irq[0];
	for (i = 0; i < PHY_MAX_ADDR; i++) {
		int n = platform_get_irq(pdev, i + 1);
		if (n < 0)
			n = PHY_POLL;
		prv->mii.irq[i] = n;
	}
	gmac_set_clk();
	gmac_phy_release();
	mdelay(500);/*ҪdelayֶȡPHYĴʧܵ*/
	
	mdiobus_register(mb);/*жȡPHY IDĲ֮ǰҪͷPHYĸλ*/
	prv->mii.bus = mb;
	ret = zx29_gmac_phy_start(ndev);
	if (ret) {
		//BUG_ON(1);
		goto errphystart;
	}
	
	platform_set_drvdata(pdev, ndev);

	tasklet_init(&prv->tasklet, zx29_gmac_tasklet, (unsigned long)ndev);
	g_gmac_tasklet = &prv->tasklet;

	prv->dma_rx_vir	= dma_alloc_coherent(NULL,
						GMAC_BUF_LEN,
						&prv->dma_rx_phy,
						GFP_KERNEL);
	if(prv->dma_rx_vir == NULL) {
		BUG_ON(1);
		goto errphystart;
	}
	prv->dma_rx_phy_init = prv->dma_rx_phy;
	prv->dma_rx_vir_init = prv->dma_rx_vir;
    prv->dma_tx_vir	= prv->dma_rx_vir + GMAC_RX_BUF_LEN;
    prv->dma_tx_phy	= prv->dma_rx_phy + GMAC_RX_BUF_LEN;

#ifdef GMAC_NO_INT
	sema_init(&prv->sem, 0);

	prv->timer = kzalloc(sizeof(struct hrtimer),GFP_KERNEL);
	if(prv->timer == NULL){
		BUG_ON(1);
		goto errmalloc;
	}
    hrtimer_init(prv->timer, CLOCK_MONOTONIC, HRTIMER_MODE_REL );

    prv->timer->function = gmac_timer_callback;
#endif

	gmac_event_init();

	g_gmac_dev = prv;

	printk("#########zx29_gmac_probe end.\n");

	return 0;
errmalloc:
	dma_free_coherent(NULL, GMAC_BUF_LEN, prv->dma_rx_vir, prv->dma_rx_phy);
errphystart:
	mdiobus_unregister(prv->mii.bus);
	mdiobus_free(prv->mii.bus);
errmii:
	unregister_netdev(ndev);
errdev:
#ifndef GMAC_NO_INT
	free_irq(ndev->irq, ndev);
#endif
errirq:
	free_netdev(ndev);

	printk("#########zx29_gmac_probe fail.\n");
	return ret;
}

static int __devexit zx29_gmac_remove(struct platform_device *pdev)
{
	struct net_device* ndev = platform_get_drvdata(pdev);
	volatile unsigned	*gmac = NULL;
	if(ndev) {
		struct zx29_gmac_dev *prv = netdev_priv(ndev);
		gmac_hw_deinit(ndev);
		mdiobus_unregister(prv->mii.bus);
#ifndef GMAC_NO_INT
		free_irq(ndev->irq, ndev);
#endif

		tasklet_disable(&prv->tasklet);
    	tasklet_kill(&prv->tasklet);

		if(prv->dma_rx_vir)
			dma_free_coherent(NULL, GMAC_BUF_LEN, prv->dma_rx_vir, prv->dma_rx_phy);   //ͷdma

		wake_unlock(&prv->wake_lock);
		wake_lock_destroy(&prv->wake_lock);

		unregister_netdev(ndev);
		free_netdev(ndev);
		platform_set_drvdata(pdev, NULL);
	}
	return 0;
}

static struct dev_pm_ops zx29_gmac_pm_ops =
{
	.suspend    = zx29_gmac_suspend,
	.resume     = zx29_gmac_resume,
};

static struct platform_driver zx29_gmac_driver = {
	.probe     = zx29_gmac_probe,
	.remove    = __devexit_p(zx29_gmac_remove),
	.driver    = {
		.name     = "zx29_gmac",
		.owner    = THIS_MODULE,
		.pm       = &zx29_gmac_pm_ops,
	},
};

static int __init zx29_gmac_init(void)
{
	printk(KERN_INFO "ZX29 GMAC ethernet driver\n");
	return platform_driver_register(&zx29_gmac_driver);
}


static void __exit zx29_gmac_exit(void)
{
	platform_driver_unregister(&zx29_gmac_driver);
}

module_init(zx29_gmac_init);
module_exit(zx29_gmac_exit);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("ZX29 on chip Ethernet driver");
MODULE_AUTHOR("zhang dongdong <zhang.dongdong16@zte.com.cn>");


