/*
 * Driver for ICPlus PHYs
 *
 * Copyright (c) 2007 Freescale Semiconductor, Inc.
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 *
 */
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/unistd.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/spinlock.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/mii.h>
#include <linux/ethtool.h>
#include <linux/phy.h>
#include <linux/gmac/gmac.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/uaccess.h>

MODULE_DESCRIPTION("ICPlus IP175C/IP101A/IP101G/IC1001 PHY drivers");
MODULE_AUTHOR("Michael Barkowski");
MODULE_LICENSE("GPL");

/* IP101A/G - IP1001 */
#define IP10XX_SPEC_CTRL_STATUS		16	/* Spec. Control Register */
#define IP1001_SPEC_CTRL_STATUS_2	20	/* IP1001 Spec. Control Reg 2 */
#define IP1001_PHASE_SEL_MASK		3	/* IP1001 RX/TXPHASE_SEL */
#define IP1001_APS_ON			11	/* IP1001 APS Mode  bit */
#define IP101A_G_APS_ON			2	/* IP101A/G APS Mode bit */
#define IP101A_G_IRQ_CONF_STATUS	0x11	/* Conf Info IRQ & Status Reg */

/*  Interrupt status Register, page16 register17. */
#define IP10XX_INT_SR             		        17
#define IP10XX_INT_SR_INTR_PIN_USED            0x8000
#define IP10XX_INT_SR_ALL_MASK                 0x800
#define IP10XX_INT_SR_SPEED_MASK               0x400
#define IP10XX_INT_SR_DUPLEX_MASK              0x200
#define IP10XX_INT_SR_LINK_MASK                0x100
#define IP10XX_INT_SR_INT_STATUS               0x8
#define IP10XX_INT_SR_SPEED_INT                0x4
#define IP10XX_INT_SR_DUPLEX_INT               0x2
#define IP10XX_INT_SR_LINK_INT                 0x1

/*  Interrupt status Register, page16 register29. */
#define IP10XX_IO_SPEC_CR             		        29
#define IP10XX_IO_SPEC_CR_SEL_INTR32               0x4

static int ip175c_config_init(struct phy_device *phydev)
{
	int err, i;
	static int full_reset_performed = 0;

	if (full_reset_performed == 0) {

		/* master reset */
		err = mdiobus_write(phydev->bus, 30, 0, 0x175c);
		if (err < 0)
			return err;

		/* ensure no bus delays overlap reset period */
		err = mdiobus_read(phydev->bus, 30, 0);

		/* data sheet specifies reset period is 2 msec */
		mdelay(2);

		/* enable IP175C mode */
		err = mdiobus_write(phydev->bus, 29, 31, 0x175c);
		if (err < 0)
			return err;

		/* Set MII0 speed and duplex (in PHY mode) */
		err = mdiobus_write(phydev->bus, 29, 22, 0x420);
		if (err < 0)
			return err;

		/* reset switch ports */
		for (i = 0; i < 5; i++) {
			err = mdiobus_write(phydev->bus, i,
					    MII_BMCR, BMCR_RESET);
			if (err < 0)
				return err;
		}

		for (i = 0; i < 5; i++)
			err = mdiobus_read(phydev->bus, i, MII_BMCR);

		mdelay(2);

		full_reset_performed = 1;
	}

	if (phydev->addr != 4) {
		phydev->state = PHY_RUNNING;
		phydev->speed = SPEED_100;
		phydev->duplex = DUPLEX_FULL;
		phydev->link = 1;
		netif_carrier_on(phydev->attached_dev);
	}

	return 0;
}

#if 1
static int  ip175l_ports_set(struct phy_device *phydev, int config) 
{
	 printk("####ip175l_port_set####\n");//cuichen
	 mdiobus_write(phydev->bus, 22, 0, 0x07ff); //22.0=0x07FF (Tag based VLAN: Port0-5, PVID classification: Port0-4, VID classification: Port5)
	if (config) {	
        // port3 wan
        /*
            22.0=0x07FF (Tag based VLAN: Port0-5, PVID classification: Port0-4, VID classification: Port5)
            22.4=0x0064 (Port0 PVID=100)
            22.5=0x0064 (Port1 PVID=100)
            22.6=0x00c8 (Port2 PVID=100)
            22.7=0x00c8 (Port3 PVID=200)
            22.10=0x000F (Enable VLAN0-3)
            22.14=0x0064 (VLAN0 VID=100)
            22.15=0x0064 (VLAN1 VID=100)
            22.16=0x0064 (VLAN2 VID=100)
            22.17=0x00c8 (VLAN3 VID=200)
            23.0=0x2727 (VLAN0 member: Port0-2 & 5, VLAN1 member: Port0-2 & 5)
            23.1=0x2827 (VLAN2 member: Port0-2 & 5, VLAN3 member: Port3 & 5)
            23.8=0x2020 (VLAN0 add tag: port5, VLAN1 add tag: port5)
            23.9=0x2020 (VLAN2 add tag: port5, VLAN3 add tag: port5)
            23.16=0x0707 (VLAN0 remove tag: port0-2, VLAN1 remove tag: port0-2)
            23.17=0x0807 (VLAN2 remove tag: port0-2, VLAN3 remove tag: port3)
            */
            //mdiobus_write(phydev->bus, 22, 0, 0x07ff);
            mdiobus_write(phydev->bus, 22, 4, 0x0064);
            mdiobus_write(phydev->bus, 22, 5, 0x0064);
            mdiobus_write(phydev->bus, 22, 6, 0x0064);
            mdiobus_write(phydev->bus, 22, 7, 0x00c8);
            mdiobus_write(phydev->bus, 22, 10, 0x000F);
            mdiobus_write(phydev->bus, 22, 14, 0x0064);
            mdiobus_write(phydev->bus, 22, 15, 0x0064);
            mdiobus_write(phydev->bus, 22, 16, 0x0064);
            mdiobus_write(phydev->bus, 22, 17, 0x00c8);
            
            mdiobus_write(phydev->bus, 23, 0, 0x2727);
            mdiobus_write(phydev->bus, 23, 1, 0x2827);
            mdiobus_write(phydev->bus, 23, 8, 0x2020);
            mdiobus_write(phydev->bus, 23, 9, 0x2020);
            mdiobus_write(phydev->bus, 23, 16, 0x0707);
            mdiobus_write(phydev->bus, 23, 17, 0x0807);
	} else {
		//port3 lan
	    /*
	    22.4=0x0064 (Port0 PVID=100)
            22.5=0x0064 (Port1 PVID=100)
            22.6=0x0064 (Port2 PVID=100)
            22.7=0x0064 (Port3 PVID=100)
            22.10=0x000F (Enable VLAN0-3)
            22.14=0x0064 (VLAN0 VID=100)
            22.15=0x0064 (VLAN1 VID=100)
            22.16=0x0064 (VLAN2 VID=100)
            22.17=0x0064 (VLAN3 VID=100)
            23.0=0x2f2f (VLAN0 member: Port0-3 & 5, VLAN1 member: Port0-3 & 5)
            23.1=0x2f2f (VLAN2 member: Port0-3 & 5, VLAN3 member: Port0-3 & 5)
            23.8=0x2020 (VLAN0 add tag: port5, VLAN1 add tag: port5)
            23.9=0x2020 (VLAN2 add tag: port5, VLAN3 add tag: port5)
            23.16=0x0f0f (VLAN0 remove tag: port0-3, VLAN1 remove tag: port0-3)
            23.17=0x0f0f (VLAN2 remove tag: port0-3, VLAN3 remove tag: port0-3)
            */
	    mdiobus_write(phydev->bus, 22, 4, 0x0064);
            mdiobus_write(phydev->bus, 22, 5, 0x0064);
            mdiobus_write(phydev->bus, 22, 6, 0x0064);
            mdiobus_write(phydev->bus, 22, 7, 0x0064);
            mdiobus_write(phydev->bus, 22, 10, 0x000F);
            mdiobus_write(phydev->bus, 22, 14, 0x0064);
            mdiobus_write(phydev->bus, 22, 15, 0x0064);
            mdiobus_write(phydev->bus, 22, 16, 0x0064);
            mdiobus_write(phydev->bus, 22, 17, 0x0064);
            
            mdiobus_write(phydev->bus, 23, 0, 0x2f2f);
            mdiobus_write(phydev->bus, 23, 1, 0x2f2f);
            mdiobus_write(phydev->bus, 23, 8, 0x2020);
            mdiobus_write(phydev->bus, 23, 9, 0x2020);
            mdiobus_write(phydev->bus, 23, 16, 0x0f0f);
            mdiobus_write(phydev->bus, 23, 17, 0x0f0f);
	}
	return 0;
}
#endif

static int ip175l_auto_adapt(struct phy_device *phydev, struct ifreq *ifr)
{
    	int config = *((int*)ifr->ifr_data);
	ip175l_ports_set(phydev, config);
	return 0;
}

static int ip175l_config_init(struct phy_device *phydev)
{
	int err, i;
	static int full_reset_performed = 0;
	printk("######175L config_init####\n");
	if (full_reset_performed == 0) {

		/* master reset */
		err = mdiobus_write(phydev->bus, 20, 2, 0x175d);
		if (err < 0)
			return err;

		/* ensure no bus delays overlap reset period */
		err = mdiobus_read(phydev->bus, 20, 2);

		/* data sheet specifies reset period is 2 msec */
		mdelay(2);

		/* enable IP175C mode */
		//err = mdiobus_write(phydev->bus, 29, 31, 0x175c);
		//if (err < 0)
			//return err;

		/* Set MII0 speed and duplex (in PHY mode) */
		//err = mdiobus_write(phydev->bus, 29, 22, 0x420);
		//if (err < 0)
			//return err;
        /* reset switch ports */ 
		for (i = 0; i < 4; i++) {
			int bmcr;
			bmcr = mdiobus_read(phydev->bus, i, MII_BMCR);
			if (bmcr < 0)
		        return bmcr;
			bmcr |= BMCR_RESET;
			bmcr = mdiobus_write(phydev->bus, i, MII_BMCR, bmcr);
			if (bmcr < 0)
			    return bmcr;
			do {
			    bmcr = mdiobus_read(phydev->bus, i, MII_BMCR);
		        if (bmcr < 0)
			        return bmcr;
	        } while (bmcr & BMCR_RESET);
		}
		printk("######175L config_init_end####\n");
		full_reset_performed = 1;
	}

	/* force mode */
	if(1)
	{
		int force_mode;

		/*force phy mode, be connected to an external mac*/
		force_mode = mdiobus_read(phydev->bus, 21, 3);
		force_mode &= ~(1<<11);
		err = mdiobus_write(phydev->bus, 21, 3, force_mode);
		if (err < 0)
			return err;

		/*MAC5 force 100M, full duplex*/
		force_mode = mdiobus_read(phydev->bus, 20, 4);
		force_mode |= (1<<15) | (1<<13);
		err = mdiobus_write(phydev->bus, 20, 4, force_mode);
		if (err < 0)
			return err;
#if 0
		/*flow control disable of mii0-2*/
		force_mode = mdiobus_read(phydev->bus, 20, 5);
		force_mode &= ~1;
		err = mdiobus_write(phydev->bus, 20, 5, force_mode);
		if (err < 0)
			return err;

		/*disable select of port5 mirror port*/
		force_mode = mdiobus_read(phydev->bus, 21, 4);
		force_mode |= 0x7<<12;
		err = mdiobus_write(phydev->bus, 21, 4, force_mode);
		if (err < 0)
			return err;

		/*config mii0 phy addr5*/
		force_mode = mdiobus_read(phydev->bus, 21, 1);
		force_mode |= 0x10;
		err = mdiobus_write(phydev->bus, 21, 1, force_mode);
		if (err < 0)
			return err;
#endif
	}

/*
22.0=0x07FF (Tag based VLAN: Port0-5, PVID classification: Port0-4, VID classification: Port5)
22.4=0x0001 (Port0 PVID=1)
22.5=0x0002 (Port1 PVID=2)
22.6=0x0003 (Port2 PVID=3)
22.7=0x0004 (Port3 PVID=4)
22.10=0x000F (Enable VLAN0-3)
22.14=0x0001 (VLAN0 VID=1)
22.15=0x0002 (VLAN1 VID=2)
22.16=0x0003 (VLAN2 VID=3)
22.17=0x0004 (VLAN3 VID=4)
23.0=0x2727 (VLAN0 member: Port0-2 & 5, VLAN1 member: Port0-2 & 5)
23.1=0x2827 (VLAN2 member: Port0-2 & 5, VLAN3 member: Port3 & 5)
23.8=0x2020 (VLAN0 add tag: port5, VLAN1 add tag: port5)
23.9=0x2020 (VLAN2 add tag: port5, VLAN3 add tag: port5)
23.16=0x0707 (VLAN0 remove tag: port0-2, VLAN1 remove tag: port0-2)
23.17=0x0807 (VLAN2 remove tag: port0-2, VLAN3 remove tag: port3)
*/
	mdiobus_write(phydev->bus, 22, 0, 0x07ff);
	mdiobus_write(phydev->bus, 22, 4, 0x0064);
	mdiobus_write(phydev->bus, 22, 5, 0x0064);
	mdiobus_write(phydev->bus, 22, 6, 0x0064);
	mdiobus_write(phydev->bus, 22, 7, 0x00c8);
	mdiobus_write(phydev->bus, 22, 10, 0x000F);
	mdiobus_write(phydev->bus, 22, 14, 0x0064);
	mdiobus_write(phydev->bus, 22, 15, 0x0064);
	mdiobus_write(phydev->bus, 22, 16, 0x0064);
	mdiobus_write(phydev->bus, 22, 17, 0x00c8);

	mdiobus_write(phydev->bus, 23, 0, 0x2727);
	mdiobus_write(phydev->bus, 23, 1, 0x2827);
	mdiobus_write(phydev->bus, 23, 8, 0x2020);
	mdiobus_write(phydev->bus, 23, 9, 0x2020);
	mdiobus_write(phydev->bus, 23, 16, 0x0707);
	mdiobus_write(phydev->bus, 23, 17, 0x0807);

	return 0;
}

typedef enum
{
	eLINK_OFF = 0,
	eLINK_ON,
	eLINK_END
} E_LINK_STATE;
static E_LINK_STATE ip175l_port_state[4]= {0};
static E_LINK_STATE phy_port_state =  0;
extern void gmac_event_notify(GMAC_NOTIFY_EVENT notify_type, void* puf);

static int ip175l_private_proc(struct phy_device *phydev)
{
	int err = 0;
	int i = 0;
	int islink = 0;
	int org_lan_status = 0;
	int org_wan_status = 0;
	int new_lan_status = 0;
	int new_wan_status = 0;

	if(phydev->bus == NULL)
		return 0;

	for(i=0; i<3; i++) {
		if(ip175l_port_state[i] == 1)
			org_lan_status = 1; 

		islink = mdiobus_read(phydev->bus, i, 1);
		islink = (islink&0x4)?1:0;
			
		//rtl8306e_port_phyLinkStatus_get(i,&islink);
		ip175l_port_state[i] = islink;
		if(islink == 1)
			new_lan_status = 1;
	}

	org_wan_status = ip175l_port_state[3];
	islink = mdiobus_read(phydev->bus, 3, 1);
	islink = (islink&0x4)?1:0;

	//rtl8306e_port_phyLinkStatus_get(i,&islink);
	ip175l_port_state[3] = islink;
	if(islink == 1)
		new_wan_status = 1;

	//for(i=0;i<4;i++)
		//printk("port.%d linkstatus=%d. ",i,ip175l_port_state[i]);

	islink = mdiobus_read(phydev->bus, 4, 1);
	islink = (islink&0x4)?1:0;
	//printk("port.4 linkstatus=%d. ",islink);

	//printk("port.5 linkstatus=%d.\n",phydev->link);

	if((org_lan_status == 0)&&(new_lan_status == 1))
		gmac_event_notify(GMAC_ETH_SW_LAN_PLUGIN, NULL);

	if((org_lan_status == 1)&&(new_lan_status == 0))
		gmac_event_notify(GMAC_ETH_SW_LAN_PLUGOUT, NULL);

	if((org_wan_status == 0)&&(new_wan_status == 1))
		gmac_event_notify(GMAC_ETH_SW_WAN_PLUGIN, NULL);

	if((org_wan_status == 1)&&(new_wan_status == 0))
		gmac_event_notify(GMAC_ETH_SW_WAN_PLUGOUT, NULL);

	return err;
}


static int ip101a_g_private_proc(struct phy_device *phydev)
{
	int err = 0;
	int i = 0;
	int islink = 0;
	int org_phy_status = 0;
	int new_phy_status = 0;

	if(phydev->bus == NULL)
		return 0;

	org_phy_status = phy_port_state;
	islink = mdiobus_read(phydev->bus, 1, 1);
	islink = (islink & 0x4)?1:0;

	phy_port_state = islink;
	if(islink == 1)
		new_phy_status = 1;

	
	//printk("port.1 linkstatus=%d. ",phy_port_state);

	//printk("portlinkstatus=%d.\n",phydev->link);

	if((org_phy_status == 0)&&(new_phy_status == 1))
		gmac_event_notify(GMAC_ETH_PHY_PLUGIN, NULL);

	if((org_phy_status == 1)&&(new_phy_status == 0))
		gmac_event_notify(GMAC_ETH_PHY_PLUGOUT, NULL);

	return err;
}


static int ip1xx_reset(struct phy_device *phydev)
{
	int bmcr;

	/* Software Reset PHY */
	bmcr = phy_read(phydev, MII_BMCR);
	if (bmcr < 0)
		return bmcr;
	bmcr |= BMCR_RESET;
	bmcr = phy_write(phydev, MII_BMCR, bmcr);
	if (bmcr < 0)
		return bmcr;

	do {
		bmcr = phy_read(phydev, MII_BMCR);
		if (bmcr < 0)
			return bmcr;
	} while (bmcr & BMCR_RESET);

	return 0;
}

static int ip1001_config_init(struct phy_device *phydev)
{
	int c;

	c = ip1xx_reset(phydev);
	if (c < 0)
		return c;

	/* Enable Auto Power Saving mode */
	c = phy_read(phydev, IP1001_SPEC_CTRL_STATUS_2);
	if (c < 0)
		return c;
	c |= IP1001_APS_ON;
	c = phy_write(phydev, IP1001_SPEC_CTRL_STATUS_2, c);
	if (c < 0)
		return c;

	if (phydev->interface == PHY_INTERFACE_MODE_RGMII) {
		/* Additional delay (2ns) used to adjust RX clock phase
		 * at RGMII interface */
		c = phy_read(phydev, IP10XX_SPEC_CTRL_STATUS);
		if (c < 0)
			return c;

		c |= IP1001_PHASE_SEL_MASK;
		c = phy_write(phydev, IP10XX_SPEC_CTRL_STATUS, c);
		if (c < 0)
			return c;
	}

	return 0;
}

static int ip101a_g_config_init(struct phy_device *phydev)
{
	int c;

	c = ip1xx_reset(phydev);
	if (c < 0)
		return c;

	/* Enable Auto Power Saving mode */
	c = phy_read(phydev, IP10XX_SPEC_CTRL_STATUS);
	c |= IP101A_G_APS_ON;
	phy_write(phydev, IP10XX_SPEC_CTRL_STATUS, c);

	/*802.3.azر*/
	phy_write(phydev, 0xd, 0x0007);
	phy_write(phydev, 0xe, 0x003C);
	phy_write(phydev, 0xd, 0x4007);
	phy_write(phydev, 0xe, 0x0000);

	/*enable interrupt*/
	c = phy_read(phydev, IP10XX_INT_SR);
    c |= IP10XX_INT_SR_INTR_PIN_USED;
    c &= ~(IP10XX_INT_SR_ALL_MASK|IP10XX_INT_SR_LINK_MASK);
	phy_write(phydev, IP10XX_INT_SR, c);

	c = phy_read(phydev, IP10XX_IO_SPEC_CR);
    c |= IP10XX_IO_SPEC_CR_SEL_INTR32;
    phy_write(phydev, IP10XX_IO_SPEC_CR, c);	

    //set auto-neg advertisement capabilities to 10/100 half/full
    c = ADVERTISE_100FULL|ADVERTISE_100HALF|ADVERTISE_10FULL|ADVERTISE_10HALF|ADVERTISE_CSMA;
    phy_write(phydev, MII_ADVERTISE, c);//0x1E1

    c	= phy_read(phydev, MII_BMCR);
    c |= BMCR_ANENABLE | BMCR_ANRESTART;
    phy_write(phydev, MII_BMCR, c);

	return 0;
}


static int ip175c_read_status(struct phy_device *phydev)
{
	if (phydev->addr == 4) /* WAN port */
		genphy_read_status(phydev);
	else
		/* Don't need to read status for switch ports */
		phydev->irq = PHY_IGNORE_INTERRUPT;

	return 0;
}

static int ip175c_config_aneg(struct phy_device *phydev)
{
	if (phydev->addr == 4) /* WAN port */
		genphy_config_aneg(phydev);

	return 0;
}


static int ip175l_read_status(struct phy_device *phydev)
{
	int status;

	status = mdiobus_read(phydev->bus, 21, 1);
	
	if ((status & (0x1<<13)) == 0)
		phydev->link = 0;
	else
		phydev->link = 1;

	phydev->speed = SPEED_100;
	phydev->duplex = DUPLEX_FULL;
	phydev->pause = phydev->asym_pause = 0;

	return 0;
}

static int ip175l_config_aneg(struct phy_device *phydev)
{

	return 0;
}

static int ip101a_g_ack_interrupt(struct phy_device *phydev)
{
	int err = phy_read(phydev, IP101A_G_IRQ_CONF_STATUS);
	if (err < 0)
		return err;

	return 0;
}

static struct phy_driver ip175l_driver = {
	.phy_id		= 0x02430d80,
	.name		= "ICPlus IP175L",
	.phy_id_mask	= 0x0fffffc0,
	.features	= SUPPORTED_100baseT_Full,
	.config_init	= &ip175l_config_init,
	.config_aneg	= &genphy_config_aneg,
	.read_status	= &ip175l_read_status,
	.private_proc	= &ip175l_private_proc,
	.auto_adapt   =  &ip175l_auto_adapt,
	.suspend	= genphy_suspend,
	.resume		= genphy_resume,
	.driver		= { .owner = THIS_MODULE,},
};

static struct phy_driver ip175c_driver = {
	.phy_id		= 0x02430d80,
	.name		= "ICPlus IP175C",
	.phy_id_mask	= 0x0ffffff0,
	.features	= PHY_BASIC_FEATURES,
	.config_init	= &ip175c_config_init,
	.config_aneg	= &ip175c_config_aneg,
	.read_status	= &ip175c_read_status,
	.suspend	= genphy_suspend,
	.resume		= genphy_resume,
	.driver		= { .owner = THIS_MODULE,},
};

static struct phy_driver ip1001_driver = {
	.phy_id		= 0x02430d90,
	.name		= "ICPlus IP1001",
	.phy_id_mask	= 0x0ffffff0,
	.features	= PHY_GBIT_FEATURES | SUPPORTED_Pause |
			  SUPPORTED_Asym_Pause,
	.config_init	= &ip1001_config_init,
	.config_aneg	= &genphy_config_aneg,
	.read_status	= &genphy_read_status,
	.suspend	= genphy_suspend,
	.resume		= genphy_resume,
	.driver		= { .owner = THIS_MODULE,},
};

static struct phy_driver ip101a_g_driver = {
	.phy_id		= 0x02430c54,
	.name		= "ICPlus IP101A/G",
	.phy_id_mask	= 0x0ffffff0,
	.features	= PHY_BASIC_FEATURES | SUPPORTED_Pause |
			  SUPPORTED_Asym_Pause,
	.flags		= PHY_HAS_INTERRUPT,
	.ack_interrupt	= ip101a_g_ack_interrupt,
	.config_init	= &ip101a_g_config_init,
	.config_aneg	= &genphy_config_aneg,
	.read_status	= &genphy_read_status,
	.private_proc	= &ip101a_g_private_proc,
	.suspend	= genphy_suspend,
	.resume		= genphy_resume,
	.driver		= { .owner = THIS_MODULE,},
};

static int __init icplus_init(void)
{
	int ret = 0;

	printk(KERN_INFO "icplus ethernet phy driver register\n");

	ret = phy_driver_register(&ip1001_driver);
	if (ret < 0)
		return -ENODEV;

	ret = phy_driver_register(&ip101a_g_driver);
	if (ret < 0)
		return -ENODEV;

	return phy_driver_register(&ip175l_driver);
}

static void __exit icplus_exit(void)
{
	phy_driver_unregister(&ip1001_driver);
	phy_driver_unregister(&ip101a_g_driver);
	//phy_driver_unregister(&ip175c_driver);
	phy_driver_unregister(&ip175l_driver);
}

subsys_initcall(icplus_init);
module_exit(icplus_exit);

static struct mdio_device_id __maybe_unused icplus_tbl[] = {
	{ 0x02430d80, 0x0ffffff0 },
	{ 0x02430d90, 0x0ffffff0 },
	{ 0x02430c54, 0x0ffffff0 },
	{ }
};

MODULE_DEVICE_TABLE(mdio, icplus_tbl);
