#ifndef _FW_ALARM_H_
#define _FW_ALARM_H_
#include "zcore_type.h"
#include "os_type.h"

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#define MAX_ALARM_NAMELEN   32          //max length of alarm name
#define MAX_ALARM_BUFFLEN   1024        //max length of alarm buff
#define MAX_ALARM_COUNT     45          //max count of alarm
#define INVALID_ALARM_ID    0xFFFFFFFF

#define MASK_SUNDAY		0x1
#define MASK_MONDAY		0x2
#define MASK_TUESDAY	0x4
#define MASK_WEDNESDAY	0x8
#define MASK_THURSDAY	0x10
#define MASK_FRIDAY 	0x20
#define MASK_SATURDAY	0x40

//default mask of workday
#define DEFAULT_WORKDAY_MASK (MASK_MONDAY | MASK_TUESDAY | MASK_WEDNESDAY | MASK_THURSDAY | MASK_FRIDAY)

//alarm frequence
typedef enum _ALARM_FREQ_
{
    AF_ONCE=0,          //alarm once
    AF_EVERYDAY=1,      //alarm everyday
    AF_WORKDAY=2,       //alarm in workday
    AF_WEEKEND=3,       //alarm in weekend
    AF_EVERYWEEK=4,     //alarm on the day of everyweek
}ALARM_FREQ;

//alarm type
typedef enum _ALARM_TYPE_
{
    ALARM_TYPE_NORMAL =     0,
    ALARM_TYPE_POWERON      ,//1
    ALARM_TYPE_POWEROFF     ,//2
}ALARM_TYPE;

//alarm infomation
typedef struct _ALARM_INFO_
{
	time_t	alarm_time;                 //alarm time is a degree of minute accuracy,based on 60
	ZCHAR owner[MAX_ALARM_NAMELEN+1];   //owner of alarm
	ALARM_FREQ alarm_freq;              //alarm frequence
    ALARM_TYPE alarm_type;              //alarm type
	ZVOID *param;                       //point to user data buff, that should be free by user
	ZUINT16 param_len;                  //the lenth of user data buff is no more than MAX_ALARM_BUFFLEN-sizeof(ALARM_INFO)
}ALARM_INFO;


//alarm initial function should be callled once
ZBOOL tp_alarm_init(ZVOID);

//alarm release function
ZBOOL tp_alarm_release(ZVOID);

//The function that add one alarm, return new alarm ID. 
//There is error if return INVALID_ALARM_ID
//return INVALID_ALARM_ID if alarm_info->param_len is more than MAX_ALARM_BUFFLEN - sizeof(ALARM_INFO)
UINT32  tp_alarm_add(CONST ALARM_INFO* alarm_info);

//remove one alarm by ID
//return FALSE if no owner matching
ZBOOL  tp_alarm_remove_byid(CONST ZCHAR* owner, UINT32 id);

//remove alarms whose owner matching
//return count of alarms removed
//return 0 if there is error
UINT32  tp_alarm_remove_byowner(CONST ZCHAR* owner);

//update alarm by ID
//return FALSE if no alarm_info->owner matching
//return FALSE if alarm_info->param_len is more than MAX_ALARM_BUFFLEN - sizeof(ALARM_INFO)
ZBOOL  tp_alarm_update(UINT32 id, ALARM_INFO* alarm_info);

//update RTC alarm
ZBOOL  tp_alarm_rtc_update(ZVOID);

//get one alarm by ID
//return FALSE if no alarm_info->owner matching
//caller must free alarm_info->param when return TRUE
ZBOOL  tp_alarm_get_byid(CONST ZCHAR* owner, UINT32 id, ALARM_INFO* alarm_info);

//The function get next alarm whose owner matching from start position, that return the alarm ID
//There is error if return INVALID_ALARM_ID
//caller must free alarm_info->param when not return INVALID_ALARM_ID
UINT32  tp_alarm_get_next_byowner(CONST ZCHAR* owner, UINT32 start, ALARM_INFO* alarm_info);

//The function get line time of alarm that cause power on
ZBOOL  tp_alarm_get_poweron_time(time_t *line_time);

//The function get next alarm whose time matching from start position, that return the alarm ID
//There is error if return INVALID_ALARM_ID
//caller must free alarm_info->param when not return INVALID_ALARM_ID
UINT32  tp_alarm_get_next_bytime(time_t line_time, UINT32 start, ALARM_INFO* alarm_info);

//get recent time after line_time, according to freq and freq_time
//the time returned is no less than line_time based on 60 if freq is not AF_ONCE
//the time returned is just freq_time if freq is AF_ONCE
//return 0 if there is error
time_t  tp_alarm_get_recenttime_byfreq(CONST ALARM_FREQ freq, time_t freq_time, CONST time_t line_time);

//compare one alarm with another
//return TRUE if equal, else return FALSE
ZBOOL  tp_alarm_is_equal(CONST ALARM_INFO *alarm_info1, CONST ALARM_INFO *alarm_info2);

//set workday mask
ZBOOL  tp_alarm_set_workdaymask(ZUINT8  workday_mask);

//get workday mask
ZUINT8  tp_alarm_get_workdaymask(ZVOID);

//return TRUE if the day is workday, else return FALSE
//weekday is between 0 and 6, 0 is Sunday
ZBOOL  tp_alarm_is_workday(UINT32 weekday);

#ifdef __cplusplus
}
#endif  /* __cplusplus */

#endif// _FW_ALARM_H_
