/**
 * @file drvs_i2s.h 
 * @brief Public APIs of i2s drivers
 *
 * Copyright (C) 2017 Sanechips Technology Co., Ltd.
 * @author  Xinqiang Xu <xu.xinqiang@sanechips.com.cn>
 * @ingroup si_cp_drv_id
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. 
 *
 */

#ifndef _HAL_I2S_H
#define _HAL_I2S_H

/*******************************************************************************
 *                           Include header files                              *
 ******************************************************************************/
#include "drvs_general.h"
#include "drvs_io_voice.h"

/*******************************************************************************
 *                             Macro definitions                               *
 ******************************************************************************/

/*******************************************************************************
 *                             Type definitions                                *
 ******************************************************************************/
/* i2s index */
typedef enum {
    I2S_1 = 0,
    I2S_2 = 1,

    MAX_I2S_ID
} T_ZDrvI2S_Id;

typedef enum {
    I2S_TX = 0,
    I2S_RX = 1
} T_ZDrvI2S_TransDirection;

typedef enum {
    I2S_NORMAL_MODE,
    I2S_LOOPBACK_MODE,

    MAX_TEST_MODE
} T_ZDrvI2S_TestMode;

typedef enum {
    PCM_NORMAL_MODE,
    PCM_TEAK_MODE,

    MAX_PCM_MODE
} T_ZDrvI2S_PcmMode;

typedef enum {
	PCM_LSB_FIRST,
    PCM_MSB_FIRST,

    MAX_PCM_FIRSTBIT
} T_ZDrvI2S_PcmFirstBitSel;

typedef enum {
    PCM_1TIME_SLOT = 0x00,
    PCM_2TIME_SLOT = 0x01,
    PCM_4TIME_SLOT = 0x03,
    PCM_8TIME_SLOT = 0x07,
    PCM_16TIME_SLOT = 0x0f,

    MAX_PCM_SLOTNUM
} T_ZDrvI2S_PcmSlotNum;

/* i2s operation mode */
typedef enum {
    I2S_DATA_16BIT = 0x0f,
    I2S_DATA_17BIT = 0x10,
    I2S_DATA_18BIT = 0x11,
    I2S_DATA_19BIT = 0x12,
    I2S_DATA_20BIT = 0x13,
    I2S_DATA_21BIT = 0x14,
    I2S_DATA_22BIT = 0x15,
    I2S_DATA_23BIT = 0x16,
    I2S_DATA_24BIT = 0x17,
    I2S_DATA_25BIT = 0x18,
    I2S_DATA_26BIT = 0x19,
    I2S_DATA_27BIT = 0x1a,
    I2S_DATA_28BIT = 0x1b,
    I2S_DATA_29BIT = 0x1c,
    I2S_DATA_30BIT = 0x1d,
    I2S_DATA_31BIT = 0x1e,
    I2S_DATA_32BIT = 0x1f,
    
    MAX_I2S_DATA_FMT
} T_ZDrvI2S_DataFmt;

typedef enum {
    I2S_TS_16CYCLE = 0x0f,
    I2S_TS_17CYCLE = 0x10,
    I2S_TS_18CYCLE = 0x11,
    I2S_TS_19CYCLE = 0x12,
    I2S_TS_20CYCLE = 0x13,
    I2S_TS_21CYCLE = 0x14,
    I2S_TS_22CYCLE = 0x15,
    I2S_TS_23CYCLE = 0x16,
    I2S_TS_24CYCLE = 0x17,
    I2S_TS_25CYCLE = 0x18,
    I2S_TS_26CYCLE = 0x19,
    I2S_TS_27CYCLE = 0x1a,
    I2S_TS_28CYCLE = 0x1b,
    I2S_TS_29CYCLE = 0x1c,
    I2S_TS_30CYCLE = 0x1d,
    I2S_TS_31CYCLE = 0x1e,
    I2S_TS_32CYCLE = 0x1f,
    
    MAX_I2S_TS_WIDTH
} T_ZDrvI2S_TsWidth;

#if 1//defined _USE_EVB2963
/*i2s track select*/
typedef enum {
    DOUBLE_TRACK = 0x00,
    LEFT_TRACK   = 0x01,
    RIGHT_TRACK  = 0x02,

    MAX_I2S_TRACK_FMT
} T_ZDrvI2S_TrackFmt;

/*i2s clock source in master mode*/
typedef enum {
    PCLK_DIVIDE      = 0x00,
    I2S_SCLK_DIVIDE  = 0x01,
    EQUAL_TO_SCLK    = 0x02,

    MAX_I2S_CLK_SOURCE
} T_ZDrvI2S_ClkSource;

/*i2s clock mode*/
typedef enum {
    TRANS_POSITIVE_NEGATIVE = 0,
    TRANS_NEGATIVE_POSITIVE ,

    MAX_EDGE_SELECT
} T_ZDrvI2S_ClkEdgeSel;

#endif

typedef VOID (*T_ZDrvI2S_AUD_PLAY_CB)(VOID);

/* i2s configuration */

/*i2s clock mode*/
typedef enum {
    I2S_TIME_MODE = 0,
    PCM_TIME_MODE ,

    MAX_I2S_TIME_MODE
} T_ZDrvI2S_ClkMode;

typedef enum {
    PCM_FSYNC_LENGTH_1CLK,
    PCM_FSYNC_LENGTH_2CLK,
    PCM_FSYNC_LENGTH_3CLK,
    PCM_FSYNC_LENGTH_4CLK,
    PCM_FSYNC_LENGTH_5CLK,
    PCM_FSYNC_LENGTH_6CLK,
    PCM_FSYNC_LENGTH_7CLK,
    PCM_FSYNC_LENGTH_8CLK,

    MAX_PCM_FSYNC
} T_ZDrvI2S_PcmFsyncLen;

typedef enum {
    FIRST_DATA_ALIGN_TO_FIRST_CYCLE,
    FIRST_DATA_ALIGN_TO_SECOND_CYCLE,

    MAX_7510_DATA_ALIGN_MODE
} T_ZDrvI2S_DataAlignMode;

typedef enum {
    TIMING_I2S_MONO_RIGHT = 0,
    TIMING_I2S_MONO_LEFT,
    TIMING_I2S_ST_2CHN_1LANE,
    TIMING_I2S_ST_4CHN_2LANE,
    TIMING_I2S_ST_6CHN_3LANE,
    TIMING_I2S_ST_8CHN_4LANE,
    TIMING_TDM_1CHN_1LANE_1TS,
    TIMING_TDM_2CHN_1LANE_2TS,

    MAX_TMING_SELECT
} T_ZDrvI2S_TimingType;

/**
 * @brief describe the i2s device timing config info
 * @param tTimingTp			tTimingTp
 * @param tTimingSel		tTimingSel		 
 * @param tChnNum 			tChnNum		 	
 * @param tLaneNum			tLaneNum		
 * @param tTsCfg			tTsCfg		
 */
typedef struct {
	T_ZDrvI2S_TimingType tTimingTp;       
	T_ZDrvI2S_ClkMode tTimingSel;
	UINT8 tChnNum;
	UINT8 tLaneNum;
	UINT8 tTsCfg;

} T_ZDrvI2S_TimingCfg;

typedef enum {
    PCM_LONG_FSYNC,
    PCM_SHORT_FSYNC,
    MAX_PCM_LS_FSYNC
} T_ZDrvI2S_PcmFsync;

typedef enum {
    TDM_TEAK_EXTRA_CYCLE,
    TDM_NO_EXTRA_CYCLE,
    MAX_TDM_EXTRA_CYCLE
} T_ZDrvI2S_TdmExtCycle;

typedef enum {
    DATA_ALIGN_STD_I2S,
    DATA_ALIGN_MSB_JUSTIF,
    DATA_ALIGN_LSB_JUSTIF,
    DATA_ALIGN_NORMAL,
    MAX_DATA_ALIGN_MODE
} T_ZDrvI2S_TransMode;

typedef enum {
    REF_CLK26M,
    REF_CLK104M,
    REF_CLK122M88,

    REF_CLK_MAX
} T_I2s_RefClkForI2s;

/**
 * @brief describe the i2s device config info
 * @param bMaster		 	TRUE: master mode; FALSE: slave mode
 * @param tDataFmt			T_ZDrvI2S_DataFmt		 
 * @param tTsWidth 			T_ZDrvI2S_TsWidth		 	
 * @param sample_rate 		sample rate of auido file
 * @param tClkMode	 		the clk mode of i2s	 	
 * @param tTansmit_edge		the clk edge select		
 * @param tTrackFmt 		tTrackFmt
 * @param tTestMode	    	tTestMode
 * @param tDataAlignMode	tDataAlignMode
 * @param tPcmMode;			tPcmMode
 * @param tPcmFirstBit		tPcmFirstBit
 * @param tPcmFsynLen 		tPcmFsynLen
 * @param tPcmSlotNum 		tPcmSlotNum
 * @param tTimingType 		tTimingType		 	
 * @param tPcmFsync			tPcmFsync				 
 * @param tTdmExtCycle	  	only exit in the 7520		 
 * @param tTransMode 		tTransMode		
 * @param refclk			refclk
 */
typedef struct {
	BOOL bMaster;       
	T_ZDrvI2S_DataFmt tDataFmt;      
	T_ZDrvI2S_TsWidth tTsWidth;	
	UINT32 sample_rate;    
	T_ZDrvI2S_ClkMode tClkMode;       
	T_ZDrvI2S_ClkEdgeSel tTansmit_edge; 
	T_ZDrvI2S_TrackFmt tTrackFmt;
	T_ZDrvI2S_TestMode tTestMode;
	T_ZDrvI2S_DataAlignMode tDataAlignMode;
	T_ZDrvI2S_PcmMode tPcmMode;
	T_ZDrvI2S_PcmFirstBitSel tPcmFirstBit;
	T_ZDrvI2S_PcmFsyncLen tPcmFsynLen;
	T_ZDrvI2S_PcmSlotNum tPcmSlotNum;

	T_ZDrvI2S_TimingType tTimingType;
	T_ZDrvI2S_PcmFsync tPcmFsync;
	T_ZDrvI2S_TdmExtCycle tTdmExtCycle;
	T_ZDrvI2S_TransMode tTransMode;
	T_I2s_RefClkForI2s refclk;

} T_ZDrvI2S_Cfg;

typedef enum {
    I2S_IDLE,
    I2S_OPEN,
    I2S_PLAY_INUSE,
    I2S_RECORD_INUSE,
    I2S_BOTH_INUSE,

    MAX_I2S_STATUS
}T_I2s_Status;

/**
 * @brief describe the codec agc Parameter 
 * @param channel		audio channel
 * @param buffersize	buffer size
 * @param p_cb 			the callback of audio play	 	
 */
typedef struct {
	T_ZDrvAudio_Channel channel;
	UINT32 buffersize;
	T_ZDrvI2S_AUD_PLAY_CB p_cb;
} T_ZDrvI2s_Params;

typedef enum {
    TRANS_CPU_MODE,
    TRANS_INT_MODE,
    TRANS_DMA_MODE,

    MAX_TRANS_FUNC
}T_ZDrvI2s_TransMode;

/*******************************************************************************
 *                       Global variable declarations                          *
 ******************************************************************************/


/*******************************************************************************
 *                       Global function declarations                          *
 ******************************************************************************/

/**
* @brief	open i2s device.
*
* @param	tId	 i2s selete struct
* @param	i2sTransMode	 T_ZDrvI2s_TransMode
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Open(T_ZDrvI2S_Id tId, T_ZDrvI2s_TransMode i2sTransMode);

/**
* @brief	reset i2s device.
*
* @param	tId	 i2s select i2s0 or i2s1
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Reset(T_ZDrvI2S_Id tId);

/**
* @brief	close i2s device.
*
* @param	tId	 i2s selete struct
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Close(T_ZDrvI2S_Id tId);

/**
* @brief 	i2s write stop ,use to stop playing or recording.
*
* @param	tId	 i2s index select i2s0 or i2s1
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Write_Stop(T_ZDrvI2S_Id tId);

/**
* @brief i2s read stop ,use to stop playing or recording.
*
* @param	tId	 i2s select i2s0 or i2s1
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Read_Stop(T_ZDrvI2S_Id tId);

/**
* @brief	This function is used to read data by i2s.
*
* @param	tId	 i2s select i2s0 or i2s1
* @param	pBuf	 one buffer is how many bytes
* @param	uiLen	 buffer length
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Read(T_ZDrvI2S_Id tId, UINT8 **pBuf, UINT32 *uiLen);

/**
* @brief	i2s trans data to fifo from ram while playing.
*
* @param	tId	 i2s selete struct
* @param	pBuf	 one buffer is how many bytes
* @param	uiLen	 buffer length
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Write(T_ZDrvI2S_Id tId, const UINT8 *pBuf, UINT32 uiLen);

/**
* @brief	i2s write start .
*
* @param	tId	 i2s selete struct
* @param	params	 pointer to struct T_ZDrvI2s_Params
* @param	ptCfg	 pointer to struct T_ZDrvI2S_Cfg
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Write_Start(T_ZDrvI2S_Id tId, T_ZDrvI2s_Params *params, T_ZDrvI2S_Cfg *ptCfg);

/**
* @brief	i2s read start.
*
* @param	tId	 i2s selete struct
* @param	params	 pointer to T_ZDrvI2s_Params
* @param	ptCfg	 pointer to T_ZDrvI2S_Cfg
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Read_Start(T_ZDrvI2S_Id tId,  T_ZDrvI2s_Params *params, T_ZDrvI2S_Cfg *ptCfg);

/**
* @brief	start to trans data,use in playing.
*
* @param	tId	 i2s selete struct
* @param	params	 T_ZDrvI2s_Params
* @param	ptCfg	 config param
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Vousb_Start(T_ZDrvI2S_Id tId, T_ZDrvI2s_Params *params, T_ZDrvI2S_Cfg *ptCfg);

/**
* @brief  	i2s get buffer ,get buffer to write playing data.
*
* @param	tId	 i2s select i2s0 or i2s1
* @param	pBuf	 playing data buffer
* @param	uiLen	 buffer length
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_GetBuf(T_ZDrvI2S_Id tId, UINT8 **pBuf, UINT32 *uiLen);

/**
* @brief	i2s free buffer ,get buffer to write playing data.
*
* @param	tId  i2s select i2s0 or i2s1
* @param	pBuf	 data buffer
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_GetRemained(T_ZDrvI2S_Id tId, UINT32 *len);

/**
* @brief 	i2s free buffer ,get buffer to write playing data.
*
* @param	tId  i2s select i2s0 or i2s1
* @param	pBuf	 data buffer
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_FreeBuf(T_ZDrvI2S_Id tId, UINT8 *pBuf);

/**
* @brief	pause while playing.
*
* @param	tId	 i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
VOID zDrvI2S_Pause(T_ZDrvI2S_Id tId);

/**
* @brief	resume playing if pause.
*
* @param	tId	 i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
VOID zDrvI2S_Resume(T_ZDrvI2S_Id tId);

/**
* @brief	Release the rx semaphore before stop.
*
* @param	tId			i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
VOID zDrvI2s_RxRlsSemaBeforeStop(T_ZDrvI2S_Id tId);

/**
* @brief	Release the tx semaphore before stop.
*
* @param	tId			i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
VOID zDrvI2s_TxRlsSemaBeforeStop(T_ZDrvI2S_Id tId);

/**
* @brief	get i2s status.
*
* @param	tId	 i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
T_I2s_Status zDrvI2s_GetI2sStatus(T_ZDrvI2S_Id tId);

#endif    /* #ifndef _HAL_I2S_H */

