/* ==========================================================================
 * $File: //dwh/usb_iip/dev/software/otg/linux/drivers/dwc_otg_driver.c $
 * $Revision: #92 $
 * $Date: 2012/08/10 $
 * $Change: 2047372 $
 *
 * Synopsys HS OTG Linux Software Driver and documentation (hereinafter,
 * "Software") is an Unsupported proprietary work of Synopsys, Inc. unless
 * otherwise expressly agreed to in writing between Synopsys and you.
 *
 * The Software IS NOT an item of Licensed Software or Licensed Product under
 * any End User Software License Agreement or Agreement for Licensed Product
 * with Synopsys or any supplement thereto. You are permitted to use and
 * redistribute this Software in source and binary forms, with or without
 * modification, provided that redistributions of source code must retain this
 * notice. You may not view, use, disclose, copy or distribute this file or
 * any information contained herein except pursuant to this license grant from
 * Synopsys. If you do not agree with this notice, including the disclaimer
 * below, then you are not authorized to use the Software.
 *
 * THIS SOFTWARE IS BEING DISTRIBUTED BY SYNOPSYS SOLELY ON AN "AS IS" BASIS
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE HEREBY DISCLAIMED. IN NO EVENT SHALL SYNOPSYS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 * ========================================================================== */

/** @file
 * The dwc_otg_driver module provides the initialization and cleanup entry
 * points for the DWC_otg driver. This module will be dynamically installed
 * after Linux is booted using the insmod command. When the module is
 * installed, the dwc_otg_driver_init function is called. When the module is
 * removed (using rmmod), the dwc_otg_driver_cleanup function is called.
 *
 * This module also defines a data structure for the dwc_otg_driver, which is
 * used in conjunction with the standard ARM lm_device structure. These
 * structures allow the OTG driver to comply with the standard Linux driver
 * model in which devices and drivers are registered with a bus driver. This
 * has the benefit that Linux can expose attributes of the driver and device
 * in its special sysfs file system. Users can then read or write files in
 * this file system to perform diagnostics on the driver components or the
 * device.
 */

#include "dwc_otg_os_dep.h"
#include "dwc_os.h"
#include "dwc_otg_dbg.h"
#include "dwc_otg_driver.h"
#include "dwc_otg_attr.h"
#include "dwc_otg_cil.h"
#include "dwc_otg_core_if.h"
#include "dwc_otg_pcd_if.h"
#include "dwc_otg_hcd_if.h"
#include "dwc_otg_pcd.h"
#include <mach/zx29_usb.h>
#include <linux/android_notify.h>
#include <mach/highspeed_debug.h>
#include <mach/iomap.h>
#include <linux/mfd/zx234290.h>
#include <linux/clockchips.h>
#include <linux/clk.h>
#include <mach/spinlock.h>

extern int zDrvPmic_SetNormal_Onoff(T_ZDrvPmic_Regulator regulator, T_ZDrvPmic_Enable enable);
extern int zx234290_setusb_v0v9_sleepmode(T_ZDrvPmic_SlpMode mode);
extern int zx234290_setusb_v3v3_sleepmode(T_ZDrvPmic_SlpMode mode);
extern int clk_is_enabled(struct clk *clk);

typedef enum    
{
    /*оƬ˯ڼ䣬ֻVDD_USB_CTRL1VIO_VDDR_USBVDD_IRAM6VIOVPLL_09
	VPLL_18VSIM0VSIM1VSSBUFVSDVDD_WKUPЩܶϵĵԴ̶ȵĽ©硣*/
    GSM_RAM_PWR = 0,
    GSM_DSP_PWR = 1,
    EDCP_PWR = 4,
    USB_CTRL_PWR = 8,
    USB_HSIC_PWR = 9,
	
    PARTITION_ALL = 10
    
} T_ZDrvPow_Partition;

typedef enum
{
    POW_MDL_1 = 0,
    POW_MDL_ALL = 1
} T_ZDrvPow_PowerModule;	/*PMU*/

typedef enum
{
    POW_ENABLE = 0,
    POW_DISABLE = 1,

    POW_ENABLE_ALL
} T_ZDrvPow_PowerEnable;

struct dwc_otg_peripheral_state
{
	u8 clk_state;
	u8 power_state;
};

static struct dwc_otg_peripheral_state dwc_otg_hal_state = {0};
/*static */struct dwc_otg_wakelock_state dwc_otg_wake_lock;
static int pclk_enable_status;
static int wclk_enable_status;

void dwc_chg_udelay(uint32_t usecs)
{
    udelay(usecs);
}
EXPORT_SYMBOL_GPL(dwc_chg_udelay);

void dwc_chg_mdelay(uint32_t usecs)
{
    mdelay(usecs);
}
EXPORT_SYMBOL_GPL(dwc_chg_mdelay);

int pow_partition_powerswitch(T_ZDrvPow_Partition part, T_ZDrvPow_PowerEnable ena)
{
    unsigned int value;
    if (POW_ENABLE == ena)
    {
        reg_spin_lock();
        value = ioread32(ZX29_POWER_DOMAIN_POWERON);
        value |= 0x1 << part;
        iowrite32(value,ZX29_POWER_DOMAIN_POWERON);
	 reg_spin_unlock();
        dwc_chg_udelay(10);

	 reg_spin_lock();
        value = ioread32(ZX29_POWER_DOMAIN_RST);
        value &= ~(0x1 << part );
        iowrite32(value,ZX29_POWER_DOMAIN_RST);
	 reg_spin_unlock();
        dwc_chg_udelay(10);

	 reg_spin_lock();
        value = ioread32(ZX29_POWER_DOMAIN_ISO);
        value &= ~(0x1 << part);
        iowrite32(value,ZX29_POWER_DOMAIN_ISO);
	 reg_spin_unlock();
		dwc_chg_udelay(10);
    }
    else
    {  
        reg_spin_lock();
        value = ioread32(ZX29_POWER_DOMAIN_ISO);
        value  |= 0x1 << part;
        iowrite32(value,ZX29_POWER_DOMAIN_ISO);
	 reg_spin_unlock();
		dwc_chg_udelay(10);

	 reg_spin_lock();
        value = ioread32(ZX29_POWER_DOMAIN_RST);
        value |= 0x1 << part;
        iowrite32(value,ZX29_POWER_DOMAIN_RST);
	 reg_spin_unlock();
		dwc_chg_udelay(10);

	 reg_spin_lock();
        value = ioread32(ZX29_POWER_DOMAIN_POWERON);
        value &= ~(0x1 << part);
        iowrite32(value,ZX29_POWER_DOMAIN_POWERON);
	 reg_spin_unlock();
		dwc_chg_udelay(10);
    }
    return 0;
}
EXPORT_SYMBOL_GPL(pow_partition_powerswitch);


int dwc_otg_open_power(void)
{
	int nRet = -1;

	if(dwc_otg_hal_state.power_state){
		USBHAL_DBG("**ALREADY OPERN POWER**");
		return 0;
	}
	dwc_otg_hal_state.power_state = 1;
#if 1
	nRet = zDrvPmic_SetNormal_Onoff(VUSB_0V9,PM_ENABLE);
	nRet += zDrvPmic_SetNormal_Onoff(VUSB_3V3,PM_ENABLE);
#endif
	nRet += pow_partition_powerswitch(USB_CTRL_PWR, POW_ENABLE);
#if 0
	nRet += Zx234290_SetLDO1Sleep_Onoff(LDOA_SLP_ECO_SLPV);
	nRet += Zx234290_SetLDO5Sleep_Onoff(LDOA_SLP_ECO_VOLT);
#endif
	zx234290_setusb_v0v9_sleepmode(PM_SLPMODE_ECO_SLPV);
    zx234290_setusb_v3v3_sleepmode(PM_SLPMODE_ECO_SLPV);
	USBHAL_DBG("**OPERN POWER RET:%d**", nRet);
	return nRet;
}

int dwc_otg_close_power(void)
{
	int nRet = -1;

	if(!dwc_otg_hal_state.power_state){
		USBHAL_DBG("**ALREADY CLOSE POWER**");
		return 0;
	}
	dwc_otg_hal_state.power_state = 0;
	
	nRet = pow_partition_powerswitch(USB_CTRL_PWR, POW_DISABLE);

	nRet += zDrvPmic_SetNormal_Onoff(VUSB_0V9,PM_DISABLE);
	nRet += zDrvPmic_SetNormal_Onoff(VUSB_3V3,PM_DISABLE);
#if 0
	nRet += Zx234290_SetLDO1Sleep_Onoff(LDOA_SLP_OFF);
	nRet += Zx234290_SetLDO5Sleep_Onoff(LDOA_SLP_OFF);
#endif
	zx234290_setusb_v0v9_sleepmode(PM_SLPMODE_OFF);
    zx234290_setusb_v3v3_sleepmode(PM_SLPMODE_OFF);
	USBHAL_DBG("**CLOSE POWER RET:%d**", nRet);
	return nRet;
}

int dwc_otg_is_clk_enable(void)
{
		struct clk* wclk;
		struct clk* pclk;
#ifdef CONFIG_ARCH_ZX297520V2
		pclk = clk_get_sys("zx29_usb.0", "12m_clk");
#else				
		pclk = clk_get_sys("zx29_usb.0", "24m_clk");
#endif
		if (IS_ERR(pclk))
			panic("failed to get usb 24m_clk wclk.");

		wclk = clk_get_sys("zx29_usb.0", "ahb_clk");
		if (IS_ERR(wclk))
			panic("failed to get usb ahb_clk wclk.");

		pclk_enable_status = clk_is_enabled(pclk);
		wclk_enable_status = clk_is_enabled(wclk);

		return	(pclk_enable_status && wclk_enable_status);
}

void dwc_otg_clk_enable(int isOn)
{
	struct clk* wclk;
	struct clk* pclk;
#ifdef CONFIG_ARCH_ZX297520V2
	pclk = clk_get_sys("zx29_usb.0", "12m_clk");
#else
	pclk = clk_get_sys("zx29_usb.0", "24m_clk");
#endif
	if (IS_ERR(pclk))
		panic("failed to get usb 24m_clk wclk.");

	wclk = clk_get_sys("zx29_usb.0", "ahb_clk");
	if (IS_ERR(wclk))
		panic("failed to get usb ahb_clk wclk.");
	
	if(isOn){
		if(dwc_otg_hal_state.clk_state){
			USBHAL_DBG("**ALREADY ENABLE CLOCK**");
			return;
		}
		dwc_otg_hal_state.clk_state = 1;
		clk_enable(pclk);
//		dwc_chg_udelay(20);
		clk_enable(wclk);
		USBHAL_DBG("**ENABLE CLOCK**");
	}else{
		if(!dwc_otg_hal_state.clk_state){
			USBHAL_DBG("**ALREADY DISABLE CLOCK**");
			return;
		}
		dwc_otg_hal_state.clk_state = 0;
//		dwc_chg_udelay(5);
		clk_disable(pclk);
		clk_disable(wclk);
		USBHAL_DBG("**DISABLE CLOCK**");
	}
}
EXPORT_SYMBOL_GPL(dwc_otg_clk_enable);

void dwc_otg_wakelock(int lock_flag,int phase)
{
	if(lock_flag){
		if(phase == 0){
			wake_lock_timeout(&dwc_otg_wake_lock.wake_lock,msecs_to_jiffies(3*1000));
			USBHAL_DBG("**dwc otg wake unlock timeout of 2 seconds**");
			dwc_otg_wake_lock.wakelock_state = 0;
		}else{
		if(dwc_otg_wake_lock.wakelock_state){
			USBHAL_DBG("**ALREADY wake lock**");
			return;
		}
		dwc_otg_wake_lock.wakelock_state = 1;
        wake_lock(&dwc_otg_wake_lock.wake_lock);
		USBHAL_DBG("**dwc otg wake lock**");
			}
	}else{
		if(phase == 0){
			wake_lock_timeout(&dwc_otg_wake_lock.wake_lock,msecs_to_jiffies(2*1000));
			USBHAL_DBG("**dwc otg wake unlock timeout of 2 seconds**");
			dwc_otg_wake_lock.wakelock_state = 0;
		}else{
			if(!dwc_otg_wake_lock.wakelock_state){
				USBHAL_DBG("**ALREADY wake unlock**");
				return;
			}
			dwc_otg_wake_lock.wakelock_state = 0;
        		wake_unlock(&dwc_otg_wake_lock.wake_lock);
			USBHAL_DBG("**dwc otg wake unlock**");
		}
	}
}
EXPORT_SYMBOL_GPL(dwc_otg_wakelock);

void dwc_otg_hal_init(void)
{
	unsigned int value;

	//open power
	dwc_otg_open_power();

	//usb power domain reset
	pow_partition_powerswitch(USB_CTRL_PWR, POW_DISABLE);
	dwc_chg_udelay(2000);
	pow_partition_powerswitch(USB_CTRL_PWR, POW_ENABLE);
    dwc_chg_udelay(2000);

	//usb  clock enable
	dwc_otg_clk_enable(1);
	
	// usb  ahb bus reset
	reg_spin_lock();
	value = ioread32(ZX29_SOC_USB_RSTEN);
	value  &= ~(1<<5);
	iowrite32(value,ZX29_SOC_USB_RSTEN);
	reg_spin_unlock();

	reg_spin_lock();
	value = ioread32(ZX29_SOC_USB_RSTEN);
	value  |= (1<<5);
	iowrite32(value,ZX29_SOC_USB_RSTEN);
	reg_spin_unlock();

	// usb  work reset  ߺ
	reg_spin_lock();
	value = ioread32(ZX29_SOC_USB_RSTEN);
	value  &= ~(1<<4);
	iowrite32(value,ZX29_SOC_USB_RSTEN);
	reg_spin_unlock();

	reg_spin_lock();
	value = ioread32(ZX29_SOC_USB_RSTEN);
	value  |= (1<<4);
	iowrite32(value,ZX29_SOC_USB_RSTEN);
	reg_spin_unlock();

	//USB PHY COMMONONN λ
	reg_spin_lock();
	value = ioread32(ZX29_USB20_MOD_CFG0);
	value  |= (1<<13);
	iowrite32(value,ZX29_USB20_MOD_CFG0);
	reg_spin_unlock();

	//release usb  phy reset 
	reg_spin_lock();
	value = ioread32(ZX29_SOC_USB_RSTEN);
	value  &= ~(1<<3);
	iowrite32(value,ZX29_SOC_USB_RSTEN);
	reg_spin_unlock();

	reg_spin_lock();
	value = ioread32(ZX29_SOC_USB_RSTEN);
	value  |= (1<<3);
	iowrite32(value,ZX29_SOC_USB_RSTEN);
	reg_spin_unlock();
	dwc_chg_udelay(10);

	value = 0;
	while((ioread32(ZX29_SOC_USB_RESET_STATE)&0x02) == 0)
	{
		value++;
		dwc_chg_udelay(100);
		if(value > 500/*500000*/) {
			USBHAL_DBG("WARNING USB_RESET_STATE, value: %d", value);
			usb_printk("%s, 0x0010C05C:BIT4, value:%d\n", __func__, value);
			break;
		}
	}
}
//EXPORT_SYMBOL_GPL(dwc_otg_hal_init);


void dwc_otg_hal_exit(void)
{
	//usb  clock disable
	dwc_otg_clk_enable(0);
	dwc_otg_close_power();
}
//EXPORT_SYMBOL_GPL(dwc_otg_hal_exit);


// for ramdump
int zDrvPmic_SetNormal_Onoff_PSM(T_ZDrvPmic_Regulator regulator, T_ZDrvPmic_Enable enable);
int dwc_otg_open_power_psm(void)
{
	int nRet = -1;

#if 1
	nRet = zDrvPmic_SetNormal_Onoff_PSM(VUSB_0V9,PM_ENABLE);
	nRet += zDrvPmic_SetNormal_Onoff_PSM(VUSB_3V3,PM_ENABLE);
#endif
	//nRet += pow_partition_powerswitch(USB_CTRL_PWR, POW_ENABLE);
	//usb power domain reset
	
	//pow_partition_powerswitch(USB_CTRL_PWR, POW_DISABLE);
	pow_partition_powerswitch(USB_CTRL_PWR, POW_ENABLE);
	
	USBHAL_DBG("**OPERN POWER RET:%d**", nRet);
	return nRet;
}

int dwc_otg_close_power_psm(void)
{
	int nRet = -1;
	
	nRet = pow_partition_powerswitch(USB_CTRL_PWR, POW_DISABLE);

	nRet += zDrvPmic_SetNormal_Onoff_PSM(VUSB_0V9,PM_DISABLE);
	nRet += zDrvPmic_SetNormal_Onoff_PSM(VUSB_3V3,PM_DISABLE);
#if 0
	nRet += Zx234290_SetLDO1Sleep_Onoff(LDOA_SLP_OFF);
	nRet += Zx234290_SetLDO5Sleep_Onoff(LDOA_SLP_OFF);
#endif
	usb_printk("**CLOSE POWER RET:%d**", nRet);
	return nRet;
}


void dwc_otg_clk_enable_psm(int isOn)
{	
	unsigned int value;
	if(isOn){		
		reg_spin_lock();
		value = ioread32(ZX29_SOC_USB_CLK);
		value  |= 1<<3;
		iowrite32(value,ZX29_SOC_USB_CLK);	
		reg_spin_unlock();
		mndelay(20);

		reg_spin_lock();
		value = ioread32(ZX29_SOC_USB_CLK);
		value  |= 1<<4;
		iowrite32(value,ZX29_SOC_USB_CLK);
		reg_spin_unlock();
		USBHAL_DBG("**ENABLE CLOCK**");
	}else{
		mndelay(5);
		reg_spin_lock();
		value = ioread32(ZX29_SOC_USB_CLK);
		value  &= ~(1<<4);
		iowrite32(value,ZX29_SOC_USB_CLK);
		reg_spin_unlock();

		reg_spin_lock();
		value = ioread32(ZX29_SOC_USB_CLK);
		value  &= ~(1<<3);
		iowrite32(value,ZX29_SOC_USB_CLK);	
		reg_spin_unlock();
		USBHAL_DBG("**DISABLE CLOCK**");
	}
}
void dwc_otg_hal_init_psm(void)
{
	unsigned int value;

	usb_printk("%s\n", __func__);
	
	//open power
	dwc_otg_open_power_psm();

	//usb power domain reset
	pow_partition_powerswitch(USB_CTRL_PWR, POW_DISABLE);
	mndelay(2000);
	pow_partition_powerswitch(USB_CTRL_PWR, POW_ENABLE);
    mndelay(2000);

	//usb  clock enable
	dwc_otg_clk_enable_psm(1);
	
	// usb  ahb bus reset
	reg_spin_lock();
	value = ioread32(ZX29_SOC_USB_RSTEN);
	value  &= ~(1<<5);
	iowrite32(value,ZX29_SOC_USB_RSTEN);
	reg_spin_unlock();

	reg_spin_lock();
	value = ioread32(ZX29_SOC_USB_RSTEN);
	value  |= (1<<5);
	iowrite32(value,ZX29_SOC_USB_RSTEN);
	reg_spin_unlock();

	// usb  work reset  ߺ
	reg_spin_lock();
	value = ioread32(ZX29_SOC_USB_RSTEN);
	value  &= ~(1<<4);
	iowrite32(value,ZX29_SOC_USB_RSTEN);
	reg_spin_unlock();

	reg_spin_lock();
	value = ioread32(ZX29_SOC_USB_RSTEN);
	value  |= (1<<4);
	iowrite32(value,ZX29_SOC_USB_RSTEN);
	reg_spin_unlock();

	//USB PHY COMMONONN λ
	reg_spin_lock();
	value = ioread32(ZX29_USB20_MOD_CFG0);
	value  |= (1<<13);
	iowrite32(value,ZX29_USB20_MOD_CFG0);
	reg_spin_unlock();

	//release usb  phy reset 
	reg_spin_lock();
	value = ioread32(ZX29_SOC_USB_RSTEN);
	value  &= ~(1<<3);
	iowrite32(value,ZX29_SOC_USB_RSTEN);
	reg_spin_unlock();

	reg_spin_lock();
	value = ioread32(ZX29_SOC_USB_RSTEN);
	value  |= (1<<3);
	iowrite32(value,ZX29_SOC_USB_RSTEN);
	reg_spin_unlock();
	dwc_chg_udelay(10);

	value = 0;
	while((ioread32(ZX29_SOC_USB_RESET_STATE)&0x02) == 0)
	{
		value++;
		dwc_chg_udelay(100);
		if(value > 500000) {
			USBHAL_DBG("WARNING USB_RESET_STATE, value: %d", value);
			usb_printk("%s, 0x0010C05C:BIT4, value:%d\n", __func__, value);
			break;
		}
	}
}

void dwc_otg_hal_exit_psm(void)
{
	usb_printk("%s\n", __func__);	
	//usb  clock disable
	dwc_otg_clk_enable_psm(0);
	dwc_otg_close_power_psm();
}
