/*
 * f_diag.c - USB diag function driver
 *
 * Copyright (C) 2019 by ZTE Corporation
 *
 * This software is distributed under the terms of the GNU General
 * Public License ("GPL") as published by the Free Software Foundation,
 * either version 2 of that License or (at your option) any later version.
 */

#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/device.h>

#include "u_diag.h"
#include "gadget_chips.h"


/*
 * This function packages a simple "diag" port with no real
 * control mechanisms, just raw data transfer over two bulk endpoints.
 *
 * Because it's not standardized, this isn't as interoperable as the
 * CDC ACM driver.  However, for many purposes it's just as functional
 * if you can arrange appropriate host side drivers.
 */
struct f_diag {
	struct u_diag   port;
	u8				data_id;
	u8				port_num;
};

static inline struct f_diag *func_to_diag(struct usb_function *f)
{
	return container_of(f, struct f_diag, port.func);
}

/*-------------------------------------------------------------------------*/

/* interface descriptor: */

static struct usb_interface_descriptor diag_interface_desc  = {

	.bLength =		USB_DT_INTERFACE_SIZE,
	.bDescriptorType =	USB_DT_INTERFACE,
	/* .bInterfaceNumber = DYNAMIC */
	.bNumEndpoints =	2,
	.bInterfaceClass =	USB_CLASS_VENDOR_SPEC,
	.bInterfaceSubClass =	USB_SUBCLASS_VENDOR_SPEC,
	.bInterfaceProtocol =	USB_SUBCLASS_VENDOR_SPEC,
	/* .iInterface = DYNAMIC */
};

/* full speed support: */

static struct usb_endpoint_descriptor diag_fs_in_desc  = {

	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bEndpointAddress =	USB_DIR_IN,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
};

static struct usb_endpoint_descriptor diag_fs_out_desc  = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bEndpointAddress =	USB_DIR_OUT,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
};

static struct usb_descriptor_header *diag_fs_function[] = {
	(struct usb_descriptor_header *) &diag_interface_desc,
	(struct usb_descriptor_header *) &diag_fs_in_desc,
	(struct usb_descriptor_header *) &diag_fs_out_desc,
	NULL,
};

/* high speed support: */

static struct usb_endpoint_descriptor diag_hs_in_desc  = {

	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize =	cpu_to_le16(512),
};

static struct usb_endpoint_descriptor diag_hs_out_desc  = {

	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize =	cpu_to_le16(512),
};

static struct usb_descriptor_header *diag_hs_function[]  = {

	(struct usb_descriptor_header *) &diag_interface_desc,
	(struct usb_descriptor_header *) &diag_hs_in_desc,
	(struct usb_descriptor_header *) &diag_hs_out_desc,
	NULL,
};

static struct usb_endpoint_descriptor diag_ss_in_desc __initdata = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize =	cpu_to_le16(1024),
};

static struct usb_endpoint_descriptor diag_ss_out_desc __initdata = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize =	cpu_to_le16(1024),
};

static struct usb_ss_ep_comp_descriptor diag_ss_bulk_comp_desc __initdata = {
	.bLength =              sizeof diag_ss_bulk_comp_desc,
	.bDescriptorType =      USB_DT_SS_ENDPOINT_COMP,
};

static struct usb_descriptor_header *diag_ss_function[] __initdata = {
	(struct usb_descriptor_header *) &diag_interface_desc,
	(struct usb_descriptor_header *) &diag_ss_in_desc,
	(struct usb_descriptor_header *) &diag_ss_bulk_comp_desc,
	(struct usb_descriptor_header *) &diag_ss_out_desc,
	(struct usb_descriptor_header *) &diag_ss_bulk_comp_desc,
	NULL,
};

/* string descriptors: */

static struct usb_string diag_string_defs[] = {
	[0].s = "Diag",
	{  } /* end of list */
};

static struct usb_gadget_strings diag_string_table = {
	.language =		0x0409,	/* en-us */
	.strings =		diag_string_defs,
};

static struct usb_gadget_strings *diag_strings[] = {
	&diag_string_table,
	NULL,
};

/*-------------------------------------------------------------------------*/

static int diag_set_alt(struct usb_function *f, unsigned intf, unsigned alt)
{
	struct f_diag		*diag = func_to_diag(f);
	struct usb_composite_dev *cdev = f->config->cdev;
#ifdef CONFIG_PM
	if(diag->port.suspend_state == 1)
		diag->port.suspend_state = 0;
#endif	
	/* we know alt == 0, so this is an activation or a reset */

	if (diag->port.in->driver_data) {
		diag_disconnect(&diag->port);
	}
	if (!diag->port.in->desc || !diag->port.out->desc) {
		if (config_ep_by_speed(cdev->gadget, f, diag->port.in) ||
		    config_ep_by_speed(cdev->gadget, f, diag->port.out)) {
			diag->port.in->desc = NULL;
			diag->port.out->desc = NULL;
			return -EINVAL;
		}
	}
	diag_connect(&diag->port, diag->port_num);
	return 0;
}

static void diag_disable(struct usb_function *f)
{
	struct f_diag	*diag = func_to_diag(f);
	struct usb_composite_dev *cdev = f->config->cdev;
#if 0
	if(diag->port.suspend_state == 1)
		return;
#endif
	diag_disconnect(&diag->port);
}

#ifdef CONFIG_PM
unsigned int g_diag_suspend_cnt = 0;
unsigned int g_diag_resume_cnt = 0;
static void diag_suspend(struct usb_function *f)
{
	struct f_diag	*diag = func_to_diag(f);
	struct usb_composite_dev *cdev = f->config->cdev;

    g_diag_suspend_cnt++;
	usb_printk( "%s, %u, suspendcnt:%d\n", __func__, __LINE__, g_diag_suspend_cnt);
	USBSTACK_DBG("%s, %u suspendcnt:%d", __func__, __LINE__, g_diag_suspend_cnt);
//	diag_disable(f);
 	//diagial_disconnect_ext(&diag->port);
    diag->port.suspend_state = 1;
	diag_disconnect_ext(&diag->port);
	
    usb_ep_disable((&diag->port)->in);
}
static void diag_resume(struct usb_function *f)
{
	struct f_diag	*diag = func_to_diag(f);
	struct usb_composite_dev *cdev = f->config->cdev;

    g_diag_resume_cnt++;
	usb_printk("%s, %u, resumecnt:%d\n", __func__, __LINE__, g_diag_resume_cnt);
	USBSTACK_DBG("%s, %u resumecnt:%d", __func__, __LINE__, g_diag_resume_cnt);
//	diag_set_alt(f, NULL, NULL);
	//usb_ep_enable((&diag->port)->in);
	usb_ep_resume_enable((&diag->port)->in);
//    usb_ep_enable((&diag->port)->out);
 	//diagial_connect_ext(&diag->port);
	diag->port.suspend_state = 0;
	diag_connect_ext(&diag->port);

}
#endif

/*-------------------------------------------------------------------------*/

/* serial function driver setup/binding */

//static int __init
static int 
diag_bind(struct usb_configuration *c, struct usb_function *f)
{
	struct usb_composite_dev *cdev = c->cdev;
	struct f_diag		*diag = func_to_diag(f);
	int			status;
	struct usb_ep		*ep;

	/* allocate instance-specific interface IDs */
	status = usb_interface_id(c, f);
	if (status < 0)
		goto fail;
	diag->data_id = status;
	diag_interface_desc.bInterfaceNumber = status;

	status = -ENODEV;

	/* allocate instance-specific endpoints */
	ep = usb_ep_autoconfig(cdev->gadget, &diag_fs_in_desc);
	if (!ep)
		goto fail;
	diag->port.in = ep;
	ep->driver_data = cdev;	/* claim */

	ep = usb_ep_autoconfig(cdev->gadget, &diag_fs_out_desc);
	if (!ep)
		goto fail;
	diag->port.out = ep;
	ep->driver_data = cdev;	/* claim */

	/* copy descriptors, and track endpoint copies */
	f->descriptors = usb_copy_descriptors(diag_fs_function);

	/* support all relevant hardware speeds... we expect that when
	 * hardware is dual speed, all bulk-capable endpoints work at
	 * both speeds
	 */
	if (gadget_is_dualspeed(c->cdev->gadget)) {
		diag_hs_in_desc.bEndpointAddress =
				diag_fs_in_desc.bEndpointAddress;
		diag_hs_out_desc.bEndpointAddress =
				diag_fs_out_desc.bEndpointAddress;

		/* copy descriptors, and track endpoint copies */
		f->hs_descriptors = usb_copy_descriptors(diag_hs_function);
	}
	if (gadget_is_superspeed(c->cdev->gadget)) {
		diag_ss_in_desc.bEndpointAddress =
			diag_fs_in_desc.bEndpointAddress;
		diag_ss_out_desc.bEndpointAddress =
			diag_fs_out_desc.bEndpointAddress;

		/* copy descriptors, and track endpoint copies */
		f->ss_descriptors = usb_copy_descriptors(diag_ss_function);
		if (!f->ss_descriptors)
			goto fail;
	}

	return 0;

fail:
	/* we might as well release our claims on endpoints */
	if (diag->port.out)
		diag->port.out->driver_data = NULL;
	if (diag->port.in)
		diag->port.in->driver_data = NULL;

	return status;
}

static void
diag_unbind(struct usb_configuration *c, struct usb_function *f)
{
	if (gadget_is_dualspeed(c->cdev->gadget))
		usb_free_descriptors(f->hs_descriptors);
	if (gadget_is_superspeed(c->cdev->gadget))
		usb_free_descriptors(f->ss_descriptors);
	usb_free_descriptors(f->descriptors);
	kfree(func_to_diag(f));
}

/**
 * diag_bind_config - add a diag function to a configuration
 * @c: the configuration to support the diag instance
 * @port_num: 
 * Context: single threaded during gadget setup
 *
 * Returns zero on success, else negative errno.
 *
 * Caller must have called @diagial_setup() with enough ports to
 * handle all the ones it binds.  Caller is also responsible
 * for calling @diagial_cleanup() before module unload.
 */
//int __init diag_bind_config(struct usb_configuration *c, u8 port_num)
int  diag_bind_config(struct usb_configuration *c, u8 port_num)

{
	struct f_diag	*diag;
	int		status;

	/* REVISIT might want instance-specific strings to help
	 * distinguish instances ...
	 */
	 USB_DEBUG("PORT_NUM:%d", port_num);

	/* maybe allocate device-global string ID */
	if (diag_string_defs[0].id == 0) {
		status = usb_string_id(c->cdev);
		if (status < 0)
			return status;
		diag_string_defs[0].id = status;
	}

	USB_DEBUG("MALLOC before");

	/* allocate and initialize one new instance */
	diag = kzalloc(sizeof *diag, GFP_KERNEL);
	if (!diag)
		return -ENOMEM;
	USB_DEBUG("MALLOC end");

	diag->port_num = port_num;

	diag->port.func.name = "diag";
	diag->port.func.strings = diag_strings;
	diag->port.func.bind = diag_bind;
	diag->port.func.unbind = diag_unbind;
	diag->port.func.set_alt = diag_set_alt;
	diag->port.func.disable = diag_disable;
#ifdef CONFIG_PM
    diag->port.func.suspend = diag_suspend;
    diag->port.func.resume = diag_resume;
#endif

	status = usb_add_function(c, &diag->port.func);
	if (status)
		kfree(diag);
	return status;
}
