/*******************************************************************************
* Copyright (C) 2013, ZTE Corporation.
*
* File Name:    io_request.c
* File Mark:
* Description:
* Others:
* Version:		 V1.0
* Author:		 geanfeng
* Date: 		 2013-10-21
********************************************************************************/

/****************************************************************************
* 	                                           Include files
****************************************************************************/
#include "drvs_general.h"
#include "drvs_assert.h"
#include "io_request.h"
#include "drvs_debug.h"
/****************************************************************************
* 	                              Local Macros
****************************************************************************/


/****************************************************************************
* 	                               Local Types
****************************************************************************/
VOID IORequest_PrintAll(VOID);

/****************************************************************************
* 	                                          Global Variables
****************************************************************************/
LIST_HEAD(g_IORequestHnd);
UINT32 g_IORequest_Cmd_Init = 0;

/****************************************************************************
* 	                                          macro Definitions
****************************************************************************/

/****************************************************************************
* 	                                          Function Definitions
****************************************************************************/
/*******************************************************************************
 * Function: fill_SgList
 * Description:fill the current request scatter list.
 * Input:
 * Output:None
 *
 * Returns:
              DRV_SUCCESS: success.
              DRV_ERROR: error, no request to deal.
 * Others:
 ********************************************************************************/
static SINT32 fill_SgList(T_Request_Queue *reqQ, UINT32 reqSize, T_Request_SgList *sgList, UINT32 sgListMaxCount)
{
    UINT32 index = 0;
    SINT32 ret = 0;
    UINT32 reqBufferPos = 0;

    while (index < sgListMaxCount)
    {
        ret = requestQueue_FetchRequest(reqQ, (T_Request *)&sgList->reqBuffer[reqBufferPos]);
        if (ret == DRV_ERROR_EMPTY)
        {
            break;
        }
        else if (ret == DRV_SUCCESS)
        {
            index++;
            reqBufferPos += reqSize;
        }
        else
        {
            zDrv_ASSERT(0);
        }
    }

    sgList->reqCount = index;

    if (sgList->reqCount)
        return DRV_SUCCESS;
    else
        return DRV_ERROR;
}
/*******************************************************************************
 * Function: free_Request
 * Description: free the request.
 * Input:
 * Output:None
 *
 * Returns:
              DRV_SUCCESS: success.
              DRV_ERROR: error.
* Others:
 ********************************************************************************/
static SINT32 free_Request(T_IO_RequestHnd * reqHnd, T_IO_REQUEST_DIRECTION reqDirection, T_Request *req)
{
    if (reqDirection == IO_REQUEST_READ)
    {
        reqHnd->readFreeCnt++;
    }
    else if (reqDirection == IO_REQUEST_WRITE)
    {
        reqHnd->writeFreeCnt++;
    }
    else
    {
        zDrv_ASSERT(0);
    }
    reqHnd->ops->free_request(reqHnd->reqHndData, reqDirection, req);
    return DRV_SUCCESS;
}
/*******************************************************************************
 * Function: free_AllRequest
 * Description:free all requests in IORequest module.
 * Input:
 * Output:None
 *
 * Returns:
              DRV_SUCCESS: success.
              DRV_ERROR: error.
 * Others:
 ********************************************************************************/
static SINT32 free_AllRequest(T_IO_RequestHnd * reqHnd)
{
    SINT32 ret = DRV_SUCCESS;
    T_Request *req = NULL;

    if (reqHnd == NULL || reqHnd->reqSize == 0)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    req = (T_Request *)zOss_Malloc(reqHnd->reqSize);
    zDrv_ASSERT(req != NULL);

    ret = DRV_SUCCESS;
    if (reqHnd->readQ)
    {
        requestQueue_Suspend(reqHnd->readQ);
        while (ret == DRV_SUCCESS)
        {
            ret = requestQueue_FetchRequest(reqHnd->readQ, req);
            if (ret == DRV_SUCCESS)
            {
                if (reqHnd->ops->request_done)
                    reqHnd->ops->request_done(reqHnd->reqHndData, req, REQUEST_NODEV);
                free_Request(reqHnd, IO_REQUEST_READ, req);
            }
        }
    }

    ret = DRV_SUCCESS;
    if (reqHnd->writeQ)
    {
        requestQueue_Suspend(reqHnd->writeQ);
        while (ret == DRV_SUCCESS)
        {
            ret = requestQueue_FetchRequest(reqHnd->writeQ, req);
            if (ret == DRV_SUCCESS)
            {
                if (reqHnd->ops->request_done)
                    reqHnd->ops->request_done(reqHnd->reqHndData, req, REQUEST_NODEV);
                free_Request(reqHnd, IO_REQUEST_WRITE, req);
            }
        }
    }

    ret = DRV_SUCCESS;
    if (reqHnd->readDoneBuffer)
    {
        while (ret == DRV_SUCCESS)
        {
            ret = ringQueue_Dequeue(reqHnd->readDoneBuffer, req);
            if (ret == DRV_SUCCESS)
            {
                if (reqHnd->ops->request_done)
                    reqHnd->ops->request_done(reqHnd->reqHndData, req, REQUEST_NODEV);
                free_Request(reqHnd, IO_REQUEST_READ, req);
            }
        }
    }

    if (reqHnd->readReqState == IO_REQUEST_START || reqHnd->readReqState == IO_REQUEST_START_TO_SUSPEND)
    {
        ret = reqHnd->readDoneFn(reqHnd, &reqHnd->readSgList, REQUEST_NODEV);
    }
    if (reqHnd->writeReqState == IO_REQUEST_START || reqHnd->writeReqState == IO_REQUEST_START_TO_SUSPEND)
    {
        ret = reqHnd->writeDoneFn(reqHnd, &reqHnd->writeSgList, REQUEST_NODEV);
    }

    return ret;
}

/*******************************************************************************
 * Function: read_Start
 * Description:start the read process.
 * Input:
 * Output:None
 *
 * Returns:
              DRV_SUCCESS: success.
              DRV_ERROR: error.
 * Others:
 ********************************************************************************/
static SINT32 read_Start(T_Request_Queue *queue)
{
    T_IO_RequestHnd * reqHnd = NULL;
    SINT32 ret = 0;

    reqHnd = (T_IO_RequestHnd*)queue->queue_data;

    if (reqHnd->readReqState == IO_REQUEST_START)
        return 0;

    if (reqHnd->readReqState == IO_REQUEST_STOP || reqHnd->readReqState == IO_REQUEST_STOP_TO_SUSPEND)
    {
        ret = fill_SgList(reqHnd->readQ, reqHnd->reqSize, &reqHnd->readSgList, reqHnd->readSgListMaxCount);
        if (ret != DRV_SUCCESS)
        {
            reqHnd->readReqState = IO_REQUEST_STOP;
            return DRV_ERROR;
        }
    }
    reqHnd->readReqState = IO_REQUEST_START;

    zDrv_ASSERT(reqHnd->halEnabled);
    ret = reqHnd->ops->start_request(reqHnd->reqHndData, IO_REQUEST_READ);

    return ret;
}

/*******************************************************************************
 * Function: read_Done
 * Description:asynchronous read process done, put the requests to the done queue and start the next request if it is necessarily .
 * Input:
 * Output:None
 *
 * Returns:
              DRV_SUCCESS: success.
              DRV_ERROR: error.
 * Others:
 ********************************************************************************/
static SINT32 read_Done(T_IO_RequestHnd *reqHnd, T_Request_SgList * reqSgList, T_DONE_STATE state)
{
    SINT32 ret = 0;
    T_Request *req = NULL;
    UINT32 index = 0;

    if (reqHnd == NULL || reqSgList == NULL || reqSgList->reqCount == 0)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    if (state == REQUEST_SUCCESS)
    {
        zDrv_ASSERT(reqHnd->readReqState == IO_REQUEST_START);
        zDrv_ASSERT(reqHnd->readDoneBuffer != NULL);

        /*add to the done request buffer*/
        while (index < reqSgList->reqCount)
        {
            req = (T_Request *)(&reqSgList->reqBuffer[index*reqHnd->reqSize]);
            ret = ringQueue_Enqueue(reqHnd->readDoneBuffer, req);
            zDrv_ASSERT(ret == DRV_SUCCESS);
            if (reqHnd->ops->request_done)
                reqHnd->ops->request_done(reqHnd->reqHndData, req, state);
            index++;
        }

        /*һ*/
        ret = fill_SgList(reqHnd->readQ, reqHnd->reqSize, &reqHnd->readSgList, reqHnd->readSgListMaxCount);
        if (ret == DRV_SUCCESS)
        {
            ret = reqHnd->ops->start_request(reqHnd->reqHndData, IO_REQUEST_READ);
        }
        else
        {
            reqHnd->readReqState = IO_REQUEST_STOP;
        }
        if (zOss_GetSemaphoreCount(reqHnd->writeReqSem) <= 0)
        {
            zOss_PutSemaphore(reqHnd->readReqSem);
        }
    }
    else if (state == REQUEST_NODEV)
    {
        /*free the request*/
        while (index < reqSgList->reqCount)
        {
            req = (T_Request *)(&reqSgList->reqBuffer[index*reqHnd->reqSize]);
            if (reqHnd->ops->request_done)
                reqHnd->ops->request_done(reqHnd->reqHndData, req, state);
            free_Request(reqHnd, IO_REQUEST_READ, req);
            index++;
        }
    }

    return ret;
}
/*******************************************************************************
 * Function: write_Start
 * Description:start the write process.
 * Input:
 * Output:None
 *
 * Returns:
              DRV_SUCCESS: success.
              DRV_ERROR: error.
 * Others:
 ********************************************************************************/
static SINT32 write_Start(T_Request_Queue *queue)
{
    T_IO_RequestHnd * reqHnd = NULL;
    SINT32 ret = 0;

    reqHnd = (T_IO_RequestHnd*)queue->queue_data;

    if (reqHnd->writeReqState == IO_REQUEST_START)
        return 0;

    if (reqHnd->writeReqState == IO_REQUEST_STOP || reqHnd->writeReqState == IO_REQUEST_STOP_TO_SUSPEND)
    {
        ret = fill_SgList(reqHnd->writeQ, reqHnd->reqSize, &reqHnd->writeSgList, reqHnd->writeSgListMaxCount);
        if (ret != DRV_SUCCESS)
        {
            reqHnd->writeReqState = IO_REQUEST_STOP;
            return DRV_ERROR;
        }
    }
    reqHnd->writeReqState = IO_REQUEST_START;

    zDrv_ASSERT(reqHnd->halEnabled);
    ret = reqHnd->ops->start_request(reqHnd->reqHndData, IO_REQUEST_WRITE);

    return ret;
}

/*******************************************************************************
 * Function: write_Done
 * Description:asynchronous write process done, free the requests and start the next request if it is necessarily .
 * Input:
 * Output:None
 *
 * Returns:
              DRV_SUCCESS: success.
              DRV_ERROR: error.
 * Others:
 ********************************************************************************/
static SINT32 write_Done(T_IO_RequestHnd *reqHnd, T_Request_SgList * reqSgList, T_DONE_STATE state)
{
    SINT32 ret = 0;
    T_Request *req = NULL;
    UINT32 index = 0;

    if (reqHnd == NULL || reqSgList == NULL || reqSgList->reqCount == 0)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    /*free the request*/
    while (index < reqSgList->reqCount)
    {
        req = (T_Request *)(&reqSgList->reqBuffer[index*reqHnd->reqSize]);
        if (reqHnd->ops->request_done)
            reqHnd->ops->request_done(reqHnd->reqHndData, req, state);
        free_Request(reqHnd, IO_REQUEST_WRITE, req);
        index++;
    }

    if (state == REQUEST_SUCCESS)
    {
        zDrv_ASSERT(reqHnd->writeReqState == IO_REQUEST_START);
        /*һ*/
        ret = fill_SgList(reqHnd->writeQ, reqHnd->reqSize, &reqHnd->writeSgList, reqHnd->writeSgListMaxCount);
        if (ret == DRV_SUCCESS)
        {
            ret = reqHnd->ops->start_request(reqHnd->reqHndData, IO_REQUEST_WRITE);
        }
        else
        {
            reqHnd->writeReqState = IO_REQUEST_STOP;
        }

        if (zOss_GetSemaphoreCount(reqHnd->writeReqSem) <= 0)
        {
            zOss_PutSemaphore(reqHnd->writeReqSem);
        }
    }

    return ret;
}



/*******************************************************************************
 * Function: drvIORequest_CmdEntry
 * Description:add IO Request debug cmd.
 * Input:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
static VOID drvIORequest_CmdEntry(T_Shell_CommandMessage *CmdMsg)
{
    IORequest_PrintAll();
    return ;
}
/*******************************************************************************
 * Function: IORequest_Create
 * Description:create the IO request handle, configure the params.
 * Input:
 * Output:None
 *
 * Returns:
    DRV_SUCCESS: success.
    DRV_ERROR: error.
 * Others:.
 ********************************************************************************/
T_IO_RequestHnd * IORequest_Create(UINT8* name,
                                   T_IO_REQUEST_DIRECTION reqDirection, UINT32 reqSize,
                                   VOID *reqHndData, T_IO_RequestOps *ops,
                                   UINT32 readReqCount, UINT32 readSgListMaxCount,
                                   UINT32 writeReqCount, UINT32 writeSgListMaxCount)
{
    T_IO_RequestHnd *reqHnd = NULL;

    reqHnd = (T_IO_RequestHnd *)zOss_Malloc(sizeof(T_IO_RequestHnd));
    if (!reqHnd || (reqSize & 0x3))
    {
        zDrv_ASSERT(0);
        return NULL;
    }
    if (!ops || !ops->hal_enable ||!ops->hal_disable ||!ops->free_request || !ops->start_request)
    {
        zDrv_ASSERT(0);
        return NULL;
    }

    zOss_Memset((VOID*)reqHnd, 0x0, sizeof(T_IO_RequestHnd));

    reqHnd->name = name;
    reqHnd->reqDirection = reqDirection;
    reqHnd->reqSize = reqSize;
    reqHnd->reqHndData = reqHndData;
    reqHnd->ops = ops;
    reqHnd->readSgListMaxCount = readSgListMaxCount;
    reqHnd->writeSgListMaxCount = writeSgListMaxCount;
    zDrv_ASSERT((reqHnd->reqSize) > sizeof(T_Request));

    /*read initialize*/
    if (reqHnd->reqDirection == IO_REQUEST_READ || reqHnd->reqDirection == IO_REQUEST_READ_WRITE)
    {
        reqHnd->readReqCount = readReqCount;
        reqHnd->readQ = requestQueue_Create(reqHnd->readReqCount, reqHnd->reqSize, read_Start, reqHnd);
        if (!reqHnd->readQ || ((UINT32)reqHnd->readQ & 0x3))
            goto error;
        reqHnd->readDoneBuffer = ringQueue_Create(reqHnd->readReqCount + reqHnd->readSgListMaxCount, reqHnd->reqSize, QUEUE_PROTECT_RAW);
        if (!reqHnd->readDoneBuffer || ((UINT32)reqHnd->readDoneBuffer & 0x3))
            goto error;
        reqHnd->readSgList.reqBuffer = (UINT8 *)zOss_Malloc(reqHnd->reqSize * reqHnd->readSgListMaxCount);
        if (!reqHnd->readSgList.reqBuffer || ((UINT32)reqHnd->readSgList.reqBuffer & 0x3))
            goto error;
        reqHnd->readReqSem = zOss_CreateSemaphore("readReqSem", 0x0);
        if (!reqHnd->readReqSem)
            goto error;

        reqHnd->readReqState = IO_REQUEST_STOP;
        reqHnd->readDoneFn = read_Done;
    }

    /*write initialize*/
    if (reqHnd->reqDirection == IO_REQUEST_WRITE || reqHnd->reqDirection == IO_REQUEST_READ_WRITE)
    {
        reqHnd->writeReqCount = writeReqCount;
        reqHnd->writeQ = requestQueue_Create(reqHnd->writeReqCount, reqHnd->reqSize, write_Start, reqHnd);
        if (!reqHnd->writeQ || ((UINT32)reqHnd->writeQ & 0x3))
            goto error;
        reqHnd->writeSgList.reqBuffer = (UINT8 *)zOss_Malloc(reqHnd->reqSize * reqHnd->writeSgListMaxCount);
        if (!reqHnd->writeSgList.reqBuffer || ((UINT32)reqHnd->writeSgList.reqBuffer & 0x3))
            goto error;
        reqHnd->writeReqSem = zOss_CreateSemaphore("writeReqSem", 0x0);
        if (!reqHnd->writeReqSem)
            goto error;

        reqHnd->writeReqState= IO_REQUEST_STOP;
        reqHnd->writeDoneFn = write_Done;
    }

    /*enable hal operations*/
    if (reqHnd->ops && reqHnd->ops->hal_enable)
        reqHnd->ops->hal_enable(reqHnd->reqHndData);
    reqHnd->halEnabled = TRUE;

    reqHnd->isSuspend = FALSE;
    reqHnd->hndState = IO_REQUEST_CREATED;

    list_add(&reqHnd->node, &g_IORequestHnd);

    if (!g_IORequest_Cmd_Init)
    {
        zOss_AddShellCmd("iorequest", drvIORequest_CmdEntry, "DRV  IO Request  Info");
        g_IORequest_Cmd_Init = 1;
    }

    return reqHnd;

error:
    if (reqHnd->readQ)
        requestQueue_Destroy(reqHnd->readQ);
    if (reqHnd->readDoneBuffer)
        ringQueue_Destroy(reqHnd->readDoneBuffer);
    if (reqHnd->readReqSem)
        zOss_DeleteSemaphore(reqHnd->readReqSem);
    if (reqHnd->writeSgList.reqBuffer)
        zOss_Free(reqHnd->writeSgList.reqBuffer);

    if (reqHnd->writeQ)
        requestQueue_Destroy(reqHnd->readQ);
    if (reqHnd->writeReqSem)
        zOss_DeleteSemaphore(reqHnd->writeReqSem);
    if (reqHnd->readSgList.reqBuffer)
        zOss_Free(reqHnd->readSgList.reqBuffer);

    zDrv_ASSERT(0);
    return NULL;
}

/*******************************************************************************
 * Function: IORequest_SubmitReadRequest
 * Description:submit the read request.
 * Input:
 *        reqHnd: the handle of request.
 *        req: the submit request.
 * Output:None
 *
 * Returns:
 *         DRV_SUCCESS: success
 ********************************************************************************/
SINT32 IORequest_SubmitReadRequest(T_IO_RequestHnd *reqHnd, T_Request *req)
{
    SINT32 ret = 0;

    if (reqHnd == NULL || req == NULL || reqHnd->reqDirection == IO_REQUEST_WRITE || reqHnd->hndState == IO_REQUEST_UNCREATED)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    ret = requestQueue_SubmitRequest(reqHnd->readQ, req);
    zDrv_ASSERT(ret == DRV_SUCCESS);

    return ret;
}
/*******************************************************************************
 * Function: IORequest_FetchDoneRequest
 * Description:fetch the completed read request.
 * Input:
 *        reqHnd: the handle of request.
 *        blockType: the block type of this operation. no wait or wait forever.
 * Output:None
 *        req: the complete read request.
 * Returns:
 *         DRV_SUCCESS: success
 *         DRV_ERROR_AGAIN: no wait, need fetch again.
 *         DRV_ERROR_ABORT: force exit.
 * Others:
 ********************************************************************************/
SINT32 IORequest_FetchDoneRequest(T_IO_RequestHnd *reqHnd, T_Request *req, T_IO_REQUEST_BLOCK_TYPE blockType)
{
    SINT32 ret = DRV_SUCCESS;

    if (reqHnd == NULL || req == NULL || reqHnd->reqDirection == IO_REQUEST_WRITE || reqHnd->hndState == IO_REQUEST_UNCREATED)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    while (1)
    {
        ret = ringQueue_Dequeue(reqHnd->readDoneBuffer, req);
        if (ret == DRV_ERROR_EMPTY)
        {
            if (blockType == IO_REQUEST_NO_WAIT)
            {
                return DRV_ERROR_AGAIN;
            }
            else
            {
                zOss_GetSemaphore(reqHnd->readReqSem, ZOSS_WAIT_FOREVER);
                if (reqHnd->hndState == IO_REQUEST_FORCE_EXIT)
                {
                    return  DRV_ERROR_ABORT;
                }
            }
        }
        else if (ret == DRV_SUCCESS)
        {
            break; /*ɹ*/
        }
        else
        {
            zDrv_ASSERT(0);
        }
    }

    return ret;
}

/*******************************************************************************
 * Function: IORequest_SubmitWriteRequest
 * Description:submit the write request.
 * Input:
 *        reqHnd: the handle of request.
 *        req: the submit request.
 *        blockType: the block type of this operation. no wait or wait forever.
 * Output:None
 *
 * Returns:
 *         DRV_SUCCESS: success
 *         DRV_ERROR_AGAIN: no wait, submit again.
 *         DRV_ERROR_ABORT: force exit.
 * Others:
 ********************************************************************************/
SINT32 IORequest_SubmitWriteRequest(T_IO_RequestHnd *reqHnd, T_Request *req, T_IO_REQUEST_BLOCK_TYPE blockType)
{
    SINT32 ret = 0;

    if (reqHnd == NULL || req == NULL || reqHnd->reqDirection == IO_REQUEST_READ || reqHnd->hndState == IO_REQUEST_UNCREATED)
    {
        zDrv_ASSERT(0);
        return 0;
    }

    while (1)
    {
        ret = requestQueue_SubmitRequest(reqHnd->writeQ, req);
        if (ret == DRV_ERROR_FULL)
        {
            /*ύʧ*/
            if (blockType == IO_REQUEST_NO_WAIT)
            {
                return DRV_ERROR_AGAIN; /*request queue is full*/
            }
            else
            {
                zOss_GetSemaphore(reqHnd->writeReqSem, ZOSS_WAIT_FOREVER);
                if (reqHnd->hndState == IO_REQUEST_FORCE_EXIT)
                {
                    return DRV_ERROR_ABORT;
                }
            }
        }
        else if (ret == DRV_SUCCESS)
        {
            break;
        }
        else
        {
            zDrv_ASSERT(0);
        }
    }

    return ret;
}
/*******************************************************************************
 * Function: IORequest_WaitRequestDone
 * Description:wait the write request done.
 * Input:
 *        reqHnd: the handle of request.
 * Output:None
 *
 * Returns:
 *         DRV_SUCCESS: success
 *         DRV_ERROR_ABORT: force exit.
 * Others:
 ********************************************************************************/
SINT32 IORequest_WaitRequestDone(T_IO_RequestHnd *reqHnd)
{

    if (reqHnd == NULL || reqHnd->reqDirection == IO_REQUEST_READ || reqHnd->hndState == IO_REQUEST_UNCREATED)
    {
        zDrv_ASSERT(0);
        return 0;
    }

    /*ȴ*/
    while (reqHnd->writeReqState != IO_REQUEST_STOP)
    {
        zOss_GetSemaphore(reqHnd->writeReqSem, ZOSS_WAIT_FOREVER);
        if (reqHnd->hndState == IO_REQUEST_FORCE_EXIT)
        {
            return DRV_ERROR_ABORT;
        }
    }

    return DRV_SUCCESS;
}

/*******************************************************************************
 * Function: IORequest_GetSgList
 * Description:get the current requests scatter list.
 * Input:
 * Output:None
 *
 * Returns:
	T_Request_SgList* : success.
	NULL: error
 * Others:
 ********************************************************************************/
T_Request_SgList* IORequest_GetSgList(T_IO_RequestHnd *reqHnd, T_IO_REQUEST_DIRECTION reqDirection)
{
    if (reqHnd == NULL || reqHnd->halEnabled != TRUE || reqHnd->hndState == IO_REQUEST_UNCREATED)
    {
        zDrv_ASSERT(0);
        return 0;
    }

    if (reqDirection == IO_REQUEST_READ)
    {
        return &reqHnd->readSgList;
    }
    else if (reqDirection == IO_REQUEST_WRITE)
    {
        return &reqHnd->writeSgList;
    }
    else
    {
        return NULL;
    }
}
/*******************************************************************************
 * Function: IORequest_Done
 * Description:has complete the request, take next proccess .
 * Input:
 * Output:None
 *
 * Returns:
    DRV_SUCCESS: success.
    DRV_ERROR: error.
 * Others:ע:ô˽ӿڵȼһģڵӿڣɱϡ
 ********************************************************************************/
SINT32 IORequest_Done(T_IO_RequestHnd *reqHnd, T_IO_REQUEST_DIRECTION reqDirection)
{
    SINT32 ret = 0;

    if (reqHnd == NULL || !reqHnd->halEnabled || reqHnd->isSuspend  || reqHnd->hndState == IO_REQUEST_UNCREATED)
    {
        zDrv_ASSERT(0);
        return 0;
    }

    if (reqDirection == IO_REQUEST_READ)
    {
        if (reqHnd->readDoneFn == NULL)
        {
            zDrv_ASSERT(0);
            return 0;
        }
        ret = reqHnd->readDoneFn(reqHnd, &reqHnd->readSgList, REQUEST_SUCCESS);
    }
    else if (reqDirection == IO_REQUEST_WRITE)
    {
        if (reqHnd->writeDoneFn == NULL)
        {
            zDrv_ASSERT(0);
            return 0;
        }
        ret = reqHnd->writeDoneFn(reqHnd, &reqHnd->writeSgList, REQUEST_SUCCESS);
    }
    else
    {
        zDrv_ASSERT(0);
    }
    return ret;
}

/*******************************************************************************
 * Function: IORequest_Suspend
 * Description:suspend the request process, stop the hal operation.
 * Input:
 * Output:None
 *
 * Returns:
    DRV_SUCCESS: success.
    DRV_ERROR: error.
 * Others:
 ********************************************************************************/
SINT32 IORequest_Suspend(T_IO_RequestHnd *reqHnd)
{
    SINT32 ret = DRV_SUCCESS;

    if (!reqHnd || reqHnd->hndState != IO_REQUEST_CREATED)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }

    if (reqHnd->isSuspend)
    {
        return DRV_SUCCESS;
    }

    /*suspend the request queue*/
    if (reqHnd->readQ)
    {
        requestQueue_Suspend(reqHnd->readQ);
    }
    if (reqHnd->writeQ)
    {
        requestQueue_Suspend(reqHnd->writeQ);
    }

    /*disable hal operations*/
    if (reqHnd->ops && reqHnd->ops->hal_disable)
    {
        ret = reqHnd->ops->hal_disable(reqHnd->reqHndData);
    }
    if (ret != DRV_SUCCESS)
    {
        zDrv_ASSERT(0);
        return DRV_ERROR;
    }
    reqHnd->halEnabled = FALSE;

    /*read request state*/
    if (reqHnd->readReqState == IO_REQUEST_START)
    {
        reqHnd->readReqState = IO_REQUEST_START_TO_SUSPEND;
    }
    else if (reqHnd->readReqState == IO_REQUEST_STOP)
    {
        reqHnd->readReqState = IO_REQUEST_STOP_TO_SUSPEND;
    }
    else
    {
        zDrv_ASSERT(0);
    }

    /*write request state*/
    if (reqHnd->writeReqState == IO_REQUEST_START)
    {
        reqHnd->writeReqState = IO_REQUEST_START_TO_SUSPEND;
    }
    else if (reqHnd->writeReqState == IO_REQUEST_STOP)
    {
        reqHnd->writeReqState = IO_REQUEST_STOP_TO_SUSPEND;
    }
    else
    {
        zDrv_ASSERT(0);
    }

    reqHnd->isSuspend = TRUE;
    return DRV_SUCCESS;
}
/*******************************************************************************
 * Function: IORequest_Resume
 * Description:resume the request process, and restart the stopped hal operation.
 * Input:
 * Output:None
 *
 * Returns:
    DRV_SUCCESS: success.
    DRV_ERROR: error.
 * Others:
 ********************************************************************************/
VOID IORequest_Resume(T_IO_RequestHnd *reqHnd)
{
    if (!reqHnd || !reqHnd->isSuspend || reqHnd->hndState != IO_REQUEST_CREATED)
    {
        zDrv_ASSERT(0);
        return ;
    }

    /*enable hal operations*/
    if (reqHnd->ops && reqHnd->ops->hal_enable)
        reqHnd->ops->hal_enable(reqHnd->reqHndData);
    reqHnd->halEnabled = TRUE;

    reqHnd->isSuspend = FALSE;

    /*can submit request to hal*/
    if (reqHnd->readQ)
    {
        requestQueue_Resume(reqHnd->readQ);
    }
    if (reqHnd->writeQ)
    {
        requestQueue_Resume(reqHnd->writeQ);
    }

    return ;
}
/*******************************************************************************
 * Function: IORequest_ForceExit
 * Description: force the higher process exit.
 * Input:
 * Output:None
 *
 * Returns:
    DRV_SUCCESS: success.
    DRV_ERROR: error.
 * Others:
 ********************************************************************************/
VOID IORequest_ForceExit(T_IO_RequestHnd *reqHnd)
{

    if (!reqHnd || reqHnd->isSuspend  || reqHnd->hndState != IO_REQUEST_CREATED)
    {
        zDrv_ASSERT(0);
        return ;
    }

    reqHnd->hndState = IO_REQUEST_FORCE_EXIT;

    if (reqHnd->readReqSem)
    {
        zOss_PutSemaphore(reqHnd->readReqSem);
    }
    if (reqHnd->writeReqSem)
    {
        zOss_PutSemaphore(reqHnd->writeReqSem);
    }

    return ;
}
/*******************************************************************************
 * Function: IORequest_CancelExit
 * Description: cancel exit mode, alloc higher process enter iorequest module.
 * Input:
 * Output:None
 *
 * Returns:
 * Others:
 ********************************************************************************/
VOID IORequest_CancelExit(T_IO_RequestHnd *reqHnd)
{

    if (!reqHnd || reqHnd->isSuspend  || reqHnd->hndState != IO_REQUEST_FORCE_EXIT)
    {
        zDrv_ASSERT(0);
        return ;
    }

    reqHnd->hndState = IO_REQUEST_CREATED;
    return ;
}
/*******************************************************************************
 * Function: IORequest_Destroy
 * Description:release the IO Request Handle.
 * Input:
 * Output:None
 *
 * Returns:

 * Others:
 	ע:ر֤û߳ʹIORequestĽӿڡ
 	߳ʹIORequestӿڣʹForceExitǿ߳˳
 	߳ȫ˳ٵDestroyӿڽ١
 ********************************************************************************/
VOID IORequest_Destroy(T_IO_RequestHnd *reqHnd)
{
    SINT32 ret = DRV_SUCCESS;

    if (!reqHnd || reqHnd->isSuspend )
    {
        zDrv_ASSERT(0);
        return ;
    }

    /*disable hal operations*/
    if (reqHnd->ops && reqHnd->ops->hal_disable)
    {
        ret = reqHnd->ops->hal_disable(reqHnd->reqHndData);
        zDrv_ASSERT(ret == DRV_SUCCESS);
    }

    reqHnd->halEnabled = FALSE;

    /*free queue resource*/
    free_AllRequest(reqHnd);

    /*free memory resource*/
    if (reqHnd->readQ)
        requestQueue_Destroy(reqHnd->readQ);
    if (reqHnd->readDoneBuffer)
        ringQueue_Destroy(reqHnd->readDoneBuffer);
    if (reqHnd->readReqSem)
        zOss_DeleteSemaphore(reqHnd->readReqSem);
    if (reqHnd->readSgList.reqBuffer)
        zOss_Free(reqHnd->readSgList.reqBuffer);

    if (reqHnd->writeQ)
        requestQueue_Destroy(reqHnd->writeQ);
    if (reqHnd->writeReqSem)
        zOss_DeleteSemaphore(reqHnd->writeReqSem);
    if (reqHnd->writeSgList.reqBuffer)
        zOss_Free(reqHnd->writeSgList.reqBuffer);

    list_del(&reqHnd->node);

    reqHnd->hndState = IO_REQUEST_UNCREATED;

    zOss_Free(reqHnd);

    return ;
}
/*******************************************************************************
 * Function: IORequest_Print
 * Description:print the debug info.
 * Input:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
VOID IORequest_Print(T_IO_RequestHnd *reqHnd)
{
    UINT32 reqCount = 0;

    if (!reqHnd)
        return ;

    zDrvDebug_Printf("----------IO REQUEST DEBUG INFO (%s)----------", reqHnd->name);

    if (reqHnd->reqDirection == IO_REQUEST_READ)
    {
        zDrvDebug_Printf("IO direciton: read only");
    }
    else if (reqHnd->reqDirection == IO_REQUEST_WRITE)
    {
        zDrvDebug_Printf("IO direciton: write only");
    }
    else if (reqHnd->reqDirection == IO_REQUEST_READ_WRITE)
    {
        zDrvDebug_Printf("IO direciton: read and write");
    }
    else
    {
        zDrvDebug_Printf("IO direciton: error");
    }

    zDrvDebug_Printf("IO request size: %d", reqHnd->reqSize);

    if (reqHnd->hndState == IO_REQUEST_CREATED)
    {
        if (reqHnd->readReqCount)
        {
            zDrvDebug_Printf( "[ IO Read Info ]");
            zDrvDebug_Printf( "Request Max Count: %d", reqHnd->readReqCount);
            zDrvDebug_Printf( "Request State: %d", reqHnd->readReqState);
            zDrvDebug_Printf( "Read Request Semaphore Count: %d,", zOss_GetSemaphoreCount(reqHnd->readReqSem));
            reqCount = (reqHnd->readQ->queue->write_pos + reqHnd->readQ->queue->unit_buffer_size - reqHnd->readQ->queue->read_pos) % reqHnd->readQ->queue->unit_buffer_size;
            reqCount = reqCount / reqHnd->readQ->queue->unit_size;
            zDrvDebug_Printf( "Read Request Queue Count: %d,", reqCount);
            reqCount = (reqHnd->readDoneBuffer->write_pos + reqHnd->readDoneBuffer->unit_buffer_size - reqHnd->readDoneBuffer->read_pos) % reqHnd->readDoneBuffer->unit_buffer_size;
            reqCount = reqCount / reqHnd->readDoneBuffer->unit_size;
            zDrvDebug_Printf( "Read Done Buffer Count: %d", reqCount);
        }

        if (reqHnd->writeReqCount)
        {
            zDrvDebug_Printf( "[ IO Write Info ]");
            zDrvDebug_Printf( "Request Max Count: %d", reqHnd->writeReqCount);
            zDrvDebug_Printf( "Request State: %d ", reqHnd->writeReqState);
            zDrvDebug_Printf( "Write Request Semaphore Count: %d,", zOss_GetSemaphoreCount(reqHnd->writeReqSem));
            reqCount = (reqHnd->writeQ->queue->write_pos + reqHnd->writeQ->queue->unit_buffer_size - reqHnd->writeQ->queue->read_pos) % reqHnd->writeQ->queue->unit_buffer_size;
            reqCount = reqCount / reqHnd->writeQ->queue->unit_size;
            zDrvDebug_Printf( "Write Request Queue Count: %d", reqCount);
        }
    }



}

/*******************************************************************************
 * Function: IORequest_PrintAll
 * Description:print the debug info.
 * Input:
 * Output:None
 *
 * Returns:

 * Others:
 ********************************************************************************/
VOID IORequest_PrintAll(VOID)
{
    struct list_head *_reqhnd, *_next_reqhnd;
    T_IO_RequestHnd *reqHnd;

    if (list_empty(&g_IORequestHnd))
    {
        zDrvDebug_Printf( "IO Request: no request handle to print !!!");
        return ;
    }

    list_for_each_safe(_reqhnd, _next_reqhnd, &g_IORequestHnd)
    {
        reqHnd = list_entry(_reqhnd, T_IO_RequestHnd, node);
        IORequest_Print(reqHnd);
    }

}

