/*
 * zx29_pcm.c  --  ZX29_PCM ALSA SoC Audio platform driver
 *
 * Copyright (C) 2017, ZTE Corporation.
 *
 * Based on dma.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/slab.h>
#include <linux/dma-mapping.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <sound/soc.h>
#include <sound/pcm_params.h>

#include <asm/dma.h>

#include <mach/clk.h>
#include <mach/gpio.h>
#include <mach/dma.h>

#include <linux/dmaengine.h>
#include <linux/scatterlist.h>
#include <mach/clk.h>
#include <mach/iomap.h>
#include "dma.h"
#include "i2s.h"

#include <linux/kthread.h>
#include <linux/semaphore.h>

#define ST_RUNNING		(1<<0)
#define ST_OPENED		(1<<1)

static const struct snd_pcm_hardware dma_hardware = {
	.info			= SNDRV_PCM_INFO_INTERLEAVED |
	SNDRV_PCM_INFO_BLOCK_TRANSFER |
	SNDRV_PCM_INFO_MMAP |
	SNDRV_PCM_INFO_MMAP_VALID,
	.formats		= SNDRV_PCM_FMTBIT_S16_LE |
	SNDRV_PCM_FMTBIT_U16_LE | SNDRV_PCM_FMTBIT_S24_LE | 
	SNDRV_PCM_FMTBIT_S32_LE,
	.channels_min		= 1,
	.channels_max		= 2,
	.buffer_bytes_max	= 9 * 1024,
	.period_bytes_min	= 32,     //PAGE_SIZE
	.period_bytes_max	= 3*1024, //PAGE_SIZE * 2
	.periods_min		= 2,
	.periods_max		= 3,
	.fifo_size		= 32,
};

struct runtime_data {
	spinlock_t lock;
	int state;
	unsigned int dma_loaded;
	unsigned int dma_period;
	dma_addr_t dma_start;
	dma_addr_t dma_pos;
	dma_addr_t dma_end;
	struct zx29_dma_params *params;
};


static int zx29_dma_trigger(struct snd_pcm_substream *substream, int cmd);
static int zx29_dma_prepare(struct snd_pcm_substream *substream);
static int zx29_preallocate_dma_buffer(struct snd_pcm *pcm, int stream);
extern SINT32 zDrv_Audio_Printf(VOID *pFormat, ...);

#ifdef CONFIG_PREEMPT_RT_FULL

static struct task_struct *pcm_pb_thread = NULL, *pcm_ca_thread = NULL;
static struct semaphore pcm_pb_complete, pcm_ca_complete;

static inline void zx29_pcm_set_sched_params(void)
{
	struct sched_param param = { .sched_priority = 40 };
	sched_setscheduler(current, SCHED_FIFO, &param);
}

static int zx29_pcm_pb_thread(void *arg)
{
	struct snd_pcm_substream *substream = (struct snd_pcm_substream *)arg;
	zx29_pcm_set_sched_params();
	
	while(1) {
		down(&pcm_pb_complete);
		if (substream)
			snd_pcm_period_elapsed(substream);
	}
}

static int zx29_pcm_ca_thread(void *arg)
{
	struct snd_pcm_substream *substream = (struct snd_pcm_substream *)arg;
	zx29_pcm_set_sched_params();

	while(1) {
		down(&pcm_ca_complete);
		if (substream)
			snd_pcm_period_elapsed(substream);
	}
}

#endif

static void audio_buffdone(void *data)
{
	struct snd_pcm_substream *substream = data;
	struct runtime_data *prtd = substream->runtime->private_data;

	if (prtd->state & ST_RUNNING) {
		prtd->dma_pos += prtd->dma_period;
		if (prtd->dma_pos >= prtd->dma_end)
			prtd->dma_pos = prtd->dma_start;
#ifdef CONFIG_PREEMPT_RT_FULL
		if (substream->stream == SNDRV_PCM_STREAM_CAPTURE)
			up(&pcm_ca_complete);
		else if(substream->stream == SNDRV_PCM_STREAM_PLAYBACK) 
			up(&pcm_pb_complete);
#else
		if (substream)
			snd_pcm_period_elapsed(substream);
#endif

	}
}

static int zx29_dma_hw_params(struct snd_pcm_substream *substream,
                              struct snd_pcm_hw_params *params)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct runtime_data *prtd = runtime->private_data;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	unsigned long totbytes = params_buffer_bytes(params);
	struct zx29_dma_params *dma =
	    snd_soc_dai_get_dma_data(rtd->cpu_dai, substream);
       unsigned int  periodsize = params_period_size(params);

	dma_cap_mask_t mask;
//	print_audio("Alsa Entered func %s\n", __func__);
	CPPS_FUNC(cpps_callbacks, zDrv_Audio_Printf)("Alsa snd_pcm_hardware buffer_bytes_max=%d,period_bytes_max=%d,periods_max=%d\n", dma_hardware.buffer_bytes_max,dma_hardware.period_bytes_max,dma_hardware.periods_max);

	dma_cap_zero(mask);
	dma_cap_set(DMA_CYCLIC, mask);
//	dma_cap_set(DMA_SLAVE, mask);

	CPPS_FUNC(cpps_callbacks, zDrv_Audio_Printf)("Alsa zx29_dma_hw_params periodsize=%d\n", periodsize);
	if (!dma)
		return 0;

	prtd->params = dma;

	prtd->params->ch = dma_request_channel(mask, zx29_dma_filter_fn, (void*)(prtd->params->channel));
	if (!(prtd->params->ch)){
		//printk(KERN_ERR "Alsa: zx29_dma_hw_params dma request chan failed!\n");
	      CPPS_FUNC(cpps_callbacks, zDrv_Audio_Printf)("Alsa: zx29_dma_hw_params dma request chan failed!\n");
		return -1;
	}

	snd_pcm_set_runtime_buffer(substream, &substream->dma_buffer);

	runtime->dma_bytes = totbytes;

	spin_lock_irq(&prtd->lock);
	prtd->dma_loaded = 0;
	prtd->dma_period = params_period_bytes(params);
	prtd->dma_start = runtime->dma_addr;
	prtd->dma_pos = prtd->dma_start;
	prtd->dma_end = prtd->dma_start + totbytes;
	spin_unlock_irq(&prtd->lock);

	//print_audio("Alsa zx29_dma_hw_params buf_len=%d,dma_period=%d\n", totbytes, prtd->dma_period);
	CPPS_FUNC(cpps_callbacks, zDrv_Audio_Printf)("Alsa zx29_dma_hw_params totbytes=%d,dma_period=%d,dma_start=0x%x\n", totbytes, prtd->dma_period,prtd->dma_start);

	return 0;
}

static int zx29_dma_hw_free(struct snd_pcm_substream *substream)
{
	struct runtime_data *prtd = substream->runtime->private_data;
//	print_audio("Alsa Entered func %s\n", __func__);

//	pr_debug("Entered %s\n", __func__);

	snd_pcm_set_runtime_buffer(substream, NULL);
	if (prtd->params)
		if(prtd->params->ch)
			dma_release_channel(prtd->params->ch);
	return 0;
}

static int zx29_dma_prepare(struct snd_pcm_substream *substream)
{
	struct runtime_data *prtd = substream->runtime->private_data;
	struct dma_async_tx_descriptor *desc = NULL;
	struct zx29_dma_channel * zx29_chan = NULL;
	int ret = 0;
	int direction =
	    (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
	     ? DMA_MEM_TO_DEV : DMA_DEV_TO_MEM);
	unsigned int limit, buf_len;
//	print_audio("Alsa Entered func %s\n", __func__);
//	pr_debug("Entered %s\n", __func__);

	limit = (prtd->dma_end - prtd->dma_start) / prtd->dma_period;
	buf_len = limit * prtd->dma_period;
	//pr_debug("Alsa %s: loaded %d, limit %d\n", __func__, prtd->dma_loaded, limit);
	
	CPPS_FUNC(cpps_callbacks, zDrv_Audio_Printf)("Alsa zx29_dma_prepare loaded %d, limit %d\n",  prtd->dma_loaded, limit);

	if (!prtd->params)
		return -1;

	prtd->dma_loaded = 0;
	prtd->dma_pos = prtd->dma_start;
//	print_audio("Alsa zx29_dma_prepare limit=%d,buf_len=%d,dma_period=%d\n", limit, buf_len, prtd->dma_period);

	dma_channel_def temp[5];
	int 	i;
	memset(temp, 0, sizeof(temp));
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		for (i = 0; i < limit; i++) {
			temp[i].src_addr 	= prtd->dma_start + i * (prtd->dma_period);
			temp[i].dest_addr	= prtd->params->dma_addr;
			temp[i].dma_control.tran_mode = TRAN_MEM_TO_PERI;
			temp[i].dma_control.src_burst_len 	= DMA_BURST_LEN_16;
			temp[i].count		= prtd->dma_period;
			//		temp[i].callback	= (dma_callback_func)dma_cb;
			/*
				temp[i].dma_control.tran_mode 		= (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
														? TRAN_MEM_TO_PERI : TRAN_PERI_TO_MEM);
				temp[i].dma_control.src_burst_len 	= (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
														? DMA_BURST_LEN_16 : DMA_BURST_LEN_8);
			*/
			temp[i].dma_control.src_burst_size 	= DMA_BURST_SIZE_32BIT;
			temp[i].dma_control.dest_burst_size = DMA_BURST_SIZE_32BIT;
			temp[i].dma_control.dest_burst_len 	= DMA_BURST_LEN_16;
			temp[i].dma_control.irq_mode 		= DMA_ALL_IRQ_ENABLE;
			temp[i].link_addr 		= 1;
		}
	} else {
		for (i = 0; i < limit; i++) {
			temp[i].src_addr 	= prtd->params->dma_addr;
			temp[i].dest_addr	= prtd->dma_start + i * (prtd->dma_period);
			temp[i].dma_control.tran_mode = TRAN_PERI_TO_MEM;
			temp[i].dma_control.src_burst_len 	= DMA_BURST_LEN_16;

			temp[i].count		= prtd->dma_period;
//		temp[i].callback	= (dma_callback_func)dma_cb;

			temp[i].dma_control.src_burst_size 	= DMA_BURST_SIZE_32BIT;
			temp[i].dma_control.dest_burst_size = DMA_BURST_SIZE_32BIT;
			temp[i].dma_control.dest_burst_len 	= DMA_BURST_LEN_16;
			temp[i].dma_control.irq_mode 		= DMA_ALL_IRQ_ENABLE;

			temp[i].link_addr 		= 1;
		}
	}
	temp[limit - 1].link_addr 	= 0;
//	print_audio("Alsa zx29_dma_prepare 1 ch=%x\n", prtd->params->ch);

	ret = dmaengine_slave_config(prtd->params->ch, (struct dma_slave_config*)&temp);

//	desc = prtd->params->ch->device->device_prep_interleaved_dma(prtd->params->ch, NULL, 0);

	desc = dmaengine_prep_dma_cyclic(
		prtd->params->ch, prtd->dma_start, buf_len,
		prtd->dma_period, direction);

	desc->callback = (dma_async_tx_callback)audio_buffdone;
	desc->callback_param = substream;

	dmaengine_submit(desc);

	return ret;
}

static int zx29_dma_trigger(struct snd_pcm_substream *substream, int cmd)
{
	struct runtime_data *prtd = substream->runtime->private_data;
	int ret = 0;
//	print_audio("Alsa Entered func %s, cmd=%d\n", __func__, cmd);

//	pr_debug("Entered %s\n", __func__);

	spin_lock(&prtd->lock);

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		prtd->state |= ST_RUNNING;
		dma_async_issue_pending(prtd->params->ch);
		break;

	case SNDRV_PCM_TRIGGER_STOP:
		prtd->state &= ~ST_RUNNING;
		dmaengine_terminate_all(prtd->params->ch);
		break;
//	case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
//		dmaengine_pause(prtd->params->ch);
//		break;
//	case SNDRV_PCM_TRIGGER_RESUME:
//		dmaengine_resume(prtd->params->ch);
//		break;
	default:
		ret = -EINVAL;
		break;
	}

	spin_unlock(&prtd->lock);

	return ret;
}

static snd_pcm_uframes_t
zx29_dma_pointer(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct runtime_data *prtd = runtime->private_data;
	unsigned long res;

//	pr_debug("Entered %s\n", __func__);

	res = prtd->dma_pos - prtd->dma_start;


	if (res >= snd_pcm_lib_buffer_bytes(substream)) {
		if (res == snd_pcm_lib_buffer_bytes(substream))
			res = 0;
	}

	return bytes_to_frames(substream->runtime, res);
}

static int zx29_dma_open(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct runtime_data *prtd;
	int ret = 0;
	
	struct snd_pcm *pcm = substream->pcm;
//	print_audio("Alsa Entered func %s\n", __func__);

	pr_debug("Alsa Entered %s\n", __func__);

	snd_pcm_hw_constraint_integer(runtime, SNDRV_PCM_HW_PARAM_PERIODS);
	snd_soc_set_runtime_hwparams(substream, &dma_hardware);

	prtd = kzalloc(sizeof(struct runtime_data), GFP_KERNEL);
	if (prtd == NULL)
		return -ENOMEM;

	spin_lock_init(&prtd->lock);

	runtime->private_data = prtd;
	ret = zx29_preallocate_dma_buffer(pcm, substream->stream );
       if(ret != 0){
		kfree(prtd);
		prtd = NULL;
	       CPPS_FUNC(cpps_callbacks, zDrv_Audio_Printf)("Alsa zx29_dma_open zx29_preallocate_dma_buffer fail\n" );		
		return -ENOMEM;
       }
	
	CPPS_FUNC(cpps_callbacks, zDrv_Audio_Printf)("Alsa zx29_dma_open zx29_preallocate_dma_buffer ret=%d, direct=%d \n",ret,substream->stream );
	return 0;
}

static int zx29_dma_close(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct runtime_data *prtd = runtime->private_data;
	struct snd_pcm *pcm = substream->pcm;
	struct snd_dma_buffer *buf;
	int stream;	
//	print_audio("Alsa Entered func %s\n", __func__);

	pr_debug("Alsa Entered %s\n", __func__);

	buf = &substream->dma_buffer;
	if (buf->area){
		dma_free_writecombine(pcm->card->dev, buf->bytes,buf->area, buf->addr);
		buf->area = NULL;
	       CPPS_FUNC(cpps_callbacks, zDrv_Audio_Printf)("Alsa zx29_dma_close dma_free_writecombine \n" );
		
	}
	if (!prtd){
		pr_debug("Alsa zx29_dma_close called with prtd == NULL\n");
	       return ENXIO;		
	}
	//dma_release_channel(prtd->params->ch);
	kfree(prtd);

	return 0;
}

static int zx29_dma_mmap(struct snd_pcm_substream *substream,
                         struct vm_area_struct *vma)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
//	print_audio("Alsa Entered func %s\n", __func__);

	pr_debug("Alsa Entered %s\n", __func__);

	return dma_mmap_writecombine(substream->pcm->card->dev, vma,
	                             runtime->dma_area,
	                             runtime->dma_addr,
	                             runtime->dma_bytes);
}

static struct snd_pcm_ops dma_ops = {
	.open		= zx29_dma_open,
	.close		= zx29_dma_close,
	.ioctl		= snd_pcm_lib_ioctl,
	.hw_params	= zx29_dma_hw_params,
	.hw_free	= zx29_dma_hw_free,
	.prepare	= zx29_dma_prepare,
	.trigger	= zx29_dma_trigger,
	.pointer	= zx29_dma_pointer,
	.mmap		= zx29_dma_mmap,
};

static int zx29_preallocate_dma_buffer(struct snd_pcm *pcm, int stream)
{
	struct snd_pcm_substream *substream = pcm->streams[stream].substream;
	struct snd_dma_buffer *buf = &substream->dma_buffer;
	size_t size = dma_hardware.buffer_bytes_max;
//	print_audio("Alsa Entered func %s\n", __func__);

	pr_debug("Alsa Entered %s\n", __func__);

	buf->dev.type = SNDRV_DMA_TYPE_DEV;
	buf->dev.dev = pcm->card->dev;
	buf->private_data = NULL;
	buf->area = dma_alloc_writecombine(pcm->card->dev, size,
	                                   &buf->addr, GFP_KERNEL);
	if (!buf->area)
		return -ENOMEM;
	buf->bytes = size;

	return 0;
}

static void zx29_dma_free_dma_buffers(struct snd_pcm *pcm)
{
	struct snd_pcm_substream *substream;
	struct snd_dma_buffer *buf;
	int stream;
//	print_audio("Alsa Entered func %s\n", __func__);
	pr_debug("Alsa Entered %s\n", __func__);
#if 0
	for (stream = 0; stream < 2; stream++) {
		substream = pcm->streams[stream].substream;
		if (!substream)
			continue;

		buf = &substream->dma_buffer;
		if (!buf->area)
			continue;

		dma_free_writecombine(pcm->card->dev, buf->bytes,
		                      buf->area, buf->addr);
		buf->area = NULL;
	}
#endif	
	if (pcm_pb_thread)
	{
		kthread_stop(pcm_pb_thread);
		pcm_pb_thread = NULL;
	}

	if (pcm_ca_thread)
	{
		kthread_stop(pcm_ca_thread);
		pcm_ca_thread = NULL;
	}
}

static u64 dma_mask = DMA_BIT_MASK(32);

static int zx29_dma_new(struct snd_soc_pcm_runtime *rtd)
{
	struct snd_card *card = rtd->card->snd_card;
	struct snd_pcm *pcm = rtd->pcm;
	int ret = 0;
//	print_audio("Alsa Entered func %s\n", __func__);
	struct snd_pcm_substream *substream;

	pr_debug("Alsa Entered %s\n", __func__);

	if (!card->dev->dma_mask)
		card->dev->dma_mask = &dma_mask;
	if (!card->dev->coherent_dma_mask)
		card->dev->coherent_dma_mask = DMA_BIT_MASK(32);

	if (pcm->streams[SNDRV_PCM_STREAM_PLAYBACK].substream) {
		#if 0
		ret = zx29_preallocate_dma_buffer(pcm,
		                                  SNDRV_PCM_STREAM_PLAYBACK);
		if (ret)
			goto out;
		#endif
	#ifdef CONFIG_PREEMPT_RT_FULL
		substream = pcm->streams[SNDRV_PCM_STREAM_PLAYBACK].substream;
		sema_init(&pcm_pb_complete, 0);

		pcm_pb_thread = kthread_run(zx29_pcm_pb_thread, substream, "pcm_pb_thread");
		if (IS_ERR(pcm_pb_thread)) {
			printk("Alsa: pcm_pb_thread create failed!\n");
			pcm_pb_thread = NULL;
			return -1;
		}
	#endif
	}

	if (pcm->streams[SNDRV_PCM_STREAM_CAPTURE].substream) {
		#if 0
		ret = zx29_preallocate_dma_buffer(pcm,
		                                  SNDRV_PCM_STREAM_CAPTURE);
		if (ret)
			goto out;
		#endif
	#ifdef CONFIG_PREEMPT_RT_FULL
		substream = pcm->streams[SNDRV_PCM_STREAM_CAPTURE].substream;

		sema_init(&pcm_ca_complete, 0);

		pcm_ca_thread = kthread_run(zx29_pcm_ca_thread, substream, "pcm_ca_thread");
		if (IS_ERR(pcm_ca_thread)) {
			printk("Alsa: pcm_ca_thread create failed!\n");
			pcm_ca_thread = NULL;
			return -1;
		}
		
	#endif
	}
out:
	return ret;
}

static struct snd_soc_platform_driver zx_asoc_platform = {
	.ops		= &dma_ops,
	.pcm_new	= zx29_dma_new,
	.pcm_free	= zx29_dma_free_dma_buffers,
};

static int __devinit zx29_asoc_platform_probe(struct platform_device *pdev)
{
//	print_audio("Alsa zx29_asoc_platform_probe start\n");
	return snd_soc_register_platform(&pdev->dev, &zx_asoc_platform);
}

static int __devexit zx29_asoc_platform_remove(struct platform_device *pdev)
{
	snd_soc_unregister_platform(&pdev->dev);
	return 0;
}

static struct platform_driver asoc_dma_driver = {
	.driver = {
		.name = "zx29-pcm-audio",
		.owner = THIS_MODULE,
	},

	.probe = zx29_asoc_platform_probe,
	.remove = __devexit_p(zx29_asoc_platform_remove),
};

module_platform_driver(asoc_dma_driver);

MODULE_DESCRIPTION("zx29 ASoC DMA Driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:zx29-audio");
