/**
* @File: sanichips_dm.c
* @Brief: Implementation of Device Manager in Sanechips
*
* Copyright (C) 2017 Sanechips Technology Co., Ltd.
* @Author: chen tianming <chen.tianming@sanechips.com.cn>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License version 2 as
* published by the Free Software Foundation.
* */

/*******************************************************************************
 *						    ͷļ   	          					       *
 ******************************************************************************/

#include <pthread.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/prctl.h>
#include "rtc_timer.h"
#include <stdio.h>
#include <error.h>
#include <unistd.h>

#include "cfg_nv_def.h"

#include "softap_api.h"
#include "unicom_dm.h"
#include "mobile_dm.h"
#include "telecom_dm.h"
/*******************************************************************************
 *                             궨                                         *
 ******************************************************************************/
#define  DM_NV_ITEM_SIZE						   64
#define  DM_INFO_MAX_SIZE                          64
#define  DM_INTERNAL_MSG_SIZE                      128
#define  DM_MSG_CMD_REGISTER_REQUEST               0x1000
#define  DM_ROUND_TOTAL						       10
#define  DM_MSG_CMD_REGISTER_THREAD_RELEASE		   0x1001
#define  DM_MSG_CMD_MAIN_THREAD_EXIT               0x1002
#define  DM_TELECOM_RETRY_TIMES                    10
#define  DM_TELECOM_RETRYTIMES_BEFORE_ACTIVED      30
/*NV name*/
#define  DM_REGISTERED_ROUND					   "registeredRound"
#define  DM_SECS_EVERY_ROUND				       "secsEveryRound"
#define  DM_SECS_EVERY_TIME						   "secsEveryTime"
#define  DM_SEC_TIMER                              "secsTime"
#define  DM_REGISTER_FLAG						   "registerFlag"
#define  DM_REGISTER_TYPE						   "registerDmType"

#define  DM_APP_KEY_MOBILE						   "appKeyMobile"
#define  DM_HOST_NAME					   		   "hostName"
#define  DM_PORT_NUM					   		   "portNum"

#define  DM_MANUF								   "Manuf"
#define  DM_MODEL								   "Model"
#define  DM_HW_VERSION							   "hw_version"
#define  DM_SW_VERSION							   "wa_version"
#define  DM_OS									   "OS"
#define  DM_OS_VERSION							   "OSVersion"
#define  DM_IN_TYPE								   "Intype"
#define  DM_BRAND								   "Brand"

#define  DM_WCDMA_ONLY							   "wcdmaModel"
#define  DM_URL_MOBILE							   "urlMobile"

#define  DM_IMEI_PREVIOUS						   "imeiPrevious"
#define  DM_ICCID_PREVIOUS						   "iccidPrevious"
#define  DM_VERSION_PREVIOU						   "versionPrevious"
#define  DM_SIM_CARD_STATUS						   "creg_stat"
#define  DM_CEREG_STATUS						   "cereg_stat"

#define  DM_REGVER                                 "regver"
#define  DM_MEID                                   "meid"
#define  DM_UETYPE                                 "uetype"

#define  DM_SYSTEM_LOCK_FILE_PATH          		   "/sys/power/wake_lock"
#define  DM_SYSTEM_UNLOCK_FILE_PATH       		   "/sys/power/wake_unlock"


#define log(...)            do { printf(__VA_ARGS__); fflush(stdout); } while(0)
#define log_err(...)        do { log("DM-ERR NOW: FILE(%s) ,LINE(%d) >>>>: ", __FILE__, __LINE__ );  printf(__VA_ARGS__); fflush(stdout);} while(0)
#ifndef __DM_NO_EBUG__
#define log_debug(...)      do { log("dm-debug now: file(%s) ,line(%d) >>>>: ", __FILE__, __LINE__); printf(__VA_ARGS__); fflush(stdout);} while(0)
#else
#define log_debug(...)
#endif

#define CALL(X) 		 do { int t_ret = (X); if (t_ret != DM_ERR_SUCCESS) { log_err("Get t_ret(%d)\n", t_ret); return t_ret;} } while(0)
#define CHECK_PARAM(X)   do { if(!(X)) {log_err("the param is null\n");  return DM_ERR_PARAM_NULL;}							    } while(0)

//  tap_log(LOG_ERR,MMI_PRINT,"Function rpc_AddObject: %s", ERROR1);
/*******************************************************************************
 *                             Ͷ                                    *
 ******************************************************************************/

enum
{
    DM_ERR_SUCCESS,
    DM_ERR_PARAM_NULL,
    DM_ERR_CREATE_THERAD_ERROR,
    DM_ERR_CREATE_MSG_QUEUE_ERROR,
    DM_ERR_DEVICE_TYPE_IS_NONE,
    DM_ERR_CREATE_TIMER_ERROR,
    DM_ERR_DEVICE_REGISTERED,
    DM_ERR_ADD_TIMER_ERROR,
    DM_ERR_SET_DOMAIN_ERR,
    DM_ERR_WAITING_NET_STATUS_CHANAGE,
    DM_ERR_QUEUE_SEND_ERROR,
    DM_ERR_CHECK_ROUTE_ERROR,
    DM_ERR_IPC_SEND_MESSAGE_ERROR,
    DM_ERR_CREATE_JSON_INFO_ERROR,
    DM_ERR_URL_INFO_CREATE_ERROR,
    DM_ERR_MOBILE_OVER_RETRY_TMIES_ERROR,
    DM_ERR_SERVER_ERROR,
    DM_ERR_SET_TIME_ROUND_ERROR,
    DM_ERR_NET_ERROR,
    DM_ERR_RETRY_ERROR,
    DM_ERR_NO_MOBILE_CARD_ERROR,
    DM_ERR_ROMAING_STATUS_ERROR,
};

/* DM ʱID*/
enum
{
    DM_TIMER_ID_UNICOM_START,
    DM_TIMER_ID_MOBILE_START,
    DM_TIMER_ID_UNICOM_EVERY_TIMES,
    DM_TIMER_ID_UNICOM_EVERY_ROUND,
    DM_TIMER_ID_MOBILE_EVERY_TIMES,
    DM_TIMER_ID_MAX,
};

/*DM ע */
enum
{
    DM_REGISTERED_NO,
    DM_REGISTERED_YES,
};

/* ǰնʽ */
enum  DM_TYPE
{
    DM_NO_ONE,
    DM_MOBILE,
    DM_UNICOM,
    DM_TELECOM,
};

/* ͨϢṹ */
enum DM_UNICOM_INFO_INDEX
{
    DM_INFO_MANUF = 0,
    DM_INFO_MODEL,
    DM_INFO_HWVISION,
    DM_INFO_SWVERSION,
    DM_INFO_OS,
    DM_INFO_OSVISION,
    DM_INFO_IMEI1,
    DM_INFO_ICCID1,
    DM_INFO_IMSI1,
    DM_INFO_MSISDN1,
    DM_INFO_MNC1,
    DM_INFO_NCLS1,
    DM_INFO_CELLID1,
    DM_INFO_LAC1,
    DM_INFO_ACCTYPE,
    DM_INFO_TIME,
    DM_INFO_TAG,
    DM_INFO_UNICOM_MAX,
};

/*struct DM_UNICOM - עϢṹ */
struct DM_UNICOM_REGISTER_INFO
{
    char value[DM_INFO_UNICOM_MAX][DM_INFO_MAX_SIZE];
};

/* ƶDMϢṹ */
typedef enum
{
    DM_INFO_MOBILE_SDKVERSION = 0,
    DM_INFO_MOBILE_IMEI1,
    DM_INFO_MOBILE_IMEI2,
    DM_INFO_MOBILE_MEID,
    DM_INFO_MOBILE_DEVICE_ID,
    DM_INFO_MOBILE_BRAND,
    DM_INFO_MOBILE_MODEL,
    DM_INFO_MOBILE_FIRMWARE_VERSION,
    DM_INFO_MOBILE_SYSTEM_VERSION,
    DM_INFO_MOBILE_CPU,
    DM_INFO_MOBILE_ROM,
    DM_INFO_MOBILE_RAM,
    DM_INFO_MOBILE_TYPE,
    DM_INFO_MOBILE_ICCID1,
    DM_INFO_MOBILE_ICCID2,
    DM_INFO_MOBILE_IMSI1,
    DM_INFO_MOBILE_IMSI2,
    DM_INFO_MOBILE_MAC,
    DM_INFO_MOBILE_CELLID,
    DM_INFO_MOBILE_LAC,
    DM_INFO_MOBILE_CHNL,
    DM_INFO_MOBILE_DATA_CARD,
    DM_INFO_MOBILE_MASTER_STATUS,
    DM_INFO_MOBILE_SEND_TIME,
    DM_INFO_MOBILE_SLOT_QUANTITY,
    DM_INFO_MOBILE_DATA_CARD2,
    DM_INFO_MOBILE_SLOT_SERVICE1,
    DM_INFO_MOBILE_SLOT_SERVICE2,
    DM_INFO_MOBILE_SLOT_NETWORK1,
    DM_INFO_MOBILE_SLOT_NETWORK2,
    DM_INFO_MOBILE_LAC2,
    DM_INFO_MOBILE_CELLID2,
    DM_INFO_MOBILE_TERMINAL_INTTYPE,
    DM_INFO_MOBILE_MAX
} DM_INFO_MOBILE_INDEX;

/*struct DM_MOBILE_INFO ƶעϢṹ */
struct  DM_MOBILE_REGISTER_INFO
{
    char value[DM_INFO_MOBILE_MAX][DM_INFO_MAX_SIZE];
};

/* Ϣṹ */
enum DM_TELECOM_INFO_INDEX
{
    DM_INFO_REGVER = 0,
    DM_INFO_MEID,
    DM_INFO_MODELSMS,
    DM_INFO_SWVER,
    DM_INFO_SIM1CDMAIMSI,
    DM_INFO_UETYPE,
    DM_INFO_SIM1ICCID,
    DM_INFO_SIM1LTEIMSI,
    DM_INFO_IMEI,
    DM_INFO_BASEID1,
    DM_INFO_SIM1TYPE,
    DM_INFO_SID,
    DM_INFO_NID,
    DM_INFO_TELECOM_MAX,
};

/*struct DM_TELECOM - עϢṹ */
struct DM_TELECOM_REGISTER_INFO
{
    char value[DM_INFO_TELECOM_MAX][DM_INFO_MAX_SIZE];
};

/**
 * @brief struct DM_CONTROl_INFO  DMģڲϢ
 * @param DM_TYPE:dm  moblie unicom telecom
 * @param registErThreadFlag:ע̱߳ǣע˳߳
 * @param mainThreadFlag:̱߳ǣע˳߳
 * @param secTimer: dmʱʱ
 * @param pid:̵߳ID
 * @param serverPid:߳ID
 * @param msgQueueId:ϢеID
 * @param msgQueueIdReg:ע߳ϢID
 * @param registerFlag:ע־ǷѾע
 * @param secsEveryRound:ÿּʱ
 * @param secsEveryTime:ÿμʱ
 * @param registerTimesEveryRound:ÿֳԵĴ
 * @param registerRoundsCurrent:ǰԵ
 * @param registeredRounds:ѾԵ
 * @param currentRouds:ǰע
 * @param currentTimes:ǰעĴ
 * @param netStatus:ǵǰ״̬
 * @param imeiPrevious:һαimei
 * @param iccidprevious:һαiccid
 * @param hostName:
 * @param ipAddr:ַ
 * @param portNum:˿
 * @note
 * @warning
 */
struct DM_CONTROl_INFO
{
    enum  DM_TYPE  dmType;
    int   registErThreadFlag;
    int   mainThreadFlag;
    int   secTimer;
    pthread_t pid;
    pthread_t serverId;
    int   msgQueueId;
    int   msgQueueIdReg;
    int   registerFlag;
    int   secsEveryRound;
    int   secsEveryTime;
    int   registeredRounds;
    int   currentRounds;
    int   currentTimes;
    int   netStatus;
    int   globalRomaing;
    int   cardType;
    int   needRegForNetStatusChange;
    char  hostName[DM_INFO_MAX_SIZE];
    char  imeiPrevious[DM_INFO_MAX_SIZE];
    char  iccidPrevious[DM_INFO_MAX_SIZE];
    char  ipAddr[32];
    unsigned long  portNum;
};

/*******************************************************************************
 *                             ̬                                    *
 ******************************************************************************/
static int g_retrytimes_before_netactived = 0;

/*******************************************************************************
 *                             ̬                                    *
 ******************************************************************************/
static int dm_control_info_init(struct DM_CONTROl_INFO *dmControlInfo);
static int dm_register_thread_init(struct DM_CONTROl_INFO *dmControlInfo);
static int dm_timer_create(struct DM_CONTROl_INFO *dmControlInfo);
static int dm_server_init(struct DM_CONTROl_INFO *dmControlInfo);

static void* dm_server_thread_entry(void* args);
static void dm_register_queue_msg_proc(MSG_BUF *pstMsg, struct DM_CONTROl_INFO *dmControlInfo);

static int dm_get_device_type(struct DM_CONTROl_INFO *dmControlInfo);
static int dm_unicom_control_info_init(struct DM_CONTROl_INFO *dmControlInfo);
static int dm_mobile_control_info_init(struct DM_CONTROl_INFO *dmControlInfo);

static void dm_wake_lock(const char *lockName);
static void dm_wake_unlock(const char *unLockName);
static void dm_write_lockfile(const char *filepath, const char *setbuf);

static int dm_timer_add(unsigned long sec, RTC_ID rtcId);

static void dm_register_flag_clean(struct DM_CONTROl_INFO *dmControlInfo);
static int dm_server_domain_config_set(MSG_BUF *pstMsg, struct DM_CONTROl_INFO *dmControlInfo);
static int dm_net_status_change_proc(struct DM_CONTROl_INFO *dmControlInfo);
static int dm_register_request_handle(struct DM_CONTROl_INFO *dmControlInfo);
static int dm_mobile_register_request_handle(struct DM_CONTROl_INFO *dmControlInfo);
static int dm_unicom_register_request_handle(struct DM_CONTROl_INFO *dmControlInfo);
static int dm_net_global_romaing_proc(struct DM_CONTROl_INFO *dmControlInfo);
static int dm_unicom_register_start(struct DM_CONTROl_INFO *dmControlInfo);

static int dm_register_msg_send(struct DM_CONTROl_INFO *dmControlInfo);
static int dm_unicom_url_info_create(struct DM_CONTROl_INFO *dmControlInfo, struct DM_UNICOM_REGISTER_INFO *dmUnicomRegInfo, DM_UNICOM_HTTP_URL_INFO *dmUnicomUrlInfo);
static int dm_mobile_url_info_create(struct DM_CONTROl_INFO *dmControlInfo, struct DM_MOBILE_REGISTER_INFO *dmMobileRegInfo, DM_MOBILE_HTTPS_URL_INIO *dmMobileUrlInfo);

static int dm_unicom_json_info_create(struct DM_UNICOM_REGISTER_INFO *dmUnicomRegInfo, DM_UNICOM_HTTP_DATA_INFO * httpsDataInfo) ;
static int dm_mobile_json_info_create(struct DM_MOBILE_REGISTER_INFO *dmMobileRegInfo, DM_MOBILE_HTTPS_DATA_INFO * httpsDataInfo);
static int dm_unicom_register_info_create(struct DM_CONTROl_INFO *dmControlInfo, struct DM_UNICOM_REGISTER_INFO *dmUnicomRegInfo);
static int dm_mobile_register_info_create(struct DM_CONTROl_INFO *dmControlInfo, struct DM_MOBILE_REGISTER_INFO *dmMobileRegInfo);
static void dm_resource_clean(struct DM_CONTROl_INFO *dmControlInfo,  void *dataInfo);
static int dm_register_time_set(struct DM_CONTROl_INFO *dmControlInfo, void *dataInfo);
static void dm_register_tag_set(struct DM_CONTROl_INFO *dmControlInfo, struct DM_UNICOM_REGISTER_INFO *dmUnicomRegInfo);
static int dm_mobile_register_start(struct DM_CONTROl_INFO *dmControlInfo);

static int dm_config_info_msg_prc(MSG_BUF *pstMsg, struct DM_CONTROl_INFO *dmControlInfo);
static int dm_time_every_round_set(MSG_BUF *pstMsg, struct DM_CONTROl_INFO *dmControlInfo);
static int dm_server_port_num_set(MSG_BUF *pstMsg, struct DM_CONTROl_INFO *dmControlInfo);
static void dm_data_channel_set(struct DM_MOBILE_REGISTER_INFO *dmMobileRegInfo);
static int dm_register_thread_release(struct DM_CONTROl_INFO *dmControlInfo);
static int  dm_main_thread_release(struct DM_CONTROl_INFO *dmControlInfo);
static void dm_rtc_timer_del();
static int dm_telecom_register_request_handle(struct DM_CONTROl_INFO *dmControlInfo);
BOOL dm_telecom_need_reg_check(struct DM_CONTROl_INFO *dmControlInfo);

/**
 * ƣ dm_get_device_type
 *  ȡǰ豸ͣDM / ƶDM
 * ˵
 *   ֵ ɹ 0: ʧ 
 * ˵
 */
static int dm_get_device_type(struct DM_CONTROl_INFO *dmControlInfo)
{
    char dmType[4] = {0};
    char registerFlag[4] = {0};

    CALL(sc_cfg_get(DM_REGISTER_TYPE, dmType, sizeof(dmType)));
    CALL(sc_cfg_get(DM_REGISTER_FLAG, registerFlag, sizeof(registerFlag)));

    dmControlInfo->dmType = atoi(dmType);
    dmControlInfo->registerFlag = atoi(registerFlag);

    return DM_ERR_SUCCESS;
}

/**
 * ƣ dm_write_lockfile
 *  ļдַ
 * ˵
 *   ֵ
 * ˵
 */
static void dm_write_lockfile(const char *filepath, const char *setbuf)
{
    int fd = -1;
    int len = 0;
    if (!filepath || !setbuf)
    {
        log_err("parm is null \n");
        return;
    }

    fd = open(filepath, O_RDWR);
    if (-1 == fd)
    {
        log_err("open lockfile fail\n");
        return;
    }

    len = strlen(setbuf);
    if (write(fd, setbuf, len) != len)
    {
        log_err("write lockfile fail\n");
    }

    close(fd);
}

/**
 * ƣdm_wake_lock
 * DMģϵͳ˯
 * ˵
 *   ֵ
 * ˵
 */
static void dm_wake_lock(const char *lockName)
{
    dm_write_lockfile(DM_SYSTEM_LOCK_FILE_PATH, lockName);
}

/**
 * ƣdm_wake_lock
 * DMģϵͳ˯߽
 * ˵
 *   ֵ
 * ˵
 */
static void dm_wake_unlock(const char *unLockName)
{
    dm_write_lockfile(DM_SYSTEM_UNLOCK_FILE_PATH, unLockName);
}

/**
 * ƣdm_timer_add
 * RTCʱ
 * ˵sec: 
 *   ֵ
 * ˵ɹ 0: ʧ 
 */
static int dm_timer_add(unsigned long sec, RTC_ID rtcId)
{
    int ret = 0;
    if (0 != (ret = rtc_timer_add(sec, rtcId, MODULE_ID_DMREG)))
    {
        log_err("dm add timer error... ret-value(%d)\n", ret);
        return DM_ERR_ADD_TIMER_ERROR;
    }
    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_create_timer
 * DMģ鴴rtcʱ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_timer_create(struct DM_CONTROl_INFO *dmControlInfo)
{
    CHECK_PARAM(dmControlInfo);

    if (DM_UNICOM == dmControlInfo->dmType && DM_REGISTERED_NO == dmControlInfo->registerFlag)
    {

        CALL(dm_timer_add(dmControlInfo->secTimer * 60, DM_TIMER_ID_UNICOM_START));
        log_debug("dm timer create success: dmType(%d), secTimer(%d)\n", dmControlInfo->dmType, dmControlInfo->secTimer);
    }
    else if (DM_MOBILE == dmControlInfo->dmType)
    {

        CALL(dm_timer_add(10, DM_TIMER_ID_MOBILE_START));
        log_debug("dm timer create success: dmType(%d), secTimer(%d)\n", dmControlInfo->dmType, dmControlInfo->secTimer);
    }
    else if(DM_TELECOM == dmControlInfo->dmType)
    {
        CALL(dm_timer_add(10, DM_TIMER_ID_MOBILE_START));
        log_debug("dm timer create success: dmType(%d), secTimer(%d)\n", dmControlInfo->dmType, dmControlInfo->secTimer);
    }
    else
    {

        log_debug("dm do not need to create timer, because dm have registered or registerRound more than 10...\n");
    }

    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_net_status_change_proc
 * PDPϢϱ״̬仯
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_net_status_change_proc(struct DM_CONTROl_INFO *dmControlInfo)
{
    CHECK_PARAM(dmControlInfo);
    char netStatus[4] = {0};
    char romaingStatus[DM_NV_ITEM_SIZE] = {0};

    CALL(sc_cfg_get(NV_PDP_STATUS, netStatus, 4));
    CALL(sc_cfg_get(NV_SIMCARD_ROAM, romaingStatus, sizeof(romaingStatus)));

    dmControlInfo->netStatus = atoi(netStatus);

    if (0 ==  strcmp(romaingStatus, "Internal"))
    {
        dmControlInfo->globalRomaing = 1;
    }
    else if (0 == strcmp(romaingStatus, "Home"))
    {
        dmControlInfo->globalRomaing = 0;
    }
    else
    {
        log_err("dm global romaing some error, romaingStatus(%s) \n ", romaingStatus);
        return DM_ERR_WAITING_NET_STATUS_CHANAGE;
    }

    /* ݿش򿪣 ҷǹ */
    if (dmControlInfo->netStatus && !dmControlInfo->globalRomaing)
    {

        /* ǵĹ: ʱʱرգȴ翪ش򿪽ע
         *  뱣֤α־ֻע߳н޸ģ(Ԥ߳̽޸)
         */
        if (dmControlInfo->needRegForNetStatusChange == 1)
        {
            /* Ϣͣʼע */
            CALL(dm_register_msg_send(dmControlInfo));
        }
    }

    log_debug("dm net change now netstatus(%d), romaing(%d), netStatus(%s), romaingStatus(%s),needRegForNetStatusChange(%d) \n", dmControlInfo->netStatus, dmControlInfo->globalRomaing, netStatus, romaingStatus, dmControlInfo->needRegForNetStatusChange);
    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_server_port_num_set
 * עķ˿ں(CP๤)
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_server_port_num_set(MSG_BUF *pstMsg, struct DM_CONTROl_INFO *dmControlInfo)
{
    char *msgData = (char *) pstMsg->aucDataBuf;
    char *portNum = NULL;
    char port[12] = {0};

    if (!(portNum = strstr(msgData, "AT+DM_REGISTER_PORT_NUM=")))
    {
        log_err("dm set portNum erro , can not find server portNum \n");
        return DM_ERR_SET_DOMAIN_ERR;
    }

    portNum = portNum + strlen("AT+DM_REGISTER_PORT_NUM=");
    sscanf(portNum, "%10[^,]", port);

    /*ȷ϶˿ںǷϷ*/
    if (!strcmp(port, "6001") || !strcmp(port, "6002"))
    {
        dmControlInfo->portNum = atol(port);
        sc_cfg_set(DM_PORT_NUM, port);
    }
    else       /* Ϸport   0 < port < 65535*/
    {
        dmControlInfo->portNum = 65539;
        sc_cfg_set(DM_PORT_NUM, "65539");
    }

    log_debug("dm set server portNum config now: server-portNum(%d)\n", dmControlInfo->portNum);
    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_set_server_domain_config
 * עķַ(cp๤)
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_server_domain_config_set(MSG_BUF *pstMsg, struct DM_CONTROl_INFO *dmControlInfo)
{
    CHECK_PARAM(pstMsg);
    CHECK_PARAM(dmControlInfo);

    char *msgData = (char *) pstMsg->aucDataBuf;
    char  *server  = NULL;

    if (!(server = strstr(msgData, "AT+DM_REGISTER_SERVER=")))
    {
        log_err("dm server domain set error\n");
        return DM_ERR_SET_DOMAIN_ERR;
    }

    memset(dmControlInfo->hostName, 0, DM_INFO_MAX_SIZE);
    server = server + strlen("AT+DM_REGISTER_SERVER=");
    sscanf(server, "%32[^,]", dmControlInfo->hostName);

    sc_cfg_set(DM_HOST_NAME, dmControlInfo->hostName);

    log_debug("dm set server domain config now: server-domain(%s)\n", dmControlInfo->hostName);
    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_clean_register_flag
 * עÿֵʱ (cpĹ)
 * ˵
 *   ֵ ɹ 0: ʧ 
 * ˵
 */
static int dm_time_every_round_set(MSG_BUF *pstMsg, struct DM_CONTROl_INFO *dmControlInfo)
{
    CHECK_PARAM(pstMsg);
    CHECK_PARAM(dmControlInfo);

    char *msgData = (char *) pstMsg->aucDataBuf;
    char  *time  = NULL;
    char  timeValue[12] = {0};

    if (!(time = strstr(msgData, "AT+DM_REGISTER_SECS_ROUND=")))
    {
        log_err("dm set time evry round error\n");
        return DM_ERR_SET_TIME_ROUND_ERROR;
    }

    time = time + strlen("AT+DM_REGISTER_SECS_ROUND=");

    sscanf(time, "%10[^,]", timeValue);
    dmControlInfo->secsEveryRound = atoi(timeValue);
    sc_cfg_set(DM_SECS_EVERY_ROUND, timeValue);

    log_debug("dm every round set now, timeValue(%s)", timeValue);
    return DM_ERR_SUCCESS;
}


/**
 * ƣdm_rtc_timer_del
 * ɾеĶʱ
 * ˵
 *   ֵ
 * ˵
 */
static void dm_rtc_timer_del(void)
{
    int index = 0;
    int ret = 0;
    for (index = DM_TIMER_ID_UNICOM_START; index < DM_TIMER_ID_MAX; index++)
    {
        ret = rtc_timer_del(index, MODULE_ID_DMREG);
        log_debug("del rtc id(%d), ret(%d)\n", index, ret);
    }
}

/**
 * ƣdm_clean_register_flag
 * ע״̬ǣ豸ûע(CP ๤)
 * ˵
 *   ֵ 
 * ˵
 */
static void dm_register_flag_clean(struct DM_CONTROl_INFO *dmControlInfo)
{
    sc_cfg_set(DM_REGISTER_FLAG, "0");
    sc_cfg_set(DM_REGISTERED_ROUND, "0");
    dmControlInfo->currentRounds = 0;
    dmControlInfo->currentTimes = 0;
    dmControlInfo->registeredRounds = 0;
    dmControlInfo->registerFlag = 0;

    if (DM_MOBILE == dmControlInfo->dmType)
    {
        sc_cfg_set(DM_IMEI_PREVIOUS, "0");
        sc_cfg_set(DM_ICCID_PREVIOUS, "0");
        sc_cfg_set(DM_VERSION_PREVIOU, "");
    }

    /*ɾеĶʱ*/
    dm_rtc_timer_del();

    dm_register_thread_release(dmControlInfo);
    log_debug("dm register flag clean now: \n");
}

#if 0   /*Ϊչ*/
/**
 * ƣdm_msg_request_info_send
 * blNVϢ
 * ˵
 *   ֵ
 * ˵ɹ 0: ʧ 
 */
static long dm_msg_request_info_send(unsigned short  targetId, unsigned short msgCmd, unsigned short dataLen, unsigned char *data, int flag)
{
    /* ctm  MODULE_ID_AT_CTL */
    long ret = 0;
    if (0 != (ret = ipc_send_message(MODULE_ID_DMREG,  targetId, msgCmd, dataLen, data, flag)))
        return DM_ERR_IPC_SEND_MESSAGE_ERROR;
    log_debug("dm msg request send now: targetId(0x%04x), msgCmd(0x%08x), dataLen(%d)\n", targetId, msgCmd, dataLen);
    return DM_ERR_SUCCESS;
}
#endif

/**
* ƣdm_register_msg_send
* עϢע߳,ע
* ˵
*   ֵɹ 0: ʧ 
* ˵
*/
static int dm_register_msg_send(struct DM_CONTROl_INFO *dmControlInfo)
{
    CHECK_PARAM(dmControlInfo);
    int ret = 0;
    long size = 0;
    MSG_BUF msgBuf = {0};

    msgBuf.usMsgCmd  = DM_MSG_CMD_REGISTER_REQUEST;
    size = sizeof(MSG_BUF) - sizeof(long);

    msgBuf.ulMagic = MSG_MAGIC_WORD;
    msgBuf.lMsgType = MSG_TYPE_DEFAULT;

    /* Ϣ ע߳ ע */
    ret = msgsnd(dmControlInfo->msgQueueIdReg, (void*) & (msgBuf), size, 0);
    if (ret < 0)
    {
        log_err("unicom startup proc send msg errro \n");
        return DM_ERR_QUEUE_SEND_ERROR;
    }

    return DM_ERR_SUCCESS;
}

/**
* ƣ dm_mobile_register_start
*  ƶDMע̿ʼ
* ˵
*   ֵ ɹ 0: ʧ 
* ˵
*/
static int dm_mobile_register_start(struct DM_CONTROl_INFO *dmControlInfo)
{
    char *dataBuf = NULL;
    int dataSize = 0;
    int ret = 0;
    char imei[DM_NV_ITEM_SIZE] = {0};
    char iccid[DM_NV_ITEM_SIZE] = {0};
    char version[DM_NV_ITEM_SIZE] = {0};
    DM_MOBILE_HTTPS_URL_INIO  dmMobileUrlInfo = {{0}, {0}, {0}, 0, 0};
    DM_MOBILE_HTTPS_DATA_INFO dmMobileHttpsDataInfo = {0};
    struct DM_MOBILE_REGISTER_INFO dmMobileRegInfo ;

    dataSize = (DM_NV_ITEM_SIZE + DM_NV_ITEM_SIZE / 4 + 5) * DM_INFO_MOBILE_MAX;

    if (!(dataBuf = (char *) malloc(dataSize)))
    {
        log_err("dm unicom start reg malloc mem error\n");
        return DM_ERR_PARAM_NULL;
    }
    memset(dataBuf, 0, dataSize);
    dmMobileHttpsDataInfo.dataBuffer = dataBuf;
    dmMobileHttpsDataInfo.dataLen = dataSize;

    ret = dm_mobile_register_info_create(dmControlInfo, &dmMobileRegInfo);
    if(DM_ERR_SUCCESS != ret)
    {
        goto ErrRet;
    }
    ret = dm_mobile_json_info_create(&dmMobileRegInfo, &dmMobileHttpsDataInfo);
    if(DM_ERR_SUCCESS != ret)
    {
        goto ErrRet;
    }
    ret = dm_mobile_url_info_create(dmControlInfo, &dmMobileRegInfo, &dmMobileUrlInfo);
    if(DM_ERR_SUCCESS != ret)
    {
        goto ErrRet;
    } //klockworks issue.

    ret = dm_mobile_https_request(&dmMobileUrlInfo, &dmMobileHttpsDataInfo);

    if (0 == ret)
    {
        /*עɹNV */
        char buf[4] = {0};
        dmControlInfo->registerFlag = DM_REGISTERED_YES;
        snprintf(buf, 3, "%d", dmControlInfo->registerFlag);
        sc_cfg_set("registerFlag", buf);             //ctm

        /*дIMEI ICCID */
        sc_cfg_get(NV_IMEI, imei, sizeof(imei));
        sc_cfg_get(NV_ZICCID, iccid, sizeof(iccid));
        sc_cfg_get(DM_SW_VERSION, version, sizeof(version));

        sc_cfg_set(DM_IMEI_PREVIOUS, imei);
        sc_cfg_set(DM_ICCID_PREVIOUS, iccid);

        /*д汾ţΪfota DM ע*/
        sc_cfg_set(DM_VERSION_PREVIOU, version);

        dm_resource_clean(dmControlInfo, (void *) &dmMobileHttpsDataInfo);
        return DM_ERR_SUCCESS;
    }

    dm_resource_clean(dmControlInfo, &dmMobileHttpsDataInfo);
    log_debug("dm start register start: debug info as following ************\n");
    log_debug("dm currentTimes(%d)\n", dmControlInfo->currentTimes);
    log_debug("dm register secTimer(%d), , secsEveryTime(%d), netStatus(%d), globalRomaing(%d)\n",
              dmControlInfo->secTimer,
              dmControlInfo->secsEveryTime,
              dmControlInfo->netStatus,
              dmControlInfo->globalRomaing);

    return DM_ERR_RETRY_ERROR;

ErrRet:
    free(dataBuf);
    return ret;
}

/**
* ƣdm_register_request_handle
* ƶdm ע
* ˵
*   ֵɹ 0: ʧ 
* ˵
*/
static int dm_mobile_register_request_handle(struct DM_CONTROl_INFO *dmControlInfo)
{
    CHECK_PARAM(dmControlInfo);

    int ret = 0;
    int checkTimes = 10;
    char romaingStatus[DM_NV_ITEM_SIZE] = {0};
    char netStatus[DM_NV_ITEM_SIZE] = {0};

    /*ȡ ״ֵ̬*/
    CALL(sc_cfg_get(NV_PDP_STATUS, netStatus, sizeof(netStatus)));
    CALL(sc_cfg_get(NV_SIMCARD_ROAM, romaingStatus, sizeof(romaingStatus)));
    dmControlInfo->netStatus = atoi(netStatus);

    if (0 ==  strcmp(romaingStatus, "Internal"))
    {
        dmControlInfo->globalRomaing = 1;
    }
    else if (0 == strcmp(romaingStatus, "Home"))
    {
        dmControlInfo->globalRomaing = 0;
    }
    else
    {
        log_err("dm global romaing some error, romaingStatus(%s) \n ", romaingStatus);
        return DM_ERR_WAITING_NET_STATUS_CHANAGE;
    }

    if (dmControlInfo->currentTimes >= 10)
    {
        log_debug("dm mobile register 10 times now...");
        return DM_ERR_MOBILE_OVER_RETRY_TMIES_ERROR;
    }

    /*  1·رգ
     *  2·򿪣ڹ
     */
    if (dmControlInfo->netStatus == 0 || (dmControlInfo->netStatus == 1 && dmControlInfo->globalRomaing == 1))
    {
        /*ƶĹ淶˵˴Ҳ볢Դ */
        dmControlInfo->currentTimes++;

        if (dmControlInfo->currentTimes < 10)
        {
            /*1ûͨ ԱвԣֵЭ涨ʵĴμʱ䣬̲ʱ
             *2Ĭֵ0,ʾʹʵĴμʱ䣬0 ʾʹôõֵ*/
            if (0 != dmControlInfo->secsEveryRound)
            {
                dmControlInfo->secsEveryTime = dmControlInfo->secsEveryRound;
            }
            else
            {
                dmControlInfo->secsEveryTime = dmControlInfo->currentTimes * 10;
            }

            dm_timer_add(dmControlInfo->secsEveryTime * 60, DM_TIMER_ID_MOBILE_EVERY_TIMES);
        }

        log_debug(" startup_proc now, net status(%d), globalRomaing(%d), currentTimes(%d)\n", dmControlInfo->netStatus, dmControlInfo->globalRomaing,  dmControlInfo->currentTimes);
        return DM_ERR_WAITING_NET_STATUS_CHANAGE;
    }

    while (checkTimes-- > 0)
    {
        if (1 == default_route_check())
            break;
        usleep(1000 * 500);
        log_debug("mobile startup check rout times(%d)\n", checkTimes);
        if (checkTimes == 0)
        {
            log_err("dm net some error occur\n");
            return DM_ERR_CHECK_ROUTE_ERROR;
        }
    }

    dmControlInfo->currentTimes++;    /*м */
    /*httpܵ */
    if (DM_ERR_SUCCESS != (ret = dm_mobile_register_start(dmControlInfo)))
    {
        /* עĴС 10 ʱ */
        if (dmControlInfo->currentTimes < 10)
        {
            /*1dmControlInfo->secsEveryRound ûͨ,ĿǷԱвԣ
             *   ֵЭ涨ʵĴμʱ䣬̲ʱ䣻
             *2Ĭֵ0,ʾʹʵĴμʱ䣬0 ʾʹôõֵ */
            if (0 != dmControlInfo->secsEveryRound)
            {
                dmControlInfo->secsEveryTime = dmControlInfo->secsEveryRound;
            }
            else
            {
                dmControlInfo->secsEveryTime = dmControlInfo->currentTimes * 10;
            }
            /* ʧÿγԶʱ */
            dm_timer_add(dmControlInfo->secsEveryTime * 60, DM_TIMER_ID_MOBILE_EVERY_TIMES);
        }

        log_debug("dm mobile register error, ret(%d), currentTimes(%d), next-rtc-time(%d)\n", ret, dmControlInfo->currentTimes, dmControlInfo->secsEveryTime);
        return ret;
    }

    return DM_ERR_SUCCESS;
}

/**
 * ƣ dm_mobile_url_info_create
 *  ƶDMķ urlϢ ַ˿ڵϢ
 * ˵
 *   ֵ ɹ 0: ʧ 
 * ˵
 */
static int dm_mobile_url_info_create(struct DM_CONTROl_INFO *dmControlInfo, struct DM_MOBILE_REGISTER_INFO *dmMobileRegInfo, DM_MOBILE_HTTPS_URL_INIO *dmMobileUrlInfo)
{
    char mobileAppkey[DM_NV_ITEM_SIZE] = {0};
    CHECK_PARAM(dmControlInfo);
    CHECK_PARAM(dmMobileRegInfo);
    CHECK_PARAM(dmMobileUrlInfo);
    CHECK_PARAM(dmMobileUrlInfo->urlBuffer);

    sc_cfg_get("appKeyMobile", mobileAppkey, DM_NV_ITEM_SIZE);

    memset(dmMobileUrlInfo, 0, sizeof(DM_MOBILE_HTTPS_URL_INIO));
    snprintf(dmMobileUrlInfo->urlBuffer, DM_HTTPS_HEAD_DATA_LENGTH - 1, "/accept/featureService?func=tsdk:postotherreg&appkey=%s", mobileAppkey);

    snprintf(dmMobileUrlInfo->hostName, DM_NV_ITEM_SIZE - 1, "%s", dmControlInfo->hostName);

    snprintf(dmMobileUrlInfo->ipAddr, 31, "%s", dmControlInfo->ipAddr);

    dmMobileUrlInfo->portNum = dmControlInfo->portNum;

    log_debug("dm unicom url-info debug: ulr(%s), portNum(%ld), hostName(%s)\n ", dmMobileUrlInfo->urlBuffer, dmMobileUrlInfo->portNum, dmMobileUrlInfo->hostName);

    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_unicom_json_info_create
 * ͨЭ淶jsonʽ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_unicom_json_info_create(struct DM_UNICOM_REGISTER_INFO *dmUnicomRegInfo, DM_UNICOM_HTTP_DATA_INFO *httpsDataInfo)
{
    CHECK_PARAM(dmUnicomRegInfo);
    CHECK_PARAM(httpsDataInfo);

    int i = 0;
    char *pTmep = NULL;
    int jsonSize = 0;
    char *unicomItemName[] = {"Manuf", "Model", "HWVersion", "SWVersion",
                              "OS", "OSVersion", "IMEI1", "ICCID1",
                              "IMSI1", "MSISDN1", "MNC1", "NCLS1",
                              "CellID1", "LAC1", "AccType", "Time", "Tag"
                             };
    /*json fromat as following
     * {"key1":"key1Value","key2":"key2Value", xxxxx}*/
    log_debug("dm unicom json infof ***********\n");

    pTmep = httpsDataInfo->dataBuffer;
    memset(pTmep, 0, httpsDataInfo->dataLen);
    for (i = 0; i < DM_INFO_UNICOM_MAX; i++)
    {
        if (i == 0)
            sprintf(pTmep, "{\"%s\":\"%s\"", unicomItemName[i], dmUnicomRegInfo->value[i]);
        else if (i == DM_INFO_UNICOM_MAX - 1)
            sprintf(pTmep, ",\"%s\":\"%s\"}", unicomItemName[i], dmUnicomRegInfo->value[i]);
        else
            sprintf(pTmep, ",\"%s\":\"%s\"", unicomItemName[i], dmUnicomRegInfo->value[i]);

        log_debug("json-value(%s)\n", pTmep);
        jsonSize = jsonSize + strlen(pTmep);
        pTmep = pTmep + strlen(pTmep);

        if (jsonSize > httpsDataInfo->dataLen)
            log_err("json buffer memory overwrite now! please check memory size, jsonSize(%d), httpsDatainfo-size(%d) !!!\n", jsonSize, httpsDataInfo->dataLen);
    }

    httpsDataInfo->dataLen = jsonSize;
    log_debug("create json info, json-szie(%d)\n",  jsonSize);

    return DM_ERR_SUCCESS;
}


/**
 * ƣdm_mobile_json_info_create
 * ƶЭ淶jsonʽ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_mobile_json_info_create(struct DM_MOBILE_REGISTER_INFO *dmMobileRegInfo, DM_MOBILE_HTTPS_DATA_INFO *httpsDataInfo)
{
    CHECK_PARAM(dmMobileRegInfo);
    CHECK_PARAM(httpsDataInfo);
    CHECK_PARAM(httpsDataInfo->dataBuffer);
    int i = 0;
    char *pTmep = NULL;
    int jsonSize = 0;

    /*淶ݣֵ */
    char *mobileItemName[] = {"sdkVersion", "imei1", "imei2", "meid", "deviceId", "brand",
                              "model", "firmwareVer", "systemVer", "cpu", "rom", "ram", "type", "iccid1", "iccid2", "imsi1", "imsi2",
                              "mac", "cellId", "lac", "channel", "dataCard", "masterStatus", "sendTime", "soltQuantity",
                              "dataCard2", "soltService1", "soltService2", "soltNetwork1", "soltNetwork2", "lac2", "cellId2", "Intype"
                             };

    log_debug("dm mobile json infof ***********\n");
    pTmep = httpsDataInfo->dataBuffer;
    memset(pTmep, 0, httpsDataInfo->dataLen);
    for (i = 0; i < DM_INFO_MOBILE_MAX; i++)
    {
        if (i == 0)
            sprintf(pTmep, "{\"%s\":\"%s\"", mobileItemName[i], dmMobileRegInfo->value[i]);
        else if (i == DM_INFO_MOBILE_MAX - 1)
            sprintf(pTmep, ",\"%s\":\"%s\"}", mobileItemName[i], dmMobileRegInfo->value[i]);
        else
            sprintf(pTmep, ",\"%s\":\"%s\"", mobileItemName[i], dmMobileRegInfo->value[i]);

        log_debug("json-value[%d](%s)\n", i, pTmep);
        jsonSize = jsonSize + strlen(pTmep);
        pTmep = pTmep + strlen(pTmep);

        if (jsonSize > httpsDataInfo->dataLen)
        {
            log_err("json buffer memory overwrite now! please check memory size, jsonSize(%d), httpsDatainfo-size(%d) !!!\n", jsonSize, httpsDataInfo->dataLen);
            return DM_ERR_CREATE_JSON_INFO_ERROR;
        }
    }

    httpsDataInfo->dataLen = jsonSize;
    log_debug("json-szie(%d)\n", jsonSize);

    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_set_register_tag
 * ͨDMЭTagֵ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static void dm_register_tag_set(struct DM_CONTROl_INFO *dmControlInfo, struct DM_UNICOM_REGISTER_INFO *dmUnicomRegInfo)
{
    snprintf(dmUnicomRegInfo->value[DM_INFO_TAG], 63, "%d,%d,%d", dmControlInfo->registeredRounds, dmControlInfo->currentRounds, dmControlInfo->currentTimes);
}

/**
 * ƣdm_register_time_set
 * DMЭʽ涨ʱʽ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_register_time_set(struct DM_CONTROl_INFO *dmControlInfo, void *dataInfo)
{
    CHECK_PARAM(dataInfo);
    char timeBuf[DM_NV_ITEM_SIZE] = {0};
    struct DM_MOBILE_REGISTER_INFO *dmMobileRegInfo = NULL;
    struct DM_UNICOM_REGISTER_INFO *dmUnicomRegInfo = NULL;
    time_t t = {0};
    struct tm *area = NULL;

    t = time(NULL);
    area = (struct tm *) localtime(&t);

    if (!area)
    {
        log_err("dm set register time error\n");
        return DM_ERR_PARAM_NULL;
    }
    strftime(timeBuf, 63, "%Y-%m-%d %H:%M:%S", area);

    if (DM_UNICOM == dmControlInfo->dmType)
    {
        dmUnicomRegInfo = (struct DM_UNICOM_REGISTER_INFO *) dataInfo;
        snprintf(dmUnicomRegInfo->value[DM_INFO_TIME], DM_NV_ITEM_SIZE - 1, "%s", timeBuf);
    }

    if (DM_MOBILE == dmControlInfo->dmType)
    {
        dmMobileRegInfo = (struct DM_MOBILE_REGISTER_INFO *) dataInfo;
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_SEND_TIME], DM_NV_ITEM_SIZE - 1, "%s", timeBuf);
    }
    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_unicom_register_info_create
 * : ͨDMЭ淶ָעϢ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_unicom_register_info_create(struct DM_CONTROl_INFO *dmControlInfo, struct DM_UNICOM_REGISTER_INFO *dmUnicomRegInfo)
{
    CHECK_PARAM(dmUnicomRegInfo);
    char  mmc[DM_NV_ITEM_SIZE] = {0};
    char dmUnicomNvInfo[DM_NV_ITEM_SIZE] = {0};
    int index = 0;

    memset(dmUnicomRegInfo, 0, sizeof(struct DM_UNICOM_REGISTER_INFO));

#ifndef DM_TEST_DEBUG
    /*ͺ*/
    CALL(sc_cfg_get(DM_MANUF, dmUnicomNvInfo, DM_NV_ITEM_SIZE));
    snprintf(dmUnicomRegInfo->value[DM_INFO_MANUF], DM_NV_ITEM_SIZE - 1,  "%s", dmUnicomNvInfo);

    /*̴*/
    memset(dmUnicomNvInfo, 0, DM_NV_ITEM_SIZE);
    CALL(sc_cfg_get(DM_MODEL, dmUnicomNvInfo, DM_NV_ITEM_SIZE));
    snprintf(dmUnicomRegInfo->value[DM_INFO_MODEL], DM_NV_ITEM_SIZE - 1,  "%s", dmUnicomNvInfo);

    /*Ӳ汾*/
    memset(dmUnicomNvInfo, 0, DM_NV_ITEM_SIZE);
    CALL(sc_cfg_get(NV_HW_VERSION, dmUnicomNvInfo, DM_NV_ITEM_SIZE));
    snprintf(dmUnicomRegInfo->value[DM_INFO_HWVISION], DM_NV_ITEM_SIZE - 1,  "%s", dmUnicomNvInfo);

    /*汾*/
    memset(dmUnicomNvInfo, 0, DM_NV_ITEM_SIZE);
    CALL(sc_cfg_get(NV_WA_VERSION, dmUnicomNvInfo, DM_NV_ITEM_SIZE));
    snprintf(dmUnicomRegInfo->value[DM_INFO_SWVERSION], DM_NV_ITEM_SIZE - 1,  "%s", dmUnicomNvInfo);

    snprintf(dmUnicomRegInfo->value[DM_INFO_OS], DM_NV_ITEM_SIZE - 1,  "%s", "TOS");

    snprintf(dmUnicomRegInfo->value[DM_INFO_OSVISION], DM_NV_ITEM_SIZE - 1,  "%s", "1.0");

    memset(dmUnicomNvInfo, 0, DM_NV_ITEM_SIZE);
    CALL(sc_cfg_get(NV_IMEI, dmUnicomNvInfo, DM_NV_ITEM_SIZE));
    snprintf(dmUnicomRegInfo->value[DM_INFO_IMEI1], DM_NV_ITEM_SIZE - 1,  "%s", dmUnicomNvInfo);

    memset(dmUnicomNvInfo, 0, DM_NV_ITEM_SIZE);
    CALL(sc_cfg_get(NV_MCC, dmUnicomNvInfo, DM_NV_ITEM_SIZE));
    CALL(sc_cfg_get(NV_MNC, mmc, DM_NV_ITEM_SIZE));
    snprintf(dmUnicomRegInfo->value[DM_INFO_MNC1], DM_NV_ITEM_SIZE - 1,  "%s%s", dmUnicomNvInfo, mmc);

    /* LTE, WCDMA, GSM, TD-CDMA, ͨҪʽ */
    memset(dmUnicomNvInfo, 0, DM_NV_ITEM_SIZE);
    CALL(sc_cfg_get(NV_NETWORK_TYPE, dmUnicomNvInfo, DM_NV_ITEM_SIZE));
    if (!strcmp(dmUnicomNvInfo, "GPRS"))   /* Э淶 GPRS->GSM */
    {
        snprintf(dmUnicomRegInfo->value[DM_INFO_NCLS1], DM_NV_ITEM_SIZE - 1,  "%s", "GSM");
    }
    else
    {
        snprintf(dmUnicomRegInfo->value[DM_INFO_NCLS1], DM_NV_ITEM_SIZE - 1,  "%s", dmUnicomNvInfo);
    }

    if (1 == dmControlInfo->cardType)   /* ƶ ,Э淶ָ Ϊ 1*/
    {
        snprintf(dmUnicomRegInfo->value[DM_INFO_ICCID1], DM_NV_ITEM_SIZE - 1,  "%s", "11111111111111111111");
        snprintf(dmUnicomRegInfo->value[DM_INFO_IMSI1], DM_NV_ITEM_SIZE - 1,  "%s", "111111111111111");
        snprintf(dmUnicomRegInfo->value[DM_INFO_MSISDN1], DM_NV_ITEM_SIZE - 1,  "%s", "11111111111");
        snprintf(dmUnicomRegInfo->value[DM_INFO_CELLID1], DM_NV_ITEM_SIZE - 1,	"%s", "1111");
        snprintf(dmUnicomRegInfo->value[DM_INFO_LAC1], DM_NV_ITEM_SIZE - 1,  "%s", "1111");

    }
    else if (3 == dmControlInfo->cardType)     /*ſЭ淶ָΪ 3*/
    {
        snprintf(dmUnicomRegInfo->value[DM_INFO_ICCID1], DM_NV_ITEM_SIZE - 1,  "%s", "33333333333333333333");
        snprintf(dmUnicomRegInfo->value[DM_INFO_IMSI1], DM_NV_ITEM_SIZE - 1,  "%s", "333333333333333");
        snprintf(dmUnicomRegInfo->value[DM_INFO_MSISDN1], DM_NV_ITEM_SIZE - 1,  "%s", "33333333333");
        snprintf(dmUnicomRegInfo->value[DM_INFO_CELLID1], DM_NV_ITEM_SIZE - 1,	"%s", "3333");
        snprintf(dmUnicomRegInfo->value[DM_INFO_LAC1], DM_NV_ITEM_SIZE - 1,  "%s", "3333");

    }
    else if (2 == dmControlInfo->cardType)
    {
        memset(dmUnicomNvInfo, 0, DM_NV_ITEM_SIZE);
        CALL(sc_cfg_get(NV_ZICCID, dmUnicomNvInfo, DM_NV_ITEM_SIZE));
        snprintf(dmUnicomRegInfo->value[DM_INFO_ICCID1], DM_NV_ITEM_SIZE - 1,	"%s", dmUnicomNvInfo);

        memset(dmUnicomNvInfo, 0, DM_NV_ITEM_SIZE);
        CALL(sc_cfg_get(NV_MSISDN, dmUnicomNvInfo, DM_NV_ITEM_SIZE));
        snprintf(dmUnicomRegInfo->value[DM_INFO_MSISDN1], DM_NV_ITEM_SIZE - 1,  "%s", dmUnicomNvInfo);

        memset(dmUnicomNvInfo, 0, DM_NV_ITEM_SIZE);
        CALL(sc_cfg_get(NV_SIM_IMSI, dmUnicomNvInfo, DM_NV_ITEM_SIZE));
        snprintf(dmUnicomRegInfo->value[DM_INFO_IMSI1], DM_NV_ITEM_SIZE - 1,  "%s", dmUnicomNvInfo);

        /*СλϢ1 */
        memset(dmUnicomNvInfo, 0, DM_NV_ITEM_SIZE);
        CALL(sc_cfg_get(NV_CELL_ID, dmUnicomNvInfo, DM_NV_ITEM_SIZE));
        snprintf(dmUnicomRegInfo->value[DM_INFO_CELLID1], DM_NV_ITEM_SIZE - 1,  "%s", dmUnicomNvInfo);

        /*СλϢ2 */
        memset(dmUnicomNvInfo, 0, DM_NV_ITEM_SIZE);
        sc_cfg_get(NV_NETWORK_TYPE, dmUnicomNvInfo, DM_NV_ITEM_SIZE);
        if (!strcmp(dmUnicomNvInfo, "LTE"))   /* 4G */
        {
            memset(dmUnicomNvInfo, 0, DM_NV_ITEM_SIZE);
            sc_cfg_get(NV_TAC_CODE, dmUnicomNvInfo, DM_NV_ITEM_SIZE);
            snprintf(dmUnicomRegInfo->value[DM_INFO_LAC1], DM_NV_ITEM_SIZE - 1,  "%s", dmUnicomNvInfo);
        }
        else      /* 4G */
        {
            memset(dmUnicomNvInfo, 0, DM_NV_ITEM_SIZE);
            sc_cfg_get(NV_LAC_CODE, dmUnicomNvInfo, DM_NV_ITEM_SIZE);
            snprintf(dmUnicomRegInfo->value[DM_INFO_LAC1], DM_NV_ITEM_SIZE - 1,  "%s", dmUnicomNvInfo);
        }
    }
    else
    {
        log_err("dm there is unknow card type\n");
        return DM_ERR_PARAM_NULL;
    }
#else // for test debug
    snprintf(dmUnicomRegInfo->value[DM_INFO_MANUF], DM_NV_ITEM_SIZE - 1,  "%s", "BestSonny");
    snprintf(dmUnicomRegInfo->value[DM_INFO_MODEL], DM_NV_ITEM_SIZE - 1,  "%s", "Best sonny LT589");
    snprintf(dmUnicomRegInfo->value[DM_INFO_HWVISION], DM_NV_ITEM_SIZE - 1,  "%s", "K312-MB-V0.2A");
    snprintf(dmUnicomRegInfo->value[DM_INFO_SWVERSION], DM_NV_ITEM_SIZE - 1,  "%s", "K312-HUASEN-BEST-SONNY-LT589-QVGA-SC-B10P02_170609_1700");
    snprintf(dmUnicomRegInfo->value[DM_INFO_OS], DM_NV_ITEM_SIZE - 1,  "%s", "TOS");
    snprintf(dmUnicomRegInfo->value[DM_INFO_OSVISION], DM_NV_ITEM_SIZE - 1,  "%s", "1.0");
    snprintf(dmUnicomRegInfo->value[DM_INFO_IMEI1], DM_NV_ITEM_SIZE - 1,  "%s", "865062030001015");
    snprintf(dmUnicomRegInfo->value[DM_INFO_MNC1], DM_NV_ITEM_SIZE - 1,  "%s", "46001");
    snprintf(dmUnicomRegInfo->value[DM_INFO_NCLS1], DM_NV_ITEM_SIZE - 1,  "%s", "LTE");

    if (1 == dmControlInfo->cardType)   /*ƶ*/
    {
        snprintf(dmUnicomRegInfo->value[DM_INFO_ICCID1], DM_NV_ITEM_SIZE - 1,  "%s", "11111111111111111111");
        snprintf(dmUnicomRegInfo->value[DM_INFO_IMSI1], DM_NV_ITEM_SIZE - 1,  "%s", "111111111111111");
        snprintf(dmUnicomRegInfo->value[DM_INFO_MSISDN1], DM_NV_ITEM_SIZE - 1,  "%s", "11111111111");
        snprintf(dmUnicomRegInfo->value[DM_INFO_CELLID1], DM_NV_ITEM_SIZE - 1,	"%s", "1111");
        snprintf(dmUnicomRegInfo->value[DM_INFO_LAC1], DM_NV_ITEM_SIZE - 1,  "%s", "1111");

    }
    else if (3 == dmControlInfo->cardType)      /*ſ*/
    {
        snprintf(dmUnicomRegInfo->value[DM_INFO_ICCID1], DM_NV_ITEM_SIZE - 1,  "%s", "33333333333333333333");
        snprintf(dmUnicomRegInfo->value[DM_INFO_IMSI1], DM_NV_ITEM_SIZE - 1,  "%s", "333333333333333");
        snprintf(dmUnicomRegInfo->value[DM_INFO_MSISDN1], DM_NV_ITEM_SIZE - 1,  "%s", "33333333333");
        snprintf(dmUnicomRegInfo->value[DM_INFO_CELLID1], DM_NV_ITEM_SIZE - 1,	"%s", "3333");
        snprintf(dmUnicomRegInfo->value[DM_INFO_LAC1], DM_NV_ITEM_SIZE - 1,  "%s", "3333");

    }
    else
    {

        snprintf(dmUnicomRegInfo->value[DM_INFO_ICCID1], DM_NV_ITEM_SIZE - 1,  "%s", "89860114834067295358");
        snprintf(dmUnicomRegInfo->value[DM_INFO_IMSI1], DM_NV_ITEM_SIZE - 1,  "%s", "460011675515843");
        snprintf(dmUnicomRegInfo->value[DM_INFO_MSISDN1], DM_NV_ITEM_SIZE - 1,  "%s", "18551770057");
        snprintf(dmUnicomRegInfo->value[DM_INFO_CELLID1], DM_NV_ITEM_SIZE - 1,	"%s", "101387885");
        snprintf(dmUnicomRegInfo->value[DM_INFO_LAC1], DM_NV_ITEM_SIZE - 1,  "%s", "53515");
    }
#endif

    snprintf(dmUnicomRegInfo->value[DM_INFO_ACCTYPE], 10,  "%s", "Mobile");
    CALL(dm_register_time_set(dmControlInfo, dmUnicomRegInfo));
    dm_register_tag_set(dmControlInfo, dmUnicomRegInfo);

    log_debug("dm unicom get register from NV now ...\n");
    for (index = 0; index < DM_INFO_UNICOM_MAX; index++)
    {
        log_debug("Unicom info VALUE[%d] = %s\n", index, (dmUnicomRegInfo->value) [index]);

    }

    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_data_channel_set
 * DMЭ淶ͨ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static void dm_data_channel_set(struct DM_MOBILE_REGISTER_INFO *dmMobileRegInfo)
{
    char dmMobileNvInfo[DM_NV_ITEM_SIZE] = {0};
    memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    sc_cfg_get(NV_SUB_NETWORK_TYPE, dmMobileNvInfo, DM_NV_ITEM_SIZE);

    if (strstr(dmMobileNvInfo, "GPRS"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 1);
    }
    else if (strstr(dmMobileNvInfo, "EDGE"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 2);
    }
    else if (strstr(dmMobileNvInfo, "UMTS"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 3);
    }
    else if (strstr(dmMobileNvInfo, "CDMA"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 4);
    }
    else if (strstr(dmMobileNvInfo, "EVDO_0"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 5);
    }
    else if (strstr(dmMobileNvInfo, "EVDO_A"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 6);
    }
    else if (strstr(dmMobileNvInfo, "HSDPA"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 8);
    }
    else if (strstr(dmMobileNvInfo, "HSUPA"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 9);
    }
    else if (strstr(dmMobileNvInfo, "HSPA"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 10);
    }
    else if (strstr(dmMobileNvInfo, "IDEN"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 11);
    }
    else if (strstr(dmMobileNvInfo, "EVDO_B"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 12);
    }
    else if (strstr(dmMobileNvInfo, "LTE"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 13);
    }
    else if (strstr(dmMobileNvInfo, "EHRPD"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 14);
    }
    else if (strstr(dmMobileNvInfo, "HSPAP"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 15);
    }
    else if (strstr(dmMobileNvInfo, "GSM"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 16);
    }
    else if (strstr(dmMobileNvInfo, "TD-SCDMA"))
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 17);
    }
    else
    {
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL], DM_NV_ITEM_SIZE - 1, "%d", 0);
        log_debug("dm data channel set , unknow the network_type(%s)\n", dmMobileNvInfo);
    }
}

/**
 * ƣdm_mobile_register_info_create
 * ƶЭ淶ָעϢ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_mobile_register_info_create(struct DM_CONTROl_INFO *dmControlInfo, struct DM_MOBILE_REGISTER_INFO *dmMobileRegInfo)
{
    CHECK_PARAM(dmMobileRegInfo);
    char dmMobileNvInfo[DM_NV_ITEM_SIZE] = {0};
    int index = 0;

    memset(dmMobileRegInfo, 0, sizeof(struct DM_MOBILE_REGISTER_INFO));

#ifndef DM_TEST_DEBUG
    /*SDK 汾*/
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_SDKVERSION], DM_NV_ITEM_SIZE - 1, "%s", "1.0.1");

    /*imei */
    memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    sc_cfg_get(NV_IMEI, dmMobileNvInfo, DM_NV_ITEM_SIZE);
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_IMEI1], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_IMEI2], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_MEID], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    /* ܻid ,iemiŴ */
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_DEVICE_ID], DM_NV_ITEM_SIZE - 1, "%s", dmMobileRegInfo->value[DM_INFO_MOBILE_IMEI1]);

    /*նƷ*/
    memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    sc_cfg_get(DM_BRAND, dmMobileNvInfo, DM_NV_ITEM_SIZE);
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_BRAND], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);

    /*նͺ*/
    memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    sc_cfg_get(DM_MODEL, dmMobileNvInfo, DM_NV_ITEM_SIZE);
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_MODEL], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);

    /*汾*/
    memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    sc_cfg_get(DM_SW_VERSION, dmMobileNvInfo, DM_NV_ITEM_SIZE);
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_FIRMWARE_VERSION], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);

    /*汾*/
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_SYSTEM_VERSION], DM_NV_ITEM_SIZE - 1, "%s", "1.0");

    /*ƽ̨CPUͺ */
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CPU], DM_NV_ITEM_SIZE - 1, "%s", "ZX297520V3");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_ROM], DM_NV_ITEM_SIZE - 1, "%s", "128MB");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_RAM], DM_NV_ITEM_SIZE - 1, "%s", "128MB");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_TYPE], DM_NV_ITEM_SIZE - 1, "%s", "1");

    memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    sc_cfg_get(NV_ZICCID, dmMobileNvInfo, DM_NV_ITEM_SIZE);
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_ICCID1], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_ICCID2], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    sc_cfg_get(NV_SIM_IMSI, dmMobileNvInfo, DM_NV_ITEM_SIZE);
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_IMSI1], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_IMSI2], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_MAC], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    sc_cfg_get(NV_CELL_ID, dmMobileNvInfo, DM_NV_ITEM_SIZE);
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CELLID], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);

    memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    sc_cfg_get(NV_NETWORK_TYPE, dmMobileNvInfo, DM_NV_ITEM_SIZE);
    if (!strcmp(dmMobileNvInfo, "LTE"))   /* 4G */
    {
        memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
        sc_cfg_get(NV_TAC_CODE, dmMobileNvInfo, DM_NV_ITEM_SIZE);
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_LAC], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);
    }
    else      /* 4G */
    {
        memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
        sc_cfg_get(NV_LAC_CODE, dmMobileNvInfo, DM_NV_ITEM_SIZE);
        snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_LAC], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);
    }

    /* 4g 3G 2G*/
    dm_data_channel_set(dmMobileRegInfo);

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_DATA_CARD], DM_NV_ITEM_SIZE - 1, "%s", "2");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_MASTER_STATUS], DM_NV_ITEM_SIZE - 1, "%s", "0");

    /*ͨ*/
    CALL(dm_register_time_set(dmControlInfo, (void *) dmMobileRegInfo));

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_SLOT_QUANTITY], DM_NV_ITEM_SIZE - 1, "%s", "1");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_DATA_CARD2], DM_NV_ITEM_SIZE - 1, "%s", "-1");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_SLOT_SERVICE1], DM_NV_ITEM_SIZE - 1, "%s", "5");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_SLOT_SERVICE2], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    /* slot-network ֵֵͨͬ(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL])*/
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_SLOT_NETWORK1], DM_NV_ITEM_SIZE - 1, "%s", dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL]);

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_SLOT_NETWORK2], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_LAC2], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CELLID2], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    sc_cfg_get(DM_IN_TYPE, dmMobileNvInfo, DM_NV_ITEM_SIZE);    /*ͺ*/
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_TERMINAL_INTTYPE], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);

    log_debug("dm mobile get register from NV now ...\n");

    for (index = 0; index < DM_INFO_MOBILE_MAX; index++)
    {
        log_debug("Mobile info VALUE[%d] = %s\n", index, (dmMobileRegInfo->value) [index]);
    }

#else

    /*SDK 汾*/
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_SDKVERSION], DM_NV_ITEM_SIZE - 1, "%s", "1.0.1");

    /*imei */
    memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    sc_cfg_get(NV_IMEI, dmMobileNvInfo, DM_NV_ITEM_SIZE);
    //snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_IMEI1], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_IMEI1], DM_NV_ITEM_SIZE - 1,  "%s", "864023030000167");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_IMEI2], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_MEID], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    /* ܻid ,iemiŴ */
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_DEVICE_ID], DM_NV_ITEM_SIZE - 1, "%s", dmMobileRegInfo->value[DM_INFO_MOBILE_IMEI1]);

    //memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    //sc_cfg_get(DM_BRAND, dmMobileNvInfo, DM_NV_ITEM_SIZE);    // նƷ
    //snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_BRAND], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_BRAND], DM_NV_ITEM_SIZE - 1, "%s", "CFZZ");

    //memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    //sc_cfg_get(DM_MODEL, dmMobileNvInfo, DM_NV_ITEM_SIZE);    //նͺ
    //snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_MODEL], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_MODEL], DM_NV_ITEM_SIZE - 1, "%s", "OK-W6");

    memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    sc_cfg_get(DM_HW_VERSION, dmMobileNvInfo, DM_NV_ITEM_SIZE); // Ӳ汾
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_FIRMWARE_VERSION], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);

    /*汾*/
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_SYSTEM_VERSION], DM_NV_ITEM_SIZE - 1, "%s", "1.0");

    /*ƽ̨CPUͺ */
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CPU], DM_NV_ITEM_SIZE - 1, "%s", "ZX297520V3");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_ROM], DM_NV_ITEM_SIZE - 1, "%s", "128MB");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_RAM], DM_NV_ITEM_SIZE - 1, "%s", "128MB");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_TYPE], DM_NV_ITEM_SIZE - 1, "%s", "1");

    //memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    //sc_cfg_get(NV_ZICCID, dmMobileNvInfo, DM_NV_ITEM_SIZE);
    //snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_ICCID1], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_ICCID1], DM_NV_ITEM_SIZE - 1, "%s", "89860065101750003921");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_ICCID2], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    sc_cfg_get(NV_SIM_IMSI, dmMobileNvInfo, DM_NV_ITEM_SIZE);
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_IMSI1], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_IMSI2], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_MAC], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    sc_cfg_get(NV_CELL_ID, dmMobileNvInfo, DM_NV_ITEM_SIZE);
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CELLID], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);

    memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    sc_cfg_get(NV_LAC_CODE, dmMobileNvInfo, DM_NV_ITEM_SIZE);
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_LAC], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);

    /* 4g 3G 2G*/
    dm_data_channel_set(dmMobileRegInfo);

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_DATA_CARD], DM_NV_ITEM_SIZE - 1, "%s", "2");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_MASTER_STATUS], DM_NV_ITEM_SIZE - 1, "%s", "0");

    CALL(dm_register_time_set(dmControlInfo, (void *) dmMobileRegInfo));

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_SLOT_QUANTITY], DM_NV_ITEM_SIZE - 1, "%s", "1");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_DATA_CARD2], DM_NV_ITEM_SIZE - 1, "%s", "-1");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_SLOT_SERVICE1], DM_NV_ITEM_SIZE - 1, "%s", "5");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_SLOT_SERVICE2], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    /* slot-network ֵֵͨͬ(dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL])*/
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_SLOT_NETWORK1], DM_NV_ITEM_SIZE - 1, "%s", dmMobileRegInfo->value[DM_INFO_MOBILE_CHNL]);

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_SLOT_NETWORK2], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_LAC2], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_CELLID2], DM_NV_ITEM_SIZE - 1, "%s", "unknow");

    //memset(dmMobileNvInfo, 0, DM_NV_ITEM_SIZE);
    //sc_cfg_get(DM_IN_TYPE, dmMobileNvInfo, DM_NV_ITEM_SIZE);    /*ͺ*/
    //snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_TERMINAL_INTTYPE], DM_NV_ITEM_SIZE - 1, "%s", dmMobileNvInfo);
    snprintf(dmMobileRegInfo->value[DM_INFO_MOBILE_TERMINAL_INTTYPE], DM_NV_ITEM_SIZE - 1, "%s", "OK-W6");

    log_debug("dm mobile get register from NV now ...\n");

    for (index = 0; index < DM_INFO_MOBILE_MAX; index++)
    {
        log_debug("Mobile info VALUE[%d] = %s\n", index, (dmMobileRegInfo->value) [index]);
    }
#endif
    return DM_ERR_SUCCESS;
}

/**
* ƣdm_unicom_url_info_create
* ͨdm ulrϢ,ַ˿ںŵ
* ˵
*   ֵɹ 0: ʧ 
* ˵
*/
static int  dm_unicom_url_info_create(struct DM_CONTROl_INFO *dmControlInfo, struct DM_UNICOM_REGISTER_INFO *dmUnicomRegInfo, DM_UNICOM_HTTP_URL_INFO *dmUnicomUrlInfo)
{
    int ret = 0;
    CHECK_PARAM(dmControlInfo);
    CHECK_PARAM(dmUnicomRegInfo);
    CHECK_PARAM(dmUnicomUrlInfo);
    CHECK_PARAM(dmUnicomUrlInfo->urlBuffer);

    memset(dmUnicomUrlInfo, 0, sizeof(DM_UNICOM_HTTP_URL_INFO));
    ret = snprintf(dmUnicomUrlInfo->urlBuffer, URL_INFO_LINE_SIZE - 1, "http://%s:%ld/register?ver=1.1&model=%s&manuf=%s&sign=", dmControlInfo->hostName, dmControlInfo->portNum, & ((dmUnicomRegInfo->value) [DM_INFO_MODEL][0]), & ((dmUnicomRegInfo->value) [DM_INFO_MANUF][0]));
    if (ret <= 0)
        return DM_ERR_URL_INFO_CREATE_ERROR;

    ret = snprintf(dmUnicomUrlInfo->hostName, DM_NV_ITEM_SIZE - 1, "%s", dmControlInfo->hostName);
    if (ret < 0)
        return DM_ERR_URL_INFO_CREATE_ERROR;

    ret = snprintf(dmUnicomUrlInfo->ipAddr, 31, "%s", dmControlInfo->ipAddr);
    if (ret < 0)
        return DM_ERR_URL_INFO_CREATE_ERROR;

    dmUnicomUrlInfo->portNum = dmControlInfo->portNum;

    log_debug("dm unicom url-info debug: ulr(%s), portNum(%ld), hostName(%s)\n ", dmUnicomUrlInfo->urlBuffer, dmUnicomUrlInfo->portNum, dmUnicomUrlInfo->hostName);
    return DM_ERR_SUCCESS;
}


/**
* ƣdm_register_thread_release
* ֪ͨע̣߳˳߳
* ˵
*   ֵɹ 0: ʧ 
* ˵ʹpthread_xxxغ˳(ϵͳ˯ڣ뱣ִ֤)
*/
static int dm_register_thread_release(struct DM_CONTROl_INFO *dmControlInfo)
{
    CHECK_PARAM(dmControlInfo);
    int ret = 0;
    long size = 0;
    MSG_BUF msgBuf = {0};
    msgBuf.usMsgCmd  = DM_MSG_CMD_REGISTER_THREAD_RELEASE;
    size = sizeof(MSG_BUF) - sizeof(long);
    msgBuf.lMsgType = 1;

    /* Ϣ ע߳ ע */
    ret = msgsnd(dmControlInfo->msgQueueIdReg, (void*) & (msgBuf), size, 0);
    if (ret < 0)
    {
        log_err("unicom thread release msg errro ret(%d) \n", ret);
        return DM_ERR_QUEUE_SEND_ERROR;
    }

    return DM_ERR_SUCCESS;
}

/**
* ƣdm_resource_clean
* עɣڴͷ, ֪ͨ߳˳
* ˵
*   ֵ
* ˵
*/
static void dm_resource_clean(struct DM_CONTROl_INFO *dmControlInfo, void *dataInfo)
{
    DM_UNICOM_HTTP_DATA_INFO *dmUnicomHttpDataInfo = NULL;
    DM_MOBILE_HTTPS_DATA_INFO *dmMobileHttpsDataInfo = NULL;

    if (DM_UNICOM == dmControlInfo->dmType)
    {
        dmUnicomHttpDataInfo  = (DM_UNICOM_HTTP_DATA_INFO *) dataInfo;
        if (dmUnicomHttpDataInfo)
        {
            if (dmUnicomHttpDataInfo->dataBuffer)
            {
                free(dmUnicomHttpDataInfo->dataBuffer);
                dmUnicomHttpDataInfo->dataBuffer = NULL;
                dmUnicomHttpDataInfo->dataLen = 0;
            }
        }
    }

    if (DM_MOBILE == dmControlInfo->dmType)
    {
        dmMobileHttpsDataInfo  = (DM_MOBILE_HTTPS_DATA_INFO *) dataInfo;
        if (dmMobileHttpsDataInfo)
        {
            if (dmMobileHttpsDataInfo->dataBuffer)
            {
                free(dmMobileHttpsDataInfo->dataBuffer);
                dmMobileHttpsDataInfo->dataBuffer = NULL;
                dmMobileHttpsDataInfo->dataLen = 0;
            }
        }
    }

    /*NV */
    if (DM_REGISTERED_YES == dmControlInfo->registerFlag)
    {
        char registerRounds[4] = {0};
        snprintf(registerRounds, 3, "%d", 0);
        sc_cfg_set(DM_REGISTERED_ROUND, registerRounds);

        dm_rtc_timer_del();
        log_debug("dm register success now...\n");
    }

    if ((DM_UNICOM == dmControlInfo->dmType) && (10 < dmControlInfo->registeredRounds || 3 < dmControlInfo->currentRounds))
    {
        dm_rtc_timer_del();
        log_debug("dm unicom have registered total registerRounds(%d) or  rounds(%d) on starting up\n", dmControlInfo->registeredRounds, dmControlInfo->currentRounds);
    }

    if ((DM_MOBILE == dmControlInfo->dmType) && (10 < dmControlInfo->currentTimes))
    {
        dm_rtc_timer_del();
        log_debug("dm unicom have registered total times(%d) on starting up\n", dmControlInfo->currentTimes);
    }
}

/**
* ƣdm_unicom_register_start
* ͨDMע̿ʼ
* ˵
*   ֵɹ 0: ʧ 
* ˵ͨDMעύʼעϸ:
*           ͨЭ淶ָע᲻ɹҪжγע
*		   1Сʱע
*           2ÿο3֣ÿּ6СʱÿְΣÿεļ6ӣһοೢ9
*           3ÿһֵĿÿһĿҪ棬ΪעϢһ֣ύͨ
*           4ο󣬵עܵﵽ10,11֣ûעɹն˽Զעᡣ
*/
static int dm_unicom_register_start(struct DM_CONTROl_INFO *dmControlInfo)
{
    char *dataBuf = NULL;
    int dataSize = 0;
    int ret = 0;
    static int currentRounds = 1;
    static int currentTimes = 1;
    DM_UNICOM_HTTP_URL_INFO  dmUnicomUrlInfo ;
    DM_UNICOM_HTTP_DATA_INFO dmUnicomHttpDataInfo ;
    struct DM_UNICOM_REGISTER_INFO dmUnicomRegInfo ;

    dataSize = (DM_NV_ITEM_SIZE + DM_NV_ITEM_SIZE / 4 + 5) * DM_INFO_UNICOM_MAX;

    if (!(dataBuf = (char *) malloc(dataSize)))
    {
        log_err("dm unicom start reg malloc mem error\n");
        return DM_ERR_PARAM_NULL;
    }
    memset(&dmUnicomHttpDataInfo, 0, sizeof(DM_UNICOM_HTTP_DATA_INFO));
    memset(dataBuf, 0, dataSize);
    dmUnicomHttpDataInfo.dataBuffer = dataBuf;
    dmUnicomHttpDataInfo.dataLen = dataSize;

    /* ն˳ܵС 10 ǰԵ3֣ÿֲ3 */
    if ((dmControlInfo->registeredRounds <= DM_ROUND_TOTAL) && (currentRounds <= 3))
    {
        /* עtagֵ(tag, ͨں˶Եǰն˳עĴ) */
        dmControlInfo->currentTimes = currentTimes;
        dmControlInfo->currentRounds = currentRounds;

        /*ܵעǰб棬Э淶ָԤĳһУûػˣҪ*/
        char registerRound[5] = {0};
        snprintf(registerRound, 4, "%d", dmControlInfo->registeredRounds);
        sc_cfg_set(DM_REGISTERED_ROUND,  registerRound);

        ret = dm_unicom_register_info_create(dmControlInfo, &dmUnicomRegInfo);
        if(DM_ERR_SUCCESS != ret)
            goto ErrRet;

        ret = dm_unicom_json_info_create(&dmUnicomRegInfo, &dmUnicomHttpDataInfo);

        ret = dm_unicom_url_info_create(dmControlInfo, &dmUnicomRegInfo, &dmUnicomUrlInfo);
        if(DM_ERR_SUCCESS != ret)
            goto ErrRet; //klockworks issue.

        ret = dm_unicom_http_request(&dmUnicomUrlInfo, &dmUnicomHttpDataInfo);
        log_debug("dm register total rounds(%d), currentRounds(%d), currentTimes(%d), http-ret(%d)\n", dmControlInfo->registeredRounds, currentRounds, currentTimes, ret);

        if (0 == ret)
        {
            char registerFlag[4] = {0};
            char registerRounds[4] = {0};

            dmControlInfo->registerFlag = DM_REGISTERED_YES;
            snprintf(registerFlag, 3, "%d", dmControlInfo->registerFlag);
            sc_cfg_set("registerFlag", registerFlag);

            snprintf(registerRounds, 3, "%d", 0);
            sc_cfg_set("registerRounds", registerRounds);

            dm_resource_clean(dmControlInfo, (void *) &dmUnicomHttpDataInfo);
            return DM_ERR_SUCCESS;
        }

        /* ǷشҪԣ˴β볢Դ */
        if (ret == ERR_SOCKET_NO_IP)
        {
            /*Ƿ󣬵ǰ ܼͳ */
            log_debug("dm start register some net's error occur \n");
            dm_resource_clean(dmControlInfo, &dmUnicomHttpDataInfo);

            /*
             *δ󣬽йܡΪѾcheck·Ok,ֲIPַ
             */
            dm_timer_add(3, 0);
            return DM_ERR_NET_ERROR;
        }

        /* ÿ波3Σÿ */
        if (currentTimes < 3)
        {
            currentTimes++;
            dmControlInfo->currentTimes = currentTimes;
            /* ÿγԶʱ */
            dm_timer_add(dmControlInfo->secsEveryTime * 60, DM_TIMER_ID_UNICOM_EVERY_TIMES);
        }
        else /* ÿ */
        {
            currentTimes = 1;  	 /* ÿУ´㿪ʼ */
            dmControlInfo->currentTimes = currentTimes;

            currentRounds++;
            dmControlInfo->currentRounds = currentRounds;

            /*Э淶һοೢ3*/
            if (dmControlInfo->currentRounds > 3)
            {
                dm_resource_clean(dmControlInfo, &dmUnicomHttpDataInfo);
                log_debug("dm unicom currentRound(%d)\n", dmControlInfo->currentRounds);
                return DM_ERR_SERVER_ERROR;
            }

            dmControlInfo->registeredRounds++;
            if (10 < dmControlInfo->registeredRounds)
            {
                log_debug("dm unicom registeredRound(%d)\n", dmControlInfo->registeredRounds);
                dm_resource_clean(dmControlInfo, &dmUnicomHttpDataInfo);
                return DM_ERR_SERVER_ERROR;
            }
            /* ÿֳԶʱ */
            dm_timer_add(dmControlInfo->secsEveryRound * 60, DM_TIMER_ID_UNICOM_EVERY_ROUND);
        }

        log_debug("dm start register start: debug info as following ************\n");
        log_debug("dm total rounds-next(%d), currentRounds-next(%d), currentTimes-next(%d)\n", dmControlInfo->registeredRounds, currentRounds, currentTimes);
        log_debug("dm register secTimer(%d), secsEveryRound(%d) , secsEveryTime(%d), netStatus(%d), globalRomaing(%d)\n",
                  dmControlInfo->secTimer,
                  dmControlInfo->secsEveryRound,
                  dmControlInfo->secsEveryTime,
                  dmControlInfo->netStatus,
                  dmControlInfo->globalRomaing);

        dm_resource_clean(dmControlInfo, &dmUnicomHttpDataInfo);
        return DM_ERR_SERVER_ERROR;
    }

    dm_resource_clean(dmControlInfo, &dmUnicomHttpDataInfo);
    return DM_ERR_RETRY_ERROR;

ErrRet:
    free(dataBuf);
    return ret; //klockworks issue.
}

/**
* ƣdm_unicom_register_request_handle
* ͨDMע
* ˵
*   ֵɹ 0: ʧ 
* ˵
*/
static int dm_unicom_register_request_handle(struct DM_CONTROl_INFO *dmControlInfo)
{
    CHECK_PARAM(dmControlInfo);

    int checkTimes = 10;
    char romaingStatus[DM_NV_ITEM_SIZE] = {0};
    char netStatus[DM_NV_ITEM_SIZE] = {0};
    char cardType[4] = {0};

    /*cmt ȡ ״ֵ̬*/
    CALL(sc_cfg_get(NV_PDP_STATUS, netStatus, sizeof(netStatus)));
    CALL(sc_cfg_get(NV_SIMCARD_ROAM, romaingStatus, sizeof(romaingStatus)));
    CALL(sc_cfg_get(NV_SIMCARD_TYPE, cardType, sizeof(cardType)));

    dmControlInfo->netStatus = atoi(netStatus);
    dmControlInfo->cardType = atoi(cardType);

    if (0 ==  strcmp(romaingStatus, "Internal"))
    {
        dmControlInfo->globalRomaing = 1;
    }
    else if (0 == strcmp(romaingStatus, "Home"))
    {
        dmControlInfo->globalRomaing = 0;
    }
    else
    {
        log_err("dm global romaing some error, romaingStatus(%s) \n ", romaingStatus);
        return DM_ERR_WAITING_NET_STATUS_CHANAGE;
    }

    /*  1·رգ
     *  2·򿪣ڹ
     *     עᡣ
     */
    if ((dmControlInfo->netStatus == 0) || ((dmControlInfo->netStatus == 1) && (dmControlInfo->globalRomaing == 1)))
    {
        /*Ƕʱʱ·ûд򿪣ȴ·Ϣ, ע */
        dmControlInfo->needRegForNetStatusChange = 1;
        log_debug(" startup_proc now, net status(%d), romaing status(%d), needRegForNetStatusChange(%d)\n", dmControlInfo->netStatus, dmControlInfo->globalRomaing, dmControlInfo->needRegForNetStatusChange);
        return DM_ERR_WAITING_NET_STATUS_CHANAGE;
    }
    dmControlInfo->needRegForNetStatusChange = 0;

    /*PDPϢϱʹ ·ɼ */
    while (checkTimes-- >= 0)
    {
        if (1 == default_route_check())
            break;
        usleep(1000 * 500);
        if (0 == checkTimes)
        {
            log_debug("unicom startup check rout times(%d)\n", checkTimes);
            return DM_ERR_CHECK_ROUTE_ERROR;
        }
    }

    /*ctm ע */
    CALL(dm_unicom_register_start(dmControlInfo));

    return DM_ERR_SUCCESS;
}

/**
* ƣdm_register_request_handle
* DMע
* ˵
*   ֵɹ 0: ʧ 
* ˵
*/
static int dm_register_request_handle(struct DM_CONTROl_INFO *dmControlInfo)
{
    CHECK_PARAM(dmControlInfo);

    if (DM_MOBILE == dmControlInfo->dmType)
    {
        char imei[DM_NV_ITEM_SIZE] = {0};
        char iccid[DM_NV_ITEM_SIZE] = {0};
        char imeiPrevious[DM_NV_ITEM_SIZE] = {0};
        char iccidPrevious[DM_NV_ITEM_SIZE] = {0};
        char cardType[4] = {0};
        char versionPrevious[DM_NV_ITEM_SIZE] = {0};
        char version[DM_NV_ITEM_SIZE] = {0};

        CALL(sc_cfg_get(NV_IMEI, imei, sizeof(imei)));
        CALL(sc_cfg_get(NV_ZICCID, iccid, sizeof(iccid)));

        CALL(sc_cfg_get(DM_IMEI_PREVIOUS, imeiPrevious, sizeof(imeiPrevious)));
        CALL(sc_cfg_get(DM_ICCID_PREVIOUS, iccidPrevious, sizeof(iccidPrevious)));
        CALL(sc_cfg_get(NV_SIMCARD_TYPE, cardType, sizeof(cardType)));

        CALL(sc_cfg_get(DM_VERSION_PREVIOU, versionPrevious, sizeof(versionPrevious)));
        CALL(sc_cfg_get(DM_SW_VERSION, version, sizeof(version)));

        snprintf(dmControlInfo->imeiPrevious, sizeof(dmControlInfo->imeiPrevious)-1, "%s", imeiPrevious);
        snprintf(dmControlInfo->iccidPrevious, sizeof(dmControlInfo->iccidPrevious)-1, "%s", iccidPrevious); //klockworks issue.

        dmControlInfo->cardType = atoi(cardType);

        /*
         *  1ƶDM ѾעǸ sim Э涨Ҫע
         *  2ƶDM fotaҪ·ע,ǰİ汾Žж
         *  eg: B11 -> B12 -> B11 ע
         */
        if ((DM_REGISTERED_YES == dmControlInfo->registerFlag)
                && (!strcmp(dmControlInfo->imeiPrevious, imei)
                    && (!strcmp(dmControlInfo->iccidPrevious, iccid)))
                && (!strcmp(versionPrevious, version)))
        {
            log_debug("dm mobile imei and iccid have no change, system have no updated now (fota), versionPrevious(%s), vesion(%s)\n", versionPrevious, version);
            return DM_ERR_DEVICE_REGISTERED;
        }

        /*ƶDM ƶ ע */
        if (1 != dmControlInfo->cardType)
        {
            log_debug("dm mobile register, no-mobile-card cardType(%s) \n", cardType);
            return DM_ERR_NO_MOBILE_CARD_ERROR;
        }

        CALL(dm_mobile_register_request_handle(dmControlInfo));
    }
    else if (DM_UNICOM == dmControlInfo->dmType)
    {
        CALL(dm_unicom_register_request_handle(dmControlInfo));
    }
    else if (DM_TELECOM == dmControlInfo->dmType)
    {
        CALL(dm_telecom_register_request_handle(dmControlInfo));
    }
    else
    {
        log_err("dm don't not know the register type(%d) now ...\n", dmControlInfo->dmType);
    }
    return DM_ERR_SUCCESS;
}


/**
 * ƣdm_main_thread_release
 * ֪ͨdmģ̣߳˳߳
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int  dm_main_thread_release(struct DM_CONTROl_INFO *dmControlInfo)
{
    CHECK_PARAM(dmControlInfo);
    int ret = 0;
    long size = 0;
    MSG_BUF msgBuf = {0};
    msgBuf.usMsgCmd  = DM_MSG_CMD_MAIN_THREAD_EXIT;
    size = sizeof(MSG_BUF) - sizeof(long);
    msgBuf.lMsgType = 1;

    dmControlInfo->registErThreadFlag = 0;    /*߳˳ǣ ͬʱϢ̣߳߳˳*/

    ret = msgsnd(dmControlInfo->msgQueueId, (void*) & (msgBuf), size, 0);
    if (ret < 0)
    {
        log_err("main thread release msg errro \n");
        return DM_ERR_QUEUE_SEND_ERROR;
    }

    return DM_ERR_SUCCESS;
}

/**
* ƣdm_register_queue_msg_proc
* DMע߳Ϣ
* ˵
*   ֵ
* ˵ע̣߳ڽ̷߳͵עϢ,߳˳Ϣ
*           (עעصıãڴ߳
* 	 		ã߳вܶ漰ͬĸֵֻܶȡ)
*/
static void dm_register_queue_msg_proc(MSG_BUF *pstMsg, struct DM_CONTROl_INFO *dmControlInfo)
{
    switch (pstMsg->usMsgCmd)
    {
    case DM_MSG_CMD_REGISTER_REQUEST:
    {
        dm_register_request_handle(dmControlInfo);
        break;
    }

    case DM_MSG_CMD_REGISTER_THREAD_RELEASE:
    {
        dm_main_thread_release(dmControlInfo);
        log_debug("dm register thread readly to exit...\n");
        break;
    }

    default:
        log_debug("dm queue msg proc\n");
    }
}

/**
 * ƣdm_server_thread_entry
 * dmע߳ں
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static void* dm_server_thread_entry(void* args)
{
    int ret = 0;
    MSG_BUF dmMsg = {0};
    long msgSize  = 0;
    pthread_attr_t  threadAttr;

    if (!args)
    {
        log_err("dm register thread param error \n");
        return NULL;
    }

    struct DM_CONTROl_INFO *dmControlInfo = (struct DM_CONTROl_INFO *) args;

    pthread_attr_init(&threadAttr);
    pthread_attr_setstacksize(&threadAttr, 16 * 1024);
    prctl(PR_SET_NAME, "dm_register_server", 0, 0, 0);

    log_debug("dm register thread start now...\n");
    dmControlInfo->registErThreadFlag = 1;
    while (dmControlInfo->registErThreadFlag)
    {
        memset((void*)(&dmMsg), 0, sizeof(dmMsg));
        msgSize = sizeof(MSG_BUF) - sizeof(long);
        if (0 < (ret = msgrcv(dmControlInfo->msgQueueIdReg, &dmMsg, msgSize, 0, MSG_NOERROR)))
        {
            log_debug("dm register thread recv msg now: ret(%d), dmMsg.src_id(%ld),  dmMsg.usMsgCmd(0x%08x), dmMsg.usDatalen(%d), dmMsg.auDataBuffer(%s)\n",
                      ret, dmMsg.src_id, dmMsg.usMsgCmd, dmMsg.usDataLen, dmMsg.aucDataBuf);
            dm_wake_lock("dm_lock_dm");
            dm_register_queue_msg_proc(&dmMsg, dmControlInfo);
            dm_wake_unlock("dm_lock_dm");
        }
        else
        {
            log_err(" dm server thread rec data error ,err-code\n");
        }
    }

    pthread_attr_destroy(&threadAttr); //klockworks issue.
    log_debug("dm reg server exit now... \n");
    return NULL;
}

/**
 * ƣdm_register_thread_init
 * DMע̳߳ʼ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_register_thread_init(struct DM_CONTROl_INFO *dmControlInfo)
{
    CHECK_PARAM(dmControlInfo);

    if (-1 == ((dmControlInfo->msgQueueIdReg) = msgget(IPC_PRIVATE, IPC_CREAT | 0600)))
    {
        log_err("Create MSG_ID_DMREG queue faild! (*msgId)(%d)\n", dmControlInfo->msgQueueIdReg);
        return DM_ERR_CREATE_MSG_QUEUE_ERROR;
    }

    if (0 != pthread_create(& (dmControlInfo->pid), NULL, dm_server_thread_entry, dmControlInfo))
    {
        log_err("dm servers pthread_create error\n");
        return DM_ERR_CREATE_THERAD_ERROR;
    }

    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_init_unicom_control_info
 * ʼDMϢ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_unicom_control_info_init(struct DM_CONTROl_INFO *dmControlInfo)
{
    char secsEveryRound[DM_NV_ITEM_SIZE] = {0};
    char secsEveryTime[DM_NV_ITEM_SIZE] = {0};
    char registeredRounds[DM_NV_ITEM_SIZE] = {0};
    char secTimer[DM_NV_ITEM_SIZE] = {0};
    char hostName[DM_NV_ITEM_SIZE] = {0};
    char portNum[12] = {0};
    char netStatus[4] = {0};

    CHECK_PARAM(dmControlInfo);

    CALL(sc_cfg_get(DM_SECS_EVERY_ROUND, secsEveryRound, DM_NV_ITEM_SIZE));
    CALL(sc_cfg_get(DM_SECS_EVERY_TIME, secsEveryTime, DM_NV_ITEM_SIZE));
    CALL(sc_cfg_get(DM_REGISTERED_ROUND, registeredRounds, DM_NV_ITEM_SIZE));
    CALL(sc_cfg_get(DM_SEC_TIMER, secTimer, DM_NV_ITEM_SIZE));
    CALL(sc_cfg_get(DM_HOST_NAME, hostName, DM_NV_ITEM_SIZE));
    CALL(sc_cfg_get(DM_PORT_NUM, portNum, sizeof(portNum))); //klockworks issue.
    CALL(sc_cfg_get(NV_PDP_STATUS, netStatus, 4));

    dmControlInfo->secsEveryRound = atoi(secsEveryRound);
    dmControlInfo->secsEveryTime = atoi(secsEveryTime);
	dmControlInfo->registeredRounds = atoi(registeredRounds);
	/*klocwork 3 SV.TAINTED.BINOP, add if condition*/
	if((dmControlInfo->registeredRounds >= 0) && (dmControlInfo->registeredRounds < 100))
	{
		dmControlInfo->registeredRounds = dmControlInfo->registeredRounds + 1;
	}
    
    dmControlInfo->secTimer = atoi(secTimer);
    dmControlInfo->netStatus = atoi(netStatus);

    dmControlInfo->portNum = atol(portNum);
    snprintf(dmControlInfo->hostName, DM_NV_ITEM_SIZE - 1, "%s", hostName);

    log_debug("dm unicom init control info: secsEveryTime(%d), secsEveryRound(%d), registeredRounds(%d), secTimer(%d), hostName(%s), portNum(%ld), netStatus(%d) \n",
              dmControlInfo->secsEveryTime,
              dmControlInfo->secsEveryRound,
              dmControlInfo->registeredRounds,
              dmControlInfo->secTimer,
              dmControlInfo->hostName,
              dmControlInfo->portNum,
              dmControlInfo->netStatus);

    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_init_mobile_control_info
 * ʼƶDMϢ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_mobile_control_info_init(struct DM_CONTROl_INFO * dmControlInfo)
{
    char secTimer[DM_NV_ITEM_SIZE] = {0};
    char hostName[DM_NV_ITEM_SIZE] = {0};
    char portNum[DM_NV_ITEM_SIZE] = {0};
    char imeiPrevious[DM_NV_ITEM_SIZE] = {0};
    char iccidPrevious[DM_NV_ITEM_SIZE] = {0};
    char cardType[4] = {0};
    char secsEveryRound[DM_NV_ITEM_SIZE] = {0};

    CHECK_PARAM(dmControlInfo);
    CHECK_PARAM(dmControlInfo);

    CALL(sc_cfg_get(DM_SEC_TIMER, secTimer, sizeof(secTimer)));
    CALL(sc_cfg_get(DM_HOST_NAME, hostName, sizeof(hostName)));
    CALL(sc_cfg_get(DM_PORT_NUM, portNum, sizeof(portNum)));
    CALL(sc_cfg_get(NV_SIMCARD_TYPE, cardType, sizeof(cardType)));
    CALL(sc_cfg_get(DM_IMEI_PREVIOUS, imeiPrevious, sizeof(imeiPrevious)));
    CALL(sc_cfg_get(DM_ICCID_PREVIOUS, iccidPrevious, sizeof(iccidPrevious)));
    CALL(sc_cfg_get(DM_SECS_EVERY_ROUND, secsEveryRound, sizeof(secsEveryRound)));

    dmControlInfo->secTimer = atoi(secTimer);
    dmControlInfo->portNum = atoi(portNum);
    snprintf(dmControlInfo->hostName, 31, "%s", hostName);
    dmControlInfo->cardType = atoi(cardType);
    dmControlInfo->secsEveryRound = atoi(secsEveryRound);

    snprintf(dmControlInfo->imeiPrevious, 31, "%s", imeiPrevious);
    snprintf(dmControlInfo->iccidPrevious, 31, "%s", iccidPrevious);

    log_debug("dm init mobile control info: secsEveryRound(%d),secTimer(%d), cardType(%d), portNum(%d), hostName(%s), iccidPrevious(%s), imeiPrevious(%s)\n",
              dmControlInfo->secsEveryRound,
              dmControlInfo->secTimer,
              dmControlInfo->cardType,
              dmControlInfo->portNum,
              dmControlInfo->hostName,
              dmControlInfo->iccidPrevious,
              dmControlInfo->imeiPrevious);
    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_telecom_control_info_init
 * ʼDMϢ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_telecom_control_info_init(struct DM_CONTROl_INFO * dmControlInfo)
{
    char hostName[DM_NV_ITEM_SIZE] = {0};
    char portNum[DM_NV_ITEM_SIZE] = {0};

    char imeiPrevious[DM_NV_ITEM_SIZE] = {0};
    char iccidPrevious[DM_NV_ITEM_SIZE] = {0};
    char cardType[4] = {0};
    char secsEveryRound[DM_NV_ITEM_SIZE] = {0};
    char secsEveryTime[DM_NV_ITEM_SIZE] = {0};

    CHECK_PARAM(dmControlInfo);

    CALL(sc_cfg_get(DM_HOST_NAME, hostName, sizeof(hostName)));
    CALL(sc_cfg_get(DM_PORT_NUM, portNum, sizeof(portNum)));
    //CALL(sc_cfg_get(NV_SIMCARD_TYPE, cardType, sizeof(cardType)));
    CALL(sc_cfg_get(DM_IMEI_PREVIOUS, imeiPrevious, sizeof(imeiPrevious)));
    CALL(sc_cfg_get(DM_ICCID_PREVIOUS, iccidPrevious, sizeof(iccidPrevious)));
    CALL(sc_cfg_get(DM_SECS_EVERY_ROUND, secsEveryRound, sizeof(secsEveryRound)));
    CALL(sc_cfg_get(DM_SECS_EVERY_TIME, secsEveryTime, sizeof(secsEveryTime)));

    dmControlInfo->portNum = atoi(portNum);
    snprintf(dmControlInfo->hostName, 31, "%s", hostName);
    dmControlInfo->cardType = atoi(cardType);
    dmControlInfo->secsEveryRound = atoi(secsEveryRound);
    dmControlInfo->secsEveryTime = atoi(secsEveryTime);

    snprintf(dmControlInfo->imeiPrevious, 31, "%s", imeiPrevious);
    snprintf(dmControlInfo->iccidPrevious, 31, "%s", iccidPrevious);

    log_debug("dm init mobile control info: secsEveryRound(%d),secTimer(%d), cardType(%d), portNum(%d), hostName(%s), iccidPrevious(%s), imeiPrevious(%s)\n",
              dmControlInfo->secsEveryRound,
              dmControlInfo->secTimer,
              dmControlInfo->cardType,
              dmControlInfo->portNum,
              dmControlInfo->hostName,
              dmControlInfo->iccidPrevious,
              dmControlInfo->imeiPrevious);
    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_control_info_init
 * ʼDMϢ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_control_info_init(struct DM_CONTROl_INFO *dmControlInfo)
{
    CHECK_PARAM(dmControlInfo);
    memset((void*) dmControlInfo, 0, sizeof(struct DM_CONTROl_INFO));

    CALL(dm_get_device_type(dmControlInfo));

    if (DM_MOBILE == dmControlInfo->dmType)
    {
        CALL(dm_mobile_control_info_init(dmControlInfo));
    }
    else if (DM_UNICOM == dmControlInfo->dmType)
    {
        CALL(dm_unicom_control_info_init(dmControlInfo));
    }
    else if (DM_TELECOM == dmControlInfo->dmType)
    {
        CALL(dm_telecom_control_info_init(dmControlInfo));
    }
    else
    {
        log_err("the device type is none \n");
        return DM_ERR_DEVICE_TYPE_IS_NONE;
    }

    log_debug("dm control info init success, device dmType(%d),registeredRounds(%d), secsEveryTime(%d), secsEveryRound(%d) ,cardType(%d), hostNameUnicom(%s), portNum(%u), imeiPrevious(%s), iccidPrevious(%s)\n",
              dmControlInfo->dmType,
              dmControlInfo->registeredRounds,
              dmControlInfo->secsEveryTime,
              dmControlInfo->secsEveryRound,
              dmControlInfo->cardType,
              dmControlInfo->hostName,
              dmControlInfo->portNum,
              dmControlInfo->imeiPrevious,
              dmControlInfo->iccidPrevious);
    return DM_ERR_SUCCESS;
}

BOOL dm_telecom_check_iccid_registed(void)
{
    char iccid[DM_NV_ITEM_SIZE] = {0};
    char iccidPrevious[DM_NV_ITEM_SIZE] = {0};
    char versionPrevious[DM_NV_ITEM_SIZE] = {0};
    char version[DM_NV_ITEM_SIZE] = {0};

    CALL(sc_cfg_get(NV_ZICCID, iccid, sizeof(iccid)));
    CALL(sc_cfg_get(DM_ICCID_PREVIOUS, iccidPrevious, sizeof(iccidPrevious)));
    CALL(sc_cfg_get(DM_VERSION_PREVIOU, versionPrevious, sizeof(versionPrevious)));
    CALL(sc_cfg_get(DM_SW_VERSION, version, sizeof(version)));

    if (0 != strcmp("", versionPrevious) &&
            0 != strcmp(versionPrevious, version))
    {
        log_debug("[dm_telecom] software version update!\n");
        sc_cfg_set(DM_ICCID_PREVIOUS, "0");
        return FALSE;
    }

    if (0 != strcmp("", iccidPrevious) &&
            0 == strcmp(iccid, iccidPrevious))
    {
        log_debug("[dm_telecom] check_ueiccid same!\n");
        return TRUE;
    }

    return FALSE;
}

BOOL dm_check_is_chinatelecom(void)
{
    char num_oper[DM_NV_ITEM_SIZE] = {0};
    sc_cfg_get(NV_OPER_NUM, num_oper, sizeof(num_oper));
    if (strcmp(num_oper, "46003") == 0 ||
            strcmp(num_oper, "46005") == 0 ||
            strcmp(num_oper, "46011") == 0)
    {
        return TRUE;
    }

    return FALSE;
}

/**
 * ƣdm_unicom_register_info_create
 * : ͨDMЭ淶ָעϢ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_telecom_register_info_create(struct DM_CONTROl_INFO *dmControlInfo, struct DM_TELECOM_REGISTER_INFO *dmTelecomRegInfo)
{
    CHECK_PARAM(dmTelecomRegInfo);
    (void)dmControlInfo;

    sc_cfg_get(DM_REGVER, dmTelecomRegInfo->value[DM_INFO_REGVER], DM_NV_ITEM_SIZE);
    sc_cfg_get(DM_MEID, dmTelecomRegInfo->value[DM_INFO_MEID], DM_NV_ITEM_SIZE);
    sc_cfg_get(DM_MODEL, dmTelecomRegInfo->value[DM_INFO_MODELSMS], DM_NV_ITEM_SIZE);
    sc_cfg_get(NV_CR_VERSION, dmTelecomRegInfo->value[DM_INFO_SWVER], DM_NV_ITEM_SIZE);
    //sc_cfg_get(NV_SIM_IMSI, dmTelecomRegInfo->value[DM_INFO_SIM1CDMAIMSI], DM_NV_ITEM_SIZE);
    sc_cfg_get(DM_UETYPE, dmTelecomRegInfo->value[DM_INFO_UETYPE], DM_NV_ITEM_SIZE);
    sc_cfg_get(NV_ZICCID, dmTelecomRegInfo->value[DM_INFO_SIM1ICCID], DM_NV_ITEM_SIZE);
    sc_cfg_get(NV_SIM_IMSI, dmTelecomRegInfo->value[DM_INFO_SIM1LTEIMSI], DM_NV_ITEM_SIZE);

    sc_cfg_get(NV_IMEI, dmTelecomRegInfo->value[DM_INFO_IMEI], DM_NV_ITEM_SIZE);
    strcpy(dmTelecomRegInfo->value[DM_INFO_SIM1TYPE], "2"); //TODO

    return DM_ERR_SUCCESS;
}


/**
 * ƣdm_unicom_json_info_create
 * ͨЭ淶jsonʽ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_telecom_json_info_create(struct DM_TELECOM_REGISTER_INFO *dmTelecomRegInfo, DM_TELECOM_HTTP_DATA_INFO *httpsDataInfo)
{
    CHECK_PARAM(dmTelecomRegInfo);
    CHECK_PARAM(httpsDataInfo);

    int i = 0;
    char *pTmep = NULL;
    int jsonSize = 0;
    char *telecomItemName[] = {"REGVER", "MEID", "MODELSMS", "SWVER",
                               "SIM1CDMAIMSI", "UETYPE", "SIM1ICCID", "SIM1LTEIMSI",
                               "IMEI1", "BASEID", "SIM1TYPE", "SID",
                               "NID"
                              };
    /*json fromat as following
     * {"key1":"key1Value","key2":"key2Value", xxxxx}*/
    log_debug("dm unicom json infof ***********\n");

    pTmep = httpsDataInfo->dataBuffer;
    memset(pTmep, 0, httpsDataInfo->dataLen);
    for (i = 0; i < DM_INFO_TELECOM_MAX; i++)
    {
        if (i == 0)
            sprintf(pTmep, "{\"%s\":\"%s\"", telecomItemName[i], dmTelecomRegInfo->value[i]);
        else if (i == DM_INFO_TELECOM_MAX - 1)
            sprintf(pTmep, ",\"%s\":\"%s\"}", telecomItemName[i], dmTelecomRegInfo->value[i]);
        else
            sprintf(pTmep, ",\"%s\":\"%s\"", telecomItemName[i], dmTelecomRegInfo->value[i]);

        log_debug("json-value(%s)\n", pTmep);
        jsonSize = jsonSize + strlen(pTmep);
        pTmep = pTmep + strlen(pTmep);

        if (jsonSize > httpsDataInfo->dataLen)
            log_err("json buffer memory overwrite now! please check memory size, jsonSize(%d), httpsDatainfo-size(%d) !!!\n", jsonSize, httpsDataInfo->dataLen);
    }

    httpsDataInfo->dataLen = jsonSize;
    log_debug("create json info, json-szie(%d)\n",  jsonSize);

    return DM_ERR_SUCCESS;
}

/**
 * ƣ dm_telecom_url_info_create
 *  ƶDMķ urlϢ ַ˿ڵϢ
 * ˵
 *   ֵ ɹ 0: ʧ 
 * ˵
 */
static int dm_telecom_url_info_create(struct DM_CONTROl_INFO *dmControlInfo, struct DM_TELECOM_REGISTER_INFO *dmMobileRegInfo, DM_TELECOM_HTTP_URL_INFO *dmMobileUrlInfo)
{
    CHECK_PARAM(dmControlInfo);
    CHECK_PARAM(dmMobileRegInfo);
    CHECK_PARAM(dmMobileUrlInfo);
    CHECK_PARAM(dmMobileUrlInfo->urlBuffer);

    memset(dmMobileUrlInfo, 0, sizeof(DM_TELECOM_HTTP_URL_INFO));

    snprintf(dmMobileUrlInfo->hostName, DM_NV_ITEM_SIZE - 1, "%s", dmControlInfo->hostName);

    snprintf(dmMobileUrlInfo->ipAddr, 31, "%s", dmControlInfo->ipAddr);

    dmMobileUrlInfo->portNum = dmControlInfo->portNum;

    log_debug("dm unicom url-info debug: ulr(%s), portNum(%ld), hostName(%s)\n ", dmMobileUrlInfo->urlBuffer, dmMobileUrlInfo->portNum, dmMobileUrlInfo->hostName);

    return DM_ERR_SUCCESS;
}

BOOL dm_telecom_need_reg_check(struct DM_CONTROl_INFO *dmControlInfo)
{
    char sim_imsi[DM_NV_ITEM_SIZE] = {0};
    char cardType[4] = {0};

    /*ǷΪſ*/
    if (FALSE == dm_check_is_chinatelecom())
    {
        log_debug("[dm_telecom] selfreg_check, not telecom sim or network, return!!\n");
        return FALSE;
    }

    CALL(sc_cfg_get(NV_SIMCARD_TYPE, cardType, sizeof(cardType)));
    dmControlInfo->cardType = atoi(cardType);
    /*DM ǵſ ע */
    if (3 != dmControlInfo->cardType)
    {
        log_debug("[dm_telecom] selfreg_check, no-telecom-card cardType(%s) \n", cardType);
        return FALSE;
    }
    /*Ѿɹע*/
    if (TRUE == dm_telecom_check_iccid_registed())
    {
        log_debug("[dm_telecom] selfreg_check, selfreg success, return!!\n");
        return FALSE;
    }
    /*Դ10*/
    if (dmControlInfo->currentTimes >= DM_TELECOM_RETRY_TIMES)
    {
        log_debug("[dm_telecom] selfreg_check, selfreg retry over, return!!\n");
        return FALSE;
    }

    sc_cfg_get(NV_SIM_IMSI, sim_imsi, sizeof(sim_imsi) - 1);
    if (0 == strcmp(sim_imsi, ""))
    {
        log_debug("[dm_telecom] selfreg_check, sim_imsi is null, return!!\n");
        return FALSE;
    }

    return TRUE;
}

/**
* ƣ dm_telecom_register_start
*  DMע̿ʼ
* ˵
*   ֵ ɹ 0: ʧ 
* ˵
*/
static int dm_telecom_register_start(struct DM_CONTROl_INFO *dmControlInfo)
{
    char *dataBuf = NULL;
    int dataSize = 0;
    int ret = 0;
    char imei[DM_NV_ITEM_SIZE] = {0};
    char iccid[DM_NV_ITEM_SIZE] = {0};
    char version[DM_NV_ITEM_SIZE] = {0};
    DM_TELECOM_HTTP_URL_INFO  dmTelecomUrlInfo = {{0}, {0}, {0}, 0};
    DM_TELECOM_HTTP_DATA_INFO dmTelecomHttpsDataInfo = {0};
    struct DM_TELECOM_REGISTER_INFO dmTelecomRegInfo ;
    printf("dm_telecom_register_start enter!\n");
    dataSize = (DM_NV_ITEM_SIZE + DM_NV_ITEM_SIZE / 4 + 5) * DM_INFO_TELECOM_MAX;

    if (!(dataBuf = (char *) malloc(dataSize)))
    {
        log_err("dm unicom start reg malloc mem error\n");
        return DM_ERR_PARAM_NULL;
    }
    memset(dataBuf, 0, dataSize);
    dmTelecomHttpsDataInfo.dataBuffer = dataBuf;
    dmTelecomHttpsDataInfo.dataLen = dataSize;

    ret = dm_telecom_register_info_create(dmControlInfo, &dmTelecomRegInfo);

    ret = dm_telecom_json_info_create(&dmTelecomRegInfo, &dmTelecomHttpsDataInfo);

    ret = dm_telecom_url_info_create(dmControlInfo, &dmTelecomRegInfo, &dmTelecomUrlInfo);
    if(DM_ERR_SUCCESS != ret)
    {
        free(dataBuf);
        return ret;
    } //klockworks issue.

    printf("dm_telecom_register_start enter: telecom_https_request!\n");
    ret = telecom_https_request(&dmTelecomUrlInfo, &dmTelecomHttpsDataInfo);

    if (0 == ret)
    {
        /*עɹNV */
        char buf[4] = {0};
        dmControlInfo->registerFlag = DM_REGISTERED_YES;
        snprintf(buf, 3, "%d", dmControlInfo->registerFlag);
        sc_cfg_set("registerFlag", buf);             //ctm

        /*дIMEI ICCID */
        sc_cfg_get(NV_IMEI, imei, sizeof(imei));
        sc_cfg_get(NV_ZICCID, iccid, sizeof(iccid));
        sc_cfg_get(DM_SW_VERSION, version, sizeof(version));

        sc_cfg_set(DM_IMEI_PREVIOUS, imei);
        sc_cfg_set(DM_ICCID_PREVIOUS, iccid);

        /*д汾ţΪfota DM ע*/
        sc_cfg_set(DM_VERSION_PREVIOU, version);

        dm_resource_clean(dmControlInfo, (void *) &dmTelecomHttpsDataInfo);
        return DM_ERR_SUCCESS;
    }

    dm_resource_clean(dmControlInfo, &dmTelecomHttpsDataInfo);
    log_debug("dm start register start: debug info as following ************\n");
    log_debug("dm currentTimes(%d)\n", dmControlInfo->currentTimes);
    log_debug("dm register secTimer(%d), , secsEveryTime(%d), netStatus(%d), globalRomaing(%d)\n",
              dmControlInfo->secTimer,
              dmControlInfo->secsEveryTime,
              dmControlInfo->netStatus,
              dmControlInfo->globalRomaing);

    return DM_ERR_RETRY_ERROR;
}

/**
* ƣdm_telecom_register_request_handle
* DMע
* ˵
*   ֵɹ 0: ʧ 
* ˵
*/
static int dm_telecom_register_request_handle(struct DM_CONTROl_INFO *dmControlInfo)
{
    int ret = 0;
    int checkTimes = 100;
    char romaingStatus[DM_NV_ITEM_SIZE] = {0};
    char netStatus[DM_NV_ITEM_SIZE] = {0};

    printf("dm_telecom_register_request_handle enter!\n");
    CHECK_PARAM(dmControlInfo);

    if(FALSE == dm_telecom_need_reg_check(dmControlInfo))   //ע
    {
        log_debug("dm_telecom_register_request_handle: reg condition check fail! \n");
        return DM_ERR_RETRY_ERROR;
    }
    /*ȡ ״ֵ̬*/
    CALL(sc_cfg_get(NV_PDP_STATUS, netStatus, sizeof(netStatus)));
    CALL(sc_cfg_get(NV_SIMCARD_ROAM, romaingStatus, sizeof(romaingStatus)));

    dmControlInfo->netStatus = atoi(netStatus);

    printf("dm_telecom_register_request_handle netStatus = %d!\n", dmControlInfo->netStatus);
    if (0 ==  strcmp(romaingStatus, "Internal"))
    {
        dmControlInfo->globalRomaing = 1;
    }
    else if (0 == strcmp(romaingStatus, "Home"))
    {
        dmControlInfo->globalRomaing = 0;
    }
    else
    {
        log_err("dm global romaing some error, romaingStatus(%s) \n ", romaingStatus);
        return DM_ERR_WAITING_NET_STATUS_CHANAGE;
    }
    /* 10*/
    log_debug("dm_telecom_register_request_handle default_route_check = %d\n", default_route_check());
    if(0 == default_route_check()) //PDPδ
    {
        if((0 == dmControlInfo->currentTimes) && (g_retrytimes_before_netactived < DM_TELECOM_RETRYTIMES_BEFORE_ACTIVED)) //ûм
        {
            g_retrytimes_before_netactived++;
            CALL(dm_timer_add(10, DM_TIMER_ID_MOBILE_START));//10S
            log_debug("dm_telecom_register_request_handle addtimer: g_retrytimes_before_netactived = %d, dmType(%d), secTimer(%d)\n", g_retrytimes_before_netactived, dmControlInfo->dmType, 10);
        }
        else
        {
            dmControlInfo->currentTimes++;    /*м */
            /* ʧÿγԶʱ */
            dm_timer_add(dmControlInfo->secsEveryTime * 60, DM_TIMER_ID_MOBILE_EVERY_TIMES);
        }
        return DM_ERR_NET_ERROR;
    }

    while (checkTimes-- > 0)
    {
        if (1 == default_route_check())
            break;
        usleep(1000 * 400);
        log_debug("mobile startup check rout times(%d)\n", checkTimes);
        if (checkTimes == 0)
        {
            log_err("dm net some error occur\n");
            return DM_ERR_CHECK_ROUTE_ERROR;
        }
    }
    dmControlInfo->currentTimes++;    /*м */
    /*http */
    if (DM_ERR_SUCCESS != dm_telecom_register_start(dmControlInfo))
    {

        /* עĴС 10 ʱ */
        if (dmControlInfo->currentTimes < DM_TELECOM_RETRY_TIMES)
        {
            /*1dmControlInfo->secsEveryRound ûͨ,ĿǷԱвԣ
             *   ֵЭ涨ʵĴμʱ䣬̲ʱ䣻
             *2Ĭֵ0,ʾʹʵĴμʱ䣬0 ʾʹôõֵ */
            if (0 != dmControlInfo->secsEveryRound)
            {
                dmControlInfo->secsEveryTime = dmControlInfo->secsEveryRound;

            }
            /* ʧÿγԶʱ */
            dm_timer_add(dmControlInfo->secsEveryTime * 60, DM_TIMER_ID_MOBILE_EVERY_TIMES);
        }

        log_debug("dm mobile register error, ret(%d), currentTimes(%d), next-rtc-time(%d)\n", ret, dmControlInfo->currentTimes, dmControlInfo->secsEveryTime);
        return ret;
    }

    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_net_global_romaing_proc
 * Ϣ , blϱ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_net_global_romaing_proc(struct DM_CONTROl_INFO *dmControlInfo)
{
    CHECK_PARAM(dmControlInfo);
    char romaingStatus[DM_NV_ITEM_SIZE] = {0};
    char netStatus[4] = {0};

    CALL(sc_cfg_get(NV_PDP_STATUS, netStatus, sizeof(netStatus)));
    CALL(sc_cfg_get(NV_SIMCARD_ROAM, romaingStatus, sizeof(romaingStatus)));

    dmControlInfo->netStatus = atoi(netStatus);

    if (0 ==  strcmp(romaingStatus, "Internal"))
    {
        dmControlInfo->globalRomaing = 1;
    }
    else if (0 == strcmp(romaingStatus, "Home"))
    {
        dmControlInfo->globalRomaing = 0;
    }
    else
    {
        log_err("dm global romaing some error, romaingStatus(%s) \n ", romaingStatus);
        return DM_ERR_ROMAING_STATUS_ERROR;
    }

    /* ݿش򿪣 ҷǹ */
    if (dmControlInfo->netStatus && !dmControlInfo->globalRomaing)
    {
        /* ǵĹ: ʱʱرգȴ翪ش򿪽ע
         *  뱣֤α־ֻע߳н޸ģ(Ԥ߳̽޸)
         */
        if (dmControlInfo->needRegForNetStatusChange == 1)
        {
            /* Ϣͣʼע */
            CALL(dm_register_msg_send(dmControlInfo));
        }
    }

    log_debug("dm global-roaming change now, romaingStatus(%s)\n", romaingStatus);
    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_config_info_msg_prc
 * CpĹϢ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_config_info_msg_prc(MSG_BUF *pstMsg, struct DM_CONTROl_INFO *dmControlInfo)
{
    CHECK_PARAM(pstMsg);
    CHECK_PARAM(dmControlInfo);
    CHECK_PARAM(pstMsg->aucDataBuf);

    unsigned short dataSize = 0;
    char *buf = NULL;

    dataSize = pstMsg->usDataLen;
    if (pstMsg->usDataLen <= 0)
    {
        log_err("dm msg some err now, data-len(%d)\n", dataSize);
        return DM_ERR_PARAM_NULL;
    }

    if (strstr((char *) pstMsg->aucDataBuf, "AT+DM_REGISTER_SERVER=")) /*CP ĹϢַͨ˿Ϣ */
    {
        CALL(dm_server_domain_config_set(pstMsg, dmControlInfo));

        if (strstr((char *) pstMsg->aucDataBuf, "AT+DM_REGISTER_PORT_NUM="))
        {
            CALL(dm_server_port_num_set(pstMsg, dmControlInfo));
        }
    }
    else if (strstr((char *) pstMsg->aucDataBuf, "AT+DM_REGISTER_FLAG="))/*CP ĹϢע */
    {
        dm_register_flag_clean(dmControlInfo);
    }
    else if (strstr((char *) pstMsg->aucDataBuf, "AT+DM_REGISTER_SECS_ROUND=")) /*CP Ĺ, ÿֵʱ  */
    {
        CALL(dm_time_every_round_set(pstMsg, dmControlInfo));
    }
    else if ((buf = strstr((char *) pstMsg->aucDataBuf, "AT+DM_REGISTER_W_ONLY=")))   	/*CP Ĺ ʽ*/
    {
        /*Ԥbufݹ󣬵дNVʧ */
        char bufData[4] = {0};
        buf = buf + strlen("AT+DM_REGISTER_W_ONLY=");
        sscanf(buf, "%3[^,]", bufData);
        sc_cfg_set("wcdmaModel", bufData);
    }

    log_debug("dm config info set now: info(%s), info-size(%d)\n", pstMsg->aucDataBuf, pstMsg->usDataLen);
    return DM_ERR_SUCCESS;
}

/**
 * ƣdm_queue_msg_proc
 * DMģ߳Ϣ ߳ڽն෽Ϣ(CPֹϢ̵߳ϢʱϢ)
 * ˵
 *   ֵ
 * ˵
 */
static void dm_queue_msg_proc(MSG_BUF *pstMsg, struct DM_CONTROl_INFO *dmControlInfo)
{
    switch (pstMsg->usMsgCmd)
    {
    case RTC_MSG_ALARM_ELAPSED:
    {
        /* rtcʱʱϢ */
        char simCardStatus[DM_NV_ITEM_SIZE] = {0};
        char cardType[DM_NV_ITEM_SIZE] = {0};
        char ceregStatus[DM_NV_ITEM_SIZE] = {0};
        sc_cfg_get(DM_SIM_CARD_STATUS, simCardStatus, sizeof(simCardStatus));
        sc_cfg_get(NV_SIMCARD_TYPE, cardType, sizeof(cardType));
        sc_cfg_get(DM_CEREG_STATUS, ceregStatus, sizeof(ceregStatus));
        log_debug("dm_queue_msg_proc: time out now, simCardStatus(%s), cardType(%s)\n", simCardStatus, cardType);
        /*
         * simCardStatus nv ֵ 1  5, עɹcarTypeգ϶û׼
         */
        if ((!strcmp(simCardStatus, "1") || !strcmp(simCardStatus, "5")) && (strcmp(cardType, "")))
        {
            dm_register_msg_send(dmControlInfo);
        }
        else if((!strcmp(ceregStatus, "1") || !strcmp(ceregStatus, "5")) && (strcmp(cardType, "")) )
        {
            dm_register_msg_send(dmControlInfo);
        }
        else
        {
            /*
             *ȴ10s, ³ԣ ֱnvϢ׼; pin 뿪߷֧
             */
            dm_timer_add(10, 0);
            log_debug("dm queue msg proc have add 10s timer now \n");
        }
        break;
    }

    case MSG_CMD_DM_CONFIG_REQ: /*CP ĹϢdmֿϢ */
    {
        dm_config_info_msg_prc(pstMsg, dmControlInfo);
        break;
    }

    case MSG_CMD_ROAM_STATUS_IND: /*Ϣ */
    {
        dm_net_global_romaing_proc(dmControlInfo);
        break;
    }

    case MSG_CMD_PDP_STATUS_IND: /*pdp·Ϣ, ƶע*/
    {
        printf("dm_queue_msg_proc: MSG_CMD_PDP_STATUS_IND \n");
        dm_net_status_change_proc(dmControlInfo);
        break;
    }

    case DM_MSG_CMD_MAIN_THREAD_EXIT:
    {
        dmControlInfo->mainThreadFlag = 0;
        log_debug("dm main thread ready to exit now \n");
        break;
    }

    default:
        log_debug("dm queue msg default proc\n");
    }

}


/**
 * ƣdm_server_init
 * DMģ̷߳ʼ
 * ˵
 *   ֵɹ 0: ʧ 
 * ˵
 */
static int dm_server_init(struct DM_CONTROl_INFO *dmControlInfo)
{
    CHECK_PARAM(dmControlInfo);
    int ret = 0;
    MSG_BUF dmMsg = {0};
    long msgSize  = 0;

    dmControlInfo->serverId = pthread_self();
    if (-1 == ((dmControlInfo->msgQueueId) = msgget(MODULE_ID_DMREG, IPC_CREAT | 0600)))
    {
        log_err("Create MSG_ID_DMREG queue faild! (*msgId)(%d)\n", dmControlInfo->msgQueueIdReg);
        return DM_ERR_CREATE_MSG_QUEUE_ERROR;
    }

    /*ʱ*/
    if (DM_ERR_SUCCESS != (ret = dm_timer_create(dmControlInfo)))
    {
        log_err("dm timer create error...\n");
        return DM_ERR_CREATE_MSG_QUEUE_ERROR;
    }

    log_debug("dm main thread start now...\n");
    dmControlInfo->mainThreadFlag = 1;
    while (dmControlInfo->mainThreadFlag)
    {
        memset((void*)(&dmMsg), 0, sizeof(MSG_BUF));
        msgSize = sizeof(MSG_BUF) - sizeof(long);
        if (0 < (ret = msgrcv(dmControlInfo->msgQueueId, &dmMsg, msgSize, 0, MSG_NOERROR)))
        {
            log_debug("dm main queue recv msg now: ret(%d), dmMsg.src_id(0x%08x),  dmMsg.usMsgCmd(0x%08x), dmMsg.usDatalen(%d), dmMsg.auDataBuffer(%s)\n",
                      ret, dmMsg.src_id, dmMsg.usMsgCmd, dmMsg.usDataLen, dmMsg.aucDataBuf);
            dm_queue_msg_proc(&dmMsg, dmControlInfo);
        }
    }

    if (msgctl(dmControlInfo->msgQueueId, IPC_RMID, NULL) < 0)
        log_err("dm remove msg now error \n");

    log_debug("dm main thread exit...\n");
    return DM_ERR_SUCCESS;
}

/*******************************************************************************
 *                             ȫֺ                                   *
 ******************************************************************************/
/**
 * ƣmain
 * DMģں
 * ˵
 *   ֵ
 * ˵
 */
int main(void)
{
    int ret = DM_ERR_SUCCESS;
    struct DM_CONTROl_INFO  dmControlInfo = {0};

    //NVʼӡ𣬲עᶯ̬ӡź
    loglevel_init();

    /* ʼDMϢ */
    ret = dm_control_info_init(&dmControlInfo);
    if (DM_ERR_SUCCESS != ret)
    {
        log_err("dm init control info error...\n");
        exit(EXIT_FAILURE);
    }

    /* ע̣߳ Ӫ̷ύע (̷ֻ߳͵Ϣ
     * עʱ̴߳*/
    if (DM_ERR_SUCCESS != (ret = dm_register_thread_init(&dmControlInfo)))
    {
        log_err("dm register thread start error...\n");
        exit(EXIT_FAILURE);
    }

    /* DṂ߳ڽոϢ(CPʱϢAPϢ) */
    if (DM_ERR_SUCCESS != (ret = dm_server_init(&dmControlInfo)))
    {
        log_err("dm server start error ...\n");
        exit(EXIT_FAILURE);
    }

    pthread_join(dmControlInfo.pid, NULL);
    return 0;
}

