/**
 * @file sc_absvr.c
 * @brief Implementation of sc_absvr.
 *
 * Copyright (C) 2023 Sanechips Technology Co., Ltd.
 * @author
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. ѡGPLv2 Licence
 *
 */


/*******************************************************************************
 *                           Include header files                              *
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <syslog.h>
#include <sys/klog.h>
#include <sys/msg.h>
#include <sys/un.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "absvr_msg.h"
#include "zxic_fota_ab_upgrade.h"
#include "softap_log.h"
#include "at_reg.h"
#include "softap_api.h"


/*******************************************************************************
 *                             Macro definitions                               *
 ******************************************************************************/
#define MSG_DATABUF_MAX_LEN (128)

#define DEBUG 1

//#define ABSVR_PRINT "SC_AB_SVR"

/*******************************************************************************
 *                             Type definitions                                *
 ******************************************************************************/
typedef struct
{
    int result; // 0: ɹ 0: ʧ
    unsigned char dataBuf[MSG_DATABUF_MAX_LEN];
} absvr_common_result_t;


typedef struct
{
    int boot_to_system;
    int reboot_flag;

    int system;
    int status;

    int status_for_nv;

    int sync_status;

    char package_path[128];
    int len;
} absvr_common_req_msg_t;


/*******************************************************************************
 *						   Local variable definitions						   *
 ******************************************************************************/


/*******************************************************************************
 *						  Global variable definitions						   *
 ******************************************************************************/


/*******************************************************************************
 * 					   Local function declarations							   *
 ******************************************************************************/
static void flush_upgrade_status(z_upgrade_status_info_t *p_status);

static int absvr_get_upgrade_type(MSG_BUF *msg);
static int absvr_verify(MSG_BUF *msg);
static int absvr_update(MSG_BUF *msg);
static int absvr_sync(MSG_BUF *msg);
static int absvr_get_upgrade_status(MSG_BUF *msg);
static int absvr_get_current_system(MSG_BUF *msg);
static int absvr_get_boot_to_system(MSG_BUF *msg);
static int absvr_set_boot_to_system(MSG_BUF *msg);
static int absvr_get_system_status(MSG_BUF *msg);
static int absvr_set_system_status(MSG_BUF *msg);
static int absvr_get_fota_status_for_nv(MSG_BUF *msg);
static int absvr_set_fota_status_for_nv(MSG_BUF *msg);
static int absvr_get_sync_status(MSG_BUF *msg);
static int absvr_set_sync_status(MSG_BUF *msg);
static int absvr_config_package_path(MSG_BUF *msg);

static int absvr_msg_proc(MSG_BUF *msg);


/*******************************************************************************
 * 					 Local function implementations 						   *
 ******************************************************************************/
static void flush_upgrade_status(z_upgrade_status_info_t *p_status)
{
    absvr_common_result_t absvr_common_result = {0};

    memcpy(absvr_common_result.dataBuf, p_status, sizeof(z_upgrade_status_info_t));

    if (0 != send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_UPDATING_RSP, sizeof(absvr_common_result),
                          (unsigned char *)(&absvr_common_result)))
    {
        slog(ABSVR_PRINT, SLOG_ERR, "Err: flush_upgrade_status send_soc_msg fail \n");
    }

    return;
}


static int absvr_get_upgrade_type(MSG_BUF *msg)
{
    absvr_common_result_t absvr_common_result = {0};
    int ret = -1;

    ret = zxic_dual_get_upgrade_type();

    absvr_common_result.result = ret;

    return send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_GET_UPGRADE_TYPE_RSP, sizeof(absvr_common_result),
                        (unsigned char *)(&absvr_common_result));
}


static int absvr_verify(MSG_BUF *msg)
{
    absvr_common_result_t absvr_common_result = {0};

    absvr_common_result.result = zxic_dual_verify();

    return send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_VRITFY_RSP, sizeof(absvr_common_result),
                        (unsigned char *)(&absvr_common_result));
}


static int absvr_update(MSG_BUF *msg)
{
    absvr_common_result_t absvr_common_result = {0};

    z_upgrade_status_info_t status = {0};
    z_upgrade_flush_status_t flush_status = {0};

    flush_status.status = &status;
    flush_status.status_cb = &flush_upgrade_status;

    absvr_common_result.result = zxic_dual_upgrade(&flush_status);

    return send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_UPDATE_RSP, sizeof(absvr_common_result),
                        (unsigned char *)(&absvr_common_result));
}


static int absvr_sync(MSG_BUF *msg)
{
    absvr_common_result_t absvr_common_result = {0};

    absvr_common_result.result = zxic_dual_sync_system();

    return send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_SYNC_RSP, sizeof(absvr_common_result),
                        (unsigned char *)(&absvr_common_result));
}


static int absvr_get_upgrade_status(MSG_BUF *msg)
{
    absvr_common_result_t absvr_common_result = {0};
    int ret = -1;

    z_upgrade_status_info_t upgrade_info = {0};

    ret = zxic_dual_get_upgrade_status(&upgrade_info);

    absvr_common_result.result = ret;
    memcpy(absvr_common_result.dataBuf, &upgrade_info, sizeof(z_upgrade_status_info_t));

    return send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_GET_UPGRADE_STATUS_RSP,
                        sizeof(absvr_common_result), (unsigned char *)(&absvr_common_result));
}


static int absvr_get_current_system(MSG_BUF *msg)
{
    absvr_common_result_t absvr_common_result = {0};
    int ret = -1;

    ret = zxic_dual_get_current_system();

    absvr_common_result.result = ret;

    return send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_GET_CURRENT_SYSTEM_RSP,
                        sizeof(absvr_common_result), (unsigned char *)(&absvr_common_result));
}


static int absvr_get_boot_to_system(MSG_BUF *msg)
{
    absvr_common_result_t absvr_common_result = {0};
    int ret = -1;

    ret = zxic_dual_get_boot_to_system();

    absvr_common_result.result = ret;

    return send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_GET_BOOT_TO_SYSTEM_RSP,
                        sizeof(absvr_common_result), (unsigned char *)(&absvr_common_result));
}


static int absvr_set_boot_to_system(MSG_BUF *msg)
{
    absvr_common_req_msg_t *common_req_msg = NULL;

    absvr_common_result_t absvr_common_result = {0};
    int ret = -1;

    common_req_msg = (absvr_common_req_msg_t *)(msg->aucDataBuf);

    ret = zxic_dual_set_boot_to_system(common_req_msg->boot_to_system, common_req_msg->reboot_flag);

    absvr_common_result.result = ret;

    return send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_SET_BOOT_TO_SYSTEM_RSP,
                        sizeof(absvr_common_result), (unsigned char *)(&absvr_common_result));
}


static int absvr_get_system_status(MSG_BUF *msg)
{
    absvr_common_result_t absvr_common_result = {0};
    int ret = -1;

    z_upgrade_system_info_t system_info = {0};

    ret = zxic_dual_get_system_status(&system_info);

    absvr_common_result.result = ret;
    memcpy(absvr_common_result.dataBuf, &system_info, sizeof(z_upgrade_system_info_t));

    return send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_GET_SYSTEM_STATUS_RSP,
                        sizeof(absvr_common_result), (unsigned char *)(&absvr_common_result));
}


static int absvr_set_system_status(MSG_BUF *msg)
{
    absvr_common_req_msg_t *common_req_msg = NULL;

    absvr_common_result_t absvr_common_result = {0};
    int ret = -1;

    common_req_msg = (absvr_common_req_msg_t *)(msg->aucDataBuf);

    ret = zxic_dual_set_system_status(common_req_msg->system, common_req_msg->status);

    absvr_common_result.result = ret;

    return send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_SET_SYSTEM_STATUS_RSP,
                        sizeof(absvr_common_result), (unsigned char *)(&absvr_common_result));
}


static int absvr_get_fota_status_for_nv(MSG_BUF *msg)
{
    absvr_common_result_t absvr_common_result = {0};
    int ret = -1;

    ret = zxic_dual_get_fota_status_for_nv();

    absvr_common_result.result = ret;

    return send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_GET_FOTA_STATUS_FOR_NV_RSP,
                        sizeof(absvr_common_result), (unsigned char *)(&absvr_common_result));
}


static int absvr_set_fota_status_for_nv(MSG_BUF *msg)
{
    absvr_common_req_msg_t *common_req_msg = NULL;

    absvr_common_result_t absvr_common_result = {0};
    int ret = -1;

    common_req_msg = (absvr_common_req_msg_t *)(msg->aucDataBuf);

    ret = zxic_dual_set_fota_status_for_nv(common_req_msg->status_for_nv);

    absvr_common_result.result = ret;

    return send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_SET_FOTA_STATUS_FOR_NV_RSP,
                        sizeof(absvr_common_result), (unsigned char *)(&absvr_common_result));
}


static int absvr_get_sync_status(MSG_BUF *msg)
{
    absvr_common_result_t absvr_common_result = {0};

    zxic_dual_get_sync_status(&(absvr_common_result.result));

    return send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_GET_SYNC_STATUS_RSP, sizeof(absvr_common_result),
                        (unsigned char *)(&absvr_common_result));
}


static int absvr_set_sync_status(MSG_BUF *msg)
{
    absvr_common_req_msg_t *common_req_msg = NULL;

    absvr_common_result_t absvr_common_result = {0};
    int ret = -1;

    common_req_msg = (absvr_common_req_msg_t *)(msg->aucDataBuf);

    ret = zxic_dual_set_sync_status(common_req_msg->sync_status);

    absvr_common_result.result = ret;

    return send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_SET_SYNC_STATUS_RSP, sizeof(absvr_common_result),
                        (unsigned char *)(&absvr_common_result));
}


static int absvr_config_package_path(MSG_BUF *msg)
{
    absvr_common_req_msg_t *common_req_msg = NULL;

    absvr_common_result_t absvr_common_result = {0};
    int ret = -1;

    common_req_msg = (absvr_common_req_msg_t *)(msg->aucDataBuf);

    ret = zxic_dual_config_package_path(common_req_msg->package_path, common_req_msg->len);

    absvr_common_result.result = ret;

    return send_soc_msg(FAR_PS, MODULE_ID_ABCLIENT, (unsigned short)ABSVR_CONFIG_PACKAGE_PATH_RSP,
                        sizeof(absvr_common_result), (unsigned char *)(&absvr_common_result));
}


static int absvr_msg_proc(MSG_BUF *msg)
{
    int ret = 0;

    if ( NULL == msg)
    {
        slog(ABSVR_PRINT, SLOG_ERR, "Err: absvr_msg_proc msg is NULL \n");
        return -1;
    }

    slog(ABSVR_PRINT, SLOG_NORMAL, "absvr_msg_proc start process msg [%hu] \n", msg->usMsgCmd);

    switch (msg->usMsgCmd)
    {
        case ABSVR_GET_UPGRADE_TYPE_REQ:
            ret = absvr_get_upgrade_type(msg);
            break;
        case ABSVR_VRITFY_REQ:
            ret = absvr_verify(msg);
            break;
        case ABSVR_UPDATE_REQ:
            ret = absvr_update(msg);
            break;
        case ABSVR_SYNC_REQ:
            ret = absvr_sync(msg);
            break;
        case ABSVR_GET_UPGRADE_STATUS_REQ:
            ret = absvr_get_upgrade_status(msg);
            break;
        case ABSVR_GET_CURRENT_SYSTEM_REQ:
            ret = absvr_get_current_system(msg);
            break;
        case ABSVR_GET_BOOT_TO_SYSTEM_REQ:
            ret = absvr_get_boot_to_system(msg);
            break;
        case ABSVR_SET_BOOT_TO_SYSTEM_REQ:
            ret = absvr_set_boot_to_system(msg);
            break;
        case ABSVR_GET_SYSTEM_STATUS_REQ:
            ret = absvr_get_system_status(msg);
            break;
        case ABSVR_SET_SYSTEM_STATUS_REQ:
            ret = absvr_set_system_status(msg);
            break;
        case ABSVR_GET_FOTA_STATUS_FOR_NV_REQ:
            ret = absvr_get_fota_status_for_nv(msg);
            break;
        case ABSVR_SET_FOTA_STATUS_FOR_NV_REQ:
            ret = absvr_set_fota_status_for_nv(msg);
            break;
        case ABSVR_GET_SYNC_STATUS_REQ:
            ret = absvr_get_sync_status(msg);
            break;
        case ABSVR_SET_SYNC_STATUS_REQ:
            ret = absvr_set_sync_status(msg);
            break;
        case ABSVR_CONFIG_PACKAGE_PATH_REQ:
            ret = absvr_config_package_path(msg);
            break;
        default:
            slog(ABSVR_PRINT, SLOG_ERR, "Err: absvr_msg_proc unknow usMsgCmd [%hu] \n", msg->usMsgCmd);
            break;
    }

    return ret;
}


/*******************************************************************************
 * 					 Global function implementations						*
 ******************************************************************************/

int main(void)
{
    int iRet = 0;

    int iMsgHandle = 0;
    MSG_BUF stMsg = {0};
    LONG msgSize =  sizeof(MSG_BUF) - sizeof(LONG);

    /*NVʼӡ𣬲עᶯ̬ӡź*/
    loglevel_init();
    slog(ABSVR_PRINT, SLOG_NORMAL, "sc_absvr begin ======= build date: %s %s \n", __DATE__, __TIME__);

    iMsgHandle = msgget(MODULE_ID_ABSVR, IPC_CREAT | 0600);
    if (-1 == iMsgHandle)
    {
        slog(ABSVR_PRINT, SLOG_ERR, "Err: can not create msg queue for sc_absvr \n");
        return -1;
    }

    for (;;)
    {
        memset(&stMsg, 0, sizeof(stMsg));
        iRet = 0;
        iRet = msgrcv(iMsgHandle, &stMsg, msgSize, 0, 0);
        if (-1 == iRet)
        {
            slog(ABSVR_PRINT, SLOG_ERR, "Err: recv msg fail errno = %d \n", errno);
            continue;
        }

        if (0 != absvr_msg_proc(&stMsg))
        {
            slog(ABSVR_PRINT, SLOG_ERR, "Err: absvr_msg_proc src_id[%d] usMsgCmd[%hu] fail \n", stMsg.src_id, stMsg.usMsgCmd);
        }
    }

    return 0;
}

