/*
 * Copyright 2008, The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <dirent.h>
#include <sys/socket.h>
#include <unistd.h>
#include <poll.h>
#include <sys/stat.h>
#include "wifi_socket.h"
#include "wpa_ctrl.h"

#include "wifi_util.h"


static const char IFNAME[]              = "IFNAME=";
#define IFNAMELEN			(sizeof(IFNAME) - 1)
static const char WPA_EVENT_IGNORE[]    = "CTRL-EVENT-IGNORE ";
#define PROPERTY_VALUE_MAX 92
pthread_mutex_t g_wpa_sock_mutex = PTHREAD_MUTEX_INITIALIZER;//multi-thread, thread1:open/close, thread2:recv

int check_alive(struct wlan_socket *skt)
{
	char  file[64]={0};
//	wf_log ("skt->pid = %d\n", skt->pid);
	if(skt->pid ==0) return 0;

	sprintf(file, "/proc/%d", skt->pid);
//	wf_log ("pidfile = %s\n", file);
	if (access(file, F_OK) != 0)
		return 0;

	return 1;
}

int wifi_connect_on_socket_path (struct wlan_socket *skt, const char *path)
{
	char supp_status[PROPERTY_VALUE_MAX] = {'\0'};
	int i = 0;
	/* Make sure hostapd is running */
	wf_log (" enter.");
	if ( !check_alive(skt)) {
		wf_log ("hostapd not running, cannot connect");
		return -1;
	}
	//wait /tmp ready
	for(i=0;i<=50;i++){
		skt->ctrl_conn = wpa_ctrl_open (path);
		if (skt->ctrl_conn == NULL) {
			if(i<50){
				wf_log ("wpa_ctrl_open ctrl_conn times= %d", i);
				usleep(100000);
				continue;
			}
			wf_log ("Unable to open connection to hostapd on \"%s\": %s \n",
			        path, strerror (errno));
			return -1;
		}
		break;
	}
	wf_log ("Success to create ctrl_conn =%p, sock=%d\n", skt->ctrl_conn, wpa_ctrl_get_fd(skt->ctrl_conn));//cov m
	
	for(i=0;i<=10;i++){
		skt->monitor_conn = wpa_ctrl_open (path);
		if (skt->monitor_conn == NULL) {
			if(i<10){
				wf_log ("wpa_ctrl_open  monitor_conn times= %d", i);
				usleep(500000);
				continue;
			}
			wf_log ("Unable to open connection to hostapd on \"%s\": %s \n",
			        path, strerror (errno));
			wpa_ctrl_close (skt->ctrl_conn);
			skt->ctrl_conn = NULL;
			return -1;
		}
		break;
	}

	wf_log ("Success to create  monitor_conn = %p, sock=%d\n", skt->monitor_conn, wpa_ctrl_get_fd(skt->monitor_conn));//cov m

	if (wpa_ctrl_attach (skt->monitor_conn) != 0) {
		wf_log ("Failed  to attach  monitor_conn= %p\n", skt->monitor_conn);//cov m

		wpa_ctrl_close (skt->monitor_conn);
		wpa_ctrl_close (skt->ctrl_conn);
		skt->ctrl_conn = skt->monitor_conn = NULL;
		return -1;
	}

	wf_log ("Success to attach  monitor_conn= %p\n", skt->monitor_conn);//cov m

	if (socketpair (AF_UNIX, SOCK_STREAM, 0, skt->exit_sockets) == -1) {
		wpa_ctrl_close (skt->monitor_conn);
		wpa_ctrl_close (skt->ctrl_conn);		
		wf_log ("Failed to socketpair, wap_ctrl_close.\n");
		skt->ctrl_conn = skt->monitor_conn = NULL;
		return -1;
	}

	return 0;
}

int wifi_connect_to_supplicant(struct wlan_socket *skt)
{
	return wifi_connect_to_hostapd(skt);
}

/* Establishes the control and monitor socket connections on the interface */
int wifi_connect_to_hostapd(struct wlan_socket *skt)
{
	static char path[PATH_MAX]={0};
	ensure_config_dir_exist(skt->sockets_dir);

	if (access (skt->sockets_dir, F_OK) == 0) {
		snprintf (path, sizeof (path), "%s/%s", skt->sockets_dir, skt->iface_name);
	}
	
	return wifi_connect_on_socket_path (skt, path);
}

int wifi_ctrl_pending(struct wpa_ctrl *ctrl, size_t time_s)
{
	struct timeval tv;
	fd_set rfds;
	int ctrlfd;
	tv.tv_sec = time_s;
	tv.tv_usec = 0;
	ctrlfd = wpa_ctrl_get_fd(ctrl);
	
	FD_ZERO(&rfds);
	FD_SET(ctrlfd, &rfds);
	select(ctrlfd + 1, &rfds, NULL, NULL, &tv);
	return FD_ISSET(ctrlfd, &rfds);
}

int wifi_restart_ctrl_socket(struct wlan_socket *skt)
{
	static char path[PATH_MAX] = {0};
	struct wpa_ctrl *tmp_ctrl_conn = NULL;
	int i = 0;
	/* Make sure hostapd is running */
	wf_log (" enter.");
	if ( !check_alive(skt)) {
		wf_log ("hostapd not running, cannot connect");
		return -1;
	}

	snprintf (path, sizeof (path), "%s/%s", skt->sockets_dir, skt->iface_name);

	for(i=0;i<=10;i++){
		tmp_ctrl_conn = wpa_ctrl_open (path);
		if (tmp_ctrl_conn == NULL) {
			if(i<10){
				wf_log ("wpa_ctrl_open ctrl_conn times= %d", i);
				usleep(500000);
				continue;
			}
			wf_log ("Unable to open connection to hostapd on \"%s\": %s \n",
			        path, strerror (errno));
			return -1;
		}
		break;
	}
	wf_log ("Success to create ctrl_conn =%p, sock=%d\n", tmp_ctrl_conn, wpa_ctrl_get_fd(tmp_ctrl_conn));//cov m

	//close fdͬ
	wpa_ctrl_close (skt->ctrl_conn);
	skt->ctrl_conn = tmp_ctrl_conn;

	return 0;
}

//wpa_supplicant߳ŶӴ,͵socketϢδʱ,
//selectʱ,װʱٴselect\recvӿ,wpa_ctrl_pending&wpa_ctrl_recv
//Returns: 0 on success, -1 on failure, -2 on timeout
int wifi_command_timeout_retry(struct wlan_socket *skt, char *reply, size_t *reply_len)
{
	if (wifi_ctrl_pending(skt->ctrl_conn, 240)) {
		if (wpa_ctrl_recv(skt->ctrl_conn, reply, reply_len) < 0) {
			wf_log("wifi_command recv fail!\n");
			return -1;
		}
		wf_log("wifi_command recv succ!\n");
		return 0;
	}
	wf_log("wifi_command pending timeout!!!!\n");
	
	if (wifi_restart_ctrl_socket(skt) == 0) {
		wf_log("wifi_restart_ctrl_socket succ!\n");
		return -1;
	}
	//station_loopպ,termination
	/* unblocks the monitor receive socket for termination */
	TEMP_FAILURE_RETRY (write (skt->exit_sockets[0], "T", 1));
	wf_log("wifi_restart_ctrl_socket failed!!!!\n");
	//assert(0);
	return -2;
}

int wifi_send_command (struct wlan_socket *skt, const char *cmd, char *reply, size_t *reply_len)
{
	int ret;
	if (skt->ctrl_conn == NULL) {
		wf_log ("Not connected to wpa_supplicant - \"%s\" command dropped.\n", cmd);
		return -1;
	}
	ret = wpa_ctrl_request (skt->ctrl_conn, cmd, strlen (cmd), reply, reply_len, NULL);
	if (ret == -2) {
		wf_log ("'%s' command timed out.\n", cmd);	
		return wifi_command_timeout_retry(skt, reply, reply_len);
	} else if (ret < 0 || strncmp (reply, "FAIL", 4) == 0) {
		return -1;
	}
	if (strncmp (cmd, "PING", 4) == 0) {
		reply[*reply_len] = '\0';
	}

	return 0;
}



int wifi_ctrl_recv (struct wlan_socket *skt, char *reply, size_t *reply_len)
{
	int res;
	int ctrlfd;
	struct pollfd rfds[2];
	wf_log ("enter ");	
	if (skt->monitor_conn == NULL) {
		wf_log ("%s: monitor_conn is NULL\n", __func__);
		return -2;
	}

	ctrlfd = wpa_ctrl_get_fd (skt->monitor_conn);
	memset (rfds, 0, 2 * sizeof (struct pollfd));
	rfds[0].fd = ctrlfd;
	rfds[0].events |= POLLIN;
	rfds[1].fd = skt->exit_sockets[1];
	rfds[1].events |= POLLIN;

	do {
		res = TEMP_FAILURE_RETRY (poll (rfds, 2, 5000));
		if (res < 0) {
			wf_log ("Error poll = %d", res);
			return res;
		} else if (res == 0) {
			/* timed out, check if supplicant is active
			 * or not ..
			 */
			if (!check_alive(skt) )
				return -2;
		}
		wf_log ("res = %d  ", res);	
	} while (res == 0);


	wf_log ("rfds changes  can recv");	
	if (rfds[0].revents & POLLIN) {
		//return wpa_ctrl_recv (skt->monitor_conn, reply, reply_len);
		int revc_res = -2;//monitor_conn is NULL
		pthread_mutex_lock(&g_wpa_sock_mutex);
		if (skt->monitor_conn != NULL) {//from AE
			//kw 3
			revc_res = wpa_ctrl_recv (skt->monitor_conn, reply, reply_len);
		}
		pthread_mutex_unlock(&g_wpa_sock_mutex);
		return revc_res;

	}

	/* it is not rfds[0], then it must be rfts[1] (i.e. the exit socket)
	 * or we timed out. In either case, this call has failed ..
	 */
	if (rfds[1].revents & POLLIN) {
		int revc_res = -2;
		//һֱᴥselect
		revc_res = recv(skt->exit_sockets[1], reply, *reply_len, 0);
		wf_log ("recv exit_sockets[1] res=%d:%s", revc_res, reply);	
	}
	return -2;
}

#if 0
static void hostapd_cli_recv_pending(struct wlan_socket *skt, char *reply, size_t *reply_len)
{
	int first = 1;
	if (skt->monitor_conn == NULL)
		return;
	while (wpa_ctrl_pending(skt->monitor_conn)) {
		char buf[256];
		size_t len = sizeof(buf) - 1;
		if (wpa_ctrl_recv(ctrl, reply, &reply_len) == 0) {
			buf[len] = '\0';
			if (action_monitor)
				hostapd_cli_action_process(buf, len);
			else {
				cli_event(buf);
				if (in_read && first)
					printf("\n");
				first = 0;
				printf("%s\n", buf);
			}
		} else {
			printf("Could not read pending message.\n");
			break;
		}
	}
}

#endif

int wifi_wait_on_socket (struct wlan_socket *skt, char *buf, size_t buflen)
{
	size_t nread = buflen - 1;
	int result;
	char *match, *match2;

	wf_log ("monitor_conn = %p\n", skt->monitor_conn);//cov m
	if (skt->monitor_conn == NULL) {
		usleep(200000);
		return snprintf (buf, buflen, "IFNAME=%s %s - connection closed",
		                 skt->iface_name, WPA_EVENT_TERMINATING);
	}

	result = wifi_ctrl_recv (skt, buf, &nread);

	/* Terminate reception on exit socket */
	if (result == -2) {
		return snprintf (buf, buflen, "IFNAME=%s %s - connection closed",
		                 skt->iface_name, WPA_EVENT_TERMINATING);
	}

	if (result < 0) {
		wf_log ("wifi_ctrl_recv failed: %s\n", strerror (errno));
		return snprintf (buf, buflen, "IFNAME=%s %s - recv error",
		                 skt->iface_name, WPA_EVENT_TERMINATING);
	}
	buf[nread] = '\0';
	/* Check for EOF on the socket */
	if (result == 0 && nread == 0) {
		/* Fabricate an event to pass up */
		wf_log ("Received EOF on supplicant socket\n");
		return snprintf (buf, buflen, "IFNAME=%s %s - signal 0 received",
		                 skt->iface_name, WPA_EVENT_TERMINATING);
	}

	/*
	 * Events strings are in the format
	 *
	 *     IFNAME=iface <N>CTRL-EVENT-XXX
	 *        or
	 *     <N>CTRL-EVENT-XXX
	 *
	 * where N is the message level in numerical form (0=VERBOSE, 1=DEBUG,
	 * etc.) and XXX is the event name. The level information is not useful
	 * to us, so strip it off.
	 */

	if (strncmp (buf, IFNAME, IFNAMELEN) == 0) {
		match = strchr (buf, ' ');
		if (match != NULL) {
			if (match[1] == '<') {
				match2 = strchr (match + 2, '>');
				if (match2 != NULL) {
					nread -= (match2 - match);
					memmove (match + 1, match2 + 1, nread - (match - buf) + 1);
				}
			}
		} else {
			return snprintf (buf, buflen, "%s", WPA_EVENT_IGNORE);
		}
	} else if (buf[0] == '<') {
		match = strchr (buf, '>');
		if (match != NULL) {
			nread -= (match + 1 - buf);
			memmove (buf, match + 1, nread + 1);
			wf_log ("supplicant generated event without interface - %s\n", buf);
		}
	} else {
		/* let the event go as is! */
		wf_log ("supplicant generated event without interface and without message level - %s\n", buf);
	}

	return nread;
}

int wifi_wait_for_event (struct wlan_socket *skt, char *buf, size_t buflen)
{
	return wifi_wait_on_socket (skt, buf, buflen);
}

void wifi_close_sockets(struct wlan_socket *skt)
{
	if (skt->ctrl_conn != NULL) {
		wpa_ctrl_close (skt->ctrl_conn);
		skt->ctrl_conn = NULL;
	}

	if (skt->monitor_conn != NULL) {
		pthread_mutex_lock(&g_wpa_sock_mutex);
		wpa_ctrl_close (skt->monitor_conn);		
		skt->monitor_conn = NULL;
		pthread_mutex_unlock(&g_wpa_sock_mutex);
		wf_log ("%s: monitor_conn is NULL\n", __func__);
	}

	if (skt->exit_sockets[0] >= 0) {
		close (skt->exit_sockets[0]);
		skt->exit_sockets[0] = -1;
	}

	if (skt->exit_sockets[1] >= 0) {
		close (skt->exit_sockets[1]);
		skt->exit_sockets[1] = -1;
	}
}



int wifi_command (struct wlan_socket *skt, const char *command, char *reply, size_t *reply_len)
{
	return wifi_send_command (skt, command, reply, reply_len);
}


static int doCommand (struct wlan_socket *skt, const char *command, char* reply, size_t reply_len)
{
	if (command == NULL) {
		return 0;
	}

	--reply_len; // Ensure we have room to add NUL termination.
	if (wifi_command (skt, command, reply, &reply_len) != 0) {
		return 0;
	}

	// Strip off trailing newline.
	if (reply_len > 0 && reply[reply_len - 1] == '\n') {
		reply[reply_len - 1] = '\0';
	} else {
		reply[reply_len] = '\0';
	}
	return 1;
}
#define REPLY_BUF_SIZE 4096 // wpa_supplicant's maximum size.

char reply[REPLY_BUF_SIZE];
// Send a command to the supplicant, and return the reply as a String.
char *docmd (struct wlan_socket *skt, const char *command)
{
	wf_log ("cmd=[%s]", command);
	memset(reply, 0, REPLY_BUF_SIZE); //klocwork
	if (!doCommand (skt, command, reply, sizeof(reply) - 1)) {
		wf_log ("fail,reply=[NULL]");
		return NULL;
	}

	wf_log ("reply=[%s]", reply);

	return reply;
}
