/**
 * @file hal_i2s.c
 * @brief Implementation of Sanechips i2s  Function
 *
 * Copyright (C) 2017 Sanechips Technology Co., Ltd.
 * @author Author: Xinqiang Xu <xu.xinqiang@sanechips.com.cn>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

/*******************************************************************************
 *                           Include header files                              *
 ******************************************************************************/
#ifdef _OS_OSE
#include "outfmt.h"
#include "ose.h"
#include "cpu.h"
#include "dda_pci.h"
#include "dda_utils.h"
#include "dda_err.h"
#include "dda_drm.h"
#include "dda_strings.h"
#include "device.sig"
#include "inputdev.sig"
#include "ramlog.h"
#include "ose_syms.h"
#endif

#include "drvs_general.h"
#include "drvs_i2s.h"
#include "drvs_pow.h"
//#include "dma_cfg.h"
//#include "drvs_dma.h"
#include "ThreadPriority.h"
#include "drvs_chip_cfg.h"
#include "drvs_sys.h"
#include "drvs_voiceprocess.h"
#include <linux/init.h>
#include <linux/platform_device.h>
//#include <cyg/hal/tos_cache.h>
/*
#include <linux/gpio.h>
#include <mach/gpio.h>
#include <mach/clk.h>
#include <mach/gpio_def.h>
#include <mach/gpio_cfg.h>
#include <mach/gpio-names.h>
#include <mach/board.h>
#include <mach/clk.h>
*/
#include <linux/slab.h>
#include <linux/dma-mapping.h>
#include <linux/module.h>
#include <linux/dmaengine.h>
#include <mach/board.h>
#include <mach/dma.h>
#include <mach/iomap.h>
#include <asm/io.h>

/*******************************************************************************
 *                             Macro definitions                               *
  *************************************************************************/
#define I2S_DRIVER_NAME             "i2s"
#define I2S_DRIVER_VERSION          1
#define I2S_VERSION     				0x0102	/*I2S Version*/

#define MALLOC_I2S_DMA_AREA

#define TOP_I2S0_PIN_CONNECT

#define ARM_I2S_LOOP_CFG	*((volatile UINT32 *)(SOC_SYS_REG_BASE+0x60))

/* process control register */
#define I2S_TX_OPEN     				(0x1<<0)	/*TX enable*/
#define I2S_RX_OPEN     				(0x1<<1)	/*RX enable*/
#define I2S_EN							(0x1<<2)

/* Timing control register */
#define I2S_MASTER_EN                   (0x1<<0)
#define I2S_TEST_MODE                  	(0x1<<1)		/*test Mode*/
#define I2S_CLK_EDGE_SEL                (0x1<<2)		/*I2S edge select*/
#define I2S_TIMING_SEL                  (0x1<<3)		/*PCM Mode or I2S Mode*/
#define I2S_LONG_FSYNC                	(0x1<<4)  /*1 long, 0 short*/
#define I2S_EXTRA_CYCLE                	(0x1<<5)  /*1 have extra cycle, 0 no extra*/
#define I2S_ALIGN_MODE(x)				(((x)&0x3)<<6) /*b00 std, b01 msb,b10 lsb,b11 normal*/

#define I2S_CHN_NUM(x)					(((x)&0x7)<<8)  /*h0~3,h7 totally 1~8 chn per frame*/
#define I2S_LANE_NUM(x)                	(((x)&0x3)<<11)  /*h0~2,h3  1~4 lane used to transfer data*/

#define I2S_TS_CFG(x)                	(((x)&0x7)<<13)  /* i2s timing or tdm cfg*/
#define I2S_TS_WIDTH(x)                	(((x)&0x1f)<<16)  /* h0~5 illeagal,hf~1f 16~32 per time slot*/
#define I2S_DATA_SIZE(x)                	(((x)&0x1f)<<21)  /* h0~5 illeagal,hf~1f data size 16~32*/
#define I2S_FMT_ERR						(0x1<<31)  /* 1 cfg illeagal,0 cfg correct*/

/* fifo control register */
#define I2S_TX_FIFO_RST                 (0x1<<0)
#define I2S_RX_FIFO_RST                 (0x1<<1)
#define I2S_TX_DMA_EN                   (0x1<<4)
#define I2S_RX_DMA_EN0                  (0x1<<5)
#define I2S_RX_DMA_EN1                  (0x1<<6)
#define I2S_TFTH(x)                     (((x)&0x1f)<<8)	/* interrupt generate setting */
#define I2S_RFTH(x)                     (((x)&0x1f)<<16)/* interrupt generate setting */

/* fifo status register */
#define I2S_TX_BEYOND_THRES				(0x1<<0)
#define I2S_RX_BEYOND_THRES				(0x1<<1)

#define I2S_TX_FIFO_EMPTY				(0x1<<4)
#define I2S_RX_FIFO_FULL				(0x1<<5)

#define  I2S_TXF_HW_FLAG				(0x1<<8)
#define I2S_TX_FIFO_CNTR(x)				(((x)&(0x3f<<9))>>9)
#define I2S_RXF_HW_FLAG					(0x1<<16)
#define I2S_RX_FIFO_CNTR(x)				(((x)&(0x3f<<17))>>17)

/* clock div register */
#define I2S_DIV(x)                      (((x)&0x7f)<<0)

/* interrupt status */
#define I2S_TX_TRIG_INT                 (0x1<<0)
#define I2S_RX_TRIG_INT                 (0x1<<1)
#define I2S_TX_UNDER_INT                (0x1<<2)	/* tx error status */
#define I2S_RX_OVER_INT                 (0x1<<3)	/* rx error status */

/* interrupt enable register */
#define I2S_TX_TRIG_INT_EN              (0x1<<0)
#define I2S_RX_TRIG_INT_EN              (0x1<<1)
#define I2S_TX_UNDER_INT_EN             (0x1<<2)
#define I2S_RX_OVER_INT_EN              (0x1<<3)

/* frame counter */
#define I2S_FRAME_CNTR(x)				((x)&0x3fffffff)
#define I2S_FRAME_CNTR_EN				(0x1<<30)
#define I2S_FRAME_CNTR_RST				(0x1<<31)

/* default rx/tx trigger threadhold */
#define I2S_RXFL_VAL                    (0x10)
#define I2S_TXFL_VAL                    (0x10)

#define I2S_ENABLE_DELAY_TIME			(10)      /* us */

#define I2S_OUT_BUF_NUM					3
#define I2S_UPLINK_SEM_NUM          	3
#define I2S_DOWNLINK_SEM_NUM        	0
#define I2S_MAX_RX_BUF_SIZE         	2048
#define I2S_MAX_TX_BUF_SIZE         	8192
#define I2S_MAX_AUDIO_PLAY_MEM_SIZE		8192
#define I2S_MAX_AUDIO_RECORD_MEM_SIZE	2048
#define I2S_MAX_VOICE_GSM_MEM_SIZE		320
#define I2S_MAX_VOICE_LTE_MEM_SIZE		640
#define RAM_ADDR_FOR_I2S1				DMA_RAM_FOR_I2S_ADDR_BASE
#define RAM_ADDR_FOR_I2S2				(DMA_RAM_FOR_I2S_ADDR_BASE + 30720)

#define I2S_SEM_WAIT_TIME				2500

#define I2S_QUEUE_SIZE					(I2S_OUT_BUF_NUM + 1)
#define I2S_WAIT_TIME					100
#define I2S_INTCTRL_TEST_MODE    		TRUE

//#define I2S0_CRM_CLKSEL     			(ZX_LSP_CRPM_BASE+0x14)
//#define I2S1_CRM_CLKSEL     			(ZX_LSP_CRPM_BASE+0x1C)

#define I2S0_CRM_CLKSEL		(LSP_CRM_REG_BASE+0x14)
#define I2S1_CRM_CLKSEL		(LSP_CRM_REG_BASE+0x1C)
#if defined _USE_AUDIO_ON_7520V2
#define I2S0_CLKDIV       *((volatile UINT32 *)(LSP_CRM_REG_BASE+0x18))
#define I2S1_CLKDIV       *((volatile UINT32 *)(LSP_CRM_REG_BASE+0x20))
#elif defined _USE_AUDIO_ON_7520V3
#define I2S0_CLKDIV_1       (LSP_CRM_REG_BASE+0x58)
#define I2S0_CLKDIV_2       (LSP_CRM_REG_BASE+0x5C)

#define I2S1_CLKDIV_1       (LSP_CRM_REG_BASE+0x60)
#define I2S1_CLKDIV_2       (LSP_CRM_REG_BASE+0x64)
#endif

/* lsp i2s0 work clk div */
#define	 I2S0_FRA_DIV      0x00000008
#define	 I2S0_FRA_BASE     0x00000808
#define	 I2S0_INTEGER_NUM  0x00001008
#define  I2S0_CLK_SEL_DIV  0x00001801
#define	 I2S0_PWRDN        0x00001901
#define	 I2S0_CLK_DIV      0x00001c04

/* lsp i2s1 work clk div */
#define	 I2S1_FRA_DIV      0x10000008
#define	 I2S1_FRA_BASE     0x10000808
#define	 I2S1_INTEGER_NUM  0x10001008
#define  I2S1_CLK_SEL_DIV  0x10001801
#define	 I2S1_PWRDN        0x10001901
#define	 I2S1_CLK_DIV      0x10001c04

/*******************************************************************************
 *                             Type definitions                                *
 ******************************************************************************/
typedef enum {
    BUFFER_NULL,
    BUFFER_WRITE,
    BUFFER_FULL,
    BUFFER_READ,

    MAX_BUFFER_STATUS
} T_I2s_BufferStatus;

typedef enum {
    CLK_8000,
    CLK_11025,
    CLK_12000,
    CLK_16000,
    CLK_22050,
    CLK_24000,
    CLK_32000,
    CLK_44100,
    CLK_48000,
    CLK_64000,
    CLK_96000,

    MAX_CLK
} T_I2s_ClkConst;

typedef enum {
    TS_16CYCLE,
    TS_17CYCLE,
    TS_18CYCLE,
    TS_19CYCLE,
    TS_20CYCLE,
    TS_21CYCLE,
    TS_22CYCLE,
    TS_23CYCLE,
    TS_24CYCLE,
    TS_25CYCLE,
    TS_26CYCLE,
    TS_27CYCLE,
    TS_28CYCLE,
    TS_29CYCLE,
    TS_30CYCLE,
    TS_31CYCLE,
    TS_32CYCLE,

    MAX_TS_WIDTH
} T_I2s_tTsWidthConst;

typedef struct {
	volatile UINT32  i2sVersion;
	volatile UINT32  processCtrl;
	volatile UINT32  timingCtrl;
	volatile UINT32  fifoCtrl;
	volatile UINT32  fifoStatus;
	volatile UINT32  intEn;
	volatile UINT32  intStatus;
	volatile UINT32  data;
	volatile UINT32  frameCntr;
	volatile UINT32  data2;

} T_I2s_Reg;


/**
 * @brief describe the i2s device info
 * @param regs		register pointer
 * @param intline	interrupt line
 * @param intprio 	interrupt priority
 * @param vector	Interrupt vector
 * @param eoiFunc	interrupt handler
 * @param eoiArg 	interrupt handler arguments
 * @param pid		the id of the process
 */
typedef struct {
	T_I2s_Reg          *regs;
	UINT32             intline;
	UINT32             intprio;
	OSVECTOR           vector;
#ifdef _OS_OSE
	DdaEndOfInt        *eoiFunc;
	DdaFuncHandle      eoiArg;
#endif
	PROCESS            pid;
} T_I2s_Device;

typedef struct {
	T_I2s_Device                *ptDev;
	T_ZDrvI2S_Cfg               tCfg;
	T_I2s_Status                i2sStatus;
	UINT16                      uiDmaRxChan;
	UINT16                      uiDmaTxChan;
//	zDrvDma_CallbackFunc        fDmaCbRx;
//	zDrvDma_CallbackFunc        fDmaCbTx;
	SINT32						DmaRxID;
	SINT32						DmaTxID;
	T_ZDrvI2s_TransMode         i2sTransMode;
	ZOSS_MUTEX_ID				rxMutex;
	ZOSS_MUTEX_ID				txMutex;
} T_I2s_Instance;

typedef struct {
	UINT8 *buf;
	T_I2s_BufferStatus i2sBufStatus;
} T_I2s_BufInfo;

typedef SINT32(*T_I2s_TxFunc)(T_ZDrvI2S_Id tId, const UINT16 *pBuf, UINT32 uiLen);

typedef struct _T_I2s_QueueNode {
	UINT16 *pBuf;
	UINT32 len;
} T_I2s_QueueNode;

typedef  struct _T_I2s_Queue {
	UINT8 readIndex;
	UINT8 writeIndex;
	T_I2s_QueueNode *data;
} T_I2s_Queue;

/**
 * @brief the data stream of transing
 * @param txBufferArray		tx buf array
 * @param txQueue			size of one buf,use in tx
 * @param txLen 			size of one buf,use in tx
 * @param txSem				for the buf management
 * @param Transing			use to check if datas is transing
 * @param pauseFlag 		pause flag
 * @param txTransmit		for the buf management
 * @param channel			audio channel
 * @param p_cb				audio playback callback function
 */
typedef struct {
	T_I2s_BufInfo         *txBufferArray;
	T_I2s_Queue           txQueue;
	UINT32                txLen;
	ZOSS_SEMAPHORE_ID     txSem;
	BOOL                  Transing;
	BOOL                  pauseFlag;
	T_I2s_TxFunc          txTransmit;
	T_ZDrvAudio_Channel   channel;
	T_ZDrvI2S_AUD_PLAY_CB p_cb;
	UINT8                 txRdIdx;
	UINT8                 txWrIdx;
} T_I2s_TxStream;


/**
 * @brief the data stream of receiving
 * @param txBufferArray		rx buf array
 * @param currentRxBuf		current rx buf
 * @param rxQueue			queue for rx
 * @param rxLen 			size of one buf,use in rx
 * @param rxIdx				idx of next rx buf
 * @param rxSem				for the buf management
 * @param rxTransmit 		trans function
 * @param channel			audio channel
 * @param p_cb				audio playback callback function
 *
 */
typedef struct {
	T_I2s_BufInfo         *rxBufferArray;
	T_I2s_BufInfo         *currentRxBuf;
	T_I2s_Queue           rxQueue;
	UINT32                rxLen;
	UINT8                 rxRdIdx;
	UINT8                 rxWrIdx;
	ZOSS_SEMAPHORE_ID     rxSem;
	T_I2s_TxFunc          rxTransmit;
	T_ZDrvAudio_Channel   channel;
	T_ZDrvI2S_AUD_PLAY_CB p_cb;
} T_I2s_RxStream;

#if defined _USE_AUDIO_ON_7520V2
typedef struct {
	UINT32   INTEGERNUM;
	UINT32   FRACADDNUM;
	UINT32   FRACBASE;
	UINT32   CLKSELDIV;
	UINT32   PWRDN;
	T_I2s_RefClkForI2s   refclk;
	UINT32   clkRatio;
}
T_I2s_ClkDiv;
#elif defined _USE_AUDIO_ON_7520V3
typedef struct {
	UINT32   FRACDIV1;
	UINT32   FRACDIV2;
	T_I2s_ClkConst i2sClockSet;
	T_I2s_tTsWidthConst i2sTsWidthSet;
	T_I2s_RefClkForI2s   refclk;
} T_I2s_ClkDiv;
#endif

/*******************************************************************************
 *                        Local function declarations                          *
 ******************************************************************************/
static SINT32 i2s_GetPcmDiv(T_ZDrvI2S_Cfg *ptCfg, T_I2s_ClkDiv *pDiv);
static SINT32 i2s_GetI2sDiv(T_ZDrvI2S_Cfg *ptCfg, T_I2s_ClkDiv *pDiv);
SINT32 i2s_SetPdiv(T_ZDrvI2S_Id tId, T_ZDrvI2S_Cfg *ptCfg);

static SINT32 i2s_QueueInit(T_I2s_Queue *i2sQueue);
static SINT32 i2s_QueuePut(T_I2s_Queue *i2sQueue, UINT16 *pBuf, UINT32 uiLen);
static SINT32 i2s_QueueGet(T_I2s_Queue *i2sQueue, T_I2s_QueueNode **queueNode);
static SINT32 i2s_QueueDestory(T_I2s_Queue *i2sQueue);

static SINT32 i2s_SetupTxBuf(T_ZDrvI2S_Id tId);
static SINT32 i2s_SetupRxBuf(T_ZDrvI2S_Id tId);
static SINT32 i2s_Reset(T_ZDrvI2S_Id tId);

static VOID i2s_IntIsr(T_ZDrvI2S_Id tId);
static VOID i2s_IntTxWriteFifo(T_ZDrvI2S_Id tId, UINT32 chanlable);
static VOID i2s_IntRxWriteBuf(T_ZDrvI2S_Id tId);
static VOID i2s_IntTxChangeBuf(T_ZDrvI2S_Id tId);
static VOID i2s_IntRxChangeBuf(T_ZDrvI2S_Id tId);

static SINT32 i2s_TxCheckState(T_ZDrvI2S_Id tId, T_ZDrvI2S_Cfg *ptCfg);
static SINT32 i2s_RxCheckState(T_ZDrvI2S_Id tId, T_ZDrvI2S_Cfg *ptCfg);

static SINT32 i2s_TxComInit(T_ZDrvI2S_Id tId, T_ZDrvI2s_Params *params);
static SINT32 i2s_RxComInit(T_ZDrvI2S_Id tId, T_ZDrvI2s_Params *params);

static VOID isr_i2s1(VOID);
static VOID isr_i2s2(VOID);

static SINT32 i2s_IntRxInit(T_ZDrvI2S_Id tId);
static SINT32 i2s_IntTxInit(T_ZDrvI2S_Id tId);
static SINT32 i2s_DmaRxInit(T_ZDrvI2S_Id tId);
static SINT32 i2s_DmaTxInit(T_ZDrvI2S_Id tId);
static SINT32 i2s_DmaTx(T_ZDrvI2S_Id tId, const UINT16 *pBuf, UINT32 uiLen);
static SINT32	i2s_DmaRx(T_ZDrvI2S_Id tId, T_I2s_Instance *ptInstance, UINT8 idx);

static SINT32 i2s_CleanTxResource(T_ZDrvI2S_Id tId);
static SINT32 i2s_CleanRxResource(T_ZDrvI2S_Id tId);

static SINT32 i2s_IntTx(T_ZDrvI2S_Id tId, const UINT16 *pBuf, UINT32 uiLen);

static VOID i2s_InitInt(VOID);
static SINT32 i2s_SetClk(T_ZDrvI2S_Id tId, T_I2s_ClkDiv *pDiv);
static SINT32 i2s_CloseClk(T_ZDrvI2S_Id i2sId);
static SINT32 i2s_SetClkDiv(UINT32 divFunc, UINT32 div);

#ifdef _USE_PSM
extern SINT32 zDrvPow_SetDevActive(T_ZDrvPow_IdleFlag devId);
extern SINT32 zDrvPow_SetDevIdle(T_ZDrvPow_IdleFlag devId);
#endif

SINT32 i2s_GetI2sTimingCfg(T_ZDrvI2S_TimingType tmType, T_ZDrvI2S_TimingCfg *pTmCfg);
SINT32 i2s_SetTiming(T_I2s_Reg *ptReg, T_ZDrvI2S_Cfg *ptCfg);
static VOID i2s_ClkInit(VOID);
static VOID i2s_ClkDeinit(T_ZDrvI2S_Id tId);

extern VOID Vp_ChangBufLen(VOID);
#ifdef _USE_VP_I2S_SYNC
extern VOID Vp_I2S1ReadChangeBufLen(VOID);
extern VOID Vp_I2S1WriteChangeBufLen(VOID);
extern VOID Vp_I2S2ReadChangeBufLen(VOID);
extern VOID Vp_I2S2WriteChangeBufLen(VOID);
#endif
SINT32 i2s_rxsem_count_Init(T_ZDrvI2S_Id tId);
SINT32 i2s_txsem_count_Init(T_ZDrvI2S_Id tId);
void bsp_udelay(U32 us);//xiu
/*******************************************************************************
 *                         Local variable definitions                          *
 ******************************************************************************/

#ifdef MALLOC_I2S_DMA_AREA
typedef struct {
//    UINT32   txtotalbuffsize;
//    UINT32   rxtotalbuffsize;
//	UINT32   txsinglebuffsize;
//    UINT32   rxsinglebuffsize;
//	dma_addr_t dma_tx_phyaddr[I2S_OUT_BUF_NUM];
//	void* dma_tx_viraddr;
//	dma_addr_t dma_rx_phyaddr[I2S_OUT_BUF_NUM];
//	void* dma_rx_viraddr;
//	T_ZDrvDma_ChannelDef rxDmaChanDef;
//	T_ZDrvDma_ChannelDef txDmaChanDef;
	UINT32   totalbuffsize;
	UINT32   singlebuffsize;
	dma_addr_t dma_phyaddr;
	void* dma_viraddr;
//	T_ZDrvDma_ChannelDef dmaChanDef;
	struct dma_chan *ch;
	int channel;
}
T_I2s_DmaInfo;

static T_I2s_DmaInfo i2sDmaState[MAX_I2S_ID][2] = {0};

#endif

static unsigned long long i2s_device_dma_mask = DMA_BIT_MASK(32);

static struct platform_device i2s_device = {
	.name = "v3-i2s",
	.id	= -1,
	.dev = {
		.dma_mask		= &i2s_device_dma_mask,
		.coherent_dma_mask	= DMA_BIT_MASK(32),
		.platform_data = NULL,
	},
};


static T_I2s_Instance s_i2sInstance[MAX_I2S_ID]  = {
	0,
};
static T_I2s_TxStream s_i2sTxStream[MAX_I2S_ID] = {
	NULL,
};
static T_I2s_RxStream s_i2sRxStream[MAX_I2S_ID] = {
	NULL,
};
static const UINT16 *s_txBuf[MAX_I2S_ID] = {
	NULL,
};
static UINT16 *s_rxBuf[MAX_I2S_ID] = {
	NULL,
};
static UINT32 s_txLen[MAX_I2S_ID] = {
	0
};
static UINT32 s_rxLen[MAX_I2S_ID] = {
	0
};
static UINT8 *s_txBufArray[MAX_I2S_ID][I2S_OUT_BUF_NUM] = {
	NULL,
};
static UINT8 *s_rxBufArray[MAX_I2S_ID][I2S_OUT_BUF_NUM] = {
	NULL,
};
#if defined _USE_AUDIO_ON_7520V2
static T_I2s_ClkDiv gPcm1SlotClkDiv[REF_CLK_MAX][MAX_CLK][MAX_PCM_MODE] = {
	{
		{{25, 199, 255, 0, 0, 0, 3}, {23, 203, 255, 1, 0, 0, 3}},
		{{18, 216, 255, 0, 0, 0, 3}, {17, 174, 255, 0, 0, 0, 3}},
		{{16, 218, 255, 1, 0, 0, 3}, {15, 220, 255, 1, 0, 0, 3}},
		{{12, 100, 255, 1, 0, 0, 3}, {11, 229, 255, 1, 0, 0, 3}},
		{{9, 108, 255, 0, 0, 0, 3}, {8, 87, 255, 1, 0, 0, 3}},
		{{8, 236, 255, 0, 0, 0, 3}, {7, 238, 255, 1, 0, 0, 3}},
		{{6, 177, 255, 0, 0, 0, 3}, {5, 242, 255, 1, 0, 0, 3}},
		{{4, 54, 255, 1, 0, 0, 3}, {4, 171, 255, 0, 0, 0, 3}},
		{{4, 118, 255, 0, 0, 0, 3}, {3, 246, 255, 1, 0, 0, 3}},
		{{3, 89, 255, 0, 0, 0, 3}, {2, 248, 255, 1, 0, 0, 3}},
		{{2, 59, 255, 0, 0, 0, 3}, {1, 251, 255, 1, 0, 0, 3}}
	}
};

static T_I2s_ClkDiv gPcm2SlotClkDiv[REF_CLK_MAX][MAX_CLK][MAX_PCM_MODE] = {
	{
		{{12, 100, 255, 1, 0, 0, 3}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}}
	}
};

static T_I2s_ClkDiv gPcm4SlotClkDiv[REF_CLK_MAX][MAX_CLK][MAX_PCM_MODE] = {
	{
		{{6, 177, 255, 0, 0, 0, 3}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}}
	}
};

static T_I2s_ClkDiv gPcm8SlotClkDiv[REF_CLK_MAX][MAX_CLK][MAX_PCM_MODE] = {
	{
		{{3, 89, 255, 0, 0, 0, 3}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}}
	}
};

static T_I2s_ClkDiv gPcm16SlotClkDiv[REF_CLK_MAX][MAX_CLK][MAX_PCM_MODE] = {
	{
		{{3, 89, 255, 0, 0, 0, 3}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}},
		{{0, 0, 0, 0, 0, 0, 0}, {0, 0, 0, 0, 0, 0, 0}}
	}
};

static T_I2s_ClkDiv gI2sClkDiv[REF_CLK_MAX][MAX_CLK][MAX_TS_WIDTH] = {
	{
		{{25, 199, 255, 0, 0, 0, 1}, {23, 203, 255, 1, 0, 0, 1}, {22, 35, 255, 1, 0, 0, 1}, {21, 195, 255, 0, 0, 0, 1}, {20, 159, 255, 0, 0, 0, 1}, {19, 176, 255, 0, 0, 0, 1}, {18, 238, 255, 0, 0, 0, 1}, {17, 83, 255, 1, 0, 0, 1}, {16, 218, 255, 1, 0, 0, 1}, {16, 128, 255, 0, 0, 0, 1}, {15, 64, 255, 1, 0, 0, 1}, {15, 24, 255, 0, 0, 0, 1}, {14, 5, 255, 1, 0, 0, 1}, {14, 4, 255, 0, 0, 0, 1}, {13, 21, 255, 1, 0, 0, 1}, {13, 53, 255, 0, 0, 0, 1}, {12, 100, 255, 1, 0, 0, 1}},
		{{18, 216, 255, 0, 0, 0, 1}, {17, 174, 255, 0, 0, 0, 1}, {16, 192, 255, 0, 0, 0, 1}, {15, 8, 255, 1, 0, 0, 1}, {14, 122, 255, 1, 0, 0, 1}, {14, 19, 255, 0, 0, 0, 1}, {13, 204, 255, 0, 0, 0, 1}, {12, 162, 255, 1, 0, 0, 1}, {12, 144, 255, 0, 0, 0, 1}, {11, 149, 255, 1, 0, 0, 1}, {11, 172, 255, 0, 0, 0, 1}, {10, 213, 255, 1, 0, 0, 1}, {10, 14, 255, 1, 0, 0, 1}, {10, 84, 255, 0, 0, 0, 1}, {9, 166, 255, 1, 0, 0, 1}, {9, 5, 255, 1, 0, 0, 1}, {9, 108, 255, 0, 0, 0, 1}},
		{{16, 218, 255, 1, 0, 0, 1}, {15, 220, 255, 1, 0, 0, 1}, {15, 24, 255, 0, 0, 0, 1}, {14, 130, 255, 0, 0, 0, 1}, {13, 21, 255, 1, 0, 0, 1}, {12, 202, 255, 1, 0, 0, 1}, {12, 158, 255, 0, 0, 0, 1}, {11, 140, 255, 1, 0, 0, 1}, {11, 145, 255, 0, 0, 0, 1}, {10, 170, 255, 1, 0, 0, 1}, {10, 213, 255, 0, 0, 0, 1}, {10, 16, 255, 0, 0, 0, 1}, {9, 88, 255, 1, 0, 0, 1}, {9, 173, 255, 0, 0, 0, 1}, {9, 14, 255, 0, 0, 0, 1}, {8, 121, 255, 1, 0, 0, 1}, {8, 236, 255, 0, 0, 0, 1}},
		{{12, 100, 255, 1, 0, 0, 1}, {11, 229, 255, 1, 0, 0, 1}, {11, 145, 255, 0, 0, 0, 1}, {10, 97, 255, 1, 0, 0, 1}, {10, 80, 255, 0, 0, 0, 1}, {9, 88, 255, 1, 0, 0, 1}, {9, 119, 255, 0, 0, 0, 1}, {8, 169, 255, 1, 0, 0, 1}, {8, 236, 255, 0, 0, 0, 1}, {8, 64, 255, 0, 0, 0, 1}, {7, 159, 255, 1, 0, 0, 1}, {7, 12, 255, 1, 0, 0, 1}, {7, 130, 255, 0, 0, 0, 1}, {7, 2, 255, 0, 0, 0, 1}, {6, 138, 255, 1, 0, 0, 1}, {6, 27, 255, 1, 0, 0, 1}, {6, 177, 255, 0, 0, 0, 1}},
		{{9, 108, 255, 0, 0, 0, 1}, {8, 87, 255, 1, 0, 0, 1}, {8, 96, 255, 0, 0, 0, 1}, {7, 131, 255, 1, 0, 0, 1}, {7, 189, 255, 0, 0, 0, 1}, {7, 10, 255, 0, 0, 0, 1}, {6, 102, 255, 1, 0, 0, 1}, {6, 208, 255, 0, 0, 0, 1}, {6, 72, 255, 0, 0, 0, 1}, {5, 202, 255, 1, 0, 0, 1}, {5, 86, 255, 1, 0, 0, 1}, {5, 234, 255, 0, 0, 0, 1}, {5, 135, 255, 0, 0, 0, 1}, {5, 42, 255, 0, 0, 0, 1}, {4, 211, 255, 1, 0, 0, 1}, {4, 130, 255, 1, 0, 0, 1}, {4, 54, 255, 1, 0, 0, 1}},
		{{8, 236, 255, 0, 0, 0, 1}, {7, 238, 255, 1, 0, 0, 1}, {7, 12, 255, 1, 0, 0, 1}, {7, 65, 255, 0, 0, 0, 1}, {6, 138, 255, 1, 0, 0, 1}, {6, 229, 255, 0, 0, 0, 1}, {6, 79, 255, 0, 0, 0, 1}, {5, 198, 255, 1, 0, 0, 1}, {5, 73, 255, 1, 0, 0, 1}, {5, 213, 255, 0, 0, 0, 1}, {5, 106, 255, 0, 0, 0, 1}, {5, 8, 255, 0, 0, 0, 1}, {4, 172, 255, 1, 0, 0, 1}, {4, 86, 255, 1, 0, 0, 1}, {4, 7, 255, 1, 0, 0, 1}, {4, 188, 255, 0, 0, 0, 1}, {4, 118, 255, 0, 0, 0, 1}},
		{{6, 177, 255, 0, 0, 0, 1}, {5, 242, 255, 1, 0, 0, 1}, {5, 73, 255, 1, 0, 0, 1}, {5, 176, 255, 0, 0, 0, 1}, {5, 40, 255, 0, 0, 0, 1}, {4, 172, 255, 1, 0, 0, 1}, {4, 59, 255, 1, 0, 0, 1}, {4, 212, 255, 0, 0, 0, 1}, {4, 118, 255, 0, 0, 0, 1}, {4, 32, 255, 0, 0, 0, 1}, {3, 207, 255, 1, 0, 0, 1}, {3, 133, 255, 1, 0, 0, 1}, {3, 65, 255, 1, 0, 0, 1}, {3, 1, 255, 1, 0, 0, 1}, {3, 197, 255, 0, 0, 0, 1}, {3, 141, 255, 0, 0, 0, 1}, {3, 89, 255, 0, 0, 0, 1}},
		{{4, 54, 255, 1, 0, 0, 1}, {4, 171, 255, 0, 0, 0, 1}, {4, 48, 255, 0, 0, 0, 1}, {3, 193, 255, 1, 0, 0, 1}, {3, 94, 255, 1, 0, 0, 1}, {3, 5, 255, 1, 0, 0, 1}, {3, 178, 255, 0, 0, 0, 1}, {3, 104, 255, 0, 0, 0, 1}, {3, 36, 255, 0, 0, 0, 1}, {2, 228, 255, 1, 0, 0, 1}, {2, 171, 255, 1, 0, 0, 1}, {2, 117, 255, 1, 0, 0, 1}, {2, 67, 255, 1, 0, 0, 1}, {2, 21, 255, 1, 0, 0, 1}, {2, 233, 255, 0, 0, 0, 1}, {2, 192, 255, 0, 0, 0, 1}, {2, 155, 255, 0, 0, 0, 1}},
		{{4, 118, 255, 0, 0, 0, 1}, {3, 246, 255, 1, 0, 0, 1}, {3, 133, 255, 1, 0, 0, 1}, {3, 32, 255, 1, 0, 0, 1}, {3, 197, 255, 0, 0, 0, 1}, {3, 114, 255, 0, 0, 0, 1}, {3, 40, 255, 0, 0, 0, 1}, {2, 226, 255, 1, 0, 0, 1}, {2, 164, 255, 1, 0, 0, 1}, {2, 106, 255, 1, 0, 0, 1}, {2, 53, 255, 1, 0, 0, 1}, {2, 4, 255, 1, 0, 0, 1}, {2, 213, 255, 0, 0, 0, 1}, {2, 171, 255, 0, 0, 0, 1}, {2, 131, 255, 0, 0, 0, 1}, {2, 94, 255, 0, 0, 0, 1}, {2, 59, 255, 0, 0, 0, 1}},
		{{3, 89, 255, 0, 0, 0, 1}, {2, 248, 255, 1, 0, 0, 1}, {2, 164, 255, 1, 0, 0, 1}, {2, 88, 255, 1, 0, 0, 1}, {2, 20, 255, 1, 0, 0, 1}, {2, 213, 255, 0, 0, 0, 1}, {2, 157, 255, 0, 0, 0, 1}, {2, 106, 255, 0, 0, 0, 1}, {2, 59, 255, 0, 0, 0, 1}, {2, 16, 255, 0, 0, 0, 1}, {1, 231, 255, 1, 0, 0, 1}, {1, 194, 255, 1, 0, 0, 1}, {1, 160, 255, 1, 0, 0, 1}, {1, 128, 255, 1, 0, 0, 1}, {1, 98, 255, 1, 0, 0, 1}, {1, 70, 255, 1, 0, 0, 1}, {1, 44, 255, 1, 0, 0, 1}},
		{{2, 59, 255, 0, 0, 0, 1}, {1, 251, 255, 1, 0, 0, 1}, {1, 194, 255, 1, 0, 0, 1}, {1, 144, 255, 1, 0, 0, 1}, {1, 98, 255, 1, 0, 0, 1}, {1, 57, 255, 1, 0, 0, 1}, {1, 20, 255, 1, 0, 0, 1}, {1, 241, 255, 0, 0, 0, 1}, {1, 209, 255, 0, 0, 0, 1}, {1, 181, 255, 0, 0, 0, 1}, {1, 154, 255, 0, 0, 0, 1}, {1, 129, 255, 0, 0, 0, 1}, {1, 107, 255, 0, 0, 0, 1}, {1, 85, 255, 0, 0, 0, 1}, {1, 66, 255, 0, 0, 0, 1}, {1, 47, 255, 0, 0, 0, 1}, {1, 30, 255, 0, 0, 0, 1}}
	}
};
#elif defined _USE_AUDIO_ON_7520V3
static T_I2s_ClkDiv gPcm1SlotClkDiv[REF_CLK_MAX][MAX_CLK][MAX_PCM_MODE] = {
	{
		{{25, 199, 0}, {23, 203, 0}},
		{{25, 199, 0}, {23, 203, 0}},
		{{25, 199, 0}, {23, 203, 0}},
		{{25, 199, 0}, {23, 203, 0}},
		{{25, 199, 0}, {23, 203, 0}},
		{{25, 199, 0}, {23, 203, 0}},
		{{25, 199, 0}, {23, 203, 0}},
		{{25, 199, 0}, {23, 203, 0}},
		{{25, 199, 0}, {23, 203, 0}},
		{{25, 199, 0}, {23, 203, 0}},
		{{25, 199, 0}, {23, 203, 0}}
	},
	{
		{{25, 199, 1}, {23, 203, 1}},
		{{25, 199, 1}, {23, 203, 1}},
		{{25, 199, 1}, {23, 203, 1}},
		{{25, 199, 1}, {23, 203, 1}},
		{{25, 199, 1}, {23, 203, 1}},
		{{25, 199, 1}, {23, 203, 1}},
		{{25, 199, 1}, {23, 203, 1}},
		{{25, 199, 1}, {23, 203, 1}},
		{{25, 199, 1}, {23, 203, 1}},
		{{25, 199, 1}, {23, 203, 1}},
		{{25, 199, 1}, {23, 203, 1}}
	},
	{
		{{25, 199, 2}, {23, 203, 2}},
		{{25, 199, 2}, {23, 203, 2}},
		{{25, 199, 2}, {23, 203, 2}},
		{{25, 199, 2}, {23, 203, 2}},
		{{25, 199, 2}, {23, 203, 2}},
		{{25, 199, 2}, {23, 203, 2}},
		{{25, 199, 2}, {23, 203, 2}},
		{{25, 199, 2}, {23, 203, 2}},
		{{25, 199, 2}, {23, 203, 2}},
		{{25, 199, 2}, {23, 203, 2}},
		{{25, 199, 2}, {23, 203, 2}}
	},

};

static T_I2s_ClkDiv gPcm2SlotClkDiv[REF_CLK_MAX][MAX_CLK][MAX_PCM_MODE] = {
	{
		{{12, 100, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}}
	},
	{
		{{12, 100, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}}
	},
	{
		{{12, 100, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}}
	}
};

static T_I2s_ClkDiv gPcm4SlotClkDiv[REF_CLK_MAX][MAX_CLK][MAX_PCM_MODE] = {
	{
		{{12, 100, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}}
	},
	{
		{{12, 100, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}}
	},
	{
		{{12, 100, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}}
	}
};

static T_I2s_ClkDiv gPcm8SlotClkDiv[REF_CLK_MAX][MAX_CLK][MAX_PCM_MODE] = {
	{
		{{12, 100, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}}
	},
	{
		{{12, 100, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}}
	},
	{
		{{12, 100, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}}
	}
};

static T_I2s_ClkDiv gPcm16SlotClkDiv[REF_CLK_MAX][MAX_CLK][MAX_PCM_MODE] = {
	{
		{{12, 100, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}},
		{{0, 0, 0}, {0, 0, 0}}
	},
	{
		{{12, 100, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}},
		{{0, 0, 1}, {0, 0, 1}}
	},
	{
		{{12, 100, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}},
		{{0, 0, 2}, {0, 0, 2}}
	}
};

static T_I2s_ClkDiv gI2sClkDiv[REF_CLK_MAX][MAX_CLK][MAX_TS_WIDTH] = {
	{
		{{0x00200019, 0x00032, CLK_8000, TS_16CYCLE, REF_CLK26M}, {0x00200019, 0x10032, CLK_8000, TS_17CYCLE, 0}, {0x00200019, 0x10032, CLK_8000, TS_18CYCLE, 0}, {0x00200019, 0x10032, CLK_8000, TS_19CYCLE, 0}, {0x00200019, 0x10032, CLK_8000, TS_20CYCLE, 0}, {0x00200019, 0x10032, CLK_8000, TS_21CYCLE, 0}, {0x00200019, 0x10032, CLK_8000, TS_22CYCLE, 0}, {0x00200019, 0x10032, CLK_8000, TS_23CYCLE, 0}, {0x00200019, 0x10032, CLK_8000, TS_24CYCLE, 0}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 0}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 0}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 0}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 0}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 0}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 0}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 0}, {0x00400019, 0x10019, CLK_8000, TS_32CYCLE, 0}},
		{{0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 0}, {0x01b90176, 0x10024, CLK_11025, TS_17CYCLE, 0}, {0x01b90176, 0x10024, CLK_11025, TS_18CYCLE, 0}, {0x01b90176, 0x10024, CLK_11025, TS_19CYCLE, 0}, {0x01b90176, 0x10024, CLK_11025, TS_20CYCLE, 0}, {0x01b90176, 0x10024, CLK_11025, TS_21CYCLE, 0}, {0x01b90176, 0x10024, CLK_11025, TS_22CYCLE, 0}, {0x01b90176, 0x10024, CLK_11025, TS_23CYCLE, 0}, {0x01b90176, 0x10024, CLK_11025, TS_24CYCLE, 0}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 0}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 0}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 0}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 0}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 0}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 0}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 0}, {0x01b900bb, 0x10012, CLK_11025, TS_32CYCLE, 0}},
		{{0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 0}, {0x00300029, 0x10021, CLK_12000, TS_17CYCLE, 0}, {0x00300029, 0x10021, CLK_12000, TS_18CYCLE, 0}, {0x00300029, 0x10021, CLK_12000, TS_19CYCLE, 0}, {0x00300029, 0x10021, CLK_12000, TS_20CYCLE, 0}, {0x00300029, 0x10021, CLK_12000, TS_21CYCLE, 0}, {0x00300029, 0x10021, CLK_12000, TS_22CYCLE, 0}, {0x00300029, 0x10021, CLK_12000, TS_23CYCLE, 0}, {0x00300029, 0x10021, CLK_12000, TS_24CYCLE, 0}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 0}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 0}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 0}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 0}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 0}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 0}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_32CYCLE, 0}},
		{{0x00400019, 0x00019, CLK_16000, TS_16CYCLE, REF_CLK26M}, {0x00400019, 0x10019, CLK_12000, TS_17CYCLE, 0}, {0x00400019, 0x10019, CLK_12000, TS_18CYCLE, 0}, {0x00400019, 0x10019, CLK_12000, TS_19CYCLE, 0}, {0x00400019, 0x10019, CLK_12000, TS_20CYCLE, 0}, {0x00400019, 0x10019, CLK_12000, TS_21CYCLE, 0}, {0x00400019, 0x10019, CLK_12000, TS_22CYCLE, 0}, {0x00400019, 0x10019, CLK_12000, TS_23CYCLE, 0}, {0x00400019, 0x10019, CLK_12000, TS_24CYCLE, 0}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 0}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 0}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 0}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 0}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 0}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 0}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 0}, {0x00800059, 0x1000c, CLK_16000, TS_32CYCLE, 0}},
		{{0x01b900bb, 0x10012, CLK_22050, TS_16CYCLE, 0}, {0x01b900bb, 0x10012, CLK_12000, TS_17CYCLE, 0}, {0x01b900bb, 0x10012, CLK_12000, TS_18CYCLE, 0}, {0x01b900bb, 0x10012, CLK_12000, TS_19CYCLE, 0}, {0x01b900bb, 0x10012, CLK_12000, TS_20CYCLE, 0}, {0x01b900bb, 0x10012, CLK_12000, TS_21CYCLE, 0}, {0x01b900bb, 0x10012, CLK_12000, TS_22CYCLE, 0}, {0x01b900bb, 0x10012, CLK_12000, TS_23CYCLE, 0}, {0x01b900bb, 0x10012, CLK_12000, TS_24CYCLE, 0}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 0}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 0}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 0}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 0}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 0}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 0}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 0}, {0x037200bb, 0x10009, CLK_22050, TS_32CYCLE, 0}},
		{{0x00600059, 0x10010, CLK_24000, TS_16CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_17CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_18CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_19CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_20CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_21CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_22CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_23CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_24CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 0}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 0}, {0x00c00059, 0x10008, CLK_24000, TS_32CYCLE, 0}},
		{{0x00800059, 0x1000c, CLK_32000, TS_16CYCLE, 0}, {0x00800059, 0x1000c, CLK_12000, TS_17CYCLE, 0}, {0x00800059, 0x1000c, CLK_12000, TS_18CYCLE, 0}, {0x00800059, 0x1000c, CLK_12000, TS_19CYCLE, 0}, {0x00800059, 0x1000c, CLK_12000, TS_20CYCLE, 0}, {0x00800059, 0x1000c, CLK_12000, TS_21CYCLE, 0}, {0x00800059, 0x1000c, CLK_12000, TS_22CYCLE, 0}, {0x00800059, 0x1000c, CLK_12000, TS_23CYCLE, 0}, {0x00800059, 0x1000c, CLK_12000, TS_24CYCLE, 0}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 0}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 0}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 0}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 0}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 0}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 0}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 0}, {0x01000059, 0x10006, CLK_32000, TS_32CYCLE, 0}},
		{{0x037200bb, 0x10009, CLK_44100, TS_16CYCLE, 0}, {0x037200bb, 0x10009, CLK_12000, TS_17CYCLE, 0}, {0x037200bb, 0x10009, CLK_12000, TS_18CYCLE, 0}, {0x037200bb, 0x10009, CLK_12000, TS_19CYCLE, 0}, {0x037200bb, 0x10009, CLK_12000, TS_20CYCLE, 0}, {0x037200bb, 0x10009, CLK_12000, TS_21CYCLE, 0}, {0x037200bb, 0x10009, CLK_12000, TS_22CYCLE, 0}, {0x037200bb, 0x10009, CLK_12000, TS_23CYCLE, 0}, {0x037200bb, 0x10009, CLK_12000, TS_24CYCLE, 0}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 0}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 0}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 0}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 0}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 0}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 0}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 0}, {0x06e4042d, 0x10004, CLK_44100, TS_32CYCLE, 0}},
		{{0x00c00059, 0x10008, CLK_48000, TS_16CYCLE, 0}, {0x00c00059, 0x10008, CLK_12000, TS_17CYCLE, 0}, {0x00c00059, 0x10008, CLK_12000, TS_18CYCLE, 0}, {0x00c00059, 0x10008, CLK_12000, TS_19CYCLE, 0}, {0x00c00059, 0x10008, CLK_12000, TS_20CYCLE, 0}, {0x00c00059, 0x10008, CLK_12000, TS_21CYCLE, 0}, {0x00c00059, 0x10008, CLK_12000, TS_22CYCLE, 0}, {0x00c00059, 0x10008, CLK_12000, TS_23CYCLE, 0}, {0x00c00059, 0x10008, CLK_12000, TS_24CYCLE, 0}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 0}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 0}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 0}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 0}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 0}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 0}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 0}, {0x01800059, 0x10004, CLK_48000, TS_32CYCLE, 0}},
		{{0x01000059, 0x10006, CLK_64000, TS_16CYCLE, 0}, {0x01000059, 0x10006, CLK_12000, TS_17CYCLE, 0}, {0x01000059, 0x10006, CLK_12000, TS_18CYCLE, 0}, {0x01000059, 0x10006, CLK_12000, TS_19CYCLE, 0}, {0x01000059, 0x10006, CLK_12000, TS_20CYCLE, 0}, {0x01000059, 0x10006, CLK_12000, TS_21CYCLE, 0}, {0x01000059, 0x10006, CLK_12000, TS_22CYCLE, 0}, {0x01000059, 0x10006, CLK_12000, TS_23CYCLE, 0}, {0x01000059, 0x10006, CLK_12000, TS_24CYCLE, 0}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 0}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 0}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 0}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 0}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 0}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 0}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 0}, {0x02000059, 0x10003, CLK_64000, TS_32CYCLE, 0}},
		{{0x01800059, 0x10004, CLK_96000, TS_16CYCLE, 0}, {0x01800059, 0x10004, CLK_12000, TS_17CYCLE, 0}, {0x01800059, 0x10004, CLK_12000, TS_18CYCLE, 0}, {0x01800059, 0x10004, CLK_12000, TS_19CYCLE, 0}, {0x01800059, 0x10004, CLK_12000, TS_20CYCLE, 0}, {0x01800059, 0x10004, CLK_12000, TS_21CYCLE, 0}, {0x01800059, 0x10004, CLK_12000, TS_22CYCLE, 0}, {0x01800059, 0x10004, CLK_12000, TS_23CYCLE, 0}, {0x01800059, 0x10004, CLK_12000, TS_24CYCLE, 0}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 0}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 0}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 0}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 0}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 0}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 0}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 0}, {0x03000059, 0x10002, CLK_96000, TS_32CYCLE, 0}}
	},
	{
		{{0x00200019, 0x10032, CLK_8000, TS_16CYCLE, REF_CLK104M}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 1}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 1}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 1}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 1}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 1}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 1}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 1}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 1}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 1}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 1}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 1}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 1}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 1}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 1}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 1}, {0x00400019, 0x10019, CLK_8000, TS_32CYCLE, 1}},
		{{0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_11025, TS_32CYCLE, 1}},
		{{0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_32CYCLE, 1}},
		{{0x00400019, 0x10019, CLK_16000, TS_16CYCLE, REF_CLK104M}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 1}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 1}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 1}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 1}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 1}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 1}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 1}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 1}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 1}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 1}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 1}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 1}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 1}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 1}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_16000, TS_32CYCLE, 1}},
		{{0x01b900bb, 0x10012, CLK_22050, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 1}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_22050, TS_32CYCLE, 1}},
		{{0x00600059, 0x10010, CLK_24000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 1}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_24000, TS_32CYCLE, 1}},
		{{0x00800059, 0x1000c, CLK_32000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 1}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_32000, TS_32CYCLE, 1}},
		{{0x037200bb, 0x10009, CLK_44100, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 1}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 1}, {0x06e4042d, 0x10004, CLK_44100, TS_32CYCLE, 1}},
		{{0x00c00059, 0x10008, CLK_48000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 1}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_48000, TS_32CYCLE, 1}},
		{{0x01000059, 0x10006, CLK_64000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 1}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 1}, {0x02000059, 0x10003, CLK_64000, TS_32CYCLE, 1}},
		{{0x01800059, 0x10004, CLK_96000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 1}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 1}, {0x03000059, 0x10002, CLK_96000, TS_32CYCLE, 1}}
	},
	{
		{{0xffff0000, 0x10078, CLK_8000, TS_16CYCLE, REF_CLK122M88}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 2}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 2}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 2}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 2}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 2}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 2}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 2}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 2}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 2}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 2}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 2}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 2}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 2}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 2}, {0x00200019, 0x10032, CLK_8000, TS_16CYCLE, 2}, {0xffff0000, 0x10078, CLK_8000, TS_32CYCLE, 2}},
		{{0xffff0000, 0x10078, CLK_11025, TS_16CYCLE, 2}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 2}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 2}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 2}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 2}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 2}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 2}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 2}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 2}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 2}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 2}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 2}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 2}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 2}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 2}, {0x01b90176, 0x10024, CLK_11025, TS_16CYCLE, 2}, {0xffff0000, 0x10078, CLK_11025, TS_32CYCLE, 2}},
		{{0xffff0000, 0x10078, CLK_12000, TS_16CYCLE, 2}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 2}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 2}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 2}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 2}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 2}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 2}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 2}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 2}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 2}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 2}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 2}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 2}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 2}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 2}, {0x00300029, 0x10021, CLK_12000, TS_16CYCLE, 2}, {0xffff0000, 0x10078, CLK_12000, TS_32CYCLE, 2}},
		{{0xffff0000, 0x10078, CLK_16000, TS_16CYCLE, REF_CLK122M88}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 2}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 2}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 2}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 2}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 2}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 2}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 2}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 2}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 2}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 2}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 2}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 2}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 2}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 2}, {0x00400019, 0x10019, CLK_12000, TS_16CYCLE, 2}, {0xffff0000, 0x10078, CLK_16000, TS_32CYCLE, 2}},
		{{0xffff0000, 0x10078, CLK_22050, TS_16CYCLE, 2}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 2}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 2}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 2}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 2}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 2}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 2}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 2}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 2}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 2}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 2}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 2}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 2}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 2}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 2}, {0x01b900bb, 0x10012, CLK_12000, TS_16CYCLE, 2}, {0xffff0000, 0x10078, CLK_22050, TS_32CYCLE, 2}},
		{{0xffff0000, 0x10078, CLK_24000, TS_16CYCLE, 2}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 2}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 2}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 2}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 2}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 2}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 2}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 2}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 2}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 2}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 2}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 2}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 2}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 2}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 2}, {0x00600059, 0x10010, CLK_12000, TS_16CYCLE, 2}, {0xffff0000, 0x10078, CLK_24000, TS_32CYCLE, 2}},
		{{0xffff0000, 0x10078, CLK_32000, TS_16CYCLE, 2}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 2}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 2}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 2}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 2}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 2}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 2}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 2}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 2}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 2}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 2}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 2}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 2}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 2}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 2}, {0x00800059, 0x1000c, CLK_12000, TS_16CYCLE, 2}, {0xffff0000, 0x10078, CLK_32000, TS_32CYCLE, 2}},
		{{0xffff0000, 0x10078, CLK_44100, TS_16CYCLE, 2}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 2}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 2}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 2}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 2}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 2}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 2}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 2}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 2}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 2}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 2}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 2}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 2}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 2}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 2}, {0x037200bb, 0x10009, CLK_12000, TS_16CYCLE, 2}, {0xffff0000, 0x10078, CLK_44100, TS_32CYCLE, 2}},
		{{0xffff0000, 0x10078, CLK_48000, TS_16CYCLE, 2}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 2}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 2}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 2}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 2}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 2}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 2}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 2}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 2}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 2}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 2}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 2}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 2}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 2}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 2}, {0x00c00059, 0x10008, CLK_12000, TS_16CYCLE, 2}, {0xffff0000, 0x10078, CLK_48000, TS_32CYCLE, 2}},
		{{0xffff0000, 0x10078, CLK_64000, TS_16CYCLE, 2}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 2}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 2}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 2}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 2}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 2}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 2}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 2}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 2}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 2}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 2}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 2}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 2}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 2}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 2}, {0x01000059, 0x10006, CLK_12000, TS_16CYCLE, 2}, {0xffff0000, 0x10078, CLK_64000, TS_32CYCLE, 2}},
		{{0xffff0000, 0x10078, CLK_96000, TS_16CYCLE, 2}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 2}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 2}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 2}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 2}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 2}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 2}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 2}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 2}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 2}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 2}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 2}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 2}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 2}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 2}, {0x01800059, 0x10004, CLK_12000, TS_16CYCLE, 2}, {0xffff0000, 0x10078, CLK_96000, TS_32CYCLE, 2}}
	}
};
#endif



static T_ZDrvI2S_TimingCfg gI2S_TimingCfg[] = {
	{TIMING_I2S_MONO_RIGHT, I2S_TIME_MODE, 0, 0, 1},
	{TIMING_I2S_MONO_LEFT, I2S_TIME_MODE, 0, 0, 2},
	{TIMING_I2S_ST_2CHN_1LANE, I2S_TIME_MODE, 1, 0, 3},
	{TIMING_I2S_ST_4CHN_2LANE, I2S_TIME_MODE, 3, 1, 3},
	{TIMING_I2S_ST_6CHN_3LANE, I2S_TIME_MODE, 5, 2, 3},
	{TIMING_I2S_ST_8CHN_4LANE, I2S_TIME_MODE, 7, 3, 3},
	{TIMING_TDM_1CHN_1LANE_1TS, PCM_TIME_MODE, 0, 0, 0},
	{TIMING_TDM_2CHN_1LANE_2TS, PCM_TIME_MODE, 1, 0, 1}
};

static UINT32 g_i2sClkDivRegs[2] = {
	LSP_CRM_REG_BASE + 0x18,
	LSP_CRM_REG_BASE + 0x20,
};

static BOOL s_bI2sInitInt = FALSE;
static BOOL s_bI2sClkInit = FALSE;

static UINT32 audioGpioWsHandle = 0;
static UINT32 audioGpioClkHandle = 0;
static UINT32 audioGpioDinHandle = 0;
static UINT32 audioGpioDoutHandle = 0;

/*
static T_ZDrvDma_ChannelDef s_i2s1Tx_DmaChlDef[6] = {0};
static T_ZDrvDma_ChannelDef s_i2s2Tx_DmaChlDef[6] = {0};
static T_ZDrvDma_ChannelDef s_i2s1Rx_DmaChlDef[6] = {0};
static T_ZDrvDma_ChannelDef s_i2s2Rx_DmaChlDef[6] = {0};
*/
/*******************************************************************************
 *                        Global variable definitions                          *
 ******************************************************************************/


/* add by lvwenhua for i2s sync */
SINT32 gI2sDmaReadIntTime[MAX_I2S_ID] = {0};
SINT32 gI2sDmaWriteIntTime[MAX_I2S_ID] = {0};
SINT32 gI2sTimes = 0;
SINT32 gI2sCntChangeTx[MAX_I2S_ID] = {0};
SINT32 gI2sCntChangeRx[MAX_I2S_ID] = {0};
//SINT32 gI2sTimeDiff = 3000;

UINT32 i2swrite_tid = 0;
UINT32 i2swrite_pbuf = 0;
UINT32 i2swrite_uilen = 0;

UINT32 freebufferadd = 0;
UINT32 freebuffertid = 0;
extern T_zDrvVoice_GbVar g_voiceVar;

/*******************************************************************************
 *                      Inline function implementations                        *
 ******************************************************************************/
/**
* @brief	check if the queue is empty.
*
* @param	i2sQueue		queue pointer
* @return	0-DRV_SUCCESS, other-error
*/
__inline BOOL i2s_IsQueueEmpty(T_I2s_Queue *i2sQueue)
{
	if (NULL == i2sQueue) { //klocwork 3 INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	return (i2sQueue->writeIndex == i2sQueue->readIndex);
}

/**
* @brief	check if the queue is full.
*
* @param	i2sQueue		queue pointer
* @return	0-DRV_SUCCESS, other-error
*/
__inline BOOL i2s_IsQueueFull(T_I2s_Queue *i2sQueue)
{
	if (NULL == i2sQueue) {	//klocwork 3 INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	return ((i2sQueue->writeIndex + 1) % I2S_QUEUE_SIZE == i2sQueue->readIndex);
}

/*******************************************************************************
 *                      Local function implementations                         *
 ******************************************************************************/
/**
* @brief	use to manage tx buffer status ,update one buffer status.
*
* @param	tId 		i2s select i2s0 or i2s1
* @param	bufAddr 	buffer pointer
* @param	bufferStatus 	buffer status
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_UpdateTxBufferStatus(T_ZDrvI2S_Id tId, UINT16 *bufAddr, T_I2s_BufferStatus bufferStatus)
{
	T_I2s_TxStream *i2sStream = &s_i2sTxStream[tId];
	UINT8 bufIndex = 0;

	if (NULL == bufAddr) { //klocwork 3 INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}
	if (i2sStream->txBufferArray == NULL) {
		return DRV_ERROR;
	}
	for (; bufIndex < I2S_OUT_BUF_NUM; bufIndex++) {
		if (i2sStream->txBufferArray[bufIndex].buf == (UINT8 *)bufAddr) {
			i2sStream->txBufferArray[bufIndex].i2sBufStatus = bufferStatus;
			break;
		}
	}

	if (bufIndex == I2S_OUT_BUF_NUM) {
		return DRV_ERROR;
	}

	return DRV_SUCCESS;
}

/**
* @brief	use to manage rx buffer status ,update one buffer status.
*
* @param	tId 		i2s select i2s0 or i2s1
* @param	bufAddr 	buffer pointer
* @param	bufferStatus	buffer status
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_UpdateRxBufferStatus(T_ZDrvI2S_Id tId, UINT16 *bufAddr, T_I2s_BufferStatus bufferStatus)
{
	T_I2s_RxStream *i2sStream = &s_i2sRxStream[tId];
	UINT8 bufIndex = 0;

	if (NULL == bufAddr) { //klocwork 3 INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}
	if (i2sStream->rxBufferArray == NULL) {
		return DRV_ERROR;
	}
	for (; bufIndex < I2S_OUT_BUF_NUM; bufIndex++) {
		if (i2sStream->rxBufferArray[bufIndex].buf == (UINT8 *)bufAddr) {
			i2sStream->rxBufferArray[bufIndex].i2sBufStatus = bufferStatus;
			break;
		}
	}

	if (bufIndex == I2S_OUT_BUF_NUM) {
		return DRV_ERROR;
	}

	return DRV_SUCCESS;
}

/**
* @brief	use to manage Tx buffer status ,update one buffer status. if one buf status is one,change to new status.
*
* @param	tId 		i2s select i2s0 or i2s1
* @param	oldStatus 	buffer status
* @param	newStatus	buffer status
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_ChangeTxBufferStatus(T_ZDrvI2S_Id tId, T_I2s_BufferStatus oldStatus, T_I2s_BufferStatus newStatus)
{
	T_I2s_TxStream *i2sStream = &s_i2sTxStream[tId];
	UINT8 bufIndex = 0;

	for (; bufIndex < I2S_OUT_BUF_NUM; bufIndex++) {
		if (i2sStream->txBufferArray[bufIndex].i2sBufStatus == oldStatus) {
			i2sStream->txBufferArray[bufIndex].i2sBufStatus = newStatus;
			break;
		}
	}

	if (bufIndex == I2S_OUT_BUF_NUM) {
		return DRV_ERR_INVALID_PARAM;
	}

	return DRV_SUCCESS;
}

/**
* @brief	this function is use to init the buffer queue.
*
* @param	i2sQueue 		queue pointer
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_QueueInit(T_I2s_Queue *i2sQueue)
{
	SINT32 ret = 0;

	if (NULL == i2sQueue) { //klocwork 3 INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	i2sQueue->data = (T_I2s_QueueNode *)zOss_Malloc(sizeof(T_I2s_QueueNode) * I2S_QUEUE_SIZE);
	if (NULL == i2sQueue->data) {
		return DRV_ERR_MEM_ALLOC;
	}

	for (ret = 0; ret < I2S_QUEUE_SIZE; ret++) {
		i2sQueue->data[ret].pBuf = NULL;
		i2sQueue->data[ret].len = 0;
	}

	i2sQueue->readIndex = 0;
	i2sQueue->writeIndex = 0;

	return DRV_SUCCESS;
}

/**
* @brief	insert the queue to queue tail.
*
* @param	*pBuf		buffer pointer
* @param	uiLen		buffer length
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_QueuePut(T_I2s_Queue *i2sQueue, UINT16 *pBuf, UINT32 uiLen)
{
	BOOL isFull = FALSE;

	if (NULL == i2sQueue || NULL == pBuf || 0 == uiLen) { //klocwork 3 INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	isFull = i2s_IsQueueFull(i2sQueue);
	if (TRUE == isFull) { //klocwork 3 INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERROR;
	}
	if (i2sQueue->data == NULL) {
		return DRV_ERROR;
	}
	i2sQueue->data[i2sQueue->writeIndex].pBuf = pBuf;
	i2sQueue->data[i2sQueue->writeIndex].len = uiLen;
	i2sQueue->writeIndex = (i2sQueue->writeIndex + 1) % I2S_QUEUE_SIZE;

	return DRV_SUCCESS;
}

/**
* @brief	Fetch the queue node from queue head .
*
* @param	i2sQueue		queue pointer
* @param	**queueNode		queue node  pointer
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_QueueGet(T_I2s_Queue *i2sQueue, T_I2s_QueueNode **queueNode)
{
	BOOL isEmpty = FALSE;

	if (NULL == i2sQueue || NULL == queueNode) { //klocwork 3 INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	isEmpty = i2s_IsQueueEmpty(i2sQueue);

	if (TRUE == isEmpty) {
		*queueNode = NULL;
		return DRV_ERROR;
	} else {
		*queueNode = &i2sQueue->data[i2sQueue->readIndex];
		i2sQueue->readIndex = (i2sQueue->readIndex + 1) % I2S_QUEUE_SIZE;
	}

	return DRV_SUCCESS;
}

/**
* @brief	destroy the queue.
*
* @param	i2sQueue		queue pointer
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_QueueDestory(T_I2s_Queue *i2sQueue)
{
	if (NULL == i2sQueue) { //klocwork 3 INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	if (i2sQueue->data != NULL) {
		zOss_Free(i2sQueue->data);
	}

	i2sQueue->data = NULL;
	i2sQueue->readIndex = 0;
	i2sQueue->writeIndex = 0;

	return DRV_SUCCESS;
}

/**
* @brief	Get pcm division coeff.
*
* @param	ptCfg		config parameter
* @param	pDiv		pcm division coeff
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_GetPcmDiv(T_ZDrvI2S_Cfg *ptCfg, T_I2s_ClkDiv *pDiv)
{
	T_I2s_ClkConst i2sClockSet = MAX_CLK;

	if (NULL == pDiv || NULL == ptCfg || ptCfg->refclk >= REF_CLK_MAX || ptCfg->tPcmMode >= MAX_PCM_MODE) {
		return DRV_ERR_INVALID_PARAM;
	}

	switch (ptCfg->sample_rate) {
	case 8000:
		i2sClockSet = CLK_8000;
		break;
	case 11025:
		i2sClockSet = CLK_11025;
		break;
	case 12000:
		i2sClockSet = CLK_12000;
		break;
	case 16000:
		i2sClockSet = CLK_16000;
		break;
	case 22050:
		i2sClockSet = CLK_22050;
		break;
	case 24000:
		i2sClockSet = CLK_24000;
		break;
	case 32000:
		i2sClockSet = CLK_32000;
		break;
	case 44100:
		i2sClockSet = CLK_44100;
		break;
	case 48000:
		i2sClockSet = CLK_48000;
		break;
	case 64000:
		i2sClockSet = CLK_64000;
		break;
	case 96000:
		i2sClockSet = CLK_96000;
		break;
	default:
		i2sClockSet = CLK_44100;
		break;
	}

	if (ptCfg->tPcmSlotNum == PCM_1TIME_SLOT) {
		*pDiv = gPcm1SlotClkDiv[ptCfg->refclk][i2sClockSet][ptCfg->tPcmMode];
	} else if (ptCfg->tPcmSlotNum == PCM_2TIME_SLOT) {
		*pDiv = gPcm2SlotClkDiv[ptCfg->refclk][i2sClockSet][ptCfg->tPcmMode];
	} else if (ptCfg->tPcmSlotNum == PCM_4TIME_SLOT) {
		*pDiv = gPcm4SlotClkDiv[ptCfg->refclk][i2sClockSet][ptCfg->tPcmMode];
	} else if (ptCfg->tPcmSlotNum == PCM_8TIME_SLOT) {
		*pDiv = gPcm8SlotClkDiv[ptCfg->refclk][i2sClockSet][ptCfg->tPcmMode];;
	} else if (ptCfg->tPcmSlotNum == PCM_16TIME_SLOT) {
		*pDiv = gPcm16SlotClkDiv[ptCfg->refclk][i2sClockSet][ptCfg->tPcmMode];
	} else {
		return DRV_ERR_INVALID_PARAM;
	}

	return DRV_SUCCESS;
}

/**
* @brief	Get i2s division coeff.
*
* @param	ptCfg		config parameter
* @param	pDiv		pcm division coeff
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_GetI2sDiv(T_ZDrvI2S_Cfg *ptCfg, T_I2s_ClkDiv *pDiv)
{
	T_I2s_ClkConst i2sClockSet = MAX_CLK;
	T_I2s_tTsWidthConst i2sTsWidthSet = MAX_TS_WIDTH;

	if (NULL == pDiv || NULL == ptCfg || ptCfg->refclk >= REF_CLK_MAX) {
		return DRV_ERR_INVALID_PARAM;
	}

	switch (ptCfg->sample_rate) {
	case 8000:
		i2sClockSet = CLK_8000;
		break;
	case 11025:
		i2sClockSet = CLK_11025;
		break;
	case 12000:
		i2sClockSet = CLK_12000;
		break;
	case 16000:
		i2sClockSet = CLK_16000;
		break;
	case 22050:
		i2sClockSet = CLK_22050;
		break;
	case 24000:
		i2sClockSet = CLK_24000;
		break;
	case 32000:
		i2sClockSet = CLK_32000;
		break;
	case 44100:
		i2sClockSet = CLK_44100;
		break;
	case 48000:
		i2sClockSet = CLK_48000;
		break;
	case 64000:
		i2sClockSet = CLK_64000;
		break;
	case 96000:
		i2sClockSet = CLK_96000;
		break;
	default:
		i2sClockSet = CLK_44100;
		break;
	}

	switch (ptCfg->tTsWidth) {
	case I2S_TS_16CYCLE:
		i2sTsWidthSet = TS_16CYCLE;
		break;
	case I2S_TS_17CYCLE:
		i2sTsWidthSet = TS_17CYCLE;
		break;
	case I2S_TS_18CYCLE:
		i2sTsWidthSet = TS_18CYCLE;
		break;
	case I2S_TS_19CYCLE:
		i2sTsWidthSet = TS_19CYCLE;
		break;
	case I2S_TS_20CYCLE:
		i2sTsWidthSet = TS_20CYCLE;
		break;
	case I2S_TS_21CYCLE:
		i2sTsWidthSet = TS_21CYCLE;
		break;
	case I2S_TS_22CYCLE:
		i2sTsWidthSet = TS_22CYCLE;
		break;
	case I2S_TS_23CYCLE:
		i2sTsWidthSet = TS_23CYCLE;
		break;
	case I2S_TS_24CYCLE:
		i2sTsWidthSet = TS_24CYCLE;
		break;
	case I2S_TS_25CYCLE:
		i2sTsWidthSet = TS_25CYCLE;
		break;
	case I2S_TS_26CYCLE:
		i2sTsWidthSet = TS_26CYCLE;
		break;
	case I2S_TS_27CYCLE:
		i2sTsWidthSet = TS_27CYCLE;
		break;
	case I2S_TS_28CYCLE:
		i2sTsWidthSet = TS_28CYCLE;
		break;
	case I2S_TS_29CYCLE:
		i2sTsWidthSet = TS_29CYCLE;
		break;
	case I2S_TS_30CYCLE:
		i2sTsWidthSet = TS_30CYCLE;
		break;
	case I2S_TS_31CYCLE:
		i2sTsWidthSet = TS_31CYCLE;
		break;
	case I2S_TS_32CYCLE:
		i2sTsWidthSet = TS_32CYCLE;
		break;
	default:
		i2sTsWidthSet = TS_16CYCLE;
		break;
	}

	*pDiv = gI2sClkDiv[ptCfg->refclk][i2sClockSet][i2sTsWidthSet];

	return DRV_SUCCESS;
}

/**
* @brief	this function is use to close i2s divide frequency.
*
* @param	tId		i2s select the i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_CloseClk(T_ZDrvI2S_Id i2sId)
{
	if (i2sId == I2S_1)
		i2s_SetClkDiv(I2S0_PWRDN, 1);
	else if (i2sId == I2S_2)
		i2s_SetClkDiv(I2S1_PWRDN, 1);
	return DRV_SUCCESS;
}

/**
* @brief	this function is use to init i2s clock.
*
* @param	VOID
* @return	0-DRV_SUCCESS, other-error
*/

static VOID i2s_ClkInit(VOID)
{

	unsigned int val;
	if (s_bI2sClkInit) {
		return;
	}
	if (g_voiceVar.isUseSlicCodec != 1) {
		val = readl(I2S0_CRM_CLKSEL);
		val &= ~(3 << 8);
		writel(val, I2S0_CRM_CLKSEL);

		val = readl(I2S0_CRM_CLKSEL);
		val |= (3 << 8);
		writel(val, I2S0_CRM_CLKSEL);

		val = readl(I2S1_CRM_CLKSEL);
		val &= ~(3 << 8);
		writel(val, I2S1_CRM_CLKSEL);

		val = readl(I2S1_CRM_CLKSEL);
		val |= (3 << 8);
		writel(val, I2S1_CRM_CLKSEL);
		zOss_Sleep(1);
	}

	val = readl(I2S0_CRM_CLKSEL);
	val |= ((3 << 0) | (3 << 10));
	writel(val, I2S0_CRM_CLKSEL);

	val = readl(I2S1_CRM_CLKSEL);
	val |= ((3 << 0) | (3 << 10));
	writel(val, I2S1_CRM_CLKSEL);

	s_bI2sClkInit = TRUE;
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_ClkInit end s_bI2sClkInit=%d\r\n", s_bI2sClkInit);

}
#if 1
/**
* @brief	this function is use to disable i2s clock.
*
* @param	tId	   i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
static VOID i2s_ClkDeinit(T_ZDrvI2S_Id tId)
{

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_ClkDeinit %d", tId);
	unsigned int val;
	if (tId == I2S_1) {
#ifndef CONFIG_SND_SOC_ES8312

		val = readl(I2S0_CRM_CLKSEL);
		val &= ~((3 << 0) | (3 << 10));
		writel(val, I2S0_CRM_CLKSEL);

		val = readl(I2S0_CRM_CLKSEL);
		val &= ~(3 << 8);
		writel(val, I2S0_CRM_CLKSEL);

		val = readl(I2S0_CRM_CLKSEL);
		val |= (3 << 8);
		writel(val, I2S0_CRM_CLKSEL);
#endif

	} else {
		val = readl(I2S1_CRM_CLKSEL);
		val &= ~((3 << 0) | (3 << 10));
		writel(val, I2S1_CRM_CLKSEL);

		val = readl(I2S1_CRM_CLKSEL);
		val &= ~(3 << 8);
		writel(val, I2S1_CRM_CLKSEL);

		val = readl(I2S1_CRM_CLKSEL);
		val |= (3 << 8);
		writel(val, I2S1_CRM_CLKSEL);
	}

	s_bI2sClkInit = FALSE;

}

/**
* @brief	this function is use to disable i2s clock.
*
* @param	tId	   i2s select i2s0 or i2s1
* @param	pDiv   pointer to T_I2s_ClkDiv
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_SetClk(T_ZDrvI2S_Id tId, T_I2s_ClkDiv *pDiv)
{

	UINT32  AmrRegBit_ClkSel;
	unsigned int val;

#ifdef _USE_AUDIO_ON_7520V2
	UINT32  AmrRegBit_ClkDiv;
#endif

	if (MAX_I2S_ID <= tId || NULL == pDiv) {
		return DRV_ERR_INVALID_PARAM;
	}

	if (tId == I2S_1) {
#if defined _USE_AUDIO_ON_7520V2

		val = readl(I2S0_CRM_CLKSEL);
		val &= 0xFFFFFFEF;
		if (pDiv->refclk == REF_CLK26M) {
			val |= 0x00000000;
		} else { /*sel 104M or 26M clk for i2s ref clk*/
			val |= 0x00000010;
		}
		writel(val, I2S0_CRM_CLKSEL);

		AmrRegBit_ClkDiv = I2S0_CLKDIV;
		AmrRegBit_ClkDiv &= 0x0FFFFFFF;
		AmrRegBit_ClkDiv |= (pDiv->clkRatio) << 28;
		I2S0_CLKDIV = AmrRegBit_ClkDiv;

		i2s_SetClkDiv(I2S0_INTEGER_NUM, pDiv->INTEGERNUM);
		i2s_SetClkDiv(I2S0_FRA_DIV, pDiv->FRACADDNUM);
		i2s_SetClkDiv(I2S0_FRA_BASE, pDiv->FRACBASE);
		i2s_SetClkDiv(I2S0_CLK_SEL_DIV, pDiv->CLKSELDIV);
		i2s_SetClkDiv(I2S0_PWRDN, pDiv->PWRDN);

		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_SetClk tId=%d,pDiv=%x! \r\n", tId, pDiv);
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_SetClk tId=%d,I2S0_CRM =%x! \r\n", tId, I2S0_CRM_CLKSEL);
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_SetClk tId=%d,I2S0_DIV =%x! \r\n", tId, I2S0_CLKDIV);
#elif defined _USE_AUDIO_ON_7520V3

		val = readl(I2S0_CRM_CLKSEL);
		if ((pDiv->refclk == REF_CLK122M88) && (pDiv->i2sClockSet == CLK_8000)) {
			val |= (3 << 16);
		} else {
			val |= (1 << 16);
		}
		if (pDiv->refclk == REF_CLK26M) {
			val |= (0 << 4);
		} else if (pDiv->refclk == REF_CLK122M88) {
			val |= (1 << 4);
		} else if (pDiv->refclk == REF_CLK104M) {
			val |= (2 << 4);
		}
		writel(val, I2S0_CRM_CLKSEL);

		val = readl(I2S0_CLKDIV_2);
		val &= (~(0x1 << 16));
		writel(val, I2S0_CLKDIV_2);

		val = pDiv->FRACDIV1;
		writel(val, I2S0_CLKDIV_1);
		val = pDiv->FRACDIV2;
		val |= (0x1 << 16);
		writel(val, I2S0_CLKDIV_2);

		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "[XK]i2s_SetClk i2s0 DIV(0x%x, 0x%x, 0x%x)LSP(0x%x, 0x%x, 0x%x)!\r\n", \
		                  pDiv->refclk, pDiv->i2sClockSet, pDiv->i2sTsWidthSet, readl(I2S0_CRM_CLKSEL), readl(I2S0_CLKDIV_1), readl(I2S0_CLKDIV_2));
#endif
	} else { //klocwork 3 INVARIANT_CONDITION.GEN  (tId == I2S_2)
#if defined _USE_AUDIO_ON_7520V2

		val = readl(I2S1_CRM_CLKSEL);
		val &= 0xFFFFFFEF;
		if (pDiv->refclk == REF_CLK26M) {
			val |= 0x00000000;
		} else { /*sel 104M or 26M clk for i2s ref clk*/
			val |= 0x00000010;
		}
		writel(val, I2S1_CRM_CLKSEL);

		AmrRegBit_ClkDiv = I2S1_CLKDIV;
		AmrRegBit_ClkDiv &= 0x0FFFFFFF;
		AmrRegBit_ClkDiv |= (pDiv->clkRatio) << 28;
		I2S1_CLKDIV = AmrRegBit_ClkDiv;

		i2s_SetClkDiv(I2S1_INTEGER_NUM, pDiv->INTEGERNUM);
		i2s_SetClkDiv(I2S1_FRA_DIV, pDiv->FRACADDNUM);
		i2s_SetClkDiv(I2S1_FRA_BASE, pDiv->FRACBASE);
		i2s_SetClkDiv(I2S1_CLK_SEL_DIV, pDiv->CLKSELDIV);
		i2s_SetClkDiv(I2S1_PWRDN, pDiv->PWRDN);
#elif defined _USE_AUDIO_ON_7520V3

		val = readl(I2S1_CRM_CLKSEL);
		if ((pDiv->refclk == REF_CLK122M88) && (pDiv->i2sClockSet == CLK_8000)) {
			val |= (3 << 16);
		} else {
			val |= (1 << 16);
		}
		if (pDiv->refclk == REF_CLK26M) {
			val |= (0 << 4);
		} else if (pDiv->refclk == REF_CLK122M88) {
			val |= (1 << 4);
		} else if (pDiv->refclk == REF_CLK104M) {
			val |= (2 << 4);
		}
		writel(val, I2S1_CRM_CLKSEL);

		val = readl(I2S1_CLKDIV_2);
		val &= (~(0x1 << 16));
		writel(val, I2S1_CLKDIV_2);

		val = pDiv->FRACDIV1;
		writel(val, I2S1_CLKDIV_1);
		val = pDiv->FRACDIV2;
		val |= (0x1 << 16);
		writel(val, I2S1_CLKDIV_2);

		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "[XK]i2s_SetClk i2s1 DIV(0x%x, 0x%x, 0x%x)LSP(0x%x, 0x%x, 0x%x)!\r\n", \
		                  pDiv->refclk, pDiv->i2sClockSet, pDiv->i2sTsWidthSet, readl(I2S1_CRM_CLKSEL), readl(I2S1_CLKDIV_1), readl(I2S1_CLKDIV_2));

#endif

	}
	return DRV_SUCCESS;
}
#endif
/**
* @brief	this function is use to reset all device,contain i2s fifo and dma.
*
* @param	tId	   i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_Reset(T_ZDrvI2S_Id tId)
{
	T_I2s_Reg *ptReg = NULL;

	if (MAX_I2S_ID <= tId || NULL == s_i2sInstance[tId].ptDev) { //klocwork 3  INVARIANT_CONDITION.GEN  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERROR;
	}

	ptReg = s_i2sInstance[tId].ptDev->regs;

	/* disable i2s; disable rx/tx; as slave: clock from external */
	ptReg->timingCtrl = 0x0;

	/* disable dma */
	ptReg->fifoCtrl &= ~(I2S_TX_DMA_EN | I2S_RX_DMA_EN0);
#ifdef _USE_AUDIO_ON_7520V2
	/* reset tx/rx fifo */
	ptReg->fifoCtrl = I2S_TX_FIFO_RST | I2S_RX_FIFO_RST;
#endif
	/* clear interrupt */
	ptReg->intStatus = 0xffff;

	/* disable interrupt */
	ptReg->intEn = 0x0;

	/* disable i2s; disable rx/tx; */
	ptReg->processCtrl = 0x0;
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_Reset id=%d", tId);

	return DRV_SUCCESS;
}


/**
* @brief	this function is use to setup the tx buffer.
*
* @param	tId	   i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_SetupTxBuf(T_ZDrvI2S_Id tId)
{
	SINT32 index = 0;
	T_I2s_TxStream *i2sStrem = &s_i2sTxStream[tId];
	if (MAX_I2S_ID <= tId) { //klocwork 3  INVARIANT_CONDITION.GEN  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	i2sStrem->txBufferArray = (T_I2s_BufInfo *)zOss_Malloc(I2S_OUT_BUF_NUM * sizeof(T_I2s_BufInfo));
	if (NULL == i2sStrem->txBufferArray) {
		return DRV_ERR_MEM_ALLOC;
	}

	for (index = 0; index < I2S_OUT_BUF_NUM; index++) {
		i2sStrem->txBufferArray[index].buf = s_txBufArray[tId][index];
		i2sStrem->txBufferArray[index].i2sBufStatus = BUFFER_NULL;
		//zOss_Memset(i2sStrem->txBufferArray[index].buf, 0, I2S_MAX_TX_BUF_SIZE);
	}

	return DRV_SUCCESS;
}

/**
* @brief	this function is use to setup the rx buffer.
*
* @param	tId	   i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_SetupRxBuf(T_ZDrvI2S_Id tId)
{
	SINT32 index = 0;
	T_I2s_RxStream *i2sStrem = &s_i2sRxStream[tId];

	if (MAX_I2S_ID <= tId) { //klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0) ;
		return DRV_ERR_INVALID_PARAM;
	}

	i2sStrem->rxBufferArray = (T_I2s_BufInfo *)zOss_Malloc(I2S_OUT_BUF_NUM * sizeof(T_I2s_BufInfo));
	if (NULL == i2sStrem->rxBufferArray) {
		return DRV_ERR_MEM_ALLOC;
	}

	for (index = 0; index < I2S_OUT_BUF_NUM; index++) {
		i2sStrem->rxBufferArray[index].buf = s_rxBufArray[tId][index];
		i2sStrem->rxBufferArray[index].i2sBufStatus = BUFFER_NULL;
		//zOss_Memset(i2sStrem->rxBufferArray[index].buf, 0, I2S_MAX_RX_BUF_SIZE);
	}

	i2sStrem->currentRxBuf = NULL;

	return DRV_SUCCESS;
}

/**
* @brief	config i2s.
*
* @param	tId	   	i2s select i2s0 or i2s1
* @param	ptCfg	   pointer to i2s config param
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_Configure(T_ZDrvI2S_Id tId, T_ZDrvI2S_Cfg *ptCfg)
{
	T_I2s_Instance *ptInstance = NULL;
	T_I2s_Reg *ptReg = NULL;
	SINT32 ret = DRV_SUCCESS;

	if (MAX_I2S_ID <= tId || NULL == ptCfg) { //klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	ptInstance = &s_i2sInstance[tId];
	ptReg = ptInstance->ptDev->regs;

	ret = i2s_SetTiming(ptReg, ptCfg);
	if (ret != DRV_SUCCESS) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_Configure  i2s_SetTiming ret=0x%x", ret);
		return DRV_ERROR;
	}

	if (ptCfg->bMaster) {
		ret = i2s_SetPdiv(tId, ptCfg);
		if (ret != DRV_SUCCESS) {
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_Configure  i2s_SetPdiverr ret=0x%x", ret);
			return DRV_ERROR;
		}
		ptReg->timingCtrl |= I2S_MASTER_EN;   /* as master */
	} else {
		ptReg->timingCtrl &= ~I2S_MASTER_EN;  /* as slave */
	}

	/*transmit data at negative edge or at positive edge*/
	if (ptCfg->tTansmit_edge == TRANS_POSITIVE_NEGATIVE) {
		ptReg->timingCtrl |= I2S_CLK_EDGE_SEL;   /*transmit data at negative edge,so recieve data at positive edge*/
	} else if (ptCfg->tTansmit_edge == TRANS_NEGATIVE_POSITIVE) {
		ptReg->timingCtrl &= ~I2S_CLK_EDGE_SEL;
	}
	/*test mode*/
	if (ptCfg->tTestMode == I2S_NORMAL_MODE) {
		ptReg->timingCtrl &= ~I2S_TEST_MODE;
	} else if (ptCfg->tTestMode == I2S_LOOPBACK_MODE) {
		ptReg->timingCtrl |= I2S_TEST_MODE;
	}

	/*data align mode*/
	if (ptCfg->tTransMode < MAX_DATA_ALIGN_MODE) {
		ptReg->timingCtrl &= ~ I2S_ALIGN_MODE(3);  /*clear 0*/
		ptReg->timingCtrl |=  I2S_ALIGN_MODE(ptCfg->tTransMode);
	} else {
		return DRV_ERR_INVALID_PARAM;
	}

	if (MAX_I2S_DATA_FMT <= ptCfg->tDataFmt) { //klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	} else {
		ptReg->timingCtrl &= ~ I2S_DATA_SIZE(0x1f);
		ptReg->timingCtrl |= I2S_DATA_SIZE(ptCfg->tDataFmt);
	}

	if (MAX_I2S_TS_WIDTH <= ptCfg->tTsWidth) { //klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	} else {
		ptReg->timingCtrl &= ~ I2S_TS_WIDTH(0x1f);
		ptReg->timingCtrl |= I2S_TS_WIDTH(ptCfg->tTsWidth);  /*I2S_TS_WIDTH*/
	}

	if (ptCfg->tPcmFsync == PCM_SHORT_FSYNC) {
		ptReg->timingCtrl &= ~I2S_LONG_FSYNC;
		ptCfg->tPcmFsynLen = PCM_FSYNC_LENGTH_1CLK;
	} else if (ptCfg->tPcmFsync ==  PCM_LONG_FSYNC) {
		ptReg->timingCtrl |= I2S_LONG_FSYNC;
		ptCfg->tPcmFsynLen = MAX_PCM_FSYNC; /*no meaning,for compatible	;7520 1ts*/
	} else { //klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	if (ptCfg->tTdmExtCycle == TDM_NO_EXTRA_CYCLE) {
		ptReg->timingCtrl &= ~I2S_EXTRA_CYCLE;

	} else if (ptCfg->tTdmExtCycle == TDM_TEAK_EXTRA_CYCLE) {
		ptReg->timingCtrl |= I2S_EXTRA_CYCLE;
	} else { //klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_Configure  success");
	return DRV_SUCCESS;
}

/**
* @brief	clean Tx Resource.
*
* @param	tId		i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_CleanTxResource(T_ZDrvI2S_Id tId)
{
	SINT32 ret = DRV_SUCCESS;

	ret = i2s_QueueDestory(&s_i2sTxStream[tId].txQueue);
	if (s_i2sTxStream[tId].txBufferArray != NULL) {
		zOss_Free(s_i2sTxStream[tId].txBufferArray);
		s_i2sTxStream[tId].txBufferArray = NULL;
	}

	return ret;
}

/**
* @brief	clean Rx Resource.
*
* @param	tId		i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_CleanRxResource(T_ZDrvI2S_Id tId)
{
	SINT32 ret = DRV_SUCCESS;

	ret = i2s_QueueDestory(&s_i2sRxStream[tId].rxQueue);
	if (s_i2sRxStream[tId].rxBufferArray != NULL) {
		zOss_Free(s_i2sRxStream[tId].rxBufferArray);
		s_i2sRxStream[tId].rxBufferArray = NULL;
	}

	return ret;
}

/**
* @brief	Tx Check i2s parameter config.
*
* @param	tId	 	i2s select i2s0 or i2s1
* @param	ptCfg	pointer to i2s config
* @return	0-DRV_SUCCESS, other-error
*/

static SINT32 i2s_TxCheckState(T_ZDrvI2S_Id tId, T_ZDrvI2S_Cfg *ptCfg)
{
	SINT32 ret = DRV_SUCCESS;
	T_I2s_Instance *ptInstance = &s_i2sInstance[tId];

	if (MAX_I2S_ID <= tId || NULL == ptCfg) { //klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	if ((ptInstance->i2sStatus != I2S_OPEN) && (ptInstance->i2sStatus != I2S_RECORD_INUSE)) {
		return DRV_ERR_NOT_OPENED;
	}

	if (ptInstance->i2sStatus != I2S_RECORD_INUSE) {
		ptInstance->tCfg = *ptCfg;
		ret = i2s_Configure(tId, ptCfg);
		if (ret != DRV_SUCCESS) {
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_TxCheckState i2s_Configure err ret=0x%x", ret);
			return DRV_ERROR;
		}
	} else {
		if ((ptCfg->bMaster != ptInstance->tCfg.bMaster) || (ptCfg->sample_rate != ptInstance->tCfg.sample_rate) || (ptCfg->tClkMode != ptInstance->tCfg.tClkMode) ||
		    (ptCfg->tTansmit_edge != ptInstance->tCfg.tTansmit_edge) || (ptCfg->tTestMode != ptInstance->tCfg.tTestMode) || (ptCfg->tTransMode != ptInstance->tCfg.tTransMode)) {
			return DRV_ERR_INVALID_PARAM;
		}

		if (ptCfg->tClkMode == I2S_TIME_MODE) {
			if ((ptCfg->tDataFmt != ptInstance->tCfg.tDataFmt) || (ptCfg->tTsWidth != ptInstance->tCfg.tTsWidth) ||
			    (ptCfg->tTrackFmt != ptInstance->tCfg.tTrackFmt)) {
				return DRV_ERR_INVALID_PARAM;
			}
		}
		if (ptCfg->tClkMode == PCM_TIME_MODE) {
			if ((ptCfg ->tPcmFirstBit != ptInstance ->tCfg.tPcmFirstBit) || (ptCfg ->tPcmFsynLen != ptInstance ->tCfg.tPcmFsynLen) ||
			    (ptCfg ->tPcmMode != ptInstance ->tCfg.tPcmMode) || (ptCfg ->tPcmSlotNum != ptInstance ->tCfg.tPcmSlotNum)) {
				return DRV_ERR_INVALID_PARAM;
			}
		}
	}

	return DRV_SUCCESS;
}

/**
* @brief	Rx Check i2s parameter config.
*
* @param	tId	 	i2s select i2s0 or i2s1
* @param	ptCfg	pointer to i2s config
* @return	0-DRV_SUCCESS, other-error
*/

static SINT32 i2s_RxCheckState(T_ZDrvI2S_Id tId, T_ZDrvI2S_Cfg *ptCfg)
{
	SINT32 ret = DRV_SUCCESS;
	T_I2s_Instance *ptInstance = &s_i2sInstance[tId];

	if (MAX_I2S_ID <= tId || NULL == ptCfg) { //klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}
	/* clear interrupt. must disable interrupt or i2s at first */
	if ((ptInstance->i2sStatus != I2S_OPEN) && (ptInstance->i2sStatus != I2S_PLAY_INUSE)) {
		return DRV_ERR_NOT_OPENED;
	}

	if (ptInstance->i2sStatus != I2S_PLAY_INUSE) {
		ptInstance->tCfg = *ptCfg;
		ret = i2s_Configure(tId, ptCfg);
		if (ret != DRV_SUCCESS) {
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_RxCheckState  failed ret is %d \r\n", ret);
			return DRV_ERROR;
		}
	} else {
		if ((ptCfg->bMaster != ptInstance->tCfg.bMaster) || (ptCfg->sample_rate != ptInstance->tCfg.sample_rate) ||
		    (ptCfg->tClkMode != ptInstance->tCfg.tClkMode) || (ptCfg->tTansmit_edge != ptInstance->tCfg.tTansmit_edge) ||
		    (ptCfg->tTestMode != ptInstance->tCfg.tTestMode) || (ptCfg->tTransMode != ptInstance->tCfg.tTransMode)) {
			return DRV_ERR_INVALID_PARAM;
		}

		if (ptCfg->tClkMode == I2S_TIME_MODE) {
			if ((ptCfg->tDataFmt != ptInstance->tCfg.tDataFmt) || (ptCfg->tTsWidth != ptInstance->tCfg.tTsWidth) ||
			    (ptCfg->tTrackFmt != ptInstance->tCfg.tTrackFmt)) {
				return DRV_ERR_INVALID_PARAM;
			}
		}

		if (ptCfg->tClkMode == PCM_TIME_MODE) {
			if ((ptCfg ->tPcmFirstBit != ptInstance ->tCfg.tPcmFirstBit) || (ptCfg ->tPcmFsynLen != ptInstance ->tCfg.tPcmFsynLen) ||
			    (ptCfg ->tPcmMode != ptInstance ->tCfg.tPcmMode) || (ptCfg ->tPcmSlotNum != ptInstance ->tCfg.tPcmSlotNum)) {
				return DRV_ERR_INVALID_PARAM;
			}
		}
	}

	return DRV_SUCCESS;
}

/**
* @brief	Initial the tx common parameter.
*
* @param	tId	 	i2s select i2s0 or i2s1
* @param	params	pointer to i2s params
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_TxComInit(T_ZDrvI2S_Id tId, T_ZDrvI2s_Params *params)
{
	T_I2s_TxStream *txStream = &s_i2sTxStream[tId];
	T_I2s_Instance *ptInstance = &s_i2sInstance[tId];
	T_I2s_Reg *ptReg = NULL;
	SINT32 ret = DRV_SUCCESS;
	UINT32 reg_val;

	if ((MAX_I2S_ID <= tId) || (NULL == params)) { //klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	ptReg = ptInstance->ptDev->regs;
	txStream->Transing = FALSE;
	txStream->pauseFlag = FALSE;
	txStream->txTransmit = NULL;
	txStream->channel = params->channel;
	txStream->txLen = params->buffersize;
	txStream->p_cb = params->p_cb;

	txStream->txRdIdx = 0;
	txStream->txWrIdx = 0;

	s_txBufArray[tId][0] = (UINT8 *)(i2sDmaState[tId][I2S_TX].dma_viraddr);
	s_txBufArray[tId][1] = s_txBufArray[tId][0] + txStream->txLen;
	s_txBufArray[tId][2] = s_txBufArray[tId][1] + txStream->txLen;

#ifdef _USE_AUDIO_ON_7520V2
	ptReg->fifoCtrl |= I2S_TX_FIFO_RST;
#endif
	//ptReg->fifoCtrl |= I2S_TFTH(30);    /*DMA MODE is also necessry*/
	reg_val = ptReg->fifoCtrl;
	reg_val &= 0xffffe0ff;
	reg_val |= I2S_TFTH(15);
	ptReg->fifoCtrl = reg_val;

	//ptReg->fifoCtrl |= I2S_TFTH(16);    /*DMA MODE is also necessry*/

	ret = i2s_SetupTxBuf(tId);
	ret += i2s_QueueInit(&(txStream->txQueue));
	if (ret != DRV_SUCCESS) {
		i2s_CleanTxResource(tId);
		return DRV_ERROR;
	}

	i2s_txsem_count_Init(tId);
	return DRV_SUCCESS;
}

/**
* @brief	Initial the rx common parameter.
*
* @param	tId	 	i2s select i2s0 or i2s1
* @param	params	pointer to i2s params
* @return	0-DRV_SUCCESS, other-error
*/

static SINT32 i2s_RxComInit(T_ZDrvI2S_Id tId, T_ZDrvI2s_Params *params)
{
	T_I2s_RxStream *rxStream = &s_i2sRxStream[tId];
	T_I2s_Instance *ptInstance = &s_i2sInstance[tId];
	T_I2s_Reg *ptReg = NULL;
	SINT32 ret = DRV_SUCCESS;
	UINT32 reg_val;

	if (MAX_I2S_ID <= tId || NULL == params) { //klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	rxStream->rxLen = params->buffersize;

	s_rxBufArray[tId][0] = (UINT8 *)(i2sDmaState[tId][I2S_RX].dma_viraddr);
	s_rxBufArray[tId][1] = s_rxBufArray[tId][0] + rxStream->rxLen;
	s_rxBufArray[tId][2] = s_rxBufArray[tId][1] + rxStream->rxLen;

	ptReg = ptInstance->ptDev->regs;

	reg_val = ptReg->fifoCtrl;
	reg_val &= 0xffe0ffff;
	reg_val |= I2S_RFTH(7);
	ptReg->fifoCtrl = reg_val;

	ret = i2s_SetupRxBuf(tId);
	ret += i2s_QueueInit(&(rxStream->rxQueue));
	if (ret != DRV_SUCCESS) {
		i2s_CleanRxResource(tId);
		return DRV_ERROR;
	}

	i2s_rxsem_count_Init(tId);
	//rxStream->rxIdx = 0;
	rxStream->rxRdIdx = 0;
	rxStream->rxWrIdx = 0;

//	rxStream->rxBufferArray[rxStream->rxIdx].i2sBufStatus = BUFFER_WRITE;
//	rxStream->currentRxBuf = &rxStream->rxBufferArray[rxStream->rxIdx];

	return DRV_SUCCESS;
}

/**
* @brief	Initial the tx interrupt common parameter.
*
* @param	tId	 	i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_IntTxInit(T_ZDrvI2S_Id tId)
{
	T_I2s_Instance *ptInstance = &s_i2sInstance[tId];
	T_I2s_TxStream *txStream = &s_i2sTxStream[tId];
	T_I2s_Reg *ptReg = NULL;

	if (MAX_I2S_ID <= tId) { //klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	ptReg = ptInstance->ptDev->regs;
	txStream->txTransmit = i2s_IntTx;
	ptReg->intEn &= ~(I2S_TX_TRIG_INT_EN | I2S_TX_UNDER_INT_EN);  /* disable tx interrupt */
	ptReg->intStatus |= (I2S_TX_TRIG_INT | I2S_TX_UNDER_INT);    /* clear interrupt. must disable interrupt or i2s at first */

	return DRV_SUCCESS;
}

/**
* @brief	Initial the rx interrupt common parameter.
*
* @param	tId	 	i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_IntRxInit(T_ZDrvI2S_Id tId)
{
#if 0
	T_I2s_Instance *ptInstance = &s_i2sInstance[tId];
	T_I2s_RxStream *rxStream = &s_i2sRxStream[tId];
	T_I2s_Reg *ptReg = NULL;

	zDrv_ASSERT(tId < MAX_I2S_ID);
	if (MAX_I2S_ID <= tId) {
		return DRV_ERR_INVALID_PARAM;
	}

	ptReg = ptInstance->ptDev->regs;

	ptReg->intStatus |= (I2S_RX_TRIG_INT | I2S_RX_OVER_INT);          /* clear interrupt */
	ptReg->intEn &= ~(I2S_RX_TRIG_INT_EN | I2S_RX_OVER_INT_EN);        /* disable rx interrupt */

	s_rxBuf[tId] = (UINT16 *)rxStream->currentRxBuf->buf;
	s_rxLen[tId] = rxStream->rxLen / sizeof(UINT16);

	rxStream->rxIdx++;
	/* enable rx interrupt */
	ptReg->intEn |= I2S_RX_TRIG_INT_EN;       /* enable rx interrupt */

	if (I2S_INTCTRL_TEST_MODE == TRUE) {
		ptReg->intEn |= I2S_RX_OVER_INT_EN; /* enable rx over interrupt */
	}
#endif
	return DRV_SUCCESS;
}

/**
* @brief	interrupt handle while in int mode.
*
* @param	tId	 	i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/

static VOID i2s_IntIsr(T_ZDrvI2S_Id tId)
{
	T_I2s_Reg *ptReg = NULL;
	T_I2s_TxStream *i2sTxStream = NULL;
	//const UINT32 uiIntStatus = 0;
	UINT32 channLable = 0;

	if (MAX_I2S_ID <= tId) { //klocwork 3  INVARIANT_CONDITION.UNREACH  delete assert
		//zDrv_ASSERT(0);
		return;
	}
	ptReg = s_i2sInstance[tId].ptDev->regs;
	i2sTxStream = &s_i2sTxStream[tId];
	const UINT32 uiIntStatus = ptReg->intStatus;

	if (I2S_INTCTRL_TEST_MODE) { //klocwork 3  INVARIANT_CONDITION.GEN
		if (uiIntStatus & I2S_TX_UNDER_INT) {       /* tx err */
			ptReg->intStatus |= I2S_TX_UNDER_INT;   /* clear interrupt */
		}
		if (uiIntStatus & I2S_RX_OVER_INT) {         /* rx err */
			ptReg->intStatus |= I2S_RX_OVER_INT;     /* clear interrupt */
		}
	}

	if (uiIntStatus & I2S_TX_TRIG_INT) {        /* playing */
		ptReg->intEn &= ~I2S_TX_TRIG_INT_EN;  /* disable tx interrupt */
		ptReg->intStatus |= I2S_TX_TRIG_INT;    /* clear interrupt. must disable interrupt or i2s at first */
		channLable = (i2sTxStream->channel == AUDIO_DUAL_CHANNEL) ? 2 : 1;
		i2s_IntTxWriteFifo(tId, channLable);
		if (0 == s_txLen[tId]) {
			i2s_IntTxChangeBuf(tId);
			return;                  /* while no data trans,then return and not enable the interrupt,so the interrupt will not generate*/
		}
		ptReg->intEn |= I2S_TX_TRIG_INT_EN ; /* enable tx interrupt */
		if (I2S_INTCTRL_TEST_MODE) { //klocwork 3  INVARIANT_CONDITION.GEN
			ptReg->intEn |= I2S_TX_UNDER_INT_EN; /* enable tx interrupt */
		}
	}

	if (uiIntStatus & I2S_RX_TRIG_INT) {              /* recording */
		ptReg->intEn &= ~I2S_RX_TRIG_INT_EN;        /* disable rx interrupt */
		ptReg->intStatus |= I2S_RX_TRIG_INT;          /* clear interrupt */
		i2s_IntRxWriteBuf(tId);

		if (0 == s_rxLen[tId]) {
			i2s_IntRxChangeBuf(tId);
		}
		ptReg->intEn |= I2S_RX_TRIG_INT_EN;        /* enable rx interrupt */
		if (I2S_INTCTRL_TEST_MODE) { //klocwork 3  INVARIANT_CONDITION.GEN
			ptReg->intEn |= I2S_RX_OVER_INT_EN; /* enable tx interrupt */
		}
	}

	if (uiIntStatus & I2S_TX_UNDER_INT) {       /* tx err */
		ptReg->intStatus |= I2S_TX_UNDER_INT;   /* clear interrupt */
#ifdef _USE_AUDIO_ON_7520V2
		ptReg->fifoCtrl |= I2S_TX_FIFO_RST;          /* reset tx fifo */
#endif
	}

	if (uiIntStatus & I2S_RX_OVER_INT) {         /* rx err */
		ptReg->intStatus |= I2S_RX_OVER_INT;     /* clear interrupt */
#ifdef _USE_AUDIO_ON_7520V2
		ptReg->fifoCtrl |= I2S_RX_FIFO_RST;           /* reset rx fifo */
#endif
	}
}

/**
* @brief	Tx Write fifo while in int mode.
*
* @param	tId	 	i2s select i2s0 or i2s1
* @param	chanlab	 	chanlab
* @return	0-DRV_SUCCESS, other-error
*/
static VOID i2s_IntTxWriteFifo(T_ZDrvI2S_Id tId, UINT32 chanlab)
{
	UINT32 uiHdledLen = I2S_TXFL_VAL;
	T_I2s_Reg *ptReg = NULL;
	UINT32 chanlable = 0;

	if (MAX_I2S_ID <= tId) { //klocwork 3  INVARIANT_CONDITION.UNREACH  delete assert
		//	zDrv_ASSERT(0);
		return;/* while no data trans,then return and not enable the interrupt,so the interrupt will not generate*/
	}
	ptReg = s_i2sInstance[tId].ptDev->regs;
	chanlable = 2;

	if (s_txLen[tId] > I2S_TXFL_VAL * chanlable) {
		s_txLen[tId] -= uiHdledLen * chanlable;
		while (0 < uiHdledLen) {
			ptReg->data = *(s_txBuf[tId] + (chanlable - 1)) | *s_txBuf[tId] << 16;
			s_txBuf[tId] += chanlable;
			uiHdledLen--;
		}
	} else {
		while (0 < s_txLen[tId]) {
			ptReg->data = *(s_txBuf[tId] + (chanlable - 1)) | *s_txBuf[tId] << 16;
			s_txBuf[tId] += chanlable;
			s_txLen[tId] -= chanlable;
		}
	}
}

/**
* @brief	Rx Write Buffer while in int mode.
*
* @param	tId	 	i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
static VOID i2s_IntRxWriteBuf(T_ZDrvI2S_Id tId)
{
	T_I2s_Reg *ptReg = NULL;
	UINT32 uiRxledLen = I2S_RXFL_VAL;
	UINT32 data = 0;

	if (MAX_I2S_ID <= tId) { //klocwork 3  INVARIANT_CONDITION.UNREACH  delete assert
		//	zDrv_ASSERT(0);
		return;
	}
	ptReg = s_i2sInstance[tId].ptDev->regs;

	if (s_rxLen[tId] > uiRxledLen * 2) {
		s_rxLen[tId] -= uiRxledLen * 2;
		while (0 < uiRxledLen) {
			data = ptReg->data;
			*(s_rxBuf[tId] + 1) = data & 0xffff;
			*s_rxBuf[tId] = (data >> 16) & 0xffff;
			s_rxBuf[tId] += 2;
			uiRxledLen--;
		}
	} else {
		while (0 < s_rxLen[tId]) {
			data = ptReg->data;
			*(s_rxBuf[tId] + 1) = data & 0xffff;
			*s_rxBuf[tId] = (data >> 16) & 0xffff;
			s_rxBuf[tId] += 2;
			s_rxLen[tId] -= 2;
		}
	}
}

/**
* @brief	Rx write Buffer while in int mode.
*
* @param	tId	 	i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
static VOID i2s_IntTxChangeBuf(T_ZDrvI2S_Id tId)
{
	T_I2s_TxStream *i2sTxStream = NULL;
	T_I2s_QueueNode *queueNode = NULL;
	UINT32 sem_cnt = 0;
	SINT32 ret = DRV_SUCCESS;

	if (MAX_I2S_ID <= tId) { //klocwork 3  INVARIANT_CONDITION.UNREACH  delete assert
		//	zDrv_ASSERT(0);
		return;
	}
	i2sTxStream = &s_i2sTxStream[tId];

	if (DRV_SUCCESS == i2s_ChangeTxBufferStatus(tId, BUFFER_READ, BUFFER_NULL)) { /*change the BUF state from READ to FULL after the data of the buf is read over,then release the sem*/
		zOss_PutSemaphore(i2sTxStream->txSem);
	}

	if (!(i2s_IsQueueEmpty(&i2sTxStream->txQueue))) {
		if (i2sTxStream->pauseFlag == FALSE) {
			ret = i2s_QueueGet(&i2sTxStream->txQueue, &queueNode);
			if (queueNode != NULL) {
				ret += i2s_UpdateTxBufferStatus(tId, queueNode->pBuf, BUFFER_READ);
			}
			//		ret += i2sTxStream->txTransmit(tId, queueNode->pBuf, queueNode->len);
			if (ret != DRV_SUCCESS) {
				return;
			}
			i2sTxStream->Transing = TRUE;
		} else {
			i2sTxStream->Transing = FALSE;
			return;
		}
	} else {
		i2sTxStream->Transing = FALSE;
		sem_cnt = zOss_GetSemaphoreCount(i2sTxStream->txSem);
		if (sem_cnt == I2S_OUT_BUF_NUM) {
			if (i2sTxStream->p_cb != NULL) {
				i2sTxStream->p_cb();
			}
		}
	}
}

/**
* @brief	Rx change Buffer while in int mode.
*
* @param	tId	 	i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
static VOID i2s_IntRxChangeBuf(T_ZDrvI2S_Id tId)
{
#if 0
	T_I2s_RxStream *i2sRxStream = &s_i2sRxStream[tId];

	zDrv_ASSERT(tId < MAX_I2S_ID);
	if (MAX_I2S_ID <= tId) {
		return;
	}

	if (i2sRxStream->rxBufferArray[i2sRxStream->rxIdx].i2sBufStatus == BUFFER_NULL) {
		i2s_QueuePut(&(i2sRxStream->rxQueue), (UINT16 *)i2sRxStream->currentRxBuf->buf, (i2sRxStream->rxLen / sizeof(UINT16)));   /*ϴյ*/
		i2s_UpdateRxBufferStatus(tId, (UINT16*)i2sRxStream->currentRxBuf->buf, BUFFER_FULL);
		zOss_PutSemaphore(i2sRxStream->rxSem);
		i2sRxStream->rxBufferArray[i2sRxStream->rxIdx].i2sBufStatus = BUFFER_WRITE;
		i2sRxStream->currentRxBuf = &i2sRxStream->rxBufferArray[i2sRxStream->rxIdx];
		s_rxBuf[tId] = (UINT16 *)i2sRxStream->currentRxBuf->buf;
		s_rxLen[tId] = i2sRxStream->rxLen / sizeof(UINT16);
		i2sRxStream->rxIdx = (i2sRxStream->rxIdx + 1) % I2S_OUT_BUF_NUM;         /* use the buf in turn*/
	} else {                   /*if the next buf is bing usedtthen rewrite the buf*/
		s_rxBuf[tId] = (UINT16 *)i2sRxStream->currentRxBuf->buf;
		s_rxLen[tId] = i2sRxStream->rxLen / sizeof(UINT16);
	}
#endif
}

/**
* @brief	i2s1 isr.
*
* @param	VOID
* @return	0-DRV_SUCCESS, other-error
*/
static VOID isr_i2s1(VOID)
{
	i2s_IntIsr(I2S_1);
}

/**
* @brief	i2s2 isr.
*
* @param	VOID
* @return	0-DRV_SUCCESS, other-error
*/
static VOID isr_i2s2(VOID)
{
	i2s_IntIsr(I2S_2);
}

/**
* @brief	i2s interrupt enable.
*
* @param	tid		i2s select
* @param	pBuf	buffer pointer
* @param	uiLen	buffer length
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_IntTx(T_ZDrvI2S_Id tId, const UINT16 *pBuf, UINT32 uiLen)
{
	T_I2s_Instance *ptInstance = NULL;
	T_I2s_Reg *ptReg = NULL;

	/*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
	if (MAX_I2S_ID <= tId || NULL == pBuf || 0 == uiLen || I2S_MAX_TX_BUF_SIZE < uiLen) {
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	ptInstance = &s_i2sInstance[tId];
	if ((ptInstance->i2sStatus != I2S_PLAY_INUSE) && (ptInstance->i2sStatus != I2S_BOTH_INUSE)) {
		return DRV_ERR_NOT_STARTED;
	}

	s_txBuf[tId] = pBuf;
	s_txLen[tId] = uiLen;

	ptReg = ptInstance->ptDev->regs;

	ptReg->fifoCtrl |= I2S_TFTH(16);    /*DMA MODE is also necessry*/
	/* clear tx interrupt */
	ptReg->intStatus |= (I2S_TX_TRIG_INT | I2S_TX_UNDER_INT);    /* clear interrupt. must disable interrupt or i2s at first */
	ptReg->intEn |= I2S_TX_TRIG_INT_EN;        /* enable tx interrupt */
	if (I2S_INTCTRL_TEST_MODE) { //klocwork 3  INVARIANT_CONDITION.GEN
		ptReg->intEn |=  I2S_TX_UNDER_INT_EN; /* enable tx under interrupt */
	}

	return DRV_SUCCESS;
}

/**
* @brief	Dma tx interrupt callback function while playing.
*
* @param	tid			i2s select
* @param	tIntStatus	pointer to T_ZDrvDma_IntStatus
* @return	0-DRV_SUCCESS, other-error
*/
static VOID i2s_DmaTxIntCb(T_ZDrvI2S_Id tId)
{

	T_I2s_Reg *ptReg = NULL;
	//const UINT32 uiIntStatus = 0;
	UINT8 bufIndex = 0;
	T_I2s_TxStream *i2sTxStream = NULL;

	if (tId >= MAX_I2S_ID) { /*klocwork 3  INVARIANT_CONDITION.UNREACH  delete assert*/
		//	zDrv_ASSERT(0) ;
		return;
	}
	ptReg = s_i2sInstance[tId].ptDev->regs;
	const UINT32 uiIntStatus  = ptReg->intStatus;

	i2sTxStream = &s_i2sTxStream[tId];
	for (; bufIndex < I2S_OUT_BUF_NUM; bufIndex++) {
		if (i2sTxStream->txBufferArray[bufIndex].i2sBufStatus == BUFFER_FULL) {
			break;
		}
	}

	if (bufIndex == I2S_OUT_BUF_NUM)
		zDrvI2S_Write_Stop(tId);

	i2s_UpdateTxBufferStatus(tId, (UINT16 *)(i2sTxStream->txBufferArray[i2sTxStream->txRdIdx].buf), BUFFER_NULL);       /*ϲ*pBufдݺ󣬸ıBUF״̬ΪFULL*/
	i2sTxStream->txRdIdx = (i2sTxStream->txRdIdx + 1) % I2S_OUT_BUF_NUM;
	zOss_PutSemaphore(i2sTxStream->txSem);

//	i2s_IntTxChangeBuf(tId);
	if (uiIntStatus & I2S_TX_UNDER_INT) {       /* tx err */
		ptReg->intStatus |= I2S_TX_UNDER_INT;   /* clear interrupt */
	}

}

/**
* @brief	Dma rx interrupt callback function while playing.
*
* @param	tid			i2s select
* @param	ptInstance	pointer to T_I2s_Instance
* @param	tIntStatus	pointer to T_ZDrvDma_IntStatus
* @return	0-DRV_SUCCESS, other-error
*/
static VOID i2s_DmaRxIntCb(T_ZDrvI2S_Id tId)
{
	T_I2s_RxStream *i2sStream = NULL;
	UINT8 rxIndex = 0;
	T_I2s_Reg *ptReg = NULL;
	//const UINT32 uiIntStatus = 0;

	if (tId >= MAX_I2S_ID) {	/*klocwork 3  INVARIANT_CONDITION.UNREACH  delete assert*/
		//	zDrv_ASSERT(0);
		return;
	}

	i2sStream = &s_i2sRxStream[tId];
	ptReg = s_i2sInstance[tId].ptDev->regs;
	const UINT32 uiIntStatus = ptReg->intStatus;
#if 0
	if (i2sStream->rxBufferArray[i2sStream->rxIdx].i2sBufStatus == BUFFER_NULL) {
		i2s_QueuePut(&i2sStream->rxQueue, (UINT16 *)i2sStream->currentRxBuf->buf, (i2sStream->rxLen / sizeof(UINT16)));   /*put the full buf to the queue*/
		i2s_UpdateRxBufferStatus(tId, (UINT16*)i2sStream->currentRxBuf->buf, BUFFER_FULL);
		zOss_PutSemaphore(i2sStream->rxSem);          /*if the next buf status is null,then release the sem so the uplayer can get the buf to read*/
		//i2s_DmaRx(tId, ptInstance, i2sStream->rxIdx);
	} else {
		/*ϲڿbufд*/
		for (rxIndex = 0; rxIndex < I2S_OUT_BUF_NUM; rxIndex++) {                    /*if the next buf is bing usedtthen rewrite the buf**/
			if (i2sStream->rxBufferArray[rxIndex].i2sBufStatus == BUFFER_WRITE) {      /*it is necessary to find the index*/
				if (uiIntStatus & I2S_RX_OVER_INT) {         /* rx err */
					ptReg->intStatus |= I2S_RX_OVER_INT;     /* clear interrupt */
#ifdef _USE_AUDIO_ON_7520V2
					ptReg->fifoCtrl |= I2S_RX_FIFO_RST;           /* reset rx fifo */
#endif
				}
				break;
			}
			if (rxIndex == 2) {
				break;
			}
		}
		i2sStream->rxIdx = rxIndex;
		//i2s_DmaRx(tId, ptInstance, rxIndex);
	}


	if (uiIntStatus & I2S_RX_OVER_INT) {         /* rx err */
		ptReg->intStatus |= I2S_RX_OVER_INT;     /* clear interrupt */
#ifdef _USE_AUDIO_ON_7520V2
		ptReg->fifoCtrl |= I2S_RX_FIFO_RST;           /* reset rx fifo */
#endif

	}
#endif
}

/**
* @brief	i2s1 rx interrupt callback function.
*
* @param	tIntStatus	pointer to T_ZDrvDma_IntStatus
* @return	0-DRV_SUCCESS, other-error
*/
static VOID dmacb_i2s1_rx(void *data)
{
	//i2s_DmaRxIntCb(I2S_1);
	UINT32 sem_cnt = 0;
	T_I2s_RxStream *i2sStream = &s_i2sRxStream[I2S_1];
	/*
		UINT8 bufIndex = 0;

		for(; bufIndex < I2S_OUT_BUF_NUM; bufIndex++) {
			if(i2sStream->rxBufferArray[bufIndex].i2sBufStatus == BUFFER_FULL) {
				break;
			}
		}
		if (bufIndex == I2S_OUT_BUF_NUM)
			zDrvI2S_Read_Stop(I2S_1);
	*/

    if(i2sStream->rxBufferArray == NULL){
        return;
    }
    
	i2s_UpdateRxBufferStatus(I2S_1, (UINT16 *)(i2sStream->rxBufferArray[i2sStream->rxWrIdx].buf), BUFFER_FULL);
	i2sStream->rxWrIdx = (i2sStream->rxWrIdx + 1) % I2S_OUT_BUF_NUM;

	sem_cnt = zOss_GetSemaphoreCount(i2sStream->rxSem);
	if (sem_cnt < I2S_OUT_BUF_NUM) {
		zOss_PutSemaphore(i2sStream->rxSem);
	} else {
		i2sStream->rxRdIdx = (i2sStream->rxRdIdx + 1) % I2S_OUT_BUF_NUM;
	}
}

/**
* @brief	i2s2 rx interrupt callback function.
*
* @param	tIntStatus	pointer to T_ZDrvDma_IntStatus
* @return	0-DRV_SUCCESS, other-error
*/
static VOID dmacb_i2s2_rx(void *data)
{
	//i2s_DmaRxIntCb(I2S_2);
	UINT32 sem_cnt = 0;
	T_I2s_RxStream *i2sStream = &s_i2sRxStream[I2S_2];
	/*
		UINT8 bufIndex = 0;

		for(; bufIndex < I2S_OUT_BUF_NUM; bufIndex++) {
			if(i2sStream->rxBufferArray[bufIndex].i2sBufStatus == BUFFER_FULL) {
				break;
			}
		}
		if (bufIndex == I2S_OUT_BUF_NUM)
			zDrvI2S_Read_Stop(I2S_2);
	*/

    if(i2sStream->rxBufferArray == NULL){
        return;
    }
    
	i2s_UpdateRxBufferStatus(I2S_2, (UINT16 *)(i2sStream->rxBufferArray[i2sStream->rxWrIdx].buf), BUFFER_FULL);
	i2sStream->rxWrIdx = (i2sStream->rxWrIdx + 1) % I2S_OUT_BUF_NUM;
	sem_cnt = zOss_GetSemaphoreCount(i2sStream->rxSem);
	if (sem_cnt < I2S_OUT_BUF_NUM) {
		zOss_PutSemaphore(i2sStream->rxSem);
	} else {
		i2sStream->rxRdIdx = (i2sStream->rxRdIdx + 1) % I2S_OUT_BUF_NUM;
	}
}

/**
* @brief	i2s1 tx interrupt callback function.
*
* @param	tid			i2s select
* @param	ptInstance	pointer to T_I2s_Instance
* @param	tIntStatus	pointer to T_ZDrvDma_IntStatus
* @return	0-DRV_SUCCESS, other-error
*/
static VOID dmacb_i2s1_tx(void *data)
{
	//i2s_DmaTxIntCb(I2S_1);
	UINT32 sem_cnt = 0;
	T_I2s_TxStream *i2sTxStream = &s_i2sTxStream[I2S_1];
	/*
		UINT8 bufIndex = 0;

		for(; bufIndex < I2S_OUT_BUF_NUM; bufIndex++) {
			if(i2sTxStream->txBufferArray[bufIndex].i2sBufStatus == BUFFER_NULL) {
				break;
			}
		}
		if (bufIndex == I2S_OUT_BUF_NUM)
			zDrvI2S_Write_Stop(I2S_1);
	*/

    if(i2sTxStream->txBufferArray == NULL){
        return;
    }
    
	i2s_UpdateTxBufferStatus(I2S_1, (UINT16 *)(i2sTxStream->txBufferArray[i2sTxStream->txRdIdx].buf), BUFFER_NULL);       /*ϲ*pBufдݺ󣬸ıBUF״̬ΪFULL*/
	i2sTxStream->txRdIdx = (i2sTxStream->txRdIdx + 1) % I2S_OUT_BUF_NUM;
	sem_cnt = zOss_GetSemaphoreCount(i2sTxStream->txSem);
	if (sem_cnt < I2S_OUT_BUF_NUM) {
		zOss_PutSemaphore(i2sTxStream->txSem);
	} else {
		i2sTxStream->txWrIdx = (i2sTxStream->txWrIdx + 1) % I2S_OUT_BUF_NUM;
	}

}

/**
* @brief	i2s2 tx interrupt callback function.
*
* @param	tid			i2s select
* @param	ptInstance	pointer to T_I2s_Instance
* @param	tIntStatus	pointer to T_ZDrvDma_IntStatus
* @return	0-DRV_SUCCESS, other-error
*/
static VOID dmacb_i2s2_tx(void *data)
{
	//i2s_DmaTxIntCb(I2S_2);
	UINT32 sem_cnt = 0;
	T_I2s_TxStream *i2sTxStream = &s_i2sTxStream[I2S_2];
	/*
		UINT8 bufIndex = 0;

		for(; bufIndex < I2S_OUT_BUF_NUM; bufIndex++) {
			if(i2sTxStream->txBufferArray[bufIndex].i2sBufStatus == BUFFER_NULL) {
				break;
			}
		}
		if (bufIndex == I2S_OUT_BUF_NUM)
			zDrvI2S_Write_Stop(I2S_2);
	*/

    if(i2sTxStream->txBufferArray == NULL){
        return;
    }
    
	i2s_UpdateTxBufferStatus(I2S_2, (UINT16 *)(i2sTxStream->txBufferArray[i2sTxStream->txRdIdx].buf), BUFFER_NULL);       /*ϲ*pBufдݺ󣬸ıBUF״̬ΪFULL*/
	i2sTxStream->txRdIdx = (i2sTxStream->txRdIdx + 1) % I2S_OUT_BUF_NUM;
	sem_cnt = zOss_GetSemaphoreCount(i2sTxStream->txSem);
	if (sem_cnt < I2S_OUT_BUF_NUM) {
		zOss_PutSemaphore(i2sTxStream->txSem);
	} else {
		i2sTxStream->txWrIdx = (i2sTxStream->txWrIdx + 1) % I2S_OUT_BUF_NUM;
	}
}

#if 0
/**
* @brief	config and start i2s dma rx .
*
* @param	tid			i2s select
* @param	ptInstance	pointer to T_I2s_Instance
* @param	idx	pointer idx
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_DmaRx(T_ZDrvI2S_Id tId, T_I2s_Instance *ptInstance, UINT8 idx)
{
	SINT32 ret = DRV_SUCCESS;
	T_I2s_RxStream *i2sStream = &s_i2sRxStream[tId];
	T_ZDrvDma_ChannelDef tChanDef = {0};

	zDrv_ASSERT((tId < MAX_I2S_ID) && (ptInstance != NULL) && (idx < I2S_OUT_BUF_NUM));
	if ((MAX_I2S_ID <= tId) || (NULL == ptInstance) || (idx >= I2S_OUT_BUF_NUM)) {
		return DRV_ERR_INVALID_PARAM;
	}

	i2sStream->rxBufferArray[idx].i2sBufStatus = BUFFER_WRITE ;
	i2sStream->currentRxBuf = &i2sStream->rxBufferArray[idx];

	if (tId == I2S_1) {
		i2sDmaState[tId].rxDmaChanDef.SrcAddr = ZX297520V3_I2S0_PHYS + 0x1c;
	} else if (tId == I2S_2) {
		i2sDmaState[tId].rxDmaChanDef.SrcAddr = ZX297520V3_I2S1_PHYS + 0x1c;
	}
//	tChanDef.DestAddr = MODEM_VA_2_PA((UINT32)i2sStream->currentRxBuf->buf);
	i2sDmaState[tId].rxDmaChanDef.DestAddr = i2sDmaState[tId].dma_rx_phyaddr[0] + idx * (i2sDmaState[tId].rxsinglebuffsize);
	if (gI2sCntChangeRx[tId] >= -80) {
		i2sDmaState[tId].rxDmaChanDef.Count = ((UINT16)i2sStream->rxLen) + gI2sCntChangeRx[tId] * 2; //*sizeof(UINT32);//uiLen*sizeof(UINT16)
		gI2sCntChangeRx[tId] = 0;//add by lvwenhua for i2s sync  2013.12.7
	} else {
		i2sDmaState[tId].rxDmaChanDef.Count = ((UINT16)i2sStream->rxLen) - 80 * 2; //*sizeof(UINT32);//uiLen*sizeof(UINT16)
		gI2sCntChangeRx[tId] = gI2sCntChangeRx[tId] + 80; //add by lvwenhua for i2s sync  2013.12.7
	}


	/*modified by miaolin 20130801*/
	ret = zDrvDma_ConfigChannel(ptInstance->DmaRxID, i2sDmaState[tId].rxDmaChanDef);
	if (ret != DRV_SUCCESS) {
		zDrvDma_DeAllocChannel(ptInstance->DmaRxID);
		return ret;
	}
#ifdef MALLOC_I2S_DMA_AREA
	//TOS_DCACHE_INVALIDATE_RANGE((UINT32)i2sStream->currentRxBuf->buf,(UINT16)i2sStream->rxLen);
	dma_sync_single_for_cpu(&i2s_device.dev, i2sDmaState[tId].rxDmaChanDef.DestAddr, i2sDmaState[tId].rxDmaChanDef.Count, DMA_FROM_DEVICE);
#endif

	return ret;
}
#endif

/**
* @brief	dma rx init.
*
* @param	tid			i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_DmaRxInit(T_ZDrvI2S_Id tId)
{
	T_I2s_Instance  *ptInstance = NULL;
	T_I2s_RxStream *i2sStream = NULL;

	if (MAX_I2S_ID <= tId) { /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	i2sStream = &s_i2sRxStream[tId];
	ptInstance = &s_i2sInstance[tId];

	struct dma_async_tx_descriptor *desc = NULL;
	dma_cap_mask_t mask;

	dma_cap_zero(mask);
	dma_cap_set(DMA_CYCLIC, mask);

	dma_channel_def temp[3];
	int 	i;
	memset(temp, 0, sizeof(temp));
	i2sDmaState[tId][I2S_RX].channel = (tId == I2S_1) ? (DMA_CH_I2S0_RX0) : (DMA_CH_I2S1_RX0);
	i2sDmaState[tId][I2S_RX].ch = dma_request_channel(mask, zx29_dma_filter_fn, (void*)(i2sDmaState[tId][I2S_RX].channel));

    if(i2sDmaState[tId][I2S_RX].ch == NULL){
        for(i = 0; i <= 5; i++){
            zOss_Sleep(1);
            i2sDmaState[tId][I2S_RX].ch = dma_request_channel(mask, zx29_dma_filter_fn, (void*)(i2sDmaState[tId][I2S_RX].channel));
            
            if(i2sDmaState[tId][I2S_RX].ch != NULL){
                break;
            }
            else if(i == 5){
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_DmaRxInit error, i2sDmaState[%d][%d].ch is NULL\n", tId, I2S_RX);
                return DRV_ERROR;
            }
        }
    }

	for (i = 0; i < I2S_OUT_BUF_NUM; i++) {
		temp[i].dest_addr 	= i2sDmaState[tId][I2S_RX].dma_phyaddr  + (i2sStream->rxLen) * i;
		temp[i].src_addr	= (tId == I2S_1) ? (ZX29_I2S0_PHYS + 0x1c) : (ZX29_I2S1_PHYS + 0x1c);
		temp[i].dma_control.tran_mode = TRAN_PERI_TO_MEM;
		temp[i].dma_control.src_burst_len 	= DMA_BURST_LEN_8;
		temp[i].count		= i2sStream->rxLen;
		//		temp[i].callback	= (dma_callback_func)dma_cb;
		/*
			temp[i].dma_control.tran_mode 		= (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
													? TRAN_MEM_TO_PERI : TRAN_PERI_TO_MEM);
			temp[i].dma_control.src_burst_len 	= (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
													? DMA_BURST_LEN_16 : DMA_BURST_LEN_8);
		*/
		temp[i].dma_control.src_burst_size 	= DMA_BURST_SIZE_32BIT;
		temp[i].dma_control.dest_burst_size = DMA_BURST_SIZE_32BIT;
		temp[i].dma_control.dest_burst_len 	= DMA_BURST_LEN_8;
		temp[i].dma_control.irq_mode 		= DMA_ALL_IRQ_ENABLE;
		temp[i].link_addr 		= 1;
	}

	temp[I2S_OUT_BUF_NUM - 1].link_addr 	= 0;

	dmaengine_slave_config(i2sDmaState[tId][I2S_RX].ch, (struct dma_slave_config*)&temp);

	desc = i2sDmaState[tId][I2S_RX].ch->device->device_prep_dma_cyclic(
	           i2sDmaState[tId][I2S_RX].ch,
	           i2sDmaState[tId][I2S_RX].dma_phyaddr, i2sStream->rxLen * 3,
	           i2sStream->rxLen,
	           DMA_DEV_TO_MEM,
	           NULL);

	desc->callback = (tId == I2S_1) ? ((dma_async_tx_callback)dmacb_i2s1_rx) : ((dma_async_tx_callback)dmacb_i2s2_rx);
	desc->callback_param = NULL;

	dmaengine_submit(desc);

	dma_async_issue_pending(i2sDmaState[tId][I2S_RX].ch);

	return DRV_SUCCESS;
}

/**
* @brief	dma tx init.
*
* @param	tid			i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_DmaTxInit(T_ZDrvI2S_Id tId)
{
	T_I2s_Instance  *ptInstance = NULL;
	T_I2s_TxStream *txStream = NULL;
	struct dma_async_tx_descriptor *desc = NULL;
	dma_cap_mask_t mask;

	if (MAX_I2S_ID <= tId) { /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	txStream = &s_i2sTxStream[tId];
	dma_cap_zero(mask);
	dma_cap_set(DMA_CYCLIC, mask);

	ptInstance = &s_i2sInstance[tId];
	dma_channel_def temp[3];
	int 	i;
	memset(temp, 0, sizeof(temp));
	i2sDmaState[tId][I2S_TX].channel = (tId == I2S_1) ? (DMA_CH_I2S0_TX) : (DMA_CH_I2S1_TX);
	i2sDmaState[tId][I2S_TX].ch = dma_request_channel(mask, zx29_dma_filter_fn, (void*)(i2sDmaState[tId][I2S_TX].channel));

    if(i2sDmaState[tId][I2S_TX].ch == NULL){
        for(i = 0; i <= 5; i++){
            zOss_Sleep(1);
            i2sDmaState[tId][I2S_TX].ch = dma_request_channel(mask, zx29_dma_filter_fn, (void*)(i2sDmaState[tId][I2S_TX].channel));
            
            if(i2sDmaState[tId][I2S_TX].ch != NULL){
                break;
            }
            else if(i == 5){
                zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_DmaTxInit error, i2sDmaState[%d][%d].ch is NULL\n", tId, I2S_TX);
                return DRV_ERROR;
            }
        }
    }

	for (i = 0; i < I2S_OUT_BUF_NUM; i++) {
		temp[i].src_addr 	= i2sDmaState[tId][I2S_TX].dma_phyaddr + (txStream->txLen) * i;
		temp[i].dest_addr	= (tId == I2S_1) ? (ZX29_I2S0_PHYS + 0x1c) : (ZX29_I2S1_PHYS + 0x1c);
		temp[i].dma_control.tran_mode = TRAN_MEM_TO_PERI;
		temp[i].dma_control.src_burst_len 	= DMA_BURST_LEN_16;
		temp[i].count		= txStream->txLen;
		//		temp[i].callback	= (dma_callback_func)dma_cb;
		/*
			temp[i].dma_control.tran_mode 		= (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
													? TRAN_MEM_TO_PERI : TRAN_PERI_TO_MEM);
			temp[i].dma_control.src_burst_len 	= (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
													? DMA_BURST_LEN_16 : DMA_BURST_LEN_8);
		*/
		temp[i].dma_control.src_burst_size 	= DMA_BURST_SIZE_32BIT;
		temp[i].dma_control.dest_burst_size = DMA_BURST_SIZE_32BIT;
		temp[i].dma_control.dest_burst_len 	= DMA_BURST_LEN_16;
		temp[i].dma_control.irq_mode 		= DMA_ALL_IRQ_ENABLE;
		temp[i].link_addr 		= 1;
	}

	temp[I2S_OUT_BUF_NUM - 1].link_addr 	= 0;

	dmaengine_slave_config(i2sDmaState[tId][I2S_TX].ch, (struct dma_slave_config*)&temp);

	desc = i2sDmaState[tId][I2S_TX].ch->device->device_prep_dma_cyclic(
	           i2sDmaState[tId][I2S_TX].ch,
	           i2sDmaState[tId][I2S_TX].dma_phyaddr, txStream->txLen * 3,
	           txStream->txLen,
	           DMA_MEM_TO_DEV,
	           NULL);

	desc->callback = (tId == I2S_1) ? ((dma_async_tx_callback)dmacb_i2s1_tx) : ((dma_async_tx_callback)dmacb_i2s2_tx);
	desc->callback_param = NULL;

	dmaengine_submit(desc);

	return DRV_SUCCESS;
}

#if 0
/**
* @brief	config and start i2s dma tx .
*
* @param	tid			i2s select
* @param	pBuf	pointer to  i2s buffer
* @param	uiLen	pointer idx
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_DmaTx(T_ZDrvI2S_Id tId, const UINT16 *pBuf, UINT32 uiLen)
{
	SINT32 ret = DRV_SUCCESS;
	//SINT32 channel_id=0;
	T_I2s_Instance  *ptInstance = &s_i2sInstance[tId];
	T_I2s_TxStream  *txStream = &s_i2sTxStream[tId];
//    T_ZDrvDma_ChannelDef tChanDef = {0};

	zDrv_ASSERT((tId < MAX_I2S_ID) && (pBuf != NULL) && (uiLen != 0) && (uiLen <= I2S_MAX_TX_BUF_SIZE));
	if (MAX_I2S_ID <= tId || NULL == pBuf || 0 == uiLen || I2S_MAX_TX_BUF_SIZE < uiLen) {
		return DRV_ERR_INVALID_PARAM;
	}

	if (pBuf == s_txBufArray[tId][0]) {
		i2sDmaState[tId].txDmaChanDef.SrcAddr = i2sDmaState[tId].dma_tx_phyaddr[0];
	} else if (pBuf == s_txBufArray[tId][1]) {
		i2sDmaState[tId].txDmaChanDef.SrcAddr = i2sDmaState[tId].dma_tx_phyaddr[1];
	} else if (pBuf == s_txBufArray[tId][2]) {
		i2sDmaState[tId].txDmaChanDef.SrcAddr = i2sDmaState[tId].dma_tx_phyaddr[2];
	}

	if (tId == I2S_1) {
		i2sDmaState[tId].txDmaChanDef.DestAddr = ZX297520V3_I2S0_PHYS + 0x1c;
	} else if (tId == I2S_2) {
		i2sDmaState[tId].txDmaChanDef.DestAddr = ZX297520V3_I2S1_PHYS + 0x1c;
	}

	if (gI2sCntChangeTx[tId] >= -80) {
		i2sDmaState[tId].txDmaChanDef.Count = uiLen * sizeof(UINT16) + gI2sCntChangeTx[tId] * 2; //ֽڴС
		gI2sCntChangeTx[tId] = 0;//add by lvwenhua for i2s sync  2013.12.7
	} else {
		i2sDmaState[tId].txDmaChanDef.Count = uiLen * sizeof(UINT16) - 80 * 2; //*sizeof(UINT32);//uiLen*sizeof(UINT16)
		gI2sCntChangeTx[tId] = gI2sCntChangeTx[tId] + 80; //add by lvwenhua for i2s sync  2013.12.7
	}

	ret = zDrvDma_ConfigChannel(ptInstance->DmaTxID, i2sDmaState[tId].txDmaChanDef);
	if (ret != DRV_SUCCESS) {
		zDrvDma_DeAllocChannel(ptInstance->DmaTxID);
		return ret;
	}

	//ret=zDrvDma_StartChannel(channel_id, NULL, TRUE);
	ret = zDrvDma_StartChannel(ptInstance->DmaTxID, ptInstance->fDmaCbTx, FALSE);
	if (ret != DRV_SUCCESS) {
		zDrvDma_DeAllocChannel(ptInstance->DmaTxID);
		return ret;
	}

	return DRV_SUCCESS;
}

#endif

/**
* @brief	init the i2s interrupt .
*
* @param	VOID			i2s select
* @return	0-DRV_SUCCESS, other-error
*/
static VOID i2s_InitInt(VOID)
{
	static T_I2s_Device s_I2sDev[MAX_I2S_ID];

	if (s_bI2sInitInt) {
		return;
	}
	memset(&s_I2sDev, 0, sizeof(s_I2sDev));
	s_I2sDev[I2S_1].intline = I2S0_INT;
	s_I2sDev[I2S_1].intprio = 15;
	s_I2sDev[I2S_1].regs = (T_I2s_Reg*)I2S0_REG_BASE;
	s_i2sInstance[I2S_1].ptDev = &s_I2sDev[I2S_1];

	s_I2sDev[I2S_2].intline = I2S1_INT;
	s_I2sDev[I2S_2].intprio = 15;
	s_I2sDev[I2S_2].regs = (T_I2s_Reg*)I2S1_REG_BASE;
	s_i2sInstance[I2S_2].ptDev = &s_I2sDev[I2S_2];


	s_i2sInstance[I2S_1].txMutex = zOss_CreateMutex("i2s1TxMutex", 1);
	zOss_ASSERT(s_i2sInstance[I2S_1].txMutex != ZOSS_NULL);

	s_i2sInstance[I2S_1].rxMutex = zOss_CreateMutex("i2s1RxMutex", 1);
	zOss_ASSERT(s_i2sInstance[I2S_1].rxMutex != ZOSS_NULL);

	s_i2sInstance[I2S_2].txMutex = zOss_CreateMutex("i2s2TxMutex", 1);
	zOss_ASSERT(s_i2sInstance[I2S_2].txMutex != ZOSS_NULL);

	s_i2sInstance[I2S_2].rxMutex = zOss_CreateMutex("i2s2RxMutex", 1);
	zOss_ASSERT(s_i2sInstance[I2S_2].rxMutex != ZOSS_NULL);
	s_bI2sInitInt = TRUE;
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_InitInt \r\n");
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(i2s_InitInt(I2S0_REG_BASE) = %x\n", I2S0_REG_BASE);
}

/**
* @brief	set clock division coeff.
*
* @param	divFunc	 the value with the divFunc
* @param	div	 	 div value
* @return	0-DRV_SUCCESS, other-error
*/
static SINT32 i2s_SetClkDiv(UINT32 divFunc, UINT32 div)
{
	UINT32 bitWidth = divFunc & 0xff;
	UINT32 maxDiv = (0x1 << bitWidth) - 1;
	UINT32 bitIndex = (divFunc >> 8) & 0xff;
	UINT32 bitMask = 0;
	UINT32 devNum = 0;
	UINT32 regBase = 0;
	UINT32 tmp = 0;

	if (div > maxDiv)
		div = maxDiv;

	if (bitIndex == 28) { /*Ƶֻ֧żƵ*/
		div &= 0xfe;
		div++;
	}

	bitMask = ((0x1 << bitWidth) - 1) << bitIndex;
	devNum = (divFunc >> 28) & 0xf;
	regBase = g_i2sClkDivRegs[devNum];
	tmp = reg32(regBase);
	tmp &= ~bitMask;
	tmp |= div << bitIndex;
	reg32(regBase) = tmp;

	return DRV_SUCCESS;
}

/*******************************************************************************
 *                      Global function implementations                        *
 ******************************************************************************/
/**
* @brief	i2s init.
*
* @param	VOID
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Init(VOID)
{
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S__Init \n");

	s_i2sRxStream[I2S_1].rxSem = zOss_CreateSemaphore("iis1_rx_sem", 0);
	if (NULL == s_i2sRxStream[I2S_1].rxSem) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "create iis1_rx_sem err \n");
		return DRV_ERROR;
	}

	s_i2sRxStream[I2S_2].rxSem  = zOss_CreateSemaphore("iis2_rx_sem", 0);
	if (NULL == s_i2sRxStream[I2S_2].rxSem) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "create iis2_rx_sem err \n");
		return DRV_ERROR;
	}

	s_i2sTxStream[I2S_1].txSem = zOss_CreateSemaphore("iis1_tx_sem", 0);
	if (NULL == s_i2sTxStream[I2S_1].txSem) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "create iis1_tx_sem err \n");
		return DRV_ERROR;
	}

	s_i2sTxStream[I2S_2].txSem = zOss_CreateSemaphore("iis2_tx_sem", 0);
	if (NULL == s_i2sTxStream[I2S_2].txSem) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "create iis2_tx_sem err \n");
		return DRV_ERROR;
	}

	return DRV_SUCCESS;
}

/**
* @brief	i2s resem count init.
*
* @param	tId	i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 i2s_rxsem_count_Init(T_ZDrvI2S_Id tId)
{
	UINT32 i = 0;
	UINT32 semCount = 0;
	SINT32 Count = 0;

	T_I2s_RxStream *rxStream = &s_i2sRxStream[tId];
	semCount = zOss_GetSemaphoreCount(rxStream->rxSem);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s%d rxStream->rxSem count=%d \r\n", tId, semCount);

	Count = semCount;
	if (Count > 0) {
		for (i = 0; i < Count; i++) {
			zOss_GetSemaphore(rxStream->rxSem, ZOSS_NO_WAIT);
		}
	}

	return DRV_SUCCESS;
}

/**
* @brief	i2s txsem count init.
*
* @param	tId	i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 i2s_txsem_count_Init(T_ZDrvI2S_Id tId)
{
	UINT32 i = 0;
	UINT32 semCount = 0;
	SINT32 Count = 0;

	T_I2s_TxStream *txStream = &s_i2sTxStream[tId];

	semCount = zOss_GetSemaphoreCount(txStream->txSem);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s%d txStream->txSem count=%d \r\n", tId, semCount);

	Count = I2S_UPLINK_SEM_NUM - semCount;
	if (Count > 0) {
		for (i = 0; i < Count; i++) {
			zOss_PutSemaphore(txStream->txSem);
		}
	} else if (Count < 0) {
		Count = 0 - Count;
		for (i = 0; i < Count; i++) {
			zOss_GetSemaphore(txStream->txSem, ZOSS_NO_WAIT);
		}
	}

	return DRV_SUCCESS;
}

/**
* @brief	Set i2s clock coeff.
*
* @param	tId		select the i2s0 or i2s1
* @param	ptCfg		pcm division coeff
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 i2s_SetPdiv(T_ZDrvI2S_Id tId, T_ZDrvI2S_Cfg *ptCfg)
{
	SINT32 ret = DRV_SUCCESS;
	T_I2s_ClkDiv div = {0};

	if (MAX_I2S_ID <= tId || NULL == ptCfg) {
		return DRV_ERR_INVALID_PARAM;
	}

	switch (ptCfg->tClkMode) {
	case I2S_TIME_MODE:
		ret = i2s_GetI2sDiv(ptCfg, &div);
		break;
	case PCM_TIME_MODE:
		ret = i2s_GetPcmDiv(ptCfg, &div);
		break;
	default:
		return DRV_ERROR;
	}

	if (ret != DRV_SUCCESS) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, " i2s_SetPdiv err ret=0x%x", ret);
		return DRV_ERROR;
	}

	ret = i2s_SetClk(tId, &div);
	if (ret != DRV_SUCCESS) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, " i2s_SetPdiv i2s_SetClk err ret=0x%x", ret);
		return DRV_ERROR;
	}

	return DRV_SUCCESS;
}

/**
* @brief	this function is use to close i2s divide frequency.
*
* @param	tmType		pointer to T_ZDrvI2S_TimingType
* @param	pTmCfg		pointer to T_ZDrvI2S_TimingCfg
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 i2s_GetI2sTimingCfg(T_ZDrvI2S_TimingType tmType, T_ZDrvI2S_TimingCfg *pTmCfg)
{
	UINT8 i = 0;
	UINT8 arraySize = sizeof(gI2S_TimingCfg) / sizeof(gI2S_TimingCfg[0]);

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_GetI2sTimingCfg   arraySize=0x%x,tmType = 0x%x\n", arraySize, tmType);

	for (i = 0; i < arraySize; i++) {
		if (gI2S_TimingCfg[i].tTimingTp == tmType) {
			*pTmCfg = gI2S_TimingCfg[i];
			break;
		}
	}

	return DRV_SUCCESS;
}

/**
* @brief	this function is set i2s timing.
*
* @param	ptReg	   pointer to T_I2s_Reg
* @param	ptCfg	   pointer to T_ZDrvI2S_Cfg
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 i2s_SetTiming(T_I2s_Reg *ptReg, T_ZDrvI2S_Cfg *ptCfg)
{
	SINT32 ret = DRV_SUCCESS;
	T_ZDrvI2S_TimingCfg tmCfg = {0};

	i2s_GetI2sTimingCfg(ptCfg->tTimingType, &tmCfg);
	/*klocwork 3  INVARIANT_CONDITION.UNREACH  delete if*/

	if (tmCfg.tTimingSel == I2S_TIME_MODE) {
		ptCfg->tClkMode = I2S_TIME_MODE;
		ptReg->timingCtrl &= ~ I2S_TIMING_SEL;
	} else if (tmCfg.tTimingSel == PCM_TIME_MODE) {
		ptCfg->tClkMode = PCM_TIME_MODE;
		ptReg->timingCtrl |= I2S_TIMING_SEL;
	} else {
		return DRV_ERR_INVALID_PARAM;
	}

	if (tmCfg.tChnNum < 8) {
		ptReg->timingCtrl &= ~ I2S_CHN_NUM(7);
		ptReg->timingCtrl |= I2S_CHN_NUM(tmCfg.tChnNum);
	} else {
		return DRV_ERR_INVALID_PARAM;
	}

	if (tmCfg.tLaneNum < 4) {
		ptReg->timingCtrl &= ~ I2S_LANE_NUM(7);
		ptReg->timingCtrl |= I2S_LANE_NUM(tmCfg.tLaneNum);
	} else {
		return DRV_ERR_INVALID_PARAM;
	}

	if ((tmCfg.tTimingSel == PCM_TIME_MODE) && (tmCfg.tTsCfg < 8)) {
		ptCfg->tPcmSlotNum = tmCfg.tTsCfg;
		ptReg->timingCtrl &= ~ I2S_TS_CFG(7);
		ptReg->timingCtrl |=  I2S_TS_CFG(tmCfg.tTsCfg);
	} else if ((tmCfg.tTimingSel == I2S_TIME_MODE) && (tmCfg.tTsCfg < 4)) {
		ptReg->timingCtrl &= ~ I2S_TS_CFG(7);
		ptReg->timingCtrl |= I2S_TS_CFG(tmCfg.tTsCfg);
		if (tmCfg.tTsCfg == 1) {
			ptCfg->tTrackFmt = RIGHT_TRACK;
		} else if (tmCfg.tTsCfg == 2) {
			ptCfg->tTrackFmt = LEFT_TRACK;
		} else if (tmCfg.tTsCfg == 3) {
			ptCfg->tTrackFmt = DOUBLE_TRACK;
		} else {
			return DRV_ERR_INVALID_PARAM;
		}
	} else {
		return DRV_ERR_INVALID_PARAM;
	}
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "i2s_SetTiming  success");

	return DRV_SUCCESS;
}

/**
* @brief	open i2s device.
*
* @param	tId	 i2s selete struct
* @param	i2sTransMode	 T_ZDrvI2s_TransMode
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Open(T_ZDrvI2S_Id tId, T_ZDrvI2s_TransMode i2sTransMode)
{
	T_I2s_Instance *ptInstance = NULL;
	T_I2s_Device *ptDev = NULL;
	SINT32 ret = DRV_SUCCESS;
	SINT32 returnValue = 0;
#ifdef MALLOC_I2S_DMA_AREA
	UINT32 bufSize;
	UINT32 txBufSize;
	UINT32 rxBufSize;
#endif
	if (MAX_I2S_ID <= tId || i2sTransMode >= MAX_TRANS_FUNC) { /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	ptInstance = &s_i2sInstance[tId];
	if (ptInstance->i2sStatus == I2S_OPEN || ptInstance->i2sStatus == I2S_PLAY_INUSE || ptInstance->i2sStatus == I2S_RECORD_INUSE || ptInstance->i2sStatus == I2S_BOTH_INUSE) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "howard zDrvI2S_Open err i2sStatus=%d \n", ptInstance->i2sStatus);
		return DRV_ERR_BUSY;
	}

	i2s_InitInt();
	i2s_ClkInit();

	ptDev = ptInstance->ptDev;
	ptInstance->i2sTransMode = i2sTransMode;

	i2s_Reset(tId);
	/*klocwork 3  INVARIANT_CONDITION.UNREACH  delete if*/
	/*if(ret != DRV_SUCCESS) {
		return DRV_ERROR;
	}*/

	/* value of control register is 0 after reseted */
	ptInstance->tCfg.bMaster  = FALSE;
	ptInstance->tCfg.tDataFmt = MAX_I2S_DATA_FMT;
	if (ptInstance->i2sTransMode == TRANS_INT_MODE) {
		if (I2S_1 == tId) {
			returnValue = zDrvInt_InstallIsr(ptDev->intline, (VOID *)isr_i2s1, "iis1", INT_HIGHLEVEL);
		} else { /*klocwork 3  INVARIANT_CONDITION.GEN  (I2S_2==tId)*/
			returnValue = zDrvInt_InstallIsr(ptDev->intline, (VOID *)isr_i2s2, "iis2", INT_HIGHLEVEL);
		}

		if (returnValue < 0) {
			return DRV_ERROR;
		}
		ptDev->pid = (PROCESS)returnValue;
	}

#ifdef MALLOC_I2S_DMA_AREA
	if ((g_voiceVar.voiceMode == VOICE_GSM_MODE) || (g_voiceVar.voiceMode == VOICE_LTE_MODE) || \
	    (g_voiceVar.voiceMode == VOICE_WCDMA_MODE) || (g_voiceVar.voiceMode == VOICE_TD_MODE)) {
		i2sDmaState[tId][I2S_TX].totalbuffsize = I2S_MAX_VOICE_LTE_MEM_SIZE * 3;
		i2sDmaState[tId][I2S_RX].totalbuffsize = I2S_MAX_VOICE_LTE_MEM_SIZE * 3;
		txBufSize = I2S_MAX_VOICE_LTE_MEM_SIZE;
		rxBufSize = I2S_MAX_VOICE_LTE_MEM_SIZE;
		i2sDmaState[tId][I2S_TX].singlebuffsize = I2S_MAX_VOICE_LTE_MEM_SIZE;
		i2sDmaState[tId][I2S_RX].singlebuffsize = I2S_MAX_VOICE_LTE_MEM_SIZE;
	} else {
		i2sDmaState[tId][I2S_TX].totalbuffsize = I2S_MAX_AUDIO_PLAY_MEM_SIZE * 3;
		i2sDmaState[tId][I2S_RX].totalbuffsize = I2S_MAX_AUDIO_RECORD_MEM_SIZE * 3;
		txBufSize = I2S_MAX_AUDIO_PLAY_MEM_SIZE;
		rxBufSize = I2S_MAX_AUDIO_RECORD_MEM_SIZE;
		i2sDmaState[tId][I2S_TX].singlebuffsize = I2S_MAX_AUDIO_PLAY_MEM_SIZE;
		i2sDmaState[tId][I2S_RX].singlebuffsize = I2S_MAX_AUDIO_RECORD_MEM_SIZE;
	}

	i2sDmaState[tId][I2S_TX].dma_viraddr = dma_alloc_writecombine(&i2s_device.dev, i2sDmaState[tId][I2S_TX].totalbuffsize,
	                                       &(i2sDmaState[tId][I2S_TX].dma_phyaddr), GFP_KERNEL);

	i2sDmaState[tId][I2S_RX].dma_viraddr = dma_alloc_writecombine(&i2s_device.dev, i2sDmaState[tId][I2S_RX].totalbuffsize,
	                                       &(i2sDmaState[tId][I2S_RX].dma_phyaddr), GFP_KERNEL);

	if ((NULL == i2sDmaState[tId][I2S_TX].dma_viraddr) || (NULL == i2sDmaState[tId][I2S_RX].dma_viraddr)) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "[XK]BUF_NUM Malloc Failed! %p,%p,%x,%x\n", i2sDmaState[tId][I2S_TX].dma_viraddr, i2sDmaState[tId][I2S_RX].dma_viraddr, \
		                  i2sDmaState[tId][I2S_TX].dma_phyaddr, i2sDmaState[tId][I2S_RX].dma_phyaddr);
		return DRV_ERR_MEM_ALLOC;
	} else {
		/*		s_txBufArray[tId][0] = (UINT8 *)(i2sDmaState[tId][I2S_TX].dma_viraddr);
				s_txBufArray[tId][1] = s_txBufArray[tId][0] + txBufSize;
				s_txBufArray[tId][2] = s_txBufArray[tId][1] + txBufSize;
				s_rxBufArray[tId][0] = (UINT8 *)(i2sDmaState[tId][I2S_RX].dma_viraddr);
				s_rxBufArray[tId][1] = s_rxBufArray[tId][0] + rxBufSize;
				s_rxBufArray[tId][2] = s_rxBufArray[tId][1] + rxBufSize;

				i2sDmaState[tId].dma_tx_phyaddr[1] = i2sDmaState[tId].dma_tx_phyaddr[0] + txBufSize;
				i2sDmaState[tId].dma_tx_phyaddr[2] = i2sDmaState[tId].dma_tx_phyaddr[1] + txBufSize;
				i2sDmaState[tId].dma_rx_phyaddr[1] = i2sDmaState[tId].dma_rx_phyaddr[0] + rxBufSize;
				i2sDmaState[tId].dma_rx_phyaddr[2] = i2sDmaState[tId].dma_rx_phyaddr[1] + rxBufSize;
		*/
	}
#endif

	ptInstance->i2sStatus = I2S_OPEN;
#ifdef _USE_PSM
	zDrvPow_SetDevActive((UINT32) IDLE_FLAG_I2S);
#endif
	return DRV_SUCCESS;
}

/**
* @brief	reset i2s device.
*
* @param	tId	 i2s select i2s0 or i2s1
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Reset(T_ZDrvI2S_Id tId)
{
	SINT32 ret = DRV_SUCCESS;
	ret = i2s_Reset(tId);
	return ret;
}


/**
* @brief	close i2s device.
*
* @param	tId	 i2s selete struct
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Close(T_ZDrvI2S_Id tId)
{
	T_I2s_Instance *ptInstance = NULL;
	SINT32 ret = DRV_SUCCESS;

	if (MAX_I2S_ID <= tId) { /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	ptInstance = &s_i2sInstance[tId];

	if (ptInstance->i2sStatus != I2S_OPEN) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Close I2S%d, i2sStatus=%d != I2S_OPEN", tId, s_i2sInstance[tId].i2sStatus);
		return DRV_ERR_NOT_OPENED;
	}

	if (ptInstance->i2sTransMode == TRANS_INT_MODE)
		/* reset tx/rx fifo to ensure there're no data of last transfer in fifo */
		/* set the FIFO int threshold */
	{
		zDrvInt_UninstallIsr(ptInstance->ptDev->intline);
		/*klocwork 3  INVARIANT_CONDITION.UNREACH  delete if*/
	}

#ifdef MALLOC_I2S_DMA_AREA
	if (i2sDmaState[tId][I2S_TX].dma_viraddr != NULL) {
		dma_free_writecombine(&i2s_device.dev, i2sDmaState[tId][I2S_TX].totalbuffsize,
		                      i2sDmaState[tId][I2S_TX].dma_viraddr, i2sDmaState[tId][I2S_TX].dma_phyaddr);
		i2sDmaState[tId][I2S_TX].dma_viraddr = NULL;
	}
	if (i2sDmaState[tId][I2S_RX].dma_viraddr != NULL) {
		dma_free_writecombine(&i2s_device.dev, i2sDmaState[tId][I2S_RX].totalbuffsize,
		                      i2sDmaState[tId][I2S_RX].dma_viraddr, i2sDmaState[tId][I2S_RX].dma_phyaddr);
		i2sDmaState[tId][I2S_RX].dma_viraddr = NULL;
	}
#endif

#ifdef CONFIG_SND_SOC_ES8312
	if (tId == I2S_2) {
		i2s_Reset(tId);
		/*klocwork 3  INVARIANT_CONDITION.UNREACH  delete if*/
		/*if(ret != DRV_SUCCESS) {
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Close i2s_Reset failed!!!\n");
		}*/
	}
#else
	i2s_Reset(tId);
	/*klocwork 3  INVARIANT_CONDITION.UNREACH  delete if*/
	/*if(ret != DRV_SUCCESS) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Close i2s_Reset failed!!!\n");
	}*/

#endif
#ifdef _USE_AUDIO_ON_7520V2
	i2s_CloseClk(tId);
#endif

	i2s_ClkDeinit(tId);

	ptInstance->i2sStatus = I2S_IDLE;

#ifdef _USE_PSM
	if ((s_i2sInstance[I2S_1].i2sStatus == I2S_IDLE) && (s_i2sInstance[I2S_2].i2sStatus == I2S_IDLE)) {
		zDrvPow_SetDevIdle((UINT32)IDLE_FLAG_I2S);
	}
#endif

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Close I2S%d, i2sStatus=%d", tId, s_i2sInstance[tId].i2sStatus);

	return ret;
}

#if 0
/**
* @brief	start to trans data,use in playing.
*
* @param	tId	 i2s selete struct
* @param	params	 T_ZDrvI2s_Params
* @param	ptCfg	 config param
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Vousb_Start(T_ZDrvI2S_Id tId, T_ZDrvI2s_Params *params, T_ZDrvI2S_Cfg *ptCfg)
{
	T_I2s_Instance *ptInstance = &s_i2sInstance[tId];
	T_I2s_TxStream *i2sTxream = &s_i2sTxStream[tId];
	UINT32 ret = DRV_SUCCESS;
	T_I2s_Reg *ptReg = NULL;
	//SINT32 waitTime = 0;//xiu

	zDrv_ASSERT((tId < MAX_I2S_ID) && (ptCfg != NULL) && (params != NULL));
	if (MAX_I2S_ID <= tId || NULL == ptCfg || NULL == params) {
		return DRV_ERR_INVALID_PARAM;
	}

	if (ptInstance->i2sStatus != I2S_OPEN) {
		return DRV_ERR_NOT_OPENED;
	}

	ptReg = ptInstance->ptDev->regs;
	ret = i2s_TxComInit(tId, params);
	ret += i2s_RxComInit(tId, params);

	if (ptInstance->i2sTransMode == TRANS_INT_MODE) {
		ret += i2s_IntTxInit(tId);
		ret += i2s_IntRxInit(tId);
		if (ret != DRV_SUCCESS) {
			i2s_CleanTxResource(tId);
			i2s_CleanRxResource(tId);
			return DRV_ERROR;
		}
	} else if (ptInstance->i2sTransMode == TRANS_DMA_MODE) {
		//	i2sTxream->txTransmit = i2s_DmaTx;
		ret += i2s_DmaTxInit(tId);
		ret += i2s_DmaRxInit(tId);
		if (ret != DRV_SUCCESS) {
			i2s_CleanTxResource(tId);
			return DRV_ERROR;
		}
#ifdef _USE_AUDIO_ON_7520V2
		ptReg->fifoCtrl = I2S_TX_FIFO_RST | I2S_RX_FIFO_RST;
#endif
		ptReg->fifoCtrl |= I2S_RFTH(30) | I2S_TFTH(30);
		ptReg->fifoCtrl |= (I2S_TX_DMA_EN | I2S_RX_DMA_EN0);             /* enable dma transfer */
	}

	ptReg->processCtrl |= (I2S_EN | I2S_TX_OPEN | I2S_RX_OPEN);                /* enable i2s; enable tx only */
	ptInstance->i2sStatus = ((ptInstance->i2sStatus) == I2S_RECORD_INUSE) ?  I2S_BOTH_INUSE : I2S_PLAY_INUSE;

	return DRV_SUCCESS;
}
#endif

/**
* @brief	i2s write prepare.
*
* @param	tId	 i2s selete struct
* @param	ptCfg	 config param
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Write_Prepare(T_ZDrvI2S_Id tId, T_ZDrvI2S_Cfg *ptCfg)
{
	T_I2s_Instance *ptInstance = NULL;
	T_I2s_Reg *ptReg = NULL;
	SINT32 ret = DRV_SUCCESS;

	if (MAX_I2S_ID <= tId || NULL == ptCfg) { /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}
	ptInstance = &s_i2sInstance[tId];

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Write_Prepare! \r\n");
	ptReg = ptInstance->ptDev->regs;
	ret = i2s_TxCheckState(tId, ptCfg);
	if (ret != DRV_SUCCESS) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Write_Prepare:  i2s_TxCheckState err ret=%d\r\n", ret);
		return DRV_ERROR;
	}

	ptReg->processCtrl |= I2S_EN;

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvI2S_Write_Prepare(i2s(%d) timing ctrl) = %x", tId, (s_i2sInstance[tId].ptDev)->regs->timingCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvI2S_Write_Prepare(i2s(%d) fifo ctrl) = %x", tId, (s_i2sInstance[tId].ptDev)->regs->fifoCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvI2S_Write_Prepare(i2s(%d) frame ctrl) = %x", tId, (s_i2sInstance[tId].ptDev)->regs->frameCntr);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvI2S_Write_Prepare(i2s(%d) fifostatus) = %x", tId, (s_i2sInstance[tId].ptDev)->regs->fifoStatus);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvI2S_Write_Prepare(i2s(%d) process ctrl) = %x", tId, (s_i2sInstance[tId].ptDev)->regs->processCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Write_Prepare(i2s(%d) reg intEn) = %x", tId, (s_i2sInstance[tId].ptDev)->regs->intEn);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Write_Prepare(i2s(%d) reg intStatus) = %x\n", tId, (s_i2sInstance[tId].ptDev)->regs->intStatus);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Write_Prepare(i2s(%d) reg data) = %x\n", tId, (s_i2sInstance[tId].ptDev)->regs->data);

	return DRV_SUCCESS;
}

/**
* @brief	i2s write start Do.
*
* @param	tId	 i2s selete struct
* @param	params	 T_ZDrvI2s_Params
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Write_Start_Do(T_ZDrvI2S_Id tId, T_ZDrvI2s_Params *params)
{

	T_I2s_Instance *ptInstance = NULL;
	T_I2s_TxStream *txStream = NULL;
	T_I2s_Reg *ptReg = NULL;
	SINT32 ret = DRV_SUCCESS;
	SINT32 waitTime = 0;

	if (MAX_I2S_ID <= tId || NULL == params) { /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	ptInstance = &s_i2sInstance[tId];
	txStream = &s_i2sTxStream[tId];

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Write_Start_Do \r\n");
	ptReg = ptInstance->ptDev->regs;
	txStream->channel = params->channel;

	ret += i2s_TxComInit(tId, params);

	if (ptInstance->i2sTransMode == TRANS_INT_MODE) {
		txStream->txTransmit = i2s_IntTx;              /*init the transter function ,it will be called in Write() function*/
		ret += i2s_IntTxInit(tId);
		if (ret != DRV_SUCCESS) {
			i2s_CleanTxResource(tId);
			return DRV_ERROR;
		}
	} else if (ptInstance->i2sTransMode == TRANS_DMA_MODE) {
		//txStream->txTransmit = i2s_DmaTx;
		ret += i2s_DmaTxInit(tId);
		if (ret != DRV_SUCCESS) {
			i2s_CleanTxResource(tId);
			return DRV_ERROR;
		}
		ptReg->fifoCtrl |= I2S_TX_DMA_EN;            /* enable dma transfer */
	}

	ptReg->processCtrl |= I2S_TX_OPEN;

	if (ptInstance->tCfg.bMaster == TRUE) {                /* wait */
		{
			bsp_udelay(I2S_ENABLE_DELAY_TIME);
			waitTime += I2S_ENABLE_DELAY_TIME;
			/*klocwork 3  INVARIANT_CONDITION.UNREACH  delete if*/
		}
	}

	ptInstance->i2sStatus = ((ptInstance->i2sStatus) == I2S_RECORD_INUSE) ? I2S_BOTH_INUSE : I2S_PLAY_INUSE;

	gI2sDmaWriteIntTime[tId] = 0;
	gI2sCntChangeTx[tId] = 0;

	return DRV_SUCCESS;
}

/**
* @brief	i2s write start .
*
* @param	tId	 i2s selete struct
* @param	params	 pointer to struct T_ZDrvI2s_Params
* @param	ptCfg	 pointer to struct T_ZDrvI2S_Cfg
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Write_Start(T_ZDrvI2S_Id tId, T_ZDrvI2s_Params *params, T_ZDrvI2S_Cfg *ptCfg)
{
	SINT32 ret = DRV_SUCCESS;
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Write_Start\r\n");

	ret = zDrvI2S_Write_Prepare(tId, ptCfg);
	if (ret != DRV_SUCCESS) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Write_Start  zDrvI2S_Write_Prepare error ret=%d \r\n", ret);
		return ret;
	}

	ret = zDrvI2S_Write_Start_Do(tId, params);
	if (ret != DRV_SUCCESS) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Write_Start  zDrvI2S_Write_Start_Do error ret=%d \r\n", ret);
		return ret;
	}

	return DRV_SUCCESS;
}

/**
* @brief	i2s read prepare.
*
* @param	tId	 i2s selete struct
* @param	ptCfg	 pointer to config param
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Read_Prepare(T_ZDrvI2S_Id tId, T_ZDrvI2S_Cfg *ptCfg)
{
	SINT32 ret = DRV_SUCCESS;
	T_I2s_Reg *ptReg = NULL;
	T_I2s_Instance *ptInstance = NULL;

	if (MAX_I2S_ID <= tId || NULL == ptCfg) { /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	ptInstance = &s_i2sInstance[tId];
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Read_Prepare \r\n");
	ptReg = ptInstance->ptDev->regs;

	ret = i2s_RxCheckState(tId, ptCfg);
	if (ret != DRV_SUCCESS) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Read_Prepare:  i2s_RxCheckState err ret=%d\r\n", ret);
		return DRV_ERROR;
	}

	ptReg->processCtrl |= I2S_EN;

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvI2S_Read_Prepare(i2s(%d) timing ctrl) = 0x%8x", tId, (s_i2sInstance[tId].ptDev)->regs->timingCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvI2S_Read_Prepare(i2s(%d) fifo ctrl) = 0x%8x", tId, (s_i2sInstance[tId].ptDev)->regs->fifoCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvI2S_Read_Prepare(i2s(%d) frame ctrl) = 0x%8x", tId, (s_i2sInstance[tId].ptDev)->regs->frameCntr);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvI2S_Read_Prepare(i2s(%d) fifostatus) = 0x%8x", tId, (s_i2sInstance[tId].ptDev)->regs->fifoStatus);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "(zDrvI2S_Read_Prepare(i2s(%d) process ctrl) = 0x%8x", tId, (s_i2sInstance[tId].ptDev)->regs->processCtrl);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Read_Prepare(i2s(%d) reg intEn) = 0x%8x", tId, (s_i2sInstance[tId].ptDev)->regs->intEn);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Read_Prepare(i2s(%d) reg intStatus) = 0x%8x\n", tId, (s_i2sInstance[tId].ptDev)->regs->intStatus);
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Read_Prepare(i2s(%d) reg data) = 0x%8x\n", tId, (s_i2sInstance[tId].ptDev)->regs->data);

	return DRV_SUCCESS;
}

/**
* @brief	i2s read start do.
*
* @param	tId	 i2s selete struct
* @param	params	 pointer to T_ZDrvI2s_Params
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Read_Start_Do(T_ZDrvI2S_Id tId, T_ZDrvI2s_Params *params)
{
	SINT32 ret = DRV_SUCCESS;
	T_I2s_Reg *ptReg = NULL;
	SINT32 waitTime = 0;
	T_I2s_Instance *ptInstance = NULL;

	if (MAX_I2S_ID <= tId || NULL == params) { /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	ptInstance = &s_i2sInstance[tId];
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Read_Start_Do \r\n");
	ptReg = ptInstance->ptDev->regs;

	ret += i2s_RxComInit(tId, params);

	if (ptInstance->i2sTransMode == TRANS_INT_MODE) {
		ret += i2s_IntRxInit(tId);                  /*will enable the RX interrupt*/
		if (ret != DRV_SUCCESS) {
			i2s_CleanRxResource(tId);
			return DRV_ERROR;
		}
	} else if (ptInstance->i2sTransMode == TRANS_DMA_MODE) {
		ret += i2s_DmaRxInit(tId);
		if (ret != DRV_SUCCESS) {
			i2s_CleanRxResource(tId);
			return DRV_ERROR;
		}
		ptReg->fifoCtrl |= I2S_RX_DMA_EN0;                          /* enable dma transfer */
	}

	ptReg->processCtrl |= I2S_RX_OPEN;
	if (ptInstance->tCfg.bMaster == TRUE) {                       /* wait */
		bsp_udelay(I2S_ENABLE_DELAY_TIME);
		waitTime += I2S_ENABLE_DELAY_TIME;
		/*klocwork 3  INVARIANT_CONDITION.UNREACH  delete if*/
	}

	ptInstance->i2sStatus = ((ptInstance->i2sStatus) == I2S_PLAY_INUSE) ? I2S_BOTH_INUSE : I2S_RECORD_INUSE;

	gI2sTimes = 0;
	gI2sDmaReadIntTime[tId] = 0;
	gI2sCntChangeRx[tId] = 0;

	return DRV_SUCCESS;
}

/**
* @brief	i2s read start.
*
* @param	tId	 i2s selete struct
* @param	params	 pointer to T_ZDrvI2s_Params
* @param	ptCfg	 pointer to T_ZDrvI2S_Cfg
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Read_Start(T_ZDrvI2S_Id tId, T_ZDrvI2s_Params *params, T_ZDrvI2S_Cfg *ptCfg)
{
	SINT32 ret = DRV_SUCCESS;
	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Read_Start \r\n");

	ret = zDrvI2S_Read_Prepare(tId, ptCfg);
	if (ret != DRV_SUCCESS) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Read_Start  zDrvI2S_Read_Prepare error ret=%d \r\n", ret);
		return ret;
	}

	ret = zDrvI2S_Read_Start_Do(tId, params);
	if (ret != DRV_SUCCESS) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Read_Start  zDrvI2S_Read_Start_Do error ret=%d \r\n", ret);
		return ret;
	}

	return DRV_SUCCESS;
}

/**
* @brief	i2s trans data to fifo from ram while playing.
*
* @param	tId	 i2s selete struct
* @param	pBuf	 one buffer is how many bytes
* @param	uiLen	 buffer length
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Write(T_ZDrvI2S_Id tId, const UINT8 *pBuf, UINT32 uiLen)
{
	T_I2s_Instance *ptInstance = NULL;
	SINT32 ret = DRV_SUCCESS;
	UINT16 bufLen = 0 ;
	T_I2s_QueueNode *queueNode = NULL;
	UINT16 *buf = NULL;
	T_I2s_TxStream *i2sStream = NULL;

	if (MAX_I2S_ID <= tId || NULL == pBuf || 0 == uiLen || I2S_MAX_TX_BUF_SIZE < uiLen) {
		i2swrite_tid = tId;
		i2swrite_pbuf = (UINT32)pBuf;//xiu
		i2swrite_uilen = uiLen;
	}

	i2sStream = &s_i2sTxStream[tId];
	/*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
	if (MAX_I2S_ID <= tId || NULL == pBuf || 0 == uiLen || I2S_MAX_TX_BUF_SIZE < uiLen) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Write invalid para tId=%d,pBuf=0x%p,uiLen=%d \n", tId, pBuf, uiLen);
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	ptInstance = &s_i2sInstance[tId];
	if ((ptInstance->i2sStatus != I2S_PLAY_INUSE) && (ptInstance->i2sStatus != I2S_BOTH_INUSE)) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Write i2sStatus not support =%d \n", ptInstance->i2sStatus);
		return DRV_ERR_NOT_STARTED;
	}

	zOss_GetMutex(ptInstance->txMutex, ZOSS_WAIT_FOREVER);
	ret = i2s_UpdateTxBufferStatus(tId, (UINT16 *)pBuf, BUFFER_FULL);      /*ϲ*pBufдݺ󣬸ıBUF״̬ΪFULL*/
	if (ret != DRV_SUCCESS) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Write i2s_UpdateTxBufferStatus err ret=%d \n", ret);
		ret =  DRV_ERROR;
		goto exit;
	}
	i2sStream->txWrIdx = (i2sStream->txWrIdx + 1) % I2S_OUT_BUF_NUM;

	if ((s_i2sTxStream[tId].pauseFlag == FALSE) && (s_i2sTxStream[tId].Transing == FALSE)) {
		i2sStream->Transing = TRUE;
		dma_async_issue_pending(i2sDmaState[tId][I2S_TX].ch);
		/*klocwork 3  INVARIANT_CONDITION.UNREACH  delete if */
	}

exit:
	zOss_PutMutex(ptInstance->txMutex);
	return ret;
}

/**
* @brief	This function is used to read data by i2s.
*
* @param	tId	 i2s select i2s0 or i2s1
* @param	pBuf	 one buffer is how many bytes
* @param	uiLen	 buffer length
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Read(T_ZDrvI2S_Id tId, UINT8 **pBuf, UINT32 *uiLen)
{
	T_I2s_Instance *ptInstance = NULL;
	SINT32 ret = DRV_SUCCESS;
	T_I2s_QueueNode *queueNode = NULL;
	T_I2s_RxStream *i2sStream = NULL;

	/*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
	if (MAX_I2S_ID <= tId || NULL == pBuf || NULL == uiLen) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Read invalid para tId=%d,pBuf=0x%p,uiLen=%p \n", tId, pBuf, uiLen);
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}
	i2sStream = &s_i2sRxStream[tId];
	ptInstance = &s_i2sInstance[tId];
	if ((ptInstance->i2sStatus != I2S_RECORD_INUSE) && (ptInstance->i2sStatus != I2S_BOTH_INUSE)) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Read i2sStatus not support =%d \n", ptInstance->i2sStatus);
		return DRV_ERR_NOT_STARTED;
	}

	if (ZOSS_SUCCESS != zOss_GetSemaphore(s_i2sRxStream[tId].rxSem, ZOSS_WAIT_FOREVER)) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Read zOss_GetSemaphore error");
		return DRV_ERROR;
	}

	zOss_GetMutex(ptInstance->rxMutex, ZOSS_WAIT_FOREVER);

	*pBuf = i2sStream->rxBufferArray[i2sStream->rxRdIdx].buf;
	*uiLen = i2sStream->rxLen;
	i2s_UpdateRxBufferStatus(tId, (UINT16 *)i2sStream->rxBufferArray[i2sStream->rxRdIdx].buf, BUFFER_READ);

#if 0
	ret = i2s_QueueGet(&(s_i2sRxStream[tId].rxQueue), &queueNode);
	if (ret != DRV_SUCCESS) {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Read i2s_QueueGet error");
		ret = DRV_ERROR;
		goto exit;
	}

	if (NULL != queueNode) {
		*pBuf = (UINT8 *)queueNode->pBuf;
		*uiLen = queueNode->len * sizeof(UINT16);
		ret = i2s_UpdateRxBufferStatus(tId, queueNode->pBuf, BUFFER_READ);
		if (ret != DRV_SUCCESS) {
			zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Read i2s_UpdateRxBufferStatus error");
			ret = DRV_ERROR;
			goto exit;
		}
	} else {
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_Read queueNode==NULL error");
		ret = DRV_ERROR;
		goto exit;
	}
#endif
exit:
	zOss_PutMutex(ptInstance->rxMutex);
	return ret;

}

/**
* @brief 	i2s write stop ,use to stop playing or recording.
*
* @param	tId	 i2s select i2s0 or i2s1
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Write_Stop(T_ZDrvI2S_Id tId)
{
	T_I2s_Instance *ptInstance = NULL;
	T_I2s_Reg *ptReg = NULL;
	SINT32 ret = DRV_SUCCESS;
	UINT32 regVal = 0, count = 9;
	T_I2s_TxStream *txStream = NULL;

	if (MAX_I2S_ID <= tId) { /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	txStream = &s_i2sTxStream[tId];
	ptInstance = &s_i2sInstance[tId];
	ptReg = ptInstance->ptDev->regs;

	if ((ptInstance->i2sStatus != I2S_PLAY_INUSE) && (ptInstance->i2sStatus != I2S_BOTH_INUSE)) {
		return DRV_ERR_NOT_STARTED;
	}

	if (g_voiceVar.isUseSlicCodec == 1) {
		while (count--) {
			regVal = ptReg->fifoStatus;
			if (((regVal >> 8) & 0x7f) == 0x20)
				break;
			zOss_Sleep(1);
		}
	}

	ptReg->processCtrl &= ~I2S_TX_OPEN;                /*disable tx only */
	if (ptInstance->i2sTransMode == TRANS_DMA_MODE) {
		ptReg->fifoCtrl &= ~I2S_TX_DMA_EN;
		dma_release_channel(i2sDmaState[tId][I2S_TX].ch);
	}
#ifdef _USE_AUDIO_ON_7520V2
	/* reset tx fifo */
	ptReg->fifoCtrl |= I2S_TX_FIFO_RST;
#endif
	ptReg->intStatus |= I2S_TX_TRIG_INT;
	ptReg->intEn &= ~I2S_TX_TRIG_INT_EN;       /* disable tx interrupt */

	zOss_GetMutex(ptInstance->txMutex, ZOSS_WAIT_FOREVER);

	i2s_CleanTxResource(tId);
	/*klocwork 3  INVARIANT_CONDITION.UNREACH  delete if*/
	/*if(ret != DRV_SUCCESS) {
		ret = DRV_ERROR;
	}*/
	zOss_PutMutex(ptInstance->txMutex);

	txStream->txTransmit = NULL;
	txStream->channel = AUDIO_MONO_CHANNEL;
	txStream->txLen = 0;
	txStream->p_cb = NULL;

	if (ptInstance->i2sStatus != I2S_BOTH_INUSE) {
		ptInstance->i2sStatus = I2S_OPEN;
	} else {
		ptInstance->i2sStatus = I2S_RECORD_INUSE;
	}

	return ret;
}

/**
* @brief i2s read stop ,use to stop playing or recording.
*
* @param	tId	 i2s select i2s0 or i2s1
*
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Read_Stop(T_ZDrvI2S_Id tId)
{
	T_I2s_Instance *ptInstance = NULL;
	T_I2s_Reg *ptReg = NULL;
	SINT32 ret = DRV_SUCCESS;

	if (MAX_I2S_ID <= tId) { /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}
	ptInstance = &s_i2sInstance[tId];
	ptReg = ptInstance->ptDev->regs;
	if ((ptInstance->i2sStatus != I2S_RECORD_INUSE) && (ptInstance->i2sStatus != I2S_BOTH_INUSE)) {
		return DRV_ERR_NOT_STARTED;
	}

	ptReg->processCtrl &= ~I2S_RX_OPEN;                /*disable tx only */

	if (ptInstance->i2sTransMode == TRANS_DMA_MODE) {
		ptReg->fifoCtrl &= ~I2S_RX_DMA_EN0;
		dma_release_channel(i2sDmaState[tId][I2S_RX].ch);
	}
#ifdef _USE_AUDIO_ON_7520V2
	ptReg->fifoCtrl |= I2S_RX_FIFO_RST;    /* reset tx/rx fifo */
#endif
	ptReg->intStatus |= I2S_RX_TRIG_INT;
	ptReg->intEn &= ~I2S_RX_TRIG_INT_EN;       /* disable tx interrupt */

	zOss_GetMutex(ptInstance->rxMutex, ZOSS_WAIT_FOREVER);

	ret = i2s_CleanRxResource(tId);
	/*klocwork 3  INVARIANT_CONDITION.UNREACH  delete if*/
	/*
	if(ret != DRV_SUCCESS) {
		ret = DRV_ERROR;
	}*/
	zOss_PutMutex(ptInstance->rxMutex);

	if (ptInstance->i2sStatus != I2S_BOTH_INUSE) {
		ptInstance->i2sStatus = I2S_OPEN;
	} else {
		ptInstance->i2sStatus = I2S_PLAY_INUSE;
	}

	return ret;
}

/**
* @brief  	i2s get buffer ,get buffer to write playing data.
*
* @param	tId	 i2s select i2s0 or i2s1
* @param	pBuf	 playing data buffer
* @param	uiLen	 buffer length
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_GetBuf(T_ZDrvI2S_Id tId, UINT8 **pBuf, UINT32 *uiLen)
{
	UINT8 txIndex = 0;
	T_I2s_Instance *ptInstance = NULL;
	T_I2s_TxStream *i2sStream = NULL;

	SINT32 ret = DRV_SUCCESS;

	/*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
	if ((NULL == pBuf) || (NULL == uiLen) || (tId >= MAX_I2S_ID)) {
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	i2sStream = &s_i2sTxStream[tId];
	ptInstance = &s_i2sInstance[tId];

	if ((ptInstance->i2sStatus != I2S_PLAY_INUSE) && (ptInstance->i2sStatus != I2S_BOTH_INUSE)) {
		return DRV_ERR_NOT_STARTED;
	}

	if (ZOSS_SUCCESS != zOss_GetSemaphore(i2sStream->txSem, I2S_SEM_WAIT_TIME)) {  /*changed by dangpeixia[2010.9.20]*/
		zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "zDrvI2S_GetBuf zOss_GetSemaphore err!\n");
		return DRV_ERROR;
	}

	zOss_GetMutex(ptInstance->txMutex, ZOSS_WAIT_FOREVER);
	if (i2sStream->txBufferArray == NULL) {
		ret = DRV_ERROR;/*  PSSTOPLTEдݷʿָ */
		goto exit;
	}

	txIndex = i2sStream->txWrIdx;

	*pBuf = i2sStream->txBufferArray[txIndex].buf;
	*uiLen = i2sStream->txLen;
	i2sStream->txBufferArray[txIndex].i2sBufStatus = BUFFER_WRITE;

exit:
	zOss_PutMutex(ptInstance->txMutex);
	return ret;
}

#if 0
/**
* @brief 	i2s both stop ,gif copying record data end, free buffer to continue record.
*
* @param	tId	 i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_Both_Stop(T_ZDrvI2S_Id tId)
{
	T_I2s_Instance *ptInstance = NULL;
	T_I2s_Reg *ptReg = NULL;
	SINT32 ret = DRV_SUCCESS;

	zDrv_ASSERT(tId < MAX_I2S_ID);
	if (MAX_I2S_ID <= tId) {
		return DRV_ERR_INVALID_PARAM;
	}

	ptInstance = &s_i2sInstance[tId];
	ptReg = ptInstance->ptDev->regs;
	if ((ptInstance->i2sStatus != I2S_RECORD_INUSE) && (ptInstance->i2sStatus != I2S_BOTH_INUSE) && (ptInstance->i2sStatus != I2S_PLAY_INUSE)) {
		return DRV_ERR_NOT_STARTED;
	}
	ptReg->processCtrl &= ~(I2S_TX_OPEN | I2S_RX_OPEN);                 /*disable I2S ,Tx ,RX */
	if (ptInstance->i2sTransMode == TRANS_DMA_MODE) {
		ptReg->fifoCtrl &= ~(I2S_RX_DMA_EN0 | I2S_TX_DMA_EN);
	}
#ifdef _USE_AUDIO_ON_7520V2
	ptReg->fifoCtrl |= I2S_RX_FIFO_RST | I2S_TX_FIFO_RST;  /* reset tx/rx fifo */
#endif
	ptReg->intStatus |= I2S_RX_TRIG_INT | I2S_TX_TRIG_INT;
	ptReg->intEn &= ~(I2S_RX_TRIG_INT_EN | I2S_TX_TRIG_INT_EN);      /* disable tx interrupt */
	ptReg->processCtrl &= ~I2S_EN;                /*disable I2S ,Tx ,RX */

	ret = i2s_CleanRxResource(tId);
	ret += i2s_CleanTxResource(tId);

	ptInstance->i2sStatus = I2S_OPEN;

	return ret;
}
#endif

/**
* @brief 	i2s free buffer ,get buffer to write playing data.
*
* @param	tId  i2s select i2s0 or i2s1
* @param	pBuf	 data buffer
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_FreeBuf(T_ZDrvI2S_Id tId, UINT8 *pBuf)
{
	SINT32 i = 0;
	T_I2s_Instance *ptInstance = NULL;
	T_I2s_RxStream *i2sStream = NULL;
	SINT32 ret = DRV_SUCCESS;

	if ((UINT8 *)NULL == pBuf || tId >= MAX_I2S_ID) { //xiu
		freebufferadd = (UINT32)pBuf;//xiu
		freebuffertid = tId;
	}

	if (NULL == pBuf || tId >= MAX_I2S_ID) { /*klocwork 3  INVARIANT_CONDITION.UNREACH  assertƵif*/
		zDrv_ASSERT(0);
		return DRV_ERR_INVALID_PARAM;
	}

	i2sStream = &s_i2sRxStream[tId];
	ptInstance = &s_i2sInstance[tId];

	if (ptInstance->i2sStatus != I2S_RECORD_INUSE && ptInstance->i2sStatus != I2S_BOTH_INUSE) {
		return DRV_ERR_NOT_STARTED;
	}


	zOss_GetMutex(ptInstance->rxMutex, ZOSS_WAIT_FOREVER);
	i2s_UpdateRxBufferStatus(tId, (UINT16 *)i2sStream->rxBufferArray[i2sStream->rxRdIdx].buf, BUFFER_NULL);
	i2sStream->rxRdIdx = (i2sStream->rxRdIdx + 1) % I2S_OUT_BUF_NUM ;

#if 0
	if (i2sStream->rxBufferArray == NULL) {
		ret = DRV_ERROR;
		goto exit;
	}
	while (i < I2S_OUT_BUF_NUM) {
		if (i2sStream->rxBufferArray[i].buf == pBuf) {
			i2sStream->rxBufferArray[i].i2sBufStatus = BUFFER_NULL;
			break;
		}
		i++;
	}

	if (i == I2S_OUT_BUF_NUM) {
		ret = DRV_ERROR;
		goto exit;
	}
#endif
exit:
	zOss_PutMutex(ptInstance->rxMutex);

	return ret;
}

#if 0
/**
* @brief	i2s free buffer ,get buffer to write playing data.
*
* @param	tId  i2s select i2s0 or i2s1
* @param	pBuf	 data buffer
* @return	0-DRV_SUCCESS, other-error
*/
SINT32 zDrvI2S_GetRemained(T_ZDrvI2S_Id tId, UINT32 *len)
{
	T_I2s_Queue *i2sQueue = &(s_i2sTxStream[tId].txQueue);
	UINT8 index = 0;
	UINT32 uiLen = 0;
	T_I2s_Instance *ptInstance = NULL;

	zDrv_ASSERT((len != NULL) && (tId < MAX_I2S_ID));
	if ((NULL == len) || (tId >= MAX_I2S_ID)) {
		return DRV_ERR_INVALID_PARAM;
	}
	ptInstance = &s_i2sInstance[tId];
	if ((ptInstance->i2sStatus != I2S_PLAY_INUSE) && (ptInstance->i2sStatus != I2S_BOTH_INUSE)) {
		return DRV_ERR_NOT_STARTED;
	}

	index = i2sQueue->readIndex;
	while (index != i2sQueue->writeIndex) {
		uiLen += i2sQueue->data[index].len;
		index = (index + 1) % I2S_QUEUE_SIZE;
	}

	*len = uiLen * sizeof(UINT16);

	return DRV_SUCCESS;
}

/**
* @brief	pause while playing.
*
* @param	tId	 i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
VOID zDrvI2S_Pause(T_ZDrvI2S_Id tId)
{
	T_I2s_Instance *ptInstance = NULL;
	zDrv_ASSERT(tId < MAX_I2S_ID);
	if (tId >= MAX_I2S_ID) {
		return;
	}

	ptInstance = &s_i2sInstance[tId];
	if ((ptInstance->i2sStatus != I2S_PLAY_INUSE) && (ptInstance->i2sStatus != I2S_BOTH_INUSE)) {
		return;
	}

	s_i2sTxStream[tId].pauseFlag = TRUE;
}

/**
* @brief	resume playing if pause.
*
* @param	tId	 i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
VOID zDrvI2S_Resume(T_ZDrvI2S_Id tId)
{
	T_I2s_Instance *ptInstance = NULL;
	T_I2s_TxStream *i2sStream = &s_i2sTxStream[tId];
	T_I2s_QueueNode *queueNode = NULL;
	SINT32 ret = DRV_SUCCESS;

	zDrv_ASSERT(tId < MAX_I2S_ID);
	if (tId >= MAX_I2S_ID) {
		return;
	}

	ptInstance = &s_i2sInstance[tId];
	if ((ptInstance->i2sStatus != I2S_PLAY_INUSE) && (ptInstance->i2sStatus != I2S_BOTH_INUSE)) {
		return;
	}

	ret = i2s_QueueGet(&(i2sStream->txQueue), &queueNode);
	if (ret != DRV_SUCCESS) {
		return;
	}

	if (queueNode != NULL) {
		i2sStream->Transing = TRUE;
		i2s_UpdateTxBufferStatus(tId, queueNode->pBuf, BUFFER_READ);
		ret = i2sStream->txTransmit(tId, queueNode->pBuf, queueNode->len);
	}

	if (ret != DRV_SUCCESS) {
		return;
	}

	i2sStream->pauseFlag = FALSE;

	return;
}
#endif

/**
* @brief	Release the rx semaphore before stop.
*
* @param	tId			i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
VOID zDrvI2s_RxRlsSemaBeforeStop(T_ZDrvI2S_Id tId)
{
	T_I2s_RxStream *i2sRxStream = &s_i2sRxStream[tId];
	zOss_PutSemaphore(i2sRxStream->rxSem);
}

/**
* @brief	Release the tx semaphore before stop.
*
* @param	tId			i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
VOID zDrvI2s_TxRlsSemaBeforeStop(T_ZDrvI2S_Id tId)
{
	T_I2s_TxStream *i2sTxStream = &s_i2sTxStream[tId];
	zOss_PutSemaphore(i2sTxStream->txSem);
}

/**
* @brief	set top i2s config.
*
* @param	VOID
* @return	0-DRV_SUCCESS, other-error
*/
VOID vp_SetTopI2sConfig(VOID)
{
	

UINT32 AmrRegBit = 0;

if (g_voiceVar.useVoiceBuffer == 1){ 



	/* inter loop */
	AmrRegBit = ARM_I2S_LOOP_CFG;

	AmrRegBit &= 0xfffffe07;
	AmrRegBit |= 0x000000a8; /* 0x000000a8 loop dsp afe(loop i2s1)--arm i2s2(loop i2s2) 0x00000150 loop dsp arm(loop i2s1)--afe i2s2(loop i2s2)*/

	ARM_I2S_LOOP_CFG = AmrRegBit;

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "vp_SetTopI2sConfig ARM_I2S_LOOP_CFG=0x%x\n", ARM_I2S_LOOP_CFG);
}
#ifdef CONFIG_VOICE_BUFFER_DRV  
	/* inter loop */
	AmrRegBit = ARM_I2S_LOOP_CFG;

	AmrRegBit &= 0xfffffe07;
	AmrRegBit |= 0x000000a8; /* 0x000000a8 loop dsp afe(loop i2s1)--arm i2s2(loop i2s2) 0x00000150 loop dsp arm(loop i2s1)--afe i2s2(loop i2s2)*/

	ARM_I2S_LOOP_CFG = AmrRegBit;

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "vp_SetTopI2sConfig ARM_I2S_LOOP_CFG=0x%x\n", ARM_I2S_LOOP_CFG);

#endif

#if 0
	UINT32 AmrRegBit = 0;
	int ret;
#if defined _USE_AUDIO_ON_7520V2
#ifdef TOP_I2S0_PIN_CONNECT
	/*i2s0 pin cfg*/
	audioGpioWsHandle = zDrvGpio_Request("i2s0_ws");
	zDrvGpio_SetFunc(audioGpioWsHandle, GPIO38_I2S0_WS);
	audioGpioClkHandle = zDrvGpio_Request("i2s0_clk");
	zDrvGpio_SetFunc(audioGpioClkHandle, GPIO39_I2S0_CLK);
	audioGpioDinHandle = zDrvGpio_Request("i2s0_din");
	zDrvGpio_SetFunc(audioGpioDinHandle, GPIO40_I2S0_DIN);
	audioGpioDoutHandle = zDrvGpio_Request("i2s0_dout");
	zDrvGpio_SetFunc(audioGpioDoutHandle, GPIO41_I2S0_DOUT);
#else
	/*i2s1 pin cfg*/
	audioGpioWsHandle = zDrvGpio_Request("i2s1_ws");
	zDrvGpio_SetFunc(audioGpioWsHandle, GPIO42_I2S1_WS);
	audioGpioClkHandle = zDrvGpio_Request("i2s1_clk");
	zDrvGpio_SetFunc(audioGpioClkHandle, GPIO43_I2S1_CLK);
	audioGpioDinHandle = zDrvGpio_Request("i2s1_din");
	zDrvGpio_SetFunc(audioGpioDinHandle, GPIO44_I2S1_DIN);
	audioGpioDoutHandle = zDrvGpio_Request("i2s1_dout");
	zDrvGpio_SetFunc(audioGpioDoutHandle, GPIO45_I2S1_DOUT);
#endif
#elif defined _USE_AUDIO_ON_7520V3
	ret = gpio_request(PIN_I2S0_WS, "i2s0_ws");
	if (ret < 0)
		BUG();
	ret = gpio_request(PIN_I2S0_CLK, "i2s0_clk");
	if (ret < 0)
		BUG();
	ret = gpio_request(PIN_I2S0_DIN, "i2s0_din");
	if (ret < 0)
		BUG();
	ret = gpio_request(PIN_I2S0_DOUT, "i2s0_dout");
	if (ret < 0)
		BUG();
	zx29_gpio_config(PIN_I2S0_WS, FUN_I2S0_WS);
	zx29_gpio_config(PIN_I2S0_CLK, FUN_I2S0_CLK);
	zx29_gpio_config(PIN_I2S0_DIN, FUN_I2S0_DIN);
	zx29_gpio_config(PIN_I2S0_DOUT, FUN_I2S0_DOUT);

	ret = gpio_request(PIN_I2S1_WS, "i2s1_ws");
	if (ret < 0)
		BUG();
	ret = gpio_request(PIN_I2S1_CLK, "i2s1_clk");
	if (ret < 0)
		BUG();
	ret = gpio_request(PIN_I2S1_DIN, "i2s1_din");
	if (ret < 0)
		BUG();
	ret = gpio_request(PIN_I2S1_DOUT, "i2s1_dout");
	if (ret < 0)
		BUG();
	zx29_gpio_config(PIN_I2S1_WS, FUN_I2S1_WS);
	zx29_gpio_config(PIN_I2S1_CLK, FUN_I2S1_CLK);
	zx29_gpio_config(PIN_I2S1_DIN, FUN_I2S1_DIN);
	zx29_gpio_config(PIN_I2S1_DOUT, FUN_I2S1_DOUT);
#endif

	/* top i2s1(no i2s2)cfg */
	AmrRegBit = ARM_I2S_LOOP_CFG;
#ifdef TOP_I2S0_PIN_CONNECT
	/*i2s0 pin cfg*/
	AmrRegBit &= 0xfffffff8;
	AmrRegBit |= 0x00000001; /* inter arm_i2s1--top i2s1 */
#else
	/*i2s1 pin cfg*/
	AmrRegBit &= 0xfff8ffff;
	AmrRegBit |= 0x00010000; /* inter arm_i2s1--top i2s2 */
#endif
	/* inter loop */
	AmrRegBit &= 0xfffffe07;
	AmrRegBit |= 0x000000a8; /* 0x000000a8 loop dsp afe(loop i2s1)--arm i2s2(loop i2s2) 0x00000150 loop dsp arm(loop i2s1)--afe i2s2(loop i2s2)*/

	ARM_I2S_LOOP_CFG = AmrRegBit;

	zDrvRamlog_PRINTF(RAMLOG_MOD_AUDIO, "vp_SetTopI2sConfig ARM_I2S_LOOP_CFG=0x%x\n", ARM_I2S_LOOP_CFG);
#endif
}

/**
* @brief	get i2s status.
*
* @param	tId	 i2s select i2s0 or i2s1
* @return	0-DRV_SUCCESS, other-error
*/
T_I2s_Status zDrvI2s_GetI2sStatus(T_ZDrvI2S_Id tId)
{
	return (s_i2sInstance[tId].i2sStatus);
}
static int __init i2s_init(void)
{
//	printk("i2s_init register\n");
	return platform_device_register(&i2s_device);
}

static void __exit i2s_exit(void)
{
//	printk("i2s_exit unregister\n");
	platform_device_unregister(&i2s_device);
}

module_init(i2s_init);
module_exit(i2s_exit);

