/* SPDX-License-Identifier: GPL-2.0-only */
/*
 *  pxa2xx_ssp.h
 *
 *  Copyright (C) 2003 Russell King, All Rights Reserved.
 *
 * This driver supports the following PXA CPU/SSP ports:-
 *
 *       PXA250     SSP
 *       PXA255     SSP, NSSP
 *       PXA26x     SSP, NSSP, ASSP
 *       PXA27x     SSP1, SSP2, SSP3
 *       PXA3xx     SSP1, SSP2, SSP3, SSP4
 */

#ifndef __LINUX_SSP_H
#define __LINUX_SSP_H

#include <linux/list.h>
#include <linux/io.h>
#include <linux/of.h>


/*
 * SSP Serial Port Registers
 * PXA250, PXA255, PXA26x and PXA27x SSP controllers are all slightly different.
 * PXA255, PXA26x and PXA27x have extra ports, registers and bits.
 */

#define TOP_CTRL		0x00	/* SSP Top Control Register */
#define FIFO_CTRL		0x04	/* SSP FIFO Control Register */
#define INT_EN			0x08    /* SSP Interrupt Enable Register */
#define TO			0x0C    /* SSP Time Out Register */
#define DATAR			0x10    /* SSP Data Register */
#define STATUS			0x14    /* SSP Stauts Register */
#define PSP_CTRL		0x18    /* SSP Programmable Serial Protocal Control Register */
#define NET_WORK_CTRL		0x1C    /* SSP NET Work Control Register */
#define NET_WORK_STATUS		0x20    /* SSP Net Work Status Register */
#define RWOT_CTRL		0x24    /* SSP RWOT Control Register */
#define RWOT_CCM		0x28    /* SSP RWOT Counter Cycles Match Register */
#define RWOT_CVWRn		0x2C    /* SSP RWOT Counter Value Write for Read Request Register */

/* 0x00 TOP_CTRL */
#define TOP_TTELP		(1 << 18)
#define TOP_TTE			(1 << 17)
#define TOP_SCFR		(1 << 16)
#define TOP_IFS			(1 << 15)
#define TOP_HOLD_FRAME_LOW	(1 << 14)
#define TOP_TRAIL		(1 << 13)
#define TOP_LBM			(1 << 12)
#define TOP_SPH			(1 << 11)
#define TOP_SPO			(1 << 10)
#define TOP_DSS(x)		((x - 1) << 5)
#define TOP_DSS_MASK		(0x1F << 5)
#define TOP_SFRMDIR		(1 << 4)
#define TOP_SCLKDIR		(1 << 3)
#define TOP_FRF_MASK		(0x3 << 1)
#define TOP_FRF_Motorola	(0x0 << 1)	/* Motorola's Serial Peripheral Interface (SPI) */
#define TOP_FRF_TI		(0x1 << 1)	/* Texas Instruments' Synchronous Serial Protocol (SSP) */
#define TOP_FRF_National	(0x2 << 1)	/* National Microwire */
#define TOP_FRF_PSP		(0x3 << 1)	/* Programmable Serial Protocol(PSP) */
#define TOP_SSE			(1 << 0)

/* 0x04 FIFO_CTRL */
#define FIFO_STRF			(1 << 19)
#define FIFO_EFWR			(1 << 18)
#define FIFO_RXFIFO_AUTO_FULL_CTRL	(1 << 17)
#define FIFO_FPCKE			(1 << 16)
#define FIFO_TXFIFO_WR_ENDIAN_MASK	(0x3 << 14)
#define FIFO_RXFIFO_RD_ENDIAN_MASK	(0x3 << 12)
#define FIFO_WR_ENDIAN_16BITS		(1 << 14)	/* Swap first 16 bits and last 16 bits */
#define FIFO_WR_ENDIAN_8BITS		(2 << 14)	/* Swap all 4 bytes */
#define FIFO_RD_ENDIAN_16BITS		(1 << 12)	/* Swap first 16 bits and last 16 bits */
#define FIFO_RD_ENDIAN_8BITS		(2 << 12)	/* Swap all 4 bytes */
#define FIFO_RSRE			(1 << 11)
#define FIFO_TSRE			(1 << 10)
#define FIFO_RFT(x)			(x << 5)
#define FIFO_TFT(x)			(x << 0)

/* 0x08 INT_EN */
#define INT_EN_EBCEI		(1 << 6)
#define INT_EN_TIM		(1 << 5)
#define INT_EN_RIM		(1 << 4)
#define INT_EN_TIE		(1 << 3)
#define INT_EN_RIE		(1 << 2)
#define INT_EN_TINTE		(1 << 1)
#define INT_EN_PINTE		(1 << 0)

/* 0x0C TO */
#define TIMEOUT(x)	((x) << 0)

/* 0x10 DATAR */
#define DATA(x)		((x) << 0)

/* 0x14 STATUS */
#define STATUS_OSS		(1 << 23)
#define STATUS_TX_OSS		(1 << 22)
#define STATUS_BCE		(1 << 21)
#define STATUS_ROR		(1 << 20)
#define STATUS_RNE		(1 << 14)
#define STATUS_RFS		(1 << 13)
#define STATUS_TUR		(1 << 12)
#define STATUS_TNF		(1 << 6)
#define STATUS_TFS		(1 << 5)
#define STATUS_EOC		(1 << 4)
#define STATUS_TINT		(1 << 3)
#define STATUS_PINT		(1 << 2)
#define STATUS_CSS		(1 << 1)
#define STATUS_BSY		(1 << 0)

/* 0x18 PSP_CTRL */
#define PSP_EDMYSTOP(x)		((x) << 27)
#define PSP_EMYSTOP(x)		((x) << 25)
#define PSP_EDMYSTRT(x)		((x) << 23)
#define PSP_DMYSTRT(x)		((x) << 21)
#define PSP_STRTDLY(x)		((x) << 18)
#define PSP_SFRMWDTH(x)		((x) << 12)
#define PSP_SFRMDLY(x)		((x) << 5)
#define PSP_SFRMP		(1 << 4)
#define PSP_FSRT		(1 << 3)
#define PSP_ETDS		(1 << 2)
#define PSP_SCMODE(x)		((x) << 0)

/* 0x1C NET_WORK_CTRL */
#define RTSA(x)			((x) << 12)
#define RTSA_BASE       12
#define RTSA_MASK		(0xFF << 12)
#define TTSA(x)			((x) << 4)
#define TTSA_BASE       4
#define TTSA_MASK		(0xFF << 4)
#define NET_FRDC(x)		((x) << 1)
#define NET_WORK_MODE		(1 << 0)

/* 0x20 NET_WORK_STATUS */
#define NET_SATUS_NMBSY		(1 << 3)
#define NET_STATUS_TSS(x)	((x) << 0)

/* 0x24 RWOT_CTRL */
#define RWOT_MASK_RWOT_LAST_SAMPLE	(1 << 4)
#define RWOT_CLR_RWOT_CYCLE		(1 << 3)
#define RWOT_SET_RWOT_CYCLE		(1 << 2)
#define RWOT_CYCLE_RWOT_EN		(1 << 1)
#define RWOT_RWOT				(1 << 0)

enum pxa_ssp_type {
	SSP_UNDEFINED = 0,
	PXA25x_SSP,  /* pxa 210, 250, 255, 26x */
	PXA25x_NSSP, /* pxa 255, 26x (including ASSP) */
	PXA27x_SSP,
	PXA3xx_SSP,
	PXA168_SSP,
	PXA910_SSP,
	CE4100_SSP,
	LPSS_SSP,
};

struct ssp_device {
	struct platform_device *pdev;
	struct list_head	node;

	struct clk	*clk;
	void __iomem	*mmio_base;
	unsigned long	phys_base;

	const char	*label;
	int		port_id;
	int		type;
	int		use_count;
	int		irq;
	int		drcmr_rx;
	int		drcmr_tx;

	u16		num_chipselect;
	u8		enable_dma;

	struct device_node	*of_node;
};

/**
 * pxa_ssp_write_reg - Write to a SSP register
 *
 * @dev: SSP device to access
 * @reg: Register to write to
 * @val: Value to be written.
 */
static inline void pxa_ssp_write_reg(struct ssp_device *dev, u32 reg, u32 val)
{
	__raw_writel(val, dev->mmio_base + reg);
}

/**
 * pxa_ssp_read_reg - Read from a SSP register
 *
 * @dev: SSP device to access
 * @reg: Register to read from
 */
static inline u32 pxa_ssp_read_reg(struct ssp_device *dev, u32 reg)
{
	return __raw_readl(dev->mmio_base + reg);
}

#if defined(CONFIG_ARCH_PXA) || defined(CONFIG_ARCH_MMP)
struct ssp_device *pxa_ssp_request(int port, const char *label);
void pxa_ssp_free(struct ssp_device *);
struct ssp_device *pxa_ssp_request_of(const struct device_node *of_node,
				      const char *label);
#else
static inline struct ssp_device *pxa_ssp_request(int port, const char *label)
{
	return NULL;
}
static inline struct ssp_device *pxa_ssp_request_of(const struct device_node *n,
						    const char *name)
{
	return NULL;
}
static inline void pxa_ssp_free(struct ssp_device *ssp) {}
#endif

#endif
