/*-----------------------------------------------------------------------------------------------*/
/**
  @file ql_audio_cfg.h
  @brief audio config API, including audio initlization, audio configuration

  @detailes
  Quectel AG55x series module AUDIO service.

  @htmlonly
  <span style="font-weight: bold">History</span>
  @endhtmlonly

  when       |   who      	|    what, where, why
  --------   |   ---      	|    ----------------------------------------------------------
  2021-11-03 |   dameng.lin |    Created .

  Copyright (c) 2019 Quectel Wireless Solution, Co., Ltd. All Rights Reserved.
  Quectel Wireless Solution Proprietary and Confidential.
-------------------------------------------------------------------------------------------------*/
#ifndef __QL_AUDIO_CFG_H__
#define __QL_AUDIO_CFG_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <math.h>
#include <stdint.h>
#include "mbtk_log.h"
#include "mbtk_audio2.h"
#include "mbtk_audio_gain.h"
#include "mbtk_audio_ubus.h"
#include "ql_v2/ql_type.h"

#define QL_AUDIO_STATE_0  0
#define QL_AUDIO_STATE_1  1

#define AUDIO_MODE_NORMAL   0
#define AUDIO_MODE_RINGTONE 1
#define AUDIO_MODE_IN_CALL  2
#define DEVICE_EARPIECE 0
#define DEVICE_SPEAKER  1
#define DEVICE_HEADSET  2

#define MIC_GAIN_MIN 0
#define MIC_GAIN_MAX 65535
#define VOLUME_MIN 0
#define VOLUME_MAX 100

static int32_t stored_mic_gain = -1;
static int32_t stored_down_volume = -1;


/**
  @brief  The audio service error callback function
  @param  error  error code.See ql_type.h for details.
*/
typedef void (*ql_audio_service_error_cb_f)(int error);

ql_audio_service_error_cb_f audio_error_callback = NULL;

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function initializes an audio service.

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_SERVICE_NOT_READY  Audio service not ready. Try again later.
  @retval  Others Failed execution.See ql_type.h for error codes
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_init(void)
{
    int result = mbtk_audio_pcm_init();
    if (result != 0) {
        LOGE("Error: mbtk_audio_wav_init failed with error code %d\n", result);
        if (audio_error_callback) {
            audio_error_callback(result);
        }
        return QL_ERR_SERVICE_NOT_READY;
    }

    return QL_ERR_OK;
}
/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function deinitializes audio services.

  @retval  QL_ERR_OK Successful
  @retval  QL_ERR_SERVICE_NOT_READY Service is not ready, need to retry
  @retval  Others Failed execution.See ql_type.h for error codes.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_deinit(void)
{
    int result = mbtk_audio_wav_deinit();
    if (result != 0) {
        LOGE("Error: mbtk_audio_wav_deinit failed with error code %d\n", result);
        if (audio_error_callback) {
            audio_error_callback(result);
        }
        return QL_ERR_SERVICE_NOT_READY;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function sets the value of a mixer.

  @param[in] control    the name of the mixer
  @param[in] val_list   the value of the mixer to be set. String type value.Multiple values are separated by spaces.

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others Failed execution.See ql_type.h for error codes.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_set_mixer_control(const char *control, const char *val_list);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function gets the value of the mixer.

  @param[in]  control       the name of the mixer
  @param[out] val_list_buf  buffer for storing mixer values
  @param[in]  buf_size      the buffer size. Unit:Byte

  @retval  QL_ERR_OK Successful execution.
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others Failed execution.See ql_type.h for error codes.

  @note  Generally, 64 bytes is enough
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_get_mixer_control(const char *control, char *val_list_buf, uint32_t buf_size);



/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function sets the microphone gain for voice call uplink.

  @param[in] mic_gain  the microphone gain to be set. range:0-65535

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.

  @note  The API should be called before a voice call
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_set_tx_voice_mic_gain(int32_t mic_gain)
{
    if (mic_gain < MIC_GAIN_MIN || mic_gain > MIC_GAIN_MAX) {
        return QL_ERR_INVALID_ARG;
    }

    stored_mic_gain = mic_gain;

    float mapped_gain = MBTK_AUDIO_GAIN_MIN + ((float)mic_gain * (MBTK_AUDIO_GAIN_MAX - MBTK_AUDIO_GAIN_MIN) / MIC_GAIN_MAX);
    int result = mbtk_dsp_gain_set(CONFIG_DSPGAIN_TX, (int)round(mapped_gain));
    if (result != 0) {
        LOGE("Error: mbtk_dsp_gain_set failed with error code %d\n", result);
        if (audio_error_callback) {
            audio_error_callback(result);
        }
        return QL_ERR_SERVICE_NOT_READY;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function gets the microphone gain for voice call uplink.

  @param[out] p_mic_gain  the current microphone gain for voice call uplink.

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_get_tx_voice_mic_gain(int32_t *p_mic_gain)
{
    if (p_mic_gain == NULL) {
        return QL_ERR_INVALID_ARG;
    }

    if (stored_mic_gain != -1) {
        *p_mic_gain = stored_mic_gain;
    } else {
        int rx_gain, tx_gain;
        int result = mbtk_dsp_gain_get(&rx_gain, &tx_gain);
        if (result != 0) {
            LOGE("Error: mbtk_dsp_gain_get failed with error code %d\n", result);
            if (audio_error_callback) {
                audio_error_callback(result);
            }
            return QL_ERR_SERVICE_NOT_READY;
        }
        *p_mic_gain = (tx_gain - MBTK_AUDIO_GAIN_MIN) * MIC_GAIN_MAX / (MBTK_AUDIO_GAIN_MAX - MBTK_AUDIO_GAIN_MIN);
    }
    LOGD("Retrieved mic gain: %d\n", *p_mic_gain);
    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function sets the speaker gain for voice call downlink.

  @param[in] spkr_gain   the speaker gain to be set. range:0-65535

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.

  @note  The API should be called before a voice call
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_set_rx_voice_spkr_gain(int32_t spkr_gain);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function gets the speaker gain for voice call downlink.

  @param[out] p_spkr_gain   the current speaker gain for voice call downlink.

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_get_rx_voice_spkr_gain(int32_t *p_spkr_gain);


/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function sets the mute state of voice call uplink.

  @param[in] mute_state  the mute state to be set.
             QL_AUDIO_STATE_0: unmute, QL_AUDIO_STATE_1: mute

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.

  @note  The API should be called during the call
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_set_tx_voice_mute_state(int32_t mute_state);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function gets the mute state of voice call uplink.

  @param[out] p_mute_state  the current mute state of voice call uplink. QL_AUDIO_STATE_0 or QL_AUDIO_STATE_1

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_get_tx_voice_mute_state(int32_t *p_mute_state);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function sets the mute state of voice call downlink.

  @param[in] mute_state  the mute state to be set.
             QL_AUDIO_STATE_0: unmute, QL_AUDIO_STATE_1: mute

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.

  @note  The API should be called during the call
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_set_rx_voice_mute_state(int32_t mute_state);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function gets the mute state of voice call downlink.

  @param[out] p_mute_state  the current mute state of voice call downlink. QL_AUDIO_STATE_0 or QL_AUDIO_STATE_1

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_get_rx_voice_mute_state(int32_t *p_mute_state);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function sets the uplink volume of a codec.

  @param[in] up_volume  the uplink volume to be set. range:0-100

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_set_codec_up_vol(int32_t up_volume);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function gets the uplink volume of a codec.

  @param[out] p_up_volume  the current uplink volume of codec. range:0-100

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_get_codec_up_vol(int32_t *p_up_volume);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function sets the downlink volume of a codec.

  @param[in] down_volume  the volume to be set. range:0-100

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_set_codec_down_vol(int32_t down_volume)
{
    if (down_volume < VOLUME_MIN || down_volume > VOLUME_MAX) {
        return QL_ERR_INVALID_ARG;
    }

    stored_down_volume = down_volume;

    float mapped_volume = MBTK_AUDIO_GAIN_MIN + ((float)down_volume * (MBTK_AUDIO_GAIN_MAX - MBTK_AUDIO_GAIN_MIN) / VOLUME_MAX);
    int result = mbtk_dsp_gain_set(CONFIG_DSPGAIN_RX, (int)round(mapped_volume));
    if (result != 0) {
        LOGE("Error: mbtk_dsp_gain_set failed with error code %d\n", result);
        if (audio_error_callback) {
            audio_error_callback(result);
        }
        return QL_ERR_SERVICE_NOT_READY;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function gets the downlink volume of a codec.

  @param[out] p_down_volume  the current downlink volume of codec. range:0-100

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_get_codec_down_vol(int32_t *p_down_volume)
{
    if (p_down_volume == NULL) {
        return QL_ERR_INVALID_ARG;
    }

    if (stored_down_volume != -1) {
        *p_down_volume = stored_down_volume;
    } else {
        int rx_gain, tx_gain;
        int result = mbtk_dsp_gain_get(&rx_gain, &tx_gain);
        if (result != 0) {
            LOGE("Error: mbtk_dsp_gain_get failed with error code %d\n", result);
            if (audio_error_callback) {
                audio_error_callback(result);
            }
            return QL_ERR_SERVICE_NOT_READY;
        }
        *p_down_volume = (rx_gain - MBTK_AUDIO_GAIN_MIN) * VOLUME_MAX / (MBTK_AUDIO_GAIN_MAX - MBTK_AUDIO_GAIN_MIN);
    }
    LOGD("Retrieved codec downlink volume: %d\n", *p_down_volume);
    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function sets the microphone mute state of a codec.

  @param[in] mute_state  the muute state to be set.
             QL_AUDIO_STATE_0: unmute, QL_AUDIO_STATE_1: mute

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.

  @note  The API should be called during the call or audio playback
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_set_codec_mic_mute_state(int32_t mute_state);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function gets the microphone mute state of a codec.

  @param[out] p_mute_state  the current microphone mute state of codec. QL_AUDIO_STATE_0 or QL_AUDIO_STATE_1

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_get_codec_mic_mute_state(int32_t *p_mute_state);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function sets the speaker mute state of a codec.

  @param[in] mute_state  the mute state to be set.
             QL_AUDIO_STATE_0: unmute, QL_AUDIO_STATE_1: mute

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.

  @note  The API should be called during the call or audio playback
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_set_codec_spk_mute_state(int32_t mute_state);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function gets the speaker mute state of a codec.

  @param[out] p_mute_state  the current speaker mute state of codec. QL_AUDIO_STATE_0 or QL_AUDIO_STATE_1

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_get_codec_spk_mute_state(int32_t *p_mute_state);


/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function enables/disables loopback.

  @param[in] enable_state  enable/disable the loopback to be set.
             QL_AUDIO_STATE_0: disable, QL_AUDIO_STATE_1: enable

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_set_loopback_enable_state(int32_t enable_state)
{
    int result;
    int device = DEVICE_SPEAKER;

    if (enable_state == QL_AUDIO_STATE_1) {
        result = mbtk_audio_set_loopback_enable_state(device, enable_state);
        if (result != 0) {
            LOGE("Error: mbtk_audio_set_loopback_enable_state failed\n");
            if (audio_error_callback) {
                audio_error_callback(result);
            }
            return QL_ERR_SERVICE_NOT_READY;
        }
        LOGD("Loopback started successfully\n");
    } else if (enable_state == QL_AUDIO_STATE_0) {
        result = mbtk_audio_set_loopback_enable_state(device, enable_state);
        if (result != 0) {
            LOGE("Error: mbtk_audio_set_loopback_enable_state failed\n");
            if (audio_error_callback) {
                audio_error_callback(result);
            }
            return QL_ERR_SERVICE_NOT_READY;
        }
    } else {
        LOGD("Error: Invalid enable_state %d\n", enable_state);
        return QL_ERR_INVALID_ARG;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  The function gets the loopback state.

  @param[out] p_enable_state  the current loopback state. QL_AUDIO_STATE_0 or QL_AUDIO_STATE_1

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_get_loopback_enable_state(int32_t *p_enable_state)
{
    if (p_enable_state == NULL) {
        LOGE("Error: p_enable_state is NULL\n");
        return QL_ERR_INVALID_ARG;
    }

    int device;
    int result = mbtk_audio_get_loopback_enable_state(&device, p_enable_state);
    if (result != 0) {
        LOGE("Error: mbtk_audio_get_loopback_enable_state failed\n");
        if (audio_error_callback) {
            audio_error_callback(result);
        }
        return QL_ERR_SERVICE_NOT_READY;
    }

    LOGD("Current loopback state: %d\n", *p_enable_state);
    return QL_ERR_OK;
}
/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function sets the sidetone gain.

  @param[in] sidetone_gain  sidetone gain to be set. range: 0-65535

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_set_sidetone_gain(int32_t sidetone_gain);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function gets the sidetone gain.

  @param[out] p_sidetone_gain  the current sidetone gain. range: 0-65535, default value: 1298

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_get_sidetone_gain(int32_t *p_sidetone_gain);


/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function sets the voice call manager state. By default, voice call services such as ringtones, ringback
tones, third-party ringtones and voice stream status control are all implemented by ql_audiod program
automatically.

  @param[in] manager_state  The manager state to be set. Voice call services include ringtones, ringback tones, the third-party
ringtones and voice stream status control. range: QL_AUDIO_STATE_0 and QL_AUDIO_STATE_1

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.

  @note  The voice service, such as call ring, beep tone, will not work during a voice call
         If the manager_state is set to QL_AUDIO_STATE_1
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_set_voice_call_manager_state(int32_t manager_state);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function gets the voice call manager state.

  @param[out] p_manager_state  the current voice call manager state.
              QL_AUDIO_STATE_0: close, QL_AUDIO_STATE_1:open

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_get_voice_call_manager_state(int32_t *p_manager_state);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function sets the voice stream state.

  @param[in] stream_state  voice stream state to be set.
             QL_AUDIO_STATE_0: close, QL_AUDIO_STATE_1:open

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.

  @note  After the voice call manager state is set to QL_AUDIO_STATE_1 by calling
ql_audio_set_voice_call_manager_state(), the service program ql_audiod will not enable voice stream
while establishing a voice call. In such a case, call ql_audio_set_voice_stream_state() to enable voice
stream state by setting stream_state to QL_AUDIO_STATE_1, and then the voice stream can be disabled
by setting stream_state to QL_AUDIO_STATE_0.*/
/*-----------------------------------------------------------------------------------------------*/
//int ql_audio_set_voice_stream_state(int32_t stream_state);
int ql_audio_set_voice_stream_state(int32_t stream_state);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  The function gets the voice stream state.

  @param[out] p_stream_state  the current voice stream state. QL_AUDIO_STATE_0 or QL_AUDIO_STATE_1

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
//int ql_audio_get_voice_stream_state(int32_t *p_stream_state);
int ql_audio_get_voice_stream_state(int32_t *p_stream_state);


/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function sets the callback function for audio service errors. 
  @param[in] cb  The callback function for audio service errors.Only when the audioservice exit abnormally,the callback function is executed.
  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_set_service_error_cb(ql_audio_service_error_cb_f cb)
{
    if (cb == NULL) {
        LOGE("Error: Callback function is NULL\n");
        return QL_ERR_INVALID_ARG;
    }

    audio_error_callback = cb;

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function switch codec.

  @param[in] codec_switch  the which codec to be switch.
             0:GSSP  1:internal codec

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_set_codec_switch(int32_t codec_switch);


/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function gets the tone state.

  @param[out] p_codec_switch  which codec has switch.    

  @retval  QL_ERR_OK Successful execution
  @retval  QL_ERR_INVALID_ARG Failed execution.Invalid arguments
  @retval  Others errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_get_codec_switch(int32_t *p_codec_switch);


typedef struct ql_audio_tone_config_struct
{
    int32_t low_fre;
    int32_t high_fre;
    int32_t on_duration;
    int32_t off_duration;
    int32_t volume;
    int32_t count;
}ql_audio_tone_config_t;

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Set tone enable state

  @param[in] low_fre   low frequency, range: 100~4000
  @param[in] high_fre  high frequency, range: 100~4000
  @param[in] on_duration  tone play time,  unit is millisecond
  @param[in] off_duration  tone pause time,  unit is millisecond
  @param[in] volume  tone play volume, range: 0~5000
  @param[in] count  tone play count

  @return  QL_ERR_OK - Successful
           QL_ERR_INVALID_ARG - Invalid arguments
           Other errorcode defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_set_tone_enable_state(int32_t enable_state,ql_audio_tone_config_t *p_tone_config);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Get tone enable state

  @param[out] p_enable_state  tone enable state,
              QL_AUDIO_STATE_0: close, QL_AUDIO_STATE_1:open

  @return  QL_ERR_OK - Successful
           QL_ERR_INVALID_ARG - Invalid arguments
           Other errorcode defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_get_tone_enable_state(int32_t *p_enable_state);

#ifdef __cplusplus
}
#endif
#endif

