/*-----------------------------------------------------------------------------------------------*/
/**
  @file ql_audio_pcm.h
  @brief playback or capture API
*/
/*-----------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  EDIT HISTORY
  This section contains comments describing changes made to the file.
  Notice that changes are listed in reverse chronological order.
  $Header: $
  when         who          what, where, why
  --------     ---          ----------------------------------------------------------
  2021-11-03   dameng.lin     Created .
-------------------------------------------------------------------------------------------------*/

#ifndef __QL_AUDIO_PCM_H
#define __QL_AUDIO_PCM_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include "mbtk_log.h"
#include "mbtk_audio2.h"
#include "ql_v2/ql_type.h"
#include "mbtk_ril_api.h"
#include "ql_v2/ql_audio_cfg.h"

typedef int ql_audio_handle_t;

static const char *g_wav_file_name = NULL;
static mbtk_ril_handle* call_info_handle = NULL;


typedef enum QL_AUDIO_STREAM_FORMAT_ENUM
{
    QL_AUDIO_STREAM_FORMAT_PCM = 1,         /**< PCM*/
    QL_AUDIO_STREAM_FORMAT_MP3,             /**< MP3*/
    QL_AUDIO_STREAM_FORMAT_AMR,             /**< AMR*/
    QL_AUDIO_STREAM_FORMAT_AMRNB,           /**< AMR-NB*/
    QL_AUDIO_STREAM_FORMAT_AMRWB,           /**< AMR-WB*/
}QL_AUDIO_STREAM_FORMAT_E;


/**The structure of PCM configuration parameters*/
typedef struct ql_audio_pcm_config_struct
{
    /**  Each sound card maintains a hardware buffer to store audio data. The hardware
    *  buffer is divided into several periods. The sound card uses period as a unit to transmit data,
    *  and one period stores some data frames.period_size sets the size of periods in hardware buffer.
    *  When period_size is set to 0,it means that the period size is calculated by the bottom layer to
    *  obtain a default value. If period_size is not 0, the recommended value is 128–960. The larger the
    *  perod_size is, the larger the kernel overhead is
    */
    uint32_t period_size;
    /**  period_count indicates the count of period that the data occupies, when the application reads data
    *  from or writes data to the hardware buffer.The size of the data that the application reads from or
    *  writes to the hardware buffer every time equals period_count multiplied by period_size. The hardware
    *  buffer has a maximum of 8 periods by default. The recommended value of period_count is 1–3.
    */
    uint32_t period_count;
    uint32_t num_channels;  /**< Number of channels. 1 Mono 2 Stereo*/
    uint32_t sample_rate;   /**< Sampling rate. A PCM interface supports 8000 and 16000,and an I2s interface supports 48000.Unit:Hz*/
    uint32_t pcm_format;        /**< PCM data format.Presently supports 2 only,which means 16-bit little endian format*/
} ql_audio_pcm_config_t;

#define QL_AUDIO_INVALID_HANDLE  ((ql_audio_handle_t)(void *)NULL)

typedef enum
{
    QL_AUDIO_STREAM_DIRECTION_PLAYBACK = 0,
    QL_AUDIO_STREAM_DIRECTION_CAPTURE,
    QL_AUDIO_STREAM_DIRECTION_MAX
}QL_AUDIO_STREAM_DIRECTION_E;


/**The enumeration of the front end PCM device types*/
typedef enum QL_AUDIO_FE_PCM_DEV_ENUM
{
    QL_AUDIO_FE_PCM_DEV_MIN = -1,
    QL_AUDIO_FE_PCM_DEV_MULTIMEDIA1 = 0,    /**< The first PCM device available for general-purpose audio playback and capturing.*/
	QL_AUDIO_FE_PCM_DEV_MULTIMEDIA2 = 1,    /**< The sencond PCM device available for general-purpose audio playback and capturing.*/
	QL_AUDIO_FE_PCM_DEV_MULTIMEDIA3 = 2,    /**< The third PCM device available for general-purpose audio playback and capturing.*/
	QL_AUDIO_FE_PCM_DEV_MAX
} QL_AUDIO_FE_PCM_DEV_E;



typedef enum QL_AUDIO_BE_DAI_ENUM
{
    QL_AUDIO_BE_DAI_MIN = -1,
    QL_AUDIO_BE_DAI_PLAYBACK_PRI_PCM = 0,   /**< Play back audio to the first PCM interface.*/
    QL_AUDIO_BE_DAI_PLAYBACK_VOICE_TX,      /**< play back audio to the voice call uplink*/
    QL_AUDIO_BE_DAI_CAPTURE_PRI_PCM,        /**< Capture audio from the first PCM interface*/
    QL_AUDIO_BE_DAI_CAPTURE_VOICE_UL,       /**< Capture voice stream from voice call uplink*/
    QL_AUDIO_BE_DAI_CAPTURE_VOICE_DL,       /**< Capture voice stream from voice call downlink*/
    QL_AUDIO_BE_DAI_MAX
}QL_AUDIO_BE_DAI_E;

#define QL_AUDIO_BE_DAI_MASK_PLAYBACK_PRI_PCM       (1 << QL_AUDIO_BE_DAI_PLAYBACK_PRI_PCM)
#define QL_AUDIO_BE_DAI_MASK_PLAYBACK_VOICE_TX      (1 << QL_AUDIO_BE_DAI_PLAYBACK_VOICE_TX)
#define QL_AUDIO_BE_DAI_MASK_CAPTURE_PRI_PCM        (1 << QL_AUDIO_BE_DAI_CAPTURE_PRI_PCM)
#define QL_AUDIO_BE_DAI_MASK_CAPTURE_VOICE_UL       (1 << QL_AUDIO_BE_DAI_CAPTURE_VOICE_UL)
#define QL_AUDIO_BE_DAI_MASK_CAPTURE_VOICE_DL       (1 << QL_AUDIO_BE_DAI_CAPTURE_VOICE_DL)

/**The enumeration of audio playback state*/
typedef enum  QL_AUDIO_PLAYBACK_STATE_ENUM
{
    QL_AUDIO_PLAYBACK_STATE_CLOSE = 0,      /**< Close*/
    QL_AUDIO_PLAYBACK_STATE_OPEN,           /**< Open*/
    QL_AUDIO_PLAYBACK_STATE_PREPARE,        /**< Ready*/
    QL_AUDIO_PLAYBACK_STATE_PLAYING,        /**< Playing*/
    QL_AUDIO_PLAYBACK_STATE_FINISHED,       /**< Finished*/
    QL_AUDIO_PLAYBACK_STATE_PAUSE,          /**< Pause*/
    QL_AUDIO_PLAYBACK_STATE_ERROR,          /**< Error*/
} QL_AUDIO_PLAYBACK_STATE_E;

/**The enumeration of audio capture state*/
typedef enum QL_AUDIO_CAPTURE_STATE_ENUM
{
    QL_AUDIO_CAPTURE_STATE_CLOSE = 0,       /**< Close*/
    QL_AUDIO_CAPTURE_STATE_OPEN,            /**< Open*/
    QL_AUDIO_CAPTURE_STATE_PREPARE,         /**< Prepare*/
    QL_AUDIO_CAPTURE_STATE_CAPTURING,       /**< Capturing*/
    QL_AUDIO_CAPTURE_STATE_FINISHED,        /**< Finished*/
    QL_AUDIO_CAPTURE_STATE_PAUSE,           /**< Pause*/
    QL_AUDIO_CAPTURE_STATE_ERROR,           /**< Error*/
} QL_AUDIO_CAPTURE_STATE_E;


#define  QL_AUDIO_PLAYBACK_NONBLOCK  0
#define  QL_AUDIO_PLAYBACK_BLOCK     1


/**
    @brief  The audio capturing state callback function
    @param  handle Recording handle,which is the return value of ql_audio_capture_open().
    @param  params Parameters carried by the callback function
    @param  state  The current audio capturing state.
*/
typedef int (*ql_audio_capture_state_cb_f)(ql_audio_handle_t handle, void *params, QL_AUDIO_CAPTURE_STATE_E state);

/**
    @brief  The playback state callback function
    @param  handle Playback handle, which is the return value of ql_audio_playback_open().
    @param  params Parameters carried by the callback function.
    @param  state  The current playback state.
*/
typedef int (*ql_audio_playback_state_cb_f)(ql_audio_handle_t handle, void *params, QL_AUDIO_PLAYBACK_STATE_E state);
/*-----------------------------------------------------------------------------------------------*/
/**
  @brief This function opens the audio context for playback.

  @param[in] fe_pcm_dev  Front end PCM device type. defined by QL_AUDIO_FE_PCM_DEV_E
  @param[in] be_dai_mask  Backend digit audio interface mask, support follow:
             QL_AUDIO_BE_DAI_MASK_PLAYBACK_PRI_PCM  Play back audio to the 1 st PCM interface
             QL_AUDIO_BE_DAI_MASK_PLAYBACK_SEC_PCM  Play back audio to the 2 nd PCM interface
             QL_AUDIO_BE_DAI_MASK_PLAYBACK_PRI_I2S  Play back audio to the 1 st I2S interface
             QL_AUDIO_BE_DAI_MASK_PLAYBACK_SEC_I2S  Play back audio to the 2 nd I2S interface
             QL_AUDIO_BE_DAI_MASK_PLAYBACK_VOICE_TX Play back audio to the voice call uplink

  @retval  A_valid_handle   Successful execution.
  @retval  QL_AUDIO_INVALID_HANDLE  Failed execution.Invalid handle
  */
/*-----------------------------------------------------------------------------------------------*/
ql_audio_handle_t ql_audio_playback_open(QL_AUDIO_FE_PCM_DEV_E fe_pcm_dev, uint32_t be_dai_mask)
{
    if (fe_pcm_dev == QL_AUDIO_FE_PCM_DEV_MULTIMEDIA1 && be_dai_mask == QL_AUDIO_BE_DAI_MASK_PLAYBACK_PRI_PCM)
    {
        int init_result = mbtk_audio_pcm_init();

        if (init_result != 0) {
            LOGE("Error: mbtk_audio_pcm_init failed with error code %d\n", init_result);
            if (audio_error_callback) {
                audio_error_callback(init_result);
            }
            return QL_AUDIO_INVALID_HANDLE;
        }
    } else {
        printf("Error: Unsupported PCM device or DAI mask\n");
        return QL_AUDIO_INVALID_HANDLE;
    }

    return (ql_audio_handle_t)1;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function prepares for audio file playback.

  @param[in] handle     The handle returned by ql_audio_playback_open().
  @param[in] file_name  The name of the file to be played back.
  @param[in] pcm_config  Pcm config, including sample rate, channel nums,
             defined by ql_audio_pcm_config_t. Generally, it is NULL.
  @param[in] playback_state_cb  Callback function to report the current playback state
             The states defined by QL_AUDIO_PLAYBACK_STATE_E
  @param[in] params Parameters carried by the callback function.

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  QL_ERR_INVALID_STATE     Failed execution. Invalid state.
  @retval  Others   Failed execution. See ql_type.h for error codes.

  @note   Before calling this function, call ql_audio_playback_open() first to obtain a handle.
          If an audio file is expected to be played back, call this function first to prepare for
          the playback and then ql_audio_playback_play() to start playback.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_playback_file_prepare(ql_audio_handle_t handle,
                                   const char *file_name,
                                   ql_audio_pcm_config_t *pcm_config,
                                   ql_audio_playback_state_cb_f playback_state_cb,
                                   void *params)
{
    if (handle == QL_AUDIO_INVALID_HANDLE) {
        LOGE("Error: Invalid handle\n");
        if (audio_error_callback) {
            audio_error_callback(-1);
        }
        return QL_ERR_SERVICE_NOT_READY;
    }

    if (file_name == NULL) {
        LOGE("Error: File name is NULL\n");
        if (audio_error_callback) {
            audio_error_callback(-1);
        }
        return QL_ERR_SERVICE_NOT_READY;
    }

    g_wav_file_name = file_name;

    if (pcm_config != NULL) {
        printf("Warning: PCM configuration provided but not supported\n");
    } else {
        LOGD("No PCM configuration provided\n");
    }

    if (playback_state_cb != NULL) {
        printf("Warning: Playback state callback provided but not supported\n");
    } else {
        LOGD("No playback state callback provided\n");
    }

    LOGD("Audio file prepared for playback: %s\n", file_name);
    return QL_ERR_OK;
}
/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function prepares for audio stream playback.

  @param[in] handle  The API ql_audio_playback_open return results
  @param[in] pcm_config  Pcm config, including sample rate, channel nums,
             defined by ql_audio_pcm_config_t. If it is NULL, the API use defaule value.
  @param[in] playback_state_cb  Callback function to report the current playback state.
  @param[in] params  Parameters carried by the callback function.

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  QL_ERR_INVALID_STATE     Failed execution. Invalid state.
  @retval  Others   Failed execution. See ql_type.h for error codes.

  @note   Before calling this function, call ql_audio_playback_open() first to obtain a handle.
          If an audio stream is expected to be played back, call this function first to prepare
          for the audio stream playback and then ql_audio_playback_push_stream() to start playback.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_playback_stream_prepare(ql_audio_handle_t handle,
                                     ql_audio_pcm_config_t *pcm_config,
                                     ql_audio_playback_state_cb_f playback_state_cb,
                                     void *params);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function starts playback of the audio data.

  @param[in] handle  The handle returned by ql_audio_playback_open().

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  QL_ERR_INVALID_STATE     Failed execution. Invalid state.
  @retval  Others   Failed execution. See ql_type.h for error codes.

  @note     Before calling this function, call ql_audio_playback_file_prepare() first to prepare the audio file
            to be played back, otherwise the audio data cannot be played back successfully.This function also supports
            playback of audio stream data. In this case, call ql_audio_playback_stream_prepare() first to prepare
            the audio stream to be played back, then this function to start playback, and finally
            ql_audio_playback_push_stream() to play back the audio stream in buffer.
 */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_playback_play(ql_audio_handle_t handle)
{
    if (handle == QL_AUDIO_INVALID_HANDLE) {
        LOGE("Error: Invalid handle\n");
        if (audio_error_callback) {
            audio_error_callback(-1);
        }
        return QL_ERR_SERVICE_NOT_READY;
    }

    if (g_wav_file_name == NULL) {
        LOGE("Error: No audio file prepared for playback\n");
        if (audio_error_callback) {
            audio_error_callback(-1);
        }
        return QL_ERR_SERVICE_NOT_READY;
    }

    int err = mbtk_audio_wav_play_start(g_wav_file_name);
    if (err != 0) {
        LOGE("Error playing audio file: %d\n", err);
        if (audio_error_callback) {
            audio_error_callback(err);
        }
        return QL_ERR_SERVICE_NOT_READY;
    } else {
        LOGD("Audio file playback started successfully\n");
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function plays back the audio stream in buffer.

  @param[in] handle  The handle returned by ql_audio_playback_open().
  @param[in] stream_buf  The buffer that stores the audio stream to be played back.
  @param[in] buf_size  The size of the audio stream to be played back. Unit: Byte.

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  QL_ERR_INVALID_STATE     Failed execution. Invalid state.
  @retval  Others   Failed execution. See ql_type.h for error codes.

  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_playback_push_stream(ql_audio_handle_t handle, void *stream_buf, uint32_t buf_size);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief    This function pauses the audio playback.

  @param[in] handle  The handle returned by ql_audio_playback_open().

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  Others   Failed execution. See ql_type.h for error codes.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_playback_pause(ql_audio_handle_t handle);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief    This function resumes the audio playback.

  @param[in] handle  The handle returned by ql_audio_playback_open().

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  Others   Failed execution. See ql_type.h for error codes.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_playback_resume(ql_audio_handle_t handle);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief This function stops the audio playback.

  @param[in] handle  The handle returned by ql_audio_playback_open().

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  Others   Failed execution. See ql_type.h for error codes.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_playback_stop(ql_audio_handle_t handle);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief    This function closes the audio context for playback.

  @param[in] handle  The handle returned by ql_audio_playback_open().

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  Others   Failed execution. See ql_type.h for error codes.

  @Note     After an audio playback ends, you must call this function to close the audio context,
            otherwise subsequent call of ql_audio_playback_open() will fail.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_playback_close(ql_audio_handle_t handle)
{
    if (handle == QL_AUDIO_INVALID_HANDLE) {
        LOGE("Error: Invalid handle\n");
        if (audio_error_callback) {
            audio_error_callback(-1);
        }
        return QL_ERR_SERVICE_NOT_READY;
    }

    int deinit_result = mbtk_audio_wav_deinit();
    if (deinit_result != 0) {
        LOGE("Error: mbtk_audio_wav_deinit failed with error code %d\n", deinit_result);
        if (audio_error_callback) {
            audio_error_callback(deinit_result);
        }
        return QL_ERR_SERVICE_NOT_READY;
    }

    LOGD("Audio playback context closed successfully\n");
    return QL_ERR_OK;
}
/*-----------------------------------------------------------------------------------------------*/
/**
  @brief This function gets the audio playback state.

  @param[in] handle  The handle returned by ql_audio_playback_open().
  @param[out] playback_state  the current audio playback state, defined by QL_AUDIO_PLAYBACK_STATE_E
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_playback_get_state(ql_audio_handle_t handle, QL_AUDIO_PLAYBACK_STATE_E *playback_state);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief This function handles the call state changes.

  @param[in] data     Pointer to the call information data.
  @param[in] data_len Length of the call information data.

  This function is a callback that processes different call states. When a call is waiting (MBTK_CLCC),
  it pauses the audio playback. When a call is disconnected (MBTK_DISCONNECTED), it resumes the audio playback.
  For other states, it logs the current call wait state.

  @note The function uses mbtk_audio_wav_play_pause() and mbtk_audio_wav_play_resume() to control audio playback.
        Error handling is performed by logging errors if the playback control functions fail.
*/
/*-----------------------------------------------------------------------------------------------*/
static void call_state_handler(const void* data, int data_len)
{
    mbtk_call_info_t *reg = (mbtk_call_info_t *)data;
    switch (reg->call_wait)
    {
        case MBTK_CLCC:
            {
                int result = mbtk_audio_wav_play_pause();
                if (result != 0) {
                    LOGE("Error: mbtk_audio_wav_play_pause failed with error code %d\n", result);
                    if (audio_error_callback) {
                        audio_error_callback(result);
                    }
                    return;
                }
                LOGD("Incoming call!\n");
            }
            break;
        case MBTK_DISCONNECTED:
            {
                int result = mbtk_audio_wav_play_resume();
                if (result != 0) {
                    LOGE("Error: mbtk_audio_wav_play_resume failed with error code %d\n", result);
                    if (audio_error_callback) {
                        audio_error_callback(result);
                    }
                    return;
                }
                LOGD("Call disconnected!\n");
            }
            break;
        default:
            LOGD("RING: None call_wait = %d", reg->call_wait);
            break;
    }
}
/*-----------------------------------------------------------------------------------------------*/
/**
  @brief This function sets the block flag for audio playback.

  @param[in] handle  The handle returned by ql_audio_playback_open().
  @param[in] flags  block flag, including QL_AUDIO_PLAYBACK_NONBLOCK and QL_AUDIO_PLAYBACK_BLOCK

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  QL_ERR_INVALID_ARG   Illegal argument.
  @retval  QL_ERR_INVALID_STATE     Failed execution. Invalid state.
  @retval  Others   Failed execution. See ql_type.h for error codes.

  @note     Call this function as per the function call sequence below, to make sure the audio playback can be blocked successfully.
                1) ql_audio_playback_open()
                2) ql_audio_playback_set_block_flag()
                3) ql_audio_playback_file_prepare()
                4) ql_audio_playback_play()
            If the audio playback is blocked successfully by calling ql_audio_playback_set_block_flag(), then
                1) If you dial a call or there is an incoming call during the audio playback, the playback will pause;
                    after the call is hung up, the playback resumes automatically.
                2) During a voice call, no function can be used to realize audio playback. In this case,
                    the call of ql_auido_palyback_file_prepare() will fail, which means audio files cannot be played back.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_playback_set_block_flag(ql_audio_handle_t handle, uint8_t flags)
{
    if (handle == QL_AUDIO_INVALID_HANDLE) {
        LOGE("Error: Invalid handle\n");
        if (audio_error_callback) {
            audio_error_callback(-1);
        }
        return QL_ERR_SERVICE_NOT_READY;
    }

    if (flags == QL_AUDIO_PLAYBACK_BLOCK) {
        LOGD("Set to blocking mode\n");

        if (call_info_handle == NULL) {
            call_info_handle = mbtk_ril_open(MBTK_AT_PORT_DEF);
            if (call_info_handle) {
                LOGD("Successfully created qser_voice_call_client_init\n");
            } else {
                LOGE("Failed to create qser_voice_call_client_init\n");
                if (audio_error_callback) {
                    audio_error_callback(-1);
                }
                return QL_ERR_SERVICE_NOT_READY;
            }
        }

        int err = mbtk_call_state_change_cb_reg(call_state_handler);
        if (err) {
            LOGE("Error registering call state change callback: %d\n", err);
            if (audio_error_callback) {
                audio_error_callback(err);
            }
            return QL_ERR_SERVICE_NOT_READY;
        }

    } else if (flags == QL_AUDIO_PLAYBACK_NONBLOCK) {
        LOGD("Set to non-blocking mode\n");
    } else {
        LOGE("Invalid flags parameter: %d\n", flags);
        if (audio_error_callback) {
            audio_error_callback(-1);
        }
        return QL_ERR_SERVICE_NOT_READY;
    }

    return QL_ERR_OK;
}
/*-----------------------------------------------------------------------------------------------*/
/**
  @brief This function opens the audio context for capturing.

  @param[in] fe_pcm_dev     Front end PCM device type.
  @param[in] be_dai_mask    Back end DAI mask,support follow:
             QL_AUDIO_BE_DAI_MASK_CAPTURE_PRI_PCM
             QL_AUDIO_BE_DAI_MASK_CAPTURE_VOICE_UL
             QL_AUDIO_BE_DAI_MASK_CAPTURE_VOICE_DL

@retval  A_valid_handle     Successful execution
@retval  QL_AUDIO_INVALID_HANDLE    Failed execution.Invalid handle
  */
/*-----------------------------------------------------------------------------------------------*/
ql_audio_handle_t ql_audio_capture_open(QL_AUDIO_FE_PCM_DEV_E fe_pcm_dev, uint32_t be_dai_mask);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function prepares for audio file capturing.

  @param[in] handle  The handle returned by ql_audio_capture_open().
  @param[in] file_name  The name of the audio file to be captured.
  @param[in] type  The format of the audio data in the audio file.
  @param[in] pcm_config  Pcm config, including sample rate, channel nums,
             defined by ql_audio_pcm_config_t, If it is NULL, the API use defaule value
  @param[in] capture_state_cb   Callback function to report the current audio capturing state.
  @param[in] params     Parameters carried by the callback function.

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  QL_ERR_INVALID_STATE     Failed execution. Invalid state.
  @retval  Others   Failed execution. See ql_type.h for error codes.

  @note     Before calling this function, call ql_audio_capture_open() first to obtain a handle.
            If an audio file is expected to be captured, call this function first to prepare for the audio file
            capturing and then ql_audio_capture_record() to start capturing.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_capture_file_prepare(ql_audio_handle_t handle,
                                  const char *file_name,
                                  QL_AUDIO_STREAM_FORMAT_E type,
                                  ql_audio_pcm_config_t *pcm_config,
                                  ql_audio_capture_state_cb_f capture_state_cb,
                                  void *params);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief    This function prepares for audio stream capturing.

  @param[in] handle  This function prepares for audio stream capturing.
  @param[in] pcm_config     PCM configuration parameters.
  @param[in] capture_state_cb   Callback function to report the current audio capturing state.
  @param[in] params     Parameters carried by the callback function.

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  QL_ERR_INVALID_STATE     Failed execution. Invalid state.
  @retval  Others   Failed execution. See ql_type.h for error codes.

  @note     Before calling this function, call ql_audio_capture_open() first to obtain a handle.
            If an audio stream is expected to be captured, call this function first to prepare for
            the audio stream capturing and then ql_audio_capture_push_stream() to start capturing.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_capture_stream_prepare(ql_audio_handle_t handle,
                                    ql_audio_pcm_config_t *pcm_config,
                                    ql_audio_capture_state_cb_f capture_state_cb,
                                    void *params);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief    This function starts to capture the audio data.

  @param[in] handle  The handle returned by ql_audio_capture_open().

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  QL_ERR_INVALID_STATE     Failed execution. Invalid state.
  @retval  Others   Failed execution. See ql_type.h for error codes.

  @note Before calling this function, call ql_audio_capture_file_prepare() first to prepare the audio
        file to be captured, otherwise the audio data cannot be captured successfully.This function also
        supports capturing of audio stream data. In this case, call ql_audio_capture_stream_prepare()
        first to prepare the audio stream to be captured, then this function to start capturing,
        and finally ql_audio_capture_pull_stream() to capture the audio stream in buffer.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_capture_record(ql_audio_handle_t handle);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function captures the audio stream data to the buffer.

  @param[in] handle  The handle returned by ql_audio_capture_open().
  @param[out] stream_buf  The buffer that stores the audio stream data to be captured.
  @param[in] buf_size  Buffer size. Unit: Byte.

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  Others   Failed execution. See ql_type.h for error codes.

  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_capture_pull_stream(ql_audio_handle_t handle, void *stream_buf, uint32_t buf_size);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief    This function pauses the audio capturing.

  @param[in] handle     The handle returned by ql_audio_capture_open().

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  Others   Failed execution. See ql_type.h for error codes.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_capture_pause(ql_audio_handle_t handle);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief    This function resumes the audio capturing.

  @param[in] handle  The handle returned by ql_audio_capture_open().

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  Others   Failed execution. See ql_type.h for error codes.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_capture_resume(ql_audio_handle_t handle);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief    This function stops the audio capturing.

  @param[in] handle  The handle returned by ql_audio_capture_open().

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  Others   Failed execution. See ql_type.h for error codes.

  @note     Calling this function will stop audio capturing regardless of whether the capturing is in
            progress or paused,and the capturing cannot be resumed after it is stopped.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_capture_stop(ql_audio_handle_t handle);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  This function closes the audio context for capturing.

  @param[in] handle  The handle returned by ql_audio_capture_open().

  @retval  QL_ERR_OK    Successful execution.
  @retval  QL_ERR_INVALID_HANDLE    Failed execution. Invalid handle.
  @retval  Others   Failed execution. See ql_type.h for error codes.
  @note     After audio capturing ends, you must call this function to close the audio context,
            otherwise subsequent call of ql_audio_capture_open() will fail.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_audio_capture_close(ql_audio_handle_t handle);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief This function gets the current audio capturing state.

  @param[in] handle     The handle returned by ql_audio_capture_open().
  @param[out] capture_state  The current audio capturing state.
  */
/*-----------------------------------------------------------------------------------------------*/
void ql_audio_capture_get_state(ql_audio_handle_t handle, QL_AUDIO_CAPTURE_STATE_E *capture_state);

#ifdef __cplusplus
}
#endif


#endif

