/*-----------------------------------------------------------------------------------------------*/
/**
  @file ql_dm.h 
  @brief device management API
*/
/*-----------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  Copyright (c) 2019 Quectel Wireless Solution, Co., Ltd. All Rights Reserved.
  Quectel Wireless Solution Proprietary and Confidential.
-------------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  EDIT HISTORY
  This section contains comments describing changes made to the file.
  Notice that changes are listed in reverse chronological order.
  $Header: $
  when       who          what, where, why
  --------   ---          ----------------------------------------------------------
  20200316   stan.li      Optimize the ql_dm_get_modem_state interface
  20191224   stan.li      Add radio on/off API
  20190625   stan.li      Created .
-------------------------------------------------------------------------------------------------*/


#ifdef __cplusplus
extern "C" {
#endif


#include <stdio.h>
#include <stdbool.h>
#include <pthread.h>

#include "mbtk_ril_api.h"
#include "ql_type.h"
#include "ql_dm.h"



#define LYNQ_AIR_PLANE_MODE_ON      4    //at+cfun = 4
#define LYNQ_AIR_PLANE_MODE_OFF     1   // at+cfun = 1



#define MBTK_RILD_ERR      -1
#define IMEI_VALID          1

static mbtk_ril_handle* ql_info_handle = NULL;
static mbtk_ril_handle* g_md_version_handle = NULL;

static ql_dm_air_plane_mode_ind_cb g_air_plane_mode_cb = NULL;
static ql_dm_service_error_cb_f global_dm_error_cb = NULL;
static ql_dm_modem_state_ind_cb global_dm_modem_cb = NULL;


static pthread_t g_air_plane_mode_thread;
static pthread_mutex_t g_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t g_cond = PTHREAD_COND_INITIALIZER;
static bool g_thread_running = false;


static void mbtk_send_singnal()
{
    pthread_mutex_lock(&g_mutex);
    pthread_cond_signal(&g_cond);
    pthread_mutex_unlock(&g_mutex);

}

static void mbtk_dm_set_service_error_func(const void* data,int data_len)
{
    if(data !=NULL && data_len == sizeof(int))
    {
        const int *state = (const int*)(data);
        if(*state)
        {
            if(global_dm_error_cb)
            {
                global_dm_error_cb(MBTK_RILD_ERR);
            }

        }
    }
        
}

static void mbtk_modem_state_change_cb(const void* data, int data_len)
{
    uint8 *ptr = (uint8*)data;
    if(global_dm_modem_cb)
    {
        global_dm_modem_cb(*ptr);
    }
    
}

void* air_plane_mode_monitor(void* arg) 
{
    int ql_info = 0;
    int ret = -1;
    mbtk_radio_state_enum mbtk_info;

    while (g_thread_running) 
    {
        pthread_mutex_lock(&g_mutex);
        pthread_cond_wait(&g_cond, &g_mutex); 
        ret = mbtk_radio_state_get(ql_info_handle, &mbtk_info);
        if (ret != 0)
        {
            LOGE("mbtk_radio_state_get fail.");
            return NULL;
        }
        
        if(mbtk_info == MBTK_RADIO_STATE_FULL_FUNC)
        {
            ql_info = QL_DM_AIR_PLANE_MODE_OFF;
        }
        
        if(mbtk_info == MBTK_RADIO_STATE_DIS_RF)
        {
            ql_info = QL_DM_AIR_PLANE_MODE_ON;
        }
        
        if(mbtk_info != MBTK_RADIO_STATE_FULL_FUNC && mbtk_info !=MBTK_RADIO_STATE_DIS_RF)
        {
            
            ql_info = QL_DM_AIR_PLANE_MODE_UNKNOWN;
        }
       
        if(g_air_plane_mode_cb) 
        {
            
            g_air_plane_mode_cb(ql_info);
        }

        pthread_mutex_unlock(&g_mutex);
    }

    return NULL;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Initialize DM service.
  @note You must call this function before other functions can be used in this module.
  @return Whether the DM service was successfully intialized.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/

int ql_dm_init(void)
{
    
    if(ql_info_handle == NULL)
    {

        mbtk_log_init("syslog", "QL_DM");

        ql_info_handle = mbtk_ril_open(MBTK_AT_PORT_DEF);
        if(ql_info_handle)
        {
           
            return QL_ERR_OK;
        } 
        else 
        {
            LOGE("mbtk_info_handle_get() fail.");
            return QL_ERR_FAILED;
        }

     }
     else
     {
        LOGE("No need init again");
        return QL_ERR_FAILED;
     }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Denitialize DM service.
  @return Whether the DM service was successfully deintialized.
  @retval QL_ERR_OK successful.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_deinit(void)
{
    if(ql_info_handle)
    {
    
        int ret = mbtk_ril_close(MBTK_AT_PORT_DEF);
        if(ret != 0) 
        {
            LOGE("mbtk_info_handle_free fail.");
            return QL_ERR_FAILED;
        }
        else 
        {
            ql_info_handle = NULL;
            LOGI("mbtk_info_handle_free success");
            return QL_ERR_OK;
        }
    }
    else
    {
        LOGE("DM not inited.");
        return QL_ERR_NOT_INIT;
    }
   
}

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get device software version.
  @param[out] soft_ver Return software version
  @param[in] soft_ver_len The length of soft_ver
  @return Whether to successfully get the software version
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/

int ql_dm_get_software_version(char *soft_ver, int soft_ver_len)
{

    int ret = -1;
    if(soft_ver == NULL || soft_ver_len <= 0)
    {
         LOGE("Bad parameters  ");
         return QL_ERR_FAILED;
    }
    if(ql_info_handle != NULL)
    {
    
        ret = mbtk_version_get(ql_info_handle,soft_ver);
        if(ret != 0)
        {
            LOGE("ql_dm_get_software_version  error.");
            return QL_ERR_FAILED;
        }
    }
    else
    {
        mbtk_ril_handle* mbtk_info_handle = NULL;
        mbtk_info_handle = mbtk_ril_open(MBTK_AT_PORT_DEF);
        if(mbtk_info_handle == NULL)
        {
            LOGE("mbtk_info_handle_get fail.");
            return QL_ERR_FAILED;
        }
        ret = mbtk_version_get(mbtk_info_handle,soft_ver);
        if(ret != 0)
        {
            LOGE("ql_dm_get_software_version  error.");
            
        }
        ret = mbtk_ril_close(MBTK_AT_PORT_DEF);
        if(ret != 0) 
        {
            LOGE("mbtk_info_handle_free fail.");
            return QL_ERR_FAILED;
        }
        
    }
    return QL_ERR_OK;
    

}

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get modem state.
  @details QL_DM_MODEM_STATE_ONLINE,if modem starts normally.
  @details QL_DM_MODEM_STATE_OFFLINE,in modem starts abnormally.
  @details QL_DM_MODEM_STATE_UNKNOWN,unknown error.
  @param[out] modem_state The state of modem
  @return Whether to successfully get the modem state
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_get_modem_state(QL_DM_MODEM_STATE_TYPE_E *modem_state)
{
    int ret = -1;
    mbtk_radio_state_enum tmp_rf;
    if(ql_info_handle == NULL)
    {
        LOGE("DM no init");
        return QL_ERR_NOT_INIT;
    }
    ret = mbtk_radio_state_get(ql_info_handle, &tmp_rf);
    if (ret != 0)
    {
        LOGE("mbtk_radio_state_get fail.");
        return QL_ERR_FAILED;
    }
    
    if(tmp_rf == 0 )
    {
        *modem_state = QL_DM_MODEM_STATE_OFFLINE;
    }
    else if(tmp_rf == 1)
    {
        *modem_state = QL_DM_MODEM_STATE_ONLINE;
    }
    else
    {
        *modem_state = QL_DM_MODEM_STATE_UNKNOWN;
    }
        
    return QL_ERR_OK;

}

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief register modem state event.
  @param[in] cb_func modem state indication callback function
  @return Whether the modem state event was successfully registered.
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/

int ql_dm_set_modem_state_change_ind_cb(ql_dm_modem_state_ind_cb cb_func)
{
    
        int ret =-1;
        
        global_dm_modem_cb = cb_func;
        
        if(ql_info_handle != NULL)
        {
            
            ret = mbtk_radio_state_change_cb_reg(mbtk_modem_state_change_cb);
            if(ret != 0)
            {
                LOGE("call mbtk_radio_state_change_cb_reg failed");
                return QL_ERR_FAILED;
            }
        }
        else 
        {
            if(g_md_version_handle == NULL)
            {
                g_md_version_handle = mbtk_ril_open(MBTK_AT_PORT_DEF);
                if(g_md_version_handle == NULL)
                {
                    LOGE("g_md_version_handle get fail.");
                    return QL_ERR_FAILED;
                }
            }
            
            ret = mbtk_radio_state_change_cb_reg(mbtk_modem_state_change_cb);
            if(ret != 0)
            {
                
                ret = mbtk_ril_close(MBTK_AT_PORT_DEF);
                if(ret < 0) 
                {
                    LOGE("mbtk_info_handle_free fail.");
                    return QL_ERR_FAILED;
                }
                
                LOGE("call mbtk_radio_state_change_cb_reg failed");
                return QL_ERR_FAILED;
            }
            
        }
        
    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get module temperature.
  @param[out] temperature The current temperature
  @return Whether to successfully get the temperature
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_get_temperature(float *temperature);

    
/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get device serial numbers.
  @param[out] p_info Pointer that point to ql_dm_device_serial_numbers_info_t
  @return Whether to successfully get the serial numbers
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_get_device_serial_numbers(ql_dm_device_serial_numbers_info_t *p_info)
{
    int ret = -1;
    if(ql_info_handle == NULL )
    {
        LOGE("DM no init");
        return QL_ERR_NOT_INIT;
    }
    
    ret = mbtk_imei_get(ql_info_handle, p_info->imei);
    if(ret != 0) 
    {
        LOGE("Error : %d\n", ret);
        return QL_ERR_FAILED;
    } 
    if(strlen(p_info->imei) > 0)
    {
        p_info->imei_valid = IMEI_VALID;
    }


    return QL_ERR_OK;
  
}

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get device firmware revision identification.
  @param[out] firmware_rev_id Return device firmware revision id
  @param[in] firmware_rev_id_len The length of firmware_rev_id
  @return Whether to successfully get the firmware revision id
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_get_device_firmware_rev_id(char *firmware_rev_id, int firmware_rev_id_len)
{
    int ret = -1;
    
    if(ql_info_handle == NULL)
    {
        LOGE("DM no init");
        return QL_ERR_NOT_INIT;
    }
    
    ret = mbtk_get_modem_version(ql_info_handle, (void *)firmware_rev_id);
    //mbtk_v2 do not have function
    if(ret < 0)
    {
        LOGE("get modem version failed");
        return QL_ERR_FAILED;
    }
    return 0;
}

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get air plane mode.
  @param[out] p_info Pointer that point to QL_DM_AIR_PLANE_MODE_TYPE_E
  @return Whether to successfully get the air plane mode
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_get_air_plane_mode(QL_DM_AIR_PLANE_MODE_TYPE_E *p_info)
{
    
    int ret = -1;
    mbtk_radio_state_enum tmp_rf;
    if(ql_info_handle == NULL)
    {
        LOGE("DM no init");
        return QL_ERR_NOT_INIT;
    }
    ret = mbtk_radio_state_get(ql_info_handle, &tmp_rf);
    if (ret != 0)
    {
        LOGE("mbtk_radio_state_get fail.");
        return QL_ERR_FAILED;
    }
    if(tmp_rf == MBTK_RADIO_STATE_FULL_FUNC )
    {
        *p_info = QL_DM_AIR_PLANE_MODE_OFF;
    }
    else if(tmp_rf == MBTK_RADIO_STATE_DIS_RF || tmp_rf == MBTK_RADIO_STATE_MINI_FUNC)
    {
        *p_info = QL_DM_AIR_PLANE_MODE_ON;
    }
        
    return 0;

}

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set air plane mode.
  @param[in] air_plane_mode 1:ON, 2:OFF
  @return Whether to successfully set the air plane mode
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/


int ql_dm_set_air_plane_mode(QL_DM_AIR_PLANE_MODE_TYPE_E air_plane_mode)
{
    //mbtk_radio_state_enum radio = MBTK_RADIO_STATE_MINI_FUNC;
    int reset = 0;
    int rf_mode = -1;
    int ret = -1;
    
    if(ql_info_handle == NULL)
    {
        LOGE("DM no init");
        return QL_ERR_NOT_INIT;
    }
    if(air_plane_mode == QL_DM_AIR_PLANE_MODE_ON)
    {
        rf_mode = MBTK_RADIO_STATE_DIS_RF;
    }
    
    if(air_plane_mode == QL_DM_AIR_PLANE_MODE_OFF)
    {
        rf_mode = LYNQ_AIR_PLANE_MODE_OFF;

    }
    
    if (rf_mode != MBTK_RADIO_STATE_DIS_RF  && rf_mode != MBTK_RADIO_STATE_FULL_FUNC) 
    {
        LOGE("Input mode is error!");
        return QL_ERR_OP_UNSUPPORTED;
    }

    
    ret = mbtk_radio_state_set(ql_info_handle, rf_mode, reset);
    if(ret != 0)
    {
        LOGE("ql_dm_set_air_plane_mode failed");
    }
    mbtk_send_singnal();
    return QL_ERR_OK;


}
/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief register air plane mode event.
  @param[in] cb_func Air plane mode indication callback function
  @return Whether the air plane mode event was successfully registered.
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/

int ql_dm_set_air_plane_mode_ind_cb(ql_dm_air_plane_mode_ind_cb cb_func) 
{
    if(ql_info_handle == NULL)
    {
        LOGE("No init ");
        return QL_ERR_NOT_INIT;
    }
    
    g_air_plane_mode_cb = cb_func;

    if (!g_thread_running) 
    {
        g_thread_running = true;
        if (pthread_create(&g_air_plane_mode_thread, NULL, air_plane_mode_monitor, NULL) != 0) 
        {
            LOGE("Failed to create air plane mode monitor thread");
            g_thread_running = false;
            return QL_ERR_FAILED;
        }
    }
    
    return QL_ERR_OK;
}


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get cpu occupancy.
  @param[out] cpu_occupancy The percentage of cpu occupancy
  @return Whether to successfully get the cpu occupancy
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_get_cpu_occupancy(float *cpu_occupancy);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get mem usage.
  @param[out] mem_use The percentage of mem usage
  @return Whether to successfully get the memory usage
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_get_mem_usage(float *mem_use);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get NV item value.
  @param[in] nv_item_name The NV item name that is either NV item id or NV item path
  @param[out] nv_item_value The NV value buf of nv_item_name
  param[in] nv_item_value_len The length of nv_item_value
  param[out] nv_len The real length of nv_item_name
  @return Whether to successfully get the NV value
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_get_nv_item_value(char *nv_item_name, unsigned char *nv_item_value, int nv_item_value_len, 
        int *nv_len);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set NV item value.
  @param[in] nv_item_name The NV item name that is either NV item id or NV item path
  @param[in] nv_item_value The NV value of nv_item_name
  @param[in] nv_item_value_len The length of nv_item_value
  param[out] nv_len The real length of nv_item_name
  @return Whether to successfully set the NV value
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_set_nv_item_value(char *nv_item_name, unsigned char *nv_item_value, int nv_item_value_len, 
        int *nv_len);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set radio on, its function is the same as at+cfun=1.
  @return Whether to successfully set the radio on
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_set_radio_on(void);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set radio off, its function is the same as at+cfun=0.
  @return Whether to successfully set the radio off
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_set_radio_off(void);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get modem mem and CPU utilization.
  @param[out] mem_use The percentage of modem utilization
  @return Whether to successfully get the modem utilization
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_get_modem_cpu_occupancy(float *cpu_occupancy);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get modem mem utilization.
  @param[out] mem_use The percentage of modem utilization
  @return Whether to successfully get the modem utilization
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_get_modem_mem_usage(float *mem_use);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get QOOS enable state
  @param[out] enable The enable state of QOOS
  @return Whether to successfully get the QOOS enable state
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_get_qoos_enable(char *enable);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set QOOS enable state
  @param[in] enable The enable state of QOOS
  @return Whether to successfully set the QOOS enable state
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_set_qoos_enable(char enable);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get QOOS configuration
  @param[out] config The configuration of QOOS
  @return Whether to successfully get the QOOS configuration
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
//int ql_dm_get_qoos_config(ql_dm_qoos_config_t *config);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set QOOS configuration
  @param[in] config The configuration of QOOS
  @return Whether to successfully set the QOOS configuration
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
//int ql_dm_set_qoos_config(ql_dm_qoos_config_t config);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get MSSR(Modem SubSysem Reset) level.
  @param[out] p_level The MSSR level
  @return Whether to successfully get the MSSR level
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_get_mssr_level(int *p_level);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set MSSR(Modem SubSysem Reset) level.
  @param[in] level The MSSR level
  @return Whether to successfully set the MSSR level
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_set_mssr_level(int level);


/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  bind subscription 
  @param[in] sub_type subscription type 
  @return Whether to successfully bind subscription.
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_bind_subscription(QL_DM_BIND_SUB_TYPE_E sub_type);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Registration server error callback. Currently, only if the server exits abnormally, 
  the callback function will be executed, and the error code is QL_ERR_ABORTED;
  @param[in] cb  Callback function 
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_set_service_error_cb(ql_dm_service_error_cb_f cb)
{
    int ret = -1;
    if(ql_info_handle == NULL)
    {
        LOGE("DM no init ");
        return QL_ERR_NOT_INIT;

    }
    global_dm_error_cb = cb;
    ret = mbtk_ril_ser_state_change_cb_reg(mbtk_dm_set_service_error_func);
    if(ret != 0)
    {
        LOGE("call mbtk_ril_server_state_change_reg failed");
        return QL_ERR_FAILED;
    }
    
    return QL_ERR_OK;
}
/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Get module the last time shutdown reason 
  @param[out] shutdown_reason the shutdown reason
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_get_shutdown_reason(QL_DM_SHUTDOWN_REASON_E *shutdown_reason);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Get module this time bootup reason 
  @param[out] bootup_reason the bootup reason
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_get_bootup_reason(QL_DM_BOOT_UP_REASON_E *bootup_reason);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief set oos config 
  @param[out] oos param
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_set_qoos_config(int p1, int p2, int p3);


/*-----------------------------------------------------------------------------------------------*/
/**
  @brief get oos config 
  @param[out] oos param
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_get_qoos_config(int *p1, int *p2, int *p3);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief set oos enable 
  @param[out] oos param
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_dm_set_qoos_enable(char enable);
#ifdef __cplusplus
}
#endif



