#include <stdio.h>

#include "ql_ecall.h"
#include "mbtk_ril_api.h"

typedef enum {
    ECALL_MSD_VEHICLE_TYPE_M1 = 1,  // 1
    ECALL_MSD_VEHICLE_TYPE_M2,
    ECALL_MSD_VEHICLE_TYPE_M3,
    ECALL_MSD_VEHICLE_TYPE_N1,
    ECALL_MSD_VEHICLE_TYPE_N2,      // 5
    ECALL_MSD_VEHICLE_TYPE_N3,
    ECALL_MSD_VEHICLE_TYPE_L1E,
    ECALL_MSD_VEHICLE_TYPE_L2E,
    ECALL_MSD_VEHICLE_TYPE_L3E,
    ECALL_MSD_VEHICLE_TYPE_L4E,     // 10
    ECALL_MSD_VEHICLE_TYPE_L5E,
    ECALL_MSD_VEHICLE_TYPE_L6E,
    ECALL_MSD_VEHICLE_TYPE_L7E
} ecall_msd_vehicle_type_enum;

typedef enum {
    ECALL_MSD_PROPULSIONTYPE_GASOLINE = 0x1,
    ECALL_MSD_PROPULSIONTYPE_DIESEL = 0x2,
    ECALL_MSD_PROPULSIONTYPE_COMPRESSED = 0x4,
    ECALL_MSD_PROPULSIONTYPE_LIQUID = 0x8,
    ECALL_MSD_PROPULSIONTYPE_ELECTRIC = 0x10,
    ECALL_MSD_PROPULSIONTYPE_HYDROGEN = 0x20,
    ECALL_MSD_PROPULSIONTYPE_OTHER = 0x40
} ecall_msd_PropulsionType_enum;

typedef enum {
    ECALL_SIM_CARD_ID_NON = 0,
    ECALL_SIM_CARD_ID_1,
    ECALL_SIM_CARD_ID_2,
    ECALL_SIM_CARD_ID_MAX
} ecall_sim_card_id_enum;

typedef struct {
    bool msd_version_valid;
    uint8_t msd_version;

    bool test_call_valid;
    bool test_call;

    bool vehicle_type_valid;
    ecall_msd_vehicle_type_enum vehicle_type;

    bool isTrusted_valid;
    bool isTrusted;

    bool automaticActivation_valid;
    QL_ECALL_MSD_TX_MODE_E automaticActivation;

    bool latitude_valid;
    int32_t latitude;

    bool longitude_valid;
    int32_t longitude;

    bool direction_valid;
    int32_t direction;

    bool PropulsionType_valid;
    ecall_msd_PropulsionType_enum PropulsionType;

    bool passengers_valid;
    uint8_t passengers;

    bool vin_valid;
    msd_Vin_t vin;

    bool ecall_cb_valid;
    ql_ecall_user_ind_f ecall_cb;
} ecall_tmp_conf_info_t;

static mbtk_ril_handle *ecall_handle = NULL;
static ecall_tmp_conf_info_t tmp_conf;

static int ecall_msd_ControlType_set()
{
     if(tmp_conf.test_call_valid && tmp_conf.isTrusted_valid
        && tmp_conf.vehicle_type_valid && tmp_conf.automaticActivation_valid) {
        mbtk_ecall_msd_cfg_info_t cfg_info;
        memset(&cfg_info, 0, sizeof(mbtk_ecall_msd_cfg_info_t));
        cfg_info.item_type = MBTK_ECALL_MSD_ITEM_CONTROLTYPE;
        // automaticActivation,testCall,positionCanBeTrustd,vehicleType
        sprintf((char*)cfg_info.data, "%d,%d,%d,%d",
            tmp_conf.automaticActivation,  // PUSH is auto ???
            tmp_conf.test_call ? 1 : 0,
            tmp_conf.isTrusted ? 1 : 0, tmp_conf.vehicle_type);
        if(mbtk_ecall_msd_item_set(ecall_handle, &cfg_info)) {
            LOGE("mbtk_ecall_msd_item_set(ControlType) fail.");
            return -1;
        } else {
            return 0;
        }
    } else {
        LOGD("ControlType not set complete, so not set.");
        return 0;
    }
}

static void ecall_state_change_cb(const void* data, int data_len)
{
    if(data)
    {
        mbtk_ril_ecall_state_info_t *ecall_data = (mbtk_ril_ecall_state_info_t*)data;
        QL_ECALL_EVENT_E event;
        switch(ecall_data->urc_id)
        {
            case MBTK_ECALL_URC_ID_MSD_SEND_START:
            {
                event = QL_ECALL_EVENT_SENDING_START;
                break;
            }
            case MBTK_ECALL_URC_ID_NACK:
            {
                event = QL_ECALL_EVENT_SENDING_MSD;
                break;
            }
            case MBTK_ECALL_URC_ID_LLACK:
            {
                event = QL_ECALL_EVENT_LLACK_RECEIVED;
                break;
            }
            case MBTK_ECALL_URC_ID_ALACK:
            {
                int urc_data = *((int *)(ecall_data->urc_data + 1));
                if((urc_data & 0x02) == 2) { // cleardown
                    event = QL_ECALL_EVENT_ALLACK_CLEARDOWN_RECEIVED;
                } else {
                    event = QL_ECALL_EVENT_ALLACK_POSITIVE_RECEIVED;
                }
                break;
            }
            case MBTK_ECALL_URC_ID_T5_EXPIRED:
            {
                event = QL_ECALL_EVENT_T5_TIMEOUT;
                break;
            }
            case MBTK_ECALL_URC_ID_T6_EXPIRED:
            {
                event = QL_ECALL_EVENT_T6_TIMEOUT;
                break;
            }
            case MBTK_ECALL_URC_ID_T7_EXPIRED:
            {
                event = QL_ECALL_EVENT_T7_TIMEOUT;
                break;
            }
            default:
            {
                LOGW(
"No process urc_id : %d", ecall_data->urc_id);
                return;
            }
        }

        if(tmp_conf.ecall_cb_valid && tmp_conf.ecall_cb) {
            tmp_conf.ecall_cb(event, NULL);
        }
    }
}

static int ecall_start_reconfig(int sim_id)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    if(sim_id >= ECALL_SIM_CARD_ID_MAX || sim_id < ECALL_SIM_CARD_ID_NON) {
        return QL_ERR_INVALID_ARG;
    }

    if(mbtk_ecall_dial_start(ecall_handle, MBTK_ECALL_DIAL_TYPE_RECONFIG)) {
        LOGE("mbtk_ecall_dial_start(RECONFIG) fail.");
        return QL_ERR_GENERIC;
    } else {
        return QL_ERR_OK;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Set the system standard.
  @param[in] system_std system standard which can be set to PAN-European or ERA-GLONASS.
  @return Whether the system standard was set successfully.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG unsupported value.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_set_system_std(ECALL_SYSTEM_STD_E system_std)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    mbtk_ecall_mode_type_enum mode;
    if(system_std == ECALL_SYSTEM_STD_PAN_EUROPEAN) {
        mode = MBTK_ECALL_MODE_TYPE_EU;
    } else if(system_std == ECALL_SYSTEM_STD_ERA_GLONASS) {
        mode = MBTK_ECALL_MODE_TYPE_ERA;
    } else {
        LOGE("Unknown system_std : %d", system_std);
        return QL_ERR_INVALID_ARG;
    }

    if(mbtk_ecall_mode_set(ecall_handle, mode)) {
        LOGE("mbtk_ecall_mode_set() fail.");
        return QL_ERR_UNKNOWN;
    } else {
        return QL_ERR_OK;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Get the system standard.
  @param[out] system_std pointer to system standard which can be PAN-European or ERA-GLONASS.
  */
/*-----------------------------------------------------------------------------------------------*/
void ql_ecall_get_system_std(ECALL_SYSTEM_STD_E* system_std)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return;
    }

    if(system_std == NULL) {
        return;
    }

    mbtk_ecall_mode_type_enum mode;
    if(mbtk_ecall_mode_get(ecall_handle, &mode)) {
        LOGE("mbtk_ecall_mode_get() fail.");
    } else {
        if(mode == MBTK_ECALL_MODE_TYPE_EU) {
            *system_std = ECALL_SYSTEM_STD_PAN_EUROPEAN;
        } else {
            *system_std = ECALL_SYSTEM_STD_ERA_GLONASS;
        }
    }
}


/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Set the MSD version.
  @param[in] msd_version MSD version which can be set to ASN1_ECALL_MSD_VERSION_1 or ASN1_ECALL_MSD_VERSION_2
  @return Whether the MSD version was set successfully.
  @retval QL_ERR_MSD_SUCCESS successful.
  @retval QL_ERR_MSD_VERSION unsupported value.
  */
/*-----------------------------------------------------------------------------------------------*/
QL_ERR_MSD_RESULT_E ql_ecall_set_msd_version(uint8_t msd_version)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_MSD_GENERIC;
    }

    if(msd_version != ASN1_ECALL_MSD_VERSION_2
        && msd_version != ASN1_ECALL_MSD_VERSION_3) {
        LOGE("Unsupport MSD version:%d", msd_version);
        return QL_ERR_MSD_VERSION;
    }

    if(tmp_conf.msd_version_valid && tmp_conf.msd_version == msd_version) {
        LOGD("msd_version not change, do nothing...");
        return QL_ERR_MSD_GENERIC;
    }

    mbtk_ecall_msd_cfg_info_t cfg_info;
    memset(&cfg_info, 0, sizeof(mbtk_ecall_msd_cfg_info_t));
    cfg_info.item_type = MBTK_ECALL_MSD_ITEM_VERSION;
    sprintf((char*)cfg_info.data, "%d", msd_version);

    if(mbtk_ecall_msd_item_set(ecall_handle, &cfg_info)) {
        LOGE("mbtk_ecall_msd_item_set() fail.");
        return QL_ERR_MSD_GENERIC;
    } else {
        tmp_conf.msd_version_valid = TRUE;
        tmp_conf.msd_version = msd_version;
        return QL_ERR_MSD_SUCCESS;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Get the MSD version.
  @param[out] msd_version pointer to MSD version.
  @return Whether the MSD version was get successfully.
  @retval QL_ERR_MSD_SUCCESS successful.
  @retval QL_ERR_MSD_VERSION unsupported value.
  */
/*-----------------------------------------------------------------------------------------------*/
QL_ERR_MSD_RESULT_E ql_ecall_get_msd_version(uint8_t* msdVersionPtr)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_MSD_GENERIC;
    }

    if(msdVersionPtr == NULL) {
        return QL_ERR_MSD_GENERIC;
    }

    if(tmp_conf.msd_version_valid) {
        *msdVersionPtr = tmp_conf.msd_version;
    } else {
        *msdVersionPtr = (uint8_t)ASN1_ECALL_MSD_VERSION_2;     // Default is 2
    }
    return QL_ERR_MSD_SUCCESS;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Sets MSD's testCall field.
  @param[in] testCall MSD call type: true = Test call false = Emergency
  */
/*-----------------------------------------------------------------------------------------------*/
void ql_ecall_set_msd_call_type(bool testCall)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return;
    }

    if(tmp_conf.test_call_valid && tmp_conf.test_call == testCall) {
        LOGD("testCall not change, do nothing...");
        return;
    }

    tmp_conf.test_call_valid = TRUE;
    tmp_conf.test_call = testCall;

    ecall_msd_ControlType_set();
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Gets MSD's testCall field.
  @param[out] testCall pointer to MSD call type: true = Test call false = Emergency
  */
/*-----------------------------------------------------------------------------------------------*/
void ql_ecall_get_msd_call_type(bool* testCall)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return;
    }

    if(testCall == NULL) {
        return;
    }

    if(tmp_conf.test_call_valid) {
        *testCall = tmp_conf.test_call;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Sets VehicleType.
  @param[in] vehicleType vehicle type
  @return Whether the vehicleType was set successfully.
  @retval QL_ERR_MSD_SUCCESS successful.
  @retval QL_ERR_MSD_BADPARM unsupported value.
  */
/*-----------------------------------------------------------------------------------------------*/
QL_ERR_MSD_RESULT_E ql_ecall_set_msd_vehicle_type(uint8_t vehicleType)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_MSD_GENERIC;
    }

    if(tmp_conf.vehicle_type_valid && tmp_conf.vehicle_type == vehicleType) {
        LOGD("vehicleType not change, do nothing...");
        return QL_ERR_MSD_GENERIC;
    }

    tmp_conf.vehicle_type_valid = TRUE;
    tmp_conf.vehicle_type = (ecall_msd_vehicle_type_enum)vehicleType;

    if(ecall_msd_ControlType_set()) {
        return QL_ERR_MSD_GENERIC;
    } else {
        return QL_ERR_MSD_SUCCESS;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Gets VehicleType.
  @param[out] vehicleType pointer to vehicle type
  @return Whether the vehicleType was get successfully.
  @retval QL_ERR_MSD_SUCCESS successful.
  @retval QL_ERR_MSD_BADPARM unsupported value.
  */
/*-----------------------------------------------------------------------------------------------*/
QL_ERR_MSD_RESULT_E ql_ecall_get_msd_vehicle_type(uint8_t* vehicleType)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_MSD_GENERIC;
    }

    if(vehicleType == NULL) {
        return QL_ERR_MSD_GENERIC;
    }

    if(tmp_conf.vehicle_type_valid) {
        *vehicleType = (uint8_t)tmp_conf.vehicle_type;
        return QL_ERR_MSD_SUCCESS;
    } else {
        return QL_ERR_MSD_GENERIC;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Sets Vehicle Identification Number
  @param[in] vin vehicle identification number
  @return Whether the vehicle identification number was set successfully.
  @retval QL_ERR_MSD_SUCCESS successful.
  @retval QL_ERR_MSD_VEH_IDENTIFI_NUMBER unsupported value.
  */
/*-----------------------------------------------------------------------------------------------*/
QL_ERR_MSD_RESULT_E ql_ecall_set_msd_vin(msd_Vin_t vin)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_MSD_GENERIC;
    }

    tmp_conf.vin_valid = TRUE;
    memcpy(&(tmp_conf.vin), &vin, sizeof(msd_Vin_t));

    mbtk_ecall_msd_cfg_info_t cfg_info;
    memset(&cfg_info, 0, sizeof(mbtk_ecall_msd_cfg_info_t));
    cfg_info.item_type = MBTK_ECALL_MSD_ITEM_VIN;
    sprintf((char*)cfg_info.data, "%s,%s,%s,%s", vin.isowmi, vin.isovds,
        vin.isovisModelyear, vin.isovisSeqPlant);
    if(mbtk_ecall_msd_item_set(ecall_handle, &cfg_info)) {
        LOGE("mbtk_ecall_msd_item_set(VIN) fail.");
        return QL_ERR_MSD_GENERIC;
    } else {
        return QL_ERR_MSD_SUCCESS;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Gets Vehicle Identification Number
  @param[out] vin pointer to vehicle identification number
  @return Whether the vehicle identification number was get successfully.
  @retval QL_ERR_MSD_SUCCESS successful.
  @retval QL_ERR_MSD_VEH_IDENTIFI_NUMBER unsupported value.
  */
/*-----------------------------------------------------------------------------------------------*/
QL_ERR_MSD_RESULT_E ql_ecall_get_msd_vin(msd_Vin_t* vin)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_MSD_GENERIC;
    }

    if(vin == NULL) {
        return QL_ERR_MSD_GENERIC;
    }

    if(tmp_conf.vin_valid) {
        memcpy(vin, &(tmp_conf.vin), sizeof(msd_Vin_t));
        return QL_ERR_MSD_SUCCESS;
    } else {
        return QL_ERR_MSD_GENERIC;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Sets Propulsion Type.
  @param[in] PropulsionType propulsion type.
  */
/*-----------------------------------------------------------------------------------------------*/
void ql_ecall_set_msd_propulsion_type(uint8_t PropulsionType)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return;
    }

    if(tmp_conf.PropulsionType_valid && tmp_conf.PropulsionType == PropulsionType) {
        LOGD("PropulsionType not change, do nothing...");
        return;
    }

    tmp_conf.PropulsionType_valid = TRUE;
    tmp_conf.PropulsionType = (ecall_msd_PropulsionType_enum)PropulsionType;

    mbtk_ecall_msd_cfg_info_t cfg_info;
    memset(&cfg_info, 0, sizeof(mbtk_ecall_msd_cfg_info_t));
    cfg_info.item_type = MBTK_ECALL_MSD_ITEM_STORAGETYPE;
    sprintf((char*)cfg_info.data, "%d,%d,%d,%d,%d,%d,%d",
        (tmp_conf.PropulsionType & ECALL_MSD_PROPULSIONTYPE_GASOLINE) ? 1 : 0,
        (tmp_conf.PropulsionType & ECALL_MSD_PROPULSIONTYPE_DIESEL) ? 1 : 0,
        (tmp_conf.PropulsionType & ECALL_MSD_PROPULSIONTYPE_COMPRESSED) ? 1 : 0,
        (tmp_conf.PropulsionType & ECALL_MSD_PROPULSIONTYPE_LIQUID) ? 1 : 0,
        (tmp_conf.PropulsionType & ECALL_MSD_PROPULSIONTYPE_ELECTRIC) ? 1 : 0,
        (tmp_conf.PropulsionType & ECALL_MSD_PROPULSIONTYPE_HYDROGEN) ? 1 : 0,
        (tmp_conf.PropulsionType & ECALL_MSD_PROPULSIONTYPE_OTHER) ? 1 : 0);
    if(mbtk_ecall_msd_item_set(ecall_handle, &cfg_info)) {
        LOGE("mbtk_ecall_msd_item_set(vehiclePropulsionStorageType) fail.");
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Gets Propulsion Type.
  @param[out] PropulsionType pointer to propulsion type.
  */
/*-----------------------------------------------------------------------------------------------*/
void ql_ecall_get_msd_propulsion_type(uint8_t* PropulsionType)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return;
    }

    if(PropulsionType == NULL) {
        return;
    }

    if(tmp_conf.PropulsionType_valid) {
        *PropulsionType = (uint8_t)(tmp_conf.PropulsionType);
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Sets number of passangers.
  @param[in] numberOfPassengers number of passangers.
  */
/*-----------------------------------------------------------------------------------------------*/
void ql_ecall_set_msd_passengers_count(uint8_t numberOfPassengers)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return;
    }

    if(tmp_conf.passengers_valid && tmp_conf.passengers == numberOfPassengers) {
        LOGD("numberOfPassengers not change, do nothing...");
        return;
    }

    tmp_conf.passengers_valid = TRUE;
    tmp_conf.passengers = numberOfPassengers;

    mbtk_ecall_msd_cfg_info_t cfg_info;
    memset(&cfg_info, 0, sizeof(mbtk_ecall_msd_cfg_info_t));
    cfg_info.item_type = MBTK_ECALL_MSD_ITEM_NUMBEROFOCCUPANTS;
    sprintf((char*)cfg_info.data, "%d", tmp_conf.passengers);
    if(mbtk_ecall_msd_item_set(ecall_handle, &cfg_info)) {
        LOGE("mbtk_ecall_msd_item_set(numberofoccupants) fail.");
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Sets vehicle location.
  @param[in] isTrusted true if the position can be trusted, false otherwise.
  @param[in] latitude latitude of vehicle's location.
  @param[in] longitude longitude of vehicle's location.
  @param[in] direction direction of vehicle's location.
  @return Whether the vehicle location was set successfully.
  @retval QL_ERR_MSD_SUCCESS successful.
  @retval QL_ERR_MSD_LATITUDE latitude unsupported value.
  @retval QL_ERR_MSD_LONGITUDE longitude unsupported value.
  @retval QL_ERR_MSD_VEH_DIRECTION direction unsupported value.
  */
/*-----------------------------------------------------------------------------------------------*/
QL_ERR_MSD_RESULT_E ql_ecall_set_msd_position(bool isTrusted, int32_t latitude, int32_t longitude, int32_t direction)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_MSD_GENERIC;
    }

    tmp_conf.isTrusted_valid = TRUE;
    tmp_conf.isTrusted = isTrusted;

    if(ecall_msd_ControlType_set()) {
        return QL_ERR_MSD_GENERIC;
    } else {
        tmp_conf.latitude_valid = TRUE;
        tmp_conf.latitude = latitude;
        tmp_conf.longitude_valid = TRUE;
        tmp_conf.longitude = longitude;

        mbtk_ecall_msd_cfg_info_t cfg_info;
        memset(&cfg_info, 0, sizeof(mbtk_ecall_msd_cfg_info_t));
        cfg_info.item_type = MBTK_ECALL_MSD_ITEM_LOCATION;
        // latitude,longitude
        sprintf((char*)cfg_info.data, "%d,%d", tmp_conf.latitude, tmp_conf.longitude);
        if(mbtk_ecall_msd_item_set(ecall_handle, &cfg_info)) {
            LOGE("mbtk_ecall_msd_item_set(vehicleLocation) fail.");
            return QL_ERR_MSD_GENERIC;
        } else {
            tmp_conf.direction_valid = TRUE;
            tmp_conf.direction = direction;

            memset(&cfg_info, 0, sizeof(mbtk_ecall_msd_cfg_info_t));
            cfg_info.item_type = MBTK_ECALL_MSD_ITEM_DIRECTION;
            // direction
            sprintf((char*)cfg_info.data, "%d", tmp_conf.direction / 2);  // Vehicle direction: [0, 179] and 255, unit: 2 degrees
            if(mbtk_ecall_msd_item_set(ecall_handle, &cfg_info)) {
                LOGE("mbtk_ecall_msd_item_set(vehicleDirection) fail.");
                return QL_ERR_MSD_GENERIC;
            } else {
                return QL_ERR_MSD_SUCCESS;
            }
        }
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Set the position Delta N-1 from position set in ql_ecall_set_msd_position() transmitted by the MSD.
  @param[in] latitudeN2 latitude delta from position set in ql_ecall_set_msd_position().
  @param[in] longitudeN2 longitude delta from position set in ql_ecall_set_msd_position().
  @return Whether the vehicle location delta was set successfully.
  @retval QL_ERR_MSD_SUCCESS successful.
  @retval QL_ERR_MSD_LATITUDE delta latitude unsupported value.
  @retval QL_ERR_MSD_LONGITUDE delta longitude unsupported value.
  */
/*-----------------------------------------------------------------------------------------------*/
QL_ERR_MSD_RESULT_E ql_ecall_set_msd_position_n1(int32_t latitudeDeltaN1, int32_t longitudeDeltaN1)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_MSD_GENERIC;
    }

    mbtk_ecall_msd_cfg_info_t cfg_info;
    memset(&cfg_info, 0, sizeof(mbtk_ecall_msd_cfg_info_t));
    cfg_info.item_type = MBTK_ECALL_MSD_ITEM_LOCATIONN1;
    sprintf((char*)cfg_info.data, "%d,%d", latitudeDeltaN1, longitudeDeltaN1);
    if(mbtk_ecall_msd_item_set(ecall_handle, &cfg_info)) {
        LOGE("mbtk_ecall_msd_item_set(recentVehicleLocationN1) fail.");
        return QL_ERR_MSD_GENERIC;
    } else {
        return QL_ERR_MSD_SUCCESS;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Set the position Delta N-2 from position set in ql_ecall_set_msd_position() transmitted by the MSD.
  @param[in] latitudeN2 latitude delta from position set in ql_ecall_set_msd_position_n1().
  @param[in] longitudeN2 longitude delta from position set in ql_ecall_set_msd_position_n1().
  @return Whether the vehicle location delta was set successfully.
  @retval QL_ERR_MSD_SUCCESS successful.
  @retval QL_ERR_MSD_LATITUDE delta latitude unsupported value.
  @retval QL_ERR_MSD_LONGITUDE delta longitude unsupported value.
  */
/*-----------------------------------------------------------------------------------------------*/
QL_ERR_MSD_RESULT_E ql_ecall_set_msd_position_n2(int32_t latitudeDeltaN2, int32_t longitudeDeltaN2)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_MSD_GENERIC;
    }

    mbtk_ecall_msd_cfg_info_t cfg_info;
    memset(&cfg_info, 0, sizeof(mbtk_ecall_msd_cfg_info_t));
    cfg_info.item_type = MBTK_ECALL_MSD_ITEM_LOCATIONN2;
    sprintf((char*)cfg_info.data, "%d,%d", latitudeDeltaN2, longitudeDeltaN2);
    if(mbtk_ecall_msd_item_set(ecall_handle, &cfg_info)) {
        LOGE("mbtk_ecall_msd_item_set(recentVehicleLocationN2) fail.");
        return QL_ERR_MSD_GENERIC;
    } else {
        return QL_ERR_MSD_SUCCESS;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Set MSD TX Mode.
  @param[in] tx_mode Transmission MSD mode, 0 - PULL, 1 - PUSH.
  @return Whether the transmission MSD mode was set successfully.
  @retval QL_ERR_MSD_SUCCESS successful.
  @retval QL_ERR_MSD_BADPARM unsupported value.
  */
/*-----------------------------------------------------------------------------------------------*/
QL_ERR_MSD_RESULT_E ql_ecall_set_msd_tx_mode(QL_ECALL_MSD_TX_MODE_E tx_mode)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_MSD_GENERIC;
    }

    if(tmp_conf.automaticActivation_valid && tmp_conf.automaticActivation == tx_mode) {
        LOGD("automaticActivation not change, do nothing...");
        return QL_ERR_MSD_GENERIC;
    }

    tmp_conf.automaticActivation_valid = TRUE;
    tmp_conf.automaticActivation = tx_mode;

    if(ecall_msd_ControlType_set()) {
        return QL_ERR_MSD_GENERIC;
    } else {
        return QL_ERR_MSD_SUCCESS;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Get MSD TX Mode.
  @param[out] tx_mode Pointer to transmission MSD mode, 0 - PULL, 1 - PUSH.
  */
/*-----------------------------------------------------------------------------------------------*/
void ql_ecall_get_msd_tx_mode(QL_ECALL_MSD_TX_MODE_E* tx_mode)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return;
    }

    if(tx_mode == NULL) {
        return;
    }

    if(tmp_conf.automaticActivation_valid) {
        *tx_mode = tmp_conf.automaticActivation;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Set the minimum interval value between dial attempts. Available for both manual and test modes.
  @param[in] pause the minimum interval value in seconds.
  @retval ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_set_interval_between_dial_attempts(uint16_t pause)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    mbtk_ecall_cfg_info_t cfg;
    memset(&cfg, 0, sizeof(cfg));
    cfg.type |= MBTK_ECALL_CFG_TIMER_DIAL;
    cfg.data[MBTK_ECALL_CFG_ITEM_TIMER_DIAL] = (uint32)(pause * 1000);
    if(mbtk_ecall_cfg_set(ecall_handle, &cfg)) {
        LOGE("mbtk_ecall_cfg_set(TIMER_DIAL) fail.");
        return QL_ERR_GENERIC;
    } else {
        return QL_ERR_OK;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Get the minimum interval value between dial attempts.
  @param[out] pause pointer to the minimum interval value in seconds.
  */
/*-----------------------------------------------------------------------------------------------*/
void ql_ecall_get_interval_between_dial_attempts(uint16_t* pause)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return;
    }

    if(pause == NULL) {
        return;
    }

    mbtk_ecall_cfg_info_t cfg;
    memset(&cfg, 0, sizeof(cfg));
    cfg.type |= MBTK_ECALL_CFG_TIMER_DIAL;
    if(mbtk_ecall_cfg_get(ecall_handle, &cfg)) {
        LOGE("mbtk_ecall_cfg_get(TIMER_DIAL) fail.");
    } else {
        if(cfg.data_valid[MBTK_ECALL_CFG_ITEM_TIMER_DIAL]) {
            *pause = (uint16_t)(cfg.data[MBTK_ECALL_CFG_ITEM_TIMER_DIAL] / 1000);
        }
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Initialize MSD to default values.
  */
/*-----------------------------------------------------------------------------------------------*/
void ql_ecall_msd_init(void)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return;
    }

    memset(&tmp_conf, 0, sizeof(ecall_tmp_conf_info_t));
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Start a test eCall session.
  @return Whether the test ecall was started successfully.
  @retval QL_ERR_MSD_SUCCESS successful.
  @retval QL_ERR_MSD_BADPARM unsupported value.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_start_test(int sim_id)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    if(sim_id >= ECALL_SIM_CARD_ID_MAX || sim_id < ECALL_SIM_CARD_ID_NON) {
        return QL_ERR_INVALID_ARG;
    }

    if(mbtk_ecall_dial_start(ecall_handle, MBTK_ECALL_DIAL_TYPE_TEST)) {
        LOGE("mbtk_ecall_dial_start(TEST) fail.");
        return QL_ERR_GENERIC;
    } else {
        return QL_ERR_OK;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Start a manual eCall session.
  @return Whether the manually triggered ecall was started successfully.
  @retval QL_ERR_MSD_SUCCESS successful.
  @retval QL_ERR_MSD_BADPARM unsupported value.
  @retval Other MSD error code defined by QL_ERR_MSD_RESULT_E.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_start_manual(int sim_id)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    if(sim_id >= ECALL_SIM_CARD_ID_MAX || sim_id < ECALL_SIM_CARD_ID_NON) {
        return QL_ERR_INVALID_ARG;
    }

    if(mbtk_ecall_dial_start(ecall_handle, MBTK_ECALL_DIAL_TYPE_MANUALLY)) {
        LOGE("mbtk_ecall_dial_start(MANUALLY) fail.");
        return QL_ERR_GENERIC;
    } else {
        return QL_ERR_OK;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Start an automatic eCall session.
  @return Whether the automatically triggered ecall was started successfully.
  @retval QL_ERR_MSD_SUCCESS successful.
  @retval QL_ERR_MSD_BADPARM unsupported value.
  @retval Other MSD error code defined by QL_ERR_MSD_RESULT_E.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_start_automatic(int sim_id)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    if(sim_id >= ECALL_SIM_CARD_ID_MAX || sim_id < ECALL_SIM_CARD_ID_NON) {
        return QL_ERR_INVALID_ARG;
    }

    if(mbtk_ecall_dial_start(ecall_handle, MBTK_ECALL_DIAL_TYPE_AUTO)) {
        LOGE("mbtk_ecall_dial_start(AUTO) fail.");
        return QL_ERR_GENERIC;
    } else {
        return QL_ERR_OK;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Initializes ecall service.
  @return Whether the ecall service was initialized successfully.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_init(void)
{
    if(ecall_handle == NULL) {
        ecall_handle = mbtk_ril_open(MBTK_AT_PORT_DEF);
        if(ecall_handle) {
            return QL_ERR_OK;
        } else {
            return QL_ERR_INTERNAL;
        }
    } else {
        return QL_ERR_GENERIC;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Deinitializes ecall service.
  @return Whether the ecall service was deinitialized successfully.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_deinit(void)
{
    if(ecall_handle) {
        if(mbtk_ril_close(MBTK_AT_PORT_DEF)) {
            return QL_ERR_INTERNAL;
        } else {
            ecall_handle = NULL;
            return QL_ERR_OK;
        }
    } else {
        return QL_ERR_GENERIC;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Sets ecall User indication callback.
  @param[in] cb call back handler.
  @param[in] userdata user data.
  @return void
  */
/*-----------------------------------------------------------------------------------------------*/
void ql_ecall_set_user_ind_cb(ql_ecall_user_ind_f cb, void *userdata)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return;
    }

    if(cb == NULL) {
        return;
    }

    if(mbtk_ecall_state_change_cb_reg(ecall_state_change_cb)) {
        LOGE("mbtk_ecall_state_change_cb_reg() fail.");
    } else {
        tmp_conf.ecall_cb_valid = TRUE;
        tmp_conf.ecall_cb = cb;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Dials eCall.
  @param[in]  sim_id sim id.
  @param[in]  p_info eCall info.
  @param[out] p_id   call id.
  @return Whether a eCall was successfully dialed.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_dial(int sim_id, const ql_voice_ecall_info_t* const p_info, uint32_t* const p_id)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    if(sim_id >= ECALL_SIM_CARD_ID_MAX || sim_id < ECALL_SIM_CARD_ID_NON
        || p_info == NULL) {
        return QL_ERR_INVALID_ARG;
    }

    int ret;
    if(p_info->msd_len > 0) {
        ret = ql_ecall_update_msd_raw(p_info->msd, p_info->msd_len);
        if(QL_ERR_OK != ret) {
            return ret;
        }
    }

    if(strlen(p_info->test_number) > 0) {
        ret = ql_ecall_set_test_number(ECALL_SIM_CARD_ID_1, p_info->test_number);
        if(QL_ERR_OK != ret) {
            return ret;
        }
    }

    if(p_info->type == QL_VOICE_ECALL_TYPE_TEST) {
        return ql_ecall_start_test(sim_id);
    } else if(p_info->type == QL_VOICE_ECALL_TYPE_RECONFIG) {
        return ecall_start_reconfig(sim_id);
    } else {
        if(p_info->auto_trigger) {
            return ql_ecall_start_automatic(sim_id);
        } else {
            return ql_ecall_start_manual(sim_id);
        }
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Hangs up eCall.
  @return Whether the eCall was successfully hung up.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_hangup(void)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    if(mbtk_ecall_dial_start(ecall_handle, MBTK_ECALL_DIAL_TYPE_RELEASE)) {
        LOGE("mbtk_ecall_dial_start(RELEASE) fail.");
        return QL_ERR_GENERIC;
    } else {
        return QL_ERR_OK;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Updates eCall MSD with raw data.
  @param[in] msd Minimum Set of Data.
  @param[in] msd_len Length of Minimum Set of Data.
  @return Whether the eCall MSD was successfully updated.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_update_msd_raw(const uint8_t* msd, uint32_t msd_len)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    if(msd == NULL || msd_len == 0) {
        return QL_ERR_INVALID_ARG;
    }

    char msd_data[500] = {0};
    int i = 0;
    while(i < msd_len) {
        sprintf(msd_data + i * 2, "%02x", *(msd + i));
        i++;
    }

    log_hex("MSD", msd, msd_len);
    LOGD("MSD[%d] : %s", strlen(msd_data), msd_data);

    if(mbtk_ecall_msd_set(ecall_handle, msd_data)) {
        LOGE("mbtk_ecall_msd_set() fail.");
        return QL_ERR_GENERIC;
    } else {
        return QL_ERR_OK;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Updates eCall MSD.
  @return Whether the eCall MSD was successfully updated.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_update_msd(void)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    if(mbtk_ecall_msd_gen(ecall_handle)) {
        LOGE("mbtk_ecall_msd_gen() fail.");
        return QL_ERR_GENERIC;
    } else {
        return QL_ERR_OK;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Pushes eCall MSD.
  @param[out] state eCall state.
  @return Whether the eCall MSD was successfully pushed.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_push_msd(void)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    if(mbtk_ecall_push(ecall_handle)) {
        LOGE("mbtk_ecall_push() fail.");
        return QL_ERR_GENERIC;
    } else {
        return QL_ERR_OK;
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Setting ecall test number. Number will be saved on the modem side.
  @param[in] number  test number
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_set_test_number(int sim_id, const char* number)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    if(sim_id >= ECALL_SIM_CARD_ID_MAX || sim_id < ECALL_SIM_CARD_ID_NON) {
        return QL_ERR_INVALID_ARG;
    }

    mbtk_ecall_only_info_t only_info;
    memset(&only_info, 0, sizeof(mbtk_ecall_only_info_t));
    if(mbtk_ecall_only_get(ecall_handle, &only_info)) {
        LOGE("mbtk_ecall_only_get() fail.");
        return QL_ERR_GENERIC;
    } else {
        mbtk_ecall_only_type_enum active = only_info.active;
        memset(&only_info, 0, sizeof(mbtk_ecall_only_info_t));
        only_info.active = active;
        memcpy(only_info.test_num, (uint8*)number, strlen(number));

        if(mbtk_ecall_only_set(ecall_handle, &only_info)) {
            LOGE("mbtk_ecall_only_set() fail.");
            return QL_ERR_GENERIC;
        } else {
            return QL_ERR_OK;
        }
    }
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Setting ecall test number. Number will be saved on the modem side.
  @param[in] number  test number
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_reset_test_number(int sim_id, const char* number)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    if(sim_id >= ECALL_SIM_CARD_ID_MAX || sim_id < ECALL_SIM_CARD_ID_NON) {
        return QL_ERR_INVALID_ARG;
    }

    return QL_ERR_UNSUPPORTED;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Reset IVS during ecall session and switch to voice.
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_reset_ivs(int sim_id)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    if(sim_id >= ECALL_SIM_CARD_ID_MAX || sim_id < ECALL_SIM_CARD_ID_NON) {
        return QL_ERR_INVALID_ARG;
    }

    return QL_ERR_UNSUPPORTED;
}

#if 0
/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Starts ecall session.
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_make_ecall(int sim_id, QL_ECALL_MAKE_ECALL_TYPE ecall_type);
#endif

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Deregister from Network.
  @return Whether a request was executed.
  @retval QL_ERR_OK successful.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ecall_terminate_nw_registration(void)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    if(mbtk_ecall_reg_set(ecall_handle, 0)) {
        LOGE("mbtk_ecall_reg_set(0) fail.");
        return QL_ERR_GENERIC;
    } else {
        return QL_ERR_OK;
    }
}

void ql_ecall_set_config_info(ql_ecall_config_t ecall_context_info)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return;
    }

    mbtk_ecall_cfg_info_t cfg;
    memset(&cfg, 0, sizeof(cfg));
    cfg.type = 0;
    if(ecall_context_info.t5_timeout_ms_valid) {
        cfg.type |= MBTK_ECALL_CFG_T5;
        cfg.data_valid[MBTK_ECALL_CFG_ITEM_T5] = (uint8)1;
        cfg.data[MBTK_ECALL_CFG_ITEM_T5] = ecall_context_info.t5_timeout_ms;
    }

    if(ecall_context_info.t6_timeout_ms_valid) {
        cfg.type |= MBTK_ECALL_CFG_T6;
        cfg.data_valid[MBTK_ECALL_CFG_ITEM_T6] = (uint8)1;
        cfg.data[MBTK_ECALL_CFG_ITEM_T6] = ecall_context_info.t6_timeout_ms;
    }

    if(ecall_context_info.t7_timeout_ms_valid) {
        cfg.type |= MBTK_ECALL_CFG_T7;
        cfg.data_valid[MBTK_ECALL_CFG_ITEM_T7] = (uint8)1;
        cfg.data[MBTK_ECALL_CFG_ITEM_T7] = ecall_context_info.t7_timeout_ms;
    }

    if(ecall_context_info.autoAnswer_timeout_ms_valid) {
        cfg.type |= MBTK_ECALL_CFG_TIMER_CALLBACK;
        cfg.data_valid[MBTK_ECALL_CFG_ITEM_TIMER_CALLBACK] = (uint8)1;
        cfg.data[MBTK_ECALL_CFG_ITEM_TIMER_CALLBACK] = ecall_context_info.autoAnswer_timeout_ms;
    }

    if(ecall_context_info.dialDurationTimer_timout_ms_valid) {
        cfg.type |= MBTK_ECALL_CFG_TIMER_DIAL;
        cfg.data_valid[MBTK_ECALL_CFG_ITEM_TIMER_DIAL] = (uint8)1;
        cfg.data[MBTK_ECALL_CFG_ITEM_TIMER_DIAL] = ecall_context_info.dialDurationTimer_timout_ms;
    }

    if(ecall_context_info.maxDialAttempts_valid) {
        cfg.type |= MBTK_ECALL_CFG_REDIALCNT;
        cfg.data_valid[MBTK_ECALL_CFG_ITEM_REDIALCNT] = (uint8)1;
        cfg.data[MBTK_ECALL_CFG_ITEM_REDIALCNT] = ecall_context_info.maxDialAttempts;
    }

    if(mbtk_ecall_cfg_set(ecall_handle, &cfg)) {
        LOGE("mbtk_ecall_cfg_set() fail.");
    }
}

void ql_ecall_get_config_info(ql_ecall_config_t* ecall_context_info)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return;
    }

    if(ecall_context_info == NULL) {
        LOGE("ecall_context_info is NULL.");
        return;
    }

    mbtk_ecall_cfg_info_t cfg;
    memset(&cfg, 0, sizeof(cfg));
    cfg.type = 0;
//    cfg.type |= MBTK_ECALL_CFG_T3;
    cfg.type |= MBTK_ECALL_CFG_T5;
    cfg.type |= MBTK_ECALL_CFG_T6;
    cfg.type |= MBTK_ECALL_CFG_T7;
//    cfg.type |= MBTK_ECALL_CFG_TH;
    cfg.type |= MBTK_ECALL_CFG_TIMER_CALLBACK;
//    cfg.type |= MBTK_ECALL_CFG_TIMER_CLEARDOWN;
//    cfg.type |= MBTK_ECALL_CFG_TIMER_DEREG;
    cfg.type |= MBTK_ECALL_CFG_TIMER_DIAL;
//    cfg.type |= MBTK_ECALL_CFG_TIMER_REDIAL;
//    cfg.type |= MBTK_ECALL_CFG_TIMER_SMS;
    cfg.type |= MBTK_ECALL_CFG_REDIALCNT;
//    cfg.type |= MBTK_ECALL_CFG_SMSPROCESS;
//    cfg.type |= MBTK_ECALL_CFG_SMSMSDCNT;

    if(mbtk_ecall_cfg_get(ecall_handle, &cfg)) {
        LOGE("mbtk_ecall_cfg_get() fail.");
    } else {
        if(cfg.data_valid[MBTK_ECALL_CFG_ITEM_T5]) {
            ecall_context_info->t5_timeout_ms = (uint16_t)cfg.data[MBTK_ECALL_CFG_ITEM_T5];
            ecall_context_info->t5_timeout_ms_valid = (uint8_t)1;
        }

        if(cfg.data_valid[MBTK_ECALL_CFG_ITEM_T6]) {
            ecall_context_info->t6_timeout_ms = (uint16_t)cfg.data[MBTK_ECALL_CFG_ITEM_T6];
            ecall_context_info->t6_timeout_ms_valid = (uint8_t)1;
        }

        if(cfg.data_valid[MBTK_ECALL_CFG_ITEM_T7]) {
            ecall_context_info->t7_timeout_ms = (uint16_t)cfg.data[MBTK_ECALL_CFG_ITEM_T7];
            ecall_context_info->t7_timeout_ms_valid = (uint8_t)1;
        }

        if(cfg.data_valid[MBTK_ECALL_CFG_ITEM_TIMER_CALLBACK]) {
            ecall_context_info->autoAnswer_timeout_ms = (uint16_t)cfg.data[MBTK_ECALL_CFG_ITEM_TIMER_CALLBACK];
            ecall_context_info->autoAnswer_timeout_ms_valid = (uint8_t)1;
        }

        if(cfg.data_valid[MBTK_ECALL_CFG_ITEM_TIMER_DIAL]) {
            ecall_context_info->dialDurationTimer_timout_ms = cfg.data[MBTK_ECALL_CFG_ITEM_TIMER_DIAL];
            ecall_context_info->dialDurationTimer_timout_ms_valid = (uint8_t)1;
        }

        if(cfg.data_valid[MBTK_ECALL_CFG_ITEM_REDIALCNT]) {
            ecall_context_info->maxDialAttempts = (uint16_t)cfg.data[MBTK_ECALL_CFG_ITEM_REDIALCNT];
            ecall_context_info->maxDialAttempts_valid = (uint8_t)1;
        }
    }
}

int ql_ecall_set_ecall_only_mode(bool ecall_only_value)
{
    if(ecall_handle == NULL) {
        LOGE("ecall_handle not init.");
        return QL_ERR_NOT_INIT;
    }

    mbtk_ecall_only_info_t only_info;
    memset(&only_info, 0, sizeof(mbtk_ecall_only_info_t));
    only_info.active = ecall_only_value ? MBTK_ECALL_ONLY_TYPE_ENABLE_ECALL : MBTK_ECALL_ONLY_TYPE_DISABLE;
    if(mbtk_ecall_only_set(ecall_handle, &only_info)) {
        LOGE("mbtk_ecall_only_set() fail.");
        return QL_ERR_GENERIC;
    } else {
        return QL_ERR_OK;
    }
}

