/*-----------------------------------------------------------------------------------------------*/
/**
  @file ql_gnss.c
  @brief SIM service API
*/
/*-----------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  Copyright (c) 2024 mobiletek Wireless Solution, Co., Ltd. All Rights Reserved.
  mobiletek Wireless Solution Proprietary and Confidential.
-------------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  EDIT HISTORY
  This section contains comments describing changes made to the file.
  Notice that changes are listed in reverse chronological order.
  $Header: $
  when       who          what, where, why
  --------   ---------    -----------------------------------------------------------------
  20241022    yq.wang      Created .
-------------------------------------------------------------------------------------------------*/
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>

#include "ql_type.h"
#include "ql_gnss.h"
#include "mbtk_type.h"
#include "mbtk_gnss.h"
#include "mbtk_log.h"

/*----------------------------------------------DEFINE-------------------------------------------*/
#define QL_GNSS_LOGE LOGE
#define QL_GNSS_LOGD LOGD

#define QL_GNSS_TIMEOUT 5
#define QL_GNSS_NMEA_SIZE_MAX 128
#define QL_GNSS_AGPS_URL_SIZE_MAX 128

#define QL_AGNSS_DOWNLOAD_TIMEPUT 60
#define QL_AGNSS_INJECT_TIMEOUT 20

#define QL_AGNSS_URL_PATH "http://uagnss.allystar.com:80/ephemeris/"

#define QL_AGNSS_EPH_GPS                "HD_GPS.hdb"
#define QL_AGNSS_EPH_BDS                "HD_BDS.hdb"
#define QL_AGNSS_EPH_GLO                "HD_GLO.hdb"
#define QL_AGNSS_EPH_GAL                "HD_GAL.hdb"
#define QL_AGNSS_EPH_QZS                "HD_QZS.hdb"
#define QL_AGNSS_EPH_GPS_BDS            "HD_GPS_BDS.hdb"
#define QL_AGNSS_EPH_GPS_GLO            "HD_GPS_GLO.hdb"
#define QL_AGNSS_EPH_GPS_GAL            "HD_GPS_GAL.hdb"
#define QL_AGNSS_EPH_GPS_QZSS           "HD_GPS_QZSS.hdb"

/*----------------------------------------------DEFINE-------------------------------------------*/

/*----------------------------------------------ENUM---------------------------------------------*/
typedef enum {
    QL_GNSS_INIT_STOP  = 0,
    QL_GNSS_INIT_START,
}ql_gnss_init_enum;

/*----------------------------------------------ENUM---------------------------------------------*/

/*----------------------------------------------STRUCT-------------------------------------------*/
typedef struct {
    ql_gnss_init_enum  status;
    ql_gnss_ind_cb_f   ind_cb;
    ql_gnss_error_cb_f server_cb;
    time_t             timestamp;
}ql_gnss_info_handle_t;

/*----------------------------------------------STRUCT-------------------------------------------*/

/*----------------------------------------------GLOBAL VARIABLE----------------------------------*/
extern long timezone;

/*----------------------------------------------GLOBAL VARIABLE----------------------------------*/

/*----------------------------------------------GLOBAL STATIC VARIABLE---------------------------*/
static ql_gnss_info_handle_t* gnss_handle = NULL;

/*----------------------------------------------GLOBAL STATIC VARIABLE---------------------------*/

/*----------------------------------------------GNSS FUNCTION-------------------------------------*/
static time_t ql_get_timestamp(char *time)
{
    char tmp_char[4] = {0};
    struct tm tmp_time;

    memset(&tmp_time, 0, sizeof(struct tm));
    memset(tmp_char, 0, sizeof(tmp_char));
    memcpy(tmp_char, &time[4], 2);
    tmp_time.tm_sec = atoi(tmp_char);

    memcpy(tmp_char, &time[2], 2);
    tmp_time.tm_min = atoi(tmp_char);

    memcpy(tmp_char, &time[0], 2);
    tmp_time.tm_hour = atoi(tmp_char);

    memcpy(tmp_char, &time[6], 2);
    tmp_time.tm_mday = atoi(tmp_char);

    memcpy(tmp_char, &time[8], 2);
    tmp_time.tm_mon = atoi(tmp_char) - 1;

    memcpy(tmp_char, &time[10], 2);
    tmp_time.tm_year = 100 + atoi(tmp_char);

    time_t _t = mktime(&tmp_time);
    tzset();
    _t = _t - timezone;

    return _t;
}

static time_t ql_get_gnss_time_sec(char *data, int data_len)
{
//    int i = 0, num = 0;
    char* previous_p = data;
    char* current_p = NULL;
    int current_get_num = 0;
    char time[15] = {0};

    //$GNRMC,024142.000,A,3039.364421,N,10403.417935,E,0.051,0.00,030124,,E,A*00

    do
    {
        current_p = strchr(previous_p, ',');
        if(current_p != NULL)
        {
            current_get_num++;
            if(current_get_num == 2)
            {
                if(previous_p != current_p)
                {
                    memcpy(time, previous_p, 6);
                    //QL_GNSS_LOGE("[%s] time:%s.", __func__, time);
                }
                else
                {
                    gnss_handle->timestamp = 0;
                    return 0;
                }
            }

            if(current_get_num == 10)
            {
                if(previous_p != current_p)
                {
                    memcpy(time + 6, previous_p, 6);
                    //QL_GNSS_LOGE("[%s] date + time:%s.", __func__, time);
                    break;
                }
                else
                {
                    gnss_handle->timestamp = 0;
                    return 0;
                }
            }

            if(current_p + 1 <= data + data_len)
            {
                previous_p = current_p + 1;
            }
            else
            {
                QL_GNSS_LOGE("[%s] data out of range.", __func__);
                gnss_handle->timestamp = 0;
                return 0;
            }
        }
    }while(current_p != NULL);

    gnss_handle->timestamp = ql_get_timestamp(time);

    return gnss_handle->timestamp;
}

static void ql_gnss_callback(uint32 ind_type, const void* data, uint32 data_len)
{
    if(data == NULL || data_len <= 0)
    {
        QL_GNSS_LOGE("[%s] data is NULL.", __func__);
        return;
    }

    if(gnss_handle->ind_cb != NULL)
    {
        if(ind_type == MBTK_GNSS_IND_NMEA)
        {
            nmea_srv_ind_msg nmea_msg = {0};
            memset(&nmea_msg, 0x0, sizeof(nmea_srv_ind_msg));
            nmea_msg.msg_id = QL_GNSS_NMEA_MSG;
            if(data_len <= QL_GNSS_NMEA_SIZE_MAX)
            {
                memcpy(nmea_msg.nmea_sentence, (const char *)data, data_len);
            }
            else
            {
                QL_GNSS_LOGE("[%s] data too long. %s", __func__, (char*)data);
                return;
            }

            if(NULL != strstr((char*)data, "RMC"))
            {
                nmea_msg.time = ql_get_gnss_time_sec(nmea_msg.nmea_sentence, data_len);
            }
            else
            {
                nmea_msg.time = gnss_handle->timestamp;
            }

            gnss_handle->ind_cb((void*)&nmea_msg);
        }
        else
        {
            QL_GNSS_LOGE("[%s] Unknown IND : %d.", __func__, ind_type);
        }
    }

}

/*----------------------------------------------GNSS FUNCTION-------------------------------------*/

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Initialize the gnss client
  @return
  QL_ERR_OK - successful
  QL_ERR_INVALID_ARG - as defined
  QL_ERR_SERVICE_NOT_READY - service is not ready, need to retry
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_init(void)
{
    if(NULL == gnss_handle)
    {
        gnss_handle = (ql_gnss_info_handle_t*)malloc(sizeof(ql_gnss_info_handle_t));
        if(NULL == gnss_handle)
        {
            QL_GNSS_LOGE("[%s] gnss handle malloc fail.", __func__);
            return QL_ERR_FAILED;
        }

        gnss_err_enum ret = mbtk_gnss_init(ql_gnss_callback);
        if(ret == GNSS_ERR_OK)
        {
            gnss_handle->status = QL_GNSS_INIT_STOP;
            ret = mbtk_gnss_ind_set(MBTK_GNSS_IND_NMEA, QL_GNSS_TIMEOUT);
            if(ret != GNSS_ERR_OK)
            {
                QL_GNSS_LOGE("[%s] gnss ind set fail.[%d]", __func__, ret);
                goto error_2;
            }
        }
        else
        {
            QL_GNSS_LOGE("[%s] gnss init fail.[%d]", __func__, ret);
            goto error_1;
        }

        gnss_handle->ind_cb = NULL;
        gnss_handle->server_cb = NULL;
        gnss_handle->timestamp = 0;
    }

    return QL_ERR_OK;
error_2:
    if(gnss_handle->status == QL_GNSS_INIT_START)
    {
        mbtk_gnss_deinit();
        gnss_handle->status = QL_GNSS_INIT_STOP;
    }
error_1:
    if(gnss_handle)
    {
        free(gnss_handle);
        gnss_handle = NULL;
    }
    return QL_ERR_FAILED;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Deinitialize the  gnss client
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_deinit(void)
{
    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    gnss_handle->timestamp = 0;
    gnss_handle->server_cb = NULL;
    gnss_handle->ind_cb = NULL;

    gnss_err_enum ret = mbtk_gnss_deinit();
    if(ret == GNSS_ERR_OK)
    {
        gnss_handle->status = QL_GNSS_INIT_STOP;
    }
    else
    {
        QL_GNSS_LOGE("[%s] gnss init fail.[%d]", __func__, ret);
        return QL_ERR_FAILED;
    }

    free(gnss_handle);
    gnss_handle = NULL;

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Start gnss
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_start(void)
{
    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    gnss_err_enum ret = GNSS_ERR_UNKNOWN;
    ret = mbtk_gnss_open(255, QL_GNSS_TIMEOUT);
    if(ret != GNSS_ERR_OK)
    {
        LOGE("[%s] gnss open fail.[%d]", __func__, ret);
        return QL_ERR_FAILED;
    }

    gnss_handle->status = QL_GNSS_INIT_START;

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Stop gnss
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_stop(void)
{
    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    gnss_err_enum ret = GNSS_ERR_UNKNOWN;
    ret = mbtk_gnss_close(QL_GNSS_TIMEOUT);
    if(ret != GNSS_ERR_OK)
    {
        LOGE("[%s] gnss close fail.[%d]", ret);
        return QL_ERR_FAILED;
    }

    gnss_handle->status = QL_GNSS_INIT_STOP;

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Set gnss start mode
  @param[in] mask Gnss start mode
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_set_start_mode(QL_GNSS_START_MODE_E mask)
{
    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    gnss_err_enum ret = GNSS_ERR_UNKNOWN;
    char param_buf[32] = {0};
//    int  length = 0;
    int  mode = 0;

    if(mask == QL_GNSS_COLD_START)
    {
        mode = 3;
    }
    else if(mask == QL_GNSS_WARM_START)
    {
        mode = 2;
    }
    else if(mask == QL_GNSS_HOT_START)
    {
        mode = 1;
    }
    else
    {
        QL_GNSS_LOGE("[%s] mask error.", __func__);
        return QL_ERR_INVALID_ARG;
    }

    snprintf(param_buf, 32, "$RESET,%d", mode);
    ret = mbtk_gnss_setting(param_buf, QL_GNSS_TIMEOUT);
    if(ret != GNSS_ERR_OK)
    {
        LOGE("[%s] gnss setting fail.[%d]", __func__);
        return QL_ERR_FAILED;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Get gnss nmea
  @param[in] nmea_str
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_qgpsgnmea(nmeasrc_sentences *nmea_str)
{
    UNUSED(nmea_str);

    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Set gnss qgpscfg
  @param[in] enable
  @param[in] str
  @param[in] len
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_set_qgpscfg(uint32_t enable, char *str, uint32_t len)
{
    UNUSED(enable);
    UNUSED(str);
    UNUSED(len);

    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Get engine state
  @param[out] state  on/ff
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_get_engine_state(QL_GNSS_ENGINE_STATE_E *state)
{
    UNUSED(state);

    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Set Constellation .
  @param[in] mask  GPS/GLO/BDS/GAL/QZSS/SBAS
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_set_constellation(QL_GNSS_CONSTELLATION_MASK_E mask)
{
    UNUSED(mask);

    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Get Constellation
  @param[out] mask   GPS/GLO/BDS/GAL/QZSS/SBAS
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_get_constellation(QL_GNSS_CONSTELLATION_MASK_E *mask)
{
    UNUSED(mask);

    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Set nmea type
  @param[in] mask type
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_set_nmea_type(uint32_t mask)
{
    UNUSED(mask);

    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Get nmea version
  @param[out] version nmea version
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_get_nmea_version(QL_GNSS_NMEA_VERSION_ID_E *version)
{
    UNUSED(version);

    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Set nmea version
  @param[in] version nmea version
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_set_nmea_version(QL_GNSS_NMEA_VERSION_ID_E version)
{
    UNUSED(version);

    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Set gnss agnss mode
  @param[in] mask Gnss agnss mode
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_set_agnss_mode(uint32_t mask)
{
    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    char eph_file[32] = {0};
    gnss_err_enum ret = GNSS_ERR_UNKNOWN;
    char write_buff[512] = {0};
//    int write_length = 0;

    if((mask & QL_GNSS_EPH_DATA_GPS) == mask)
    {
        memcpy(eph_file, QL_AGNSS_EPH_GPS, strlen(QL_AGNSS_EPH_GPS));
    }
    else if((mask & QL_GNSS_EPH_DATA_BDS) == mask)
    {
        memcpy(eph_file, QL_AGNSS_EPH_BDS, strlen(QL_AGNSS_EPH_BDS));
    }
    else if((mask & QL_GNSS_EPH_DATA_GLO) == mask)
    {
        memcpy(eph_file, QL_AGNSS_EPH_GLO, strlen(QL_AGNSS_EPH_GLO));
    }
    else if((mask & QL_GNSS_EPH_DATA_GAL) == mask)
    {
        memcpy(eph_file, QL_AGNSS_EPH_GAL, strlen(QL_AGNSS_EPH_GAL));
    }
    else if((mask & QL_GNSS_EPH_DATA_QZSS) == mask)
    {
        memcpy(eph_file, QL_AGNSS_EPH_QZS, strlen(QL_AGNSS_EPH_QZS));
    }
    else if((mask & (QL_GNSS_EPH_DATA_GPS | QL_GNSS_EPH_DATA_BDS)) == mask)
    {
        memcpy(eph_file, QL_AGNSS_EPH_GPS_BDS, strlen(QL_AGNSS_EPH_GPS_BDS));
    }
    else if((mask & (QL_GNSS_EPH_DATA_GPS | QL_GNSS_EPH_DATA_GLO)) == mask)
    {
        memcpy(eph_file, QL_AGNSS_EPH_GPS_GLO, strlen(QL_AGNSS_EPH_GPS_GLO));
    }
    else if((mask & (QL_GNSS_EPH_DATA_GPS | QL_GNSS_EPH_DATA_GAL)) == mask)
    {
        memcpy(eph_file, QL_AGNSS_EPH_GPS_GAL, strlen(QL_AGNSS_EPH_GPS_GAL));
    }
    else if((mask & (QL_GNSS_EPH_DATA_GPS | QL_GNSS_EPH_DATA_QZSS)) == mask)
    {
        memcpy(eph_file, QL_AGNSS_EPH_GPS_QZSS, strlen(QL_AGNSS_EPH_GPS_QZSS));
    }
    else
    {
        QL_GNSS_LOGE("[%s] unknown mask.", __func__);
        return QL_ERR_INVALID_ARG;
    }

    snprintf(write_buff, 512, "$AGPSCFG,%s%s,NULL,NULL", QL_AGNSS_URL_PATH, eph_file);
    ret = mbtk_gnss_setting(write_buff, QL_GNSS_TIMEOUT);
    if(ret != GNSS_ERR_OK)
    {
        LOGE("[%s] gnss setting fail.[%d]", __func__, ret);
        return QL_ERR_FAILED;
    }
    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Inject  agnss data
  @param[in] data agnss data
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_inject_agnss_data(void)
{
    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    gnss_err_enum ret = GNSS_ERR_UNKNOWN;
    ret = mbtk_gnss_eph_download(QL_AGNSS_DOWNLOAD_TIMEPUT);
    if(ret != GNSS_ERR_OK)
    {
        LOGE("[%s] gnss eph download fail.[%d]", __func__, ret);
        return QL_ERR_FAILED;
    }

    ret = mbtk_gnss_eph_inject(QL_AGNSS_INJECT_TIMEOUT);
    if(ret != GNSS_ERR_OK)
    {
        LOGE("[%s] gnss eph inject fail.[%d]", __func__, ret);
        return QL_ERR_FAILED;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Inject UTC time to GNSS Engine.
  @param[in] timestamp ,unit: ms
  @return Whether to successfully Start Gnss
  @retval QL_ERR_OK successful
  @retval QL_ERR_FAILED  Start Gnss Failed
  @retval Other error code defined by ql_gnss.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_inject_utc_time(uint64_t timestamp)
{
    UNUSED(timestamp);

    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Set gnss suspend
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_suspend(void)
{
    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Set gnss resume
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_resume(void)
{
    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}




/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Register gnss callback
  @param[in] cb
  @return
  QL_ERR_OK - successful
  QL_ERR_NOT_INIT - uninitialized
  QL_ERR_SERVICE_NOT_READY - service is not ready
  QL_ERR_INVALID_ARG - Invalid arguments
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_set_ind_cb(ql_gnss_ind_cb_f cb)
{
    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    gnss_handle->ind_cb = cb;

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Registration server error callback. Currently, only if the server exits abnormally,
  the callback function will be executed, and the error code is QL_ERR_ABORTED;
  @param[in] cb  Callback function
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_gnss_set_service_error_cb(ql_gnss_error_cb_f cb)
{
    if(NULL == gnss_handle)
    {
        QL_GNSS_LOGE("[%s] gnss handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    gnss_handle->server_cb = cb;

    return QL_ERR_OK;
}

