#ifndef __QL_GPIO_INTERNAL_H
#define __QL_GPIO_INTERNAL_H

#ifdef __cplusplus
extern "C" {
#endif

#include "ql_gpio.h"

typedef enum{
    INTERNAL_BEGIN = -1,
    INTERNAL_GPIO_10 = 10,
    INTERNAL_GPIO_11 = 11,
    INTERNAL_GPIO_19 = 19,
    INTERNAL_GPIO_21 = 21,
    INTERNAL_GPIO_22 = 22,
    INTERNAL_GPIO_23 = 23,
    INTERNAL_GPIO_24 = 24,
    INTERNAL_GPIO_25 = 25,
    INTERNAL_GPIO_26 = 26,
    INTERNAL_GPIO_27 = 27,
    INTERNAL_GPIO_31 = 31,
    INTERNAL_GPIO_32 = 32,
    INTERNAL_GPIO_33 = 33,
    INTERNAL_GPIO_34 = 34,
    INTERNAL_GPIO_35 = 35,
    INTERNAL_GPIO_36 = 36,
    INTERNAL_GPIO_37 = 37,
    INTERNAL_GPIO_38 = 38,
    INTERNAL_GPIO_39 = 39,
    INTERNAL_GPIO_40 = 40,
    INTERNAL_GPIO_43 = 43,
    INTERNAL_GPIO_48 = 48,
    INTERNAL_GPIO_49 = 49,
    INTERNAL_GPIO_50 = 50,
    INTERNAL_GPIO_51 = 51,
    INTERNAL_GPIO_52 = 52,
    INTERNAL_GPIO_53 = 53,
    INTERNAL_GPIO_54 = 54,
    INTERNAL_GPIO_55 = 55,
    INTERNAL_GPIO_56 = 56,
    INTERNAL_GPIO_57 = 57,
    INTERNAL_GPIO_58 = 58,
    INTERNAL_GPIO_59 = 59,
    INTERNAL_GPIO_99 = 99,
    INTERNAL_GPIO_117 = 117,
    INTERNAL_GPIO_118 = 118,
    INTERNAL_GPIO_119 = 119,
    INTERNAL_GPIO_120 = 120,
    INTERNAL_GPIO_122 = 122,
    INTERNAL_GPIO_123 = 123,
    INTERNAL_GPIO_126 = 126,
    INTERNAL_END = 127
}INTERNAL_ENUM_PINNAME;


/*****************************************************************
* Function:     __ql_gpio_init 
* 
* Description:
*               This function enables the GPIO function of the specified pin,
*               and initialize the configurations, including direction,
*               level and pull selection.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of INTERNAL_ENUM_PINNAME.
*               dir:
*                   The initial direction of GPIO, one value of ENUM_PIN_DIRECTION.
*               level:
*                   The initial level of GPIO, one value of ENUM_PIN_LEVEL. 
*               pullSel:
*                   Pull selection, one value of ENUM_PIN_PULLSEL.
* Return:        
*               RES_OK, this function succeeds.
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid. 
*               RES_IO_ERR, the function failed
*               other place. For example this GPIO has been using as EINT.
*****************************************************************/
int __ql_gpio_init(INTERNAL_ENUM_PINNAME       pin_name, 
                 ENUM_PIN_DIRECTION  dir, 
                 ENUM_PIN_LEVEL      level, 
                 ENUM_PIN_PULLSEL    pull_sel
                 );

/*****************************************************************
* Function:     __ql_gpio_base_init 
* 
* Description:
*               This function enables the GPIO function of the specified pin.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of INTERNAL_ENUM_PINNAME.
*            
* Return:        
*               RES_OK, this function succeeds.
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid. 
*               RES_IO_ERR, the function failed
*****************************************************************/
int __ql_gpio_base_init(INTERNAL_ENUM_PINNAME pin_name );

/*****************************************************************
* Function:     __ql_gpio_set_level 
* 
* Description:
*               This function sets the level of the specified GPIO.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of INTERNAL_ENUM_PINNAME.
*               level:
*                   The initial level of GPIO, one value of ENUM_PIN_LEVEL. 
* Return:        
*               RES_OK, this function succeeds.
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid. 
*               RES_IO_ERR, the function failed
*               other place. For example this GPIO has been using as EINT.
*****************************************************************/
int __ql_gpio_set_level(INTERNAL_ENUM_PINNAME pin_name, ENUM_PIN_LEVEL level);

/*****************************************************************
* Function:     __ql_gpio_get_level 
* 
* Description:
*               This function gets the level of the specified GPIO.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of INTERNAL_ENUM_PINNAME.
* Return:        
*               The level value of the specified GPIO, which is 
*               nonnegative integer.
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid.
*****************************************************************/
int __ql_gpio_get_level(INTERNAL_ENUM_PINNAME pin_name);

/*****************************************************************
* Function:     __ql_gpio_set_direction 
* 
* Description:
*               This function sets the direction of the specified GPIO.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of INTERNAL_ENUM_PINNAME.
*               dir:
*                   The initial direction of GPIO, one value of ENUM_PIN_DIRECTION.
* Return:        
*               RES_OK, this function succeeds.
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid. 
*               RES_IO_ERR, the function failed
*               other place. For example this GPIO has been using as EINT.
*****************************************************************/
int __ql_gpio_set_direction(INTERNAL_ENUM_PINNAME pin_name, ENUM_PIN_DIRECTION dir);

/*****************************************************************
* Function:     __ql_gpio_get_direction 
* 
* Description:
*               This function gets the direction of the specified GPIO.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of INTERNAL_ENUM_PINNAME.
* Return:        
*               0  INPUT
*               1  OUTPUT
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid. 
*               other place. For example this GPIO has been using as EINT.
*****************************************************************/
int __ql_gpio_get_direction(INTERNAL_ENUM_PINNAME pin_name);


/*****************************************************************
* Function:     __ql_gpio_set_pull_selection 
* 
* Description:
*               This function sets the pull selection of the specified GPIO.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of INTERNAL_ENUM_PINNAME.
*               ENUM_PIN_PULLSEL:
*                   Pull selection, one value of ENUM_PIN_PULLSEL.
* Return:        
*               RES_OK, this function succeeds.
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid. 
*               RES_IO_ERR, the function failed
*               other place. For example this GPIO has been using as EINT.
*****************************************************************/
int __ql_gpio_set_pull_selection(INTERNAL_ENUM_PINNAME pin_name, ENUM_PIN_PULLSEL pull_sel);

/*****************************************************************
* Function:     __ql_gpio_get_pull_selection 
* 
* Description:
*               This function gets the pull selection of the specified GPIO.
*
* Parameters:
*               pin_name:
*                   Pin name, one value of INTERNAL_ENUM_PINNAME.
* Return:        
*               0   no pull
*               1   pull down
*               3   pull up
*****************************************************************/
int __ql_gpio_get_pull_selection(INTERNAL_ENUM_PINNAME pin_name);

/*****************************************************************
* Function:     __ql_gpio_uninit 
* 
* Description:
*               This function releases the specified GPIO that was 
*               initialized by calling ql_gpio_init() previously.
*               After releasing, the GPIO can be used for other purpose.
* Parameters:
*               pin_name:
*                   Pin name, one value of INTERNAL_ENUM_PINNAME.
* Return:        
*               RES_OK, this function succeeds.
*               RES_IO_NOT_SUPPORT, the input GPIO is invalid. 
*               RES_IO_ERR, the function failed
*               other place. For example this GPIO has been using as EINT.
*****************************************************************/
int __ql_gpio_uninit(INTERNAL_ENUM_PINNAME pin_name);

//------------------------------------------------------------------------------
/*
* Function:     __ql_eint_enable 
* 
* Description:
*               Set the interrupt sense mode, and enable interrupt. 
*
* Parameters:
*               eint_pin_name:
*                   EINT pin name, one value of INTERNAL_ENUM_PINNAME that has 
*                   the interrupt function.
*
*               eint_type:
*                   Interrupt type, level-triggered or edge-triggered.
*                   Now, only edge-triggered interrupt is supported.
*
*               eint_callback:
*                   call back function
*
* Return:        
*               RES_OK, this function succeeds.
*               else failed to execute the function. 
*/
//------------------------------------------------------------------------------
int __ql_eint_enable(INTERNAL_ENUM_PINNAME eint_pin_name, ENUM_EINT_TYPE eint_type, ql_eint_callback eint_callback);


//------------------------------------------------------------------------------
/*
* Function:     __ql_eint_disable 
* 
* Description:
*               Disable the interrupt sense. 
*
* Parameters:
*               eint_pin_name:
*                   EINT pin name, one value of INTERNAL_ENUM_PINNAME that has 
*                   the interrupt function.
*
* Return:        
*               RES_OK, this function succeeds.
*               else failed to execute the function. 
*/
//------------------------------------------------------------------------------
int __ql_eint_disable(INTERNAL_ENUM_PINNAME eint_pin_name);


/*-----------------------------------------------------------------------------------------------*/
/**
Function:     __ql_set_gpio_function 
* 
* Description:
*             set pin function. 
*
* Parameters:
*            pin_name:
*            EINT pin name, one value of INTERNAL_ENUM_PINNAME.
*		 func:
*		 pin function,value range from 0 to 7
*
* Return:        
*               RES_OK, this function succeeds.
*               else failed to execute the function. 
*/
/*-----------------------------------------------------------------------------------------------*/
int __ql_set_gpio_function(INTERNAL_ENUM_PINNAME pin_name,unsigned int func);


/*-----------------------------------------------------------------------------------------------*/
/*Function:     __ql_get_gpio_function 
* 
* Description:
*               get pin function. 
*
* Parameters:
*               pin_name:
*               EINT pin name, one value of INTERNAL_ENUM_PINNAME.
*
* Return:        
*               return 0-7, this function succeeds.
*               else failed to execute the function. 
*/
/*-----------------------------------------------------------------------------------------------*/
int __ql_get_gpio_function(INTERNAL_ENUM_PINNAME pin_name);

/*-----------------------------------------------------------------------------------------------*/
/*Function:     __ql_check_pin_function_status 
* 
* Description:
*               Check whether the pin is gpio function. 
*
* Parameters:
*               pin_name:
*               EINT pin name, one value of INTERNAL_ENUM_PINNAME.
*
* Return:        
*               return 1, this pin is gpio function.
*               return 0, this pin is not gpio function. 
*		     else failed to execute the function.
*/
/*-----------------------------------------------------------------------------------------------*/
int __ql_check_pin_function_status(INTERNAL_ENUM_PINNAME  pin_name);

#ifdef __cplusplus
}
#endif

#endif
