/*-----------------------------------------------------------------------------------------------*/
/**
  @file ql_dm.h 
  @brief device management API
*/
/*-----------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  Copyright (c) 2021 Quectel Wireless Solution, Co., Ltd. All Rights Reserved.
  Quectel Wireless Solution Proprietary and Confidential.
-------------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  EDIT HISTORY
  This section contains comments describing changes made to the file.
  Notice that changes are listed in reverse chronological order.
  $Header: $
  when       who          what, where, why
  --------   ---          ----------------------------------------------------------
  20200316   stan.li      Optimize the ql_dm_get_modem_state interface
  20191224   stan.li      Add radio on/off API
  20190625   stan.li      Created .
-------------------------------------------------------------------------------------------------*/

#ifndef __QL_MS_DM_H__
#define __QL_MS_DM_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "ql_type.h"
#include "ql_dm.h"

typedef void (*ql_ms_dm_air_plane_mode_ind_cb)(int sim_id, QL_DM_AIR_PLANE_MODE_TYPE_E air_plane_mode);
typedef void (*ql_ms_dm_modem_state_ind_cb)(int statue);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Initialize DM service.
  @note You must call this function before other functions can be used in this module.
  @return Whether the DM service was successfully intialized.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_init(void);


/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Deinitialize DM service.
  @return Whether the DM service was successfully deintialized.
  @retval QL_ERR_OK successful.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_deinit(void);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get device software version. 
  @param[out] soft_ver Return software version
  @param[in] soft_ver_len The length of soft_ver
  @return Whether to successfully get the software version
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_get_software_version(char *soft_ver, int soft_ver_len);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get modem state.
  @details QL_DM_MODEM_STATE_ONLINE,if modem starts normally.
  @details QL_DM_MODEM_STATE_OFFLINE,in modem starts abnormally.
  @details QL_DM_MODEM_STATE_UNKNOWN,unknown error. 
  @param[out] modem_state The state of modem
  @return Whether to successfully get the modem state
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_get_modem_state(QL_DM_MODEM_STATE_TYPE_E *modem_state);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief register modem state event. 
  @param[in] cb_func modem state indication callback function
  @return Whether the modem state event was successfully registered.
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_set_modem_state_change_ind_cb(ql_dm_modem_state_ind_cb cb_func);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get module temperature. 
  @param[out] temperature The current temperature
  @return Whether to successfully get the temperature
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_get_temperature(float *temperature);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get device serial numbers. 
  @param[out] p_info Pointer that point to ql_dm_device_serial_numbers_info_t
  @return Whether to successfully get the serial numbers
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_get_device_serial_numbers(ql_dm_device_serial_numbers_info_t *p_info);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get device firmware revision identification. 
  @param[out] firmware_rev_id Return device firmware revision id
  @param[in] firmware_rev_id_len The length of firmware_rev_id
  @return Whether to successfully get the firmware revision id
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_get_device_firmware_rev_id(char *firmware_rev_id, int firmware_rev_id_len);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get air plane mode. 
  @param[in] sim id number
  @param[out] p_info Pointer that point to QL_DM_AIR_PLANE_MODE_TYPE_E
  @return Whether to successfully get the air plane mode
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_get_air_plane_mode(int sim_id, QL_DM_AIR_PLANE_MODE_TYPE_E *p_info);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set air plane mode. 
  @param[in] sim id number
  @param[in] air_plane_mode 1:ON, 2:OFF
  @return Whether to successfully set the air plane mode
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_set_air_plane_mode(int sim_id, QL_DM_AIR_PLANE_MODE_TYPE_E air_plane_mode);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief register air plane mode event.
  @param[in] sim_id Sim id number
  @param[in] cb_func Air plane mode indication callback function
  @return Whether the air plane mode event was successfully registered.
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_set_air_plane_mode_ind_cb(int sim_id, ql_ms_dm_air_plane_mode_ind_cb cb_func);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get cpu occupancy. 
  @param[out] cpu_occupancy The percentage of cpu occupancy
  @return Whether to successfully get the cpu occupancy
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_get_cpu_occupancy(float *cpu_occupancy);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get mem usage. 
  @param[out] mem_use The percentage of mem usage
  @return Whether to successfully get the memory usage
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_get_mem_usage(float *mem_use);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get NV item value. 
  @param[in] sim id number
  @param[in] nv_item_name The NV item name that is either NV item id or NV item path
  @param[out] nv_item_value The nv value buf of nv_item_name
  param[in] nv_item_value_len The length of nv_item_value
  param[out] nv_len The real length of nv_item_name
  @return Whether to successfully get the NV value
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_get_nv_item_value(char *nv_item_name, unsigned char *nv_item_value, int nv_item_value_len,  int *nv_len);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set NV item value. 
  @param[in] sim id number
  @param[in] nv_item_name The NV item name that is either NV item id or NV item path
  @param[in] nv_item_value The NV value of nv_item_name
  @param[in] nv_item_value_len The length of nv_item_value
  param[out] nv_len The real length of nv_item_name
  @return Whether to successfully set the NV value
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_set_nv_item_value(char *nv_item_name, unsigned char *nv_item_value, int nv_item_value_len, int *nv_len);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set radio on, its function is the same as at+cfun=1. 
  @param[in] sim id number
  @return Whether to successfully set the radio on
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_set_radio_on(int sim_id);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set radio off, its function is the same as at+cfun=0. 
  @param[in] sim id number
  @return Whether to successfully set the radio off
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_set_radio_off(int sim_id);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get modem CPU utilization. 
  @param[out] mem_use The percentage of modem utilization
  @return Whether to successfully get the modem utilization
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_get_modem_cpu_occupancy(float *cpu_occupancy);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get modem mem utilization. 
  @param[out] mem_use The percentage of modem utilization
  @return Whether to successfully get the modem utilization
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_get_modem_mem_usage(float *mem_use);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get QOOS enable state 
  @param[in] sim id number
  @param[out] enable The enable state of QOOS
  @return Whether to successfully get the QOOS enable state
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_get_qoos_enable(int sim_id, QL_DM_QOOS_ENABLE_TYPE_E *enable);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set QOOS enable state 
  @param[in] sim id number
  @param[in] enable The enable state of QOOS
  @return Whether to successfully set the QOOS enable state
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_set_qoos_enable(int sim_id, QL_DM_QOOS_ENABLE_TYPE_E enable);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get QOOS configuration 
  @param[in] sim id number
  @param[out] config The configuration of QOOS
  @return Whether to successfully get the QOOS configuration
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_get_qoos_config(int sim_id, ql_dm_qoos_config_t *config);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set QOOS configuration 
  @param[in] sim id number
  @param[in] config The configuration of QOOS
  @return Whether to successfully set the QOOS configuration
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_set_qoos_config(int sim_id, char enable ,int p1, int p2, int p3);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get MSSR(Modem SubSysem Reset) level. 
  @param[out] p_level The MSSR level
  @return Whether to successfully get the MSSR level
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_get_mssr_level(int *p_level);


/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set MSSR(Modem SubSysem Reset) level. 
  @param[in] level The MSSR level
  @return Whether to successfully set the MSSR level
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_set_mssr_level(int level);


/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Registration server error callback. Currently, only if the server exits abnormally, 
  the callback function will be executed, and the error code is QL_ERR_ABORTED;
  @param[in] cb  Callback function 
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_dm_set_service_error_cb(ql_dm_service_error_cb_f cb);


#ifdef __cplusplus
}
#endif


#endif

