/*-----------------------------------------------------------------------------------------------*/
/**
  @file ql_ms_nw.h 
  @brief network registration service API, support Dual Sim Dual Active(DSDA).
*/
/*-----------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  Copyright (c) 2021 Quectel Wireless Solution, Co., Ltd. All Rights Reserved.
  Quectel Wireless Solution Proprietary and Confidential.
-------------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  EDIT HISTORY
  This section contains comments describing changes made to the file.
  Notice that changes are listed in reverse chronological order.
  $Header: $
  when       who          what, where, why
  --------   ---          ----------------------------------------------------------
  20200518   rambo.shan   Add WEA and ETWS alert indication.
  20191225   stan.li      Modify function description.
  20191202   stan.li      change the name of ql_nw_pref_nwmode_roming_info_t
  20190528   stan.li      Created .
-------------------------------------------------------------------------------------------------*/

#ifndef __QL_MS_NW_H__
#define __QL_MS_NW_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "ql_type.h"
#include "ql_nw.h"

typedef void (*ql_ms_nw_voice_reg_ind_cb)(int sim_id, ql_nw_reg_status_info_t *p_info);

typedef void (*ql_ms_nw_data_reg_ind_cb)(int sim_id, ql_nw_reg_status_info_t *p_info);

typedef void (*ql_ms_nw_signal_strength_ind_cb)(int sim_id, ql_nw_signal_strength_info_t *p_info, QL_NW_SIGNAL_STRENGTH_LEVEL_E level);

typedef void (*ql_ms_nw_cell_access_status_ind_cb)(int sim_id, QL_NW_CELL_ACCESS_STATE_TYPE_E status);

typedef void (*ql_ms_nw_nitz_time_update_ind_cb)(int sim_id, ql_nw_nitz_time_info_t *p_info);

typedef void (*ql_ms_nw_network_scan_async_cb)(int sim_id, int async_index, ql_nw_scan_result_list_info_t *p_info);

typedef void (*ql_ms_nw_wea_reg_ind_cb)(int sim_id, ql_nw_wea_alert_info_t *p_info);

typedef void (*ql_ms_nw_etws_reg_ind_cb)(int sim_id, ql_nw_etws_alert_info_t *p_info);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief Initialize NW service.
  @note You must call this function before other functions can be used in this module.
  @return Whether the NW service was successfully intialized.
  @retval QL_ERR_OK successful
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_init(void);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Deinitializes NW service.
  @return Whether the NW service was deinitialized successfully.
  @retval QL_ERR_OK successful.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_deinit(void);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief scan network status.
  @param[in] sim id number
  @param[out] async_index The index of request msg
  @param[in] async_cb The callback function of request msg
  @return Whether to successfully trigger the network scan operation
  @retval QL_ERR_OK  successful
  @retval QL_ERR_NOT_INIT  uninitialized
  @retval QL_ERR_SERVICE_NOT_READY  service is not ready
  @retval QL_ERR_INVALID_ARG  Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_network_scan(int sim_id, int *async_index, ql_ms_nw_network_scan_async_cb async_cb);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set power mode.
  @param[in] sim id number
  @param[in] lower_mode, defined by QL_NW_LOWER_POWER_MASK_XXX
  @return Whether to successfully set the power mode
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_set_power_mode(int sim_id, uint8_t lower_mode);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set perferred NW mode and roaming indicator.
  @param[in] sim id number
  @param[in] p_info Pointer that point to ql_nw_pref_nwmode_roaming_info_t
  @return Whether to successfully set nwmode and roaming
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_set_pref_nwmode_roaming(int sim_id, ql_nw_pref_nwmode_roaming_info_t *p_info);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get perferred NW mode and roaming indicator.
  @param[in] sim id number
  @param[out] p_info Pointer that point to ql_nw_pref_nwmode_roaming_info_t 
  @return Whether to successfully get nwmode and roaming
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_get_pref_nwmode_roaming(int sim_id, ql_nw_pref_nwmode_roaming_info_t *p_info);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get mobile operator name.
  @param[in] sim id number
  @param[out] p_info Pointer that point to ql_nw_mobile_operator_name_info_t
  @return Whether to successfully get the mobile operator name
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_get_mobile_operator_name(int sim_id, ql_nw_mobile_operator_name_info_t *p_info);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get cell information.
  @param[in] sim id number
  @param[out] p_info Pointer that point to ql_nw_cell_info_t
  @return Whether to successfully get the cell information
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_get_cell_info(int sim_id, ql_nw_cell_info_t *p_info);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get voice registration status.
  @param[in] sim id number
  @param[out] p_info Pointer that point to ql_nw_reg_status_info_t
  @return Whether to successfully get the voice registration status
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_get_voice_reg_status(int sim_id, ql_nw_reg_status_info_t *p_info);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get data registration status.
  @param[in] sim id number
  @param[out] p_info Pointer that point to ql_nw_reg_status_info_t
  @return Whether to successfully get the data registration status
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_get_data_reg_status(int sim_id, ql_nw_reg_status_info_t *p_info);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get current signal strength.
  @param[in] sim id number
  @param[out] p_info Pointer that point to ql_nw_signal_strength_info_t
  @param[out] p_level: signal strength level
  @return Whether to successfully get the signal strength
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_get_signal_strength(int sim_id, ql_nw_signal_strength_info_t *p_info, QL_NW_SIGNAL_STRENGTH_LEVEL_E* p_level);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get current cell acccess status.
  @param[in] sim id number
  @param[out] p_info Pointer that point to QL_NW_CELL_ACCESS_STATE_TYPE_E
  @return Whether to successfully get the cell access status
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_get_cell_access_status(int sim_id, QL_NW_CELL_ACCESS_STATE_TYPE_E *p_info);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get network time.
  @param[in] sim id number
  @param[out] p_info Pointer that point to ql_nw_nitz_time_info_t 
  @return Whether to successfully get the network time
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_get_nitz_time_info(int sim_id, ql_nw_nitz_time_info_t *p_info);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief register voice registration event.
  @param[in] sim id number
  @param[in] cb_func Voice registration indication callback function
  @return Whether the voice registration event was successfully registered.
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
 */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_set_voice_reg_ind_cb(int sim_id, ql_ms_nw_voice_reg_ind_cb cb_func);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief register data registration event.
  @param[in] sim id number
  @param[in] cb_func Data registration indication callback function
  @return Whether the data registration event was successfully registered.
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_set_data_reg_ind_cb(int sim_id, ql_ms_nw_data_reg_ind_cb cb_func);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief register signal strength event.
  @param[in] sim id number
  @param[in] cb_func Signal strength indication callback function
  @return Whether the signal strength event was successfully registered
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_set_signal_strength_ind_cb(int sim_id, ql_ms_nw_signal_strength_ind_cb cb_func);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief register cell access status event.
  @param[in] sim id number
  @param[in] cb_func Cell access status indication callback function
  @return Whether the cell access status event was successfully registered
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_set_cell_access_status_ind_cb(int sim_id, ql_ms_nw_cell_access_status_ind_cb cb_func);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief register network time event.
  @param[in] sim id number
  @param[in] cb_func nitz time update indication callback function
  @return Whether the network time event was successfully registered
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_set_nitz_time_update_ind_cb(int sim_id, ql_ms_nw_nitz_time_update_ind_cb cb_func);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief register wea alert event.
  @param[in] sim id number
  @param[in] cb_func wea alert indication callback function
  @return Whether the network time event was successfully registered
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_set_wea_alert_ind_cb(int sim_id, ql_ms_nw_wea_reg_ind_cb cb_func);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief register etws alert event.
  @param[in] sim id number
  @param[in] cb_func etws alert indication callback function
  @return Whether the network time event was successfully registered
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_set_etws_alert_ind_cb(int sim_id, ql_ms_nw_etws_reg_ind_cb cb_func);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set wea alert config.
  @param[in] sim id number
  @param[in] item Items to set.
  @param[in] p_info Pointer that point to ql_nw_wea_config_t.
  @return Whether to successfully set the wea config.
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_set_wea_config(int sim_id, int item, ql_nw_wea_config_t *p_info);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Gets wea config.
  @param[in] sim id number
  @param[out] p_config wea config.
  @return Whether the wea config was successfully obtained.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_get_wea_config(int sim_id, ql_nw_wea_config_t *p_config);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set etws alert config.
  @param[in] sim id number
  @param[in] etws config.
  @return Whether to successfully set the etws config.
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_set_etws_config(int sim_id, uint8_t enable_etws);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get etws alert config.
  @param[in] sim id number
  @param[out] p_enable_etws Pointer.
  @return Whether to successfully set the etws config.
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_get_etws_config(int sim_id, uint8_t* p_enable_etws);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief get high capability subscription.
  @param[in] sim id number
  @param[out] p_high_cap pointer that point to QL_NW_BIND_SUB_TYPE_E
  @return Whether the high capability subscription was successfully obtained.
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_get_high_cap_sub(QL_NW_BIND_SUB_TYPE_E *p_high_cap);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief set the SIM card index that can uniquely register to the 5G network. 
  the main difference between high and non-high capability subscription is that high capability 
  subscription can register to 5G network while non-high capability subscription can only 
  register to LTE or GSM.
  @param[in] sim id number
  @param[in] high_cap high capability subscription 
  @return Whether to successfully set the high capability subscription.
  @retval QL_ERR_OK successful
  @retval QL_ERR_NOT_INIT uninitialized
  @retval QL_ERR_SERVICE_NOT_READY service is not ready
  @retval QL_ERR_INVALID_ARG Invalid arguments
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_set_high_cap_sub(QL_NW_BIND_SUB_TYPE_E high_cap);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Registration server error callback. Currently, only if the server exits abnormally, 
  the callback function will be executed, and the error code is QL_ERR_ABORTED;
  @param[in] sim id number
  @param[in] cb  Callback function 
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_nw_set_service_error_cb(ql_nw_service_error_cb_f cb);

#ifdef __cplusplus
}
#endif


#endif

