/*-----------------------------------------------------------------------------------------------*/
/**
  @file ql_ms_sms.h 
  @brief short message service API, support Dual Sim Dual Active(DSDA). 
*/
/*-----------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  Copyright (c) 2021 Quectel Wireless Solution, Co., Ltd. All Rights Reserved.
  Quectel Wireless Solution Proprietary and Confidential.
-------------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  EDIT HISTORY
  This section contains comments describing changes made to the file.
  Notice that changes are listed in reverse chronological order.
  $Header: $
  when       who          what, where, why
  --------   ---          ----------------------------------------------------------
  2021069   Stan.li       Created.
-------------------------------------------------------------------------------------------------*/
#ifndef __QL_MS_SMS_H__
#define __QL_MS_SMS_H__
#ifdef __cplusplus
extern "C" {
#endif

#include "ql_type.h"
#include "ql_sms.h"


typedef void (*ql_ms_sms_msg_async_cb_f)(int sim_id, int id, int result);
typedef void (*ql_ms_sms_msg_recv_cb_f)(int sim_id, ql_sms_msg_t *p_msg, ql_sms_timestamp_t *timestamp,
                                             ql_sms_user_data_head_t *head);
typedef void (*ql_ms_sms_pdu_async_cb_f)(int sim_id, int id, int result);
typedef void (*ql_ms_sms_pdu_recv_cb_f)(int sim_id, ql_sms_pdu_t *p_pdu);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Initializes SMS service.
  @return Whether the SMS service was initialized successfully.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_sms_init(void);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Deinitializes SMS service.
  @return Whether the SMS service was deinitialized successfully.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_sms_deinit(void);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Sets the service center address.
  @param[in] sim id number
  @param[in] addr service center address.
  @param[in] len  service center address length.
  @return Whether the service center address was set successfully.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_sms_set_service_center_addr(int sim_id, char *addr, int len);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Gets the service center address.
  @param[in] sim id number
  @param[out] addr service center address.
  @param[in] len  service center address length.
  @return Whether the service center address was successfully obtained.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_sms_get_service_center_addr(int sim_id, char *addr, int len);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Sends message synchronously.
  @param[in] sim id number
  @param[in] p_msg pointer to ql_ms_sms_msg_t.
  @return Whether the message was successfully sent synchronously.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_sms_send_msg(int sim_id, ql_sms_msg_t *p_msg);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Sends message asynchronously.
  @param[in] sim id number
  @param[in] p_msg  pointer to ql_ms_sms_msg_t
  @param[out] id    id for this async operation
  @param[in] cb     async callback
  @return Whether the message was successfully sent asynchronously.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_sms_send_msg_async(int sim_id, ql_sms_msg_t *p_msg, int *id, ql_ms_sms_msg_async_cb_f cb);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Sets SMS message reception callback hanlder.
  @param[in] sim id number
  @param[in] cb message reception callback handler.
  @return Whether the message reception callback hanlder was set successfully.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_sms_set_msg_recv_cb(int sim_id, ql_ms_sms_msg_recv_cb_f cb);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Sends PDU synchronously.
  @param[in] sim id number
  @param[in] p_pdu SMS PDU.
  @return Whether the PDU was successfully sent synchronously.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_sms_send_pdu(int sim_id, ql_sms_pdu_t *p_pdu);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Sends PDU asynchronously.
  @param[in] sim id number
  @param[in] p_pdu sms pdu.
  @param[out] id id for this async operation.
  @param[in] cb async callback.
  @return Whether the PDU was successfully sent asynchronously.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_sms_send_pdu_async(int sim_id, ql_sms_pdu_t *p_pdu, int *id, ql_ms_sms_pdu_async_cb_f cb);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Sets SMS PDU reception callback hanlder.
  @param[in] sim id number
  @param[in] cb PDU reception callback handler.
  @return Whether the PDU reception callback hanlder was set successfully.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_sms_set_pdu_recv_cb(int sim_id, ql_ms_sms_pdu_recv_cb_f cb);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Registration server error callback. Currently, only if the server exits abnormally, 
  the callback function will be executed, and the error code is QL_ERR_ABORTED;
  @param[in] cb  Callback function 
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_sms_set_service_error_cb(ql_sms_service_error_cb_f cb);

#ifdef __cplusplus
}
#endif


#endif

