/*-----------------------------------------------------------------------------------------------*/
/**
  @file ql_ms_voice.h 
  @brief Voice service DSDA API. 
*/
/*-----------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  Copyright (c) 2019 Quectel Wireless Solution, Co., Ltd. All Rights Reserved.
  Quectel Wireless Solution Proprietary and Confidential.
-------------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  EDIT HISTORY
  This section contains comments describing changes made to the file.
  Notice that changes are listed in reverse chronological order.
  $Header: $
  when       who          what, where, why
  --------   ---          ----------------------------------------------------------
  20210622   Rambo.shan   Created, Added Voice DSDA API.
-------------------------------------------------------------------------------------------------*/

#ifndef __QL_MS_VOICE_H__
#define __QL_MS_VOICE_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include "ql_voice.h"

typedef void (*ql_ms_voice_ecall_status_cb_f)(int sim_id, uint32_t id, 
                                                   QL_VOICE_ECALL_MSD_TRANS_STATUS_E status);

typedef void (*ql_ms_voice_ecall_event_cb_f)(int sim_id, ql_voice_ecall_event_t *p_event);

typedef void (*ql_ms_voice_call_cb_f)(int sim_id, ql_voice_record_array_t *p_arr);

/** voice dtmf event callback */
typedef void (*ql_ms_voice_dtmf_cb_f)(int sim_id, uint32_t id, QL_VOICE_DTMF_EVENT_E event, 
                                        char *digits, int len); /**< len: length of digits */

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Initializes voice service.
  @return Whether the voice service was initialized successfully.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_init(void);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Deinitializes voice service.
  @return Whether the voice service was deinitialized successfully.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_deinit(void);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Dials a call.
  @param[in] num    phone number to dial.
  @param[in] len    length of phone number, should be less than 
                    or euqnal to QL_VOICE_MAX_PHONE_NUMBER.
  @param[out] id    call id.
  @return Whether a voice call was successfully dialed.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_dial(int sim_id, char *num, int len, uint32_t *id);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Cancels dialing with given id.
  @param[in] id call id returned from dial.
  @return Whether the voice call was successfully cancelled.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_cancel_dial(int sim_id, uint32_t id);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  hangup all dialing.
  @return Whether all voice call were successfully hangup.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_hangup_all(int sim_id);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Answers the call.
  @param[in] id call id returned from dial.
  @return Whether the voice call was successfully answered.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_answer(int sim_id, uint32_t id);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Hangs up the call.
  @param[in] id call id returned from dial.
  @return Whether the voice call was successfully hung up.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_hangup(int sim_id, uint32_t id);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Holds the call when mutil calls is activated.
  @param[in] id call id returned from dial.
  @return Whether the voice call was successfully held.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_hold(int sim_id, uint32_t id);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Releases the call from hold when mutil calls is activated.
  @param[in] id call id returned from dial.
  @return Whether the voice call was successfully unheld.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_unhold(int sim_id, uint32_t id);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Gets call records.
  @param[in] p_arr  pointer to ql_voice_record_array_t.
  @return Whether the call records were successfully obtained.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_get_records(int sim_id, ql_voice_record_array_t *p_arr);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Registers or Unregisters forwarding.
  @param[in] reg    0 - unregister, 1 - register.
  @param[in] cond   forwarding condition.
  @param[in] num    phone number.
  @param[in] len    length of phone numebr.
  @return Whether the voice call forward was registered or unregistered successfully.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_forwarding(int sim_id, int reg, QL_VOICE_FW_COND_E cond, char *num, int len);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Gets forwarding status.
  @param[in]  cond      forwarding condition.
  @param[out] p_status  pointer to ql_voice_fw_status_t.
  @return Whether the voice call forward status was successfully obtained.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_get_forwarding_status(int sim_id, QL_VOICE_FW_COND_E cond, ql_voice_fw_status_t *p_status);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Enables or disables call waiting.
  @param[in] enable 0 - disable, other - enable.
  @return Whether the voice call waiting was enabled or disabled successfully.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_waiting(int sim_id, int enable);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Gets call waiting status.
  @param[out] enabled 0 - waiting is disabled, 1 - waiting is enabled.
  @return Whether the voice call waiting status was successfully obtained.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_get_waiting_status(int sim_id, int *enabled);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Enables or disables auto answer.
  @param[in] enable 0 - disable, other - enable.
  @param[in] sec    wait this `sec' seconds before auto answer.
  @return Whether the voice call autoanswer was enabled or disabled successfully.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_autoanswer(int sim_id, int enable, uint32_t sec);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Sends a DTMF(Dual Tone Multi Frequency) character over the call ID.
  @param[in] id call id returned from dial.
  @param[in] c DTMF character to be sent. Valid DTMF characters are 0-9, A-D, '*', '#'.
  @return Whether a DTMF character was successfully sent.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_send_dtmf_char(int sim_id, uint32_t id, char c);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Sets voice call callback handler.
  @param[in] cb call back handler.
  @return Whether the voice call callback handler was successfully set.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_set_call_cb(int sim_id, ql_ms_voice_call_cb_f cb);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Sets voice dtmf callback handler.
  @param[in] cb call back handler.
  @return Whether the voice call DTMF repcetion callback handler was successfully set.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_set_dtmf_cb(int sim_id, ql_ms_voice_dtmf_cb_f cb);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Dials eCall.
  @param[in] p_info eCall info.
  @return Whether a eCall was successfully dialed.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_ecall_dial(int sim_id, ql_voice_ecall_info_t *p_info);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Hangs up eCall.
  @return Whether the eCall was successfully hung up.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_ecall_hangup(int sim_id);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Updates eCall MSD.
  @param[in] msd Minimum Set of Data.
  @param[in] msd_len Length of Minimum Set of Data.
  @return Whether the eCall MSD was successfully updated.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_ecall_update_msd(int sim_id, const uint8_t *msd, uint32_t msd_len);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Pushes eCall MSD.
  @param[out] state eCall state.
  @return Whether the eCall MSD was successfully pushed.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_ecall_push_msd(int sim_id, QL_VOICE_ECALL_STATE_E *state);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Gets eCall config.
  @param[in] p_config eCall config.
  @return Whether the eCall config was successfully obtained.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_ecall_get_config(int sim_id, ql_voice_ecall_config_t *p_config);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Sets eCall config.
  @param[in] item Items to set.
  @param[in] p_config eCall config.
  @return Whether the eCall config was successfully set.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_ecall_set_config(int sim_id, int item, ql_voice_ecall_config_t *p_config);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Sets eCall event callback handler.
  @param[in] cb call back handler.
  @return Whether the eCall event callback handler was successfully set.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_ecall_set_event_cb(int sim_id, ql_ms_voice_ecall_event_cb_f cb);

/*-----------------------------------------------------------------------------------------------*/
/** 
  @brief  Sets eCall status callback handler.
  @param[in] cb call back handler.
  @return Whether the eCall status callback handler was successfully set.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_INVALID_ARG invalid argument.
  @retval QL_ERR_UNKNOWN unknown error, failed to connect to service.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_ecall_set_status_cb(int sim_id, ql_ms_voice_ecall_status_cb_f cb);

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Registration server error callback. Currently, only if the server exits abnormally, 
  the callback function will be executed, and the error code is QL_ERR_ABORTED;
  @param[in] cb  Callback function 
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_ms_voice_set_service_error_cb(ql_voice_service_error_cb_f cb);

#ifdef __cplusplus
}
#endif

#endif  /* __QL_VOICE_H__ */

