/*-----------------------------------------------------------------------------------------------*/
/**
  @file ql_sim.h
  @brief SIM service API
*/
/*-----------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  Copyright (c) 2024 mobiletek Wireless Solution, Co., Ltd. All Rights Reserved.
  mobiletek Wireless Solution Proprietary and Confidential.
-------------------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------------------
  EDIT HISTORY
  This section contains comments describing changes made to the file.
  Notice that changes are listed in reverse chronological order.
  $Header: $
  when       who          what, where, why
  --------   ---------    -----------------------------------------------------------------
  20241022    yq.wang      Created .
-------------------------------------------------------------------------------------------------*/

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/epoll.h>
#include <unistd.h>
#include <errno.h>
#include <pthread.h>
#include <string.h>
#include <fcntl.h>

#include "ql_sim.h"
#include "ql_type.h"
#include "mbtk_type.h"
#include "mbtk_ril_api.h"
#include "mbtk_list.h"
#include "mbtk_log.h"

/*----------------------------------------------DEFINE-------------------------------------------*/
#define MBTK_ERR_OK             0

#define MBTK_PIN_VALUE_LENGTH   16
#define MBTK_BUFF_TEMP_SIZE_16  16
#define MBTK_BUFF_TEMP_SIZE_32  32
#define MBTK_BUFF_TEMP_SIZE_64  64
#define MBTK_BUFF_TEMP_SIZE_128 128

#define MBTK_READ_EVENT_SIZE    1
#define MBTK_WRITE_EVENT_SIZE   1

#define QL_SIM_LOGE LOGE
#define QL_SIM_LOGD LOGD

#define MBTK_PIN_ENABLE 1
/*----------------------------------------------DEFINE-------------------------------------------*/

/*----------------------------------------------ENUM-------------------------------------------*/
typedef enum {
    QL_SIM_EVENT_THREAD_QUIT = 0,
    QL_SIM_EVENT_STATE_CHANGE,
    QL_SIM_EVENT_MAX = 255,
}ql_sim_event_enum;

/*----------------------------------------------ENUM-------------------------------------------*/

/*----------------------------------------------STRUCT-------------------------------------------*/
typedef struct {
    mbtk_ril_handle*       handle;
    int                       control[2];
    pthread_t                 cb_thread_id;
    ql_sim_card_status_cb_f   status_cb;
    ql_sim_service_error_cb_f server_cb;
}ql_sim_info_handle_t;

/*----------------------------------------------STRUCT-------------------------------------------*/

/*----------------------------------------------GLOBAL STATIC VARIABLE---------------------------*/
static ql_sim_info_handle_t* sim_handle = NULL;

/*----------------------------------------------GLOBAL STATIC VARIABLE---------------------------*/

/*----------------------------------------------SIM FUNCTION-------------------------------------*/

static bool check_slot_valid(QL_SIM_SLOT_E slot)
{
    if (slot != QL_SIM_SLOT_1 && slot != QL_SIM_SLOT_2)
    {
        QL_SIM_LOGE("bad slot: %d, slot should be 1 or 2", slot);
        return false;
    }
    
    return true;
}

static void ql_slot_convert_to_mbtk(QL_SIM_SLOT_E ql_slot,mbtk_sim_type_enum *mbtk_slot)
{
    if(ql_slot == QL_SIM_SLOT_1)
    {
        *mbtk_slot = MBTK_SIM_1;
    }
    
    if(ql_slot == QL_SIM_SLOT_2)
    {
        *mbtk_slot = MBTK_SIM_2;
    }
    return;
    
}

static void ql_sim_state_change_cb(const void* data, int data_len)
{
    if(data_len != sizeof(mbtk_ril_sim_state_info_t))
    {
        QL_SIM_LOGE("[%s] data_len[%d] than sim[%d] fail. ", __func__, data_len, sizeof(mbtk_ril_sim_state_info_t));
    }
    else
    {
        mbtk_ril_sim_state_info_t *ptr = (mbtk_ril_sim_state_info_t*)data;
        QL_SIM_LOGD("[%s] state[%d] type[%d].", __func__, ptr->sim_state , ptr->sim_type);
        ql_sim_event_enum cmd = QL_SIM_EVENT_STATE_CHANGE;
        if(sim_handle->control[0] >= 0 && ptr->sim_state != MBTK_SIM_STATE_UNKNOWN)
        {
            int ret = write(sim_handle->control[0], &cmd, MBTK_WRITE_EVENT_SIZE);
            if(ret != MBTK_WRITE_EVENT_SIZE)
            {
                QL_SIM_LOGE("[%s] write fail.[%d]", __func__, ret);
            }
        }
    }
}

static void ql_sim_server_change_cb(const void* data, int data_len)
{
    if(data) {
        const uint8 *ptr = (const uint8*)data;
        mbtk_ril_ser_state_enum state = (mbtk_ril_ser_state_enum)ptr[0];
        LOGD("ril server state : %d.\n", state);
        if(sim_handle->server_cb && MBTK_RIL_SER_STATE_EXIT == state) {
            sim_handle->server_cb(QL_ERR_ABORTED);
        }
    }
}

static int ql_epoll_register(int epoll_fd, int fd)
{
    struct epoll_event  ev;
    int    ret, flags;

    /* important: make the fd non-blocking */
    flags = fcntl(fd, F_GETFL);
    fcntl(fd, F_SETFL, flags | O_NONBLOCK);

    ev.events  = EPOLLIN;
    ev.data.fd = fd;

    do
    {
        ret = epoll_ctl(epoll_fd, EPOLL_CTL_ADD, fd, &ev);
        if(ret < 0)
        {
            QL_SIM_LOGE("[%s] epoll_ctl fail.[%s]", __func__, strerror(errno));
        }
    } while (ret < 0 && errno == EINTR);

    return QL_ERR_OK;
}

static void* ql_cb_thread(void* arg)
{
    ql_sim_info_handle_t* info = (ql_sim_info_handle_t*)arg;

    if(info->control[0] < 0 && info->control[0] < 0)
    {
        if(socketpair(AF_LOCAL, SOCK_STREAM, 0, info->control ) < 0)
        {
            QL_SIM_LOGE("[%s] control creat fail.", __func__);
            return NULL;
        }
    }

    int ne = 0;
    int nevents = 0;
    int ret = 0;
    struct epoll_event events[3];
    int control_fd = info->control[1];
    ql_sim_event_enum cmd = QL_SIM_EVENT_THREAD_QUIT;

    //A maximum of three events can be processed simultaneously
    int epoll_fd = epoll_create(3);
    ql_epoll_register(epoll_fd, control_fd);

    QL_SIM_LOGE("[%s] thread start run.", __func__);
    while(1)
    {
        nevents = epoll_wait(epoll_fd, events, 3, -1);
        if(nevents < 0)
        {
            if(errno != EINTR)
            {
                QL_SIM_LOGE("[%s] epoll_wait fail.[%s]", __func__, strerror(errno));
            }
            continue;
        }

        for (ne = 0; ne < nevents; ne++)
        {
            if((events[ne].events & (EPOLLERR|EPOLLHUP)) != 0)
            {
                QL_SIM_LOGE("[%s] EPOLLERR or EPOLLHUP event error.", __func__);
                break;
            }

            if ((events[ne].events & EPOLLIN) != 0)
            {
                if (events[ne].data.fd == control_fd)
                {
                    ret = read(control_fd, &cmd, MBTK_READ_EVENT_SIZE);
                    if(ret != MBTK_READ_EVENT_SIZE)
                    {
                        QL_SIM_LOGE("[%s] read fail.[%d]", __func__, ret);
                    }
                    else
                    {
                        switch(cmd)
                        {
                            case QL_SIM_EVENT_THREAD_QUIT:
                            {
                                close(epoll_fd);
                                epoll_fd = -1;
                                if(info->control[0] >= 0)
                                {
                                    close(info->control[0]);
                                    info->control[0] = -1;
                                }
                                if(info->control[1] >= 0)
                                {
                                    close(info->control[1]);
                                    info->control[1] = -1;
                                }
                                QL_SIM_LOGD("[%s] thread quit.", __func__);
                                return NULL;
                            }
                            case QL_SIM_EVENT_STATE_CHANGE:
                            {
                                if(info->status_cb)
                                {
                                    ql_sim_card_info_t sim_info = {0};
                                    ql_sim_get_card_info(QL_SIM_SLOT_1, &sim_info);
                                    info->status_cb(QL_SIM_SLOT_1, &sim_info);
                                    ql_sim_get_card_info(QL_SIM_SLOT_2, &sim_info);
                                    info->status_cb(QL_SIM_SLOT_2, &sim_info);
                                }
                                
                                break;
                            }
                            default:
                            {
                                QL_SIM_LOGE("[%s] unknown event.[%d]", __func__, cmd);
                                break;
                            }
                        }
                    }
                }
            }
        }
    }
}

static int ql_cb_thread_creat(ql_sim_info_handle_t *info)
{
    if(info->cb_thread_id == 0)
    {
        pthread_attr_t thread_attr;
        pthread_attr_init(&thread_attr);
        if(pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED))
        {
            QL_SIM_LOGE("[%s] pthread_attr_setdetachstate fail.", __func__);
            return QL_ERR_FAILED;
        }
        if(pthread_create(&(info->cb_thread_id), &thread_attr, ql_cb_thread, info))
        {
            QL_SIM_LOGE("[%s] pthread_create fail.[%ld]", __func__, info->cb_thread_id);
            return QL_ERR_FAILED;
        }
        QL_SIM_LOGD("[%s] pthread_create success.[%ld]", __func__, info->cb_thread_id);
    }

    return QL_ERR_OK;
}

static int ql_cb_thread_free(ql_sim_info_handle_t *info)
{
    int ret = 0;
    if(info->cb_thread_id != 0)
    {
        ql_sim_event_enum cmd = QL_SIM_EVENT_THREAD_QUIT;
        if(info->control[0] >= 0)
        {
            ret = write(info->control[0], &cmd, MBTK_WRITE_EVENT_SIZE);
            if(ret != MBTK_WRITE_EVENT_SIZE)
            {
                QL_SIM_LOGE("[%s] write fail.[%d]", __func__, ret);
                return QL_ERR_FAILED;
            }
        }
        info->cb_thread_id = 0;
        QL_SIM_LOGE("[%s] pthread quit success.", __func__);
    }

    return QL_ERR_OK;
}

/*----------------------------------------------SIM FUNCTION-------------------------------------*/

/*----------------------------------------------SIM API------------------------------------------*/

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Initializes SIM service.
  @note You must call this function before other functions can be used in this module.
  @return Whether the SIM service was intialized successfully.
  @retval QL_ERR_OK successful
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_init(void)
{
    if(NULL == sim_handle)
    {
        sim_handle = (ql_sim_info_handle_t*)malloc(sizeof(ql_sim_info_handle_t));
        if(NULL == sim_handle)
        {
            QL_SIM_LOGE("[%s] sim handle malloc fail.", __func__);
            return QL_ERR_FAILED;
        }

        sim_handle->handle = mbtk_ril_open(MBTK_AT_PORT_DEF);
        if(NULL == sim_handle->handle)
        {
            QL_SIM_LOGE("[%s] mbtk handle init fail.", __func__);
            goto error_1;
        }

        int ret = mbtk_ds_sim_state_change_cb_reg(MBTK_SIM_1,ql_sim_state_change_cb);
        if(ret != MBTK_ERR_OK)
        {
            QL_SIM_LOGE("[%s] set sim state cb fail.[%d]", __func__, ret);
            goto error_2;
        }

        ret = mbtk_ds_sim_state_change_cb_reg(MBTK_SIM_2,ql_sim_state_change_cb);
        if(ret != MBTK_ERR_OK)
        {
            QL_SIM_LOGE("[%s] set sim state cb fail.[%d]", __func__, ret);
            goto error_2;
        }
        
        ret = mbtk_ril_ser_state_change_cb_reg(ql_sim_server_change_cb);
        if(ret != MBTK_ERR_OK)
        {
            QL_SIM_LOGE("[%s] set sim server cb fail.[%d]", __func__, ret);
            goto error_2;
        }

        sim_handle->status_cb = NULL;
        sim_handle->server_cb = NULL;
        sim_handle->control[0] = -1;
        sim_handle->control[1] = -1;
        sim_handle->cb_thread_id = 0;
    }

    return QL_ERR_OK;

error_2:
    if(sim_handle->handle)
    {
        mbtk_ril_close(MBTK_AT_PORT_DEF);
        sim_handle->handle = NULL;
    }
error_1:
    if(sim_handle)
    {
        free(sim_handle);
        sim_handle = NULL;
    }
    return QL_ERR_FAILED;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Deinitializes SIM service.
  @return Whether the SIM service was deintialized successfully.
  @retval QL_ERR_OK successful
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_deinit(void)
{
    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    int ret = 0;
    sim_handle->server_cb = NULL;
    sim_handle->status_cb = NULL;

    if(NULL != sim_handle->handle)
    {
        ret = mbtk_ril_close(MBTK_AT_PORT_DEF);
        if(ret != MBTK_ERR_OK)
        {
            QL_SIM_LOGE("[%s] mbtk handle deinit fail.[%d]", __func__, ret);
            return QL_ERR_FAILED;
        }
        sim_handle->handle = NULL;
    }

    ret = ql_cb_thread_free(sim_handle);
    if(ret != QL_ERR_OK)
    {
        QL_SIM_LOGE("[%s] cb thread free deinit fail.", __func__);
        return QL_ERR_FAILED;
    }

    free(sim_handle);
    sim_handle = NULL;
    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Gets the IMSI (for 3GPP) or IMSI_M (for 3GPP2) from the SIM in ASCII form.
  @param [in] slot Slot to be used.
  @note This function is only supported by Identify card in slot 1
  @param [in] app_type Application type.
  @note This function is only supported by 3GPP applications
  @param [out] imsi Buffer to fill IMSI data.
  @param [in] imsi_len Buffer length.
  @return Whether the IMSI was successfully obtained.
  @retval QL_ERR_OK successful
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_get_imsi(QL_SIM_SLOT_E slot, QL_SIM_APP_TYPE_E app_type, char *imsi, int imsi_len)
{
    //UNUSED(slot);
    UNUSED(app_type);

    mbtk_sim_type_enum sim_id = -1;
    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    if(NULL == imsi)
    {
        QL_SIM_LOGE("[%s] imsi is NULL.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    
    if(imsi_len < QL_SIM_IMSI_LENGTH)
    {
        QL_SIM_LOGE("[%s] imsi buf is too short.", __func__);
        return QL_ERR_BUF_OVERFLOW;
    }
    
    if (!check_slot_valid(slot))
    {
         QL_SIM_LOGE("[%s] check_slot_valid failed.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    ql_slot_convert_to_mbtk(slot,&sim_id);
    
    char temp_buff[MBTK_BUFF_TEMP_SIZE_32] = {0};
    memset(imsi, 0x0, imsi_len);
    memset(temp_buff, 0x0, MBTK_BUFF_TEMP_SIZE_32);
    int ret = mbtk_ds_imsi_get(sim_handle->handle,sim_id,(void *)temp_buff);
    if(ret != MBTK_ERR_OK)
    {
        QL_SIM_LOGE("[%s] imsi get fail.[%d]", __func__, ret);
        return QL_ERR_FAILED;
    }

    memcpy(imsi, temp_buff, QL_SIM_IMSI_LENGTH);
    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Retrieves the Integrated Circuit Card ID (ICCID) stored on the card.
  @param [in] slot Slot to be used(the parameter currently does not support).
  @note This function is only supported by Identify card in slot 1
  @param [out] iccid Buffer to fill ICCID data.
  @param [in] iccid_len Buffer length.
  @return Whether the ICCID was successfully obtained.
  @retval QL_ERR_OK successful
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_get_iccid(QL_SIM_SLOT_E slot, char *iccid, int iccid_len)
{
    //UNUSED(slot);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    if(NULL == iccid)
    {
        QL_SIM_LOGE("[%s] iccid is NULL.", __func__);
        return QL_ERR_INVALID_ARG;
    }

    if(iccid_len < QL_SIM_ICCID_LENGTH)
    {
        QL_SIM_LOGE("[%s] imsi buf is too short.", __func__);
        return QL_ERR_BUF_OVERFLOW;
    }
    mbtk_sim_type_enum sim_id = -1;
    if(!check_slot_valid(slot))
    {
        QL_SIM_LOGE("[%s] check_slot_valid failed.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    ql_slot_convert_to_mbtk(slot,&sim_id);
    char temp_buff[MBTK_BUFF_TEMP_SIZE_32] = {0};
    memset(iccid, 0x0, iccid_len);
    memset(temp_buff, 0x0, MBTK_BUFF_TEMP_SIZE_32);
    int ret = mbtk_ds_iccid_get(sim_handle->handle,sim_id, (void *)temp_buff);
    if(ret != MBTK_ERR_OK)
    {
        QL_SIM_LOGE("[%s] iccid get fail.[%d]", __func__, ret);
        return QL_ERR_FAILED;
    }

    memcpy(iccid, temp_buff, QL_SIM_ICCID_LENGTH);
    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Retrieves the device phone number stored on the card.
  @param [in] slot Slot to be used.
  @note This function is only supported by Identify card in slot 1
  @param [in] app_type Application type.
  @note This function is only supported by 3GPP applications
  @param [out] phone_num Buffer to fill phone number.
  @param [in] phone_num_len Buffer length.
  @return Whether the phone number was successfully retrieved.
  @retval QL_ERR_OK successful
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_get_phone_num(QL_SIM_SLOT_E slot, QL_SIM_APP_TYPE_E app_type,
                          char *phone_num, int phone_num_len)
{
    //UNUSED(slot);
    UNUSED(app_type);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    if(NULL == phone_num)
    {
        QL_SIM_LOGE("[%s] phone_num is NULL.", __func__);
        return QL_ERR_INVALID_ARG;
    }

    if(phone_num_len < QL_SIM_PHONE_NUMBER_MAX)
    {
        QL_SIM_LOGE("[%s] phone_num buf is too short.", __func__);
        return QL_ERR_BUF_OVERFLOW;
    }

    mbtk_sim_type_enum sim_id = -1;
    if(!check_slot_valid(slot))
    {
        QL_SIM_LOGE("[%s] check_slot_valid failed.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    ql_slot_convert_to_mbtk(slot,&sim_id);
    
    char temp_buff[MBTK_BUFF_TEMP_SIZE_128] = {0};
    memset(phone_num, 0x0, phone_num_len);
    memset(temp_buff, 0x0, MBTK_BUFF_TEMP_SIZE_128);
    int ret = mbtk_ds_phone_number_get(sim_handle->handle,sim_id, (void *)temp_buff);
    if(ret != MBTK_ERR_OK)
    {
        QL_SIM_LOGE("[%s] phone_num get fail.[%d]", __func__, ret);
        return QL_ERR_FAILED;
    }

    memcpy(phone_num, temp_buff, QL_SIM_PHONE_NUMBER_MAX);
    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Retrieves the preferred operators stored on the card.
  @param [in] slot Slot to be used.
  @note This function is only supported by Identify card in slot 1
  @param [out] list Buffer to hold operators.
  @note This function is only supported by 3GPP applications
  @return Whether the preferred operators were successfully retrieved.
  @retval QL_ERR_OK successful
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry
  @retval Other error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_get_operators(QL_SIM_SLOT_E slot, ql_sim_operator_list_t *list)
{
    //UNUSED(slot);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    if(NULL == list)
    {
        QL_SIM_LOGE("[%s] list is NULL.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    mbtk_sim_type_enum sim_id = -1;
    if(!check_slot_valid(slot))
    {
        QL_SIM_LOGE("[%s] check_slot_valid failed.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    ql_slot_convert_to_mbtk(slot,&sim_id);
#if 0
    list_node_t* operators_list = NULL;
    mbtk_net_info_t* operator = NULL;
    char temp_buff[MBTK_BUFF_TEMP_SIZE_16] = {0};
    int operators_size = 0;
    int ret = mbtk_available_net_get(sim_handle->handle, &operators_list);
    if(ret != MBTK_ERR_OK)
    {
        QL_SIM_LOGE("[%s] available operators get fail.[%d]", __func__, ret);
        if(NULL != operators_list)
        {
            list_free(operators_list);
            operators_list = NULL;
        }
        return QL_ERR_FAILED;
    }

    if(NULL == operators_list)
    {
        QL_SIM_LOGE("[%s] operators list is NULL.", __func__);
        return QL_ERR_FAILED;
    }
    else
    {
        list_first(operators_list);
        memset(list, 0x0, sizeof(ql_sim_operator_list_t));
        while ((operator = (mbtk_net_info_t*) list_next(operators_list)))
        {
            QL_SIM_LOGD("[%s]operators: %d, %d, %d, %d.", __func__, operator->net_sel_mode, operator->net_type, operator->net_state, operator->plmn);
            memset(temp_buff, 0x0, MBTK_BUFF_TEMP_SIZE_16);
            ret = sprintf(temp_buff, "%d", operator->plmn);
            list->operators[operators_size].mnc_len = ret - QL_SIM_MCC_LENGHT;
            if(ret > QL_SIM_MCC_LENGHT && list->operators[operators_size].mnc_len <= QL_SIM_MNC_MAX)
            {
                memcpy(list->operators[operators_size].mcc, temp_buff, QL_SIM_MCC_LENGHT);
                memcpy(list->operators[operators_size].mnc, temp_buff + QL_SIM_MCC_LENGHT, list->operators[operators_size].mnc_len);
            }
            else
            {
                QL_SIM_LOGE("[%s] MCC MNC length error.", __func__);
                list_free(operators_list);
                operators_list = NULL;
                return QL_ERR_FAILED;
            }
            operators_size++;
        }
        list->len = operators_size;
    }
    list_free(operators_list);
    operators_list = NULL;
#else
    mbtk_net_info_array_t net_list;
    mbtk_ril_err_enum err = mbtk_ds_available_net_get(sim_handle->handle,sim_id,&net_list);
    if(err != MBTK_RIL_ERR_SUCCESS) {
        LOGE("Error : %d", err);
    } else {
        LOGD("Available net number:%d\n", net_list.num);
        int i = 0;
        char temp_buff[MBTK_BUFF_TEMP_SIZE_16] = {0};
        int ret;
        memset(list, 0x0, sizeof(ql_sim_operator_list_t));
        while(i < net_list.num) {
            memset(temp_buff, 0x0, MBTK_BUFF_TEMP_SIZE_16);
            LOGD("NET : %d,%d,%d,%d\n", net_list.net_info[i].net_sel_mode,
                net_list.net_info[i].net_type, net_list.net_info[i].net_state,
                net_list.net_info[i].plmn);

            ret = sprintf(temp_buff, "%d", net_list.net_info[i].plmn);
            list->operators[i].mnc_len = ret - QL_SIM_MCC_LENGHT;
            if(ret > QL_SIM_MCC_LENGHT && list->operators[i].mnc_len <= QL_SIM_MNC_MAX)
            {
                memcpy(list->operators[i].mcc, temp_buff, QL_SIM_MCC_LENGHT);
                memcpy(list->operators[i].mnc, temp_buff + QL_SIM_MCC_LENGHT, list->operators[i].mnc_len);
            }
            else
            {
                QL_SIM_LOGE("[%s] MCC MNC length error.", __func__);
                return QL_ERR_FAILED;
            }
            i++;
        }
        list->len = net_list.num;
    }

#endif
    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Enables the PIN on an application.
  @param [in] slot Slot to be used.
  @note This function is only supported by Identify card in slot 1
  @param [in] app_type Application type.
  @note This function is only supported by 3GPP applications
  @param [in] pin PIN to be used.
  @note This function is only supported by Level 1 user verification
  @param [in] pin_value PIN value. NULL terminated.
  @return Whether the PIN was successfully enabled.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_enable_pin(QL_SIM_SLOT_E slot, QL_SIM_APP_TYPE_E app_type,
                             QL_SIM_PIN_E pin, const char *pin_value)
{
    //UNUSED(slot);
    UNUSED(app_type);
    UNUSED(pin);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    if(NULL == pin_value)
    {
        QL_SIM_LOGE("[%s] pin_value is NULL.", __func__);
        return QL_ERR_INVALID_ARG;
    }

    int pin_len = strlen(pin_value);
    if(pin_len > QL_SIM_PIN_MAX)
    {
        QL_SIM_LOGE("[%s] pin length is too long.", __func__);
        return QL_ERR_ARG_TOO_LONG;
    }
    else if(pin_len <= 0)
    {
        QL_SIM_LOGE("[%s] pin length is too short.", __func__);
        return QL_ERR_INVALID_ARG;
    }

    mbtk_sim_type_enum sim_id = -1;
    if(!check_slot_valid(slot))
    {
        QL_SIM_LOGE("[%s] check_slot_valid failed.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    ql_slot_convert_to_mbtk(slot,&sim_id);
    mbtk_sim_lock_info_t info = {0};
    info.type = MBTK_SIM_LOCK_TYPE_ENABLE;
    memcpy(info.pin1, pin_value, pin_len);

    int err = mbtk_ds_sim_lock_set(sim_handle->handle,sim_id, &info);
    if(err)
    {
        QL_SIM_LOGE("[%s] enable pin fail.[%d]", __func__, err);
        return QL_ERR_FAILED;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Disables the PIN on an application.
  @param [in] slot Slot to be used.
  @note This function is only supported by Identify card in slot 1
  @param [in] app_type Application type.
  @note This function is only supported by 3GPP applications
  @param [in] pin PIN to be used.
  @note This function is only supported by Level 1 user verification
  @param [in] pin_value PIN value. NULL terminated.
  @return Whether the PIN was successfully disabled.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_disable_pin(QL_SIM_SLOT_E slot, QL_SIM_APP_TYPE_E app_type,
                             QL_SIM_PIN_E pin, const char *pin_value)
{
    //UNUSED(slot);
    UNUSED(app_type);
    UNUSED(pin);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    if(NULL == pin_value)
    {
        QL_SIM_LOGE("[%s] pin_value is NULL.", __func__);
        return QL_ERR_INVALID_ARG;
    }

    int pin_len = strlen(pin_value);
    if(pin_len > QL_SIM_PIN_MAX)
    {
        QL_SIM_LOGE("[%s] pin length is too long.", __func__);
        return QL_ERR_ARG_TOO_LONG;
    }
    else if(pin_len <= 0)
    {
        QL_SIM_LOGE("[%s] pin length is too short.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    mbtk_sim_type_enum sim_id = -1;
    if(!check_slot_valid(slot))
    {
        QL_SIM_LOGE("[%s] check_slot_valid failed.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    ql_slot_convert_to_mbtk(slot,&sim_id);
    
    mbtk_sim_lock_info_t info = {0};
    info.type = MBTK_SIM_LOCK_TYPE_DISABLE;
    memcpy(info.pin1, pin_value, pin_len);

    int err = mbtk_ds_sim_lock_set(sim_handle->handle,sim_id, &info);
    if(err)
    {
        QL_SIM_LOGE("[%s] disenable pin fail.[%d]", __func__, err);
        return QL_ERR_FAILED;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Verifies the PIN value of an application.
  @param [in] slot Slot to be used.
  @note This function is only supported by Identify card in slot 1
  @param [in] app_type Application type.
  @note This function is only supported by 3GPP applications
  @param [in] pin PIN to be used.
  @note This function is only supported by Level 1 user verification
  @param [in] pin_value PIN value. NULL terminated.
  @note PIN must be enabled before calling this function.
  @return Whether the PIN was successfully verified.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_verify_pin(QL_SIM_SLOT_E slot, QL_SIM_APP_TYPE_E app_type,
                             QL_SIM_PIN_E pin, const char *pin_value)
{
    //UNUSED(slot);
    UNUSED(app_type);
    UNUSED(pin);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    if(NULL == pin_value)
    {
        QL_SIM_LOGE("[%s] pin_value is NULL.", __func__);
        return QL_ERR_INVALID_ARG;
    }

    int pin_len = strlen(pin_value);
    if(pin_len > QL_SIM_PIN_MAX)
    {
        QL_SIM_LOGE("[%s] pin length is too long.", __func__);
        return QL_ERR_ARG_TOO_LONG;
    }
    else if(pin_len <= 0)
    {
        QL_SIM_LOGE("[%s] pin length is too short.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    mbtk_sim_type_enum sim_id = -1;
    if(!check_slot_valid(slot))
    {
        QL_SIM_LOGE("[%s] check_slot_valid failed.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    ql_slot_convert_to_mbtk(slot,&sim_id);
    
    mbtk_sim_lock_info_t info = {0};
    info.type = MBTK_SIM_LOCK_TYPE_VERIFY_PIN;
    memcpy(info.pin1, pin_value, pin_len);

    int err = mbtk_ds_sim_lock_set(sim_handle->handle,sim_id, &info);
    if(err)
    {
        QL_SIM_LOGE("[%s] verify pin fail.[%d]", __func__, err);
        return QL_ERR_FAILED;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Changes the PIN value of an application.
  @param [in] slot Slot to be used.
  @note This function is only supported by Identify card in slot 1
  @param [in] app_type Application type.
  @note This function is only supported by 3GPP applications
  @param [in] pin PIN to be used.
  @note This function is only supported by Level 1 user verification
  @param [in] old_pin_value Old PIN value. NULL terminated.
  @param [in] new_pin_value New PIN value. NULL terminated.
  @return Whether the PIN was successfully changed.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_change_pin(QL_SIM_SLOT_E slot, QL_SIM_APP_TYPE_E app_type,
                             QL_SIM_PIN_E pin, const char *old_pin_value, const char *new_pin_value)
{
    //UNUSED(slot);
    UNUSED(app_type);
    UNUSED(pin);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    if(NULL == old_pin_value || NULL == new_pin_value)
    {
        QL_SIM_LOGE("[%s] pin_value is NULL.", __func__);
        return QL_ERR_INVALID_ARG;
    }

    int old_pin_len = strlen(old_pin_value);
    int new_pin_len = strlen(new_pin_value);
    if(old_pin_len > QL_SIM_PIN_MAX || new_pin_len > QL_SIM_PIN_MAX)
    {
        QL_SIM_LOGE("[%s] pin length is too long.", __func__);
        return QL_ERR_ARG_TOO_LONG;
    }
    else if(old_pin_len <= 0 || new_pin_len <= 0)
    {
        QL_SIM_LOGE("[%s] pin length is too short.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    mbtk_sim_type_enum sim_id = -1;
    if(!check_slot_valid(slot))
    {
        QL_SIM_LOGE("[%s] check_slot_valid failed.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    ql_slot_convert_to_mbtk(slot,&sim_id);
    
    mbtk_sim_lock_info_t info = {0};
    info.type = MBTK_SIM_LOCK_TYPE_CHANGE;
    memcpy(info.pin1, old_pin_value, old_pin_len);
    memcpy(info.pin2, new_pin_value, new_pin_len);

    int err = mbtk_ds_sim_lock_set(sim_handle->handle,sim_id, &info);
    if(err)
    {
        QL_SIM_LOGE("[%s] change pin fail.[%d]", __func__, err);
        return QL_ERR_FAILED;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Unblocks a blocked PIN using the PUK code.
  @param [in] slot Slot to be used.
  @note This function is only supported by Identify card in slot 1
  @param [in] app_type Application type.
  @note This function is only supported by 3GPP applications
  @param [in] pin PIN to be used.
  @note This function is only supported by Level 1 user verification
  @param [in] puk_value PUK value.  NULL terminated.
  @param [in] pin_value New PIN value.  NULL terminated.
  @note The user must pass PUK1 to unblock PIN1 or PUK2 to unblock PIN2.
  @return Whether the PIN was successfully unblocked.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_unblock_pin(QL_SIM_SLOT_E slot, QL_SIM_APP_TYPE_E app_type,
                             QL_SIM_PIN_E pin, const char *puk_value, const char *pin_value)
{
    //UNUSED(slot);
    UNUSED(app_type);
    UNUSED(pin);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    if(NULL == puk_value || NULL == pin_value)
    {
        QL_SIM_LOGE("[%s] value is NULL.", __func__);
        return QL_ERR_INVALID_ARG;
    }

    int puk_len = strlen(puk_value);
    int pin_len = strlen(pin_value);
    if(puk_len > QL_SIM_PUK_LENGTH || pin_len > QL_SIM_PIN_MAX)
    {
        QL_SIM_LOGE("[%s] length is too long.", __func__);
        return QL_ERR_ARG_TOO_LONG;
    }
    else if(puk_len <= 0 || pin_len <= 0)
    {
        QL_SIM_LOGE("[%s] length is too short.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    mbtk_sim_type_enum sim_id = -1;
    if(!check_slot_valid(slot))
    {
        QL_SIM_LOGE("[%s] check_slot_valid failed.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    ql_slot_convert_to_mbtk(slot,&sim_id);
    
    mbtk_sim_lock_info_t info = {0};
    info.type = MBTK_SIM_LOCK_TYPE_VERIFY_PUK;
    memcpy(info.pin1, pin_value, pin_len);
    memcpy(info.puk, puk_value, puk_len);

    int err = mbtk_ds_sim_lock_set(sim_handle->handle,sim_id, &info);
    if(err)
    {
        QL_SIM_LOGE("[%s] unblock pin fail.[%d]", __func__, err);
        return QL_ERR_FAILED;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Retrieves the card info stored on a card.
  @param [in] slot Slot to be used.
  @note This function is only supported by Identify card in slot 1
  @param [out] p_info Pointer of ql_sim_card_info_t.
  @return Whether the card info was successfully retrieved.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_get_card_info(QL_SIM_SLOT_E slot, ql_sim_card_info_t *p_info)
{
    //UNUSED(slot);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    if(NULL == p_info)
    {
        QL_SIM_LOGE("[%s] p_info is NULL.", __func__);
        return QL_ERR_INVALID_ARG;
    }

    mbtk_sim_type_enum sim_id = -1;
    if(!check_slot_valid(slot))
    {
        QL_SIM_LOGE("[%s] check_slot_valid failed.", __func__);
        return QL_ERR_INVALID_ARG;
    }
    ql_slot_convert_to_mbtk(slot,&sim_id);
    
    mbtk_sim_state_enum sim;
    int ret = mbtk_ds_sim_state_get(sim_handle->handle,sim_id, &sim);
    if(ret != MBTK_ERR_OK)
    {
        QL_SIM_LOGE("[%s] sim state get fail.[%d]", __func__, ret);
        return QL_ERR_FAILED;
    }
    
    mbtk_sim_card_type_enum sim_card_type;
    ret = mbtk_ds_sim_type_get(sim_handle->handle,sim_id, &sim_card_type);
    if(ret)
    {
        QL_SIM_LOGE("[%s] sim type get fail.[%d]", __func__, ret);
        return QL_ERR_FAILED;
    }

    mbtk_pin_puk_last_times_t sim_last_times = {0};
    ret = mbtk_ds_sim_lock_retry_times_get(sim_handle->handle,sim_id, &sim_last_times);
    if(ret)
    {
        QL_SIM_LOGE("[%s] sim pin puk times get fail.[%d]", __func__, ret);
        return QL_ERR_FAILED;
    }

    int pin_state;
    ret = mbtk_ds_sim_lock_get(sim_handle->handle,sim_id, &pin_state);
    if(ret)
    {
        QL_SIM_LOGE("[%s] sim pin state get fail.[%d]", __func__, ret);
        return QL_ERR_FAILED;
    }

    memset(p_info, 0x0, sizeof(ql_sim_card_info_t));
    switch(sim)
    {
        case MBTK_SIM_STATE_ABSENT:                   //CARD ABSENT
        {
            p_info->state = QL_SIM_CARD_STATE_ABSENT;
            p_info->app_3gpp.app_state = QL_SIM_APP_STATE_UNKNOWN;
            p_info->app_3gpp.pin1_state = QL_SIM_PIN_STATE_UNKNOWN;
            p_info->app_3gpp.pin2_state = QL_SIM_PIN_STATE_UNKNOWN;
            break;
        }
        case MBTK_SIM_STATE_READY:                    //CARD READY
        {
            p_info->state = QL_SIM_CARD_STATE_PRESENT;
            p_info->app_3gpp.app_state = QL_SIM_APP_STATE_READY;
            if(pin_state == MBTK_PIN_ENABLE)
            {
                p_info->app_3gpp.pin1_state = QL_SIM_PIN_STATE_ENABLED_VERIFIED;
            }
            else
            {
                p_info->app_3gpp.pin1_state = QL_SIM_PIN_STATE_DISABLED;
            }
            p_info->app_3gpp.pin2_state = QL_SIM_PIN_STATE_UNKNOWN;
            break;
        }
        case MBTK_SIM_STATE_SIM_PIN:                      //SIM PIN
        {
            p_info->state = QL_SIM_CARD_STATE_PRESENT;
            p_info->app_3gpp.app_state = QL_SIM_APP_STATE_PIN1_REQ;
            if(pin_state == MBTK_PIN_ENABLE)
            {
                p_info->app_3gpp.pin1_state = QL_SIM_PIN_STATE_ENABLED_NOT_VERIFIED;
            }
            else
            {
                p_info->app_3gpp.pin1_state = QL_SIM_PIN_STATE_DISABLED;
            }
            p_info->app_3gpp.pin2_state = QL_SIM_PIN_STATE_UNKNOWN;
            break;
        }
        case MBTK_SIM_STATE_SIM_PUK:                      //SIM PUK
        {
            p_info->state = QL_SIM_CARD_STATE_PRESENT;
            p_info->app_3gpp.app_state = QL_SIM_APP_STATE_PUK1_REQ;
            if(pin_state == MBTK_PIN_ENABLE)
            {
                p_info->app_3gpp.pin1_state = QL_SIM_PIN_STATE_BLOCKED;
            }
            else
            {
                p_info->app_3gpp.pin1_state = QL_SIM_PIN_STATE_DISABLED;
            }
            p_info->app_3gpp.pin2_state = QL_SIM_PIN_STATE_UNKNOWN;
            break;
        }
        case MBTK_SIM_STATE_PH_NET_PIN:  //NETWORK PERSON PIN
        {
            p_info->state = QL_SIM_CARD_STATE_PRESENT;
            p_info->app_3gpp.app_state = QL_SIM_APP_STATE_UNKNOWN;
            if(pin_state == MBTK_PIN_ENABLE)
            {
                p_info->app_3gpp.pin1_state = QL_SIM_PIN_STATE_ENABLED_NOT_VERIFIED;
            }
            else
            {
                p_info->app_3gpp.pin1_state = QL_SIM_PIN_STATE_DISABLED;
            }
            p_info->app_3gpp.pin2_state = QL_SIM_PIN_STATE_UNKNOWN;
            break;
        }
        case MBTK_SIM_STATE_NOT_READY:                //UNKNOWN STATE
        default:
        {
            p_info->state = QL_SIM_CARD_STATE_UNKNOWN;
            p_info->app_3gpp.app_state = QL_SIM_APP_STATE_UNKNOWN;
            p_info->app_3gpp.pin1_state = QL_SIM_PIN_STATE_UNKNOWN;
            p_info->app_3gpp.pin2_state = QL_SIM_PIN_STATE_UNKNOWN;
            break;
        }
    }

    switch(sim_card_type)
    {
        case MBTK_SIM:
        case MBTK_TEST_SIM:
        {
            p_info->type = QL_SIM_CARD_TYPE_ICC;
            break;
        }
        case MBTK_USIM:
        case MBTK_TEST_USIM:
        {
            p_info->type = QL_SIM_CARD_TYPE_UICC;
            break;
        }
        default:
        {
            p_info->type = QL_SIM_CARD_TYPE_UNKNOWN;
            break;
        }
    }

    p_info->app_3gpp.pin1_num_retries = sim_last_times.p1_retry;
    p_info->app_3gpp.puk1_num_retries = sim_last_times.puk1_retry;
    p_info->app_3gpp.pin2_num_retries = sim_last_times.p2_retry;
    p_info->app_3gpp.puk2_num_retries = sim_last_times.puk2_retry;

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Reads data from a specific file on a specified application on the card.
  @param [in] slot Slot to be used.
  @param [in] app_type Application type.
  @param [inout] p_file Pointer of ql_sim_file_t.
  @return Whether the file was successfully read.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_read_file(QL_SIM_SLOT_E slot, QL_SIM_APP_TYPE_E app_type, ql_sim_file_t *p_file)
{
    UNUSED(slot);
    UNUSED(app_type);
    UNUSED(p_file);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Writes data to a specific file on a specified application on the card.
  @param [in] slot Slot to be used.
  @param [in] app_type Application type.
  @param [in] p_file Pointer of ql_sim_file_t
  @note The type of file is determined by the record number field,
    which indicates a transparent file when zero and a record-based file otherwise.
  @return Whether the file was successfully written.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_write_file(QL_SIM_SLOT_E slot, QL_SIM_APP_TYPE_E app_type, ql_sim_file_t *p_file)
{
    UNUSED(slot);
    UNUSED(app_type);
    UNUSED(p_file);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Retrieves the info of a specific file on a specified application on the card.
  @param [in] slot Slot to be used.
  @param [in] app_type Application type.
  @param [inout] p_info Pointer of ql_sim_file_info_t.
  @return Whether the file info was successfully retrieved.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_get_file_info(QL_SIM_SLOT_E slot, QL_SIM_APP_TYPE_E app_type,
                                 ql_sim_file_info_t *p_info)
{
    UNUSED(slot);
    UNUSED(app_type);
    UNUSED(p_info);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Reads phone book on a specified application on the card.
  @param [in] slot Slot to be used.
  @param [in] app_type Spplication type.
  @param [in] pb_path Phone book path. NULL terminated.
  @param [in] record_idx Record index to read. Starts from 1.
  @param [out] p_record Pointer of ql_sim_phone_book_record_t.
  @return Whether the phone book record was successfully retrieved.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_read_phone_book(QL_SIM_SLOT_E slot, QL_SIM_APP_TYPE_E app_type,
                                    const char *pb_path, uint8_t record_idx,
                                    ql_sim_phone_book_record_t *p_record)
{
    UNUSED(slot);
    UNUSED(app_type);
    UNUSED(pb_path);
    UNUSED(record_idx);
    UNUSED(p_record);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Writes phone book on a specified application on the card.
  @param [in] slot Slot to be used.
  @param [in] app_type Application type.
  @param [in] pb_path Phone book path. NULL terminated.
  @param [in] record_idx Record index to write. Starts from 1.
  @param [in] p_record Pointer of ql_sim_phone_book_record_t.
  @note If p_record->name[0] = 0 and p_record->number[0] = 0, record will be deleted.
  @return Whether the phone book record was successfully saved.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_write_phone_book(QL_SIM_SLOT_E slot, QL_SIM_APP_TYPE_E app_type,
                                    const char *pb_path, uint8_t record_idx,
                                    ql_sim_phone_book_record_t *p_record)
{
    UNUSED(slot);
    UNUSED(app_type);
    UNUSED(pb_path);
    UNUSED(record_idx);
    UNUSED(p_record);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Opens a logical channel on a UICC card.
  @param [in] slot Slot to be used.
  @param [out] channel_id Channel opened.
  @return Whether the logical channel was successfully opened.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_open_logical_channel(QL_SIM_SLOT_E slot, uint8_t *channel_id)
{
    UNUSED(slot);
    UNUSED(channel_id);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Closes a logical channel on a UICC card.
  @param [in] slot Slot to be used.
  @param [in] channel_id Channel to be closed.
  @return Whether the logical channel was successfully closed.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_close_logical_channel(QL_SIM_SLOT_E slot, uint8_t channel_id)
{
    UNUSED(slot);
    UNUSED(channel_id);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Sends an APDU to the card.
  @param [in] slot Slot to be used.
  @param [in] channel_id Channel to be used.
  @param [inout] p_apdu Pointer of ql_sim_apdu_t.
  @note You must call ql_sim_open_logical_channel before sending an APDU.
  @return Whether the APDU was successfully sent.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_send_apdu(QL_SIM_SLOT_E slot, uint8_t channel_id, ql_sim_apdu_t *p_apdu)
{
    UNUSED(slot);
    UNUSED(channel_id);
    UNUSED(p_apdu);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Sets SIM card status callback handler
  @param[in] cb call back handler.
  @return Whether the card status callback handler was successfully set.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_set_card_status_cb(ql_sim_card_status_cb_f cb)
{
    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    int ret = QL_ERR_OK;
    if(NULL != cb)
    {
        ret = ql_cb_thread_creat(sim_handle);
        if(ret != QL_ERR_OK)
        {
            QL_SIM_LOGE("[%s] cb thread creat fail.", __func__);
        }
    }
    else
    {
        ret = ql_cb_thread_free(sim_handle);
        if(ret != QL_ERR_OK)
        {
            QL_SIM_LOGE("[%s] cb thread free deinit fail.", __func__);
            return QL_ERR_FAILED;
        }
    }
    sim_handle->status_cb = cb;

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Switches slot.
  @param [in] log_slot Logical slot to be switched.
  @param [in] phy_slot Physical slot to be switched.
  @return Whether the slot was successfully switched.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_switch_slot(QL_SIM_SLOT_E log_slot, QL_SIM_PHY_SLOT_E phy_slot)
{
    UNUSED(log_slot);
    UNUSED(phy_slot);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Gets active slots.
  @param [Out] p_active_slots Active slots.
  @return Whether the active slots were successfully obtained.
  @retval QL_ERR_OK successful.
  @retval QL_ERR_SERVICE_NOT_READY service is not ready, need to retry.
  @retval Other error code defined by ql_type.h.
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_get_active_slots(ql_sim_active_slots_t *p_active_slots)
{
    UNUSED(p_active_slots);

    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    return QL_ERR_OK;
}

/*-----------------------------------------------------------------------------------------------*/
/**
  @brief  Registration server error callback. Currently, only if the server exits abnormally,
  the callback function will be executed, and the error code is QL_ERR_ABORTED;
  @param[in] cb  Callback function
  @return
  QL_ERR_OK - successful
  Other - error code defined by ql_type.h
  */
/*-----------------------------------------------------------------------------------------------*/
int ql_sim_set_service_error_cb(ql_sim_service_error_cb_f cb)
{
    if(NULL == sim_handle)
    {
        QL_SIM_LOGE("[%s] sim handle not init.", __func__);
        return QL_ERR_NOT_INIT;
    }

    sim_handle->server_cb = cb;

    return QL_ERR_OK;
}

//int ql_sim_switch_slot(QL_SIM_SLOT_E log_slot, QL_SIM_PHY_SLOT_E phy_slot);
//int ql_sim_get_active_slots(ql_sim_active_slots_t *p_active_slots);
/*----------------------------------------------SIM API------------------------------------------*/

